// ################################################################################
//
//  WII_ESP32C3_Transceiver vR.0
//
//  Released:  16/10/2024
//
//  Author: TechKnowTone
//
// ################################################################################
/*
  TERMS OF USE: This software is furnished "as is", without technical support, and
  with no warranty, expressed or implied, as to its usefulness for any purpose. In
  no event shall the author or copyright holder be liable for any claim, damages,
  or other liability, whether in an action of contract, tort or otherwise, arising
  from, out of or in connection with the software or the use or other dealings in
  the software.

  Micro:  ESP32C3 Dev Module

  Note ESP32-C3 does not have an external USB chip, so in IDE\Tools\ you must enable
  the following three options:
  
    USB CDC On Boot: "Enabled"
    JTAG Adapter: "Disabled"
    Upload Speed: "115200"

  As the HWCDC takes some time to initialise, so you may need a delay in setup.

  This code sets up an ESP32-C3 Zero as an ESP-NOW transceiver.
  EsPressif changed the way in which ESP-NOW works from v3.0+ so this code has had
  to be altered to enable it to be compiled.
*/
// Declare Libraries
#include <Arduino.h>            // needed when using an ESP32 micro
#include <Wire.h>               // library needed to run the I2C bus interface
#include "WiFi.h"               // Wi-Fi library
#include <esp_now.h>            // ESP-NOW peer-to-peer library

// Configuration
String Release = "Release: R0";
String Date = "16/10/2024";

#define I2Cdel 20         // I2C bus timing delay for Wii
#define RGB_ON 10         // Change white brightness (max 255)
#define RGB_PIN 10        // GPIO10 is connected to the WS2812B RGB LED
#define WiiErrRet 19      // error threshold used in I2C initialisation

// I2C interface
const int SCL_PIN =  8;
const int SDA_PIN =  9;

// remote device MAC address is nulled to start with
uint8_t broadcastAddress[] = {0,0,0,0,0,0};

// 250 byte max for Tx/Rx data
// It will be used to create the Tx/Rx data structures
typedef struct ESPNOW_Buff250 {
    char ESPdata[250];
} ESPNOW_Buff250;

// Create a 250 byte Tx buffer
ESPNOW_Buff250 Tx_Buff;

// Create a 250 byte Rx buffer
ESPNOW_Buff250 Rx_Buff;

// Create a variable of type esp_now_peer_info_t to store information about the peer.
esp_now_peer_info_t peerInfo;

// Declare and initialise global variables
bool Blink;               // LED state ON/OFF
unsigned long BlinkInt;   // blink interval, default = 1000 (1 sec)
esp_err_t ESP_Err;        // returned success value
bool ESP_NOW_Init;        // == true once ESP-NOW has been initiaiised
byte I2CErr;              // flag returned from I2C comms
byte JoyXOff;             // value to correct stick centre X offset
byte JoyYOff;             // value to correct stick centre Y offset
uint32_t Millis;          // current millis()
String myMAC;             // MAC address if this device
uint32_t next2ms;         // 2ms loop timer trigger
uint32_t next40ms;        // 40ms loop timer trigger
uint32_t nextBlink;       // LED blink timer
int nunchuk_address;      // Nunchuk I2C address (0x52)
byte received_data[16];   // Array storing data received from Wii controller
int reportEn;             // reports data for testing if >= 1. Set this to 0 for transparent telemetry mode
int Rx_Chk;               // Rx checksum
int Rx_len;               // >0 when a packet has been received
int Rx_Pnt;               // receive pointer for data management
bool RxRecvEvent;         // set == true when a OnDataRecv() call back event has occured
uint32_t t0,t1;           // microseconnd function timers
long t1_t0;               // calc of t1 - t0
long t1t0Cnt;             // number of t1-t0 readings
long t1t0Max;             // max of t1-t0 times
long t1t0Min;             // min of t1-t0 times
long t1t0Sum;             // sum of t1-t0 for averaging
int Task40ms;             // task pointer for 40ms tasks
bool Tx;                  // status of transmission, true == ok, false == failed
int Tx_CB;                // == 1 call back is clear to send, == -1 resend
char Tx_Char;             // serial byte received from usb port, to be sent over WiFi
int Tx_Chk;               // Tx checksum
bool Tx_Echo;             // ==true to echo serial data at port level, default false
int Tx_ErrCnt;            // disconnects if error count is too high
bool Tx_Hd;               // == true if a '$' text header needs to be included in a block
int Tx_len;               // > 0 when WiFi Tx buffer contains data to be sent
bool WiFiConnected;       // == true if a WiFi connection has been established
String WiFi_Dev;          // the name of the remote device establishing the connection
byte WiFiRxMacAddr[6];    // senders MAC address, which could be broadcast FF:FF:FF:FF:FF:FF
int WiFi_State;           // WiFi connection state machine
byte WiFiTxMacAddr[6];    // transmit MAC address, which should match myMAC address
int WiiCal;               // counter used in offset calibration
bool Wii_Connected;       // == true if a Wii contoller is detected
int Wii_Cnt;              // counter used in Wii connection code
int WiiErrCnt;            // number of attempts to initialse the Nunchuk
int WiiTask;              // initialisation task pointer
char WiiType;             // device type, 'N' == Nunchuk, 'C' == Classic
bool Wii_Tx;              // ==true to load I2C data into the Tx buffer
bool XYOffAdj;            // if == true adjust XY centre offsets

// --------------------------------------------------------------------------------

void setup() {
  // Initialise default USB serial port
  Set_Defaults();
  Millis = millis();
  Serial.begin(115200); // baud rate for serial monitor USB Rx/Tx comms
  // Allow time for the USB hardware to be initialised, but don't simply wait for
  // Serial, otherwise you can't simply power the transceiver off a USB battery
  // source, as it will fail to run, as it will wait for Serial indefinately.
  while (!Serial && ((millis() - Millis) < 200)) {yield();}

  Serial.print("\n\n"); // double line feed
  Serial.println("Wii_ESP32C3_Transceiver");
  Serial.println(Release);
  Serial.println(Date);
  Serial.println("HWCDC in " + String(Millis) + "ms");

  Wire.setPins(SDA_PIN, SCL_PIN);   // Set the I2C pins before begin
  Wire.begin();                     // join i2c bus (address optional for master)

  // initialise WiFi link
  WiFi.mode(WIFI_MODE_STA);
  // new code for ESP32 lib v3.0+
  while (!WiFi.STA.started()) {delay(10);}
  // Once started we can get the micros MAC address
  myMAC = WiFi.macAddress();
  Serial.println(String() + "MAC: " + myMAC);
  // WiFi.disconnect(); // I don't think we need this
  // For this board we got: EC:DA:3B:BD:4B:AC

  runPOST();

  Sync_Timers();
}

// --------------------------------------------------------------------------------

void loop() {
  // The loop function runs over and over again forever
  //##############################################################################
  // Blink RGB LED
  //##############################################################################
  // The RGB LED colour reflects the connection status of the transceiver
  // Red    - no I2C, and no Wi-Fi connection
  // Purple - Wii I2C connected, but no Wi-Fi connection
  // Blue   - Wii I2C connected and Wi-Fi connection
  // Green  - no I2C, but Wi-Fi connection
  Millis = millis();
  if ((Millis - nextBlink) >= BlinkInt) {
    nextBlink = Millis;
    if (Blink) {
      //                                              Grn,Red,Blu
      if (WiFiConnected) {
        // Wi-Fi is connected
        if (Wii_Connected) {rgbLedWrite(RGB_PIN,  0,  0,RGB_ON);}         // Blue
                      else {rgbLedWrite(RGB_PIN,RGB_ON,  0,  0);}         // Green
      } else {
        // Wi-Fi is not connected
        if (Wii_Connected) {rgbLedWrite(RGB_PIN,  0,RGB_ON/2,RGB_ON/2);}  // Purple
                      else {rgbLedWrite(RGB_PIN,  0,RGB_ON,  0);}         // Red
      }
    } else {
      rgbLedWrite(RGB_PIN,0,0,0);                    // Off / black}
    }
    // Serial.print("");
    Blink = !Blink;
  }


  //##############################################################################
  // 40ms tasks
  //##############################################################################
  if (Task40ms > 0) {
    switch(Task40ms) {
      case 1: // handle Wii on I2C
        if (Wii_Connected) {
          // if Wii controller is connected read I2C data and send it over the
          // the WiFi link if a remote device is connected, by setting a flag
          Wii_Tx = ReadI2CData();
        } else {
          // if the Wii is not connected then test for it every 400ms
          WiiInitialiseTask();
          WiiTask++; if (WiiTask >= 10) {
            WiiTask = 0;    // reset Wii initialisation task pointer
            Wii_Tx = true;  // send default data every 400ms
          }
        }
        
        if ((reportEn > 0) && Wii_Connected) {
          // reports the contents of the Wii controller data registers, if a controller
          // is connected to I2C
          reportEn--;
          if (reportEn < 1) {
            reportEn = 5;
            Serial.println(F("\nWii I2C Data:")); // print title
            ReportWiiData();
          }
        } break;
    } Task40ms--;
  } else if ((millis() - next40ms) >= 40) {
      // test for 40ms time-out
      next40ms = millis();
      Task40ms = 1; // reset loop timer
  }

  //##############################################################################
  // Handle serial communications
  //##############################################################################
  // A OnDataRecv() event has occured so handle it
  if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events

  // Rx/Tx is handled as a function so that it can be called from code that is waiting
  if (WiFiConnected) {WiFiHandleRXTx();}
}

// --------------------------------------------------------------------------------

void OnDataRecv(const esp_now_recv_info_t *info, const uint8_t *incomingData, int len) {
  // Call-back function oocurs when data has been received
  // This needs to be treated like an interrupt, so speed is the essence here
  // memcpy moves data in a block, instead of individual bytes if using for() loop
  memcpy(WiFiRxMacAddr, info->src_addr, 6);   // get the senders MAC/broadcast address
  memcpy(WiFiTxMacAddr, info->des_addr, 6);   // get destination MAC address, which should be assigned peer
  memcpy(&Rx_Buff, incomingData, len);        // copy the received data into Rx_Buff array
  Rx_len = len - 1; Rx_Pnt = 0;               // len contain the length of data
  RxRecvEvent = true;                         // set the flag to say data received
}

// --------------------------------------------------------------------------------

void OnDataRecvHandler() {
  // Called from main loop() when data is received in OnDataRecv().
  RxRecvEvent = false;
  // incoming data will always be two or more characters, ie. text == $+++
  if (Rx_len < 2) {Serial.println("Rx Error - no Rx data"); return;}

  // when not connected data will be accepted from any remote device
  // once connected the MAC address must match the registered device
  if (!WiFiConnected) {
    // currently not conencted so listen to any remote device that tries to connect
    // record remote devices MAC address
    broadcastAddress[0] = WiFiRxMacAddr[0]; broadcastAddress[1] = WiFiRxMacAddr[1];
    broadcastAddress[2] = WiFiRxMacAddr[2]; broadcastAddress[3] = WiFiRxMacAddr[3];
    broadcastAddress[4] = WiFiRxMacAddr[4]; broadcastAddress[5] = WiFiRxMacAddr[5];
  } else {
    // connected so check that the data is from the established connection
    // ignore Rx data if MAC address does not match stored device MAC
    if (broadcastAddress[0] != WiFiRxMacAddr[0]) {return;}
    if (broadcastAddress[1] != WiFiRxMacAddr[1]) {return;}
    if (broadcastAddress[2] != WiFiRxMacAddr[2]) {return;}
    if (broadcastAddress[3] != WiFiRxMacAddr[3]) {return;}
    if (broadcastAddress[4] != WiFiRxMacAddr[4]) {return;}
    if (broadcastAddress[5] != WiFiRxMacAddr[5]) {return;}
  }
    
  // while (Rx_len > 0) {
  //   // data has been received previously, so complete passing it to the serial port
  //   // before using the new data
  //   // reset the data in Rx_Buff[] to nulls as we go, then clear the flag
  //   if (Serial.availableForWrite() > 0) {
  //     Serial.write(Rx_Buff.ESPdata[Rx_Pnt]); Rx_Buff.ESPdata[Rx_Pnt] = 0;
  //     Rx_Pnt++; if (Rx_Pnt >= Rx_len) {Rx_len = 0;} // stop at end of data
  //   }
  // }
  
  // // we are now ready to handle the new packet of data
  // memcpy(&Rx_Buff, incomingData, len);  // copies received data into Rx buffer
  
  if (WiFiConnected) {
    // WiFi connected so perform checksum check and if ok main loop will extract
    // data if Rx_len > 0
    WiFiRxGetChecksum(Rx_len);  // generate a checksum from Rx data
    if (Rx_Chk != Rx_Buff.ESPdata[Rx_len]) {
      // Rx_len = 0;
      // Serial.println("Rx Error - checksum");
    }
  } else {
    // If not connected this must be the first packet received, which is the name of
    // the remote device used to initiate a connection, so accept this name and 
    // enter connected transparent mode
    WiFiGetRemoteName();
    WiFiRegisterPeer();
    WiFiConnected = true;
  }
  // Serial.println("Rx");
}

// --------------------------------------------------------------------------------

void OnDataSent(const uint8_t *mac_addr, esp_now_send_status_t status) {
  // Callback when data is sent
  t1 = micros();  // record the loop back time to get here in microseconds
  // Serial.print("Send Status: ");
  if (status == 0){
    // No Tx errors detected so examine the loopback time
    // Tests on different packet data sizes revealed the following delays:
    // Data   Avg   Min    Max    Baud
    //   8    864   558   37228    74 kb/s avg
    //  16    896   573   34580   143 kb/s avg
    //  64    976   655   37116   656 kb/s avg
    //  96    1070  701   38033   897 kb/s avg
    // 128    1055  734   37447   1.2 Mb/s avg
    // 160    1039  786   39145   1.5 Mb/s avg
    // 192    1102  835   37813   1.7 Mb/s avg
    // 224    1232  870   37684   1.8 Mb/s avg
    // 250    1272  902   37289   1.9 Mb/s avg
    //
    // loop delay increases with packet size, as does efficiency
    // above baud rate is based on 10 bits/byte
    // t1_t0 = t1 - t0;
    // t1t0Cnt++; t1t0Sum += t1_t0;
    // if (t1_t0 < t1t0Min) {t1t0Min = t1_t0;}
    // if (t1_t0 > t1t0Max) {t1t0Max = t1_t0;}
    // Serial.println(String(DataTxLen) + "\t" + String(t1_t0) + "\t" + String(t1t0Sum/t1t0Cnt) + "\t" + String(t1t0Min) + "\t" + String(t1t0Max));
    Tx = true;
    Tx_CB = 1;      // set flag to indicate that a call back state has occured
    Tx_ErrCnt = 0;  // clear the error flag in case of a glitch
  }
  else{
    Serial.println("Tx Failed " + String(t1 - t0));
    Tx = false;
    Tx_CB = -1;  // set flag to indicate that a resend is required
    Tx_ErrCnt++; if (Tx_ErrCnt >= 5) {
      // too many send errors so break this connection
      WiFiConnected = false;
      // clear the MAC address of the previous remote device
      broadcastAddress[0] = 0; broadcastAddress[1] = 0;
      broadcastAddress[2] = 0; broadcastAddress[3] = 0;
      broadcastAddress[4] = 0; broadcastAddress[5] = 0;
      Serial.println("WiFi Disconnected");
    }
  }
  // Serial.println("Tx");
}

// --------------------------------------------------------------------------------

void runPOST() {
  // called when the code first runs
  //##############################################################################
  // Initialise the Nunchuk controller
  //##############################################################################
  // iniaitalise the Wii controller and confirm its presence by setting a flag
  Serial.println(F("\nInitialising Wii controller over I2C..."));
  WiiInitialise();
  if (WiiErrCnt > WiiErrRet) {
    Serial.println(F("Wii controller initialisation failed!"));
    Wii_Connected = false;
  } else {
    Wii_Connected = true;
    Serial.print(F("Wii controller initialised in ")); Serial.println(WiiErrCnt);
    ReadI2CIdent();   // determine the controller type
    // ReadI2CCalData(); // read calibration data
    XYOffAdj = false; WiiCal = 0; ReadI2CData();
  }

  WiFi_Init_ESP_NOW();

  Serial.println(F("\nPOST completed."));
  Serial.println(F("Listening..."));
  delay(100);
}

// --------------------------------------------------------------------------------

void Set_Defaults() {
  // define initial conditions for power-up and soft RESET
  Blink = false;              // LED state ON/OFF
  BlinkInt = 1000;            // blink interval, default = 1000 (1 sec)
  ESP_NOW_Init = false;       // == true once ESP-NOW has been initiaiised
  I2CErr = 1;                 // used to detect transmission errors
  JoyXOff = 142;              // value to correct stick centre X offset
  JoyYOff = 137;              // value to correct stick centre Y offset
  nextBlink = millis();       // timer for LED blink
  nunchuk_address = 0x52;     // Nunchuk I2C address (0x52)
  reportEn = 0;               // reports data for testing if >= 1. Set this to 0 for transparent telemetry mode
  Rx_Chk = 0;                 // Rx checksum
  Rx_len = 0;                 // >0 when a packet has been received
  Rx_Pnt = 0;                 // receive pointer for data management
  RxRecvEvent = false;        // set == true when a OnDataRecv() call back event has occured
  Task40ms = 0;               // task pointer for 40ms tasks
  Tx = false;                 // status of transmission, true == ok, false == failed
  Tx_CB = 1;                  // == 1 call back is clear to send, == -1 resend
  Tx_Char = 0;                // serial byte received from usb port
  Tx_Chk = 0;                 // Tx checksum
  Tx_Echo = false;            // ==true to echo serial data at port level, default false
  Tx_ErrCnt = 0;              // disconnects if error count is too high
  Tx_Hd = false;              // == true if a $ text header needs to be included in a block
  Tx_len = 0;                 // >0 when WiFi Tx buffer contains data to be sent
  WiFiConnected = false;      // == true if a WiFi connection has been established
  WiFi_State = 0;             // WiFi connection state machine
  WiiCal = 0;                 // counter used in offset calibration
  Wii_Connected = false;      // true if a Wii contoller is detected
  Wii_Cnt = 0;                // counter used in Wii connection code
  WiiErrCnt = 0;              // number of attempts to initialse the Nunchuk
  WiiTask = 0;                // initialisation task pointer
  WiiType = ' ';              // device type, 'N' == Nunchuk, 'C' == Classic
  Wii_Tx = false;             // ==true to load I2C data into the Tx buffer
  XYOffAdj = false;           // if == true adjust XY centre offsets
}

// --------------------------------------------------------------------------------

void Sync_Timers() {
  // Initialise loop timers
  while (millis() < 10) {yield();}
  next2ms = millis() - 1;     // prime 2ms loop timer
  next40ms = millis() - 10;   // 40ms loop timer trigger
  nextBlink = millis();       // LED blink timer
}

// --------------------------------------------------------------------------------


