
// --------------------------------------------------------------------------------

void ClearI2CData() {
  // whenh Wii device is not connected we set the Rx buffer data to defaults
  // this assumed Nunchuk data
  received_data[0] = 128;
  received_data[1] = 128;
  received_data[2] = 0;
  received_data[3] = 0;
  received_data[4] = 0;
  received_data[5] = 3;
}

// --------------------------------------------------------------------------------

String GetBIN(byte zVal) {
  // returns an 8 character binary representation of zVal
  String zBIN$ = "";
  // convert zVal into binary
  if (zVal & 0B10000000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B01000000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00100000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00010000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00001000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000100) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000010) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000001) {zBIN$ += "1";} else {zBIN$ += "0";}
  return zBIN$;
}

// --------------------------------------------------------------------------------

String getHex(uint8_t zV) {
  // Returns a string in HEX == zV
  String zRet$ = ""; int zL = zV/16;
  if (zL < 10) {zRet$ = char(48 + zL);} else {zRet$ = char(55 + zL);}
  zL = zV - (16 * zL);
  if (zL < 10) {zRet$ += char(48 + zL);} else {zRet$ += char(55 + zL);}
  return zRet$;
}

// --------------------------------------------------------------------------------

void PackI2CData() {
  // Copy the Nunchuk data into the 8 byte transmit buffer with a checksum byte
  // use first byte as a header to define the type of data
  Tx_Buff.ESPdata[0] = WiiType; 
  Tx_Buff.ESPdata[1] = received_data[0];
  Tx_Buff.ESPdata[2] = received_data[1];
  Tx_Buff.ESPdata[3] = received_data[2];
  Tx_Buff.ESPdata[4] = received_data[3];
  Tx_Buff.ESPdata[5] = received_data[4];
  Tx_Buff.ESPdata[6] = received_data[5];

  // calculate checksum byte for Wii data only, ignore WiiType
  Tx_Buff.ESPdata[7] = (received_data[0] ^ received_data[1] ^ received_data[2] ^ received_data[3] ^ received_data[4] ^ received_data[5]);
  Tx_len = 8; // adjust the pointer for potential serial data to be added
}

// --------------------------------------------------------------------------------

void ReadI2CCalData() {
  // read a block of data from the Nunchuk calibration registers, defined as:
  /*
    1   0G value of X-axis [9:2]
    2   0G value of Y-axis [9:2]
    3   0G value of Z-axis [9:2]
    4   LSB of Zero value of X,Y,Z axes
    5   1G value of X-axis [9:2]
    6   1G value of Y-axis [9:2]
    7   1G value of Z-axis [9:2]
    8   LSB of 1G value of X,Y,Z axes
    9   Joystick X-axis maximum
    10  Joystick X-axis minimum
    11  Joystick X-axis center
    12  Joystick Y-axis maximum
    13  Joystick Y-axis minimum
    14  Joystick Y-axis center
    15  Checksum
    16  Checksum

  */
  Wire.beginTransmission(nunchuk_address);  // Start communication with the Nunchuck.
  Wire.write(0x20);                         // We want to start reading at register (20 hex)
  I2CErr = Wire.endTransmission();        // End the transmission and check ACK
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond
  //
  // check to ensure that the Nunchuk has responded
  if (I2CErr == 0) {
    // no error reported so read the sixteen Nunchuk registers
    Wire.requestFrom(nunchuk_address,16);    // Request 16 bytes from the Nunchuck
    for(int i = 0; i < 16; i++) received_data[i] = Wire.read(); //Copy the bytes to the received_data array

    if (reportEn > 0) {
      Serial.println(F("\nWii Calibration Data:")); // send title
      ReportWiiCalData();
    }
  }
}

// --------------------------------------------------------------------------------

bool ReadI2CData() {
  // read a block of data from the Nunchuk controller
  Wire.beginTransmission(nunchuk_address);  // Start communication with the Nunchuck.
  Wire.write(0x00);                         // We want to start reading at register (00 hex)
  I2CErr = Wire.endTransmission();        // End the transmission and check ACK
  if (I2CErr != 0) {Serial.println("ReadI2CData Err " + String(I2CErr));}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond
  //
  // check to ensure that the Nunchuk has responded
  if (I2CErr == 0) {
    // no error reported so read the six Nunchuk registers
    Wire.requestFrom(nunchuk_address,6);     // Request 6 bytes from the Nunchuck
    // Wii device may send less than requested
    byte zB = 0;
    while(Wire.available()) {
      received_data[zB] = Wire.read();
      zB++;
    }
    // if (zB != 6) {Serial.println("only " + String(zB) + " data bytes");}
    // check data validity in case Nunchuk has crashed. Under such cercumstances
    // all data registers will appear to hold a value of 255
    // we ignore the joystick registers as they may actually be at 255
    if ((received_data[2] == 255) && (received_data[3] == 255) && (received_data[4] == 255)) {
      I2CErr = 1;
    } else {
      // Nunchuk appears to be working correctly
      // we will use the 4th reading as being the centre value
      // both Nunchuk and Classic are mapped from 0 - 255
      if (WiiCal < 5) {
        WiiCal++;
        if ((WiiCal == 4) && (WiiType == 'N')) {
          JoyXOff = received_data[0];
          JoyYOff = received_data[1];
          XYOffAdj = true;
        }
      }
      if (XYOffAdj) {
        // XY offset adjustment enabled, so re-map received values
        // this only applies to a Nunchuk controller
        if (received_data[0] <= JoyXOff) {
          received_data[0] = map(received_data[0],0,JoyXOff,0,128);
        } else {received_data[0] = map(received_data[0],JoyXOff,255,128,255);}
        if (received_data[1] <= JoyYOff) {
          received_data[1] = map(received_data[1],0,JoyYOff,0,128);
        } else {received_data[1] = map(received_data[1],JoyYOff,255,128,255);}
      }
    }
    return true;  // data read successfully so send over WiFi
  } else {
    // we re-initialise the Nunchuk if a failure in the I2C comms occurs
    // if it fails to reinitialise we flag the failure
    WiiInitialise();
    if (WiiErrCnt > WiiErrRet) {
      if (Wii_Connected) {Serial.println(F("Wii disconnected!"));}
      // clear all of the data
      ClearI2CData(); Wii_Connected = false; WiiType = '-';
    }
    return false;  // data not read successfully so don't send over WiFi
  }
}

// --------------------------------------------------------------------------------

void ReadI2CIdent() {
  // read a block of data from the Nunchuk extension register
  Wire.beginTransmission(nunchuk_address);  // Start communication with the Nunchuck.
  Wire.write(0xFA);                         // We want to start reading at register (FA hex)
  I2CErr = Wire.endTransmission();          // End the transmission and check ACK
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond
  //
  // check to ensure that the Nunchuk has responded
  if (I2CErr == 0) {
    // no error reported so read the six Nunchuk registers
    WiiType = 0;  // clear the current type flag incase controllers are swapped out
    Wire.requestFrom(nunchuk_address,6);      //Request 6 bytes from the Nunchuck
    for(int i = 5; i >=0; i--) received_data[i] = Wire.read(); //Copy the bytes to the received_data array
    // check data validity in case Nunchuk has crashed. Under such cercumstances
    // all data registers will appear to hold a value of 255
    // we ignore the joystick registers as they may actually be at 255
    WiiType = '-';  // Wii device unknown
    if ((received_data[2] == 255) && (received_data[3] == 255) && (received_data[4] == 255)) {
      I2CErr = 1;
    } else {
      // looks ok so determine the controller type
      // 0xA4200000 - Nunchuk
      // 0xA4200101 - Classic Controller
      if ((received_data[5] == 0) && (received_data[4] == 0) && (received_data[3] == 164) && (received_data[2] == 32) && (received_data[1] == 0) && (received_data[0] == 0)) {
        // Nunchuck controller - 0xA4200000
        Serial.println(F("\nWii Nunchuk connected")); // send title
        WiiType = 'N';
      } else if ((received_data[5] == 0) && (received_data[4] == 0) && (received_data[3] == 164) && (received_data[2] == 32) && (received_data[1] == 1) && (received_data[0] == 1)) {
        // Classic controller - 0xA4200101
        Serial.println(F("\nWii Classic connected")); // send title
        WiiType = 'C';
      } else if ((received_data[5] == 1) && (received_data[4] == 0) && (received_data[3] == 164) && (received_data[2] == 32) && (received_data[1] == 1) && (received_data[0] == 1)) {
        // Classic controller - 0xA4200101
        Serial.println(F("\nWii Classic Pro connected")); // send title
        WiiType = 'P';
      } else {
        // unrecognised type, so send data to serial port
        Serial.println("Unknown Wii type:");
        Serial.println("Rx[0]:" + String(received_data[0]));
        Serial.println("Rx[1]:" + String(received_data[1]));
        Serial.println("Rx[2]:" + String(received_data[2]));
        Serial.println("Rx[3]:" + String(received_data[3]));
        Serial.println("Rx[4]:" + String(received_data[4]));
        Serial.println("Rx[5]:" + String(received_data[5]));
        Serial.println();
      }
    }
  }
  if (I2CErr == 0) {
    // Wii device appears to have responded correctly so report device ident
    if (reportEn > 0) {
      Serial.println(F("\nWii Ident Data:")); // send title
      ReportWiiData();
    }
  }
}

// --------------------------------------------------------------------------------

void ReportWiiCalData() {
  // send Wii calibration data to the serial port, with formatting
  for(byte i = 0; i < 16; i++) {ReportWiiDataLn(i);}
}

// --------------------------------------------------------------------------------

void ReportWiiData() {
  // send Wii data to the serial port, with formatting
  for(byte i = 0; i < 6; i++) {ReportWiiDataLn(i);}
}

// --------------------------------------------------------------------------------

void ReportWiiDataLn(int zP) {
  // reports a single line for a given register zP, range 0 - 15
  Serial.print(F("Bite ")); Serial.print(zP); Serial.print(F(" = "));
  if (received_data[zP] < 100) Serial.print(F(" "));
  if (received_data[zP] < 10) Serial.print(F(" "));
  Serial.print(received_data[zP]);       //output I2C data from nunchuk
  Serial.print(F("\t"));
  // this section packs out the binary field to a constant 8 characters
  if (received_data[zP] < 128) Serial.print(F("0"));
  if (received_data[zP] < 64) Serial.print(F("0"));
  if (received_data[zP] < 32) Serial.print(F("0"));
  if (received_data[zP] < 16) Serial.print(F("0"));
  if (received_data[zP] < 8) Serial.print(F("0"));
  if (received_data[zP] < 4) Serial.print(F("0"));
  if (received_data[zP] < 2) Serial.print(F("0"));
  Serial.println(received_data[zP],BIN);       //output I2C data from nunchuk
}

// --------------------------------------------------------------------------------

void WiiInitialise() {
  // initialise the Wii over the I2C bus
  I2CErr = 1; WiiErrCnt = 0; WiiTask = 0; XYOffAdj = false; WiiCal= 0;
  Serial.println("WiiInitialise()");
  while (I2CErr != 0) {
    // wait in this loop initialising the Wii until it responds
    Wire.beginTransmission(nunchuk_address);    // Start communication with the Nunchuck
    Wire.write((byte)0xF0);                     // We want to write to register (F0 hex)
    Wire.write((byte)0x55);                     // Set the register bits as 01010101
    I2CErr = Wire.endTransmission();            // End the transmission
    delayMicroseconds(I2Cdel);                  // Allow MPU time to respond
    if (I2CErr == 0) {
      Wire.beginTransmission(nunchuk_address);  // Start communication with the Nunchuck
      Wire.write((byte)0xFB);                   // We want to write to register (FB hex)
      Wire.write((byte)0x00);                   // Set the register bits as 00000000
      I2CErr = Wire.endTransmission();          // End the transmission
      delayMicroseconds(I2Cdel);                // Allow MPU time to respond
    }
    WiiErrCnt++;                                // Track the number of attempts
    if (WiiErrCnt > WiiErrRet) {
      // no Wii device detected after 20 attempts
      Serial.println("Wii did not respond");
      break;
    }
  }
  // if called from the main loop we need to reset the timers
  Sync_Timers();
}

// --------------------------------------------------------------------------------

void WiiInitialiseTask() {
  // innitialise the Wii over the I2C bus as a series of tasks when not connected
  // this function is called every 10ms
  switch (WiiTask) {
    case 0:
      Wire.beginTransmission(nunchuk_address);  // Start communication with the Nunchuck
      Wire.write((byte)0xF0);                   // We want to write to register (F0 hex)
      Wire.write((byte)0x55);                   // Set the register bits as 01010101
      I2CErr = Wire.endTransmission();          // End the transmission
      delayMicroseconds(I2Cdel);                // Allow MPU time to respond
      break;
    case 1:
      Wire.beginTransmission(nunchuk_address);  // Start communication with the Nunchuck
      Wire.write((byte)0xFB);                   // We want to write to register (FB hex)
      Wire.write((byte)0x00);                   // Set the register bits as 00000000
      I2CErr = Wire.endTransmission();          // End the transmission
      delayMicroseconds(I2Cdel);                // Allow MPU time to respond
      if (I2CErr == 0) {
        ReadI2CIdent();
        if (WiiType != 0) {
          Wii_Connected = true;
          // Serial.println("Wii initialised");
          XYOffAdj = false; WiiCal = 0; ReadI2CData();
        } else {I2CErr = 2;}
      }
      if (I2CErr != 0) {
        // Serial.println("WiiInitialiseTask failed");
        WiiType = '-'; ClearI2CData(); Wii_Connected = false;} 
      break;
  }
}




  //###############################################################################
  //
  //  WiFi Code
  //
  //###############################################################################



// --------------------------------------------------------------------------------

void WiFiClearRxBuff(int zNum) {
  // clears the contents of the receive buffer
  for (int zP = 0; zP < zNum;zP++) {Rx_Buff.ESPdata[zP] = 0;}
}

// --------------------------------------------------------------------------------

void WiFiClearTxBuff(int zNum) {
  // clears the contents of the transmit buffer
  for (int zP = 0; zP < zNum;zP++) {Tx_Buff.ESPdata[zP] = 0;}
}

// --------------------------------------------------------------------------------

void WiFiDisconnect() {
  // Enter the WiFi disconnected state
  esp_now_del_peer(broadcastAddress);
  WiFiConnected = false;
  WiFiClearRxBuff(Tx_len);
  WiFiClearTxBuff(Tx_len);
  Tx_CB = 1;    // set ready to send flag
  // Serial.println("WiFiDisconnected!");
}

// --------------------------------------------------------------------------------

void WiFiGetRemoteName() {
  // called from the receive function when in the disconnected state. It is
  // assumed that the remote device will be establishing a connection using its
  // own name, so extract that from the Rx buffer
  WiFi_Dev = "";
  for (int zP = 0; zP < Rx_len; zP++) {
      WiFi_Dev += Rx_Buff.ESPdata[zP];
  }
  // clear Rx buffer for next Rx
  WiFiClearRxBuff(Rx_len); Rx_len = 0;
  // Serial.println("Remote: " + WiFi_Dev);
}

// --------------------------------------------------------------------------------

void WiFiHandleRXTx() {
  // called when WiFi is connected to perform Rx and Tx comms
  //##########################
  // Receive handler
  //##########################
  // If not connected to a remote device Rx_len == 0 and the following code will
  // not run. So nothing will come out of the serial port, other than connection
  // status messages, until a connection is established
  if (Rx_len > 0) {
    // Data has been received from the remote device, so pass it to the serial port
    // reset the data in Rx_Buff[] to nulls as we go, then clear the flag
    if (Serial.availableForWrite() > 0) {
      // serial buffer has space in it so send a character
      Serial.write(Rx_Buff.ESPdata[Rx_Pnt]); Rx_Buff.ESPdata[Rx_Pnt] = 0;
      Rx_Pnt++; if (Rx_Pnt >= Rx_len) {Rx_len = 0;} // stop at end of data
    }
  }

  //##########################
  // Transmit handler
  //##########################
  // If not connected, nothing will be broadcast over WiFi, and the link status will
  // be 'Disconnected', 'Listening'
  // A link has been established with a remote device so send data to it
  // At 115200 baud we could receive 14 bytes every millisecond from serial port
  // hence we will transmit the buffer contents every 2 ms, approx 30 bytes
  if ((Serial.available() > 0) && (Tx_len < 32)) {
    // serial data available so load up to 32 bytes into the WiFi Tx buffer
    if ((Tx_len == 0) || Tx_Hd) {
      // start with text header byte, or place one after Wii data
      Tx_Buff.ESPdata[Tx_len] = '$'; Tx_len++; Tx_Hd = false;
    }
    Tx_Char = Serial.read();  // get the char from the serial port
    Tx_Buff.ESPdata[Tx_len] = Tx_Char; Tx_len++;  // put it in the Tx buffer
    if (Tx_Echo) {Serial.write(Tx_Char);} // echo the received character on the serial port
  }

  if ((millis() - next2ms) >= 2) {
    next2ms = millis(); // reset the timer
    if (Tx_len > 0) {
      // The buffer contains data so add a checksum and send it
      t0 = micros();
      WiFiTxGetChecksum(Tx_len); Tx_Buff.ESPdata[Tx_len] = Tx_Chk; Tx_len++;
      ESP_Err = esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, Tx_len);
      Tx_len = 0;
      // Serial.println("WiFi Tx");
    }
    // Do we need to send Nunchuk data in the next frame?
    if (Wii_Tx) {
      // yes we do, so it is placed at the front of the buffer, with Tx_len adjusted
      // and sent with the next frame in 2ms time
      PackI2CData(); Wii_Tx = false;
      Tx_Hd = true; // set flag to include '$' at head of text block
    }
  }
}

// --------------------------------------------------------------------------------

void WiFi_Init_ESP_NOW() {
  //##############################################################################
  // Initialise ESP-NOW
  //##############################################################################
  // if we have previously tried a connection then we need to de-initialise it
  if (ESP_NOW_Init) {
    // already attempted to initialise ESP-NOW so strip everything back
  //  Serial.println("Closing ESP-NOW");
    esp_now_del_peer(broadcastAddress);
    // WiFi.disconnect();
  }

  // Init ESP-NOW and returns its status
  if (esp_now_init() != ESP_OK) {
    // Serial.println("Error initializing ESP-NOW");
    ESP_NOW_Init = false; return;
  } else {
    // Serial.println("\nESP-NOW Activated");
    // Once ESPNow is successfully Init, we will register for Send CB to
    // get the status of Trasnmitted packet
    ESP_Err = esp_now_register_send_cb(OnDataSent);
    // Serial.print("Registered send cb ");
    //      if (ESP_Err == ESP_OK) {Serial.println("OK");}
    // else if (ESP_Err == ESP_ERR_ESPNOW_NOT_INIT) {Serial.println("ESPNOW not initialised");}
    // else if (ESP_Err == ESP_ERR_ESPNOW_INTERNAL) {Serial.println("internal error");}

    // Register for a callback function that will be called when data is received
    ESP_Err = esp_now_register_recv_cb(OnDataRecv);
    // Serial.print("Registered recv cb ");
    //      if (ESP_Err == ESP_OK) {Serial.println("OK");}
    // else if (ESP_Err == ESP_ERR_ESPNOW_NOT_INIT) {Serial.println("ESPNOW not initialised");}
    // else if (ESP_Err == ESP_ERR_ESPNOW_INTERNAL) {Serial.println("internal error");}
    ESP_NOW_Init = true;
  }
}

// --------------------------------------------------------------------------------

void WiFiRegisterPeer() {
  // Register peer device
  memcpy(peerInfo.peer_addr, broadcastAddress, 6);
  peerInfo.channel = 0;  
  peerInfo.encrypt = false;

  // Add peer        
  if (esp_now_add_peer(&peerInfo) != ESP_OK){
    Serial.println("Failed to add peer");
    return;
  }
  // Register for a callback function that will be called when data is received
  esp_now_register_recv_cb(OnDataRecv);
}

// --------------------------------------------------------------------------------

void WiFiRxGetChecksum(int zlen) {
  // Performs an XOR checksum calc on the Rx buffer of length zlen
  Rx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    Rx_Chk = Rx_Chk ^ Rx_Buff.ESPdata[zP];
  }
}

// --------------------------------------------------------------------------------

void WiFiTxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Tx buffer of length zlen
  Tx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    Tx_Chk = Tx_Chk ^ Tx_Buff.ESPdata[zP];
  }
}

// --------------------------------------------------------------------------------

