
/* microSD card functions:
  Web link: https://www.arduino.cc/en/Reference/FilePrint

  appendFile(SD,path,text)        - append text string to a file defined by path
  createDir(SD,path)              - create a directory defined by 'path'
  deleteFile(SD,filepath)         - delete a file defined by 'path'
  getSDsize()                     - returns the size of the microSD card as unit64_t
  initSD()                        - initialise the microSD card
  listDir(SD,path,level)          - list a directory at 'path'
  readFile(SD,path)               - read the contents of a file to serial monitor
  removeDir(SD,path)              - remove a directory defined by 'path'
  renameFile(SD,pathOld,pathNew)  - rename file pathOld as pathNew
  SD.begin()                      - initialises the SD card
  SD.cardSize                     - returns size as uint64_t
  SD.cardType()                   - returns uint8_t for cardTypes, CARD_MMC, CARD_SD, CARD_SDHC
  SD.end()                        - call this when a card is removed. It will allow you to insert and initialise a new card
  SD.exists(file/dir path)        - tests whether a file or directory exists on the SD card
  SD.open(filepath)               - opens a file for reading
  SD.open(filepath, mode)         - opens file in FILE_READ, FILE_WRITE or FILE_APPEND modes
    file.available()              - check if there are any bytes available for reading from the file
    file.close()                  - closes the file, and ensure that any data written to it is physically saved to the SD card
    file.flush()                  - ensures that any bytes written to the file are physically saved to the SD card
    file.isDirectory()            - directories (or folders) are special kinds of files, this function reports if the current file is a directory or not
    file.name()                   - returns the file name
    file.openNextFile()           - reports the next file or folder in a directory
    file.peek()                   - read a byte from the file without advancing to the next one
    file.position()               - get the current position within the file (i.e. the location to which the next byte will be read from or written to)
    file.print(data)              - print data to the file, which must have been opened for writing. Prints numbers as a sequence of digits
    file.print(data, BASE)        - BASE (optional): the base in which to print numbers
    file.read()                   - read from the file
    file.read(buf, len)           - buf: an array of characters or bytes, len: the number of elements in buf
    file.rewindDirectory()        - rewindDirectory() will bring you back to the first file in the directory, used in conjunction with openNextFile()
    file.seek(pos)                - seek to a new position in the file, which must be between 0 and the size of the file (inclusive)
    file.size()                   - get the size of the file
    file.write(data)              - write data to the file, data: the byte, char, or string (char *) to write
    file.write(buf, len)          - buf: an array of characters or bytes, len: the number of elements in buf
  SD.remove(filename)             - removes a file from the SD card
  SD.size(filepath)               - returns the size of a file
  testFileIO(SD,path)             - test a files read/write functions
  writeFile(SD,path,text)         - write text string to file defined by 'path'
*/

// ---------------------------------------------------------------------------------

void appendFile(fs::FS &fs, const char * path, const char * message){
  // append data to a file
  if (PrintEn) {Serial.printf("Appending to file: %s\n", path);}

  File file = fs.open(path, FILE_APPEND);
  if (!file){
    Serial.println("Failed to open file for appending");
    return;
  }
  if (file.print(message)){
    if (PrintEn) {Serial.println("Message appended");}
  } else {
    Serial.println("Append failed");
  }
  file.close();
}

// ---------------------------------------------------------------------------------

void createDir(fs::FS &fs, const char * path){
  // create a directory defined by 'path'
  Serial.printf("Creating Dir: %s\n", path);
  if(fs.mkdir(path)){
    Serial.println("Dir created");
  } else {
    Serial.println("mkdir failed");
  }
}

// ---------------------------------------------------------------------------------

void deleteFile(fs::FS &fs, const char * path){
  // delete file defined by 'path'
  Serial.printf("Deleting file: %s\n", path);
  if(fs.remove(path)){
    Serial.println("File deleted");
  } else {
    Serial.println("Delete failed");
  }
}

// ---------------------------------------------------------------------------------

uint64_t getSDsize() {
  // returns the size of the SD card
  uint64_t cardSize = SD.cardSize() / (1024 * 1024);
  return cardSize;
}

// ---------------------------------------------------------------------------------

void initSD() {
  // initialise the microSD card
  ErrSD = false;  // clear the error flag
  if(!SD.begin(5)){
    Serial.println("Card Mount Failed");
    ErrSD = true; return;
  }
  cardType = SD.cardType();

  if(cardType == CARD_NONE){
    Serial.println("No SD card attached");
    ErrSD = true; return;
  }

  Serial.print("SD Card Type: " + String(cardType) + " ");
  if(cardType == CARD_MMC){
    Serial.println("MMC");
  } else if(cardType == CARD_SD){
    Serial.println("SDSC");
  } else if(cardType == CARD_SDHC){
    Serial.println("SDHC");
  } else {
    Serial.println("UNKNOWN");
  }
}

// ---------------------------------------------------------------------------------

void listDir(fs::FS &fs, const char * dirname, uint8_t levels){
  // list a directory at 'path'
  Serial.printf("Listing directory: %s\n", dirname);

  File root = fs.open(dirname);
  if(!root){
    Serial.println("Failed to open directory");
    return;
  }
  if(!root.isDirectory()){
    Serial.println("Not a directory");
    return;
  }

  File file = root.openNextFile();
  while(file){
    if(file.isDirectory()){
      Serial.print("  DIR : ");
      Serial.println(file.name());
      if(levels){
        listDir(fs, file.name(), levels -1);
      }
    } else {
      Serial.print("  FILE: ");
      Serial.print(file.name());
      Serial.print("  SIZE: ");
      Serial.println(file.size());
    }
    file = root.openNextFile();
  }
}

// ---------------------------------------------------------------------------------

void loadDir(fs::FS &fs, const char * dirname, int16_t zListPnt){
  // load the contents of a directory at 'path'
  // Serial.printf("Loading directory: %s\n", dirname);

  File root = fs.open(dirname);
  if(!root){
    Serial.println("Failed to open directory");
    return;
  }
  if(!root.isDirectory()){
    Serial.println("Not a directory");
    return;
  }

  ListCnt = 0;        // counter of files in this directory
  int16_t zNum = 0;   // counter of up to 6 files
  File file = root.openNextFile();
  while(file){
    if(file.isDirectory()){
      // Ignore directories if they exist
    } else {
      if ((ListCnt >= zListPnt) && (zNum < 6)) {
        FileList$[zNum] = file.name();  // load the filename
        zNum++;                         // list pointer 0 - 5
      }
      ListCnt++;
    }
    file = root.openNextFile();
  }
}

// ---------------------------------------------------------------------------------

void readFile(fs::FS &fs, const char * path){
  // read the contents of a file to serial monitor
  Serial.printf("Reading file: %s\n", path);

  File file = fs.open(path);
  if(!file){
    Serial.println("Failed to open file for reading");
    return;
  }

  Serial.print("Read from file: ");
  while(file.available()){
    Serial.write(file.read());
  }
  file.close();
}

// ---------------------------------------------------------------------------------

void removeDir(fs::FS &fs, const char * path){
  // remove a directory defined by 'path'
  Serial.printf("Removing Dir: %s\n", path);
  if(fs.rmdir(path)){
    Serial.println("Dir removed");
  } else {
    Serial.println("rmdir failed");
  }
}

// ---------------------------------------------------------------------------------

void renameFile(fs::FS &fs, const char * path1, const char * path2){
  // rename file pathOld as pathNew
  Serial.printf("Renaming file %s to %s\n", path1, path2);
  if (fs.rename(path1, path2)) {
    Serial.println("File renamed");
  } else {
    Serial.println("Rename failed");
  }
}

// ---------------------------------------------------------------------------------

void testFileIO(fs::FS &fs, const char * path){
  // test a files read/write funstions
  File file = fs.open(path);
  static uint8_t buf[512];
  size_t len = 0;
  uint32_t start = millis();
  uint32_t end = start;
  if(file){
    len = file.size();
    size_t flen = len;
    start = millis();
    while(len){
      size_t toRead = len;
      if(toRead > 512){
        toRead = 512;
      }
      file.read(buf, toRead);
      len -= toRead;
    }
    end = millis() - start;
    Serial.printf("%u bytes read for %u ms\n", flen, end);
    file.close();
  } else {
    Serial.println("Failed to open file for reading");
  }


  file = fs.open(path, FILE_WRITE);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }

  size_t i;
  start = millis();
  for(i=0; i<2048; i++){
    file.write(buf, 512);
  }
  end = millis() - start;
  Serial.printf("%u bytes written for %u ms\n", 2048 * 512, end);
  file.close();
}

// ---------------------------------------------------------------------------------

void writeFile(fs::FS &fs, const char * path, const char * message){
  // write string data to file defined by 'path'
  // Serial.printf("Writing file: %s\n", path);

  File file = fs.open(path, FILE_WRITE);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }
  if(file.print(message)){
    Serial.println("File written");
  } else {
    Serial.println("Write failed");
  }
  file.close();
}

// ---------------------------------------------------------------------------------
