
// --------------------------------------------------------------------------------

void AutoMoveTask() {
  // Called when vehicle movements are wanted during playing tunes, using drive tasks
  //  1 - auto forward drive task
  //  2 - auto reverse drive task
  //  3 - turn right drive task
  //  4 - turn left drive task
  switch (AutoMove) {
    case 1: // Simple dance movements.
      // This case acts as the executive for the rest. All movements are selected at
      // random, with each one returinging to the start position, before another
      // random movement is chosen. The distance moved in each case is also random.
      // Once initiated we wait at AutoMove = 99 until the DriveTask ends.
      AutoMove = 10 * (1 + random(4));    // select the movement
      AutoStep = 35 + (random(25));       // select the 'slot' distance to travel
      Slots_ON();                         // reset and enable slot counters
      break;

    case 10: // move forwards
      AutoStopFL = AutoStep; AutoStopFR = AutoStep;
      DriveTask = 1; AutoMoveNext = 11; AutoMove = 99; break;
    case 11: // return backwards
      AutoStopFL = -AutoStep; AutoStopFR = -AutoStep;
      DriveTask = 2; AutoMoveNext = 1; AutoMove = 99; break;

    case 20: // move backwards
      AutoStopFL = -AutoStep; AutoStopFR = -AutoStep;
      DriveTask = 2; AutoMoveNext = 21; AutoMove = 99; break;
    case 21: // return forwards
      AutoStopFL = AutoStep; AutoStopFR = AutoStep;
      DriveTask = 1; AutoMoveNext = 1; AutoMove = 99; break;

    case 30: // turn right
      if (random(9) > 7) {AutoStep *= 3;}
      AutoStopFL = AutoStep; AutoStopFR = -AutoStep;
      DriveTask = 3; AutoMoveNext = 31; AutoMove = 99; break;
    case 31: // return left
      AutoStopFL = -AutoStep; AutoStopFR = AutoStep;
      DriveTask = 4; AutoMoveNext = 1; AutoMove = 99; break;

    case 40: // turn left
      if (random(9) > 7) {AutoStep *= 3;}
      AutoStopFL = -AutoStep; AutoStopFR = AutoStep;
      DriveTask = 4; AutoMoveNext = 41; AutoMove = 99; break;
    case 41: // return right
      AutoStopFL = AutoStep; AutoStopFR = -AutoStep;
      DriveTask = 3; AutoMoveNext = 1; AutoMove = 99; break;
    
    case 99: // Wait for DriveTask movement to end.
      // Then return for next movement selection.
      if (!DriveTask) {AutoMove = AutoMoveNext; Slots_ON();}
      break;
  }
}

// --------------------------------------------------------------------------------

void DriveAuto() {
  // Perform automatic driving tasks depending on task pointer.
  // Called every 20ms from loop() when DriveTask > 0.
  //  1 - auto forward drive task
  //  2 - auto reverse drive task
  //  3 - turn right drive task
  //  4 - turn left drive task
  //
  switch (DriveTask) {
    case 1: // auto forward drive task, set initial PWM values
      PwmFwdMax = 115; PWM_Lft = PWM_StartMax; PWM_Rht = PWM_StartMax; DriveTask = 10; break;
    case 2: // auto backward drive task, set initial PWM values
      PwmFwdMax = 115; PWM_Lft = -PWM_StartMax; PWM_Rht = -PWM_StartMax; DriveTask = 20; break;
    case 3: // auto turn right drive task, set initial PWM values
      PwmFwdMax = 115; PWM_Lft = PWM_StartMax; PWM_Rht = -PWM_StartMax; DriveTask = 30; break;
    case 4: // auto turn left drive task, set initial PWM values
      PwmFwdMax = 115; PWM_Lft = -PWM_StartMax; PWM_Rht = PWM_StartMax; DriveTask = 40; break;

    // subtasks associated with DriveTask = 1
    case 10:  // drive forward towards target counts
      // this task relies on slot counter readings turning OFF PWM demands
      if (PWM_Lft != 0) {
        if (IntP34Cnt > (AutoStopFL-10)) {if (PWM_Lft > PWM_StartMax) {PWM_Lft -= 2;}} // slow down
        else if (PWM_Lft < PwmFwdMax) {PWM_Lft += 1;} // slowly speed up towards target point
      }
      if (PWM_Rht != 0) {
        if (IntP39Cnt > (AutoStopFR-10)) {if (PWM_Rht > PWM_StartMax) {PWM_Rht -= 2;}} // slow down
        else if (PWM_Rht < PwmFwdMax) {PWM_Rht += 1;} // slowly speed up towards target point
      }
      else if (PWM_Lft == 0) {DriveTask = 0;}
      break;

    // subtasks associated with DriveTask = 2
    case 20:  // drive backwards towards target counts
      // this task relies on slot counter readings turning OFF PWM demands
      if (PWM_Lft != 0) {
        if (IntP34Cnt < (AutoStopFL+10)) {if (PWM_Lft < -PWM_StartMax) {PWM_Lft += 2;}} // slow down
        else if (PWM_Lft > -PwmFwdMax) {PWM_Lft -= 1;} // slowly speed up towards target point
      }
      if (PWM_Rht != 0) {
        if (IntP39Cnt < (AutoStopFR+10)) {if (PWM_Rht < -PWM_StartMax) {PWM_Rht += 2;}} // slow down
        else if (PWM_Rht > -PwmFwdMax) {PWM_Rht -= 1;} // slowly speed up towards target point
      }
      else if (PWM_Lft == 0) {DriveTask = 0;}
      break;
    // subtasks associated with DriveTask = 2

    case 30:  // turn right towards target counts
      // this task relies on slot counter readings turning OFF PWM demands
      if (PWM_Lft != 0) {
        if (IntP34Cnt > (AutoStopFL-10)) {if (PWM_Lft > PWM_StartMax) {PWM_Lft -= 2;}} // slow down
        else if (PWM_Lft < PwmFwdMax) {PWM_Lft += 1;} // slowly speed up towards target point
      }
      if (PWM_Rht != 0) {
        if (IntP39Cnt < (AutoStopFR+10)) {if (PWM_Rht < -PWM_StartMax) {PWM_Rht += 2;}} // slow down
        else if (PWM_Rht > -PwmFwdMax) {PWM_Rht -= 1;} // slowly speed up towards target point
      }
      else if (PWM_Lft == 0) {DriveTask = 0;}
      break;

    case 40:  // turn left towards target counts
      // this task relies on slot counter readings turning OFF PWM demands
      if (PWM_Lft != 0) {
        if (IntP34Cnt < (AutoStopFL+10)) {if (PWM_Lft < -PWM_StartMax) {PWM_Lft += 2;}} // slow down
        else if (PWM_Lft > -PwmFwdMax) {PWM_Lft -= 1;} // slowly speed up towards target point
      }
      if (PWM_Rht != 0) {
        if (IntP39Cnt > (AutoStopFR-10)) {if (PWM_Rht > PWM_StartMax) {PWM_Rht -= 2;}} // slow down
        else if (PWM_Rht < PwmFwdMax) {PWM_Rht += 1;} // slowly speed up towards target point
      }
      else if (PWM_Lft == 0) {DriveTask = 0;}
      break;

    default: DriveTask = 0; break;  // catch-all clears the task
  }
  // Serial.println(DriveTask);
}

// --------------------------------------------------------------------------------

void GoToSleep() {
  // called when powering down robot  
}

// --------------------------------------------------------------------------------

void MainTask_DriveDemo() {
  // Performs an on-the-spot drive demo using the AutoMoveTask() and DriveTask()
  // functions. Starts with a 5 - 1 count down, which can be cancelled.
  // Called every 20ms from loop().
  if (SubDel > 0) {SubDel--; return;}

  switch (SubTask) {
    case 0: // Initialise
      Display_Text2x16("DRIVE","DEMO");
      DispDel = 500; SubDel = 100;   // wait 2s. over-ride display delay
      DispClk = 5; SubTask++; break;
    case 1: // Count-down
      DisplayText1S24(String( DispClk));
      DispDel = 500; SubDel = 50;   // wait 1s. over-ride display delay
      DispClk--; if (DispClk < 0) {SubTask++;}
      break;
    case 2: // Run...
      DispMode = 61; DispDel = 25;
      AutoMove = 1; SubTask++;
      break;
    case 3: // Monitor
      AutoMoveTask();                   // call the move executive task function
      if (!DriveTask) {SubDel = 50;}    // delay between movements
      break;
  }
}

// --------------------------------------------------------------------------------

void MainTask_Intro() {
  // This task talks or plays a tune at the start, after a delay
  // When active, this task is called every 20ms from the main loop
  if (SubDel > 0) {SubDel--; return;}

  switch (SubTask) {
    case 0: // set the initial pause of 3 second
      SubDel = 50; SubTask++; break;
    case 1: // play audio
      if (SDin)  {
        // SD card has not been detected during POST checks
        // play two notes, with the attenuator ON
        TonePlay(1047); delay(100);
        TonePlay(523); delay(150);
        ToneDetach();
        synchLoopTimers(); SubDel = 5; SubTask = 3;
      } else {
        // SD card was detected, so play intro audio
        SetLedMode(6);                  // set LEds to peak voice tracking
        SetHeadMode(2);                 // talking, Head movement
        TalkAdd("/P/HE0005.WAV[50]");   // Hello. Seargent TankBot here. +1 sec delay
        TalkAdd("/P/HO0000.WAV");       // How are you today
      } 
      SubTask++; break;
    case 2: // wait for audio to finish
      if (!Talk) {SubTask++; SubDel = 5;}
      break;
    case 3: // audio has finished talking
      SetHeadMode(1); SubDel = 60; SubTask++; break;
    case 4: // go to default sleep mode
      SetMainMode(0); break;
  }
}

// --------------------------------------------------------------------------------

void MainTask_TrackWall(int zMode) {
  // LTOF/Sonar demo, dual function depending on zMode value
  // zMode = 0 backs away from an approaching target
  // zMode = 1 maintains a fixed distance from a wall, stays in this mode until reset
  // refer to Calibration document for ranging diagrams and limits
  if (MainRun) {return;}
  
  MainRun = true; // prevent re-entry of task
  switch (SubTask) {
    case 0: // initialise, set range limits and LED mode
      // R0 - minimum range, stopping point
      // RM - max range, object detected
      Status$ = "Initialising";
      R0 = 80; RM = 500; Drive = 0; SetLedMode(5);
      DrivePWM = PWM_StartMax; PWM_Start = PWM_StartMin; Slots_ON();
      if (zMode == 0) {
        // Back-away mode, so load range data for this
        // Serial.println(F("> MainTask_BackAway()"));
        // R1 - range below which we slow down
        // R2 - range which detemines max backaway speed
        // R3 - range as which we start to backaway
        R0 = 80; R1 = 150; R2 = 180; R3 = 450;
        PwmFwdMax = 140; PwmBckMax = 160;
      } else {
        // Tracking mode, so load range data for this
        // Serial.println(F("> MainTask_TrackWall()"));
        // R1 - range below which we slow down
        // R2 - range which detemines max backaway speed
        // R3 - range at which we start to backaway
        // R4 - range at which we stop going forward
        // R5 - range at which we reach max speed going forward
        // R6 - range at which we start going forward or slow as we lose the target
        R0 = 80; R1 = 120; R2 = 150; R3 = 265; R4 = 300; R5 = 400; R6 = 435;
        PwmFwdMax = 140; PwmBckMax = 140;
      } SubTask++; break;
    case 1:
      // initial entry point or after a movement
      if (LedMode != 5) {SetLedMode(5);}
      Status$ = "Scanning"; SubTask = 2; break;
    case 2: // wait for a wall to appear and if necessary change direction
      // only move onto SubTask 3 when a wall appears
      if (LedMode != 5) {SetLedMode(5);}
      if ((Range >= R0) && (Range <= R3)) {
        // target approaching in either mode
        Status$ = "Backaway";
        SubTask = 3; Drive = abs(Drive);
        if (Drive != 3) {
          // change drive direction from forwards to backwards
          Drive = 3;
        }
      } else if ((Range >= R4) && (Range <= RM) && (zMode == 1)) {
        // in tracking mode and target is within range to follow
        Status$ = "Tracking";
        SubTask = 3; Drive= abs(Drive);
        if (Drive != 1) {
          // change drive direction from backards to forwards
          Drive = 1;
        }
      } break;
    case 3: // scan for wall and respond
      if (LedMode != 5) {SetLedMode(5);}
      if (Range >= R0) {
        // range is above minimum value
        if (Range > RM) {
          // target has moved out of range so stop
          Status$ = "Target Lost";
          PWM_Lft = 0; PWM_Rht = 0;
          SubTask = 4;
        } else if ((zMode > 0) && (Range > R6)) {
          // target tracking mode
          // if driving forward let it continue, but if not start moderately
          Status$ = "Target >= " + String(R6);
          Drive = 1;  // set Drive as forward
          DrivePWM = map(Range,R6,RM,PwmFwdMax,PWM_StartMin);
          PWM_Lft = DrivePWM; PWM_Rht = DrivePWM;
        } else if ((zMode > 0) && (Range >= R4) && (Range <= R6)) {
          // far wall detected so drive speed depends on how close it is
          Status$ = "Target >= " + String(R4);
          DrivePWM = map(Range, R4,R5,PWM_StartMin,PwmFwdMax);
          if (Range >= R5) {DrivePWM = PwmFwdMax;} // cap max forward drive value
          PWM_Lft = DrivePWM; PWM_Rht = DrivePWM;
          Drive = 1;
        } else if ((zMode > 0) && (Range < R4) && (Range > R3)) {
          // in mode 1 centre deadband
          Status$ = "Centre Deadband";
          PWM_Lft = 0; PWM_Rht = 0; Drive = 0;
        } else if ((Range <= R3) && (Range >= R1)) {
          // in either mode a close wall is detected so backaway speed
          // depends on how close it is
          Status$ = "Target <= " + String(R3);
          DrivePWM = map(Range,R3,R2,PWM_StartMin,PwmBckMax);
          if (Range <= R2) {Status$ = "Target <= R2"; DrivePWM = PwmBckMax;} // cap max drive value
          PWM_Lft = -DrivePWM; PWM_Rht = -DrivePWM;
          Drive = 3;
        } else if (Range < R1) {
          // in either mode a very close wall detected so slow down backaway speed
          Status$ = "Target < " + String(R1);
          if (Range > RangeMin) {DrivePWM = map(Range, R0,R1,PWM_StartMin,PwmBckMax);}
          else {DrivePWM = 0;}  //STOP, too close
          PWM_Lft = -DrivePWM; PWM_Rht = -DrivePWM;
          Drive = 3;
        } else {
          // in centre deadband so stop
          PWM_Lft = 0; PWM_Rht = 0; Drive = 0;
          SubTask = 4;
        }
      } else {
        // no wall detected but one had been previously
        SubTask = 4;
      } break;
    case 4: // stop moving and wait
      Status$ = "Watching"; Drive = -abs(Drive);
      PWM_Lft = 0; PWM_Rht = 0; Drive = 0;
      SubTask++; anyMilli = millis(); break;  // start timer for pause period
    case 5: // wait for a valid range then return to previous movement
      if (LedMode != 5) {SetLedMode(5);}
      if ((Range >= R0) && (Range <= R3)) {
        // valid reverse range
        if (Drive == -3) {
          // previously driving backwards
          TgtCnt = 1; SubTask = 3;
        } else {SubTask = 2;} // driving direction changing
      } else if ((zMode > 0) && (Range >= R4) && (Range <= RM)) {
        // valid forward range
        if (Drive == -1) {
          // previously driving forwards so continue
          TgtCnt = 1; SubTask = 3;
        } else {SubTask = 2;} // driving direction changing
      }
      if (zMode == 0) {
        // zMode = 0 back away from an approaching target
        // check after a delay to see if a return movement is needed
        Status$ = "Time: " + String((float)(millis() - anyMilli)/1000.0);
        if ((millis() - anyMilli) >= 5000) {
          if (IntP34Cnt < -2) {SubTask++;} // at least half a wheel turn backwards
          else {anyMilli = millis();}
        }
      } else if (zMode == 1) {
        // zMode = 1 maintains a fixed distance from target
        // if stood still after a delay return to ready state
        if ((millis() - anyMilli) >= 5000) {
          anyMilli = millis(); SubTask = 2;
        }
      }
      break;
    case 6:
      // prepare for return to start movement
      DrivePWM = PWM_StartMax; Drive = 0; SubTask++; break;
    case 7:
      // change direction of drive movement to forward
      SetLedMode(21); Drive = 1; SubTask++; break;
    case 8: 
      // check reached beginning or seen another wall?
      // Serial.println("Driven = " + String(IntP36Cnt));
      if (((Range >= R0) && (Range <= R3)) || ((IntP34Cnt + IntP39Cnt) >= 0)) {
        // stop driving forward
        Status$ = "Stopping";
        PWM_Lft = 0; PWM_Rht = 0; Drive = 0;
        if ((IntP34Cnt + IntP39Cnt) >= 0) {Reset_Slots();}
        SubTask = 1; SetLedMode(5);
      } else {
        // continue driving forwards, back to start point
        // if approaching start point slow down within 2 wheel turns
        Status$ = "Returning " + String(IntP34Cnt + IntP39Cnt);
        if (IntP34Cnt > -80) {if (DrivePWM > PWM_StartMax) {DrivePWM -= 2;}} // slow down
        else if (DrivePWM < PwmFwdMax) {DrivePWM += 1;} // slowly speed up towards start point
        PWM_Lft = DrivePWM; PWM_Rht = DrivePWM;
      } break;
  } MainRun = false;
}

// --------------------------------------------------------------------------------

void MotorDriveTask() {
  // Called at 50Hz from the main task loop to control motor PWMs and dither
  // Control variables are set through the Motor(,) function
  // Generate dither signal automatically, note h/w PWM is 30kHz
  // if (IsrRun) {return;}   // changes to PWM when playing audio crash the system

  Dither = -Dither; // alternating 30Hz signal
  DithON = false;   // clear the applied flag
  
  // now create PWM counter signals
  // left track motors
  int16_t zPWM = PWM_Lft; int16_t zStart = PWM_Start + 25;
  if (zPWM >  0) {
    if (zPWM < zStart) {zPWM += Dither; zPWM = constrain(zPWM,0,255); DithON = true;}
    PwmLftB = 255; PwmLftA = 255 - zPWM; DIR_FL = 1;
  } else if (PWM_Lft == 0) {
    PwmLftA = 0; PwmLftB = 0; DIR_FL = 0;
  } else {
    if (zPWM > -zStart) {zPWM += Dither; zPWM = constrain(zPWM,-255,0); DithON = true;}
    PwmLftB = 255 + zPWM; PwmLftA = 255; DIR_FL = -1;
  }
  
  // right track motors
  zPWM = PWM_Rht; 
  if (zPWM >  0) {
    if (zPWM < zStart) {zPWM += Dither; zPWM = constrain(zPWM,0,255); DithON = true;}
    PwmRhtB = 255 - zPWM; PwmRhtA = 255; DIR_FR = 1;
  } else if (PWM_Rht == 0) {
    PwmRhtA = 0; PwmRhtB = 0; DIR_FR = 0;
  } else {
    if (zPWM > -zStart) {zPWM += Dither; zPWM = constrain(zPWM,-255,0); DithON = true;}
    PwmRhtB = 255; PwmRhtA = 255 + zPWM; DIR_FR = -1;
  }
  // Now load changed PWM values here if not playing audio, otherwise do during ISR function
  if (IsrRun) {
    // If audio playback is running we can't make calls to ledcWrite() whilst interrupts
    // are occuring, so we set a flag so that they can be updated during the ISR
    uint8_t zMask = 0;  // clear the temporary mask
    if (PwmLftAL != PwmLftA) {zMask |= 0b00001000;}
    if (PwmLftBL != PwmLftB) {zMask |= 0b00000100;}
    if (PwmRhtAL != PwmRhtA) {zMask |= 0b00000010;}
    if (PwmRhtBL != PwmRhtB) {zMask |= 0b00000001;}
    PwmMask = zMask;    // update the actual mask in one step, so that the ISR gets all changes
  } else {
    if (PwmLftAL != PwmLftA) {ledcWrite(PinLftA, PwmLftA);}
    if (PwmLftBL != PwmLftB) {ledcWrite(PinLftB, PwmLftB);}
    if (PwmRhtAL != PwmRhtA) {ledcWrite(PinRhtA, PwmRhtA);}
    if (PwmRhtBL != PwmRhtB) {ledcWrite(PinRhtB, PwmRhtB);}
  }
  // remember the last values written
  PwmLftAL = PwmLftA; PwmLftBL = PwmLftB;
  PwmRhtAL = PwmRhtA; PwmRhtBL = PwmRhtB;
}

// --------------------------------------------------------------------------------

void SuspendMainTask() {
  // called when running a task and button SW1 is pressed
  // we effectively stop all activity
  MainTask = 0;               // set Null task
  // HeadTask = 0;               // effectively turn off head movement tasks
  // HeadTgtCnt = 0;             // stop any head auto-movement
  servoTgt[0] = Gun0;         // set target for centre position
  servoTgt[1] = Sen0;         // set target for centre position
  MotorPWM (0,0);                           // stop PWM drives
  if (VL53L1X_Task > 0) {VL53L1X_OFF();}    // laser disabled by default
  LedMode = 0;                              // stop LED activity
}

// --------------------------------------------------------------------------------

void ToneTask() {
  // play a tone ON/OFF
  // called every 40ms from loop()
  if (ToneDel > 0) {ToneDel--; return;} // used as a timing function
  
  switch(TonePnt) {
    case 0:
      ToneAttach();
      ledcWriteNote(PinTone, NOTE_C, 4);  // play note C, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 1:
      ledcWriteNote(PinTone, NOTE_D, 4);  // play note D, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 2:
      ledcWriteNote(PinTone, NOTE_E, 4);  // play note E, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 3:
      ledcWriteNote(PinTone, NOTE_F, 4);  // play note E, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 4:
      ledcWriteNote(PinTone, NOTE_G, 4);  // play note E, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 5:
      ledcWriteNote(PinTone, NOTE_A, 4);  // play note E, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 6:
      ledcWriteNote(PinTone, NOTE_B, 4);  // play note E, octave 4
      ToneDel = 10; TonePnt++; break;       // play for 1 sec
    case 7:
      ToneDetach();
      ToneDel = 100; TonePnt = 0; break;     // off for 2 sec
  }
}

// --------------------------------------------------------------------------------
