// ################################################################################
//
//  TankBot ESP32 v0.0 (R0)
//
//  Released:  13/08/2024
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Microcontroller:  ESP32 Dev Module  v3.0.2

    CPU Freq 240 MHz

    This code controls a tracked chassis, providing the following functions:
    * Wi-Fi control using Nunchuk or Wii Classic controllers
    * OLED Display + mirroring to PC using a custom Windows app
    * Button switches for mode & function selections
    * Two-channel RGB LED animations
    * Wheel slot sensors, left and right, for distance, speed, rpm, etc
    * Laser ranging & tracking
    * Sonar ranging & tracking & target engagement
    * Audio streamed from SD card + peak amplitude tracking
      * This include speech and the playing of music files
    
    IMPORTANT - Espressif changed the way ESP-NOW works from v3.0, which broke the original
    code and caused compiler errors. This version has been modified to work with v3.0+.
    Enjoy!
*/
// Declare Libraries
#include <FastLED.h>          // Neopixel library           v3.7.1
#include "FS.h"               // file system library
#include "SD.h"               // SD card library
#include <ESP32Servo.h>       // servo library              v3.0.5
#include "rtc_wdt.h"          // watchdog timer library
#include "SSD1306Wire.h"      // OLED display library       v4.6.1
#include "SparkFun_VL53L1X.h" // VL53L1X laser ranger lib   v1.2.12

#include <esp_now.h>          // ESP-NOW WiFi link library
#include <WiFi.h>             // ESP WiFi library

// Configuration
String Release = "TankBot ESP32 (R0)";
String Date = "13/08/2024";

File SDfile;                  // create a file object, named SDfile

// I2C address map
// 0x29 VL53L1X laser range finder
// 0x3C OLED display

// Display Settings
const int I2C_DISPLAY_ADDRESS = 0x3c;
const int SDA_PIN = 21;         // I2C SDA GPIO pin
const int SCL_PIN = 22;         // I2C SCL GPIO pin

// Initialize the oled display for address 0x3c
SSD1306Wire display(I2C_DISPLAY_ADDRESS, SDA_PIN, SCL_PIN);

// Define a task for core 0
TaskHandle_t Core0;

// Define constants for Display Monitor app.
// Items commented out // are already defined in SSD1306Wire.h
// #define BLACK 0
#define BLUE 5
#define GREEN 4
// #define INVERSE 2
#define MAGENTA 8
#define RED 3
// #define TEXT_ALIGN_LEFT 0
// #define TEXT_ALIGN_RIGHT 1
// #define TEXT_ALIGN_CENTER 2
// #define TEXT_ALIGN_CENTER_BOTH 3
#define TURQUOISE 7
// #define WHITE 1
#define YELLOW 6

// Define audio constants
#define AttPin 4              // audio attenuation pin is GPIO4
#define AudBufLen 512         // size of each audio byte buffer
#define AudBufLen_1 511       // size of each audio byte buffer - 1
#define AudPwmBits  8         // number of bits in audio PWM data
#define IsrTimeCnt 125        // set Timer limit, giving 62.5µs ISR interrupts @ 2MHz/125
#define PinTone 26            // PWM audio and tones will be sent to GPIO26
#define PwmBits  8            // number of bits in motor PWM data
#define PwmAudFreq 312500     // output PWM frequency, from 240MHz base clock

hw_timer_t *AudioTimer = NULL;

// Define compiler constants
#define autoRestMax 32000 // autoRest max counter value
#define BatCal 365.9      // calibrated voltage multiplying factor
#define BatCritical 2415  // critical battery threshold @6.6v, default = 2672, set == 0 to ignore
#define BatMax 3000       // A0 for fully charged battery voltage, == 8.2v
#define BatPin 36         // analog input pin ADC1_0 on GPIO36
#define BatWarn 2561      // battery threshold of 7.0v gives low warning
#define DbLLX 119         // JoyX deadband lower limit
#define DbULX 136         // JoyX deadband Upper limit
#define DbLLY 119         // JoyY deadband lower limit
#define DbULY 136         // JoyY deadband Upper limit
#define Gun0   1561       // servo 0 PWM value for gun centre
#define Gun60N 2218       // servo 0 PWM value for gun left -60°
#define Gun60P  958       // servo 0 PWM value for gun right +60°
#define INT_P34 34        // speed sensor interupt input FL
#define INT_P39 39        // speed sensor interupt input FR
#define INT_PIN 0         // VL53L1X GPIO interrupt to ADC input not actually used
#define LED_Pin0 16       // assign GPIO16 for front face RGB LEDs
#define LED_Pin1 17       // assign GPIO17 for weapon RGB LEDs
#define LtofDet 350       // max range for target detection in head scanning
#define LtofGnd 350       // caliubration range for ground reflection
#define LtofLimit 500     // Range max value is always limited to this figure
#define LtofMax 400       // out of range value for head scanning
#define LtofMin 80        // minimum range value
#define LtofStop 140      // range value at which autonomous movement stops
#define Max0 1800         // max value for gun servo 0
#define Max1 2400         // max value for LTOF servo 1
#define Min0 1200         // min value for gun servo 0
#define Min1  600         // min value for LTOF servo 1
#define NumStrips 2       // number of independent strips
#define PinEcho 35        // assign GPIO35 to HC-SR04 Echo pin
#define PinLftA 27        // assign GPIO 27 to PWM left motors
#define PinLftB 12        // assign GPIO 12 to PWM left motors
#define PinRhtA 33        // assign GPIO 33 to PWM right motors
#define PinRhtB 25        // assign GPIO 25 to PWM right motors
#define PinTrig  2        // assign GPIO2 to the HC-SR04 Trig pin
#define PrintEn false     // set true for debugging purposes

#define PwmFreq 30000     // PWM frequency is 30 kHz for motor H-bridge drivers
#define PWM_StartMax 55   // PWM to overcome start inertia
#define PWM_StartMin 35   // PWM to maintain movement
#define RngDpth 41        // size of RangeData[] array
#define Sen0   1438       // servo 1 PWM value for sensors centre
#define Sen60N 2004       // servo 1 PWM value for sensors left -60°
#define Sen60P  851       // servo 1 PWM value for sensors right +60°
#define sw0Pin  0         // left switch SW0 assigned to GPIO 0
#define sw1Pin 15         // right switch SW1 assigned to GPIO 15
#define TalkDepth 20      // depth of the TalkEngine() filepath ring array
#define XSHUT_PIN 14      // GPIO 14 controls VL53L1X XSHUT HIGH enable pin
 
// create an array of FastLED controller instances, so that we can use separate show events
CRGB Face_LEDs[7];  // reserve memory for front strip RGB data
CRGB Gear_LEDs[1];  // reserve memory for gear strip RGB data
CRGB Guns_LEDs[7];  // reserve memory for weapon strip RGB data, we only need 2
CRGB  Temp_LED[8];  // temp store used in LED animation

// create VL53L1X range sensor instance, and define its pins
// Note that INT_PIN is declared for the library, but not used in the code
SFEVL53L1X VL53L1X(Wire, XSHUT_PIN, INT_PIN);

// create servo instances
Servo servoInst[2]; // define gun and LTOF servos

// define servo arrays
// there are 2 servos in this robot, a gun (S0), + Head (S1)
bool servoAtt[] = {false,false};        // true if servo is attached
int16_t servoCnt[] = {0,0};             // counter used in servo movements
int16_t servoMax[] = {Gun60N,Sen60N};   // max cal. values
int16_t servoMem[2];                    // record of values sent to servos, used in fidgeting
int16_t servoMin[] = {Gun60P,Sen60P};   // min cal. values
//                     00,01 servo ref numbers 
int16_t servoPins[] = {32,13};          // servo output pins assigned
int16_t servoVal[] = {Gun0,Sen0};       // value sent to servos
int16_t servoTgt[] = {Gun0,Sen0};       // target values for servos

// Create ESP-NOW Wi-Fi structure
uint8_t broadcastAddress[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

// 250 byte max for Tx/Rx data
// It will be used to create the Tx/Rx data structures
typedef struct ESPNOW_Buff250 {
    char ESPdata[250];
} ESPNOW_Buff250;

// Create a 250 byte Tx buffer
ESPNOW_Buff250 Tx_Buff;

// Create a 250 byte receive
ESPNOW_Buff250 Rx_Buff;

// Declare audio variables
bool AttEn = true;            // audio attenuator ON flag
bool AttenRef;                // master audio attenuator setting
int16_t AudioDel = 0;         // audio task delay counter
bool AudioEn = false;         // == true once a channel has been assigned for playing audio PWM
bool AudioErr = false;        // result from setting audio PWM
int16_t AudioPause = false;   // set == true to temporarily inhibit audio
bool AudioPkEn = false;       // if set == true a peak scan will be performed when playing audio
uint8_t AudioPkVal;           // peek value obtained from scanning an audio Buffer[] 0 - 128
int16_t AudioPTsk = 0;        // if != 0 then value = playing task
int16_t AudioTask = 0;        // if != 0 then play an audio file
unsigned long Bytes_A;        // bytes stored in Buffer_A[]
unsigned long Bytes_B;        // bytes stored in Buffer_B[]
bool IsrBuf = false;          // pointer to either Buffer_A[] or Buffer_B[]
volatile uint8_t IsrEnd = 0;  // ISR play auto-ramp at end of buffer, == 0 whilst playing
bool IsrESC = false;          // ISR play autostop at end of rampd down, == false whilst playing
int16_t IsrLdn = 0;           // flag set > 0 to load more audio data
int16_t IsrPnt = 0;           // pointer used by ISR to read Buffer_n[] data
volatile bool IsrRun = false; // if == true then run ISR code, default = false
bool IsrRunF;                 // a temporary copy of IsrRun, when IsrRun is switched OFF
bool IsrStarted = false;      // == true once ISR timer has been initialised
int IsrTimer = 0;             // one of four hardware timer used for interrupts 0-3
unsigned long IsrT0,IsrT1;    // ISR performance timer used in development
int16_t Jukebox;              // audio jukebox mode, 0 - 2
int16_t JukeboxAEN;           // audio engine number used in jukebox pause
int16_t JukeboxMax;           // audio jukebox, number of tracks in this mode
int16_t JukeboxSTN;           // audio engine suspended task number used in jukebox pause
int16_t JukeboxTrk;           // audio jukebox track number
int16_t Play = 0;             // task pointer for PlayEngine() function
bool PwmAtt = false;          // == true once audio PWM is attached for the 1st time
int16_t Talk = 0;             // task pointer for TalkEngine() function
String TalkArray[TalkDepth];  // ring buffer containing filepaths to be played
bool TalkAtt = false;         // attenuator setting used with Talk
int16_t TalkCnt = 0;          // counter/pointer used in storing audio filepaths
int16_t TalkDel = 0;          // TalkEngine() pause counter
int16_t TalkPause = 0;        // interphrase pause in 20ms increments
int16_t TalkPnt = 0;          // pointer used in retrieving audio filepaths
bool ToneEn = false;          // == true if Tone PWM has been configured

// Declare SD reader variables
bool Atten;                   // == true if attenuator is ON, == false if not
String Aud$ = "";             // used with plotting function to look at recorded data
String Dir$;                  // directory part of filepath, as in /A
char DirLast = ' ';           // previous directory pointer, controls list loading
char DirPnt = 'A';            // pointer to audio folder ie. /A
uint8_t cardType;             // numeric ref to card type
bool ErrSD;                   // general error flag
String FileList$[] = {"","","","","",""};  // dir file list array
String FileName$ = "";        // name without the path and .WAV extension
String FilePath$ = "";        // SD file path
unsigned long FilePos = 0;    // position in openned file
unsigned long FileSize = 0;   // openned file size
int16_t ListCnt;              // directory file list counter
int16_t ListPnt = 0;          // pointer to first listed file
int16_t ListSel = -1;         // Pointer to selected file
bool SDin = false;            // == true if SD card is attached
bool SDopen = false;          // == true when a file has been openned
unsigned long WavEnd = 0;     // openned file PCM data end pointer
unsigned long WavSize = 0;    // openned file PCM data size

// Declare and initialise global variables
String Any$;                  // temporary string
long AnyLong;                 // any temp long value
unsigned long anyMilli;       // any millisecond timer
int anyVal;                   // any temp integer
int16_t any16t;               // any temp int16_t value
int16_t App;                  // >0 if an app is connected
int16_t AutoMove;             // if == true then move vehicle whilst playing a tune
int16_t AutoMoveNext;         // next task in move sequence
String Auton$;                // autonomous mode string
int16_t autoRest;             // returns to a rest position after a timeout
int16_t AutoStep;             // number of slot counts in auto-drives
int16_t AutoStopFL;           // flag/count for left-hand drive auto-stop function
int16_t AutoStopFR;           // flag/count for right-hand drive auto-stop function
bool BA;                      // Wii Classic button BA flag
int32_t BatAvg;               // preload averaging function with high value
int32_t BatAvgMax;            // maximum average value in this period
int16_t BatCnt;               // delay counter to set battery reading intervals
int16_t BatCritCnt;           // timer used in baterycritical testing
int16_t BatData[60];          // battery averages at 1 second intervals
int16_t BatDel;               // set a 10 second delay in 'life' prediction
int16_t BatDP;                // battery data pointer, 0 - 59
int16_t BatLast;              // previous battery average reading
bool BattNow;                 // == true causes battery reading
bool BatPwr;                  // true if battery supply detected, false if USB assumed
int16_t BatQ;                 // a measure of battery quality 1 - 5
int16_t BatRead;              // timer counter used in reading battery
unsigned long BatRep;         // time when to report battery reading over serial link
int32_t BatSum;               // preload averaging sum with 20x max value
unsigned long BatT0;          // time of first battery measurement in life prediction forecast
String BatTime$;              // time left displayed
unsigned long BatTms;         // time when battery reading was taken
int16_t BatV0;                // reference voltage used in prediction forecast
int16_t BatV1;                // lowest voltage used in prediction forecast
int16_t BatVol;               // instantaneous battery voltage
bool BB;                      // Wii Classic button BB flag
bool BDD;                     // Wii Classic button BDD flag
bool BDL;                     // Wii Classic button BDL flag
int16_t BDLcnt;               // Wii Classic button BDL counter
int BDLdwn;                   // Wii Classic button BDL timer
bool BDR;                     // Wii Classic button BDR flag
int16_t BDRcnt;               // Wii Classic button BDR counter
int BDRdwn;                   // Wii Classic button BDR timer
bool BDU;                     // Wii Classic button BDU flag
bool Bhome;                   // Wii Classic home button flag
bool BLT;                     // Wii Classic button BLT flag
bool Blink;                   // LED blink clock
int16_t BlinkCnt;             // counter for LED Blink clock
bool Border;                  // if true then display has a border
bool BotActive;               // == true if active, otherwise false
uint8_t Bright = 255;         // FastLED brightness
uint8_t Brightness;           // Display brightness
bool Bsel;                    // Wii Classic select button flag
bool Bstrt;                   // Wii Classic start button flag
bool BX;                      // Wii Classic button BX flag
int16_t BX_Cnt;               // Wii Classic button BX timer counter
bool BY;                      // Wii Classic button BY flag
int CalcCnt;                  // tracks the number of head calculations in autonomous mode
int16_t C_Cnt;                // counter used in 'C' button detection
byte C_Dn;                    // counter for 'C' button down period
byte C_Up;                    // counter for 'C' button up period
char cmdMode;                 // command mode
int16_t cmdSgn;               // cmdVal sign, 0 or -1
char cmdType;                 // command mode type
int cmdVal;                   // value associated with a cmdType
byte Col_B,Col_G,Col_R;       // current colours
int16_t CZ;                   // received button values
bool CZ_Wait;                 // flag used for button release wait function
int16_t DCX,DCY;              // display temp pointers
int16_t DIR_FL;               // direction sign, +ve forward, -ve backwards
int16_t DIR_FR;               // direction sign, +ve forward, -ve backwards
bool DispBack;                // flag used to draw a back arrow on the display
int16_t DispClk;              // a counter used for displaying time in 1 sec incs
bool DispClr;                 // set == true to clear the memory contents
int16_t DispCnt;              // counter used in display updating
int16_t DispCX,DispCY;        // display cursor coordinates
int16_t DispDel;              // >0 display delay counter, stops updates over I2C
bool DispDisp;                // set = true to immediately display memory contents
int16_t DispLast;             // display mode in previous cycle
bool DispLock;                // if == true then don't allow display mode changes
int16_t DispMode;             // the type of display for a given mode
int16_t DispModeCnt;          // counter used in display updating
bool DispMon;                 // == true if connected to OLED Mirror app
bool DispNOP;                 // set == true to stop display I2C updates
int16_t DispNow;              // if > 0 display special modes
int16_t DispNxtCnt;           // counter used in auto-branching of display
int16_t DispOveride;          // if DispLock == true then use this overide mode
int16_t DispPnt;              // points at the current display mode
int16_t DispRet;              // pointer used to return to a previous DispMode
int16_t DispTask;             // task pointer for display functions
unsigned long DispTime;       // millisecond timer used in display updates
int16_t DispTx;               // counter used with display mirror app
int16_t DispVal;              // general display value, passed from tasks
int16_t Dither;               // overlay signal used to reduce stiction
bool DithON;                  // dither applied flag, true = on, false = off
int16_t DM_AudEng;            // Display_Audio_Engine() DispMode pointer
int16_t DM_FleLst;            // Display_File_List() DispMode pointer
int16_t DML,DMR;              // DispMode pointers on mouse click
int16_t DM_TalkEng;           // Display_Talk_Engine() DispMode pointer
int16_t DmB,DmX,DmY;          // button and X,Y values associated with a display monitor mouse click
int16_t Drive;                // if > 0 then driving in various directions
int16_t DriveLast;            // previous driving direction
int16_t DriveLft;             // left-side PWM value
int16_t DrivePWM;             // PWM drive value for both wheels
int16_t DriveRht;             // right-side PWM value
int16_t DriveTask;            // task pointer for auto-drive tasks
long duration;                // length of the echo pulse returned from HC-SR04
bool ESC;                     // true to escape all functions
int16_t ESP_NOW_BA;           // broadcast address pointer, -1 to start with
bool ESP_NOW_Init;            // == true once ESP-NOW has been initiaiised
int16_t Gear;                 // value used to determine PWM Max values
uint8_t gBrightness = 255;    // brightness used in FastLED show events
bool GunAirEn;                // enables Gun movement in air defence mode
int16_t GunAmp;               // amplitude variable used in Gun moves
int16_t GunAng0,GunAng1;      // guntarget detection angles
int16_t GunCnt;               // counter variable used in Gun moves
int16_t GunDel;               // 40ms delay counter used by Gun sub-tasks
int16_t GunFire;              // if > 0 the run gun firing sequence
int16_t GunMode;              // controls the Gun movement tasks
int16_t GunNext;              // subtask pointer used in Gun moves
int16_t GunSub;               // Gun movement sub-task pointer
bool GunTgt0,GunTgt1;         // target detection flags
int16_t GunTgtAng;            // gun target engagement angle in degrees
bool HCSR04;                  // if == true then run sonar range measurement
bool HeadAirEn;               // enables Head movement in air defence mode
int16_t HeadAngle;            // head angle in degrees, 0 centre
int16_t HeadDel;              // 40ms delay counter used by head sub-tasks
int16_t HeadInc;              // head angle increment in degrees, 2 default
int16_t HeadMode;             // controls the head movement tasks
int16_t HeadSub;              // head movement sub-task pointer
byte I2C_Error;               // flag returned from I2C comms
bool I2C_Disp;                // true if display is detected during POST
bool I2C_LTOF;                // true if VL53L1X is detected during POST
int16_t Inc = -1;             // increment value
bool Int_En;                  // interrupts enabled flag, == true if enabled
unsigned long IntLast[] = {0,0};  // previous interrupt times in micros()
volatile uint8_t IntMask;     // 4-bit mask used to flag the occurence of interrupts
volatile unsigned long IntMicros[2];   // interrupt times in micros()
bool  IntP34Ch;               // if == true then a slot transition has occured
int16_t IntP34Cnt;            // pulse count for pin INT_P34
int16_t Int34Rev;             // slot counter for pin INT_34, looking for 20 slots == 1 revolution
int16_t Int34RPM;             // calculated rpm for INT_34 slot, actually rpm x 10
bool IntP34State;             // latest state of the INT_P34 input HIGH/LOW
bool  IntP39Ch;               // if == true then a slot transition has occured
int16_t IntP39Cnt;            // pulse count for pin INT_P39
int16_t Int39Rev;             // slot counter for pin INT_39, looking for 20 slots == 1 revolution
int16_t Int39RPM;             // calculated rpm for INT_39 slot, actually rpm x 10
bool IntP39State;             // latest state of the INT_P39 input HIGH/LOW
unsigned long IntLast40[2];   // revolution period in ms, 20 slots, 40 counts
unsigned long IntPeriods[2];  // inter-slot periods in µs
bool JoyActive;               // flag used to prevent stack overflow during moves
int16_t JoyD;                 // direction value 0 - 8
int16_t JoyDL;                // direction 1 counter
int16_t JoyLX;                // Wii Classic left joystick demand
int16_t JoyLY;                // Wii Classic right joystick demand
int16_t JoyMode;              // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
int16_t JoySpd;               // speed vector = max(JoyX,JoyY)
int16_t JoyV;                 // joystick vector  = SQRT(SQ(JX) + SQ(JY))
int16_t JoyX;                 // received value
int JoyXV;                    // converted value 0 - 127
int16_t JoyXC;                // received value offset
int16_t JoyY;                 // received value
int16_t JoyYC;                // received value offset
int JoyYi;                    // tracking filter for received JoyY value
int JoyYV;                    // converted value 0 - 128
boolean JoyYT;                // tracking filter flag; true if not on track
int JX;                       // global variable used with Joystick app
int JY;                       // global variable used with Joystick app
char keyChar;                 // any keyboard character
int16_t keyVal;               // any keyboard value
int16_t LED_Cnt;              // counter used in LED sequencing
int16_t LED_Cycles;           // counter used in LED sequencing
int16_t LED_Del0;             // delay timer used in front LED functions
int16_t LED_Del1;             // delay timer used in gun LED functions
int16_t LedGunMode;           // gun LED mode
int16_t LED_Inc;              // increment value used in LED tasks
int16_t LedLast;              // saved LedMode in case we want to return to it
int16_t LED_Max;              // max value used in LED functions
int16_t LedMode;              // a value > 0 determines LED flash sequence; if = 0 disabled
bool LEDnow0 = false;         // == true if LED_Run0 caused an update
bool LEDnow1 = false;         // == true if LED_Run1 caused an update
unsigned long LED_Period0;    // front LED timer period, default = 20ms
unsigned long LED_Period1;    // gun LED timer period, default = 20ms
int16_t LED_Pnt;              // pointer used in LED animations
bool LED_Run0;                // == true if face LED_Tasks are to run
bool LED_Run1;                // == true if gun LED_Tasks are to run
bool LEDshow0 = false;        // front LED show flag
bool LEDshow1 = false;        // gun LED show flag
bool LED_SWL;                 // LED flag set during button switch press
int16_t LedSpd;               // motor speed value used by LED animation
int16_t LED_Task;             // LED task pointer
bool LedTick;                 // set true during LED cycles
unsigned long LED_Timer0;     // 20+ms LED timer for front LEDs
unsigned long LED_Timer1;     // 20+ms LED timer for gun LEDs
byte LEDVal;                  // values used in LED tasks
bool LTOFcalc;                // if == true then calculate speed of ranged object in m/s
int16_t LTOFfps;              // ranger sensor measurement rate in fps
int16_t LTOFperiod;           // range sensor inter-measurementg period in millis()
int16_t LTOFspd10;            // speed of ranged target in m/s x 10
int16_t LTOFspd10L;           // previous speed of ranged target in m/s x 10
int16_t LTOFspdCnt;           // speed peak detector sustain limit
int16_t LTOFt0,LTOFt1;        // range sensor timers for fps
int16_t MainMode;             // PixyBot main state, 0 = sleep, 1 = Drive, 2 = LTOF, 3 = Pixy2
bool MainRun;                 // set true when a MainTask is running to prevent re-entry
int16_t MainTask;             // main task pointer
int16_t MainTaskNext;         // the task to run after an arrest; default is 0
int16_t MainType;             // a number representing the type of task, default = 0
bool MDEn;                    // motor drive task enable, default = true
int16_t ModeSel;              // MainMode selection
int16_t ModeWas;              // previous MainMode selection
int16_t MoveState;            // move state machine pointer
String myMAC;                 // MAC address if this device
unsigned long next20ms;       // 20ms loop timer
note_t NoteLast;              // most recently played note
unsigned long Often;          // temporary millis() counter used in debuging code
uint8_t Octave;               // most recently played octave
int16_t Ping;                 // Rx 'ping' counter
unsigned long print40ms;      // 40ms variable print timer
int16_t PrintTgt;             // 0 == serial port, 1 == WiFi ESP-NOW
int16_t PwmBckMax;            // max reverse PWM used in LTOF demo
bool PwmEn;                   // if true motor PWM is enabled
int16_t PWM_Freq;             // servo PWM frequency, default = 100;
int16_t PwmFwdMax;            // max forward PWM used in LTOF demo
int16_t PwmInt;               // integrator used in motor control
int16_t PWM_Lft;              // PWM demand applied to left motor drives +/-255
uint8_t PwmLftA, PwmLftB;     // values written using ledcWrite()
uint8_t PwmLftAL, PwmLftBL;   // previous values written using ledcWrite()
volatile uint8_t PwmMask;     // an 8-bit ISR mask for writing to PWM channels
int16_t PwmPnt;               // point to PWM channel for manual spoeed adjustment 0 - 3
int16_t PWM_Rht;              // PWM demand applied to right motor drives +/-255
uint8_t PwmRhtA, PwmRhtB;     // values written using ledcWrite()
uint8_t PwmRhtAL, PwmRhtBL;   // previous values written using ledcWrite()
int16_t PWM_Start;            // value needed to ensure motor start
String PrintTx = "";          // printed strings
int16_t R0,R1,R2,R3,R4,R5,R6,RM; // range values for backaway, target tracking and autonomous
int Range;                    // range value limited to RangeMax
int16_t RangeAng;             // angle at which minimum range was measured
int16_t RangeAngMin;          // angle at which minimum range was recorded
int16_t RangeCentre;          // distance value looking forward
int16_t RangeCentreTracker;   // slugged distance value looking forward
int16_t RangeData[RngDpth];   // range stores at 3° intervals
int16_t RangeDataDP;          // pointer to RangeData[], effectively an angle/3
int16_t RangeDataDPL;         // previous pointer to RangeData[], effectively an angle/3
bool RangeEn;                 // == true if VL53L1X ranging is active
int RangeDiff;                // difference in RangeRaw from RangeLast
bool RangeFps;                // == true if LTOF fps is to be displayed
long RangeIntL;               // L-H integrator used in head centring
long RangeIntLL;              // previous averaged L-H integrator used in head centring
long RangeIntR;               // R-H integrator used in head centring
long RangeIntRL;              // previous averaged R-H integrator used in head centring
int RangeLast;                // previous range measurement
int16_t RangeLeft;            // distance value looking left
bool RangeLL;                 // == true if range < RangeMin
int RangeMax;                 // the current limited to Range measurements
int RangeMin;                 // the current lower limit of Range measurements
unsigned long RangeMinMin;    // min recorded range value during close range scanning 
unsigned long RangeMinVal;    // min range value during close range scanning 
int16_t RangeMinAng;          // angle at which RangeMin occured
int16_t RangeMinMem;          // RangeMin at end of each half sweep
bool RangeNew;                // set == true when a new range is available
bool RangePnt;                // true when a centre range value is recorded, or extremes of sweep
int RangeRate;                // max rate of inter-measurement Range change, default = 64
int RangeRaw;                 // range value obtained from VL53L1X
int RangeRawF;                // rate limited RangeRaw value
int16_t RangeRight;           // distance value looking right
int16_t RangeTrig;            // range trigger point used in head scanning
bool RangeUL;                 // == true if range exceeds RangeMax
bool readWiiCall;             // flag prevents call stack overflow
int16_t Rx_Chk;               // Rx checksum
bool RxClass;                 // set == true when a Wii Classic command has been receieved
bool RxNunch;                 // set == true when a Wii Nunchuk command has been receieved
int16_t RxCZ;                 // buffered received CZ value
int16_t RxJoyX;               // buffered received JoyX value
int16_t RxJoyY;               // buffered received JoyY value
int16_t Rx_len;               // length of WiFi received data block
int16_t Rx_Pnt;               // data pointer user by Rx state machine
bool RxRec;                   // set true when a valid frame of data is received
bool RxRecvEvent;             // set == true when a OnDataRecv() call back event has occured
int16_t RxState;              // receiver state machine state
int16_t Rx_Task;              // task pointer user by Rx state machine
int16_t RTxTimeout;           // counter used to auto-reset if WiFi fails
int16_t RxVal;                // value received from serial Rx
byte RxWiFi[6];               // array holding 6 bytes of Wii received data
bool SerialRx;                // == true if any character has been received
bool ServoAttOnce;            // == true once servos have been attached once
bool ServoEn;                 // false = OFF, true = ON enabled status
int16_t ServoPnt;             // servo target pointer for manual control 0 - 1
int16_t ServoAngTgtAvg;       // averaged target position
volatile uint8_t ServoMask;   // binary mask used in ISR to change servo values
uint16_t ServoVal;            // value sent servos, default = 500
unsigned long Slack20ms;      // measured slack time in 20ms tasks
bool SlotEn;                  // if == true then constantly monitor wheel slot encoders
uint8_t Snore;                // snore animation character
String Status$ = "";          // any status message to be displayed on OLED
int16_t Steer;                // value applied when steering left/right
int16_t SubCnt;               // sub task general counter
int16_t SubDel;               // subtask delay counter
int16_t SubNext;              // pointer to next sub task
int16_t SubRpt;               // subroutine repeat counter
int16_t SubTask;              // sub task pointer
int16_t SubTask1;             // level 1 sub task pointer, used when Pixy2 Pan/Tilt is using SubTask
int16_t SubVal;               // anyvalue used in a subroutine
int16_t SubWait;              // SubTask wait period
int16_t SubWaitNext;          // SubTask to goto after wait period
int16_t sw0Cnt;               // button switch counter
int16_t sw0DwnTime;           // button pressed down time
bool sw0LastState;            // previous state of button switch, HIGH/LOW
bool sw0_Nop;                 // if == true don't perform switch functions
bool sw0State;                // state of read button switch pin
int16_t sw0Timer;             // timer used to detemine button sequences
bool sw0Wup;                  // set == true for SW0 to wait for button release
int16_t sw1Cnt;               // button switch counter
int16_t sw1DwnTime;           // button pressed down time
bool sw1LastState;            // previous state of button switch, HIGH/LOW
bool sw1_Nop;                 // if == true don't perform switch functions
bool sw1State;                // state of read button switch pin
int16_t sw1Timer;             // timer used to detemine button sequences
bool sw1Wup;                  // set == true for SW1 to wait for button release
unsigned int t0,t1;           // code execution timers
bool Task40ms;                // 40ms phase changer tested during 20ms loop
int16_t TaskPnt;             // 20ms subtask pointer
bool TEST = false;            // set == true during assembly, then false for normal use
int16_t TgtCnt;               // counter used in automatic target and drive actions
bool ToneAtt;                 // if == true then tone channel is attached, default = false
int16_t ToneDel;              // tone delay counter
int ToneFreq;                 // frequency used in TonyPlay() function
int16_t TonePnt;              // tone task pointer
bool Turn;                    // default = false; if true turns continuously until = false
int16_t Tx_PingCnt;           // 1 second Tx ping timer
bool USBPwr;                  // true if POST detects USB power voltage
uint8_t VL53L1X_OC;           // VL53L1X ROI SPAD centre
uint16_t VL53L1X_ROIX;        // VL53L1X ROI SPAD width
uint16_t VL53L1X_ROIY;        // VL53L1X ROI SPAD height
unsigned long VL53L1X_Skip;   // read function skip timer
unsigned long VL53L1X_SkipDel;// read function skip timer delay tracker
int16_t VL53L1X_Status;       // tracks value of status register
int16_t VL53L1X_Task;         // if > 0 then run VL53L1X tasks
unsigned long VL53L1X_Timer;  // general timer
int16_t WiFiCntC;             // C button counter for WiFi enable
int16_t WiFiConCnt;           // counter used in connection retry times
bool WiFiConnected;           // == true if a WiFi connection has been established
bool WiFiEn;                  // true when WiFi is enabled with prolonged 'C' button activity
int16_t WiFiPing;             // Rx counter used to maintain connected state
int16_t WiFiRestCnt;          // counter used to goi to rest if no WiFi demands
bool WiFiRx;                  // true when a block of data has been received
long WiFiRxBytes;             // number of bytes received over the WiFi link
int16_t WiFiRxErr;            // number of bytes received over the WiFi link
bool WiFiRxRec;               // true when a block of data has been successfully received
byte WiFiRxMacAddr[6];        // senders MAC address, which could be broadcast FF:FF:FF:FF:FF:FF
int16_t WiFiTryCnt;           // WiFi try to connect count on error count
int16_t WiFiTryNum;           // WiFi try to connect total count
bool WiFiTryOnce;             // == true if first pass of trying to connect
bool WiFiTx;                  // true when a block of data has been sent successfully
int16_t WiFiTx_CB;            // == 1 call back is clear to send, == -1 resend
int16_t WiFiTx_Chk;           // Tx checksum
int16_t WiFiTx_len;           // >0 when WiFi Tx buffer contains data to be sent
long WiFiTxBytes;             // number of bytes received over the WiFi link
int16_t WiFiTxErr;            // WiFi data error
int16_t WiFiTxErrCnt;         // WiFi Tx error count, leads to disconnection
byte WiFiTxMacAddr[6];        // transmit MAC address, which should match myMAC address
uint8_t WiiBYAX;              // Wii Classic right-hand XYAB buttons
uint8_t WiiDigi;              // Wii Classic digital pad bits
uint8_t WiiSHS;               // Wii Classic centre select,home,start bits
int16_t WiiType;              // device type, 0 == Nunchuk, 1 == Classic
byte Z_Dn;                    // button pressed state; 0 == UP, >= 1 == Down
bool Z_Mode;                  // joystick motion flag, normally false, if == true then slide mode

uint8_t Buffer_A[AudBufLen];  // RAM buffer A for SD card read data
uint8_t Buffer_B[AudBufLen];  // RAM buffer B for SD card read data

// ---------------------------------------------------------------------------------

void IRAM_ATTR onTimer() {
  // This is the Timer ISR routine, which is called every 62.5µs (16 kHz).
  // This code MUST be placed before the setup() code for the compiler to recognise it.
  // For the code to run IsrRun == true and flags must have been set beforehand
  // the code must be very efficient, as it is an overhead on everything else
  IsrT1 = micros() - IsrT0; // get timer period
  IsrT0 = micros();         // performance timer, comment out normally
  if (IsrRun) {
    if (!IsrBuf) {
      ledcWrite(PinTone,Buffer_A[IsrPnt]);
      // ledcWrite(PinTone,0);   // silent output
      // Aud$+= String(Buffer_A[IsrPnt]) + ",0,255\n";  // output for serial monitor
    } else {
      ledcWrite(PinTone,Buffer_B[IsrPnt]);
      // ledcWrite(PinTone,0);   // silent output
      // Aud$+= String(Buffer_B[IsrPnt]) + ",0,255\n";  // output for serial monitor
    }
    IsrPnt++;                             // increment the buffer pointer
    if (IsrPnt > AudBufLen_1) {
      //#######################################################
      // Switch between audio buffers A and B
      //#######################################################
      // Check that the buffer load function has been serviced?
      // IsrLdn == 0 if buffer loading is working correctly
      if (IsrLdn) {Aud$ = "IsrLdn = " + String(IsrLdn) + "\n";}
      // check for end of buffer and switch over
      if (IsrESC) {IsrRun = false; IsrRunF = false; IsrEnd = 3; return;}  // playing autostops at end of buffer when IsrESC == true

      IsrBuf = !IsrBuf;                   // swap the buffer pointer
      IsrPnt = 0;                         // reset the data pointer
      // Check for end of the file loading actions, and ramp down.
      if (IsrEnd == 2) {IsrESC = true; IsrLdn = 0;} else {IsrLdn = 1;}
    } else if (PwmMask) {
      //#######################################################
      // Update motor PWM drives?
      //#######################################################
      // When not swapping audio buffers, we check the motor PwmMask for ledcWrite() operations.
      // This mask is set during calls to the MotorDriveTask() function, if IsrRun == true.
      if (PwmMask & 0b00001000) {ledcWrite(PinLftA, PwmLftA);}
      if (PwmMask & 0b00000100) {ledcWrite(PinLftB, PwmLftB);}
      if (PwmMask & 0b00000010) {ledcWrite(PinRhtA, PwmRhtA);}
      if (PwmMask & 0b00000001) {ledcWrite(PinRhtB, PwmRhtB);}
      PwmMask = 0;  // clear the mask, so that it is not re-used
    } else if (ServoMask) {
      //#######################################################
      // Update servo PWM drives?
      //#######################################################
      // Check if the ServoMask is set for servo operations.
      // This mask is set when a servo move is needed and IsrRun == true.
      if (ServoMask & 0b00000001) {
        // Gun servo change bit is set
        if (!servoAtt[0]) {attachServoN(0);}
        else {servoInst[0].writeMicroseconds(servoVal[0]);}
      }
      if (ServoMask & 0b00000010) {
        // Head servo change bit is set
        if (!servoAtt[1]) {attachServoN(1);}
        else {servoInst[1].writeMicroseconds(servoVal[1]);}
      }
      ServoMask = 0;  // clear the mask, so that it is not re-used
    }
  }

  // when read from elsewhere the IsrTx value will give you the ISR code time IsrT1 in µs
  // tests showed that when IsrRun == false, IsrT1 = 1µs, and when IsrRun == true, IsrT1 = 7µs
  // so when playing audio at 62.5µs per sample, we are losing about 10% of available Core1 processing power
  // IsrT1 = micros(); IsrT1 = IsrT1 - IsrT0; // performance timer, comment out normally
}

// ---------------------------------------------------------------------------------

void setup() {
  // Setup code, runs once:
  // Allow allocation of timers for ESP32Servo.h
  ESP32PWM::allocateTimer(0);
  ESP32PWM::allocateTimer(1);
  ESP32PWM::allocateTimer(2);
  ESP32PWM::allocateTimer(3);

  // Initialise PWM pins and output states
  pinMode(PinLftA, OUTPUT); digitalWrite(PinLftA,LOW);
  pinMode(PinLftB, OUTPUT); digitalWrite(PinLftB,LOW);
  pinMode(PinRhtA, OUTPUT); digitalWrite(PinRhtA,LOW);
  pinMode(PinRhtB, OUTPUT); digitalWrite(PinRhtB,LOW);

  // initialise other pins
  AttenON();                      // audio attenuator set to output LOW by default
  pinMode(BatPin,INPUT);          // ADC1_0  GPIO36 battery divider
  pinMode(INT_P34,INPUT_PULLUP);  // speed sensor interupt pin front-left
  pinMode(INT_P39,INPUT_PULLUP);  // speed sensor interupt pin front-right
  pinMode(PinEcho,INPUT);         // HC-SR04 echo pin input
  pinMode(PinTone,OUTPUT); digitalWrite(PinTone,LOW);     // set tone output initial condition
  pinMode(PinTrig,OUTPUT); digitalWrite(PinTrig,LOW);     // set HC-SR04 trig pin LOW
  pinMode(sw0Pin,INPUT_PULLUP);   // left button switch sw0 Pin
  pinMode(sw1Pin,INPUT_PULLUP);   // right button switch sw1 Pin
  pinMode(XSHUT_PIN,OUTPUT); digitalWrite(XSHUT_PIN,LOW); // switch VL53L1X OFF

  // ensure servo signals are OFF
  pinMode(servoPins[0],OUTPUT); digitalWrite(servoPins[0],LOW); // set gun PWM LOW on reset
  pinMode(servoPins[1],OUTPUT); digitalWrite(servoPins[1],LOW); // set LTOF PWM LOW on reset

  setDefaults();  // reset variables

  Serial.begin(115200);

  Wire.begin();

  // initialise the FastLED component
  FastLED.addLeds<WS2812,LED_Pin0, GRB>(Face_LEDs, 7); // define the front LED strip
  FastLED.addLeds<WS2812,LED_Pin1, GRB>(Guns_LEDs, 2); // define the weapon LED strip

  // initialise the four motor PWM channels, setting each to max
  for (int zP = 0;zP < 4;zP++) {
    PwmSetup(zP);
    PwmAttach(zP);
  }

  // grab initial battery voltage
  // take 20 readings over a 20ms period and average
  BatSum = 0;
  for (int zI = 0;zI < 20;zI++) {BatSum+= analogRead(BatPin); delay(1);}
  BatAvg = BatSum/20; BatVol = BatAvg;
    
  // initialise WiFi link
  WiFi.mode(WIFI_STA);
  // new code for ESP32 lib v3.0+
  while (!WiFi.STA.started()) {delay(10);}
  // Once started we can get the micros MAC address
  myMAC = WiFi.macAddress();
  Serial.println("my MAC= " + String(myMAC));
  // WiFi.disconnect(); // I don't think we need this

  // initialise the ESP-NOW link for the first time
  // Init_ESP_NOW();
  WiFiConCnt = 75;      // 3 sec delay between trying cycles

  RunPOST();  // initialise I2C devices, set intro display, etc

  synchLoopTimers();
  
  // define the Core 0 task parameters
  // Core 0 is normally used for WiFi, but here we use it to read sensors
  // leaving Core 1 to create the LED patterns and clock them out
  // Note, this code starts and runs immediately, before loop() does
  xTaskCreatePinnedToCore(
    loop0,       // Task function.
    "Core0",     // name of task.
    10000,       // Stack size of task
    NULL,        // parameter of the task
    1,           // priority of the task
    &Core0,      // Task handle to keep track of created task
    0);          // pin task to core 0
}

// ---------------------------------------------------------------------------------

void loop() {
  // Main loop code, runs repeatedly after setup().
  // When playing audio, filling the buffers is the highest priority task
  // Therefore the IsrLdn checks, for AudioFill() are performed at regular intervals
  if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
  if (SlotEn) {Read_Slots();} // read slot sensors for wheel movement

  //###############################################################################
  //
  //  Front RGB LED 20+ms Tasks (50Hz variable)
  //
  //###############################################################################
  // The LED tasks can run at a variable rate
  if (LED_Run0) {LED_Main();}
  else if ((millis() - LED_Timer0) >= LED_Period0) {
    LED_Timer0 = millis();
    if (LEDnow0) {LEDnow0 = false; LEDshow0 = true;} // trigger 'timed' show process
    else {LED_Run0 = true;} // not a show event, so reset run flag for delayed events
  }
  if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
  if (SlotEn) {Read_Slots();} // read slot sensors for wheel movement
  
  //###############################################################################
  //
  //  Gun RGB LED 20+ms Tasks (50Hz variable)
  //
  //###############################################################################
  // The LED tasks can run at a variable rate
  if (LED_Run1) {GunLEDs();}
  else if ((millis() - LED_Timer1) >= LED_Period1) {
    LED_Timer1 = millis();
    if (LEDnow1) {LEDnow1 = false; LEDshow1 = true;} // trigger 'timed' show process
    else {LED_Run1 = true;} // not a show event, so reset run flag for delayed events
  }
  if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
  if (SlotEn) {Read_Slots();} // read slot sensors for wheel movement

  //###############################################################################
  //
  //  WiFi comms check & respond
  //
  //###############################################################################
  // if WiFi is conencted then check Rx buffer on every cycle, so that we respond to
  // received packets as soon as possible
  if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events

  if (WiFiConnected) {
    if (WiFiRx) {readWiFiRx();} // WiFi Rx buffer contains unread data so deal with it before adding new
  }
  if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
  if (SlotEn) {Read_Slots();} // read slot sensors for wheel movement
 
  // Perform the following tasks on a 20ms time frame
  // Tasks in the range 1 - 9 will be performed every 20ms
  // Tasks in the range 10 - 19 will be performed every 40ms in phase 0
  // Tasks in the range 20+ will be performed every 40ms in phase 1
  // This keeps the tasks nicely spaced in time and prevents drift
  if (TaskPnt > 0) {
    if (TaskPnt < 20) {
      //###############################################################################
      //
      //  20ms Task sequence
      //
      //###############################################################################
      // tasks that are performed every 20ms are contained within this code block
      if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
      switch(TaskPnt) {
        case  1: // 40ms task switcher, depending on Task40ms phase state
          // the branch TaskPnt pointers need to be +1, as they are decremented on each pass
          if (Task40ms) {TaskPnt = 27;} else {TaskPnt = 43;}
          break;
        case  2: // perform sonar ranging if enabled
          if (HCSR04) {Read_Echo();} break;
        case  3: readSW1(); break;                      // read right button switch SW1
        case  4: readSW0(); break;                      // read left button switch SW0
        case  5: if (MDEn) {MotorDriveTask();} break;   // control motor PWM and dither signals
        case  6:  // move Head servo
          if (servoVal[1] != servoTgt[1]) {MoveHead();}; break;
        case  7:  // move Gun servo
          if (servoVal[0] != servoTgt[0]) {MoveGun();}; break;
        case  8: // drive auto-tasks
          if (DriveTask) {DriveAuto();} break;          // perform auto-drive tasks
        case  9:
          if (WiFiConnected) {
                 if (RxClass) {DoClassic();}   // respond to Wii Classic demands
            else if (RxNunch) {DoNunchuk();}   // respond to Wii Nunchuk demands
          } break;
        case 10: // if connected to WiFi read Wii data and respond
          if (WiFiConnected) {readWii();} break;
        case 11: // if WiFi fails stop movement and trigger autoRest
          // if readWiiCall == true then we are currently locked in a movement, so ignore timeout
          if (BotActive && (RTxTimeout > 0) & !readWiiCall) {
            RTxTimeout--;
            if (RTxTimeout == 1) {
              Serial.println("WiFi failed!...");
              LedMode = 0; autoRest = 0;
            }
          } break;
        case 12: // main tasks
          //###############################################################################
          //
          //  Main Tasks
          //
          //###############################################################################
          // Perform these tasks at 20ms intervals.
          // Note that not all options are supported.
          switch (MainTask) {
            case  0: break;                         // default null task
            case  1: MainTask_Intro(); break;       // intro task

            case 11: break;                         // Sonar fixed head ranging
            case 12: MainTask_TrackWall(0); break;  // Sonar fixed head, backaway 
            case 13: MainTask_TrackWall(1); break;  // Sonar fixed head, track wall
            case 14: break;                         // Sonar fixed head, autonomous

            case 21: break;                         // Sonar auto head ranging
            case 22: break;                         // Sonar auto head, backaway 
            case 23: break;                         // Sonar auto head, track wall
            case 24: break;                         // Sonar auto head, air defence mode
            case 25: break;                         // Sonar auto head, autonomous

            case 31: break;                         // LTOF fixed head ranging
            case 32: MainTask_TrackWall(0); break;  // LTOF fixed head, backaway 
            case 33: MainTask_TrackWall(1); break;  // LTOF fixed head, track wall
            case 34: break;                         // LTOF fixed head, autonomous

            case 41: break;                         // LTOF auto head ranging
            case 42: break;                         // LTOF auto head, backaway 
            case 43: break;                         // LTOF auto head, track wall
            case 44: break;                         // LTOF auto head, air defence mode
            case 45: break;                         // LTOF auto head, autonomous

            case 50: break;                         // default audio task, do nothing

            case 62: MainTask_DriveDemo(); break;   // Drive demo task

            case 71: break;                         // Wi-Fi control via Wii controller
          } break;
          //###############################################################################
          //
          //  Audio Tasks
          //
          //###############################################################################
          // perform these tasks at 20ms intervals whenever AudioTask != 0
          // run as a separate task so that audio can be played by tasks
        case 13:
               if (Talk) {TalkEngine();}                // when Talk != 0 perform talking tasks
          else if (Play) {PlayEngine();}                // when Play != 0 play a music file
          else if (AudioTask) {AudioTask_PlayAudio();}  // play a music file
          // IsrRun = false; Talk = 0;
          break;
      }
    } else if (TaskPnt < 40) {
      //###############################################################################
      //
      //  40ms Tasks (25Hz) Phase 0
      //
      //###############################################################################
      // These tasks are run in a consequitive ripple to reduce the overhead on more
      // critical code. They are performed in reverse case order and will all complete
      // within the 40ms period
      if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
      if (TEST) {
        // tasks performed during TEST mode
        switch(TaskPnt) {
          case 10: if (I2C_Disp) {Display_40ms();} break;    // update display if necessary
          case 11: ToneTask(); break;                        // play a tone
        }
      } else {
        // normal 40ms tasks
        switch(TaskPnt) {
          case 20: // Terminate phase 0 task sequence
            TaskPnt = 0;  break;
          case 21:
            BatRead++; if (BatRead >= 5) {BatRead = 0; readBattery();} break; // 200ms battery read
          case 22: // perform gun movement tasks
            if (GunMode) {Gun_Moves();} break;
          case 23: if (Ping > 0) {Ping--;} break;                             // time-out data received flag
          case 24: // if not connected to WiFi run then try to connect every second
            if (!WiFiConnected) {
              // only try to connect WiFi if no 'Pings' are being received
              if (Ping == 0) {
                WiFiConCnt--;
                if (WiFiConCnt < 1) {
                  WiFiConCnt = 10; WiFiTryToConnect();
                  Serial.println("Trying to connect: " + getBAhex());
                  // Serial.println("Trying to connect WiFi");
                }
              }
            } break;
          case 25:
            if (WiFiPing > 0) {
              WiFiPing--;
              if (WiFiPing < 1) {
                // we have stopped receiving so disconnect WiFi
                WiFiDisconnect();
                ESP_NOW_BA = -1;
                WiFiTryOnce = true;
                // Serial.println("WiFiPing == 0");
              }
            } break;
          case 26: // perform head movement tasks
            if (HeadMode) {Head_Moves();} break;
        }
      }
    } else {
      //###############################################################################
      //
      //  40ms Tasks (25Hz) Phase 1
      //
      //###############################################################################
      if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
      switch(TaskPnt) {
        case 40: // Terminating phase 1 task sequence
          TaskPnt = 0;  break;
        case 41:
          // check display mirror auto time-out
          // when count reaches zero DispMon is set to false and assumed not connected
          if (DispTx > 0) {DispTx--; if (DispTx == 0) {DispMon = false;}}
          break;
        case 42: if (I2C_Disp) {Display_40ms();} break;   // update display if necessary
      }
    }
    TaskPnt--; // count down the 20ms sub-tasks as we loop
    if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
  // } else if (Slack20ms == 0) {Slack20ms = millis() - next20ms;
  } else if ((millis() - next20ms) >= 20) {
  //###############################################################################
  //
  //  20ms loop timer
  //
  //###############################################################################
    // do this every 20ms
    next20ms = millis();  // reset next time point
    Task40ms = !Task40ms; // toggle 40ms phase pointer
    TaskPnt = 13;         // initiate 20ms task sequence
    // Serial.println(Slack20ms);
    // Slack20ms = 0;        // clear the slack monitor
  }
  if (IsrLdn) {AudioFill(); yield();} // called on a regular basis when this data load flag is set
  if (SlotEn) {Read_Slots();} // read slot sensors for wheel movement

  //###############################################################################
  //
  //  40 ms Print Task - Variable
  //
  //###############################################################################
  // this code uses either the serial port or the WiFi link to send messages
  // at 115200 baud we can send 11.5 chars/ms, so 64 chars takes 5.5ms.
  if ((millis() - print40ms)  >= 40) {
    print40ms = millis();
    if (PrintTx.length() > 0) {
      // characters in string buffer so send some/all of them
      print40ms = millis() - 34;  // shorten the delay to 6ms for more data to send
      Tx_PingCnt = 0;             // reset the WiFi ping counter
      if (PrintTgt == 0) {
        // default uses the serial port
        if (Serial.availableForWrite() >= 64) {
          if (PrintTx.length() <= 64) {
            Serial.print(PrintTx);
            PrintTx = "";   // empty the buffer
          } else {
            Serial.print(PrintTx.substring(0,64));
            PrintTx = PrintTx.substring(64);
          }
        }
      } else {
        // text data has been received from WiFi so respond using ESP_NOW
        // ensure that we have had a call back from the previous frame
        if (WiFiConnected) {
          // only trannsmit PrintTx data if connected
          if (WiFiTx_CB == 1) {
            WiFiTx_CB = 0;  // clear the call back flag when sending
            if (PrintTx.length() <= 32) {
              WiFiTx_len = PrintTx.length();
              PrintTx.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
              WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
              WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
              // t0 = micros();
              esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
              PrintTx = "";   // empty the buffer
            } else {
              WiFiTx_len = 32;
              Any$ = PrintTx.substring(0,WiFiTx_len);
              Any$.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
              WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
              WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
              // t0 = micros();
              esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
              PrintTx = PrintTx.substring(32);
            }
          } else if (WiFiTx_CB == -1) {
            // an error has occured so resend the data
            if (WiFiTryCnt < 5) {
              WiFiTryCnt++;   // track the number of retries
              WiFiTx_CB = 0;  // clear the call back flag when sending
              t0 = micros();
              esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
            } else {
              // too many resends so disconnect WiFi link
              WiFiDisconnect();
              Serial.println("Failed to Tx!");
            }
          }
        } else {
          // not connected so clear PrintTxdata
          PrintTx = "";
        }
      }
    }
  }

  //###############################################################################
  //
  //  Slack space tasks
  //
  //###############################################################################
  // these functions perform asynchronous measurements and need to run often to
  // improve their accuracy
  if (IsrLdn) {AudioFill(); yield();}     // called on a regular basis when this data load flag is set

  readSerial();                           // check serial port on every loop cycle

  if (VL53L1X_Task > 0) {VL53L1X_Run();}  // perform laser measurement tasks
  if (SlotEn) {Read_Slots();}             // read slot sensors for wheel movement
}

// ---------------------------------------------------------------------------------

void loop0(void * pvParameters) {
  // These functions runs on Core 0, which is also used for ESP32 WiFi ESP-NOW
  // *** WARNING *** do not use for time consuming code like display, I2C, etc
  // Here we read the wheel slot encoders and drive the RGB LEDs
  // the following message, sent once at boot, confirms core 0 assignment
  Serial.print("Core0 task running on core "); Serial.println(xPortGetCoreID());

  for(;;){
    // infinite loop, just like loop() but running in Core 0
    // you must never use 'break' or 'return' statement here or it will crash
    // we tell the free RTOS system to call this function every 1 ms = 1 kHz

    // //###############################################################################
    // //
    // //  Load audio buffers?
    // //
    // //###############################################################################
    // // if enables slots are read every loop, 1ms intgervals
    // if (IsrLdn) {AudioFill();} // called on a regular basis when this data load flag is set

    //###############################################################################
    //
    //  Read slot sensors
    //
    //###############################################################################
    // if enables slots are read every loop, 1ms intgervals
    if (SlotEn) {Read_Slots();}


    //###############################################################################
    //
    //  RGB LEDs
    //
    //###############################################################################
    // this loop runs on 1ms intervals, and triggers a 'show' event when a flag is set
    // because we are running two strips we need to call two show events for each strip
    // to make the trigger happen in the FastLED library function
    if ((LedGunMode == 99) && !GunFire) {
      // in combined LED mode, and not firing at the moment
      if (LEDshow0) {
        FastLED[0].showLeds(255); FastLED[1].showLeds(255);
        LEDshow0 = false; LED_Run0 = true;
        LEDshow1 = false; LED_Run1 = true;
      }
    } else if (LEDshow0) {
      FastLED[0].showLeds(255); FastLED[0].showLeds(255);
      LEDshow0 = false; LED_Run0 = true;
    }
    else if (LEDshow1) {
      FastLED[1].showLeds(255); FastLED[1].showLeds(255);
      LEDshow1 = false; LED_Run1 = true;
    }
    
    // Print to the serial port any change to Aud$ in the last millisecond
    if (Aud$.length() > 0) {Serial.print(Aud$); Aud$ = "";}
    // return to RTOS system with a 1ms call back, should allow rpm sensing <750rpm
    vTaskDelay(1/portTICK_PERIOD_MS);
  }
}

// --------------------------------------------------------------------------------

void OnDataRecv(const esp_now_recv_info_t *info, const uint8_t *incomingData, int len) {
  // Call-back function oocurs when data has been received
  // This needs to be treated like an interrupt, so speed is the essence here
  // memcpy moves data in a block, instead of individual bytes if using for() loop
  memcpy(WiFiRxMacAddr, info->src_addr, 6);   // get the senders MAC/broadcast address
  memcpy(WiFiTxMacAddr, info->des_addr, 6);   // get destination MAC address, which should be assigned peer
  memcpy(&Rx_Buff, incomingData, len);        // copy the received data into Rx_Buff array
  Rx_len = len - 1; Rx_Pnt = 0;               // len contain the length of data
  RxRecvEvent = true;                         // set the flag to say data received
}

// --------------------------------------------------------------------------------

void OnDataRecvHandler() {
  // Called from the main loop() when has been set, to handle the received data
  RxRecvEvent = false;
  // incoming data will always be two or more characters, ie. text == $+++
  if (Rx_len < 2) {Serial.println("Rx Error - no Rx data"); return;}

  RxGetChecksum(Rx_len);  // generate a checksum from Rx data
  if (Rx_Chk != Rx_Buff.ESPdata[Rx_len]) {
    Serial.println("Rx Error - checksum");
  }
  // set flags before exiting
  WiFiRxBytes += Rx_len; Rx_Task = 0; WiFiRx = true;
  WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
  WiFiRxRec = true; // set flag indicating data has been recieved from Wii Transceiver
  //  Serial.print("Bytes received: "); Serial.println(Rx_len);
}

// --------------------------------------------------------------------------------

void OnDataSent(const uint8_t *mac_addr, esp_now_send_status_t status) {
  // Callback when data is sent over Wi-Fi, determines success or failure
  // t1 = micros();
  // Serial.print("Send Status: ");
  if (WiFiConnected) {
    // the link is already active so handle normal success/failure scenarios
    if (status == 0){
      // Serial.println("Success " + String(t1-t0));
      WiFiTx = true;    // flag WiFi Tx success
      WiFiTx_CB = 1;    // set flag to indicate that a call back state has occured
      WiFiTxErrCnt = 0; // clear the error flag in case of a glitch
      WiFiPing = 50;    // set link time-out to 2 seconds
    } else {
      // Serial.println("Fail " + String(t1 - t0));
      WiFiTx = false;   // flag WiFi Tx failure
      WiFiTx_CB = -1;   // set flag to indicate that a resend is required
      WiFiTxErrCnt++;
    }
  } else {
    // not connected so we must have tried to connect and may get an error
    if (status == 0){
      // Successful, so enter the connected state
      Serial.println("Connected!");
      WiFiConnected = true;
      WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
      WiFiTxErrCnt = 0; // reset error count for this connection
    } else {
      // Serial.println("Status = " + String(status));
      WiFiTxErrCnt++;
    }
    WiFiTx_CB = 1;
  }
}

// ---------------------------------------------------------------------------------

void RunPOST() {
  // start-up initialisation processes, mainly testing I2C addresses
  // 0x3C OLED display
  // 0x68 MPU 6050 3-axis motion sensor
  // 0x29 VL53L1X laser range finder
  // 0x54 Pixy2 I2C UART
  Serial.println("\n\n" + Release + "  " + Date + "\n");
  
  // Enter TEST mode?... holding down SW1 during boot forces this condition
  // Note do not hold down SW0 during RESET as boot failure will occur
  if (!TEST) {if (digitalRead(sw1Pin) == LOW) {TEST = true;}}
  if (TEST) {Serial.println("TEST mode active");}

  // display indicates initial battery reading at boot
  BatAvg = BatVol; BatSum = BatVol * 20;  // preload the battery sum for averaging
  AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
  if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
  if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
  Serial.println("Battery " + String(BatVol) + "\t" + String(float(BatAvg)/BatCal) + "v\t" + String(AnyLong) + "%");
  randomSeed(BatVol); // use the battery voltage as the random seed factor
  // if battery is less than 5.5v then USB mode is assumed, which makes servos inoperable
  if (BatAvg <= 2111) {USBPwr = true;}
  randomSeed(BatVol);
  
  // Test for display and initialise
  Wire.beginTransmission(0x3C);
  I2C_Error = Wire.endTransmission();
  if (I2C_Error == 0) {
    I2C_Disp = true; Display_Init();
    Display_Intro();
    Serial.println("Display initialised");
  } else {I2C_Disp = false; Serial.println("Display Not Found!");}

  // initilaise RGB LED arrays for both strips to OFF
  // by calling 'show' here for each strip, we can show them separately later by doing it twice
  fill_solid(Face_LEDs, 7, CRGB::Black);
  FastLED[0].showLeds(gBrightness);
  fill_solid(Guns_LEDs, 2, CRGB::Black);
  FastLED[1].showLeds(gBrightness);
  Serial.println("RGB LEDs cleared.");

  // Test for VL53L1X laser sensor and initialise
  digitalWrite(XSHUT_PIN,HIGH);  // enable VL53L1X sensor
  Wire.beginTransmission(0x29);
  I2C_Error = Wire.endTransmission();
  if (I2C_Error == 0) {
    I2C_LTOF = true; Serial.println("Initialising VL53L1X...");
    VL53L1X_ON();   // turn it ON briefly to confirm that it is there
    if (!TEST) {VL53L1X_OFF();}  // turn the laser OFF by default if not in TEST mode
  } else {I2C_LTOF = false; Serial.println("VL53L1X Not Found!");}

  // Initialise SD card reader
  // note ErrSD will return true if an error orrurs
  initSD();
  SDin = ErrSD; // retain the error status, mounted = false, not mounted = true
  if (SDin)  {
    Serial.println("SD card FAILED to mount.");  
  } else {
    Serial.println("SD card mounted OK.");
  }

  // Centre turret servos
  attachServos(10); delay(200); detachServos();

  // get ready to start
  SetMainMode(0);                     // set default mode 0
  DispDel = 50;                       // wait 2 seconds before displaying anything else
  display.setBrightness(Brightness);  // have a bright display for the intro
  if (TEST) {
    // run in TEST mode
    TEST_Init();
  } else {
    // normal startup
    SetMainTask(1);                     // start with default audio intro task
  }

  Serial.println("POST completed");
}

// ---------------------------------------------------------------------------------

void setDefaults() {
  // load default values
  App = 0;                // > 0 if an app is connected
  AttenON();              // == true if attenuator is ON, == false if not
  AttenRef = Atten;       // make a not of the master setting
  AudioPause = false;     // set == true to temporarily inhibit audio
  AutoMove = 0;           // if == true then move vehicle whilst playing a tune
  Auton$ = "Autonomous";  // autonomous mode string
  autoRest = autoRestMax; // returns to a rest position after a timeout
  AutoStopFL = 0;         // flag/count for left-hand drive auto-stop function
  AutoStopFR = 0;         // flag/count for right-hand drive auto-stop function
  BatAvgMax = 0;          // maximum average value in this period
  BatCnt = 0;             // pointer to array values and critical timer
  BatCritCnt = 0;         // timer used in baterycritical testing
  BatDel = 50;            // set a 10 second initial delay in 'life' prediction
  BatDP = 0;              // battery data pointer
  BatLast = 0;            // previous battery average reading
  BatPwr = false;         // true if battery supply detected, false if USB assumed
  BatRead = 0;            // timer counter used in reading battery every 200ms
  BatT0 = 0;              // time of first battery measurement in life prediction forecast
  BatTime$ = "---";       // time left displayed
  BB = false;             // Wii Classic button BB flag
  BDD = false;            // Wii Classic button BDD flag
  BDL = false;            // Wii Classic button BDL flag
  BDLcnt = 0;             // Wii Classic button BDL timer
  BDLdwn = 50;            // Wii Cl;assic button BDL down timer
  BDR = false;            // Wii Classic button BDR flag
  BDRcnt = 0;             // Wii Classic button BDR timer
  BDRdwn = 50;            // Wii Cl;assic button BDR down timer
  BDU = false;            // Wii Classic button BDU flag
  Bhome = false;          // Wii Classic home button flag
  Blink = false;          // LED blink clock
  BlinkCnt = 0;           // counter for LED Blink clock
  Border = false;         // if true then display has a border
  BotActive = false;      // = true if active, otherwise false
  Bright = 255;           // FastLED brightness
  Brightness = 255;       // Display brightness
  Bsel = false;           // Wii Classic select button flag
  Bstrt = false;          // Wii Classic start button flag
  BX = false;             // Wii Classic button BX flag
  BX_Cnt = 0;             // Wii Classic button BX timer counter
  BY = false;             // Wii Classic button BY flag
  C_Cnt = 0;              // counter used in 'C' button detection
  C_Dn = 0;               // counter for 'C' button down period
  cmdMode = ' ';          // command mode
  cmdSgn = 1;             // cmdVal sign, 1 or -1
  cmdType = ' ';          // command mode type
  cmdVal = 0;             // value associated with a cmdType
  CZ = 0;                 // received button values
  CZ_Wait = false;        // flag used for button release wait function
  DIR_FL = 0;             // direction sign, +ve forward, -ve backwards, 0 stopped
  DIR_FR = 0;             // direction sign, +ve forward, -ve backwards, 0 stopped
  DispBack = false;       // flag used to draw a back arrow on the display
  DispClk = 0;            // a counter used for displaying time in 1 sec incs
  DispClr = false;        // set == true to clear the memory contents
  DispCnt = 0;            // counter used in display updating
  DispDel = 0;            // >0 display delay counter, stops updates over I2C
  DispDisp = false;       // set = true to immediately display memory contents
  DispLast = -1;          // display mode in previous cycle
  DispLock = false;       // if == true then don't allow display mode changes
  DispMode = 13;          // the type of display for a given mode
  DispModeCnt = 0;        // counter used in display updating
  DispMon = false;        // == true if connected to OLED Mirror app
  DispNOP = false;        // set == true to stop display I2C updates
  DispNow = 0;            // if > 0 display special modes
  DispNxtCnt = 0;         // counter used in auto-branching of display
  DispOveride = 0;        // if DispLock == true then use this overide mode
  DispPnt = 0;            // points at the current display mode, default = 0 batteries
  DispTask = 0;           // task pointer for display functions
  DispTime = millis();    // millisecond timer used in display updates
  DispTx = 0;             // counter used with display mirror app
  DispVal = 0;            // general display value, passed from tasks
  Dither = 25;            // overlay signal used to reduce stiction
  DriveTask = 0;          // task pointer for auto-drive tasks
  GunAirEn = false;       // enables Gun movement in air defence mode
  GunDel = 0;             // 40ms delay counter used by Gun sub-tasks
  GunMode = 0;            // controls the Gun movement tasks
  GunSub = 0;             // Gun movement sub-task pointer
  gBrightness = 255;      // brightness used in FastLED show events
  Gear = 1;               // value used to determine PWM Max values
  GunFire = 0;            // if > 0 the run gun firing sequence
  HCSR04 = false;         // if == true then run sonar range measurement task
  HeadAirEn = false;      // enables head movement in air defence mode
  HeadAngle = 0;          // head angle in degrees, 0 centre
  HeadDel = 0;            // 40ms delay counter used by head sub-tasks
  HeadInc = 2;            // head angle increment in degrees, 2 default
  HeadMode = 0;           // controls the head movement tasks
  HeadSub = 0;            // head movement sub-task pointer
  I2C_Disp = false;       // true if display is detected during POST
  I2C_LTOF = false;       // true if VL53L1X is detected during POST
  JoyActive = true;       // flag used to prevent stack overflow during moves
  JoyDL = 0;              // direction 1 counter
  JoyMode = 0;            // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
  JoySpd = 0;             // speed vector = max(JoyX,JoyY)
  JoyX = 0;               // received value, default
  JoyXV = 0;              // converted value 0 - 127
  JoyY = 0;               // received value, default
  JoyYi = 0;              // tracking filter for received JoyY value
  JoyYV = 0;              // converted value 0 - 128
  Jukebox = 0;            // audio jukebox mode, 0 - 2
  JukeboxAEN = 0;         // audio engine number used in jukebox pause
  JukeboxMax = 0;         // audio jukebox, number of tracks in this mode
  JukeboxSTN = 0;         // audio engine suspended task number used in jukebox pause
  JukeboxTrk = 0;         // audio jukebox track number
  JX = 0;                 // global variable used in turning, with Joystick app
  JY = 0;                 // global variable used in turning, with Joystick app
  LED_Cnt = 0;            // counter used in LED sequencing
  LED_Del0 = 0;           // delay timer used in front LED functions
  LED_Del1 = 0;           // delay timer used in gun LED functions
  LedGunMode = 0;         // gun LED mode
  LED_Inc = 1;            // increment value used in LED tasks
  LedLast = 0;            // saved LedMode in case we want to return to it
  LEDnow0 = false;        // == true if LED_Run0 caused an update
  LEDnow1 = false;        // == true if LED_Run1 caused an update
  LED_Period0 = 20;       // front LED timer period, default = 20ms
  LED_Period1 = 20;       // gun LED timer period, default = 20ms
  LED_Pnt = 0;            // pointer used in LED animations
  LED_Run0 = false;       // == true if LED_Task is to run
  LED_Run1 = false;       // == true if LED_Task is to run
  LEDshow0 = false;       // LED show flag for face LEDs
  LEDshow1 = false;       // LED show flag for gun LEDs
  LED_SWL = false;        // LED flag set during button switch press
  LED_Task = 0;           // LED task pointer
  LedTick = false;        // set true during LED cycles
  LEDVal = 0;             // values used in LED tasks
  LTOFcalc = false;       // if == true then calculate speed of ranged object in m/s
  LTOFspd10 = 0;          // speed of ranged target in m/s x 10
  LTOFspd10L = 0;         // previous speed of ranged target in m/s x 10
  LTOFspdCnt = 0;         // speed peak detector sustain limit
  MainMode = 0;           // TankBot main state, 0 = sleep, 1 = Drive, etc
  MainTask = 0;           // main task pointer
  MainTaskNext = 0;       // the task to run after an arrest; default is 0
  MainType = 0;           // a number representing the type of task, default = 0
  MDEn = true;            // motor drive task enable, default = true
  ModeSel = 0;            // MainMode selection
  ModeWas = -1;           // previous MainMode selection
  MoveState = 0;          // move state machine pointer
  NoteLast = NOTE_C;      // most recently played note, default = middle 'C'
  Octave = 4;             // most recently played octave, default
  Often = 0;              // temporary millis() counter used in debuging code
  Ping = 0;               // 'ping' counter
  PwmEn = false;          // if true motor PWM is enabled
  PWM_Freq = 100;         // servo PWM frequency, default = 100;
  PwmInt = 0;             // integrator used in motor control
  PwmMask = 0;            // an 8-bit ISR mask for writing to PWM channels
  PwmPnt = 0;             // point to PWM channel for manual spoeed adjustment 0 - 3
  RangeEn = false;        // == true if VL53L1X ranging is active
  RangeFps = false;       // == true if LTOF fps is to be displayed
  RangeMax = LtofLimit;   // the current limited to Range measurements
  RangeMin = LtofMin;     // the current lower limited to Range measurements
  RangeRate = 64;         // max rate of inter-measurement Range change, default = 64
  RangeRawF = 1000;       // rate limited RangeRaw value (default)
  readWiiCall = false;    // flag prevents call stack overflow
  RxClass = false;        // set == true when a Wii Classic command has been receieved
  Rx_Chk = 0;             // Rx checksum
  RxCZ = 0;               // buffered received CZ value
  RxJoyX = 0;             // buffered received JoyX value
  RxJoyY = 0;             // buffered received JoyY value
  Rx_len = 0;             // length of WiFi received data block
  RxNunch = false;        // set == true when a Wii Nunchuk command has been receieved
  RxRec = false ;         // set true when a valid frame of data is received
  RxRecvEvent = false;    // set == true when a OnDataRecv() call back event has occured
  RxState = 0;            // receiver state machine state
  Rx_Task = 0;            // pointer user by Rx state machine
  RTxTimeout = 0;         // counter used to auto-reset if WiFi fails
  RxVal = -1;             // value received from serial Rx
  SDin = false;           // == true if SD card is attached
  SerialRx = false;       // == true if any character has been received
  ServoAttOnce = false;   // == true once servos have been attached once
  ServoEn = false;        // false = OFF, true = ON enabled status
  ServoMask = 0;          // binary mask used in ISR to change servo values
  ServoPnt = 0;           // servo target pointer for manual control, default = 0
  
  // servo start values are offset by IO1 to ensure some movement occurs at reset
  servoVal[0] = Gun0;     // value sent to gun servos
  servoVal[1] = Sen0;     // value sent to LTOF servos
  servoTgt[0] = Gun0;     // target value for gun servo
  servoTgt[1] = Sen0;     // target value for LTOF servo

  Slack20ms = 0;          // measured slack time in 20ms tasks
  SlotEn = false;         // if == true then constantly monitor wheel slot encoders
  Snore = 3;              // snore animation character
  SubCnt = 0;             // sub task general counter
  SubDel = 0;             // subtask delay counter
  SubNext = 0;            // pointer to next sub task
  SubTask = 0;            // sub task pointer
  sw0Cnt = 0;             // button switch counter
  sw0DwnTime = 0;         // button pressed down time
  sw0LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw0_Nop = false;        // if == true don't perform switch functions
  sw0State = HIGH;        // state of read button switch pin
  sw0Timer = 0;           // timer used to detemine button sequences
  sw0Wup = false;         // set == true for SW0 to wait for button release
  sw1Cnt = 0;             // button switch counter
  sw1LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw1_Nop = false;        // if == true don't perform switch functions
  sw1State = HIGH;        // state of read button switch pin
  sw1Timer = 0;           // timer used to detemine button sequences
  sw1Wup = false;         // set == true for SW1 to wait for button release
  TalkAtt = false;        // attenuator setting used with Talk
  TalkCnt = 0;            // counter/pointer used in storing audio filepaths
  TalkDel = 0;            // TalkEngine() pause counter
  TalkPause = 0;          // interphrase pause in 20ms increments
  TalkPnt = 0;            // pointer used in retrieving audio filepaths
  Task40ms = false;       // 40ms phase changer tested during 20ms loop
  TaskPnt = 0;            // 20ms subtask timer
  ToneAtt = false;        // if == true then tone channel is attached, default = false
  ToneDel = 0;            // tone delay counter
  ToneFreq = 0;           // frequency used in TonyPlay() function
  TonePnt = 0;            // tone task pointer
  Tx_PingCnt = 0;         // 1 second Tx ping timer
  USBPwr = false;         // true if POST detects USB power voltage
  VL53L1X_OC = 199;       // VL53L1X ROI SPAD centre
  VL53L1X_ROIX = 16;      // VL53L1X ROI SPAD width
  VL53L1X_ROIY = 16;      // VL53L1X ROI SPAD height
  VL53L1X_SkipDel = 33;   // read function skip timer delay tracker, nominally 33
  VL53L1X_Task = 0;       // if > 0 then run LV53L1X tasks
  WiFiCntC = 0;           // C button counter for WiFi enable
  WiFiConCnt = 0;         // counter used in connection retry times
  WiFiConnected = false;  // == true if a WiFi connection has been established
  WiFiEn = false;         // true when WiFi is enabled with prolonged 'Z' button activity
  WiFiPing = 0;           // Rx counter used to maintain connected state
  WiFiRx = false;         // true when a block of data has been received
  WiFiRxBytes = 0;        // number of bytes receiv ed over the WiFi link
  WiFiRxErr;              // number of receive errors in the WiFi link
  WiFiRxRec = false;      // true when a block of data has been successfully received
  WiFiTryCnt = 0;         // WiFi try to connect count
  WiFiTryNum = 0;         // WiFi try to connect total count
  WiFiTryOnce = true;     // == true if first pass of trying to connect
  WiFiTx = false;         // true when a block of data has been sent successfully
  WiFiTxBytes = 0;        // number of bytes receiv ed over the WiFi link
  WiFiTx_CB = 1;          // == 1 call back is clear to send, == -1 resend
  WiFiTx_Chk = 0;         // Tx checksum
  WiFiTx_len = 0;         // >0 when WiFi Tx buffer contains data to be sent
  WiFiTxErr = 0;          // WiFi data error
  WiFiTxErrCnt = 0;       // WiFi Tx error count, leads to disconnection
  WiiBYAX = 0b00001111;   // Wii Classic right-hand XYAB buttons
  WiiDigi = 0b00001111;   // Wii Classic digital pad bits
  WiiSHS  = 0b00000111;   // Wii Classic centre select,home,start bits
  WiiType = 0;            // device type, 0 == Nunchuk, 1 == Classic
  Z_Dn = 0;               // button pressed state; 0 = UP, 1 = Down
  Z_Mode = false;         // joystick motion flag, normally false, if == true then slide mode
}

// ---------------------------------------------------------------------------------

void synchLoopTimers() {
  // reset the main loop timers
  while (millis() < 40) {}    // ensure that millis() is larger than presets

  LED_Timer0 = millis() - 10;  // reset LED timer with 10ms phase shift
  LED_Timer1 = millis() - 14;  // reset LED timer with 14ms phase shift
  next20ms   = millis() -  2;  // reset 20ms task timer with 2ms phase shift
  print40ms  = millis() -  6;  // reset 40ms printer timer with 6ms phase shift
}

// --------------------------------------------------------------------------------
