
/* SSD1306Wire.h functions, see https://github.com/ThingPulse/esp8266-oled-ssd1306

  display.clear()        - clear the local pixel buffer
  display.display()      - write the buffer to the display
  display.displayOff()   - turn the display OFF
  display.displayOn()    - turn the display ON
  display.drawCircle(int16_t x, int16_t y, int16_t radius)
  display.drawFastImage(int16_t x, int16_t y, int16_t width, int16_t height, const uint8_t *image)
  display.drawHorizontalLine(int16_t x, int16_t y, int16_t length)
  display.drawLine(int16_t x0, int16_t y0, int16_t x1, int16_t y1)  - draw from x0k,y0 to x1,y1
  display.drawProgressBar(uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint8_t progress)
  display.drawRect(int16_t x, int16_t y, int16_t width, int16_t height) - draw a rectangle
  display.drawString(int16_t x, int16_t y, String text)
  display.drawStringMaxWidth(int16_t x, int16_t y, int16_t maxLineWidth, String text)
  display.drawVerticalLine(int16_t x, int16_t y, int16_t length)
  display.drawXbm(int16_t x, int16_t y, int16_t width, int16_t height, const char* xbm)
  display.fillCircle(int16_t x, int16_t y, int16_t radius)
  display.fillRect(int16_t x, int16_t y, int16_t width, int16_t height)
  display.flipScreenVertically()   - flip the current screen upside down
  disply.init()         - initialise the display
  disply.invertDisplay()- inverted display mode
  disply.end()          - free memory used by the display
  disply.displayOn()    - turn the display ON
  display.mirrorScreen() - draw the screen mirrored
  disply.normalDisplay()- normal display mode
  disply.reconnect()    - connect again through I2C
  display.setBrightness(uint8_t)
  display.setColor(OLEDDISPLAY_COLOR color) - BLACK, WHITE, INVERSE
  display.setContrast(uint8_t contrast, uint8_t precharge = 241, uint8_t comdetect = 64)
  display.setFont(const uint8_t* fontData)
  display.setPixel(int16_t x, int16_t y)  - set a pixel at x,y
  display.setTextAlignment(OLEDDISPLAY_TEXT_ALIGNMENT textAlignment)  - TEXT_ALIGN_LEFT, TEXT_ALIGN_CENTER, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER_BOTH

  W = display.getStringWidth(const char* text, uint16_t length)
  W = display.getStringWidth(String text)
  
  Display size is 128 x 64 (w x h) pixels
  Default fonts are ArialMT at 10, 16 & 24 pixels high
  Text options include:
    2 x 24, + 1 x 10 pnt lines
    1 x 24, + 2 x 16 pnt lines
    1 x 24, + 3 x 10 pnt lines
    4 x 16 pnt lines, no line spaces, decending characters will merge
    3 x 16, + 1 x 10 pnt lines
    6 x 10 pnt lines, no line spaces, decending characters will merge
*/

// --------------------------------------------------------------------------------

void Display_40ms() {
  // called every 40ms from the main loop
  // if DispDel > 0 then we wait for DispDel cycles to complete before update
  if (DispNOP) {return;}  // this function is aborted, no I2C activity
  
  if (DispDel > 0) {DispDel--; return;} // don't update display if DispDel > 0

  // ***WARNING*** you can not allow the audio ISR to run whilst using I2C.
  // So here we temporarily disable it, to prevent the system from crashing.
  // We store the IsrRun flag, and this must be replaced before leaving this function.
  // So if you add to this code, ensure that you respect that process.
  if (DispClr) {Display_clear(); DispClr = false; IsrRun = IsrRunF; return;}
  if (DispDisp) {Display_display(); DispDisp = false; DispClr = true; IsrRun = IsrRunF; return;}

  // Serial.println("DispMode=" + String(DispMode));
  DispCnt--;
  if (DispCnt < 1) {
    // DispCnt is reset for display updates, so can be different for each display mode
    DispCnt = 1;  // default, will trigger on each cycle
    if (DispLock) {DispMode = DispOveride;} // force locked mode
    if (DispBack) {
      // draw a back arrow on the screen
      // display.clear();
      // display.drawLine(48,32,80,16);
      // display.drawLine(48,32,80,48);
      // display.drawLine(80,16,80,48);
      // DispCnt = 10;
    } else {
      if (DispNow > 0) {
        // special display modes used in calibration etc.
        switch(DispNow) {
          // case 1: DispCnt = 3; Display_ServoVals(); break;
        }
      } else {
        if (TEST) {
          // in TEST mode we use a special display
          switch(DispMode) {
            case 0: Display_Test_Battery(); break;
            // case 1: Display_Test_PWM(); break;
            case 2: Display_Test_VLX(); break;
            case 3: Display_Slot_Sens(); break;
            case 4: Display_Counter(); break; // simple 1 second counter
          }
          DispModeCnt++; if (DispModeCnt >= 10) {
            DispModeCnt = 0; DispMode++;
            if (DispMode > 4) {DispMode = 0;}
          }
        } else {
          // Normal menu screens, display depends on menu modes
          if (DispLast != DispMode) {Display_clear();}
          DispLast = DispMode;
          // Some cases are used to wrap display options
          switch(DispMode) {
            case  -3: Display_File_List();        DML =  4; DMR = -2; break;  // File list display
            case  -2: Display_Audio_Engine();     DML = -3; DMR = -1; break;  // Audio engine display
            case  -1: Display_TalkEngine();       DML = -2; DMR =  0; break;  // Talk Engine flags
            case   0: Display_Sleep_Mode();       DML = -1; DMR =  1; break;  // default Sleep Mode
            case   1: Display_Tasks();            DML =  0; DMR =  2; break;  // tasks pointers
            case   2: Display_Counter();          DML =  1; DMR =  3; break;  // simple 1 second counter
            case   3: Display_Battery();          DML =  2; DMR =  4; break;  // full battery reading
            case   4: Display_Battery_Ret();      DML =  3; DMR =  0; break;  // brief battery reading
            
            case  10: Display_Sonar_Fixed_Mode(); DML = 11; DMR = 11; break;  // Sonar fixed head mode intro display
            case  11: Display_Range();            DML = 11; DMR = 11; break;  // LTOF range
            
            case  20: Display_Sonar_Auto_Mode();  DML = 22; DMR = 21; break;  // Sonar fixed head mode intro display
            case  21: Display_Range();            DML = 22; DMR = 22; break;  // LTOF range
            case  22: Display_LTOF_Data();        DML = 21; DMR = 21; break;  // LTOF data
            
            case  30: Display_LTOF_Fixed_Mode();  DML = 31; DMR = 31; break;  // LTOF fixed head mode intro display
            case  31: Display_Range();            DML = 31; DMR = 31; break;  // LTOF range
            
            case  40: Display_LTOF_Auto_Mode();   DML = 42; DMR = 41; break;  // LTOF auto head mode intro display
            case  41: Display_Range();            DML = 42; DMR = 42; break;  // LTOF range
            case  42: Display_LTOF_Data();        DML = 41; DMR = 41; break;       // LTOF data
            
            case  50: Display_Audio_Mode();       DML = 54; DMR = 52; break;  // audio mode intro display
            case  51: Display_Select_Audio();     DML = 54; DMR = 52; break;  // Audio tune select display
            case  52: Display_Audio_Play();       DML = 51; DMR = 53; break;  // Audio tune playing display
            case  53: Display_Jukebox();          DML = 52; DMR = 54; break;  // jukebox mode display
            case  54: Display_Audio_Engine();     DML = 53; DMR = 51; break;  // jukebox mode display
            
            case  60: Display_Drive_Mode();       DML = 69; DMR = 61; break;  // Drive mode intro display
            case  61: Display_Motor_PWM();        DML = 69; DMR = 62; break;  // motor PWM
            case  62: Display_Motor_PWM_Waves();  DML = 61; DMR = 63; break;  // motor PWM as waveforms
            case  63: Display_Slot_Sens();        DML = 62; DMR = 64; break;  // slot sensors
            case  64: Display_Slot_Periods();     DML = 63; DMR = 65; break;  // slot periods
            case  65: Display_Slot_Counts();      DML = 64; DMR = 66; break;  // slot counts
            case  66: Display_Slot_RPM();         DML = 65; DMR = 67; break;  // slot rpms
            case  67: Display_Joysticks();        DML = 66; DMR = 68; break;  // X/Y joystick demands
            case  68: Display_ServoMoves();       DML = 67; DMR = 69; break;  // servo moves
            case  69: Display_Wii_Ctrl();         DML = 68; DMR = 61; break;  // Wi-Fi control
            
            case  80: Display_WiFi_Mode();        DML = 80; DMR = 80; break;  // WiFi drive mode intro display

            case  99: Display_Select_Mode();      DML = 99; DMR = 99; break;  // Mainmode select display
          }
        }
      }
    }
    // the following line puts a rectangle around every display, showing the boundry region
    // Border = true;
    if ((Border) || TEST) {Display_drawRect(0,0,128,64);}
  }
}

// --------------------------------------------------------------------------------

void Display_Audio_Engine() {
  // MainMode 5 display
  DM_AudEng = DispMode;
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  2222222222  3333333333  4444444444  5555555555
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Audio Engine");
  Display_drawString(64,14, "Isr Period:" + String(IsrT1) + "us");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  if (FilePath$.length() > 0) {
    Display_setColor(BLUE);
    Display_drawString( 19, 2, "Play");
    Display_drawString(127, 2, "Talk");
    Display_setColor(WHITE);
  }
  Display_drawString( 40,26, "IsrRunF:");
  Display_drawString( 40,38, "IsrBuf:");
  Display_drawString(100,26, "Time:");
  Display_drawString(100,38, "IsrPnt:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 42,26, String(IsrRunF));
  if (IsrBuf) {Display_drawString( 42,38, "B");} else {Display_drawString( 42,38, "A");}
  if (FileSize > 0) {
    // calculate time remaining in MM:SS
    AnyLong = (FileSize - FilePos)/16000; // time in seconds
    if (AnyLong > 59) {
      Any$ = String(AnyLong/60) + ":";
      AnyLong = AnyLong % 60;
    } else {Any$ = "0:";}
    if (AnyLong < 10) {Any$+= "0";}
    Any$+= String(AnyLong);
    Display_drawString(102,26, Any$);
  } else {Display_drawString(102,26, "00:00");}
  Display_drawString(102,38, String(IsrPnt));
  Display_display();  // display this immediately
  DispDel = 5;        // wait 800ms before displaying audio select screen
}

// --------------------------------------------------------------------------------

void Display_Audio_Mode() {
  // MainMode 5 display  
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111111111111111111222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Audio");
  Display_drawString(64, 26, "Mode");
  Display_display();  // display this immediately
  DispDel = 20;       // wait 800ms before displaying audio select screen
  DispMode = 51;      // set the next display mode after delay DisDel
}

// --------------------------------------------------------------------------------

void Display_Audio_Play() {
  // Displays the current audio track and runtime info
  // called at 40ms intervals from the main loop
  // we want to update this display every 1 second
  if ((millis() - DispTime) < 1000) {return;} // too early, only update once every 1 second
  if ((millis() - DispTime) > 2000) {DispTime = millis();} // synch clock
  else {DispTime += 1000;}

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111    2222222222222222    3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 2,16,4,16,4,16
  Display_drawString(64, 2, "Playing...");
  if (IsrRunF) {
    Display_drawString(64, 22, FileName$);
    if (FileSize > 0) {
      // calculate time remaining in MM:SS
      AnyLong = (FileSize - FilePos)/16000; // time in seconds
      if (AnyLong > 59) {
        Any$ = String(AnyLong/60) + ":";
        AnyLong = AnyLong % 60;
      } else {Any$ = "0:";}
      if (AnyLong < 10) {Any$+= "0";}
      Any$+= String(AnyLong);
    }
  } else {
    if (JukeboxSTN) {Display_drawString(64, 22, "paused"); Any$ = "-:--";}
    else {Display_drawString(64, 22, "completed"); Any$ = "0:00";}
  }
  Display_drawString(64, 42, Any$);
  Display_display();  // display this immediately
  DispDel = 16;       // wait before refresh
}

// --------------------------------------------------------------------------------

void Display_Battery() {
  // display battery information
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111     2222222222222222     3333333333333333
  Brightness = 255; Display_setBrightness(255);   // ensure display bright
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 02, "Battery");
  if (BatAvg > BatCritical) {
    // battery powered mode
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_drawString(0, 23, String(BatAvg));
    Display_setTextAlignment(TEXT_ALIGN_CENTER);
    Display_drawString(64, 23, String(float(BatAvg)/BatCal) + "v");
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
    if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
    if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
    Display_drawString(127, 23, String(AnyLong) + "%");
    // display the time
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_drawString(0, 44, getHmsTime(millis()));
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    Display_drawString(127, 44, BatTime$);
  } else {
    // USB powered mode
    Display_setTextAlignment(TEXT_ALIGN_CENTER);
    Display_drawString(64, 23, "USB Pwr");
    // display the time
    Display_drawString(64, 44, getHmsTime(millis()));
  }
  Display_display();  // display this immediately
  DispMode = DispRet; // rfeturn to previous calling display mode
  DispDel = 50;       // wait 1s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Battery_Ret() {
  // display battery status, then return to previous mode
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //               1111111111111111    2222222222222222
  Brightness = 255; Display_setBrightness(255);   // ensure display bright
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 14,16,4,16,14
  Display_drawString(64, 14, "Battery");
  if (USBPwr) {
    Display_drawString(64, 34, "USB Mode");
  } else {
    AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
    if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
    if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
    Display_drawString(64, 34, String(float(BatAvg)/BatCal) + "v  " + String(AnyLong) + "%");
  }
  Display_display();  // display this immediately
  DispMode = DispRet; // rfeturn to previous calling display mode
  DispDel = 50;       // wait 1s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Counter() {
  // displays a simple counter at 1 second intervals
  // used mainly for test purposes
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                         111111111111111111111111
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 20,24,20
  Display_drawString(64,24, String(DispClk));
  if ((Border) || TEST) {Display_drawRect(0,0,128,64);}
  Display_display();
  DispClk++;
  DispDel = 23;    // wait 1 second before next update
}

// --------------------------------------------------------------------------------

void Display_Drive_Mode() {
  // MainMode 6 display
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //       111111111111111111111111    222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 6, "Drive");
  Display_drawString(64,34, "Mode");
  Display_setFont(16);
  Display_display();  // display immediately
  DispDel = 4;        // wait 120ms before displaying next count
  DispNxtCnt++; if (DispNxtCnt > 10) {DispMode = 61;} // auto branch to PWM display
}

// --------------------------------------------------------------------------------

void Display_File_List() {
  // Display a list of selectable files
  DM_FleLst = DispMode;
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111122222222223333333333444444444455555555556666666666
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 0, 2, "Dir.");
  Display_drawString( 0,42, "Cnt.");
  Display_drawString( 0,52, String(ListCnt));
  Display_setColor(BLUE);
  // display Directory value
  switch (DirPnt) {
    case 'A' : Dir$ = "/A"; break;
    case 'M' : Dir$ = "/M"; break;
    case 'N' : Dir$ = "/N"; break;
    case 'P' : Dir$ = "/P"; break;
  } Display_drawString(16, 2, Dir$);
  // Attenuator
  if (Atten) {Display_drawString(0,22,"Vol-");} else {Display_drawString(0,22,"Vol+");}
  // get file list if not loaded
  if (DirPnt != DirLast) {
    for (int zF = 0;zF < 6;zF++) {FileList$[zF] = "";}  // clear existing list
    switch (DirPnt) {
      case 'A': loadDir(SD,"/A",ListPnt); break;
      case 'M': loadDir(SD,"/M",ListPnt); break;
      case 'N': loadDir(SD,"/N",ListPnt); break;
      case 'P': loadDir(SD,"/P",ListPnt); break;
    }
    DirLast = DirPnt;           // set the same to prevent further loading
  }
  // display loaded list
  DCY = 2;
  for (int zF = 0;zF < 6;zF++) {
    if ((ListPnt + zF) == ListSel) {
      Display_setColor(YELLOW);
      Display_drawString( 28,DCY, "/" + FileList$[zF]);
           if (Talk != 0) {Display_drawString( 98,DCY, "...T");}
      else if (Play != 0) {Display_drawString( 98,DCY, "P...");}
            else {Display_drawString( 98,DCY, "P/T");}
      Display_setColor(BLUE);
    } else {
      Display_drawString( 28,DCY, "/" + FileList$[zF]);
      Display_drawString( 98,DCY, "P/T");
    }
    DCY += 10;
  }
  // Draw scroll bars & arrows
  Display_drawLine(123, 5,127, 5); Display_drawLine(123, 5,125, 0); Display_drawLine(125, 0,127, 5);
  Display_drawRect(123, 7, 4,49);
  if (ListCnt > 6) {
    // Draw in Slider
    int16_t zH = (49 * 6)/ListCnt;
    int16_t zY = 7 + ((ListPnt * (49-zH)/ListCnt));
    Display_fillRect(123,zY,4,zH);
  }
  Display_drawLine(123,58,127,58); Display_drawLine(123,58,125,63); Display_drawLine(125,63,127,58);

  Display_setColor(WHITE);
  Display_display();  // display immediately
  DispDel = 21;       // wait 1s before displaying next count, so as not to affect audio quality
}

// --------------------------------------------------------------------------------

void Display_Init() {
  // initialise the OLED display
  display.init();
  display.clear();
  display.setContrast(255);
  // display.flipScreenVertically();
  display.display();
  DispClr = false; DispDisp = false;
}

// --------------------------------------------------------------------------------

void Display_Intro() {
  // First screen to be displayed
  // default fonts are ArialMT_Plain_10, ArialMT_Plain_16 and ArialMT_Plain_24
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  111111111111111111111111 222222222222222222222222  3333333333
  Display_setColor(WHITE);
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "TankBot");
  Display_drawString(64, 26, "ESP32");
  Display_setFont(10);
  Display_drawString(64, 52, "Batt.= " + String(float(BatAvg)/BatCal) + "v  " + String(AnyLong) + "%");
  Display_display();
}

// --------------------------------------------------------------------------------

void Display_Joysticks() {
  // display joystick data
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111    2222222222222222    3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  if (!WiFiEn) {
    // problems with WiFi
    Display_drawString(64, 2, "Joysticks");
    if (WiFiConnected) {
      Display_drawString( 64, 22, "WiFi Connected");
      Display_drawString( 64, 42, "NOT Enabled");
    } else {
      Display_drawString( 64, 22, "No WiFi Link");
      Display_drawString( 64, 42, "Inaccessible");
    }
  } else {
    // Y64 vertical text spacing, from Y=0 to Y=63
    // 0123456789012345678901234567890123456789012345678901234567890123
    //   1111111111111111    2222222222    3333333333    4444444444
    Display_drawString(64, 0, "Joysticks");
    Display_setFont(10);
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    Display_drawString( 32, 22, "JLX:");
    Display_drawString( 32, 36, "JRX:");
    Display_drawString( 32, 50, "CZ:");
    Display_drawString( 96, 22, "JLY:");
    Display_drawString( 96, 36, "JRY:");
    Display_drawString( 96, 50, " D:");
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_drawString( 32, 22, String(JoyLX));
    Display_drawString( 32, 36, String(JoyX));
    Any$ = "0"; if (CZ & 2) {Any$ = "1";}
    if (CZ & 1) {Any$ += "1";} else {Any$ += "0";}
    Display_drawString( 32, 50, Any$);
    Display_drawString( 96, 22, String(JoyLY));
    Display_drawString( 96, 36, String(JoyY));
    Display_drawString( 96, 50, String(JoyMode));
  }
  Display_display();  // display immediately
  DispDel = 4;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Jukebox() {
  // MainMode 5 display  
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     111111111111111111111111 222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Jukebox");
  Display_drawString(64, 29, "Mode " + String(Jukebox));
  Display_display();  // display this immediately
  Display_display();  // display immediately
  DispDel = 30;       // wait 1200ms before displaying audio select screen
  DispMode = 52;      // set the next display mode after delay DisDel
}

// --------------------------------------------------------------------------------

void Display_LTOF_Auto_Mode() {
  // MainMode 4 display  
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     111111111111111111111111 222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "LTOF Auto");
  Display_drawString(64, 29, "Mode");
  Display_display();  // display immediately
  DispDel = 3;        // wait 120ms before displaying next count
  DispNxtCnt++; if (DispNxtCnt > 10) {DispMode = 41;} // auto branch to Range display
}

// --------------------------------------------------------------------------------

void Display_LTOF_Data() {
  // displays the data stored in the RangeData[] array
  // the table consists of 41 range values, from RangeMin to RangeOL
  // as it draws the array it also determines min/max ranges from data
  Display_clear();    // remove current display content
  if (Brightness != 255) {Display_setBrightness(255);}
  int zMax = 0;
  int zMin = RangeMax;
  DispCX = 23; int zXMin = DispCX;
  for (int zD = 0; zD < RngDpth; zD++) {
    int zY = map(RangeData[zD],RangeMin,RangeMax,63,0);
    if (RangeData[zD] < zMin) {zMin = RangeData[zD]; zXMin = DispCX;}
    else if (RangeData[zD] > zMax) {zMax = RangeData[zD];}
    Display_drawRect(DispCX,zY,2,64-zY);
    DispCX+=2;
  }
  // draw the min range pointer
  if (zMin < RangeMax ) {
    Display_setColor(BLACK);
    Display_drawLine(zXMin,61,zXMin,63);
  }

  Display_setColor(WHITE);
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 1111111111  2222222222                   3333333333  4444444444
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(0, 0, "Max");
  Display_drawString(0, 12, String(zMax));
  Display_drawString(0, 41, "Min");
  Display_drawString(0, 53, String(zMin));
  
  Display_display();  // display immediately
  DispDel = 3;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_LTOF_Fixed_Mode() {
  // MainMode 3 display  
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //      111111111111111111111111    222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 5, "LTOF Fixed");
  Display_drawString(64, 33, "Mode");
  Display_display();  // display immediately
  DispDel = 3;        // wait 120ms before displaying next count
  DispNxtCnt++; if (DispNxtCnt > 10) {DispMode = 31;} // auto branch to Range display
}

// --------------------------------------------------------------------------------

void Display_Mirrored() {
  // Display robot type when DispMon is first connected.
  // We clear current flags so that this is displayed immediately.

  DispClr = false; DispDisp = false; DispCnt = 1;
  
  Display_clear();    // remove current display content
  Display_display();  // clears moirror display immediately
  if (Brightness != 255) {Display_setBrightness(255);}
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  111111111111111111111111 222222222222222222222222   3333333333
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "TankBot");
  Display_drawString(64, 26, "ESP32");
  Display_setFont(10);
  Display_drawString(64,53, "Display Mirror");
  if (DispMon) {PrintTx+= "~Goff\n";} // disable graphing mode in Monitor+
  Display_display();  // display immediately
  DispDel = 50;       // wait 2s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Motor_PWM() {
  // display the PWM values being applied to the motors
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111  2222222222222222
  //                             3333333333333333
  //                                          4444444444444444
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Motor  PWM");
  Display_drawString( 64, 22, String(Steer));
  Display_drawString( 24, 28, String(PWM_Lft));
  Display_drawString(103, 28, String(PWM_Rht));
  if (WiFiEn) {Display_drawString( 64, 41, String(Drive));}
  if (DithON) {Display_drawRect(0,24,128,40);}
  Display_display();  // display this immediately
  DispDel = 3;        // wait 40ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Motor_PWM_Waves() {
  // Display the PWM values being applied to the motors as waveforms
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111  2222222222222222
  //                             3333333333333333
  //                                          4444444444444444
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Motor  PWM");
  Display_drawString( 64, 22, String(Steer));
  Display_Wave( 4,28,abs(PWM_Lft));   // drawing function
  Display_Wave(91,28,abs(PWM_Rht));   // drawing function
  if (WiFiEn) {Display_drawString( 64, 41, String(Drive));}
  Display_display();  // display this immediately
  DispDel = 3;        // wait 80ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Question_YN(String zQ1,String zQ2) {
  // display NO or YES question screen  
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111     2222222222222222     3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, zQ1);
  Display_drawString(64, 25, zQ2);
  // draw the button bars
  //  display.setColor(INVERSE);  // BLACK, WHITE, INVERSE
  Display_fillRect( 0,46,37,18);
  Display_fillRect(91,46,37,18);
  Display_setColor(BLACK);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(  7, 46, "NO");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(126, 46, "YES");
  Display_setColor(WHITE);  // BLACK, WHITE, INVERSE
  Display_display();  // display this immediately
  DispDel = 25;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void Display_Range() {
  // display VL53L1X range measurement
  String zSL = ""; String zSR = "";
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111      2222222222222222      3333333333333333
  //                                                   4444444444
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Display the mode of operationg or simply 'Range'
       if (MainType == 2) {Display_drawString(64, 2, "Back Away");}
  else if (MainType == 3) {Display_drawString(64, 2, "Track Mode");}
  else if (MainType == 4) {Display_drawString(64, 2, Auton$);}
  else {Display_drawString(64, 2, "Range");}
  // If in ranging mode display the current range value
  if (RangeEn || HCSR04) {
         if (RangeUL) {zSL = "< "; zSR = " >";}
    else if (RangeLL) {zSL = "> "; zSR = " <";}
    Display_drawString(64,24, zSL + String(Range) + " mm" + zSR);
  } else {Display_drawString(65,24, "---");}  // not ranging at present
  if (LTOFcalc) {
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_drawString(0,46,String(LTOFfps) + " fps");
    
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if ((LTOFspd10 == 0) || RangeUL) {
      Display_drawString(127,46, "--- m/s");
    } else {
      Display_drawString(127,46, String(float(LTOFspd10)/10.0,1) + " m/s");
    }
  } else {
    if (RangeFps) {Display_drawString(64,46,String(LTOFfps) + " fps");}
    else {Display_setFont(10); Display_drawString(64,50,Status$);}
  }
  Display_display();  // display this immediately
  DispDel = 3;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Select_Audio() {
  // display mode select screen for audio, mode item to choose
  // NO + YES select buttons
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111    2222222222222222      3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Select  Audio");
  switch (ModeSel) {
    case  0: Display_drawString(64, 24, "MT000"); break;
    case  1: Display_drawString(64, 24, "MT001"); break;
    case  2: Display_drawString(64, 24, "MT002"); break;
    case  3: Display_drawString(64, 24, "MT003"); break;
    case  4: Display_drawString(64, 24, "MT004"); break;
    case  5: Display_drawString(64, 24, "MT005"); break;
    case  6: Display_drawString(64, 24, "MT006"); break;
    case  7: Display_drawString(64, 24, "MT007"); break;
    case  8: Display_drawString(64, 24, "MT008"); break;
    case  9: Display_drawString(64, 24, "MT009"); break;
    case 10: Display_drawString(64, 24, "MT010"); break;
  }
  // draw the button bars
  Display_fillRect( 0,46,37,18);
  Display_fillRect(91,46,37,18);
  Display_setColor(BLACK);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(  7, 46, "NO");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(126, 46, "YES");
  Display_setColor(WHITE);  // BLACK, WHITE, INVERSE
  Display_display();
  DispDel = 5;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Select_Mode() {
  // display mode select screen, mode item to choose
  // NO + YES select buttons
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111    2222222222222222      3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Select  Mode");
  switch (ModeSel) {
    case 0: Display_drawString(64, 24, "Sleep"); break;
    case 1: Display_drawString(64, 24, "Sonar-F"); break;
    case 2: Display_drawString(64, 24, "Sonar-A"); break;
    case 3: Display_drawString(64, 24, "LTOF-F"); break;
    case 4: Display_drawString(64, 24, "LTOF-A"); break;
    case 5: Display_drawString(64, 24, "Audio"); break;
    case 6: Display_drawString(64, 24, "Drive"); break;
  }
  // draw the button bars
  Display_fillRect( 0,46,37,18);
  Display_fillRect(91,46,37,18);
  Display_setColor(BLACK);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(  7, 46, "NO");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(126, 46, "YES");
  Display_setColor(WHITE);  // BLACK, WHITE, INVERSE
  Display_display();
  DispDel = 5;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_ServoMoves() {
  // Display servo move data
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111    2222222222222222    3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Servo Moves");
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111    2222222222    3333333333    4444444444
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 36, 22, "Sr0Val:");
  Display_drawString( 36, 36, "Sr0Tgt:");
  Display_drawString( 36, 50, "Sr0Cnt:");
  Display_drawString(100, 22, "Sr1Val:");
  Display_drawString(100, 36, "Sr1Tgt:");
  Display_drawString(100, 50, "Sr1Cnt:");
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 36, 22, String(servoVal[0]));
  Display_drawString( 36, 36, String(servoTgt[0]));
  Display_drawString( 36, 50, String(servoCnt[0]));
  Display_drawString(100, 22, String(servoVal[1]));
  Display_drawString(100, 36, String(servoTgt[1]));
  Display_drawString(100, 50, String(servoCnt[1]));
  Display_display();  // display immediately
  DispDel = 4;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Sleep_Mode() {
  // default state display, called every 40ms
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   11111111111111111111111122222222222222222222222233333333333333..
  if (Brightness != 64) {Display_setBrightness(64);}  // ensure display is dim
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Sleep");
  Display_drawString(64, 26, "Mode");
  Snore++; if (Snore > 3) {Snore = 0;}
  switch (Snore) {
    case 0:
      Display_setFont(10); Display_drawString(64, 50, "."); break;
    case 1:
      Display_setFont(10); Display_drawString(64, 50, "z"); break;
    case 2:
      Display_setFont(10); Display_drawString(64, 50, "Z"); break;
    case 3:
      Display_setFont(16); Display_drawString(64, 49, "Z"); break;
  }
  Display_display();
  DispDel = 6;                // wait 200ms before displaying next count
  // branch to displaying the battery every 10 seconds
  DispNxtCnt++; if (DispNxtCnt >= 50) {DispNxtCnt = 0; DispRet = DispMode; DispMode = 3;}
}

// --------------------------------------------------------------------------------

void Display_Slot_Counts() {
  // display the slot counts from the speed sensors
  // Y64 vertical text spacing, from Y=0 to Y=63
  Display_clear();
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111        2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Slot  Counts");
  Display_drawString(32, 28, String(IntP34Cnt));  // FL
  Display_drawString(95, 28, String(IntP39Cnt));  // FR
  Display_display();
  DispDel = 2;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Slot_Periods() {
  // display the slot periods from the speed sensors
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111        2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Slot  Periods");
  Display_drawString( 32, 28, String(IntPeriods[0]));  // FL
  Display_drawString( 95, 28, String(IntPeriods[1]));  // FR
  Display_display();
  DispDel = 2;        // wait 120ms before displaying next count
  if (!IntP34Ch && !IntP39Ch) {
    // no change in the previous period so clear displayed values
    Reset_Slots();
  }
  IntP34Ch = false; IntP39Ch = false;
}

// --------------------------------------------------------------------------------

void Display_Slot_RPM() {
  // display the slot speed sensors RPM values
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111        2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Slot  RPMs");
  Display_drawString( 32, 28, String(0.1 * Int34RPM,1));  // FL
  Display_drawString( 95, 28, String(0.1 * Int39RPM,1));  // FR
  Display_display();
  DispDel = 2;        // wait 120ms before displaying next count
  if (!IntP34Ch && !IntP39Ch) {
    // no change in the previous period so clear displayed values
    Reset_Slots();
  }
  IntP34Ch = false; IntP39Ch = false;
}

// --------------------------------------------------------------------------------

void Display_Slot_Sens() {
  // display the state of the two slot sensors  
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111    2222222222222222    3333333333333333
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Slot Sensors");
  // INT_P34  speed sensor interupt pin front-left
  // INT_P39  speed sensor interupt pin front-right
  Display_drawString(64, 22, "F-L = " + String(digitalRead(INT_P34)));
  Display_drawString(64, 42, "F-R = " + String(digitalRead(INT_P39)));
  Display_display();
  DispDel = 2;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Sonar_Auto_Mode() {
  // MainMode 2 display  
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111111111111111111222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Sonar Auto");
  Display_drawString(64, 26, "Mode");
  Display_display();
  DispDel = 3;        // wait 160ms before displaying next count
  DispNxtCnt++; if (DispNxtCnt > 10) {DispMode = 21;} // auto branch to Range display
}

// --------------------------------------------------------------------------------

void Display_Sonar_Fixed_Mode() {
  // MainMode 1 display  
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111111111111111111222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Sonar Fixed");
  Display_drawString(64, 26, "Mode");
  Display_display();
  DispDel = 3;        // wait 160ms before displaying next count
  DispNxtCnt++; if (DispNxtCnt > 10) {DispMode = 11;} // auto branch to Range display
}

// --------------------------------------------------------------------------------

void Display_TalkEngine() {
  // Display flags associated with the Talk Engine
  DM_TalkEng = DispMode;
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111    2222222222  3333333333  4444444444  5555555555
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString( 64,  2, "Talk Engine");
  Display_drawString( 64, 16, "File$: " + FilePath$);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  if (FilePath$.length() > 0) {
    Display_setColor(BLUE);
    Display_drawString( 19, 2, "Play");
    Display_drawString(127, 2, "Talk");
    Display_setColor(WHITE);
  }
  Display_drawString( 44, 28, "Talk:");
  Display_drawString( 44, 40, "TalkPnt:");
  Display_drawString( 44, 54, "TalkDel:");
  Display_drawString(108, 28, "TalkCnt:");
  Display_drawString(108, 40, "AudPkEn:");
  Display_drawString(108, 54, "IsrRunF:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 46, 28, String(Talk));
  Display_drawString( 46, 40, String(TalkPnt));
  Display_drawString( 46, 54, String(TalkDel));
  Display_drawString(110, 28, String(TalkCnt));
  if (AudioPkEn) {Display_drawString(110, 40, "1");} else {Display_drawString(110, 40, "0");}
  if (IsrRunF) {Display_drawString(110, 54, "1");} else {Display_drawString(110, 54, "0");}
  Display_display();
  DispDel = 3;        // wait 160ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Tasks() {
  // display the task values
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111    2222222222  3333333333  4444444444  5555555555
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Tasks");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 48, 16, "MainTsk:");
  Display_drawString(112, 16, "SubTsk:");
  Display_drawString( 48, 28, "SubCnt:");
  Display_drawString(112, 28, "SubNxt:");
  Display_drawString( 48, 40, "LEDMde:");
  Display_drawString(112, 40, "SubWait:");
  Display_drawString( 48, 52, "MainMde:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 50, 16, String(MainTask));
  Display_drawString(114, 16, String(SubTask));
  Display_drawString( 50, 28, String(SubCnt));
  Display_drawString(114, 28, String(SubNext));
  Display_drawString( 50, 40, String(LedMode));
  Display_drawString(114, 40, String(SubWait));
  Display_drawString( 50, 52, String(MainMode));
  Display_display();
  DispDel = 4;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Test_Battery() {
  // display battery voltage  
  // Y64 vertical text spacing, from Y=0 to Y=63
   Display_clear();
  // 0123456789012345678901234567890123456789012345678901234567890123
  //         1111111111111111                2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 8, "Battery");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62, 40, "Batt. =");
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(66, 40, String(float(BatAvg)/BatCal) + "v");
  Display_display();
  DispDel = 20;        // wait 840ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Test_VLX() {
  // display Ltof range
  if (!I2C_LTOF) {DispModeCnt = 20; return;}
  
   Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111111111111111111        222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,  0, "LTOF");
  Display_setFont(16);
  Display_drawString(64, 32, "Range " + String(Range));
  Display_display();
  DispDel = 3;        // wait 160ms before displaying next count
}

// --------------------------------------------------------------------------------

void DisplayText1S24(String zT0) {
  // display one line of text at 24 pt

  // ***WARNING*** you can't allow the audio ISR to run whilst using I2C
  // so here we temporarily disable it, to prevent the system from crashing.
  // We store the IsrRun flag, and this must be replaced before leaving this function.
  // So if you add to this code, ensure that you respect that process.
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                     111111111111111111111111
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 20, zT0);
  Display_setBrightness(Brightness);
  Display_display();  // display this immediately
  DispDel = 24;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void DisplayText2(String zT0,String zT1) {
  // display two lines of text of equal height

  // ***WARNING*** you can't allow the audio ISR to run whilst using I2C
  // so here we temporarily disable it, to prevent the system from crashing.
  // We store the IsrRun flag, and this must be replaced before leaving this function.
  // So if you add to this code, ensure that you respect that process.
  bool zIsrRun = IsrRun; IsrRun = false; // temporarily block the audio ISR
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //               1111111111111111    2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 14, zT0);
  Display_drawString(64, 34, zT1);
  if (Border) {Display_drawRect(0,0,128,64); Display_drawRect(1,1,126,62);} // thick boarder
  Display_setBrightness(Brightness);
  Display_display();  // display this immediately
  DispDel = 24;       // wait 1 sec before display next screen
  IsrRun = zIsrRun;
}

// --------------------------------------------------------------------------------

void Display_Text2x16(String zT0$,String zT1$) {
  // called to display a two lines of text centre screen in font 16

  // ***WARNING*** you can't allow the audio ISR to run whilst using I2C
  // so here we temporarily disable it, to prevent the system from crashing.
  // We store the IsrRun flag, and this must be replaced before leaving this function.
  // So if you add to this code, ensure that you respect that process.
  bool zIsrRun = IsrRun; IsrRun = false; // temporarily block the audio ISR
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //               1111111111111111    2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(16);  // fonts are ArialMT_Plain_10,16,24
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 14, zT0$);
  Display_drawString(64, 34, zT1$);
  Display_setBrightness(Brightness);
  Display_display();
  DispDel = 24;       // wait 1 sec before display next screen
  IsrRun = zIsrRun;
}

// --------------------------------------------------------------------------------

void DisplayText2S24(String zT0,String zT1) {
  // display two lines of text, 1st line is large

  // ***WARNING*** you can't allow the audio ISR to run whilst using I2C
  // so here we temporarily disable it, to prevent the system from crashing.
  // We store the IsrRun flag, and this must be replaced before leaving this function.
  // So if you add to this code, ensure that you respect that process.
  bool zIsrRun = IsrRun; IsrRun = false; // temporarily block the audio ISR
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //         111111111111111111111111        2222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,  8, zT0);
  Display_setFont(16);
  Display_drawString(64, 40, zT1);
  Display_setBrightness(Brightness);
  Display_display();  // display this immediately
  DispDel = 24;       // wait 1 sec before display next screen
  IsrRun = zIsrRun;
}

// --------------------------------------------------------------------------------

void Display_Wave(int16_t zX,int16_t zY,int16_t zPWM) {
  // Display sub-task
  // Draw a PWM waveform representing zPWM at zX,zY 
  // The waveform will consist of 2 x Marks + 2 x Spaces
  int16_t zH = 8; // height of pulses in pixels
  int16_t zMw = (zPWM * 16)/255;
  int16_t zSw = ((255 - zPWM) * 16)/255;
  if (zPWM == 0) {
    Display_drawLine(zX,zY+zH,zX+32,zY+zH); // flat line
  } else {
    Display_drawLine(zX,zY,zX,zY+zH);             // leading edge 1st Mark
    Display_drawLine(zX,zY,zX+zMw,zY); zX+=zMw;   // top of 1st Mark
    Display_drawLine(zX,zY,zX,zY+zH);             // trailing edge 1st Mark
    Display_drawLine(zX,zY+zH,zX+zSw,zY+zH);      // bottom of 1st Space
    zX+=zSw;
    Display_drawLine(zX,zY,zX,zY+zH);             // leading edge 2nd Mark
    Display_drawLine(zX,zY,zX+zMw,zY); zX+=zMw;   // top of 2nd Mark
    Display_drawLine(zX,zY,zX,zY+zH);             // trailing edge 2nd Mark
    Display_drawLine(zX,zY+zH,zX+zSw,zY+zH);      // bottom of 1st Space
  }
}

// --------------------------------------------------------------------------------

void Display_Wii_Ctrl() {
  // display 6 bytes received from Wii over ESP-NOW
  // display I2C values as decimal 0 - 255
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111122222222223333333333444444444455555555556666666666
  // 7777777777  8888888888
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(10); DispCX = 75;
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(DispCX,  0, String(RxWiFi[0]));
  Display_drawString(DispCX, 10, String(RxWiFi[1]));
  Display_drawString(DispCX, 20, String(RxWiFi[2]));
  Display_drawString(DispCX, 30, String(RxWiFi[3]));
  Display_drawString(DispCX, 40, String(RxWiFi[4]));
  Display_drawString(DispCX, 50, String(RxWiFi[5]));
  // display I2C values as binary 00000000 - 11111111
  Display_setTextAlignment(TEXT_ALIGN_LEFT); DispCX = 82;
  Display_drawString(DispCX,  0, GetBIN(RxWiFi[0]));
  Display_drawString(DispCX, 10, GetBIN(RxWiFi[1]));
  Display_drawString(DispCX, 20, GetBIN(RxWiFi[2]));
  Display_drawString(DispCX, 30, GetBIN(RxWiFi[3]));
  Display_drawString(DispCX, 40, GetBIN(RxWiFi[4]));
  Display_drawString(DispCX, 50, GetBIN(RxWiFi[5]));
  // display 'I2C' device in top left corner
  Display_drawString(0, 0, "Wii");
       if (WiiType == 'C') {Display_drawString(0, 12, "Classic");}
  else if (WiiType == 'N') {Display_drawString(0, 12, "Nunchuk");}
  else if (WiiType == 'P') {Display_drawString(0, 12, "Classic Pro");}
  else {Display_drawString(0, 12, "Disconnected");}
  Display_display();
  DispDel = 4;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_WiFi_Mode() {
  // MainMode 7 display
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111111111111111111222222222222222222222222
  if (Brightness != 255) {Display_setBrightness(255);}
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 0,24,2,16,4,10
  Display_drawString(64, 2, "Wi-Fi");
  Display_drawString(64, 26, "Mode");
  Display_setFont(16);
  Display_display();
  DispDel = 3;        // wait 160ms before displaying next count
  DispNxtCnt++; if (DispNxtCnt > 10) {DispMode = 61;} // auto branch to Drive display
}

// --------------------------------------------------------------------------------

void SetDispMode(int16_t zM) {
  // Sets the display mode and resets flags, and display to clear
  DispMode = zM; DispNOP = false; DispDel = 0;
  DispClr = false; DispDisp = false; DispCnt = 0;
}

// --------------------------------------------------------------------------------

