/*
  These functions enable the playing of audio files from an SD card, using a
  system timer, triggering interrupts ever 62.5s (16kHz).
*/

// --------------------------------------------------------------------------------

void AttenOFF() {
  // removes the drive from the attenuator pin, and clears the flag
  pinMode(AttPin,INPUT); Atten = false;
}

// --------------------------------------------------------------------------------

void AttenON() {
  // drives the attenuator pin LOW, and sets the flag
  digitalWrite(AttPin,LOW); // set the state of the GPIO register before turning on the pin
  pinMode(AttPin,OUTPUT); digitalWrite(AttPin,LOW); Atten = true;
}

// --------------------------------------------------------------------------------

void AudioFill() {
  // Called from the main loop whenever 'IsrLdn' > 0
  // Read more data into the correct Buffer_n[]
  // Note we are reading audio data whilst doing this
  if (IsrLdn == 1) {
    // this value has been set by the ISR to load data into one of the buffers
    switch (IsrEnd) {
      case 0: // not yet reached the end of the file so keep loading data
        if (IsrBuf) {
          // ISR is taking data from Buffer_B[] so fill Buffer_A[]
          readSDtoBuffer(0);
        } else {
          // ISR is taking data from Buffer_A[] so fill Buffer_B[]
          readSDtoBuffer(1);
        } 
        // check to see if we want to determine peak amplitude
        if (AudioPkEn) {IsrLdn = 2;} else {IsrLdn = 0;}
        break;
      case 1: // reached the end of the file, so add a ramp down
        if (IsrBuf) {
          // ISR is taking data from Buffer_B[] so ramp down Buffer_A[]
          rampSDbufferDwn(0);
          // AudioFillBuf(0,0,128);
        } else {
          // ISR is taking data from Buffer_A[] so ramp down Buffer_B[]
          rampSDbufferDwn(1);
          // AudioFillBuf(1,0,128);
        }
        AudioPkEn = false; AudioPkVal = 0;
        IsrLdn = 0; IsrEnd = 2;
        break;
      // case 2:
      //   IsrESC = true; IsrLdn = 0; break;  // set IsrESC to stop at then end of this buffer
    }
    // Serial.println("AudioFill()");
  } else {
    // IsrLdn == 2, so determine the peak value in the buffer just loaded
    // the variable AudioPkVal will be between 0 - 127
    AudioPkVal = 0; uint8_t zPk;
    if (IsrBuf) {
      // ISR is taking data from Buffer_B[] so examine Buffer_A[]
      for (int zP = 0;zP < AudBufLen;zP++) {
        zPk = abs((int16_t)Buffer_A[zP] - 128);
        if (zPk > AudioPkVal) {AudioPkVal = zPk;}
      }
    } else {
      // ISR is taking data from Buffer_A[] so examine Buffer_B[]
      for (int zP = 0;zP < AudBufLen;zP++) {
        zPk = abs((int16_t)Buffer_B[zP] - 128);
        if (zPk > AudioPkVal) {AudioPkVal = zPk;}
      }
    }
    IsrLdn = 0; // clear the load flag when the peak scan has been done
  }
}

// --------------------------------------------------------------------------------

void AudioFillBuf(uint8_t zB,int zS,uint8_t zD) {
  // Fills one of the data buffers, defined by zB, with value zD starting at zS
  if (zB == 0) {for (int zP = zS;zP < AudBufLen;zP++) {Buffer_A[zP] = zD;}}
          else {for (int zP = zS;zP < AudBufLen;zP++) {Buffer_B[zP] = zD;}}
}

// --------------------------------------------------------------------------------

void AudioIsr100ms() {
  // set up a timer to run the ISR interrupt @ 100 ms intervals
  IsrRun = false; delayMicroseconds(100);         // block the ISR code from running
  // set up AudioTimer
  if (!IsrStarted) {
    AudioTimer = timerBegin(2000000);             // set the timer frequency to 2MHz
    timerAttachInterrupt(AudioTimer, &onTimer);   // attach onTimer() function to our AudioTimer
    IsrStarted = true;
  }
  timerAlarm(AudioTimer, 200000, true, 0);        // set timer alarm to 100ms @ 2MHz
  // Serial.println("Isr@10Hz");
}

// --------------------------------------------------------------------------------

void AudioIsrEn() {
  // set up a timer to run the ISR interrupt @ 16kHz
  IsrRun = false; delayMicroseconds(100);           // block the ISR code from running
  // set up AudioTimer
  if (!IsrStarted) {
    AudioTimer = timerBegin(2000000);               // set the timer frequency to 2MHz
    timerAttachInterrupt(AudioTimer, &onTimer);     // attach ISR onTimer() function to our AudioTimer
    IsrStarted = true;
  }
  timerAlarm(AudioTimer, IsrTimeCnt, true, 0);      // set the interrupt to occur at 65.2µs for 16kHz audio samples
  // timerAlarm(AudioTimer, 1375, true, 0);      // set the interrupt to occur at 625µs for 1.6kHz audio samples
  // Serial.println("Isr@16kHz");
}

// --------------------------------------------------------------------------------

void AudioIsrSet(uint64_t zCnt) {
  // ISR timer to run @ zCnt intervals @ 2 MHz
  // 1 sec == 2000000 counts
  IsrRun = false; delayMicroseconds(100);         // block the ISR code from running
  // set up AudioTimer, if not already started
  if (!IsrStarted) {
    AudioTimer = timerBegin(2000000);             // set the timer frequency to 2MHz
    timerAttachInterrupt(AudioTimer, &onTimer);   // attach onTimer() function to our AudioTimer
    IsrStarted = true;
  }
  timerAlarm(AudioTimer, zCnt, true, 0);          // set timer alarm to zCnt @ 2MHz
  // Serial.println("Isr@"+ String(2000000/zCnt) + "Hz");
}

// --------------------------------------------------------------------------------

void AudioPauseNow() {
  // Called to temporarily inhibit the playing of audio. eg. when gun firing
  Serial.println("AudioPauseNow(" + String(AudioPause) + ")");
  IsrRun = false;     // stop the 16 kHz samples
  AttenRef = Atten;   // remember the attenuator setting
       if (AudioTask) {AudioPause = 1; AudioPTsk = AudioTask; AudioTask = 99; AudioDel = 0;}
  else if (Play) {AudioPause = 2; AudioPTsk = Play; Play = 99;}
  else if (Talk) {AudioPause = 3; AudioPTsk = Talk; Talk = 99; TalkDel = 0;}
  // slow down the ISR frequency and detach the pin
  AudioIsr100ms();            // set interrupt to 100ms intervals frees up cpu
  ledcWrite(PinTone,0);     // effectively turn OFF the LOW speed PWM
  // ledcDetach(PinTone);    // turn OFF the high speed PWM
}

// --------------------------------------------------------------------------------

void AudioPlay(String zFile) {
  // Called to start playing an audio file
  // Serial.println("AudioPlay()");
  if (IsrRun) {AudioStop();}  // ensure nothing is playing

  // attempt to open the audio file and read its data
  readSDfileInfo(zFile);
  if (ErrSD) {
    // exit on error
    Serial.println("ErrSD: readSDfileInfo() failed");
    return;
  }
  // Serial.println("File was openned");

  // the file looks good, so we now start loading it
  AudioPwmEn();         // initialise and enable the 312.5kHz PWM channel
  // Serial.println("AudioPwmEn");
  rampSDbufferUp(0);    // load Buffer_A[] with audio ramp
  readSDtoBuffer(1);    // load Buffer_B[] with 1st audio data block
  // Serial.println("Buffer 0 loaded");
  // Initialise the ISR timer to run at 16kHz
  AudioIsrEn();
  // Serial.println("ISR created");
  // set flags to start playing audio
  IsrBuf = false;   // point at buffer A to start with
  IsrESC = false;   // reset the ESCape flag
  IsrLdn = 0;       // reset the data load flag
  IsrEnd = 0;       // reset the 'end' task pointer
  IsrPnt = 0;       // point at the 1st byte in the buffer
  IsrRun = true;    // enable the ISR function
}

// --------------------------------------------------------------------------------

void AudioPlayNow() {
  // Called to continue the playing of audio. eg. when gun firing caused a pause
  Serial.println("AudioPlayNow()");
       if (AudioPause == 1) {AudioTask = AudioPTsk;}
  else if (AudioPause == 2) {Play = AudioPTsk;}
  else if (AudioPause == 3) {Talk = AudioPTsk;}
  // restart the high frequency PWM output

  if (!AudioEn) {AudioPwmEn();}
  if (!IsrBuf) {ledcWrite(PinTone,Buffer_A[IsrPnt]);
  } else {ledcWrite(PinTone,Buffer_B[IsrPnt]);}

  // restart the ISR 16 kHz timer
  AudioIsrEn();
  IsrRun = true; AudioPause = 0; 
}

// --------------------------------------------------------------------------------

void AudioPwmEn() {
  // Initialise the 8-bit PWM channel to run at 312,500Hz
  // This gives 19.5 PWM pulses per ISR period, and is well above audio frrequency.
  // The initial PWM value = 0, so no output, audio will be ramped up to 128
  if (!PwmAtt) {
    // Set up the audio PWM channel once only
    AudioErr = ledcAttach(PinTone, PwmAudFreq, AudPwmBits);
    if (AudioErr) {Serial.println("Audio Freq set");}
    else {Serial.println("AudioErr: Freq not set");}
    AudioErr = ledcWrite(PinTone, 0);
    if (AudioErr) {Serial.println("Audio Duty set to 0");}
    else {Serial.println("AudioErr: Duty not set");}
    PwmAtt = true; AudioEn = true; ToneAtt = false;
  } else {
    // Audio PWM channel has been attached, but changed for Tone purposes
    // So here we re-configure it for audio
    if (!AudioEn) {
      AudioErr = ledcChangeFrequency(PinTone, PwmAudFreq, AudPwmBits);
      if (AudioErr) {Serial.println("Audio Freq reset");}
      else {Serial.println("AudioErr: Freq not reset");}
      AudioErr = ledcWrite(PinTone, 0);
      if (AudioErr) {Serial.println("Audio Duty set to 0");}
      else {Serial.println("AudioErr: Duty not set");}
      AudioEn = true; ToneAtt = false;
    }
  }
}

// --------------------------------------------------------------------------------

void AudioStop() {
  // called to ensure that nothing is playing at the moment
    // ensure that the ISR code is blocked
  IsrRun = false;           // turn off the ISR
  delayMicroseconds(100);   // wait at least 62.5µs
  IsrRunF = false;          // turn off the ISR temp flag
  IsrLdn = 0;               // stop loading more audio data
  IsrEnd = 0;               // reset the end of load flag
  AudioIsr100ms();          // set interrupt to 100ms intervals frees up cpu
  ledcWrite(PinTone, 0);    // effectively turn OFF the high speed PWM
  SDfile.close();           // close the open file
  SDopen = false;           // reset the open flag
}

// --------------------------------------------------------------------------------

void AudioTask_PlayAudio() {
  // starts audio playing, monitors and returns display to selection mode once done
  // if AudioTask < 0 then initialisation is performed
  // if (GunFire) {AudioTask = 0; return;}     // prevent talking when gun is being fired
  if (Talk) {AudioTask = 0; return;}        // cancel if talking is invoked
  if (AudioTask < 0) {AudioDel = 0;}        // new task so cancel any delays
  if (AudioDel > 0) {AudioDel--; return;}   // delay counter

  switch (AudioTask) {
    case -1: // initialisation
      AudioTask = 1; break;
    case 1: // set up LEDs and select track based on Modesel
  	  AutoMove = 0;               // always start with auto-moves turned OFF
      if (IsrRun) {AudioStop();}  // ensure nothing is playing
      switch (ModeSel) {
        case  0: FilePath$ = "/M/MT000.WAV"; FileName$ = "MT000"; break;
        case  1: FilePath$ = "/M/MT001.WAV"; FileName$ = "MT001"; break;
        case  2: FilePath$ = "/M/MT002.WAV"; FileName$ = "MT002"; break;
        case  3: FilePath$ = "/M/MT003.WAV"; FileName$ = "MT003"; break;
        case  4: FilePath$ = "/M/MT004.WAV"; FileName$ = "MT004"; break;
        case  5: FilePath$ = "/M/MT005.WAV"; FileName$ = "MT005"; break;
        case  6: FilePath$ = "/M/MT006.WAV"; FileName$ = "MT006"; break;
        case  7: FilePath$ = "/M/MT007.WAV"; FileName$ = "MT007"; break;
        case  8: FilePath$ = "/M/MT008.WAV"; FileName$ = "MT008"; break;
        case  9: FilePath$ = "/M/MT009.WAV"; FileName$ = "MT009"; break;
        case 10: FilePath$ = "/M/MT010.WAV"; FileName$ = "MT010"; break;
      }
      DispMode = 52; DispDel = 0; DispCnt = 0;
      SetLedMode(50);
      // Serial.println("Playing audio " + FilePath$);
      AudioTask++; SubDel = 5; break;
    case 2: // play audio file
      // Serial.println("AudioPlay() " + FilePath$);
      AudioPlay(FilePath$);
      if (AttenRef) {AttenON();} else {AttenOFF();}
      SetHeadMode(2);   // talking, Head movement
      SetGunMode(3);    // crude sin(), Gun movement
      AudioTask++; break;
    case 3: // wait for end of audio track
      if (AutoMove) {AutoMoveTask();}
      if (!IsrRun) {AudioTask++; SubDel = 10;}
      break;
    case 4: // stop LED animations, then exit
      // Serial.println("Playing audio is complete");
      FileSize = 0;     // clear openned file size
      LedGunMode = 0; SetMainMode(5);
      SetGunMode(1);    // centre Gun
      SetHeadMode(1);   // centre Head
      AudioTask = 0;    // suspend this task
      break;  // go back to audio play list selection

    case 99: // suspended state, caused by jukebox pause
      break;
  }
}

// --------------------------------------------------------------------------------

void JukeboxGet(int16_t zD,int16_t zN) {
  // gets the zN filepath for jukebox zD
  Jukebox = zD;
  switch (zD) {
    case 0:
      JukeboxMax = 3; // number of tracks -1 in this section
      zN = constrain(zN,0,JukeboxMax); JukeboxTrk = zN;
      switch (zN) {
        case 0:
          // say "Hello, sargent Tankbot here."
          TalkAdd("/P/HE0005.WAV"); FileName$ = "HE0005"; break;
        case 1:
          // say "My name is sargent Tankbot."
          TalkAdd("/P/MY0001.WAV"); FileName$ = "MY0001";
          // say "My brain is an ESP32 micrcontroller."
          TalkAdd("/P/MY0000.WAV"); FileName$ = "MY0000";
          // say "I can see using sonar or a laser range finder.""
          TalkAdd("/P/IC0000.WAV"); FileName$ = "IC0000";
          break;
        case 2:
          // say "My tracks are driven by 4 DC motors."
          TalkAdd("/P/MY0010.WAV"); FileName$ = "MY0010";
          // say "My wheel slot counters can be used to measure speed and distance."
          TalkAdd("/P/MY0008.WAV"); FileName$ = "MY0008";
          // say "You can control me, using a Wii Classic controller, over Wi-Fi."
          TalkAdd("/P/YO0002.WAV"); FileName$ = "YO0002";
          break;
        case 3:
          // say "All of my voices and sounds are stored on a micro SD card."
          TalkAdd("/P/AL0010.WAV"); FileName$ = "AL0010";
          // say "Woof! Woof!"
          TalkAdd("/A/DO0000.WAV"); FileName$ = "DO0000";
          // say "Oops! I hope that didn't scare you?"
          TalkAdd("/P/OO0000.WAV"); FileName$ = "OO0000";
          break;
      } break;
    case 1:
      JukeboxMax = 4; // number of tracks -1 in this section
      zN = constrain(zN,0,JukeboxMax); JukeboxTrk = zN;
      switch (zN) {
        case 0: FilePath$ = "/M/TT000.WAV"; FileName$ = "TT000"; break;
        case 1: FilePath$ = "/M/TT001.WAV"; FileName$ = "TT001"; break;
        case 2: FilePath$ = "/M/TT002.WAV"; FileName$ = "TT002"; break;
        case 3: FilePath$ = "/M/TT003.WAV"; FileName$ = "TT003"; break;
        case 4: FilePath$ = "/M/TT004.WAV"; FileName$ = "TT004"; break;
      } break;
    case 2:
      JukeboxMax = 10;  // number of tracks -1 in this section
      zN = constrain(zN,0,JukeboxMax); JukeboxTrk = zN;
      switch (zN) {
        case  0: FilePath$ = "/M/MT000.WAV"; FileName$ = "MT000"; break;
        case  1: FilePath$ = "/M/MT001.WAV"; FileName$ = "MT001"; break;
        case  2: FilePath$ = "/M/MT002.WAV"; FileName$ = "MT002"; break;
        case  3: FilePath$ = "/M/MT003.WAV"; FileName$ = "MT003"; break;
        case  4: FilePath$ = "/M/MT004.WAV"; FileName$ = "MT004"; break;
        case  5: FilePath$ = "/M/MT005.WAV"; FileName$ = "MT005"; break;
        case  6: FilePath$ = "/M/MT006.WAV"; FileName$ = "MT006"; break;
        case  7: FilePath$ = "/M/MT007.WAV"; FileName$ = "MT007"; break;
        case  8: FilePath$ = "/M/MT008.WAV"; FileName$ = "MT008"; break;
        case  9: FilePath$ = "/M/MT009.WAV"; FileName$ = "MT009"; break;
        case 10: FilePath$ = "/M/MT010.WAV"; FileName$ = "MT010"; break;
      } break;
  }
}

// --------------------------------------------------------------------------------

void JukeboxNext() {
  // Get the next filepath using button counter BDRcnt
  JukeboxTrk += BDRcnt;
  if (JukeboxTrk > JukeboxMax) {JukeboxTrk = 0;}  // wrap round file list
  if (Jukebox == 0) {TalkStop();}                 // clear pointers in Talk engine
  JukeboxGet(Jukebox,JukeboxTrk);                 // load the audio file
  // check to see if we are currently playing a jukebox track?
  if (Play) {PlayAdd(FilePath$);}                 // playing, so stop it and play this one
}

// --------------------------------------------------------------------------------

void JukeboxPause() {
  // pause or continue playing the jukebox file
  // the action taken depends on which audio engine is playing
  if (JukeboxSTN) {
    // we are already in a paused state, so continue the playing process
         if (JukeboxAEN == 0) {AudioTask = JukeboxSTN; IsrRun = true;}
    else if (JukeboxAEN == 1) {Play = JukeboxSTN; IsrRun = true;}
    else if (JukeboxAEN == 2) {Talk = JukeboxSTN; IsrRun = true;}
    JukeboxSTN = 0; // clear the paused flag
  } else {
    // stop the audio engine that is playing, and record its flags
    // the audio is stopped by inhibiting the ISR, which would normally cancel the
    // audio play, so we need to suspend it first
         if (AudioTask) {JukeboxAEN = 0; JukeboxSTN = AudioTask; AudioTask = 99; AudioDel = 0;}
    else if (Play) {JukeboxAEN = 1; JukeboxSTN = 5; Play = 3;}
    else if (Talk) {JukeboxAEN = 2; JukeboxSTN = Talk; Talk = 99; TalkDel = 0;}
    else {JukeboxSTN= 0;}             // nothing was playing
    if (JukeboxSTN) {IsrRun = false;} // stop ISR from running
  }
}

// --------------------------------------------------------------------------------

void JukeboxPlay() {
  // play or replay an audio file based on the current jukebox selection
  if (Jukebox == 0) {TalkStop();}     // clear pointers in Talk engine
  JukeboxGet(Jukebox,JukeboxTrk); JukeboxSTN = 0;
  // phrases are loaded using one or more TalkAdd(FilePath$),
  // where as single music files use the PlayAdd(FilePath$) function
  if (Jukebox != 0) {PlayAdd(FilePath$);}
}

// --------------------------------------------------------------------------------

void JukeboxPrev() {
  // get the previous filepath using button counter BDLdwn
  JukeboxTrk -= BDLcnt;
  if (JukeboxTrk < 0) {JukeboxTrk = JukeboxMax;} // wrap round file list]
  if (Jukebox == 0) {TalkStop();}                 // clear pointers in Talk engine
  JukeboxGet(Jukebox,JukeboxTrk);                 // load the audio file
  // check to see if we are currently playing a jukebox track?
  if (Play) {PlayAdd(FilePath$);}                 // playing, so stop it and play this one
}

// --------------------------------------------------------------------------------

void JukeboxSet(int16_t zS) {
  // change the jukebox to zS and load the first filepath
  Jukebox = zS;
  // stop all audio playing
  AudioStop(); if (Talk) {TalkStop();} AudioTask = 0;
  // set up new jukebox mode, and mention it
  switch(Jukebox) {
    case 0: 
      // say "Jukebox 0"
      TalkAdd("/P/DU0050.wav"); TalkAdd("/N/NU000.wav");
      // we don't use JukeboxGet() here as we don't want to say anything immediately
      Jukebox = 0; JukeboxTrk = 0; break;
    case 1:
      // say "Jukebox 1"
      JukeboxGet(1,0); TalkAdd("/P/DU0050.wav"); TalkAdd("/N/NU001.wav"); break; 
    case 2: 
      // say "Jukebox 2"
      JukeboxGet(2,0); TalkAdd("/P/DU0050.wav"); TalkAdd("/N/NU002.wav"); break;
  }
  // if in Mode == 0 then display jukebox and drop into audio player display.
  // but don't change mode as jukebox is not a particular mode
  if (MainMode == 0) {
    // display 'Jukebox mode'
    SetDispMode(53);
  }
}

// --------------------------------------------------------------------------------

bool openSDpath(String zPath) {
  // Attempts to open a file path
  // returns 'true' if file is openned, 'false' if not
  // also gets FileSize if openned
  FilePath$ = zPath;
  SDfile = SD.open(zPath);
  if (!SDfile) {
    FileSize = 0; ErrSD = true;
    // Serial.println("Failed to open file for reading"); ErrSD = true;
  } else {
    // we reduce FileSize so that metadata at the end of the file is not read
    FileSize = SDfile.size();  ErrSD = false;
    // Serial.println("Opened path: " + zPath); ErrSD = false;
  }
  SDfile.close(); SDopen = false;
  return ErrSD;
}

// --------------------------------------------------------------------------------

bool openSDread(String zPath) {
  // Attempts to open a file path for reading and leaves it open
  // returns ErrSD = 'false' if file is openned, 'true' if not, failed
  // FileSize is set if the file exists
  FilePath$ = zPath;
  SDfile = SD.open(zPath,FILE_READ);
  if (!SDfile) {
    // Serial.println("Failed to open file for reading");
    FileSize = 0;
    SDfile.close();           // close the open file
    SDopen = false;           // reset the open flag
    ErrSD = true;
  } else {
    // Serial.println("Openned path: " + zPath);
    FileSize = SDfile.size();
    // Serial.println("File size = " + String(FileSize));
    SDopen = true;           // set the open flag
    ErrSD = false;
  }
  FilePos = 0;
  return ErrSD;
}

// --------------------------------------------------------------------------------

void PlayAdd(String zPath) {
  // Starts the PlayEngine() to play file zPath$.
  // If called whilst playing audio, then stop it.
  if (IsrRun) {
    AudioStop();
    TalkStop();
  }
  // start the PlayEngine()
  FilePath$ = zPath; Play = 1;
}

// --------------------------------------------------------------------------------

void PlayEngine() {
  // audio file player, initiated by passing a filepath to PlayAdd()
  switch (Play) {
      case 0: break;  // default, do nothing state
      case 1: // start playing FilePath$
        AudioPlay(FilePath$); Play++; break;
      case 2: // Wait here for audio to finish
       if (AutoMove) {AutoMoveTask();}
       if (IsrEnd == 3) {
         // Exit once file is played out
         AudioStop();          // reset audio system
         Play = 0;
         if (AutoMove) {
          SetGunMode(1);    // centre Gun
          SetHeadMode(1);   // centre Head
          AutoMove = 0;     // terminate vehicle automovement
         }
        } break;
      case 3: // temporarily stop playing, PAUSE
        if (GunMode) {GunMode = -GunMode;}
        if (HeadMode) {HeadMode = -HeadMode;}
        IsrRun = false; Play++; break;
      case 4: // Play is active, but doing nothing
        break;
      case 5: // resume playing
        if (GunMode < 0) {GunMode = -GunMode;}
        if (HeadMode < 0) {HeadMode = -HeadMode;}
        IsrRun = true; Play = 2; break;

      case 99: // suspended state, caused by jukebox pause
        break;
  }
}

// --------------------------------------------------------------------------------

void PlayStop() {
  // Stops the play engine and resets flags
  AudioStop();
  Play = 0;
  if (AutoMove) {
    SetGunMode(1);    // centre Gun
    SetHeadMode(1);   // centre Head
    AutoMove = 0;     // terminate vehicle automovement
  }
}

// --------------------------------------------------------------------------------

void rampSDbufferDwn(bool zBp) {
  // Fills the audio buffer zBp with a ramp from 128 down to 0
  // dividing the (buffer_length - pointer) by 4, gives 128 - 0
  // Serial.println("rampSDbufferDwn(" + String(zBp) + ")");
  if (!zBp) {
    for (uint16_t zP = 0;zP < AudBufLen;zP++) {Buffer_A[zP] = (AudBufLen - zP)>>2;}
    Buffer_A[AudBufLen_1] = 0;
  } else {
    for (uint16_t zP = 0;zP < AudBufLen;zP++) {Buffer_B[zP] = (AudBufLen - zP)>>2;}
    Buffer_B[AudBufLen_1] = 0;
  }
}

// --------------------------------------------------------------------------------

void rampSDbufferUp(bool zBp) {
  // fills the audio buffer zBp with a ramp from 0 to 128
  // dividing the pointer by 4 gives 0 - 127
  // Serial.println("rampSDbufferUp(" + String(zBp) + ")");
  if (!zBp) {
    for (uint16_t zP = 0;zP < AudBufLen;zP++) {Buffer_A[zP] = zP>>2;}
    Buffer_A[AudBufLen_1] = 128;
  } else {
    for (uint16_t zP = 0;zP < AudBufLen;zP++) {Buffer_B[zP] = zP>>2;}
    Buffer_B[AudBufLen_1] = 128;
  }
}

// --------------------------------------------------------------------------------

bool readSDfileInfo(String zPath) {
  // Opens file zPath and reads the header information into Buffer_A[], as 72 bytes.
  // Returns ErrSD = false if all OK
  //
  // Position     Example     Description
  //
  //  0 -  3	    “RIFF”	    Marks the file as a riff file. Characters are each 1 byte long.
  //  4 -  7	    integer     File size of the overall file, in bytes (32-bit integer). Typically, you’d fill this in after creation.
  //  8 - 11	    “WAVE”	    File Type Header. For our purposes, it always equals “WAVE”.
  // 12 - 15	    “fmt "	    Format chunk marker. Includes trailing null
  // 16 - 19	    16	        Length of format data as listed above
  // 20 - 21	    1         	Type of format (1 is PCM) - 2 byte integer
  // 22 - 23	    2         	Number of Channels - 2 byte integer
  // 24 - 27	    44100	      Sample Rate, 32 bit integer. Common values are 44100 (CD), 48000 (DAT). Sample Rate = Number of Samples per second, or Hertz.
  // 28 - 31	    176400	    (Sample Rate * BitsPerSample * Channels) / 8.
  // 32 - 33	    4	          (BitsPerSample * Channels) / 8.1 - 8 bit mono2 - 8 bit stereo/16 bit mono4 - 16 bit stereo
  // 34 - 35	    16	        Bits per sample
  // 36 - 39	    “data”	    “data” chunk header. Marks the beginning of the data section.
  // 40 - 43	    integer     Size of the WAV data section.

  bool zPrint = false;  // set == true to list WAV file header information
  FilePath$ = zPath; ErrSD = false;
  // start by opnning the file, to confirm it exists
  SDfile = SD.open(zPath,FILE_READ);
  if (!SDfile) {
    // Openning file failed!
    Serial.println("Failed to open file " + FilePath$);
    SDfile.close(); SDopen = false;
    FileSize = 0; ErrSD = true; return ErrSD;
  } else {
    // File openned successfully
    if (zPrint) {Serial.println("Openned path: " + FilePath$);}
    SDopen = true;
    FileSize = SDfile.size();
    if (zPrint) {Serial.println("File size = " + String(FileSize));}  // this is total file size on SD
    // 0 - 3 now get first data block into Buffer_A[], which should be 'RIFF'
    SDfile.read(Buffer_A,96);
    char zHeadStr[4] = {'R','I','F','F'};
    for (int zP = 0; zP < 4; zP++) {
      if (Buffer_A[zP] != zHeadStr[zP]) {
        Serial.println("Not a RIFF file");
        SDfile.close(); SDopen = false;
        ErrSD = true; return ErrSD;
      }
    }
    if (zPrint) {Serial.println(" 0 -  3 valid RIFF");}
    // 4 -  7 we ignore the file size data
    uint32_t zData4 = Buffer_A[7]; zData4 = (zData4 << 8) + Buffer_A[6];
    zData4 = (zData4 << 8) + Buffer_A[5]; zData4 = (zData4 << 8) + Buffer_A[4];
    if (zPrint) {Serial.println(" 4 -  7 Filesize:" + String(zData4));}
    // 8 = 11 now get data from Buffer_A[], which should be 'WAVE'
    char zWavStr[4] = {'W','A','V','E'};
    for (int zP = 0; zP < 4; zP++) {
      if (Buffer_A[8+zP] != zWavStr[zP]) {
        Serial.println("Not a WAVE file");
        SDfile.close(); SDopen = false;
        ErrSD = true; return ErrSD;
      }
    }
    if (zPrint) {Serial.println(" 8 - 11 .WAV type confirmed");}
    // 12 = 15 now get data from Buffer_A[], which should be 'Fmt '
    char zFmtStr[4] = {'f','m','t',' '};
    for (int zP = 0; zP < 4; zP++) {
      if (Buffer_A[12+zP] != zFmtStr[zP]) {
        Serial.println("12 - 15 fmt header missing");
        SDfile.close(); SDopen = false;
        ErrSD = true; return ErrSD;
      }
    }
    if (zPrint) {Serial.println("12 - 15 fmt header found");}
    // 16 - 19 get length of format sub-chunk data
    zData4 = Buffer_A[19]; zData4 = (zData4 << 8) + Buffer_A[18];
    zData4 = (zData4 << 8) + Buffer_A[17]; zData4 = (zData4 << 8) + Buffer_A[16];
    uint32_t zFmtOff = zData4;
    if (zPrint) {Serial.println("16 - 19 Format data length:" + String(zFmtOff));}
    // 20 - 21 get the type of format
    uint16_t zData2 = Buffer_A[21]; zData2 = (zData2 << 8) + Buffer_A[20];
    if (zPrint) {
      if (zData2 == 1) {Serial.println("20 - 21 PCM data confirmed");}
      else {Serial.println("20 - 21 Data type " + String(zData2) + " unknown"); ErrSD = true; return ErrSD;}
    }
    // 22 - 23 get the number of channels
    zData2 = Buffer_A[23]; zData2 = (zData2 << 8) + Buffer_A[22];
    if (zPrint) {
      if (zData2 == 1) {Serial.println("22 - 23 1 channel mono");}
      else {Serial.println("22 - 23 " + String (zData2) + " channels, not supported here"); ErrSD = true; return ErrSD;}
    }
    // 24 - 27 get sample playback frequency in Hz
    zData4 = Buffer_A[27]; zData4 = (zData4 << 8) + Buffer_A[26];
    zData4 = (zData4 << 8) + Buffer_A[25]; zData4 = (zData4 << 8) + Buffer_A[24];
    if (zPrint) {Serial.println("24 - 27 Sample Freq = " + String(zData4) + "Hz");}
    // 28 - 31 get (Sample Rate * BitsPerSample * Channels) / 8 should be 16000 for (16000 * 8 * 1)/8
    zData4 = Buffer_A[31]; zData4 = (zData4 << 8) + Buffer_A[30];
    zData4 = (zData4 << 8) + Buffer_A[29]; zData4 = (zData4 << 8) + Buffer_A[28];
    if (zPrint) {
      if (zData4 == 16000) {Serial.println("28 - 31 8-bit PCM @ 1600 Hz confirmed");}
      else {Serial.println("28 - 31 " +String (zData4) + "-bits data not supported"); ErrSD = true; return ErrSD;}
    }
    // 32 - 33	(BitsPerSample * Channels) / 8.1 - 8 bit mono2 - 8 bit stereo/16 bit mono4 - 16 bit stereo
    zData2 = Buffer_A[33]; zData2 = (zData2 << 8) + Buffer_A[32];
    if (zPrint) {Serial.println("32 - 33 (BitsPerSample * Channels) / 8 = " + String(zData2));}
    // 34 - 35 get Bits per sample
    zData2 = Buffer_A[35]; zData2 = (zData2 << 8) + Buffer_A[34];
    if (zPrint) {Serial.println("34 - 35 Bits per sample = " + String(zData2));}
    // 36 - 39 confirm start of data block by searching for 'data' check
    char zDataStr[4] = {'d','a','t','a'};
    uint32_t zOff = 36;
    bool zFound = true;
    while (zOff < 60) {
      zFound = true;
      for (int zP = 0; zP < 4; zP++) {
        if (Buffer_A[zOff+zP] != zDataStr[zP]) {zFound = false; break;}
      } if (zFound) {break;}  // success! found 'data' at zOff
      zOff++;
    }
    if (!zFound) {
      if (zPrint) {Serial.println("36 - 39 'data' check was not found");}
      ErrSD = true; return ErrSD;
    } else if (zPrint) {Serial.println("36 - 39 'data' check was found");}
    // 40-43 now get size of playback data block, immediately after 'data' check
    zData4 = Buffer_A[zOff+7]; zData4 = (zData4 << 8) + Buffer_A[zOff+6];
    zData4 = (zData4 << 8) + Buffer_A[zOff+5]; zData4 = (zData4 << 8) + Buffer_A[zOff+4];
    if (zPrint) {Serial.println("40 - 43 WAV data block size = " + String(zData4));}
    // Increase WavSize by the size of the header information block
    WavSize = zData4;
    WavEnd  = zData4 + zOff + 8;
    // Serial.println("Playback size = " + String(FileSize) + " bytes");
    // audio data starts immediately after the size value
    FilePos = zOff + 8;   // add 8 bytes for 'data' + block size
    SDfile.close(); SDopen = false;
  }
  return ErrSD;
}

// --------------------------------------------------------------------------------

void readSDtoBuffer(bool zBp) {
  // Read up to AudBufLen bytes into either Buffer_A[] or Buffer_B[]
  // when we run out of data, we complete the fill with 128 and set  IsrEnd = 1 to
  // invoke a ramp down 128 - 0 on next data load
  // unsigned long zRd = micros(); // used to time this function
  if (!SDopen) {
    SDfile = SD.open(FilePath$,FILE_READ);
    if (!SDfile) {
      Serial.println("ErrSD: Failed to open file " + FilePath$);
      FileSize = 0; ErrSD = true; return;
    }
    FileSize = SDfile.size();
    SDfile.seek(FilePos);
    SDopen = true;
  }
  // Serial.println(FileSize);
  if (FilePos > WavEnd) {
    Serial.println("ErrSD: FilePos exceeds WavEnd");
    SDfile.close(); SDopen = false;
    ErrSD = true; return;
  }
  unsigned long zNum = AudBufLen;
  if ((WavEnd - FilePos) < zNum) {zNum = WavEnd - FilePos;} // remaining data is less than buffer
  if (!zBp) {
    // read into Buffer_A[]
    if (zNum > 0) {SDfile.read(Buffer_A,zNum);}
    Bytes_A = zNum;
    // If buffer is not full, then copy in 128 values as silence
    if (Bytes_A < AudBufLen) {
      // AudioFillBuf(0,Bytes_A,128);
      for (int zP = Bytes_A;zP < AudBufLen;zP++) {Buffer_A[zP] = 128;}
      IsrEnd = 1;
    }
    // Serial.println("Bytes_A=" + String(Bytes_A));
    FilePos += Bytes_A;
  } else {
    // read into Buffer_B[]
    if (zNum > 0) {SDfile.read(Buffer_B,zNum);}
    Bytes_B = zNum;
    // If buffer is not full, then copy in 128 values as silence
    if (Bytes_B < AudBufLen) {
      // AudioFillBuf(1,Bytes_B,128);
      for (int zP = Bytes_B;zP < AudBufLen;zP++) {Buffer_B[zP] = 128;}
      IsrEnd = 1;
    }
    // Serial.println("Bytes_B=" + String(Bytes_B));
    FilePos += Bytes_B;
  }
  // Aud$ += String(micros() - zRd) + "\n";
}

// --------------------------------------------------------------------------------

void seakSDbytes(long zPos) {
  // move the file data pointer
  FilePos = zPos;
  ErrSD = SDfile.seek(zPos);
  if (ErrSD) {Serial.println("ErrSD: seakSDbytes() failed");}
}

// --------------------------------------------------------------------------------

bool TalkAdd(String zFilePath) {
  // This function adds the filepath to the TalkArray[] and invokes the TalkEngine
  // if it is not already playing. Up to 10 audio files can be queued for playing.
  // The array[] acts as a ring buffer, if overloaded filepaths will be missed
  TalkArray[TalkCnt] = zFilePath;
  TalkCnt++; if (TalkCnt >= TalkDepth) {TalkCnt = 0;}
  Serial.println("TalkAdd = " + zFilePath);
  // ensure that the TalkEngine is active, or activate it
  if (!Talk) {Talk = 1;}
  return true;
}

// --------------------------------------------------------------------------------

void TalkEngine() {
  // The TalkEngine is called from the main loop every 20ms whenever Talk != 0
  // It manages the playing of one or more WAV filenames pre-loaded into an array
  // To invoke the TalkEngine you simply call TalkAdd() with a filepath
  // If (GunFire) {Talk = 0; return;}  // prevent talking when gun is being fired

  if (TalkDel > 0) {TalkDel--; return;}   // Pre-play pause delay

  switch (Talk) {
    case 0: break;  // do nothing, default state
    case 1: // get the next file
      if (TalkPnt != TalkCnt) {
        // there are still files in the ring buffer to play
        FilePath$ = TalkArray[TalkPnt];
        TalkPause = 5;  // set minimum inter-phrase delay
        // test for a delay [nn] attached to the end of the filepath
        anyVal = FilePath$.indexOf('[');
        if (anyVal >= 0) {
          // Filepath$ has a [delay] attached to it, so extract it as TalkPause
          Any$ = FilePath$.substring(anyVal+1,FilePath$.length()-1);
          TalkPause = Any$.toInt(); // get the [nn] delay value x 20ms
          // Serial.println(Any$);
          FilePath$ = FilePath$.substring(0,anyVal);
        }
        Serial.println("TalkEngine = " + FilePath$);
        TalkPnt++; if (TalkPnt >= TalkDepth) {TalkPnt = 0;}
        AudioPkEn = true;   // Turn on the LED peak detector
        // ensure audio attenuator is set correctly
        if (TalkAtt != Atten) {if (TalkAtt) {AttenON();} else {AttenOFF();}}
        AudioPlay(FilePath$);
        // if no error, go into monitor end of play mode
        if (!ErrSD) {Talk++; SetLedMode(6); TalkDel = 5;}
      } else {
        // reached the end of the ring buffer, so terminate the TalkEngine
        Talk = 0; SetLedMode(LedLast); AudioPkEn = false;
      }
      break;
    case 2: // Monitor for end of play, when IsrEnd == 3
      if (IsrEnd == 3) {
        AudioStop();          // reset audio system
        Talk = 1;             // reset TalkEngine task pointer
        TalkDel = TalkPause;  // delay next, by previous TalkPause value
      } break;

      case 99: // suspended state, caused by jukebox pause
        break;
  }
}

// --------------------------------------------------------------------------------

void TalkStop() {
  // called to terminate speech mid-track
  AudioStop();
  AudioPkEn = false; AudioPkVal = 0;
  Talk = 0; TalkCnt = 0; TalkDel = 0; TalkPnt = 0;
}

// ---------------------------------------------------------------------------------

void ToneAttach() {
  // Attaches a tone channel for playing musical notes using ledcWriteNote()
  // PWM frequency is 100 KHz, with 12-bit resolution
  if (!PwmAtt) {AudioPwmEn();}  // PWM channel has not been configured for audio

  if (!ToneAtt) {
    ledcAttach(PinTone,1E5,12);
    AudioErr = ledcChangeFrequency(PinTone, 1E5, 12);
    if (AudioErr) {Serial.println("Tone freq reset");}
    else {Serial.println("AudioErr: Tone freq not set");}
    ledcWriteTone(PinTone, 0);  // set Tone duty to off
    AudioEn = false; ToneAtt = true;
  }
}

// ---------------------------------------------------------------------------------

void ToneDetach() {
  // Effectively detaches Tone functions by reverting the channel back to audio PWM
  AudioPwmEn();
  ToneAtt = false; ToneFreq = 0;
}

// ---------------------------------------------------------------------------------

void ToneNote(note_t zNote,uint8_t zOct) {
  // play a note using the tone library
  if (!ToneAtt) {ToneAttach();}  // ensure PWM channel is assigned
  ledcWriteNote(PinTone, zNote, zOct);
  NoteLast = zNote;
}

// ---------------------------------------------------------------------------------

void TonePlay(int zFreq) {
  // Play a note of a given frequency zFreq, on the tone channel
  // If not already attached, then attach it
  if (!ToneAtt) {ToneAttach();}
  ledcWriteTone(PinTone,zFreq);
  ToneFreq = zFreq; // record tone value for sweeping later
}

// --------------------------------------------------------------------------------

