// ################################################################################
//
//  Wii Classic Reach Robot v0.00 Beta
//
//  Released:  13/04/2018
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls a reach robot, driven by 4 servo motors. You can control it
    directly from keyboard commands via the serial monitor interface or via a Wii
    Classic controller using the I2C interface. With a series of simple commands you
    can direct a 'Move' engine to perform a series of tasks, including randomly
    selecting predefined positions.

    This version focuses on using the Wii controller. If a controller is detected it
    polls the deveice every 20ms and respoinds accordingly. If the controlled was and
    is removed it moves to the reset position and powers off servos. Then waits for a
    controller to be re-connected.
    
    Keyboard Commands:
    !       - forces a 'soft' runPOST RESET
    #       - <BACKSPACE> so reset receive buffer
    HM.     - move to HOME position
    K<.     - move arm left [IR] Servo 0
    K>.     - move arm right [IR] Servo 0
    K^.     - move vertical arm down, head moves up, Servo 2
    Kv.     - move vertical arm up, head moves down, Servo 2
    KA.     - open jaws, Servo 3
    KD.     - close jaws, Servo 3
    KE.     - enable all servos PWM
    KL.     - speed limit mode, clears speed increments
    KO.     - disable all servos PWM
    KS.     - move arm backwards, Servo 1
    KW.     - move arm forwards, Servo 1
    KZ.     - dither last value to see if central
    ME0.    - stop the move engine
    ME1.    - restart the move engine
    MLnn.   - load values for move routine nn, but don't run
    MTnn.   - goto predefined position nn
    MXnn.   - load and run a move sequence, 0 - n
    RM.     - move to RESET position
    RP0.    - report current servo values, S0 - S3
    RP1.    - report current servo values, S0 - S3 relative to the Home values
    RP2.    - report current servo values as a moveLoadPosRFV() statement
    SAnn.   - set servo angle in degrees 0 - 180
    SD.     - detach the active servo pin
    SLnn.   - set target servo lower limit in microseconds
    SMnn.   - set servoMain angle in microseconds servoLL - servoUL
    SNnn.   - set the target servo number 0 - 3
    SUnn.   - set target servo upper limit in microseconds
    SVnn.   - set target servo angle in microseconds
*/
// Declare and initialise global variables
#include <Servo.h>
#include <EEPROM.h>
#include <Wire.h>     // Use I2C to communicate with the Wii Classic
#include "Commands.h"

// Define servo calibration constants
#define fwdArmMax 1995 // forward arm Max servo value
#define fwdArmMin 654 // forward arm Min servo value
#define fwdArmVert 1050 // forward arm vertical servo value
#define gripClose 962 // jaws closed servo value
#define gripOpen 1465 // jaws moderately open value (23%)
#define gripWide 2168 // jaws wide open valuess
#define turntableCtr 1511 // turntable servo centre value
#define turntableMax 2131 // turntable servo Max value
#define turntableMin 908 // turntable servo Min value
#define vertArmMaxA 1698 // vertical arm Max 'A' servo value
#define vertArmMaxB 2280 // vertical arm Max 'B' servo value
#define vertArmMinA 1037 // vertical arm Max 'A' servo value
#define vertArmMinB 1377 // vertical arm Max 'B' servo value
#define vertArmMinC 1913 // vertical arm Max 'C' servo value

#define Home0 turntableCtr // home position for servo 0
#define Home1 fwdArmVert // home position for servo 1
#define Home2 vertArmMinC // home position for servo 2
#define Home3 gripOpen // home position for servo 3
#define Reset0 turntableCtr // RESET position for servo 0
#define Reset1 fwdArmMin // RESET position for servo 1
#define Reset2 vertArmMinC // RESET position for servo 2
#define Reset3 gripClose // RESET position for servo 3
#define servoOffMax 0 // sets maximum thermal drift offset for servo 0
#define servoOffRmpDwn 100000 // sets thermal offset ramp down time in miliseconds
#define servoOffRmpUp 10000 // sets thermal offset ramps up time in miliseconds

// Define general constants
#define LEDPin 13 // onboard LED output pin
#define moveArraySize 300 // number of elements in the move sequence
#define moveDefInterval 10000 // default interval for move loop
#define posMax 32 // depth of movePos arrays
#define wiiAdd 0x52 // Wii controller I2C bus address

// define arrays
int moves[moveArraySize]; // memory reserved for movement commands/data
int moveLabels[10]; // move play sequence labels
int movePosF[posMax]; // store for servo[1] target values
int movePosR[posMax]; // store for servo[0] target values
int movePosV[posMax]; // store for servo[2] target values
int servoPins[] = {  2,   3,   4,   5};  // servo output pins assigned
int servoMax[] = {turntableMax,fwdArmMax,vertArmMaxB,gripWide};  // servo max pulse
int servoMin[] = {turntableMin, fwdArmMin, vertArmMinA, gripClose};  // servo max pulse
int servoCtr[] = {Reset0,Reset1,Reset2,Reset3}; // value sent to servos
int servoTrg[] = {   0,   0,   0,   0}; // servos target values
int servoVal[] = {Reset0,Reset1,Reset2,Reset3}; // value sent to servos
int servoTgt[4]; // target values for servos

// define Wii specific items
# define WiiLED A3    // LED used to indicate Wii plugged in
byte WiiCnt;          // received byte counter
uint8_t WiiData[6];   // array to store Wii output
byte WiiError = true; // used to detect transmission errors
int WiiLX,WiiLY;     // left joystick values
byte wiiPhase = 0; // used to perform wii functions at 50 Hz
int WiiRX,WiiRY;     // right joystick values


// Declare and initialise global variables
int Angle;  // start with servo in the centre pos
int anyAny = 0; // any temporary variable
int anyVal = 0; // any temporary variable
int Calibrated = true; // limits functionality, set to true once calibrated
char cmdMode; // command mode
int cmdRec; // > 0 if a '~' has been received
char cmdType; // command type
int cmdVal; // value associated with a cmdType
int incOffset; // > 0 flag which controls thermal offset increments
int incS0; // servo[0] increment
int incS1; // servo[1] increment
int incS2; // servo[2] increment
int incS3; // servo[3] increment
unsigned long interval; // main loop interval in microseconds
char keyChar; // any keyboard character
int keyPnt; // pointer to last servo affected by keyboard demand
int keyRec; // > 0 if a key has been received
int keyVal; // any keyboard value
int MasterMode; // mode to be run from RESET
int moveCmd; // the current move command
int moveCnt; // counter used in moving to target values
int moveForLp; // counter value for For...Next loop
int moveForPnt; // pointer value in For...Next loop
int moveInc; // temp move value
unsigned long moveInterval; // move engine interval in microseconds
int moveLast; // previous movePnt value
int moveMem0,moveMem1,moveMem2; // temporary pushed values
unsigned long moveMicros; // move engine interval timer in microseconds
int movePause; // pause after move in loop cycles, default = 0;
int movePnt; // movement array pointer, default is -1
int moveReturn; // used to store return line in move engine
int moveRpt; // flag set to report servo values
int moveRun; // -1 = run, 0 = stop, >0 = run n then stop
int moveTask; // main task pointer in movement engine
int moveSubTask; // sub-task pointer used in moves
long moveWait; // wait counter in cycles, could be quite long. ie. minutes
unsigned long nextMicros; // main loop interval timer in microseconds
int Once = 1; // only = 1 from power reset
long ON_Timer = 0; // time from power-on in loop 10 ms loop cycles
long ON_Timer_Last = 0; // records previous time event
int PWR_timeout; // disables servos after a time-out, when in manual mode
int servoAtt; // 0 = OFF, 1 = ON attached status
int servoEn; // 0 = OFF, 1 = ON enabled status
int servoLL = 544; // servo min time setting
int servoNum = 0; // target servo number 0 - 3 for setting LL/UL limits, default = 0
int servoOff0; // thermal offset for servo 0
long servoOffDec; // thermal offset decrement counter
long servoOffDecT; // thermal offset decrement counter total count
long servoOffInc; // thermal offset increment counter
long servoOffIncT; // thermal offset increment counter total count
int servoPin;  // servo output pin
int servoUL = 2400; // servo max time setting
int shift; // any shift << or >> value
int timeDelay = 1; // delay in milliseconds

// Declare objects
Servo servoMain; // define temp Servo case
Servo servoInst[4]; // define Reach Robot servos

void setup() {
  // put your setup code here, to run once:
  pinMode(LEDPin, OUTPUT);
  pinMode(WiiLED, OUTPUT);  // set output pin for WiiLED
  digitalWrite(WiiLED,HIGH);   // turn OFF WiiLED
  Serial.begin(57600); // high baud rate for sweeping functions
  runPOST();
}

void loop() {
  // put your main code here, to run repeatedly:
  if (micros() >= moveMicros) {
    // normally runs every 50ms, but can be varied to suit speed of
    // robot movements
    moveMicros = micros() + moveInterval;
    if (movePnt >= 0) {moveEngine();}
  }
  
  // read keyboard input when not doing other things
  readKey(); // empty Rx buffer
  if (keyVal != -1) {keyRec = 10;} // set a timer-out for adjustment speed
  
  if (micros() >= nextMicros) {
    // do these every 10ms
    nextMicros = micros() + interval; // set next time point
    ON_Timer++; // increment ON timer
    if (keyRec > 0) {
      // key released time-out so reset increment values
      keyRec--; if (keyRec == 0) {incS0 = 1; incS1 = 1; incS2 = 1;  incS3 = 1;}
    }
    if (WiiError == 0) {
      // Wii attached and responding so scan it
      digitalWrite(WiiLED,LOW);   // turn ON WiiLED
      wiiPhase--; 
      if (wiiPhase < 1) {
        // in wiiPhase 0 every 20ms
        wiiPhase = 2; // reset phase
        WiiUpdate(); // read Wii in this phase
        WiiLX = WiiLeftStickX(); // decode LX 0/32/63
        WiiLY = WiiLeftStickY(); // decode LY 0/32/63
        WiiRX = WiiRightStickX(); // decode RX 0/16/31
        WiiRY = WiiRightStickY(); // decode RY 0/16/31
      } else {
        // in wiiPhase 1 every 20ms or nmore often if WiiError > 0
        if (wiiPhase == 1) {
          // ensures we are in phase 1
          // check buttons
          shift = 0;
          if (WiiPressedRowBit(4,3)) {moveGoHome();} // Home - goto Home
          if (WiiPressedRowBit(4,1)) {shift = 2;} // RT - Set shift to reduce movement
          if (WiiPressedRowBit(5,2)) {shift = 3;} // RZ - Set shift to further reduce movement
          if (WiiPressedRowBit(4,4)) {moveGoRESET();} // Select - goto RESET
          if (WiiPressedRowBit(5,7) && WiiPressedRowBit(4,2) && WiiPressedRowBit(5,2)) {
            // Lz+Start+Rz so run moveEngine
            MasterMode = 2; moveToTest00 (); moveStart();
          }

          // check joysticks if moveEngine is not active
          if (movePnt < 0) {
            if (WiiLX > 32) {JoyJawClose((WiiLX-32)>>shift);} 
            else if (WiiLX < 32) {JoyJawOpen((32-WiiLX)>>shift);} 
            if (WiiLY > 32) {JoyArmFwd((WiiLY-32)>>shift);} 
            else if (WiiLY < 32) {JoyArmBck((32-WiiLY)>>shift);} 
            if (WiiRX > 16) {JoyTurnRight((WiiRX-16)>>shift);} 
            else if (WiiRX < 16) {JoyTurnLeft((16-WiiRX)>>shift);} 
            if (WiiRY > 16) {JoyVertDwn((WiiRY-16)>>shift);} 
            else if (WiiRY < 16) {JoyVertUp((16-WiiRY)>>shift);}
          }
        }
      }
    } else {
      // Wii not attached so keep polling the address every second
      digitalWrite(WiiLED,HIGH);   // turn OFF WiiLED
      wiiPhase--; if (wiiPhase < 1) {
        wiiInitialise();
        if (WiiError != 0) {wiiPhase = 100;}
      }
    }
    if ((PWR_timeout > 0) && (movePnt < 0)) {
      // previously servo code received so detach servos after a 2 sec timeout delay
      PWR_timeout--; if (PWR_timeout == 1) {detachServos();}
    }
    if (incOffset > 0) {incMoveOffset();}
    else {decMoveOffset();}
  }
}

// ----------------------------------------------------------------------

void doCmd() {
  // a '.' has been received so execute command if valid
  int zPrint = 1; // set flag for echo Rx response
  switch (cmdMode) {
    case ' ': break;
    case 'H':
      switch (cmdType) {
        case 'M':
          // move to HOME position
          if (servoEn > 0) {
            moveGoHome();
          } break;
      } zPrint = 0; break;
    case 'K':
      zPrint = 0;
      if (servoEn > 0) {
        switch (cmdType) {
          case '<': keyBdTurnLeft(); break;
          case '>': keyBdTurnRight(); break;
          case '^': keyBdVertDwn(); break;
          case 'V': keyBdVertUp(); break;
          case 'A': keyBdJawOpen(); break;
          case 'D': keyBdJawClose(); break;
          case 'L': incS0 = 1; incS1 = 1; incS2 = 1;  incS3 = 1; break;
          case 'S': keyBdArmBck(); break;
          case 'W': keyBdArmFwd(); break;
        }
      }
      switch (cmdType) {
        case 'E': servoEn = 1; PWR_timeout = 200; attachServos(100); movePnt = -1; zPrint = 1; break;
        case 'O': servoEn = 0; detachServos(); zPrint = 1; break;
        case 'Z': keyBdDither(); break;
      } break;
    case 'M':
      switch (cmdType) {
        case 'E':
          switch (cmdVal) {
            case 0:
              // stop the move engine
              moveLast = movePnt; movePnt = -1;
              detachServos(); break;
            case 1:
              // restart the move engine
              movePnt = moveLast; moveRun = -1; attachServos(10);
              break;
          } break;
        case 'L':
          // load a move sequence but don't run it
          switch (cmdVal) {
            case 0:
              moveToTest00 (); break;
          } moveLast = 0; movePnt = -1; break;
        case 'T':
          // load a predefined position and move to it
          if ((cmdVal >= 0) && (cmdVal < posMax)) {
            movePnt = 0; moveLoad2(cmdGoPos, cmdVal);
            moveLoad1(cmdEndOn); moveStart();
          } break;
        case 'X':
          // load and eXecute a move run sequence
          switch (cmdVal) {
            case 0:
              moveToTest00(); moveStart(); break;
            case 1:
              moveToTest00(); movePause = 20; moveStart(); break;
          } break;
      } zPrint = 0; break;
    case 'R':
      switch (cmdType) {
        case 'M':
          // move to RESET position
          if (servoEn > 0) {
            moveGoRESET();
          } break;
        case 'P':
          if (movePnt >= 0) {moveRun++; moveRpt = 1;}
          else {
            if (cmdVal == 0 ) {reportValues();}
            if (cmdVal == 1 ) {reportOffsets();}
            if (cmdVal == 2) {reportMoveData();}
          } break;
      } zPrint = 0; break;
    case 'S':
      switch (cmdType) {
        case ' ': break;
        case 'A':
          Angle = cmdVal; servoMain.write(Angle);
          // report the angle in microseconds
          Serial.print(F("SM="));
          Serial.print(servoMain.readMicroseconds());
          Serial.println("."); zPrint = 0;
          break;
        case 'D': servoPin = -1; servoMain.detach(); break;
        case 'L':
          // set the lower limit of the target servo 0 - 3
          servoMin[servoNum] = cmdVal;
          if (servoVal[servoNum] < cmdVal) {
            // if servo is outside of this limit move it
            PWR_timeout = 200; if (servoAtt < 1) {attachServos(0); movePnt = -1;}
            servoVal[servoNum] = cmdVal;
            servoInst[servoNum].writeMicroseconds(servoVal[servoNum]);
            Serial.print(F("SV")); Serial.print(servoNum); Serial.print(F("="));
            Serial.println(servoVal[servoNum]); zPrint = 0;
          } break;
        case 'M':
          Angle = cmdVal; servoMain.writeMicroseconds(Angle);
          // report the angle in degrees
          Serial.print(F("SA="));
          Serial.print(servoMain.read());
          Serial.println("."); zPrint = 0;
          break;
        case 'N':
          // set the target servo number 0 - 3
          servoNum = cmdVal; break;
        case 'U':
          // set the upper limit of the target servo 0 - 3
          servoMax[servoNum] = cmdVal;
          if (servoVal[servoNum] > cmdVal) {
            // if servo is outside of this limit move it
            PWR_timeout = 200; if (servoAtt < 1) {attachServos(0); movePnt = -1;}
            servoVal[servoNum] = cmdVal;
            servoInst[servoNum].writeMicroseconds(servoVal[servoNum]);
            Serial.print(F("SV")); Serial.print(servoNum); Serial.print(F("="));
            Serial.println(servoVal[servoNum]); zPrint = 0;
          } break;
        case 'V':
          // set the angle of the target servo 0 - 3
          // ensure that it is within the current limits
          cmdVal = max(servoMin[servoNum], cmdVal);
          cmdVal = min(servoMax[servoNum], cmdVal);
          servoVal[servoNum] = cmdVal;
          PWR_timeout = 200; if (servoAtt < 1) {attachServos(0); movePnt = -1;}
          if (Calibrated && (servoNum == 1)) {
            setVertMinMax(); servoInst[2].writeMicroseconds(servoVal[2]);
            Serial.print(F("SV2=")); Serial.println(servoVal[2]);
          } 
          if (servoNum == 0) {
            incOffset = true;  // turn demand so increment thermal offset counter
            servoInst[0].writeMicroseconds(servoVal[0] - servoOff0);
          } else {
            servoInst[servoNum].writeMicroseconds(servoVal[servoNum]);
          }
          Serial.print(F("SV")); Serial.print(servoNum); Serial.print(F("="));
          Serial.println(servoVal[servoNum]); zPrint = 0;
          break;
      } break;
  }
  if (zPrint > 0) {
    // is not a reporting function, echo the received command
    Serial.print(cmdMode); Serial.print(cmdType); Serial.print(cmdVal); Serial.println('.');
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0;
}

// ----------------------------------------------------------------------

void emptySerial() {
  // empty the serial input buffer
  keyVal = 0;
  while (keyVal != -1) {
    keyVal = Serial.read();
  } cmdRec = 0; // clear the recevied 
}

// ----------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = (cmdVal * 10) + zVal;  
}

// ----------------------------------------------------------------------

void readKey() {
  // reads a key from the keyboard annd reacts accordingly
  keyVal = Serial.read();
  if (keyVal != -1) {
    keyChar = char(keyVal); cmdRec = 1; // value received
//      Serial.print("keyVal=");
//      Serial.print(keyChar);
//      Serial.print("   ASCII=");
//      Serial.println(keyVal);
    int zNF = 0;
    switch (keyChar) {
      case '.': doCmd(); zNF = 1; break;
      case '!': runPOST(); zNF = 1; break;
      case '#': cmdMode = ' '; cmdType = ' '; cmdVal = 0; zNF = 1; break;
      case '~': zNF = 1; Serial.println("~"); break; // null tick received so respond
      case '0': extendCmdVal(0); zNF = 1;break;
      case '1': extendCmdVal(1); zNF = 1;break;
      case '2': extendCmdVal(2); zNF = 1;break;
      case '3': extendCmdVal(3); zNF = 1;break;
      case '4': extendCmdVal(4); zNF = 1;break;
      case '5': extendCmdVal(5); zNF = 1;break;
      case '6': extendCmdVal(6); zNF = 1;break;
      case '7': extendCmdVal(7); zNF = 1;break;
      case '8': extendCmdVal(8); zNF = 1;break;
      case '9': extendCmdVal(9); zNF = 1;break;
    }
    if (zNF == 0) {
      if (cmdMode == ' ') {
        // test for new Command Mode char?
        switch (keyChar) {
          case 'e': cmdMode = 'E'; break;
          case 'E': cmdMode = 'E'; break;
          case 'h': cmdMode = 'H'; break;
          case 'H': cmdMode = 'H'; break;
          case 'k': cmdMode = 'K'; break;
          case 'K': cmdMode = 'K'; break;
          case 'm': cmdMode = 'M'; break;
          case 'M': cmdMode = 'M'; break;
          case 'r': cmdMode = 'R'; break;
          case 'R': cmdMode = 'R'; break;
          case 's': cmdMode = 'S'; break;
          case 'S': cmdMode = 'S'; break;
        } cmdType = ' '; cmdVal = 0;
      } else {
        // test for Command Type char?
        switch (keyChar) {
          case '<': cmdType = '<'; break;
          case '>': cmdType = '>'; break;
          case '^': cmdType = '^'; break;
          case 'A': cmdType = 'A'; break;
          case 'd': cmdType = 'D'; break;
          case 'D': cmdType = 'D'; break;
          case 'e': cmdType = 'E'; break;
          case 'E': cmdType = 'E'; break;
          case 'g': cmdType = 'G'; break;
          case 'G': cmdType = 'G'; break;
          case 'l': cmdType = 'L'; break;
          case 'L': cmdType = 'L'; break;
          case 'm': cmdType = 'M'; break;
          case 'M': cmdType = 'M'; break;
          case 'n': cmdType = 'N'; break;
          case 'N': cmdType = 'N'; break;
          case 'o': cmdType = 'O'; break;
          case 'O': cmdType = 'O'; break;
          case 'p': cmdType = 'P'; break;
          case 'P': cmdType = 'P'; break;
          case 's': cmdType = 'S'; break;
          case 'S': cmdType = 'S'; break;
          case 't': cmdType = 'T'; break;
          case 'T': cmdType = 'T'; break;
          case 'u': cmdType = 'U'; break;
          case 'U': cmdType = 'U'; break;
          case 'v': cmdType = 'V'; break;
          case 'V': cmdType = 'V'; break;
          case 'w': cmdType = 'W'; break;
          case 'W': cmdType = 'W'; break;
          case 'x': cmdType = 'X'; break;
          case 'X': cmdType = 'X'; break;
          case 'z': cmdType = 'Z'; break;
          case 'Z': cmdType = 'Z'; break;
        }
      }
    }
  }
}

// ----------------------------------------------------------------------

void RESET() {
  // perform a soft RESET
  servoMain.detach();
  detachServos();
  digitalWrite(WiiLED,HIGH);   // turn OFF WiiLED
  setDefaults();
}

// ----------------------------------------------------------------------

void runPOST() {
  // called during start-up
  Serial.print(F("\n\n\n\n\n\n\n\n\n\n"));
  Serial.println(F("Reach Robot v0.11"));
  Serial.println(F("Starting POST..."));
  RESET();
  emptySerial();
  Serial.println(F("Centring Servos..."));
  attachServos(100); delay(200); detachServos();
  Serial.println(F("Pause 2s before RESET test..."));
  digitalWrite(LEDPin, HIGH); delay(1000);
  testCtrlApp(); // see if connected to Windows app?
  digitalWrite(LEDPin, LOW);
  if ((cmdRec < 1) && (Once > 0) && Calibrated) {
    // Power-On reset or no serial link so enter demo modes
    // but only if the robot has been calibrated
    // read the MasterMode value then reset it to zero
    testResetState(); // read EEPROM and count RESETs
    MasterMode = EEPROM.read(3); EEPROM.update(3, 0);
  } else {
    // on serial link so force into mode 1
    MasterMode = 1;
  }
  Serial.print(F("Demo mode ")); Serial.print(MasterMode);
  Serial.println(F(" initiated!"));
  switch (MasterMode) {
    case 1:
      // single RESET so do nothing
      if (cmdRec < 1) {
        // no serial link
        Serial.println(F("Sleeping...zzz"));
        servoMain.detach(); detachServos();
      } else {
        // serial link so send initialisation data
        Serial.print(F("SN0LL")); Serial.println(servoMin[0]);
        Serial.print(F("SN0UL")); Serial.println(servoMax[0]);
        Serial.print(F("SN1LL")); Serial.println(servoMin[1]);
        Serial.print(F("SN1UL")); Serial.println(servoMax[1]);
        Serial.print(F("SN2LL")); Serial.println(servoMin[2]);
        Serial.print(F("SN2UL")); Serial.println(servoMax[2]);
        Serial.print(F("SN3LL")); Serial.println(servoMin[3]);
        Serial.print(F("SN3UL")); Serial.println(servoMax[3]);
        reportEachValue();
        reportHomeValues();
      } break;
    case 2:
      // start demo mode 1
//      moveToTest00 (); moveStart();
      break;
  }
  Serial.println(F("POST complete!"));
  Once = 0;
}

