// ################################################################################
//
//  Reach Robot Cal v0.00 Beta
//
//  Released:  23/02/2018
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls a reach robot, driven by 4 servo motors. You can control it
    directly from keyboard commands via the serial monitor interface or via an
    infrared remote control. With a series of simple commands you can direct a 'Move'
    engine to perform a series of tasks, including randomly selecting predefined
    positions.

    This version utilises IR control, where received codes are used to effectively
    trigger keyboard functions, as indicated by [IR]. There by replacing the Windows
    app for calibration and sequence creation. You can now initiate 'Move' sequences
    with the 1,2,3 IR keys.
    
    Keyboard Commands:
    !       - forces a 'soft' runPOST RESET
    #       - <BACKSPACE> so reset receive buffer
    HM.     - move to HOME position
    K<.     - move arm left [IR] Servo 0
    K>.     - move arm right [IR] Servo 0
    K^.     - move vertical arm down, head moves up, Servo 2
    Kv.     - move vertical arm up, head moves down, Servo 2
    KA.     - open jaws, Servo 3
    KD.     - close jaws, Servo 3
    KE.     - enable all servos PWM
    KL.     - speed limit mode, clears speed increments
    KO.     - disable all servos PWM
    KS.     - move arm backwards, Servo 1
    KW.     - move arm forwards, Servo 1
    KZ.     - dither last value to see if central
    ME0.    - stop the move engine
    ME1.    - restart the move engine
    MLnn.   - load values for move routine nn, but don't run
    MTnn.   - goto predefined position nn
    MXnn.   - load and run a move sequence, 0 - n
    RM.     - move to RESET position
    RP0.     - report current servo values, S0 - S3
    RP1.     - report current servo values, S0 - S3 relative to the Home values
    RP2.     - report current servo values as a moveLoadPosRFV() statement
    SAnn.   - set servo angle in degrees 0 - 180
    SD.     - detach the active servo pin
    SLnn.   - set target servo lower limit in microseconds
    SMnn.   - set servoMain angle in microseconds servoLL - servoUL
    SNnn.   - set the target servo number 0 - 3
    SUnn.   - set target servo upper limit in microseconds
    SVnn.   - set target servo angle in microseconds
*/
// Declare and initialise global variables
#include <Servo.h>
#include <EEPROM.h>
#include "Commands.h"
#include "IRremote.h"

// Define servo calibration constants
#define fwdArmMax 1450 // forward arm Max servo value
#define fwdArmMin 900 // forward arm Min servo value
#define fwdArmVert 1100 // forward arm vertical servo value
#define gripClose 970 // jaws closed servo value
#define gripOpen 1400 // jaws moderately open value (23%)
#define gripWide 2000 // jaws wide open value
#define turntableCtr 1500 // turntable servo centre value
#define turntableMax 1900 // turntable servo Max value
#define turntableMin 1100 // turntable servo Min value
#define vertArmMaxA 1900 // vertical arm Max 'A' servo value
#define vertArmMaxB 2000 // vertical arm Max 'B' servo value
#define vertArmMinA 1650 // vertical arm Max 'A' servo value
#define vertArmMinB 1650 // vertical arm Max 'B' servo value
#define vertArmMinC 1900 // vertical arm Max 'C' servo value

#define Home0 turntableCtr // home position for servo 0
#define Home1 fwdArmVert // home position for servo 1
#define Home2 1900 // home position for servo 2
#define Home3 gripOpen // home position for servo 3
#define Reset0 turntableCtr // RESET position for servo 0
#define Reset1 1200 // RESET position for servo 1
#define Reset2 1900 // RESET position for servo 2
#define Reset3 gripOpen // RESET position for servo 3
#define servoOffMax 0 // sets maximum thermal drift offset for servo 0
#define servoOffRmpDwn 60000 // sets thermal offset ramp down time in miliseconds
#define servoOffRmpUp 10000 // sets thermal offset ramps up time in miliseconds

// Define general constants
#define IR_Inc_Max 12 // Max servo increment used in IR auto move mode
#define LEDPin 13 // onboard LED output pin
#define moveArraySize 300 // number of elements in the move sequence
#define moveDefInterval 10000 // default interval for move loop
#define posMax 32 // depth of movePos arrays

// define arrays
int moves[moveArraySize]; // memory reserved for movement commands/data
int moveLabels[10]; // move play sequence labels
int movePosF[posMax]; // store for servo[1] target values
int movePosR[posMax]; // store for servo[0] target values
int movePosV[posMax]; // store for servo[2] target values
int servoPins[] = {  2,   3,   4,   5};  // servo output pins assigned
int servoMax[] = {turntableMax,fwdArmMax,vertArmMaxB,gripWide};  // servo max pulse
int servoMin[] = {turntableMin, fwdArmMin, vertArmMinA, gripClose};  // servo max pulse
int servoCtr[] = {Reset0,Reset1,Reset2,Reset3}; // value sent to servos
int servoTrg[] = {   0,   0,   0,   0}; // servos target values
int servoVal[] = {Reset0,Reset1,Reset2,Reset3}; // value sent to servos
int servoTgt[4]; // target values for servos

// Declare and initialise global variables
int Angle;  // start with servo in the centre pos
int anyAny = 0; // any temporary variable
int anyVal = 0; // any temporary variable
int Calibrated = false; // limits functionality, set to true once calibrated
char cmdMode; // command mode
int cmdRec; // > 0 if a '~' has been received
char cmdType; // command type
int cmdVal; // value associated with a cmdType
int incOffset; // > 0 flag which controls thermal offset increments
int incS0; // servo[0] increment
int incS1; // servo[1] increment
int incS2; // servo[2] increment
int incS3; // servo[3] increment
int IR_Inc = 1; // IR mov ement increment
int IR_Last; // previous IR code
int IR_Match; // detects consequitive matching IR codes
int IR_Num = 1; // multiplier used in IR auto-movement
int IR_Repeat; // auto-IR code repeat counter
unsigned long interval; // main loop interval in microseconds
char keyChar; // any keyboard character
int keyPnt; // pointer to last servo affected by keyboard demand
int keyRec; // > 0 if a key has been received
int keyVal; // any keyboard value
int MasterMode; // mode to be run from RESET
int moveCmd; // the current move command
int moveCnt; // counter used in moving to target values
int moveForLp; // counter value for For...Next loop
int moveForPnt; // pointer value in For...Next loop
int moveInc; // temp move value
unsigned long moveInterval; // move engine interval in microseconds
int moveLast; // previous movePnt value
int moveMem0,moveMem1,moveMem2; // temporary pushed values
unsigned long moveMicros; // move engine interval timer in microseconds
int movePause; // pause after move in loop cycles, default = 0;
int movePnt; // movement array pointer, default is -1
int moveReturn; // used to store return line in move engine
int moveRpt; // flag set to report servo values
int moveRun; // -1 = run, 0 = stop, >0 = run n then stop
int moveTask; // main task pointer in movement engine
int moveSubTask; // sub-task pointer used in moves
long moveWait; // wait counter in cycles, could be quite long. ie. minutes
unsigned long nextMicros; // main loop interval timer in microseconds
int Once = 1; // only = 1 from power reset
long ON_Timer = 0; // time from power-on in loop 10 ms loop cycles
long ON_Timer_Last = 0; // records previous time event
int PWR_timeout; // disables servos after a time-out, when in manual mode
int receiver = 8; // pin 3 of IR receiver to Arduino digital pin 8
int servoAtt; // 0 = OFF, 1 = ON attached status
int servoEn; // 0 = OFF, 1 = ON enabled status
int servoLL = 544; // servo min time setting
int servoNum = 0; // target servo number 0 - 3 for setting LL/UL limits, default = 0
int servoOff0; // thermal offset for servo 0
long servoOffDec; // thermal offset decrement counter
long servoOffDecT; // thermal offset decrement counter total count
long servoOffInc; // thermal offset increment counter
long servoOffIncT; // thermal offset increment counter total count
int servoPin;  // servo output pin
int servoUL = 2400; // servo max time setting
int timeDelay = 1; // delay in milliseconds

// Declare objects
Servo servoMain; // define temp Servo case
Servo servoInst[4]; // define Reach Robot servos
IRrecv irrecv(receiver);  // create instance of 'irrecv'
decode_results results;   // create instance of IR 'decode_results'

void setup() {
  // put your setup code here, to run once:
  pinMode(LEDPin, OUTPUT);
  Serial.begin(57600); // high baud rate for sweeping functions
  runPOST();
  irrecv.enableIRIn(); // Start the receiver
  irrecv.blink13(true);
}

void loop() {
  // put your main code here, to run repeatedly:
  if (micros() >= moveMicros) {
    // normally runs every 50ms, but can be varied to suit speed of
    // robot movements
    moveMicros = micros() + moveInterval;
    if (movePnt >= 0) {moveEngine();}
  }
  
  // read keyboard input when not doing other things
  readKey(); // empty Rx buffer
  if (keyVal != -1) {keyRec = 10;} // set a timer-out for adjustment speed
  
  if (micros() >= nextMicros) {
    // do these every 10ms
    nextMicros = micros() + interval; // set next time point
    ON_Timer++; // increment ON timer
    if (keyRec > 0) {
      // key released time-out so reset increment values
      keyRec--; if (keyRec == 0) {incS0 = 1; incS1 = 1; incS2 = 1;  incS3 = 1;}
    }
    if (irrecv.decode(&results)) {
      // IR code received from TL1838 sensor
      if (MasterMode < 2) {
        // in normal reset mode so respond to IR joystick commands
  //      Serial.print("IR Period="); Serial.println(ON_Timer - ON_Timer_Last); 
        ON_Timer_Last = ON_Timer;
        incS0 = 1; incS1 = 1; incS2 = 1;  incS3 = 1;
        if (IR_Last == (int)results.value) {
          // same IR code received more than once consequitively
          IR_Match++; if (IR_Match >= 2) {
            if (IR_Match == 2) {Serial.println(F("Auto"));}
            IR_Repeat = 15; if (IR_Inc < IR_Inc_Max) {IR_Num++;}
            IR_Inc = 1 + (IR_Num / 32);
            incS0 = IR_Inc; incS1 = IR_Inc; incS2 = IR_Inc;  incS3 = IR_Inc;
          }
        } else {
          IR_Match = 0; // clear the matching codes counter
          IR_Repeat = 0; // clear the auto-repeat counter
          IR_Inc = 1; IR_Num = 1; // reset auto-increments
        }
        translateIR_Samsung_0111(results.value);
        IR_Last = (int)results.value;
        irrecv.resume(); // prepare to receive the next value
      } else {
          // already running a move sequence so only respond to PWR
          translateIR_Samsung_0111_PWR(results.value);
          irrecv.resume(); // prepare to receive the next value
      }
    } else {
      if (IR_Repeat > 0) {
        // same code received more than once to auto-repeat movement
        IR_Repeat--; // repeat movement will time-out  at IR_Repeat = 0
        if (IR_Inc < IR_Inc_Max) {IR_Num++;}
        IR_Inc = 1 + (IR_Num / 32);
        incS0 = IR_Inc; incS1 = IR_Inc; incS2 = IR_Inc;  incS3 = IR_Inc;
        translateIR_Samsung_0111(IR_Last);
      }
    }
    if ((PWR_timeout > 0) && (movePnt < 0)) {
      // previously servo code received so detach servos after a 2 sec timeout delay
      PWR_timeout--; if (PWR_timeout == 1) {detachServos();}
    }
    if (incOffset > 0) {incMoveOffset();}
    else {decMoveOffset();}
  }
}

// ----------------------------------------------------------------------

void doCmd() {
  // a '.' has been received so execute command if valid
  int zPrint = 1; // set flag for echo Rx response
  switch (cmdMode) {
    case ' ': break;
    case 'H':
      switch (cmdType) {
        case 'M':
          // move to HOME position
          if (servoEn > 0) {
            moveInit(); moveLoad1(cmdHome); moveLoad1(cmdEndOn); moveStart();
            reportEachValue();
            Serial.println(F("Home Position"));
          } break;
      } zPrint = 0; break;
    case 'K':
      zPrint = 0;
      if (servoEn > 0) {
        switch (cmdType) {
          case '<': keyBdTurnLeft(); break;
          case '>': keyBdTurnRight(); break;
          case '^': keyBdVertDwn(); break;
          case 'V': keyBdVertUp(); break;
          case 'A': keyBdJawOpen(); break;
          case 'D': keyBdJawClose(); break;
          case 'L': incS0 = 1; incS1 = 1; incS2 = 1;  incS3 = 1; break;
          case 'S': keyBdArmBck(); break;
          case 'W': keyBdArmFwd(); break;
        }
      }
      switch (cmdType) {
        case 'E': servoEn = 1; PWR_timeout = 200; attachServos(100); movePnt = -1; zPrint = 1; break;
        case 'O': servoEn = 0; detachServos(); zPrint = 1; break;
        case 'Z': keyBdDither(); break;
      } break;
    case 'M':
      switch (cmdType) {
        case 'E':
          switch (cmdVal) {
            case 0:
              // stop the move engine
              moveLast = movePnt; movePnt = -1;
              detachServos(); break;
            case 1:
              // restart the move engine
              movePnt = moveLast; moveRun = -1; attachServos(10);
              break;
          } break;
        case 'L':
          // load a move sequence but don't run it
          switch (cmdVal) {
            case 0:
              moveToTest00 (); break;
          } moveLast = 0; movePnt = -1; break;
        case 'T':
          // load a predefined position and move to it
          if ((cmdVal >= 0) && (cmdVal < posMax)) {
            movePnt = 0; moveLoad2(cmdGoPos, cmdVal);
            moveLoad1(cmdEndOn); moveStart();
          } break;
        case 'X':
          // load and eXecute a move run sequence
          switch (cmdVal) {
            case 0:
              moveToTest00(); moveStart(); break;
            case 1:
              moveToTest00(); movePause = 20; moveStart(); break;
          } break;
      } zPrint = 0; break;
    case 'R':
      switch (cmdType) {
        case 'M':
          // move to RESET position
          if (servoEn > 0) {
            moveInit(); moveLoad1(cmdReset); moveLoad1(cmdEnd); moveStart();
            reportEachValue();
            Serial.println(F("Reset Position"));
          } break;
        case 'P':
          if (movePnt >= 0) {moveRun++; moveRpt = cmdVal+1;}
          else {
            if (cmdVal == 0 ) {reportValues();}
            if (cmdVal == 1 ) {reportOffsets();}
            if (cmdVal == 2) {reportMoveData();}
          } break;
      } zPrint = 0; break;
    case 'S':
      switch (cmdType) {
        case ' ': break;
        case 'A':
          Angle = cmdVal; servoMain.write(Angle);
          // report the angle in microseconds
          Serial.print(F("SM="));
          Serial.print(servoMain.readMicroseconds());
          Serial.println("."); zPrint = 0;
          break;
        case 'D': servoPin = -1; servoMain.detach(); break;
        case 'L':
          // set the lower limit of the target servo 0 - 3
          servoMin[servoNum] = cmdVal;
          if (servoVal[servoNum] < cmdVal) {
            // if servo is outside of this limit move it
            PWR_timeout = 200; if (servoAtt < 1) {attachServos(0); movePnt = -1;}
            servoVal[servoNum] = cmdVal;
            servoInst[servoNum].writeMicroseconds(servoVal[servoNum]);
            Serial.print(F("SV")); Serial.print(servoNum); Serial.print(F("="));
            Serial.println(servoVal[servoNum]); zPrint = 0;
          } break;
        case 'M':
          Angle = cmdVal; servoMain.writeMicroseconds(Angle);
          // report the angle in degrees
          Serial.print(F("SA="));
          Serial.print(servoMain.read());
          Serial.println("."); zPrint = 0;
          break;
        case 'N':
          // set the target servo number 0 - 3
          servoNum = cmdVal; break;
        case 'U':
          // set the upper limit of the target servo 0 - 3
          servoMax[servoNum] = cmdVal;
          if (servoVal[servoNum] > cmdVal) {
            // if servo is outside of this limit move it
            PWR_timeout = 200; if (servoAtt < 1) {attachServos(0); movePnt = -1;}
            servoVal[servoNum] = cmdVal;
            servoInst[servoNum].writeMicroseconds(servoVal[servoNum]);
            Serial.print(F("SV")); Serial.print(servoNum); Serial.print(F("="));
            Serial.println(servoVal[servoNum]); zPrint = 0;
          } break;
        case 'V':
          // set the angle of the target servo 0 - 3
          // ensure that it is within the current limits
          cmdVal = max(servoMin[servoNum], cmdVal);
          cmdVal = min(servoMax[servoNum], cmdVal);
          servoVal[servoNum] = cmdVal;
          PWR_timeout = 200; if (servoAtt < 1) {attachServos(0); movePnt = -1;}
          if (Calibrated && (servoNum == 1)) {
            setVertMinMax(); servoInst[2].writeMicroseconds(servoVal[2]);
            Serial.print(F("SV2=")); Serial.println(servoVal[2]);
          } 
          if (servoNum == 0) {
            incOffset = true;  // turn demand so increment thermal offset counter
            servoInst[0].writeMicroseconds(servoVal[0] + servoOff0);
          } else {
            servoInst[servoNum].writeMicroseconds(servoVal[servoNum]);
          }
          Serial.print(F("SV")); Serial.print(servoNum); Serial.print(F("="));
          Serial.println(servoVal[servoNum]); zPrint = 0;
          break;
      } break;
  }
  if (zPrint > 0) {
    // is not a reporting function, echo the received command
    Serial.print(cmdMode); Serial.print(cmdType); Serial.print(cmdVal); Serial.println('.');
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0;
}

// ----------------------------------------------------------------------

void emptySerial() {
  // empty the serial input buffer
  keyVal = 0;
  while (keyVal != -1) {
    keyVal = Serial.read();
  } cmdRec = 0; // clear the recevied 
}

// ----------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = (cmdVal * 10) + zVal;  
}

// ----------------------------------------------------------------------

void readKey() {
  // reads a key from the keyboard annd reacts accordingly
  keyVal = Serial.read();
  if (keyVal != -1) {
    keyChar = char(keyVal); cmdRec = 1; // value received
//      Serial.print("keyVal=");
//      Serial.print(keyChar);
//      Serial.print("   ASCII=");
//      Serial.println(keyVal);
    int zNF = 0;
    switch (keyChar) {
      case '.': doCmd(); zNF = 1; break;
      case '!': runPOST(); zNF = 1; break;
      case '#': cmdMode = ' '; cmdType = ' '; cmdVal = 0; zNF = 1; break;
      case '~': zNF = 1; Serial.println("~"); break; // null tick received so respond
      case '0': extendCmdVal(0); zNF = 1;break;
      case '1': extendCmdVal(1); zNF = 1;break;
      case '2': extendCmdVal(2); zNF = 1;break;
      case '3': extendCmdVal(3); zNF = 1;break;
      case '4': extendCmdVal(4); zNF = 1;break;
      case '5': extendCmdVal(5); zNF = 1;break;
      case '6': extendCmdVal(6); zNF = 1;break;
      case '7': extendCmdVal(7); zNF = 1;break;
      case '8': extendCmdVal(8); zNF = 1;break;
      case '9': extendCmdVal(9); zNF = 1;break;
    }
    if (zNF == 0) {
      if (cmdMode == ' ') {
        // test for new Command Mode char?
        switch (keyChar) {
          case 'e': cmdMode = 'E'; break;
          case 'E': cmdMode = 'E'; break;
          case 'h': cmdMode = 'H'; break;
          case 'H': cmdMode = 'H'; break;
          case 'k': cmdMode = 'K'; break;
          case 'K': cmdMode = 'K'; break;
          case 'm': cmdMode = 'M'; break;
          case 'M': cmdMode = 'M'; break;
          case 'r': cmdMode = 'R'; break;
          case 'R': cmdMode = 'R'; break;
          case 's': cmdMode = 'S'; break;
          case 'S': cmdMode = 'S'; break;
        } cmdType = ' '; cmdVal = 0;
      } else {
        // test for Command Type char?
        switch (keyChar) {
          case '<': cmdType = '<'; break;
          case '>': cmdType = '>'; break;
          case '^': cmdType = '^'; break;
          case 'A': cmdType = 'A'; break;
          case 'd': cmdType = 'D'; break;
          case 'D': cmdType = 'D'; break;
          case 'e': cmdType = 'E'; break;
          case 'E': cmdType = 'E'; break;
          case 'g': cmdType = 'G'; break;
          case 'G': cmdType = 'G'; break;
          case 'l': cmdType = 'L'; break;
          case 'L': cmdType = 'L'; break;
          case 'm': cmdType = 'M'; break;
          case 'M': cmdType = 'M'; break;
          case 'n': cmdType = 'N'; break;
          case 'N': cmdType = 'N'; break;
          case 'o': cmdType = 'O'; break;
          case 'O': cmdType = 'O'; break;
          case 'p': cmdType = 'P'; break;
          case 'P': cmdType = 'P'; break;
          case 's': cmdType = 'S'; break;
          case 'S': cmdType = 'S'; break;
          case 't': cmdType = 'T'; break;
          case 'T': cmdType = 'T'; break;
          case 'u': cmdType = 'U'; break;
          case 'U': cmdType = 'U'; break;
          case 'v': cmdType = 'V'; break;
          case 'V': cmdType = 'V'; break;
          case 'w': cmdType = 'W'; break;
          case 'W': cmdType = 'W'; break;
          case 'x': cmdType = 'X'; break;
          case 'X': cmdType = 'X'; break;
          case 'z': cmdType = 'Z'; break;
          case 'Z': cmdType = 'Z'; break;
        }
      }
    }
  }
}

// ----------------------------------------------------------------------

void RESET() {
  // perform a soft RESET
  servoMain.detach();
  detachServos();
  setDefaults();
}

// ----------------------------------------------------------------------

void runPOST() {
  // called during start-up
  Serial.print(F("\n\n\n\n\n\n\n\n\n\n"));
  Serial.println(F("Reach Robot v0.11"));
  Serial.println(F("Starting POST..."));
  RESET();
  emptySerial();
  Serial.println(F("Centring Servos..."));
  attachServos(100); delay(200); detachServos();
  Serial.println(F("Pause 2s before RESET test..."));
  digitalWrite(LEDPin, HIGH); delay(1000);
  testCtrlApp(); // see if connected to Windows app?
  digitalWrite(LEDPin, LOW);
  if ((cmdRec < 1) && (Once >0) && Calibrated) {
    // Power-On reset or no serial link so enter demo modes
    // but only if the robot has been calibrated
    // read the MasterMode value then reset it to zero
    testResetState(); // read EEPROM and count RESETs
    MasterMode = EEPROM.read(3); EEPROM.update(3, 0);
  } else {
    // on serial link so force into mode 1
    MasterMode = 1;
  }
  Serial.print(F("Demo mode ")); Serial.print(MasterMode);
  Serial.println(F(" initiated!"));
  switch (MasterMode) {
    case 1:
      // single RESET so do nothing
      if (cmdRec < 1) {
        // no serial link
        Serial.println(F("Sleeping...zzz"));
        servoMain.detach(); detachServos();
      } else {
        // serial link so send initialisation data
        Serial.print(F("SN0LL")); Serial.println(servoMin[0]);
        Serial.print(F("SN0UL")); Serial.println(servoMax[0]);
        Serial.print(F("SN1LL")); Serial.println(servoMin[1]);
        Serial.print(F("SN1UL")); Serial.println(servoMax[1]);
        Serial.print(F("SN2LL")); Serial.println(servoMin[2]);
        Serial.print(F("SN2UL")); Serial.println(servoMax[2]);
        Serial.print(F("SN3LL")); Serial.println(servoMin[3]);
        Serial.print(F("SN3UL")); Serial.println(servoMax[3]);
        reportEachValue();
        reportHomeValues();
      } break;
     case 2:
      // start demo mode 1
//      moveToTest00 (); moveStart();
      break;
  }
  Serial.println(F("POST complete!"));
  Once = 0;
}

