// ################################################################################
//
//  Quadruped Release v0.1
//
//  Released:  30/07/2018
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls a Quadruped critter robot, driven by 8 servo motors, using
    two main functions, one for walking and one that moves servo angles towards a
    set of target values. The rate at which these functions run can be varied which
    means that the speed of the robot can also be vaied simply by changing the rate.

    The code also reads codes from an infrared receiver, distance measurements from
    an ultrasonic sensor, and it also responds to a button press and messages it 
    receives over the USB serial port. As this robot is intended for use as a 
    classroom demonstrator, it naturally resets to a safe state and only responds
    to user demandsd once initiated. There are comments throughout the code to help
    explain this.

    Keyboaard commands:
    . - print servo values

*/
// Declare and initialise global variables
#include <Servo.h>
#include "IRremote.h"

// Define servo constants. These will be determined by you from servo calibration.
#define Ang1_45 1117  // this is the 45 degree value for servo 1
#define Ang1_135 2035 // this is the 135 degree value for servo 1
#define Ang2_65 2164 // this is the 65 degree value for servo 2
#define Ang2_153 1189 // this is the 153 degree value for servo 2
#define Ang3_45 1887  // this is the 45 degree value for servo 3
#define Ang3_135 950 // this is the 135 degree value for servo 3
#define Ang4_65 715 // this is the 65 degree value for servo 4
#define Ang4_153 1624 // this is the 153 degree value for servo 4
#define Ang5_45 1179  // this is the 45 degree value for servo 5
#define Ang5_135 2107 // this is the 135 degree value for servo 5
#define Ang6_65 2212 // this is the 65 degree value for servo 6
#define Ang6_153 1313 // this is the 153 degree value for servo 6
#define Ang7_45 2002  // this is the 45 degree value for servo 7
#define Ang7_135 1040 // this is the 135 degree value for servo 7
#define Ang8_65 801 // this is the 65 degree value for servo 8
#define Ang8_153 1676 // this is the 153 degree value for servo 8
#define BattMin 804 // minimum A0 reading to trigger a battery LOW event
#define LegD 110 // angle for leg down in 'Move Engine' Walk
#define LegU 130 // angle for leg raised in 'Move Engine' Walk
#define Max1 2174 // max value for servo 1
#define Max2 2164 // max value for servo 2
#define Max3 2026 // max value for servo 3
#define Max4 2317 // max value for servo 4
#define Max5 2265 // max value for servo 5
#define Max6 2212 // max value for servo 6
#define Max7 2116 // max value for servo 7
#define Max8 2298 // max value for servo 8
#define Min1 1007 // min value for servo 1
#define Min2 682 // min value for servo 2
#define Min3 835 // min value for servo 3
#define Min4 715 // min value for servo 4
#define Min5 1150 // min value for servo 5
#define Min6 686 // min value for servo 6
#define Min7 911 // min value for servo 7
#define Min8 801 // min value for servo 8
#define Rst1 1423 // reset value for servo 1
#define Rst2 2164 // reset value for servo 2
#define Rst3 1575 // reset value for servo 3
#define Rst4 715 // reset value for servo 4
#define Rst5 1488 // reset value for servo 5
#define Rst6 2212 // reset value for servo 6
#define Rst7 1685 // reset value for servo 7
#define Rst8 801 // reset value for servo 8

// Define general constants, that are not servo related
#define echoMax 2500 // maximum wait for echo signal
#define echoPin 8 // ultrasonic echo pin
#define IRPin A1 // IR receiver INPUT pin, also centre LED OUTPUT pin
#define LEDCtr A1 // centre LED output pin
#define LEDLft 3 // left LED output pin
#define LEDPin 13 // onboard LED output pin
#define LEDRht 2 // right LED output pin
#define SwPin 10 // button switch pin
#define trigPin 9 // ultrasonic trigger pin

// define servo arrays. See calibration diagrams for servo number assignments
//                 01,02,03,04,05,06,07,08 
int servoMem[8]; // record of values sent to servos, used in fidgeting
int servoPins[] = { 5, 4,A4,A5,A3,A2, 6, 7};  // servo output pins assigned
int servoVal[8]; // value sent to servos
int servoTgt[8]; // target values for servos

// Declare and initialise global variables
int anyDel; // any delay value
int anyFor; // any for loop value
unsigned long anyMilli; // any millisecond timer
int anyRnd; // any random integer
int anyVal;  // any temp variable
boolean atTgt; // true if servos are at target values
int autoOFF; // if set > 0 then counts down to servo switch OFF, if -1 disabled
int BattAv; // average battery voltage
int BattCnt; // battery failed 1s time-out counter
int BattSum; // cumulative battery voltage
int BattVol; // instantaneous battery voltage
char cmdMode; // command mode
char cmdType; // command type
int cmdVal; // value associated with a cmdType
unsigned long distance; // calculated distance value from echo
unsigned long duration; // duration of echo in microseconds
int echoTO; // counter providing squelsh function
boolean ESC; // true to escape all functions
int HandCnt; // counter used in hand detection
boolean HandEn; // true if hand detection is needed; default = false
int HandTask; // used to sequence hand detection
unsigned long interval; // main loop interval in microseconds
int IrCode; // valid code received from IR receiver, otherwise -1
int IrCodeLast; // previous valid code received from IR receiver, otherwise -1
boolean IrEn; // IR enable, default false, mode 1 = true
boolean IrEnM; // IR enable Master state, default false, mode 1 = true
int IrPWR; // timer used to enable IR modes
int IrRxTimer; // counter used to detect end of received IR codes
char keyChar; // any keyboard character
int keyVal; // any keyboard value
int LastTask; // record of the previous MainTask
boolean  LEDLftSt; // HIGH/LOWE state of LED pin
int LedCnt; // counter used in LED sequencing
boolean  LEDCtrSt; // HIGH/LOWE state of LED pin
int LedMode; // a value > 0 determines LED flash sequence; if = 0 disabled
boolean  LEDRhtSt; // HIGH/LOWE state of LED pin
int LegDn; // dynamic angle for leg down in 'Move Engine'
int LegUp; // dynamic angle for raised down in 'Move Engine'
boolean MainRun; // set true when a MainTask is running to prevent re-entry
int MainTask; // main task pointer
int MainTaskNext; // the task to run after an arrest; default is 0
unsigned long nextMicros; // main loop interval timer in microseconds
int Once; // counter for demo moves
boolean QuadActive; // true if active, otherwise false
int RadarCnt; // used to set the frequency of radar triggering
int receiver = IRPin; // create an active IR receiver pin
int servoDelta; // temporary delta value used in target movements
boolean servoEn; // set true if servos are attached
int SubTask; // sub task pointer
int SubVal; // anyvalue used in a subroutine
int swCnt; // button switch counter
boolean swLastState; // previous state of button switch, HIGH/LOW
boolean swState; // state of read button switch pin
int swTimer; // timer used to detemine button sequences
int tgtCnt; // counter used in automatic servo move to target actions
boolean Turn; // default = false; if true turns continuously until = false
int Walk; // if >0 then walk in one of four directions
int WalkCnt; // a counter which controls the walking process
long Walked; // counter used to return robot to a given position
unsigned long walkInterval; // main loop walking interval in microseconds
unsigned long walkMicros; // main loop walking task interval timer in microseconds

// Declare library related objects
Servo servoInst[8]; // define Quadruped servos
IRrecv irrecv(receiver);  // create instance of 'irrecv'
decode_results results;   // create instance of 'decode_results'

void setup() {
  // runs once after reset to initialise variables, port pins and timers
  setDefaults(); // assign default values to all variables
  pinMode(echoPin, INPUT);
  pinMode(LEDCtr, OUTPUT); digitalWrite(LEDCtr, HIGH); // LED OFF initially
  pinMode(LEDLft, OUTPUT); digitalWrite(LEDLft, HIGH); // LED OFF initially
  pinMode(LEDPin, OUTPUT);
  pinMode(LEDRht, OUTPUT); digitalWrite(LEDRht, HIGH); // LED OFF initially
  pinMode(SwPin, INPUT_PULLUP); // button switch has pullup resistor
  pinMode(trigPin, OUTPUT);
  Serial.begin(57600); // high baud rate for sweeping functions
  runPOST();
  irrecv.enableIRIn(); // Start the receiver
  synchLoopTimers(); // reset loop timers
}

void loop() {
  // this main loop is called by the Arduino kernal, but also from functions
  // it has two H/W timers, one for servo movement and one for task scheduling
  if (micros() >= walkMicros) {
    // walking uses a separate timer so that the speed of walking can be controlled
    // before walking is enabled ensure that start conditions are set
    // walk movements are based on an 88 step counter with 1 degree accuracy
    walkMicros = walkMicros + walkInterval; // set next time pointer
    if (Walk > 0) {
      // to stop walking we set Walk = 0; values 1 - 4 result in servo movement
      if (autoOFF >= 0) {autoOFF = 5;} // set servo detach auto-OFF timeout
      WalkCnt++; if (WalkCnt > 87) {WalkCnt = 0;} // increment and restart
      switch (Walk) {
        case 1: // walking forwards
          // front left leg & rear right leg
          Walked++; // increase how far walked counter
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt < 46) {SetAngle(0,90+WalkCnt); SetAngle(4,135-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(0,177-WalkCnt); SetAngle(4,48+WalkCnt);}
          // front right leg & rear left leg
          if (WalkCnt > 43) {SetAngle(2,46+WalkCnt); SetAngle(3,LegDn); SetAngle(6,179-WalkCnt); SetAngle(7,LegDn);}
          else if (WalkCnt > 2) {SetAngle(2,133-WalkCnt); SetAngle(6,92+WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,134+WalkCnt); SetAngle(3,LegDn); SetAngle(6,91-WalkCnt); SetAngle(7,LegDn);}
          break;
        case 2: // walking towards the right
          // front right leg & rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle(2,90-WalkCnt); SetAngle(6,30+WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,3+WalkCnt); SetAngle(6,132-WalkCnt);}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(4,134-WalkCnt); SetAngle(5,LegDn); SetAngle(0,1+WalkCnt); SetAngle(1,LegDn);}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(4,47+WalkCnt); SetAngle(0,88-WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(4,46-WalkCnt); SetAngle(5,LegDn); SetAngle(0,89+WalkCnt); SetAngle(1,LegDn);}
          break;
        case 3: // walking backwards
          // front left leg & rear right leg
          Walked--; // decrease how far walked counter
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt < 46) {SetAngle(4,90+WalkCnt); SetAngle(0,135-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(4,177-WalkCnt); SetAngle(0,48+WalkCnt);}
          // front right leg & rear left leg
          if (WalkCnt > 43) {SetAngle(6,46+WalkCnt); SetAngle(3,LegDn); SetAngle(2,179-WalkCnt); SetAngle(7,LegDn);}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(6,133-WalkCnt); SetAngle(2,92+WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(6,134+WalkCnt); SetAngle(3,LegDn); SetAngle(2,91-WalkCnt); SetAngle(7,LegDn);}
          break;
        case 4: // walking towards the leftt
          // front right leg & rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle(2,45+WalkCnt); SetAngle(6,90-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,132-WalkCnt); SetAngle(6,3+WalkCnt);}
          // front left leg & rear right leg
          if (WalkCnt > 43) {SetAngle(0,134-WalkCnt); SetAngle(1,LegDn); SetAngle(4,1+WalkCnt); SetAngle(5,LegDn);}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(0,47+WalkCnt); SetAngle(4,88-WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(0,46-WalkCnt); SetAngle(1,LegDn); SetAngle(4,89+WalkCnt); SetAngle(5,LegDn);}
          break;
      }
    } else if (tgtCnt > 0) {
      // if not walking we can move servos towards target values
      // normally occures every 20ms, but walkInterval can be reduced to speed it up
      // now move servos if a target move has been set
      if (autoOFF >= 0) {autoOFF = 5;} // set servo detach auto-OFF timeout
      if (tgtCnt == 1) {
        // move values immediately to the target value, if you want quickest moves
        for (int zP = 0; zP < 8;zP++) {
          servoVal[zP] = servoTgt[zP]; // set values to target values
          servoInst[zP].writeMicroseconds(servoVal[zP]);
        }
      } else {
        // move values progressively towards the target values
        // servo values will be moved on a 2 to N step basis set by tgtCnt
        // hence if tgtCnt is large, say 50, this will take 1 second to complete
        atTgt = true; // set a test flag for achieving target values
        for (int zP = 0; zP < 8;zP++) {
          servoDelta = servoTgt[zP] - servoVal[zP];
          if (abs(servoDelta) > 0) {
            servoVal[zP] = servoVal[zP] + (servoDelta / tgtCnt);
            servoInst[zP].writeMicroseconds(servoVal[zP]); atTgt = false;
          }
        } if (atTgt) {tgtCnt = 0;} // already at target values so clear the move count
      } tgtCnt--; // progressivley reduce the target factor
    }
  }

  if (micros() >= nextMicros) {
    // do these tasks every 20ms if interval = 20000
    // this timer is used for all non-servo tasks
    // as loop() is called from functions to keep the servos moving, flags are used
    // to prevent stack overflow due to re-entrant code nesting
    nextMicros = nextMicros + interval; // set next time point

    // perform hand detection task, normall used in MainTask = 2
    if (HandEn) {detectHand();}
    
    // perform 'Move Engine' tasks
    // once entered these tasks will sustain unless interrupted by IR or button
    switch (MainTask) {
      case -1: // clearing active tasks, we ant to interrupt the current task
        SubTask--; if (SubTask < 1) {
          flashLEDs(MainTaskNext); SetMainTask(MainTaskNext); synchLoopTimers();
        } break;
      case 0: break; // default null task
      case 1: // goto QuadActive default standing position, then to MainTask = 0
        GoToStand(50); MainTask = 0; break;
      case 2: // on the spot exercises, responding to close hand
        MainTask_OnTheSpot(); break;
      case 3: // backaway from target mode
        MainTask_BackAway(); break;
      case 4: // follow wall target mode
        MainTask_TrackWall(); break;
      case 98: // do a test move if coded?
        MainTask_Test(50); SetMainTask(0); break;
      case 99: // move to standing position
        MainTask_StandAndStretch(50); MainTask = 0; break;
      default:
        break;
    }

    // if not moving servos perform an auto-detach process after a time-out
    if ((Walk < 1) && (tgtCnt < 1)) {
      // return to servo Power-OFF state?
      if (autoOFF > 0) {autoOFF--; if (autoOFF == 0) {detachServos();}}
    }

    // perform these tasks only when active, ie. QuadActive == true
    // for safety reasons the robot sensors are deactivated by default
    if (QuadActive) {
      // check radar
      RadarCnt--; if (RadarCnt < 1) {RadarCnt = 3; trigRadar();}
      
      // check infrared?
      if (IrEn) {
        if (irrecv.decode(&results)) {
          // IR code received from TL1838 sensor.
          // Codes are received every 100-120ms from the remote control.
          // Random values may be received in certain lighting conditions, or when
          // servos are attached and moving.
          IrEn = false;
          translateIR_Samsung_0111(results.value);
          irrecv.resume(); // prepare to receive the next value BEFORE decoding
          decodeIrFunction(); IrRxTimer = 10; // timer detects IR button released
        } else {
          // no IR code received in this cycle
          if (IrRxTimer > 0) {
            IrRxTimer--; if (IrRxTimer < 1) {
              // no more valid codes received
              Serial.println(F("IR Up!"));
              if ((Walk > 0) && (MainTask == 0)) {
                // stop a walking function, assuming it may be temporary
                Walk = -abs(Walk); LedMode = 0; MemStore(); // store current values
                GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,1); // put all feet down immediately
                loopWhiletgtCnt(); walkInterval = 20000; // restart at normal speed
              }
              if (Turn) {Turn = false;}
              IrCode = -1; IrCodeLast = -1;
            }
          }
        } IrEn = true;
      }
    }
    
    // Always do the following every 20ms...

    // Check battery voltage averaged over 16 cycles.
    // This avoids transients and brown-outs tripping out the robot.
    BattVol = analogRead(A0);
    BattAv = BattSum >> 4; BattSum = BattSum + BattVol - BattAv;
    if (BattAv <= BattMin) {
      // Battery avreage has reached minimum set level.
      // A further one second timeout counter avoids brownouts and glitches.
      BattCnt++; if (BattCnt > 50) {BatteryFailed();}
    } else {BattCnt = 0;} // reset battery LOW timeout
//    Serial.print(F("0,")); Serial.print(BattAv); Serial.println(F(",816"));
    
    // Read the button switch; used for setting MainTasks and power-up
    readSwitch();
    
    // Read keyboard input, useful in debugging to demand servo values
    readKey();
    
    // Update LEDs and auto-clear ON flags.
    // Left LED is driven LOW by the button switch state, normally HIGH
    if (LedMode > 0) {
      // Setting LED modes provides light sequences during movement
      switch (LedMode) {
        case 1: // LED move centre to out
          if (LedCnt <= 8) {LEDCtrSt = LOW;} else if (LedCnt <= 16) {LEDLftSt = LOW; LEDCtrSt = LOW; LEDRhtSt = LOW;}
          else if (LedCnt <= 32) {LEDLftSt = LOW; LEDRhtSt = LOW;} else if (LedCnt >= 40) {LedCnt = 0;} break;
        case 2: // LED move left to right
          if (LedCnt <= 8) {LEDLftSt = LOW;} else if (LedCnt <= 16) {LEDCtrSt = LOW;}
          else if (LedCnt <= 24) {LEDRhtSt = LOW;} else if (LedCnt >= 32) {LedCnt = 0;} break;
        case 3: // LED move out to centre
          if (LedCnt <= 8) {LEDLftSt = LOW; LEDRhtSt = LOW;} else if (LedCnt <= 16) {LEDLftSt = LOW; LEDCtrSt = LOW; LEDRhtSt = LOW;}
          else if (LedCnt <= 32) {LEDCtrSt = LOW;} else if (LedCnt >= 40) {LedCnt = 0;} break;
        case 4: // LED move right to left
          if (LedCnt <= 8) {LEDRhtSt = LOW;} else if (LedCnt <= 16) {LEDCtrSt = LOW;}
          else if (LedCnt <= 24) {LEDLftSt = LOW;} else if (LedCnt >= 32) {LedCnt = 0;} break;
        case 5: // IR PWR state achieved, so light RH and LF LEDs
          if (LedCnt < 50) {LEDLftSt = LOW; LEDRhtSt = LOW;} else {LedMode = 0;}
      } LedCnt++;
    }
    // Note that the centre LED is also wired to the infrared receiver.
    // We can pull it LOW to turn it ON, but should never drive it HIGH as we may
    // damage the IR receiver device.
    digitalWrite(LEDLft, LEDLftSt); LEDLftSt = HIGH;
    if ((!IrEn) && (LEDCtrSt == LOW)) {pinMode(LEDCtr, OUTPUT); digitalWrite(LEDCtr, LEDCtrSt); LEDCtrSt = HIGH;}
    else {pinMode(LEDCtr, INPUT_PULLUP);}
    digitalWrite(LEDRht, LEDRhtSt); LEDRhtSt = HIGH;
  }
}

// ----------------------------------------------------------------------

void delayLoop(unsigned long zDel) {
  // Loops() called from any function for a delay period of time.
  zDel = millis() + zDel;
  while (millis() < zDel) {
    if (ESC) return;
    loop();
  }  
}

// ----------------------------------------------------------------------

void loopWhiletgtCnt() {
  // Used to call loop() during a 'Move' function, keeps servos moving.
//  Serial.println(F("loopWhiletgtCnt"));
  while (tgtCnt > 0) {
    if (ESC) return;
    loop();
  }
}

// ----------------------------------------------------------------------

void loopWhileWalk(int zWalkCnt) {
  // Walk for a number of WalkCnt clicks, there are 88 in a walk cycle.
//  Serial.println(F("loopWhileWalk"));
  int zWCL;
  while (zWalkCnt > 0) {
    if (ESC) return;
    zWCL = WalkCnt; loop();
    if (zWCL != WalkCnt) {zWalkCnt--;}  
  }
}

// ----------------------------------------------------------------------

void runPOST() {
  // Called during startup().
  // Signs onto the serial monitor, flashes the LEDs and moves servos to rest.
  Serial.println(F("\n\nQuadruped v0.16b"));
  Serial.println(F("Starting POST..."));
  Serial.println(F("Testing LEDs..."));
  digitalWrite(LEDLft, LOW); delay(200); digitalWrite(LEDLft, HIGH); LEDLftSt= HIGH;
  digitalWrite(LEDCtr, LOW); delay(200); pinMode(LEDCtr, INPUT_PULLUP); LEDCtrSt= HIGH;
  digitalWrite(LEDRht, LOW); delay(200); digitalWrite(LEDRht, HIGH); LEDRhtSt= HIGH;
  // empty Rx buffer
  keyVal = 0; while (keyVal != -1) {readKey();}
  Serial.println(F("Reseting Servos..."));
  GoToRest(50); autoOFF = 5;
  Serial.println(F("POST complete!"));
}

// ----------------------------------------------------------------------

void setDefaults() {
  // Load default values, not set in their declaration.
  // So we can call this at any time to set default values.
  atTgt = false; // true if servos are at target values
  autoOFF = 0; // if set > 0 then counts down to servo switch off
  BattCnt = 0; // battery failed 1s time-out counter
  BattSum = BattMin << 4; //  pre-set cumulative battery voltage
  echoTO = 0; // counter providing squelsh function
  ESC = false; // true to escape all functions
  HandCnt = 0; // counter used in hand detection
  HandEn = false; // true if hand detection is needed; default = false
  HandTask = 0; // used to sequence hand detection
  interval = 20000; // main loop interval in microseconds, 50Hz
  IrEn = false; // IR enable, default false, mode 1 = true
  IrEnM = false; // IR enable Master state, default false, mode 1 = true
  IrCodeLast = -1; // previous valid code received from IR receiver, otherwise -1
  IrPWR = 0; // timer used to enable IR modes
  IrRxTimer = 0; // counter used to detect end of received IR codes
  LedCnt = 0; // counter used in LED sequencing
  LedMode = 0; // a value > 0 determines LED flash sequence; if = 0 disabled
  LegDn = LegD; // dynamic angle for leg down in 'Move Engine'
  LegUp = LegU; // dynamic angle for raised down in 'Move Engine'
  MainRun = false; // set true when a MainTask is running to prevent re-entry
  MainTask = 0; // main task pointer
  MainTaskNext = 0; // the task to run after an arrest; default is 0
  Once = 0; // counter for demo moves
  QuadActive = false; // true if active, otherwise false
  RadarCnt = 0; // used to set the frequency of radar triggering
  servoEn = false; // set true if servos are attached
  servoVal[0] = Rst1; // value sent to servos
  servoVal[1] = Rst2; // value sent to servos
  servoVal[2] = Rst3; // value sent to servos
  servoVal[3] = Rst4; // value sent to servos
  servoVal[4] = Rst5; // value sent to servos
  servoVal[5] = Rst6; // value sent to servos
  servoVal[6] = Rst7; // value sent to servos
  servoVal[7] = Rst8; // value sent to servos
  servoTgt[0] = Rst1; // target values for servos
  servoTgt[1] = Rst2; // target values for servos
  servoTgt[2] = Rst3; // target values for servos
  servoTgt[3] = Rst4; // target values for servos
  servoTgt[4] = Rst5; // target values for servos
  servoTgt[5] = Rst6; // target values for servos
  servoTgt[6] = Rst7; // target values for servos
  servoTgt[7] = Rst8; // target values for servos
  SubTask = 0; // sub task pointer
  swCnt = 0; // button switch counter
  swLastState = HIGH; // previous state of button switch, HIGH/LOW
  swState = HIGH; // stat of read button switch pin
  swTimer = 0; // timer used to detemine button sequences
  tgtCnt = 0; // counter used in automatic servo move to target actions
  Turn = false; // default = false; if true turns continuously until = false
  Walk = 0; // if >0 then walk in one of four directions
  Walked = 0; // counter used to return robot to a given position
  walkInterval = 20000; // main loop walking interval in microseconds
}

// ----------------------------------------------------------------------

void synchLoopTimers() {
  // Reset main loop timers to the H/W micros() timer.
  // Needed if you ever use long delay() code, otherwise servo movements will be
  // accelerated by the time-lapse that has occured.
  nextMicros = micros() + interval; // set loop timer
  walkMicros = micros() + walkInterval; // set loop timer
}

