// ################################################################################
//
//  Quadruped RC Release v0.1
//
//  Released:  24/08/2018
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls a Quadruped critter robot, driven by 8 servo motors, and 
    controlled via a Wii Nunchuk controller over a serial 2.4GHz wireless interface.
    After RESET the robot must be brought into the QuadActive state by pressing and
    holding the Nunchuk 'C' button, before any movements can be controlled.

    The servo motors must be calibrated during construction, and the values taken
    need to be inserted in the angle definitions included in this code. Functions
    that are commented-out, are not needed but you may find useful to use in your
    code. Enjoy!

*/
// Declare and initialise global variables
#include <Servo.h>

// Define servo constants
#define Ang1_45 1021
#define Ang1_135 1944
#define Ang2_65 2026
#define Ang2_153 1079
#define Ang3_45 1944
#define Ang3_135 983
#define Ang4_65 921
#define Ang4_153 1815
#define Ang5_45 1012
#define Ang5_135 2002
#define Ang6_65 2140
#define Ang6_153 1246
#define Ang7_45 1973
#define Ang7_135 1031
#define Ang8_65 849
#define Ang8_153 1724
#define BattMin 804 // minimum A0 reading to trigger a battery LOW event
#define LegD 110 // angle for leg down in 'Move Engine' Walk
#define LegU 130 // angle for leg raised in 'Move Engine' Walk
#define Max1 2088 // max value for servo 1
#define Max2 2026 // max value for servo 2
#define Max3 2050 // max value for servo 3
#define Max4 2298 // max value for servo 4
#define Max5 2136 // max value for servo 5
#define Max6 2140 // max value for servo 6
#define Max7 2045 // max value for servo 7
#define Max8 2308 // max value for servo 8
#define Min1 930 // min value for servo 1
#define Min2 715 // min value for servo 2
#define Min3 859 // min value for servo 3
#define Min4 921 // min value for servo 4
#define Min5 945 // min value for servo 5
#define Min6 686 // min value for servo 6
#define Min7 878 // min value for servo 7
#define Min8 849 // min value for servo 8
#define Rst1 1423 // reset value for servo 1
#define Rst2 2164 // reset value for servo 2
#define Rst3 1575 // reset value for servo 3
#define Rst4 715 // reset value for servo 4
#define Rst5 1488 // reset value for servo 5
#define Rst6 2212 // reset value for servo 6
#define Rst7 1685 // reset value for servo 7
#define Rst8 801 // reset value for servo 8

// Define general constants
#define LED0 8 // leftmost LED output pin
#define LED1 9 // left LED output pin
#define LED2 10 // centre LED output pin
#define LED3 11 // right LED output pin
#define LED4 12 // rightmost LED output pin

// define servo arrays
//                 01,02,03,04,05,06,07,08 
int servoMem[8]; // record of values sent to servos, used in fidgeting
int servoPins[] = { 5, 4,A4,A5,A3,A2, 6, 7};  // servo output pins assigned
int servoVal[8]; // value sent to servos
int servoTgt[8]; // target values for servos

// Declare and initialise global variables
int anyDel; // any delay value
int anyFor; // any for loop value
unsigned long anyMilli; // any millisecond timer
int anyRnd; // any random integer
int anyVal;  // any temp variable
boolean atTgt; // true if servos are at target values
int autoOFF; // if set > 0 then counts down to servo switch OFF, if -1 disabled
int autoRest; // returns to a rest position after a timeout
int BattAv; // average battery voltage
int BattCnt; // battery failed 1s time-out counter
int BattSum; // cumulative battery voltage
int BattVol; // instantaneous battery voltage
int C_Cnt; // counter used in 'C' button detection
byte C_Dn; // counter for 'C' button down period
byte C_Up; // counter for 'C' button up period
byte checksum; // XOR Rx data to detect corruption at edge of range
char cmdMode; // command mode
char cmdType; // command type
int cmdVal; // value associated with a cmdType
int CZ = 0; // received button values
boolean CZ_Wait; // flag used for button release wait function
boolean ESC; // true to escape all functions
unsigned long interval; // main loop interval in microseconds
boolean JoyActive; // flag used to prevent stack overflow during moves
int JoyMode; // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
int JoySpd; // speed vector = max(JoyX,JoyY)
int JoyX; // received value
int JoyXV; // converted value 0 - 127
int JoyY; // received value
int JoyYi; // tracking filter for received JoyY value
int JoyYV; // converted value 0 - 128
boolean JoyYT; // tracking filter flag; true if not on track
int LastTask; // record of the previous MainTask
boolean  LED0St; // HIGH/LOWE state of LED pin
boolean  LED1St; // HIGH/LOWE state of LED pin
boolean  LED2St; // HIGH/LOWE state of LED pin
boolean  LED3St; // HIGH/LOWE state of LED pin
boolean  LED4St; // HIGH/LOWE state of LED pin
int LedCnt; // counter used in LED sequencing
int LedMode; // a value > 0 determines LED flash sequence; if = 0 disabled
boolean LedTick; // set true every 'Walk' cycle
int LegDn; // dynamic angle for leg down in 'Move Engine'
int LegUp; // dynamic angle for raised down in 'Move Engine'
boolean MainRun; // set true when a MainTask is running to prevent re-entry
int MainTask; // main task pointer
int MainTaskNext; // the task to run after an arrest; default is 0
unsigned long nextMicros; // main loop interval timer in microseconds
int Once; // counter for demo moves
boolean QuadActive; // true if active, otherwise false
boolean RxRec = false ; // set true when a valid frame of data is received
int RxState = 0; // receiver state machine state
int RxVal = -1; // value received from serial Rx
int servoDelta; // temporary delta value used in target movements
boolean servoEn; // set true if servos are attached
int SubTask; // sub task pointer
int SubVal; // anyvalue used in a subroutine
int tgtCnt; // counter used in automatic servo move to target actions
boolean Turn; // default = false; if true turns continuously until = false
int Walk; // if >0 then walk in one of four directions
long WalkCnt; // a counter which controls the walking process
unsigned long walkInterval; // main loop walking interval in microseconds
int WalkLast; // previousl Walk value if any
long WalkLftDrive; // drive factor for left-hand side, 128 = max, 0 = min
byte WalkMax; // speed factor 4 - 12
unsigned long walkMicros; // main loop walking task interval timer in microseconds
int WalkSpeed; // value used to determine WalkMax
long WalkRgtDrive; // drive factor for right-hand side, 128 = max, 0 = min
int Z_Cnt; // counter used in 'Z' button detection
byte Z_Dn; // counter for 'Z' button down period
byte Z_Up; // counter for 'Z' button up period

// Declare objects
Servo servoInst[8]; // define Quadruped servos

void setup() {
  // put your setup code here, to run once:
  setDefaults();
  pinMode(LED0, OUTPUT); digitalWrite(LED0, HIGH); // LED0 OFF initially
  pinMode(LED1, OUTPUT); digitalWrite(LED1, HIGH); // LED1 OFF initially
  pinMode(LED2, OUTPUT); digitalWrite(LED2, HIGH); // LED2 OFF initially
  pinMode(LED3, OUTPUT); digitalWrite(LED3, HIGH); // LED3 OFF initially
  pinMode(LED4, OUTPUT); digitalWrite(LED4, HIGH); // LED4 OFF initially
  Serial.begin(9600); // baud rate for Rx/Tx comms
  runPOST();
  synchLoopTimers(); // reset loop timers
}

void loop() {
  // put your main code here, to run repeatedly:
  if (micros() >= walkMicros) {
    // walking uses a separate timer so that the speed of walking can be controlled
    // before walking is enabled ensure that start conditions are set
    walkMicros = walkMicros + walkInterval; // set next time point
    LedTick = true; // perform LED sequences at same rate as walking
    if (Walk > 0) {
      if (autoOFF >= 0) {autoOFF = 5;} // set auto-OFF timeout
      WalkCnt++; if (WalkCnt > 87) {WalkCnt = 0;} // increment and restart
      switch (Walk) {
        case 1: // walking forwards
          // front left leg &  rear right leg
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt < 46) {SetAngle(0,90+((WalkCnt * WalkLftDrive)/128));
            SetAngle(4,90+(((45-WalkCnt)*WalkRgtDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(0,90+(((87-WalkCnt)*WalkLftDrive)/128));
            SetAngle(4,90-(((42-WalkCnt)*WalkRgtDrive)/128));}
          // front right leg &  rear left leg
          if (WalkCnt > 43) {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,90+(((WalkCnt-44)*WalkRgtDrive)/128));
            SetAngle(6,90+(((89-WalkCnt)*WalkLftDrive)/128));}
          else if (WalkCnt > 2) {SetAngle(2,90+(((43-WalkCnt)*WalkRgtDrive)/128));
            SetAngle(6,90+(((WalkCnt+2)*WalkLftDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,90+(((WalkCnt+44)*WalkRgtDrive)/128));
            SetAngle(6,90+(((1-WalkCnt)*WalkLftDrive)/128));} break;
            
        case 2: // walking towards the right
          // front right leg &  rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle(2,90-((WalkCnt*WalkLftDrive)/128));
            SetAngle(6,90-(((45-WalkCnt)*WalkRgtDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,90-(((87-WalkCnt)*WalkLftDrive)/128));
            SetAngle(6,90-(((WalkCnt-42)*WalkRgtDrive)/128));}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,90+(((WalkCnt-88)*WalkLftDrive)/128));
            SetAngle(4,90-(((WalkCnt-44)*WalkRgtDrive)/128));}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(0,90-(((2+WalkCnt)*WalkLftDrive)/128));
            SetAngle(4,90-(((43-WalkCnt)*WalkRgtDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,90+(((WalkCnt-1)*WalkLftDrive)/128));
            SetAngle(4,90-(((WalkCnt+44)*WalkRgtDrive)/128));} break;
            
        case 3: // walking backwards
          // front left leg &  rear right leg
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt < 46) {SetAngle(0,90+(((45-WalkCnt)*WalkLftDrive)/128));
            SetAngle(4,90+((WalkCnt * WalkRgtDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(0,90-(((42-WalkCnt)*WalkLftDrive)/128));
            SetAngle(4,90+(((87-WalkCnt)*WalkRgtDrive)/128));}
          // front right leg &  rear left leg
          if (WalkCnt > 43) {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,90+(((89-WalkCnt)*WalkRgtDrive)/128));
            SetAngle(6,90+(((WalkCnt-44)*WalkLftDrive)/128));}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(2,90+(((WalkCnt+2)*WalkRgtDrive)/128));
            SetAngle(6,90+(((43-WalkCnt)*WalkLftDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,90+(((1-WalkCnt)*WalkRgtDrive)/128));
            SetAngle(6,90+(((WalkCnt+44)*WalkLftDrive)/128));} break;
            
        case 4: // walking towards the left
          // front right leg & rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle(2,90-(((45-WalkCnt)*WalkRgtDrive)/128));
            SetAngle(6,90-((WalkCnt*WalkLftDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,90-(((WalkCnt-42)*WalkRgtDrive)/128));
            SetAngle(6,90-(((87-WalkCnt)*WalkLftDrive)/128));}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,90-(((WalkCnt-44)*WalkRgtDrive)/128));
            SetAngle(4,90+(((WalkCnt-88)*WalkLftDrive)/128));}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(0,90-(((43-WalkCnt)*WalkRgtDrive)/128));
            SetAngle(4,90-(((2+WalkCnt)*WalkLftDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,90-(((WalkCnt+44)*WalkRgtDrive)/128));
            SetAngle(4,90+(((WalkCnt-1)*WalkLftDrive)/128));} break;
            
        case 5: // turning towards the left
          // front left leg & rear right leg
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt < 46) {SetAngle(0,90-WalkCnt); SetAngle(4,90-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(0,WalkCnt+3); SetAngle(4,WalkCnt+3);}
          // front right leg &  rear left leg
          if (WalkCnt > 43) {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,WalkCnt+1); SetAngle(6,WalkCnt+1);}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(2,88-WalkCnt); SetAngle(6,88-WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,WalkCnt+89); SetAngle(6,WalkCnt+89);} break;
            
        case 6: // turning towards the right
          // front right leg & rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle(2,90-WalkCnt); SetAngle(6,90-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,WalkCnt+3); SetAngle(6,WalkCnt+3);}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,WalkCnt+1); SetAngle(4,WalkCnt+1);}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(0,88-WalkCnt); SetAngle(4,88-WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,WalkCnt+89); SetAngle(4,WalkCnt+89);} break;
      }
    } else if (tgtCnt > 0) {
      // if not walking see if we need to move the servos?
      if (autoOFF >= 0) {autoOFF = 5;} // set auto-OFF timeout
      if (tgtCnt == 1) {
        // move values immediately to the target value
        for (int zP = 0; zP < 8;zP++) {
          servoVal[zP] = servoTgt[zP]; // set values to target values
          servoInst[zP].writeMicroseconds(servoVal[zP]);
        }
      } else {
        // move values progressively towards the target values
        atTgt = true; // set a test flag for achieving target values
        for (int zP = 0; zP < 8;zP++) {
          servoDelta = servoTgt[zP] - servoVal[zP];
          if (abs(servoDelta) > 0) {
            servoVal[zP] = servoVal[zP] + (servoDelta / tgtCnt);
            servoInst[zP].writeMicroseconds(servoVal[zP]); atTgt = false;
          }
        } if (atTgt) {tgtCnt = 0;} // already at target values so clear the move count
      } tgtCnt--; // progressivley reduce the target factor
    }
  }

  // regularly check serial receive buffer
  RxRec = false; ReadRx();
  if (RxRec) {decodeCZ();} // regularly track button presses

  // perform these tasks only when active
  if ((QuadActive) && (JoyActive)) {
    // read the X,Y joysticks and walk if demanded
    JoyActive = false; // prevent stack overflow during moves that call loop()
    if (RxRec) {
      // respond to joystick demands every 40ms (25Hz)
      // note direction of travel will only change once stick has been centred
      if ((JoyX == 127) && (JoyY == 128)) {
        // joystick is in centre deadband position
        JoyMode = 0; // reset the direction of travel mode
        if (Walk != 0) {
          Serial.println(F("Stopping"));
          walkInterval = 20000; SetLegsDown();
          JoyMode = 0; WalkLast = Walk; Walk = 0; // stop walking if active
          LedMode = 0;
        }
      } else if (JoyMode == 0) {
        autoRest = 0; // reset user time-out 
        if (Z_Dn < 1) {
          // 'Z' button is not pressed so normal walking movements
          if (JoyY > 150) {JoyMode = 1;} // set direction of travel as forward
          else if (JoyY < 106) {JoyMode = -1;} // set direction of travel as backward
          else if (JoyX < 105) {JoyMode = 3;} // set direction of travel as turn left
          else if (JoyX > 149) {JoyMode = 2;} // set direction of travel as turn right
        } else {
          // 'Z' button pressed so perform special actions
          if (JoyX < 105) {JoyMode = 5;} // set direction of travel as walk to left
          else if (JoyX > 149) {JoyMode = 4;} // set direction of travel as walk to right
          else if (JoyY > 149) {JoyMode = 6;} // perform a bow
          else if (JoyY < 106) {JoyMode = 7;} // perform a hello wave
        }
      } else {
        autoRest = 0; // reset user time-out 
        switch (JoyMode) {
          case -1: JoyMoveBwd(); break;
          case 1: JoyMoveFwd(); break;
          case 2: JoyTurnRgt(); break;
          case 3: JoyTurnLft(); break;
          case 4: JoyMoveRgt(); break;
          case 5: JoyMoveLft(); break;
          case 6: DemoMove_Bow(); break;
          case 7: DemoMove_SitNWave(); break;
        }
      }
    } JoyActive = true; // re-enable this code block
  }

  if (micros() >= nextMicros) {
    // do these tasks every 20ms 
    nextMicros = nextMicros + interval; // set next time point
    
    // perform 'Move Engine' tasks
    switch (MainTask) {
      case 0: break;
    }

    // if not moving do the following:
    if ((Walk < 1) && (tgtCnt < 1)) {
      // return to Power-OFF state?
      if (autoOFF > 0) {autoOFF--; if (autoOFF < 1) {detachServos();}}
    }
        
    // always do the following every 20ms
    // return to standing position after autoRest time-out
    if ((QuadActive) && (autoRest < 32000)) {
      autoRest++;
      if (autoRest == 500) {
        // move to the standing position after 5 seconds
        Serial.println(F("Going To Rest..."));
        MoveToAngles(90,90,90,90,90,90,90,90); WalkLast = 0;
      } else if (autoRest == 15000) {
        // move to the reset position 5 minutes
        Serial.println(F("Sleeping..."));
        GoToRest(50); QuadActive = false;
      }
    }

    // check battery voltage averaged over 16 cycles
    BattVol = analogRead(A0);
    BattAv = BattSum >> 4; BattSum = BattSum + BattVol - BattAv;
    if (BattAv <= BattMin) {
      // battery voltage has reached minimum level
      // a one second timeout counter avoids brownouts and glitches
      BattCnt++; if (BattCnt > 50) {BatteryFailed();}
    } else {BattCnt = 0;} // reset battery LOW timeout
//    Serial.print(F("0,")); Serial.print(BattAv); Serial.println(F(",816"));
  }

  if (LedTick) {
    // LEDs are update once every Walk cycle to keep rates the same
    LedTick = false;
    // update LEDs and clear ON flags
    if (LedMode >= 0) {
      switch (LedMode) {
        case 0: 
          anyVal = 200; anyRnd = 5; if (QuadActive) {anyVal = 50;}
          if (C_Dn > 0) {anyVal = 10; anyRnd = 15;}
          LedCnt++; if (LedCnt >= anyVal) {
            LedCnt = 0; anyVal = random(anyRnd);
            switch(anyVal) {
              case 0: LED0St = LOW; break;
              case 1: LED1St = LOW; break;
              case 2: LED2St = LOW; break;
              case 3: LED3St = LOW; break;
              case 4: LED4St = LOW; break;
              case 5: LED0St = LOW; LED1St = LOW; break;
              case 6: LED1St = LOW; LED2St = LOW; break;
              case 7: LED2St = LOW; LED3St = LOW; break;
              case 8: LED3St = LOW; LED4St = LOW; break;
              case 9: LED0St = LOW; LED2St = LOW; break;
              case 10: LED0St = LOW; LED3St = LOW; break;
              case 11: LED0St = LOW; LED4St = LOW; break;
              case 12: LED1St = LOW; LED3St = LOW; break;
              case 13: LED1St = LOW; LED4St = LOW; break;
              case 14: LED2St = LOW; LED4St = LOW; break;
            }
          } break;
        case 1: // LED move from centre to out
          if (LedCnt <= 10) {LED2St = LOW;} 
          else if (LedCnt <= 16) {LED1St = LOW; LED3St = LOW;}
          else if (LedCnt <= 20) {LED0St = LOW;LED4St = LOW;}
          else if (LedCnt >= 23) {LedCnt = 0;} break;
        case 2: // LED move left to right
          if (LedCnt <= 8) {LED0St = LOW;} 
          else if (LedCnt <= 16) {LED1St = LOW;}
          else if (LedCnt <= 24) {LED2St = LOW;} 
          else if (LedCnt <= 32) {LED3St = LOW;} 
          else if (LedCnt <= 40) {LED4St = LOW;} 
          else if (LedCnt > 45) {LedCnt = 0;} break;
        case 3: // LED move from outer to centre
          if (LedCnt <= 4) {LED0St = LOW; LED4St = LOW;} 
          else if (LedCnt <= 10) {LED1St = LOW; LED3St = LOW;}
          else if (LedCnt <= 20) {LED2St = LOW;} 
          else if (LedCnt >= 23) {LedCnt = 0;} break;
        case 4: // LED move right to left
          if (LedCnt <= 8) {LED4St = LOW;} 
          else if (LedCnt <= 16) {LED3St = LOW;}
          else if (LedCnt <= 24) {LED2St = LOW;} 
          else if (LedCnt <= 32) {LED1St = LOW;} 
          else if (LedCnt <= 40) {LED0St = LOW;} 
          else if (LedCnt > 45) {LedCnt = 0;} break;
      } LedCnt++;
    }
    digitalWrite(LED0, LED0St); LED0St = HIGH;
    digitalWrite(LED1, LED1St); LED1St = HIGH;
    digitalWrite(LED2, LED2St); LED2St = HIGH;
    digitalWrite(LED3, LED3St); LED3St = HIGH;
    digitalWrite(LED4, LED4St); LED4St = HIGH;
  }
}

// ----------------------------------------------------------------------

void delayLoop(unsigned long zDel) {
  // loop for a delay period
  zDel = millis() + zDel;
  while (millis() < zDel) {
    if (ESC) return;
    loop();
  }  
}

// ----------------------------------------------------------------------

void loopWhiletgtCnt() {
  // used to call loop during a 'Move' function
//  Serial.println(F("loopWhiletgtCnt"));
  while (tgtCnt > 0) {
    if (ESC) return;
    loop();
  }
}

// ----------------------------------------------------------------------

void loopWhileWalk(int zWalkCnt) {
  // walk for a number of WalkCnt clicks
//  Serial.println(F("loopWhileWalk"));
  int zWCL;
  while (zWalkCnt > 0) {
    if (ESC) return;
    zWCL = WalkCnt; loop();
    if (zWCL != WalkCnt) {zWalkCnt--;}  
  }
}

// ----------------------------------------------------------------------

void runPOST() {
  // called during start-up
  Serial.println(F("\n\nQuadruped 'RC' v0.00a"));
  Serial.println(F("Starting POST..."));
  Serial.println(F("Testing LEDs..."));
  digitalWrite(LED0, LOW); delay(200); digitalWrite(LED0, HIGH); LED0St= HIGH;
  digitalWrite(LED1, LOW); delay(200); digitalWrite(LED1, HIGH); LED1St= HIGH;
  digitalWrite(LED2, LOW); delay(200); digitalWrite(LED2, HIGH); LED2St= HIGH;
  digitalWrite(LED3, LOW); delay(200); digitalWrite(LED3, HIGH); LED3St= HIGH;
  digitalWrite(LED4, LOW); delay(200); digitalWrite(LED4, HIGH); LED4St= HIGH;
  Serial.println(F("Reseting Servos..."));
  GoToRest(50); autoOFF = 5;
  Serial.println(F("POST complete!"));
}

// ----------------------------------------------------------------------

void setDefaults() {
  // load default values
  atTgt = false; // true if servos are at target values
  autoOFF = 0; // if set > 0 then counts down to servo switch off
  autoRest = 0; // returns to a rest position after a timeout
  BattCnt = 0; // battery failed 1s time-out counter
  BattSum = BattMin << 4; //  pre-set cumulative battery voltage
  C_Cnt = 0; // counter used in 'C' button detection
  C_Dn = 0; // counter for 'C' button down period
  C_Up = 50; // counter for 'C' button up period
  CZ_Wait = false; // flag used for button release wait function
  ESC = false; // true to escape all functions
  LedTick = false; // set true every 'Walk' cycle
  interval = 20000; // main loop interval in microseconds, 50Hz
  JoyActive = true; // flag used to prevent stack overflow during moves
  JoyMode = 0; // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
  JoySpd = 0; // speed vector = max(JoyX,JoyY)
  JoyX= 0; // received value
  JoyXV = 0; // converted value 0 - 127
  JoyY = 0; // received value
  JoyYi = 0; // tracking filter for received JoyY value
  JoyYV = 0; // converted value 0 - 128
  LedCnt = 0; // counter used in LED sequencing
  LedMode = 0; // a value > 0 determines LED flash sequence; if = 0 disabled
  LegDn = LegD; // dynamic angle for leg down in 'Move Engine'
  LegUp = LegU; // dynamic angle for raised down in 'Move Engine'
  MainRun = false; // set true when a MainTask is running to prevent re-entry
  MainTask = 0; // main task pointer
  MainTaskNext = 0; // the task to run after an arrest; default is 0
  Once = 0; // counter for demo moves
  QuadActive = false; // true if active, otherwise false
  servoEn = false; // set true if servos are attached
  servoVal[0] = Rst1; // value sent to servos
  servoVal[1] = Rst2; // value sent to servos
  servoVal[2] = Rst3; // value sent to servos
  servoVal[3] = Rst4; // value sent to servos
  servoVal[4] = Rst5; // value sent to servos
  servoVal[5] = Rst6; // value sent to servos
  servoVal[6] = Rst7; // value sent to servos
  servoVal[7] = Rst8; // value sent to servos
  servoTgt[0] = Rst1; // target values for servos
  servoTgt[1] = Rst2; // target values for servos
  servoTgt[2] = Rst3; // target values for servos
  servoTgt[3] = Rst4; // target values for servos
  servoTgt[4] = Rst5; // target values for servos
  servoTgt[5] = Rst6; // target values for servos
  servoTgt[6] = Rst7; // target values for servos
  servoTgt[7] = Rst8; // target values for servos
  SubTask = 0; // sub task pointer
  tgtCnt = 0; // counter used in automatic servo move to target actions
  Turn = false; // default = false; if true turns continuously until = false
  Walk = 0; // if >0 then walk in one of four directions
  walkInterval = 20000; // main loop walking interval in microseconds
  WalkLast = 0; // previousl Walk value if any
  WalkLftDrive = 128; // drive factor for left-hand side, 128 = max, 0 = min
  WalkMax = 4; // speed factor 4 - 12
  WalkRgtDrive = 128; // drive factor for right-hand side, 128 = max, 0 = min
  WalkSpeed = 1; // value used to determine WalkMax
  Z_Cnt = 0; // counter used in 'Z' button detection
  Z_Dn = 0; // counter for 'Z' button down period
  Z_Up = 50; // counter for 'Z' button up period
}

// ----------------------------------------------------------------------

void synchLoopTimers() {
  // reset main loop timers
  nextMicros = micros() + interval; // set loop timer
  walkMicros = micros() + walkInterval; // set loop timer
}


