// ################################################################################
//
//  Balance Bot And Critter Remote v0.01 Beta
//
//  Released:  18/08/2018
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Elements of this software were taken from the public domain. It uses
    libraries which must be installed on your system.

    This code is for the Nunchuk wireless controller unit. It sends user demands
    over the Wi-Fi and gives indication of this on the LED. Data is sent for a
    joystick demand and for buttons C and Z. Depending on the mode of operation it
    will either send data formated for the Balance Bot or data formatted for the
    critter 'RC' robot. Data received from either robot is sent transparently to
    the serial monitor. This uses the Z button during POST to set the mode of
    operation and therefore the type of data sent.

*/
// Declare constants
#define interval 40 // main loop period is 40 ms (25Hz)
#define LEDPin 13   // LED pin

// Declare libraries
#include <Wire.h>           // Include the Wire.h library for I2C comms
#include <SoftwareSerial.h> // Included serial for WiFi comms

SoftwareSerial SoftSerial(12, 11);  // Rx pin = 12, Tx pin = 11

// Declare and initialise global variables
byte checksum;                      // checksum byte for Critter RC data
char cmdMode;                       // keyboard command
byte DbLL = 80;                     // joystick deadband lower limit
byte DbUL = 170;                    // joystick deadband upper limit
int LedON = 0;                      // LED flash counter
int Mode = 0;                       // data mode, 0 = Balance Bot, 1 = Critter
unsigned long nextMillis = 0;       // loop timer trigger
int nunchuk_address = 0x52;         //Nunchuk I2C address (0x52)
byte received_data[6];              //Declare some global byte variables
int reportEn = 0;                   //reports data for testing if > 0
byte RxByte = 0;                    // byte received from WiFi serial link
byte send_byte;                     // bit pattern sent to BalanceBot over WiFi
byte WiiError = 1;                  // used to detect transmission errors

void setup(){
  // Start serial devices and connect to the Nunchuk over I2C
  pinMode(LEDPin, OUTPUT); digitalWrite(LEDPin, HIGH); // turn LED ON initially
  Serial.begin(9600);                           //Start the serial port at 9600 kbps
  SoftSerial.begin(9600);                       //Start the software serial port at 9600 kbps
  delay(200); digitalWrite(LEDPin, LOW);         // LED ON for minimum of 200ms
  WiiError = 1;
  while (WiiError != 0) {
    // wait in this loop initialising the Wii until it responds
    Wire.begin();                                 //Start the I2C as master
    TWBR = 12;                                    //Set the I2C clock speed to 400kHz
    Wire.begin();                                 //Start the I2C bus as master
    Wire.beginTransmission(nunchuk_address);      //Start communication with the Nunchuck
    Wire.write((byte)0xF0);                       //We want to write to register (F0 hex)
    Wire.write((byte)0x55);                       //Set the register bits as 01010101
    Wire.write((byte)0xFB);                       //We want to write to register (FB hex)
    Wire.write((byte)0x00);                       //Set the register bits as 00000000
    WiiError = Wire.endTransmission();            //End the transmission
    delay(10);                                    //Short delay
  }
  runPOST();                                    // send power-on message
  nextMillis = millis() + interval;             // prime loop timer
}

// --------------------------------------------------------------------------------

void loop(){
  // read the Nunchuk every 40ms and send joystick data if detected
  // X/Y joystick readings are 0 - 255, centre 127/128 respectively.
  // We include a deadband of 80 - 170 for the BalanceBot.
  // The LED is ON when a joystick/button demand is being sent
  // Data is continuously sent for the Critter RC robot, including a checksum byte.
  ReadNunchukData(); // read the NUnchuk data registers
  // default is to send Balance Bot data
  send_byte = B00000000;                        //Set the send_byte variable to 0
  if(received_data[0] < DbLL)send_byte |= B00000001;   //If the variable received_data[0] is smaller then 80 set bit 0 of the send byte variable
  if(received_data[0] > DbUL)send_byte |= B00000010;  //If the variable received_data[0] is larger then 170 set bit 1 of the send byte variable
  if(received_data[1] < DbLL)send_byte |= B00001000;   //If the variable received_data[1] is smaller then 80 set bit 3 of the send byte variable
  if(received_data[1] > DbUL)send_byte |= B00000100;  //If the variable received_data[1] is larger then 170 set bit 2 of the send byte variable
  if((received_data[5] & 1) == 0)send_byte |= B00010000;  //If Z button pressed set bit 4
  if((received_data[5] & 2) == 0)send_byte |= B00100000;  //If C button pressed set bit 5
  if(send_byte > 0){
      // valid data to be sent
    LedON = 0;          // Set LED to come ON constantly
    if (Mode < 1) {
        // Mode == 0 hence send Balance Bot data
        SoftSerial.print((char)send_byte);    //Send the send_byte variable if it's value is larger then 0
    }
  }
  if (Mode > 0) {
    // Mode == 1 hence always send Critter RC data
    // baud rate = 9600, gives approx 960 characters per second
    // at 25Hz we can send blocks of 38 characters max, so 6 is no problem
    // we send a 6 byte frame = [ 2 byte key + 3 bytes data + 1 byte checksum ]
    SoftSerial.write(0x0AA);    // send 1st part of key 1010 1010 = AA Hex
    SoftSerial.write(0x055);    // send 2nd part of key 0101 0101 = 55 Hex
    SoftSerial.write(received_data[0]);    // send Nunchuk joystick-X
    SoftSerial.write(received_data[1]);    // send Nunchuk joystick-Y
    SoftSerial.write(received_data[5]);    // send Nunchuk buttons C and Z
    checksum = 0xAA ^ 0x55 ^ received_data[0] ^ received_data[1] ^ received_data[5];
    SoftSerial.write(checksum);    // send frame checksum byte
  }
  // see if Nunchuk data reporting is enabled?
  if (reportEn > 0) {
    while (SoftSerial.available()) {
      RxByte = SoftSerial.read(); // discard any received data from BalanceBot
    } reportEn++;
    if (reportEn > 25) {
      // report every second
      reportEn = 1; ReportNunchukData();
    }
  }

  // now wait for loop timer
  while (nextMillis > millis()) {
    // wait for timer to catch up
    if (reportEn < 1) {
      // Rx transparent mode
      if (SoftSerial.available()) {RxByte = SoftSerial.read(); Serial.write(RxByte);}
    }
  }
  nextMillis = nextMillis + interval;     //Create a 40 millisecond loop delay
  
  // flash LED if Nunchuk data is being sent
  // or flash once every second if in Balance Bot mode or twice if in Critter mode
  if (LedON == 4) {
    if (Mode > 0) {digitalWrite(LEDPin, HIGH);} // turn ON LED if in critter mode
  } else if (LedON <= 0) {
    // blink every second if no demand is being sent
    digitalWrite(LEDPin, HIGH); LedON = 25; // reset 1 sec timer
  }  else {digitalWrite(LEDPin, LOW);} // turn the LED OFF
  LedON--;
}

// --------------------------------------------------------------------------------

void ReadNunchukData() {
  // read a block of data from the Nunchuk controller
  Wire.beginTransmission(nunchuk_address); //Start communication with the Nunchuck.
  Wire.write(0x00);                        //We want to start reading at register (00 hex)
  Wire.endTransmission();                  //End the transmission
  delay(1);
  Wire.requestFrom(nunchuk_address,6);      //Request 6 bytes from the Nunchuck
  for(byte i = 0; i < 6; i++) received_data[i] = Wire.read(); //Copy the bytes to the received_data array
  received_data[5] = received_data[5] & 0x03; // mask off accelerometer bits
}

// --------------------------------------------------------------------------------

void ReportNunchukData() {
  // send nunchuk data to the serial port, with formatting
  Serial.print("\n\n\n\n\n"); // send 5 blank lines
  Serial.println("Nunchuk Data:"); // send title
  for(byte i = 0; i < 6; i++) {
    Serial.print("Bite "); Serial.print(i); Serial.print(" = ");
    if (received_data[i] < 100) Serial.print(" ");
    if (received_data[i] < 10) Serial.print(" ");
    Serial.print(received_data[i]);       //output I2C data from nunchuk
    Serial.print("\t");
    if (received_data[i] < 128) Serial.print("0");
    if (received_data[i] < 64) Serial.print("0");
    if (received_data[i] < 32) Serial.print("0");
    if (received_data[i] < 16) Serial.print("0");
    if (received_data[i] < 8) Serial.print("0");
    if (received_data[i] < 4) Serial.print("0");
    if (received_data[i] < 2) Serial.print("0");
    Serial.println(received_data[i],BIN);       //output I2C data from nunchuk
  }
}

// --------------------------------------------------------------------------------

void runPOST() {
  // send power-on messages  
  Serial.println("\n\n\n\nNunchuk Complex Remote v0.01\n");
  Serial.println("Checking 'Z' button for Mode change...");
  // check the Z button 100 times over a 1 second period
  int zCnt = 0;
  for (int zI = 0; zI < 100; zI++) {
    // after the 'blink' in setup() we have 1 second to hold down the 'Z' button
    // to force a Mode change. If that opportunity is missed then Mode = 0.
    ReadNunchukData();
    // note buttons are active LOW
    if((received_data[5] & 1) == 0) zCnt++;  //If Z button pressed increment counter
    delay(10); // wait 10ms between readings
  }
  // if the button was held for at least 100ms switch to Critter 'RC' mode
  if (zCnt >= 10) {
    Serial.println("Mode = 1 for Critter 'RC'");
    Mode = 1; DbLL = 125; DbUL = 130; // narrow the deadband to get more range
    digitalWrite(LEDPin, HIGH); // turn ON the LED and wait for button release
    ReadNunchukData();
    while ((received_data[5] & 1) == 0) {
      // stay here until the 'Z' button is released
      delay(10); ReadNunchukData(); // read the Nunchuk every 10ms
    } digitalWrite(LEDPin, LOW);
  } else {
    Serial.println("Mode = 0 for BalanceBot");
  }
  if (reportEn < 1) {
    Serial.println("Entering transparent mode...\n");
  }
}

// --------------------------------------------------------------------------------


