// ######################################################################
//
//  Servo Motor Sketch For Quadruped v0.1 Beta
//
//  Released:  09/11/2018
//
// ######################################################################
/*
    This program simply drives the servo motor to a position
    determined by a soft potentiometer on the PC screen. The code reacts
    directly to commands sent over the serial interface.

    Servo pins are mapped as follows for Quadruped LTOF:

    00  - not mapped
    01  - D7
    02  - D6
    03  - A2
    04  - A3
    05  - A1
    06  - A0
    07  - D8
    08  - D9
    09  - not mapped 
    10  - not mapped 
    11  - not mapped 
    12  - not mapped 
    13  - not mapped 
    
    Commands:
    !   - forces a 'soft' RESET
    SAnn.   - set servo angle in degrees 0 - 180
    SD.     - detach the active servo pin
    SLnn.   - set attached servo lower limit in microseconds
    SMnn.   - set servo angle in microseconds servoLL - servoUL
    SPnn.   - set the active servo pin number
    SUnn.   - set attached servo upper limit in microseconds
*/
// Declare and initialise global variables
#include <Servo.h>
int Angle;  // start with servo in the centre pos
char cmdMode; // command mode
char cmdType; // command type
int cmdVal; // value associated with a cmdType
char keyChar; // any keyboard character
int keyVal; // any keyboard value
int servoLL = 400; // servo min time setting
int servoPin;  // servo output pin
int servoUL = 2600; // servo max time setting
int timeDelay = 1; // delay in milliseconds
Servo servoMain; // Define our Servo

void setup() {
  // put your setup code here, to run once:
  Serial.begin(19200); // high baud rate for sweeping functions
  runPOST();
  delay(1000);          // wait 1 second
}

void loop() {
  // put your main code here, to run repeatedly:
  keyVal = 0; while (keyVal != -1) {readKey();} // empty Rx buffer
  delay(timeDelay);  // Wait nns
}

// ----------------------------------------------------------------------

void doCmd() {
  // a '.' has been received so execute command if valid
  int zPrint = 1;
  switch (cmdMode) {
    case ' ': break;
    case 'S':
      switch (cmdType) {
        case ' ': break;
        case 'A':
          Angle = cmdVal; servoMain.write(Angle);
          // report the angle in microseconds
          Serial.print("SM=");
          Serial.print(servoMain.readMicroseconds());
          Serial.println("."); zPrint = 0;
          break;
        case 'D': servoPin = -1; servoMain.detach(); break;
        case 'L':
          servoLL = cmdVal;
          if (servoPin > -1) {servoMain.attach(servoPin,servoLL,servoUL);}
          break;
        case 'M':
          Angle = cmdVal; servoMain.writeMicroseconds(Angle);
          // report the angle in degrees
          Serial.print("SA=");
          Serial.print(servoMain.read());
          Serial.println("."); zPrint = 0;
          break;
        case 'P':
          servoPin = -1; servoMain.detach();
          switch (cmdVal) {
            case 1: servoPin = 7; break;
            case 2: servoPin = 6; break;
            case 3: servoPin = A2; break;
            case 4: servoPin = A3; break;
            case 5: servoPin = A1; break;
            case 6: servoPin = A0; break;
            case 7: servoPin = 8; break;
            case 8: servoPin = 9; break;
            default: cmdVal = -1; break;
          }
           if (cmdVal > 0) {servoMain.attach(servoPin,servoLL,servoUL);}
          break;
        case 'U':
          servoUL = cmdVal;
          if (servoPin > -1) {servoMain.attach(servoPin,servoLL,servoUL);}
          break;
      } break;
  }
  if (zPrint > 0) {
    // is not a reporting function, echo the received command
    Serial.print(cmdMode); Serial.print(cmdType); Serial.print(cmdVal); Serial.println('.');
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0;
}

// ----------------------------------------------------------------------

void emptySerial() {
  // empty the serial input buffer
  keyVal = 0;
  while (keyVal != -1) {
    keyVal = Serial.read();
  }
}

// ----------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = (cmdVal * 10) + zVal;  
}

// ----------------------------------------------------------------------

void readKey() {
  // reads a key from the keyboard annd reacts accordingly
  keyVal = Serial.read();
  if (keyVal != -1) {
    keyChar = char(keyVal);
//      Serial.print("keyVal=");
//      Serial.print(keyChar);
//      Serial.print("   ASCII=");
//      Serial.println(keyVal);
    int zNF = 0;
    switch (keyChar) {
      case '!': RESET(); zNF = 1;break;
      case '0': extendCmdVal(0); zNF = 1;break;
      case '1': extendCmdVal(1); zNF = 1;break;
      case '2': extendCmdVal(2); zNF = 1;break;
      case '3': extendCmdVal(3); zNF = 1;break;
      case '4': extendCmdVal(4); zNF = 1;break;
      case '5': extendCmdVal(5); zNF = 1;break;
      case '6': extendCmdVal(6); zNF = 1;break;
      case '7': extendCmdVal(7); zNF = 1;break;
      case '8': extendCmdVal(8); zNF = 1;break;
      case '9': extendCmdVal(9); zNF = 1;break;
    }
    if (zNF == 0) {
      if (cmdMode == ' ') {
        // test for new Command Mode char?
        switch (keyChar) {
          case 's': cmdMode = 'S'; break;
          case 'S': cmdMode = 'S'; break;
        } cmdType = ' '; cmdVal = 0;
      } else {
        // test for Command Type char?
        switch (keyChar) {
          case 'a': cmdType = 'A'; break;
          case 'A': cmdType = 'A'; break;
          case 'd': cmdType = 'D'; break;
          case 'D': cmdType = 'D'; break;
          case 'l': cmdType = 'L'; break;
          case 'L': cmdType = 'L'; break;
          case 'm': cmdType = 'M'; break;
          case 'M': cmdType = 'M'; break;
          case 'p': cmdType = 'P'; break;
          case 'P': cmdType = 'P'; break;
          case 'u': cmdType = 'U'; break;
          case 'U': cmdType = 'U'; break;
          case '.': doCmd(); break;
        }
      }
    } else if (keyChar == '.') {doCmd();}
  }
}

// ----------------------------------------------------------------------

void RESET() {
  // perform a soft RESET
  servoMain.detach();
  setDefaults();
}

// ----------------------------------------------------------------------

void runPOST() {
  // called during start-up
  Serial.println("Starting POST...");
  emptySerial();
  RESET();
  Serial.println("POST complete!");
}

// ----------------------------------------------------------------------

void setDefaults() {
  // set defaults at start-up and in soft RESET
  Angle  = 90;  // start with servo in the centre pos
  cmdMode = ' '; // command mode
  cmdType = ' '; // command type
  cmdVal = 0; // value associated with a cmdType
  servoPin = -1;  // servo output pin is undefined
}

