// ################################################################################
//
//  Quadruped Auto Release v0.01
//
//  Released:  19/12/2018
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls an autonomous Quadruped critter robot, driven by 9 servo
    motors, 8 for leags and one for a head. It has limited vision using a laser
    ranging device, VL53L0X. 

    The servo motors must be calibrated during construction, and the values taken
    need to be inserted in the angle definitions included in this code. Functions
    that are commented-out, are not needed but you may find useful to use in your
    code. Enjoy!

    New servo added and recalibrated

*/
// Declare and initialise global variables
#include <Wire.h>
#include <Servo.h>

// Define servo constants
#define Ang1_45 1131
#define Ang1_135 2040
#define Ang2_65 2179
#define Ang2_153 1222
#define Ang3_45 1906
#define Ang3_135 946
#define Ang4_65 841
#define Ang4_153 1736
#define Ang5_45 1174
#define Ang5_135 2131
#define Ang6_65 2207
#define Ang6_153 1323
#define Ang7_45 1839
#define Ang7_135 926
#define Ang8_65 787
#define Ang8_153 1672
#define Head_0 1155 // mid point head angle, zero degrees
#define Head_65N 690 // -65 degrees, lower limit
#define Head_90N 520 // -90 degrees, lower limit
#define Head_65P 1651 // -65 degrees, lower limit
#define Head_90P 1849 // +90 degrees, upper limit
#define LegDwnMin 110 // min angle for leg down in 'Move Engine' Walk
#define LegDwnMax 130 // max angle for leg down in 'Move Engine' Walk
#define LegUpMin 130 // min angle for leg raised in 'Move Engine' Walk
#define LegUpMax 153 // max angle for leg raised in 'Move Engine' Walk
#define Max1 2179 // max value for servo 1
#define Max2 2179 // max value for servo 2
#define Max3 1906 // max value for servo 3
#define Max4 2284 // max value for servo 4
#define Max5 2265 // max value for servo 5
#define Max6 2207 // max value for servo 6
#define Max7 1839 // max value for servo 7
#define Max8 2303 // max value for servo 8
#define Min1 1131 // min value for servo 1
#define Min2 701 // min value for servo 2
#define Min3 844 // min value for servo 3
#define Min4 841 // min value for servo 4
#define Min5 1174 // min value for servo 5
#define Min6 696 // min value for servo 6
#define Min7 796 // min value for servo 7
#define Min8 787 // min value for servo 8
#define Rst1 1423 // reset value for servo 1
#define Rst2 2164 // reset value for servo 2
#define Rst3 1575 // reset value for servo 3
#define Rst4 715 // reset value for servo 4
#define Rst5 1488 // reset value for servo 5
#define Rst6 2212 // reset value for servo 6
#define Rst7 1685 // reset value for servo 7
#define Rst8 801 // reset value for servo 8

// define VL53LOX constants
#define ADDRESS_DEFAULT 0b0101001
#define calcMacroPeriod(vcsel_period_pclks) ((((uint32_t)2304 * (vcsel_period_pclks)* 1655) + 500) / 1000)
#define checkTimeoutExpired() (io_timeout > 0 && ((uint16_t)millis() - timeout_start_ms) > io_timeout)
#define decodeVcselPeriod(reg_val) (((reg_val) + 1) << 1)
#define encodeVcselPeriod(period_pclks) (((period_pclks) >> 1) - 1)
#define startTimeout() (timeout_start_ms = millis())
#define XSHUT 12  // D12 is connected to the XSHUT pin

// Declare register list from VL53L0X.h API, ordered as listed there
#define SYSRANGE_START 0x00
#define SYSTEM_THRESH_HIGH 0x0C
#define SYSTEM_THRESH_LOW 0x0E
#define SYSTEM_SEQUENCE_CONFIG 0x01
#define SYSTEM_RANGE_CONFIG 0x09
#define SYSTEM_INTERMEASUREMENT_PERIOD 0x04
#define SYSTEM_INTERRUPT_CONFIG_GPIO 0x0A
#define GPIO_HV_MUX_ACTIVE_HIGH 0x84
#define SYSTEM_INTERRUPT_CLEAR 0x0B
#define RESULT_INTERRUPT_STATUS 0x13
#define RESULT_RANGE_STATUS 0x14
#define RESULT_CORE_AMBIENT_WINDOW_EVENTS_RTN 0xBC
#define RESULT_CORE_RANGING_TOTAL_EVENTS_RTN 0xC0
#define RESULT_CORE_AMBIENT_WINDOW_EVENTS_REF 0xD0
#define RESULT_CORE_RANGING_TOTAL_EVENTS_REF 0xD4
#define RESULT_PEAK_SIGNAL_RATE_REF 0xB6
#define ALGO_PART_TO_PART_RANGE_OFFSET_MM 0x28
#define I2C_SLAVE_DEVICE_ADDRESS 0x8A
#define MSRC_CONFIG_CONTROL 0x60
#define PRE_RANGE_CONFIG_MIN_SNR 0x27
#define PRE_RANGE_CONFIG_VALID_PHASE_LOW 0x56
#define PRE_RANGE_CONFIG_VALID_PHASE_HIGH 0x57
#define PRE_RANGE_MIN_COUNT_RATE_RTN_LIMIT 0x64
#define FINAL_RANGE_CONFIG_MIN_SNR 0x67
#define FINAL_RANGE_CONFIG_VALID_PHASE_LOW 0x47
#define FINAL_RANGE_CONFIG_VALID_PHASE_HIGH 0x48
#define FINAL_RANGE_CONFIG_MIN_COUNT_RATE_RTN_LIMIT 0x44
#define PRE_RANGE_CONFIG_SIGMA_THRESH_HI 0x61
#define PRE_RANGE_CONFIG_SIGMA_THRESH_LO 0x62
#define PRE_RANGE_CONFIG_VCSEL_PERIOD 0x50
#define PRE_RANGE_CONFIG_TIMEOUT_MACROP_HI 0x51
#define PRE_RANGE_CONFIG_TIMEOUT_MACROP_LO 0x52
#define SYSTEM_HISTOGRAM_BIN 0x81
#define HISTOGRAM_CONFIG_INITIAL_PHASE_SELECT 0x33
#define HISTOGRAM_CONFIG_READOUT_CTRL 0x55
#define FINAL_RANGE_CONFIG_VCSEL_PERIOD 0x70
#define FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI 0x71
#define FINAL_RANGE_CONFIG_TIMEOUT_MACROP_LO 0x72
#define CROSSTALK_COMPENSATION_PEAK_RATE_MCPS 0x20
#define MSRC_CONFIG_TIMEOUT_MACROP 0x46
#define SOFT_RESET_GO2_SOFT_RESET_N 0xBF
#define IDENTIFICATION_MODEL_ID 0xC0
#define IDENTIFICATION_REVISION_ID 0xC2
#define OSC_CALIBRATE_VAL 0xF8
#define GLOBAL_CONFIG_VCSEL_WIDTH 0x32
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_0 0xB0
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_1 0xB1
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_2 0xB2
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_3 0xB3
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_4 0xB4
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_5 0xB5
#define GLOBAL_CONFIG_REF_EN_START_SELECT 0xB6
#define DYNAMIC_SPAD_NUM_REQUESTED_REF_SPAD 0x4E
#define DYNAMIC_SPAD_REF_EN_START_OFFSET 0x4F
#define POWER_MANAGEMENT_GO1_POWER_FORCE 0x80
#define VHV_CONFIG_PAD_SCL_SDA__EXTSUP_HV 0x89
#define ALGO_PHASECAL_LIM 0x30
#define ALGO_PHASECAL_CONFIG_TIMEOUT 0x30

// define VL53L0C data structures
struct SequenceStepEnables {boolean tcc, msrc, dss, pre_range, final_range;};
struct SequenceStepTimeouts {
      uint16_t pre_range_vcsel_period_pclks, final_range_vcsel_period_pclks;
      uint16_t msrc_dss_tcc_mclks, pre_range_mclks, final_range_mclks;
      uint32_t msrc_dss_tcc_us, pre_range_us, final_range_us;};

// declare VL53L0X variables
uint8_t address = ADDRESS_DEFAULT;
bool did_timeout;
uint16_t io_timeout;
uint8_t last_status; // status of last I2C transmission
uint16_t Range; // 16-bit range value
uint8_t stop_variable; // read by init and used when starting measurement; is StopVariable field of VL53L0X_DevData_t structure in API
uint32_t measurement_timing_budget_us;
uint16_t timeout_start_ms;
enum vcselPeriodType {VcselPeriodPreRange, VcselPeriodFinalRange};
//void VL53L0X_setAddress(uint8_t new_addr);
//inline uint8_t VL53L0X_getAddress(void) { return address; }
//bool VL53L0X_init(bool io_2v8 = true);
//void VL53L0X_writeReg(uint8_t reg, uint8_t value);
//void VL53L0X_writeReg16Bit(uint8_t reg, uint16_t value);
//void VL53L0X_writeReg32Bit(uint8_t reg, uint32_t value);
//uint8_t VL53L0X_readReg(uint8_t reg);
//uint16_t VL53L0X_readReg16Bit(uint8_t reg);
//uint32_t VL53L0X_readReg32Bit(uint8_t reg);
//void VL53L0X_writeMulti(uint8_t reg, uint8_t const * src, uint8_t count);
//void VL53L0X_readMulti(uint8_t reg, uint8_t * dst, uint8_t count);
//bool VL53L0X_setSignalRateLimit(float limit_Mcps);
//float VL53L0X_getSignalRateLimit(void);
//bool VL53L0X_setMeasurementTimingBudget(uint32_t budget_us);
//uint32_t VL53L0X_getMeasurementTimingBudget(void);
//bool VL53L0X_setVcselPulsePeriod(vcselPeriodType type, uint8_t period_pclks);
//uint8_t VL53L0X_getVcselPulsePeriod(vcselPeriodType type);
void VL53L0X_startContinuous(uint32_t period_ms = 0);
//void VL53L0X_stopContinuous(void);
//uint16_t VL53L0X_readRangeContinuousMillimeters(void);
//uint16_t VL53L0X_readRangeSingleMillimeters(void);
inline void VL53L0X_setTimeout(uint16_t timeout) { io_timeout = timeout; }
//inline uint16_t VL53L0X_getTimeout(void) { return io_timeout; }
bool VL53L0X_timeoutOccurred(void);
bool VL53L0X_getSpadInfo(uint8_t * count, bool * type_is_aperture);
//void VL53L0X_getSequenceStepEnables(SequenceStepEnables * enables);
//void VL53L0X_getSequenceStepTimeouts(SequenceStepEnables const * enables, SequenceStepTimeouts * timeouts);
//bool VL53L0X_performSingleRefCalibration(uint8_t vhv_init_byte);
//static uint16_t VL53L0X_decodeTimeout(uint16_t value);
//static uint16_t VL53L0X_encodeTimeout(uint16_t timeout_mclks);
//static uint32_t VL53L0X_timeoutMclksToMicroseconds(uint16_t timeout_period_mclks, uint8_t vcsel_period_pclks);
//static uint32_t VL53L0X_timeoutMicrosecondsToMclks(uint32_t timeout_period_us, uint8_t vcsel_period_pclks);

// Define general constants
#define AngDifMax 250 // maximum sweep angle from centre difference
#define autoRestMax 32000 // autoRest max counter value
#define BattMin 804 // minimum A0 reading to trigger a battery LOW event
#define DbLLX 117 // JoyX deadband lower limit
#define DbULX 137 // JoyX deadband Upper limit
#define DbLLY 118 // JoyY deadband lower limit
#define DbULY 138 // JoyY deadband Upper limit
#define HeadSpd 24 // speed used in autonomous mode
#define LED0 3 // leftmost LED output pin
#define LED1 4 // left LED output pin
#define LED2 5 // centre LED output pin
#define LED3 10 // right LED output pin
#define LED4 11 // rightmost LED output pin
#define LtofDet 350 // max range for target detection in head scanning
#define LtofLimit 500 // Range max value is always limited to this figure
#define LtofMax 400 // out of range value for head scanning
#define LtofMin 55 // minimum range value
#define LtofStop 230 // range value at which autonomous walking stops
#define SpdRmp 4 // walking speed acceleration damping factor
#define servoHeadPin 2 // pin for head servo
#define SwPin A6 // button switch pin
#define WalkSpeedToMax 250; // set WalkSpeed auto-reduction as 5 seconds

// define servo arrays
//                 01,02,03,04,05,06,07,08 
int servoMem[8]; // record of values sent to servos, used in fidgeting
int servoPins[] = { 7, 6,A2,A3,A1,A0, 8, 9};  // servo output pins assigned
int servoVal[8]; // value sent to servos
int servoTgt[8]; // target values for servos

// Declare and initialise global variables
int AngDif, AngNeg, AngPos, AngRef; // temporary head angles
int anyDel; // any delay value
int anyFor; // any for loop value
int anyInt; // any temporary integer value
long anyLong; // any long value
unsigned long anyMilli; // any millisecond timer
int anyRnd; // any random integer
int anyVal;  // any temp variable
bool atHeadTgt; // true if head servo is at target value
bool atTgt; // true if servos are at target values
int autoHeadOFF; // if set > 0 then counts down to head servo switch OFF, if -1 disabled
int automCnt; // delay counter used in autonomous mode
int automRnd; // random walking distance in autonomous mode
int automTurn; // flag used in autonomous turning
int autoOFF; // if set > 0 then counts down to servo switch OFF, if -1 disabled
int autoRest; // returns to a rest position after a timeout
int BattAv; // average battery voltage
int BattCnt; // battery failed 1s time-out counter
int BattSum; // cumulative battery voltage
int BattVol; // instantaneous battery voltage
int CalcCnt; // tracks the number of head calculations in autonomous mode
byte checksum; // XOR Rx data to detect corruption at edge of range
bool ESC; // true to escape all functions
bool HeadActive; // true if head is in scan mode for tasks
bool HeadAdj; // flag used in head scanning 
int HeadCnt; // counter used in head movements
int HeadCyc; // cycle counter used in head movements
int HeadDir; // head movement direction, +1 == RH, -1 == LH
long HeadErr; // value used in head angle adjustment
bool HeadErrSt; // trip flag used in head scanning
long HeadErrTrip; // error tripping value used in head angle adjustment
long HeadErrTrk; // error tracking value used in head angle adjustment
unsigned long headInterval; // main loop head movement interval in microseconds
unsigned long headMicros; // main loop head movement timer in microseconds
int HeadMode; // pointer used in head scanning
int HeadPause; // 20ms counter to temporarily pause head movement
int HeadSubTask; // head movement sub task pointer
int HeadTask; // head movement task pointer
int HeadTaskNext; // head movement next task pointer
bool HeadTaskRun; // set to true during head tasks
int headTgtCnt; // counter used in automatic head servo move to target actions
int HeadVal = Head_0 + 200; // default head servo angle, set to 200 > Head_0
int HeadTgt = HeadVal; // default head servo angle
unsigned long interval; // main loop interval in microseconds
int LastTask; // record of the previous MainTask
bool  LED0St; // HIGH/LOWE state of LED pin
bool  LED1St; // HIGH/LOWE state of LED pin
bool  LED2St; // HIGH/LOWE state of LED pin
bool  LED3St; // HIGH/LOWE state of LED pin
bool  LED4St; // HIGH/LOWE state of LED pin
bool LedBlink; // HIGH/LOW LED blinking state
int LedCnt; // counter used in LED sequencing
int LedMode; // a value > 0 determines LED flash sequence; if = 0 disabled
int LedMsk; // a mask used to overide LedMode functions
int LedMskCnt; // a counter used with LedMsk overide time-out
bool LedTick; // set true every 'Walk' cycle
int LegDn; // dynamic angle for leg down in 'Move Engine'
int LegSt1,LegSt3,LegSt5,LegSt7; // up down states of legs
int LegUp; // dynamic angle for leg raised in 'Move Engine'
int LegUpFwd; // dynamic angle for forward leg raised in 'Move Engine'
int LTOF_On; // > 0 when laser ranging is enabled
unsigned long LTOF_Wait; // delay used to prevent wasted time in LTOF readings
bool MainRun; // set true when a MainTask is running to prevent re-entry
int MainTask; // main task pointer
int MainTaskNext; // the task to run after an arrest; default is 0
byte nextLoop; // flag used to trigger a head task during the next loop if > 0
unsigned long nextMicros; // main loop interval timer in microseconds
int Once; // counter for demo moves
bool QuadActive; // true if active, otherwise false
int RangeAng; // angle at which minimum range was measured
int RangeCentre; // distance value looking forward
bool RangePnt; // true when a centre range value is recorded, or extremes of sweep
int RangeCentreTracker; // slugged distance value looking forward
long RangeIntL; // L-H integrator used in head centring
long RangeIntR; // R-H integrator used in head centring
int RangeLeft; // distance value lookin left
unsigned long RangeMin; // min range value during close range scanning 
int RangeMinAng; // angle at which RangeMin occured
int RangeMinMem; // RangeMin at end of each half sweep
int RangeRight; // distance value lookin right
int RangeTrig; // range trigger point used in head scanning
int R0,R1,R2,R3,R4,R5,R6,RM; // range values for backaway and target tracking
int servoDelta; // temporary delta value used in target movements
bool servoHeadEn; // set true if head servo is attached
bool servoEn; // set true if servos are attached
int SubTask; // sub task pointer
int SubVal; // anyvalue used in a subroutine
int swCnt; // button switch counter
bool swLastState; // previous state of button switch, HIGH/LOW
bool swState; // state of read button switch pin
int swTimer; // timer used to detemine button sequences
int tgtCnt; // counter used in automatic servo move to target actions
bool TrgtDet; // true if a target has been detected in autonomous mode
bool Turn; // default = false; if true turns continuously until = false
int TurnCnt; // counter used in neutral turning, inversely proportional to speed
bool Turning; // set = true so that ESC can be used to break out of a turn cycle
int Walk; // if >0 then walk in one of four directions
long WalkCnt; // a counter which controls the walking process
long Walked; // counter used to return robot to a given position
unsigned long walkInterval; // main loop walking interval in microseconds
int WalkLast; // previousl Walk value if any
long WalkLftDrive; // drive factor for left-hand side, 128 = max, 0 = min
byte WalkMax; // speed factor 4 - 12
unsigned long walkMicros; // main loop walking task interval timer in microseconds
int WalkNext; // used to store the next walking mode
int WalkSpeed; // value used to determine WalkMax
int WalkSpeedTO; // Time-Out counter for WalkSpeed auto-reduction, counts to zero
long WalkRgtDrive; // drive factor for right-hand side, 128 = max, 0 = min


// Declare objects
Servo servoInst[8]; // define Quadruped servos
Servo servoHead; // define head servo

void setup() {
  // put your setup code here, to run once:
  setDefaults();
  pinMode(LED0, OUTPUT); digitalWrite(LED0, HIGH); // LED0 OFF initially
  pinMode(LED1, OUTPUT); digitalWrite(LED1, HIGH); // LED1 OFF initially
  pinMode(LED2, OUTPUT); digitalWrite(LED2, HIGH); // LED2 OFF initially
  pinMode(LED3, OUTPUT); digitalWrite(LED3, HIGH); // LED3 OFF initially
  pinMode(LED4, OUTPUT); digitalWrite(LED4, HIGH); // LED4 OFF initially
  pinMode(XSHUT,OUTPUT); digitalWrite(XSHUT,LOW); // reset the VL53L0X
  Serial.begin(115200); // baud rate for Rx/Tx comms
  Wire.begin();
  runPOST();
  synchLoopTimers(); // reset loop timers
}

void loop() {
  // main code which runs repeatedly as multi-tasking functions
  // continuously read the laser range finder, when enabled
  switch(LTOF_On) {
    // this is the VL53L0X_readRangeContinuousMillimeters() function broken down
    // into a state machine which take measurements when LTOF_On > 0
    // use the VL53L0X_Enable/Disable functions to switch the laser ON/OFF
    case 0: break;
    case 1: 
      // Restart the timeout period during which a measurement should occur, as the
      // dev ice is opperating in continuous measurement mode
      startTimeout(); LTOF_On++; break;
    case 2:
      // Here we read the status of the interrupt register over I2C to see if a
      // measurement has been completed. We also check for a timeout occuring.
      if ((VL53L0X_readReg(RESULT_INTERRUPT_STATUS) & 0x07) == 0) {
        if (checkTimeoutExpired()) {
          did_timeout = true;
          return 65535;
        }
      } else {LTOF_On++;} // an interrupt has occured so move onto reading it
      break;
    case 3:
      // assumptions: Linearity Corrective Gain is 1000 (default);
      // fractional ranging is not enabled
      Range = VL53L0X_readReg16Bit(RESULT_RANGE_STATUS + 10);
      VL53L0X_writeReg(SYSTEM_INTERRUPT_CLEAR, 0x01);
      LTOF_Wait = millis() + 28;
//      Serial.println(Range);
      Range = min(Range,LtofLimit);
//      Serial.print(Range); Serial.print("\t"); Serial.print(HeadVal); Serial.print("\t"); Serial.print(RangeAng); Serial.print("\t"); Serial.println(RangeCentreTracker);
      LTOF_On++; break;
    case 4:
      // as the sensor only returns measurements every 30-32ms, there is no point in
      // spending time reading interrupt registers ov er I2C, so we wait here until
      // the wait period expires
      if (millis() >= LTOF_Wait) {LTOF_On = 1;} // wait time-out reached
      break;
  }

  if (micros() >= walkMicros) {
    // walking uses a separate timer so that the speed of walking can be controlled
    // before walking is enabled ensure that start conditions are set
    walkMicros = walkMicros + walkInterval; // set next time point
    LedTick = true; // perform LED sequences at same rate as walking

    
//    Serial.print(RangeIntL);  Serial.print("\t");  Serial.print(RangeIntR);  Serial.print("\t");  Serial.print(RangeAng);  Serial.println("\t 0 \t 12000");
//    Serial.print(Head_65P);  Serial.print("\t");  Serial.print(Head_65N);  Serial.print("\t");  Serial.print(RangeAng);  Serial.println("\t 520 \t 1849");
    

    
    if (Walk > 0) {
      if (autoOFF >= 0) {autoOFF = 5;} // set auto-OFF timeout
      WalkCnt++; if (WalkCnt > 87) {WalkCnt = 0;} // increment and restart
      switch (Walk) {
        case 1: // walking forwards
          // front left leg &  rear right leg
          Walked++; // increase how far walked counter
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt == 22) {SetAngle(3,LegUpFwd); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt < 46) {SetAngle90To135(0,90+((WalkCnt * WalkLftDrive)/128));
            SetAngle90To135(4,90+(((45-WalkCnt)*WalkRgtDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(1,LegUpFwd); SetAngle(5,LegUp);}
          else {SetAngle90To135(0,90+(((87-WalkCnt)*WalkLftDrive)/128));
            SetAngle90To135(4,90-(((42-WalkCnt)*WalkRgtDrive)/128));}
          // front right leg &  rear left leg
          if (WalkCnt > 43) {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle90To135(2,90+(((WalkCnt-44)*WalkRgtDrive)/128));
            SetAngle90To135(6,90+(((89-WalkCnt)*WalkLftDrive)/128));}
          else if (WalkCnt > 2) {SetAngle90To135(2,90+(((43-WalkCnt)*WalkRgtDrive)/128));
            SetAngle90To135(6,90+(((WalkCnt+2)*WalkLftDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(3,LegUpFwd); SetAngle(7,LegUp);}
          else {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle90To135(2,90+(((WalkCnt+44)*WalkRgtDrive)/128));
            SetAngle90To135(6,90+(((1-WalkCnt)*WalkLftDrive)/128));} break;
            
        case 2: // walking towards the right
          // front right leg &  rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle45To90(2,90-((WalkCnt*WalkLftDrive)/128));
            SetAngle45To90(6,90-(((45-WalkCnt)*WalkRgtDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(3,LegUpFwd); SetAngle(7,LegUp);}
          else {SetAngle45To90(2,90-(((87-WalkCnt)*WalkLftDrive)/128));
            SetAngle45To90(6,90-(((WalkCnt-42)*WalkRgtDrive)/128));}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle45To90(0,90+(((WalkCnt-88)*WalkLftDrive)/128));
            SetAngle45To90(4,90-(((WalkCnt-44)*WalkRgtDrive)/128));}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUpFwd);} // Up start condition
          else if (WalkCnt > 2) {SetAngle45To90(0,90-(((2+WalkCnt)*WalkLftDrive)/128));
            SetAngle45To90(4,90-(((43-WalkCnt)*WalkRgtDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUpFwd);}
          else {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle45To90(0,90+(((WalkCnt-1)*WalkLftDrive)/128));
            SetAngle45To90(4,90-(((WalkCnt+44)*WalkRgtDrive)/128));} break;
            
        case 3: // walking backwards
          // front left leg &  rear right leg
           Walked--; // decrease how far walked counter
         if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt < 46) {SetAngle90To135(0,90+(((45-WalkCnt)*WalkLftDrive)/128));
            SetAngle90To135(4,90+((WalkCnt * WalkRgtDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUpFwd);}
          else {SetAngle90To135(0,90-(((42-WalkCnt)*WalkLftDrive)/128));
            SetAngle90To135(4,90+(((87-WalkCnt)*WalkRgtDrive)/128));}
          // front right leg &  rear left leg
          if (WalkCnt > 43) {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle90To135(2,90+(((89-WalkCnt)*WalkRgtDrive)/128));
            SetAngle90To135(6,90+(((WalkCnt-44)*WalkLftDrive)/128));}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUpFwd);} // Up start condition
          else if (WalkCnt > 2) {SetAngle90To135(2,90+(((WalkCnt+2)*WalkRgtDrive)/128));
            SetAngle90To135(6,90+(((43-WalkCnt)*WalkLftDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUpFwd);}
          else {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle90To135(2,90+(((1-WalkCnt)*WalkRgtDrive)/128));
            SetAngle90To135(6,90+(((WalkCnt+44)*WalkLftDrive)/128));} break;
            
        case 4: // walking towards the left
          // front right leg & rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle45To90(2,90-(((45-WalkCnt)*WalkRgtDrive)/128));
            SetAngle45To90(6,90-((WalkCnt*WalkLftDrive)/128));}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUpFwd);}
          else {SetAngle45To90(2,90-(((WalkCnt-42)*WalkRgtDrive)/128));
            SetAngle45To90(6,90-(((87-WalkCnt)*WalkLftDrive)/128));}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle45To90(0,90-(((WalkCnt-44)*WalkRgtDrive)/128));
            SetAngle45To90(4,90+(((WalkCnt-88)*WalkLftDrive)/128));}
          else if (WalkCnt == 22) {SetAngle(1,LegUpFwd); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle45To90(0,90-(((43-WalkCnt)*WalkRgtDrive)/128));
            SetAngle45To90(4,90-(((2+WalkCnt)*WalkLftDrive)/128));}
          else if (WalkCnt > 1) {SetAngle(1,LegUpFwd); SetAngle(5,LegUp);}
          else {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle45To90(0,90-(((WalkCnt+44)*WalkRgtDrive)/128));
            SetAngle45To90(4,90+(((WalkCnt-1)*WalkLftDrive)/128));} break;
            
        case 5: // turning towards the left
          // front left leg & rear right leg
          if (WalkCnt < 1) {SetAngle(1, LegDn); SetAngle(5, LegDn);}
          else if (WalkCnt < 46) {SetAngle(0,90-WalkCnt); SetAngle(4,90-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(0,WalkCnt+3); SetAngle(4,WalkCnt+3);}
          // front right leg &  rear left leg
          if (WalkCnt > 43) {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,WalkCnt+1); SetAngle(6,WalkCnt+1);}
          else if (WalkCnt == 22) {SetAngle(3,LegUp); SetAngle(7,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(2,88-WalkCnt); SetAngle(6,88-WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(3,LegDn); SetAngle(7,LegDn);
            SetAngle(2,WalkCnt+89); SetAngle(6,WalkCnt+89);} break;
            
        case 6: // turning towards the right
          // front right leg & rear left leg
          if (WalkCnt < 1) {SetAngle(3, LegDn); SetAngle(7, LegDn);}
          else if (WalkCnt < 46) {SetAngle(2,90-WalkCnt); SetAngle(6,90-WalkCnt);}
          else if (WalkCnt == 46) {SetAngle(3,LegUp); SetAngle(7,LegUp);}
          else {SetAngle(2,WalkCnt+3); SetAngle(6,WalkCnt+3);}
          // front left leg &  rear right leg
          if (WalkCnt > 43) {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,WalkCnt+1); SetAngle(4,WalkCnt+1);}
          else if (WalkCnt == 22) {SetAngle(1,LegUp); SetAngle(5,LegUp);} // Up start condition
          else if (WalkCnt > 2) {SetAngle(0,88-WalkCnt); SetAngle(4,88-WalkCnt);}
          else if (WalkCnt > 1) {SetAngle(1,LegUp); SetAngle(5,LegUp);}
          else {SetAngle(1,LegDn); SetAngle(5,LegDn);
            SetAngle(0,WalkCnt+89); SetAngle(4,WalkCnt+89);} break;
      }
    } else if (tgtCnt > 0) {
      // if not walking see if we need to move the servos?
      if (autoOFF >= 0) {autoOFF = 5;} // set auto-OFF timeout
      if (tgtCnt == 1) {
        // move values immediately to the target value
        for (int zP = 0; zP < 8;zP++) {
          servoVal[zP] = servoTgt[zP]; // set values to target values
          servoInst[zP].writeMicroseconds(servoVal[zP]);
        }
      } else {
        // move values progressively towards the target values
        atTgt = true; // set a test flag for achieving target values
        for (int zP = 0; zP < 8;zP++) {
          servoDelta = servoTgt[zP] - servoVal[zP];
          if (abs(servoDelta) > 0) {
            servoVal[zP] = servoVal[zP] + (servoDelta / tgtCnt);
            servoInst[zP].writeMicroseconds(servoVal[zP]); atTgt = false;
          }
        } if (atTgt) {tgtCnt = 0;} // already at target values so clear the move count
      }
      if (tgtCnt > 0) {tgtCnt--;} // progressivley reduce the target factor
    }
//    Serial.println(tgtCnt);
  }


  if (micros() >= headMicros) {
    // move the head towards the target if counter is set
    headMicros = headMicros + headInterval;
    if (HeadPause > 0) {
      // reduce headPause to zero before doing anything
      // headPause counts in 20ms increments, 50 == 1 second
      HeadPause--;
    } else {
      if (headTgtCnt > 0) {
  //      Serial.println(headTgtCnt);
        if (autoHeadOFF >= 0) {autoHeadOFF = 20;} // set auto-OFF timeout
        atHeadTgt = true; // set a test flag for achieving target values
        if (headTgtCnt == 1) {
          // move head immediately to the target value
          HeadVal = HeadTgt; // set values to target values
          servoHead.writeMicroseconds(HeadVal);
        } else {
          // move head value progressively towards the target value
          servoDelta = HeadTgt - HeadVal;
          if (abs(servoDelta) > 0) {
            HeadVal = HeadVal + (servoDelta / headTgtCnt);
            servoHead.writeMicroseconds(HeadVal); atHeadTgt = false;
          } if (atHeadTgt) {headTgtCnt = 0;} // already at target value so clear the move count
        }
        if (headTgtCnt > 0) {headTgtCnt--;} // progressivley reduce the target factor
  //      Serial.println(HeadVal);
      }
    }
  }

  
  if (nextLoop > 0) {
    // head movement tasks performed in the next loop after the main tasks so as
    // not to impact on movement task timings too much
    nextLoop = 0; DoHeadTasks();
  }

  
  if (micros() >= nextMicros) {
    // do these tasks every 20ms 
    nextMicros = nextMicros + interval; // set next time point
    nextLoop = 1; // initiate a head movement task during the next loop cycle
    
    // perform 'Move Engine' tasks
    switch (MainTask) {
      case 0: break; // default null task
      case 1: // goto QuadActive default standing position
        GoToStand(50); MainTask = 0; break;
      case 2: // backaway from target mode, then return to start
        MainTask_TrackWall(0); break;
      case 3: // track wall target mode
        MainTask_TrackWall(1); break;
      case 4: // autonomous demo mode
        MainTask_Autonomous(); break;
    }

    // perform auto-speed reduction if needed, applies to speeds 4 & 5 only
    // this is aimed at preventing servo burn-out, limiting high speed bursts
    if (WalkSpeedTO > 0) {
      WalkSpeedTO--; if (WalkSpeedTO == 0) {
        if (WalkSpeed == 4) {WalkSpeed = 3; LedMsk = 7; LedMskCnt = 20; WalkMax = 10; WalkSpeedTO = 2 * WalkSpeedToMax;}
        if (WalkSpeed == 5) {WalkSpeed = 4; LedMsk = 15; LedMskCnt = 20; WalkMax = 13; WalkSpeedTO = WalkSpeedToMax;}
      }
    }
    
    // if not moving do the following:
    if ((Walk < 1) && (tgtCnt < 1)) {
      // return legs to Power-OFF state?
      if (autoOFF > 0) {autoOFF--; if (autoOFF < 1) {detachServos();}}
    }
    if (headTgtCnt < 1) {
      // return head to Power-OFF state?
      if (autoHeadOFF > 0) {autoHeadOFF--; if (autoHeadOFF < 1) {detachHeadServo();}}
    }
        
    // always do the following every 20ms
    // return to standing position after autoRest time-out
    if ((QuadActive) && (autoRest < autoRestMax)) {
      autoRest++;
      if (autoRest == 500) {
        // move to the standing position after 5 seconds
//        Serial.println(F("Resting..."));
        MoveToAngles(90,90,90,90,90,90,90,90); WalkLast = 0;
      } else if (autoRest == 15000) {
        // move to the reset position 5 minutes
//        Serial.println(F("Sleeping..."));
        GoToRest(50);  loopWhiletgtCnt();
        setDefaults();
      }
    }

    // read button switch
    BattVol = analogRead(A6);
//    Serial.println(BattVol);
    readSwitch();

    if (swState == HIGH) {
      // only perform battery voltage check when switch is not pressed
      // check battery voltage averaged over 16 cycles
      BattAv = BattSum >> 4; BattSum = BattSum + BattVol - BattAv;
//      Serial.println(BattAv);
      if (BattAv <= BattMin) {
        // battery voltage has reached minimum level
        // a one second timeout counter avoids brownouts and glitches
        BattCnt++; if (BattCnt > 50) {BatteryFailed();}
      } else {BattCnt = 0;} // reset battery LOW timeout
//      Serial.print(F("0,")); Serial.print(BattAv); Serial.println(F(",816"));
    }
  }

  if (LedTick) {
    // LEDs are update once every Walk cycle to keep rates the same
    LedTick = false;
    // update LEDs and clear ON flags
    if (LedMode >= 0) {
      switch (LedMode) {
        case 0: 
          // random LED sequence to indicate Quadruped is functional
          anyVal = 200; anyRnd = 5; if (QuadActive) {anyVal = 50;}
          LedCnt++; if (LedCnt >= anyVal) {
            LedCnt = 0; anyVal = random(anyRnd);
            switch(anyVal) {
              case 0: LED0St = LOW; break;
              case 1: LED1St = LOW; break;
              case 2: LED2St = LOW; break;
              case 3: LED3St = LOW; break;
              case 4: LED4St = LOW; break;
              case 5: LED0St = LOW; LED1St = LOW; break;
              case 6: LED1St = LOW; LED2St = LOW; break;
              case 7: LED2St = LOW; LED3St = LOW; break;
              case 8: LED3St = LOW; LED4St = LOW; break;
              case 9: LED0St = LOW; LED2St = LOW; break;
              case 10: LED0St = LOW; LED3St = LOW; break;
              case 11: LED0St = LOW; LED4St = LOW; break;
              case 12: LED1St = LOW; LED3St = LOW; break;
              case 13: LED1St = LOW; LED4St = LOW; break;
              case 14: LED2St = LOW; LED4St = LOW; break;
            }
          } break;
        case 1: // LED move from centre to out
          if (LedCnt <= 10) {LED2St = LOW;} 
          else if (LedCnt <= 16) {LED1St = LOW; LED3St = LOW;}
          else if (LedCnt <= 20) {LED0St = LOW;LED4St = LOW;}
          else if (LedCnt >= 23) {LedCnt = 0;} break;
        case 2: // LED move left to right
          if (LedCnt <= 8) {LED0St = LOW;} 
          else if (LedCnt <= 16) {LED1St = LOW;}
          else if (LedCnt <= 24) {LED2St = LOW;} 
          else if (LedCnt <= 32) {LED3St = LOW;} 
          else if (LedCnt <= 40) {LED4St = LOW;} 
          else if (LedCnt > 45) {LedCnt = 0;} break;
        case 3: // LED move from outer to centre
          if (LedCnt <= 4) {LED0St = LOW; LED4St = LOW;} 
          else if (LedCnt <= 10) {LED1St = LOW; LED3St = LOW;}
          else if (LedCnt <= 20) {LED2St = LOW;} 
          else if (LedCnt >= 23) {LedCnt = 0;} break;
        case 4: // LED move right to left
          if (LedCnt <= 8) {LED4St = LOW;} 
          else if (LedCnt <= 16) {LED3St = LOW;}
          else if (LedCnt <= 24) {LED2St = LOW;} 
          else if (LedCnt <= 32) {LED1St = LOW;} 
          else if (LedCnt <= 40) {LED0St = LOW;} 
          else if (LedCnt > 45) {LedCnt = 0;} break;
        case 5:
          // display laser range as LED bars
          LedCnt++; if (LedCnt >= 10) {LedCnt = 0;}
          LedBlink = LOW; if (LedCnt > 5) {LedBlink = HIGH;}
          if (Range <= LtofMax) {
            anyVal = map(Range,LtofMin,LtofMax,10,0);
            if (anyVal == 1) {LED0St = LedBlink;}
            if (anyVal >= 2) {LED0St = LOW;}
            if (anyVal == 3) {LED1St = LedBlink;}
            if (anyVal >= 4) {LED1St = LOW;}
            if (anyVal == 5) {LED2St = LedBlink;}
            if (anyVal >= 6) {LED2St = LOW;}
            if (anyVal == 7) {LED3St = LedBlink;}
            if (anyVal >= 8) {LED3St = LOW;}
            if (anyVal == 9) {LED4St = LedBlink;}
            if (anyVal >= 10) {LED4St = LOW;}
          }
          break;
        case 99:
          if (LedCnt > 0) {LedCnt--;}
          else {LedMode = 0;}
          break;
      } LedCnt++;
    }
    if (LedMskCnt > 0) {
      // an LED mask has been set which overides the LedMode function
      LedMskCnt--;
      switch (LedMsk) {
        case 1: LED0St = LOW; break;
        case 3: LED0St = LOW; LED1St = LOW; break;
        case 7: LED0St = LOW; LED1St = LOW; LED2St = LOW; break;
        case 15: LED0St = LOW; LED1St = LOW; LED2St = LOW; LED3St = LOW; break;
        case 31: LED0St = LOW; LED1St = LOW; LED2St = LOW; LED3St = LOW; LED4St = LOW; break;
      }
    }
    // LED states are cleared every cycle
    digitalWrite(LED0, LED0St); LED0St = HIGH;
    digitalWrite(LED1, LED1St); LED1St = HIGH;
    digitalWrite(LED2, LED2St); LED2St = HIGH;
    digitalWrite(LED3, LED3St); LED3St = HIGH;
    digitalWrite(LED4, LED4St); LED4St = HIGH;
  }
}

// ----------------------------------------------------------------------

void delayLoop(unsigned long zDel) {
  // loop for a delay period
  zDel = millis() + zDel;
  while (millis() < zDel) {
    if (ESC) {return;}
    loop();
  }  
}

// ----------------------------------------------------------------------

void loopWhiletgtCnt() {
  // used to call loop during a 'Move' function
//  Serial.println(F("loopWhiletgtCnt"));
  while (tgtCnt > 0) {
    if (ESC) {return;}
    loop();
  }
}

// ---------------------------------------------------------------------------------

void loopWhileWaiting() {
  // this loop is called from VL53L0X_readRangeContinuousMillimeters(void) whilst
  // waiting for a measurement. It is called every 500 us. So put multi-tasking
  // code here, rather than waiting for a measurement to complete.
  loop();
}

// ----------------------------------------------------------------------

void loopWhileWalk(int zWalkCnt) {
  // walk for a number of WalkCnt clicks
//  Serial.println(F("loopWhileWalk"));
  int zWCL;
  while (zWalkCnt > 0) {
    if (ESC) return;
    zWCL = WalkCnt; loop();
    if (zWCL != WalkCnt) {zWalkCnt--;}  
  }
}

// ----------------------------------------------------------------------

void loopWhileWalkEcho(int zWalkCnt) {
  // walk for a number of WalkCnt clicks whilst reading the radar distance
//  Serial.println(F("loopWhileWalk"));
  int zWCL;
  while (zWalkCnt > 0) {
    if (ESC) return;
    zWCL = WalkCnt; loop();
    if (zWCL != WalkCnt) {
      // a walk step has occured in the main loop
      zWalkCnt--;
      // check Range reading and exit if an object is detected
      if (Range <= 325) {break;}
      // slowly increase walking speed
      if (walkInterval > 15000) {walkInterval = walkInterval -100;}
    }  
  }
}

// ----------------------------------------------------------------------

void runPOST() {
  // called during start-up
  Serial.println(F("\n\nQuadruped 'Auto' Release v0.01"));
  Serial.println(F("Starting POST..."));
  Serial.println(F("Testing LEDs..."));
  digitalWrite(LED0, LOW); delay(200); digitalWrite(LED0, HIGH); LED0St= HIGH;
  digitalWrite(LED1, LOW); delay(200); digitalWrite(LED1, HIGH); LED1St= HIGH;
  digitalWrite(LED2, LOW); delay(200); digitalWrite(LED2, HIGH); LED2St= HIGH;
  digitalWrite(LED3, LOW); delay(200); digitalWrite(LED3, HIGH); LED3St= HIGH;
  digitalWrite(LED4, LOW); delay(200); digitalWrite(LED4, HIGH); LED4St= HIGH;
  Serial.println(F("Reseting Servos..."));
  HeadToRest(); // centre head servo
  GoToRest(50); autoOFF = 5;
  Serial.println(F("POST complete!"));
}

// ----------------------------------------------------------------------

void setDefaults() {
  // load default values
  atTgt = false; // true if servos are at target values
  autoHeadOFF = 0; // if set > 0 then counts down to head servo switch OFF, if -1 disabled
  autoOFF = 0; // if set > 0 then counts down to servo switch off
  autoRest = autoRestMax; // returns to a rest position after a timeout
  BattAv = 915; // average battery voltage
  BattCnt = 0; // battery failed 1s time-out counter
  BattSum = BattMin << 4; //  pre-set cumulative battery voltage
  ESC = false; // true to escape all functions
  HeadActive = false; // true if head is in scan mode for tasks
  HeadDir = 0; // head movement direction, +1 == RH, -1 == LH
  headInterval = 20000; // main loop head movement interval in microseconds
  HeadSubTask = 0; // head movement sub task pointer
  HeadTask = 0; // head movement task pointer
  HeadTaskRun = false; // set to true during head tasks
  LedMsk = 0; // a mask used to overide LedMode functions
  LedMskCnt = 0; // a counter used with LedMsk overide time-out
  LedTick = false; // set true every 'Walk' cycle
  LTOF_On; // > 0 when laser ranging is enabled
  interval = 20000; // main loop interval in microseconds, 50Hz
  LedCnt = 0; // counter used in LED sequencing
  LedMode = 0; // a value > 0 determines LED flash sequence; if = 0 disabled
  LegDn = LegDwnMin; // dynamic angle for leg down in 'Move Engine'
  LegUp = LegUpMin; // dynamic angle for raised down in 'Move Engine'
  LegUpFwd = LegUpMin; // dynamic angle for forward leg raised in 'Move Engine'
  LTOF_On = 0; // 1 or 2 when laser ranging is enabled, default is 0 == OFF
  MainRun = false; // set true when a MainTask is running to prevent re-entry
  MainTask = 0; // main task pointer
  MainTaskNext = 0; // the task to run after an arrest; default is 0
  nextLoop = 0; // flag used to trigger a head task during the next loop if > 0
  Once = 0; // counter for demo moves
  QuadActive = false; // true if active, otherwise false
  servoEn = false; // set true if servos are attached
  servoHeadEn = false; // set true if head servo is attached
  servoVal[0] = Rst1; // value sent to servos
  servoVal[1] = Rst2; // value sent to servos
  servoVal[2] = Rst3; // value sent to servos
  servoVal[3] = Rst4; // value sent to servos
  servoVal[4] = Rst5; // value sent to servos
  servoVal[5] = Rst6; // value sent to servos
  servoVal[6] = Rst7; // value sent to servos
  servoVal[7] = Rst8; // value sent to servos
  servoTgt[0] = Rst1; // target values for servos
  servoTgt[1] = Rst2; // target values for servos
  servoTgt[2] = Rst3; // target values for servos
  servoTgt[3] = Rst4; // target values for servos
  servoTgt[4] = Rst5; // target values for servos
  servoTgt[5] = Rst6; // target values for servos
  servoTgt[6] = Rst7; // target values for servos
  servoTgt[7] = Rst8; // target values for servos
  SubTask = 0; // sub task pointer
  swCnt = 0; // button switch counter
  swLastState = HIGH; // previous state of button switch, HIGH/LOW
  swState = HIGH; // stat of read button switch pin
  swTimer = 0; // timer used to detemine button sequences
  tgtCnt = 0; // counter used in automatic servo move to target actions
  Turn = false; // default = false; if true turns continuously until = false
  Walk = 0; // if >0 then walk in one of four directions
  Walked = 0; // counter used to return robot to a given position
  walkInterval = 20000; // main loop walking interval in microseconds
  WalkLast = 0; // previousl Walk value if any
  WalkLftDrive = 128; // drive factor for left-hand side, 128 = max, 0 = min
  WalkMax = 4; // speed factor 4 - 12
  WalkRgtDrive = 128; // drive factor for right-hand side, 128 = max, 0 = min
  WalkSpeed = 1; // value used to determine WalkMax, 1 - 5
  WalkSpeedTO = 0; // Time-Out counter for WalkSpeed auto-reduction, counts to zero
}

// ----------------------------------------------------------------------

void synchLoopTimers() {
  // reset main loop timers
  headMicros = micros() + headInterval; // set head loop timer
  nextMicros = micros() + interval; // set task loop timer
  walkMicros = micros() + walkInterval; // set walk loop timer
}


