// ################################################################################
//
//  Quadruped Auton ESP32 vR4
//
//  Released:  17/08/2025
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    microcontroller:  ESP32 Dev Module  v3.0.7

    Board:  ESP32 Dev Module

    This code controls a quadruped robot, giving it the ability to walk and respond to
    commands from a Wii controller over ESPNOW WiFi. The robot has a rotating head
    which contains a laser ranging device. RGB LEDs are used to provide good visual
    effects for ranging and motion.

    The code is multi-tasking and does the following:
    * Periodically reads button switch SW0 and responds to actions
    * Performs a range of RGB LED animationss
    * Controls a VL53L0X laser distance sensor mounted in the head
    * Two move engines, which control the head and leg servos
    * Able to connect to multiple Wii WEMOS D1 Transcievers, MAC addresses stored in code

    The quadruped is controlled via a Wii Nunchuk controller, over a 2.4GHz wireless
    link. After RESET the robot must be brought into the 'Active' state by pressing
    and holding the Nunchuk 'C' button, before any movements can be controlled. The
    following control features are included:

    C(H)      If C is held for >1 second whilst inactive, enter the active state.
    C(H)+Z(H) If C and Z are held for >2 seconds in the active state, enter the
              inactive resting state.
    C         Each C press will increase the responsiveness of the robot from 1–5 (max).
    C(H)+Joy  If C is held in min speed the leg up height will be increased to enable
              it to climb over small objects and uneven surface.
    Z         Each Z press will decrease the responsiveness of the robot from 1–5 (max).
    Z(H)+Joy  Holding Z will change the right/left walking modes from turning to walking
              sideways, forwards will become a ‘bow’ and reverse will become a ‘Bye wave’.
    
    The servo motors must be calibrated during construction, and the values noted
    need to be inserted in the angle definitions included in this code. Functions
    that are commented-out, are not needed but you may find useful to use in your
    code.
    
    IMPORTANT - Espressif changed the way ESP-NOW works from v3.0, which broke the original
    code and caused compiler errors. This version has been modified to work with v3.0+.
    Enjoy!
*/
// Declare Libraries
#include <Wire.h>               // I2C library
#include <FastLED.h>            // Neopixel library     v3.9.3
#include <ESP32Servo.h>         // servo library        v3.0.5

#include <esp_now.h>            // ESP-NOW WiFi link library
#include <WiFi.h>               // ESP WiFi library

// Configuration
String Release = "Quadruped Auton ESP32 (R4)";
String Date = "17/08/2025";

// Define a task for core 0
TaskHandle_t Core0;

// Define servos constants gained from your servo calibration work
// All servos are different and leaver arm positions will not match
// These values must be those taken from PWM tests performed on your robot
// If you ever change a servo you will need to re-calibrate those values
#define Ang1_45 960     // front left shoulder
#define Ang1_135 1940   // front left shoulder
#define Ang2_65 2267    // front left leg pinch
#define Ang2_147 1357   // front left leg belly down
#define Ang3_45 1916    // front right shoulder
#define Ang3_135 935    // front right shoulder
#define Ang4_65 680     // front right leg pinch
#define Ang4_147 1586   // front right leg belly down
#define Ang5_45 920     // rear right shoulder
#define Ang5_135 1916   // rear right shoulder
#define Ang6_65 2234    // rear right leg pinch
#define Ang6_147 1351   // rear right leg belly down
#define Ang7_45 2097    // rear left shoulder
#define Ang7_135 1174   // rear left shoulder
#define Ang8_65 713     // rear left leg pinch
#define Ang8_147 1598   // rear left leg belly down
#define Head_0 1552     // mid point head angle, zero degrees
#define Head_60N 890    // -60 degrees, lower limit
#define Head_60P 2203   // +60 degrees, upper limit
#define HeadPin 2       // head servo is connected to GPIO2
#define LegDwnMin 110   // min angle for leg down in 'Move Engine' Walk
#define LegDwnMax 130   // max angle for leg down in 'Move Engine' Walk
#define LegUpMin 130    // min angle for leg raised in 'Move Engine' Walk
#define LegUpMax 153    // max angle for leg raised in 'Move Engine' Walk
#define Max1 2093       // max value for servo 1
#define Max2 2267       // max value for servo 2
#define Max3 1916       // max value for servo 3
#define Max4 2400       // max value for servo 4
#define Max5 2069       // max value for servo 5
#define Max6 2234       // max value for servo 6
#define Max7 2097       // max value for servo 7
#define Max8 2400       // max value for servo 8
#define Min1 960        // min value for servo 1
#define Min2 500        // min value for servo 2
#define Min3 809        // min value for servo 3
#define Min4 680        // min value for servo 4
#define Min5 920        // min value for servo 5
#define Min6 500        // min value for servo 6
#define Min7 1042       // min value for servo 7
#define Min8 713        // min value for servo 8
#define Rst0 1366       // reset value for servo 0 - Head_0 - 20
#define Rst1 1450       // reset value for servo 1 - front left hip
#define Rst2 1990       // reset value for servo 2 - front left leg
#define Rst3 1426       // reset value for servo 3 - front right hip
#define Rst4 956        // reset value for servo 4 - front right leg
#define Rst5 1418       // reset value for servo 5 - rear right hip
#define Rst6 1965       // reset value for servo 6 - rear right leg
#define Rst7 1636       // reset value for servo 7 - rear left hip
#define Rst8 982        // reset value for servo 8 - rear left leg

// define VL53LOX constants
#define ADDRESS_DEFAULT 0b0101001 // VL53L0X address: 29Hex
#define calcMacroPeriod(vcsel_period_pclks) ((((uint32_t)2304 * (vcsel_period_pclks)* 1655) + 500) / 1000)
#define checkTimeoutExpired() (io_timeout > 0 && ((uint16_t)millis() - timeout_start_ms) > io_timeout)
#define decodeVcselPeriod(reg_val) (((reg_val) + 1) << 1)
#define encodeVcselPeriod(period_pclks) (((period_pclks) >> 1) - 1)
#define startTimeout() (timeout_start_ms = millis())
#define XSHUT 14  // GPIO14 is connected to the XSHUT pin

// Declare register list from VL53L0X.h API, ordered as listed there
#define SYSRANGE_START 0x00
#define SYSTEM_THRESH_HIGH 0x0C
#define SYSTEM_THRESH_LOW 0x0E
#define SYSTEM_SEQUENCE_CONFIG 0x01
#define SYSTEM_RANGE_CONFIG 0x09
#define SYSTEM_INTERMEASUREMENT_PERIOD 0x04
#define SYSTEM_INTERRUPT_CONFIG_GPIO 0x0A
#define GPIO_HV_MUX_ACTIVE_HIGH 0x84
#define SYSTEM_INTERRUPT_CLEAR 0x0B
#define RESULT_INTERRUPT_STATUS 0x13
#define RESULT_RANGE_STATUS 0x14
#define RESULT_CORE_AMBIENT_WINDOW_EVENTS_RTN 0xBC
#define RESULT_CORE_RANGING_TOTAL_EVENTS_RTN 0xC0
#define RESULT_CORE_AMBIENT_WINDOW_EVENTS_REF 0xD0
#define RESULT_CORE_RANGING_TOTAL_EVENTS_REF 0xD4
#define RESULT_PEAK_SIGNAL_RATE_REF 0xB6
#define ALGO_PART_TO_PART_RANGE_OFFSET_MM 0x28
#define I2C_SLAVE_DEVICE_ADDRESS 0x8A
#define MSRC_CONFIG_CONTROL 0x60
#define PRE_RANGE_CONFIG_MIN_SNR 0x27
#define PRE_RANGE_CONFIG_VALID_PHASE_LOW 0x56
#define PRE_RANGE_CONFIG_VALID_PHASE_HIGH 0x57
#define PRE_RANGE_MIN_COUNT_RATE_RTN_LIMIT 0x64
#define FINAL_RANGE_CONFIG_MIN_SNR 0x67
#define FINAL_RANGE_CONFIG_VALID_PHASE_LOW 0x47
#define FINAL_RANGE_CONFIG_VALID_PHASE_HIGH 0x48
#define FINAL_RANGE_CONFIG_MIN_COUNT_RATE_RTN_LIMIT 0x44
#define PRE_RANGE_CONFIG_SIGMA_THRESH_HI 0x61
#define PRE_RANGE_CONFIG_SIGMA_THRESH_LO 0x62
#define PRE_RANGE_CONFIG_VCSEL_PERIOD 0x50
#define PRE_RANGE_CONFIG_TIMEOUT_MACROP_HI 0x51
#define PRE_RANGE_CONFIG_TIMEOUT_MACROP_LO 0x52
#define SYSTEM_HISTOGRAM_BIN 0x81
#define HISTOGRAM_CONFIG_INITIAL_PHASE_SELECT 0x33
#define HISTOGRAM_CONFIG_READOUT_CTRL 0x55
#define FINAL_RANGE_CONFIG_VCSEL_PERIOD 0x70
#define FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI 0x71
#define FINAL_RANGE_CONFIG_TIMEOUT_MACROP_LO 0x72
#define CROSSTALK_COMPENSATION_PEAK_RATE_MCPS 0x20
#define MSRC_CONFIG_TIMEOUT_MACROP 0x46
#define SOFT_RESET_GO2_SOFT_RESET_N 0xBF
#define IDENTIFICATION_MODEL_ID 0xC0
#define IDENTIFICATION_REVISION_ID 0xC2
#define OSC_CALIBRATE_VAL 0xF8
#define GLOBAL_CONFIG_VCSEL_WIDTH 0x32
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_0 0xB0
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_1 0xB1
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_2 0xB2
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_3 0xB3
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_4 0xB4
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_5 0xB5
#define GLOBAL_CONFIG_REF_EN_START_SELECT 0xB6
#define DYNAMIC_SPAD_NUM_REQUESTED_REF_SPAD 0x4E
#define DYNAMIC_SPAD_REF_EN_START_OFFSET 0x4F
#define POWER_MANAGEMENT_GO1_POWER_FORCE 0x80
#define VHV_CONFIG_PAD_SCL_SDA__EXTSUP_HV 0x89
#define ALGO_PHASECAL_LIM 0x30
#define ALGO_PHASECAL_CONFIG_TIMEOUT 0x30

// define VL53L0C data structures
struct SequenceStepEnables {boolean tcc, msrc, dss, pre_range, final_range;};
struct SequenceStepTimeouts {
      uint16_t pre_range_vcsel_period_pclks, final_range_vcsel_period_pclks;
      uint16_t msrc_dss_tcc_mclks, pre_range_mclks, final_range_mclks;
      uint32_t msrc_dss_tcc_us, pre_range_us, final_range_us;};

// declare VL53L0X variables
uint8_t address = ADDRESS_DEFAULT;
bool did_timeout;
uint16_t io_timeout;
uint8_t last_status;    // status of last I2C transmission
uint32_t measurement_timing_budget_us;
uint16_t Range;         // 16-bit modified range value used by the code
uint16_t RangeRaw;      // 16-bit range value read from VL53L0X sensor
uint16_t RangeRawLast;  // 16-bit previous range value read from VL53L0X sensor
uint8_t stop_variable;  // read by init and used when starting measurement; is StopVariable field of VL53L0X_DevData_t structure in API
uint16_t timeout_start_ms;
enum vcselPeriodType {VcselPeriodPreRange, VcselPeriodFinalRange};
void VL53L0X_startContinuous(uint32_t period_ms = 0);
inline void VL53L0X_setTimeout(uint16_t timeout) { io_timeout = timeout; }
bool VL53L0X_timeoutOccurred(void);
bool VL53L0X_getSpadInfo(uint8_t * count, bool * type_is_aperture);

// Define general constants
#define AngDifMax 250     // maximum sweep angle from centre difference, in microseconds
#define AngDifMin 100     // minimum sweep angle from centre difference, in microseconds
#define autoRestMax 32000 // autoRest max counter value
#define BatCal 383.9      // calibrated voltage multiplying factor
#define BatCritical 2534  // critical battery threshold @6.6v, default = 2672, set == 0 to ignore
#define BatMax 3148       // A0 for fully charged battery voltage, == 8.2v
#define BatPin 36         // analog input pin ADC0 on GPIO36
#define BatWarn 2688      // battery threshold of 7.0v gives low warning
#define DbLLX 117         // JoyX deadband lower limit
#define DbULX 137         // JoyX deadband Upper limit
#define DbLLY 118         // JoyY deadband lower limit
#define DbULY 138         // JoyY deadband Upper limit
#define HeadSpd 24        // speed used in autonomous mode
#define LED_Pin 5         // assign GPIO5 for RGB LEDs
#define LtofDet 350       // max range for target detection in head scanning
#define LtofLimit 500     // Range max value is always limited to this figure
#define LtofMax 400       // out of range value for head scanning
#define LtofMin 55        // minimum range value
#define LTOF_Offset 50    // range correction factor, subtracted from measurement
#define LtofStop 140      // range value at which autonomous walking stops
#define NumLEDs  5        // number of neopixel LEDs
#define SpdRmp 4          // walking speed acceleration damping factor
#define sw0Pin 15         // left switch SW0 assigned to GPIO15
#define TESTmode false    // true during assembly, false for normal use
#define TEXT_ALIGN_LEFT 0
#define TEXT_ALIGN_RIGHT 1
#define TEXT_ALIGN_CENTER 2
#define TEXT_ALIGN_CENTER_BOTH 3
#define WalkSpeedToMax 250; // set WalkSpeed auto-reduction as 5 seconds
#define XSHUT_PIN 14      // GPIO14 controls VL53L1X XSHUT HIGH enable pin
 
// create a FastLED instance and define the total number of LEDs on the robot
CRGB LED[NumLEDs];  // LED array on front of robot
CRGB TMP[NumLEDs];  // temp LED array for animation effects

// define a data structure matching an LED, used in LED pattern generation
struct LED_rgb {
  uint8_t r;
  uint8_t g;
  uint8_t b;
};
struct LED_rgb LEDX;      // create LEDX as a remporary RGB store

// create servo instances
Servo servoInst[9]; // define head and leg servos

// define servo arrays
// there are 9 servos in this robot, 8 legs (S1 - S8) + 1 head (S0)
bool servoAtt[9] = {false,false,false,false,false,false,false,false,false};  // true if servo is attached
int16_t servoLast[9]; // previous value sent to the servo
int16_t servoMax[9];  // max values in manual adjustment
int16_t servoMem[9];  // record of values sent to servos, used in fidgeting
int16_t servoMin[9];  // min values in manual adjustment
//                     00,01,02,03,04,05,06,07,08 servo ref numbers 
int16_t servoPins[] = {02,25,33,18,19,17,16,26,27};  // servo output pins assigned
int16_t servoVal[9];  // value sent to servos
int16_t servoTgt[9];  // target values for servos

// you need to build a Wii Transceiver and get its MAC address
uint8_t broadcastAddress[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

// 250 byte max for Tx/Rx data
// It will be used to create the Tx/Rx data structures
typedef struct ESPNOW_Buff250 {
    char ESPdata[250];
} ESPNOW_Buff250;

// Create a 250 byte Tx buffer
ESPNOW_Buff250 Tx_Buff;

// Create a 250 byte receive
ESPNOW_Buff250 Rx_Buff;

// Declare and initialise global variables
String Any$;                  // any temp string
int16_t AngDif, AngNeg, AngPos, AngRef; // temporary head angles
////int16_t Any;                  // any temp value
//int16_t AnyCnt;               // general counter
int16_t anyDel;               // any delay value
int16_t anyFor;               // any for loop value
int anyInt;                   // any temporary integer value
long AnyLong;                 // any temp value
unsigned long anyMilli;       // any millisecond timer
int16_t anyRnd;               // any random integer
int16_t anyVal;               // any temp variable
bool atHeadTgt;               // true if head servo is at target value
bool atTgt;                   // true if servos are at target values
int16_t autoHeadOFF;          // if set > 0 then counts down to head servo switch OFF, if -1 disabled
int16_t automCnt;             // delay counter used in autonomous mode
int16_t automRnd;             // random walking distance in autonomous mode
int16_t automTurn;            // flag used in autonomous turning
int16_t autoOFF;              // if set > 0 then counts down to servo switch OFF, if < 1 disabled
int16_t autoRest;             // returns to a rest position after a timeout
int32_t BatAvg;               // preload averaging function with high value
int32_t BatDist;              // used battery discharge testing
int32_t BatSum;               // preload averaging sum with 20x max value
int32_t BatVol;               // instantaneous battery voltage
bool Blink;                   // LED blink clock
int16_t BlinkCnt;             // counter for LED Blink clock
uint8_t Bright;               // FastLED brightness
uint8_t Brightness;           // Display brightness
int CalcCnt;                  // tracks the number of head calculations in autonomous mode
int16_t C_Cnt;                // counter used in 'C' button detection
byte C_Dn;                    // counter for 'C' button down period
byte Col_B,Col_G,Col_R;       // current colours
byte C_Up;                    // counter for 'C' button up period
char cmdMode;                 // command mode
int16_t cmdSgn;               // cmdVal sign, 0 or -1
char cmdType;                 // command mode type
int cmdVal;                   // value associated with a cmdType
int16_t CZ;                   // received button values
bool CZ_Wait;                 // flag used for button release wait function
char Deg = 176;               // ASCII for "°"
bool DispClr;                 // set == true to clear the memory contents
int16_t DispCnt;              // counter used in display updating
byte DispCX,DispCY;           // display cursor coordinates
int16_t DispDel;              // >0 display delay counter, stops updates over I2C
int16_t DispDelGbl;           // global DispDel offset, used to adjust display frame rates
bool DispDisp;                // set = true to immediately display memory contents
bool DispLock;                // if == true then don't allow display mode changes
int16_t DispMode;             // the type of display for a given mode
bool DispMon;                 // == true if connected to OLED Mirror app
int16_t DispNext;             // the next DispMode to be used after delay
int16_t DispOveride;          // if DispLock == true then use this overide mode
int16_t DispTx;               // counter used with display mirror app
int16_t DmB,DmX,DmY;          // button and X,Y values associated with a display monitor mouse click
int16_t DM_Min,DM_Max;        // min/max Display Mode pointer values
int16_t ESC;                  // > 0 to escape all functions
bool Echo;                    // if true send print messages
int16_t ESP_NOW_BA;           // broadcast address pointer, -1 to start with
bool ESP_NOW_Init;            // == true once ESP-NOW has been initiaiised
gpio_num_t GPIOpin;           // used to read GPIO output register
bool HeadActive;              // = true if head is in scan mode for tasks
bool HeadAdj;                 // flag used in head scanning 
int HeadAng;                  // servoVal[0] converter to degrees
int16_t HeadCnt;              // counter used in head movements
int16_t HeadCyc;              // cycle counter used in head movements
int16_t HeadDel;              // head movement delay
int16_t HeadDir;              // head movement direction, +1 == RH, -1 == LH
long HeadErr;                 // value used in head angle adjustment
bool HeadErrSt;               // trip flag used in head scanning
long HeadErrTrip;             // error tripping value used in head angle adjustment
long HeadErrTrk;              // error tracking value used in head angle adjustment
unsigned long headInterval;   // main loop head movement interval in microseconds
unsigned long headMicros;     // main loop head movement timer in microseconds
int16_t HeadMode;             // pointer used in head scanning
int16_t HeadPause;            // 20ms counter to temporarily pause head movement
int16_t HeadSubTask;          // head movement sub task pointer
int16_t HeadTask;             // head movement task pointer
int16_t HeadTaskLast;         // previous head movement task pointer
int16_t HeadTaskNext;         // head movement next task pointer
bool HeadTaskRun;             // set to true during head tasks
int16_t HeadTgtCnt;           // counter used in automatic head servo move to target actions
bool I2C_Err;                 // Flag used to detect VL53L0X errors
unsigned long interval;       // main loop interval in microseconds
bool JoyActive;               // flag used to prevent stack overflow during moves
int16_t JoyD;                 // direction value 0 - 8
int16_t JoyDL;                // direction 1 counter
int16_t JoyMode;              // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
int JoySpd;                   // speed vector = max(JoyX,JoyY)
int16_t JoyV;                 // joystick vector  = SQRT(SQ(JX) + SQ(JY))
int16_t JoyX;                 // received value
int JoyXV;                    // converted value 0 - 127
int16_t JoyXC;                // received value offset
int16_t JoyY;                 // received value
int16_t JoyYC;                // received value offset
int JoyYi;                    // tracking filter for received JoyY value
int JoyYV;                    // converted value 0 - 128
bool JoyYT;                   // tracking filter flag; true if not on track
char keyChar;                 // any keyboard character
int16_t keyVal;               // any keyboard value
int16_t LastTask;             // record of the previous MainTask
int16_t LedCnt;               // counter used in LED sequencing
int16_t LED_Del;              // delay timer used in LED functions
int16_t LedLast;              // previous task when change set
int16_t LedMem;               // previous recorded LED variable, default -1
int16_t LedMsk;               // a mask used to overide LedMode functions
int16_t LedMskCnt;            // a counter used with LedMsk overide time-out
int16_t LED_Max;              // max value used in LED functions
int16_t LedMode;              // a value > 0 determines LED flash sequence; if = 0 disabled
int16_t LedNext;              // next LedMode to be used after a priotity task
bool LedNop;                  // if == true then skip LED modes
unsigned long LED_Period;     // LED timer period, default = 20ms
int16_t LedRstCnt;            // counter used in LED auto reset
bool LEDshow;                 // LED show flag
int16_t LED_Task;             // LED task pointer
bool LedTick;                 // set true every 'Walk' cycle
unsigned long LED_Timer;      // 20+ms LED timer
byte LEDVal;                  // values used in LED tasks
int16_t LegDn;                // dynamic angle for leg down in 'Move Engine'
int16_t LegSt2,LegSt4,LegSt6,LegSt8; // up down states of legs
int16_t LegUp;                // dynamic angle for leg raised in 'Move Engine'
int16_t LegUpFwd;             // dynamic angle for forward leg raised in 'Move Engine'
bool LTOFcalc;                // if == true then calculate speed of ranged object in m/s
int16_t LTOF_ESC;             // a time-out escape counter for the VL53L0X
int16_t LTOFfps;              // ranger sensor measurement rate in fps
int16_t LTOF_On;              // > 0 when laser ranging is enabled
int LTOFperiod;               // range sensor inter-measurementg period in millis()
int16_t LTOFspd10;            // speed of ranged target in m/s x 10
int LTOFt0,LTOFt1;            // range sensor timers for fps
unsigned long LTOF_Wait;      // delay used to prevent wasted time in LTOF readings
int16_t MainDel;              // MainTask delay counter, default = 0
bool MainRun;                 // set true when a MainTask is running to prevent re-entry
int16_t MainTask;             // MainTask pointer used to run tasks
int16_t MainTaskNext;         // the task to run after an arrest; default is 0
int16_t MainTaskRef;          // record of MainTask pointer when mode was set
int16_t MoveState;            // move state machine pointer
String MoveType;              // type of move taking place
String myMAC;                 // MAC address if this device
unsigned long next20ms;       // 20ms loop timer
unsigned long next40ms;       // 40ms loop timer
byte nextLoop;                // flag used to trigger a head task during the next loop if > 0
int16_t Once;                 // counter for demo moves
int16_t Ping;                 // Rx 'ping' counter
unsigned long print40ms;      // 40ms variable print timer
int16_t PrintTgt;             // 0 == serial port, 1 == WiFi ESP-NOW
String PrintTx = "";          // printed strings
int16_t PWM_Freq;             // servo PWM frequency, default = 100;
bool QuadActive; // true if active, otherwise false
int16_t RangeAng;             // angle at which minimum range was measured
int16_t RangeCentre;          // distance value looking forward
int16_t RangeDiff;            // difference in RangeRaw from RangeLast
bool RangeEn;                 // == true if VL53L1X ranging is active
bool RangePnt;                // true when a centre range value is recorded, or extremes of sweep
int16_t RangeCentreTracker;   // slugged distance value looking forward
long RangeIntL;               // L-H integrator used in head centring
long RangeIntR;               // R-H integrator used in head centring
uint16_t RangeLast;           // previous range measurement
int16_t RangeLeft;            // distance value lookin left
bool RangeLL;                 // == true if range < RangeMin
int RangeMax;                 // the current limited to Range measurements
int RangeMin;                 // the current lower limit of Range measurements
int16_t RangeMinAng;          // angle at which RangeMin occured
int16_t RangeMinMem;          // RangeMin at end of each half sweep
int16_t RangeRight;           // distance value lookin right
int16_t RangeTrig;            // range trigger point used in head scanning
bool RangeUL;                 // == true if range exceeds RangeMax
int16_t R0,R1,R2,R3,R4,R5,R6,RM; // range values for backaway and target tracking
bool readWiiCall;             // flag prevents call stack overflow
int16_t Rx_Chk;               // Rx checksum
int16_t RxCZ;                 // buffered received CZ value
int16_t RxJoyX;               // buffered received JoyX value
int16_t RxJoyY;               // buffered received JoyY value
int16_t Rx_len;               // length of WiFi received data block
int16_t Rx_Pnt;               // data pointer user by Rx state machine
bool RxRec;                   // set true when a valid frame of data is received
bool RxRecvEvent;             // set == true when a OnDataRecv() call back event has occured
int16_t RxState;              // receiver state machine state
int16_t Rx_Task;              // task pointer user by Rx state machine
int16_t RTxTimeout;           // counter used to auto-reset if WiFi fails
int16_t RxVal;                // value received from serial Rx
byte RxWiFi[6];               // array holding 6 bytes of Wii received data
bool SerialRx;                // == true if any character has been received
int16_t servoDelta;           // temporary delta value used in target movements
bool ServoEn;                 // false = OFF, true = ON enabled status
int16_t ServoInc;             // servo angle increment pointer
int16_t ServoPnt;             // servo target pointer for manual control
int16_t servoTgtLft;          // leftt most target PWM angle
int16_t servoTgtRht;          // right most target PWM angle
bool ShowSkip;                // flag used in a FastLED show() event
bool Sit;                     // used in backaway tasks
int16_t SubLast;              // previous sub task pointer
int16_t SubTask;              // sub task pointer
int16_t SubVal;               // anyvalue used in a subroutine
int16_t sw0Cnt;               // button switch counter
int16_t sw0DwnTime;           // button pressed down time
bool sw0LastState;            // previous state of button switch, HIGH/LOW
bool sw0_Nop;                 // == true to block SW0 read tasks, default = false
bool sw0State;                // state of read button switch pin
int16_t sw0Timer;             // timer used to detemine button sequences
unsigned long t0;             // general timer
unsigned long t1;             // general timer
int16_t Task10ms;             // 10ms subtask timer
int16_t Task20ms;             // 20ms subtask timer
int16_t Task40ms;             // 40ms subtask timer
bool TEST;                    // normally false, set = true for assembly test mode
byte Testmode;                // defines the state of TEST
int16_t TestNext;             // task pointer used by the TEST mode
int16_t TestTask;             // task pointer used by the TEST mode
int16_t TgtCnt;               // counter used in automatic servo move to target actions
bool Tick20ms;                // if == true then run LED tasks at 20ms rate
bool TrgtDet;                 // true if a target has been detected in autonomous mode
bool Turn;                    // default = false; if true turns continuously until = false
int TurnCnt;                  // counter used in neutral turning, inversely proportional to speed
bool Turning;                 // set = true so that ESC can be used to break out of a turn cycle
int16_t Tx_PingCnt;           // 1 second Tx ping timer
bool USB;                     // == true if battery voltage is below critical value at outset
unsigned long VL53L0Xms;      // 1ms timer for reading VL53L0X device
int16_t VL53L0X_Show;         // timer used to hold off VL53L0X when using FastLED
int16_t VL53L0X_Skip;         // timer used to hold off VL53L0X when running sub-tasks
int16_t Walk;                 // if >0 then walk in one of four directions
long WalkCnt;                 // a counter which controls the walking process
long Walked;                  // counter used to return robot to a given position
unsigned long walkInterval;   // main loop walking interval in microseconds
int16_t WalkLast;             // previousl Walk value if any
int WalkLftDrive;             // drive factor for left-hand side, 128 = max, 0 = min
byte WalkMax;                 // speed factor 4 - 15
unsigned long walkMicros;     // main loop walking task interval timer in microseconds
int16_t WalkNext;             // used to store the next walking mode
int16_t WalkSpeed;            // value used to determine WalkMax
int16_t WalkSpeedTO;          // Time-Out counter for WalkSpeed auto-reduction, counts to zero
int WalkRgtDrive;             // drive factor for right-hand side, 128 = max, 0 = min
int16_t WiFiCntC;             // C button counter for WiFi enable
int16_t WiFiConCnt;           // counter used in connection retry times
bool WiFiConnected;           // == true if a WiFi connection has been established
bool WiFiEn;                  // true when WiFi is enabled with prolonged 'Z' button activity
int16_t WiFiPing;             // Rx counter used to maintain connected state
int16_t WiFiRestCnt;          // counter used to goi to rest if no WiFi demands
bool WiFiRx;                  // true when a block of data has been received
long WiFiRxBytes;             // number of bytes received over the WiFi link
int16_t WiFiRxErr;            // number of errors received over the WiFi link
bool WiFiRxRec;               // true when a block of data has been successfully received
byte WiFiRxMacAddr[6];        // senders MAC address, which could be broadcast FF:FF:FF:FF:FF:FF
int16_t WiFiTryCnt;           // WiFi try to connect count on error count
int16_t WiFiTryNum;           // WiFi try to connect total count
bool WiFiTryOnce;             // == true if first pass of trying to connect
bool WiFiTx;                  // true when a block of data has been sent successfully
int16_t WiFiTx_CB;            // == 1 call back is clear to send, == -1 resend
int16_t WiFiTx_Chk;           // Tx checksum
int16_t WiFiTx_len;           // >0 when WiFi Tx buffer contains data to be sent
long WiFiTxBytes;             // number of bytes received over the WiFi link
int16_t WiFiTxErr;            // WiFi data error
int16_t WiFiTxErrCnt;         // WiFi Tx error count, leads to disconnection
byte WiFiTxMacAddr[6];        // transmit MAC address, which should match myMAC address
int16_t WiiType;              // device type, 0 == Nunchuk, 1 == Classic
int WL_C, WR_C;               // walk engine counters
int WL_CM, WR_CM;             // walk engine counters modified by steering
int WL_I, WR_I;               // walk engine counter increments
int W_Up;                     // leg up angle for walk engine
byte Z_Dn;                    // button pressed state; 0 == UP, >= 1 == Down

// ---------------------------------------------------------------------------------

void setup() {
  // Setup code, runs once, immediately after reset
  // Allow allocation of all timers
  ESP32PWM::allocateTimer(0);
  ESP32PWM::allocateTimer(1);
  ESP32PWM::allocateTimer(2);
  ESP32PWM::allocateTimer(3);

  pinMode(sw0Pin,INPUT_PULLUP);                             // button switch sw0 Pin
  pinMode(BatPin,INPUT);                                    // ADC0  battery divider
  pinMode(XSHUT_PIN,OUTPUT); digitalWrite(XSHUT_PIN,LOW);   // switch VL53L0X OFF
  pinMode(HeadPin,OUTPUT); digitalWrite(HeadPin,LOW);       // set head PWM LOW on reset
  for (int zS = 1; zS <= 8; zS++) {
      pinMode(servoPins[zS],OUTPUT); digitalWrite(servoPins[zS],LOW); // set leg PWM LOW on reset
  }

  setDefaults();        // reset variables

  Wire.begin();         // initialise the I2C interface

  Serial.begin(115200); // baud rate for serial monitor USB Rx/Tx comms

  // initialise the FastLED component
  FastLED.addLeds<WS2812B, LED_Pin, GRB>(LED, NumLEDs);
  FastLED.setBrightness(Bright);
    
  // initialise WiFi link
  WiFi.mode(WIFI_STA);
  // new code for ESP32 lib v3.0+
  while (!WiFi.STA.started()) {delay(10);}
  // Once started we can get the micros MAC address
  myMAC = WiFi.macAddress();
  Serial.println("my MAC= " + String(myMAC));
  // WiFi.disconnect(); // I don't think we need this

  // initialise the ESP-NOW link for the first time
  Init_ESP_NOW();
  
  RunPOST();  // do some basic checks
  
  // set loop timers
  synchLoopTimers();
  
  // define the Core 0 task parameters
  // Core 0 is normally used for WiFi, but here we also use it drive RGB LEDs,
  // having created the LED patterns and clock them out
  // Note, this code starts and runs immediately, before loop() does
  xTaskCreatePinnedToCore(
    loop0,       // Task function.
    "Core0",     // name of task.
    10000,       // Stack size of task
    NULL,        // parameter of the task
    1,           // priority of the task
    &Core0,      // Task handle to keep track of created task
    0);          // pin task to core 0

  Serial.println("Setup() complete");

}

// ---------------------------------------------------------------------------------

  // the controller will react to a soft RESET event
  // this function effectively restarts the code as if from power-up
void(* resetFunc) (void) = 0; //declare reset function at address 0

// ---------------------------------------------------------------------------------

void loop() {
  // Main loop code, runs repeatedly.

  //###############################################################################
  //
  //  WiFi comms check & respond
  //
  //###############################################################################
  // if WiFi is conencted then check Rx buffer on every cycle, so that we respond to
  // received packets as soon as possible
  if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events

  if (WiFiConnected) {
    if (WiFiRx) {readWiFiRx();} // WiFi Rx buffer contains unread data so deal with it before adding new
  }


  //###############################################################################
  //
  //  Read serial comms
  //
  //###############################################################################
  readSerial();                 // check serial port on every loop cycle


  //###############################################################################
  //
  //  LTOF check
  //
  //###############################################################################
  // continuously read the laser range finder, when enabled at 1ms intervals
  if (millis() != VL53L0Xms) {VL53L0Xms = millis(); if (LTOF_On > 0) {VL53L0X_Run();}}


  //###############################################################################
  //
  //  Walk engine check
  //
  //###############################################################################
  // based on a variable rate hardware timer, perform walking tasks
  if ((micros() - walkMicros) >= walkInterval) {
    // walking uses a separate timer so that the speed of walking can be controlled
    // before walking is enabled ensure that start conditions are set
    walkMicros = micros(); // set next time point
    if (!Tick20ms) {LedTick = true;} // perform LED sequences at same rate as walking
    Walk_Engine();
  }


  //###############################################################################
  //
  //  Head engine check
  //
  //###############################################################################
  // the Head Engine moves the head servo towards a target position, but it does not
  // control the head behaviour, which is done in DoHeadTasks() every 20ms
  if ((micros() - headMicros) >= headInterval) {
    // move the head towards the target if target counter is > 0
    headMicros = micros();
  //  Serial.println(headMicros);
    Head_Engine();
  }

  if (nextLoop > 0) {
    // head movement tasks performed in the next loop after the main tasks so as
    // not to impact on movement task timings too much
    nextLoop = 0; DoHeadTasks();
  }

  
  //###############################################################################
  //
  //  20ms tasks
  //
  //###############################################################################
  // perform these cascading tasks every 20ms
  if (Task20ms > 0) {
    switch(Task20ms) {
      case  1: // if WiFi fails stop movement and trigger autoRest
        // if readWiiCall == true then we are currently locked in a movement, so ignore timeout
        if (QuadActive && (RTxTimeout > 0) & !readWiiCall) {
          RTxTimeout--;
          if (RTxTimeout == 1) {
            Serial.println("WiFi failed!...");
            LedMode = 0; autoRest = 0;  // start teh autorest timeout cycle
          }
        } break;
      case 2: battRead(); break;    // read analogue input
      case 3: readSW0(); break;     // read button switch
      case 4: // return to standing position after autoRest time-out
        if ((QuadActive) && (autoRest < autoRestMax)) {
          autoRest++;
          if (autoRest == 500) {
            // move to the standing position after 10 seconds
    //        Serial.println(F("Resting..."));
            walkInterval = 20000;   // set to default speed
            MoveToAngles(90,90,90,90,90,90,90,90); WalkLast = 0;
            autoOFF = 10; // turn off servos after movement stops
          } else if (autoRest == 6000) {
            // move to the reset position 2 minutes
    //        Serial.println(F("Reseting..."));
            GoToRest(100);
            // resetFunc();
          }
        } break;
      case 5:// return head to Power-OFF state?
        if (HeadTgtCnt < 1) {
          // if (autoHeadOFF > 0) {autoHeadOFF--; if (autoHeadOFF < 1) {detachHeadServo();}}
        }
        break;

      case 6: DoHeadTasks(); break;   // perform head movement tasks
        
      case 7: // if not moving detach the legs if autoOFF count is set
        if ((Walk < 1) && (TgtCnt < 1)) {
          // return legs to Power-OFF state if standing or at rest?
          if (autoOFF > 0) {
            autoOFF--;
            if (autoOFF < 1) {
              SetLedMode(4);
              detachServos();
              // if (!QuadActive) {detachHeadServo();}
              Serial.println("AutoOFF - servos detached");
            }
          }
        }
        break;

      case 8: // perform auto-speed reduction if needed, applies to speeds 4 & 5 only
        // this is aimed at preventing servo burn-out, limiting high speed bursts
        if (WalkSpeedTO > 0) {
          WalkSpeedTO--; if (WalkSpeedTO == 0) {
            if (WalkSpeed == 4) {WalkSpeed = 3; LedMsk = 7; LedMskCnt = 20; WalkMax = 10; WalkSpeedTO = 2 * WalkSpeedToMax;}
            if (WalkSpeed == 5) {WalkSpeed = 4; LedMsk = 15; LedMskCnt = 20; WalkMax = 13; WalkSpeedTO = WalkSpeedToMax;}
          }
        } break;
      case 9:
        if (Tick20ms) {LedTick = true;}
        break;
    } Task20ms--;
  } else if ((millis() - next20ms) >= 20) {

    // do this every 20ms plus cascade of tasks
    next20ms = millis();  // set next time point
    // Serial.println(servoVal[0]);
    // Serial.print("2");
    Task20ms = 9;         // initiate 20ms task sequence

    // if connected to WiFi read Wii data and respond
    if (WiFiConnected) {readWii();}

    //###############################################################################
    //
    //  Main tasks
    //
    //###############################################################################
    // perform 'Move Engine' tasks
    // if the code is already in a MainTask we wait for that code to complete first
    // not doing so could result in a stacj overflow
    // Serial.print(MainTask);
    if (!MainRun) {
      switch (MainTask) {
        case -2: // run immediately after POST when in TEST mode
          MainTask_TEST(); break;
        case -1: // run immediately after normal RESET to go to rest position
          MainTask_POST(); break;
        case 0: break; // default null task
        case 1: // goto QuadActive default standing position
          MainTask_Range(); break;
        case 2: // backaway from target mode, then return to start
          MainTask_BackAway(); break;
        case 3: // track wall target mode
          MainTask_TrackWall(1); break;
        case 4: // autonomous demo mode
          MainTask_Autonomous(); break;
        case 99: // handle ESC exits
          MainTask_ESC(); break;
      default:
        Serial.println("MainTask crashed!");
        MainTask = 0; break;
      }
    }
  }


  //###############################################################################
  //
  //  40 ms Tasks (25Hz)
  //
  //###############################################################################
  // These tasks are run in a consequitive ripple to reduce the overhead on more
  // critical code. They are performed in reverse case order and will all complete
  // within the 40ms period
  if (Task40ms > 0) {
    switch(Task40ms) {
      case  1: if (DispMon) {Display_40ms();} break;  // update display if necessary
      case  2: if (Ping > 0) {Ping--;} break;  // time-out data received flag
      case  3: // if not connected to WiFi run then try to connect every second
        if (!WiFiConnected) {
          // only try to connect WiFi if no 'Pings' are being received
          if (Ping == 0) {
            WiFiConCnt--; if (WiFiConCnt < 1) {
              WiFiConCnt = 10; WiFiTryToConnect();  // 400 ms retry time-out
              if (WiFiTryOnce) {Serial.println("Trying to connect: " + getBAhex());}
            //  Serial.println("Trying to connect WiFi");
            }
          }
        } break;
      case  4:
        if (WiFiPing > 0) {
          WiFiPing--;
          if (WiFiPing < 1) {
             // we have stopped receiving so disconnect WiFi
            WiFiDisconnect();
            ESP_NOW_BA = -1;
            WiFiTryOnce = true;
          //  Serial.println("WiFiPing == 0");
          }
        } break;
    } Task40ms--;
  } else if ((millis() - next40ms) >= 40) {
    next40ms = millis(); Task40ms = 4;
    // Serial.print("4");
    // check display mirror auto time-out
    // when count reaches zero DispMon is set to false
    if (DispTx > 0) {DispTx--; if (DispTx == 0) {DispMon = false;}}
  }


  //###############################################################################
  //
  //  40 ms Print Task - Variable
  //
  //###############################################################################
  // this code uses either the serial port or the WiFi link to send messages
  if ((millis() - print40ms) >= 40) {
    print40ms = millis();
    if (!Echo) {PrintTx = "";}  // Echo is false so empty print buffer
    if (PrintTx.length() > 0) {
      // characters in string buffer so send some/all of them
      print40ms = millis() - 34; // shorten the delay to 6ms for more data to send
      Tx_PingCnt = 0;           // reset the WiFi ping counter
      if (PrintTgt == 0) {
        // default uses the serial port
        if (Serial.availableForWrite() >= 64) {
          if (PrintTx.length() <= 64) {
            Serial.print(PrintTx);
            PrintTx = "";   // empty the buffer
          } else {
            Serial.print(PrintTx.substring(0,64));
            PrintTx = PrintTx.substring(64);
          }
        }
      } else {
        // text data has been received from WiFi so respond using ESP_NOW
        // ensure that we have had a call back from the previous frame
        if (WiFiConnected) {
          // only trannsmit PrintTx data if connected
          if (WiFiTx_CB == 1) {
            WiFiTx_CB = 0;  // clear the call back flag when sending
            if (PrintTx.length() <= 32) {
              WiFiTx_len = PrintTx.length();
              PrintTx.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
              WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
              WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
            //  t0 = micros();
              esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
              PrintTx = "";   // empty the buffer
            } else {
              WiFiTx_len = 32;
              Any$ = PrintTx.substring(0,WiFiTx_len);
              Any$.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
              WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
              WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
            //  t0 = micros();
              esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
              PrintTx = PrintTx.substring(32);
            }
          } else if (WiFiTx_CB == -1) {
            // an error has occured so resend the data
            if (WiFiTryCnt < 5) {
              WiFiTryCnt++;   // track the number of retries
              WiFiTx_CB = 0;  // clear the call back flag when sending
            //  t0 = micros();
              esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
            } else {
              // too many resends so disconnect WiFi link
              WiFiDisconnect();
              Serial.println("Failed to Tx!");
            }
          }
        } else {
          // not connected so clear PrintTxdata
          PrintTx = "";
        }
      }
    }
  }
}

// ---------------------------------------------------------------------------------

void loop0(void * pvParameters) {
  // this function runs on Core 0, also used for ESP32 WiFi NOW
  // *** WARNING *** do not use for time consuming code like display, I2C, FastLED, etc
  // the following message, sent once at boot, confirms core 0 assignment
  Serial.print("Core0 task running on core "); Serial.println(xPortGetCoreID());

  for(;;){
    // infinite loop, just like loop() but running in Core 0
    // you must never use 'break' or 'return' statement here or it will crash
    // we tell the free RTOS system to call this function every 1 ms
  
  //  t1[2] = micros(); // end time
  //  t0[3] = micros(); // start time

    //###############################################################################
    //
    //  Control RGB LEDs
    //
    //###############################################################################
    // the process is triggered by LedTick, which is set by the Walk_Engine, so as to
    // synchronise the timing of the LEDs with the speed of the legs
    if (LedTick) {
      // LEDs are update once every Walk cycle to keep rates the same
      LedTick = false;
      LED_Main();
    }
    // Whenever LEDshow is set in the main code we update the RGB LED strip.
    // However this appear5s to conflict and crash VL53L0X I2C comms.
    // So we have to block the VL53L0X code during this process
    if (LEDshow) {
      if (ShowSkip) {ShowSkip = false; VL53L0X_Show = 3;}
      else {LEDshow = false; FastLED.show();
      }
    } else {ShowSkip = true;}

  //  t1[3] = micros(); // end time
  //  t0[2] = micros(); // start time

    // return to RTOS system with a 1ms call back, will give responseive LEDs
    vTaskDelay(1/portTICK_PERIOD_MS);
  }
}

// --------------------------------------------------------------------------------

void delayLoop(unsigned long zDel) {
  // loop for a delay period
  zDel = millis() + zDel;
  while (millis() < zDel) {
    if (ESC) return;
    loop(); yield();
  }  
}

// --------------------------------------------------------------------------------

void loopWhileTgtCnt() {
  // used to call loop() code during a 'Move' function
  // Serial.println(F("loopWhiletgTCnt"));
  while (TgtCnt > 0) {
    if (ESC) {return;}
    loop(); yield();
  }
}

// --------------------------------------------------------------------------------

void loopWhileTurn(long zWalkCnt) {
  // count number of Walked clicks whilst turning, ignoring range data
  // from testing the corection factor for 360° = 2.3
  // zWalkCnt is effective the angle passed to this funtion
  zWalkCnt = (zWalkCnt * 23)/10;
  Walked = 0; // clear the walked counter
  while (zWalkCnt > Walked) {
    if (ESC) return;
    loop();
    // slowly increase walking speed
    if (walkInterval > 15000) {walkInterval = walkInterval -100;}
    yield();
  }
}

// --------------------------------------------------------------------------------

void loopWhileTurnEcho(long zWalkCnt) {
  // count number of Walked clicks whilst turning
  // from testing the corection factor for 360° = 2.3
  // zWalkCnt is effective the angle passed to this funtion
  // if there is no wall/target detected then return by setting ESC
  zWalkCnt = (zWalkCnt * 23)/10;
  Walked = 0; // clear the walked counter
  while (zWalkCnt > Walked) {
    if (ESC) return;
    loop(); yield();
    // check range
    if (RangeMinMem > LtofStop){ESC = 1;} // no wall detected, so stop turning
    // slowly increase walking speed
    if (walkInterval > 15000) {walkInterval = walkInterval -100;}
  }
}

// --------------------------------------------------------------------------------

void loopWhileWalkEcho(long zWalkCnt) {
  // walk for a number of Walked clicks whilst reading the radar distance
  // there are nominally 32 clicks in a walk leg stride
  // Serial.println("loopWhileWalk");
  Walked = 0; // clear the walked counter
  while (zWalkCnt > Walked) {
    if (ESC) return;
    loop();
    // check Range reading and exit if an object is detected
    if (Range <= 325) {break;}
    // slowly increase walking speed
    if (walkInterval > 15000) {walkInterval = walkInterval -100;}
    yield();
  }
}

// --------------------------------------------------------------------------------

void OnDataRecv(const esp_now_recv_info_t *info, const uint8_t *incomingData, int len) {
  // Call-back function oocurs when data has been received
  // This needs to be treated like an interrupt, so speed is the essence here
  // memcpy moves data in a block, instead of individual bytes if using for() loop
  memcpy(WiFiRxMacAddr, info->src_addr, 6);   // get the senders MAC/broadcast address
  memcpy(WiFiTxMacAddr, info->des_addr, 6);   // get destination MAC address, which should be assigned peer
  memcpy(&Rx_Buff, incomingData, len);        // copy the received data into Rx_Buff array
  Rx_len = len - 1; Rx_Pnt = 0;               // len contain the length of data
  RxRecvEvent = true;                         // set the flag to say data received
}

// --------------------------------------------------------------------------------

void OnDataRecvHandler() {
  // Called from the main loop() when has been set, to handle the received data
  RxRecvEvent = false;
  // incoming data will always be two or more characters, ie. text == $+++
  if (Rx_len < 2) {Serial.println("Rx Error - no Rx data"); return;}

  RxGetChecksum(Rx_len);  // generate a checksum from Rx data
  if (Rx_Chk != Rx_Buff.ESPdata[Rx_len]) {
    Serial.println("Rx Error - checksum");
  }
  // set flags before exiting
  WiFiRxBytes += Rx_len; Rx_Task = 0; WiFiRx = true;
  WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
  WiFiRxRec = true; // set flag indicating data has been recieved from Wii Transceiver
  //  Serial.print("Bytes received: "); Serial.println(Rx_len);
}

// --------------------------------------------------------------------------------

void OnDataSent(const uint8_t *mac_addr, esp_now_send_status_t status) {
  // Callback when data is sent, determines success or failure
  // t1 = micros();
  // Serial.print("Send Status: ");
  if (WiFiConnected) {
    // the link is already active so handle normal success/failure scenarios
    if (status == 0){
      // Serial.println("Success " + String(t1-t0));
      WiFiTx = true;    // flag WiFi Tx success
      WiFiTx_CB = 1;    // set flag to indicate that a call back state has occured
      WiFiTxErrCnt = 0; // clear the error flag in case of a glitch
      WiFiPing = 50;    // set link time-out to 2 seconds
    } else {
    //  Serial.println("Fail " + String(t1 - t0));
      WiFiTx = false;   // flag WiFi Tx failure
      WiFiTx_CB = -1;   // set flag to indicate that a resend is required
      WiFiTxErrCnt++;
    }
  } else {
    // not connected so we must have tried to connect and may get an error
    if (status == 0){
      // successful so enter the connected state
      Serial.println("Connected! to " + getBAhex());
      WiFiConnected = true;
      WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
      WiFiTxErrCnt = 0; // reset error count for this connection
    } else {
    //  Serial.println("Status = " + String(status));
      WiFiTxErrCnt++;
    }
    WiFiTx_CB = 1;
  }
}

// ---------------------------------------------------------------------------------

void RunPOST() {
  // start-up initialisation processes  
  Serial.println(Release);
  Serial.println("Compiled: " + Date);

  // set up battery quick averaging
  BatVol = analogRead(BatPin);
  // if connected to USB only, no battery, then BatVol = 0
  // under these conditions the robot shuts down
  if ((BatVol < BatCritical) && !TEST) {USB = true;}
  
  BatAvg = BatVol; BatSum = BatVol * 50;  // preload the battery sum for averaging
  AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
  if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
  if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
  Serial.println("Battery " + String(BatVol) + "\t" + String(float(BatAvg)/BatCal) + "v\t" + String(AnyLong) + "%");
  randomSeed(BatVol); // use the battery voltage as the random seed factor
  if (USB) {Serial.println("Working in USB mode");}

  // initilaise RGB LED array
  FastLED.clear();
  FastLED.show(); // This sends the updated pixel color to the hardware.
  Serial.println("RGB LEDs cleared.");
  LED_Set_All(0,1,0); // start all green

  // Test for the presence of the VL53L0X on I2C
  digitalWrite(XSHUT_PIN,HIGH);   // switch VL53L0X ON
  delay(1);
  Wire.beginTransmission(ADDRESS_DEFAULT);
  I2C_Err = Wire.endTransmission();
  if (I2C_Err == 0) {Serial.println("VL53L0X found on I2C buss");}
  else {Serial.println("VL53L0X not found");}
  digitalWrite(XSHUT_PIN,LOW);   // switch VL53L0X OFF

  // check for SW0 button being pressed when coming out of RESET
  if (digitalRead(sw0Pin) == LOW) {
    // SW0 is being pressed so force TEST mode
    TEST = true;
  }

  // get ready to run
  if (TEST) {
    // in TEST mode, used during assembly
    // LEDs cycle RGB colours
    // Display has a border and cycles through settings
    // SW0 sets servos to 1500 constant or short sweep
    // SW1 toggle LTOF sensor ON/OFF
    Serial.println("Running in TEST mode");
    delay(1000); //display intro for 1 second
    // set all servos to their default values, for fitting limbs etc
    for (int zS = 0; zS <= 8; zS++) {
      servoVal[zS] = 1500;  // set default servo values
      servoAtt[zS] = false; // set 'attached' flag
    }
    Wait_ButtonsDwn();      // ensure button switch is not pressed before continuing
    SetMainTask(-2);        // run TEST mode task
  } else {
    // normal mode so display battery voltage after intro screen
    Serial.println("Running in normal mode");
    MainTask = -1;  // run MainTask_POST()
   }
  Serial.println("POST completed");
}

// ---------------------------------------------------------------------------------

void setDefaults() {
  // load default values
  atTgt = false;          // true if servos are at target values
  autoHeadOFF = 0;        // if set > 0 then counts down to head servo switch OFF, if -1 disabled
  autoOFF = 0;            // if set > 0 then counts down to servo switch OFF, if -1 disabled
  autoRest = autoRestMax; // returns to a rest position after a timeout
  atHeadTgt = false;      // = true if head servo is at target value
  atTgt = false;          // = true if servos are at target values
  BatAvg = 4095;          // preload averaging function with high value
  BatAvg = 0;             // preload averaging function with high value
  BatSum = 72320;         // preload averaging sum with 20x max value
  Blink = false;          // LED blink clock
  BlinkCnt = 0;           // counter for LED Blink clock
  Bright = 255;           // FastLED brightness
  Brightness = 128;       // Display brightness
  C_Cnt = 0;              // counter used in 'C' button detection
  C_Dn = 0;               // counter for 'C' button down period
  cmdMode = ' ';          // command mode
  cmdSgn = 1;             // cmdVal sign, 1 or -1
  cmdType = ' ';          // command mode type
  cmdVal = 0;             // value associated with a cmdType
  CZ = 0;                 // received button values
  CZ_Wait = false;        // flag used for button release wait function
  DispClr = false;        // set == true to clear the memory contents
  DispCnt = 0;            // counter used in display updating
  DispDel = 0;            // >0 display delay counter, stops updates over I2C
  DispDelGbl = 0;         // global DispDel offset, used to adjust display frame rates
  DispDisp = false;       // set = true to immediately display memory contents
  DispLock = false;       // if == true then don't allow display mode changes
  DispMode = 0;           // the type of display for a given mode
  DispMon = false;        // == true if connected to OLED Mirror app
  DispOveride = 0;        // if DispLock == true then use this overide mode
  DispTx = 0;             // counter used with display mirror app
  Echo = true;            // if true send PrintTx print messages
  ESC = 0;                // set > 0 to escape all functions
  ESP_NOW_BA = -1;        // broadcast address pointer, -1 to start with
  ESP_NOW_Init = false;   // == true once ESP-NOW has been initiaiised
  HeadActive = false;     // true if head is in scan mode for tasks
  HeadDir = 0;            // head movement direction, +1 == RH, -1 == LH
  headInterval = 20000;   // main loop head movement interval in microseconds
  HeadPause = 0;          // 20ms counter to temporarily pause head movement
  HeadSubTask = 0;        // head movement sub task pointer
  HeadTask = 0;           // head movement task pointer
  HeadTaskRun = false;    // set to true during head tasks
  HeadTgtCnt = 0;         // counter used in automatic head servo move to target actions
  interval = 20000;       // main loop interval in microseconds, 50Hz
  JoyActive = true;       // flag used to prevent stack overflow during moves
  JoyDL = 0;              // direction 1 counter
  JoyMode = 0;            // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
  JoySpd = 0;             // speed vector = max(JoyX,JoyY)
  JoyX = 128;             // received value, default
  JoyXV = 0;              // converted value 0 - 127
  JoyY = 128;             // received value, default
  JoyYi = 0;              // tracking filter for received JoyY value
  JoyYV = 0;              // converted value 0 - 128
  LED_Del = 0;            // delay timer used in LED functions
  LedLast = 0;            // previous task when change set
  LedMem = -1;            // previous recorded LED variable, default -1
  LedNop = true;          // if == true then skip LED modes
  LED_Period = 20;        // LED timer period, default = 20ms
  LedRstCnt = 0;          // counter used in LED auto reset
  LEDshow = false;        // LED show flag for mouth LEDs
  LedTick = false;        // set true every 'Walk' cycle
  LEDVal = 0;             // values used in LED tasks
  LegDn = LegDwnMin;      // dynamic angle for leg down in 'Move Engine'
  LegUp = LegUpMin;       // dynamic angle for leg raised in 'Move Engine'
  LegUpFwd = LegUpMin;    // dynamic angle for forward leg raised in 'Move Engine'
  LTOFcalc = false;       // if == true then calculate speed of ranged object in m/s
  LTOF_On = 0;            // 1 or 2 when laser ranging is enabled, default is 0 == OFF
  LTOFspd10 = 0;          // speed of ranged target in m/s x 10
  MainDel = 0;            // MainTask delay counter, default = 0
  MainRun = false;        // set true when a MainTask is running to prevent re-entry
  MainTask = 0;           // main task pointer
  MainTaskNext = 0;       // the task to run after an arrest; default is 0
  MainTaskRef = 0;        // record of MainTask pointer when mode was set
  MoveState = 0;          // move state machine pointer
  MoveType = "";          // type of move taking place
  Once = 0;               // counter for demo moves
  Ping = 0;               // 'ping' counter
  PWM_Freq = 50;          // servo PWM frequency, default = 100;
  QuadActive = false;     // true if active, otherwise false
  RangeEn = false;        // == true if VL53L1X ranging is active
  RangeMax = LtofLimit;   // the current limited to Range measurements
  RangeMin = LtofMin;     // the current lower limited to Range measurements
  readWiiCall = false;    // flag prevents call stack overflow in readWii()
  RTxTimeout = 0;         // counter used to auto-reset if WiFi fails
  Rx_Chk = 0;             // Rx checksum
  RxCZ = 0;               // buffered received CZ value
  RxJoyX = 0;             // buffered received JoyX value
  RxJoyY = 0;             // buffered received JoyY value
  Rx_len = 0;             // length of WiFi received data block
  RxRec = false ;         // set true when a valid frame of data is received
  RxRecvEvent = false;    // set == true when a OnDataRecv() call back event has occured
  RxState = 0;            // receiver state machine state
  Rx_Task = 0;            // pointer user by Rx state machine
  RxVal = -1;             // value received from serial Rx
  SerialRx = false;       // == true if any character has been received
  ServoEn = false;        // false = OFF, true = ON enabled status
  ServoPnt = 0;           // servo target pointer for manual control, default = 0
  
  // servo start values are offset by IO1 to ensure some movement occurs at reset
  servoVal[0] = Rst0; // value sent to servos
  servoVal[1] = Rst1; // value sent to servos
  servoVal[2] = Rst2; // value sent to servos
  servoVal[3] = Rst3; // value sent to servos
  servoVal[4] = Rst4; // value sent to servos
  servoVal[5] = Rst5; // value sent to servos
  servoVal[6] = Rst6; // value sent to servos
  servoVal[7] = Rst7; // value sent to servos
  servoVal[8] = Rst8; // value sent to servos
  servoTgt[0] = Rst0; // target values for servos
  servoTgt[1] = Rst1; // target values for servos
  servoTgt[2] = Rst2; // target values for servos
  servoTgt[3] = Rst3; // target values for servos
  servoTgt[4] = Rst4; // target values for servos
  servoTgt[5] = Rst5; // target values for servos
  servoTgt[6] = Rst6; // target values for servos
  servoTgt[7] = Rst7; // target values for servos
  servoTgt[8] = Rst8; // target values for servos
  
  Sit = false;            // used in backaway tasks
  SubLast = 0;            // previous sub task pointer
  SubTask = 0;            // sub task pointer
  sw0Cnt = 0;             // button switch counter
  sw0DwnTime = 0;         // button pressed down time
  sw0LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw0_Nop = false;        // == true to block SW0 read tasks, default = false
  sw0State = HIGH;        // state of read button switch pin
  sw0Timer = 0;           // timer used to detemine button sequences
  TEST = TESTmode;        // normally false, set = true for assembly test mode
  Tick20ms = false;       // if == true then run LED tasks at 20ms rate
  Turn = false;           // default = false; if true turns continuously until = false
  Tx_PingCnt = 0;         // 1 second Tx ping timer
  USB = false;            // == true if battery voltage is below critical value at outset
  VL53L0X_Show = 0;       // timer used to hold off VL53L0X when using FastLED
  VL53L0X_Skip = 0;       // timer used to hold off VL53L0X when using FastLED
  Walk = 0;               // if >0 then walk in one of four directions
  WalkCnt = 0;            // a counter which controls the walking process
  Walked = 0;             // counter used to return robot to a given position
  walkInterval = 20000;   // main loop walking interval in microseconds
  WalkLast = 0;           // previousl Walk value if any
  WalkLftDrive = 128;     // drive factor for left-hand side, 128 = max, 0 = min
  WalkMax = 4;            // speed factor 4 - 12
  WalkRgtDrive = 128;     // drive factor for right-hand side, 128 = max, 0 = min
  WalkSpeed = 1;          // value used to determine WalkMax, 1 - 5
  WalkSpeedTO = 0;        // Time-Out counter for WalkSpeed auto-reduction, counts to zero
  WiFiCntC = 0;           // C button counter for WiFi enable
  WiFiConCnt = 0;         // counter used in connection retry times
  WiFiConnected = false;  // == true if a WiFi connection has been established
  WiFiEn = false;         // true when WiFi is enabled with prolonged 'Z' button activity
  WiFiPing = 0;           // Rx counter used to maintain connected state
  WiFiRx = false;         // true when a block of data has been received
  WiFiRxBytes = 0;        // number of bytes receiv ed over the WiFi link
  WiFiRxErr = 0;          // number of receive errors in the WiFi link
  WiFiRxRec = false;      // true when a block of data has been successfully received
  WiFiTryCnt = 0;         // WiFi try to connect count
  WiFiTryNum = 0;         // WiFi try to connect total count
  WiFiTryOnce = true;     // == true if first pass of trying to connect
  WiFiTx = false;         // true when a block of data has been sent successfully
  WiFiTxBytes = 0;        // number of bytes receiv ed over the WiFi link
  WiFiTx_CB = 1;          // == 1 call back is clear to send, == -1 resend
  WiFiTx_Chk = 0;         // Tx checksum
  WiFiTx_len = 0;         // >0 when WiFi Tx buffer contains data to be sent
  WiFiTxErr = 0;          // WiFi data error
  WiFiTxErrCnt = 0;       // WiFi Tx error count, leads to disconnection
  WiiType = 0;            // device type, 0 == Nunchuk, 1 == Classic
  Z_Dn = 0;               // button pressed state; 0 = UP, 1 = Down
}

// --------------------------------------------------------------------------------

void synchLoopTimers() {
  // reset main loop timers
  // note that millis() timers are offset so that they don't burden each other
  while (millis() < 50) {}              // ensure there is at least 50ms on the hardware timer

  headMicros = micros();                // set head loop timer (default 20ms variable)
  next20ms = millis() - 15;             // set task loop timer to 5ms from now (default 20ms fixed)
  next40ms = millis() - 25;             // set task loop timer to 15ms from now (default 40ms fixed)
  print40ms = millis() - 30;            // serial PrintTx management for WiFi to 10ms from now
  walkMicros = micros();                // set walk loop timer (default 20ms variable)
}

// ---------------------------------------------------------------------------------
