
// ----------------------------------------------------------------------

void AnyLongDamp() {
  // called after setting AnyLong to a value based on the following:
  // AnyLong = map(RangeCentreTracker,RM,R5,1,12);
  AnyLong = 60000/AnyLong;
  AnyLong = long(walkInterval) + ((AnyLong - long(walkInterval)) / SpdRmp);
  walkInterval = AnyLong;
}

// ----------------------------------------------------------------------

void CalcHeadAng() {
  // something detected so calculate a steering angle, which is used by
  // the autonomous code
  AnyLong = (512 * (RangeIntR - RangeIntL)) / (RangeIntR + RangeIntL);
  RangeAng = map(AnyLong,128,-128,Head_60P,Head_60N);
  // now set variables for next half-cycle
  CalcCnt++; // track the number of calculation modes performed
}

// ----------------------------------------------------------------------

void CheckMinRange() {
  // used in autonomous mode to keep an eye on a target appearing during a
  // half sweep, so this is called at the extremes of the sweep
  if (RangeMin <= LtofDet) {
    // something has been detected
    TrgtDet = true;
    // new steering angle is only calculated when something is within range
  } else {
    TrgtDet = false; // nothing has been detected
  }
  if ((Turning) && (RangeMin >= LtofStop)) {ESC = 1;} // if turning then stop turning
  RangeMinMem = RangeMin; RangeMin = LtofMax;         // store & reset the minimum range
}

// ----------------------------------------------------------------------

void CheckWalkDir(int zW) {
  // ensure walk is set for the right direction
  if (abs(Walk) == zW) {Walk = zW;} else {StartWalk(zW);}
}

// ----------------------------------------------------------------------

void DemoMove_Bow() {
  // bow movement used to introduce demo move
  walkInterval = 20000;   // set to default speed
  SetLedMode(21);
  MoveType = "Bowing";
  MoveToAngles(90,LegDn,90,LegDn,90,LegDn,90,LegDn);
  GoToAngles(45,120,45,120,90,90,90,90,50); loopWhileTgtCnt();
  GoToAngles(90,163,90,163,-1,-1,-1,-1,60); loopWhileTgtCnt();
  SetLedMode(0); delayLoop(1000); // pause
  
  SetLedMode(23);
  GoToAngles(45,120,45,120,-1,-1,-1,-1,30); loopWhileTgtCnt();
  GoToAngles(90,90,90,90,-1,-1,-1,-1,30); loopWhileTgtCnt();
  SetLedMode(0); autoOFF = 10;  // in standing position, so turn off legs
  Walk = 0; WalkLast = 0;       // clear walk pointers, so that next move positions legs first
}

// ----------------------------------------------------------------------

void DemoMove_SitNWave() {
  // sit and wave the right hand
  walkInterval = 20000;   // set to default speed
  SetLedMode(23);
  MoveType = "Waving";
  MoveToAngles(90,LegDn,90,LegDn,90,LegDn,90,LegDn);
  GoToAngles(90,90,90,90,145,142,145,142,50); loopWhileTgtCnt();
  SetLedMode(0); delayLoop(1000); // pause
  if (random(2) == 0) {
    // wave with right-hand leg
    GoToAngles(90,90,80,160,-1,-1,-1,-1,20); loopWhileTgtCnt();
    delayLoop(200); // pause
    for (anyFor=0; anyFor<2; anyFor++) {
      GoToAngles(90,90,60,180,-1,-1,-1,-1,3); loopWhileTgtCnt();
      GoToAngles(90,90,40,150,-1,-1,-1,-1,3); loopWhileTgtCnt();
      delayLoop(100); // pause
      GoToAngles(90,90,60,180,-1,-1,-1,-1,3); loopWhileTgtCnt();
      GoToAngles(90,90,80,150,-1,-1,-1,-1,3); loopWhileTgtCnt();
      delayLoop(100); // pause
    }
  } else {
    // wave with left-hand leg
    GoToAngles(80,160,90,90,-1,-1,-1,-1,20); loopWhileTgtCnt();
    delayLoop(200); // pause
    for (anyFor=0; anyFor<2; anyFor++) {
      GoToAngles(60,180,90,90,-1,-1,-1,-1,3); loopWhileTgtCnt();
      GoToAngles(40,150,90,90,-1,-1,-1,-1,3); loopWhileTgtCnt();
      delayLoop(100); // pause
      GoToAngles(60,180,90,90,-1,-1,-1,-1,3); loopWhileTgtCnt();
      GoToAngles(80,150,90,90,-1,-1,-1,-1,3); loopWhileTgtCnt();
      delayLoop(100); // pause
    }
  }
  delayLoop(1000); // pause
  SetLedMode(21); GoToAngles(90,90,90,90,-1,-1,-1,-1,50); loopWhileTgtCnt();
  SetLedMode(0); delayLoop(1000); // pause
  SetLedMode(21);GoToStand(50); loopWhileTgtCnt();
  SetLedMode(0); autoOFF = 10;  // in standing position, so turn off legs
  Walk = 0; WalkLast = 0;       // clear walk pointers, so that next move positions legs first
}

// ----------------------------------------------------------------------

void DoHeadTasks() {
  // perform a range of head movement tasks depending on the MainTask
  // in non-autonomous mode the head will look for targets and track them
  // in autonomous mode the head does not seek to find/track a target, it simply
  // scans over a given range to provide steering information
  // HeadTask = 0       get centre range and exit
  // HeadTask = 1       initialise head tasks
  // HeadTask = 2 - 4   random head movements
  // HeadTask = 10 - 22 scan a detected target, whilst in range
  // HeadTask = 30 - 37 left/right scanning for autonomous mode, steering and stop
  // HeadTask = 99      small left to right movements, to prevent servo static jitter
  // in WiFi control mode, simply scan randomly, without ranging
  
  if (HeadTaskRun) return;
  if (HeadTask < 1) {RangeCentreTracker = Range; return;}

  HeadTaskRun = true; // prevent stack overflow due to nesting
  if (!servoAtt[0]) {attachHeadServo();}
  // if (HeadTaskLast != HeadTask) {Serial.println("HeadTask " + String(HeadTask));}
  HeadTaskLast = HeadTask;
  switch(HeadTask) {
    // in this range of cases the head scans randomly left to right looking
    // if a target is detected by HeadMode code then do different actions
    case 1:
      // set random first head movement and head mode
      // also called when target has been lost
      RangeAng = Head_0;            // default angle for no detected target
      RangeCentre = LtofMax;        // reset minimum range tracker
      RangeCentreTracker = LtofMax; // reset minimum range tracker
      RangeMin = LtofMax;           // reset minimum range tracker
      autoHeadOFF = -1;
      HeadAdj = false; // disable calculations for 1st half sweep
      if (QuadActive &&  WiFiEn && (MainTask == 0)) {
        // robot is being controlled via WiFi so simply move head randomly
        HeadDir = random(2);  // set random head direction to start with
        HeadCyc = 0;          // if > 0 then perform full left/right sweep
        HeadMode = -1;        // no high level control task
        HeadTask++;           // next task is 2
      } else if (MainTask < 4) {
        // target scanning & tracking mode
        HeadDir = random(2);  // set random head direction to start with
        HeadCyc = 4;          // if > 0 then perform full left/right sweep
        HeadMode = 0;         // waits for target detection
        HeadTask++;           // next task is 2
      } else {
        // MainTask == 4 autonomous scanning mode
        TrgtDet = false;      // clear target detected flag
        CalcCnt = 0;          // counter used in autonomous mode
        HeadAdj = false;      // reset head sweep flag
        HeadMode = 3;         // autonomous mode, looks for target
        HeadTask = 30;
      } 
      break;
    case 2:
      // toggle head direction and set new target angle
      // if HeadCyc  > 0 perform full left/right sweeps
      // if HeadCyc == 0 perform sweeps to random angles in opposite quadrants
      if (HeadDir > 0) {
        // look to the right quadrant
        HeadDir = -1; // flag indicates moving to the right
        // set random head target angle between 0 to 60° in 10° steps
        if (HeadCyc > 0) {servoTgt[0] = Head_60P; servoTgtRht = Head_60P;}
        else {servoTgt[0] = (((Head_60P - Head_0) * random(8))/7) + Head_0;}
      } else {
        // look to the left quadrant
        HeadDir = 1;  // flag indicates moving to the left
        // set random head target angle between -60 to 0° in 10° steps
        if (HeadCyc > 0) {servoTgt[0] = Head_60N; servoTgtLft = Head_60N;}
        else {servoTgt[0] =  Head_0 - (((Head_0 - Head_60N) * random(8))/7);}
      } 
      if (HeadCyc > 0) {HeadCyc--;} // if full-cycling reduce counter
      // set head steps based on angle to move through
      HeadTgtCnt = abs(((servoTgt[0] - servoVal[0]) * 50)/(Head_60P - Head_60N));
      HeadTask++; break;
    case 3:
      // wait for head to reach end of movement
      if (HeadTgtCnt < 1) {
        // end of movement reached, set a random delay and HeadTask = 4
        HeadCnt = 10 + random(20); HeadTask++;
        if (HeadCyc > 0) {HeadCnt = 10;}  // full sweeps, so don't pause for long
      } break;
    case 4:
      // pause temporarily at this angle whilst HeadCnt > 0
      HeadCnt--; if (HeadCnt < 1) {
        if (HeadCyc > 0) {HeadTask = 2;}  // full sweep so go back to changing direction
        else {
          // complete the sweep
          if (HeadDir > 0) {
            // complete the move to the left
            servoTgt[0] = Head_60N;  servoTgtLft = Head_60N;
          } else {
            // complete the move to the right
            servoTgt[0] = Head_60P; servoTgtRht = Head_60P;
          }
          HeadTgtCnt = abs(((servoTgt[0] - servoVal[0]) * 50)/(Head_60P - Head_60N));
          HeadTask++;
        }
      } break;
    case 5:
      // wait for head to reach end of movement
      if (HeadTgtCnt < 1) {
        // end of movement reached, set a random delay and HeadTask = 4
        HeadCnt = 10 + random(20); HeadTask++;
        if (HeadCyc > 0) {HeadCnt = 10;}  // full sweeps, so don't pause for long
      } break;
    case 6:
      // pause temporarily at this angle whilst HeadCnt > 0
      HeadCnt--; if (HeadCnt < 1) {HeadTask = 2;} // change direction
      break;


    case 10:
      // scan a detected target whilst it remains in range, centred on the detected
      // angle as the starting point. Immediately stop the random sweep. Set the
      // sweep range to max.
      HeadMode = 1;           // set HeadMode to record minimum value
      HeadCnt = 0;            // clear the target lost counter
      HeadTgtCnt = 0;         // stop the current head movement
      RangeAng = RangeMinAng; // record head angle at which a target was detected
      RangeMin = LtofMax;     // reset minimum range tracker
      AngDif = AngDifMin;     // set the sweep angle delta value
      HeadAdj = false;        // prevent target angle adjustment until a full sweep is done
      HeadErrTrk = 0;         // zero error tracker
      HeadErrSt = false;      // clear the head error status flag
      HeadTask++; break;
    case 11:
      // this is a centre scan point, prior to sweeping right
      RangeCentre = min(int(Range),int(RangeMin)); // use the closest range
      RangePnt = true; // flag used by tracker function
      servoTgt[0] = min(Head_60P, RangeAng + AngDif); // define sweep to right range
      servoTgtRht = servoTgt[0];                      // note right most angle
      SetHeadTgtCnt(servoTgt[0] - servoVal[0]);       // set sweep speed
      HeadTask++; break;
    case 12:
      // turning to the right
      // if a target has been detected then slips away, we stop turning right,
      // and shift the target angle to the left, and don't calcuylate the error
      if (RangeMin <= LtofDet) {
        if (Range > (RangeMin + 10)) {HeadTgtCnt = 0; RangeAng -= AngDif;}
      }
      // wait for the rightmost target angle to be reached
      if (HeadTgtCnt < 1) {
        // rightmost point has been reached
        if (HeadAdj) {HeadMode = 2;} // adjust only if full sweep has been completed
        HeadAdj = true; // a full sweep has now been completed
        HeadTask++; // move on once target is reached
      } RangeIntR += long(RangeMax - Range); // integrate the right-hand range values
      break;
    case 13:
      // adjustment may have moved the rightmost point, so move there before
      // starting a new sweep
      servoTgt[0] = min(Head_60P, RangeAng + AngDif); // define sweep to right range
      servoTgtRht = servoTgt[0];                      // note right most angle
      SetHeadTgtCnt(servoTgt[0] - servoVal[0]);       // set sweep speed
      HeadTask++; break;
    case 14:
      // wait for new rightmost end point to be reached  
      if (HeadTgtCnt < 1) {HeadTask++;} // rightmost point reached
      break;
    case 15:
      // now turn left towards the centre point
      RangePnt = true;                  // flag used by tracker function
      servoTgt[0] = RangeAng;           // centre angle is min range point
      SetHeadTgtCnt(servoTgt[0] - servoVal[0]); // set sweep speed
      RangeIntR = 0; RangeIntL = 0;     // clear the integrators
      RangeMin = LtofMax;               // reset minimum range tracker
      HeadTask++; break;
    case 16:
      // wait until the centre point has been reached
      // integrate the right-hand range value as we go
      if (HeadTgtCnt < 1) {HeadTask++;} // centre point achieved
      RangeIntR += long(RangeMax - Range); // integrate the right-hand range values
      break;
    case 17:
      // this is a centre scan point, prior to sweeping right
      RangeCentre = min(int(Range),int(RangeMin));        // use the closest range
      RangePnt = true;                                    // flag used by tracker function
      servoTgt[0] = max(Head_60N, RangeAng - AngDif);     // define sweep to left range
      servoTgtLft = servoTgt[0];                          // note left most angle
      SetHeadTgtCnt(servoTgt[0] - servoVal[0]);           // set sweep speed
      HeadTask++; break;
    case 18:
      // turning to the right
      // if a target has been detected then slips away, we stop turning left,
      // and shift the target angle to the right, and don't calculate the error
      if (RangeMin <= LtofDet) {
        if (Range > (RangeMin + 10)) {HeadTgtCnt = 0; RangeAng += AngDif;}
      }
      // wait for the leftmost target angle to be reached
      // integrate the left-hand range value as we go
      if (HeadTgtCnt < 1) {
        // leftmost point has been reached
        if (HeadAdj) {HeadMode = 2;}  // calculate centre adjustments based on this full sweep
        HeadAdj = true; // a full sweep has now been completed
        HeadTask++;     // move on once target is reached
      } RangeIntL += long(RangeMax - Range); // integrate the left-hand range values
      break;
    case 19:
      // adjustment may have moved the leftmost point, so move there before
      // starting a new sweep
      servoTgt[0] = max(Head_60N, RangeAng - AngDif);  // define sweep to right range
      servoTgtLft = servoTgt[0];                       // note left most angle
      SetHeadTgtCnt(servoTgt[0] - servoVal[0]);        // set sweep speed
      HeadTask++; break;
    case 20:
      // wait for new leftmost end point to be reached  
      if (HeadTgtCnt < 1) {HeadTask++;} // rightmost point reached
      break;
    case 21:
      // now turn right towards the centre point
      RangePnt = true;                  // flag used by tracker function
      servoTgt[0] = RangeAng;           // centre angle on min range point
      SetHeadTgtCnt(servoTgt[0] - servoVal[0]); // set sweep speed
      RangeIntR = 0; RangeIntL = 0;     // clear the integrators
      RangeMin = LtofMax;               // reset minimum range tracker
      HeadTask++; break;
    case 22:
      // wait until the centre point has been reached
      // integrate the left-hand range value as we go
      if (HeadTgtCnt < 1) {HeadTask = 11;}  // centre point achieved
      RangeIntL += long(RangeMax - Range);             // integrate the left-hand range values
      break;


    case 30:
      // these set of cases perform simple left to right scanning for autonomous
      // target detection.
      // centre point continue towards left-most point
      if (HeadAdj) {CalcHeadAng();} // calculate adjustments based on this half sweep
      RangeIntL = 0; // clear left-hand integrator
      servoTgt[0] = Head_60N; HeadTgtCnt = HeadSpd; HeadTask++; break;
    case 31:
      // wait for new leftmost end point to be reached  
      if (HeadTgtCnt < 1) {HeadTask++;} // left-most point reached
      RangeIntL += long(Range); // integrate the left-hand range values
      break;
    case 32:
      // left-most point reached
      // turn right towards centre position
      CheckMinRange();
      servoTgt[0] = Head_0; HeadTgtCnt = HeadSpd; HeadTask++; break;
    case 33:
      // wait for centre point to be reached  
      if (HeadTgtCnt < 1) {HeadTask++;} // centre point reached
      RangeIntL += long(Range); // integrate the left-hand range values
      break;
    case 34:
      // centre point continue right towards right-most position
      CalcHeadAng(); // calculate adjustments based on this half sweep
      RangeIntR = 0; // clear right-hand integrator
      servoTgt[0] = Head_60P; HeadTgtCnt = HeadSpd; HeadTask++; break;
    case 35:
      // wait for right-most point to be reached  
      if (HeadTgtCnt < 1) {HeadTask++;} // right-most point reached
      RangeIntR += long(Range); // integrate the right-hand range values
      break;
    case 36:
      // right-most point reached
      // turn towards centre position
      CheckMinRange();
      HeadAdj = true; // full half sweeps can now be integrated
      servoTgt[0] = Head_0; HeadTgtCnt = HeadSpd; HeadTask++; break;
    case 37:
      // wait for centre point to be reached  
      if (HeadTgtCnt < 1) {HeadTask = 30;} // right-most point reached
      RangeIntR += long(Range); // integrate the right-hand range values
      break;

    case 99:  // static slow moving
      // simply set new target once there
      if (HeadCyc == 0) {HeadToAngle(10, 50); HeadCyc = 1;}
      else {HeadToAngle(-10, 50); HeadCyc = 0;}
      HeadDel = 0;                                // no high level control task
      RangeCentreTracker = Range;
      HeadTask++; break;
    case 100: // wait for target to be reached
      if (HeadTgtCnt < 1) {
        HeadDel++; if (HeadDel >= 100) {HeadDel = 0; HeadTask++;}
      }
      RangeCentreTracker = Range; break;
    case 101: // pause at this position
      HeadDel++; if (HeadDel >= 100) {HeadTask = 99;}
      RangeCentreTracker = Range; break;

  }

  switch(HeadMode) {
    // this section oversees what results from above HeadTask scanning code
    // here we use Range values to switch between different head scanning tasks
    case -1:
      // not in ranging mode
      break;
    case 0:
      // wait here whilst no target is being detected whilst scanning
      RangeCentre = Range;
      // track the minimum whilst sweeping
      if (Range < RangeMin) {RangeMin = Range; RangeMinAng = servoVal[0];}
      if (RangeMin <= LtofDet) {
        // a target has been detected, so wait for a minimum
        // once range is increasing, switch to close range scanning
        // we include a 10mm dead band for noise
        if (Range > (RangeMin + 10)) {HeadTask = 10;}
      }
      break;
    case 1:
      // we are performing a target scan
      // only record minimum during full sweep
      if (Range < RangeMin) {RangeMin = Range; RangeMinAng = servoVal[0];}
      break;
    case 2:
      // recalculate centre angle at both sweep end points
      // the adjustment is based on the difference in L/R integrators
      if ((RangeIntR + RangeIntL) != 0) {
        // integrators have accumulated values
        HeadErr = ((RangeIntR - RangeIntL) * 100)/(RangeIntR + RangeIntL);
        RangeAng = RangeAng + int(HeadErr);
        if (abs(HeadErr) > HeadErrTrk) {HeadErrTrk++;} else {HeadErrTrk--;}
      }
      HeadErrTrip = abs(abs(HeadErr) - HeadErrTrk);
    //  Serial.print(RangeIntL); Serial.print("\t"); Serial.println(RangeIntR);
    //  Serial.print(HeadErrTrip); Serial.print("\t"); Serial.println(AngDif);
      // adjust sweep width depending on angle adjustment needed
      if ((HeadErrTrip > 10) && (HeadErrSt)) {
        // significant head angle adjustment so increase sweep angle
        AngDif = AngDifMax; // increase the sweep angle if target is moving
        HeadErrSt = false;  // clear the head error status flag
      } else {
        // decrease the sweep angle progressively to reduce the error
        // then wait for  errors to occur due to target movement
        if (AngDif > AngDifMin) {
          AngDif = max(AngDifMin,AngDif - (AngDif/8));
        } else if (HeadErrTrip < 4) {HeadErrSt = true;}
      }
      // check for lost target in either phase
      if (RangeMin >= LtofDet) {
        // nothing detected in this half cycle, so increment a counter
        HeadCnt++;  // increment the target lost counter
        // could be slightly off centre so widen the sweep
        if (HeadCnt == 1) {AngDif = AngDifMax + 300;};  // no target detected in 1/2 cycle
        if (HeadCnt >= 2) {HeadTask = 1;}               // 1 dead cycles, target has been lost
      } else {HeadCnt = 0;}                             // zero the target lost counter
      RangeAng = min(int(RangeAng),Head_60P); RangeAng = max(int(RangeAng),Head_60N);
      HeadMode = 1; break;
    
    case 3:
      // mode used for autonomous
      // RangeMin tracks Range for lowest value and flags target detection
      if (Range < RangeMin) {
        RangeMin = Range;
        // check to see if something has been detected
        if (RangeMin <= LtofDet) {TrgtDet = true;}
      } break;
  }
  // before leaving track the centre positions
  if (RangePnt) {
    RangePnt = false; // reset the point flag
    anyInt = (RangeCentre - RangeCentreTracker)/2; RangeCentreTracker += anyInt;
  }
  // release stack overflow block and allow this function to be called again from
  // the main loop()
  HeadTaskRun = false;
}

// --------------------------------------------------------------------------------

void GetHeadAngle() {
  // returns servoVal[0] converted to an angle as HeadAng
  if (servoVal[0] < Head_0) {HeadAng = map(servoVal[0],Head_60N,Head_0,30,90);}
  else {HeadAng = map(servoVal[0],Head_0,Head_60P,90,150);}
}

// ----------------------------------------------------------------------

void GetSteeringDrive(int zDir) {
  // calculates steering angle when head is active
  // zDir defines direction, 1 == forward, -1 == backwards
  // at centre RangeAng == Head_0, full left == Head_65N, full right == 65P
  WalkLftDrive = 128; WalkRgtDrive = 128; // set drive factors to max as default
  if (!HeadActive) {return;} // head is not active so no steering needed
  if (zDir == 0) {return;} // set drives equal
  if (zDir > 0) {
    // walking forwards
    if (RangeAng >= Head_0) {
      // target is on the right-hand side
      WalkRgtDrive = map(RangeAng,Head_60P,Head_0,0,128);
    } else {
      // target is on the left-hand side
      WalkLftDrive = map(RangeAng,Head_60N,Head_0,0,128);
    }
  } else {
    // walking backwards
    if (RangeAng >= Head_0) {
      // target is on the right-hand side
      WalkLftDrive = map(RangeAng,Head_60P,Head_0,0,128);
    } else {
      // target is on the left-hand side
      WalkRgtDrive = map(RangeAng,Head_60N,Head_0,0,128);
    }
  }
  WalkLftDrive = max(WalkLftDrive,0); WalkLftDrive = min(WalkLftDrive,128);
  WalkRgtDrive = max(WalkRgtDrive,0); WalkRgtDrive = min(WalkRgtDrive,128);
}


// ----------------------------------------------------------------------

void GoToAngles(int zA1,int zA2,int zA3,int zA4,int zA5,int zA6,int zA7,int zA8, int zCnt) {
  // move to specific servo angles, or -1 means no change
  // number of 20ms steps in movement set by zCnt
  // if zCnt > 0 then loop main loop() within this function
  // if zCnt < 0 then movement will be performed by normal use of loop()
  if (ESC) return; // exit if this flag is set
  
  if (!servoAtt[1]) {attachServos(0);}
  if (zA1 >= 0) {servoTgt[1] = map(zA1,45,135,Ang1_45,Ang1_135);}
  if (zA2 >= 0) {servoTgt[2] = map(zA2,65,147,Ang2_65,Ang2_147);}
  if (zA3 >= 0) {servoTgt[3] = map(zA3,45,135,Ang3_45,Ang3_135);}
  if (zA4 >= 0) {servoTgt[4] = map(zA4,65,147,Ang4_65,Ang4_147);}
  if (zA5 >= 0) {servoTgt[5] = map(zA5,45,135,Ang5_45,Ang5_135);}
  if (zA6 >= 0) {servoTgt[6] = map(zA6,65,147,Ang6_65,Ang6_147);}
  if (zA7 >= 0) {servoTgt[7] = map(zA7,45,135,Ang7_45,Ang7_135);}
  if (zA8 >= 0) {servoTgt[8] = map(zA8,65,147,Ang8_65,Ang8_147);}
  checkTgts(); // ensure target values are within limits
  TgtCnt = abs(zCnt);
  // if TgtVnt is +ve we run loop() from here
  if (zCnt > 0) {loopWhileTgtCnt();} // included here for code efficiency
  // Serial.println(F("GoToAngles Set"));
  // Serial.println(tgtCnt);
}

// ----------------------------------------------------------------------

void GoToRest(int zCnt) {
  // move to the resting position
  // number of 20ms steps in movement set by zCnt
  if (ESC) return;
  
  walkInterval = 20000;   // set to default speed
  // Serial.println(F("> GoToRest()"));
  GoToAngles(68,68,68,68,68,68,68,68,-zCnt);  // go to the resting position
  autoRest = autoRestMax;                     // no need to rest from this position
  autoOFF = 10;                               // servos disabled 200ms after movement stops
}

// ----------------------------------------------------------------------

void GoToStand(int zCnt) {
  // move to the normal standing position
  // number of 20ms steps in movement set by zCnt
  if (ESC) return;
  
  walkInterval = 20000;   // set to default speed
  // Serial.println(F("> GoToStand()"));
  GoToAngles(90,90,90,90,90,90,90,90,-zCnt);  // go to the resting position
  autoRest = autoRestMax;                     // no need to rest from this position
  autoOFF = 10;                               // servos disabled 200ms after movement stops
}

// ----------------------------------------------------------------------

void Head_Engine() {
  // called form the main loop to perform head movement tasks
  if (ESC) return; // exit if this flag is set

  if (HeadPause > 0) {
    // reduce headPause to zero before doing anything
    // headPause counts in 20ms increments, 50 == 1 second
    HeadPause--;
  } else {
    if (HeadTgtCnt > 0) {
      //  Serial.print(HeadTgtCnt); Serial.print("\t");
      if (autoHeadOFF >= 0) {autoHeadOFF = 20;}   // reset auto-OFF timeout
      if (!servoAtt[0]) {attachHeadServo();}
      atHeadTgt = true;                           // set a test flag for achieving target values
      if (HeadTgtCnt == 1) {
        // move head immediately to the target value
        servoVal[0] = servoTgt[0];                // set values to target values
        if (servoLast[0] != servoVal[0]) {servoInst[0].writeMicroseconds(servoVal[0]); servoLast[0] = servoVal[0];}
      }
      else {
        // move head value progressively towards the target value
        servoDelta = servoTgt[0] - servoVal[0];
        if (abs(servoDelta) > 0) {
          servoVal[0] = servoVal[0] + (servoDelta / HeadTgtCnt);
          if (servoLast[0] != servoVal[0]) {servoInst[0].writeMicroseconds(servoVal[0]); servoLast[0] = servoVal[0];}
          atHeadTgt = false;
        }
        if (atHeadTgt) {HeadTgtCnt = 0;} // already at target value so clear the move count
      }
      HeadTgtCnt--; // progressivley reduce the target factor
    //  Serial.println(servoVal[0]);
    }
    // Serial.println(HeadTgtCnt);
    //  Serial.print(HeadTgtCnt); Serial.print("\t");
    //  Serial.println(servoDelta);
  }
}

// ----------------------------------------------------------------------

void HeadToAngle(int zA, int zCnt) {
  // set a specific head angle to move to, +/- 60 degrees
  // number of steps in head movement set by zCnt
  if (ESC) return; // exit if this flag is set
  
  if (!servoAtt[0]) {attachHeadServo();}
  // assume non-linearity and map angle into the correct quadrant for best
  // accuracy
  if (zA >= 0) {
    servoTgt[0] = map(zA,0,60,Head_0,Head_60P);
  } else {
    servoTgt[0] = map(zA,-60,0,Head_60N,Head_0);
  }
  // checkTgts(); // ensure target values are within limits
  HeadTgtCnt = zCnt;
  // Serial.println(F("HeadToAngles Set"));
  // Serial.println(HeadTgtCnt);
}

// ----------------------------------------------------------------------

void HeadToRest() {
  // move the head to the straight forward rest position
  HeadToAngle(0,20); autoHeadOFF = 5; // centre head servo
}

// ----------------------------------------------------------------------

void JoyMoveBwd() {
  // called from main loop to respond to joystick demands
  // this function walks and steers in a backwards direction
  WalkLftDrive = 128; WalkRgtDrive = 128; MoveType = "Backwards";
  if (Walk == 0) {
    // the start of a new walk movement
    if (WalkLast != 3) {
      // previously walking in a different direction
    //  Serial.println(F("Walking Backwards"));
      walkInterval = 60000; StartWalk(3);
    } else {
      // continue walking in same direction
      SetLegsPrevious(); Walk = WalkLast;
    } JoyYi = 0; JoyYT = true; SetLedMode(23);
  } else {
    // ensure joystick values are in the required range
    if (JoyX < 127) {JoyXV = 128 - JoyX;} // convert -ve vector to +ve
    else {JoyXV = JoyX - 127;} // ensure JoyX = 0 - 128
    if (JoyY < 128) {JoyYV = 128 - JoyY;} // convert -ve vector to +ve
    else {JoyYV = JoyY - 127;} // ensure JoyY = 0 - 128
    JoySpd = JoyYV; if (JoyXV > JoyYV) {JoySpd = JoyXV;}
    // we use the larger of the two joystick values to set the speed
    // JoyYi is a tracking filter to slug the initial joystick response
    if (JoyYT) {
      // initially slug the response until it catches up
      if (JoyYi < (JoySpd-2)) {JoyYi += 2;}
      else if (JoyYi > (JoySpd+2)) {JoyYi -= 2;}
      else {JoyYi = JoySpd; JoyYT = false;}
    } else {JoyYi = JoySpd;}
    walkInterval = map((unsigned long)JoyYi,0,128,1,WalkMax);
    walkInterval = 60000/walkInterval;
    // now apply steering
    if (JoyX <= 127) {WalkLftDrive = JoyX + 1; MoveType += " left";}
    else if (JoyX > 128) {WalkRgtDrive = 127-(JoyX-128); MoveType += " right";}
    if ((WalkSpeed == 1) && (C_Dn == 25)) {
      // 'C' held during walking
      LegUpFwd = LegUpMax; WalkLftDrive /= 2; WalkRgtDrive /= 2;
    } else {
      LegUpFwd = LegUpMin;
    }
  }
}

// ----------------------------------------------------------------------

void JoyMoveFwd() {
  // called from main loop to respond to joystick demands
  // this function walks and steers in a backwards direction
  WalkLftDrive = 128; WalkRgtDrive = 128; MoveType = "Forward";
  if (Walk == 0) {
    // the start of a new walk movement
    if (WalkLast != 1) {
      // previously walking in a different direction
    //  Serial.println(F("Walking Forwards"));
      walkInterval = 60000; StartWalk(1);
    } else {
      // continue walking in same direction
      SetLegsPrevious(); Walk = WalkLast;
    } JoyYi = 0; JoyYT = true; SetLedMode(21);
  } else {
    // ensure joystick values are in the required range
    if (JoyX < 127) {JoyXV = 128 - JoyX;} // convert -ve vector to +ve
    else {JoyXV = JoyX - 127;} // ensure JoyX = 0 - 128
    if (JoyY < 128) {JoyYV = 128 - JoyY;} // convert -ve vector to +ve
    else {JoyYV = JoyY - 127;} // ensure JoyY = 0 - 128
    JoySpd = JoyYV; if (JoyXV > JoyYV) {JoySpd = JoyXV;}
    // we use the larger of the two joystick values to set the speed
    // JoyYi is a tracking filter to slug the initial joystick response
    if (JoyYT) {
      // initially slug the response until it catches up
      if (JoyYi < (JoySpd-2)) {JoyYi += 2;}
      else if (JoyYi > (JoySpd+2)) {JoyYi -= 2;}
      else {JoyYi = JoySpd; JoyYT = false;}
    } else {JoyYi = JoySpd;}
    walkInterval = map((unsigned long)JoyYi,0,128,1,WalkMax);
    walkInterval = 60000/walkInterval;
    // now apply steering
    if (JoyX <= 127) {WalkLftDrive = JoyX; MoveType += " left";}
    else if (JoyX > 128) {WalkRgtDrive = 127-(JoyX-128); MoveType += " right";}
    // now set leg up height if in lowest speed
    if ((WalkSpeed == 1) && (C_Dn == 25)) {
      // 'C' held during walking
      LegUpFwd = LegUpMax; WalkLftDrive /= 2; WalkRgtDrive /= 2;
    } else {
      LegUpFwd = LegUpMin;
    }
  }
}

// ----------------------------------------------------------------------

void JoyMoveLft() {
  // called from main loop to respond to joystick demands
  // this function walks and steers in a leftwards direction
  WalkLftDrive = 128; WalkRgtDrive = 128; MoveType = "Walk Left";
  if (Walk == 0) {
    // the start of a new walk movement
    if (WalkLast != 4) {
      // previously walking in a different direction
    //  Serial.println(F("Walking Leftwards"));
      walkInterval = 60000; StartWalk(4);
    } else {
      // continue walking in same direction
      SetLegsPrevious(); Walk = WalkLast;
    } JoyYi = 0; JoyYT = true; SetLedMode(24);
  } else {
    // ensure joystick values are in the required range
    if (JoyX < 127) {JoyXV = 128 - JoyX;} // convert -ve vector to +ve
    else {JoyXV = JoyX - 127;} // ensure JoyX = 0 - 128
    if (JoyY < 128) {JoyYV = 128 - JoyY;} // convert -ve vector to +ve
    else {JoyYV = JoyY - 127;} // ensure JoyY = 0 - 128
    JoySpd = JoyYV; if (JoyXV > JoyYV) {JoySpd = JoyXV;}
    // we use the larger of the two joystick values to set the speed
    // JoyYi is a tracking filter to slug the initial joystick response
    if (JoyYT) {
      // initially slug the response until it catches up
      if (JoyYi < (JoySpd-2)) {JoyYi += 2;}
      else if (JoyYi > (JoySpd+2)) {JoyYi -= 2;}
      else {JoyYi = JoySpd; JoyYT = false;}
    } else {JoyYi = JoySpd;}
    walkInterval = map((unsigned long)JoyYi,0,128,1,WalkMax);
    walkInterval = 60000/walkInterval;
    // now apply steering set by y-axis
    if (JoyY <= 127) {WalkLftDrive = JoyY;  MoveType += " down";}
    else if (JoyY > 128) {WalkRgtDrive = 127-(JoyY-128);  MoveType += " up";}
    if ((WalkSpeed == 1) && (C_Dn == 25)) {
      // 'C' held during walking
      LegUpFwd = LegUpMax; WalkLftDrive /= 2; WalkRgtDrive /= 2;
    } else {
      LegUpFwd = LegUpMin;
    }
  }
}

// ----------------------------------------------------------------------

void JoyMoveRgt() {
  // called from main loop to respond to joystick demands
  // this function walks and steers in a rightwards direction
  WalkLftDrive = 128; WalkRgtDrive = 128; MoveType = "Walk Right";
  if (Walk == 0) {
    // the start of a new walk movement
    if (WalkLast != 2) {
      // previously walking in a different direction
    //  Serial.println(F("Walking Rightwards"));
      walkInterval = 60000; StartWalk(2);
    } else {
      // continue walking in same direction
      SetLegsPrevious(); Walk = WalkLast;
    } JoyYi = 0; JoyYT = true; SetLedMode(22);
  } else {
    // ensure joystick values are in the required range
    if (JoyX < 127) {JoyXV = 128 - JoyX;} // convert -ve vector to +ve
    else {JoyXV = JoyX - 127;} // ensure JoyX = 0 - 128
    if (JoyY < 128) {JoyYV = 128 - JoyY;} // convert -ve vector to +ve
    else {JoyYV = JoyY - 127;} // ensure JoyY = 0 - 128
    // we use the larger of the two joystick values to set the speed
    JoySpd = JoyYV; if (JoyXV > JoyYV) {JoySpd = JoyXV;}
    // JoyYi is a tracking filter to slug the initial joystick response
    if (JoyYT) {
      // initially slug the response until it catches up
      if (JoyYi < (JoySpd-2)) {JoyYi += 2;}
      else if (JoyYi > (JoySpd+2)) {JoyYi -= 2;}
      else {JoyYi = JoySpd; JoyYT = false;}
    } else {JoyYi = JoySpd;}
    walkInterval = map((unsigned long)JoyYi,0,128,1,WalkMax);
    walkInterval = 60000/walkInterval;
    // now apply steering set by y-axis
    if (JoyY <= 127) {WalkRgtDrive = JoyY;  MoveType += " down";}
    else if (JoyY > 128) {WalkLftDrive = 127-(JoyY-128);  MoveType += " up";}
    if ((WalkSpeed == 1) && (C_Dn == 25)) {
      // 'C' held during walking
      LegUpFwd = LegUpMax; WalkLftDrive /= 2; WalkRgtDrive /= 2;
    } else {
      LegUpFwd = LegUpMin;
    }
  }
}

// ----------------------------------------------------------------------

void JoyTurnLft() {
  // called from main loop to respond to joystick demands
  // this function turns in a leftwards direction
  MoveType = "Neutral Left";
  if (Walk == 0) {
    // the start of a new walk movement
    if (WalkLast != 5) {
      // previously walking in a different direction
     Serial.println(F("Turning Left"));
      walkInterval = 60000; StartWalk(5);
    } else {
      // continue walking in same direction
      SetLegsPrevious(); Walk = WalkLast;
    } JoyYi = 0; JoyYT = true; SetLedMode(25);
  } else {
    // ensure joystick values are in the required range
    if (JoyX < 127) {JoyXV = 128 - JoyX;} // convert -ve vector to +ve
    else {JoyXV = JoyX - 127;} // ensure JoyXV = 0 - 128
    // we use the joystick value to set the speed
    JoySpd = JoyXV;
    // JoyYi is a tracking filter to slug the initial joystick response
    if (JoyYT) {
      // initially slug the response until it catches up
      if (JoyYi < (JoySpd-2)) {JoyYi += 2;}
      else if (JoyYi > (JoySpd+2)) {JoyYi -= 2;}
      else {JoyYi = JoySpd; JoyYT = false;}
    } else {JoyYi = JoySpd;}
    walkInterval = map((unsigned long)JoyYi,0,128,1,WalkMax);
    walkInterval = 60000/walkInterval;
  }
}

// ----------------------------------------------------------------------

void JoyTurnRgt() {
  // called from main loop to respond to joystick demands
  // this function turns in a rightwards direction
  MoveType = "Neutral Right";
  if (Walk == 0) {
    // the start of a new walk movement
    if (WalkLast != 6) {
      // previously walking in a different direction
    //  Serial.println(F("Turning Right"));
      walkInterval = 60000; StartWalk(6);
    } else {
      // continue walking in same direction
      SetLegsPrevious(); Walk = WalkLast;
    } JoyYi = 0; JoyYT = true; SetLedMode(26);
  } else {
    // ensure joystick values are in the required range
    if (JoyX < 127) {JoyXV = 128 - JoyX;} // convert -ve vector to +ve
    else {JoyXV = JoyX - 127;} // ensure JoyXV = 0 - 128
    // we use the joystick value to set the speed
    JoySpd = JoyXV;
    // JoyYi is a tracking filter to slug the initial joystick response
    if (JoyYT) {
      // initially slug the response until it catches up
      if (JoyYi < (JoySpd-2)) {JoyYi += 2;}
      else if (JoyYi > (JoySpd+2)) {JoyYi -= 2;}
      else {JoyYi = JoySpd; JoyYT = false;}
    } else {JoyYi = JoySpd;}
    walkInterval = map((unsigned long)JoyYi,0,128,1,WalkMax);
    walkInterval = 60000/walkInterval;
  }
}

// ----------------------------------------------------------------------

void MainTask_Autonomous() {
  // autonomous discovery mode, stays in this mode until reset
  // starts with looking in front of itself
  // if clear, then move forward, a random distance if fixed head
  // if not clear, rotate and look again
  // if rotating head, walk and scan at the same time
  if (MainRun) {return;}
  MainRun = true; // prevent re-entry of task

  if (SubTask != SubLast) {Serial.println("SubTask=" + String(SubTask));}
  SubLast = SubTask;
  switch (SubTask) {
    case 0: // initialise
      Serial.println(F("> MainTask_Autonomous()")); 
      automRnd = 1; SubTask++; break;
    case 1:
      // set legs for going forward, prior to radar scanning
      // branchg to either fixed head (2) or moving head (50) tasks
      automTurn = 0; Walk = 0; walkInterval = 18000; StartWalk(-1); 
      if (HeadActive) {
        SubTask = 50; // autonomous with rotating head
      } else {
        SubTask = 2; // autonomous with fixed head, twisting body
      } SetLedMode(5);
      break;
    case 2:
      //###############################################################################
      //
      //  Fixed head twisting body
      //
      //###############################################################################
      // twist & scan randomly and turning code
      // Start with ranges at max.
      RangeCentre = LtofMax;  RangeLeft = LtofMax; RangeRight = LtofMax;
      SetLedMode(5);
      if (random(2) < 1) {
        // twist left, then right, taking radar readings
        // twist to the left
      //  Serial.println(F("Looking left...")); 
        GoToAngles(45,-1,135,-1,45,-1,135,-1,30);
        delayLoop(150); if (Range <= LtofDet) {RangeLeft = Range;}
      //  Serial.println(Range); 
        // twist to the right
      //  Serial.println(F("Looking right...")); 
        GoToAngles(135,-1,45,-1,135,-1,45,-1,60);
        delayLoop(150); if (Range <= LtofDet) {RangeRight = Range;}
       Serial.println(Range); 
      } else {
        // twist right, then left, taking radar readings
        // twist to the right
       Serial.println(F("Looking right...")); 
        GoToAngles(135,-1,45,-1,135,-1,45,-1,30);
        delayLoop(150); if (Range <= LtofDet) {RangeRight = Range;}
       Serial.println(Range); 
        // twist to the left
       Serial.println(F("Looking left...")); 
        GoToAngles(45,-1,135,-1,45,-1,135,-1,60);
        delayLoop(150); if (Range <= LtofDet) {RangeLeft = Range;}
      //  Serial.println(Range); 
      }
      // back to centre
    //  Serial.println(F("Looking forward...")); 
      GoToAngles(90,-1,90,-1,90,-1,90,-1,30);
      delayLoop(150); if (Range <= LtofDet) {RangeCentre = Range;}
    //  Serial.println(Range); 
      // now check range readings and respond
      if (automTurn == 0) {
        // haven't just turned, so determine direction
        if (RangeLeft <= 325) {
          // something is to my left so turn right and repeat process measurements
          automRnd = 1; // reset walking to short distance at first
          if (RangeRight > RangeLeft) {
            // object on the left is closer, so turn right
            NeutralTurnRightEcho(60,15000); automTurn = 1;
          } else {
            // object on the right is closer, so turn left
            NeutralTurnLeftEcho(60,15000); automTurn = -1;
          } break;
        }
        else if (RangeRight <= 325) {
          // something is to my right so turn left and repeat process measurements
          automRnd = 1; // reset walking to short distance at first
          if (RangeLeft > RangeRight) {
            // object on the right is closer, so turn left
            NeutralTurnLeftEcho(60,15000); automTurn = -1;
          } else {
            // object on the left is closer, so turn right
            NeutralTurnRightEcho(60,15000); automTurn = 1;
          } break;
        }
        else if (RangeCentre <= 325) {
          // something is in front of me, but sides are clear
          // so turn randomly left/right
        //  Serial.println(F("Object ahead...")); 
          automRnd = 1; // reset walking to short distance at first
          if (random(2) > 0) {
          //  Serial.println(F("Turning Left...")); 
            NeutralTurnLeftEcho(90,15000); automTurn = -1;
          } else {
          //  Serial.println(F("Turning Right...")); 
            NeutralTurnRightEcho(90,15000); automTurn = 1;
          }
        } 
        else {SubTask = 3; break;}
      }
      else if ((RangeCentre <= 325) || (RangeLeft <= 325) || (RangeRight <= 325)) {
        // despite turning an object has still been detected
        // so we continue turning in the same direction
        automRnd = 1; // reset walking to short distance at first
        if  (automTurn > 0) {
          // just turned to the right, so continue turning if object is present
          NeutralTurnRightEcho(60,15000); Walk = 0;
        } else {
          // just turned to the left, so continue turning if object is present
          NeutralTurnLeftEcho(60,15000); Walk = 0;
        }
      }
      else {SubTask = 3;}
      break;
    case 3:
      // all is clear so walk for a random distance, then re-measure
      // random distance increases provided nothing is detected along the way
      //  Serial.println(F("Walking forward..."));
      walkInterval = 15000 + (1000 * random(7)); // vary walking speed
      StartWalk(-1); Walk = 1;
      loopWhileWalkEcho(88 * automRnd);
      if (automRnd < 10) {automRnd++;}  // increase walking range
      delayLoop(200);                   // short pause before range scanning
      SubTask = 1;                      // restart scanning process
      break;

    case 50:
      //###############################################################################
      //
      //  Moving head moving body
      //
      //###############################################################################
      // autonomouse with moving head
      // at this point the robot is stationary and checking for a wall
      // the head is constantly scanning left to right
      if (CalcCnt > 3) {
        // head has performed at least 4 half-sweeps
        CalcCnt = 0;
        if ((TrgtDet) && (RangeMin <= LtofStop)) {
          // if wall or target is close go into 360 turning mode
          // whilst in this mode move the head more rapidly
          headInterval = 10000;
          if (RangeAng >= Head_0) {
            // wall is closer on the left
            Serial.println("Turning right");
            NeutralTurnRight(60,15000); // start with a short turn no range detection
            NeutralTurnRightEcho(360,15000);
            Walk = 0;
          } else {
            // wall is closer on the right
            Serial.println("Turning left");
            NeutralTurnLeft(60,15000); // start with a short turn no range detection
            NeutralTurnLeftEcho(360,15000);
            Walk = 0;
          }
          if (ESC == 0) {SubTask = 100;} // no way out found
          else {
            // found a way out so slow down head and start walking again
            ESC = 0; headInterval = 20000;
            StartWalk(1); SubTask++;
            Serial.println("Walking forward");
          }
        } else {
          // no target or wall was detected so start walking
          StartWalk(1); SetLedMode(5); SubTask++;
        }
      } break;
    case 51:
      // walking whilst looking for targets/walls
      if (CalcCnt > 0) {
        // a new sweep has been completed so recalculate steering
        CalcCnt = 0;
        GetSteeringDrive(1); // get steering in forward direction
      }
      // whilst walking constrantly look for a close wall or target
      Serial.println("RangeMin = " + String(RangeMin));
      if (RangeMin <= LtofStop) {
        // a target or wall was detected close range so stop
        Serial.println("Target detected");
        Walk = 0; SetLedMode(5);
        GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1); // put all feet down immediately
        SubTask = 50; // go back to start point
      }
      break;

    case 100:
      // here because turning 360 did not find a way out
      HeadTask = 0; HeadToRest();
      automCnt = 500 + random(1000); // create a delay for a minimum of 10 seconds
      break;
    case 101:
      // wait here then restart
      automCnt--; if (automCnt < 1) {SubTask = 0;}
      break;

  } MainRun = false;
}

// ----------------------------------------------------------------------

void MainTask_BackAway() {
  // Backs away from an approaching target, later recovers position
  // Stays in these modes until reset
  if (MainRun) {return;}
  
  MainRun = true; // prevent re-entry of task
  // Use a tracking filter on the range to prevent sudden movements during tracking
  // LED range mode sustained throughout, unless returning to start position
  switch (SubTask) {
    case 0: // initialise
      // set the range limits for this mode
      DispMode = 1;     // display movements in monitor app if connected
      R0 = 55; RM = 335; Walked = 0;
      Serial.println(F("> MainTask_BackAway()"));
      R1 = 125; R2 = 160; R3 = 325;
      SubTask = 1; break;
      
    case 1: // initialise legs positions
      // Set a walk direction as backwards
      autoRest = 501;  // set auto-rest, robot is already standing
      MoveType = "Setting";   // type of move taking place
      // start expecting to go backward
      StartWalk(-3);
      SetLedMode(5);  // set default LED range mode
      SubTask = 2; break;
      
    case 2: // wait for a wall to appear and if necessary change direction
      // only move onto SubTask 3 when a wall appears
      // auto-rest will time-out if left in this state
      MoveType = "Looking";   // type of move taking place
      walkInterval = 60000;   // extend interval to slow robot down
    //  Serial.println(Walk);
      if ((RangeCentreTracker >= R0) && (RangeCentreTracker <= R3)) {
        // this applies to both zMode == 0 and zMode == 1
        SubTask = 3; Walk = abs(Walk);
        if (Walk != 3) {
          // change walk direction from forwards to backwards
          Walk = 0; WalkCnt += 44; if (WalkCnt > 87) {WalkCnt -= 88;}
          WalkNext = 3; SetLegStates(); // set leg heights
          SubTask = 99;
        }
      }
      break;
      
    case 3: // scan for object and respond by walking backwards
      autoRest = 501;  // set auto-rest, robot is already standing
      SetLedMode(5);  // keep in ranging mode whilst responding to targets
      if (RangeCentreTracker >= R0) {
        AnyLong = RangeCentreTracker - 10; // introduce 10mm of hysteresis
        if (RangeCentreTracker > RM) {
          // target moved out of range so stop
          SubTask = 4;
        } else if ((AnyLong <= R3) && (AnyLong >= R1)) {
          // close wall detected so backaway speed depends on how close it is
          MoveType = "Retreat";  // type of move taking place
          CheckWalkDir(3); // ensure we are going to walk backward
          GetSteeringDrive(-1);
          AnyLong = map(AnyLong,R3,R2,1,12);
          AnyLong = min(AnyLong,12L); // set max speed == 12
          AnyLongDamp();
        } else if ((AnyLong <= R1) && (AnyLong >= R0)) {
          // too close wall detected so slow down backaway speed
          MoveType = "Stop!";  // type of move taking place
          CheckWalkDir(3); // ensure we are going to walk backward
          GetSteeringDrive(-1);
          AnyLong = map(AnyLong, R0,R1,1,12);
          AnyLongDamp();
        } else {
          // in centre deadband so stop
          SubTask = 4;}
      } else {
        // no wall detected but one had been previously
        SubTask = 4;
      } break;
      
    case 4: // stop moving and put feet down and wait
      // reset speed to slow
      MoveType = "Waiting";          // type of move taking place
      Walk = -abs(Walk); walkInterval = 60000; SetLedMode(5);
      MemStore(); // store current values
      GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1); // put all feet down immediately
      Sit = false; SubTask++; anyMilli = millis(); break;
      
    case 5: // wait for a valid range then return to previous movement
      // the robot has moved, but is now stationary
      autoRest = 501;  // set auto-rest, robot is already standing
      if ((RangeCentreTracker >= R0) && (RangeCentreTracker <= R3)) {
        // valid reverse range
        if (Walk == -3) {
          // previously walking backwards so raise lowered legs and continue
          MemRead(); TgtCnt = 1; SubTask = 3; Walk = 3;
        } else {SubTask = 2;} // walking direction changing
      } else {
        // object has stopped approaching, but wait before sitting down
        if (((millis() - anyMilli) > 2000) && !Sit) {
          // waited 2 seconds before sitting
          // GoToAngles(-1,LegDn,-1,LegDn,90,165,90,165,10);     // put back legs down to improve ranging
          Sit = true;
        }
      }
      if (!HeadActive) {
        // Check after a delay to see if a return to start movemnent is needed
        // This feature does not work in head active mode, which incorporates steering
        autoRest = 501;  // set auto-rest, robot is already standing
        MoveType = "Timing " + String((millis() - anyMilli)/1000);  // type of move taking place
        if ((millis() - anyMilli) >= 5000) {
          if (Walked < -44) {SubTask++;} // at least half a stride backwards, so do it
          else {
            // barely moved so go back into looking mode
            Walk = 0; SubTask = 1;
            GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1);
          }
        }
      } break;
      
    case 6: // stop waiting
      walkInterval = 40000; Walk = 0; // stop walk engine
      WalkCnt -= 44; if (WalkCnt < 0) {WalkCnt += 88;}
      SetLegStates(); // set leg heights
      SetLedMode(5); SubTask++; break;
      
    case 7:
      // prepare to walk back to the start point
      // change direction of walk movement to forward
      Walk = 1; SetLedMode(21); // start moving forward
      MainDel = 0; SubTask++; break;
      
    case 8: // loop here until we have reached start point, or seen another wall
    //  Serial.print(F("Walked = ")); Serial.println(Walked);
      autoRest = 501;  // set auto-rest, robot is already standing
      if (((RangeCentreTracker >= R0) && (RangeCentreTracker <= R3)) || (Walked >= -1)) {
        // Object has re-appeared so start backing away again after so many counts
        MainDel++; if (MainDel > 3) {
          Walk = 0; Walked = 0; SubTask = 1;
          GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1);
        }
      } else {MainDel = 0;} // zero timer
      // If some way from start point, slowly speed up
      // If approaching start point slow down
      MoveType = "Return";          // type of move taking place
      if (Walked > -100) {
        // slow down, nearing end of count
        if (walkInterval < 40000) {walkInterval += 200;}
      } else if (
        // slowly speed up whilst walking
        walkInterval > 10000) {walkInterval -= 100;
      } break;
    case 99:
      // one cycle to allow leg heights to be set before moving
      Walk = WalkNext; SubTask = 3; break;
  } MainRun = false;
  // Serial.print("0"); Serial.print("\t"); Serial.println(RangeCentreTracker);
}

// ----------------------------------------------------------------------

void MainTask_ESC() {
  // called when a task change is needed in the middle of a moving task
  // setting ESC > 0 will exit the current task quickly, within one
  // cycle of the main task loops. After that we move to standing position
  // Serial.println("ESC " + String(ESC));
  if (ESC > 0) {ESC--; SubTask = 0; return;}
  
  // any movements should have now ended
  if ((TgtCnt == 0) && (HeadTgtCnt == 0)) {SetMainTask(0);}
}

// ----------------------------------------------------------------------

void MainTask_POST() {
  // run immediately after normal RESET to go to rest position
  // called every 20ms from main loop
  if (MainDel > 0) {MainDel--; return;}
  
  MainRun = true; // inhibit re-entry of this code
  switch (SubTask) {
    case 0: // start with a 2s delay
      Serial.println("Going to rest...");
      MainDel = 100; SubTask++; break;
    case 1: // set all LEDs red as warning of pending movement
      LED_Set_All(5,0,0);
      MainDel = 50; SubTask++;
      break;
    case 2: // move head and legs to rest potition
      SubTask++;
      HeadToRest();   // centre head servo
      GoToRest(50);   // rest legs beneath robot body
      break;
    case 3: // wait for movement to end
      if ((HeadTgtCnt == 0) && (TgtCnt == 0)) {SubTask++;}
      break;
    case 4: // set default task 0
      Serial.println("At rest");
      SetMainTask(0);
      break;
  }
  MainRun = false;
}

// ----------------------------------------------------------------------

void MainTask_Range() {
  // default ranging task, robot is in standing position with laser active
  switch (SubTask) {
    case 0: // initialise
      GoToStand(50);
      DispMode = 1;           // display range in monitor app if connected
      autoRest = 501;        // set auto-rest, robot is already standing
      RangeMax = LtofLimit;   // the current limited to Range measurements
      RangeMin = LtofMin;     // the current lower limited to Range measurements
      SubTask++; break;
    case 1:
      // monitor ranging function for moving targets
      // if movement detected then reset auto-rest function, otherwise
      // the robot will power down if range remains constannt
      if (abs(RangeDiff) > 20) {autoRest = 501;}
    //  Serial.println("RangeDiff = " + String(RangeDiff));
    //  Serial.println("autoRest = " + String(autoRest));
      break;
  }
}

// ----------------------------------------------------------------------

void MainTask_TEST() {
  // run immediately after POST when in TEST mode
  // called every 20ms from main loop
  // LEDs will start with random pattern
  if (MainDel > 0) {MainDel--; return;}
  
  switch (SubTask) {
    case 0: // start with a 2s delay
      Serial.println("MainTask_TEST active...");
      MainDel = 100; SubTask++; break;  // 2 sec delay
    case 1: // send range value to the serial monitor
      Serial.println("Entering ranging mode");
      SetLedMode(5); SubTask++; break;
    case 2:
      Serial.print("Range ="); Serial.println(Range);
      MainDel = 10; break;  // update at 5 Hz
  } 
}

// ----------------------------------------------------------------------

void MainTask_TrackWall(int zMode) {
  // this code is dual function, depending on the zMode value
  // zMode = 0 backs away from an approaching target, later recovers position
  // zMode = 1 maintains a fixed distance from a wall, unless wall disappears
  // both stay in these modes until reset
  if (MainRun) {return;}
  
  MainRun = true; // prevent re-entry of task
  // use a tracking filter on the range to prevent sudden movements during tracking
  // LED range mode sustained throughout, unless returning to start position
  switch (SubTask) {
    case 0: // initialise
      // set the range limits for either mode
      DispMode = 1;     // display movements in monitor app if connected
      R0 = 55; RM = 335; Walked = 0;
      if (zMode == 0) {
        Serial.println(F("> MainTask_BackAway()"));
        R1 = 125; R2 = 160; R3 = 325;
      } else {
        Serial.println(F("> MainTask_TrackWall()"));
        R1 = 80; R2 = 130; R3 = 170; R4 = 200; R5 = 230; R6 = 305;
      } SubTask = 1; break;
      
    case 1: // initialise legs positions
      // depending on zMode set a walk direction
      autoRest = 501;  // set auto-rest, robot is already standing
      MoveType = "Setting";   // type of move taking place
      if (zMode == 0) {
        // start expecting to go backward
        StartWalk(-3);
      } else {
        // start expecting to go forward
        StartWalk(-1);
      }
      SetLedMode(5);  // set default LED range mode
      SubTask = 2; break;
      
    case 2: // wait for a wall to appear and if necessary change direction
      // only move onto SubTask 3 when a wall appears
      // auto-rest will time-out if left in this state
      MoveType = "Looking";   // type of move taking place
      walkInterval = 60000;   // extend interval to slow robot down
    //  Serial.println(Walk);
      if ((RangeCentreTracker >= R0) && (RangeCentreTracker <= R3)) {
        // this applies to both zMode == 0 and zMode == 1
        SubTask = 3; Walk = abs(Walk);
        if (Walk != 3) {
          // change walk direction from forwards to backwards
          Walk = 0; WalkCnt += 44; if (WalkCnt > 87) {WalkCnt -= 88;}
          WalkNext = 3; SetLegStates(); // set leg heights
          SubTask = 99;
        }
      } else if ((zMode == 1) && (RangeCentreTracker >= R4) && (RangeCentreTracker <= RM)) {
        // this only applies to zMode == 1
        SubTask = 3; Walk = abs(Walk);
        if (Walk != 1) {
          // change walk direction from backards to forwards
          Walk = 0; WalkCnt -= 44; if (WalkCnt < 0) {WalkCnt += 88;}
          WalkNext = 1; SetLegStates(); // set leg heights
          SubTask = 99;
        }
      } break;
      
    case 3: // scan for wall and respond by walking
      autoRest = 501;  // set auto-rest, robot is already standing
      SetLedMode(5);  // keep in ranging mode whilst responding to targets
      if (RangeCentreTracker >= R0) {
        AnyLong = RangeCentreTracker - 10; // introduce 10mm of hysteresis
        if (RangeCentreTracker > RM) {
          // target moved out of range so stop
          SubTask = 4;
        } else if ((zMode > 0) && (AnyLong >= R6)) {
          // if walking forward let it continue, but if not start moderately
          MoveType = "Chase";  // type of move taking place
          CheckWalkDir(1); // ensure we are going to walk forward
          GetSteeringDrive(1);
          AnyLong = map(AnyLong,RM,R5,1,12);
          AnyLongDamp();
        } else if ((zMode > 0) && (AnyLong >= R5)) {
          // if walking forward let it continue, but if not start moderately
          MoveType = "Chase";  // type of move taking place
          CheckWalkDir(1); // ensure we are going to walk forward
          GetSteeringDrive(1);
          AnyLong = 12; // set to max speed
          AnyLongDamp();
        } else if ((zMode > 0) && (AnyLong >= R4)) {
          // far wall detected so walking speed depends on how close it is
          MoveType = "Chase";  // type of move taking place
          CheckWalkDir(1); // ensure we are going to walk forward
          GetSteeringDrive(1);
          AnyLong = map(AnyLong, R4,R5,1,12);
          AnyLongDamp();
        } else if ((AnyLong <= R3) && (AnyLong >= R1)) {
          // close wall detected so backaway speed depends on how close it is
          MoveType = "Retreat";  // type of move taking place
          CheckWalkDir(3); // ensure we are going to walk backward
          GetSteeringDrive(-1);
          AnyLong = map(AnyLong,R3,R2,1,12);
          AnyLong = min(AnyLong,12L); // set max speed == 12
          AnyLongDamp();
        } else if ((AnyLong <= R1) && (AnyLong >= R0)) {
          // too close wall detected so slow down backaway speed
          MoveType = "Stop!";  // type of move taking place
          CheckWalkDir(3); // ensure we are going to walk backward
          GetSteeringDrive(-1);
          AnyLong = map(AnyLong, R0,R1,1,12);
          AnyLongDamp();
        } else {
          // in centre deadband so stop
          SubTask = 4;}
      } else {
        // no wall detected but one had been previously
        SubTask = 4;
      } break;
      
    case 4: // stop moving and put feet down and wait
      // reset speed to slow
      MoveType = "Waiting";          // type of move taking place
      Walk = -abs(Walk); walkInterval = 60000; SetLedMode(5);
      MemStore(); // store current values
      GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1); // put all feet down immediately
      SubTask++; anyMilli = millis(); break;
      
    case 5: // wait for a valid range then return to previous movement
      // the robot has moved, but is now stationary
      autoRest = 501;  // set auto-rest, robot is already standing
      if ((RangeCentreTracker >= R0) && (RangeCentreTracker <= R3)) {
        // valid reverse range
        if (Walk == -3) {
          // previously walking backwards so raise lowered legs and continue
          MemRead(); TgtCnt = 1; SubTask = 3; Walk = 3;
        } else {SubTask = 2;} // walking direction changing
      } else if ((zMode > 0) && (RangeCentreTracker >= R4) && (RangeCentreTracker <= RM)) {
        // valid forward range
        if (Walk == -1) {
          // previously walking forwards so lower legs and continue
          MemRead(); TgtCnt = 1; SubTask = 3; Walk = 1;
        } else {SubTask = 2;} // walking direction changing
      } else if ((zMode == 0) && (!HeadActive)) {
        // check after a delay to see if a return to start movemnent is needed
        // this feature does not work in head active mode, which incorporates steering
        autoRest = 501;  // set auto-rest, robot is already standing
        MoveType = "Timing " + String((millis() - anyMilli)/1000);  // type of move taking place
        if ((millis() - anyMilli) >= 5000) {
          if (Walked < -44) {SubTask++;} // at least half a stride backwards, so do it
          else {
            // barely moved so go back into looking mode
            Walk = 0; SubTask = 1;
            GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1);
          }
        }
      } break;
      
    case 6: // stop waiting
      walkInterval = 40000; Walk = 0; // stop walk engine
      WalkCnt -= 44; if (WalkCnt < 0) {WalkCnt += 88;}
      SetLegStates(); // set leg heights
      SetLedMode(5); SubTask++; break;
      
    case 7:
      // prepare to walk back to the start point
      // change direction of walk movement to forward
      Walk = 1; SetLedMode(21); // start moving forward
      SubTask++; break;
      
    case 8: // loop here until we have reached start point, or seen another wall
    //  Serial.print(F("Walked = ")); Serial.println(Walked);
      autoRest = 501;  // set auto-rest, robot is already standing
      if (((RangeCentreTracker >= R0) && (RangeCentreTracker <= R3)) || (Walked >= -1)) {
        Walk = 0; SubTask = 1;
        GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-1);
      }
      // if some way from start point, slowly speed up
      // if approaching start point slow down
      MoveType = "Return";          // type of move taking place
      if (Walked > -100) {
        // slow down, nearing end of count
        if (walkInterval < 40000) {walkInterval += 200;}
      } else if (
        // slowly speed up whilst walking
        walkInterval > 10000) {walkInterval -= 100;
      } break;
    case 99:
      // one cycle to allow leg heights to be set before moving
      Walk = WalkNext; SubTask = 3; break;
  } MainRun = false;
  // Serial.print("0"); Serial.print("\t"); Serial.println(RangeCentreTracker);
}

// ----------------------------------------------------------------------

void MoveLegsToTgt() {
  // moves all 8 legs to their target positions using TgtCnt
  if (TgtCnt == 1) {
    // move values immediately to the target value
    for (int zP = 1; zP <= 8;zP++) {
      servoVal[zP] = servoTgt[zP]; // set values to target values
      servoInst[zP].writeMicroseconds(servoVal[zP]); servoLast[zP] = servoVal[zP];
    }
  } else {
    // move values progressively towards the target values
    atTgt = true; // set a test flag for achieving target values
    for (int zP = 1; zP <= 8;zP++) {
      servoDelta = servoTgt[zP] - servoVal[zP];
      if (abs(servoDelta) > 0) {
        servoVal[zP] = servoVal[zP] + (servoDelta / TgtCnt);
        if (servoLast[zP] != servoVal[zP]) {servoInst[zP].writeMicroseconds(servoVal[zP]); servoLast[zP] = servoVal[zP];}
        atTgt = false;
      }
    }
    if (atTgt) {TgtCnt = 0;} // already at target values so clear the move count
  }
  if (TgtCnt > 0) {TgtCnt--;} // progressivley reduce the target factor
  // Serial.println(TgtCnt);
}

// ----------------------------------------------------------------------

void MoveToAngles(int zA1,int zA2,int zA3,int zA4,int zA5,int zA6,int zA7,int zA8) {
  // move to specific servos angle by raising and lowering legs
  // number of 20ms steps in movement set by zCnt
  if (ESC) return; // exit if this flag is set
  
  if (!servoAtt[0]) {attachServos(0);}
  anyDel = 120;
  GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,3);
  SetAngle(11,zA1); // get the target angle in microseconds, but don't move the servo
  if (servoVal[1] != servoTgt[1]) {
    SetAngle(2,LegUp); delayLoop(anyDel); SetAngle(1,zA1); delayLoop(anyDel);
  } SetAngle(2,zA2);
  SetAngle(13,zA3); // get the target angle in microseconds, but don't move the servo
  if (servoVal[3] != servoTgt[3]) {
    SetAngle(4,LegUp); delayLoop(anyDel); SetAngle(3,zA3); delayLoop(anyDel);
  } SetAngle(4,zA4);
  SetAngle(15,zA5); // get the target angle in microseconds, but don't move the servo
  if (servoVal[5] != servoTgt[5]) {
    SetAngle(6,LegUp); delayLoop(anyDel); SetAngle(5,zA5); delayLoop(anyDel);
  } SetAngle(6,zA6);
  SetAngle(17,zA7); // get the target angle in microseconds, but don't move the servo
  if (servoVal[7] != servoTgt[7]) {
    SetAngle(8,LegUp); delayLoop(anyDel); SetAngle(7,zA7); delayLoop(anyDel);
  } SetAngle(8,zA8);  delayLoop(anyDel);
}

// --------------------------------------------------------------------------------

void NeutralTurnLeft(int16_t zAng, unsigned long zInterval) {
  // neutral turn to the left by a given angle, and speed
  // any range values are ignored
  unsigned long zwalkInterval = walkInterval; // save current walk period
  walkInterval = zInterval;
  StartWalk(-5); Walk = 5;
  loopWhileTurn(zAng); Walk = 0;
  walkInterval = zwalkInterval;
}

// --------------------------------------------------------------------------------

void NeutralTurnLeftEcho(int16_t zAng, unsigned long zInterval) {
  // neutral turn to the left by a given angle, and speed
  // note that speed can also be set by walkInterval loop period
  // the actual angle turned is approximately (zAng * 28)/45
  // if zAng < 0 then don't move legs to the drop position, assume already there
  unsigned long zwalkInterval = walkInterval; // save current walk period
  walkInterval = zInterval;
  StartWalk(-5); Walk = 5;
  loopWhileTurnEcho(zAng); Walk = 0;
  walkInterval = zwalkInterval;
}

// --------------------------------------------------------------------------------

void NeutralTurnRight(int16_t zAng, unsigned long zInterval) {
  // neutral turn to the right by a given angle, step and speed
  // any range values are ignored
  unsigned long zwalkInterval = walkInterval; // save current walk period
  walkInterval = zInterval;
  StartWalk(-6); Walk = 6;
  loopWhileTurn(zAng); Walk = 0;
  walkInterval = zwalkInterval;
}

// --------------------------------------------------------------------------------

void NeutralTurnRightEcho(int16_t zAng, unsigned long zInterval) {
  // neutral turn to the right by a given angle, step and speed
  // any range values are ignored
  unsigned long zwalkInterval = walkInterval; // save current walk period
  walkInterval = zInterval;
  StartWalk(-6); Walk = 6;
  loopWhileTurnEcho(zAng); Walk = 0;
  walkInterval = zwalkInterval;
}

// --------------------------------------------------------------------------------

void SetHeadTgtCnt(int zAngDif) {
  // sets the movement count value based on the displacement AngDif
  HeadTgtCnt = (16 * abs(zAngDif))/AngDifMax;
  if (HeadTgtCnt < 1) {HeadTgtCnt = 1;}
}

// ----------------------------------------------------------------------

void SetLegStates() {
  // set leg heights depending on WalkCnt, used when reversing direction
  LegSt2 = LegDn; LegSt4 = LegDn; LegSt6 = LegDn; LegSt8 = LegDn;
  if ((WalkCnt > 1) && (WalkCnt < 44)) {LegSt4 = LegUp; LegSt8 = LegUp;}
  else if (WalkCnt > 45) {LegSt2 = LegUp; LegSt6 = LegUp;}
  GoToAngles(-1,LegSt2,-1,LegSt4,-1,LegSt6,-1,LegSt8,-1);
}

// ----------------------------------------------------------------------

void SetLegsDown() {
  // stores the leg positions then puts them all down on the floor
  MemStore();
  SetAngle(2,LegDn); SetAngle(4,LegDn); SetAngle(6,LegDn); SetAngle(8,LegDn);
}

// ----------------------------------------------------------------------

void SetLegsPrevious() {
  // recover stored leg positions
  MemRead(); SetToTgtNow();
}

// --------------------------------------------------------------------------------

void SetWalkMax() {
  // set the value of WalkMax based on WalkSpeed
  // think of WalkSpeed as the gear you are in
  switch(WalkSpeed) {
    case 1: WalkMax =  4; break;
    case 2: WalkMax =  7; break;
    case 3: WalkMax = 10; break;
    case 4: WalkMax = 13; break;
    case 5: WalkMax = 15; break;
  }
  // if not currently changing the speed, then record LedMode so we can return to it
  if (LedMode != 3) {LedNext = LedMode; SetLedMode(3);}
}

// ----------------------------------------------------------------------

void StartWalk(int zWalk) {
  // initiates one of four walk directions
  // if zWalk is -ve then position feet but don't walk
  // 0 - stationary
  // 1 - walk forward
  // 2 - walk to the right
  // 3 - walk backwards
  // 4 - walk to the left
  // 5 - neutral turn to the left
  // 6 - neutral turn to the right
  Walk = 0; WalkCnt = 21; // stop walking and set start points
  switch (abs(zWalk)) {
    case 0: MoveToAngles(111,LegDn,112,LegDn,114,LegDn,113,LegDn); break;
    case 1: MoveToAngles(111,LegDn,112,LegDn,114,LegDn,113,LegDn); break;
    case 2: MoveToAngles(67,LegDn,69,LegDn,68,LegDn,66,LegDn); break;
    case 3: MoveToAngles(114,LegDn,113,LegDn,111,LegDn,112,LegDn); break;
    case 4: MoveToAngles(68,LegDn,66,LegDn,67,LegDn,69,LegDn); break;
    case 5: MoveToAngles(69,LegDn,67,LegDn,69,LegDn,67,LegDn); break;
    case 6: MoveToAngles(67,LegDn,69,LegDn,67,LegDn,69,LegDn); break;
  }
  Walk = zWalk; // start walking if Walk > 0
  if (zWalk >= 0) {
    SetLedMode(Walk + 20);
  }
}

// ----------------------------------------------------------------------

void Walk_Engine() {
  // called from the main loop on a timer
  // Walk values:
  // 1  walking forwards
  // 2  walking towards the right, sideways
  // 3  walking backwards
  // 4  walking towards the left, sideways
  // 5  neutral turning towards the left
  // 6  neutral turning towards the right
  if (ESC) {return;}
  
  if (Walk > 0) {
    WalkCnt++; if (WalkCnt > 87) {WalkCnt = 0;} // increment and restart
    // Serial.println(WalkCnt);
    switch (Walk) {
      case 1: // walking forwards
        // front left leg &  rear right leg
        Walked++; // increase how far walked counter
        if (WalkCnt < 1) {SetAngle(2, LegDn); SetAngle(6, LegDn);}
        else if (WalkCnt == 22) {SetAngle(4,LegUpFwd); SetAngle(8,LegUp);} // Up start condition
        else if (WalkCnt < 46) {SetAngle90To135(1,90+((WalkCnt * WalkLftDrive)/128));
          SetAngle90To135(5,90+(((45-WalkCnt)*WalkRgtDrive)/128));}
        else if (WalkCnt == 46) {SetAngle(2,LegUpFwd); SetAngle(6,LegUp);}
        else {SetAngle90To135(1,90+(((87-WalkCnt)*WalkLftDrive)/128));
          SetAngle90To135(5,90-(((42-WalkCnt)*WalkRgtDrive)/128));}
        // front right leg &  rear left leg
        if (WalkCnt > 43) {SetAngle(4,LegDn); SetAngle(8,LegDn);
          SetAngle90To135(3,90+(((WalkCnt-44)*WalkRgtDrive)/128));
          SetAngle90To135(7,90+(((89-WalkCnt)*WalkLftDrive)/128));}
        else if (WalkCnt > 2) {SetAngle90To135(3,90+(((43-WalkCnt)*WalkRgtDrive)/128));
          SetAngle90To135(7,90+(((WalkCnt+2)*WalkLftDrive)/128));}
        else if (WalkCnt > 1) {SetAngle(4,LegUpFwd); SetAngle(8,LegUp);}
        else {SetAngle(4,LegDn); SetAngle(8,LegDn);
          SetAngle90To135(3,90+(((WalkCnt+44)*WalkRgtDrive)/128));
          SetAngle90To135(7,90+(((1-WalkCnt)*WalkLftDrive)/128));} break;
          
      case 2: // walking towards the right
        // front right leg &  rear left leg
        if (WalkCnt < 1) {SetAngle(4, LegDn); SetAngle(8, LegDn);}
        else if (WalkCnt < 46) {SetAngle45To90(3,90-((WalkCnt*WalkLftDrive)/128));
          SetAngle45To90(7,90-(((45-WalkCnt)*WalkRgtDrive)/128));}
        else if (WalkCnt == 46) {SetAngle(4,LegUpFwd); SetAngle(8,LegUp);}
        else {SetAngle45To90(3,90-(((87-WalkCnt)*WalkLftDrive)/128));
          SetAngle45To90(7,90-(((WalkCnt-42)*WalkRgtDrive)/128));}
        // front left leg &  rear right leg
        if (WalkCnt > 43) {SetAngle(2,LegDn); SetAngle(6,LegDn);
          SetAngle45To90(1,90+(((WalkCnt-88)*WalkLftDrive)/128));
          SetAngle45To90(5,90-(((WalkCnt-44)*WalkRgtDrive)/128));}
        else if (WalkCnt == 22) {SetAngle(2,LegUp); SetAngle(6,LegUpFwd);} // Up start condition
        else if (WalkCnt > 2) {SetAngle45To90(1,90-(((2+WalkCnt)*WalkLftDrive)/128));
          SetAngle45To90(5,90-(((43-WalkCnt)*WalkRgtDrive)/128));}
        else if (WalkCnt > 1) {SetAngle(2,LegUp); SetAngle(6,LegUpFwd);}
        else {SetAngle(2,LegDn); SetAngle(6,LegDn);
          SetAngle45To90(1,90+(((WalkCnt-1)*WalkLftDrive)/128));
          SetAngle45To90(5,90-(((WalkCnt+44)*WalkRgtDrive)/128));} break;
          
      case 3: // walking backwards
        // front left leg &  rear right leg
         Walked--; // decrease how far walked counter
       if (WalkCnt < 1) {SetAngle(2, LegDn); SetAngle(6, LegDn);}
        else if (WalkCnt < 46) {SetAngle90To135(1,90+(((45-WalkCnt)*WalkLftDrive)/128));
          SetAngle90To135(5,90+((WalkCnt * WalkRgtDrive)/128));}
        else if (WalkCnt == 46) {SetAngle(2,LegUp); SetAngle(6,LegUpFwd);}
        else {SetAngle90To135(1,90-(((42-WalkCnt)*WalkLftDrive)/128));
          SetAngle90To135(5,90+(((87-WalkCnt)*WalkRgtDrive)/128));}
        // front right leg &  rear left leg
        if (WalkCnt > 43) {SetAngle(4,LegDn); SetAngle(8,LegDn);
          SetAngle90To135(3,90+(((89-WalkCnt)*WalkRgtDrive)/128));
          SetAngle90To135(7,90+(((WalkCnt-44)*WalkLftDrive)/128));}
        else if (WalkCnt == 22) {SetAngle(4,LegUp); SetAngle(8,LegUpFwd);} // Up start condition
        else if (WalkCnt > 2) {SetAngle90To135(3,90+(((WalkCnt+2)*WalkRgtDrive)/128));
          SetAngle90To135(7,90+(((43-WalkCnt)*WalkLftDrive)/128));}
        else if (WalkCnt > 1) {SetAngle(4,LegUp); SetAngle(8,LegUpFwd);}
        else {SetAngle(4,LegDn); SetAngle(8,LegDn);
          SetAngle90To135(3,90+(((1-WalkCnt)*WalkRgtDrive)/128));
          SetAngle90To135(7,90+(((WalkCnt+44)*WalkLftDrive)/128));} break;
          
      case 4: // walking towards the left
        // front right leg & rear left leg
        if (WalkCnt < 1) {SetAngle(4, LegDn); SetAngle(8, LegDn);}
        else if (WalkCnt < 46) {SetAngle45To90(3,90-(((45-WalkCnt)*WalkRgtDrive)/128));
          SetAngle45To90(7,90-((WalkCnt*WalkLftDrive)/128));}
        else if (WalkCnt == 46) {SetAngle(4,LegUp); SetAngle(8,LegUpFwd);}
        else {SetAngle45To90(3,90-(((WalkCnt-42)*WalkRgtDrive)/128));
          SetAngle45To90(7,90-(((87-WalkCnt)*WalkLftDrive)/128));}
        // front left leg &  rear right leg
        if (WalkCnt > 43) {SetAngle(2,LegDn); SetAngle(6,LegDn);
          SetAngle45To90(1,90-(((WalkCnt-44)*WalkRgtDrive)/128));
          SetAngle45To90(5,90+(((WalkCnt-88)*WalkLftDrive)/128));}
        else if (WalkCnt == 22) {SetAngle(2,LegUpFwd); SetAngle(6,LegUp);} // Up start condition
        else if (WalkCnt > 2) {SetAngle45To90(1,90-(((43-WalkCnt)*WalkRgtDrive)/128));
          SetAngle45To90(5,90-(((2+WalkCnt)*WalkLftDrive)/128));}
        else if (WalkCnt > 1) {SetAngle(2,LegUpFwd); SetAngle(6,LegUp);}
        else {SetAngle(2,LegDn); SetAngle(6,LegDn);
          SetAngle45To90(1,90-(((WalkCnt+44)*WalkRgtDrive)/128));
          SetAngle45To90(5,90+(((WalkCnt-1)*WalkLftDrive)/128));} break;
          
      case 5: // neutral turning towards the left
        // front left leg & rear right leg
        Walked++; // increase how far walked counter
        if (WalkCnt < 1) {SetAngle(2, LegDn); SetAngle(6, LegDn);}
        else if (WalkCnt < 46) {SetAngle(1,90-WalkCnt); SetAngle(5,90-WalkCnt);}
        else if (WalkCnt == 46) {SetAngle(2,LegUp); SetAngle(6,LegUp);}
        else {SetAngle(1,WalkCnt+3); SetAngle(5,WalkCnt+3);}
        // front right leg &  rear left leg
        if (WalkCnt > 43) {SetAngle(4,LegDn); SetAngle(8,LegDn);
          SetAngle(3,WalkCnt+1); SetAngle(7,WalkCnt+1);}
        else if (WalkCnt == 22) {SetAngle(4,LegUp); SetAngle(8,LegUp);} // Up start condition
        else if (WalkCnt > 2) {SetAngle(3,88-WalkCnt); SetAngle(7,88-WalkCnt);}
        else if (WalkCnt > 1) {SetAngle(4,LegUp); SetAngle(8,LegUp);}
        else {SetAngle(4,LegDn); SetAngle(8,LegDn);
          SetAngle(3,WalkCnt+89); SetAngle(7,WalkCnt+89);} break;
          
      case 6: // neutral turning towards the right
        // front right leg & rear left leg
        Walked++; // increase how far walked counter
        if (WalkCnt < 1) {SetAngle(4, LegDn); SetAngle(8, LegDn);}
        else if (WalkCnt < 46) {SetAngle(3,90-WalkCnt); SetAngle(7,90-WalkCnt);}
        else if (WalkCnt == 46) {SetAngle(4,LegUp); SetAngle(8,LegUp);}
        else {SetAngle(3,WalkCnt+3); SetAngle(7,WalkCnt+3);}
        // front left leg &  rear right leg
        if (WalkCnt > 43) {SetAngle(2,LegDn); SetAngle(6,LegDn);
          SetAngle(1,WalkCnt+1); SetAngle(5,WalkCnt+1);}
        else if (WalkCnt == 22) {SetAngle(2,LegUp); SetAngle(6,LegUp);} // Up start condition
        else if (WalkCnt > 2) {SetAngle(1,88-WalkCnt); SetAngle(5,88-WalkCnt);}
        else if (WalkCnt > 1) {SetAngle(2,LegUp); SetAngle(6,LegUp);}
        else {SetAngle(2,LegDn); SetAngle(6,LegDn);
          SetAngle(1,WalkCnt+89); SetAngle(5,WalkCnt+89);} break;
    }
  } else if (TgtCnt > 0) {
    // if not walking see if we need to move the servos?
    MoveLegsToTgt();
  }
    // Serial.println(TgtCnt);
}

// ----------------------------------------------------------------------

void Walk_Stop() {
  // called when button pressed during walking, so we stop
  // stop the current walk sequence and ensure all legs are down
  Walk = 0; WalkCnt = 0;
  GoToAngles(-1,LegDn,-1,LegDn,-1,LegDn,-1,LegDn,-5);
}

// ----------------------------------------------------------------------
