
// ----------------------------------------------------------------------

void arrest(int zMT) {
  // called on the button count == 1 to ensure it is arrested from any movement
  MainTask = -1; SubTask = 0; ESC = 1; walkInterval = 20000;
  if (HeadActive) {HeadTask = 0; servoTgt[0] = Head_0; HeadTgtCnt= 10;} // centre head
  MainTaskNext = zMT;
}

// ---------------------------------------------------------------------------------

void attachHeadServo() {
  // head servo and set its position to servoVal[0]
  if (ESC || USB) return;
  
  servoAtt[0] = true;
  servoInst[0].setPeriodHertz(PWM_Freq);    // set PWM frequency
  servoInst[0].attach(HeadPin,0,2400);
  servoInst[0].writeMicroseconds(servoVal[0]); servoLast[0] = servoVal[0];
  servoTgt[0] = servoVal[0]; // correct target pointer
  Serial.println("Head servo attached.");
}

// ---------------------------------------------------------------------------------

void attachServos(int zDel) {
  // attach all 8 leg servos
  // delay used during power-up reset, otherwiswe set to 0
  if (ESC || USB) return;
  
  ServoEn = true; // flag servos are attached
  // attach the leg servos
  for (int zS = 1; zS <= 8; zS++) {
    servoInst[zS].setPeriodHertz(PWM_Freq);    // set PWM frequency
    servoInst[zS].attach(servoPins[zS],500,2400);
    servoInst[zS].writeMicroseconds(servoVal[zS]); servoLast[zS] = servoVal[zS];
    servoAtt[zS] = true;
    delay(zDel);
  }
  Serial.println("Leg servos attached.");
}

// --------------------------------------------------------------------------------

void battRead() {
  // read the analogue voltage connected to the battery every 20ms
  BatVol = analogRead(BatPin);
  // BatVol-= BatDist; BatDist++;  // uncomment this line to simulate rapid battery discharge
  BatSum = BatSum + BatVol - BatAvg;
  BatAvg = BatSum/50;     // ADC is averaged over 50 readings to remove noise
  // Serial.println("BatAvg " + String(BatAvg));
  if ((BatAvg >= BatWarn) && USB) {USB = false;}  // battery voltage returned by switch-on
  if (USB) {return;}  // battery voltage was low when powered up

  // detect critical battery condition and disable Quadruped
  // this is ignored when in TEST == true mode
  if ((BatAvg <= BatCritical) && !TEST){
    // critical battery voltage reached
    FastLED.clear(); FastLED.show();  // turn OFF LEDs
    VL53L0X_Disable();                    // turn OFF LTOF
    GoToRest(50);
    while (TgtCnt > 0) {
      // complete the goto rest from here
      MoveLegsToTgt(); delay(20);
    }
    // and that's it
    while (true) {yield();}
  }
}

// --------------------------------------------------------------------------------

void checkTgts() {
  // ensure that set leg targets are within servo min/max limits
  servoTgt[ 1] = min16(servoTgt[ 1],Max1); servoTgt[ 1] = max16(servoTgt[ 1],Min1);
  servoTgt[ 2] = min16(servoTgt[ 2],Max2); servoTgt[ 2] = max16(servoTgt[ 2],Min2);
  servoTgt[ 3] = min16(servoTgt[ 3],Max3); servoTgt[ 3] = max16(servoTgt[ 3],Min3);
  servoTgt[ 4] = min16(servoTgt[ 4],Max4); servoTgt[ 4] = max16(servoTgt[ 4],Min4);
  servoTgt[ 5] = min16(servoTgt[ 5],Max5); servoTgt[ 5] = max16(servoTgt[ 5],Min5);
  servoTgt[ 6] = min16(servoTgt[ 6],Max6); servoTgt[ 6] = max16(servoTgt[ 6],Min6);
  servoTgt[ 7] = min16(servoTgt[ 7],Max7); servoTgt[ 7] = max16(servoTgt[ 7],Min7);
  servoTgt[ 8] = min16(servoTgt[ 8],Max8); servoTgt[ 8] = max16(servoTgt[ 8],Min8);
}

// ---------------------------------------------------------------------------------

void decodeKey(int zkeyVal) {
  // decodes zkeyVal and excutes commands on '.'
  // every time we receive a character set response to 10ms later, just in case
  // there is another packet behind it coming from the controller
  next40ms = millis() - 10;
  if (zkeyVal == 10) {return;}
  if (zkeyVal == 13) {return;}
  keyChar = char(zkeyVal);
  switch (keyChar) {
    case '.': doCmd(); return;          // command terminator
    case '~': // connected to OLED Mirror app via USB or WiFi
      if (!DispMon) {DispMon = true; PrintTx = "\n\n"; DispNext = DispMode; Display_Mirrored();} // display mirrored message
      DispTx = 50; return;              // set the DispMOn time-out counter
  //  case '+': cmdVal = -1; return;      // set flag for ML+. command
    case '-': cmdSgn = -1; return;      // numeric sign
    case '!': esp_restart();             // RESET cmd received
    case '#':  // 'ping' sent from application, disables auto-rest function and Wi-Fi link
      Ping = 100; autoRest = autoRestMax;  LedMode = 0;
      WiFiPing = 0; if (WiFiConnected) {WiFiDisconnect();}
      return;
    case '*': cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1; return; // ESC abort command
    case '0': extendCmdVal(0); return;
    case '1': extendCmdVal(1); return;
    case '2': extendCmdVal(2); return;
    case '3': extendCmdVal(3); return;
    case '4': extendCmdVal(4); return;
    case '5': extendCmdVal(5); return;
    case '6': extendCmdVal(6); return;
    case '7': extendCmdVal(7); return;
    case '8': extendCmdVal(8); return;
    case '9': extendCmdVal(9); return;
  }
  if (cmdMode == ' ') {
    // test for new Command Mode char?
    // only accept certain characters as valid commands, or ignore them
    // delay battery update by 10 seconds whilst receiving commands
    cmdMode = keyChar;
    switch (keyChar) {
      // check for valid mode and convert lower-case
      case 'd': cmdMode = 'D'; break;
      case 's': cmdMode = 'S'; break;
      case 'v': cmdMode = 'V'; break;
      case 'w': cmdMode = 'W'; break;
    } cmdType = ' '; cmdVal = 0;
  } else {
    // test for Command Type char?
    cmdType = keyChar;
    switch (keyChar) {
      // check for lower-case and convert to upper-case
      case 'a': cmdType = 'A'; break;
      case 'c': cmdType = 'C'; break;
      case 'd': cmdType = 'D'; break;
      case 'h': cmdType = 'H'; break;
      case 'l': cmdType = 'L'; break;
      case 'm': cmdType = 'M'; break;
      case 'o': cmdType = 'O'; break;
      case 'r': cmdType = 'R'; break;
      case 's': cmdType = 'S'; break;
      case 't': cmdType = 'T'; break;
      case 'u': cmdType = 'U'; break;
      case 'v': cmdType = 'V'; break;
      case 'x': cmdType = 'X'; break;
      case 'y': cmdType = 'Y'; break;
    }
  }
}

// ---------------------------------------------------------------------------------

void detachHeadServo() {
  // Detach head servo GH-S37D
  // The micro servo uses GPIO2, which is also connected to the buitlin LED
  // Removing PWM from this servo does not appear to power it down, so we
  // just centralise it, looking forward
  if (!servoAtt[0]) {return;} // already detached
  
  // Serial.println("detachHeadServo()");
  servoInst[0].writeMicroseconds(Head_0);
  servoLast[0] = Head_0;
  servoAtt[0] = false;
}

// ---------------------------------------------------------------------------------

void detachServos() {
  // Detatch all of the leg servos, normally done when at rest
  if (!ServoEn) {return;}
  
  // Detach the leg servos in quick succession
  for (int zS = 1; zS <= 8; zS++) {
    // if (servoAtt[zS]) {servoInst[zS].detach();}
    if (servoAtt[zS]) {servoInst[zS].release();}
    servoAtt[zS] = false;
  }
  ServoEn = false; // flag all servos are detached
  Serial.println("detachServos()");
}

// ---------------------------------------------------------------------------------

void disableWiFi() {
  // when either a switch is pressed or WiFi disable is required
  WiFiEn = false; WiFiCntC = 0; WiFiRestCnt = 0; C_Dn = 8; Z_Dn = 8; HeadTask = 0;
}

// --------------------------------------------------------------------------------

void displayLft() {
  // change the DispMode pointer to the previous display
  if (DispLock) {return;} // ignore mouse click when display is locked

  DispMode--; if (DispMode < DM_Min) {DispMode = DM_Max;}
  DispDel = 0; DispCnt = 0;
}

// --------------------------------------------------------------------------------

void displayRht() {
  // change the DispMode pointer to the next display
  if (DispLock) {return;} // ignore mouse click when display is locked

  DispMode++; if (DispMode > DM_Max) {DispMode = DM_Min;}
  DispDel = 0; DispCnt = 0;
}

// ---------------------------------------------------------------------------------

void doCmd() {
  // a '.' has been received on the serial port so execute command if valid
  // Commands:
  // DMxxyy.  - display monitor left mouse click, xx = 0-99, yy = 0-99
  // SAnn.  - set target servo angle in degrees
  // SD.    - detach the current target servo pin
  // SLnn.  - set target servo lower limit in microseconds
  // SRn.   - set the state of the echo flag, 0 = OFF, 1 = ON
  // STnn.  - set the target servo number 0 - 16, but don't feedback value
  // SUnn.  - set target servo upper limit in microseconds
  // SVnn.  - set target servo angle in microseconds
  // V?.    - report VL53L1X status
  // VOn.   - turns VL53L1X n = 1 ON or n = 0 OFF

  cmdVal*= cmdSgn;  // correct cmdVal with the sign value
  
  switch (cmdMode) {
    case ' ': break;
    case 'D':
      switch (cmdType) {
        case 'D': // DDn. command, adjust global delay offset
          if (cmdVal > 0) {DispDelGbl ++;} else {DispDelGbl --;}
          if (DispDelGbl < 0) {DispDelGbl = 0;}
          // display this in the Rx field of Monitor+
          PrintTx+= "$DispDelGbl: " + String(DispDelGbl) +"\n";
          break;
        case 'L': // DLn. command, toggle display lock
          if (cmdVal != 0) {DispLock = true; DispOveride = DispMode;}
          else {DispLock = false;}
          break;
        case 'M':
          if (cmdSgn < 0) {}  // mouse click released
          else {
            if (cmdVal >= 10000) {
              // this is a Display Monitor mouse right-click
              // determine whether it is a left or right button click
              cmdVal /= 10000; cmdVal --; DmB = 1;
              DmY = cmdVal % 100; DmX = cmdVal/100;
            } else {
              // this is a Display Monitor mouse left-click
              DmY = cmdVal % 100; DmX = cmdVal/100; DmB = 0;
            }
            //      if (DispMode == DM_MPU_Acc) {DM_MPU_Acc_();}
            // else {
              // this is not a special 'clickable' display, so simply change it
              if (DmX < 50) {displayLft();  // left side click
              } else {displayRht(); }       // right side click
            // }
          } break;
      } break;
    case 'S':
      // keyboard/serial commands used to set servo values
      switch (cmdType) {
        case 'A': // set target servo angle in degrees
          if (ServoPnt == 0) {if (!servoAtt[0]) {attachHeadServo();}}
          else if (!servoAtt[ServoPnt]) {
            // if not attached, attach it
            servoAtt[ServoPnt] = true;
            servoInst[ServoPnt].setPeriodHertz(PWM_Freq);    // set PWM frequency
            servoInst[ServoPnt].attach(servoPins[ServoPnt],500,2400);
          }
          SetAngle(ServoPnt,cmdVal); break;
        case 'D': // detach the current target servo pin
          if (cmdVal < 0) {cmdVal = ServoPnt;}  // ignore out of range values
          if (cmdVal > 8) {cmdVal = ServoPnt;}  // ignore out of range values
          ServoPnt = cmdVal;
          if (ServoPnt == 0) {if (!servoAtt[0]) {detachHeadServo();}}
          else {servoAtt[ServoPnt] = false; servoInst[ServoPnt].detach();}
          break;
        case 'L': // set target servo lower limit in microseconds
          if (ServoPnt <= 8) {servoMin[ServoPnt] = cmdVal;}
          break;
        case 'R': // set the state of the Echo flag
          if (cmdVal > 0) {Echo = true; PrintTx += "Echo ON\n";} else {Echo = false;}
          break;
        case 'T': // set target servo pointer in range 0 - 9
          if (cmdVal < 0) {cmdVal = ServoPnt;}  // ignore out of range values
          if (cmdVal > 8) {cmdVal = ServoPnt;}  // ignore out of range values
          ServoPnt = cmdVal; break;
        case 'U': // set target servo upper limit in microseconds
          if (ServoPnt <= 8) {servoMax[ServoPnt] = cmdVal;}
          break;
        case 'V': // set target servo angle in microseconds
          if (ServoPnt == 0)  {
            // ServoPnt is pointing at the head servo
            servoVal[0] = cmdVal;
            if (!servoAtt[0]) {attachHeadServo();}
            servoInst[0].writeMicroseconds(servoVal[0]);
            servoTgt[0] = servoVal[0]; // correct target pointer
          }
          else if (ServoPnt <= 8) {
            // ServoPnt is pointing at a leg servo
            servoVal[ServoPnt] = cmdVal;
            if (!servoAtt[ServoPnt]) {
              // if not attached, attach it
              servoAtt[ServoPnt] = true;
              servoInst[ServoPnt].setPeriodHertz(PWM_Freq);    // set PWM frequency
              servoInst[ServoPnt].attach(servoPins[ServoPnt],500,2400);
            }
            servoInst[ServoPnt].writeMicroseconds(servoVal[ServoPnt]);
            servoTgt[ServoPnt] = servoVal[ServoPnt]; // correct target pointer
          }
          Serial.println("ST" + String(ServoPnt) + ".SV" + String(cmdVal) + ".");
          break;
      } break;
    case 'V':
      // VL53L1X commands
      switch (cmdType) {
        case '?': // report VL53L1X status
        //  Serial.println("VL53L1X Task " + String(VL53L1X_Task));
        //  Serial.println("VL53L1X OC   " + String(VL53L1X_OC));
        //  Serial.println("VL53L1X ROIX " + String(VL53L1X_ROIX));
        //  Serial.println("VL53L1X ROIY " + String(VL53L1X_ROIY));
          break;
        case 'O': // turn VL53L1X sensor ON/OFF
          if (cmdVal == 0) {VL53L0X_Enable();} else {VL53L0X_Disable();}
          Serial.println("VO" + String(cmdVal) + ".");
          break;
      }
      break;
    default:
      PrintTx += "Unknown cmd: " + cmdMode + cmdType + String(cmdVal) + "\n";
      break;
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1;
}

// ---------------------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = abs(cmdVal * 10) + zVal;
}

// ----------------------------------------------------------------------

void flashLEDs(int zCnt) {
  // flash the all LEDs blue swCnt times
  // this is a blocking function, hence we need to reset timers once done
  LedNop = true;
  while (zCnt > 0) {
    FastLED.clear();
    LED[0].setRGB(0,0,8); LED[1].setRGB(0,0,8); LED[2].setRGB(0,0,8); LED[3].setRGB(0,0,8); LED[4].setRGB(0,0,8);
    FastLED.show(); delay(40);
    FastLED.clear(); FastLED.show();
    delay(300);
    zCnt--;
  }
  synchLoopTimers();
  LedNop = false;
}

// ----------------------------------------------------------------------

void flashWhilstDown(int zDel) {
  // flash LEDs randomly whilst waiting for button switch to be released
  while (digitalRead(sw0Pin) == LOW) {
    GetRndCol(2); LED[random(5)].setRGB(Col_R,Col_G,Col_B);
    GetRndCol(4); LED[random(5)].setRGB(Col_R,Col_G,Col_B);
    GetRndCol(8); LED[random(5)].setRGB(Col_R,Col_G,Col_B);
    LEDshow = true;
    delay(zDel);
    if (digitalRead(sw0Pin) == HIGH) {FastLED.clear(); LEDshow = true; break;}
  }
}

// --------------------------------------------------------------------------------

String getBAhex() {
  // Rerturns the current Broardcast Address MAC, ie: 50:02:91:68:F7:3F
  String zRet$ = ""; uint8_t zV;
  for (int zP = 0;zP < 6;zP++) {
    zV = broadcastAddress[zP]/16;
    if (zV < 10) {zRet$ += char(zV + 48);} else {zRet$ += char(zV + 55);}
    zV = broadcastAddress[zP] - (zV *16);
    if (zV < 10) {zRet$ += char(zV + 48);} else {zRet$ += char(zV + 55);}
    if (zP < 5) {zRet$ += ":";}
  }
  return zRet$;
}

// --------------------------------------------------------------------------------

void GetRndCol(byte zB) {
  // return a random colour of strength zB
  if (zB < 2) {zB = 2;}
  switch(random(6)) {
    case 0: Col_R =   zB; Col_G =    0; Col_B =    0; break;  // red
    case 1: Col_R = zB/2; Col_G = zB/2; Col_B =    0; break;  // yellow
    case 2: Col_R =    0; Col_G =   zB; Col_B =    0; break;  // green
    case 3: Col_R =    0; Col_G = zB/2; Col_B = zB/2; break;  // cyan
    case 4: Col_R =    0; Col_G =    0; Col_B =   zB; break;  // blue
    case 5: Col_R = zB/2; Col_G =    0; Col_B = zB/2; break;  // magenta
  }
}

// --------------------------------------------------------------------------------

void Init_ESP_NOW() {
  // called to initiate an ESEP-NOW link or connect to another trannsceiver
  // each time this function is called it will attempt to connect to a different
  // boradcast address, cycling through them, over and over
  // Serial.println("Init_ESP_NOW()");

  // if we have previously tried a connection then we need to de-initialise it
  if (ESP_NOW_Init) {
    // already attempted to initialise ESP-NOW so strip everything back
    Serial.println("Closing ESP-NOW");
    esp_now_del_peer(broadcastAddress);
    // WiFi.disconnect();
  }
  
  // Init ESP-NOW and returns its status
  if (esp_now_init() != ESP_OK) {
    Serial.println("Error initializing ESP-NOW");
    return;
  }
  ESP_NOW_Init = true;
  
  // change the broadcast address pointer each time
  ESP_NOW_BA++;
  // Serial.println("ESP_NOW_BA = " + String(ESP_NOW_BA));
  
  switch (ESP_NOW_BA) {
    case 0: // Wii D1 Mk1 Transciever unique MAC: 50:02:91:68:F7:3F
      broadcastAddress[0] = 0x50;
      broadcastAddress[1] = 0x02;
      broadcastAddress[2] = 0x91;
      broadcastAddress[3] = 0x68;
      broadcastAddress[4] = 0xF7;
      broadcastAddress[5] = 0x3F;
      break;
    case 1: // Wii D1 Mk1 Transciever unique MAC: 50:02:91:68:F0:D2
      broadcastAddress[0] = 0x50;
      broadcastAddress[1] = 0x02;
      broadcastAddress[2] = 0x91;
      broadcastAddress[3] = 0x68;
      broadcastAddress[4] = 0xF0;
      broadcastAddress[5] = 0xD2;
      break;
    case 2: // Wii D1 Mk1 Transciever unique MAC: 58:BF:25:DB:27:A2
      broadcastAddress[0] = 0x58;
      broadcastAddress[1] = 0xBF;
      broadcastAddress[2] = 0x25;
      broadcastAddress[3] = 0xDB;
      broadcastAddress[4] = 0x27;
      broadcastAddress[5] = 0xA2;
      break;
    case 3: // Wii ESP32-C3 Zero Transciever unique MAC: EC:DA:3B:BD:4B:AC
      broadcastAddress[0] = 0xEC;
      broadcastAddress[1] = 0xDA;
      broadcastAddress[2] = 0x3B;
      broadcastAddress[3] = 0xBD;
      broadcastAddress[4] = 0x4B;
      broadcastAddress[5] = 0xAC;
      break;
    default:
      // End of list reached, so extend pause time and reset cycle
      // Turn off serial port reporting after first pass, unless connection fails
      ESP_NOW_BA = -1;      // 1st case will be 0
      WiFiConCnt = 75;      // 3 sec delay between trying cycles
      WiFiTryOnce = false;  // don't report trying after one cycle
      break;
  }

  // Once ESPNow is successfully Init, we will register for Send CB to
  // get the status of Trasnmitted packet
  esp_now_register_send_cb(OnDataSent);

  // Register peer
  esp_now_peer_info_t peerInfo; //initialize and assign the peer information as a pointer to an addres
  memset(&peerInfo, 0, sizeof(peerInfo));
  memcpy(peerInfo.peer_addr, broadcastAddress, 6); //copy the value of  broadcastAddress with 6 bytes to peerInfo.peer_addr
  peerInfo.channel = 0;     //channel at which the esp talk. 0 means undefined and data will be sent on the current channel. 1-14 are valid channels which is the same with the local device 
  peerInfo.encrypt = false; //not encrypted

  //Add the device to the paired device list 
  if (esp_now_add_peer(&peerInfo) != ESP_OK){ 
    Serial.println("Failed to add peer");
    return;
  }

  // Register for a callback function that will be called when data is received
  esp_now_register_recv_cb(OnDataRecv);
}

// --------------------------------------------------------------------------------

void LED_Main() {
  // tasks for manipulating RGB LED patterns
  // Blink clock and modes
  if (LedNop) {return;}
  
  Blink = false; BlinkCnt++; if (BlinkCnt > 30) {BlinkCnt = 0; Blink = true;}
  // switch over-rides
  if (sw0State == LOW) {
    FastLED.clear();
    LED[3].setRGB(0,0,8); LED[4] = LED[3];
    LEDshow = true; LED_Del = 50; return;
  }
  if (LED_Del > 0) {LED_Del--; return;} // delay by skipping tasks

  // Serial.println(LedMode);
  switch(LedMode) {
    case 0: // doing nothing, display battery status
      LED_Store(); break;
    case 1: // just standing 60/120
      LED_Stand(); break;
    case 2: // standing ready 90/90
      LED_Ready(); break;
    case 3: // display WalkSpeed setting
      LED_WalkSpeed(); break;
    case 4: // disabling servos
      LED_Rest(); break;
    case 5: // LTOF range
      LED_Range(); break;
    case 6: // TEST mode default
      LED_Test(); break;
    case 10: // random LED colours
      GetRndCol(2); LED[random(5)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(4); LED[random(5)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(8); LED[random(5)].setRGB(Col_R,Col_G,Col_B);
      LEDshow = true; LED_Del = 2;
      break;
    case 21: // walking forward
      LED_WalkFwd(); break;
    case 22: // walking turning right
      LED_NeutRht(); break;
    case 23: // walking backward
      LED_WalkBck(); break;
    case 24: // walking turning left
      LED_NeutLft(); break;
    case 25: // walking neutral turn left
      LED_NeutLft(); break;
    case 26: // walking neutral turn right
      LED_NeutRht(); break;
  }
}

// --------------------------------------------------------------------------------

void LED_NeutLft() {
  // generates a neutral turn left pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      TMP[0].setRGB(16,16,0); TMP[1].setRGB(8,4,0); TMP[2].setRGB(3,0,0); TMP[3].setRGB(1,0,0);
      LedCnt = 0; LED_Task++; break;
    case 1: // rotate LED pattern
      FastLED.clear();
      switch (LedCnt) {
        case 0:
          LED[4] = TMP[0];break;
        case 1:
          LED[3] = TMP[0]; LED[4] = TMP[1]; break;
        case 2:
          LED[2] = TMP[0]; LED[3] = TMP[1];
          LED[4] = TMP[2]; break;
        case 3:
          LED[1] = TMP[0]; LED[2] = TMP[1];
          LED[3] = TMP[2]; LED[4] = TMP[3]; break;
        case 4:
          LED[0] = TMP[0]; LED[1] = TMP[1];
          LED[2] = TMP[2]; LED[3] = TMP[3]; break;
        case 5:
          LED[0] = TMP[1]; LED[1] = TMP[2];
          LED[2] = TMP[3]; LED[4] = TMP[0]; break;
        case 6:
          LED[0] = TMP[2]; LED[1] = TMP[3];
          LED[3] = TMP[0]; LED[4] = TMP[1]; break;
        case 7:
          LED[0] = TMP[3]; LED[2] = TMP[0];
          LED[3] = TMP[1]; LED[4] = TMP[2]; break;
        case 8:
          LED[1] = TMP[0]; LED[2] = TMP[1];
          LED[3] = TMP[2]; LED[4] = TMP[3]; break;
      }
      LedCnt++; if (LedCnt > 8) {LedCnt = 4;}
      LEDshow = true; LED_Del = 6; break;
  }
}

// --------------------------------------------------------------------------------

void LED_NeutRht() {
  // generates a neutral turn right pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      TMP[0].setRGB(16,16,0); TMP[1].setRGB(8,4,0); TMP[2].setRGB(3,0,0); TMP[3].setRGB(1,0,0);
      LedCnt = 0; LED_Task++; break;
    case 1: // rotate LED pattern
      FastLED.clear();
      switch (LedCnt) {
        case 0:
          LED[0] = TMP[0];break;
        case 1:
          LED[0] = TMP[1]; LED[1] = TMP[0]; break;
        case 2:
          LED[0] = TMP[2]; LED[1] = TMP[1];
          LED[2] = TMP[0]; break;
        case 3:
          LED[0] = TMP[3]; LED[1] = TMP[2];
          LED[2] = TMP[1]; LED[3] = TMP[0]; break;
        case 4:
          LED[1] = TMP[3]; LED[2] = TMP[2];
          LED[3] = TMP[1]; LED[4] = TMP[0]; break;
        case 5:
          LED[0] = TMP[0]; LED[2] = TMP[3];
          LED[3] = TMP[2]; LED[4] = TMP[1];break;
        case 6:
          LED[0] = TMP[1]; LED[1] = TMP[0];
          LED[3] = TMP[3]; LED[4] = TMP[2];break;
        case 7:
          LED[0] = TMP[2]; LED[1] = TMP[1];
          LED[2] = TMP[0]; LED[4] = TMP[3];break;
        case 8:
          LED[0] = TMP[3]; LED[1] = TMP[2];
          LED[2] = TMP[1]; LED[3] = TMP[0];break;
      }
      LedCnt++; if (LedCnt > 8) {LedCnt = 4;}
      LEDshow = true; LED_Del = 6; break;
  }
}

// --------------------------------------------------------------------------------

void LED_Range() {
  // drive LEDs in response to LTOF ranging   
  FastLED.clear();
  if (Range < RangeMax) {
    anyVal = map(Range,LtofMin,LtofLimit,110,0);
         if (anyVal >= 100) {LED[4].setRGB(5,0,0); LED[3].setRGB(5,0,0); LED[2].setRGB(5,0,0); LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  90) {LED[4].setRGB(1,1,0); LED[3].setRGB(5,0,0); LED[2].setRGB(5,0,0); LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  80) {LED[3].setRGB(5,0,0); LED[2].setRGB(5,0,0); LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  70) {LED[3].setRGB(1,1,0); LED[2].setRGB(5,0,0); LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  60) {LED[2].setRGB(5,0,0); LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  50) {LED[2].setRGB(1,1,0); LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  40) {LED[1].setRGB(5,0,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  30) {LED[1].setRGB(1,1,0); LED[0].setRGB(5,0,0);}
    else if (anyVal >=  20) {LED[0].setRGB(5,0,0);}
    else if (anyVal >=  10) {LED[0].setRGB(1,1,0);}
  } else {
    if (Blink) {LED[2].setRGB(2,0,0);}
  }
  LEDshow = true; LED_Del = 0; Tick20ms = true;
}

// --------------------------------------------------------------------------------

void LED_Ready() {
  // Red walking dots right-left-right-...
  FastLED.clear();
  switch(LedCnt) {
    case 0: LED[0].setRGB(3,0,0); break;
    case 1: LED[1].setRGB(3,0,0); break;
    case 2: LED[2].setRGB(3,0,0); break;
    case 3: LED[3].setRGB(3,0,0); break;
    case 4: LED[4].setRGB(3,0,0); break;
    case 5: LED[3].setRGB(3,0,0); break;
    case 6: LED[2].setRGB(3,0,0); break;
    case 7: LED[1].setRGB(3,0,0); LedCnt = -1; break;
  } LedCnt++;
  LEDshow = true; LED_Del = 6;
}

// --------------------------------------------------------------------------------

void LED_Rest() {
  // indicate servos have been disabled
  FastLED.clear();
  switch (LED_Task) {
    case 0: LED[0].setRGB(64,64, 0); LED[4].setRGB(64,64, 0); break;
    case 1: LED[1].setRGB(64, 0, 0); LED[3].setRGB(64, 0, 0); break;
    case 2: LED[2].setRGB(64, 0, 0); break;
    case 3: LED[2].setRGB(32, 0, 0); break;
    case 4: LED[2].setRGB(16, 0, 0); break;
    case 5: LED[2].setRGB( 8, 0, 0); break;
    case 6: LED[2].setRGB( 2, 0, 2); break;
    case 7: LED[2].setRGB( 0, 0, 1); break;
    case 8: SetLedMode(LedLast); break;
  }LED_Task++;
  LEDshow = true; LED_Del = 6;
}

// --------------------------------------------------------------------------------

void LED_Set_All(byte zR,byte zG,byte zB) {
  // sets all of the LEDs to one colour immediately
  for (byte zL = 0; zL <= 4;zL++) {LED[zL].setRGB(zR,zG,zB);}
  FastLED.show();
}

// --------------------------------------------------------------------------------

void LED_Stand() {
  // Purple walking dots when servos are powered, otherwise green
  if (!ServoEn) {LED_Store(); return;}
  
  FastLED.clear();
  switch(LedCnt) {
    case 0: LED[0].setRGB(1,0,1); break;
    case 1: LED[1].setRGB(1,0,1); break;
    case 2: LED[2].setRGB(1,0,1); break;
    case 3: LED[3].setRGB(1,0,1); break;
    case 4: LED[4].setRGB(1,0,1); break;
    case 5: LED[3].setRGB(1,0,1); break;
    case 6: LED[2].setRGB(1,0,1); break;
    case 7: LED[1].setRGB(1,0,1); break;
    default: LedCnt = -1; break;  // zero the count and restart cycle
  } LedCnt++;
  LEDshow = true; LED_Del = 6;
}

// --------------------------------------------------------------------------------

void LED_Store() {
  // default, slow green/yellow/red walking dot as battery status
  // BatAvg will normally be between BatMax and BatWarn
  int16_t zMax = (5 * (BatAvg - BatCritical))/(BatMax - BatCritical);
  if (zMax > 4) {zMax = 4;}
  FastLED.clear();
  switch(zMax) {
    case 4:  TMP[0].setRGB(0,6,0); break;
    case 3:  TMP[0].setRGB(2,4,0); break;
    case 2:  TMP[0].setRGB(2,2,0); break;
    case 1:  TMP[0].setRGB(2,1,0); break;
    case 0:  TMP[0].setRGB(2,0,0); break;
  }
   LED_Del = 10; zMax++;
  switch(LedCnt) {
    case 0: LED_Del = 20; break; // put in a pause
    case 1: LED[0] = TMP[0]; break;
    case 2: LED[1] = TMP[0]; break;
    case 3: LED[2] = TMP[0]; break;
    case 4: LED[3] = TMP[0]; break;
    case 5: LED[4] = TMP[0]; break;
  } LedCnt++;
  if (LedCnt > zMax) {LedCnt = 0;} // limit the sweep based on zMax
  LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Test() {
  // scrolling RGB sequence in TEST mode
  // scroll LEDs up
  LED[4] = LED[3]; LED[3] = LED[2]; LED[2] = LED[1]; LED[1] = LED[0];
  
  switch(LED_Task) {
    case 0: LED[0].setRGB(2,0,0); break;
    case 3: LED[0].setRGB(0,2,0); break;
    case 6: LED[0].setRGB(0,0,2); break;
  }
  LED_Task++; if (LED_Task > 8) {LED_Task = 0;}
  LEDshow = true; LED_Del = 6;
}

// --------------------------------------------------------------------------------

void LED_WalkBck() {
  // generates a walking backward pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      TMP[0].setRGB(16,16,0); TMP[1].setRGB(8,4,0); TMP[2].setRGB(3,0,0); TMP[3].setRGB(1,0,0);
      LedCnt = 0; LED_Task++; break;
    case 1: // rotate LED pattern
      FastLED.clear();
      switch (LedCnt) {
        case 0:
          LED[0] = TMP[0]; LED[4] = TMP[0]; break;
        case 1:
          LED[1] = TMP[0]; LED[3] = TMP[0];
          LED[0] = TMP[1]; LED[4] = TMP[1]; break;
        case 2:
          LED[2] = TMP[0];
          LED[1] = TMP[1]; LED[3] = TMP[1];
          LED[0] = TMP[2]; LED[4] = TMP[2]; break;
        case 3:
          LED[2] = TMP[1];
          LED[1] = TMP[2]; LED[3] = TMP[2];
          LED[0] = TMP[3]; LED[4] = TMP[3]; break;
        case 4:
          LED[2] = TMP[2];
          LED[1] = TMP[3]; LED[3] = TMP[3]; break;
        case 5:
          LED[2] = TMP[3];
          LED[0] = TMP[0]; LED[4] = TMP[0]; break;
      }
      LedCnt++; if (LedCnt > 5) {LedCnt = 1;}
      LEDshow = true; LED_Del = 6; break;
  }
}

// --------------------------------------------------------------------------------

void LED_WalkFwd() {
  // generates a walking forward pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      TMP[0].setRGB(16,16,0); TMP[1].setRGB(8,4,0); TMP[2].setRGB(3,0,0); TMP[3].setRGB(1,0,0);
      LedCnt = 0; LED_Task++; break;
    case 1: // rotate LED pattern
      FastLED.clear();
      switch (LedCnt) {
        case 0:
          LED[2] = TMP[0]; break;
        case 1:
          LED[2] = TMP[1];
          LED[1] = TMP[0]; LED[3] = TMP[0]; break;
        case 2:
          LED[2] = TMP[2];
          LED[1] = TMP[1]; LED[3] = TMP[1];
          LED[0] = TMP[0]; LED[4] = TMP[0]; break;
        case 3:
          LED[2] = TMP[3];
          LED[1] = TMP[2]; LED[3] = TMP[2];
          LED[0] = TMP[1]; LED[4] = TMP[1]; break;
        case 4:
          LED[1] = TMP[3]; LED[3] = TMP[3];
          LED[0] = TMP[2]; LED[4] = TMP[2]; break;
        case 5:
          LED[2] = TMP[0];
          LED[0] = TMP[3]; LED[4] = TMP[3]; break;
      }
      LedCnt++; if (LedCnt > 5) {LedCnt = 1;}
      LEDshow = true; LED_Del = 6; break;
  }
}

// --------------------------------------------------------------------------------

void LED_WalkSpeed() {
  // display the current/changing WalkSpeed value as a blue bar
  // after 2 seconds the bar times out and reverts back to the previous mode
  LedCnt++; if (LedCnt > 100) {SetLedMode(LedNext); return;}
  
  if (LedMem != WalkSpeed) {
    FastLED.clear();
    LedCnt = 0;     // zero the count when a speed change occurs
    LED[0].setRGB(0,0,1);
    if (WalkSpeed >= 2) {LED[1].setRGB(0,0,2);}
    if (WalkSpeed >= 3) {LED[2].setRGB(0,0,4);}
    if (WalkSpeed >= 4) {LED[3].setRGB(0,0,8);}
    if (WalkSpeed >= 5) {LED[4].setRGB(0,0,16);}
    LEDshow = true; // only update if changed
  } LedMem = WalkSpeed;
}

// --------------------------------------------------------------------------------

int16_t max16(int16_t zA,int16_t zB) {
  // returns the larger of zA or zB
  if (zA >= zB) {return zA;} else {return zB;}  
}

// --------------------------------------------------------------------------------

void MemRead() {
  // reads the stored leg servo values into target values
  for (anyFor = 1;anyFor <= 8;anyFor++) {servoTgt[anyFor] = servoMem[anyFor];}
}

// --------------------------------------------------------------------------------

void MemStore() {
  // stores the current leg servo values in a memory store
  for (anyFor = 1;anyFor <= 8;anyFor++) {servoMem[anyFor] = servoVal[anyFor];}
}

// --------------------------------------------------------------------------------

int16_t min16(int16_t zA,int16_t zB) {
  // returns the smaller of zA or zB
  if (zA <= zB) {return zA;} else {return zB;}  
}

// --------------------------------------------------------------------------------

void readSerial() {
  // reads characters from the serial port and responds to commands
  // set PrintTgt to indicate source as serial port for responses
  keyVal = Serial.read();
  if (keyVal != -1) {SerialRx = true; PrintTgt = 0; decodeKey(keyVal);}
}

// ---------------------------------------------------------------------------------

void readSW0() {
  // called from the main loop every 20ms
  // read the left button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task
  sw0State = digitalRead(sw0Pin); // record button state
  if (sw0_Nop) {return;}          // block this function whilst == true
  
  if (sw0State == LOW) {
    //##############################################################################
    //
    // SW0 button is pressed down
    //
    //##############################################################################
    if (sw0LastState == HIGH) {
      // SW0 has just been pressed down
  //    Serial.println("SW0 Hi-LO");
      if ((sw0Cnt == 0) && QuadActive) {
        // ensure that if it was moving stop it!
        // are we walking?
        if (Walk > 0) {Walk_Stop();}  // if yes then stop it
        // centre head
        if (HeadTask != 0) {HeadTask = 0; HeadToRest();}
        // disconnect WiFi?
        if (WiFiEn) {disableWiFi();}
        // go into ESCape mode
        SetMainTask(99);
      }
      sw0Cnt++;       // count on the falling edge
      sw0DwnTime = 0; // restart button down time counter
      sw0Timer = 0;   // reset the timer
      Display_Text1x16("Sw = " + String(sw0Cnt),25);
    } else {
      // whilst the button is down adjust the timer at 10ms steps
      // if Wii is not connected test for a long press to invoke demo mode
      sw0DwnTime++; // track button pressed time
      if (TEST) {
        // look for long press >1 sec in TEST mode
        if (sw0DwnTime == 100) {
          if (Testmode == 1) {TestTask = 2;}
          else if (Testmode == 2) {TestTask = 10;} else {TestTask = 2;}
          sw0Cnt = 0; sw0Timer = 0;
        }
      } else {
        // not in TEST mode
        if (sw0Timer >= 60) {
          // button held down for >= 1.2 seconds, a long press
          sw0Cnt = 0; sw0Timer = 0; SetLedMode(0);
          if (!QuadActive) {
            // robot was inactive so stand it up and enable head scanning
            HeadActive = true; flashWhilstDown(100);
            SetMainTask(1); // go to default LTOF state
            Display_Text2x16("Waking Up!", "Head Active",50);
      //      Serial.println(F("Standing Up!"));
            QuadActive = true; synchLoopTimers();
          } else {
            // robot was active so return it to the reset state
            flashWhilstDown(100);
            synchLoopTimers();
            SetMainTask(0); // terminate current tasks
            Display_Text2x16("Going To", "Sleep...",50);
            HeadToRest();   // return head to the forward position
            GoToRest(50);   // go to the resting position within 1 second
            DispMode = 0;   // display monitor battery mode
      //      Serial.println(F("Going To Rest..."));
            loopWhileTgtCnt();
            QuadActive = false; VL53L0X_Disable(); HeadActive = false;
          }
        }
      }
    }
  } else {
    //##############################################################################
    //
    // SW0 button is released
    //
    //##############################################################################
    if (sw0LastState == LOW) {
      // SW0 has just been released
      FastLED.clear();  FastLED.show();
  //    Serial.println("SW0 Lo-Hi");
    }
    if (sw0Timer >= 50) {
      // button released for 1 sec so assume valid button counts
    //    Serial.print(F("swCnt = ")); Serial.println(swCnt);
      if (TEST && (sw0Cnt > 0)) {
        // single button press during test mode
        if (Testmode == 1) {
          if (ServoInc == 0) {ServoInc = 1;} else {ServoInc = 0;}
        }
      } else {
        // button released for 1 sec so assume valid button count
      //  Serial.print(F("swCnt = ")); Serial.println(swCnt);
        flashLEDs(sw0Cnt); // indicate count by flashing
        if (sw0Cnt == 1) {
          // goto default state
          SetMainTask(1);
          if (!QuadActive) {Display_Text2x16("Waking Up!", "Head Static",50);}
          QuadActive = true;
        } else if (QuadActive) {
          // only respond to these counts if stood up and active
          // swCnt == 2 backaway from target mode
          // swCnt == 3 track wall target mode
          // swCnt == 4 autonomous discovery mode
          if (sw0Cnt > 4) {sw0Cnt = 1;} // ignore excessive button presses
          SetMainTask(sw0Cnt);
        } else {Display_Text2x16("Not Active", "Ignored",50);}
        sw0Cnt = 0; sw0Timer = 0; synchLoopTimers();
      }
    }
  }
  sw0LastState = sw0State; // record current state
  if (sw0Cnt > 0) {sw0Timer++;}
}

// ---------------------------------------------------------------------------------

void readWiFiRx() {
  // Called from the main loop every cycle to check for ESP-NOW data  
  // a state machine is used to separate the data blocks by header type
  switch(Rx_Task) {
    case 0: // determine the type of data to handle
      // B  - binary data block
      // C  - Wii Classic controller
      // N  - Wii Nunchuk controller
      // P  - Wll Classic Pro controller
      // -  - unknown type, transceiver did not recognise
           if (Rx_Buff.ESPdata[Rx_Pnt] == 'B') {Rx_Task = 3; break;}                // binary data block
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'C') {WiiType = 'C'; Rx_Task = 1; break;} // Classic controller
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'N') {WiiType = 'N'; Rx_Task = 1; break;} // Nunchuk controller
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'P') {WiiType = 'P'; Rx_Task = 1; break;} // Classic Pro controller
      else if (Rx_Buff.ESPdata[Rx_Pnt] == '-') {WiiType = '-'; Rx_Task = 1; break;} // unknown type
      else if (Rx_Buff.ESPdata[Rx_Pnt] == '$') {Rx_Task = 2; break;}
      // oh dear! bloick header not recognised so discard the block and flag error
      else {WiFiRxErr++; WiFiRx = false;}
      break;
      
    case 1: // load Wii I2C data
      // After the header byte there are 6 data bytes + checksum
      // perform XOR on 6 bytes to check for errors
      if (Rx_Buff.ESPdata[7] == (Rx_Buff.ESPdata[1] ^ Rx_Buff.ESPdata[2] ^ Rx_Buff.ESPdata[3] ^ Rx_Buff.ESPdata[4] ^ Rx_Buff.ESPdata[5] ^ Rx_Buff.ESPdata[6])) {
        // data has passed the checksum test
        RxWiFi[0] = Rx_Buff.ESPdata[1];
        RxWiFi[1] = Rx_Buff.ESPdata[2];
        RxWiFi[2] = Rx_Buff.ESPdata[3];
        RxWiFi[3] = Rx_Buff.ESPdata[4];
        RxWiFi[4] = Rx_Buff.ESPdata[5];
        RxWiFi[5] = Rx_Buff.ESPdata[6];

        // if Wii data strip off the JX,JY and CZ values
        RxRec = true; // indicate a valid frame
        if (WiiType == 'N') {
          JoyX = RxWiFi[0];   // joystick X 0-255
          JoyY = RxWiFi[1];   // joystick Y 0-255
          CZ = RxWiFi[5] & 3; // CZ buttons, C = 2/0, Z = 1/0, active low
        } else if (WiiType == 'C') { // Wii Classic
          JoyX = WiiRightStickX() * 8;
          JoyY = WiiRightStickY() * 8;
          CZ = ((RxWiFi[4] & 0b100000)>>5) + (RxWiFi[4] & 0b10); // read LT and RT buttons as C,Z
        } else if (WiiType == 'P') { // Wii Classic Pro
          JoyX = WiiRightStickX() * 8;
          JoyY = WiiRightStickY() * 8;
          CZ = (RxWiFi[4] & 0b10) + ((RxWiFi[5] & 0b100)>2); // read RT and ZR buttons as C,Z
        } else {
          // not Nunchuk so set defaults
          JoyX = 128; JoyY = 128; CZ = 3;
        }
      } else {
        WiFiRxErr++;
      }
      Rx_Task = 0;  // reset the task pointer to lok at another block if needed
      Rx_Pnt += 8;  // increase the data pointer beyond this block
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; Rx_len = 0;} // reached end of buffer
      break;

    case 2: // load text data as if from a serial port
      // read characters up to the end of the buffer
      Rx_Pnt++;
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; Rx_len = 0; break;} // reached end of buffer
      keyVal = Rx_Buff.ESPdata[Rx_Pnt]; PrintTgt = 1; decodeKey(keyVal);
    //  Serial.write(keyVal);
      break;

    case 3: // load binary data block
      // data length is specified in second byte
      Serial.println("Bin");
      Rx_Pnt++;
      // at the moment just jump over it, binary transfer is not needed
      Rx_Pnt += Rx_Buff.ESPdata[Rx_Pnt] + 1;
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; break;} // reached end of buffer
      break;
  }
}

// ---------------------------------------------------------------------------------

void readWii() {
  // called every 20ms, but wireless transceiver only transmits every 40ms
  // considers data received from the Wii Nunchuk controller over WiFi and 
  // manages all movements demanded by the controller
  if (readWiiCall) {return;}
  
  readWiiCall = true; // prevent re-entry of this function
  // Serial.println("readWii()");
  
  if (RxRec) {
    //##############################################################################
    //
    //  WiFi Nunchuk data received
    //
    //##############################################################################
    // Wii data received but we may not have enabled WiFi mode yet
    RxRec = false; RTxTimeout = 25; // set timeout to approx 500ms
  //  PrintTx += "JX" + String(JoyX) + " JY" + String(JoyY) + " CZ" + String(CZ) + "\n";
  //  Serial.print("CZ "); Serial.println(CZ);
    if (!WiFiEn) {
      //##############################################################################
      //
      //  WiFi is currently disabled, listening
      //
      //##############################################################################
      // we have not received enough 'C' buttons to make the system active
      // the 'C' button alone must be held in to activate the Wii WiFi link
      if (CZ == 1) {
        // C button only is being pressed
        WiFiCntC += 2;
      //  Serial.println("WiFiCnt = " + String(WiFiCntC));
        if (WiFiCntC > 60) {
          // going active, user has held in 'C' button for sufficient time
        //  Serial.println("WiFiEn = true");
          SetMainTask(0); // go to default state
          QuadActive = true; WiFiEn = true; C_Dn = 8; WalkSpeed = 1; HeadTask = 1;
          SetLedMode(1);  // reset LED tasks
          DispMode = 2;   // display monitor Wi-Fi mode
          MoveType = "---";
          // set defaults here
          GoToStand(50);  // go to standing position
          synchLoopTimers(); QuadActive = true;
        } else {
          // still counting C buttons
          LEDVal = WiFiCntC/10; SetLedMode(10);
        }
      } else {
        // C button not pressed or released early so check for a inadequate count
        if (WiFiCntC > 0) {
          // reduce the C-button counter, which has to be sustained to enable WiFi mode
          WiFiCntC--; LEDVal = 1 + (WiFiCntC/10);
          if (WiFiCntC < 1) {SetLedMode(0);} else {SetLedMode(10);}
        }
      }
    } else {
      
      //##############################################################################
      //
      //  WiFi is enabled
      //
      //##############################################################################
      // WiFi is enabled, so check for power down request C + Z together
      if (CZ == 0) {
        // both C and Z buttons are being pressed so move towards powering down
        if (WiFiCntC > 0) {
          WiFiCntC -= 2;
        //  Serial.println("WiFiCnt = " + String(WiFiCntC));
          if (WiFiCntC <= 0) {
            // power-down condition satisfied, so go to rest
          //  Serial.println("WiFi mode disabled!");
            WiFiCntC = 0; disableWiFi(); WalkSpeed = 1;
            HeadToRest();   // return head to the forward position
            GoToRest(50); // go to the resting position within 1 second
            // set defaults here
            QuadActive = false;  WiFiEn = false; HeadTask = 0; SetLedMode(0);
            DispMode = 0;   // display monitor battery mode

          } else {
            // still counting down CZ buttons
            SetLedMode(10);
          }
        }
      } else {
        // check for C + Z being released part way through power-down
        if (LedMode == 10) {SetLedMode(1); WiFiCntC = 62;}

        
        // Check CZ buttons to see if a speed change is wanted
        if ((CZ & 2) == 0) {
          // C button is pressed, assume a speed change is needed
          C_Dn++;
          if (C_Dn == 2) {
            C_Dn = 8;
            if (WalkSpeed < 5) {WalkSpeed++; SetWalkMax();}
          }
        //  Serial.println("WalkSpeed =" + String(WalkSpeed));
        } else {C_Dn = 0;} // C button released
        // Check Z button. It has two functions. Short press change speed, but if
        // held in it changes the mode of the joysticks
        if ((CZ & 1) == 0) {
          // Z button is pressed
          Z_Dn++; if (Z_Dn > 12) {Z_Dn = 50;} // Z_Dn capped at 50, held for >= 2 seconds
        //  Serial.print("Z_Dn "); Serial.println(Z_Dn);
        } else {
          // Z button is not pressed
          if ((Z_Dn > 1) && (Z_Dn <= 10)) {
            // held down for less than 400 ms @ 40ms Rx cycle
            if (WalkSpeed > 1) {WalkSpeed--; SetWalkMax();}
          } Z_Dn = 0; // Z button released so clear down counter
        }
      //  Serial.print("Z_Dn "); Serial.println(Z_Dn);

        if (WiFiCntC < 60) {
          // C-button counter is not at full strength, assume WiFi off released prematurely
          WiFiCntC++; LEDVal = 1 + (WiFiCntC/10);
          if (WiFiCntC == 60) {LED_Task = -1;} else {LED_Task = -10;}
        }
      }
    }
      
    //##############################################################################
    //
    //  Wii joystick demands
    //
    //##############################################################################
    // joystick movements are ignored if:
    //  - Quad is not active
    //  - WiFiEn = false;
    //  - Wii Joystick has not sent data.
    //  - a MainTask is running
    // Serial.println(String(JoyMode) + "\t" + String(JoyX) + "\t" + String(JoyY));
    if ((QuadActive) &&  WiFiEn && (JoyActive) && (MainTask == 0)) {
      // read the X,Y joysticks and walk if demanded
      JoyActive = false; // prevent stack overflow during moves that call loop()
      // respond to joystick demands every 40ms (25Hz)
      // note direction of travel will only change once stick has been centred
      if ((JoyX >= DbLLX) && (JoyX <= DbULX) && (JoyY >= DbLLY) && (JoyY <= DbULY)) {
        // joystick is in centre deadband position
        if (LedRstCnt > 0) {
          LedRstCnt--; if (LedRstCnt == 0) {SetLedMode(2);}
        }
        JoyMode = 0;      // reset the direction of travel mode
        MoveType = "---"; // default display monitor action
        //  Serial.print(F("#"));
        if (Walk != 0) {
        //  Serial.println(F("Stopping"));
          walkInterval = 20000; SetLegsDown(); SetLedMode(2);
          // stop walking if active, but remember direction
          WalkLast = Walk; Walk = 0;
        }
      } else if (JoyMode == 0) {
        autoRest = 0;   // reset user time-out
        LedRstCnt = 2;  // set LED reset count
        if (Z_Dn < 1) {
          // 'Z' button is not pressed so normal walking movements
               if (JoyY > 150) {JoyMode =  1;} // set direction of travel as forward
          else if (JoyY < 106) {JoyMode = -1;} // set direction of travel as backward
          else if (JoyX < 105) {JoyMode =  3;} // set direction of travel as turn left
          else if (JoyX > 149) {JoyMode =  2;} // set direction of travel as turn right
        } else {
          // 'Z' button pressed so perform special actions
               if (JoyX < 105) {JoyMode = 5;} // set direction of travel as walk to left
          else if (JoyX > 149) {JoyMode = 4;} // set direction of travel as walk to right
          else if (JoyY > 149) {JoyMode = 6;} // perform a bow
          else if (JoyY < 106) {JoyMode = 7;} // perform a hello wave
        }
      } else {
        autoRest = 0;     // reset user time-out
        // Some Wii controllers put out erronious values. Hence we keep checking
        // that JoyMode is correct. If not, we clear it, to correct it.
        switch (JoyMode) {
          case -1: JoyMoveBwd(); if (JoyY > 106) {JoyMode = 0;  Walk = 0;} break;
          case  1: JoyMoveFwd(); if (JoyY < 150) {JoyMode = 0;  Walk = 0;} break;
          case  2: JoyTurnRgt(); if (JoyX < 149) {JoyMode = 0;  Walk = 0;} break;
          case  3: JoyTurnLft(); if (JoyX > 105) {JoyMode = 0;  Walk = 0;} break;
          case  4: JoyMoveRgt(); if (JoyX < 149) {JoyMode = 0;  Walk = 0;} break;
          case  5: JoyMoveLft(); if (JoyX > 105) {JoyMode = 0;  Walk = 0;} break;
          
          case 6: JoyMode = 0; DemoMove_Bow(); break;
          case 7: JoyMode = 0; DemoMove_SitNWave(); break;
        }
      } JoyActive = true; // re-enable this code block
    }
  } else {
  //##############################################################################
  //
  //  No WiFi Nunchuk received recently
  //
  //##############################################################################
    if (WiFiEn) {
      // HexBot is enabled for Wii WiFi control
      // maintain Wii enabled counter
      if ((WiFiCntC < 64) && (CZ != 0)) {
        WiFiCntC++; LEDVal = WiFiCntC/10;
        if (WiFiCntC == 64) {LED_Task = -1;} else {LED_Task = -10;}
      }
    }
  }

  if (WiFiEn) {
  //##############################################################################
  //
  //  WiFi background tasks when WiFi is enabled
  //
  //##############################################################################
    // only do these functions if WiFi is enabled
    if (MoveState == 20) {
      // long un-demanded moves like bowing are watched until they complete
      MoveState = 0; LED_Task = -1;
    }
    
    if (MoveState == -1) {
      // at the end of a joystick move we want to set things back to normal
      LED_Task = -1;
    //  SetSpeedDefaults();
      MoveState = 0;
    }
  }
  
  readWiiCall = false; // re-allow calls to this function
  // Serial.print("CntZ "); Serial.println(WiFiCntZ);
  // Serial.println(MoveState);
}

// --------------------------------------------------------------------------------

void RxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Rx buffer of length zlen
  Rx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    Rx_Chk = Rx_Chk ^ Rx_Buff.ESPdata[zP];
  }
}

// --------------------------------------------------------------------------------

void SetAngle(int zS, int zA) {
  // sets a specific servo zS to an angle zA immediately
  // if zS >= 10 then set target only and don't move the servo
  // note leg servos are numbered 1 - 8
  if (ESC) return;
  
  anyVal = 0; if (zS >= 10) {zS = zS - 10; anyVal = 1;}
  if (!servoAtt[1]) {attachServos(0);}  // servos were detached
  if (zA >= 0) {
    switch (zS) {
      case 0: servoTgt[1] = map(zA,-60,60,Head_60N,Head_60P); break;
      case 1: servoTgt[1] = map(zA,45,135,Ang1_45,Ang1_135); break;
      case 2: servoTgt[2] = map(zA,65,147,Ang2_65,Ang2_147); break;
      case 3: servoTgt[3] = map(zA,45,135,Ang3_45,Ang3_135); break;
      case 4: servoTgt[4] = map(zA,65,147,Ang4_65,Ang4_147); break;
      case 5: servoTgt[5] = map(zA,45,135,Ang5_45,Ang5_135); break;
      case 6: servoTgt[6] = map(zA,65,147,Ang6_65,Ang6_147); break;
      case 7: servoTgt[7] = map(zA,45,135,Ang7_45,Ang7_135); break;
      case 8: servoTgt[8] = map(zA,65,147,Ang8_65,Ang8_147); break;
    }
    if (anyVal < 1) {
      // zS is not >= 10 so set servo immediately
      if (servoVal[zS] != servoTgt[zS]) {
        servoVal[zS] = servoTgt[zS]; servoInst[zS].writeMicroseconds(servoVal[zS]);
        // Serial.println(servoVal[zS]);
      }
    }
  }
}

// ----------------------------------------------------------------------

void SetAngle45To90(int zS, int zA) {
  // this is called to correct leg swing non-linearity, which occurs near
  // the 90 degree angle range. Angles near 45 degrees are not affected.
  if (zA >= 74) {zA++;
    if (zA >= 80) {zA++;
      if (zA >= 84) {zA++;
        if (zA >= 87) {zA++;
          if (zA >= 90) {zA++;
          }
        }
      }
    }
  }
  // now set the angle as if normal
  SetAngle(zS,zA);
}

// ----------------------------------------------------------------------

void SetAngle90To135(int zS, int zA) {
  // this is called to correct leg swing non-linearity, which occurs near
  // the 90 degree angle range. Angles near 135 degrees are not affected.
  if (zA <= 106) {zA--;
    if (zA <= 100) {zA--;
      if (zA <= 96) {zA--;
        if (zA <= 93) {zA--;
          if (zA <= 90) {zA--;
          }
        }
      }
    }
  }
  // now set the angle as if normal
  SetAngle(zS,zA);
}

// --------------------------------------------------------------------------------

void SetLedMode(int zM) {
  // sets the LED mode and clears flags & counters if mode changes
  Tick20ms = false;     // if == true then run LED tasks at 20ms rate
  LedLast = LedMode;    // save current task as previous task when changing
  if (zM != LedMode) {LedCnt = 0; LedMem = -1; LED_Task = 0; LED_Period = 20;}
  LedMode = zM;
  // Serial.println("SetLedMode" + String(LedMode));
}

// --------------------------------------------------------------------------------

void SetMainTask(int zMT) {
  // sets the main task and associated flags
  // immediately stop any movement
  if (zMT == 99) {MainTask = 99; ESC = 5; return;}

  DispMode = 0;         // the type of display for a given mode
  TgtCnt = 0;           // clear servo target counter
  Walk = 0;             // stop walking 
  HeadPause = 100;      // don't move head for 2 seconds
  ESC = 0;              // remove ESC flag if set
  HeadMode = -1;        // default is non-ranging mode
  HeadTask = 0;         // effectively turn off head movement tasks
  HeadTaskLast = 0;     // previous head task pointer
  headInterval = 20000; // main loop head movement interval in microseconds, variable
  LedNop = false;       // if == true then skip LED modes completely
  LTOFcalc = false;     // if == true then calculate speed of ranged object in m/s
  MainDel = 0;          // MainTask delay counter, default = 0
  MainRun = false;      // remove MainTask inhibit flag
  MainTask = zMT;       // point to new task
  MainTaskRef = zMT;    // record which task was set here
  MoveType = "";        // type of move taking place
  Range = LtofMax;      // set range to default max
  RangeAng = Head_0;    // centre the head range PWM angle
  RangeMax = LtofLimit; // the current limited to Range measurements
  RangeMin = LtofMin;   // the current lower limited to Range measurements
  RangeRawLast = 1000;  // 16-bit previous range value read from VL53L0X sensor
  servoTgtLft = Head_60N; // set right most PWM angle
  servoTgtRht = Head_60P; // set right most PWM angle
  SetLedMode(0);        // remove any LED flashing
  SubLast = -1;         // reset previous subtask pointer
  SubTask = 0;          // zero subtask pointer
  Turn = false;         // default = false; if true turns continuously until = false
  VL53L0X_Disable();    // laser ranging task disabled by default
  Walked = 0;           // counter used to return robot to a given position
  walkInterval = 20000; // main loop leg movement interval in microseconds, variable
  synchLoopTimers();    // reset all multi-tasking timers
  WalkLftDrive = 128;   // drive factor for left-hand side, 128 = max, 0 = min
  WalkRgtDrive = 128;   // drive factor for right-hand side, 128 = max, 0 = min
  if (HeadActive) {if (HeadTask != 0) {HeadTask = 0; HeadToRest();}}
  
  switch(MainTask) {
    case -2:  // TEST mode
      VL53L0X_Enable();               // enable laser ranging task
      SetLedMode(10); break;          // start with random pattern
    case 1:
      VL53L0X_Enable();               // enable laser ranging task
      if (HeadActive) {HeadTask = 1;} // activate scanning
      else {HeadTask = 99;}
      SetLedMode(5); break;
    case 2:
      VL53L0X_Enable(); // enable back away task
      if (HeadActive) {HeadTask = 1;} // activate scanning
      else {HeadTask = 99;}
      SetLedMode(5); break;
    case 3:
      VL53L0X_Enable(); // enable target tracking task
      if (HeadActive) {HeadTask = 1;} // activate scanning
      else {HeadTask = 99;}
      SetLedMode(5); break;
    case 4:
      VL53L0X_Enable(); // enable autonomous task
      if (HeadActive) {HeadTask = 1;} // activate scanning
      else {HeadTask = 99;}
      SetLedMode(5); break;
  }
  Serial.println("MainTask = " + String(MainTask));
}

// --------------------------------------------------------------------------------

void SetToTgtNow() {
  // set servos to target values immediately, if not at targets
  if (!servoAtt[1]) {attachServos(0);}  // servos were detached
  for (anyFor = 1;anyFor <= 8;anyFor++) {
    if (servoVal[anyFor] != servoTgt[anyFor]) {
      servoVal[anyFor] = servoTgt[anyFor];
      if (servoLast[anyFor] != servoVal[anyFor]) {servoInst[anyFor].writeMicroseconds(servoVal[anyFor]); servoLast[anyFor] = servoVal[anyFor];}
    }
  }
}

// ----------------------------------------------------------------------

void VL53L0X_Disable() {
  // switches OFF the VL53L0X device inhibiting the laser
  digitalWrite(XSHUT,LOW); // disable the VL53L0X
  LTOF_On = 0;      // reset the measurement statemachine
  RangeEn = false;  // ranging is OFF
  // Serial.println("VL53L0X disabled");
}

// ----------------------------------------------------------------------

void VL53L0X_Enable() {
  // switches ON and initialises the VL53L0X device
  digitalWrite(XSHUT,HIGH); // enable the VL53L0X
  delay(1);
  VL53L0X_init(true); delay(10);
  VL53L0X_setTimeout(500);

  // Start continuous back-to-back mode (take readings as
  // fast as possible).  To use continuous timed mode
  // instead, provide a desired inter-measurement period in
  // ms (e.g. sensor.startContinuous(100)).
  VL53L0X_startContinuous(); delay(10);
  LTOF_On = 1;     // restart the measurement statemachine
  RangeEn = true;  // ranging is ON
  // Serial.println("VL53L0X enabled");
}

// --------------------------------------------------------------------------------

void VL53L0X_Run() {
  // called from the main loop every millisecond when LTOF_On > 0
  // When updating LEDs we stop I2C action
  if (VL53L0X_Show > 0) {VL53L0X_Show--; return;}
  if (VL53L0X_Skip > 0) {VL53L0X_Skip--; return;}

  switch(LTOF_On) {
    // this is the VL53L0X_readRangeContinuousMillimeters() function broken down
    // into a state machine which take measurements when LTOF_On > 0
    // use the VL53L0X_Enable/Disable functions to switch the laser ON/OFF
    case 0: break;
    case 1: 
      // Restart the timeout period during which a measurement should occur, as the
      // dev ice is opperating in continuous measurement mode
      startTimeout(); LTOF_ESC = 0; LTOF_On++; break;
    case 2:
      // Here we read the status of the interrupt register over I2C to see if a
      // measurement has been completed. We also check for a timeout occuring.
      if ((VL53L0X_readReg(RESULT_INTERRUPT_STATUS) & 0x07) == 0) {
        if (checkTimeoutExpired()) {
          did_timeout = true;
        //  return 65535;
        }
      } else {LTOF_On++;} // an interrupt has occured so move onto reading it
      // Increment the LTOF_ESC counter on every pass
      LTOF_ESC++; if (LTOF_ESC>= 40) {LTOF_On = 99;} // forced reset
      break;
    case 3:
      // assumptions: Linearity Corrective Gain is 1000 (default);
      // fractional ranging is not enabled
      RangeRaw = VL53L0X_readReg16Bit(RESULT_RANGE_STATUS + 10) - LTOF_Offset;
      LTOFt1 = millis();  // record time of reading
      VL53L0X_writeReg(SYSTEM_INTERRUPT_CLEAR, 0x01);
      LTOF_Wait = LTOFt1 + 28;
      // limits and filtering
      // when the range is long we can get transient errors,l so we use rate
      // limiting to minimise this effect
      if (RangeRaw > 1000) {RangeRaw = 1000;}                               // limit raw range measurements to 1000
      if ((RangeRawLast - RangeRaw) > 100) {RangeRaw = RangeRawLast - 100;} // rate limiting
      RangeRawLast = RangeRaw;
      Range = (RangeRaw + RangeLast)>>1;      // average range value over two readings
      RangeUL = false;
      if (Range > LtofLimit) {Range = LtofLimit; RangeUL = true;}
      RangeLL = false;
      if (Range < LtofMin) {Range = LtofMin; RangeLL = true;}
    //  Serial.println(Range);
    //  Serial.println(String(RangeRaw) + "," + String(Range) + ",0,500");
    //  Serial.print(Range); Serial.print("\t"); Serial.print(servoVal[0]); Serial.print("\t"); Serial.print(RangeAng); Serial.print("\t"); Serial.println(RangeCentreTracker);
      LTOF_On++; break;
    case 4:
      // calculate range difference
      RangeDiff = (int)Range - (int)RangeLast;
      RangeLast = Range;
      // calculate measurement period
      LTOFperiod = LTOFt1 - LTOFt0; LTOFt0 = LTOFt1;
      // calculate the fps frequency
      LTOFfps = 1000/LTOFperiod;
      LTOF_On++; break;
    case 5:
      // as the sensor only returns measurements every 30-32ms, there is no point in
      // spending time reading interrupt registers over I2C, so we wait here until
      // the wait period expires
      if (millis() >= LTOF_Wait) {LTOF_On = 1;} // wait for time-out reached
      break;
    
    case 99:
      // The device has timed-out. so we re-initialise it 
      // switches ON and initialises the VL53L0X device
      digitalWrite(XSHUT,LOW);  // disnable the VL53L0X
      LTOF_On++; break;
    case 100:
      digitalWrite(XSHUT,HIGH); // enable the VL53L0X
      LTOF_On++; break;
    case 101:
      VL53L0X_init(true); VL53L0X_Skip = 10;
      LTOF_On++; break;
    case 102:
      VL53L0X_setTimeout(500);
      VL53L0X_startContinuous(); VL53L0X_Skip = 10;
      LTOF_On = 1;     // restart the measurement statemachine
      RangeEn = true;  // ranging is ON
      break;
  }
}

// --------------------------------------------------------------------------------

void Wait_ButtonsDwn() {
  // wait in this function until SW0 is released
  // Serial.println("Wait_ButtonsDwn()");
  while (!digitalRead(sw0Pin)) {
    // one or both buttons held down
    yield();  // let system functions run in background
  }
  sw0Cnt = 0; sw0DwnTime = 0; sw0LastState = HIGH; sw0_Nop = false; sw0State = HIGH; sw0Timer = 0;
  synchLoopTimers();
}

// ----------------------------------------------------------------------

int WiiRightStickX() {
  // returns a Wii Classic RX value 0/16/31
  return ((RxWiFi[0] & (byte)0xc0) >> 3) + ((RxWiFi[1] & (byte)0xc0) >> 5) +  ((RxWiFi[2] & (byte)0x80) >> 7);
}

// ----------------------------------------------------------------------

int WiiRightStickY() {
  // returns a Wii Classic RY value 0/16/31
  return RxWiFi[2] & (byte)0x1f;    
}

// --------------------------------------------------------------------------------




  //###############################################################################
  //
  //  WiFi Code
  //
  //###############################################################################

void WiFiClearRxBuff(int zNum) {
  // clears the contents of the receive buffer
  for (int zP = 0; zP < zNum;zP++) {Rx_Buff.ESPdata[zP] = 0;}
}

// --------------------------------------------------------------------------------

void WiFiClearTxBuff(int zNum) {
  // clears the contents of the transmit buffer
  for (int zP = 0; zP < zNum;zP++) {Tx_Buff.ESPdata[zP] = 0;}
}

// --------------------------------------------------------------------------------

void WiFiDisconnect() {
  // enter the WiFi disconnected state
  WiFiConnected = false;
  WiFiClearRxBuff(WiFiTx_len);
  WiFiClearTxBuff(WiFiTx_len);
  WiFiTryCnt = 0;     // reset the connection retry counter
  WiFiTx_CB = 1;      // set ready to send flag
  WiFiRxRec = false;  // clear the data received flag
  Serial.println("WiFiDisconnected!");
}

// --------------------------------------------------------------------------------

void WiFiTryToConnect() {
  // try to connect with the Wii Transceiver by sending devices name
  // initialise ESP-NOW link first
  Init_ESP_NOW();
  
  Tx_Buff.ESPdata[0] = 'Q';
  Tx_Buff.ESPdata[1] = 'u';
  Tx_Buff.ESPdata[2] = 'a';
  Tx_Buff.ESPdata[3] = 'd';
  Tx_Buff.ESPdata[4] = 'A';
  Tx_Buff.ESPdata[5] = 'u';
  Tx_Buff.ESPdata[6] = 'o';
  WiFiTx_len = 7;
  // t0 = micros();
  esp_err_t result = esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
  if (result == ESP_OK) {
  //  Serial.println("Sent with success");
  } else {
  //  Serial.println("Error sending the data");
  } 

  WiFiTryNum++; // count the total number of attempts
  // Serial.println("WiFiTryToConnect...");
}

// --------------------------------------------------------------------------------

void WiFiTxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Tx buffer of length zlen
  WiFiTx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    WiFiTx_Chk = WiFiTx_Chk ^ Tx_Buff.ESPdata[zP];
  }
}

// ---------------------------------------------------------------------------------
