
/* SSD1306Wire.h functions, see https://github.com/ThingPulse/esp8266-oled-ssd1306

  disply.clear()        - clear the local pixel buffer
  disply.display()      - write the buffer to the display
  disply.displayOff()   - turn the display OFF
  disply.displayOn()    - turn the display ON
  display.drawCircle(int16_t x, int16_t y, int16_t radius)
  display.drawFastImage(int16_t x, int16_t y, int16_t width, int16_t height, const uint8_t *image)
  display.drawHorizontalLine(int16_t x, int16_t y, int16_t length)
  display.drawLine(int16_t x0, int16_t y0, int16_t x1, int16_t y1)  - draw from x0k,y0 to x1,y1
  display.drawProgressBar(uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint8_t progress)
  display.drawRect(int16_t x, int16_t y, int16_t width, int16_t height)
  display.drawString(int16_t x, int16_t y, String text)
  display.drawStringMaxWidth(int16_t x, int16_t y, int16_t maxLineWidth, String text)
  display.drawVerticalLine(int16_t x, int16_t y, int16_t length)
  display.drawXbm(int16_t x, int16_t y, int16_t width, int16_t height, const char* xbm)
  display.fillCircle(int16_t x, int16_t y, int16_t radius)
  display.fillRect(int16_t x, int16_t y, int16_t width, int16_t height)
  display.flipScreenVertically()   - flip the current screen upside down
  disply.init()         - initialise the display
  disply.invertDisplay()- inverted display mode
  disply.end()          - free memory used by the display
  disply.displayOn()    - turn the display ON
  display.mirrorScreen() - draw the screen mirrored
  disply.normalDisplay()- normal display mode
  disply.reconnect()    - connect again through I2C
  display.setBrightness(uint8_t)
  display.setColor(OLEDDISPLAY_COLOR color) - BLACK, WHITE, INVERSE
  display.setContrast(uint8_t contrast, uint8_t precharge = 241, uint8_t comdetect = 64)
  display.setFont(const uint8_t* fontData)
  display.setPixel(int16_t x, int16_t y)  - set a pixel at x,y
  display.setTextAlignment(OLEDDISPLAY_TEXT_ALIGNMENT textAlignment)  - TEXT_ALIGN_LEFT, TEXT_ALIGN_CENTER, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER_BOTH

  W = display.getStringWidth(const char* text, uint16_t length)
  W = display.getStringWidth(String text)
  
  Display size is 128 x 64 (w x h) pixels
  Default fonts are ArialMT at 10, 16 & 24 pixels high
  Text options include:
    2 x 24, + 1 x 10 pnt lines
    1 x 24, + 2 x 16 pnt lines
    1 x 24, + 3 x 10 pnt lines
    4 x 16 pnt lines, no line spaces, decending characters will merge
    3 x 16, + 1 x 10 pnt lines
    6 x 10 pnt lines, no line spaces, decending characters will merge
*/

// --------------------------------------------------------------------------------

void Display_40ms() {
  // called every 40ms from the main loop if DispMon == true
  // if DispDel > 0 then we wait for DispDel cycles to complete before update
  
  if (DispDel > 0) {DispDel--; return;} // don't update display if DispDel > 0
  if (DispClr) {Display_clear(); DispClr = false; return;}
  if (DispDisp) {Display_display(); DispDisp = false; DispClr = true; return;}

  // we refresh individual displays based on a counter, DispCnt
  // set DispCnt -ve to prevent the display update process
  DispCnt--;
  if (DispCnt < 1) {
    // DispCnt is reset for display updates, so can be different for each
    if (DispLock) {DispMode = DispOveride;} // force locked mode
    // normal menu screens, display depends on menu modes
    switch(DispMode) {
      case 0: Display_Battery(); break;
      case 1: Display_Range(); break;
      case 2: Display_WiFi(); break;
      case 3: Display_Status_A(); break;
    }
    DM_Min = 0; DM_Max = 3;
    // the following line puts a rectangle around every display, showing the boundry region
    // Border = true;
    // if ((Border) || TEST) {Display_drawRect(0,0,128,64);}
  }

  // Add a global frame delay offset, if set
  DispDel += DispDelGbl;
}

// --------------------------------------------------------------------------------

void Display_Battery() {
  // normal screen to be displayed
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //    1111111111111111    2222222222222222    3333333333333333
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 3, "QuadAuto");
  if (!USB) {Display_drawString(64,23, "Battery");} else {Display_drawString(64,23, "USB Mode");}
  AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
  if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
  if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
  Display_drawString(64,43, String(float(BatAvg)/BatCal) + "v  " + String(AnyLong) + "%");
  Display_display();
  DispDel = 25; // update every 1s
}

// --------------------------------------------------------------------------------

void Display_Mirrored() {
  // display robot type when DispMon is first connected
  // we clear current flags so that this is displayed immediately
  DispClr = false; DispDisp = false; DispCnt = 1;
  
  Display_clear();    // remove current display content
  Display_display();  // clears mirror display immediately
  if (Brightness != 255) {Display_setBrightness(255);}  // display may be in sleep mode
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111111111111111111 222222222222222222222222   3333333333
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "QuadAuto");
  Display_drawString(64, 25, "ESP32");
  Display_setFont(10);
  Display_drawString(64,52, "Display Mirror");
  Display_display();  // display immediately
  DispMode = DispNext;
  DispDel = 100;       // wait 4s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Range() {
  // display VL53L1X range measurement
  // MainTask:
  // 1 - LTOF ranging only
  // 2 - Backaway
  // 3 - Track target
  // 4 - Autonomous
  String zSL = ""; String zSR = "";
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111      2222222222222222      3333333333333333
  Display_clear();    // remove current display content
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
       if (MainTask == 2) {Display_drawString(64, 2, "Back Away");}
  else if (MainTask == 3) {Display_drawString(64, 2, "Track Mode");}
  else {Display_drawString(64, 2, "Range");}
  if (RangeEn) {
    if (RangeUL) {zSL = "< "; zSR = " >";}
    if (RangeLL) {zSL = "> "; zSR = " <";}
    Display_drawString(64,24, zSL + String(Range) + " mm" + zSR);
  } else {Display_drawString(65,24, "---");}
  if (MainTask == 1) {
    if (HeadActive) {
      // moving head, so draw a box representing angles
      int16_t zCX0 = map(servoTgtLft,Head_60N,Head_60P,0,127);
      int16_t zCX1 = map(servoTgtRht,Head_60N,Head_60P,0,127);
      int16_t zCXT = map(RangeAng,Head_60N,Head_60P,0,127);
      Display_drawRect(zCX0,46,zCX1 - zCX0 + 1,17);
      Display_drawLine(zCXT,46,zCXT,63);
      Display_drawLine(zCXT-1,49,zCXT-1,60);
      Display_drawLine(zCXT+1,49,zCXT+1,60);
    } else {Display_drawString(64,46,String(LTOFfps) + " fps");}
  } else {
    // in Backaway or Tracking mode
    Display_setFont(10);
    // show move type, like 'Looking'
    Display_drawString(64,46,MoveType);
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    // show SubTask value
    Display_drawString( 0,46,String(SubTask));
    if ((Walk == 1) || (Walk == 3) || (Walked != 0)) {
      // show distance travelled
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(127,46,String(Walked));
    }
  }
  Display_setFont(5);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(127,58,String(LTOF_On));

  Display_display();  // display this immediately
  DispDel = 4;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Status_A() {
  // Display a status screen
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111    22222222 33333333 44444444 55555555 66666666
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(63, 2, "Status A");

  Display_setFont(8);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 45,16, "MainTask:");
  Display_drawString(104,16, "JoyMode:"); 
  Display_drawString( 44,25, "LedMode:"); 
  Display_drawString(104,25, "LedTask:"); 
  Display_drawString( 45,34, "Walk:"); 
  Display_drawString(104,34, "Interval:"); 
  Display_drawString( 45,43, "WiFiEn:"); 
  Display_drawString(104,43, "Connected:"); 
  Display_drawString( 45,52, "RxErr:"); 
  Display_drawString(104,52, "TxErrCnt:"); 

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 46,16, String(MainTask)); 
  Display_drawString(105,16, String(JoyMode)); 
  Display_drawString( 46,25, String(LedMode)); 
  Display_drawString(105,25, String(LED_Task)); 
  Display_drawString( 46,34, String(Walk)); 
  Display_drawString(105,34, String(walkInterval)); 
  Display_drawString( 46,43, String(WiFiEn)); 
  Display_drawString(105,43, String(WiFiConnected)); 
  Display_drawString( 46,52, String(WiFiRxErr)); 
  Display_drawString(105,52, String(WiFiTxErrCnt)); 

  Display_display();  // display this immediately
  DispDel = 5;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Text1x16(String zLine0,int16_t zDel) {
  // display one line of text in 16 font
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                        1111111111111111
  Display_clear();    // remove current display content
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,23, zLine0);
  Display_display();  // display this immediately
  DispDel = zDel;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Text2x16(String zLine0,String zLine1,int16_t zDel) {
  // display two lines of text in 16 font
  // 0123456789012345678901234567890123456789012345678901234567890123
  //            1111111111111111    2222222222222222
  Display_clear();    // remove current display content
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,11, zLine0);
  Display_drawString(64,31, zLine1);
  Display_display();  // display this immediately
  DispDel = zDel;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_WiFi() {
  // display WiFi control
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 1111111111111111    2222222222     3333333333    4444444444
  Display_clear();    // remove current display content
  String zC = "1"; String zZ = "1";
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "Wi-Fi");
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(10,20, "JX:" + String(JoyX));
  Display_drawString(50,20, "JY:" + String(JoyY));
  if ((CZ & 2) == 0) {zC = "0";}
  if ((CZ & 1) == 0) {zZ = "0";}
  Display_drawString(90,20, "CZ: " + zC + zZ);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,35,MoveType);
  AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
  if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
  if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 0, 49, "Gear: " + String(WalkSpeed));
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(127, 49, "Bat: " + String(float(BatAvg)/BatCal) + "v  " + String(AnyLong) + "%");
  Display_display();  // display this immediately
  DispDel = 4;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

