
/* ###############################################################################
  PLAY functions

  These functions implement the record and playback features, used with a Wii
  Classic and Classic Pro controllers. The method equally applies to other projects
  using ESP-NOW WiFi link and Wii controllers, like SpidaBot for example.

  A text loading and display function was built into the Monitor+ app, so that you
  can effectively transfer, save edit and reuse controller sequences.

  In order to use this cope the following definitions and variables need to be
  defined at the front of your code:

  #define RECnum 5000           // depth of RECord arrays 

  bool BLT;                     // Wii Classic front L button states
  bool BRT;                     // Wii Classic front R button states

  DM_Recall;                    // RECall display assigned Display Mode

  bool PLAY;                    // set == true to play a recorded controller sequence
  int16_t PLAYinc;              // PLAY time tracker
  int16_t PLAYpnt;              // PLAY step pointer
  bool PLAYstart;               // if == true then waiting to start PLAY mode
  int32_t PLAYtime;             // PLAY step time counter
  int32_t PLAYtotal;            // PLAY total time

  bool REC;                     // RECording flag, default = false
  byte RECmem0[RECnum];         // RECord array element 0
  byte RECmem1[RECnum];         // RECord array element 1
  byte RECmem2[RECnum];         // RECord array element 2
  byte RECmem3[RECnum];         // RECord array element 3
  byte RECmem4[RECnum];         // RECord array element 4
  byte RECmem5[RECnum];         // RECord array element 5
  int16_t RECpnt;               // RECord array pointer
  bool RECstart;                // if == true then waiting to start RECord mode
  int32_t RECtime[RECnum];      // RECord time counters
  int32_t RECtotal;             // RECord total time

  As the PLAY features essentially breaks into the normal flow of data used by
  the WiFi control system, you also need to insert those elements of code. You will
  find them under the Functions.ino tab, if you search for: PLAY####

   ###############################################################################
*/

// --------------------------------------------------------------------------------

void PLAY_CLR() {
  // Resets PLAY flags and counters
  PLAY = false;       // disable play mode
  PLAYpnt = 0;        // reset play pointer
  PLAYstart = true;   // if == true then waiting to start PLAY mode, default
  PLAYtime = 0;       // reset 1st record play time
  PLAYtotal = 0;      // reset play total
}

// ---------------------------------------------------------------------------------

void PLAY_Loader(byte zR0,byte zR1,byte zR2,byte zR3,byte zR4,byte zR5,uint32_t zTime) {
  // This functions loaded the REDCmem[] arrays with data, as if from the Wii
  // controller, but sourced from a recording.
  // Before calling this function, for the first element, set RECpnt = 0.
  RECmem0[RECpnt] = zR0;
  RECmem1[RECpnt] = zR1;
  RECmem2[RECpnt] = zR2;
  RECmem3[RECpnt] = zR3;
  RECmem4[RECpnt] = zR4;
  RECmem5[RECpnt] = zR5;
  RECtime[RECpnt] = zTime;
  RECpnt++;                 // RECord array pointer, points at end of the array stack

}

// ---------------------------------------------------------------------------------

void PLAY_now() {
  // Play controller recorded data. until end, or stops if user operates controller.
  // PLAY mode does not start until controller operations ceases.
  // During playback any movement of the controller will cancel playback mode.
  if (Wii_Op_Check()) {
    if (PLAYstart) {return;}    // waiting for controller to be free
    PLAY = false; return;       // stop playing sequence
  }
  if (PLAYstart) {
    PLAYstart = false;            // clear the start flag
    DispMode = DM_Recall;
  }
  // Play the recorded data
  PLAYtime++; PLAYtotal++;
  // Look for end of whole recording
  if (PLAYtotal >= RECtotal) {PLAY_CLR(); return;}
  // Look for end of this step
  if (PLAYtime >= RECtime[PLAYpnt]) {
    // At the end of the current step, so move to the next
    PLAYtime = 1; PLAYpnt++;
  }
  // Now load recorded controller register values, replacing those received
  // from the Wii over WiFi, providing the playback control.
  RxWiFi[0] = RECmem0[PLAYpnt];
  RxWiFi[1] = RECmem1[PLAYpnt];
  RxWiFi[2] = RECmem2[PLAYpnt];
  RxWiFi[3] = RECmem3[PLAYpnt];
  RxWiFi[4] = RECmem4[PLAYpnt];
  RxWiFi[5] = RECmem5[PLAYpnt];
}

// ---------------------------------------------------------------------------------

void PLAY_STOP() {
  // Called to reset PLAY flags and load defaults into Rx buffer
  PLAY_CLR();
  RxWiFi[0] = 160;
  RxWiFi[1] =  32;
  RxWiFi[2] =  16;
  RxWiFi[3] =   0;
  RxWiFi[4] = 255;
  RxWiFi[5] = 255;
}

// ---------------------------------------------------------------------------------

void PLAY_Toggle() {
  // Toggles the PLAY flag, and clears the REC flag
  if (PLAY) {PLAY_STOP();}
  else {if (RECtotal > 0) {PLAY = true;}}
  REC = false;
}

// ---------------------------------------------------------------------------------

void PLAY_Upload() {
  // Sends recorded data to Monitor+ app.
  // This method can be used to send other text data to Monitor+.
  // Note this blocks other functions, whilst uploading, and can take some time,
  // depending on the size of recorded data.
  BLOCKED = true;   // Block Monitor+ updates in response to ACKs
  OLED_Text2S10("Recording","Uploaded");
  PrintTx+= "<$<$\n"; // send the text block start command
  // Now add header information
  PrintTx+= "// If not already recording, reset the record pointer\n";
  PrintTx+= "if (!REC) {RECpnt = 0;}\n";
  PrintTx+= "//         (zR0,zR1,zR2,zR3,zR4,zR5,zTime)\n";
  for (int16_t zL = 0;zL < RECpnt; zL++) {
    // Ensure string buffer does not get overloaded
    while (PrintTx.length() > 128) {
      // handle received packets as soon as possible
      if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events
      delay(8);         // give the handler time to send data
      PrintTxHandler();
    }
    // Include the next line
    PrintTx+= "PLAY_Loader(" + ret3Char(RECmem0[zL]) + "," + ret3Char(RECmem1[zL]) + "," + ret3Char(RECmem2[zL]) + "," + ret3Char(RECmem3[zL]) + "," + ret3Char(RECmem4[zL]) + "," + ret3Char(RECmem5[zL]) + "," + String(RECtime[zL]) + ")\n";
    // handle received packets as soon as possible
    if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events
    yield();
  }
  PrintTx+= "$>$>\n"; // send the text block end command
  RTxTimeout = 50;    // reset timeout to approx 1 sec
  BLOCKED = false;    // Release Monitor+ function
  DisplayText2S16("Recording","Loaded");
}

// --------------------------------------------------------------------------------

void REC_Chng_Check() {
  // Called from readWiFiRx(), when in RECording mode.
  // RECording mode does not start until controller actions cease.
  if (RECstart) {
    if (Wii_Op_Check()) {return;}   // controller is still in use, so abort.
    RECstart = false;               // clear flag to start RECording
    DispMode = DM_Recall;           // switch display to recording
  }
  bool zDif = false;  // RX data is different flag
  // Look for changes
       if (RECmem0[RECpnt] != RxWiFi[0]) {zDif = true;}
  else if (RECmem1[RECpnt] != RxWiFi[1]) {zDif = true;}
  else if (RECmem2[RECpnt] != RxWiFi[2]) {zDif = true;}
  else if (RECmem3[RECpnt] != RxWiFi[3]) {zDif = true;}
  else if (RECmem4[RECpnt] != RxWiFi[4]) {zDif = true;}
  else if (RECmem5[RECpnt] != RxWiFi[5]) {zDif = true;}

  if (!zDif) {
    // Data received matches data stored, so no controller change.
    // Simply increment the time counter.
    RECtime[RECpnt]++; RECtotal++;
    // Check for 59min 59sec == 89,975 counts at 25Hz (40ms)
    if (RECtotal >= 89975) {REC = false;} // time limit reached
  } else {
    // New controller action has occured.
    RECpnt++; // increment the memory pointer
    // Check for end of memory
    if (RECpnt == RECnum) {REC = false; return;}  // end of memory reached
    // Store latest values
    RECmem0[RECpnt] = RxWiFi[0];
    RECmem1[RECpnt] = RxWiFi[1];
    RECmem2[RECpnt] = RxWiFi[2];
    RECmem3[RECpnt] = RxWiFi[3];
    RECmem4[RECpnt] = RxWiFi[4];
    RECmem5[RECpnt] = RxWiFi[5];
    // Initialise the time count, for this new change
    RECtime[RECpnt] = 1;
  }
}

// --------------------------------------------------------------------------------

void REC_CLR() {
  // Clears recorded controller changes, and resets flags
  REC = false;      // disable record mode
  RECpnt = 0;       // reset RECord step pointer
  RECstart = false; // if == true then waiting to start RECord mode, default
  RECtime[0] = 0;   // reset 1st RECord time
  RECtotal = 0;     // reset RECord total time
  // Set the 1st array elements to the default values
  RECmem0[0] = 160;
  RECmem1[0] =  32;
  RECmem2[0] =  16;
  RECmem3[0] =   0;
  RECmem4[0] = 255;
  RECmem5[0] = 255;
}

// --------------------------------------------------------------------------------

void REC_Toggle() {
  // Toggles the REC flag, and clears the PLAY flag
  if (REC) {REC = false;} else {REC = true;}
  PLAY = false;
}

// --------------------------------------------------------------------------------

String ret3Char(byte zV) {
  // Returns a 3 character representation of zV, with leading spaces as padding.
  // zV is in the range 0 - 255
  String zRet = String(zV);
  if (zV <  10) {zRet = " " + zRet;}
  if (zV < 100) {zRet = " " + zRet;}
  return zRet;
}

// --------------------------------------------------------------------------------

bool Wii_Op_Check() {
  // Called to check if a Wii Classic controller is being operated.
  // Returns true if a button or joystick is operated
  bool zDif = false;  // RX data is different flag
  // Look for changes
       if (RxWiFi[0] != 160) {zDif = true;}
  else if (RxWiFi[1] !=  32) {zDif = true;}
  else if (RxWiFi[2] !=  16) {zDif = true;}
  else if (RxWiFi[3] !=   0) {zDif = true;}
  else if (RxWiFi[4] != 255) {zDif = true;}
  else if (RxWiFi[5] != 255) {zDif = true;}
  return zDif;
}

// ---------------------------------------------------------------------------------

