 // ################################################################################
//
//  Omni-Bot v0.0 (R1)
//
//  Released:  28/11/2025
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Micro:  ESP32 Dev Module      v3.3.4      IDE: v2.3.6

      Partition scheme: "Huge APP (3MB No OTA/1MB SPIFFS)"

    This code controls a 3x3 robot chassis, with an OLED display, light sensors, acoustic
    range sensors, and 3-axis motion sensor.
    Functions provided are as follows:
    * Motor drive via Wii controllers over Wi-Fi, Nunchuk and Classics supported
    * 64x48 OLED Display
    * Light sensor tracks torch activity
    * Range sensors respond to hand movements
    * WiFi remote control using Wii Nunchuk
    * Able to connect to multiple Wii WEMOS D1 Transcievers
    
    IMPORTANT - library updates occur on a regular basis, and sometimes cause problems
    with code, that no longer compiles. So here I have included the version numbers, so
    that if you experience problems, you can revert back to the original versions.

    This version includes a fix to ESPNOW, to compile in v3.3.4
    Enjoy!
*/
// Declare Libraries
#include <Adafruit_GFX.h>     // Adafruit graphics lib      v1.12.4
#include "SSD1306Wire.h"      // OLED display library       v4.6.1
#include <FastLED.h>          // Neopixel library           v3.10.3

#include <esp_now.h>          // ESP-NOW WiFi library       v3.3.4
#include <WiFi.h>             // ESP WiFi library           v3.3.4

// Configuration
String Rev = "vR1";
String Release = "Omni-Bot ESP32 " + Rev;
String Date = "28/11/2025";

// I2C address map
// 0x3C OLED display
// 0x68 MPU 6050 3-axis motion sensor

// Display Settings
const int I2C_DISPLAY_ADDRESS = 0x3c;
const int SDA_PIN = 21;         // I2C SDA GPIO pin
const int SCL_PIN = 22;         // I2C SCL GPIO pin

// Initialize the oled display for address 0x3c
SSD1306Wire display(I2C_DISPLAY_ADDRESS, SDA_PIN, SCL_PIN);

// Define a task for core 0
TaskHandle_t Core0;

// 3-axis ICM-42607-P variables
const int MPU_addr=0x68;      // I2C address of the MPU-6050
long AcAvgDiv = 0;            // dividing factor used in averaging
long AcAvgDivMax = 250;       // max dividing factor for averaging
int16_t AccRawX = 0;          // ICM raw accelerometer value
int16_t AccRawY = 0;          // ICM raw accelerometer value
int16_t AccRawZ = 0;          // ICM raw accelerometer value
int16_t AcX = 0;              // corrected X-acceleration value
long AcXAvg = 0;              // averaged AcX value to reduce noise
int16_t AcXL = 0;             // previous MPU X-acceleration value
int16_t AcXOff = -187;        // offset to be added to raw X-acceleration value
long AcXSum = 0;              // averaging accumulator
int16_t AcY = 0;              // MPU raw Y-acceleration value
long AcYAvg = 0;              // averaged AcY value to reduce noise
float AcYF = 0.0;             // floating point Y acceleration value
int16_t AcYL = 0;             // previous MPU Y-acceleration value
int16_t AcYOff = -111;        // offset to be added to raw X-acceleration value
bool AcYsgn = true;           // positive/negative sign flag
long AcYSum = 0;              // averaging accumulator
int16_t AxCnt = 0;            // small offset adjustnment counter
int16_t AyCnt = 0;            // small offset adjustnment counter
float DisX = 0.0;             // distance travelled on X-axis in mm
float DisY = 0.0;             // distance travelled on Y-axis in mm
long GyAvgDiv = 0;            // dividing factor used in averaging
int16_t GyCal = 250;          // counter used to determine gyro calibration period
int16_t GyCnt = 0;            // small offset adjustnment counter
int16_t GyrRawX;              // ICM raw gyro reading
int16_t GyrRawY;              // ICM raw gyro reading
int16_t GyrRawZ;              // ICM raw gyro reading
unsigned long GyT = 0;        // time of gyro readings in milliseconds
unsigned long GyTD = 1;       // time between successive gyro readings in milliseconds
int16_t GyZ = 0;              // MPU raw Z-gyroscope value
long GyZAvg = 0;              // averaged GyZ value to reduce noise
int16_t GyZL = 0;             // previous MPU Z-gyroscope value
int16_t GyZnt = 12;           // noise threshold for Z gyro move detection
int16_t GyZOff = 0;           // offset to be added to raw Z-gyroscope value
int16_t GyZOffCnt = 0;        // offset counter for gyro centering
long GyZSum = 0;              // averaging accumulator
uint32_t ICMtd;               // time between ICM readings in microseconds
uint32_t ICMtr;               // ICM readings time reference in microseconds
int16_t ICMove = 0;           // set > 0 when movement is detected
float VelX = 0.0;             // estimated veloxity on X axis
float VelY = 0.0;             // estimated veloxity on Y axis
float YawAcc;                 // yaw angle in degrees based on accelerometers
int16_t YawClr = 0;           // auto zero offset counter for Z-axis
int16_t YawCnt = 0;           // auto zero delay counter for Z-axis
float YawDif = 0.0;           // rotation angle difference in degrees based on gyro + acc
bool YawESC = false;          // rotation exit flag during edge detection
float YawGyr;                 // yaw angle in degrees based on gyro + acc
int YawGyrInt = 0;            // integer version of YawGyr for quicker calcs
float YawTgt = 0.0;                 // target angle in degrees, when turning

// Define constants for Monitor+ app
// Some definitions are commented out when using SSD1306Wire.h library
// #define BLACK 0
#define BLUE 5
#define GREEN 4
#define GREY 9
#define MAGENTA 8
#define RED 3
#define TURQUOISE 7
#define YELLOW 6

// Define constants
#define acc_1g  8192      // accelerometer 1g value
#define AngDifMax 250     // maximum head sweep angle from centre difference, in microseconds
#define AngDifMin 100     // minimum head sweep angle from centre difference, in microseconds
#define At_DRIVE 1        // AtState value for drive mode
#define At_LDR   4        // AtState value for LDR sense mode
#define At_OMNI  3        // AtState value for omni drive mode
#define At_READY 2        // AtState value for being ready
#define At_REST  0        // AtState value for resting
#define At_SONAR 5        // AtState value for sonar modes
#define autoRestMax 32000 // autoRest max counter value
#define Bat6v6 2460       // 6.6v critical battery threshold, shutdown (single cell 3.3v)
#define Bat7v2 2710       // 7.2v battery threshold 20% warning (single cell 3.6v)
#define Bat7v6 2872       // 7.6v battery threshold 80% (single cell 3.8v)
#define Bat8v2 3136       // 8.2v fully charged battery voltage, light load
#define Bat8v4 3226       // 8.4v fully charged battery voltage, O/C
#define BatUSB 1500       // minimum triggers USB supply assumption
#define BatPin 36         // analog input pin ADC1_0 on GPIO36
#define DbLLX 126         // JoyX deadband lower limit
#define DbULX 130         // JoyX deadband Upper limit
#define DbLLY 126         // JoyY deadband lower limit
#define DbULY 130         // JoyY deadband Upper limit
#define Echo0 16          // GPIO used for front sensor
#define Echo1  4          // GPIO used for right sensor
#define Echo2  2          // GPIO used for left sensor
#define echoLimit 4000    // pulse time-out limit in microseconds
#define echoMax 500       // Failed range response, 
#define GyAvgDivMax 2500  // max averaging divider, 2500 == 10 second average
#define I2Cdel 10         // I2C bus timing delay for ICM-42607-P, in microseconds
#define ICM_address 0x68  // ICM-42607-P I2C address (0x68 or 0x69)
#define ICMMoveMax 50     // value of ICMove set when moving
#define ICMMoveMin 10     // value of ICMove set when stopping after a move
#define INT_P34 34        // speed sensor interupt input FR
#define INT_P35 35        // speed sensor interupt input RL
#define INT_P36 36        // speed sensor interupt input FL
#define INT_P39 39        // speed sensor interupt input RR
#define INT_PIN 0         // VL53L1X GPIO interrupt to ADC input not actually used
#define LDR0_Pin 39       // GPIO for left-hand sensor
#define LDR1_Pin 34       // GPIO for right-hand sensor
#define LED_Pin 5         // assign GPIO5 for RGB LEDs
#define MoveNum 100       // depth of movement array
#define NumLEDs 15        // number of WS2812B LEDs
#define Pin_A0 33         // PWM pin for motor A front
#define Pin_A1 32         // PWM pin for motor A front
#define Pin_B0 26         // PWM pin for motor B right
#define Pin_B1 25         // PWM pin for motor B right
#define Pin_C0 13         // PWM pin for motor C rear
#define Pin_C1 27         // PWM pin for motor C rear
#define PwmBits 8         // PWM counter width 8 bits, 0 - 255
#define PwmFreq 30000     // PWM frequency is 30 kHz
#define PWM_StartMax 45   // PWM to overcome start inertia
#define PWM_StartMin 30   // PWM to maintain movement
#define RangeLL 20        // Range finder lower limit
#define RangeMax 500      // Failed range reading, set to 500 mm
#define RangeUL 300       // Range finder upper limit
#define RECnum 5000       // depth of RECord arrays 
#define sw0Pin 12         // left switch SW0 assigned to GPIO14
#define sw1Pin 15         // right switch SW1 assigned to GPIO15
#define Trig0 18          // GPIO used for trigger pulse
#define Trig1 17          // GPIO used for trigger pulse
#define Trig2 14          // GPIO used for trigger pulse
#define TrigDel 10        // RCWL trigger pulse width
#define XSHUT_PIN 32      // GPIO32 controls VL53L1X XSHUT HIGH enable pin
#define Xoff 32           // X offset needed in 64x48 OLED display
#define Yoff 16           // Y offset needed for flipped 64x48 OLED display
 
// create a FastLED instance and define the total number of LEDs on the robot
CRGB Mem[NumLEDs];        // LEDs saved, to recall later
CRGB LED[NumLEDs];        // LEDs connected to micro for o/p
CRGB LEDX[1];             // temp LED value

// my Wii Transciever unique MAC: 50:02:91:68:F7:3F
// you need to build a Wii Transceiver and get its MAC address
uint8_t broadcastAddress[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

// 250 byte max for Tx/Rx data
// It will be used to create the Tx/Rx data structures
typedef struct ESPNOW_Buff250 {
    char ESPdata[250];
} ESPNOW_Buff250;

// Create a 250 byte Tx buffer
ESPNOW_Buff250 Tx_Buff;

// Create a 250 byte receive
ESPNOW_Buff250 Rx_Buff;

// Declare and initialise global variables
int16_t ACK = 0;              // acknowledgement flag used with Monitor+
String Any$ = "";             // temporary string
uint8_t AnyByte;              // any byte value
int16_t AngDif, AngNeg, AngPos, AngRef; // temporary head angles
int anyInt = 0;               // any temporary integer value
long AnyLong = 0;             // any temp value
unsigned long anyMilli;       // any millisecond timer
int16_t anyVal = 0;           // any temp variable
int16_t AppCnt = 0;           // if > 0 then connected to a Windows app via USB or WiFi
int16_t AtState = At_REST;    // current at state, ie. REST, READY,STAND, etc...
String Auton$ = "";           // autonomous mode string
bool AutonCal = false;        // == true when ground calibration has been achieved
int16_t AutoMove = 0;         // autonomouse move status for display
int32_t BatAvg = 0;           // preload averaging function with high value
float BatMult = 1.0;          // battery multiplier used in PWM limiting
int32_t BatPc = 0;            // battery percentage of charge, as integer
int32_t BatSum = Bat8v2 * 20; // cumulative battery voltage
String BatTime$ = "";         // time left displayed
int32_t BatV = 0;             // battery voltage * 10, as in 10.0v == 100
float BatVfp = 0.0;           // battery voltage as a decimal number. ie. 11.2v
int32_t BatVol = 0;           // instantaneous battery voltage
bool BDD = false;             // Wiii Classic digital pad buttons
bool BDD_ = false;            // Wiii Classic digital pad, button down flags
bool BDL = false;             // Wiii Classic digital pad buttons
bool BDL_ = false;            // Wiii Classic digital pad, button down flags
bool BDR = false;             // Wiii Classic digital pad buttons
bool BDU = false;             // Wiii Classic digital pad buttons
bool BDU_ = false;            // Wiii Classic digital pad, button down flags
bool BDR_ = false;            // Wiii Classic digital pad, button down flags
bool BDLst = false;           // Wiii Classic digital pad button states
bool BDRst = false;           // Wiii Classic digital pad button states
bool BHm = false;             // Wii Classic Select, Home and Start buttons
bool Blink = false;           // LED blink clock
int16_t BlinkCnt = 0;         // counter for LED Blink clock
bool BLT = false;             // Wii Classic front L button states
bool BLOCKED = false;         // if == true then block functions
bool Boot = true;             // if == true then currently in Boot phase
bool Border = false;          // if true then display has a border
bool BotActive = false;       // == true if active, otherwise false
uint8_t Bright = 255;         // FastLED brightness
uint8_t Brightness = 255;     // Display brightness
bool BRT = false;             // Wii Classic front R button states
bool BSe = false;             // Wii Classic Select, Home and Start buttons
bool BSt = false;             // Wii Classic Select, Home and Start buttons
bool BX,BY,Ba,Bb;             // Wii Classic right-hand buttons, Y,X,b,a
bool BXDwn = false;           // Wii Classic right-hand X button states
bool BYDwn = false;           // Wii Classic right-hand Y button states
bool BaDwn = false;           // Wii Classic right-hand A button states
bool BbDwn;                   // Wii Classic right-hand B button states
bool BZL = false;             // Wii Classic front ZL button states
bool BZR = false;             // Wii Classic front ZR button states
int16_t C_Cnt = 0;            // counter used in 'C' button detection
byte C_Dn = 0;                // counter for 'C' button down period
byte C_Up = 0;                // counter for 'C' button up period
char cmdMode = ' ';           // command mode
int16_t cmdSgn = 1;           // cmdVal sign, 0 or -1
char cmdType = ' ';           // command mode type
int32_t cmdVal = 0;           // value associated with a cmdType
bool Cnt8ms = false;          // 8ms counter flag
int16_t Cnt16ms = 4;          // counts 4ms down to 16ms for task scheduler
int16_t Cnt20ms = 0;          // counts 4ms down to 20ms for task scheduler
int16_t Cnt40ms = 1;          // counts 4ms down to 40ms for task scheduler
byte Col_B = 0;               // current colours
byte Col_G = 0;               // current colours
byte Col_R = 0;               // current colours
int16_t CZ = 0;               // received button values
bool CZ_Wait = false;         // flag used for button release wait function
char Deg = 176;               // ASCII for "°"
int16_t DIR_FL = 1;           // direction sign, +ve forward, -ve backwards
int16_t DIR_FR = 1;           // direction sign, +ve forward, -ve backwards
int16_t DIR_RR = 1;           // direction sign, +ve forward, -ve backwards
bool DispBack = false;        // flag used to draw a back arrow on the display
bool DispChng = false;        // flag invokes initialisation code
bool DispClr = false;         // set == true to clear the memory contents
int16_t DispCnt = 0;          // counter used in display updating
int16_t DispCX = 0;           // display cursor X coordinate
int16_t DispCY = 0;           // display cursor Y coordinate
int16_t DispDel = 0;          // >0 display delay counter, stops updates over I2C
int16_t DispDelGbl = 0;       // global DispDel offset, used to adjust display frame rates
bool DispDisp = false;        // set = true to immediately display memory contents
bool DispEn = true;           // if == true then generate display messages, otherwise inhibit
int16_t DispLast = 0;         // previous DispMode value
bool DispLock = false;        // if == true then don't allow display mode changes
int16_t DispH = 48;           // display rectangle height
byte DispMenu = 0;            // group type, 0 = sensors, 1 = tasks, etc
int16_t DispMode = 0;         // the type of display for a given mode
int16_t DispModeCnt = 0;      // counter used in display updating
bool DispMon = false;         // == true if connected to OLED Mirror app
int16_t DispNext = 0;         // next value of DispMode
bool DispNOP = false;         // set == true to stop display I2C updates
int16_t DispNow = 0;          // if > 0 display special modes
int16_t DispNxtCnt = 0;       // counter used in auto-branching of display
int16_t DispOveride = 0;      // if DispLock == true then use this overide mode
int16_t DispPnt = 0;          // points at the current display mode
int16_t DispTask = 0;         // task pointer for display functions
int16_t DispTx = 0;           // counter used with display mirror app
int16_t DispVal = 0;          // general display value, passed from tasks
int16_t DispW = 64;           // display rectangle width
int16_t DM_Auton;             // Auton display assigned Display Mode
int16_t DmB = 0;              // screen mouse button associated with a display monitor mouse click
int16_t DmX = 0;              // screen mouse button X values associated with a display monitor mouse click
int16_t DmY = 0;              // screen mouse button Y values associated with a display monitor mouse click
int16_t DM_Batt;              // Battery display assigned Display Mode
int16_t DM_BatPlus;           // Battery plus display assigned Display Mode
int16_t DM_Control;           // Control display assigned to Display Mode
bool DmDwn = false;           // == true whilst mouse is clicked in Monitor+ mode
int16_t DM_ICM;               // ICM sensor display assigned Display Mode
int16_t DM_ICM_Mov;           // ICM motion display assigned Display Mode
int16_t DM_ICM_Rot;           // ICM rotation display assigned Display Mode
int16_t DM_JoyNorm;           // Joystick normal values display assigned Display Mode
int16_t DM_JoyOff;            // Joystick values and offsets display assigned Display Mode
int16_t DM_LDR;               // LDR sensor display assigned Display Mode
int16_t DM_LED;               // LED modes display assigned Display Mode
int16_t DM_Min,DM_Max;        // min/max Display Mode pointer values
int16_t DM_MtrPWM;            // motor PWM display assigned Display Mode
int16_t DM_MoveEng;           // Move Engine display assigned Display Mode
int16_t DM_Range;             // ranges display assigned Display Mode
int16_t DM_Recall;            // RECall display assigned Display Mode
int16_t DM_Tacho;             // Tachos display assigned Display Mode
int16_t DM_Tune;              // test display assigned Display Mode
int16_t DM_Time;              // timing display assigned Display Mode
bool DmSft = false;           // == true if SHIFT key held down when mouse clicked in Monitor+
int16_t DM_Wii_Ctrl;          // Wii controller display aassigned Display Mode
int16_t Drive = 0;            // if > 0 then driving in various directions
int16_t DriveBck = 0;         // rear PWM value
int16_t DriveLast = 0;        // previous driving direction
int16_t DriveFL = 0;          // front left-side PWM value
int16_t DriveFR = 0;          // front right-side PWM value
int16_t DriveLft = 0;         // left-side PWM value
int16_t DrivePWM = 0;         // PWM drive value for all 4 wheels
int16_t DriveRht = 0;         // right-side PWM value
int16_t DriveRL = 0;          // rear left-side PWM value
int16_t DriveRR = 0;          // rear right-side PWM value
int16_t DriveTmp = 0;         // temp value used in drive modes
String DriveTxt = "";         // indicates drive mode
bool Echo = true;             // if true send print messages
int16_t EchoPin;              // Pointer to echo pin in use
int16_t EndTask = 0;          // points to the last task in the switch() schedule
bool ESC = false;             // true to escape all functions
int16_t ESP_NOW_BA = -1;      // broadcast address pointer, -1 to start with
bool ESP_NOW_Init = false;    // == true once ESP-NOW has been initiaiised
int16_t GAScnt = 0;           // counter used in getAtState messages
int16_t Gear = 1;             // value used to determine PWM Max values
int16_t GearMax = 2;          // restricts Gear selection, default = 2, max = 5
bool G_once = true;           // set == true to force initial graphing mode, default = false
int16_t GPmax = 0;            // end of graphing list, for Scope mode
int16_t GPmin = 0;            // start of graphing list, for Scope mode
int16_t GPnn = 0;             // pointer to current graphing items
bool GP_Title = true;         // if == true then report a title in Scope graphing mode
int gyro_Z_data_raw;          // raw Z gyro value read from MPU6050
byte I2C_Err;                 // flag returned from I2C comms
bool I2C_OLED = false;        // true if display is detected during POST
bool I2C_ICM = false;         // true if ICM-42607-P is detected during POST
bool ICM_En = true;           // if == true then ICM I2C is enabled, default = true
int16_t Inc = -1;             // increment value
bool JoyActive = true;        // flag used to prevent stack overflow during moves
int16_t JoyAng;               // left joystick angle
int16_t JoyCnt = 0;           // controller glitch filter
int16_t JoyD = 0;             // direction value 0 - 8
int16_t JoyDL = 0;            // direction 1 counter
int16_t JoyLX = 32;           // Wii Classic left joystick values
int16_t JoyLY = 32;           // Wii Classic left joystick values
int16_t JoyMode = 0;          // direction of travel flag, 0 = stationary, 1 = forward, -1 = backward
int16_t JoySpd = 0;           // speed vector = max(JoyX,JoyY)
int16_t JoyV;                 // joystick vector  = SQRT(SQ(JX) + SQ(JY))
int16_t JoyX = 0;             // received value
int JoyXV = 0;                // converted value 0 - 127
int16_t JoyXC;                // received value offset
int16_t JoyY;                 // received value
int16_t JoyYC;                // received value offset
int JoyYi = 0;                // tracking filter for received JoyY value
int JoyYV = 0;                // converted value 0 - 128
boolean JoyYT;                // tracking filter flag; true if not on track
int16_t JX = 0;               // global variable used with Joystick app
int16_t JY = 0;               // global variable used with Joystick app
char keyChar;                 // any keyboard character
int16_t keyVal;               // any keyboard value
int16_t LDR0 = 0;             // ADC value for LDR0_Pin
int32_t LDR0_Avg = 0;         // averaged ADC value for LDR0_Pin
int32_t LDR0_Sum = 0;         // accumulator for averaging LDR0
int32_t LDR1 = 0;             // ADC value for LDR1_Pin
int32_t LDR1_Avg = 0;         // averaged ADC value for LDR1_Pin
int32_t LDR1_Sum = 0;         // accumulator for averaging LDR1
int32_t LDR_Div = 20;         // LDR averaging coefficient 1 - 100
bool LDR_En = true;           // if == true then LDR reading takes place, default = true
int16_t LDR_Ga = 70;          // LDR gain factor
int16_t LDR_Task = 0;         // 10ms task pointer for reading light sensors
int16_t LED_Cnt = 0;          // counter used in LED sequencing
int16_t LED_Del = 0;          // delay timer used in LED functions
int16_t LED_Max = 0;          // max value used in LED functions
bool LedMem = false;          // if == true then values have been saved to Mem[]
int16_t LedMode = 0;          // a value > 0 determines LED flash sequence; if = 0 disabled
int16_t LedNext = 0;          // next LED routine pionter
bool LedNop = false;          // if true then don't update LEDs
uint32_t LED_Period = 20;     // LED timer period, default = 20ms
bool LED_Run = false;         // == true if LED_Task is to run
bool LEDshow = false;         // LED show flag
int16_t LedSpd = PWM_StartMin;  // motor speed value used by LED animation
int16_t LED_Task = 0;         // LED task pointer
uint32_t LED_Timer = 0;       // 20+ms LED timer
byte LEDVal = 0;              // values used in LED tasks
uint32_t loopDelay;           // RTOS task timer
int16_t MainDel = 0;          // if > 0 this is the MainTask delay timer
int16_t MainCnt = 0;          // counter used in MainTasks
int16_t MainMode = 0;         // main state, 0 = sleep
bool MainPWM = false;         // MainTask flag indicates PWM drive has been applied
bool MainRun = false;         // set true when a MainTask is running to prevent re-entry
int16_t MainSub = 0;          // MasinTask sub-task pointer
int16_t MainTask = 0;         // main task pointer
int16_t MainTaskNext = 0;     // the task to run after an arrest; default is 0
int16_t MainTime = 0;         // Timer/counter used in tasks
int16_t MotorPnt = 0;         // motor pointers for app adjustment
int16_t MoveCnt = 0;          // number of movements loaded
uint16_t MoveCtrl[MoveNum];   // movement control
int16_t MoveFL[MoveNum];      // front left motor movements
int16_t MoveFR[MoveNum];      // front right motor movements
int16_t MovePnt = -1;         // pointer to moves in the arrays, default -1
int16_t MoveRC[MoveNum];      // front rear centre motor movements
int16_t MoveState = 0;        // move state machine pointer
uint16_t MoveStp = 0;         // 16s step down counter
uint16_t MoveTime[MoveNum];   // movement time, in units of 16ms
String myMAC;                 // MAC address if this device
int16_t NewMode = 0;          // normally = MainMode, but can be set to a run mode
int16_t NewTask = 0;          // normally MainTask, but can be set to a run task
uint32_t next4ms;             // 4ms loop timer
uint32_t next16ms;            // 16.666ms loop timer
uint32_t next20ms;            // 20ms loop timer
uint32_t next40ms;            // 40ms loop timer
int16_t OLED_Del = 0;         // >0 OLED display delay counter, stops updates over I2C
int16_t OLED_Mode = 0;        // OLED display pointer
bool OLED_NOP = false;        // set == true to stop OLED display I2C updates
bool Omni = false;            // == true when left joystick forces Omni mode
uint32_t period16ms = 16;     // a period which varies between 16-17ms to acjhieve 60Hz
uint8_t phase16ms = 0;        // a 3-phase pointer which adjusts the 16ms value to get 16.6666
int16_t Ping = 0;             // Rx 'ping' counter
bool PLAY = false;            // set == true to play a recorded controller sequence
int16_t PLAYinc = 0;          // PLAY time tracker
int16_t PLAYpnt = 0;          // PLAY step pointer
bool PLAYstart = true;        // if == true then waiting to start PLAY mode
int32_t PLAYtime = 0;         // PLAY step time counter
int32_t PLAYtotal = 0;        // PLAY total time
int16_t Pnt40ms = 1;          // counts 4ms down to 20ms (variable) for print scheduler
bool Print40ms = false;       // if == true then call the print scheduler
int16_t PrintTgt = 0;         // 0 == serial port, 1 == WiFi ESP-NOW
String PrintTx = "";          // printed strings
uint32_t PulseWidth;          // width of RCWL pulse in microseconds
int16_t PWM_A = 0;            // PWM drive for motor A
int16_t PWM_A0 = 0;           // PWM value written to H-bridge driver
int16_t PWM_A1 = 0;           // PWM value written to H-bridge driver
int16_t PWM_A_ = 0;           // PWM off state value, 0 = 00 or 1 = FF
int16_t PWM_B = 0;            // PWM drive for motor B
int16_t PWM_B0 = 0;           // PWM value written to H-bridge driver
int16_t PWM_B1 = 0;           // PWM value written to H-bridge driver
int16_t PWM_B_ = 0;           // PWM off state value, 0 = 00 or 1 = FF
int16_t PWM_C = 0;            // PWM drive for motor C
int16_t PWM_C0 = 0;           // PWM value written to H-bridge driver
int16_t PWM_C1 = 0;           // PWM value written to H-bridge driver
int16_t PWM_C_ = 0;           // PWM off state value, 0 = 00 or 1 = FF
bool PwmEn = false;           // if true motor PWM is enabled
int16_t PWM_Max = 255;        // max PWM value, constrained by V_max
int16_t PWM_Start = 0;        // value needed to ensure motor start
int32_t Range[3];             // RCWL range values
int32_t RangeAvg[3];          // averaged range values
int16_t RangeCnt[3];          // out of range discriminator count
int32_t RangeDiv = 3;         // value used in averaging ranges
int16_t RangeEx = 5;          // out of range discriminator limit
int32_t RangeOld[3];          // previous range values
int16_t RangePnt = 0;         // pointer to Range[] variable
int32_t RangeSum[3];          // accumulated range values, for averaging
int32_t RangeVal[3];          // RCWL range values, used by other functions
int16_t RCWL = 0;             // phase pointer for RCWL measurements
int16_t RCWL_Cnt = 16;        // value used as 4ms count-down, sets RCWL cycle rate
bool RCWL_En = false;         // if == true then HCWL acoustic ranging is enabled
uint8_t RCWL_Mask = 0;        // bity mask representing enabled RCWL devices
int16_t RCWL_Max = 12;        // value used to load 4ms count-down, sets RCWL cycle rate
int16_t RCWL_Mode = 0;        // Value configures order in which sensors are read
bool readWiiCall = false;     // flag prevents call stack overflow
bool REC = false;             // RECording flag, default = false
byte RECmem0[RECnum];         // RECord array element 0
byte RECmem1[RECnum];         // RECord array element 1
byte RECmem2[RECnum];         // RECord array element 2
byte RECmem3[RECnum];         // RECord array element 3
byte RECmem4[RECnum];         // RECord array element 4
byte RECmem5[RECnum];         // RECord array element 5
int16_t RECpnt = 0;           // RECord array pointer
bool RECstart = false;        // if == true then waiting to start RECord mode
uint32_t RECtime[RECnum];     // RECord time counters
int32_t RECtotal = 0;         // RECord total time
int16_t Rng0 = 0;             // range limits used in tasks
int16_t Rng1 = 0;             // range limits used in tasks
int16_t Rng2 = 0;             // range limits used in tasks
int16_t Rng3 = 0;             // range limits used in tasks
int16_t Rng4 = 0;             // range limits used in tasks
int16_t Rng5 = 0;             // range limits used in tasks
int16_t Rng6 = 0;             // range limits used in tasks
int16_t Rng7 = 0;             // range limits used in tasks
int16_t Rx_Chk = 0;           // Rx checksum
int16_t RxCZ = 0;             // buffered received CZ value
int16_t RxJoyX = 0;           // buffered received JoyX value
int16_t RxJoyY = 0;           // buffered received JoyY value
int16_t Rx_len = 0;           // length of WiFi received data block
uint32_t RxMs = 0;            // timer for Rx period
uint32_t RxPeriod = 40;       // time between Rx Wii data blocks
int16_t Rx_Pnt = 0;           // data pointer user by Rx state machine
bool RxRec = false;           // set true when a valid frame of data is received
bool RxRecvEvent = false;     // set == true when a OnDataRecv() call back event has occured
int16_t RxState = 0;          // receiver state machine state
int16_t Rx_Task = 0;          // task pointer user by Rx state machine
int16_t RTxTimeout = 0;       // counter used to auto-reset if WiFi fails
int16_t RxVal = -1;           // value received from serial Rx
byte RxWiFi[6];               // array holding 6 bytes of Wii received data
bool SerialRx = false;        // == true if any character has been received
uint8_t Snore = 3;            // snore animation character
int16_t SpdAdj = 0;           // Speed adjustment factor used in omi steering
String Status$ = "";          // any status message to be displayed on OLED
int16_t Steer = 0;            // value applied when steering left/right
int16_t SteerAng = 0;         // steer angle at which minimum range was measured
bool StopDet = false;         // edge/wall stop flag used in autonomouse mode
int16_t SubCnt = 0;           // sub task general counter
int16_t SubDel = 0;           // subtask delay counter
int16_t SubNext = 0;          // pointer to next sub task
int16_t SubRpt = 0;           // subroutine repeat counter
int16_t SubTask = 0;          // sub task pointer
int16_t SubTask1 = 0;         // level 1 sub task pointer
int16_t SubVal = 0;           // anyvalue used in a subroutine
int16_t SubWait = 0;          // SubTask wait period
int16_t SubWaitNext = 0;      // SubTask to goto after wait period
int16_t sw0Cnt = 0;           // button switch counter
int16_t sw0DwnTime = 0;       // button pressed down time
bool sw0LastState = HIGH;     // previous state of button switch, HIGH/LOW
bool sw0_Nop = false;         // if == true don't perform switch functions
bool sw0State = HIGH;         // state of read button switch pin
int16_t sw0Timer = 0;         // timer used to detemine button sequences
bool sw0Wup = false;          // set == true for SW0 to wait for button release
int16_t sw1Cnt = 0;           // button switch counter
int16_t sw1DwnTime = 0;       // button pressed down time
bool sw1LastState = HIGH;     // previous state of button switch, HIGH/LOW
bool sw1_Nop = false;         // if == true don't perform switch functions
bool sw1State = HIGH;         // state of read button switch pin
int16_t sw1Timer = 0;         // timer used to detemine button sequences
bool sw1Wup = false;          // set == true for SW1 to wait for button release
byte t;                       // any timer pointer
uint32_t t0[4];               // any function timer
uint32_t t1[4];               // any function timer
uint32_t t2[4];               // any function timer
int32_t TachoA = 0;           // motor tracho counter, for motor 'A'
int32_t TachoB = 0;           // motor tracho counter, for motor 'B'
int32_t TachoC = 0;           // motor tracho counter, for motor 'C'
int16_t Task4ms = 0;          // 4ms task scheduler pointer
String TaskMsg = "";          // Any task message
bool TEST = false;            // if == true then in limited TEST mode
int16_t TEST_PWM = 0;         // +/- PWM value used during TEST mode
uint32_t tMax = 0;            // tracks the maximum period spent performing tasks
uint32_t tRef = 0;            // 4ms loop zero time reference
bool Trig16ms = 0;            // if == true then run a designated 16ms tasks
bool Trig20ms = 0;            // if == true then run a designated 20ms tasks
bool Trig40ms = 0;            // if == true then run a designated 40ms tasks
bool TrigICM = false;         // if == true then run the ICM sensor read every 8ms
bool TrigOLED = false;        // if == true then run OLED display updates every 40ms
int16_t TrigPin = Trig0;      // pointer to pin used by distance sensors
uint32_t tSch = 0;            // time taken to complete the task schedule
float TuneFpt[4];             // temp tune values
int16_t TuneInt[4];           // temp tune values
String TuneRef[4];            // temp tune reference
bool Turn = false;            // default = false; if true turns continuously until = false
bool Turning = false;         // set == true so that ESC can be used to break out of a turn cycle
int16_t Tx_PingCnt = 0;       // 1 second Tx ping timer
bool USB = false;             // == true if initial battery is below 5.5v
float V_max;                  // calculated voltage limit for motor PWM

int16_t WiFiCntC = 0;         // C button counter for WiFi enable
int16_t WiFiConCnt = 0;       // counter used in connection retry times
bool WiFiConnected = false;   // == true if a WiFi connection has been established
bool WiFiEn = false;          // true when WiFi is enabled with prolonged 'C' button activity
int16_t WiFiPing = 0;         // Rx counter used to maintain connected state
int16_t WiFiRestCnt;          // counter used to go to rest if no WiFi demands
bool WiFiRx = false;          // == true when a block of data has been received
long WiFiRxBytes = 0;         // number of bytes received over the WiFi link
int16_t WiFiRxErr = 0;        // number of bytes received over the WiFi link
bool WiFiRxRec = false;       // true when a block of data has been successfully received
byte WiFiRxMacAddr[6];        // senders MAC address, which could be broadcast FF:FF:FF:FF:FF:FF
int16_t WiFiTryCnt = 0;       // WiFi try to connect count on error count
int16_t WiFiTryNum = 0;       // WiFi try to connect total count
bool WiFiTryOnce = true;      // == true if first pass of trying to connect
bool WiFiTx = false;          // == true when a block of data has been sent successfully
int16_t WiFiTx_CB = 1;        // == 1 call back is clear to send, == -1 resend
int16_t WiFiTx_Chk;           // Tx checksum
int16_t WiFiTx_len = 0;       // >0 when WiFi Tx buffer contains data to be sent
long WiFiTxBytes = 0;         // number of bytes received over the WiFi link
int16_t WiFiTxErr = 0;        // WiFi data error
int16_t WiFiTxErrCnt = 0;     // WiFi Tx error count, leads to disconnection
byte WiFiTxMacAddr[6];        // transmit MAC address, which should match myMAC address
int16_t WiiType = 0;          // device type, 0 == Nunchuk, 1 == Classic
byte Z_Dn = 0;                // button pressed state; 0 == UP, >= 1 == Down
bool Z_Mode = false;          // joystick motion flag, normally false, if == true then slide mode

// ---------------------------------------------------------------------------------

void setup() {
  // Setup code, runs once:
  // Initialise PWM pins and output states
  pinMode(Pin_A0, OUTPUT); digitalWrite(Pin_A0,LOW);
  pinMode(Pin_A1, OUTPUT); digitalWrite(Pin_A1,LOW);
  pinMode(Pin_B0, OUTPUT); digitalWrite(Pin_B0,LOW);
  pinMode(Pin_B1, OUTPUT); digitalWrite(Pin_B1,LOW);
  pinMode(Pin_C0, OUTPUT); digitalWrite(Pin_C0,LOW);
  pinMode(Pin_C1, OUTPUT); digitalWrite(Pin_C1,LOW);

  // initialise other pins
  pinMode(BatPin,INPUT);          // ADC1_0  GPIO36 battery divider
  pinMode(Echo0, INPUT_PULLUP);   // echo pin for front sensor
  pinMode(Echo1, INPUT_PULLUP);   // echo pin for right-hand sensor
  pinMode(Echo2, INPUT_PULLUP);   // echo pin for left-hand sensor
  pinMode(sw0Pin,INPUT_PULLUP);   // button switch sw0 Pin
  pinMode(sw1Pin,INPUT_PULLUP);   // button switch sw1 Pin
  pinMode(Trig0, OUTPUT); digitalWrite(Trig0,LOW);
  pinMode(Trig1, OUTPUT); digitalWrite(Trig1,LOW);
  pinMode(Trig2, OUTPUT); digitalWrite(Trig2,LOW);

  getDispRef();                   // read references, without displaying screens

  Serial.begin(115200);
  delay(100);

  I2C_Err = Wire.begin();   // Start the I2C bus as master
  if (!I2C_Err) {Serial.println("I2C Wire.begin() Error!");} else {Serial.println("I2C started");}

  // initialise the FastLED component
  FastLED.addLeds<WS2812B, LED_Pin, GRB>(LED, NumLEDs);
  FastLED.setBrightness(Bright);
  FastLED.clear();
  FastLED.show();

  // initialise the six PWM channels, setting each to max
  for (int zP = 0;zP < 8;zP++) {
    PwmSetup(zP);
    PwmAttach(zP);
  }

  // initialise WiFi link
  WiFi.mode(WIFI_STA);
  myMAC = WiFi.macAddress();
  Serial.println("my MAC= " + String(myMAC));
  // WiFi.disconnect(); // I don't think we need this

  // initialise the ESP-NOW link for the first time
  Init_ESP_NOW();

  RunPOST();  // initialise I2C devices, set intro display, etc

  // define the Core 0 task parameters
  // Core 0 is normally used for WiFi, but here we use it to read sensors
  // leaving Core 1 to create the LED patterns and clock them out
  // Note, this code starts and runs immediately, before loop() does
  xTaskCreatePinnedToCore(
    loop0,       // Task function.
    "Core0",     // name of task.
    10000,       // Stack size of task
    NULL,        // parameter of the task
    1,           // priority of the task
    &Core0,      // Task handle to keep track of created task
    0);          // pin task to core 0

  Boot = false;
  synchLoopTimers();
}

// ---------------------------------------------------------------------------------

void loop() {
  // Main loop code, runs repeatedly.
  // A 4ms task scheduler controls all major activities
  // t1[0] = micros(); // end time
  // t0[1] = micros(); // start time

  //###############################################################################
  //
  //  WiFi comms check & respond
  //
  //###############################################################################
  // if WiFi is conencted then check Rx buffer on every cycle, so that we respond to
  // received packets as soon as possible
  if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events

  readSerial();                             // check serial port on every loop cycle
  
  //###############################################################################
  //
  //  4ms Task scheduler (250Hz)
  //
  //###############################################################################
  // These tasks run as a cascade, starting with Task4ms == 1, up to the end of the
  // switch() list. We start by servicing the ICM-42607-P functions, which we want
  // to run at a rate pf 125Hz. Other tasks run at a slower rate, depending on flags
  // set when the schedule reloads, every 4ms.
  if (Task4ms > 0) {
    switch (Task4ms) {
      case  1: if (TrigICM) {readICM();} break;                       // read IC registers
      case  2: if (TrigICM) {AvgICM();} break;                        // average yaw sensor values
      case  3: if (TrigICM) {calcYawAng();} break;                    // calculate yaw, pitch and turning
      case  4: if (TrigICM) {calcSpeed();} break;                     // average sensor Gyro values
      case  5: if (TrigOLED) {OLED_40ms();} break;

      case  6: RGB_RunTask(); break;                                  // update LEDs

      case  7: if (Trig16ms) {runMainTasks();} break;                 // perform MainTasks 
      case  8: if (Trig16ms) {readSW0();} break;                      // read button SW0  
      case  9: if (Trig16ms) {readSW1();} break;                      // read button SW1 
      case 10: if (Trig16ms) {read_Battery();} break;                 // read battery voltage
      case 11: if (Trig16ms && (MovePnt >= 0)) {MoveEngine();} break; // run the MoveEngine
      case 12: if (Trig16ms) {MotorDriveTask();} break;               // update motor PWM values

      case 13: if (Trig20ms) {CheckWiFiConnected();} break;
      
      case 14: if (Trig40ms) {if (DispMon) {DispMon_40ms();}} break;
      case 15: if (Trig40ms) {CheckDispTx();} break;

      // This should be the last task in the list
      case 16: if (Print40ms) {PrintTxHandler();} break;
    }
    // Increment task pointer up to the end
    Task4ms++;
    if (Task4ms > EndTask) {
      Task4ms = 0;                        // stop the switch() scheduler from running
      tSch = micros() - tRef;             // time spent running the scheduled tasks
      if (tSch > tMax) {tMax = tSch;}
      // Serial.println("tSch " + String(tSch));
    }
  } else {
    // This is where we are waiting for the next 4ms time event.
    // So we keep checking the 4ms timer.
    if ((millis() - next4ms) >= 4) {
      //####################################################################
      // Master scheduler config
      //####################################################################
      // Reset 4ms timer and adjust task pointers.
      // Note that tRef acts as the zero time reference for the 4ms period
      tRef = micros(); next4ms = millis(); Task4ms= 1;
      // Toggle the TrigICM flag, so that it runs every 8ms 125Hz.
      // It is synchronised to an 8ms counter.
      if (Cnt8ms) {Cnt8ms = false;} else {Cnt8ms = true;}
      if (I2C_ICM && ICM_En) {TrigICM = Cnt8ms;} else {TrigICM = false;}    // set 8ms task trigger for ICM
      // Clear the task run flags
      Trig16ms = false; Trig20ms = false; Trig40ms = false; TrigOLED = false; Print40ms = false;
      // Here we set task flags depending on their timers.
      Cnt16ms--; if (Cnt16ms < 1) {Cnt16ms =  4; Trig16ms = true;}  // set 16ms task trigger
      Cnt20ms--; if (Cnt20ms < 1) {Cnt20ms =  5; Trig20ms = true;}  // set 20ms task trigger
      Cnt40ms--; if (Cnt40ms < 1) {Cnt40ms = 10; Trig40ms = true;}  // set 40ms task trigger
      if (Cnt40ms == 8) {
        // Set a trigger for the OLED display, in phase with the ICM
        if (!TrigICM) {Cnt40ms--;}               // add a phase shift to Cnt40ms
        else if (I2C_OLED) {TrigOLED = true;}   // set 40ms OLED task trigger
      }
      RCWL_Cnt--; if (RCWL_Cnt < 1) {RCWL_Cnt = RCWL_Max; if (RCWL_En) {RCWL = 1;}}    // set RCWL task trigger
      Pnt40ms--; if (Pnt40ms < 1) {Pnt40ms = 10; Print40ms = true;}                   // set PrintTx task trigger
      // *** IMPORTANT *** - set the last task in the scheduled switch() list
      EndTask = 16;
      // Time-out any connected app
      if (AppCnt > 0) {AppCnt--; if (AppCnt == 0) {PWM_OFF(); OLED_Text2S1610("App","Time-out"); OLED_Mode = 0;}}
    } else {
      //####################################################################
      // 'Slack' space'
      //####################################################################
      // This time period is considered as 4ms 'slack space' or free time.
      // We are here because all of the scheduled tasks have been completed.
      // Once connected to WiFi, look for a WiFiRx flag, and respond.
      if (WiFiConnected) {
        if (WiFiRx) {readWiFiRx();} // WiFi Rx buffer contains unread data so deal with it before adding new
      }
      // We look at RCWL counts during this slack space. They are critical in 
      // timing the RCWL trigger and read process
      if (RCWL > 0) {RCWL_Tasks();}
    }
  }
}

// ---------------------------------------------------------------------------------

void loop0(void * pvParameters) {
  // this function runs on Core 0, also used for ESP32 WiFi NOW
  // *** WARNING *** do not use for time consuming code like display, I2C, FastLED, etc
  // the following message, sent once at boot, confirms core 0 assignment
  Serial.print("Core0 task running on core "); Serial.println(xPortGetCoreID());

  for(;;){
    // infinite loop, just like loop() but running in Core 0
    // you must never use 'break' or 'return' statement here or it will crash
    // we tell the free RTOS system to call this function every 1 ms
  
    // t1[2] = micros(); // end time
    // t0[3] = micros(); // start time
    //###################################################################
    // LDR tasks
    //###################################################################
    // Here we read an LDR pin, then switch the mux to the next pin ready for
    // reading on the next cycle. Both sensors are read at 100 Hz, 5ms apart.
    int16_t zV = 0;   // dummy read variable
    if (LDR_En) {
      switch (LDR_Task) {
        case 0: // read LDR0_Pin
          LDR0 = 4095 - analogRead(LDR0_Pin); zV = analogRead(LDR1_Pin); break;
        case 1: // calculate rolling average
          LDR0_Avg = LDR0_Sum/LDR_Div; LDR0_Sum = LDR0_Sum + LDR0 - LDR0_Avg; break;
        case 5: // filter readings
          LDR1 = 4095 - analogRead(LDR1_Pin); zV = analogRead(LDR0_Pin); break;
        case 6: // calculate rolling average
          LDR1_Avg = LDR1_Sum/LDR_Div; LDR1_Sum = LDR1_Sum + LDR1 - LDR1_Avg; break;
      }
      LDR_Task++; if (LDR_Task > 9) {LDR_Task = 0;}
    } else {LDR_Task = 0;}

    //###################################################################
    // 20ms tasks
    //###################################################################
    // These are short tasks, triggered from the Core 1 main loop() every 20ms.
    // The Core 1 flag stays active for 4ms, that's long enough to be seen here.
    // However, the BLOCKED flag can stop these tasks.
    if (Trig20ms && !BLOCKED) {
      CheckWiFiFailed();
    }

    //###################################################################
    // 40ms tasks
    //###################################################################
    // These are short tasks, triggered from the Core 1 main loop() every 40ms
    // The Core 1 flag stays active for 4ms, that's long enough to be seen here.
    // However, the BLOCKED flag can stop these tasks.
    if (Trig40ms && !BLOCKED) {
      CheckWiFiPing();
      CheckWiFiReconnect();
      if (Ping > 0) {Ping--;}
    }
    // t1[3] = micros(); // end time
    // t0[2] = micros(); // start time

    // Return to RTOS system, with a 1ms call back, allowing WiFi tasks to run
    vTaskDelay(1/portTICK_PERIOD_MS);
  }
}

// --------------------------------------------------------------------------------

void OnDataRecv(const esp_now_recv_info_t *info, const uint8_t *incomingData, int len) {
  // Call-back function oocurs when data has been received
  // This needs to be treated like an interrupt, so speed is the essence here
  // memcpy moves data in a block, instead of individual bytes if using for() loop
  memcpy(WiFiRxMacAddr, info->src_addr, 6);   // get the senders MAC/broadcast address
  memcpy(WiFiTxMacAddr, info->des_addr, 6);   // get destination MAC address, which should be assigned peer
  memcpy(&Rx_Buff, incomingData, len);        // copy the received data into Rx_Buff array
  Rx_len = len - 1; Rx_Pnt = 0;               // len contain the length of data
  RxRecvEvent = true;                         // set the flag to say data received
}

// --------------------------------------------------------------------------------

void OnDataRecvHandler() {
  // Called from the main loop() when RxRecvEvent has been set, to handle the received data
  RxRecvEvent = false;
  // incoming data will always be two or more characters, ie. text == $+++
  if (Rx_len < 2) {Serial.println("Rx Error - no Rx data"); return;}

  RxGetChecksum(Rx_len);  // generate a checksum from Rx data
  if (Rx_Chk != Rx_Buff.ESPdata[Rx_len]) {
    Serial.println("Rx Error - checksum");
  }
  // set flags before exiting
  WiFiRxBytes += Rx_len; Rx_Task = 0; WiFiRx = true;
  WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
  WiFiRxRec = true; // set flag indicating data has been recieved from Wii Transceiver
  //  Serial.print("Bytes received: "); Serial.println(Rx_len);
}

// --------------------------------------------------------------------------------

void OnDataSent(const uint8_t *mac_addr, esp_now_send_status_t status) {
  // Callback when data is sent over Wi-Fi, determines success or failure
  // t1 = micros();
  // Serial.print("Send Status: ");
  if (WiFiConnected) {
    // the link is already active so handle normal success/failure scenarios
    if (status == 0){
      // Serial.println("Success " + String(t1-t0));
      WiFiTx = true;    // flag WiFi Tx success
      WiFiTx_CB = 1;    // set flag to indicate that a call back state has occured
      WiFiTxErrCnt = 0; // clear the error flag in case of a glitch
      WiFiPing = 50;    // set link time-out to 2 seconds
    } else {
      // Serial.println("Fail " + String(t1 - t0));
      WiFiTx = false;   // flag WiFi Tx failure
      WiFiTx_CB = -1;   // set flag to indicate that a resend is required
      WiFiTxErrCnt++;
    }
  } else {
    // not connected so we must have tried to connect and may get an error
    if (status == 0){
      // successful so enter the connected state
      // Serial.println("Connected!");
      WiFiConnected = true;
      WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
      WiFiTxErrCnt = 0; // reset error count for this connection
    } else {
      // Serial.println("Status = " + String(status));
      WiFiTxErrCnt++;
    }
    WiFiTx_CB = 1;
  }
}

// --------------------------------------------------------------------------------

void PrintTxHandler() {
  // This function handles the contents of the PrintTx string, which is used
  // to send data to either the serial port or WiFi system.
  // It is called at a 40ms rate, but speeds up when more data is to be sent.
  // It is called as a function, so that it can be called during battery failure.
  if (!Echo) {PrintTx = "";}   // Echo is false so empty print buffer
  if (PrintTx.length() > 0) {
    // characters in string buffer so send some/all of them
    Pnt40ms = 2;          // shorten the delay to 8ms for more data to send
    Tx_PingCnt = 0;       // reset the WiFi ping counter
    // PrintTgt = 0;      // force printing to serial port
    if (PrintTgt == 0) {
      // default uses the serial port
      if (Serial.availableForWrite() >= 64) {
        if (PrintTx.length() <= 64) {
          Serial.print(PrintTx);
          // PrintTx = "";   // empty the buffer
          PrintTx.clear();   // empty the buffer
        } else {
          Serial.print(PrintTx.substring(0,64));
          PrintTx = PrintTx.substring(64);
        }
      }
    } else {
      // text data has been received previously from WiFi, so respond using ESP_NOW
      // ensure that we have had a call back from the previous frame
      if (WiFiConnected) {
        // only trannsmit PrintTx data if connected
        if (WiFiTx_CB == 1) {
          WiFiTx_CB = 0;  // clear the call back flag when sending
          if (PrintTx.length() <= 32) {
            WiFiTx_len = PrintTx.length();
            PrintTx.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
            WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
            WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
          //  t0 = micros();
            esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
            // PrintTx = "";   // empty the buffer
            PrintTx.clear();   // empty the buffer
          } else {
            WiFiTx_len = 32;
            Any$ = PrintTx.substring(0,WiFiTx_len);
            Any$.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
            WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
            WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
          //  t0 = micros();
            esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
            PrintTx = PrintTx.substring(32);
          }
        } else if (WiFiTx_CB == -1) {
          // an error has occured so resend the data
          if (WiFiTryCnt < 5) {
            WiFiTryCnt++;   // track the number of retries
            WiFiTx_CB = 0;  // clear the call back flag when sending
          //  t0 = micros();
            esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
          } else {
            // too many resends so disconnect WiFi link
            WiFiDisconnect();
            Serial.println("Failed to Tx!");
          }
        }
      } else {
        // not connected so clear PrintTxdata
        PrintTx = "";
      }
    }
  }
}

// ---------------------------------------------------------------------------------

void RunPOST() {
  // Start-up initialisation processes, mainly testing I2C addresses
  // 0x3C OLED display
  // 0x68 ICM-42607P 3-axis motion sensor
  Serial.println("\n\n\n" + Release + "\n");
  Serial.print(F("Released: ")); Serial.println(Date);
  Serial.println(F("Running POST..."));

  // Test for TEST mode, if either SW0 or SW1 are pressed coming out of reset
  if (!digitalRead(sw0Pin) || !digitalRead(sw1Pin)) {TEST = true;} else {TEST = false;}
  
  // Test for OLED display and initialise
  Wire.beginTransmission(0x3C);
  I2C_Err = Wire.endTransmission();
  if (I2C_Err == 0) {
    I2C_OLED = true; Display_Init();
    Display_Intro();
    Serial.println("Display initialised");
  } else {I2C_OLED = false; Serial.println("Display Not Found!");}

  // Set the LED ring low green
  LED_Fill(0,8,0);
  FastLED.show(); LED_Del = 100;  // hold this for 2 seconds
  Serial.println("RGB LEDs set Green.");

  // Test for ICM-42607-P 3-axis motion sensor and initialise
  Wire.beginTransmission(ICM_address);
  I2C_Err = Wire.endTransmission();
  if (I2C_Err == 0) {
    Serial.println(F("Initialising ICM..."));
    I2C_ICM = true; ICM_42607_Initialise();
  } else {I2C_ICM = false; Serial.println(F("ICM-42607-P Not Found!"));}
  
  // Set up battery reading
  initBatteryRead();
  
  // Initialise OLED display
  OLED_Init();
  OLED_Update();
  Serial.println("OLED initialised");

  if (TEST) {
    // forced into TEST mode
  } else {
    // normal startup
    SetMainMode(0);
    OLED_setBrightness(255);  // have a bright display for the intro
    OLED_Del = 50;             // wait 2 seconds before displaying anything else
  }

  Serial.println("POST completed");
  WaitSwUp();

  // clear timers used to assess system slack time
  for (int zT = 0;zT < 4; zT++) {t0[zT] = micros(); t2[zT] = 0;}  // clear the timings
}

// --------------------------------------------------------------------------------

void setDMs() {
  // Sets the DM references to their default values of -99
  DM_Auton = -99;             // Auton display assigned Display Mode
  DM_Batt = -99;              // Battery display assigned Display Mode
  DM_BatPlus = -99;           // Battery plus display assigned Display Mode
  DM_Control = -99;           // Control display assigned Display Mode
  DM_ICM = -99;               // ICM-42607-P values display
  DM_ICM_Mov = -99;           // ICM motion display assigned Display Mode
  DM_ICM_Rot = -99;           // ICM rotation display assigned Display Mode
  DM_JoyNorm = -99;           // Joystick normal values display assigned Display Mode
  DM_JoyOff = -99;            // Joystick values and offsets display assigned Display Mode
  DM_LDR = -99;               // LDR values display
  DM_LED = -99;               // LED modes display
  DM_MtrPWM = -99;            // motor PWM values
  DM_MoveEng = -99;           // motor MoveEng display assigned Display Mode
  DM_Range = -99;             // ranges display assigned Display Mode
  DM_Recall = -99;            // RECall display assigned Display Mode
  DM_Tacho = -99;             // Tachos display assigned Display Mode
  DM_Tune = -99;              // test display assigned Display Mode
  DM_Time = -99;              // timing display assigned Display Mode
  DM_Wii_Ctrl = -99;          // Wii controller display aassigned Display Mode
  // DM_WiFi = -99;              // WiFi display assigned Display Mode
}

// ---------------------------------------------------------------------------------

void synchLoopTimers() {
  // Synchronise loop timers
  // Ensure that there is at least 40ms on the counter
  while (millis() < 4) {yield();}

  next4ms = millis();         // set 4ms task timer
}

// --------------------------------------------------------------------------------

/*
  Timing analysis used to determine which functions are taking the longest. This is
  done using the t0[],t1[] and t2[] arrays, allowing you to place up to 10 measurements
  in each compiled version.

  System          20    µs
  ReadWiFiRx      76    µs
  Print40ms       321   µs
  FastLED.show()  542   µs
  Display_40ms()  4513  µs
  Task_16ms()     24636 µs
  Loop()          25052 µs

  Conclusion, these delays impact on slot reading, so move that task to Core 0
  
*/

// --------------------------------------------------------------------------------
