/*
  Normally my projects use a 128x64 OLED display, and its contents is mapped into
  the onitor+ application. However, in this project the OLED display is smaller
  and there fore treated as a separate entity, with its own set of display
  functions.
*/

// --------------------------------------------------------------------------------

void OLED_40ms() {
  // Called every 40ms from the main loop.
  // If DispDel > 0 then we wait for DispDel cycles to complete before an update.
  // Display updates can take several milliseconds, so we avoid updates during ranging.
  if (OLED_NOP) {return;}                   // this function can be disabled/aborted
  
  if (OLED_Del > 0) {OLED_Del--; return;}   // don't update display if OLED_Del > 0
  if (RCWL > 0) {return;}                   // don't run when sonar cycle is active

  t0[3] = micros();
  // When in RECord or PLAY modes we force the OLED display
  if (PLAY) {OLED_PLAY(); return;}
  if (REC) {OLED_RECord(); return;}
  // Otherwise display as normal
  switch (OLED_Mode) {
    case -1: OLED_DriveMode(); break;
    case  0: OLED_Update(); break;
    case  1: OLED_AppConnected(); break;
    
    case 10: OLED_LDR_Sense(); break;
    case 11: OLED_LDR_Track(); break;
    case 12: OLED_LDR_Follow(); break;

    case 20: OLED_Sonar_Sense(); break;
    case 21: OLED_Sonar_Turn(); break;
    case 22: OLED_Sonar_Backaway(); break;
    case 23: OLED_Sonar_Track(); break;
  }
  t1[3] = micros() - t0[3];
}

// --------------------------------------------------------------------------------

void OLED_AppConnected() {
  // App connected screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //     1111111111  2222222222        3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+4, "App");
  display.drawString(Xoff+32,Yoff+16, "Connected");
  display.drawString(Xoff+32,Yoff+34, String(BatVfp,1) + "v  " + String(BatPc) + "%");

  display.display();
  OLED_Del = 5;
}

// --------------------------------------------------------------------------------

void OLED_Battery_Low() {
  // WARNING screen before shut down
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //           1111111111  2222222222
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+10, "BATTERY");
  display.drawString(Xoff+32,Yoff+22, "LOW");
  display.display();
}

// --------------------------------------------------------------------------------

void OLED_BatWarning() {
  // WARNING screen battery < 20%
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //           1111111111  2222222222  3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+10, "BATTERY");
  display.drawString(Xoff+32,Yoff+22, "LOW");
  display.drawString(Xoff+32,Yoff+34, "<20%>");
  display.display();
}

// --------------------------------------------------------------------------------

void OLED_DriveMode() {
  // Drive Mode screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //     1111111111  2222222222        3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+4, "DRIVE");
  display.drawString(Xoff+32,Yoff+16, "Mode");
  display.drawString(Xoff+32,Yoff+34, String(BatVfp,1) + "v  " + String(BatPc) + "%");

  display.display();
  OLED_Del = 5;
}

// --------------------------------------------------------------------------------

void OLED_Gear() {
  // Display Gear screen
  display.clear();
  display.setFont(ArialMT_Plain_16);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111111111  222222222222222222222222
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+5, "GEAR");

  display.setFont(ArialMT_Plain_24);
  display.drawString(Xoff+32,Yoff+23, String(Gear));

  display.display();  // display this immediately
  OLED_Del = 20;      // wait before display next screen
}

// --------------------------------------------------------------------------------

void OLED_GearMax() {
  // Display Gear Max
  display.clear();
  display.setFont(ArialMT_Plain_16);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111111111  222222222222222222222222
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+5, "GEAR");

  display.setFont(ArialMT_Plain_24);
  display.drawString(Xoff+32,Yoff+23, "Max");

  display.display();  // display this immediately
  OLED_Del = 20;      // wait before display next screen
}

// --------------------------------------------------------------------------------

void OLED_GearMin() {
  // Display Gear Min
  display.clear();
  display.setFont(ArialMT_Plain_16);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111111111  222222222222222222222222
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+5, "GEAR");

  display.setFont(ArialMT_Plain_24);
  display.drawString(Xoff+32,Yoff+23, "Min");

  display.display();  // display this immediately
  OLED_Del = 20;      // wait before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Init() {
  // Initialise the OLED display
  display.init();
  display.clear();
  display.setContrast(31);
  display.flipScreenVertically();   // use this with Yoff=16, when display is upside down
  display.display();
}

// --------------------------------------------------------------------------------

void OLED_LDR_Follow() {
  // Display LDR screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //            1111111111  2222222222    3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+11, "LDR");
  display.drawString(Xoff+32,Yoff+23, "Follow");

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(Xoff+0,Yoff+37, String(LDR1));

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  display.drawString(Xoff+63,Yoff+37, String(LDR0));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_LDR_Sense() {
  // Display LDR screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //            1111111111  2222222222    3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+11, "LDR");
  display.drawString(Xoff+32,Yoff+23, "Sense");

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(Xoff+0,Yoff+37, String(LDR1));

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  display.drawString(Xoff+63,Yoff+37, String(LDR0));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_LDR_Track() {
  // Display LDR track screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //            1111111111  2222222222    3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+11, "LDR");
  display.drawString(Xoff+32,Yoff+23, "Track");

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(Xoff+0,Yoff+37, String(LDR1));

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  display.drawString(Xoff+63,Yoff+37, String(LDR0));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_PLAY() {
  // Display PLAYback screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111 2222222222    3333333333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+ 5, "PLAY");
  display.drawString(Xoff+32,Yoff+16, "Mode");

  display.setFont(ArialMT_Plain_16);
  display.drawString(Xoff+32,Yoff+27, getMinSec(RECtotal - PLAYtotal));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_RECord() {
  // Display RECording screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111 2222222222    3333333333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+ 5, "RECORD");
  display.drawString(Xoff+32,Yoff+16, "Mode");

  display.setFont(ArialMT_Plain_16);
  display.drawString(Xoff+32,Yoff+27, getMinSec(RECtotal));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_setBrightness(uint8_t zB) {
  // set the brightness level of the OLED display
  Brightness = zB; display.setBrightness(zB);
}

// --------------------------------------------------------------------------------

void OLED_Sonar_Backaway() {
  // Display sonar backaway screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111 2222222222 3333333333 4444444444
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+ 5, "Sonar");
  display.drawString(Xoff+32,Yoff+16, "Backaway");

  display.drawString(Xoff+32,Yoff+27, String(RangeVal[0]));

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(Xoff+0,Yoff+38, String(RangeVal[1]));

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  display.drawString(Xoff+63,Yoff+38, String(RangeVal[2]));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Sonar_Sense() {
  // Display sonar sense screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111 2222222222 3333333333 4444444444
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+ 5, "Sonar");
  display.drawString(Xoff+32,Yoff+16, "Sense");

  if (RCWL_Mask & 0b001) {display.drawString(Xoff+32,Yoff+27, String(RangeVal[0]));}
  else {display.drawString(Xoff+32,Yoff+27, "---");}

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  if (RCWL_Mask & 0b010) {display.drawString(Xoff+0,Yoff+38, String(RangeVal[1]));}
  else {display.drawString(Xoff+0,Yoff+38, "---");}

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  if (RCWL_Mask & 0b100) {display.drawString(Xoff+63,Yoff+38, String(RangeVal[2]));}
  else {display.drawString(Xoff+63,Yoff+38, "---");}

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Sonar_Track() {
  // Display sonar track screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111 2222222222 3333333333 4444444444
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+ 5, "Sonar");
  display.drawString(Xoff+32,Yoff+16, "Track");

  display.drawString(Xoff+32,Yoff+27, String(RangeVal[0]));

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(Xoff+0,Yoff+38, String(RangeVal[1]));

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  display.drawString(Xoff+63,Yoff+38, String(RangeVal[2]));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Sonar_Turn() {
  // Display sonar turn screen
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //      1111111111 2222222222 3333333333 4444444444
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+ 5, "Sonar");
  display.drawString(Xoff+32,Yoff+16, "Turn");

  display.drawString(Xoff+32,Yoff+27, String(RangeVal[0]));

  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(Xoff+0,Yoff+38, String(RangeVal[1]));

  display.setTextAlignment(TEXT_ALIGN_RIGHT);
  display.drawString(Xoff+63,Yoff+38, String(RangeVal[2]));

  display.display();  // display this immediately
  OLED_Del = 5;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Text1S24(String zT0) {
  // Display one lines of text, in 24 font
  display.clear();
  display.setFont(ArialMT_Plain_24);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //               111111111111111111111111
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+14, zT0);

  display.display();  // display this immediately
  OLED_Del = 50;      // wait 2 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Text2S10(String zT0,String zT1) {
  // Display two lines of text, in 10 font
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //               1111111111    2222222222
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+14, zT0);
  display.drawString(Xoff+32,Yoff+28, zT1);

  display.display();  // display this immediately
  OLED_Del = 25;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Text2S1610(String zT0,String zT1) {
  // Display two lines of text, in 10 font
  display.clear();
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //         1111111111111111    2222222222
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.setFont(ArialMT_Plain_16);
  display.drawString(Xoff+32,Yoff+8, zT0);
  display.setFont(ArialMT_Plain_10);
  display.drawString(Xoff+32,Yoff+28, zT1);

  display.display();  // display this immediately
  OLED_Del = 25;      // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void OLED_Update() {
  // First screen to be displayed
  // default fonts are ArialMT_Plain_10, ArialMT_Plain_16 and ArialMT_Plain_24
  display.clear();
  display.setFont(ArialMT_Plain_10);
  // Y48 vertical text spacing, from Y=0 to Y=47
  // 012345678901234567890123456789012345678901234567
  //     1111111111  2222222222        3333333333
  display.setTextAlignment(TEXT_ALIGN_CENTER);
  display.drawString(Xoff+32,Yoff+4, "Omni-Bot");
  display.drawString(Xoff+32,Yoff+16, Rev);
  display.drawString(Xoff+32,Yoff+34, String(BatVfp,1) + "v  " + String(BatPc) + "%");

  if (TEST) {display.drawRect(Xoff,Yoff,64,48);} // draw 'Test' box outline
  display.display();
  OLED_Del = 5;
}

// --------------------------------------------------------------------------------

