/*
  OLED Display Mirroring functions
  Rel: 13/03/2022

  Note the names of these functions start with a capital 'D' and use an underscore
  '_' instead of a period '.' so as not to be confused with the functions of the
  SSD1306Wire.h library. These functions only work when connected to a PC running
  the OLED Mirror app. I have only included the functions needed by this project
  but more could easily be added, using the same principles. Not all features can
  be included due to the limitations of Processing, like inverse ink colour.

  Display_clear()                 - clear the mirror pixel buffers
  Display_display()               - write the mirrored buffer to the display
  Display_drawCircle(X0,Y0,R0)    - draw a circle radius R0 at X0,Y0
  Display_drawLine(X0,Y0,X1,Y1)   - draw a line from X0,Y0 to X1,Y1
  Display_drawRect(x,y,w,h)       - draw a rectangle in the OLED Mirror app
  Display_drawString(x,y,S$)      - draw text string on teh mirrored display
  Display_fillRect(X,Y,W,H)       - draw a filled rectangle in the OLED Mirror app
  Display_setBrightness(Bright)   - sets teh brightness of the display 0 - 255
  Display_setColor(Ink)           - set the colour of the ink in OLED Mirror app, BLACK, WHITE
  Display_setFont(FontSize)       - set the point size of the mirrored font
  display.setTextAlignment(Align) - TEXT_ALIGN_LEFT, TEXT_ALIGN_CENTER, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER_BOTH

*/

// --------------------------------------------------------------------------------

void Display_ACK() {
  // Request and ACK from Monitor+ app.
  // Used to improve Wi-Fi link stability. It does not affect OLED display.
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~ACK\n";
  ACK = 25; // set ACK to 1 sec timeout, in case Monitor+ does not respond
}

// --------------------------------------------------------------------------------

void Display_clear() {
  // Clear the Monitor+ memory
  // if DispMon then clear the mirror pixel buffers
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~Clr\n";
}

// --------------------------------------------------------------------------------

void Display_display() {
  // Display update over I2C
  // if DisMon then write the mirrored commands to the display in Monitor+ app
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~Disp\n";
}

// --------------------------------------------------------------------------------

void Display_drawCircle(int16_t zX0,int16_t zY0,int16_t zR0) {
  // Draw a circle radius zR0 at zX0,zY0
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwCle(" + String(zX0) + "," + String(zY0) + "," + String(zR0) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_drawLine(int16_t zX0,int16_t zY0,int16_t zX1,int16_t zY1) {
  // Draw a line from X0,Y0 to X1,Y1
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwLin(" + String(zX0) + "," + String(zY0) + "," + String(zX1) + "," + String(zY1) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_drawRect(int16_t zX,int16_t zY,int16_t zW,int16_t zH) {
  // Draw a rectangle in Monitor+
  // if DispMOn then draw a rectangle in the OLED Mirror app
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwRec(" + String(zX) + "," + String(zY) + "," + String(zW) + "," + String(zH) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_drawString(int16_t zX,int16_t zY,String zS$) {
  // Draw text at zX,zY in Monitor+
  // if DispMOn then draw text string on the mirrored display
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwStr(" + String(zX) + "," + String(zY) + "," + zS$ + ")\n";
}

// --------------------------------------------------------------------------------

void Display_fillRect(int16_t zX,int16_t zY,int16_t zW,int16_t zH) {
  // draw a filled rectangle in OLED memory
  // if DispMon then draw a filled rectangle in the OLED Mirror app
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwBar(" + String(zX) + "," + String(zY) + "," + String(zW) + "," + String(zH) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setBrightness(uint8_t zB) {
  // Set the brightness level of the Monitor+ display
  Brightness = zB;
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetLux(" + String(zB) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setColor(uint8_t zInk) {
  // set the colour of the ink used in OLED display graphics
 if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetCol(" + String(zInk) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setFont(uint8_t zFp) {
  // set the font size of the OLED text
  // if DispMon then set the point size of the mirrored font
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetFnt(" + String(zFp) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setTextAlignment(uint8_t zFa) {
  // set the font alignment
  // if DispMOn then set text alignment for OLED Mirror app 
  // zFa = 0  TEXT_ALIGN_LEFT
  // zFa = 1  TEXT_ALIGN_RIGHT
  // zFa = 2  TEXT_ALIGN_CENTER
  // zFa = 3  TEXT_ALIGN_CENTER_BOTH
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetAlg(" + String(zFa) + ")\n";
}

// --------------------------------------------------------------------------------
