
/* SSD1306Wire.h functions, see https://github.com/ThingPulse/esp8266-oled-ssd1306

  display.clear()        - clear the local pixel buffer
  display.display()      - write the buffer to the display
  display.displayOff()   - turn the display OFF
  display.displayOn()    - turn the display ON
  display.drawCircle(int16_t x, int16_t y, int16_t radius)
  display.drawFastImage(int16_t x, int16_t y, int16_t width, int16_t height, const uint8_t *image)
  display.drawHorizontalLine(int16_t x, int16_t y, int16_t length)
  display.drawLine(int16_t x0, int16_t y0, int16_t x1, int16_t y1)  - draw from x0k,y0 to x1,y1
  display.drawProgressBar(uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint8_t progress)
  display.drawRect(int16_t x, int16_t y, int16_t width, int16_t height) - draw a rectangle
  display.drawString(int16_t x, int16_t y, String text)
  display.drawStringMaxWidth(int16_t x, int16_t y, int16_t maxLineWidth, String text)
  display.drawVerticalLine(int16_t x, int16_t y, int16_t length)
  display.drawXbm(int16_t x, int16_t y, int16_t width, int16_t height, const char* xbm)
  display.fillCircle(int16_t x, int16_t y, int16_t radius)
  display.fillRect(int16_t x, int16_t y, int16_t width, int16_t height)
  display.flipScreenVertically()   - flip the current screen upside down
  disply.init()         - initialise the display
  disply.invertDisplay()- inverted display mode
  disply.end()          - free memory used by the display
  disply.displayOn()    - turn the display ON
  display.mirrorScreen() - draw the screen mirrored
  disply.normalDisplay()- normal display mode
  disply.reconnect()    - connect again through I2C
  display.setBrightness(uint8_t)
  display.setColor(OLEDDISPLAY_COLOR color) - BLACK, WHITE, INVERSE
  display.setContrast(uint8_t contrast, uint8_t precharge = 241, uint8_t comdetect = 64)
  display.setFont(const uint8_t* fontData)
  display.setPixel(int16_t x, int16_t y)  - set a pixel at x,y
  display.setTextAlignment(OLEDDISPLAY_TEXT_ALIGNMENT textAlignment)  - TEXT_ALIGN_LEFT, TEXT_ALIGN_CENTER, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER_BOTH

  W = display.getStringWidth(const char* text, uint16_t length)
  W = display.getStringWidth(String text)
  
  Display size is 128 x 64 (w x h) pixels
  Default fonts are ArialMT at 10, 16 & 24 pixels high
  Text options include:
    2 x 24, + 1 x 10 pnt lines
    1 x 24, + 2 x 16 pnt lines
    1 x 24, + 3 x 10 pnt lines
    4 x 16 pnt lines, no line spaces, decending characters will merge
    3 x 16, + 1 x 10 pnt lines
    6 x 10 pnt lines, no line spaces, decending characters will merge
*/

// --------------------------------------------------------------------------------

void DispMon_40ms() {
  // Called every 40ms from the main loop, when Monitor+ app is connected
  // if DispDel > 0 then we wait for DispDel cycles to complete before update
  if (DispNOP) {return;}          // this function is aborted, no I2C activity
  if (ACK > 0) {ACK--; return;}   // waiting for an acknowledgement from Monitor+ app
  
  if (DispDel > 0) {DispDel--; return;} // don't update display if DispDel > 0
  if (DispClr) {Display_clear(); DispClr = false; return;}
  if (DispDisp) {Display_display(); DispDisp = false; DispClr = true; return;}

  // we refresh individual displays based on a counter, DispCnt
  // Serial.println("DispMode=" + String(DispMode));
  DispCnt--;
  if (DispCnt < 1) {
    // DispCnt is reset for display updates, so can be different for each display mode
    DispCnt = 1;  // default, will trigger on each cycle
    if (DispLock) {DispMode = DispOveride;} // force locked mode
    // Menu screens, display depends on menu modes
    switch(DispMode) {
      case -4: Display_Tune(); break;
      case -3: Display_Timing(); break;
      case -2: Display_Recall(); break;
      case -1: Display_Control(); break;
      case  0: Display_Battery(); break;
      case  1: Display_LDR(); break;
      case  2: Display_Ranges(); break;
      case  3: Display_Auton(); break;
      case  4: Display_ICM(); break;
      case  5: Display_ICM_Move(); break;
      case  6: Display_ICM_Rot(); break;
      case  7: Display_Motor_PWM(); break;
      case  8: Display_Move_Engine(); break;
      case  9: Display_Tachos(); break;
      case 10: Display_Joystick(); break;
      case 11: Display_Joystick_Norm(); break;
      case 12: Display_LED(); break;
      case 13: Display_Wii_Ctrl(); break;
    }
    DM_Min = -4; DM_Max = 13;
    // the following line puts a rectangle around every display, showing the boundry region
  //  Border = true;
    if (Border) {Display_drawRect(0,0,128,64);}

    // Add a global frame delay offset, if set
    DispDel += DispDelGbl;
  }
}

// --------------------------------------------------------------------------------

void Display_Auton() {
  // normal screen to be displayed
  DM_Auton = DispMode;    // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111  2222222222222222   3333333333    4444444444
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  if (MainTask == 22) {Display_drawString(64, 2, "Backaway");}
  else {Display_drawString(64, 2, "Auton");}
  // Next display range data
  if (RangeAvg[0] <= 400) {Display_drawString(64, 20, String(RangeAvg[0]));}
  else {Display_drawString(64, 20, "< 400 >");}

  Display_setFont(10);
  Display_drawString(32, 39, String(TachoC));
  Display_drawString(95, 39, String(TachoA));
  Display_drawString(64, 53, TaskMsg);

  Display_setColor(GREEN);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 0, 53, String(MainSub));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update delay
}

// --------------------------------------------------------------------------------

void Display_Battery() {
  // normal screen to be displayed
  DM_Batt = DispMode;     // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()
  if (TEST) {Display_Battery_Plus(); return;}

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111     2222222222  3333333333     4444444444
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Omni-Bot 3x3");
  Display_setFont(10);
  Display_drawString(64,23, "Battery");
  if (!USB) {
    if (BatVfp > 9.99) {Any$= String(BatVfp,1) + "v  ";}
    else {Any$ = String(BatVfp,2) + "v  ";}
    Display_drawString(64,35, Any$ + String(BatPc) + "%");
  } else {
    Display_drawString(64,35, "USB 5v");
  }
  Display_drawString(64,50, getAtState());

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 25; // update every 1s
}

// --------------------------------------------------------------------------------

void Display_Battery_Plus() {
  // DEtailed battery screen to be displayed
  DM_BatPlus = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111    2222222222222222        3333333333333333
  //                           4444444444444444
  Display_setFont(16); Display_setColor(WHITE);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Battery+");
  if ((BatAvg > Bat6v6) || TEST) {
    // battery powered mode
    if (BatVfp > 9.99) {Display_drawString(64, 21, String(BatVfp,1) + "v");}
    else {Display_drawString(64,21, String(BatVfp,2) + "v");}

    Display_setFont(10);
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_drawString( 2,26, String(BatAvg));
    Display_drawString( 2,45, getHmsTime(millis()));

    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    Display_drawString(125,26, String(BatPc) + "%");
    Display_drawString(125,45, BatTime$);
  } else {
    // USB powered mode
    Display_setTextAlignment(TEXT_ALIGN_CENTER);
    Display_drawString(64,24, "USB Pwr");
    // display the time
    Display_drawString(64,45, getHmsTime(millis()));
  }
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 25; // update every 1s
}

// --------------------------------------------------------------------------------

void Display_Batt_Low() {
  // Battery = 0%
  // This is the last message sent over the comms link
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                1111111111111111    2222222222222222
  Display_setFont(16); Display_setColor(RED);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,15, "Battery Low");
  Display_drawString(64,35, "0%");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
}

// --------------------------------------------------------------------------------

void Display_Control() {
  // Control screen options
  DM_Control = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111  2222222222 3333333333 4444444444 5555555555
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(63, 1, "Control");

  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62,19, "RESET:");
  Display_drawString(62,30, "Batt:");
  Display_drawString(62,41, "N/T:");
  Display_drawString(62,52, "Mode:");
  Display_drawString(99,52, "Task:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_setColor(BLUE);
  Display_drawString( 63,19, "Now!");
  Display_drawString( 63,30, "Flat");
  if (TEST) {Display_drawString( 63,41, "Normal");} else {Display_drawString( 63,41, "TEST");}
  Display_drawString( 63,52, String(NewMode));
  Display_drawString(100,52, String(NewTask));
  Display_drawString(111,52, "Go!");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);}
  Display_display();
  DispDel = 10; // update every 400ms
}

// --------------------------------------------------------------------------------

void Display_ICM() {
  // Display ICM-42607-P accelerometer & gyro information
  DM_ICM = DispMode;      // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111    2222222222    3333333333    4444444444
  //  55555555 66666666
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "ICM");

  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(31, 21, "AcX:");
  Display_drawString(89, 21, "Off:");
  Display_drawString(31, 35, "AcY:");
  Display_drawString(89, 35, "Off:");
  Display_drawString(31, 49, "GyZ:");
  Display_drawString(89, 49, "Off:");

  Display_setFont(8);
  Display_drawString(127, 1, String(1000000/ICMtd) + "Hz");
  Display_setColor(GREEN);
  Display_drawString(127, 49, String(GyZnt));

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  if (ICMove > 0) {Display_drawString(0,10, "M");}

  Display_setColor(BLUE);
  if (ICM_En) {Display_drawString(12, 1, "Y");} else {Display_drawString(12, 1, "N");}
  Display_setColor(WHITE);
  Display_drawString( 0, 1, "En:");

  Display_setFont(10);
  Display_drawString(33, 21, String(AcX));
  Display_drawString(90, 21, String(AcXOff));
  Display_drawString(33, 35, String(AcY));
  Display_drawString(90, 35, String(AcYOff));
  Display_drawString(33, 49, String(GyZ));
  Display_drawString(90, 49, String(GyZOff));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 3;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_ICM_Move() {
  // display MPU6050 accelerometer information
  DM_ICM_Mov = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111    2222222222    3333333333    4444444444
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Motion");

  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(126, 7, "mm/s");
  Display_drawString(26, 21, "VelX:");
  Display_drawString(88, 21, "VelY:");
  Display_drawString(26, 35, "DisX:");
  Display_drawString(88, 35, "DisY:");
  Display_drawString(26, 49, "Clr:");
  Display_drawString(88, 49, "Clr:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  if (ICMove > 0) {
    Display_setColor(GREEN);
    Display_drawString(2, 1, "M");
    Display_setColor(WHITE);
  }
  Display_drawString(27, 21, String(VelX,1));
  Display_drawString(89, 21, String(VelY,1));
  Display_drawString(27, 35, String(DisX,1));
  Display_drawString(89, 35, String(DisY,1));

  Display_setColor(BLUE);
  Display_drawString(27, 49, "[!]");
  Display_drawString(89, 49, "[!]");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 3;        // wait 120ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_ICM_Rot() {
  // display ICM-42607-P rotation angle
  DM_ICM_Rot = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                        1111111111111111     2222222222222222
  //                                                       3333333333
  Display_setFont(16);  // fonts are ArialMT_Plain_10, ArilMT_Plain_24
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 24, String(YawGyr,1) + Deg);

  Display_setFont(10);  // fonts are ArialMT_Plain_10, ArilMT_Plain_24
  if (YawTgt != 0.0) {Display_drawString(64, 44, String(YawTgt,1) + Deg);}

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  if (YawDif >= 0.0) {Display_drawString(3, 54, String(YawDif,3) + Deg);}
  else {Display_drawString(0, 54, String(YawDif,3) + Deg);}

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(127, 54, "GyZ:" + String(GyZOff));
  
  Display_drawCircle(64,32,31);
  Display_drawLine(21,32,30,32);  // left horizontal line
  Display_drawLine(97,32,106,32); // right horizontal line
  if (YawDif > 0.01) {
    // right arrow points up, left arrow points down
    Display_drawLine(21,32,26,21); Display_drawLine(26,21,30,32);
    Display_drawLine(97,32,101,43); Display_drawLine(101,43,106,32);
  } else if (YawDif < -0.01) {
    // right arrow points down, left arrow points up
    Display_drawLine(21,32,26,43); Display_drawLine(26,43,30,32);
    Display_drawLine(97,32,101,21); Display_drawLine(101,21,106,32);
  }

  Display_setColor(BLUE);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 0, 0, "CLR");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 3;        // wait 80ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Init() {
  // initialise the OLED display
  display.init();
  display.clear();
  display.setContrast(255);
  // display.flipScreenVertically();
  display.display();
  DispClr = false; DispDisp = false;
}

// --------------------------------------------------------------------------------

void Display_Intro() {
  // first screen to be displayed
  // default fonts are ArialMT_Plain_10, ArialMT_Plain_16 and ArialMT_Plain_24
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111111111111111111 2222222222222222          3333333333
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 0,24,2,16,4,10
  Display_drawString(64, 0, "Omni-Bot");
  Display_setFont(16);
  Display_drawString(64, 25, Rev);
  Display_setFont(10);
  Display_drawString(64, 51, "Batt: " + String(BatVfp) + "v  " + String(BatPc) + "%");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 20;
}

// --------------------------------------------------------------------------------

void Display_Joystick() {
  // display joystick valus and offsets
  DM_JoyOff = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111     2222222222222222     3333333333333333
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Joysticks");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_setFont(10);
  if (WiiType == 'N') {
    // Wii Nunchuk joystick data
    Display_drawString(16,25, "X = ");
    Display_drawString(44,25, String(RxWiFi[0]));
    Display_drawString(16,46, "Y = ");
    Display_drawString(44,46, String(RxWiFi[1]));
    } else if (WiiType == 'C') {
    // Wii Classic joystick data
    Display_drawString( 8,25, "XL= ");
    Display_drawString(40,25, String(RxWiFi[0] & 63));
    Display_drawString(72,25, "XR= ");
    AnyByte = ((RxWiFi[0] & 0b11000000)>>3) + ((RxWiFi[1] & 0b11000000)>>5) + ((RxWiFi[2] & 0b10000000)>>7);
    Display_drawString(104,25, String(AnyByte));
    Display_drawString( 8,46, "YL= ");
    Display_drawString(40,46, String(RxWiFi[1] & 63));
    Display_drawString(72,46, "YR= ");
    Display_drawString(104,46, String(RxWiFi[2] & 31));
    } else if (WiiType == 'P') {
    // Wii Classic Pro joystick data
    Display_drawString( 8,25, "XL= ");
    Display_drawString(40,25, String(RxWiFi[0] & 63));
    Display_drawString(72,25, "XR= ");
    AnyByte = ((RxWiFi[0] & 0b11000000)>>3) + ((RxWiFi[1] & 0b11000000)>>5) + ((RxWiFi[2] & 0b10000000)>>7);
    Display_drawString(104,25, String(AnyByte));
    Display_drawString( 8,46, "YL= ");
    Display_drawString(40,46, String(RxWiFi[1] & 63));
    Display_drawString(72,46, "YR= ");
    Display_drawString(104,46, String(RxWiFi[2] & 31));
  } else {
    // Wii type undefined
    Display_drawString(44,25, "-?-");
    Display_drawString(44, 46, "-?-");
    Display_drawString(86,25, "-?-");
    Display_drawString(86, 46, "-?-");
  }
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 3;
}

// --------------------------------------------------------------------------------

void Display_Joystick_Norm() {
  // display normalised joystick valus and offsets
  DM_JoyNorm = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111     2222222222222222     3333333333333333
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Joysticks");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_setFont(10);
  if (WiiType == 'N') {
    // Wii Nunchuk joystick data
    Display_drawString(16,25, "X = ");
    Display_drawString(44,25, String(RxWiFi[0]));
    Display_drawString(16,46, "Y = ");
    Display_drawString(44,46, String(RxWiFi[1]));
    } else if (WiiType == 'C') {
    // Wii Classic joystick data
    Display_drawString( 8,25, "LX= ");
    Display_drawString(40,25, String(JoyLX));
    Display_drawString(72,25, "JX= ");
    Display_drawString(104,25, String(JoyX));
    Display_drawString( 8,46, "LY= ");
    Display_drawString(40,46, String(JoyLY));
    Display_drawString(72,46, "JY= ");
    Display_drawString(104,46, String(JoyY));
    } else if (WiiType == 'P') {
    // Wii Classic Pro joystick data
    Display_drawString( 8,25, "LX= ");
    Display_drawString(40,25, String(JoyLX));
    Display_drawString(72,25, "JX= ");
    Display_drawString(104,25, String(JoyX));
    Display_drawString( 8,46, "LY= ");
    Display_drawString(40,46, String(JoyLY));
    Display_drawString(72,46, "JY= ");
    Display_drawString(104,46, String(JoyY));
  } else {
    // Wii type undefined
    Display_drawString(44,25, "-?-");
    Display_drawString(44, 46, "-?-");
    Display_drawString(86,25, "-?-");
    Display_drawString(86, 46, "-?-");
  }
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 3;
}

// --------------------------------------------------------------------------------

void Display_LDR() {
  // LDR sensors display
  DM_LDR = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  111111111111  22222222  33333333  44444444           55555555
  Display_setFont(12);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "LDRs");
  Display_setFont(8);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(31, 15, "LDR0:");
  Display_drawString(95, 15, "LDR1:");
  Display_drawString(31, 25, "Avg0:");
  Display_drawString(95, 25, "Avg1:");
  Display_drawString(62, 35, "Div:");
  Display_drawString(62, 54, "Gain:");
  Display_drawString( 7, 44, "L");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(33, 15, String(LDR0));
  Display_drawString(97, 15, String(LDR1));
  Display_drawString(33, 25, String(LDR0_Avg));
  Display_drawString(97, 25, String(LDR1_Avg));
  Display_drawString(120,44, "R");

  Display_setColor(BLUE);
  Display_drawString(64, 35, get3Char(LDR_Div));
  Display_drawString(64, 54, get3Char(LDR_Ga));
  // Draw direction bar
  Display_setColor(WHITE);
  Display_drawLine(63,48,63,50);  // draw line vertical centre
  Display_drawLine( 8,49,118,49); // draw horizontal line
  int16_t zX = (LDR_Ga * (LDR1_Avg - LDR0_Avg))/(1 + (abs(LDR0_Avg + LDR1_Avg)/2));
  if (zX > 55) {zX = 55;} else if (zX < -55) {zX = -55;}
  zX += 62; // offset is 2 short to account for with of rectangle
  Display_drawRect(zX,46,2,6);    // draw box value

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 4;
}

// --------------------------------------------------------------------------------

void Display_LED() {
  // LED mode display
  DM_LED = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111  2222222222  3333333333
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "LEDs");
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62, 19, "LedMode:");
  Display_drawString(62, 31, "Task:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(64, 19, String(LedMode));
  Display_drawString(64, 31, String(LED_Task));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 4;
}

// --------------------------------------------------------------------------------

void Display_Mirrored() {
  // display robot type when DispMon is first connected
  // we clear current flags so that this is displayed immediately
  DispClr = false; DispDisp = false; DispCnt = 1;
  
  PrintTx = "\n\n";   // remove any previous messages
  Display_clear();    // remove current display content
  if (Brightness != 255) {Display_setBrightness(255);}  // display may be in sleep mode
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111111111111111111 2222222222222222        3333333333
  Display_display();  // clears moirror display immediately
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Omni-Bot");
  Display_setFont(16);
  Display_drawString(64, 27, Rev);
  Display_setFont(10);
  Display_drawString(64,53, "Display Mirror");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();    // display immediately
  DispMode = DispNext;  // return to previous display mode
  DispDel = 50;         // wait 2s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Motor_PWM() {
  // display the PWM values being applied to the motors
  DM_MtrPWM = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111  2222222222  3333333333  4444444444  5555555555
  //                6666666
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Motor  PWM");

  Display_drawString( 24, 13, "C:" + String(PWM_C));
  Display_drawString(103, 13, "A:" + String(PWM_A));
  Display_drawString( 64, 25, "B:" + String(PWM_B));
  Display_drawString( 64, 37, "S:" + String(Steer));

  Display_setFont(7);
  if (WiFiEn) {
    Display_setColor(GREEN);
    Display_drawString( 64, 15, DriveTxt);
    Display_drawString( 64, 49, "D:" + String(Drive));
  } else {
    Display_setColor(RED);
    Display_drawString( 64, 15, "No RC");
  }
  Display_setColor(WHITE);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 3;        // wait 80ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Move_Engine() {
  // Display Move Engine values
  DM_MoveEng = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111  2222222222  3333333333  4444444444      5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Move Engine");

  Display_drawString( 24, 13, "C:" + String(PWM_C));
  Display_drawString(103, 13, "A:" + String(PWM_A));
  Display_drawString( 64, 25, "B:" + String(PWM_B));
  Display_drawString( 24, 37, "Cnt:" + String(MoveCnt));
  Display_drawString( 64, 37, "Pnt:" + String(MovePnt));
  Display_drawString(103, 37, "Stp:" + String(MoveStp));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 3;        // wait 80ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Ranges() {
  // Ranges mode display
  DM_Range = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111    2222222222   3333333333   4444444444
  //  5555555555
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Ranges");
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 10,  1, "M:");
  Display_drawString( 62, 21, "Fnt:");
  Display_drawString( 31, 34, "Lft:");
  Display_drawString( 95, 34, "Rht:");
  Display_drawString( 19, 47, "Ex:");
  Display_drawString( 58, 47, "Rate:");
  Display_drawString(117, 47, "Avg:");

  Display_drawString(118,  1, "En:");
  Display_setColor(BLUE);
  Display_drawString(16,  1, String(RCWL_Mode));
  String zT$ = "N"; if (RCWL_En) {zT$ = "Y";}
  Display_drawString(127,  1, zT$);

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 20, 47, String(RangeEx));
  float zRate = 250.0/(float)RCWL_Max;
  if (zRate >= 100.0) {Display_drawString( 59, 47, String(zRate,0) + "Hz");}
  else {Display_drawString( 59, 47, String(zRate,1) + "Hz");}
  Display_drawString(118, 47, String(RangeDiv));

  Display_setColor(WHITE);
  if (RCWL_En) {
    if (RCWL_Mask & 0b001) {Display_drawString(63, 21, String(RangeVal[0]));} else {Display_drawString(63, 21, "---");}
    if (RCWL_Mask & 0b010) {Display_drawString(96, 34, String(RangeVal[1]));} else {Display_drawString(96, 34, "---");}
    if (RCWL_Mask & 0b100) {Display_drawString(32, 34, String(RangeVal[2]));} else {Display_drawString(32, 34, "---");}
  } else {
    Display_drawString(63, 21, "---");
    Display_drawString(32, 34, "---");
    Display_drawString(96, 34, "---");
  }

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5;
}

// --------------------------------------------------------------------------------

void Display_Recall() {
  // Display 6 bytes received from Wii over ESP-NOW.
  // Display I2C values as decimal 0 - 255.
  DM_Recall = DispMode;     // note Display Mode assignment
  if (!DispEn) {return;}    // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 1111111111 222222223333333344444444555555556666666677777777
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  if (!PLAY) {Display_drawString(63,  0, "Record");}
  else {Display_drawString(63,  0, "Recall");}

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_setFont(8); DispCX = 89;
  Display_drawString(DispCX, 11, String(RxWiFi[0]));
  Display_drawString(DispCX, 19, String(RxWiFi[1]));
  Display_drawString(DispCX, 27, String(RxWiFi[2]));
  Display_drawString(DispCX, 35, String(RxWiFi[3]));
  Display_drawString(DispCX, 43, String(RxWiFi[4]));
  Display_drawString(DispCX, 51, String(RxWiFi[5]));
  // display I2C values as binary 00000000 - 11111111
  Display_setTextAlignment(TEXT_ALIGN_LEFT); DispCX = 93;
  Display_drawString(DispCX, 11, GetBIN(RxWiFi[0]));
  Display_drawString(DispCX, 19, GetBIN(RxWiFi[1]));
  Display_drawString(DispCX, 27, GetBIN(RxWiFi[2]));
  Display_drawString(DispCX, 35, GetBIN(RxWiFi[3]));
  Display_drawString(DispCX, 43, GetBIN(RxWiFi[4]));
  Display_drawString(DispCX, 51, GetBIN(RxWiFi[5]));
  // display Wii type in top left corner
  if (!WiFiEn) {
    Display_setColor(RED);
    Display_drawString(0, 11, "WiFi Disabled");
    Display_setColor(WHITE);
  } else {
         if (WiiType == 'C') {Display_drawString(0, 11, "Classic");}
    else if (WiiType == 'P') {Display_drawString(0, 11, "Classic Pro");}
    else if (WiiType == 'N') {Display_drawString(0, 11, "Nunchuk");}
  }

  Display_drawString(2, 27, "Size:" + String(RECpnt) + "/" + String(RECnum) + "  " + String((100.0 * RECpnt)/RECnum,0) + "%");
  Display_drawString(1, 35, "REC:");
  Display_drawString(34, 35, "PLAY:");
  if (!PLAY) {
    Display_drawString( 0, 43, "Time:" + String(RECtime[RECpnt]));
    Display_drawString(49, 43, getMinSec(RECtime[RECpnt]));
    Display_drawString(10, 51, "At:" + String(RECtotal));
    Display_drawString(49, 51, getMinSec(RECtotal));
  } else {
    Display_drawString( 0, 43, "Time:" + String(RECtime[PLAYpnt]));
    Display_drawString(49, 43, getMinSec(RECtime[PLAYpnt]));
    Display_drawString(10, 51, "At:" + String(PLAYtotal));
    Display_drawString(49, 51, getMinSec(PLAYtotal));
  }

  if (WiFiEn) {Display_setColor(BLUE);} else {Display_setColor(GREY);}
  Display_drawString(  0, 0, "COPY");
  Display_drawString(113, 0, "CLR");
  if (REC) {Display_drawString(19, 35, "Y");} else {Display_drawString(19, 35, "N");}
  if (PLAY) {Display_drawString(54, 35, "Y");} else {Display_drawString(54, 35, "N");}

  Display_setColor(GREEN);
  Display_drawString( 0,19, String(1000/RxPeriod) + "Hz");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 8;        // wait before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Tachos() {
  // Tacho counters display
  DM_Tacho = DispMode;    // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111    2222222222   3333333333   4444444444
  //                     55555 66666  77777 88888
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Tachos");

  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(32, 21, "MtrC:");
  Display_drawString(94, 21, "MtrA:");
  Display_drawString(62, 34, "MtrB:");

  Display_setFont(6);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(32, 20, String(PWM_C));
  Display_drawString(32, 26, String(TachoC));
  Display_drawString(94, 20, String(PWM_A));
  Display_drawString(94, 26, String(TachoA));
  Display_drawString(62, 33, String(PWM_B));
  Display_drawString(62, 39, String(TachoB));

  Display_setFont(10);
  Display_setColor(BLUE);
  Display_drawString( 0, 1, "CLR");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5;
}

// --------------------------------------------------------------------------------

void Display_Test_Battery() {
  // display battery voltage  
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //         1111111111111111      2222222222222222
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 8, "Battery");
  // Y64 = 16,16,16,16
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62, 40, "Batt. =");
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(66, 40, String(BatVfp) + "v");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 20;        // wait 800ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Test_MPU() {
  // display servoAny
  if (!I2C_ICM) {DispModeCnt = 20; return;}
  
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 1111111111111111                3333333333333333
  //                 2222222222222222                4444444444444444
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "MPU");
  // Y64 = 16,16,16,16
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62, 16, "Acc X =");
  Display_drawString(62, 32, "Acc Y =");
  Display_drawString(62, 48, "Gyr Z =");
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(64, 16, String(AcX));
  Display_drawString(64, 32, String(AcY));
  Display_drawString(64, 48, String(GyZ));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 2;        // wait 80ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Test_PWM() {
  // display servoAny
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                     111111111111111111111111
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 6,24,4,24,6
  int16_t zY = 20;
  Display_drawString(64, zY, "PWM " + String(TEST_PWM));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 2;        // wait 80ms before displaying next count
}

// --------------------------------------------------------------------------------

void DisplayText1S24(String zT0) {
  // display one line of text at 24 pt
  if (DispLock) {return;} // prompts are disabled in DispLock mode
  if (DispNOP) {return;}  // this function is aborted if DispNOP == true

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                     111111111111111111111111
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 20,24,20
  Display_drawString(64, 20, zT0);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 25;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void DisplayText2(String zT0,String zT1) {
  // display two lines of text of equal height
  if (DispLock) {return;} // prompts are disabled in DispLock mode
  if (DispNOP) {return;}  // this function is aborted if DispNOP == true

  Display_clear();
  Display_setBrightness(255);   // ensure display is bright
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //               1111111111111111    2222222222222222
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 14, zT0);
  Display_drawString(64, 34, zT1);
  if (Border) {Display_drawRect(0,0,128,64); Display_drawRect(1,1,126,62);} // thick boarder

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 50;       // wait 2 sec before display next screen
}

// --------------------------------------------------------------------------------

void DisplayText2S16(String zT0,String zT1) {
  // display two lines of text, in 16 font
  if (DispLock) {return;} // prompts are disabled in DispLock mode
  if (DispNOP) {return;}  // this function is aborted if DispNOP == true

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //            1111111111111111    2222222222222222
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 11, zT0);
  Display_drawString(64, 31, zT1);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 25;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void DisplayText2S24(String zT0,String zT1) {
  // display two lines of text, 1st line is large
  if (DispLock) {return;} // prompts are disabled in DispLock mode
  if (DispNOP) {return;}  // this function is aborted if DispNOP == true

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //         111111111111111111111111        2222222222222222
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,  8, zT0);
  Display_setFont(16);
  Display_drawString(64, 40, zT1);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 25;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void Display_Tune() {
  // Display the test values
  DM_Tune = DispMode;     // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111 2222222222 3333333333 4444444444 5555555555
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Tune");

  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 62, 18, TuneRef[0]);
  Display_drawString( 62, 29, TuneRef[1]);
  Display_drawString( 62, 40, TuneRef[2]);
  Display_drawString( 62, 51, TuneRef[3]);

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_setColor(BLUE);
  Display_drawString( 63, 18, String(TuneInt[0]));
  Display_drawString( 63, 29, String(TuneInt[1]));
  Display_drawString( 63, 40, String(TuneInt[2]));
  Display_drawString( 63, 51, String(TuneInt[3]));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 6;        // wait 40ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Timing() {
  // display the task values
  DM_Time = DispMode;     // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111 2222222222 3333333333 4444444444 5555555555
  //  66666666
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Timing");

  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 2, 18, "ICM:");
  Display_drawString( 2, 29, "LDR:");
  Display_drawString( 2, 40, "RCWL:");
  Display_drawString( 2, 51, "OLED:");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 48, 18, "t1:");
  Display_drawString( 96, 18, "t2:");
  Display_drawString( 48, 29, "t1:");
  Display_drawString( 96, 29, "t2:");
  Display_drawString( 48, 40, "t1:");
  Display_drawString( 96, 40, "t2:");
  Display_drawString( 48, 51, "t1:");
  Display_drawString( 96, 51, "t2:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 49, 18, String(t1[0]));
  Display_drawString( 97, 18, String(t2[0]));
  Display_drawString( 49, 29, String(t1[1]));
  Display_drawString( 97, 29, String(t2[1]));
  Display_drawString( 49, 40, String(t1[2]));
  Display_drawString( 97, 40, String(t2[2]));
  Display_drawString( 49, 51, String(t1[3]));
  Display_drawString( 97, 51, String(t2[3]));

  Display_setColor(BLUE);
  if (ICM_En)   {Display_drawString(120, 18, "Y");} else {Display_drawString(120, 18, "N");}
  if (LDR_En)   {Display_drawString(120, 29, "Y");} else {Display_drawString(120, 29, "N");}
  if (RCWL_En)  {Display_drawString(120, 40, "Y");} else {Display_drawString(120, 40, "N");}
  if (OLED_NOP) {Display_drawString(120, 51, "N");} else {Display_drawString(120, 51, "Y");}

  Display_setFont(8);
  Display_setColor(GREEN);
  Display_drawString(  0,  1, String(tSch));
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(127,  1, String(tMax));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 6;        // wait 40ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Wii_Ctrl() {
  // Display 6 bytes received from Wii over ESP-NOW.
  // Display I2C values as decimal 0 - 255.
  DM_Wii_Ctrl = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}    // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111122222222223333333333444444444455555555556666666666
  //   7777777777777777  8888888888888888
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_setFont(10); DispCX = 68;
  Display_drawString(DispCX,  2, String(RxWiFi[0]));
  Display_drawString(DispCX, 12, String(RxWiFi[1]));
  Display_drawString(DispCX, 22, String(RxWiFi[2]));
  Display_drawString(DispCX, 32, String(RxWiFi[3]));
  Display_drawString(DispCX, 42, String(RxWiFi[4]));
  Display_drawString(DispCX, 52, String(RxWiFi[5]));
  // display I2C values as binary 00000000 - 11111111
  Display_setTextAlignment(TEXT_ALIGN_LEFT); DispCX = 76;
  Display_drawString(DispCX,  2, GetBIN(RxWiFi[0]));
  Display_drawString(DispCX, 12, GetBIN(RxWiFi[1]));
  Display_drawString(DispCX, 22, GetBIN(RxWiFi[2]));
  Display_drawString(DispCX, 32, GetBIN(RxWiFi[3]));
  Display_drawString(DispCX, 42, GetBIN(RxWiFi[4]));
  Display_drawString(DispCX, 52, GetBIN(RxWiFi[5]));
  // display 'I2C' in top left corner
  Display_setFont(16);
  Display_drawString(1, 2, "Wii");
  Display_setFont(8);
       if (WiiType == 'C') {Display_drawString(1, 18, "Classic");}
  else if (WiiType == 'P') {Display_drawString(1, 18, "Classic Pro");}
  else if (WiiType == 'N') {Display_drawString(1, 18, "Nunchuk");}

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 3;        // wait before displaying next count
}

// --------------------------------------------------------------------------------

void getDispRef() {
  // Called during setup to extract all of the display screen references
  // This calls Disp_40ms multiple times, with DispEnd = false
  // It is called when entering and leaving TEST mode as it has differnt display options
  int16_t zDM = DispMode;         // preserve DispMode
  DispCnt = 0; DispDel = 0; DispClr = false; DispDisp = false; // clear flags
  DispEn = false;                 // disable the generation of display data
  setDMs();                       // set all DM references to their default value -99
  DispMode = 0; DispMon_40ms();   // read the default screen and get DM_Min and DM_Max
  for (DispMode = DM_Min;DispMode <= DM_Max; DispMode++) {
    if (DispMode != 0) {DispMon_40ms();} // extract all of the other references
  }
  DispMode = zDM;                 // restore DispMode
  DispEn = true;                  // enable the generation of display data
  DispDel = 1;
}

// --------------------------------------------------------------------------------
