
// ---------------------------------------------------------------------------------

void AvgICM() {
  // Called as a 4ms subtask, keeps a rolling average of the ICM sensor offsets
  // initially the ICM is not calibrated, so we run a timer before using its outputs
  // as offset values. After that we do a long average, when not moving. This allows
  // for the Omni-Bot moving around on a surface which is not quite level.
  if (GyCal > 0) {
    //######################################
    // Sensor calibration process is running
    //######################################
    GyCal--; 
    AcXSum += AccRawX;
    AcYSum += AccRawY;
    GyZSum += GyrRawZ;
    if (GyAvgDiv < GyAvgDivMax) {
      // increase the scope of the averaging factor if necessary
      // note at 4ms sample, AvgDiv = 250 == 1 second rolling average
      GyAvgDiv++;
    } else {
      // once the accumulator has the correct number of samples roll out the average
      AcXSum -= AcXAvg;
      AcYSum -= AcYAvg;
      GyZSum -= GyZAvg;
    }
    AcXAvg = AcXSum/GyAvgDiv;
    AcYAvg = AcYSum/GyAvgDiv;
    GyZAvg = GyZSum/GyAvgDiv;
    YawGyr = 0.0;         // zero the rotation gyro
  }
  if (GyCal == 1) {
    // calibration period complete, so use offsets determined by averaging
    AcXOff = -AcXAvg; AxCnt = 0; DisX = 0.0;
    AcYOff = -AcYAvg; AyCnt = 0; DisY = 0.0;
    GyZOff = -GyZAvg; GyCnt = 0; YawGyr = 0.0;
  } else {
    //######################################
    // Auto-offset adjustment
    //######################################
    // Perform vey long average when not moving
    if (abs(GyZ) <= GyZnt) {
      // Very small movement on gyro suggests stationary.
      if (ICMove > 0) {ICMove--;}
      else {
        // In this condition we perform long averaging.
        if (AcX > 0) {
          AxCnt++; if (AxCnt > 50) {AxCnt = 0; AcXOff--;}
        }
        else if (AcX < 0) {
          AxCnt--; if (AxCnt < -50) {AxCnt = 0; AcXOff++;}
        }
        if (AcY > 0) {
          AyCnt++; if (AyCnt > 50) {AyCnt = 0; AcYOff--;}
        }
        else if (AcY < 0) {
          AyCnt--; if (AyCnt < -50) {AyCnt = 0; AcYOff++;}
        }
        if (GyZ > 0) {
          GyCnt++; if (GyCnt > 50) {GyCnt = 0; GyZOff--;}
        }
        else if (GyZ < 0) {
          GyCnt--; if (GyCnt < -50) {GyCnt = 0; GyZOff++;}
        }
      }
    } else {
      // Significant changes in GyZ trigger the ICMove flag
      if (abs(GyZ) > (GyZnt + 2)) {ICMove = ICMMoveMax;}
      else if (ICMove == 0) {GyZnt++;} // tweak the threshold if not moving
    }
  }
}

// ---------------------------------------------------------------------------------

void calcSpeed() {
  // Calculates speed and distance from accelerometer values.
  // 1g acceleration == 9.8 m/s/s
  // ICM is set to +/- 2g fsd +32,767 to -32,766
  // Therefore Acc = A * 9.8 /16384 m/s/s =  A * 98/16384 cm/s/s = A * 980/16384 mm/s/s
  // This function only works when the Omni-Bot is thought to be moving.
  if (ICMove == 0) {
    // No movement, so zero the speed values
    VelX = 0.0; VelY = 0.0;
  } else {
    // Calculate speeds in X and Y
    float zTime = 0.000001 * ICMtd; // time since last reading
    float zMult = 0.05981 * zTime;   // (980/16384) * zTime
    VelX += zMult * AcX;
    VelY += zMult * AcY;

    // Calculate incremental distance in mm if velocity >= 0.01mm/s
    if (fabs(VelX) >= 0.01) {DisX += VelX*zTime;}
    if (fabs(VelY) >= 0.01) {DisY += VelY*zTime;}
  }

  // Time the end of all ICM related tasks
  t2[0] = micros() - ICMtr;
}

// ---------------------------------------------------------------------------------

void calcYawAng() {
  // Adds the drift offset to the gyro and calculates change in angle
  if (GyCal > 0) {
    // during calibration gyros are locked to zero
    YawGyr = 0.0;
  } else {
    float zDiv = (float)GyTD/131068000;               // nominally 0.0003051 at 4ms
    YawDif = (float)GyZ * zDiv;
    // Serial.println(String(fabs(RotDif)) + "\t" + String(RotCnt));
    YawGyr += YawDif;
    // If angular movement is significant then zero the YawClr counter
    // If no movement then angle is cleared after 2 seconds
    if (fabs(YawDif) > 0.05) {YawClr = 0;}
    else {YawClr++; if (YawClr >= 500) {YawGyr = 0.0; YawClr = 0;}}
    // limit the Yaw angle
    if (YawGyr > 360.0) {YawGyr -= 360.0;}
    else if (YawGyr < -360.0) {YawGyr += 360.0;}
    YawGyrInt = int(YawGyr);
  }
}

// ---------------------------------------------------------------------------------

void ClearTachos() {
  // Clear the Tacho values
  TachoA = 0;   // motor tracho counter, for motor 'A'
  TachoB = 0;   // motor tracho counter, for motor 'B'
  TachoC = 0;   // motor tracho counter, for motor 'C'
}

// ---------------------------------------------------------------------------------

void decodeKey(int zkeyVal) {
  // Decodes zkeyVal and excutes commands on receipt of '.' characters.
  // Every time we receive a character set response to 10ms later, just in case
  // there is another packet behind it coming from the controller
  next40ms = millis() - 10;
  if (zkeyVal ==  9) {ACK = 0; DispMon_40ms(); return;} // Monitor+ has sent an ACK
  if (zkeyVal == 10) {return;}
  if (zkeyVal == 13) {return;}
  keyChar = char(zkeyVal);
  switch (keyChar) {
    case '.': doCmd(); return;                        // command terminator
    case '!': PWM_OFF(); MotorPnt = 0; return;        // RESET motor drive
    case '#': // if AppCnt > 0, conected to Windows app via USB or WiFi
      if (AppCnt == 0) {OLED_Text2S1610("App","Connected"); OLED_Mode = 1;}
      AppCnt = 500; return;                   
    case '~': // connected to Monitor+ app via USB or WiFi
      if (!DispMon) {DispMon = true; DispNext= DispMode; Display_Mirrored();} // display mirrored message
      DispTx = 50; return;
    case '-': cmdSgn = -1; return;                    // numeric sign
    case '*': cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1; return; // ESC abort command
    case '0': extendCmdVal(0); return;
    case '1': extendCmdVal(1); return;
    case '2': extendCmdVal(2); return;
    case '3': extendCmdVal(3); return;
    case '4': extendCmdVal(4); return;
    case '5': extendCmdVal(5); return;
    case '6': extendCmdVal(6); return;
    case '7': extendCmdVal(7); return;
    case '8': extendCmdVal(8); return;
    case '9': extendCmdVal(9); return;
  }
  if (cmdMode == ' ') {
    // test for new Command Mode char?
    // only accept certain characters as valid commands, or ignore them
    // delay battery update by 10 seconds whilst receiving commands
    cmdMode = keyChar;
    switch (keyChar) {
      // convert lower-case to upper-case, for recognised cmdModes
      case 'd': cmdMode = 'D'; break;
      case 'g': cmdMode = 'G'; break;
      case 'j': cmdMode = 'J'; break;
      case 'p': cmdMode = 'P'; break;
    } cmdType = ' '; cmdVal = 0;
  } else {
    // test for Command Type char?
    cmdType = keyChar;
    switch (keyChar) {
      // check for lower-case and convert to upper-case
      case 'd': cmdType = 'D'; break;
      case 'e': cmdType = 'E'; break;
      case 'l': cmdType = 'L'; break;
      case 'm': cmdType = 'M'; break;
      case 'p': cmdType = 'P'; break;
      case 't': cmdType = 'T'; break;
      case 'v': cmdType = 'V'; break;
      case 'x': cmdType = 'X'; break;
      case 'y': cmdType = 'Y'; break;
    }
  }
}

// --------------------------------------------------------------------------------

void displayLft() {
  // change the DispMode pointer to the previous display
  if (DispLock) {return;} // ignore mouse click when display is locked
  if (DmDwn) {return;}    // Only respond to one mouse down event

  DispMode--; if (DispMode < DM_Min) {DispMode = DM_Max;}
  DispDel = 1; DispCnt = 0; DmDwn = true;
}

// --------------------------------------------------------------------------------

void displayRht() {
  // Change the DispMode pointer to the next display
  if (DispLock) {return;} // ignore mouse click when display is locked
  if (DmDwn) {return;}    // Only respond to one mouse down event

  DispMode++; if (DispMode > DM_Max) {DispMode = DM_Min;}
  DispDel = 1; DispCnt = 0; DmDwn = true;
}

// ---------------------------------------------------------------------------------

void DM_Control_() {
  // Control screen functions
  DispDel = 2;  // Update after 80ms
       if ((DmX >= 49) && (DmX <= 61) && (DmY >= 33) && (DmY <= 44)) {esp_restart();}
  else if ((DmX >= 49) && (DmX <= 58) && (DmY >= 50) && (DmY <= 60)) {BatSum = 0;}
  else if ((DmX >= 49) && (DmX <= 73) && (DmY >= 63) && (DmY <= 74)) {DM_ToggleTEST();}
  else if ((DmX >= 49) && (DmX <= 53) && (DmY >= 83) && (DmY <= 94)) {DM_NewMode();}
  else if ((DmX >= 78) && (DmX <= 83) && (DmY >= 83) && (DmY <= 94)) {DM_NewTask();}
  else if ((DmX >= 86) && (DmX <= 97) && (DmY >= 83) && (DmY <= 94)) {SetMainMode(NewMode);}
  else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_ICM_() {
  // Toggle ICM_En flag
  DispDel = 2;  // Update after 80ms
  if ((DmX >= 9) && (DmX <= 13) && (DmY <= 10)) {
    if (ICM_En) {ICM_En = false;} else {ICM_En = true;}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_ICM_Mov_() {
  // Clears distance variables
  DispDel = 2;  // Update after 80ms
  if (DmY >= 79) {
    if (DmX < 50) {
      if ((DmX >= 21) && (DmX <= 28)) {DisX = 0.0;}
      else {displayLft();}
    } else {
      if ((DmX >= 69) && (DmX <= 85)) {DisY = 0.0;}
      else {displayRht();}
    }
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_ICM_Rot_() {
  // Clear the Yaw gyros value
  DispDel = 2;  // Update after 80ms
  if ((DmX <= 15) && (DmY <= 12)) {YawGyr = 0.0; YawGyrInt = 0; YawTgt = 0.0;}
  else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_LDR_() {
  // Adjust LDR values
  DispDel = 2;  // Update after 80ms
  if ((DmY >= 56) && (DmY <= 65)) {
    // Adjusting Div values
         if ((DmX >= 50) && (DmX < 53)) {DM_LDR_Div_Inc(100);}
    else if ((DmX >= 53) && (DmX < 57)) {DM_LDR_Div_Inc(10);}
    else if ((DmX >= 57) && (DmX < 60)) {DM_LDR_Div_Inc(1);}
    else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if ((DmY >= 86) && (DmY <= 94)) {
    // Adjust Gain value
         if ((DmX >= 50) && (DmX < 53)) {DM_LDR_Ga_Inc(100);}
    else if ((DmX >= 53) && (DmX < 57)) {DM_LDR_Ga_Inc(10);}
    else if ((DmX >= 57) && (DmX < 60)) {DM_LDR_Ga_Inc(1);}
    else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_LDR_Div_Inc(int16_t zV) {
  // Increment/decrement the LDR_Div value
  if (DmB) {LDR_Div -= zV;} else {LDR_Div += zV;}
  if (LDR_Div > 500) {LDR_Div = 500;}
  else if (LDR_Div < 1) {LDR_Div = 1;}
}

// ---------------------------------------------------------------------------------

void DM_LDR_Ga_Inc(int16_t zV) {
  // Increment/decrement the LDR_Ga value
  if (DmB) {LDR_Ga -= zV;} else {LDR_Ga += zV;}
  if (LDR_Ga > 200) {LDR_Ga = 200;}
  else if (LDR_Ga < 1) {LDR_Ga = 1;}
}

// ---------------------------------------------------------------------------------

void DM_NewMode() {
  // Display click changes selected NewMode
  if (DmB) {NewMode--;} else {NewMode++;}
  NewTask = 0;
  if (NewMode < -1) {NewMode = -1;}
}

// ---------------------------------------------------------------------------------

void DM_NewTask() {
  // Display click changes selected NewTask
  if (DmB) {NewTask--;} else {NewTask++;}
  if (NewTask < 0) {NewTask = 0;}
}

// ---------------------------------------------------------------------------------

void DM_Range_() {
  // Toggle Range enable
  DispDel = 2;  // Update after 80ms
  if (DmY <= 16) {
    // Set modes and toggle enable
         if ((DmX >= 9) && (DmX <= 12)) {DM_Range_Mode();}
    else if (DmX >= 93) {RCWL_En = !RCWL_En;}
    else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if ((DmY >= 76) && (DmY <= 86)) {
    // Rate adjust
         if ((DmX >= 16) && (DmX <= 22)) {DM_RangeEx_Inc();}    // Exclusion count
    else if ((DmX >= 46) && (DmX <= 70)) {DM_Range_Inc();}      // Measurement period
    else if ((DmX >= 92) && (DmX <= 98)) {DM_RangeDiv_Inc();}   // Averaging divider
    else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_RangeDiv_Inc() {
  // Increment/decrement the RangeDiv value
  int16_t zInc = 1; if (DmSft) {zInc = 10;}
  if (DmB) {RangeDiv-= zInc;} else {RangeDiv+= zInc;}
  if (RangeDiv > 50) {RangeDiv = 50;}
  else if (RangeDiv < 1) {RangeDiv = 1;}
}

// ---------------------------------------------------------------------------------

void DM_RangeEx_Inc() {
  // Increment/decrement the RangeEx value
  int16_t zInc = 1; if (DmSft) {zInc = 10;}
  if (DmB) {RangeEx-= zInc;} else {RangeEx+= zInc;}
  if (RangeEx > 20) {RangeEx = 20;}
  else if (RangeEx < 1) {RangeEx = 0;}
}

// ---------------------------------------------------------------------------------

void DM_Range_Inc() {
  // Increment/decrement the RCWL_Cnt value
  int16_t zInc = 1; if (DmSft) {zInc = 10;}
  if (DmB) {RCWL_Max+= zInc;} else {RCWL_Max-= zInc;}
  if (RCWL_Max > 100) {RCWL_Max = 100;}
  else if (RCWL_Max < 4) {RCWL_Cnt = 4;}
}

// ---------------------------------------------------------------------------------

void DM_Range_Mode() {
  // Change the RCWL mode value
  // The order in which sensors are read is determined by RCWL_Mode:
  // 0        - read front sensor only
  // 1        - read right-hand sensor only
  // 2        - read left-hand sensor only
  // 3/4/5    - read sensors in sequence, front, right, left, front,...
  // 6/7/8/9  -  read sensors in sequence, front, right, left, front,...
  if (!DmB) {
    // Left click, so increase
         if (RCWL_Mode <= 2) {RCWL_Mode++;}
    else if (RCWL_Mode <= 5) {RCWL_Mode = 6;}
    else {RCWL_Mode = 0;}
  } else {
    // Right click, so decrease
         if (RCWL_Mode >= 6) {RCWL_Mode = 3;}
    else if (RCWL_Mode >= 3) {RCWL_Mode = 2;}
    else if (RCWL_Mode >= 1) {RCWL_Mode--;}
    else {RCWL_Mode = 6;}
  }
  SetRcwlMode(RCWL_Mode);
}

// ---------------------------------------------------------------------------------

void DM_Recall_() {
  // This is a PLAY#### function.
  // Toggle Recall flags, when in WiFi mode
  DispDel = 2;  // Update after 80ms
  if (WiFiEn) {
    if (DmY <= 16) {
      // COPY and CLR functions
           if (DmX <= 16) {PLAY_Upload();}
      else if (DmX >= 87) {PLAY_STOP(); REC_CLR();}
      else if (DmX < 50) {displayLft();} else {displayRht();}
    } else if ((DmY >= 54) && (DmY <= 62)) {
      // RECall and PLAY enables
          if ((DmX >= 15) && (DmX <= 19)) {REC_Toggle();}     // Toggle REC flag
      else if ((DmX >= 42) && (DmX <= 46)) {PLAY_Toggle();}    // Toggle PLAY flag
      else if (DmX < 50) {displayLft();} else {displayRht();}
    } else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_Tacho_() {
  // Clear Tachos values
  DispDel = 2;  // Update after 80ms
  if ((DmX <= 15) && (DmY <= 12)) {ClearTachos();}
  else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_Time_() {
  // Toggle enable flags values
  DispDel = 2;  // Update after 80ms
  if ((DmX >= 93) && (DmY >= 29)) {
         if (DmY < 42) {if (ICM_En) {ICM_En = false;} else {ICM_En = true;}}
    else if (DmY < 59) {if (LDR_En) {LDR_En = false;} else {LDR_En = true;}}
    else if (DmY < 75) {if (RCWL_En) {RCWL_En = false;} else {RCWL_En = true;}}
                  else {if (OLED_NOP) {OLED_NOP = false;} else {OLED_NOP = true;}}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void DM_ToggleTEST() {
  // Toggle TEST mode
  if (TEST) {TEST = false;} else {TEST = true;}
  LED_Task = 0;   // reset LED task
}

// ---------------------------------------------------------------------------------

void DM_Tune_() {
  // Adjust TuneInt[] values
  DispDel = 2;  // Update after 80ms
  int16_t zInc = 1; if (DmSft) {zInc = 10;}
  if ((DmX >= 49) && (DmX <= 65)) {
    // TuneInt[0]
    if ((DmY >= 31) && (DmY <= 42)) {
      if (DmB) {TuneInt[0]-= zInc;} else {TuneInt[0]+= zInc;}
    }
    // TuneInt[1]
    else if ((DmY >= 49) && (DmY <= 59)) {
      if (DmB) {TuneInt[1]-= zInc;} else {TuneInt[1]+= zInc;}
    }
    // TuneInt[2]
    else if ((DmY >= 65) && (DmY <= 76)) {
      if (DmB) {TuneInt[2]-= zInc;} else {TuneInt[2]+= zInc;}
    }
    // TuneInt[3]
    else if ((DmY >= 82) && (DmY <= 92)) {
      if (DmB) {TuneInt[3]-=zInc;} else {TuneInt[3]+=zInc;}
    }
    else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
}

// ---------------------------------------------------------------------------------

void doCmd() {
  // a '.' has been received on the serial port so execute command if valid
  // Commands:
  // DDn.     - increase (1) or decrease (-1) DispDel global delay offset DispDelGbl
  // DLn.     - toggle display lock, n=0 OFF, n=1 ON
  // DMxxyy.  - display monitor left mouse click, xx = 0-99, yy = 0-99
  // DM-.     - display monitor mouse up event
  // DS.      - SHIFT-key added to DMxxyy. message
  // G.       - report Scope data for graphing app
  // GE.      - end of Scope graphing mode
  // GPnn.    - nn points to data to be sent
  // JXnn.    - set global variable JX to nn
  // JYnn.    - set global variable JY to nn
  // PTnn.    - set target wheel PWM pointer
  // PVnn.    - set PWM value

  cmdVal*= cmdSgn;  // correct cmdVal with the sign value
  
  switch (cmdMode) {
    case ' ': break;
    case 'D':
      switch (cmdType) {
        case 'D': // DDn. command, adjust global delay offset
          if (cmdVal > 0) {DispDelGbl ++;} else {DispDelGbl --;}
          if (DispDelGbl < 0) {DispDelGbl = 0;}
          // display this in the Rx field of Monitor+
          PrintTx+= "$DispDelGbl: " + String(DispDelGbl) +"\n";
          break;
        case 'L':
          if (cmdVal != 0) {DispLock = true; DispOveride = DispMode;}
          else {DispLock = false;}
          break;
        case 'M':
          if (cmdSgn < 0) {  // mouse click released
            DmDwn = false;
            // Serial.println("DwDwn=false");
          } else {
            if (cmdVal >= 10000) {
              // this is a Display Monitor mouse right-click
              // determine whether it is a left or right button click
              cmdVal /= 10000; cmdVal --; DmB = 1;
              DmY = cmdVal % 100; DmX = cmdVal/100;
            } else {
              // this is a Display Monitor mouse left-click
              DmY = cmdVal % 100; DmX = cmdVal/100; DmB = 0;
            }
            if (!DmDwn) {
                   if (DispMode == DM_Control) {DM_Control_();}
              else if (DispMode == DM_ICM) {DM_ICM_();}
              else if (DispMode == DM_ICM_Mov) {DM_ICM_Mov_();}
              else if (DispMode == DM_ICM_Rot) {DM_ICM_Rot_();}
              else if (DispMode == DM_LDR) {DM_LDR_();}
              else if (DispMode == DM_Range) {DM_Range_();}
              else if (DispMode == DM_Recall) {DM_Recall_();}
              else if (DispMode == DM_Tacho) {DM_Tacho_();}
              else if (DispMode == DM_Time) {DM_Time_();}
              else if (DispMode == DM_Tune) {DM_Tune_();}
              else {
                // this is not a special 'clickable' display, so simply change it
                if (DmX < 50) {displayLft();  // left side click
                } else {displayRht();}       // right side click
              }
            }
            // Serial.println("DwDwn=true");
          } DmSft = false; break;
        case 'S': // embedded SHIFT key
          DmSft = true; break;
      } break;
    case 'G':
      // Scope graphing app report commands
      switch (cmdType) {
        case ' ': 
          // Report the same set of values twice as WiFi data can easily be
          // corrupted. Therefore two matching consequitives messages must be
          // received for the graphing function to work
          // if in Display Monitor mode then switch that off
          if (DispMon) {DispMon = false; PrintTx = ""; DispTx = -1;}
          // RPms = millis() - RPt0; RPt0 = millis();
          ScopeReport(GPnn); ScopeReport(GPnn);
          break;
        case 'E': // GE. enter monitor mode when dropping out of graphing mode
          // return to Display Monitor mode if it was previously active
          if (DispTx < 0) {DispMon = true; PrintTx = ""; DispTx = 50; DispDel = 0; DispCnt = 0;}
          break;
        case 'P': // GPnn. sets the reporting mode and triggers title reporting
          if (DispMon) {DispMon = false; PrintTx = ""; DispTx = -1;}
          GPnn = cmdVal; 
          if (G_once) {G_once = false; GPnn = 0;}
          GP_Title = true;
          PrintTx += "GP" + String(GPnn) + "\n";
          break;
      } break;
    case 'J':
      // commands used to set global variables in association with Joystick app
      switch (cmdType) {
        case 'X': // set global variable JX to cmdVal
          JX = cmdVal; //Serial.println("JX=" + String(JX));
          break;
        case 'Y': // set global variable JY to cmdVal
          JY = cmdVal; //Serial.println("JY=" + String(JY));
          break;
      } break;
    case 'P':
      // keyboard/serial commands used to set target wheel PWM pointer
      switch (cmdType) {
        case 'T': // set target servo pointer in range 0 - 5
          if (cmdVal < 0) {cmdVal = MotorPnt;}
          if (cmdVal > 5) {cmdVal = MotorPnt;}
          MotorPnt = cmdVal; break;
        case 'V': // set wheel PWM values 0 - 255
               if (MotorPnt ==  0) {analogWrite(Pin_A0, cmdVal);}
          else if (MotorPnt ==  1) {analogWrite(Pin_A1, cmdVal);}
          else if (MotorPnt ==  2) {analogWrite(Pin_B0, cmdVal);}
          else if (MotorPnt ==  3) {analogWrite(Pin_B1, cmdVal);}
          else if (MotorPnt ==  4) {analogWrite(Pin_C0, cmdVal);}
          else if (MotorPnt ==  5) {analogWrite(Pin_C1, cmdVal);}
          break;
      } break;
    default:
      PrintTx += "Unknown cmd: " + cmdMode + cmdType + String(cmdVal) + "\n";
      break;
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1;
}

// ---------------------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = abs(cmdVal * 10) + zVal;
}

// --------------------------------------------------------------------------------

String get3Char(int16_t zV) {
  // returns a 3 character representation of zV, with leading 0's
  String zRet = String(zV);
  if (zV <  10) {zRet = "0" + zRet;}
  if (zV < 100) {zRet = "0" + zRet;}
  return zRet;
}

// // --------------------------------------------------------------------------------

String getAtState() {
  // returns the AtState value as a string
  String zT$ = "";
  Any$ = "???";
  switch (AtState) {
    case At_REST:
      GAScnt++; if (GAScnt > 2) {GAScnt = 0;}
      switch (GAScnt) {
        case 0: zT$ = "..z  "; break;
        case 1: zT$ = ".zz "; break;
        case 2: zT$ = "zzZ"; break;
      } Any$ = "Resting " + zT$; break;
    case At_DRIVE:  Any$ = "Drive Mode"; break;
    case At_LDR:
      switch(MainTask) {
        case 10: Any$ = "LDR Sense"; break;
        case 11: Any$ = "LDR Track"; break;
        case 12: Any$ = "LDR Follow"; break;
       } break;
    case At_OMNI:   Any$ = "Omni Mode"; break;
    case At_READY:  Any$ = "I'm Ready"; break;
    case At_SONAR:
      switch(MainTask) {
        case 20: Any$ = "Sonar Sense"; break;
        case 21: Any$ = "Sonar Backaway"; break;
        case 22: Any$ = "Sonar Track"; break;
       } break;
    // case CAL:   Any$ = "READY!"; break;
    // case READY:   Any$ = "READY!"; break;
    // case RESET:   Any$ = "RESET..."; break;
  }
  return Any$;
}

// --------------------------------------------------------------------------------

String getBAhex() {
  // Rerturns the current Broardcast Address MAC, ie: 50:02:91:68:F7:3F
  String zRet$ = ""; uint8_t zV;
  for (int zP = 0;zP < 6;zP++) {
    zV = broadcastAddress[zP]/16;
    if (zV < 10) {zRet$ += char(zV + 48);} else {zRet$ += char(zV + 55);}
    zV = broadcastAddress[zP] - (zV *16);
    if (zV < 10) {zRet$ += char(zV + 48);} else {zRet$ += char(zV + 55);}
    if (zP < 5) {zRet$ += ":";}
  }
  return zRet$;
}

// --------------------------------------------------------------------------------

int16_t getBatV(int32_t zAvg) {
  // Determine the battery voltage, and percentage, by mapping the ADC readings onto threshold Voltages
  // These threshold voltages were determined using a multimeter, to calibrate the ESP32's ADC
  // if the voltage exceeds BatMax, the estimated voltage is displayed, but battery capacity will be limited to 100%
       if (zAvg > Bat8v2)  {BatV = map(zAvg,Bat7v6,Bat8v4,760,840); BatPc = 100;} // >= 100%
  else if (zAvg > Bat7v6) {BatV = map(zAvg,Bat7v6,Bat8v2,760,820); BatPc = map(zAvg,Bat7v6,Bat8v2,80,100);}  // >= 80%
  else if (zAvg > Bat7v2) {BatV = map(zAvg,Bat7v2,Bat7v6,720,760); BatPc = map(zAvg,Bat7v2,Bat7v6,20, 80);}  // >= 20%
  else {BatV = map(zAvg,Bat6v6,Bat7v2,660,720); BatPc = map(zAvg,Bat6v6,Bat7v2,0,20);}  // < 20%
  if (BatPc < 0) {BatPc = 0;}   // Ignore < 0 percentages
  return BatV;
}

// --------------------------------------------------------------------------------

String GetBIN(byte zVal) {
  // returns an 8 character binary representation of zVal
  String zBIN$ = "";
  // convert zVal into binary
  if (zVal & 0B10000000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B01000000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00100000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00010000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00001000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000100) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000010) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000001) {zBIN$ += "1";} else {zBIN$ += "0";}
  return zBIN$;
}

// --------------------------------------------------------------------------------

String getHmsTime(unsigned long zMillis) {
  // returns a 'time' string in Hrs:Min:sec base on zMillis
  String zTime$ = "";
  long zU = zMillis/3600000; zMillis = zMillis%3600000;
  if (zU > 0) {zTime$ = String(zU) + ":";}
  zU = zMillis/60000; zMillis = zMillis%60000;
  if (zU < 10) {zTime$ += "0";}
  zTime$ += String(zU);
  zU = zMillis/1000;
  if (zU < 10) {zTime$ += ":0";} else {zTime$ += ":";}
  zTime$ += String(zU);
  return zTime$;
}

// --------------------------------------------------------------------------------

String getListLineNum(int16_t zL) {
  // Returns a number as a 4-digit string
  String zT$ = String(zL);
       if (zL <   10) {zT$ = "000" + zT$;}
  else if (zL <  100) {zT$ = "00" + zT$;}
  else if (zL < 1000) {zT$ = "0" + zT$;}
  return zT$;
}

// --------------------------------------------------------------------------------

String getMinSec(int32_t zTick) {
  // Used by the RECall function, to convert RECtime[RECpnt] into min:sec
  // zTick are in 40ms units
  String zT = "";
  int16_t zMin = zTick/1500;
  if (zMin > 0) {zTick -= (zMin * 1500);}
  int16_t zSec = zTick/25;
  if (zMin < 10) {zT = "0" + String(zMin);} else {zT = String(zMin);}
  zT+= ":";
  if (zSec < 10) {zT+= "0" + String(zSec);} else {zT+= String(zSec);}
  return zT;
}

// --------------------------------------------------------------------------------

void GetRndCol(byte zB) {
  // return a random colour of strength zB
  if (zB < 2) {zB = 2;}
  switch(random(6)) {
    case 0: Col_R =   zB; Col_G =    0; Col_B =    0; break;  // red
    case 1: Col_R = zB/2; Col_G = zB/2; Col_B =    0; break;  // yellow
    case 2: Col_R =    0; Col_G =   zB; Col_B =    0; break;  // green
    case 3: Col_R =    0; Col_G = zB/2; Col_B = zB/2; break;  // cyan
    case 4: Col_R =    0; Col_G =    0; Col_B =   zB; break;  // blue
    case 5: Col_R = zB/2; Col_G =    0; Col_B = zB/2; break;  // magenta
  }
}

// --------------------------------------------------------------------------------

void ICM_42607_Initialise() {
  // set up gyro hardware
  // By default the MPU-6050 sleeps. So we have to wake it up.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search.
  Wire.write(0x1F);                     // We want to write to the PWR_MGMT_0 register (1F hex)
  Wire.write(0b00001111);               // Turn ON gyros and accelerometers in Low Noise mode
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(200);               // Allow ICM time to turn ON sensors

  // Set the full scale of the gyro to +/- 250 degrees per second, ODR = 800Hz
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search.
  Wire.write(0x20);                     // We want to write to the GYRO_CONFIG register (20 hex)
  Wire.write(0b01100110);               // Set the gyro bits as +/-250dps full scale, 800Hz ODR
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);            // Allow ICM time to respond

  //Set the full scale of the accelerometer to +/-2g. ODR = 800Hz
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search.
  Wire.write(0x21);                     // We want to write to the ACCEL_CONFIG register (21 hex)
  // Wire.write(0b01000110);               // Set the accel bits as +/-4g full scale range, 800Hz ODR
  Wire.write(0b11000110);               // Set the accel bits as +/-2g full scale range, 800Hz ODR
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);            // Allow ICM time to respond

  //Set gyro filtering to improve the raw data.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search
  Wire.write(0x23);                     // We want to write to the CONFIG register (23 hex)
  Wire.write(0b00000100);               // Set Digital Low Pass Filter to ~53Hz
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);            // Allow ICM time to respond

  //Set accelerometer filtering to improve the raw data.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search
  Wire.write(0x24);                     // We want to write to the CONFIG register (24 hex)
  Wire.write(0b00000100);               // Set 2x averaging and Low Pass Filter to ~53Hz
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);            // Allow ICM time to respond

  // Bypass FIFO mode
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search
  Wire.write(0x28);                     // We want to write to the CONFIG register (28 hex)
  Wire.write(0b00000001);               // Set FIFO to bypassed
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);            // Allow ICM time to respond

  // APEX config1
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search
  Wire.write(0x26);                     // We want to write to the CONFIG register (26 hex)
  Wire.write(0b00000000);               // All motion detection disabled, ODR = 25Hz
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);            // Allow ICM time to respond
}

// --------------------------------------------------------------------------------

void initBatteryRead() {
  // called during runPOST() or on a soft reset
  // take 100 battery readings over a 100ms period to get an average
  BatVol = 0;
  for (int zI = 0;zI < 100;zI++) {BatVol+= analogRead(BatPin); delay(1);}
  BatVol/= 100;

  BatAvg = BatVol; BatSum = BatVol * 50;  // preload the battery sum for averaging
  BatV = getBatV(BatVol); BatVfp = (float)BatV/100.0;
  Serial.println("Battery " + String(BatVol) + "\t" + String(BatVfp) + "v\t" + String(BatPc) + "%");
  randomSeed(BatVol); // use the battery voltage as the random seed factor
  // if battery is less than 5.5v then USB mode is assumed, which makes servos inoperable
  if (BatAvg <= BatUSB) {USB = true; Serial.println(F("USB power assumed."));}
  // Convert BatAvg to a mapped voltage and floating point equivalent
  BatV = getBatV(BatAvg); BatVfp = (float)BatV/10.0;
}

// --------------------------------------------------------------------------------

void Init_ESP_NOW() {
  // called to initiate an ESEP-NOW link or connect to another trannsceiver
  // each time this function is called it will attempt to connect to a different
  // boradcast address, cycling through them, over and over
  //  Serial.println("Init_ESP_NOW()");

  // if we have previously tried a connection then we need to de-initialise it
  if (ESP_NOW_Init) {
    // already attempted to initialise ESP-NOW so strip everything back
    // Serial.println("Closing ESP-NOW");
    esp_now_del_peer(broadcastAddress);
    // WiFi.disconnect();
  }
  
  // Init ESP-NOW and returns its status
  if (esp_now_init() != ESP_OK) {
    Serial.println("Error initializing ESP-NOW");
    return;
  }
  ESP_NOW_Init = true;
  
  // change the broadcast address pointer each time
  ESP_NOW_BA++;
  //Serial.println("ESP_NOW_BA = " + String(ESP_NOW_BA));
  
  switch (ESP_NOW_BA) {
    case 0: // Wii Transciever unique MAC: 50:02:91:68:F7:3F
      broadcastAddress[0] = 0x50;
      broadcastAddress[1] = 0x02;
      broadcastAddress[2] = 0x91;
      broadcastAddress[3] = 0x68;
      broadcastAddress[4] = 0xF7;
      broadcastAddress[5] = 0x3F;
      break;
    case 1: // Wii Transciever unique MAC: 50:02:91:68:F0:D2
      broadcastAddress[0] = 0x50;
      broadcastAddress[1] = 0x02;
      broadcastAddress[2] = 0x91;
      broadcastAddress[3] = 0x68;
      broadcastAddress[4] = 0xF0;
      broadcastAddress[5] = 0xD2;
      break;
    case 2: // Wii D1 Mk1 Transciever unique MAC: 58:BF:25:DB:27:A2
      broadcastAddress[0] = 0x58;
      broadcastAddress[1] = 0xBF;
      broadcastAddress[2] = 0x25;
      broadcastAddress[3] = 0xDB;
      broadcastAddress[4] = 0x27;
      broadcastAddress[5] = 0xA2;
      break;
    case 3: // Wii ESP32-C3 Zero Transciever unique MAC: EC:DA:3B:BD:4B:AC
      broadcastAddress[0] = 0xEC;
      broadcastAddress[1] = 0xDA;
      broadcastAddress[2] = 0x3B;
      broadcastAddress[3] = 0xBD;
      broadcastAddress[4] = 0x4B;
      broadcastAddress[5] = 0xAC;
      break;
    default:
      // End of list reached, so extend pause time and reset cycle
      // Turn off serial port reporting after first pass, unless connection fails
      ESP_NOW_BA = -1;      // 1st case will be 0
      WiFiConCnt = 75;      // 3 sec delay between trying cycles
      WiFiTryOnce = false;  // don't report trying after one cycle
      break;
  }

  // Once ESPNow is successfully Init, we will register for Send CB to
  // get the status of Trasnmitted packet
  esp_now_register_send_cb((esp_now_send_cb_t)OnDataSent);

  // Register peer
  esp_now_peer_info_t peerInfo; //initialize and assign the peer information as a pointer to an addres
  memset(&peerInfo, 0, sizeof(peerInfo));
  memcpy(peerInfo.peer_addr, broadcastAddress, 6); //copy the value of  broadcastAddress with 6 bytes to peerInfo.peer_addr
  peerInfo.channel = 0;     //channel at which the esp talk. 0 means undefined and data will be sent on the current channel. 1-14 are valid channels which is the same with the local device 
  peerInfo.encrypt = false; //not encrypted

  //Add the device to the paired device list 
  if (esp_now_add_peer(&peerInfo) != ESP_OK){ 
    Serial.println("Failed to add peer");
    return;
  }

  // Register for a callback function that will be called when data is received
  esp_now_register_recv_cb(OnDataRecv);
}

// --------------------------------------------------------------------------------

void LED_Blue_Rot() {
  // Rotating Blue band
  switch (LED_Task) {
    case 0: // set initial state
      LED_Fill(0,0,0);
      LED[ 0].setRGB(0,0,0); LED[ 1].setRGB(0,0,1); LED[ 2].setRGB(0,0,2); LED[ 3].setRGB(0,0,4); LED[ 4].setRGB(0,0,16);
      LED[ 5].setRGB(0,0,0); LED[ 6].setRGB(0,0,1); LED[ 7].setRGB(0,0,2); LED[ 8].setRGB(0,0,4); LED[ 9].setRGB(0,0,16);
      LED[10].setRGB(0,0,0); LED[11].setRGB(0,0,1); LED[12].setRGB(0,0,2); LED[13].setRGB(0,0,4); LED[14].setRGB(0,0,16);
      LED_Task++; break;
    case 1: // rotate LEDs
      LED_Rot_Rht(); break;
  }
  LED_Del = 6; LED_Run = false; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Col_Ring() {
  // Display a coloured ring, set by Col_B,Col_G,Col_R
  // Then branch to another LED function if LedNext >= 0
  LED_Fill(Col_R,Col_G,Col_B);
  if (LedNext >= 0) {SetLedMode(LedNext);}
  LED_Del = 20; LED_Run = false; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_DriveAuto() {
  // LED drive patterns are derived from motor drive values
  int16_t zLED = 2;
       if ((PWM_A == 0) && (PWM_B == 0) && (PWM_C == 0)) {zLED = 2;}   // stationary
  else if ((PWM_C > 0) && (PWM_A < 0) && (PWM_B == 0)) {zLED = 21;}    // forwards
  else if ((PWM_C < 0) && (PWM_A > 0) && (PWM_B == 0)) {zLED = 25;}    // backwards
  else if ((PWM_C > 0) && (PWM_A > 0) && (PWM_B > 0)) {zLED = 23;}     // neutral right
  else if ((PWM_C < 0) && (PWM_A < 0) && (PWM_B < 0)) {zLED = 27;}     // neutral left

  if (zLED != LedNext) {
    LedNext = zLED;
    // Change of mode, so reset the flags and set the new mode
    LED_Cnt = 0;
    LED_Del = 0;
    LED_Task = 0;
    LED_Period = 20;
  }
  switch (LedNext) {
    case  2: LED_Ready(); break;
    case 21: LED_DriveFwd(); break;
    case 23: LED_NeutRht(); break;
    case 25: LED_DriveBck(); break;
    case 27: LED_NeutLft(); break;
  }
}

// --------------------------------------------------------------------------------

void LED_DriveBck() {
  // generates a driving backward pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 5].setRGB(16,16,0); LED[13].setRGB(16,16,0); // yellow
      LED[ 6].setRGB(0,0,16); LED[12].setRGB(0,0,16); // bright blue
      LED[ 7].setRGB(0,0,4); LED[11].setRGB(0,0,4); // mid blue
      LED[ 8].setRGB(0,0,1); LED[10].setRGB(0,0,1); // min blue
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[5]; // grab the last body LED rgb values
        LED[ 5] = LED[ 6]; LED[ 6] = LED[ 7]; LED[ 7] = LED[ 8]; LED[ 8] = LED[ 9]; LED[ 9] = LEDX[0];
        LED[14] = LED[ 5]; LED[13] = LED[ 6]; LED[12] = LED[ 7]; LED[11] = LED[ 8]; LED[10] = LED[ 9];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the red eyes
    LED[0].setRGB( 0,0,0); LED[2] = LED[0]; LED[4] = LED[0];
    LED[1].setRGB(32,0,0); LED[3] = LED[1];
  }
}

// --------------------------------------------------------------------------------

void LED_DriveFwd() {
  // generates a driving forward pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 9].setRGB(16,16,0); LED[10].setRGB(16,16,0); // yellow
      LED[ 8].setRGB(0,0,16); LED[11].setRGB(0,0,16); // bright blue
      LED[ 7].setRGB(0,0,4); LED[12].setRGB(0,0,4); // mid blue
      LED[ 6].setRGB(0,0,1); LED[13].setRGB(0,0,1); // min blue
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[9]; // grab the last body LED rgb values
        LED[ 9] = LED[ 8]; LED[ 8] = LED[ 7]; LED[ 7] = LED[ 6]; LED[ 6] = LED[ 5]; LED[ 5] = LEDX[0];
        LED[10] = LED[ 9]; LED[11] = LED[ 8]; LED[12] = LED[ 7]; LED[13] = LED[ 6]; LED[14] = LED[ 5];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the red eyes
    LED[0].setRGB( 0,0,0); LED[2] = LED[0]; LED[4] = LED[0];
    LED[1].setRGB(32,0,0); LED[3] = LED[1];
  }
}

// --------------------------------------------------------------------------------

void LED_Eyes() {
  // Two red eyes at the front of the robot
  // When moving eyes are brighter
  LED_Fill(0,0,0); LED_Run = false; LEDshow = true;
  // Blink occasionally
  if (0 == random(10)) {return;}
  // Now change eye positions
  int16_t zTask = random(3);
  TryAgain: if (zTask == LED_Task) {goto TryAgain;}
  if (LED_Task > zTask) {LED_Task--;} else {LED_Task++;}
  LEDX[0].setRGB(16,0,0);
  if (REC) {LEDX[0].setRGB(16,0,16);}
  switch (LED_Task) {
    case 0: // look left
      LED[0] = LEDX[0]; LED[2] = LEDX[0]; break;
    case 1: // look forward
      LED[1] = LEDX[0]; LED[3] = LEDX[0]; break;
    case 2: // look right
      LED[2] = LEDX[0]; LED[4] = LEDX[0]; break;
  }
  LED_Del = 20 + random(20);
}

// --------------------------------------------------------------------------------

void LED_Fill(uint8_t zR,uint8_t zG,uint8_t zB) {
  // Fill the LED array with a specific colour
  for (int zL = 0;zL < NumLEDs;zL++) {LED[zL].setRGB(zR,zG,zB);}
}

// --------------------------------------------------------------------------------

void LED_Green_Rot() {
  // Rotating Green band. But in TEST mode change to blue.
  switch (LED_Task) {
    case 0: // set initial state
      LED_Fill(0,0,0);
      if (!TEST) {LED[ 0].setRGB(0, 0,0); LED[ 1].setRGB(0,1,0); LED[ 2].setRGB(0,2,0); LED[ 3].setRGB(0,4,0); LED[ 4].setRGB(0,16, 0);}
            else {LED[ 0].setRGB(0, 0,0); LED[ 1].setRGB(0,0,1); LED[ 2].setRGB(0,0,2); LED[ 3].setRGB(0,0,4); LED[ 4].setRGB(0, 0,16);}
      LED[ 5] = LED[0]; LED[ 6] = LED[1]; LED[ 7] = LED[2]; LED[ 8] = LED[3]; LED[ 9] = LED[4];
      LED[10] = LED[0]; LED[11] = LED[1]; LED[12] = LED[2]; LED[13] = LED[3]; LED[14] = LED[4];
      LED_Task++; break;
    case 1: // rotate LEDs
      LED_Rot_Rht(); break;
  }
  LED_Del = 6; LED_Run = false; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Main() {
  // tasks for manipulating RGB LED patterns
  // Blink clock and modes
  BlinkCnt++; if (BlinkCnt > 30) {BlinkCnt = 0; Blink = true;}
  // Button switches over-ride current mode
  if ((sw0State == LOW) || (sw1State == LOW)) {
    if (!LedMem) {
      SaveLEDs();
      if (sw0State == LOW) {LED_SW0();}  // force left side LEDs
      if (sw1State == LOW) {LED_SW1();}  // force right side LEDs
      LED_Run = false; LEDshow = true; LED_Del = 1;
    } return;
  }
  if (LedMem) {Recall_LEDs();}
  if (LED_Del > 0) {LED_Del--; LED_Run = false; return;} // delay by skipping tasks

  //  Serial.println(LedMode);
  // When performing drive tasks vary rate with LedSpd
  if (LedMode >= 21) {LED_Period = map(LedSpd,PWM_StartMin,255,20,10);}
  
  switch(LedMode) {
    case 0: // doing nothing
      LED_Green_Rot(); break;
    case 1: // awake, moving eyes
      LED_Eyes(); break;
    case 2: // ready, waiting for action
      LED_Ready(); break;
    case 3: // Sonar mode
      LED_Sonar(); break;
    case 4: // Blue rotate
      LED_Blue_Rot(); break;
    case 5: // Coloured ring
      LED_Col_Ring(); break;
    case 10: // random LED colours
      GetRndCol(16); LED[random(15)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(32); LED[random(15)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(64); LED[random(15)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(16); LED[random(15)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(32); LED[random(15)].setRGB(Col_R,Col_G,Col_B);
      GetRndCol(64); LED[random(15)].setRGB(Col_R,Col_G,Col_B);
      LED_Run = false; LEDshow = true; LED_Del = 0;
      break;
    case 12: // Pulsing red ring
      LED_Red_Pulse(); break;
    case 13: // Three red pulses
      LED_Red_Warning(); break;

    // The following tasks, 20 onwards, are all drive tasks
    case 20: // auton driving
      LED_DriveAuto(); break;
    // Normal driving
    case 21: // driving forward
      LED_DriveFwd(); break;
    case 23: // driving neutral turn right
      LED_NeutRht(); break;
    case 25: // driving backward
      LED_DriveBck(); break;
    case 27: // driving neutral turn left
      LED_NeutLft(); break;

    // Omni-wheel driving
    case 31: // driving forward
      LED_Omni_Fwd(); break;
    case 32: // driving forward NNE
      LED_Omni_NNE(); break;
    case 33: // driving forward ENE
      LED_Omni_ENE(); break;
    case 34: // driving sideways to the right
      LED_Omni_Rht(); break;
    case 35: // driving backward ESE
      LED_Omni_ESE(); break;
    case 36: // driving backward SSE
      LED_Omni_SSE(); break;
    case 37: // driving backward
      LED_Omni_Bck(); break;
    case 38: // driving backward SSW
      LED_Omni_SSW(); break;
    case 39: // driving backward WSW
      LED_Omni_WSW(); break;
    case 40: // driving sideways to the left
      LED_Omni_Lft(); break;
    case 41: // driving forward WNW
      LED_Omni_WNW(); break;
    case 42: // driving forward NNW
      LED_Omni_NNW(); break;
  }
}

// --------------------------------------------------------------------------------

void LED_NeutLft() {
  // generates a neutral turn leftt pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 5].setRGB(16,16,0); LED[10].setRGB(16,16,0); // yellow
      LED[ 6].setRGB(0,0,16); LED[11].setRGB(0,0,16); // bright blue
      LED[ 7].setRGB(0,0,4); LED[12].setRGB(0,0,4); // mid blue
      LED[ 8].setRGB(0,0,1); LED[13].setRGB(0,0,1); // min blue
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[5]; // grab the last body LED rgb values
        LED[ 5] = LED[ 6]; LED[ 6] = LED[ 7]; LED[ 7] = LED[ 8]; LED[ 8] = LED[ 9]; LED[ 9] = LEDX[0];
        LED[10] = LED[ 5]; LED[11] = LED[ 6]; LED[12] = LED[ 7]; LED[13] = LED[ 8]; LED[14] = LED[ 9];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the red eyes
    LED[1].setRGB( 0,0,0); LED[3] = LED[1]; LED[4] = LED[1];
    LED[0].setRGB(32,0,0); LED[2] = LED[0];
  }
}

// --------------------------------------------------------------------------------

void LED_NeutRht() {
  // generates a neutral turn right pattern
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 9].setRGB(16,16,0); LED[14].setRGB(16,16,0); // yellow
      LED[ 8].setRGB(0,0,16); LED[13].setRGB(0,0,16); // bright blue
      LED[ 7].setRGB(0,0,4); LED[12].setRGB(0,0,4); // mid blue
      LED[ 6].setRGB(0,0,1); LED[11].setRGB(0,0,1); // min blue
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[9]; // grab the last body LED rgb values
        LED[ 9] = LED[ 8]; LED[ 8] = LED[ 7]; LED[ 7] = LED[ 6]; LED[ 6] = LED[ 5]; LED[ 5] = LEDX[0];
        LED[14] = LED[ 9]; LED[13] = LED[ 8]; LED[12] = LED[ 7]; LED[11] = LED[ 6]; LED[10] = LED[ 5];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the red eyes
    LED[0].setRGB( 0,0,0); LED[1] = LED[0]; LED[3] = LED[0];
    LED[2].setRGB(32,0,0); LED[4] = LED[2];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_Bck() {
  // driving backwards
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 7].setRGB(16,16,0); LED[12] = LED[7];  // yellow
      LED[ 6].setRGB( 0,16,0); LED[13] = LED[6];  // bright green
      LED[ 5].setRGB( 0, 4,0); LED[14] = LED[5];  // mid green
      LED[ 4].setRGB( 0, 1,0); LED[ 0] = LED[4];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[3]; // grab the last body LED rgb values
        LED[ 3] = LED[ 4]; LED[ 4] = LED[ 5]; LED[ 5] = LED[ 6]; LED[ 6] = LED[ 7]; LED[ 7] = LEDX[0];
        LED[12] = LED[ 7]; LED[13] = LED[ 6]; LED[14] = LED[ 5]; LED[ 0] = LED[ 4]; LED[ 1] = LED[ 3];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[8].setRGB( 0,0,0); LED[11] = LED[8]; LED[2] = LED[8];
    LED[9].setRGB(16,0,16); LED[10] = LED[9];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_ENE() {
  // driving forward ENE
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 2].setRGB(16,16,0); LED[ 7] = LED[ 2]; // yellow
      LED[ 1].setRGB( 0,16,0); LED[ 8] = LED[ 1]; // bright green
      LED[ 0].setRGB( 0,04,0); LED[ 9] = LED[ 0]; // mid green
      LED[14].setRGB( 0,01,0); LED[10] = LED[14]; // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[11]; // grab the last body LED rgb values
        LED[11] = LED[10]; LED[10] = LED[ 9]; LED[ 9] = LED[ 8]; LED[ 8] = LED[ 7]; LED[ 7] = LEDX[0];
        LED[13] = LED[11]; LED[14] = LED[10]; LED[ 0] = LED[ 9]; LED[ 1] = LED[ 8]; LED[ 2] = LED[ 7];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[3].setRGB( 0,0,0); LED[6] = LED[3]; LED[12] = LED[3];
    LED[4].setRGB(16,0,16); LED[5] = LED[4];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_ESE() {
  // driving backwards ESE
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 0].setRGB(16,16,0); LED[14] = LED[7];  // yellow
      LED[ 1].setRGB( 0,16,0); LED[13] = LED[6];  // bright green
      LED[ 2].setRGB( 0, 4,0); LED[12] = LED[5];  // mid green
      LED[ 3].setRGB( 0, 1,0); LED[11] = LED[4];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[4]; // grab the last body LED rgb values
        LED[ 4] = LED[ 3]; LED[ 3] = LED[ 2]; LED[ 2] = LED[ 1]; LED[ 1] = LED[ 0]; LED[ 0] = LEDX[0];
        LED[10] = LED[ 4]; LED[11] = LED[ 3]; LED[12] = LED[ 2]; LED[13] = LED[ 1]; LED[14] = LED[ 0];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[5].setRGB( 0,0,0); LED[7] = LED[5]; LED[9] = LED[5];
    LED[6].setRGB(16,0,16); LED[8] = LED[6];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_Fwd() {
  // driving forwards
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 9].setRGB(16,16,0); LED[10] = LED[9]; // yellow
      LED[ 8].setRGB( 0,16,0); LED[11] = LED[8]; // bright green
      LED[ 7].setRGB( 0,04,0); LED[12] = LED[7]; // mid green
      LED[ 6].setRGB( 0,01,0); LED[13] = LED[6]; // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[9]; // grab the last body LED rgb values
        LED[ 9] = LED[ 8]; LED[ 8] = LED[ 7]; LED[ 7] = LED[ 6]; LED[ 6] = LED[ 5]; LED[ 5] = LEDX[0];
        LED[10] = LED[ 9]; LED[11] = LED[ 8]; LED[12] = LED[ 7]; LED[13] = LED[ 6]; LED[14] = LED[ 5];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[0].setRGB( 0,0,0); LED[2] = LED[0]; LED[4] = LED[0];
    LED[1].setRGB(16,0,16); LED[3] = LED[1];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_Lft() {
  // sliding sideways to the left
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 0].setRGB(16,16,0); LED[10] = LED[0];  // yellow
      LED[ 1].setRGB( 0,16,0); LED[ 9] = LED[1];  // bright green
      LED[ 2].setRGB( 0, 4,0); LED[ 8] = LED[2];  // mid green
      LED[ 3].setRGB( 0, 1,0); LED[ 7] = LED[3];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[4]; // grab the last body LED rgb values
        LED[ 4] = LED[ 3]; LED[ 3] = LED[ 2]; LED[ 2] = LED[ 1]; LED[ 1] = LED[ 0]; LED[ 0] = LEDX[0];
        LED[10] = LED[ 0]; LED[ 9] = LED[ 1]; LED[ 8] = LED[ 2]; LED[ 7] = LED[ 3]; LED[ 6] = LED[ 4];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[11].setRGB( 0,0,0); LED[13] = LED[11]; LED[5] = LED[11];
    LED[12].setRGB(16,0,16); LED[14] = LED[12];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_NNE() {
  // driving forward NNE
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 5].setRGB(16,16,0); LED[ 0] = LED[ 5]; // yellow
      LED[ 6].setRGB( 0,16,0); LED[14] = LED[ 6]; // bright green
      LED[ 7].setRGB( 0,04,0); LED[13] = LED[ 7]; // mid green
      LED[ 8].setRGB( 0,01,0); LED[12] = LED[ 8]; // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[ 9]; // grab the last body LED rgb values
        LED[ 9] = LED[ 8]; LED[ 8] = LED[ 7]; LED[ 7] = LED[ 6]; LED[ 6] = LED[ 5]; LED[ 5] = LEDX[0];
        LED[11] = LED[ 9]; LED[12] = LED[ 8]; LED[13] = LED[ 7]; LED[14] = LED[ 6]; LED[ 0] = LED[ 5];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[1].setRGB( 0,0,0); LED[3] = LED[1]; LED[10] = LED[1];
    LED[2].setRGB(16,0,16); LED[4] = LED[2];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_NNW() {
  // driving forward NNW
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 4].setRGB(16,16,0); LED[14] = LED[ 4]; // yellow
      LED[ 5].setRGB( 0,16,0); LED[13] = LED[ 5]; // bright green
      LED[ 6].setRGB( 0,04,0); LED[12] = LED[ 6]; // mid green
      LED[ 7].setRGB( 0,01,0); LED[11] = LED[ 7]; // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[ 8]; // grab the last body LED rgb values
        LED[ 8] = LED[ 7]; LED[ 7] = LED[ 6]; LED[ 6] = LED[ 5]; LED[ 5] = LED[ 4]; LED[ 4] = LEDX[0];
        LED[14] = LED[ 4]; LED[13] = LED[ 5]; LED[12] = LED[ 6]; LED[11] = LED[ 7]; LED[10] = LED[ 8];
        LED_Run = false; LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[1].setRGB( 0,0,0); LED[3] = LED[1]; LED[9] = LED[1];
    LED[0].setRGB(16,0,16); LED[2] = LED[0];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_Rht() {
  // sliding sideways to the right
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 4].setRGB(16,16,0); LED[ 9] = LED[4];  // yellow
      LED[ 3].setRGB( 0,16,0); LED[10] = LED[3];  // bright green
      LED[ 2].setRGB( 0, 4,0); LED[11] = LED[2];  // mid green
      LED[ 1].setRGB( 0, 1,0); LED[12] = LED[1];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[0]; // grab the last body LED rgb values
        LED[ 0] = LED[ 1]; LED[ 1] = LED[ 2]; LED[ 2] = LED[ 3]; LED[ 3] = LED[ 4]; LED[ 4] = LEDX[0];
        LED[ 9] = LED[ 4]; LED[10] = LED[ 3]; LED[11] = LED[ 2]; LED[12] = LED[ 1]; LED[13] = LED[ 0];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[6].setRGB( 0,0,0); LED[8] = LED[6]; LED[14] = LED[6];
    LED[5].setRGB(16,0,16); LED[7] = LED[5];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_SSE() {
  // driving backwards SSE
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 5].setRGB(16,16,0); LED[10] = LED[5];  // yellow
      LED[ 4].setRGB( 0,16,0); LED[11] = LED[4];  // bright green
      LED[ 3].setRGB( 0, 4,0); LED[12] = LED[3];  // mid green
      LED[ 2].setRGB( 0, 1,0); LED[13] = LED[2];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[1]; // grab the last body LED rgb values
        LED[ 1] = LED[ 2]; LED[ 2] = LED[ 3]; LED[ 3] = LED[ 4]; LED[ 4] = LED[ 5]; LED[ 5] = LEDX[0];
        LED[10] = LED[ 5]; LED[11] = LED[ 4]; LED[12] = LED[ 3]; LED[13] = LED[ 2]; LED[14] = LED[ 1];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[6].setRGB( 0,0,0); LED[8] = LED[6]; LED[0] = LED[6];
    LED[7].setRGB(16,0,16); LED[9] = LED[7];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_SSW() {
  // driving backwards SSW
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 9].setRGB(16,16,0); LED[14] = LED[9];  // yellow
      LED[ 8].setRGB( 0,16,0); LED[ 0] = LED[8];  // bright green
      LED[ 7].setRGB( 0, 4,0); LED[ 1] = LED[7];  // mid green
      LED[ 6].setRGB( 0, 1,0); LED[ 2] = LED[6];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[5]; // grab the last body LED rgb values
        LED[ 5] = LED[ 6]; LED[ 6] = LED[ 7]; LED[ 7] = LED[ 8]; LED[ 8] = LED[ 9]; LED[ 9] = LEDX[0];
        LED[14] = LED[ 9]; LED[ 0] = LED[ 8]; LED[ 1] = LED[ 7]; LED[ 2] = LED[ 6]; LED[ 3] = LED[ 5];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[11].setRGB( 0,0,0); LED[13] = LED[11]; LED[14] = LED[11];
    LED[10].setRGB(16,0,16); LED[12] = LED[10];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_WNW() {
  // driving forward WNW
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 2].setRGB(16,16,0); LED[12] = LED[ 2]; // yellow
      LED[ 3].setRGB( 0,16,0); LED[11] = LED[ 3]; // bright green
      LED[ 4].setRGB( 0,04,0); LED[10] = LED[ 4]; // mid green
      LED[ 5].setRGB( 0,01,0); LED[ 9] = LED[ 5]; // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[ 6]; // grab the last body LED rgb values
        LED[ 6] = LED[ 5]; LED[ 5] = LED[ 4]; LED[ 4] = LED[ 3]; LED[ 3] = LED[ 2]; LED[ 2] = LEDX[0];
        LED[12] = LED[ 2]; LED[11] = LED[ 3]; LED[10] = LED[ 4]; LED[ 9] = LED[ 5]; LED[ 8] = LED[ 6];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[13].setRGB( 0,0,0); LED[ 1] = LED[13]; LED[7] = LED[13];
    LED[ 0].setRGB(16,0,16); LED[14] = LED[0];
  }
}

// --------------------------------------------------------------------------------

void LED_Omni_WSW() {
  // driving backward WSW
  switch (LED_Task) {
    case 0: // create LED pattern
      FastLED.clear();
      LED[ 0].setRGB(16,16,0); LED[ 9] = LED[0];  // yellow
      LED[ 1].setRGB( 0,16,0); LED[ 8] = LED[1];  // bright green
      LED[ 2].setRGB( 0, 4,0); LED[ 7] = LED[2];  // mid green
      LED[ 3].setRGB( 0, 1,0); LED[ 6] = LED[3];  // min green
      LED_Run = false; LEDshow = true; LED_Task++; break;
    case 1: // rotate LED pattern
      LED_Cnt++;
      if (LED_Cnt >= 4) {
        LEDX[0] = LED[4]; // grab the last body LED rgb values
        LED[ 4] = LED[ 3]; LED[ 3] = LED[ 2]; LED[ 2] = LED[ 1]; LED[ 1] = LED[ 0]; LED[ 0] = LEDX[0];
        LED[ 9] = LED[ 0]; LED[ 8] = LED[ 1]; LED[ 7] = LED[ 2]; LED[ 6] = LED[ 3]; LED[ 5] = LED[ 4];
        LEDshow = true; LED_Cnt = 0;
      } LED_Run = false; break;
  }
  if (LEDshow) {
    // Create the purple eyes
    LED[10].setRGB( 0,0,0); LED[12] = LED[10]; //LED[ 4] = LED[10];
    LED[11].setRGB(16,0,16); LED[13] = LED[11];
  }
}

// --------------------------------------------------------------------------------

void LED_Ready() {
  // Two blue eyes at the front of the robot
  // When moving eyes are fixed and brighter using another function
  LED_Fill(0,0,0); LED_Run = false; LEDshow = true;
  // Blink occasionally
  if (0 == random(10)) {return;}
  // Now change eye positions
  int16_t zTask = random(3);
  TryAgain: if (zTask == LED_Task) {goto TryAgain;}
  if (LED_Task > zTask) {LED_Task--;} else {LED_Task++;}
  switch (LED_Task) {
    case 0: // look left
      LED[0].setRGB(0,0,24); LED[2] = LED[0]; break;
    case 1: // look forward
      LED[1].setRGB(0,0,24); LED[3] = LED[1]; break;
    case 2: // look right
      LED[2].setRGB(0,0,24); LED[4] = LED[2]; break;
  }
  LED_Del = 20 + random(20);
}

// --------------------------------------------------------------------------------

void LED_Red_Pulse() {
  // Pulsing red band
  if (LED_Cnt < 1) {LED_Cnt = 1;}
  LED_Fill(LED_Cnt,0,0);
  LED_Del = 6; LED_Run = false; LEDshow = true;
  LED_Cnt*= 2; if (LED_Cnt > 64) {LED_Cnt = 0;}
}

// --------------------------------------------------------------------------------

void LED_Red_Warning() {
  // Three bright Red flashes before branching to LedNext
  LED_Run = false; LEDshow = true;
  switch (LED_Task) {
    case 0: // 1st red
      LED_Fill(64,0,0); LED_Del = 5; LED_Task++; break;
    case 1: // all off
      LED_Fill( 0,0,0); LED_Del = 10; LED_Task++; break;
    case 2: // 2nd red
      LED_Fill(64,0,0); LED_Del = 5; LED_Task++; break;
    case 3: // all off
      LED_Fill( 0,0,0); LED_Del = 10; LED_Task++; break;
    case 4: // 2nd red
      LED_Fill(64,0,0); LED_Del = 5; LED_Task++; break;
    case 5: // all off
      LED_Fill( 0,0,0); LED_Del = 20;
      SetLedMode(LedNext); break;
  }
}

// --------------------------------------------------------------------------------

void LED_Rot_Lft() {
  // Rotate all of the LEDs by one to the left, anti-clockwise when viewed from above
  LEDX[0] = LED[0];
  for (int zL = 0; zL < 14;zL++) {LED[zL] = LED[zL+1];}
  LED[14] = LEDX[0];
}

// --------------------------------------------------------------------------------

void LED_Rot_Rht() {
  // Rotate all of the LEDs by one to the right, clockwise when viewed from above
  LEDX[0] = LED[14];
  for (int zL = 14; zL > 0;zL--) {LED[zL] = LED[zL-1];}
  LED[0] = LEDX[0];
}

// --------------------------------------------------------------------------------

void LED_Set_All(byte zR,byte zG,byte zB) {
  // sets all of the LEDs to one colour immediately
  for (byte zL = 0; zL < 15;zL++) {LED[zL].setRGB(zR,zG,zB);}
  FastLED.show();
}

// --------------------------------------------------------------------------------

void LED_Sonar() {
  // Display sonar ranges, on the Range 0 - 400mm
  FastLED.clear(); LED_Run = false; LEDshow = true;
  // Start with the front
  if (RangeVal[0] > 300) {if (Blink) {LED[2].setRGB(16,8,0);}}
  else {
    LED_Cnt = map(RangeAvg[0],10,300,10,1);
    if (LED_Cnt >  0) {LED[0].setRGB(16,16,0);}
    if (LED_Cnt >  1) {LED[0].setRGB(32, 0,0);}
    if (LED_Cnt >  2) {LED[1].setRGB(16,16,0);}
    if (LED_Cnt >  3) {LED[1].setRGB(32, 0,0);}
    if (LED_Cnt >  4) {LED[2].setRGB(16,16,0);}
    if (LED_Cnt >  5) {LED[2].setRGB(32, 0,0);}
    if (LED_Cnt >  6) {LED[3].setRGB(16,16,0);}
    if (LED_Cnt >  7) {LED[3].setRGB(32, 0,0);}
    if (LED_Cnt >  8) {LED[4].setRGB(16,16,0);}
    if (LED_Cnt >  9) {LED[4].setRGB(32, 0,0);}
  }
  if (RangeVal[1] > 300) {if (Blink) {LED[7].setRGB(16,8,0);}}
  else {
    LED_Cnt = map(RangeAvg[1],10,300,10,1);
    if (LED_Cnt >  0) {LED[5].setRGB(16,16,0);}
    if (LED_Cnt >  1) {LED[5].setRGB(32, 0,0);}
    if (LED_Cnt >  2) {LED[6].setRGB(16,16,0);}
    if (LED_Cnt >  3) {LED[6].setRGB(32, 0,0);}
    if (LED_Cnt >  4) {LED[7].setRGB(16,16,0);}
    if (LED_Cnt >  5) {LED[7].setRGB(32, 0,0);}
    if (LED_Cnt >  6) {LED[8].setRGB(16,16,0);}
    if (LED_Cnt >  7) {LED[8].setRGB(32, 0,0);}
    if (LED_Cnt >  8) {LED[9].setRGB(16,16,0);}
    if (LED_Cnt >  9) {LED[9].setRGB(32, 0,0);}
  }
  if (RangeVal[2] > 300) {if (Blink) {LED[12].setRGB(16,8,0);}}
  else {
    LED_Cnt = map(RangeAvg[2],10,300,10,1);
    if (LED_Cnt >  0) {LED[10].setRGB(16,16,0);}
    if (LED_Cnt >  1) {LED[10].setRGB(32, 0,0);}
    if (LED_Cnt >  2) {LED[11].setRGB(16,16,0);}
    if (LED_Cnt >  3) {LED[11].setRGB(32, 0,0);}
    if (LED_Cnt >  4) {LED[12].setRGB(16,16,0);}
    if (LED_Cnt >  5) {LED[12].setRGB(32, 0,0);}
    if (LED_Cnt >  6) {LED[13].setRGB(16,16,0);}
    if (LED_Cnt >  7) {LED[13].setRGB(32, 0,0);}
    if (LED_Cnt >  8) {LED[14].setRGB(16,16,0);}
    if (LED_Cnt >  9) {LED[14].setRGB(32, 0,0);}
  }
  Blink = false; LED_Del = 2;
}

// --------------------------------------------------------------------------------

void LED_SW0() {
  // Called as an over-ride when SW0 is pressed
  FastLED.clear(); LED[14].setRGB(0,0,64);
}

// --------------------------------------------------------------------------------

void LED_SW1() {
  // Called as an over-ride when SW1 is pressed
  FastLED.clear(); LED[5].setRGB(0,0,64);
}

// --------------------------------------------------------------------------------

void LED_Test() {
  // Function called when in TEST mode
  LED_Cnt++; if (LED_Cnt > 14) {LED_Cnt = 0;}
  FastLED.clear(); LED[LED_Cnt].setRGB(0,16,0);
  LED_Run = false; LEDshow = true;
}

// --------------------------------------------------------------------------------

int16_t map16(int16_t x,int16_t in_min,int16_t in_max,int16_t out_min,int16_t out_max) {
  // standard map function, but for int16_t variables
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// --------------------------------------------------------------------------------

bool MotorToTgt(int16_t zFL,int16_t zFR,int16_t zRL,int16_t zRR) {
  // // moves motor demands towards target values
  // // uses intelligent start and stop
  bool zAtTGt = true; //int16_t zI = 2;
  // if (zFL > PWM_FL) { // increase front left value
  //        if (zFL > 0) {PWM_FL += zI; if (PWM_FL < PWM_StartMax) {PWM_FL = PWM_StartMax;}}
  //   else if (zFL < 0) {PWM_FL += zI; if (PWM_FL > zFL) {PWM_FL = zFL;}}
  //   else {if (PWM_FL < -PWM_StartMax) {PWM_FL += zI;} else {PWM_FL = 0;}}
  //   zAtTGt = false;
  // } else if (zFL < PWM_FL) { // decrease front left value
  //        if (zFL < 0) {PWM_FL -= zI; if (PWM_FL > -PWM_StartMax) {PWM_FL = -PWM_StartMax;}}
  //   else if (zFL > 0) {PWM_FL -= zI; if (PWM_FL < zFL) {PWM_FL = zFL;}}
  //   else {if (PWM_FL > PWM_StartMax) {PWM_FL -= zI;} else {PWM_FL = 0;}}
  //   zAtTGt = false;
  // }
  // if (zFR > PWM_FR) { // increase front right value
  //        if (zFR > 0) {PWM_FR += zI; if (PWM_FR < PWM_StartMax) {PWM_FR = PWM_StartMax;}}
  //   else if (zFR < 0) {PWM_FR += zI; if (PWM_FR > zFR) {PWM_FR = zFR;}}
  //   else {if (PWM_FR < -PWM_StartMax) {PWM_FR += zI;} else {PWM_FR = 0;}}
  //   zAtTGt = false;
  // } else if (zFR < PWM_FR) { // decrease front right value
  //        if (zFR < 0) {PWM_FR -= zI; if (PWM_FR > -PWM_StartMax) {PWM_FR = -PWM_StartMax;}}
  //   else if (zFR > 0) {PWM_FR -= zI; if (PWM_FR < zFR) {PWM_FR = zFR;}}
  //   else {if (PWM_FR > PWM_StartMax) {PWM_FR -= zI;} else {PWM_FR = 0;}}
  //   zAtTGt = false;
  // }
  // if (zRL > PWM_RL) { // increase rear left value
  //        if (zRL > 0) {PWM_RL += zI; if (PWM_RL < PWM_StartMax) {PWM_RL = PWM_StartMax;}}
  //   else if (zRL < 0) {PWM_RL += zI; if (PWM_RL > zRL) {PWM_RL = zRL;}}
  //   else {if (PWM_RL < -PWM_StartMax) {PWM_RL += zI;} else {PWM_RL = 0;}}
  //   zAtTGt = false;
  // } else if (zRL < PWM_RL) { // decrease rear left value
  //        if (zRL < 0) {PWM_RL -= zI; if (PWM_RL > -PWM_StartMax) {PWM_RL = -PWM_StartMax;}}
  //   else if (zRL > 0) {PWM_RL -= zI; if (PWM_RL < zRL) {PWM_RL = zRL;}}
  //   else {if (PWM_RL > PWM_StartMax) {PWM_RL -= zI;} else {PWM_RL = 0;}}
  //   zAtTGt = false;
  // }
  // if (zRR > PWM_RR) { // increase rear right value
  //        if (zRR > 0) {PWM_RR += zI; if (PWM_RR < PWM_StartMax) {PWM_RR = PWM_StartMax;}}
  //   else if (zRR < 0) {PWM_RR += zI; if (PWM_RR > zRR) {PWM_RR = zRR;}}
  //   else {if (PWM_RR < -PWM_StartMax) {PWM_RR += zI;} else {PWM_RR = 0;}}
  //   zAtTGt = false;
  // } else if (zRR < PWM_RR) { // decrease rear right value
  //        if (zRR < 0) {PWM_RR -= zI; if (PWM_RR > -PWM_StartMax) {PWM_RR = -PWM_StartMax;}}
  //   else if (zRR > 0) {PWM_RR -= zI; if (PWM_RR < zRR) {PWM_RR = zRR;}}
  //   else {if (PWM_RR > PWM_StartMax) {PWM_RR -= zI;} else {PWM_RR = 0;}}
  //   zAtTGt = false;
  // }
  
  // //  Serial.println(String(PWM_FL) + "," + String(PWM_FR) + "," + String(PWM_RL) + "," + String(PWM_RR));
  return zAtTGt;  // zAtTGt == true if at target
}

// --------------------------------------------------------------------------------

void MotorPWM (int16_t zC,int16_t zA,int16_t zB) {
  // Sets the four H-bridge PWM values for drive control
  // zV > 0 - move wheel in direction for forward motion
  // zV = 0 - lock wheel in brake condition
  // zV < 0 - move wheel in direction for reverse motion
  if (!PwmEn) {zA = 0; zB = 0; zC = 0;} // motor PWM is disabled

  // Store variables for display and counters
  // PWM applied in MotorDriveTask() function, which also applied dither
  // Sign of motor drive needs to be reversed, given the way in which they are wired
  PWM_C = -zC; PWM_A = -zA; PWM_B = -zB;
}

// ---------------------------------------------------------------------------------

void PwmAttach(uint8_t zCh) {
  // attached the micros internal PWM counter to an output pin
  switch (zCh) {
    case 0: analogWrite(Pin_A0, 0); break;
    case 1: analogWrite(Pin_A1, 0); break;
    case 2: analogWrite(Pin_B0, 0); break;
    case 3: analogWrite(Pin_B1, 0); break;
    case 4: analogWrite(Pin_C0, 0); break;
    case 5: analogWrite(Pin_C1, 0); break;
  }
}

// --------------------------------------------------------------------------------

void PWM_OFF() {
  // turn OFF motor drives
  PWM_A0 = 0; PWM_A1 = 0; PWM_A = 0;  // stop Motor A front
  PWM_B0 = 0; PWM_B1 = 0; PWM_B = 0;  // stop Motor B right
  PWM_C0 = 0; PWM_C1 = 0; PWM_C = 0;  // stop Motor C rear
  analogWrite(Pin_A0, PWM_A0); analogWrite(Pin_A1, PWM_A1);
  analogWrite(Pin_B0, PWM_B0); analogWrite(Pin_B1, PWM_B1);
  analogWrite(Pin_C0, PWM_C0); analogWrite(Pin_C1, PWM_C1);
}

// ---------------------------------------------------------------------------------

void PwmSetup(uint8_t zCh) {
  // setup the PWM counter to an output pin
  switch (zCh) {
    case 0: analogWriteFrequency(Pin_A0, PwmFreq); analogWriteResolution(Pin_A0, PwmBits); break;
    case 1: analogWriteFrequency(Pin_A1, PwmFreq); analogWriteResolution(Pin_A1, PwmBits); break;
    case 2: analogWriteFrequency(Pin_B0, PwmFreq); analogWriteResolution(Pin_B0, PwmBits); break;
    case 3: analogWriteFrequency(Pin_B1, PwmFreq); analogWriteResolution(Pin_B1, PwmBits); break;
    case 4: analogWriteFrequency(Pin_C0, PwmFreq); analogWriteResolution(Pin_C0, PwmBits); break;
    case 5: analogWriteFrequency(Pin_C1, PwmFreq); analogWriteResolution(Pin_C1, PwmBits); break;
  }
}

// --------------------------------------------------------------------------------

void Range_Reset() {
  // Reset range values to default limits
  for (int zP = 0;zP < 3; zP++) {
    RangeVal[zP] = RangeMax;              // set range to out of limits
    RangeAvg[zP] = RangeMax;              // set range average to out of limits
    RangeSum[zP] = RangeMax * RangeDiv;   // set averaging accumulator to full
  }
}

// --------------------------------------------------------------------------------

void read_Battery() {
  // Called to read the battery voltage every 8ms
  // In TEST mode we don't apply critical checks, so that we can adjust voltage
  // feeds freely for test purposes.
  BatVol = analogRead(BatPin);
  BatSum = BatSum + BatVol - BatAvg;
  BatAvg = BatSum/50;     // ADC is rolling averaged over 50 readings to remove noise
  // Serial.println(BatAvg);

  if ((BatAvg > Bat6v6) && (USB)) {
    // power has returned
    USB = false;
    synchLoopTimers();
  } 
  if (USB) {return;}

  BatV = getBatV(BatAvg); BatVfp = (float)BatV/100.0;

  // Determine the max PWM value for this battery voltage
       if (BatVfp <= V_max) {PWM_Max = 255;}
  else if (BatVfp <= 8.20) {PWM_Max = (BatMult/BatVfp);}  // BatMult = 255.0 * V_max
  else {PWM_Max = 236;}

  // Report WARNING states, if not in TEST mode
  if (!TEST) {
    if (BatPc < 20) {OLED_BatWarning();}

    // If battery voltage falls below critical threshold, trigger a shutdown
    if (BatAvg < Bat6v6) {
      // Below critical threshold
      // Ensure motors are OFF
      PWM_OFF();
      // Turn OFF LEDs
      FastLED.clear(); FastLED.show(); LedNop = true; LEDshow= false;
      if (DispMon) {
        // If Monitor+ is running, send warning, then terminate
        DispDel = 100; PrintTx = ""; Display_Batt_Low();
        while (PrintTx.length() > 0) {PrintTxHandler(); delay(40);}
        DispMon = false;
      }
      // Put a message on the OLED display
      OLED_Battery_Low();
      delay(1000);  // allow time for messages
      // Enter low power state
      esp_deep_sleep_start();
      // Endless loop... that never runs!
      while (true) {yield();}
    }
  }
}

// --------------------------------------------------------------------------------

void readICM() {
  // Read ICM-42607-P registers.
  // As all register addresse are sequential, we read all 12 registers in one go
  Wire.beginTransmission(ICM_address);                    // Start communication with the MPU
  Wire.write(0x0B);                                       // Start reading ACCEL_XOUT_H at register OB hex
  I2C_Err = Wire.endTransmission(false);                  // End the transmission
  delayMicroseconds(I2Cdel);                              // Allow MPU time to respond

  // Update timers for rate measurements
  ICMtd = micros() - ICMtr;   	// time since previous call to readICM()
  ICMtr = ICMtd + ICMtr;        // set ICMtr == micros(), for next period

  Wire.requestFrom(ICM_address, 12);                      // Request 6 bytes from the MPU
  AccRawX = Wire.read()<<8|Wire.read();                   // Combine the two X-axis bytes to make one integer
  AccRawY = Wire.read()<<8|Wire.read();                   // Combine the two Y-axis bytes to make one integer
  AccRawZ = Wire.read()<<8|Wire.read();                   // Combine the two Z-axis bytes to make one integer
  GyrRawX = Wire.read()<<8|Wire.read();                   // Combine the two bytes to make one integer
  GyrRawY = Wire.read()<<8|Wire.read();                   // Combine the two bytes to make one integer
  GyrRawZ = Wire.read()<<8|Wire.read();                   // Combine the two bytes to make one integer

  t1[0] = micros() - ICMtr;   // t1 = 434µs @ 400kHz I2C, and 1466 µs @ 100kHz I2C
  
  // apply calibration offsets to these raw values
  AcX = AccRawX + AcXOff;   // note AcXOff should be zero initially in order to determin its value
  AcY = AccRawY + AcYOff;   // note AcYOff should be zero initially in order to determin its value
  GyZ = GyrRawZ + GyZOff;   // note AcYOff should be zero initially in order to determin its value

  // determine intermeasurement time for increased gyro angle precision
  if (GyT > 0) {
    GyTD = micros() - GyT;  // get the differential time between readings
    GyT = GyTD + GyT;       // set the timer for the next reading
  } else {
    // 1st reading after reset
    GyT = micros();
    GyTD = 4000;    // set 1st period to 4ms
  }
}

// --------------------------------------------------------------------------------

void readSerial() {
  // reads characters from the serial port and responds to commands
  // set PrintTgt to indicate source as serial port for responses
  keyVal = Serial.read();
  if (keyVal != -1) {SerialRx = true; PrintTgt = 0; decodeKey(keyVal);}
}

// ---------------------------------------------------------------------------------

void readSW0() {
  // Read the left button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task and WiFi control
  // SW0 is read every 16ms (62.5Hz) to debounce
  sw0State = digitalRead(sw0Pin); // record button state
  if (sw0_Nop) {return;}          // block this function whilst == true
  
  if (sw0State == LOW) {
    //##############################################################################
    //
    // SW0 button is pressed down
    //
    //##############################################################################
    if (sw0Wup) {return;} // Wait for sw0 to go HIGH again
    
    if (sw0LastState == HIGH) {
      // SW0 has just been pressed down
  //      Serial.println("SW0 Hi-LO");
      sw0DwnTime = 0; // restart button down time counter
      sw0Cnt++;       // count on the falling edge
      sw0Timer = 0;   // reset the timer
      if (MainMode < 0) {SetMainMode(0); sw0Wup = true; return;}  // Exit WiFi mode
      // Change MainMode on button press
      // 0 - default resting mode
      // 1 - LDR light modes
      // 2 - RCWL sonar modes
      // 3 - autonomous drive modes
      NewMode++; if (NewMode > 3) {NewMode = 0;}
      NewTask = 0; SetMainMode(NewMode);
    } else {
      // Whilst the button is down adjust the timer at 16ms steps
      sw0DwnTime++; // track button pressed time
      if (sw0Timer > 120) {
        // button SW0 held down for 2 seconds
        if (MainMode != 0) {SetMainMode(0);}
        // block further actions until released
        sw0Cnt = 0; sw0Timer = 0; sw0Wup = true; return;
      }
    }
  } else {
    //##############################################################################
    //
    // SW0 button is released
    //
    //##############################################################################
    if (sw0Wup) {
      // waited for button release
      sw0Wup = false; sw0Cnt = 0; sw0Timer = 0;
    } else {
      if (sw0LastState == LOW) {
        // SW0 has just been released
  //        Serial.println("SW0 Lo-Hi");
      }
      // Run timer code here
      sw0Cnt = 0; sw0Timer = 0;
    }
  }
  sw0LastState = sw0State; // record current state
  if (sw0Cnt > 0) {sw0Timer++;}
}

// ---------------------------------------------------------------------------------

void readSW1() {
  // Read the right button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the WiFi mode
  // SW1 is read every 16ms (62.5Hz) to debounce
  sw1State = digitalRead(sw1Pin); // record button state
  if (sw1_Nop) {return;}          // block this function whilst == true
  
  if (sw1State == LOW) {
    //##############################################################################
    //
    // SW1 button is pressed down
    //
    //##############################################################################
    if (sw1Wup) {return;}
    
    if (sw1LastState == HIGH) {
      // SW1 has just been pressed down
      //  Serial.println("SW1 Hi-LO");
      sw1DwnTime = 0; // restart button down time counter
      sw1Cnt++; // count on the falling edge
      sw1Timer = 0; // reset the timer
      if (MainMode < 0) {SetMainMode(0); sw1Wup = true; return;}  // Exit WiFi mode
      // If not in rest MainMode == 0, we cycle through mode task options
      // The options available are dependant on the MainMode value
      if (MainMode > 0) {
        NewTask++;
        switch(MainMode) {
          case 1:
            // 0 - LDR sense
            // 1 - LDR track
            // 2 - LDR follow
            if (NewTask > 2) {NewTask = 0;} break;
          case 2:
            // 0 - Sonar sense
            // 1 - Sonar turn
            // 2 - Sonar backaway /turn
            // 3 - Sonar follow
            if (NewTask > 3) {NewTask = 0;} break;
          case 3:
            // 0 - Auton rest
            break;
        }
        SetMainMode(MainMode);
      }
    } else {
      sw1DwnTime++; // track button pressed time
      if(sw1Timer >= 120) {
        // button held down for >= 2 seconds
        NewTask = 0; SetMainMode(MainMode);
        sw1Cnt = 0; sw1Timer = 0; sw1Wup = true; return;
      }
    }
  } else {
    //##############################################################################
    //
    // sw1State == HIGH as SW1 button is not pressed
    //
    //##############################################################################
    if (sw1Wup) {
      // release locked-out state
      sw1Wup = false;
      sw1Cnt = 0; sw1Timer = 0;
    } else {
      // normal SW1 release
      if (sw1LastState == LOW) {
        // SW1 has just been released
        // Serial.println("SW1 Lo-Hi");
      }
    }
  }
  sw1LastState = sw1State; // record current state
  if (sw1Cnt > 0) {sw1Timer++;}
}

// ---------------------------------------------------------------------------------

void readWiFiRx() {
  // Called from loop() whenever WiFiRx == true, to decode ESP-NOW data.
  // A state machine is used to separate the data blocks by header type.
  switch(Rx_Task) {
    case 0: // determine the type of data to handle
           if (Rx_Buff.ESPdata[Rx_Pnt] == 'B') {Rx_Task = 3; break;}
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'C') {WiiType = 'C'; Rx_Task = 1; break;} // Classic
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'P') {WiiType = 'P'; Rx_Task = 1; break;} // Classic Pro
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'N') {WiiType = 'N'; Rx_Task = 1; break;} // Nunchuk
      else if (Rx_Buff.ESPdata[Rx_Pnt] == '-') {WiiType = '-'; Rx_Task = 1; break;}
      else if (Rx_Buff.ESPdata[Rx_Pnt] == '$') {Rx_Task = 2; break;}
      // oh dear! bloick header not recognised so discard the block and flag error
      else {WiFiRxErr++; WiFiRx = false;}
      break;
      
    case 1: // Load Wii I2C data.
      // After the header byte there are 6 data bytes + checksum
      // perform XOR on 6 bytes to check for errors
      if (Rx_Buff.ESPdata[7] == (Rx_Buff.ESPdata[1] ^ Rx_Buff.ESPdata[2] ^ Rx_Buff.ESPdata[3] ^ Rx_Buff.ESPdata[4] ^ Rx_Buff.ESPdata[5] ^ Rx_Buff.ESPdata[6])) {
        // Data has passed the checksum test.
        RxWiFi[0] = Rx_Buff.ESPdata[1];
        RxWiFi[1] = Rx_Buff.ESPdata[2];
        RxWiFi[2] = Rx_Buff.ESPdata[3];
        RxWiFi[3] = Rx_Buff.ESPdata[4];
        RxWiFi[4] = Rx_Buff.ESPdata[5];
        RxWiFi[5] = Rx_Buff.ESPdata[6];


        // This is PLAY#### related code.
        //#######################################################################
        // Adjust period timer, to display Rx rate in Hz
        RxPeriod = millis() - RxMs; RxMs += RxPeriod;
        if (WiFiEn) {
          // WiFi mode must be active for these functions to run.
          // If in RECording mode, compare this with the previous values.
          // If different, then store the change.
               if (REC) {REC_Chng_Check();}
          else if (PLAY) {PLAY_now();}
        } else if (PLAY) {PLAY_STOP();}
        //#######################################################################

        // Depending on Wii controller type, data strip off the JX,JY and CZ values
        RxRec = true; // indicate a valid frame
        if (WiiType == 'N') {           // Wii Nunchuk
          JoyX = RxWiFi[0];             // joystick X 0-255
          JoyY = RxWiFi[1];             // joystick Y 0-255
          CZ = RxWiFi[5] & 3;           // CZ buttons, C = 2/0, Z = 1/0, active low
        } else if (WiiType == 'C') {    // Wii Classic
          JoyX = WiiRightStickX();      // joystick X 0-248
               if (JoyX > 16) {JoyX = map(JoyX,17,31,129,255);}
          else if (JoyX < 16) {JoyX = map(JoyX, 0,15,  0,127);}
             else {JoyX = 128;}
          JoyY = WiiRightStickY();      // joystick Y 0-248
               if (JoyY > 16) {JoyY = map(JoyY,17,31,129,255);}
          else if (JoyY < 16) {JoyY = map(JoyY, 0,15,  0,127);}
             else {JoyY = 128;}
          CZ = ((RxWiFi[4] & 0b100000)>>5) + (RxWiFi[4] & 0b10); // read LT and RT buttons as C,Z
        } else if (WiiType == 'P') {    // Wii Classic Pro
          JoyX = WiiRightStickX();      // joystick X 0-248
               if (JoyX > 16) {JoyX = map(JoyX,17,31,129,255);}
          else if (JoyX < 16) {JoyX = map(JoyX, 0,15,  0,127);}
             else {JoyX = 128;}
          JoyY = WiiRightStickY();      // joystick Y 0-248
               if (JoyY > 16) {JoyY = map(JoyY,17,31,129,255);}
          else if (JoyY < 16) {JoyY = map(JoyY, 0,15,  0,127);}
             else {JoyY = 128;}
          CZ = (RxWiFi[4] & 0b10) + ((RxWiFi[5] & 0b100)>2); // read RT and ZR buttons as C,Z
        } else {
          // not Nunchuk so set defaults
          JoyX = 128; JoyY = 128; CZ = 3;
        }
      } else {
        WiFiRxErr++;
      }
      Rx_Task = 0;  // reset the task pointer to lok at another block if needed
      Rx_Pnt += 8;  // increase the data pointer beyond this block
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; Rx_len = 0;} // reached end of buffer
      break;

    case 2: // load text data as if from a serial port
      // read characters up to the end of the buffer
      Rx_Pnt++;
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; Rx_len = 0; break;} // reached end of buffer
      keyVal = Rx_Buff.ESPdata[Rx_Pnt]; PrintTgt = 1; decodeKey(keyVal);
  //      Serial.write(keyVal);
      break;

    case 3: // load binary data block
      // data length is specified in second byte
      Rx_Pnt++;
      // at the moment just jump over it, binary transfer is not needed
      Rx_Pnt += Rx_Buff.ESPdata[Rx_Pnt] + 1;
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; break;} // reached end of buffer
      break;
  }
}

// ---------------------------------------------------------------------------------

void readWii() {
  // Called every 20ms, but wireless transceiver only transmits every 40ms
  // Considers data received from the Wii Nunchuk controller over WiFi and 
  // manages all movements demanded by the controller
  if (readWiiCall) {return;}  // avoid stack overflow

  readWiiCall = true;         // prevent re-entry of this function
  
  if (RxRec) {
  //##############################################################################
  //
  //  WiFi Nunchuk data received
  //
  //##############################################################################
    // Wii data received but we may not have enabled WiFi mode yet
    RxRec = false; RTxTimeout = 50; // set timeout to approx 1 sec
  //    PrintTx += "JX" + String(JoyX) + " JY" + String(JoyY) + " CZ" + String(CZ) + "\n";
  //    Serial.print("CZ "); Serial.println(CZ);
    if (!WiFiEn) {
  //##############################################################################
  //
  //  WiFi is currently disabled, listening
  //
  //##############################################################################
      // We have not received enough 'C' buttons to make the system active
      // the 'C' button alone must be held in to activate the Wii WiFi link
      if (CZ == 1) {
        // C button only is being pressed
        if (WiFiCntC < 1) {DisplayText2("Waking","Up!"); OLED_Text2S10("Waking","Up!");}
        WiFiCntC += 2;
        //  Serial.println("WiFiCnt = " + String(WiFiCntC));
        if (WiFiCntC > 60) {
          // going active, user has held in 'C' button for sufficient time
          //  Serial.println("WiFiEn = true");
          SetMainMode(-1); // go to Drive mode
          C_Dn = 8;
        } else {
          // still counting C buttons
          LEDVal = WiFiCntC/10; SetLedMode(10);
        }
      } else {
        // C button not pressed or released early so check for a inadequate count
        if (WiFiCntC > 0) {
          // reduce the C-button counter, which has to be sustained to enable WiFi mode
          WiFiCntC--; LEDVal = 1 + (WiFiCntC/10);
          if (WiFiCntC < 1) {
            DisplayText2("Wakeup","Cancelled");
            SetMainMode(0);
          } else {SetLedMode(10);}
        }
      }
    } else {
      
  //##############################################################################
  //
  //  WiFi is enabled
  //
  //##############################################################################
      // WiFi enabled so check for power down
      // CZ button stats are active LOW
      if (CZ == 0) {
        // both C and Z buttons are being pressed so move towards powering down
        if (WiFiCntC > 0) {
          if (WiFiCntC == 64) {DisplayText2("Going To","Sleep..."); DispMode = 0;}
          WiFiCntC -= 2;
        //  Serial.println("WiFiCnt = " + String(WiFiCntC));
          if (WiFiCntC <= 0) {
            // Power-down condition satisfied, so go to rest
            //  Serial.println("WiFi mode disabled!");
             SetMainMode(0);
          } else {
            // still counting down CZ buttons
            SetLedMode(10);
          }
        }
      } else {
        // Check CZ buttons to see if a speed change is wanted
        if ((CZ & 2) == 0) {
          // C button is pressed, assume a speed change is needed
          C_Dn++;
          if (C_Dn == 2) {
            C_Dn = 8;
            if (!BSe) {
              // Normal Gear select
              if (Gear < GearMax) {Gear++; SetMaxSpeed();}
              if (DispMon) {DisplayText2S24(String(Gear),"Gear");}
              OLED_Gear();
            } else {
              // SELECT + C so change to GearMax
              GearMax = 5;
              if (DispMon) {DisplayText2S24("Gear","Max");}
              OLED_GearMax();
            }
          }
          //  Serial.println("Gear =" + String(Gear));
        } else {C_Dn = 0;} // C button released
        // Check Z button. It has two functions. Short press change speed, but if
        // held in it changes the mode of the joysticks
        if ((CZ & 1) == 0) {
          // Z button is pressed
          Z_Dn++; if (Z_Dn >= 12) {
            // Switching between slide mode and normal only works for Nunchuk controllers
            if ((Z_Dn == 12) && (WiiType == 'N')) {
              Z_Mode = !Z_Mode; // toggle joystick motion mode flag
              Border = true;    // turn on boarder
              if (Z_Mode) {DisplayText2("Slide","Drive");}
              else {DisplayText2("Normal","Drive");} DispDel = 12;        // short display 480ms
              Border = false;   // thick boarder
            } Z_Dn = 50;        // Z_Dn capped at 50, held for >= 2 seconds
          }
          //  Serial.print("Z_Dn "); Serial.println(Z_Dn);
        } else {
          // Z button is not pressed
          if ((Z_Dn > 1) && (Z_Dn <= 10)) {
            // held down for less than 400 ms @ 40ms Rx cycle
            if (!BSe) {
              // Normal Gear down select
              if (Gear > 1) {Gear--; SetMaxSpeed();}
              if (DispMon) {DisplayText2S24(String(Gear),"Gear");}
              OLED_Gear();
            } else {
              // SE£LECT button held down, so reduce GearMax
              GearMax = 2;
              if (Gear > GearMax) {Gear = GearMax;}
              if (DispMon) {DisplayText2S24("Gear","Min");}
              OLED_GearMin();
            }
          } Z_Dn = 0; // Z button released so clear down counter
        }
        //  Serial.print("Z_Dn "); Serial.println(Z_Dn);

        if (WiFiCntC < 60) {
          // C-button counter is not at full strength, assume WiFi off released prematurely
          WiFiCntC++; LEDVal = 1 + (WiFiCntC/10);
          if (WiFiCntC == 60) {SetLedMode(-1);} else {SetLedMode(-10);}
        }
      }
    }
      
  //##############################################################################
  //
  //  Wii joystick demands
  //
  //##############################################################################
    // joystick movements are ignored if:
    // Omni-Bot is not active
    // Wii Joystick has not sent new data
    // a MainTask is running
    if (BotActive && WiFiEn && JoyActive && (MainMode == -1)) {
      // read the X,Y joysticks and drive if demanded
      JoyActive = false; // prevent stack overflow during moves that call loop()
      // respond to joystick demands every 40ms (25Hz)
      // note direction of travel will only change once stick has been centred
      if ((JoyLX != 32) || (JoyLY != 32)) {
        // The presence of a left Classic joystick is detected, so over-ride right
        // joystick functions. We need to re-scale its value first.
        Omni = true; ICMove = ICMMoveMax;
        int16_t zJLX, zJLY;
             if (JoyLX > 32) {zJLX = map(JoyLX,33,63,129,255);}
        else if (JoyLX < 32) {zJLX = map(JoyLX,0,31,0,127);}
           else {zJLX = 128;}
             if (JoyLY > 32) {zJLY = map(JoyLY,33,63,129,255);}
        else if (JoyLY < 32) {zJLY = map(JoyLY,0,31,0,127);}
           else {zJLY = 128;}
        JoyOmniMove(zJLX,zJLY);
      } else {
        // No left joystick action
        // When coming out of Omni mode we force a joystick null, to stop the motors, before
        // responding to any right stick demands
        // Clear previous Omni-move conditions.
        if (Omni) {JoyX = 128; JoyY = 128; Omni = false;}
        // Now chjeck for joystick centre condition
        if ((JoyX >= DbLLX) && (JoyX <= DbULX) && (JoyY >= DbLLY) && (JoyY <= DbULY)) {
          // Joystick is in centre deadband position
          JoyMode = 0;  // reset the direction of travel mode
          JoyCnt = 0;   // reset the controller glitch filter
          AtState = At_DRIVE;
            //  Serial.print(F("#"));
          if (Drive != 0) {
            // If we were moving, then stop the Omni-Bot
            //  Serial.println(F("Stopping"));
            SetLedMode(1);
            // stop driving if active, but remember direction
            DriveLast = Drive; Drive = 0;
            //  Serial.println("MotorPWM (0,0,0,0)");
            MotorPWM (0,0,0); Steer = 0;
            ICMove = ICMMoveMin;        // crank down the move counter
            DriveTxt = "-";
          }
        } else {
          // Joystick not in centre deadband position. So set movement flag, and
          // respond to joystick demands
          ICMove = ICMMoveMax;
          if (JoyMode == 0) {
            // in normal mode a joystick selection locks us into that mode, until released
            JoyCnt++;
            if (JoyCnt == 3) {
              // We have ignored the first 2 values
              // we apply a deadband to the joystick before triggering movement
              // there are potentially 8 directions of travel
              if (!Z_Mode) {
                // Normal drive mode
                    if (JoyY > DbULY) {JoyMode = 1;} // drive forward
                else if (JoyY < DbLLY) {JoyMode = 5;} // drive backwards
                else if (JoyX > DbULX) {JoyMode = 3;} // neutral turn right
                else if (JoyX < DbLLX) {JoyMode = 7;} // neutral turn left
              }
            }
          }
          if (Z_Mode) {
            // Nunchuk slide drive mode, joystick can change direction of travel at any time
            // JoyModes range from 10 to 18 in this active Z_Mode
            // priority testing goes to forward/reverse and left/right directions
            JoyOmniMove(JoyX,JoyY);
          }
        }
        switch (JoyMode) {
          case 1: JoyMoveFwd(); break;
          case 3: JoyTurnRgt(); break;
          case 5: JoyMoveBwd(); break;
          case 7: JoyTurnLft(); break;
        }
      } JoyActive = true; // re-enable this code block
    }
  } else {
  //##############################################################################
  //
  //  No WiFi Nunchuk received recently
  //
  //##############################################################################
    if (WiFiEn) {
      // Robot is enabled for Wii WiFi control
      // maintain Wii enabled counter
      if ((WiFiCntC < 64) && (CZ != 0)) {
        WiFiCntC++; LEDVal = WiFiCntC/10;
        if (WiFiCntC == 64) {SetLedMode(-1);} else {SetLedMode(-10);}
      }
    }
  }
  
  readWiiCall = false; // re-allow calls to this function
  //  Serial.print("CntZ "); Serial.println(WiFiCntZ);
  //  Serial.println(MoveState);
}

// --------------------------------------------------------------------------------

void ReadWiiClassic() {
  // Called when the Wii Classic controller is connected over Wi-Fi
  // The 'BLT', 'BRT' and JRX/JRY functions have already been handled
  // Rx bits are active LOW, but these button flags are active HIGH
  //##########################################################
  // front Z buttons
  //##########################################################
  BLT = !(RxWiFi[4]>>5 & 1);  // read L button
  BRT = !(RxWiFi[4]>>1 & 1);  // read R button
  //##########################################################
  // front Z buttons
  //##########################################################
  BZL = !(RxWiFi[5]>>7 & 1);  // read ZL button
  BZR = !(RxWiFi[5]>>2 & 1);  // read ZR button
  //##########################################################
  // Digital pad
  //##########################################################
  BDD = !(RxWiFi[4]>>6 & 1); // read BDD button
  BDL = !(RxWiFi[5]>>1 & 1); // read BDL button
  BDR = !(RxWiFi[4]>>7 & 1); // read BDR button
  BDU = !(RxWiFi[5]    & 1); // read BDU button

  // PLAY#### related code.
  if (WiFiEn) {
    //##########################################################
    // PLAYback functions
    //##########################################################
    // When in WiFi drive mode.
    // Handle digital pad special movements.
    if (BDD) {Move_BDD();} else {BDD_ = false;}
    if (BDL) {Move_BDL();} else {BDL_ = false;}
    if (BDR) {Move_BDR();} else {BDR_ = false;}
    if (BDU) {Move_BDU();} else {BDU_ = false;}
  }

  //##########################################################
  // Left joystick
  //##########################################################
  JoyLX = RxWiFi[0]; JoyLX &= 0x3f; // extract left joystick value
  JoyLY = RxWiFi[1]; JoyLY &= 0x3f; // extract right joystick value
  //##########################################################
  // Centre Buttons, Select,Home,Start
  //##########################################################
  BSe = !(RxWiFi[4]>>4 & 1);  // read select button
  BHm = !(RxWiFi[4]>>3 & 1);  // read home button
  BSt = !(RxWiFi[4]>>2 & 1);  // read start button

  // PLAY#### related code.
  if (WiFiEn) {
    //##########################################################
    // RECord & PLAYback functions
    //##########################################################
    // When in WiFi drive mode.
    // Handle RECording and PLAYback for Classic controllers
    if (WiiType == 'C') {    // Wii Classic
      // Pressing BZL + Start will start a RECording process.
      // Pressing BZL on its own will stop a RECording.
      // Pressing Start on its own will PLAY a recording.
      if (!BZL && BSt) {if (!REC && !PLAY && (RECtotal > 0)) {PLAY_CLR(); PLAY = true; Gear = 1;}}
      if ( BZL && BSt) {if (!REC && !RECstart) {REC_CLR(); REC = true; RECstart = true; Gear = 1;}}
      if ( BZL && REC && !RECstart) {REC = false;}
    }
    if (WiiType == 'P') {    // Wii Classic Pro
      // Pressing L + Start will start a RECording process.
      // Pressing L on its own will stop a RECording.
      // Pressing Start on its own will PLAY a recording.
      if (!BLT && BSt) {if (!REC && !PLAY && (RECtotal > 0)) {PLAY_CLR(); PLAY = true; Gear = 1;}}
      if ( BLT && BSt) {if (!REC && !RECstart) {REC_CLR(); REC = true; RECstart = true; Gear = 1;}}
      if ( BLT && REC && !RECstart) {REC = false;}
    }
  }

  //##########################################################
  // Right-hand Buttons, Y,X,b,a
  //##########################################################
  Ba = !(RxWiFi[5]>>4 & 1);  // read Ba button
  Bb = !(RxWiFi[5]>>6 & 1);  // read Bb button
  BX = !(RxWiFi[5]>>3 & 1);  // read BX button
  BY = !(RxWiFi[5]>>5 & 1);  // read BY button
}

// --------------------------------------------------------------------------------

void Recall_LEDs() {
  // Recalls previously saved LED states and clears a flag
  for (int zL = 0;zL < NumLEDs; zL++) {LED[zL] = Mem[zL];}
  LedMem = false;  // set flag to indicate values recalled
}

// --------------------------------------------------------------------------------

void RxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Rx buffer of length zlen
  Rx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    Rx_Chk = Rx_Chk ^ Rx_Buff.ESPdata[zP];
  }
}

// --------------------------------------------------------------------------------

void SaveLEDs() {
  // Saves the current state of each LED and sets a flag
  for (int zL = 0;zL < NumLEDs; zL++) {Mem[zL] = LED[zL];}
  LedMem = true;  // set flag to indicate stored values
}

// // --------------------------------------------------------------------------------

void SetLedMode(int16_t zM) {
  // Sets the LED mode and clears the variables.
  // if zM matches current LedMode then do nothing.
  // if zM is -ve then reset that mode.
  // If Zm is new, then reset that mode.
  if (zM == LedMode) {return;}
  
  // Change of mode, or zM is -ve, so reset the flags and set the new mode
  LED_Cnt = 0;
  LED_Del = 0;
  LED_Task = 0;
  LED_Period = 20;

  LedMode = abs(zM);
  // Serial.println("SetLedMode" + String(LedMode));
}

// --------------------------------------------------------------------------------

void SetMainMode(int16_t zM) {
  // Sets the default conditions for a new mode
  // MainMode affects display functions and MainTask settings
  // The value of NewTask will determine which main task is run
  DispNOP = false;          // if == true then don't update the display functions
  DisX = 0.0;               // distance travelled on X-axis in mm
  DisY = 0.0;               // distance travelled on Y-axis in mm
  Drive = 0;                // reset drive direction
  DriveLast = 0;            // reset previous driving direction
  DriveTxt = "-";           // indicates drive mode
  ESC = false;              // clear the ESCape flag
  Gear = 1;                 // start in lowest speed range
  JoyLX = 32;               // Wii Classic left joystick X value
  JoyLY = 32;               // Wii Classic left joystick Y value
  MainDel = 0;              // if > 0 this is the MainTask delay timer
  MainRun = false;          // clear MainTask run flag
  MainSub = 0;              // MasinTask sub-task pointer
  OLED_Mode = 0;            // set OLED to default screen
  PLAY_STOP();              // reset the PLAY mode
  PwmEn = false;            // if == true motor PWM is enabled
  PWM_OFF();                // turn OFF motor PWMs
  PWM_Start = PWM_StartMax; // default start value
  Range_Reset();            // reset range values
  REC_CLR();                // reset the RECord mode
  RCWL_En = false;          // disable trigger the RCWL ranging devices
  SetRcwlMode(0);           // set default RCWL sensor mode to front only
  Steer = 0;                // zero steering
  SubCnt = 0;               // sub task general counter
  SubDel = 0;               // subtask delay counter
  SubTask = 0;              // reset sub task pointer
  SubTask1 = 0;             // level 1 sub task pointer
  TaskMsg = "";             // Any task message
  TuneClr();                // rewset Tune[] values
  Turn = false;             // default = false; if true turns continuously until = false
  YawTgt = 0.0;             // reset rotation gyro target
  Z_Mode = false;           // joystick motion flag, normally false, if == true then slide mode

  switch (zM) {
    case -1: // Wi-Fi drive mode
      DisplayText2S16("WiFi","Mode");
      OLED_Text2S1610("WiFi","Mode");
      DispMode = DM_MtrPWM;     // set Monitor+ display to LDR
      OLED_Mode = -1;           // OLED DRive Mode
      AtState = At_DRIVE;       // Set state for display
      BotActive = true;         // Set active flag
      PwmEn = true;             // if == true motor PWM is enabled
      PWM_Start = PWM_StartMax; // default start value
      WiFiEn = true;            // Enable Wii functions
      MainTask = 0;             // define the MainTask to run
      NewTask = 0;              // clear the task pointer
      SetLedMode(1); break;

    case 0: // Resting
      WiFiDisable();            // Disable WiFi functions
      DispMode = DM_Batt;       // set Monitor+ display to LDR
      if (!Boot) {
        DisplayText2S16("Rest","Mode");
        OLED_Text2S1610("Rest","Mode");
      }
      AtState = At_REST;        // Set state for display
      BotActive = false;        // Clear active flag
      MainTask = 0;             // define the MainTask to run
      NewTask = 0;              // clear the task pointer
      SetLedMode(0); break;

    case 1: // LDR modes
      DispMode = DM_LDR;        // set Monitor+ display to LDR
           if (NewTask == 0) {DisplayText2S16("LDR","Sense"); OLED_Text2S1610("LDR","Sense");}
      else if (NewTask == 1) {DisplayText2S16("LDR","Track"); OLED_Text2S1610("LDR","Track");}
      else if (NewTask == 2) {DisplayText2S16("LDR","Follow"); OLED_Text2S1610("LDR","Follow");}
      MainTask = 10 + NewTask;  // define the MainTask to run
      OLED_Mode = MainTask;
      break;

    case 2: // RCWL sonar modes
      DispMode = DM_Range;      // set Monitor+ display to Range
           if (NewTask == 0) {SetRcwlMode(3); DisplayText2S16("Sonar","Sense"); OLED_Text2S1610("Sonar","Sense");}
      else if (NewTask == 1) {SetRcwlMode(3); DisplayText2S16("Sonar","Turn"); OLED_Text2S1610("Sonar","Turn");}
      else if (NewTask == 2) {SetRcwlMode(0); DisplayText2S16("Sonar","Backaway"); OLED_Text2S1610("Sonar","Backaway");}
      else if (NewTask == 3) {SetRcwlMode(0); DisplayText2S16("Sonar","Track"); OLED_Text2S1610("Sonar","Track");}
      RCWL_En = true;           // enable trigger the RCWL ranging devices
      MainTask = 20 + NewTask;  // define the MainTask to run
      OLED_Mode = MainTask;
      break;

    case 3: // Autonomous modes
      DispMode = DM_MoveEng;    // set Monitor+ display to Move Engine
           if (NewTask == 0) {DisplayText2S16("Auton","Rest"); OLED_Text2S1610("Auton","Rest");}
      else if (NewTask == 1) {DisplayText2S16("Auton","Drive"); OLED_Text2S1610("Auton","Drive");}
      MainTask = 30 + NewTask;  // define the MainTask to run
      SetLedMode(4); break;
  }
  MainMode = zM; NewMode = MainMode;
  //  Serial.println("SetMainMode = " + String(MainMode));
}

// --------------------------------------------------------------------------------

void SetMaxSpeed() {
  // called when changing 'Gear' to set drive limitations
}

// --------------------------------------------------------------------------------

void SetRcwlMode(int16_t zMode) {
  // Called to set the RCWL_Mode variable and count value
  RCWL_Mode = zMode;
       if (zMode < 3) {RCWL_Max = 12;}
  else if (zMode < 6) {RCWL_Max = 10;}
  else {RCWL_Max = 8;}
}

// --------------------------------------------------------------------------------

void sw0_Reset() {
  // reset SW0 scan variables
  sw0Cnt = 0;             // button switch counter
  sw0DwnTime = 0;         // button pressed down time
  sw0LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw0_Nop = false;        // if == true don't perform switch functions
  sw0State = HIGH;        // state of read button switch pin
  sw0Timer = 0;           // timer used to detemine button sequences
  sw0Wup = false;         // set == true for SW0 to wait for button release
}

// --------------------------------------------------------------------------------

void sw1_Reset() {
  // reset SW1 scan variables
  sw1Cnt = 0;             // button switch counter
  sw1DwnTime = 0;         // button pressed down time
  sw1LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw1_Nop = false;        // if == true don't perform switch functions
  sw1State = HIGH;        // state of read button switch pin
  sw1Timer = 0;           // timer used to detemine button sequences
  sw1Wup = false;         // set == true for SW0 to wait for button release
}

// ---------------------------------------------------------------------------------

void TuneClr() {
  // Reset Tune[] values
  for (int16_t zT = 0; zT < 4;zT++) {
    TuneRef[zT] = "---:"; TuneInt[zT] = 0; TuneFpt[zT] = 0.0;
  }
}

// ---------------------------------------------------------------------------------

void WaitSwUp() {
  // Wait here for both switches to be released
  while (!digitalRead(sw0Pin) || !digitalRead(sw1Pin)) {yield();}
}
// ---------------------------------------------------------------------------------



//###############################################################################
//
//  WiFi Code
//
//###############################################################################


// --------------------------------------------------------------------------------

void WiFiClearRxBuff(int zNum) {
  // clears the contents of the receive buffer
  for (int zP = 0; zP < zNum;zP++) {Rx_Buff.ESPdata[zP] = 0;}
}

// --------------------------------------------------------------------------------

void WiFiClearTxBuff(int zNum) {
  // clears the contents of the transmit buffer
  for (int zP = 0; zP < zNum;zP++) {Tx_Buff.ESPdata[zP] = 0;}
}

// ---------------------------------------------------------------------------------

void WiFiDisable() {
  // when either a switch is pressed or WiFi disable is required
  WiFiEn = false; WiFiCntC = 0; WiFiRestCnt = 0; C_Dn = 8; Z_Dn = 8;
}

// --------------------------------------------------------------------------------

void WiFiDisconnect() {
  // enter the WiFi disconnected state
  WiFiConnected = false;
  WiFiClearRxBuff(WiFiTx_len);
  WiFiClearTxBuff(WiFiTx_len);
  WiFiTryCnt = 0;     // reset the connection retry counter
  WiFiTx_CB = 1;      // set ready to send flag
  WiFiRxRec = false;  // clear the data received flag
  Serial.println("WiFiDisconnected!");
}

// ----------------------------------------------------------------------

int WiiLeftStickX() {
  // returns a Wii Classic LX value 0/32/63
  return  ((RxWiFi[0] & (byte)0x3f));
}

// ----------------------------------------------------------------------

int WiiLeftStickY() {
  // returns a Wii Classic LY value 0/32/63
  return  ((RxWiFi[1] & (byte)0x3f));       
}

// ----------------------------------------------------------------------

boolean WiiPressedRowBit(byte row, byte bit) {
  // Wii Classic, read a bit to test for button presses.
  // Call directly using the following:
  //  values  (row,bit):
  //  RT      (4,1)
  //  Start   (4,2)
  //  Home    (4,3)
  //  Select  (4,4)
  //  LT      (4,5)
  //  D-Down  (4,6)
  //  D-Right (4,7)
  //  D-Up    (5,0)
  //  D-Left  (5,1)
  //  RZ      (5,2)
  //  X       (5,3)
  //  A       (5,4)
  //  Y       (5,5)
  //  B       (5,6)
  //  LZ      (5,7)

  byte mask = (1 << bit);           // gives 00001000 mask for (X,3) 
  return (!(RxWiFi[row] & mask ));  // gives 11110111 or 11111111
}

// --------------------------------------------------------------------------------

void WiFiTryToConnect() {
  // try to connect with the Wii Transceiver by sending devices name
  // initialise ESP-NOW link first
  Init_ESP_NOW();

  Tx_Buff.ESPdata[0] = 'O';
  Tx_Buff.ESPdata[1] = 'm';
  Tx_Buff.ESPdata[2] = 'n';
  Tx_Buff.ESPdata[3] = 'i';
  Tx_Buff.ESPdata[4] = '-';
  Tx_Buff.ESPdata[5] = 'B';
  Tx_Buff.ESPdata[6] = 'o';
  Tx_Buff.ESPdata[7] = 't';
  WiFiTx_len = 8;
  //  t0 = micros();
  esp_err_t result = esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
  if (result == ESP_OK) {
  //    Serial.println("Sent with success");
  } else {
  //    Serial.println("Error sending the data");
  } 

  WiFiTryNum++; // count the total number of attempts
  //  Serial.println("WiFiTryToConnect...");
}

// --------------------------------------------------------------------------------

void WiFiTxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Tx buffer of length zlen
  WiFiTx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    WiFiTx_Chk = WiFiTx_Chk ^ Tx_Buff.ESPdata[zP];
  }
}

// ----------------------------------------------------------------------

int WiiRightStickX() {
  // returns a Wii Classic RX value 0/16/31
  return ((RxWiFi[0] & (byte)0xc0) >> 3) + ((RxWiFi[1] & (byte)0xc0) >> 5) +  ((RxWiFi[2] & (byte)0x80) >> 7);
}

// ----------------------------------------------------------------------

int WiiRightStickY() {
  // returns a Wii Classic RY value 0/16/31
  return RxWiFi[2] & (byte)0x1f;    
}

// ---------------------------------------------------------------------------------
