 // ################################################################################
//
//  TDM Sword MK1 (R0)
//
//  Released:  18/05/2025
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Micro:  ESP32 Dev Module    Version: 3.2.0

    This program writes patterns to a row of RGB LEDs mounted on a
    3D printed sword blade. This version duplicates the modes of the previous wooden
    version and TDM patterns.

    Standard modes (when selected horizontal) are:

    Mode 0 - default random pattern
    Mode 1 - fixed pattern sequence
    Mode 2 - random patterns and speeds
    Mode 3 - accelerometer bars X,Y,Z
    Mode 4 - accelerometer moving stripes
    Mode 5 - random fixed patterns if moving
    Mode 6 - faster random colours if moving
    Mode 7 - test tasks

    Holding the sword, with the blade pointing vertically upwards, and pressing the
    button once for Mode 0, will enter TDM mode. This displays coloured patterns
    for each quadrent. Pressing the button again will display the word "HELLO"
*/
// Declare Libraries
#include <Wire.h>             // I2C library      version 3.2.0
#include <FastLED.h>          // Neopixel library version 3.19.6

// Configuration
String Release = "TDM Sword MK1 (R0)";
String Date = "18/05/2025";

// Define a task for core 0
TaskHandle_t Core0;

// 3-axis MPU 6050 variables
long AcAvgDiv;                // dividing factor used in averaging
long AcAvgDivMax;             // max dividing factor for averaging
int16_t AcX;                  // MPU raw X-acceleration value + offset trim
int16_t AcXV;                 // MPU X-acceleration value converted to an angle vector
long AcXAvg;                  // averaged AcX value to reduce noise
int16_t AcXL;                 // previous MPU X-acceleration value
long AcXSum;                  // averaging accumulator
int16_t AcY;                  // MPU raw Y-acceleration value + offset trim
int16_t AcYV;                 // MPU Y-acceleration value converted to an angle vector
long AcYAvg;                  // averaged AcY value to reduce noise
float AcYF;                   // floating point Y acceleration value
int16_t AcYL;                 // previous MPU Y-acceleration value
bool AcYsgn;                  // positive/negative sign flag
long AcYSum;                  // averaging accumulator
int16_t AcZ;                  // MPU raw Z-acceleration value + offset trim
int16_t AcZV;                 // MPU Z-acceleration value converted to an angle vector
long AcZAvg;                  // averaged AcZ value to reduce noise
int16_t AcZL;                 // previous MPU Z-acceleration value
long AcZSum;                  // averaging accumulator
int16_t AngL;                 // previous AngZI
float AngX;                   // accelerometer X angle
float AngZ;                   // gyro Z angle
int16_t AngZI;                // integer of AngZ
long GyAvgDiv;                // dividing factor used in averaging
int GyCal;                    // counter used to determine gyro calibration period
unsigned long GyT;            // time of gyro readings in microseconds
unsigned long GyTD;           // time between successive gyro readings in microseconds
int16_t GyX;                  // MPU raw X-gyroscope value
long GyXAvg;                  // averaged GyX value to reduce noise
int16_t GyXL;                 // previous MPU X-gyroscope value
int16_t GyXOffCnt;            // offset counter for gyro centering
long GyXSum;                  // averaging accumulator
int16_t GyY;                  // MPU raw Y-gyroscope value
long GyYAvg;                  // averaged GyY value to reduce noise
int16_t GyYL;                 // previous MPU Y-gyroscope value
int16_t GyYOffCnt;            // offset counter for gyro centering
long GyYSum;                  // averaging accumulator
int16_t GyZ;                  // MPU raw Z-gyroscope value
long GyZAvg;                  // averaged GyZ value to reduce noise
int16_t GyZL;                 // previous MPU Z-gyroscope value
int16_t GyZOffCnt;            // offset counter for gyro centering
long GyZSum;                  // averaging accumulator
bool MPUCal;                  // == true during calibration
float PitchAcc;               // pitch angle in degrees base on accelerometer
float PitchGyr;               // pitch angle in degrees base on gyro + acc
int PitchGyrInt;              // integer version of PitchGyr for quicker calcs
bool UpRht;                   // set == true when the robot is the correct way up
float YawAcc;                 // yaw angle in degrees based on accelerometers
float YawGyr;                 // yaw angle in degrees based on gyro + acc
int YawGyrInt;                // integer version of YawGyr for quicker calcs

// Define constants
#define acc_1g  8192              // accelerometer 1g value at +/-4g full scale
#define acc_1gF  8192.0           // float accelerometer 1g value at +/-4g full scale
#define acc_calibration_valueX 0  // Enter the accelerometer calibration value, default 0
#define acc_calibration_valueY 0  // Enter the accelerometer calibration value, default 0
#define AcXOff -495               // offset to be added to raw X-acceleration value
#define AcYOff -39                // offset to be added to raw Y-acceleration value
#define AcZOff 0                  // offset to be added to raw Z-acceleration value
#define BatCal 383.9      // calibrated voltage multiplying factor
#define BatCritical 2534  // critical battery threshold @6.6v, default = 2672, set == 0 to ignore
#define BatMax 3148       // A0 for fully charged battery voltage, == 8.2v
#define BatPin 35         // analog input pin ADC15 on GPIO12
#define BatWarn 2687      // battery threshold of 7.0v gives low warning
#define ColBlu 0x000088;  // ref colour
#define ColCya 0x004444;  // ref colour
#define ColGrn 0x008800;  // ref colour
#define ColPur 0x440044;  // ref colour
#define ColRed 0x880000;  // ref colour
#define ColYel 0x444400;  // ref colour
#define GyAvgDivMax 250   // max averaging divider, 250 == 10 second average
#define GyXOff 0          // offset to be added to raw X-gyroscope value
#define GyYOff 0          // offset to be added to raw Y-gyroscope value
#define GyZOff -40        // offset to be added to raw Z-gyroscope value
#define I2Cdel 20         // I2C bus timing delay for MPU6050
#define LED_Pin 4         // assign GPIO12 for RGB LEDs
#define MPU_address 0x68  // MPU-6050 I2C address (0x68 or 0x69)
#define NumLEDs 17        // number of neopixel LEDs
#define NumLEDs_1 16      // number of neopixel LEDs - 1
#define NumLEDs_2 15      // number of neopixel LEDs - 2
#define sw0Pin 33         // switch SW0 assigned to GPIO33
#define sw1Pin 25         // switch SW1 assigned to GPIO25
#define sw2Pin 26         // switch SW2 assigned to GPIO26
#define TEXT_ALIGN_LEFT 0
#define TEXT_ALIGN_RIGHT 1
#define TEXT_ALIGN_CENTER 2
#define TEXT_ALIGN_CENTER_BOTH 3
 
// create a FastLED instance and define the total number of LEDs on the robot
CRGB LED[NumLEDs];        // LEDs connected to micro for o/p
CRGB LEDX[1];             // temp LED value
CRGB FRAME[(NumLEDs - 1) * 16]; // image used in TDM mode

// Declare and initialise global variables
int Any;                      // any temporary integer value
String Any$;                  // temporary string
long AnyLong;                 // any temp value
unsigned long anyMilli;       // any millisecond timer
int16_t anyVal;               // any temp variable
int32_t BatAvg;               // preload averaging function with high value
int32_t BatDist;              // used battery discharge testing
int32_t BatSum;               // preload averaging sum with 20x max value
int32_t BatVol;               // instantaneous battery voltage
uint8_t Bright=32;               // FastLED brightness
uint8_t Brightness;           // Display brightness
uint8_t colBlu[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; // colour blue
uint8_t colGrn[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; // colour green
uint8_t colRed[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; // colour red

// colours
int Blu,Cyn,Grn,Prl,Red,Ylw;  // colours used in TDM graphics

int coreLB;                   // core 0 loopback time
char cmdMode;                 // command mode
int16_t cmdSgn;               // cmdVal sign, 0 or -1
char cmdType;                 // command mode type
int cmdVal;                   // value associated with a cmdType
byte Col_B,Col_G,Col_R;       // current colours
char Deg = 176;               // ASCII for "°"
bool ESC;                     // true to escape all functions
int gyro_Z_data_raw;          // raw Z gyro value read from MPU6050
byte I2CErr;                  // flag returned from I2C comms
bool I2C_MPU;                 // true if MPU6050 is detected during POST
int16_t JX;                   // global variable used with Joystick app
int16_t JY;                   // global variable used with Joystick app
char keyChar;                 // any keyboard character
int16_t keyVal;               // any keyboard value
bool LEDclr;                  // == true if LEDs are to be cleared in TDM mode
int16_t LED_Cnt;              // counter used in LED sequencing
int16_t LED_Del;              // delay timer used in LED functions
int16_t LED_Max;              // max value used in LED functions
int16_t LedMode;              // a value > 0 determines LED flash sequence; if = 0 disabled
unsigned long LED_Period;     // LED timer period, default = 20ms
bool LED_Run;                 // == true if LED_Task is to run
bool LEDshw;                  // LED show flag in TDM mode
int16_t LED_Task;             // LED task pointer
bool LedTick;                 // set true during LED cycles
unsigned long LED_Timer;      // 20+ms LED timer
byte LEDVal;                  // values used in LED tasks
unsigned long loopDelay;      // RTOS task timer
int16_t mainMode;             // pointer to display mode, default = 0+
int16_t maxBright;            // maximum brightness
int16_t maxBright2;           // half of maximum brightness
int16_t maxBright3;           // third of maximum brightness
bool ModeEn;                  // if == true the do mainMode tasks
String myMAC;                 // MAC address if this device
unsigned long next5ms;        //  5ms loop timer
unsigned long next10ms;       // 10ms loop timer
unsigned long next20ms;       // 20ms loop timer
unsigned long next40ms;       // 40ms loop timer
long patCol0;                 // any pattern colour
int16_t patTask;              // any pattern task pointer
int16_t patTaskEnd;           // any pattern end pointer
int16_t patTaskFlg;           // any pattern task flag
int16_t patTaskNxt;           // any pattern task pointer record
int16_t patTaskPnt;           // any pattern pointer
int16_t Ping;                 // Rx 'ping' counter
int16_t PlayCnt;              // any play counter
int16_t PlayDel;              // number of wait cycles in play sequence >= 0
int16_t PlayEn;               // >0 enables play mode
int16_t PlayF;                // any play end finish offset
int16_t PlayI;                // play increment from PlayS to PlayE
int16_t PlayS;                // any play start offset, also used as play pointer
unsigned long print40ms;      // 40ms variable print timer
int16_t PrintTgt;             // 0 == serial port, 1 == WiFi ESP-NOW
String PrintTx = "";          // printed strings
bool readWiiCall;             // flag prevents call stack overflow
int16_t RotClr;               // auto zero offset counter for Z-axis
float RotGyr;                 // rotation angle in degrees based on gyro + acc
int16_t RotGyrInt;            // rotation angle in degrees as an integer
int16_t Rx_Chk;               // Rx checksum
int16_t RxCZ;                 // buffered received CZ value
int16_t RxJoyX;               // buffered received JoyX value
int16_t RxJoyY;               // buffered received JoyY value
int16_t Rx_len;               // length of WiFi received data block
int16_t Rx_Pnt;               // data pointer user by Rx state machine
bool RxRec;                   // set true when a valid frame of data is received
int16_t RxState;              // receiver state machine state
int16_t Rx_Task;              // task pointer user by Rx state machine
int16_t RTxTimeout;           // counter used to auto-reset if WiFi fails
int16_t RxVal;                // value received from serial Rx
byte RxWiFi[6];               // array holding 6 bytes of Wii received data
bool SerialRx;                // == true if any character has been received
int16_t subTask;              // sub task pointer used in everey mode
int16_t subTaskNxt;           // next sub task pointer used in everey mode
int16_t sw0Cnt;               // button switch counter
int16_t sw0DwnTime;           // button pressed down time
bool sw0LastState;            // previous state of button switch, HIGH/LOW
bool sw0_Nop;                 // if == true don't perform switch functions
bool sw0State;                // state of read button switch pin
int16_t sw0Timer;             // timer used to detemine button sequences
bool sw0Wup;                  // set == true for SW0 to wait for button release
int16_t sw1Cnt;               // button switch counter
int16_t sw1DwnTime;           // button pressed down time
bool sw1LastState;            // previous state of button switch, HIGH/LOW
bool sw1_Nop;                 // if == true don't perform switch functions
bool sw1State;                // state of read button switch pin
int16_t sw1Timer;             // timer used to detemine button sequences
bool sw1Wup;                  // set == true for SW1 to wait for button release
int16_t sw2Cnt;               // button switch counter
int16_t sw2DwnTime;           // button pressed down time
bool sw2LastState;            // previous state of button switch, HIGH/LOW
bool sw2_Nop;                 // if == true don't perform switch functions
bool sw2State;                // state of read button switch pin
int16_t sw2Timer;             // timer used to detemine button sequences
bool sw2Wup;                  // set == true for SW0 to wait for button release
int16_t swEn;                 // if >0 enables up/down switch reading
int16_t swPnt;                // switch task poionter for core 0 tasks
int16_t Task5ms;              //  5ms subtask pointer
int16_t Task10ms;             // 10ms subtask pointer
int16_t Task20ms;             // 20ms subtask pointer
int16_t Task40ms;             // 40ms subtask pointer
int16_t taskCnt0;             // any subtask counter
int16_t taskCnt1;             // any subtask counter
int16_t taskDel0;             // any subtask delay counter
int16_t taskInc0;             // any task increment
int16_t taskOff0;             // any task offset
int16_t taskSpeed;            // nominal speed of task
bool TDM;                     // == true if in TDM mode
int16_t TdmBeat;              // TDM mode LED[0] red value
int16_t TDMcnt;               // counter used to hold off heartbeat when moving
int16_t TdmInc;               // increment used in TDM tasks
bool TEST = false;            // set == true during assembly, then false for normal use
int16_t Tx_PingCnt;           // 1 second Tx ping timer
bool USB;                     // == true if initial battery is below 5.5v
bool Vertical;                // == true if AcY >= 5500
bool VertLast;                // previous Vertical state

// ---------------------------------------------------------------------------------

void setup() {
  // Setup code, runs once:
  // initialise other pins
  pinMode(BatPin,INPUT);        // ADC2_4  GPIO13 battery divider
  pinMode(sw0Pin,INPUT_PULLUP); // button switch sw0 Pin
  pinMode(sw1Pin,INPUT_PULLUP); // button switch sw1 Pin
  pinMode(sw2Pin,INPUT_PULLUP); // button switch sw2 Pin
  
  setDefaults();  // reset variables

  Serial.begin(115200);

  Wire.begin();

  // initialise the FastLED component
  // a chain of 17 LEDs = 24 x 17 = 408 bits, taking 510us to show at 800kbps
  FastLED.addLeds<WS2812B, LED_Pin, GRB>(LED, NumLEDs);
  FastLED.setBrightness(Bright);

  // grab initial battery voltage before WiFi kills the ADC2 function
  // take 10 readings over a 10ms period
  for (int zI = 0;zI < 10;zI++) {BatVol+= analogRead(BatPin); delay(1);}
  BatVol/= 10;

  RunPOST();      // initialise I2C devices, set intro display, etc

  synchLoopTimers(0);
  
  // define the Core 0 task parameters
  // Core 0 is normally used for WiFi, but here we use it to read sensors
  // leaving Core 1 to create the LED patterns and clock them out
  // Note, this code starts and runs immediately, before loop() does
  xTaskCreatePinnedToCore(
    loop0,       // Task function.
    "Core0",     // name of task.
    10000,       // Stack size of task
    NULL,        // parameter of the task
    1,           // priority of the task
    &Core0,      // Task handle to keep track of created task
    0);          // pin task to core 0
}

// ---------------------------------------------------------------------------------

  // the controller will react to a soft RESET event
  // this function effectively restarts the code as if from power-up
void(* resetFunc) (void) = 0; //declare reset function at address 0

// ---------------------------------------------------------------------------------

void loop() {
  // Main loop code, runs repeatedly.
  // t1[0] = micros(); // end time
  // t0[1] = micros(); // start time

  //###############################################################################
  //
  //  WiFi comms check & respond
  //
  //###############################################################################
  // if WiFi is conencted then check Rx buffer on every cycle, so that we respond to
  // received packets as soon as possible
  // if (WiFiConnected) {
  //   if (WiFiRx) {readWiFiRx();} // WiFi Rx buffer contains unread data so deal with it before adding new
  // }

  readSerial();                 // check serial port on every loop cycle


  //###############################################################################
  //
  //  5ms TDM MPU sensor tasks (200Hz)
  //
  //###############################################################################
  // These tasks are run in a consequitive ripple to reduce the overhead on more
  // critical code. They are performed in reverse case order and will all complete
  // within the 5ms period
  if (TDM) {
    if (Task5ms > 0) {
      switch(Task5ms) {
        case  1: loadFRAME(); break;     // load a strip from the frame
        case  2: calcAngle(); break;     // calculate yaw and Z gyro turning
        case  3: readGyroZ(); break;     // read the Z-axis gyro
        case  4: if (I2C_MPU) {readAccelAll();} break;  // read MPU6050 accelerometers
       }
      if (I2CErr) {Serial.println("I2CErr");}
      Task5ms--;
    } else {Task5ms = 4;} // run unconstrained
  }

  
  //###############################################################################
  //
  //  20ms Tasks
  //
  //###############################################################################
  // perform these tasks at 20ms intervals
  if (Task20ms > 0) {
    switch(Task20ms) {
      case  1: if (TDM) {doTdmTask();} break;
      case  2: if (!TDM && ModeEn && (PlayEn > 0)) {playTask();} break;
      case  3: if (!TDM && ModeEn) {doModeTask();} break;
      case  6: if (!TDM && I2C_MPU) {readAccelAll();} break;  // read MPU6050 accelerometers
    } Task20ms--; // count down the sub-tasks as we loop
   } else if (millis() - next20ms >= 20) {
    // do this every 20ms
    next20ms = millis();  // set next time point
    Task20ms = 6;         // initiate 20ms task sequence
   }


  //###############################################################################
  //
  //  40 ms Print Task - Variable
  //
  //###############################################################################
  // this code uses either the serial port or the WiFi link to send messages
  // at 115200 baud we can send 11.5 chars/ms, so 64 chars takes 5.5ms.
  if (millis() >= print40ms) {
    print40ms = millis() + 40;
    if (PrintTx.length() > 0) {
      // characters in string buffer so send some/all of them
      print40ms = millis() + 6; // shorten the delay for more data to send
      Tx_PingCnt = 0;           // reset the WiFi ping counter
      if (PrintTgt == 0) {
        // default uses the serial port
        if (Serial.availableForWrite() >= 64) {
          if (PrintTx.length() <= 64) {
            Serial.print(PrintTx);
            PrintTx = "";   // empty the buffer
          } else {
            Serial.print(PrintTx.substring(0,64));
            PrintTx = PrintTx.substring(64);
          }
        }
      } else {
      }
    }
  }
}

// ---------------------------------------------------------------------------------

void loop0(void * pvParameters) {
  // this function runs on Core 0, also used for ESP32 WiFi NOW
  // *** WARNING *** do not use for time consuming code like display, I2C, FastLED, etc
  // the following message, sent once at boot, confirms core 0 assignment
  Serial.print("Core0 task running on core "); Serial.println(xPortGetCoreID());

  for(;;){
    // infinite loop, just like loop() but running in Core 0
    // you must never use 'break' or 'return' statement here or it will crash
    // we tell the free RTOS system to call this function every 1 ms
    switch (swPnt) {  
      case  0: readSW0(); coreLB = 3; break;  // read left button switch SW0
      case  1: readSW1(); coreLB = 3; break;  // read middle button switch SW1
      case  2: readSW2(); coreLB = 3; break;  // read right button switch SW2
    }
    swPnt++; if (swPnt > 2) {swPnt = 0;}

    // return to RTOS system with a call back, based on the switch task completed
    // the coreLB values will add to give a loop delat of approx 10ms
    vTaskDelay(coreLB/portTICK_PERIOD_MS);
  }
}

// ---------------------------------------------------------------------------------

void RunPOST() {
  // start-up initialisation processes, mainly testing I2C addresses
  Serial.println("\n\n" + Release + "  " + Date + "\n");

  // BatAvg = BatVol; BatSum = BatVol * 50;  // preload the battery sum for averaging
  // AnyLong = (BatAvg - BatCritical)*100/(BatMax - BatCritical);
  // if (AnyLong > 100) {AnyLong = 100;} // limit max to 100%
  // if (AnyLong < 0) {AnyLong = 0;}     // limit min to 0%
  // Serial.println("Battery " + String(BatVol) + "\t" + String(float(BatAvg)/BatCal) + "v\t" + String(AnyLong) + "%");
  // randomSeed(BatVol); // use the battery voltage as the random seed factor
  // // if battery is less than 5.5v then USB mode is assumed, which makes servos inoperable
  // if (BatAvg <= 2111) {USB = true;}

  // initilaise RGB LED array
  FastLED.clear();
  LED[0].setRGB(3,0,0); // set level shifter red
  FastLED.show();       // This sends the updated pixel color to the hardware.
  Serial.println("RGB LEDs cleared.");

  // Test for MPU 6050 3-axis motion sensor and initialise
  Wire.beginTransmission(MPU_address);
  I2CErr = Wire.endTransmission();
  if (I2CErr == 0) {
    Serial.println("Initialising Gyros...");
    I2C_MPU = true; MPU_6050_Initialise();
    // check orientation for main mode on reset
    Serial.println("Determining mode...");
    for (Any = 0;Any < 10;Any++) {readAccelAll(); delay(10);}
    if (Vertical) {
      TDM = true; Serial.println("TDM mode active"); ModeEn = false;
      SetTdmMode(0);
    } else {
      TDM = false; Serial.println("Normal mode active");
      setMainMode(0);  // start in Mode 0
    }
  } else {I2C_MPU = false; Serial.println("MPU6050 Not Found!");}

  // determine offsets when in TEST mode
  if (TEST && I2C_MPU) {
    // ensure that the sword is placed on a level table for X,Y offsets
    // ensure that the sword is placed at 90 degrees for Z offset
    Serial.println("Collecting offsets...");
    while (true) {
      AcXSum = 0; AcYSum = 0; AcZSum = 0; GyZSum = 0;
      // Ignore the first 50 readings
      for (Any = 0;Any < 50;Any++) {readAccelAll(); readGyroZ(); delay(20);}
      for (Any = 0;Any < 100;Any++) {
        readAccelAll(); AcXSum+= AcX;  AcYSum+= AcY;  AcZSum+= AcZ;
        readGyroZ(); GyZSum+= GyZ;
        delay(20);
      }
      Serial.print("AcXOff= "); Serial.print(AcXSum/100);
      Serial.print("  |  AcYOff= "); Serial.print(AcYSum/100);
      Serial.print("  |  AcZOff= "); Serial.println(AcZSum/100);
      Serial.print("GyZOff= "); Serial.println(GyZSum/100);
    }
  }

  Serial.println("POST completed");

}

// ---------------------------------------------------------------------------------

void setDefaults() {
  // load default values
  AcAvgDiv = 0;           // dividing factor used in averaging
  AcAvgDivMax = 250;      // max dividing factor for averaging
  AcXAvg = 0;             // averaged AcX value to reduce noise
  AcYAvg = 0;             // averaged AcX value to reduce noise
  AcZAvg = 0;             // averaged AcX value to reduce noise
  AngZ = 0;               // gyro Z angle
  Bright = 255;           // FastLED brightness
  Brightness = 255;       // Display brightness
  cmdMode = ' ';          // command mode
  cmdSgn = 1;             // cmdVal sign, 1 or -1
  cmdType = ' ';          // command mode type
  cmdVal = 0;             // value associated with a cmdType
  
  for (Any = 0; Any < NumLEDs; Any++) {colBlu[Any] = 0;}  //colour blue
  for (Any = 0; Any < NumLEDs; Any++) {colGrn[Any] = 0;}//colour green
  for (Any = 0; Any < NumLEDs; Any++) {colRed[Any] = 0;}//colour red

  GyAvgDiv = 0;           // dividing factor used in averaging
  GyCal = 250;            // counter used to determine gyro calibration period
  GyT = 0;                // time of gyro readings in microseconds
  GyX = 0;                // MPU raw X-gyroscope value
  GyXAvg = 0;             // averaged GyX value to reduce noise
  GyXOffCnt = 0;          // offset counter for gyro centering
  GyXSum = 0;             // averaging accumulator
  GyY = 0;                // MPU raw Y-gyroscope value
  GyYAvg = 0;             // averaged GyY value to reduce noise
  GyYOffCnt = 0;          // offset counter for gyro centering
  GyYSum = 0;             // averaging accumulator
  GyZ = 0;                // MPU raw Z-gyroscope value
  GyZAvg = 0;             // averaged GyZ value to reduce noise
  GyZOffCnt = 0;          // offset counter for gyro centering
  GyZSum = 0;             // averaging accumulator
  I2C_MPU = false;        // true if MPU6050 is detected during POST
  JX = 0;                 // global variable used in turning, with Joystick app
  JY = 0;                 // global variable used in turning, with Joystick app
  LED_Cnt = 0;            // counter used in LED sequencing
  LED_Del = 0;            // delay timer used in LED functions
  LED_Period = 20;        // LED timer period, default = 20ms
  LED_Run = false;        // == true if LED_Task is to run
  LEDshw = false;         // LED show flag in TDM mode
  LED_Task = 0;           // LED task pointer
  LedTick = false;        // set true during LED cycles
  LEDVal = 0;             // values used in LED tasks
  mainMode = 0;           // pointer to display mode, default = 0+
  maxBright = 150;        // maximum brightness
  ModeEn = true;          // if == true the do mainMode tasks
  MPUCal = false;         // == true during calibration
  patCol0 = 0;            // any pattern colour
  patTask = 0;            // any pattern task pointer
  patTaskEnd = 0;         // any pattern end pointer
  patTaskFlg = 0;         // any pattern task flag
  patTaskPnt = 0;         // any pattern pointer
  Ping = 0;               // 'ping' counter
  PitchAcc = 0.0;         // pitch angle in degrees base on accelerometer
  PitchGyr = 0.0;         // pitch angle in degrees base on gyro + acc
  PlayCnt = 0;            // any play counter
  PlayDel = 5;            // number of wait cycles in play sequence >= 0
  PlayEn = 0;             // >0 enables play mode
  PlayF = 0;              // any play end finish offset
  PlayI = -1;             // play increment from PlayS to PlayE
  PlayS = 0;              // any play start offset, also used as play pointer
  readWiiCall = false;    // flag prevents call stack overflow
  RotClr = 0;             // auto zero offset counter for Z-axis
  Rx_Chk = 0;             // Rx checksum
  RxCZ = 0;               // buffered received CZ value
  RxJoyX = 0;             // buffered received JoyX value
  RxJoyY = 0;             // buffered received JoyY value
  Rx_len = 0;             // length of WiFi received data block
  RxRec = false ;         // set true when a valid frame of data is received
  RxState = 0;            // receiver state machine state
  Rx_Task = 0;            // pointer user by Rx state machine
  RTxTimeout = 0;         // counter used to auto-reset if WiFi fails
  RxVal = -1;             // value received from serial Rx
  SerialRx = false;       // == true if any character has been received
  subTask = 0;            // sub task pointer used in everey mode
  subTaskNxt = 0;         // next sub task pointer used in everey mode
  sw0Cnt = 0;             // button switch counter
  sw0DwnTime = 0;         // button pressed down time
  sw0LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw0_Nop = false;        // if == true don't perform switch functions
  sw0State = HIGH;        // state of read button switch pin
  sw0Timer = 0;           // timer used to detemine button sequences
  sw0Wup = false;         // set == true for SW0 to wait for button release
  sw1Cnt = 0;             // button switch counter
  sw1LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw1_Nop = false;        // if == true don't perform switch functions
  sw1State = HIGH;        // state of read button switch pin
  sw1Timer = 0;           // timer used to detemine button sequences
  sw1Wup = false;         // set == true for SW1 to wait for button release
  sw2Cnt = 0;             // button switch counter
  sw2DwnTime = 0;         // button pressed down time
  sw2LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw2_Nop = false;        // if == true don't perform switch functions
  sw2State = HIGH;        // state of read button switch pin
  sw2Timer = 0;           // timer used to detemine button sequences
  sw2Wup = false;         // set == true for SW0 to wait for button release
  swPnt = 0;              // switch task poionter for core 0 tasks
  Task20ms = 0;           // 20ms subtask timer
  Task40ms = 0;           // 40ms subtask timer
  taskCnt0 = 0;           // any subtask counter
  taskCnt1 = 0;           // any subtask counter
  taskDel0 = 0;           // any subtask delay counter
  taskOff0 = 0;           // any task offset
  taskSpeed = 0;          // nominal speed of task
  TDM = false;            // == true if in TDM mode
  TdmBeat = 0;            // TDM mode LED[0] red value
  TDMcnt = 0;             // counter used to hold off heartbeat when moving
  TdmInc = 1;             // increment used in TDM tasks
  Tx_PingCnt = 0;         // 1 second Tx ping timer
  UpRht = true;           // set == true by AcZ when the robot is the correct way up
  USB = false;            // == true if initial battery is below 5.5v
  Vertical = false;       // == true if AcY >= 5500
  VertLast = false;       // previous Vertical state
  YawAcc = 0.0;           // yaw angle in degrees based on accelerometers
  YawGyr = 0.0;           // yaw angle in degrees based on gyro + acc
}

// ---------------------------------------------------------------------------------

void synchLoopTimers(unsigned long zAdvance) {
  // reset or adjust main loop timers, based on zAdvance
  // if  0 then a 2ms phase offset is introduced to prevent simultaneous triggering
  // if >0 then add zAdvance to the current timer values
  if (zAdvance == 0) {
    // setting up fresh loop timers
    while (millis() < 10) {}    // let millis() get some time on
    
    LED_Timer = millis() + 26;  // reset 20+ms loop timer with 3ms phase to 10ms timer
    loopDelay = millis();
    next5ms = millis() - 1;     // reset 5ms task timer with 1ms phase to loopDelay
    next20ms = millis() - 6;    // reset 20ms task timer with 4ms phase to loopDelay
    next40ms = millis() - 8;    // reset 40ms task timer with 4ms phase to loopDelay
    print40ms = millis() + 46;  // reset 40ms printer loop timer with 4ms phase to loopDelay
  } else {
    // advancing loop timers
    LED_Timer += zAdvance;
    loopDelay += zAdvance;
    next5ms += zAdvance;
    next20ms += zAdvance;
    next40ms += zAdvance;
    print40ms += zAdvance;
  }
}

// --------------------------------------------------------------------------------
