// ######################################################################
//
//  Magical Sword v0.08 Beta
//
//  Released:  01/06/2016
//
//  Author: TechKnowTone
//
// ######################################################################
/*
    This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any
    purpose.

    Elements of this software were taken from the public domain. It uses
    libraries which must be installed on your system.

    This program writes patterns to a row of RGB LEDs mounted on a
    simulated sword blade. This version reads up/down switches in all modes
    and inclides addiotnal fixed/ramdom patterns.

    Mode 0 - default random pattern
    Mode 1 - fixed pattern seqeunce
    Mode 2 - random patterns and speeds
    Mode 3 - accelerometer bars X,Y,Z
    Mode 4 - accelerometer moving stripes
    Mode 5 - random fixed patterns if moving
    Mode 6 - faster random colours if moving
    Mode 7 - test tasks
*/
// Declare library functions
#include <Adafruit_NeoPixel.h>
#include <Wire.h>

// Define labels and constants
#define ColBlu 0x000088;
#define ColCya 0x004444;
#define ColGrn 0x008800;
#define ColPur 0x440044;
#define ColRed 0x880000;
#define ColYel 0x444400;
#define ledPin 12 // RGB LED serial pin
#define ledNum 15 // number of RGB LEDs connected
#define sw0Pin 5 // switch 0 input
#define sw1Pin 6 // switch 1 input - Mode switch
#define sw2Pin 7 // switch 2 input

// MPU variables
const int MPU_addr=0x68;  // I2C address of the MPU-6050
int16_t AcX;  // MPU raw X-acceleration value
int16_t AcXL;  // last X-acceleration value
int16_t AcY;  // MPU raw Y-acceleration value
int16_t AcYL;  // last Y-acceleration value
int16_t AcZ;  // MPU raw Z-acceleration value
int16_t AcZL;  // last Z-acceleration value

// NeoPixel parameters
// Parameter 1 = number of pixels in strip
// Parameter 2 = pin number (most are valid)
// Parameter 3 = pixel type flags, add together as needed:
//   NEO_RGB     Pixels are wired for RGB bitstream
//   NEO_GRB     Pixels are wired for GRB bitstream
//   NEO_KHZ400  400 KHz bitstream (e.g. FLORA pixels)
//   NEO_KHZ800  800 KHz bitstream (e.g. High Density LED strip)
//Adafruit_NeoPixel strip = Adafruit_NeoPixel(ledNum, ledPin, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel strip = Adafruit_NeoPixel(ledNum, ledPin, NEO_GRB + NEO_KHZ400);

// Declare and initialise global variables
uint8_t colBlu[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; //colour blue
uint8_t colGrn[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; //colour green
uint8_t colRed[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; //colour red
unsigned long interval; // main loop interval in microseconds
int mainMode; // pointer to display mode, default = 0+
int maxBright; // maximum brightness
int maxBright2; // half of maximum brightness
int maxBright3; // third of maximum brightness
unsigned long nextMicros; // main loop interval in microseconds
int overFlow; // main loop overflow warning flag
int overFlowEn; // main loop overflow warning flag enable
long patCol0; // any pattern colour
int patTask; // any pattern task pointer
int patTaskEnd; // any pattern end pointer
int patTaskFlg; // any pattern task flag
int patTaskNxt; // any pattern task pointer record
int patTaskPnt; // any pattern pointer
int PlayCnt; // any play counter
int PlayDel; // number of wait cycles in play sequence >= 0
int PlayEn; // >0 enables play mode
int PlayF; // any play end finish offset
int PlayI; // play increment from PlayS to PlayE
int PlayS; // any play start offset, also used as play pointer
int swPress; // >0 if switch pressed
int subTask; // sub task pointer used in everey mode
int subTaskNxt; // next sub task pointer used in everey mode
int swCnt; // number of switch presses
int swDel; // delay between siwtch reads
int swEn; // >0 enables up/down switch reading
int swRead; // latest input value
int swState; //previous switch state
int swTimeout; // inactivity timeout timer
int swVal; // temporary switch value or inactivity timer
int taskCnt0; // any subtask counter
int taskCnt1; // any subtask counter
int taskDel0; // any subtask delay counter
int taskInc0; // any task increment
int taskOff0; // any task offset
int taskSpeed; // nominal speed of task

void setup() {
  // put your setup code here, to run once:
  setDefaults();
  Serial.begin(9600);
  pinMode(sw0Pin, INPUT_PULLUP);
  pinMode(sw1Pin, INPUT_PULLUP);
  pinMode(sw2Pin, INPUT_PULLUP);
  strip.begin();
  strip.show(); // Initialize all pixels to 'off'
  delay(100);
  setMode(0); // start in Mode 0
}

// ----------------------------------------------------------------------

void loop() {
  // main loop runs every 20 ms (50Hz):
  doModeTask();
  if (PlayEn > 0) {playTask();}
  readModeSwitch();
  readAccelAll();
  if (swEn > 0) {readUpDwnSwitches();}
  while (micros() < nextMicros) {
    // wait for remainder of 20ms period to sync overall period
    overFlow = 0;
  } nextMicros = micros() + interval;
  if (overFlow > 0 && overFlowEn > 0) {Serial.println("Loop Overflow!");};
  overFlow = 1;
}

// ----------------------------------------------------------------------

void calAccVals() {
  // converts the raw accelerometer values as -1 to 15  
  AcX = ((abs(AcX - 128))/4) - 16; AcX = max(AcX,-1); AcX = min(AcX,14); // calibrate AcX
  AcY = 16 - ((abs(AcY - 128))/5); AcY = max(AcY,-1); AcY = min(AcY,14); // calibrate AcY
  AcZ = 16 - ((abs(AcZ - 128))/5); AcZ = max(AcZ,-1); AcZ = min(AcZ,14); // calibrate AcZ
//      Serial.print("AX="); Serial.print(AcX);
//      Serial.print("  |  AY="); Serial.print(AcY);
//      Serial.print("  |  AZ="); Serial.print(AcZ);
//      Serial.print("\n");
}

// ----------------------------------------------------------------------

void clrArray(){
  // sets all array colour values to 0
  for (int zP = 0; zP < 15; zP++) {colRed[zP] = 0; colGrn[zP] = 0; colBlu[zP] = 0;}
}

// ----------------------------------------------------------------------

void cls(){
  // sets all array colour values to 0 and clear the display
  clrArray(); showColArray(0);
}

// ----------------------------------------------------------------------

void doModeTask() {
  // main task routine  
  switch(mainMode) {
    case 0:
      // Mode 0 - default random pattern
      doModeTask0(); break;
    
    case 1:
      // Mode 1 - fixed pattern sequnce
      doModeTask1(); break;

    case 2:
      // Mode 2 - random patterns and speeds
      doModeTask2(); break;

    case 3:
      // Mode 3 - accelerometer bars X,Y,Z
      doModeTask3(); break;

    case 4:
      // Mode 4 - accelerometer moving stripes
      doModeTask4(); break;

    case 5:
      // Mode 5
      doModeTask5(); break;

    case 6:
      // Mode 6
      doModeTask6(); break;

    case 7:
      // Mode 7 - test tasks
      doModeTask7(); break;

     case 80:
      // switch modes based on mode switch count
      setMode(swCnt); swCnt = 0; break;

    case 99:
      break; // null task called when the Mode switch is pressed
  }
}

// ----------------------------------------------------------------------

void getRndCol() {
  // generate a random colour for patCol0
  int zCol = random(7); // choose colour at random
  long zmaxBright = maxBright;
  long zmaxBright2 = maxBright2;
  long zmaxBright3 = maxBright3;
  switch(zCol) {
    case 0: patCol0 = zmaxBright<<16; break;
    case 1: patCol0 = (zmaxBright2<<16) + (zmaxBright2<<8); break;
    case 2: patCol0 = (zmaxBright<<8); break;
    case 3: patCol0 = (zmaxBright2<<8) + zmaxBright2; break;
    case 4: patCol0 = zmaxBright; break;
    case 5: patCol0 = (zmaxBright2<<16) + zmaxBright2; break;
    case 6: patCol0 = (zmaxBright3<<16) + (zmaxBright3<<8) + zmaxBright3; break;
  }
}

// ----------------------------------------------------------------------

void loadColArray(long zC0,long zC1,long zC2,long zC3,long zC4,long zC5,long zC6,long zC7,long zC8,long zC9,long zC10,long zC11,long zC12,long zC13,long zC14) {
  // loads 15 hexadecimal colours into the colour array
  colBlu[0] = zC0 & 0xFF; colGrn[0] = (zC0>>8) & 0xFF; colRed[0] = (zC0>>16) & 0xFF;
  colBlu[1] = zC1 & 0xFF; colGrn[1] = (zC1>>8) & 0xFF; colRed[1] = (zC1>>16) & 0xFF;
  colBlu[2] = zC2 & 0xFF; colGrn[2] = (zC2>>8) & 0xFF; colRed[2] = (zC2>>16) & 0xFF;
  colBlu[3] = zC3 & 0xFF; colGrn[3] = (zC3>>8) & 0xFF; colRed[3] = (zC3>>16) & 0xFF;
  colBlu[4] = zC4 & 0xFF; colGrn[4] = (zC4>>8) & 0xFF; colRed[4] = (zC4>>16) & 0xFF;
  colBlu[5] = zC5 & 0xFF; colGrn[5] = (zC5>>8) & 0xFF; colRed[5] = (zC5>>16) & 0xFF;
  colBlu[6] = zC6 & 0xFF; colGrn[6] = (zC6>>8) & 0xFF; colRed[6] = (zC6>>16) & 0xFF;
  colBlu[7] = zC7 & 0xFF; colGrn[7] = (zC7>>8) & 0xFF; colRed[7] = (zC7>>16) & 0xFF;
  colBlu[8] = zC8 & 0xFF; colGrn[8] = (zC8>>8) & 0xFF; colRed[8] = (zC8>>16) & 0xFF;
  colBlu[9] = zC9 & 0xFF; colGrn[9] = (zC9>>8) & 0xFF; colRed[9] = (zC9>>16) & 0xFF;
  colBlu[10] = zC10 & 0xFF; colGrn[10] = (zC10>>8) & 0xFF; colRed[10] = (zC10>>16) & 0xFF;
  colBlu[11] = zC11 & 0xFF; colGrn[11] = (zC11>>8) & 0xFF; colRed[11] = (zC11>>16) & 0xFF;
  colBlu[12] = zC12 & 0xFF; colGrn[12] = (zC12>>8) & 0xFF; colRed[12] = (zC12>>16) & 0xFF;
  colBlu[13] = zC13 & 0xFF; colGrn[13] = (zC13>>8) & 0xFF; colRed[13] = (zC13>>16) & 0xFF;
  colBlu[14] = zC14 & 0xFF; colGrn[14] = (zC14>>8) & 0xFF; colRed[14] = (zC14>>16) & 0xFF;
}

// --------------------------------------------------------------------

void readAccelAll() {
  // read the X,Y,Z-axis acceleration values from the MPU
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3B);  // starting with register 0x3B (ACCEL_XOUT_H)
  Wire.endTransmission(false);
  Wire.requestFrom(MPU_addr,6,true);  // request 1 register read
  AcX=Wire.read()<<8|Wire.read();  // 0x3B (ACCEL_XOUT_H) & 0x3C (ACCEL_XOUT_L)
  AcY=Wire.read()<<8|Wire.read();  // 0x3D (ACCEL_YOUT_H) & 0x3E (ACCEL_YOUT_L)
  AcZ=Wire.read()<<8|Wire.read();  // 0x3F (ACCEL_ZOUT_H) & 0x40 (ACCEL_ZOUT_L)
  AcX=abs((16384 - AcX)/128);  // condition AcX
  AcY=abs((16384 - AcY)/128);  // condition AcY
  AcZ=abs((16384 - AcZ)/128);  // condition AcZ
//  Serial.print("AX="); Serial.print(AcX);
//  Serial.print("  |  AY="); Serial.print(AcY);
//  Serial.print("  |  AZ="); Serial.print(AcZ);
//  Serial.print("\n");
}

// --------------------------------------------------------------------

void readAccelX() {
  // read the X-axis acceleration value from the MPU
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3B);  // starting with register 0x3B (ACCEL_XOUT_H)
  Wire.endTransmission(false);
  Wire.requestFrom(MPU_addr,2,true);  // request 1 register read
  AcX=Wire.read()<<8|Wire.read();  // 0x3B (ACCEL_XOUT_H) & 0x3C (ACCEL_XOUT_L)
  AcX=abs((16384 - AcX)/128);  // condition AcX
}

// --------------------------------------------------------------------

void readAccelY() {
  // read the Y-axis acceleration value from the MPU
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3D);  // starting with register 0x3D (ACCEL_YOUT_H)
  Wire.endTransmission(false);
  Wire.requestFrom(MPU_addr,2,true);  // request 1 register read
  AcY=Wire.read()<<8|Wire.read();  // 0x3D (ACCEL_YOUT_H) & 0x3E (ACCEL_YOUT_L)
  AcY=abs((16384 - AcY)/128);  // condition AcY
}

// --------------------------------------------------------------------

void readAccelZ() {
  // read the Z-axis acceleration value from the MPU
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3F);  // starting with register 0x3F (ACCEL_ZOUT_H)
  Wire.endTransmission(false);
  Wire.requestFrom(MPU_addr,2,true);  // request 1 register read
  AcZ=Wire.read()<<8|Wire.read();  // 0x3F (ACCEL_ZOUT_H) & 0x40 (ACCEL_ZOUT_L)
  AcZ=abs((16384 - AcZ)/128);  // condition AcZ
}

// --------------------------------------------------------------------

void readDwnSwitch() {
  // read down switch and adjust speed
  if (digitalRead(sw2Pin) == LOW) {
      // down switch pressed
      if (taskSpeed > 1) {taskSpeed--;}
      swPress = 1; setPixelOFF();
      for (int zP=0; zP<taskSpeed; zP++) {setPixiCol(zP,150,0,0);}
      while(digitalRead(sw2Pin) == LOW){
      // wait for release of switch
        delay(10); // 10ms sample period
      } delay(100); // debounce on exit
  }
}

// --------------------------------------------------------------------

void readModeSwitch() {
  // reads a single switch and counts the number of presses
  swRead = digitalRead(sw1Pin);
  if (swRead != swState) {
    // a transition has occured!
    if (swRead == LOW) {
      // switch depressed
      swCnt++; swVal = 0; mainMode = 99; setPixelOFF(); PlayEn = 0; // stop other tasks
      // display count in binary
      if (swCnt == 1) {setPixiCol(0,80,0,0); setPixiCol(1,0,0,0); setPixiCol(2,0,0,0);}
      if (swCnt == 2) {setPixiCol(0,0,0,0); setPixiCol(1,80,0,0); setPixiCol(2,0,0,0);}
      if (swCnt == 3) {setPixiCol(0,80,0,0); setPixiCol(1,80,0,0); setPixiCol(2,0,0,0);}
      if (swCnt == 4) {setPixiCol(0,0,0,0); setPixiCol(1,0,0,0); setPixiCol(2,80,0,0);}
      if (swCnt == 5) {setPixiCol(0,80,0,0); setPixiCol(1,0,0,0); setPixiCol(2,80,0,0);}
      if (swCnt == 6) {setPixiCol(0,0,0,0); setPixiCol(1,80,0,0); setPixiCol(2,80,0,0);}
      if (swCnt == 7) {setPixiCol(0,80,0,0); setPixiCol(1,80,0,0); setPixiCol(2,80,0,0);}
      if (swCnt > 7) setPixelOFF();
      Serial.println(swCnt);
    }
  swState = swRead; // remember current state
  }
  if (swCnt > 0) {
    // switch pressed once or more
    swVal++; // increment timeout delay
    if (swVal >= 75) {
      // no more changes after ~1.5 seconds
      if (swRead == LOW) swCnt = 0; // long press so RESET
      if (swCnt > 7) swCnt = 0; // RESET mode on overflow
      swVal = 0; mainMode = 80; setPixelOFF(); waitSwitchUp();
    }
  }
}

// --------------------------------------------------------------------

void readUpDwnSwitches() {
  // called from main loop to read the up/down switches  
  swPress = 0; readUpSwitch(); readDwnSwitch();
  if (swPress > 0) {
    swTimeout = 50;
    while (swTimeout > 0) {
      swPress = 0; readUpSwitch(); readDwnSwitch();
      if (swPress > 0) {swTimeout = 50;}
      swTimeout--; delay(20);
    }
  } overFlow = 0; // ignore overflow on this cycle
}

// --------------------------------------------------------------------

void readUpSwitch() {
  // read up switch and adjust speed
  if (digitalRead(sw0Pin) == LOW) {
    // up switch pressed
    if (taskSpeed < 15) {taskSpeed++;}
    swPress = 1; setPixelOFF();
    for (int zP=0; zP<taskSpeed; zP++) {setPixiCol(zP,150,0,0);}
    while(digitalRead(sw0Pin) == LOW){
      // wait for release of switch
      delay(10); // 10ms sample period
    } delay(100); // debounce on exit
  }
}

// ----------------------------------------------------------------------

void RotColArrLft() {
  // rotate the contents of the colour array towards the sword point
  int zPP;
  uint8_t zR = colRed[14]; uint8_t zG = colGrn[14]; uint8_t zB = colBlu[14];
  for (int zP = 14; zP > 0; zP--) {
    zPP = zP-1; colRed[zP] = colRed[zPP]; colGrn[zP] = colGrn[zPP]; colBlu[zP] = colBlu[zPP];
  } colRed[0] = zR; colGrn[0] = zG; colBlu[0] = zB;
}

// ----------------------------------------------------------------------

void RotColArrRht() {
  // rotate the contents of the colour array towards the sword handle
  int zPP;
  uint8_t zR = colRed[0]; uint8_t zG = colGrn[0]; uint8_t zB = colBlu[0];
  for (int zP = 0; zP < 14; zP++) {
    zPP = zP+1; colRed[zP] = colRed[zPP]; colGrn[zP] = colGrn[zPP]; colBlu[zP] = colBlu[zPP];
  } colRed[14] = zR; colGrn[14] = zG; colBlu[14] = zB;
}

// ----------------------------------------------------------------------

void runPOST() {
  // indicate all LEDs are working
  Serial.print("\n\n\n\n\n\n\n\n\n\n"); //CLS
  Serial.println("Starting POST...");
  for (int zLED = 0; zLED < 15; zLED++) {
    strip.setPixelColor(zLED,strip.Color(maxBright, 0, 0)); strip.show();
    delay(50);
  }
  for (int zLED = 0; zLED < 15; zLED++) {
    strip.setPixelColor(zLED,strip.Color(0, maxBright, 0)); strip.show();
    delay(50);
  }
  for (int zLED = 0; zLED < 15; zLED++) {
    strip.setPixelColor(zLED,strip.Color(0, 0, maxBright)); strip.show();
    delay(50);
  }
  for (int zLED = 0; zLED < 15; zLED++) {
    strip.setPixelColor(zLED,strip.Color(0, 0, 0)); strip.show();
    delay(50);
  }
  wakeUpMPU();
  Serial.println("...POST completed");
  overFlow = 0; // ignore overflow on first cycle
}

// ----------------------------------------------------------------------

void setColVal(int zP,long zC) {
  // set an array point zP to a specific Hex value zC
  colBlu[zP] = zC & 0xFF; colGrn[zP] = (zC>>8) & 0xFF; colRed[zP] = (zC>>16) & 0xFF;
}

// ----------------------------------------------------------------------

void setDefaults() {
  // set default variable values
  int zP;
  for (zP = 0; zP < 15; zP++) {colBlu[zP] = 0;}//colour blue
  for (zP = 0; zP < 15; zP++) {colGrn[zP] = 0;}//colour green
  for (zP = 0; zP < 15; zP++) {colRed[zP] = 0;}//colour red
  interval = 20000; // main loop interval in microseconds
  mainMode = 0; // pointer to display mode, default = 0+
  maxBright = 150; // maximum brightness
  nextMicros = micros(); // main loop count interval in microseconds
  overFlowEn = 1; // main loop overflow warning flag enable
  patCol0 = 0; // any pattern colour
  patTask = 0; // any pattern task pointer
  patTaskEnd = 0; // any pattern end pointer
  patTaskFlg = 0; // any pattern task flag
  patTaskPnt = 0; // any pattern pointer
  PlayCnt = 0; // any play counter
  PlayDel = 5; // number of wait cycles in play sequence >= 0
  PlayEn = 0; // >0 enables play mode
  PlayF = 0; // any play end finish offset
  PlayI = -1; // play increment from PlayS to PlayE
  PlayS = 0; // any play start offset, also used as play pointer
  subTask = 0; // sub task pointer used in everey mode
  subTaskNxt = 0; // next sub task pointer used in everey mode
  swCnt = 0; // number of switch presses
  swDel = 20; // delay between switch reads
  swEn = 1; // >0 enables up/down switch reading
  swRead = HIGH; // latest input value
  swState = HIGH; //previous switch state
  swTimeout = 0; // inactivity timeout timer
  swVal = 0; // temporary switch value or inactivity timer
  taskCnt0 = 0; // any subtask counter
  taskCnt1 = 0; // any subtask counter
  taskDel0 = 0; // any subtask delay counter
  taskOff0 = 0; // any task offset
  taskSpeed = 0; // nominal speed of task
}

// ----------------------------------------------------------------------

void setMode(int zMode) {
  // called to set starting conditions for a specific mode
  switch(zMode) {
    case 0:
      // Mode 0 - default random pattern
      setDefaults(); runPOST();
      maxBright = 10; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 0; swEn = 1; taskCnt1 = 1; taskInc0 = 10; taskSpeed = 1; patTask = 0; overFlowEn = 1;
      break;
    case 1:
      // Mode 1 - fixed pattern sequnce
      maxBright = 128; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 0; swEn = 1; taskSpeed = 15; patTask = 0; overFlowEn = 1;
      break;
    case 2:
      // Mode 2 - random patterns and speeds
      maxBright = 128; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 0; swEn = 1; taskSpeed = 15; patTask = 0; overFlowEn = 1;
      break;
    case 3:
      // Mode 3 - accelerometer bars X,Y,Z
      maxBright = 128; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 0; swEn = 0; taskCnt1 = 1; taskInc0 = 10; taskSpeed = 1; patTask = 0; overFlowEn = 1;
      break;
    case 4:
      // Mode 4 - accelerometer moving stripes
      maxBright = 10; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 0; swEn = 1; taskCnt1 = 1; taskInc0 = 10; taskSpeed = 15; patTask = 0; overFlowEn = 1;
      break;
    case 5:
      // Mode 5 - random fixed pattern if moving
      maxBright = 128; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 0; swEn = 1; taskSpeed = 15; patTask = 0; overFlowEn = 1;
      break;
    case 6:
      // Mode 6 - faster random colours if moving
      maxBright = 10; maxBright2 = maxBright/2; maxBright3 = maxBright/3;
      subTask = 99; swEn = 1; taskCnt0 = 50; taskSpeed = 15; patTask = 0; overFlowEn = 1;
      break;
    case 7:
      // Mode 7 - test tasks
      subTask = 0; overFlowEn = 0; break;
  }
  mainMode = zMode; subTask = 0;
}

// ----------------------------------------------------------------------

void setPixelBlu(int zP,uint8_t zVal) {
  // set the pixel zP Blue colour to zVal
  colRed[zP] = 0; colGrn[zP] = 0; colBlu[zP] = zVal;
  strip.setPixelColor(zP,strip.Color(colRed[0], colGrn[0], colBlu[0]));
  strip.show();
}

// ----------------------------------------------------------------------

void setPixelGrn(int zP,uint8_t zVal) {
  // set the pixel zP Green colour to zVal
  colRed[zP] = 0; colGrn[zP] = zVal; colBlu[zP] = 0;
  strip.setPixelColor(zP,strip.Color(colRed[zP], colGrn[zP], colBlu[zP]));
  strip.show();
}

// ----------------------------------------------------------------------

void setPixelOFF(){
  // turns OFF all LED values
  for (int zP = 0; zP < 15; zP++) {setPixiCol(zP,0,0,0);}
}

// ----------------------------------------------------------------------

void setPixelRed(int zP,uint8_t zVal) {
  // set the pixel zP Red colour to zVal
  colRed[zP] = zVal; colGrn[zP] = 0; colBlu[zP] = 0;
  strip.setPixelColor(zP,strip.Color(colRed[zP], colGrn[zP], colBlu[zP]));
  strip.show();
}

// ----------------------------------------------------------------------

void setPixiCol(int zP, uint8_t zRed, uint8_t zGrn, uint8_t zBlu) {
  // set the pixel zP colour values and the array
  colRed[zP] = zRed; colGrn[zP] = zGrn; colBlu[zP] = zBlu;
  strip.setPixelColor(zP,strip.Color(zRed, zGrn, zBlu));
  strip.show();
}

// ----------------------------------------------------------------------

void showColArray(int zOff) {
  // loads the colour array values into the strip and shows it, with an
  // offset zOff
  for (int zP = 0; zP < 15; zP++) {
    int zAdd = zP + zOff;
    if (zAdd >= 0 && zAdd <=14) {
      strip.setPixelColor(zP,strip.Color(colRed[zAdd], colGrn[zAdd], colBlu[zAdd]));
    } else {
      strip.setPixelColor(zP,strip.Color(0,0,0));
    }
  } strip.show();
}

// ----------------------------------------------------------------------

void waitSwitchUp() {
  // wait here while switch is pressed
  while(digitalRead(sw1Pin) == LOW){
    delay(10); // 10ms sample period
  } delay(10); // debounce on exit
}

// ----------------------------------------------------------------------

void wakeUpMPU() {
  // initialize the MPU6050
  Serial.println("Starting MPU.");
  Wire.begin();
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x6B);  // PWR_MGMT_1 register
  Wire.write(0);     // set to zero (wakes up the MPU-6050)
  Wire.endTransmission(true);
}


