// ################################################################################
//
//  TrackBot_2 v0.13
//
//  Released:  09/01/2020
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls a line tracking robot. It reads analogue values from IR
    sensors which are powered from separate digital outputs, so that the light from
    one does not interfer with an adjacent sensor. Each sensor is switched ON in
    turn and a reading taken just before being switched OFF again. Time is needed for
    the sensor output to settle down

    This version contains:
    * auto-assisted calibration of each servos deadband potentiometer
    * auto-assisted calibration of wheel servo stall current
    * auto-assisted sensor black/white calibration measurements
    * automatic rotational sensor black/white calibration measurements
    * centre sensor acts as a tachometer counter for rotational measurements
    * calibration coefficients are store in EEPROM
    * propotional steering using the analogue A0 sensor outputs
    * sustained cornering for the prevention of overshoots
    * battery health measurements and auto-shutdown
    * start with 'no line' but steer once centre sensor sees a track
    * steering bias algorithm to compensate for motor mis-matches
    * track cross-over allowances, prior to emergency stops

*/
// Declare Libraries
#include <EEPROM.h>
#include <Servo.h>

// Define constants
#define BackTimerRes 1250     // reset value for LED background timer
#define BattLowLim 540        // critical limit or battery voltage
#define EEPROM_Key B10101010  // EEPROM valid key, normally B10101010
#define LED_OffMax 4294967295 // a value which can not be exceeded by micros()
#define PrintEn false         // set = true to continuously print something
#define PWM_Max 1800          // maximum PWM value to be applied to servos
#define PWM_LftSpd0Def 1520   // default left forwards stall value
#define PWM_LftSpd1Def 1530   // default left forward speed 1 value
#define PWM_LftSpd2Def 1540   // default left forward speed 2 value
#define PWM_LftSpd3Def 1550   // default left forward speed 3 value
#define PWM_LftSpd4Def 1560   // default left forward speed 4 value
#define PWM_LftSpd5Def 1570   // default left forward speed 5 value
#define PWM_LftSpd6Def 1580   // default left forward speed 6 value
#define PWM_LftSpd7Def 1590   // default left forward speed 7 value
#define PWM_RhtSpd0Def 1520   // default right forwards stall value
#define PWM_RhtSpd1Def 1530   // default right forward speed 1 value
#define PWM_RhtSpd2Def 1540   // default right forward speed 2 value
#define PWM_RhtSpd3Def 1550   // default right forward speed 3 value
#define PWM_RhtSpd4Def 1560   // default right forward speed 4 value
#define PWM_RhtSpd5Def 1570   // default right forward speed 5 value
#define PWM_RhtSpd6Def 1580   // default right forward speed 6 value
#define PWM_RhtSpd7Def 1590   // default right forward speed 7 value
#define PWM_StallOffset 4     // a value allowing for user pressing the button
#define PWM_Start 1570        // when offset calibrated this should start servoes in a forward direction
#define PWM_StopRef 1500      // nominal PWM no drive value
#define Sen_LvlMax 160        // max sensor output for black line
#define Sen_LvlMin 35         // set at sensor minimum level + 5
#define Sen_LvlOff 500        // robot lifted off the table threshold
#define TachoCntMax 2500      // Tacho max period set at 10 seconds

// Define I/O pins
#define BattPin A6            // analogue pin connected to potential divider
#define LED_Brk 13            // brake LED pin
#define LED_Lft 12            // left LED pin
#define LED_Ctr 11            // centre LED pin
#define LED_Rht 8             // right LED pin
#define Sens_LftInp A3        // left sensor analog pin
#define Sens_LftPwr A0        // left sensor pin
#define Sens_CtrInp A4        // left sensor analog pin
#define Sens_CtrPwr A1        // centre sensor pin
#define Sens_RhtInp A5        // right sensor analog pin
#define Sens_RhtPwr A2        // right sensor pin
#define ServoLftPin 9         // left-hand servo pin
#define ServoRhtPin 10        // left-hand servo pin
#define SWL_Pin 6             // pin connected to SWL button switch
#define SWR_Pin 7             // pin connected to SWR button switch

// Declare variables for Servo Test functions, which can be removed later if code
// space is limited, once the servos have been calibrated.
int Angle;  // start with servo in the centre pos
char cmdMode; // command mode
char cmdType; // command type
int cmdVal; // value associated with a cmdType
char keyChar; // any keyboard character
int keyVal; // any keyboard value
int servoLL = 544; // servo min time setting
int servoPin;  // servo output pin
int servoUL = 2400; // servo max time setting
int timeDelay = 1; // delay in milliseconds
Servo servoMain; // Define our Servo


// Declare and initialise global variables
int AnyInt;                   // temporary integer value
int BackTimer;                // background timer used to display Curve & Speed settings
int BatteryAv;                // averaged battery voltage
int BattLow = BattLowLim;     // set battery critical limit
unsigned int Battery_Sum;     // accumulator for battery averaging
int BatteryVal;               // battery voltage reading
byte Brake;                   // a timer used in soft braking
byte BrakeMx;                 // emergency stop count threshold
byte BrakeT1;                 // Speed 1 emergency stop trigger
byte BrakeT2;                 // Speed 2 emergency stop trigger
byte BrakeT3;                 // Speed 3 emergency stop trigger
byte BrakeT4;                 // Speed 4 emergency stop trigger
byte BrakeT5;                 // Speed 5 emergency stop trigger
byte BrakeT6;                 // Speed 6 emergency stop trigger
byte BrakeT7;                 // Speed 7 emergency stop trigger
byte Curve;                   // value 1 - 7 for turning rate
byte CurveLast;               // previous value 1 - 7 for turning rate
bool IgnoreLift;              // set = true to ignore lifted auto-stop
unsigned long Interval = 1333;// set main loop frequency at 750 Hz
byte LED_CTR;                 // flag controlling centre LED ON/OFF state
bool LED_BRK;                 // flag controlling brake LED ON/OFF state
int LED_LvlMax;               // max level used by LED driver code
byte LED_LFT;                 // flag controlling left LED ON/OFF state
byte LED_Mask;                // if > 0 then right 3 bits define the L-C-R ON/OFF states
unsigned long LED_OffTime;    // timer used to dim and turn OFF LEDs
bool LED_ON;                  // flag used in LED brightness control
byte LED_RHT;                 // flag controlling right LED ON/OFF state
bool Lifted;                  // true if lifted off the surface
bool LiftedLast;              // previous Lifted state to track lifted transition
int LiftedLvl;                // sensor limit for being lifted off the surface
byte MainMode;                // mode of operation
byte MaskCnt;                 // reset mask counter for LED flashing
unsigned long nextMicros;     // main loop event timer in microseconds
int NoLineCnt;                // timer counter used in line tracking mode
int NoLineSTOP;               // speed dependant no line stopping count threshold
byte Phase;                   // sensor phase pointer
bool PWM_Change;              // set true if a PWM value has changed
int PWM_Lft;                  // left servo angle in microseconds
float PWM_LftGain;            // speed dependant steering gain multiplier
int PWM_LftInc;               // left servo incrementing value
int PWM_LftSpeed;             // max left wheel PWM speed used in line tracking mode
int PWM_LftSpd0;              // calibrated stall speed
int PWM_LftSpd1;              // calibrated speed 1 PWM
int PWM_LftSpd2;              // calibrated speed 2 PWM
int PWM_LftSpd3;              // calibrated speed 3 PWM
int PWM_LftSpd4;              // calibrated speed 4 PWM
int PWM_LftSpd5;              // calibrated speed 5 PWM
int PWM_LftSpd6;              // calibrated speed 6 PWM
int PWM_LftSpd7;              // calibrated speed 7 PWM
int PWM_Rht;                  // right servo angle in microseconds
float PWM_RhtGain;            // speed dependant steering gain multiplier
int PWM_RhtInc;               // left servo incrementing value
int PWM_RhtSpeed;             // max right wheel PWM speed used in line tracking mode
int PWM_RhtSpd0;              // calibrated stall speed
int PWM_RhtSpd1;              // calibrated speed 1 PWM
int PWM_RhtSpd2;              // calibrated speed 2 PWM
int PWM_RhtSpd3;              // calibrated speed 3 PWM
int PWM_RhtSpd4;              // calibrated speed 4 PWM
int PWM_RhtSpd5;              // calibrated speed 5 PWM
int PWM_RhtSpd6;              // calibrated speed 6 PWM
int PWM_RhtSpd7;              // calibrated speed 7 PWM
int PWM_Throttle;             // a PWM value depenant on Speed
bool Sen_CTR;                 // state of sensor based on threshold value
bool Sen_CtrLast;             // previous sensor state
int Sen_CtrLvlMax;            // maximum 'black' value for centre sensor
int Sen_CtrLvlMin;            // minimum 'white' value for centre sensor
int Sen_CtrThreshLvl;         // 30% threshold level for centre sensor
int Sen_CtrVal;               // analogue value read from centre sensor
bool Sen_LFT;                 // state of sensor based on threshold value
bool Sen_LftLast;             // previous sensor state
int Sen_LftLvlMax;            // maximum 'black' value for left sensor
int Sen_LftLvlMin;            // minimum 'white' value for left sensor
int Sen_LftThreshLvl;         // 30% threshold level for left sensor
int Sen_LftVal;               // analogue value read from left sensor
bool Sen_RHT;                 // state of sensor based on threshold value
bool Sen_RhtLast;             // previous sensor state
int Sen_RhtLvlMax;            // maximum 'black' value for right sensor
int Sen_RhtLvlMin;            // minimum 'white' value for right sensor
int Sen_RhtThreshLvl;         // 30% threshold level for right sensor
int Sen_RhtVal;               // analogue value read from right sensor
bool Sensor_Cal;              // true if in sensor calibration mode, otherwise false
bool Servos_Attached;         // true when servos are attached
byte Speed;                   // value 1 - 7 for car speed setting
int SpdSt;                   // historic flag which tracks speed increase/decrease
int SpdUp;                   // flag which tracks speed increase/decrease
int SWL_Cnt;                  // button pressed counter
bool SWL_Last;                // previous state of button SWL
byte SWL_Num;                 // number of left button presses count        
bool SWL_Skip;                // flag used to skip button actions
bool SWL_State;               // current read state of button SWL
int SWR_Cnt;                  // button pressed counter
bool SW_Rel;                  // flags used in checking both button switches released
bool SWR_Last;                // previous state of button SWL
byte SWR_Num;                 // number of right button presses count        
bool SWR_Skip;                // flag used to skip button actions
bool SWR_State;               // current read state of button SWL
unsigned long T1,T2;          // code microsecond timers
int TachoCnt;                 // centre sensor tachometer counter
byte Tacho_En;                // set > 0 to enable Tacho function
bool TachoHi;                 // tacho threshold state
bool TachoLast;               // previous tacho threshold state
bool TachoNew;                // set = true when a new Tacho reading is recorded
int TachoPeriod;              // centre sensor tachometer period
int Tacho_ThreshLvl;          // trigger threshold value used in tacho function
int TaskClock;                // task duration in 4ms ticks
int TaskCnt;                  // general counter used in tasks
int TaskCnt0;                 // general counter used in tasks
byte TaskESC;                 // set true to control task ext mode
int Task_Inc;                 // any increment value used in a task
byte TaskNext;                // task pointer to be used after a delay task
byte TaskPnt;                 // main task pointer
int Task_PWM;                 // any PWM value used in tasks
int Task_PwmMax;              // the maximum PWM value to be used in a task
int Task_PwmMin;              // the minimum PWM value to be used in a task
int Task_T0;                  // task interval timers
int Task_T1;                  // task interval timers
int Task_T2;                  // task interval timers
int Task_T3;                  // task interval timers
int Task_T4;                  // task interval timers
int Task_T5;                  // task interval timers
int Task_Trgt;                // any target defined by the task
bool TOF = false;             // set true when hardware counter overflows
byte TurnCnt;                 // turning speed counter
bool TurnEn;                  // if true enable steering frunctions
int TurnFlg;                  // -1 = turn left, 0 = straight, +1 = turn right
byte TurnPlus;                // set whilst turning if sustained turning is needed
int Val_Mux;                  // analogue reading immediate after switching input, ignored

// Declare servo objects
Servo servoLft; // create left servo
Servo servoRht; // create right servo

void setup() {
  // Setup code, runs once:
  detachServos(); // set servo signals LOW at start
  pinMode(Sens_LftInp,INPUT);     // define left sensor input mode
  pinMode(Sens_CtrInp,INPUT);     // define centre sensor input mode
  pinMode(Sens_RhtInp,INPUT);     // define right sensor input mode
  pinMode(SWL_Pin,INPUT_PULLUP);  // define left button switch pin mode
  pinMode(SWR_Pin,INPUT_PULLUP);  // define right button switch pin mode
  pinMode(LED_Lft,OUTPUT); digitalWrite(LED_Lft,LOW); // set LED pin output mode
  pinMode(LED_Ctr,OUTPUT); digitalWrite(LED_Ctr,LOW); // set LED pin output mode
  pinMode(LED_Rht,OUTPUT); digitalWrite(LED_Rht,LOW); // set LED pin output mode
  pinMode(Sens_LftPwr,OUTPUT); digitalWrite(Sens_LftPwr,LOW);  // define left PWR pin
  pinMode(Sens_CtrPwr,OUTPUT); digitalWrite(Sens_CtrPwr,LOW);  // define centre PWR pin
  pinMode(Sens_RhtPwr,OUTPUT); digitalWrite(Sens_RhtPwr,LOW);  // define right PWR pin
  Serial.begin(115200);
  Serial.println("TrackBot_2_13\n");
  Run_Post();   // perform power-on self checks
  SyncTimers(); // set the main loop timer
}

void loop() {
  // Main loop code, runs repeatedly.
  // read all three sensors first
  if (TOF) {
    while (micros() > nextMicros) {}  // wait for hardware timer to overflow
    TOF = false;                    // reset overflow flag
  }
  if (micros() >= nextMicros) {
    nextMicros = nextMicros + Interval;
    if (nextMicros < micros()) {TOF = true;}  // hardware counter is about to overflow
    
    LEDs_OFF(); // ensure all LEDs are OFF before taking a reading
    switch(Phase) {
      // the Main Loop timer is divided into 3 phases, which switch between one of
      // the 3 sensors in sequence. Other functions like button reading are placed
      // into one of these phases too.
      case 0:
        // take a left sensor reading, which was switched ON in Pase == 2
        Sen_LftVal = analogRead(Sens_LftInp);   // read left sensor
        digitalWrite(Sens_LftPwr,LOW);          // switch OFF left sensor
        digitalWrite(Sens_CtrPwr,HIGH);         // switch ON centre sensor
        Val_Mux = analogRead(Sens_CtrInp);      // switch multiplexer to centre sensor
        if (Sensor_Cal) {
          // in calibration mode so adjust min/max values for all three sensors
          if (Sen_LftVal > Sen_LftLvlMax) {Sen_LftLvlMax = Sen_LftVal;}
          if (Sen_LftVal < Sen_LftLvlMin) {Sen_LftLvlMin = Sen_LftVal;}
        } else {
          // set flag based on previously calculated threshold values
          if (Sen_LftVal < Sen_LftThreshLvl) {Sen_LFT = true;} else {Sen_LFT = false;}
        }

        // set left LED brightness
        LED_ON = LOW; LED_OffTime = 0;      // default is LED == OFF
        if (!Lifted) {
          // Robot is on a surface so drive LEDs normally
          // when lifted away from the surface all LEDs will be OFF
          if (MainMode != 10) {
            // not in line tracking mode so LEDs have various functions
            // a background timer is used to display the Curve and Speed settings
            // periodically to remind the user of these settings
            BackTimer--;
            if (BackTimer < 1) {Reset_BackTimer();}
            if (BackTimer <= 125) {
              LED_Mask = B10000000;   // turn all LEDs OFF
            } else if (BackTimer <= 150) {
              LED_Mask = B10000000;
              if ((Speed == 1) || (Speed == 3) || (Speed == 5) || (Speed == 7)){LED_Mask += 4;}
              if ((Speed == 2) || (Speed == 3) || (Speed == 6) || (Speed == 7)){LED_Mask += 2;}
              if (Speed >= 4) {LED_Mask += 1;}
            } else if (BackTimer <= 213) {
              LED_Mask = B10000000;   // turn all LEDs OFF
            } else if (BackTimer <= 238) {
              LED_Mask = B10000000;
              if ((Curve == 1) || (Curve == 3) || (Curve == 5) || (Curve == 7)){LED_Mask += 4;}
              if ((Curve == 2) || (Curve == 3) || (Curve == 6) || (Curve == 7)){LED_Mask += 2;}
              if (Curve >= 4) {LED_Mask += 1;}
            } else if (BackTimer <= 363) {
              LED_Mask = B10000000;   // turn all LEDs OFF
            }
            if (LED_Mask) {
              // LED mask 8-bits 1xxxxLCR
              if ((LED_Mask & 4) == 4) {LED_ON = HIGH; LED_OffTime = micros() + 800;}
            } else if (!SWL_State) {
              // left switch pressed so set left LED on bright
              LED_ON = HIGH; LED_OffTime = LED_OffMax;
            } else if (SWL_Num || SWR_Num) {
              // switch sequence in progress so blank LEDs
            } else if (!SWR_State) {
              // right button pressed only so set left LED OFF
            } else if (Sensor_Cal) {
               // we are in calibration mode so adjust LED PWM depending on value
              if (Sen_LftVal < (Sen_LftLvlMax - 32)) {
                LED_ON = HIGH; LED_OffTime = micros() + ((Sen_LftLvlMax - Sen_LftVal) * 2.);
              }
            } else if (Sen_LFT) {
              // not in calibration mode so LED is either ON of OFF
              // define LED as being ON
              LED_ON = HIGH; LED_OffTime = micros() + 500;
            }
          } else if (Sen_LFT) {
            // in Line Tracking Mode so represent sensor status
            LED_ON = HIGH; LED_OffTime = micros() + 500;
          }
        } digitalWrite(LED_Lft,LED_ON);
        // track sensor transitions
        if (Sen_LftLast != Sen_LFT) {Reset_BackTimer();}
        Sen_LftLast = Sen_LFT;

        // read the button switches in this phase
        Read_Button_Switches();
        Phase ++;
        
        if (PrintEn) {
          Serial.print("0,600,"); Serial.print(Sen_LftVal); Serial.print(",");
//          Serial.print(PWM_Lft); Serial.print(F(","));
//          Serial.print(nextMicros - micros()); Serial.print(",");
        } break;

        
      case 1:
        // take a centre sensor reading, which was switched ON in Pase == 0
        Sen_CtrVal = analogRead(Sens_CtrInp);   // read centre sensor
        digitalWrite(Sens_CtrPwr,LOW);          // switch OFF centre sensor
        digitalWrite(Sens_RhtPwr,HIGH);         // switch ON right sensor

        BatteryVal = analogRead(BattPin);       // switch analog mux to battery voltage
        BatteryVal = analogRead(BattPin);       // read battery voltage
        
        Val_Mux = analogRead(Sens_RhtInp);      // switch multiplexer to right sensor

        if (Sensor_Cal) {
          // in calibration mode so adjust min/max values for all three sensors
          if (Sen_CtrVal > Sen_CtrLvlMax) {Sen_CtrLvlMax = Sen_CtrVal;}
          if (Sen_CtrVal < Sen_CtrLvlMin) {Sen_CtrLvlMin = Sen_CtrVal;}
        }
        // set flag based on previously calculated threshold values
        if (Sen_CtrVal < Sen_CtrThreshLvl) {Sen_CTR = true;} else {Sen_CTR = false;}

        if (Tacho_En) {
          // perform tachometer function if centre sensor levels are valid
          if ((Sen_CtrLvlMax - Sen_CtrLvlMin) > 50) {
            // sufficient range to give a tachometer measurement
            Tacho_ThreshLvl = Sen_CtrLvlMin + ((Sen_CtrLvlMax - Sen_CtrLvlMin)/4);
            if (Sen_CtrVal < Tacho_ThreshLvl) {TachoHi = true;} else {TachoHi = false;}
            TachoCnt++; // increment the Tacho 4ms counter
            if (TachoCnt > TachoCntMax) {TachoCnt = TachoCntMax;}  // prevent overflow or large values
            if (TachoHi != TachoLast) {
              // an edge has been detected
              if (TachoHi) {
                // we have just gone from black to white
                // we ignore the first counter value, which will be incorrect
                if (Tacho_En > 1) {
                  TachoPeriod = TachoCnt; TachoNew = true;
                } TachoCnt = 0; Tacho_En = 2;
              }
            } TachoLast = TachoHi;  // track the state of this flag
          } else {TachoCnt = 0;}  // suppress the Tacho counter when limits are invalid
        }
        
        // set centre LED brightness
        LED_ON = LOW; LED_OffTime = 0;      // default is LED == OFF
        if (!Lifted) {
          // Robot is on a surface so drive LEDs normally
          // when lifted away from the surface all LEDs will be OFF
          if (MainMode != 10) {
            // not in line tracking mode so LEDs have various functions
            if (LED_Mask) {
              // LED mask 8-bits 1xxxxLCR
              if ((LED_Mask & 2) == 2) {LED_ON = HIGH; LED_OffTime = micros() + 800;}
            } else if (SWL_Num || SWR_Num) {
            } else if ((!SWL_State) || (!SWR_State)) {
              // left or right switch pressed so set centre LED OFF
            } else if (Sensor_Cal) {
               // we are in calibration mode so adjust LED PWM depending on value
              if (Sen_CtrVal < (Sen_CtrLvlMax - 32)) {
                LED_ON = HIGH; LED_OffTime = micros() + ((Sen_CtrLvlMax - Sen_CtrVal) * 2);
              }
            } else if (Sen_CTR) {
              // define LED as being ON
              LED_ON = HIGH; LED_OffTime = micros() + 500;
            }
          } else if (Sen_CTR) {
            // in Line Tracking Mode so represent sensor status
            LED_ON = HIGH; LED_OffTime = micros() + 500;
          }
        } digitalWrite(LED_Ctr,LED_ON);
        // track sensor transitions
        if (Sen_CtrLast != Sen_CTR) {Reset_BackTimer();}
        Sen_CtrLast = Sen_CTR;

  //#############################################################################
  //  Read serial port
  //#############################################################################
        // read the serial port to support servo testing
        readKey(); Phase ++;
        
  //#############################################################################
  //  Read Battery voltage
  //#############################################################################
        // track battery average voltage over time
        Battery_Sum = Battery_Sum - BatteryAv + BatteryVal;
        BatteryAv = Battery_Sum/32; // get the average from 32 samples
        if (BatteryAv < BattLow) {Battery_Critical();}

        if (PrintEn) {
          Serial.print(Sen_CtrVal); Serial.print(",");
//          Serial.print(nextMicros - micros()); Serial.print(",");
        } break;

        
      case 2:
        // take a right sensor reading, which was switched ON in Pase == 1
        Sen_RhtVal = analogRead(Sens_RhtInp);   // read right sensor
        digitalWrite(Sens_RhtPwr,LOW);          // switch OFF right sensor
        digitalWrite(Sens_LftPwr,HIGH);         // switch ON left sensor
        Val_Mux = analogRead(Sens_LftInp);      // switch multiplexer to left sensor
        if (Sensor_Cal) {
          // in calibration mode so adjust min/max values for all three sensors
          if (Sen_RhtVal > Sen_RhtLvlMax) {Sen_RhtLvlMax = Sen_RhtVal;}
          if (Sen_RhtVal < Sen_RhtLvlMin) {Sen_RhtLvlMin = Sen_RhtVal;}
        } else {
          // set flag based on previously calculated threshold values
          if (Sen_RhtVal < Sen_RhtThreshLvl) {Sen_RHT = true;} else {Sen_RHT = false;}
        }
        // test the lifted from surface threshold
        Lifted = (Sen_LftVal >= LiftedLvl) & (Sen_CtrVal >= LiftedLvl) & (Sen_RhtVal >= LiftedLvl);

        // set right LED brightness
        LED_ON = LOW; LED_OffTime = 0;      // default is LED == OFF
        if (!Lifted) {
          // Robot is on a surface so drive LEDs normally
          // when lifted away from the surface all LEDs will be OFF
          if (MainMode != 10) {
            // not in line tracking mode so LEDs have various functions
            if (LED_Mask) {
              // LED mask 8-bits 1xxxxLCR
              if ((LED_Mask & 1) == 1) {LED_ON = HIGH; LED_OffTime = micros() + 800;}
            } else if (!SWR_State) {
              // right switch pressed so set right LED on bright
              LED_ON = HIGH; LED_OffTime = LED_OffMax;
            } else if (SWL_Num || SWR_Num) {
              // switch seqwuence in progress so blank LEDs
            } else if (!SWL_State) {
              // left button pressed only so set right LED OFF
            } else if (Sensor_Cal) {
               // we are in calibration mode so adjust LED PWM depending on value
              if (Sen_RhtVal < (Sen_RhtLvlMax - 32)) {
                LED_ON = HIGH; LED_OffTime = micros() + ((Sen_RhtLvlMax - Sen_RhtVal) * 2);
              }
            } else if (Sen_RHT) {
              // define LED as being ON
              LED_ON = HIGH; LED_OffTime = micros() + 500;
            }
          } else if (Sen_RHT) {
            // in Line Tracking Mode so represent sensor status
            LED_ON = HIGH; LED_OffTime = micros() + 500;
          }
        } digitalWrite(LED_Rht,LED_ON);
        // track sensor transitions
        if (Sen_RhtLast != Sen_RHT) {Reset_BackTimer();}
        Sen_RhtLast = Sen_RHT;

        // in Phase == 2 control the brake LEDs
        if (LED_BRK) {digitalWrite(LED_Brk,HIGH);} else {digitalWrite(LED_Brk,LOW);}

        // perform main tasks during this phase
        if (MainMode > 0) {if (Lifted && !IgnoreLift) {MainMode = 0; Set_MainMode();}}
        if (Lifted != LiftedLast) {Blink();}  // flash LEDs as we go through the transition point
        LiftedLast = Lifted;                  // record current state


  //#############################################################################
  //  MainMode tasks
  //#############################################################################
        switch (MainMode) {
          case 0: break;                        // default task, do nothing
          case 10:                              // active line tracking mode
            Task_Line_Track(); if (TurnEn) {Steer();}
            break;
          case 20: Task_Sens_Cal_Man(); break;  // enster manual sensor calibration mode
          case 30: Task_Sens_Cal_Auto(); break; // enster automatic sensor calibration mode
          case 40: Task_Cal_Speed(); break;     // servo automatic speed calibration
          case 50: Task_Cal_Crossover(); break; // 'crossover' calibration
          case 60: Task_Cal_Centre(); break;    // servo centre calibration function
        } Phase = 0 ;
        
        if (PrintEn) {
          Serial.println(Sen_RhtVal);
//          Serial.println(PWM_Rht);
//          Serial.println(nextMicros - micros());
        } break;
    }
  }

  //#############################################################################
  //  Slack time
  //#############################################################################
  // during free time we switch OFF LEDs to provide a dimming effect
  switch(Phase) {
    case 0:
      // we have just read the value of Sen_RhtVal so set right LED brightness
      if (micros() > LED_OffTime) {digitalWrite(LED_Rht,LOW); LED_OffTime = LED_OffMax;}
      break;
    case 1:
      // we have just read the value of Sen_LftVal so set left LED brightness
      if (micros() > LED_OffTime) {digitalWrite(LED_Lft,LOW); LED_OffTime = LED_OffMax;}
      break;
    case 2:
      // we have just read the value of Sen_CtrVal so set centre LED brightness
      if (micros() > LED_OffTime) {digitalWrite(LED_Ctr,LOW); LED_OffTime = LED_OffMax;}
      break;
  }
}

// --------------------------------------------------------------------------------

void Run_Post() {
  // flash LEDs at bootup
  Serial.println(F("Running POST..."));
  SetDefaults(); SetBaseline();
  Serial.println(F("Flashing LEDs"));
  digitalWrite(LED_Lft,HIGH); delay(300); digitalWrite(LED_Lft,LOW);
  digitalWrite(LED_Ctr,HIGH); delay(300); digitalWrite(LED_Ctr,LOW);
  digitalWrite(LED_Rht,HIGH); delay(300); digitalWrite(LED_Rht,LOW);
  digitalWrite(LED_Brk,HIGH); delay(300); digitalWrite(LED_Brk,LOW);
  Serial.println(F("Outputting PWM 1500µs"));
  attachServos(); delay(500); detachServos();

  // load EEPROM values
  EEPROM_Load();
  Set_LiftedLvl();
  Set_Thresholds();
  Report_Sen_MinMax();

  // check battery
  for (int zI = 1; zI < 100; zI++) {
    Battery_Sum = Battery_Sum - BatteryAv + analogRead(BattPin);
    BatteryAv = Battery_Sum/32; // get the average from 32 samples
  }
  if (BatteryAv < BattLow) {
    Serial.println(F("Battery LOW... USB connection assunmed."));
    BattLow = BattLow/2;  // set very low limt
  } else {
    Serial.print(F("Battery = ")); Serial.println((float)BatteryAv * 0.0122);
  }
  Serial.println(F("POST complete!\n"));
}

// --------------------------------------------------------------------------------

void SetBaseline() {
  // set default speeds and sensor trigger points
  BrakeT1 = 21;                 // Speed 1 emergency stop trigger
  BrakeT2 = 22;                 // Speed 2 emergency stop trigger
  BrakeT3 = 23;                 // Speed 3 emergency stop trigger
  BrakeT4 = 24;                 // Speed 4 emergency stop trigger
  BrakeT5 = 25;                 // Speed 5 emergency stop trigger
  BrakeT6 = 26;                 // Speed 6 emergency stop trigger
  BrakeT7 = 27;                 // Speed 7 emergency stop trigger
  LiftedLvl = 260;              // sensor limit for being lifted off the surface
  PWM_Lft = PWM_StopRef;        // left servo angle in microseconds
  PWM_LftInc = 1;               // left servo incrementing value
  PWM_LftSpd0 = PWM_LftSpd0Def; // forward PWM stall value
  PWM_LftSpd1 = PWM_LftSpd1Def; // forward PWM speed 1 value
  PWM_LftSpd2 = PWM_LftSpd2Def; // forward PWM speed 2 value
  PWM_LftSpd3 = PWM_LftSpd3Def; // forward PWM speed 3 value
  PWM_LftSpd4 = PWM_LftSpd4Def; // forward PWM speed 4 value
  PWM_LftSpd5 = PWM_LftSpd5Def; // forward PWM speed 5 value
  PWM_LftSpd6 = PWM_LftSpd6Def; // forward PWM speed 6 value
  PWM_LftSpd7 = PWM_LftSpd7Def; // forward PWM speed 7 value
  PWM_Rht = PWM_StopRef;        // right servo angle in microseconds
  PWM_RhtInc = 1;               // left servo incrementing value
  PWM_RhtSpd0 = PWM_RhtSpd0Def; // forward PWM stall value
  PWM_RhtSpd1 = PWM_RhtSpd1Def; // forward PWM speed 1 value
  PWM_RhtSpd2 = PWM_RhtSpd2Def; // forward PWM speed 2 value
  PWM_RhtSpd3 = PWM_RhtSpd3Def; // forward PWM speed 3 value
  PWM_RhtSpd4 = PWM_RhtSpd4Def; // forward PWM speed 4 value
  PWM_RhtSpd5 = PWM_RhtSpd5Def; // forward PWM speed 5 value
  PWM_RhtSpd6 = PWM_RhtSpd6Def; // forward PWM speed 6 value
  PWM_RhtSpd7 = PWM_RhtSpd7Def; // forward PWM speed 7 value
  Sen_CtrLvlMax = Sen_LvlMax;   // default maximum 'black' value for centre sensor
  Sen_CtrLvlMin = Sen_LvlMin;   // default minimum 'white' value for centre sensor
  Sen_LftLvlMax = Sen_LvlMax;   // default maximum 'black' value for left sensor
  Sen_LftLvlMin = Sen_LvlMin;   // default minimum 'white' value for left sensor
  Sen_RhtLvlMax = Sen_LvlMax;   // default maximum 'black' value for right sensor
  Sen_RhtLvlMin = Sen_LvlMin;   // default minimum 'white' value for right sensor
}

// --------------------------------------------------------------------------------

void SetDefaults() {
  // set default values here to allow for a soft reset
  BackTimer = 0;                // background timer used to display Curve & Speed settings
  BatteryAv = 1024;             // averaged battery voltage
  Battery_Sum = 32768;          // accumulator for battery averaging
  Brake = 0;                    // a timer used in soft braking
  Curve = 7;                    // value 1 - 7 turning rate
  LED_CTR = LOW;                // flag controlling centre LED ON/OFF state
  LED_BRK = false;              // flag controlling brake LED ON/OFF state
  LED_LvlMax = Sen_LvlMax;      // max level used by LED driver code
  LED_LFT = LOW;                // flag controlling left LED ON/OFF state
  LED_Mask = 0;                 // if > 0 then right 3 bits define the L-C-R ON/OFF states
  LED_OffTime = 0;              // timer used to dim and turn OFF LEDs
  LED_RHT = LOW;                // flag controlling right LED ON/OFF state
  Lifted = false;               // default true if lifted off the surface
  LiftedLast = false;           // previous Lifted state to track lifted transition
  MainMode = 0;                 // mode of operation
  MaskCnt = 0;                  // reset mask counter for LED flashing
  Phase = 0;                    // sensor phase pointer
  Sensor_Cal = false;           // true if in sensor calibration mode, otherwise false
  Speed = 1;                    // value 1 - 7 for car speed setting
  SWL_Cnt = 0;                  // button pressed counter
  SWL_Last = true;              // previous state of button SWL
  SWL_Num = 0;                  // number of left button presses count        
  SWL_Skip = false;             // flag used to skip button actions
  SWL_State = true;             // current read state of button SWL
  SWR_Cnt = 0;                  // button pressed counter
  SW_Rel = false;               // flags used in checking both button switches released
  SWR_Last = true;              // previous state of button SWL
  SWR_Num = 0;                  // number of right button presses count        
  SWR_Skip = false;             // flag used to skip button actions
  SWR_State = true;             // current read state of button SWL
  TachoNew = false;             // set = true when a new Tacho reading is recorded
  TurnEn = false;               // if true enable steering frunctions
  TurnFlg = 0;                  // -1 = turn left, 0 = straight, +1 = turn right
  TurnPlus = 0;                 // set whilst turning if sustained turning is needed
}

// --------------------------------------------------------------------------------
