
// --------------------------------------------------------------------------------

void Task_Cal_Centre() {
  // in this mode we apply a saw-tooth PWM signal to both servos, centred around
  // the nominal 1500µs centre value. The user can therefore adjust the pots in the
  // base of the FS90R servos.
  Flash_All_LEDs_Masked();  // use the LED mask flashing mode
  Extend_BackTimer();       // set LED background task delay to hold it off
  switch (TaskPnt) {
    case 0: 
      // initialise variables
      Serial.println(F("Servo centre calibration mode"));
      Task_PWM = 1500; PWM_Lft = Task_PWM;  PWM_Rht = Task_PWM;
      // deband is 60, or +/-30, so we will use a window of 80, +/-40
      Task_PwmMax = PWM_StopRef + 40; Task_PwmMin = PWM_StopRef - 40; Task_Inc = 1;
      TaskPnt = 3; break;
    case 1:
      Task_PWM += Task_Inc;                         // change PWM value
      if (Task_PWM >= Task_PwmMax) {Task_Inc = -1;} // set increment -ve
      if (Task_PWM <= Task_PwmMin) {Task_Inc = 1;}  // set increment +ve
      Serial.println(Task_PWM);
      PWM_Lft = Task_PWM;  PWM_Rht = Task_PWM; Set_Servos();
      TaskCnt = 2; TaskPnt = 2;
      break;
    case 2:
      // this task performs an inter-step delay
      if (TaskCnt > 0) {TaskCnt--;} // decrement timer counter
      else {TaskPnt = 1;}           // TaskCnt == 0 so return to task 1
      break;
    case 3:
      // wait for the right button to be released before starting the process
      // a 1 second delay is added after the button release
      if (SWR_State) {TaskCnt = 250; TaskPnt = 2;}
      break;
  }
}

// --------------------------------------------------------------------------------

void Task_Cal_Crossover() {
  // Perform a track cross-over calibration at the current speed setting, so the
  // calibration needs to be repeated for all 7 speeds using a special track.
  // The robot measures the width of 4 cross-over tracks and finally a double
  // width track in order for it to store the brake figure in 4ms cycles.
  Extend_BackTimer();       // set LED background task delay to hold it off
  TaskClock++;              // increment the task duration clock
  switch (TaskPnt) {
    case 0:
      // initialise task goes onto wait for button release
      Flash_All_LEDs_Masked();  // use the LED mask flashing mode
      TaskCnt = 0; TaskPnt = 90; break; // wait for button switch to be released
    case 1:
      // we initialise variables and start the robot at the current speed
      // the robot must be placed over a track line or we abort this mode
      // [ ][#][ ]
      if (Sen_LFT && !Sen_CTR && Sen_RHT) {
        // centre sensor is OFF, so we are on a line and can start
        // for this process we apply a specific Curve value
        CurveLast = Curve; Curve = 7;
        Serial.println(F("Calibrate Crossover Mode"));
        Start_PWM(); TaskPnt++; TaskCnt = 25; LED_Mask = 0;
      } else {TaskPnt = 70;}  // abort if not on a line
      break;
    case 2:
      // allow a short delay before applying steering to keep it on track
      TaskCnt--; if (TaskCnt < 1) {TaskPnt++; TurnEn = true;}
      break;
    case 3:
      // drive along looking for a crossover track
      if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
        // [#][#][#]  - all sensors are dark
        Task_T0 = TaskClock; // record the dark edge
        Brake = 1;            // set the robot going straight
        TaskPnt++;
      } break;
    case 4:
      // driving over crossover track looking for it to end
      if (Sen_LFT || Sen_CTR || Sen_RHT) {
        // [ ||x][ ||x][ ||x] - at least 1 sensor is not dark
        Task_T1 = TaskClock - Task_T0; // record the dark track width in 4ms clocks
        Brake = 0;            // remove braking effect on steering
        TaskPnt++;
      } break;
    case 5:
      // drive along looking for a crossover track
      if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
        // [#][#][#]  - all sensors are dark
        Task_T0 = TaskClock; // record the dark edge
        Brake = 1;            // set the robot going straight
        TaskPnt++;
      } break;
    case 6:
      // driving over crossover track looking for it to end
      if (Sen_LFT || Sen_CTR || Sen_RHT) {
        // [ ||x][ ||x][ ||x] - at least 1 sensor is not dark
        Task_T2 = TaskClock - Task_T0; // record the dark track width in 4ms clocks
        Brake = 0;            // remove braking effect on steering
        TaskPnt++;
      } break;
    case 7:
      // drive along looking for a crossover track
      if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
        // [#][#][#]  - all sensors are dark
        Task_T0 = TaskClock; // record the dark edge
        Brake = 1;            // set the robot going straight
        TaskPnt++;
      } break;
    case 8:
      // driving over crossover track looking for it to end
      if (Sen_LFT || Sen_CTR || Sen_RHT) {
        // [ ||x][ ||x][ ||x] - at least 1 sensor is not dark
        Task_T3 = TaskClock - Task_T0; // record the dark track width in 4ms clocks
        Brake = 0;            // remove braking effect on steering
        TaskPnt++;
      } break;
    case 9:
      // drive along looking for a crossover track
      if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
        // [#][#][#]  - all sensors are dark
        Task_T0 = TaskClock; // record the dark edge
        Brake = 1;            // set the robot going straight
        TaskPnt++;
      } break;
    case 10:
      // driving over crossover track looking for it to end
      if (Sen_LFT || Sen_CTR || Sen_RHT) {
        // [ ||x][ ||x][ ||x] - at least 1 sensor is not dark
        Task_T4 = TaskClock - Task_T0; // record the dark track width in 4ms clocks
        Brake = 0;            // remove braking effect on steering
        TaskPnt++;
      } break;
    case 11:
      // drive along looking for a double crossover track
      if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
        // [#][#][#]  - all sensors are dark
        Task_T0 = TaskClock; // record the dark edge
        Brake = 1;            // set the robot going straight
        TaskPnt++;
      } break;
    case 12:
      // driving over double crossover track looking for it to end
      if (Sen_LFT || Sen_CTR || Sen_RHT) {
        // [ ||x][ ||x][ ||x] - at least 1 sensor is not dark
        Task_T5 = TaskClock - Task_T0;  // record the dark track width in 4ms clocks
        Brake = 0;  TurnEn = false;     // remove braking and steering
        detachServos();                 // stop the robot
        TaskPnt++;
      } break;
    case 13:
      // now we calculate the stopping point for this speed
      Task_T0 = max(Task_T1,Task_T2);
      Task_T0 = max(Task_T0,Task_T3);
      Task_T0 = max(Task_T0,Task_T4);
      // we now have the longest delay measured inf Task_T0
      if (Task_T0 >= Task_T5) {TaskPnt = 70;} // abort if last track was not thicker
      else {
        // last double line is thicker so set braking point
        Task_T0 = Task_T0 + ((Task_T5 - Task_T0)/3);
        if (Speed == 1) {BrakeT1 = Task_T0;}  // set brake time for Speed 1
        if (Speed == 2) {BrakeT2 = Task_T0;}  // set brake time for Speed 2
        if (Speed == 3) {BrakeT3 = Task_T0;}  // set brake time for Speed 3
        if (Speed == 4) {BrakeT4 = Task_T0;}  // set brake time for Speed 4
        if (Speed == 5) {BrakeT5 = Task_T0;}  // set brake time for Speed 5
        if (Speed == 6) {BrakeT6 = Task_T0;}  // set brake time for Speed 6
        if (Speed == 7) {BrakeT7 = Task_T0;}  // set brake time for Speed 7
        TaskPnt++;
      } break;
    case 14:
      // save crossover value for this speed
      EEPROM_Save(); Report_Sen_MinMax();
      MainMode = 0; Set_MainMode();
      break;
      
    case 70:
      // aborting this mode due to a fault condition
      Curve = CurveLast; MainMode = 0; Set_MainMode();
      break;
    case 80:
      // wait here for timer count to expire then switch to TaskNext
      TaskCnt--; if (TaskCnt < 1) {TaskPnt = TaskNext;}
      break;
    case 90:
      // we wait here checking for a button release. If this does not occur soon
      // we switch to the servo centre calibration task mode
      Flash_All_LEDs_Masked();  // use the LED mask flashing mode
      if (!SWR_State) {
        TaskCnt++;
        if (TaskCnt >= 500) {MainMode = 60; Set_MainMode(); Blink_Long();} // timed-out so switch modes
      } else {TaskCnt = 250; TaskPnt = 80; TaskNext = 1;} // button released so perform cal stall
      break;
  }
  // robot steering is applied whenever TurnEn == true
  if (TurnEn) {Steer();}
}

// --------------------------------------------------------------------------------

void Task_Cal_Speed() {
  // performs an automatic setting of the servo motor speed points, by determining
  // the PWM values required to achieve the max, mid and min points on the desired
  // speed curve. From this the remaining PW values are calculed, assuming a
  // straight line characteristic, including the stall points. Each wheel is driven
  // separately, rotating the vehicle around a common centre point and a special
  // test sheet.
  if (TaskPnt != 6) {Flash_All_LEDs_Masked();}  // use the LED mask flashing mode
  else {Flash_Rht_LED_Masked();}  // only flash the right-hand LED
  Extend_BackTimer();             // set LED background task delay to hold it off
  switch (TaskPnt) {
    case 0:
      // initialise task
      TaskCnt = 0; TaskPnt = 90; break; // wait for button switch to be released
    case 1:
      // start right servo at default start speed
      PWM_Rht = PWM_Start; Servos_Attached = true;
      servoRht.attach(ServoRhtPin); servoRht.writeMicroseconds(3000 - PWM_Rht);
      TaskPnt++; TaskCnt = 2500; break;
    case 2:
      // initiate the decrease of speed to min value
      Set_Tacho_ON(); TaskESC = 0;
      Task_Trgt = 214; Task_Inc = 8; TaskNext = 3; TaskPnt = 70; break;
    case 3:
      // store the 1st speed value, which is the lowest speed value
      PWM_RhtSpd1 = PWM_Rht;
      // decrease the speed to 4th value
      Set_Tacho_ON(); TaskESC = 0;
      Task_Trgt = 107; Task_Inc = 8; TaskNext = 4; TaskPnt = 70; break;
    case 4:
      // store the 4th speed value, which is the middle speed value
      PWM_RhtSpd4 = PWM_Rht;
      // decrease the speed to 1st value
      Set_Tacho_ON(); TaskESC = 0;
      Task_Trgt = 71; Task_Inc = 8; TaskNext = 5; TaskPnt = 70; break;
    case 5:
      // store the 7th speed value
      PWM_RhtSpd7 = PWM_Rht;
      // now stop right motor
      detachServos();
      TaskPnt++; IgnoreLift = true; break;
    case 6:
      // wait for right button to be pressed whilst flashing right LED
      if (!SWR_State) {TaskPnt++; IgnoreLift = false;}
      break;
    case 7:
      // wait for right button to be released
      if (SWR_State) {TaskPnt++;}
      break;
    case 8:
      // a short delay before setting off again
      TaskCnt = 250; TaskNext = 20; TaskPnt = 80;
      break;
      
    case 20:
      // start right servo at default start speed
      PWM_Lft = PWM_Start; Servos_Attached = true;
      servoLft.attach(ServoLftPin); servoLft.writeMicroseconds(PWM_Lft);
      TaskPnt++; TaskCnt = 2500; break;
    case 21:
      // initiate the decrease of speed to min value
      Set_Tacho_ON(); TaskESC = 0;
      Task_Trgt = 214; Task_Inc = 8; TaskNext = 22; TaskPnt = 60; break;
    case 22:
      // store the 1st speed value
      PWM_LftSpd1 = PWM_Lft;
      // increase the speed to 4th value
      Set_Tacho_ON(); TaskESC = 0;
      Task_Trgt = 107; Task_Inc = 8; TaskNext = 23; TaskPnt = 60; break;
    case 23:
      // store the 4th speed value
      PWM_LftSpd4 = PWM_Lft;
      // increase the speed to max value
      Set_Tacho_ON(); TaskESC = 0;
      Task_Trgt = 71; Task_Inc = 8; TaskNext = 24; TaskPnt = 60; break;
    case 24:
      // store the 7th speed value
      PWM_LftSpd7 = PWM_Lft;
      // now stop left motor
      detachServos();
      TaskPnt++; break;
    case 25:
      // calculate unmeasured speeds from values taken
      // current range is wheel rpm = 30 to 90, for speeds = 1 to 7
      // so stall points are set at 1/3rd of pwm range
      // if you change the range you will need to change this calculation
      PWM_LftSpd0 = PWM_LftSpd1 - (PWM_LftSpd4 - PWM_LftSpd1);  // left stall speed
      PWM_RhtSpd0 = PWM_RhtSpd1 - (PWM_RhtSpd4 - PWM_RhtSpd1);  // right stall speed
      
      // all other points are based on a linear speed curve
      PWM_LftSpd2 = PWM_LftSpd1 + ((PWM_LftSpd4 - PWM_LftSpd1)/3);  // left speed 2
      PWM_RhtSpd2 = PWM_RhtSpd1 + ((PWM_RhtSpd4 - PWM_RhtSpd1)/3);  // right speed 2
      PWM_LftSpd3 = PWM_LftSpd2 + ((PWM_LftSpd4 - PWM_LftSpd1)/3);  // left speed 2
      PWM_RhtSpd3 = PWM_RhtSpd2 + ((PWM_RhtSpd4 - PWM_RhtSpd1)/3);  // right speed 2
      PWM_LftSpd5 = PWM_LftSpd4 + ((PWM_LftSpd7 - PWM_LftSpd4)/3);  // left speed 2
      PWM_RhtSpd5 = PWM_RhtSpd4 + ((PWM_RhtSpd7 - PWM_RhtSpd4)/3);  // right speed 2
      PWM_LftSpd6 = PWM_LftSpd5 + ((PWM_LftSpd7 - PWM_LftSpd4)/3);  // left speed 2
      PWM_RhtSpd6 = PWM_RhtSpd5 + ((PWM_RhtSpd7 - PWM_RhtSpd4)/3);  // right speed 2
      TaskPnt++; break;
    case 26:
      // Now save the results and report them
      EEPROM_Save(); Report_Sen_MinMax();
      MainMode = 0; Set_MainMode();
      break;


    case 30:
      // process has been aborted due to a timeout, so we need to recover the 
      // previously stored values and exit
      EEPROM_Load(); Report_Sen_MinMax();
      MainMode = 0; Set_MainMode();
      break;

    
    case 60:
      // this subtask adjusts the left motor speed until its target is achieved
      // if this takes to long abort the process
      TaskCnt--; if (TaskCnt < 1) {TaskPnt = 30; return;}
      if (TachoNew) {
        // use the Tacho period count to offset and delay the time-out function
        TaskCnt += TachoPeriod;
        LED_BRK = !LED_BRK; // toggle the brake lights
        // Tacho period has just been updated so use this to effect speed control
        if (TachoPeriod > Task_Trgt) {PWM_Lft += Task_Inc; SpdUp = 1;}  // speed up
        else if (TachoPeriod < Task_Trgt) {PWM_Lft -= Task_Inc; SpdUp = -1;}  // slow down
        PWM_Lft = min(PWM_Lft,PWM_Max);        // limit PWM max value
        PWM_Lft = max(PWM_Lft,PWM_StopRef);    // limit PWM min value
        servoLft.writeMicroseconds(PWM_Lft); TachoNew = false;
        // check for overshoot and correct speed increment
        if (SpdSt = 0) {} // ignore the first increase/decrease direction
        else if (SpdUp != SpdSt) {if (Task_Inc > 2) {Task_Inc /= 2;}}
        // target speed is assumed to be achieved when increment == 1
        if (Task_Inc <= 2) {
          TaskESC++; if ((TaskESC >= 10) && (SpdUp != SpdSt) && (SpdUp < 0)) {
            // exit when we have just exceeded the target speed
            TaskPnt = TaskNext; TaskCnt = 2500; Blink_Long();
          }
        } SpdSt = SpdUp;  // track speed change flag
      } break;
    
    case 70:
      // this subtask adjusts the right motor speed until its target is achieved
      // if this takes to long abort the process
      TaskCnt--; if (TaskCnt < 1) {TaskPnt = 30; return;}
      if (TachoNew) {
        // use the Tacho period count to offset and delay the time-out function
        TaskCnt += TachoPeriod;
        LED_BRK = !LED_BRK; // toggle the brake lights
        // Tacho period has just been updated so use this to effect speed control
        if (TachoPeriod > Task_Trgt) {PWM_Rht += Task_Inc; SpdUp = 1;}  // speed up
        else if (TachoPeriod < Task_Trgt) {PWM_Rht -= Task_Inc; SpdUp = -1;}  // slow down
        PWM_Rht = min(PWM_Rht,PWM_Max);        // limit PWM max value
        PWM_Rht = max(PWM_Rht,PWM_StopRef);    // limit PWM min value
        servoRht.writeMicroseconds(3000 - PWM_Rht); TachoNew = false;
        // check for overshoot and correct speed increment
        if (SpdSt = 0) {} // ignore the first increase/decrease direction
        else if (SpdUp != SpdSt) {if (Task_Inc > 2) {Task_Inc /= 2;}}
        // target speed is assumed to be achieved when increment == 1
        if (Task_Inc <= 2) {
          TaskESC++; if ((TaskESC >= 10) && (SpdUp != SpdSt) && (SpdUp < 0)) {
            // exit when we have just exceeded the target speed
            TaskPnt = TaskNext; TaskCnt = 2500; Blink_Long();
          }
        } SpdSt = SpdUp;  // track speed change flag
      } break;
      
    case 80:
      // wait here for timer count to expire then switch to TaskNext
      TaskCnt--; if (TaskCnt < 1) {TaskPnt = TaskNext;}
      break;
    case 90:
      // we wait here checking for a button release. If this does not occur soon
      // we switch to the crossover calibration task mode
      if (!SWR_State) {
        TaskCnt++;
        if (TaskCnt >= 500) {MainMode = 50; Set_MainMode(); Blink_Long();} // timed-out so switch modes
      } else {TaskCnt = 250; TaskPnt = 80; TaskNext = 1;} // button released so perform cal stall
      break;
  }
}

// --------------------------------------------------------------------------------

void Task_Line_Track() {
  // perform a line tracking task
//  Serial.println(TaskPnt);
  switch (TaskPnt) {
      case 0:
        // initialise the task
        // [ ][#][ ]
        if (Sen_LFT && !Sen_CTR && Sen_RHT) {
          // centre sensor is OFF, so we are on a line and can start
          // NoLineCnt is used as a delay to allow the bot to get up to speed
          // Brake is used as a glitch filter and has a stopping rate
          Serial.println(F("Line Tracking Mode"));
          Serial.print(F("Curve = ")); Serial.println(Curve);
          Serial.print(F("Speed = ")); Serial.println(Speed);
          Start_PWM(); TaskPnt++; NoLineCnt = 50; Brake = 0;
        } else if (Sen_LFT && Sen_CTR && Sen_RHT) {
          // [ ][ ][ ]  no track detected
          // the robot will start but with steering disabled until centre sensor
          // sees a track to follow. It will run like this for 5 seconds
          Serial.println(F("Blind Driving Mode"));
          Serial.print(F("Curve = ")); Serial.println(Curve);
          Serial.print(F("Speed = ")); Serial.println(Speed);
          Start_PWM(); TaskPnt = 10; NoLineCnt = 250 * 5; Brake = 0;
        } else {
          // start conditions not met, so exit
          Serial.println(F("Start conditions not met!"));
          Serial.print(Sen_LFT); Serial.print("\t");
          Serial.print(Sen_CTR); Serial.print("\t");
          Serial.println(Sen_RHT);
          TaskPnt = 4;
        } break;
      case 1:
        // allow time for motors to start and to speed up
        if (NoLineCnt > 0) {NoLineCnt--;}
        else {TaskPnt++; TurnEn = true;}
        break;
      case 2:
        // in full line tracking mode, monitoring sensors
        break;
      case 3:
        // switch OFF auto speed controllers
        // reduucing speed to stop at a rate of 'Brake'
        TurnEn = false;
        if (PWM_Lft > PWM_LftSpd0) {PWM_Lft -= Brake;} else {PWM_Lft = PWM_StopRef;}
        if (PWM_Rht > PWM_RhtSpd0) {PWM_Rht -= Brake;} else {PWM_Rht = PWM_StopRef;}
        if ((PWM_Lft == PWM_StopRef) && (PWM_Rht == PWM_StopRef)) {TaskPnt++;}
        Set_Servos(); LED_BRK = true; break;
      case 4:
        // brake sharply and exit this task
        TurnEn = false;
        MainMode = 0; Set_MainMode();
        break;
      case 10:
        // drive until a centre line is detected or stop after time-out
        // if all three sensors see black stop immediately
        if (NoLineCnt > 0) {
          NoLineCnt--;
          if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
            // [#][#][#]
            TaskPnt = 4; return;  // STOP immediately
          }
          if (!Sen_CTR) {
            // [x][#][x]    centre line detected so switch to line following mode
            TaskPnt = 2; TurnEn = true; NoLineCnt = 0;
            Brake = BrakeMx - 8;  // prepare for an emergency stop
          }
        } else {TaskPnt = 3; Brake = 5;} // end of time-out so stop
        break; 
  }
  // sensor monitor code once started
  if (TaskPnt == 2) {
    // monitor whilst moving
    if (!Sen_LFT && !Sen_CTR && !Sen_RHT) {
      // [#][#][#]
      // all black so STOP! quickly after a Brake count
      // any steering is immediately cancelled so that the robot will cross a black line square on
      Brake++;  // apply a 250ms time-out as an anti-glitch filter
      if (Brake >= BrakeMx) {
        // STOP!, set task pointer and exit
        Serial.println(F("All black so STOP!"));
        LED_BRK = true; TaskPnt = 4; return;
      } NoLineCnt = 0;
    } else {
      // not all black so clear brake flag and return to steering mode
      Brake = 0;
    }
    if (Sen_LFT && Sen_CTR && Sen_RHT) {
      // [ ][ ][ ]
      // no black line so stop after 2 seconds
      NoLineCnt++;
      if (NoLineCnt >= NoLineSTOP) {
        Serial.println(F("All white so STOP!"));
        Brake = 2; TaskPnt = 3; return;
      }
      return; // exit, don't do anything else
    }
    if (TurnFlg != 0) {
      // in sustained turning, so look for centre sensor for when to stop
      // [ ][#][ ]
      if (Sen_LFT && !Sen_CTR && Sen_RHT) {
        // back on track so cancel turning condition
        TurnFlg = 0;      // clear the flag
        LED_BRK = false;  // turn OFF brake lights
        NoLineCnt = 0;    // we are back on track
      }
    } else {
      // TurnFlg == 0 so check for line using the sensor threshold
      if (!Sen_LFT) {
        // [#][x][x]
        TurnFlg = -1; LED_BRK = true; NoLineCnt = 0;
      } else if (!Sen_RHT) {
        // [x][x][#]
        TurnFlg = 1; LED_BRK = true; NoLineCnt = 0;
      }
    }
  }
}

// --------------------------------------------------------------------------------

void Task_Sens_Cal_Auto() {
  // perform an automated rotational calibration of sensors  
  Extend_BackTimer();   // set LED background task delay to hold it off
  switch (TaskPnt) {
    case 0:
      // on entering this mode we wait for the right button to be released
      Flash_All_LEDs_Masked();  // use the LED mask flashing mode
      TaskCnt++;
      if (!SWR_State) {
        if (TaskCnt >= 500) {MainMode = 0; Set_MainMode(); Blink_Long();} // timed-out so switch modes
      } else {
        // button released so perform automated sensor calibration
        TaskCnt = 500; TaskPnt++;
      } break;
    case 1:
      // wait for 2 seconds before moving
      Flash_All_LEDs_Masked();  // use the LED mask flashing mode
      TaskCnt--; if (TaskCnt < 1) {TaskPnt++;}
      break;
    case 2:
      // entering automated sensor calibration routine
      LED_Mask = 0; // remove any masking effects
      // centre the sensor limits on their current values
      Sen_LftLvlMax = Sen_LftVal; Sen_LftLvlMin = Sen_LftVal;
      Sen_CtrLvlMax = Sen_CtrVal; Sen_CtrLvlMin = Sen_CtrVal;
      Sen_RhtLvlMax = Sen_RhtVal; Sen_RhtLvlMin = Sen_RhtVal;
      Sensor_Cal = true;    // indicate start of sensor calibration mode
      // start the right-hand servo motor
      PWM_Rht = PWM_Start; Servos_Attached = true;
      servoRht.attach(ServoRhtPin); servoRht.writeMicroseconds(3000 - PWM_Rht);
      // set up the process timer
      Task_Trgt = 250; TaskCnt = 250 * 12; Task_Inc = 8; Set_Tacho_ON();
      TaskPnt++; break;  // set a 12 second timer
    case 3:
      // wait here until the time is up or if the user presses a button
      // once the sensor limits begin to separate we use the Tachop to adjust the speed
      TaskCnt--;
      if (TaskCnt < 1) {TaskPnt++;} // time is up so conclude calibration
      if (TachoNew) {
        // Tacho period has just been updated so use this to effect speed control
        if (TachoPeriod > Task_Trgt) {PWM_Rht += Task_Inc; SpdUp = 1;}        // speed up
        else if (TachoPeriod < Task_Trgt) {PWM_Rht -= Task_Inc; SpdUp = -1;}  // slow down
        PWM_Rht = min(PWM_Rht,PWM_Max);        // limit PWM max value
        PWM_Rht = max(PWM_Rht,PWM_StopRef);    // limit PWM min value
        servoRht.writeMicroseconds(3000 - PWM_Rht); TachoNew = false;
        // check for overshoot and reduce speed increment each time
        if (SpdSt == 0) {SpdSt = SpdUp;}
        else if (SpdUp != SpdSt) {if (Task_Inc > 1) {Task_Inc--;}}
        SpdSt = SpdUp;  // track speed change flag
      } //else {LED_BRK = false;}
      break;
    case 4:
      // slow down servo then stop it
      if (PWM_Rht >= PWM_RhtSpd0) {
        // slowing servo down
        PWM_Rht--; servoRht.writeMicroseconds(3000 - PWM_Rht);
      } else {
        // speed reduced to stall speed so now stop
        detachServos(); TaskPnt++;
      } break;
    case 5:
      // wrap things up
      Sensor_Cal = false;   // indicate end of sensor calibration mode
      Blink(); Blink();     // flash the LEDs brightly twice
      Validate_Limits();    // check values and set limits
      Report_Sen_MinMax();  // report the new limits to the monitor
      MainMode = 0; Set_MainMode();
      break;
  }
}

// --------------------------------------------------------------------------------

void Task_Sens_Cal_Man() {
  // perform manual sensor calibration task
  // if right button continues to be held down it will switch to auto mode
  Extend_BackTimer();   // set LED background task delay to hold it off
  switch (TaskPnt) {
      case 0:
        // initialise the task
        Blink();  // flash the LEDs brightly once
        TaskCnt = 500; TaskPnt++; break;
      case 1:
        // has the left button been released within 2 second window?
        if (SWL_State) {TaskPnt++; break;}  // Yes!
        TaskCnt--;
        // if button not released in time we will jump to automatic mode
        if (TaskCnt < 1) {
          // switching to automatic mode
          Blink_Long(); MainMode = 30; Set_MainMode();}
        break;
      case 2:
        // entering manual calibration routine
        Sen_LftLvlMax = Sen_LftVal; Sen_LftLvlMin = Sen_LftVal;
        Sen_CtrLvlMax = Sen_CtrVal; Sen_CtrLvlMin = Sen_CtrVal;
        Sen_RhtLvlMax = Sen_RhtVal; Sen_RhtLvlMin = Sen_RhtVal;
        Sensor_Cal = true;    // indicate start of sensor calibration mode
        TaskCnt = 250 * 12; TaskPnt++; break;  // set a 12 second timer
      case 3:
        // wait here until the time is up or if the user presses a button
        TaskCnt--;
        if (TaskCnt < 1) {TaskPnt++;} // time is up so conclude calibration
        break;
      case 4:
        // wrap things up
        Sensor_Cal = false;   // indicate start of sensor calibration mode
        Blink(); Blink();     // flash the LEDs brightly twice
        Validate_Limits();    // check values and set limits
        Report_Sen_MinMax();  // report the new limits to the monitor
        MainMode = 0; Set_MainMode();
        break;
  }
}
