
void attachServos() {
  // attach both servos and write PWM values
  servoLft.attach(ServoLftPin); servoLft.writeMicroseconds(PWM_Lft);
  servoRht.attach(ServoRhtPin); servoRht.writeMicroseconds(PWM_Rht);
  Servos_Attached = true;
}

// --------------------------------------------------------------------------------

void Battery_Critical() {
  // battery has reached critical level so stop everything
  MainMode = 0; Set_MainMode();
  LEDs_OFF();
  while (true) {
    // loop forever, and hope someone notices
  }
}

// --------------------------------------------------------------------------------

void Blink() {
  // flash all three LEDs brightly for 100ms, then resynch and exit
  LEDs_ON(); delay(100);
  LEDs_OFF(); delay(200);
  SyncTimers();
}

// --------------------------------------------------------------------------------

void Blink_Long() {
  // flash all three LEDs brightly for 200ms, then resynch and exit
  LEDs_ON(); delay(200);
  LEDs_OFF(); delay(300);
  SyncTimers();
}

// --------------------------------------------------------------------------------

void detachServos() {
  // detach both servos
  servoLft.detach(); servoRht.detach();
  pinMode(ServoLftPin,OUTPUT); digitalWrite(ServoLftPin,LOW); // set servo pin output mode
  pinMode(ServoRhtPin,OUTPUT); digitalWrite(ServoRhtPin,LOW); // set servo pin output mode
  Servos_Attached = false;
}

// --------------------------------------------------------------------------------

void EEPROM_Load() {
  // load values from the EEPROM memory area, or write defaults if first time
  if (EEPROM.read(0) != EEPROM_Key) {SetBaseline(); EEPROM_Save(); return;}

  Sen_CtrLvlMax = (int)EEPROM.read( 1) + ((int)EEPROM.read( 2) << 8);
  Sen_CtrLvlMin = (int)EEPROM.read( 3) + ((int)EEPROM.read( 4) << 8);
  Sen_LftLvlMax = (int)EEPROM.read( 5) + ((int)EEPROM.read( 6) << 8);
  Sen_LftLvlMin = (int)EEPROM.read( 7) + ((int)EEPROM.read( 8) << 8);
  Sen_RhtLvlMax = (int)EEPROM.read( 9) + ((int)EEPROM.read(10) << 8);
  Sen_RhtLvlMin = (int)EEPROM.read(11) + ((int)EEPROM.read(12) << 8);
  PWM_LftSpd0 = (int)EEPROM.read(13) + ((int)EEPROM.read(14) << 8);
  PWM_RhtSpd0 = (int)EEPROM.read(15) + ((int)EEPROM.read(16) << 8);
  PWM_LftSpd1 = (int)EEPROM.read(17) + ((int)EEPROM.read(18) << 8);
  PWM_RhtSpd1 = (int)EEPROM.read(19) + ((int)EEPROM.read(20) << 8);
  PWM_LftSpd2 = (int)EEPROM.read(21) + ((int)EEPROM.read(22) << 8);
  PWM_RhtSpd2 = (int)EEPROM.read(23) + ((int)EEPROM.read(24) << 8);
  PWM_LftSpd3 = (int)EEPROM.read(25) + ((int)EEPROM.read(26) << 8);
  PWM_RhtSpd3 = (int)EEPROM.read(27) + ((int)EEPROM.read(28) << 8);
  PWM_LftSpd4 = (int)EEPROM.read(29) + ((int)EEPROM.read(30) << 8);
  PWM_RhtSpd4 = (int)EEPROM.read(31) + ((int)EEPROM.read(32) << 8);
  PWM_LftSpd5 = (int)EEPROM.read(33) + ((int)EEPROM.read(34) << 8);
  PWM_RhtSpd5 = (int)EEPROM.read(35) + ((int)EEPROM.read(36) << 8);
  PWM_LftSpd6 = (int)EEPROM.read(37) + ((int)EEPROM.read(38) << 8);
  PWM_RhtSpd6 = (int)EEPROM.read(39) + ((int)EEPROM.read(40) << 8);
  PWM_LftSpd7 = (int)EEPROM.read(41) + ((int)EEPROM.read(42) << 8);
  PWM_RhtSpd7 = (int)EEPROM.read(43) + ((int)EEPROM.read(44) << 8);
  BrakeT1 = EEPROM.read(45);
  BrakeT2 = EEPROM.read(46);
  BrakeT3 = EEPROM.read(47);
  BrakeT4 = EEPROM.read(48);
  BrakeT5 = EEPROM.read(49);
  BrakeT6 = EEPROM.read(50);
  BrakeT7 = EEPROM.read(51);
  Serial.println(F("EEPROM loaded"));
}

// --------------------------------------------------------------------------------

void EEPROM_Save() {
  // save values to the EEPROM memory area
  // if the memory was empty during EEPROM_Load() then default values will be written
  EEPROM.update(0,EEPROM_Key);
  EEPROM.update( 1,(byte)Sen_CtrLvlMax); EEPROM.update( 2,(byte)(Sen_CtrLvlMax >> 8));
  EEPROM.update( 3,(byte)Sen_CtrLvlMin); EEPROM.update( 4,(byte)(Sen_CtrLvlMin >> 8));
  EEPROM.update( 5,(byte)Sen_LftLvlMax); EEPROM.update( 6,(byte)(Sen_LftLvlMax >> 8));
  EEPROM.update( 7,(byte)Sen_LftLvlMin); EEPROM.update( 8,(byte)(Sen_LftLvlMin >> 8));
  EEPROM.update( 9,(byte)Sen_RhtLvlMax); EEPROM.update(10,(byte)(Sen_RhtLvlMax >> 8));
  EEPROM.update(11,(byte)Sen_RhtLvlMin); EEPROM.update(12,(byte)(Sen_RhtLvlMin >> 8));
  EEPROM.update(13,(byte)PWM_LftSpd0); EEPROM.update(14,(byte)(PWM_LftSpd0 >> 8));
  EEPROM.update(15,(byte)PWM_RhtSpd0); EEPROM.update(16,(byte)(PWM_RhtSpd0 >> 8));
  EEPROM.update(17,(byte)PWM_LftSpd1); EEPROM.update(18,(byte)(PWM_LftSpd1 >> 8));
  EEPROM.update(19,(byte)PWM_RhtSpd1); EEPROM.update(20,(byte)(PWM_RhtSpd1 >> 8));
  EEPROM.update(21,(byte)PWM_LftSpd2); EEPROM.update(22,(byte)(PWM_LftSpd2 >> 8));
  EEPROM.update(23,(byte)PWM_RhtSpd2); EEPROM.update(24,(byte)(PWM_RhtSpd2 >> 8));
  EEPROM.update(25,(byte)PWM_LftSpd3); EEPROM.update(26,(byte)(PWM_LftSpd3 >> 8));
  EEPROM.update(27,(byte)PWM_RhtSpd3); EEPROM.update(28,(byte)(PWM_RhtSpd3 >> 8));
  EEPROM.update(29,(byte)PWM_LftSpd4); EEPROM.update(30,(byte)(PWM_LftSpd4 >> 8));
  EEPROM.update(31,(byte)PWM_RhtSpd4); EEPROM.update(32,(byte)(PWM_RhtSpd4 >> 8));
  EEPROM.update(33,(byte)PWM_LftSpd5); EEPROM.update(34,(byte)(PWM_LftSpd5 >> 8));
  EEPROM.update(35,(byte)PWM_RhtSpd5); EEPROM.update(36,(byte)(PWM_RhtSpd5 >> 8));
  EEPROM.update(37,(byte)PWM_LftSpd6); EEPROM.update(38,(byte)(PWM_LftSpd6 >> 8));
  EEPROM.update(39,(byte)PWM_RhtSpd6); EEPROM.update(40,(byte)(PWM_RhtSpd6 >> 8));
  EEPROM.update(41,(byte)PWM_LftSpd7); EEPROM.update(42,(byte)(PWM_LftSpd7 >> 8));
  EEPROM.update(43,(byte)PWM_RhtSpd7); EEPROM.update(44,(byte)(PWM_RhtSpd7 >> 8));
  EEPROM.update(45,BrakeT1);
  EEPROM.update(46,BrakeT2);
  EEPROM.update(47,BrakeT3);
  EEPROM.update(48,BrakeT4);
  EEPROM.update(49,BrakeT5);
  EEPROM.update(50,BrakeT6);
  EEPROM.update(51,BrakeT7);
  Serial.println(F("EEPROM saved"));
}

// --------------------------------------------------------------------------------

void Extend_BackTimer() {
  // set the LED background timer to 10 seconds to hold it off
  BackTimer = BackTimerRes;
}

// --------------------------------------------------------------------------------

void Flash_All_LEDs() {
  // flash warning that robot is about to move off  
  for (int zL = 0; zL < 20; zL++) {
    LEDs_ON(); delay(32);
    LEDs_OFF(); delay(120);
  }
  delay(200); // pause after sequence
  SyncTimers();
}

// --------------------------------------------------------------------------------

void Flash_All_LEDs_Masked() {
  // flash warning that robot servos are powered up
  if (MaskCnt == 0) {LED_Mask = B10000111;}       // turn all 3 LEDs ON
  else if (MaskCnt == 8) {LED_Mask = B10000000;}  // turn all 3 LEDs OFF
  MaskCnt++;
  if (MaskCnt >= 30) {MaskCnt = 0;}  
}

// --------------------------------------------------------------------------------

void Flash_Binary_LEDs() {
  // display 3 times the Curve and Speed values as binary numbers on LEDs 1 - 7
  LEDs_OFF(); delay(500);
  for (int zL = 0; zL < 4; zL++) {
    if ((Curve == 1) || (Curve == 3) || (Curve == 5) || (Curve == 7)){digitalWrite(LED_Lft,HIGH);}
    if ((Curve == 2) || (Curve == 3) || (Curve == 6) || (Curve == 7)){digitalWrite(LED_Ctr,HIGH);}
    if (Curve >= 4) {digitalWrite(LED_Rht,HIGH);}
    delay(100); LEDs_OFF(); delay(250);
    if ((Speed == 1) || (Speed == 3) || (Speed == 5) || (Speed == 7)){digitalWrite(LED_Lft,HIGH);}
    if ((Speed == 2) || (Speed == 3) || (Speed == 6) || (Speed == 7)){digitalWrite(LED_Ctr,HIGH);}
    if (Speed >= 4) {digitalWrite(LED_Rht,HIGH);}
    delay(100); LEDs_OFF(); delay(500);
  }
  delay(200); // pause after sequence
  SyncTimers();
}

// --------------------------------------------------------------------------------

void Flash_LftLED() {
  // this function flashes the left LED based onm the SWL_Num value
  LEDs_OFF(); // ensure they are all OFF
  delay(200); // pause with all LED's OFF before flashing sequence
  for (int zL = 0; zL < SWL_Num; zL++) {
    digitalWrite(LED_Lft,HIGH); delay(100);
    digitalWrite(LED_Lft,LOW); delay(300);
  }
  delay(200); // pause after sequence
  SyncTimers();
}

// --------------------------------------------------------------------------------

void Flash_RhtLED() {
  // this function flashes the right LED based onm the SWR_Num value
  LEDs_OFF(); // ensure they are all OFF
  delay(200); // pause with all LED's OFF before flashing sequence
  for (int zL = 0; zL < SWR_Num; zL++) {
    digitalWrite(LED_Rht,HIGH); delay(100);
    digitalWrite(LED_Rht,LOW); delay(300);
  }
  delay(200); // pause after sequence
  SyncTimers();
}

// --------------------------------------------------------------------------------

void Flash_Rht_LED_Masked() {
  // flash right LED only using masked function
  if (MaskCnt == 0) {LED_Mask = B10000001;}       // turn right LED ON
  else if (MaskCnt == 8) {LED_Mask = B10000000;}  // turn all 3 LEDs OFF
  MaskCnt++;
  if (MaskCnt >= 30) {MaskCnt = 0;}  
}

// --------------------------------------------------------------------------------

void LEDs_OFF() {
  // turns OFF all front LEDs
  digitalWrite(LED_Ctr,LOW); // drive centre LED OFF
  digitalWrite(LED_Lft,LOW); // drive left LED OFF
  digitalWrite(LED_Rht,LOW); // drive right LED OFF
}

// --------------------------------------------------------------------------------

void LEDs_ON() {
  // turns ON all front LEDs
  digitalWrite(LED_Ctr,HIGH); // drive centre LED ON
  digitalWrite(LED_Lft,HIGH); // drive left LED ON
  digitalWrite(LED_Rht,HIGH); // drive right LED ON
}

// --------------------------------------------------------------------------------

void Read_Button_Switches() {
  // called from the Main Loop this code responds to button presses
  if (SW_Rel) {Wait_ButtonsUp(); return;}
  
  SWL_State = digitalRead(SWL_Pin);
  if (!SWL_State && SWL_Last) {
    // button SWL has just been pressed
    Reset_BackTimer();  // suspend LED background tasks
    if (MainMode > 0) {
      // already in an active mode, so exit that function unless in 'speed' cal.
      if  (MainMode != 40) {MainMode = 0; Set_MainMode();}
      SWL_Skip = true;
    }
  } else if (SWL_State && !SWL_Last) {
    // button SWL has just been released
    Extend_BackTimer();  // hold off LED background tasks
    if (SWL_Skip) {
      SWL_Skip = false; // clear the skip flag
      SWL_Cnt = 0;      // clear the timer
      SWL_Num = 0;      // clear the button pressed number
    } else {
      // release skip was not set, so count
      SWL_Num++;        // increment the button pressed count
      SWL_Cnt = 250;    // set a time to occur after button is released
    }
  } else if (!SWL_State && !SWL_Last) {
    Extend_BackTimer();  // hold off LED background tasks
    if (!SWL_Skip) {
      // button SWL is held down, so run a 2 sec time-out on it
      if (SWL_Cnt < 500) {SWL_Cnt++;}
      if (!SWR_State) {
        // right button has also been pressed so exit into tracking mode
        LEDs_ON(); delay(1000);         // switch LEDs ON to recognise buttons
        LEDs_OFF(); Wait_UntilUp();     // switch LEDs OFF and wait for release
//        Flash_Binary_LEDs();            // indicate settings before starting
        Flash_All_LEDs();               // flash LED warning
        MainMode = 10; Set_MainMode();  // enter line tracking mode
        return;
      }
      if (SWL_Cnt == 500) {
        // button held down so enter calibration mode
        SWL_Cnt++;  // prevent actions repeating
        MainMode = 20; Set_MainMode();  // invoke manual sensor calibration mode
        SWL_Skip = true;  // set button release skiup flag
      }
    }
  } else if (SWL_Cnt > 0) {
    // counter has been set to time-out following a button release
    SWL_Cnt--;  // decrement timer to zero
    if (SWL_Cnt < 1) {
      // time-out has been reached so use the button pressed number
      if (SWL_Num > 7) {SWL_Num = 7;} // limit Curve to 10 - 70 range
      Curve = SWL_Num;
      Serial.print(F("Curve = ")); Serial.println(Curve);
      Flash_LftLED(); // flash SWL_Num count for user feedback
      SWL_Num = 0;    // clear the number for next count
    }
  }
  SWL_Last = SWL_State; // track previous switch state to capture edges


  // right-hand 'select' button
  SWR_State = digitalRead(SWR_Pin);
  if (!SWR_State && SWR_Last) {
    // button SWR has just been pressed
    Reset_BackTimer();  // suspend LED background tasks
    if (MainMode > 0) {
      // already in an active mode, so exit that function unless in 'speed' cal.
      if  (MainMode != 40) {MainMode = 0; Set_MainMode();}
      SWR_Skip = true;
    }
  } else if (SWR_State && !SWR_Last) {
    // button SWR has just been released
    Extend_BackTimer();  // hold off LED background tasks
    if (SWR_Skip) {
      SWR_Skip = false; // clear the skip flag
      SWR_Cnt = 0;      // clear the timer
      SWR_Num = 0;      // clear the button pressed number
    } else {
      // release skip was not set, so count
      SWR_Num++;        // increment the button pressed count
      SWR_Cnt = 250;    // set a time to occur after button is released
    }
  } else if (!SWR_State && !SWR_Last) {
    Extend_BackTimer();  // hold off LED background tasks
    if (!SWR_Skip) {
      // button SWR is held down, so run a 2 sec time-out on it
      if (SWR_Cnt < 500) {SWR_Cnt++;}
      if (!SWL_State) {
        // left button has also been pressed so exit into tracking mode
        LEDs_ON(); delay(1000);         // switch LEDs ON to recognise buttons
        LEDs_OFF(); Wait_UntilUp();     // switch LEDs OFF and wait for release
//        Flash_Binary_LEDs();          // indicate settings before starting
        Flash_All_LEDs();             // flash LED warning
        MainMode = 10; Set_MainMode(); // enter line tracking mode
        return;
      }
      if (SWR_Cnt == 500) {
        // button held down so enter calibration mode
        SWR_Cnt++;  // prevent actions repeating
        MainMode = 40; Set_MainMode();  // invoke servo 'stall' calibration mode
        SWR_Skip = true;  // set button release skiup flag
      }
    }
  } else if (SWR_Cnt > 0) {
    // counter has been set to time-out following a button release
    SWR_Cnt--;  // decrement timer to zero
    if (SWR_Cnt < 1) {
      // time-out has been reached so use the button pressed number
      if (SWR_Num > 7) {SWR_Num = 7;} // limit speed setting to 1 - 7
      Speed = SWR_Num;
      Serial.print(F("Speed = ")); Serial.println(Speed);
      Flash_RhtLED(); // flash SWR_Num count for user feedback
      SWR_Num = 0;    // clear the number for next count
    }
  }
  SWR_Last = SWR_State; // track previous switch state to capture edges
}

// --------------------------------------------------------------------------------

void Report_Sen_MinMax() {
  // report the sensor min/max limits
  Serial.print(F("Lft Max = ")); Serial.print(Sen_LftLvlMax);
  Serial.print(F("\tMin = ")); Serial.println(Sen_LftLvlMin);
  Serial.print(F("Ctr Max = ")); Serial.print(Sen_CtrLvlMax);
  Serial.print(F("\tMin = ")); Serial.println(Sen_CtrLvlMin);
  Serial.print(F("Rht Max = ")); Serial.print(Sen_RhtLvlMax);
  Serial.print(F("\tMin = ")); Serial.println(Sen_RhtLvlMin);
  Serial.print(F("LiftedLvl = ")); Serial.println(LiftedLvl);
  Serial.print(F("Thresholds:\t")); Serial.print(Sen_LftThreshLvl);
  Serial.print(F("\t")); Serial.print(Sen_CtrThreshLvl);
  Serial.print(F("\t")); Serial.println(Sen_RhtThreshLvl);
  Serial.print(F("Curve = ")); Serial.println(Curve);
  Serial.print(F("Speed = ")); Serial.println(Speed);
  Serial.print(F("Lft PWM Spd0 = ")); Serial.println(PWM_LftSpd0);
  Serial.print(F("Rht PWM Spd0 = ")); Serial.println(PWM_RhtSpd0);
  Serial.print(F("Lft PWM Spd1 = ")); Serial.println(PWM_LftSpd1);
  Serial.print(F("Rht PWM Spd1 = ")); Serial.println(PWM_RhtSpd1);
  Serial.print(F("Lft PWM Spd2 = ")); Serial.println(PWM_LftSpd2);
  Serial.print(F("Rht PWM Spd2 = ")); Serial.println(PWM_RhtSpd2);
  Serial.print(F("Lft PWM Spd3 = ")); Serial.println(PWM_LftSpd3);
  Serial.print(F("Rht PWM Spd3 = ")); Serial.println(PWM_RhtSpd3);
  Serial.print(F("Lft PWM Spd4 = ")); Serial.println(PWM_LftSpd4);
  Serial.print(F("Rht PWM Spd4 = ")); Serial.println(PWM_RhtSpd4);
  Serial.print(F("Lft PWM Spd5 = ")); Serial.println(PWM_LftSpd5);
  Serial.print(F("Rht PWM Spd5 = ")); Serial.println(PWM_RhtSpd5);
  Serial.print(F("Lft PWM Spd6 = ")); Serial.println(PWM_LftSpd6);
  Serial.print(F("Rht PWM Spd6 = ")); Serial.println(PWM_RhtSpd6);
  Serial.print(F("Lft PWM Spd7 = ")); Serial.println(PWM_LftSpd7);
  Serial.print(F("Rht PWM Spd7 = ")); Serial.println(PWM_RhtSpd7);
  Serial.print(F("BrakeT1 = ")); Serial.println(BrakeT1);
  Serial.print(F("BrakeT2 = ")); Serial.println(BrakeT2);
  Serial.print(F("BrakeT3 = ")); Serial.println(BrakeT3);
  Serial.print(F("BrakeT4 = ")); Serial.println(BrakeT4);
  Serial.print(F("BrakeT5 = ")); Serial.println(BrakeT5);
  Serial.print(F("BrakeT6 = ")); Serial.println(BrakeT6);
  Serial.print(F("BrakeT7 = ")); Serial.println(BrakeT7);
}

// --------------------------------------------------------------------------------

void Reset_BackTimer() {
  // called from a switch operation to prevent LED background task
  BackTimer = BackTimerRes; LED_Mask = 0;
}

// --------------------------------------------------------------------------------

void Set_LiftedLvl() {
  // set the lifted off the surface level
  LiftedLvl = Sen_LftLvlMax;
  if (LiftedLvl < Sen_CtrLvlMax) {LiftedLvl = Sen_CtrLvlMax;}
  if (LiftedLvl < Sen_RhtLvlMax) {LiftedLvl = Sen_RhtLvlMax;}
  LED_LvlMax = LiftedLvl; // grab the max level for LED driver code
  LiftedLvl += 100; // set if 100 above darkest level
}

// --------------------------------------------------------------------------------

void Set_MainMode() {
  // called when changing modes
  detachServos();   // turn OFF servo motor drive

  BackTimer = BackTimerRes; // set LED background task delay
  Brake = 0;                // reset the brake counter
  IgnoreLift = false;       // set = true to ignore lifted auto-stop
  LED_BRK = false;          // turn OFF brake lights
  LED_Mask = 0;             // remove any masking effects
  MaskCnt = 0;              // reset mask counter for LED flashing
  PWM_Change = false;       // set true if a PWM value has changed
  Tacho_En = 0;             // turn OFF the tacho function, > 0 == ON
  TachoNew = false;         // reset the Tacho new value flag
  TaskClock = 0;            // reset the task duration clock
  TaskCnt = 0;              // reset the general counter used in tasks
  TaskESC = 0;              // changed to control task exit mode
  TaskPnt = 0;              // reset main task pointer
  TurnEn = false;           // disable steering functions
  TurnFlg = 0;              // reset, -1 = turn left, 0 = straight, +1 = turn right
  TurnPlus = false;         // set whilst turning if sustained turning is needed
  
  Serial.print(F("MainMode = ")); Serial.println(MainMode);
  SyncTimers();
}

// --------------------------------------------------------------------------------

void Set_Servos() {
  // output the servo values, reversing the right-hand side to match the left,
  // centred around the 1500µs mid point
  if (!Servos_Attached) {attachServos(); }  // ensure servos are attached
  servoLft.writeMicroseconds(PWM_Lft);
  servoRht.writeMicroseconds(3000 - PWM_Rht);

//  Serial.print(PWM_Lft); Serial.print(F("\t")); Serial.print(PWM_Rht);
}

// --------------------------------------------------------------------------------

void Set_Tacho_ON() {
  // initiates the tacho function in the main loop
  Tacho_En = 1; TachoHi = false; TachoLast = false; SpdSt = 0; SpdUp = 0;
}

// --------------------------------------------------------------------------------

void Set_Thresholds() {
  // sets the trigger thresholds for the three sensors based on calibration values
  // the centre threshold is set much closer to the black line limit
  Sen_LftThreshLvl = Sen_LftLvlMin + ((Sen_LftLvlMax - Sen_LftLvlMin)/4);
  Sen_CtrThreshLvl = Sen_CtrLvlMin + ((Sen_CtrLvlMax - Sen_CtrLvlMin)/16);
  Sen_RhtThreshLvl = Sen_RhtLvlMin + ((Sen_RhtLvlMax - Sen_RhtLvlMin)/4);
}

// --------------------------------------------------------------------------------

void Start_PWM() {
  // called from line tracking task. It determines the appropriate PWM values and
  // starts the motors
  switch(Speed) {
    case 1: PWM_LftSpeed = PWM_LftSpd1; PWM_RhtSpeed = PWM_RhtSpd1; BrakeMx = BrakeT1; break;
    case 2: PWM_LftSpeed = PWM_LftSpd2; PWM_RhtSpeed = PWM_RhtSpd2; BrakeMx = BrakeT2; break;
    case 3: PWM_LftSpeed = PWM_LftSpd3; PWM_RhtSpeed = PWM_RhtSpd3; BrakeMx = BrakeT3; break;
    case 4: PWM_LftSpeed = PWM_LftSpd4; PWM_RhtSpeed = PWM_RhtSpd4; BrakeMx = BrakeT4; break;
    case 5: PWM_LftSpeed = PWM_LftSpd5; PWM_RhtSpeed = PWM_RhtSpd5; BrakeMx = BrakeT5; break;
    case 6: PWM_LftSpeed = PWM_LftSpd6; PWM_RhtSpeed = PWM_RhtSpd6; BrakeMx = BrakeT6; break;
    case 7: PWM_LftSpeed = PWM_LftSpd7; PWM_RhtSpeed = PWM_RhtSpd7; BrakeMx = BrakeT7; break;
  }

  // turning variable Curve is in the range 1 - 7, but the turning rate also needs
  // to be a function of speed. The sustain TurnFlg also kicks in at the threshold
  // point for each sensor.
  PWM_LftGain = 0.143 * (float)((PWM_LftSpeed - PWM_LftSpd0) * Curve)/(float)(Sen_LftThreshLvl - Sen_LftLvlMin);
  PWM_RhtGain = 0.143 * (float)((PWM_RhtSpeed - PWM_RhtSpd0) * Curve)/(float)(Sen_RhtThreshLvl - Sen_RhtLvlMin);

  Serial.print(F("PWM_LftSpeed = ")); Serial.println(PWM_LftSpeed);
  Serial.print(F("PWM_RhtSpeed = ")); Serial.println(PWM_RhtSpeed);
  Serial.print(F("PWM_LftGain = ")); Serial.println(PWM_LftGain);
  Serial.print(F("PWM_RhtGain = ")); Serial.println(PWM_RhtGain);
  Serial.print(F("BrakeMx = ")); Serial.println(BrakeMx);
  TurnCnt = 8;  // reseed the curve rate counter

  // set stop counters
  NoLineSTOP = (5 * 250) - (Speed * 36);  // stop sooner if speed is higher
  
  PWM_Lft = PWM_LftSpeed; PWM_Rht = PWM_RhtSpeed;
  Set_Servos();
  SyncTimers(); // re-synch main loop counter due to print statements
}

// --------------------------------------------------------------------------------

void Steer() {
  // provides steering functions for several modes  
  TurnCnt--; PWM_Change = false;
  if (Brake > 0) {
    // Brake counter is active due to a black crossover line, so ensure we are
    // going in a straight line
    if (PWM_Lft != PWM_LftSpeed) {PWM_Lft = PWM_LftSpeed; PWM_Change = true;}
    if (PWM_Rht != PWM_RhtSpeed) {PWM_Rht = PWM_RhtSpeed; PWM_Change = true;}
  } else {
    // if not braking then apply normal steering
    if (Sen_LftVal > Sen_LftLvlMin) {
      // left sensor is detecting a reduction in light level
      AnyInt = PWM_LftSpeed - (int)(PWM_LftGain * (float)(Sen_LftVal - Sen_LftLvlMin));
      if (AnyInt < PWM_LftSpd0) {AnyInt = PWM_StopRef;}  // PWM dropped below stall so kill it
      if (TurnFlg < 0) {
        // if turn flag is -ve maintain a sustained turn based on max value
        if (AnyInt < PWM_Lft) {PWM_Lft = AnyInt; PWM_Change = true;}
        else if (TurnCnt == Curve) {
          // sensor may not have taken turning to its maximum so continue the trend
          if (PWM_Lft <= PWM_LftSpd0) {PWM_Lft = PWM_StopRef; PWM_Change = true;}
          if (PWM_Lft > PWM_LftSpd0) {PWM_Lft--; PWM_Change = true;}
        }
      } else {
        // if TurnFlg is >= 0 then PWM tracks proportional control
        if (AnyInt != PWM_Lft) {PWM_Lft = AnyInt; PWM_Change = true;}
      }
    } else if (PWM_Lft != PWM_LftSpeed) {PWM_Lft = PWM_LftSpeed; PWM_Change = true;}
    if (Sen_RhtVal > Sen_RhtLvlMin) {
      // right sensor is detecting a reduction in light level
      AnyInt = PWM_RhtSpeed - (int)(PWM_RhtGain * (float)(Sen_RhtVal - Sen_RhtLvlMin));
      if (AnyInt < PWM_RhtSpd0) {AnyInt = PWM_StopRef;}  // PWM dropped below stall so kill it
      if (TurnFlg > 0) {
        // if turn flag is +ve maintain a sustained turn based on max value
        if (AnyInt < PWM_Rht) {PWM_Rht = AnyInt; PWM_Change = true;}
        else if (TurnCnt == Curve) {
          // sensor may not have taken turning to its maximum so continue the trend
          if (PWM_Rht <= PWM_RhtSpd0) {PWM_Rht = PWM_StopRef; PWM_Change = true;}
          if (PWM_Rht > PWM_RhtSpd0) {PWM_Rht--; PWM_Change = true;}
        }
      } else {
        // if TurnFlg is <= 0 then PWM tracks proportional control
        if (AnyInt != PWM_Rht) {PWM_Rht = AnyInt; PWM_Change = true;}
      }
    } else if (PWM_Rht != PWM_RhtSpeed) {PWM_Rht = PWM_RhtSpeed; PWM_Change = true;}
  }
  if (PWM_Change) {Set_Servos();}
  if (TurnCnt <= Curve) {TurnCnt = 8;}
}

// --------------------------------------------------------------------------------

void SyncTimers() {
  // re-synch the main loop timer
  nextMicros = micros() + Interval;   // prime the loop timer
}

// --------------------------------------------------------------------------------

void Validate_Limits() {
  // check sensor limts and determine other settings
  bool Good = true;
  if ((Sen_LftLvlMax / Sen_LftLvlMin) < 3) {Good = false;}
  if ((Sen_CtrLvlMax / Sen_CtrLvlMin) < 3) {Good = false;}
  if ((Sen_RhtLvlMax / Sen_RhtLvlMin) < 3) {Good = false;}

  if (Good) {
    // values have sufficinet ratio
    Serial.println(F("Sensor Cal. values are good!"));
    // add a small offset to minimum values to remove ADC noise
    Sen_LftLvlMin += 1;
    Sen_CtrLvlMin += 1;
    Sen_RhtLvlMin += 1;
    EEPROM_Save();
  } else {
    // insufficient ratio so reload stored values
    Serial.println(F("Sensor Cal. values are poor!"));
    Report_Sen_MinMax();
    Serial.println(F("Reloading previous values"));
    EEPROM_Load();
  }

  // now set additional limits
  // set the lifted off the surface level
  Set_LiftedLvl();
  Set_Thresholds();
}

// --------------------------------------------------------------------------------

void Wait_ButtonsUp() {
  // invoked by button scanner and waits until both switches are released  
  SWL_State = digitalRead(SWL_Pin); SWL_Last = SWL_State;
  SWR_State = digitalRead(SWR_Pin); SWR_Last = SWR_State;
  if (SWL_State & SWR_State) {
    // both buttons released
    SWL_Skip = false; SWR_Skip = false; // clear the skip flags
    SWL_Cnt = 0; SWR_Cnt = 0;           // clear the timers
    SWL_Num = 0; SWR_Num = 0;           // clear the button pressed numbers
    SW_Rel = false;                     // finally exit this mode
  }
}

// --------------------------------------------------------------------------------

void Wait_UntilUp() {
  // code will not return until both button switches are released
  // all flags are reset
  SWL_State = HIGH; SWL_Last = SWL_State;
  SWR_State = HIGH; SWR_Last = SWR_State;
  SWL_Skip = false; SWR_Skip = false; // clear the skip flags
  SWL_Cnt = 0; SWR_Cnt = 0;           // clear the timers
  SWL_Num = 0; SWR_Num = 0;           // clear the button pressed numbers
  SW_Rel = false;                     // finally exit this mode
  while (!digitalRead(SWL_Pin) || !digitalRead(SWR_Pin)) {
    // do nothing
  }
}
