/*
  Move functions and (parameters) are:
  
  CurveLeft(mm,aa)      Draw a curve of radius mm, through angle aa
  CurveRight(mm,aa)     Draw a curve of radius mm, through angle aa
  DrawLine(mm)          Draw a line in the current heading
  DrawToPoint(x,y)      Draw a line from the current position to x,y
  GoBackward(mm)        Move backwards without drawing
  GoForward(mm)         Move forward without drawing
  GoToPoint(x,y)        Move to the point to x,y
  PenDwn()              Put pen to paper
  PenMax()              Raise pen to max height
  PenUp()               Raise pen from paper
  PolyLeft(n,mm)        Draw an 'n' sided polygon, side length mm
  PolyRight(n,mm)       Draw an 'n' sided polygon, side length mm
  RectLeft(mm,mm)       Draw a rectangle to the left
  RectRight(mm,mm)      Draw a rectangle to the right
  SetOrigin(x,y,ang)    Store current position as reference point
  TurnLeft(Deg,Pen)     Turn left without drawing
  TurnRight(Deg,Pen)    Turn right without drawing
  TurnTo(aa)            Turn at the current position to a new heading
  
*/

// ----------------------------------------------------------------------

void CurveLeft(float zRad,float zAng) {
  // draw a curve of radius zRad, turning left through zAng degrees
  float zOD = (zAng / 360.0) * (6.28319 * (zRad + (axleWidth / 2)));
  zOD = (zOD * 4096.0 * ScaleXY) / (PI * wheelDia);
  long zOP = int(zOD); // distance in motor pulses
//  Serial.print("zOP = "); Serial.println(zOP);
  float zID = (zAng / 360.0) * (6.28319 * (zRad - (axleWidth / 2)));
  int zDir = 1; if (zID < 0.0) {zDir = -1;}
  zID = (abs(zID) * 4096.0 * ScaleXY) / (PI * wheelDia);
//  Serial.print("zIP = "); Serial.println(zIP);
  if (penDrawMode == 0) {PenUp();}
  if (penDrawMode > 0) {PenDwn();}
  interval = intervalMax;
  Brake = 0; long zB = abs(zOP - intervalBTT); // set brake threshold
  if (zOP < 136) {zB = (zOP * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  long zLastIP = 0; // set tracking count
  for (long zL = 0; zL < zOP; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    long zIncIP = ((zL + 1) * zID) / zOD;
    if (zIncIP > zLastIP) {
      zLastIP = zIncIP;
      if (zDir > 0) {incPhaseL();} else {decPhaseL();}
    } decPhaseR();
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void CurveRight(float zRad,float zAng) {
  // draw a curve of radius zRad, turning right through zAng degrees
  float zOD = (zAng / 360.0) * (6.28319 * (zRad + (axleWidth / 2)));
  zOD = (zOD * 4096.0 * ScaleXY) / (PI * wheelDia);
  long zOP = int(zOD); // distance in motor pulses
  float zID = (zAng / 360.0) * (6.28319 * (zRad - (axleWidth / 2)));
  int zDir = 1; if (zID < 0.0) {zDir = -1;}
  zID = (abs(zID) * 4096.0 * ScaleXY) / (PI * wheelDia);
  if (penDrawMode == 0) {PenUp();}
  if (penDrawMode > 0) {PenDwn();}
  interval = intervalMax;
  Brake = 0; long zB = abs(zOP - intervalBTT); // set brake threshold
  if (zOP < 136) {zB = (zOP * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  long zLastIP = 0; // set tracking count
  for (long zL = 0; zL < zOP; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    long zIncIP = ((zL + 1) * zID) / zOD;
    if (zIncIP > zLastIP) {
      zLastIP = zIncIP;
      if (zDir > 0) {decPhaseR();} else {incPhaseR();}
    } incPhaseL();
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void DrawLine(float zDistance) {
  // move directly forward whilst drawing a line
//  Serial.print("Distance="); Serial.println(zDistance);
  zDistance = (zDistance * 4096.0 * ScaleXY) / (PI * wheelDia);
  anyL = int(zDistance); // distance in motor pulses
//  Serial.print("Steps="); Serial.println(anyD); Serial.println("");
  if (penDrawMode == 0) {PenUp();}
  if (penDrawMode > 0) {PenDwn();}
  interval = intervalMax;
  Brake = 0; int zB = abs(anyL - intervalBTT); // set brake threshold
  if (anyL < 136) {zB = (anyL * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  for (int zL = 0; zL < anyL; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    incPhaseL(); decPhaseR(); 
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void DrawToPoint(float zX, float zY) {
  // draw from the current x,y position to zX, zY
  // first calculate the distance to travel
  Serial.println("\nDrawToPoint...");
  float zDX = zX - CartX; float zDY = zY - CartY;
  float zDH = sqrt((zDX * zDX) + (zDY * zDY));
  float zAng = 0.0; // create temp turn angle variable
  float zOrAng = 0.0; // create temp origin angle variable
  // now calculate the turn angle
  if (zDX == 0.0) {
    // new point is directly above or below current point
    if (zDY > 0.0) {
      // new point is directly above current point
      zOrAng = 0.0; zAng = CartAng;
      if (CartAng != 0.0) {zAng = -CartAng;} 
    } else if (zDY < 0.0) {
      // new point is directly below current point
      zOrAng = 180.0; zAng = 180.0;
      if (CartAng > 0.0) {zAng = 180.0 - CartAng;} 
      if (CartAng < 0.0) {zAng = -180.0 - CartAng;} 
    } else {
      // new point is at current point
    }
  }
  else if (zDY == 0.0) {
    // new point is directly to left or right of current point
    if (zDX > 0.0) {
      // new point is directly to the right of current point
      zOrAng = 90.0; zAng = 0.0;
      if (CartAng > 90.0) {zAng = -(CartAng - 90.0);} 
      if (CartAng < 90.0) {zAng = 90.0 - CartAng;} 
    } else if (zDX < 0.0) {
      // new point is directly to the left of current point
      zOrAng = -90.0; zAng = 0.0;
      if (CartAng < -90.0) {zAng = -CartAng - 90.0;} 
      if (CartAng > -90.0) {zAng = -90.0 - CartAng;} 
    } else {
      // new point is at current point
    }
  }
  else if (zDX > 0.0) {
    // new point is in 1st or 2nd quadrant
    if (zDY > 0.0) {
      // new point is in 1st quadrant
      Serial.println("Quad = 1");
      zOrAng = (atan(zDX / zDY) * 180) / PI; zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    } else if(zDY < 0.0) {
      // new point is in 2nd quadrant
      Serial.println("Quad = 2");
      zOrAng = 180.0 +((atan(zDX / zDY) * 180) / PI); zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    }
  }
  else if (zDX < 0.0) {
    // new point is in 3rd or 4th quadrant
    if (zDY > 0.0) {
      // new point is in 4th quadrant
      Serial.println("Quad = 4");
      zOrAng = (atan(zDX / zDY) * 180) / PI; zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    } else if(zDY < 0.0) {
      // new point is in 3rd quadrant
      Serial.println("Quad = 3");
      zOrAng = -180.0 + ((atan(zDX / zDY) * 180) / PI); zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    }
  }
  // check angle isn't > 180?
  if (abs(zAng) > 180.0) {
    if (zAng > 0.0) {zAng = -(360.0 - zAng);}
    else if (zAng < 0.0) {zAng = 360.0 + zAng;}
  }
  // now turn and draw as necessary
  if (zAng > 0.0) {TurnRight(zAng, 1); Serial.print("TR = "); Serial.println(zAng);}
  if (zAng < 0.0) {TurnLeft(-zAng, 1); Serial.print("TL = "); Serial.println(zAng);}
  if (zDH > 0.0) {DrawLine(zDH); Serial.print("zDH = "); Serial.println(zDH);}
  // store new position and heading
  CartX = zX; CartY = zY; CartAng = zOrAng; CheckCartAng();
  Serial.print("CartX = "); Serial.println(CartX);
  Serial.print("CartY = "); Serial.println(CartY);
  Serial.print("OrAng = "); Serial.println(zOrAng);
  Serial.print("CAng = "); Serial.println(CartAng);
}

// ----------------------------------------------------------------------

void DrawToPoint2(float zX0,float zY0,float zX1,float zY1) {
  // perform 2 x DrawToPoint functions
  DrawToPoint(zX0,zY0);
  DrawToPoint(zX1,zY1);
}

// ----------------------------------------------------------------------

void DrawToPoint3(float zX0,float zY0,float zX1,float zY1,float zX2,float zY2) {
  // perform 3 x DrawToPoint functions
  DrawToPoint(zX0,zY0);
  DrawToPoint(zX1,zY1);
  DrawToPoint(zX2,zY2);
}

// ----------------------------------------------------------------------

void DrawToPoint4(float zX0,float zY0,float zX1,float zY1,float zX2,float zY2,float zX3,float zY3) {
  // perform 4 x DrawToPoint functions
  DrawToPoint(zX0,zY0);
  DrawToPoint(zX1,zY1);
  DrawToPoint(zX2,zY2);
  DrawToPoint(zX3,zY3);
}

// ----------------------------------------------------------------------

void DrawToPoint5(float zX0,float zY0,float zX1,float zY1,float zX2,float zY2,float zX3,float zY3,float zX4,float zY4) {
  // perform 5 x DrawToPoint functions
  DrawToPoint(zX0,zY0);
  DrawToPoint(zX1,zY1);
  DrawToPoint(zX2,zY2);
  DrawToPoint(zX3,zY3);
  DrawToPoint(zX4,zY4);
}

// ----------------------------------------------------------------------

void GoBackward(float zDistance) {
  // move directly backwards
  zDistance = (zDistance * 4096.0 * ScaleXY) / (PI * wheelDia);
  anyL = int(zDistance);
  PenUp();
  interval = intervalMax;
  Brake = 0; int zB = abs(anyL - intervalBTT); // set brake threshold
  if (anyL < 136) {zB = (anyL * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  for (int zL = 0; zL < anyL; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    decPhaseL(); incPhaseR(); 
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void GoForward(float zDistance) {
  // move directly forward
  nextMicros = micros() + interval;
  zDistance = (zDistance * 4096.0 * ScaleXY) / (PI * wheelDia);
  anyL = int(zDistance);
  PenUp();
  interval = intervalMax;
  Brake = 0; int zB = abs(anyL - intervalBTT); // set brake threshold
  if (anyL < 136) {zB = (anyL * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  for (int zL = 0; zL < anyL; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    incPhaseL(); decPhaseR(); 
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void GoToPoint(float zX, float zY) {
  // move from the current x,y position to zX, zY
  // first calculate the distance to travel
  Serial.println("\nGoToPoint...");
  float zDX = zX - CartX; float zDY = zY - CartY;
  float zDH = sqrt((zDX * zDX) + (zDY * zDY));
  float zAng = 0.0; // create temp turn angle variable
  float zOrAng = 0.0; // create temp origin angle variable
  // now calculate the turn angle
  if (zDX == 0.0) {
    // new point is directly above or below current point
    if (zDY > 0.0) {
      // new point is directly above current point
      zOrAng = 0.0; zAng = CartAng;
      if (CartAng != 0.0) {zAng = -CartAng;} 
    } else if (zDY < 0.0) {
      // new point is directly below current point
      zOrAng = 180.0; zAng = 180.0;
      if (CartAng > 0.0) {zAng = 180.0 - CartAng;} 
      if (CartAng < 0.0) {zAng = -180.0 - CartAng;} 
    } else {
      // new point is at current point
    }
  }
  else if (zDY == 0.0) {
    // new point is directly to left or right of current point
    if (zDX > 0.0) {
      // new point is directly to the right of current point
      zOrAng = 90.0; zAng = 0.0;
      if (CartAng > 90.0) {zAng = -(CartAng - 90.0);} 
      if (CartAng < 90.0) {zAng = 90.0 - CartAng;} 
    } else if (zDX < 0.0) {
      // new point is directly to the left of current point
      zOrAng = -90.0; zAng = 0.0;
      if (CartAng < -90.0) {zAng = -CartAng - 90.0;} 
      if (CartAng > -90.0) {zAng = -90.0 - CartAng;} 
    } else {
      // new point is at current point
    }
  }
  else if (zDX > 0.0) {
    // new point is in 1st or 2nd quadrant
    if (zDY > 0.0) {
      // new point is in 1st quadrant
      Serial.println("Quad = 1");
      zOrAng = (atan(zDX / zDY) * 180) / PI; zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    } else if(zDY < 0.0) {
      // new point is in 2nd quadrant
      Serial.println("Quad = 2");
      zOrAng = 180.0 +((atan(zDX / zDY) * 180) / PI); zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    }
  }
  else if (zDX < 0.0) {
    // new point is in 3rd or 4th quadrant
    if (zDY > 0.0) {
      // new point is in 4th quadrant
      Serial.println("Quad = 4");
      zOrAng = (atan(zDX / zDY) * 180) / PI; zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    } else if(zDY < 0.0) {
      // new point is in 3rd quadrant
      Serial.println("Quad = 3");
      zOrAng = -180.0 + ((atan(zDX / zDY) * 180) / PI); zAng = 0.0;
      if (CartAng < zOrAng) {zAng = zOrAng - CartAng;} 
      if (CartAng > zOrAng) {zAng = -(CartAng - zOrAng);} 
    }
  }
  // check angle isn't > 180?
  if (abs(zAng) > 180.0) {
    if (zAng > 0.0) {zAng = -(360.0 - zAng);}
    else if (zAng < 0.0) {zAng = 360.0 + zAng;}
  }
  // now turn and move as necessary
  if (zAng > 0.0) {TurnRight(zAng, 0); Serial.print("TR = "); Serial.println(zAng);}
  if (zAng < 0.0) {TurnLeft(-zAng, 0); Serial.print("TL = "); Serial.println(zAng);}
  if (zDH > 0.0) {GoForward(zDH); Serial.print("zDH = "); Serial.println(zDH);}
  // store new position and heading
  CartX = zX; CartY = zY; CartAng = zOrAng; CheckCartAng();
  Serial.print("CartX = "); Serial.println(CartX);
  Serial.print("CartY = "); Serial.println(CartY);
  Serial.print("OrAng = "); Serial.println(zOrAng);
  Serial.print("CAng = "); Serial.println(CartAng);
}

// ----------------------------------------------------------------------

void joyForward() {
  // perform joystick move forward function
  if (MainTask == 50) {
    // already in moving state, so check direction & vary speed
    if (PhaseDirL == 1) {
      // increase forward left wheel speed up to a maximum
      intervalJL = (intervalJL * 7) / 8;
      intervalJL = max(intervalJL, 1000);
    } else {
      // slow down reverse left movement
      intervalJL = (intervalJL * 9) / 8;
      if (intervalJL > 40000) {PhaseDirL = 1;}
      intervalJL = min(intervalJL, 40000);
    } Serial.print("Interval L = "); Serial.println(intervalJL);
    if (PhaseDirR == 1) {
      // increase forward left wheel speed up to a maximum
      intervalJR = (intervalJR * 7) / 8;
      intervalJR = max(intervalJR, 1000);
    } else {
      // slow down reverse right movement
      intervalJR = (intervalJR * 9) / 8;
      if (intervalJR > 40000) {PhaseDirR = 1;}
      intervalJR = min(intervalJR, 40000);
    } Serial.print("Interval R = "); Serial.println(intervalJR);
    intervalJC = (intervalJL + intervalJR)/2;
  } else if (MainTask == -1) {
    // in idle state so enter moving forward state
    Serial.println("FORWARD");
    MainTask = 50; stepperON(); PenMax(); ServoOFF();
    PhaseDirL = 1; PhaseDirR = 1; // set direction of stepper
    intervalJC = 40000;
    intervalJL = 40000; nextJLMicros = micros() + intervalJL;
    intervalJR = 40000; nextJRMicros = micros() + intervalJR;
    interval = 40000; nextMicros = micros() + interval;
  }
}

// ----------------------------------------------------------------------

void joyLeft() {
  // perform joystick move left function  
  if (MainTask == 50) {
    // already in moving state, so check direction & vary speed
    if ((PhaseDirL == PhaseDirR) && (PhaseDirL != 0)) {
      // going in same direction so steer to the left by slowing it down
      intervalJL = (intervalJL * 9)/8;
      intervalJL = min(intervalJL, 40000);
      // increase speed of R-H wheel
      intervalJR = (intervalJR * 7)/8;
      intervalJR = max(intervalJR, 1000);
    }
    else {
      // already neutral turning so increase speed of both wheels
      PhaseDirL = -1; PhaseDirR = 1;
      intervalJL = (intervalJL * 7)/8;
      intervalJL = max(intervalJL, 1000);
      intervalJR = (intervalJR * 7)/8;
      intervalJR = max(intervalJR, 1000);
    }
  } else if (MainTask == -1) {
    // in idle state so enter rotating state
    MainTask = 50; stepperON(); PenMax(); ServoOFF();
    PhaseDirL = -1; PhaseDirR = 1; // set direction of stepper
    intervalJC = 0; // combined forward/reverse speed = 0
    intervalJL = 17500; nextJLMicros = micros() + intervalJL;
    intervalJR = 17500; nextJRMicros = micros() + intervalJR;
    interval = 20000; nextMicros = micros() + interval;
  } joyKey = 5; // set steering correction delay
}

// ----------------------------------------------------------------------

void joyReverse() {
  // perform joystick move in reverse function  
  if (MainTask == 50) {
    // already in moving state, so check direction & vary speed
    if (PhaseDirL == -1) {
      intervalJL = (intervalJL * 7) / 8;
      intervalJL = max(intervalJL, 1000);
    } else {
      intervalJL = (intervalJL * 9) / 8;
      if (intervalJL > 20000) {PhaseDirL = -1;}
      intervalJL = min(intervalJL, 40000);
    } Serial.print("Interval L = "); Serial.println(intervalJL);
    if (PhaseDirR == -1) {
      intervalJR = (intervalJR * 7) / 8;
      intervalJR = max(intervalJR, 1000);
    } else {
      intervalJR = (intervalJR * 9) / 8;
      if (intervalJR > 20000) {PhaseDirR = -1;}
      intervalJR = min(intervalJR, 40000);
    } Serial.print("Interval R = "); Serial.println(intervalJR);
    intervalJC = (intervalJL + intervalJR)/2;
  } else if (MainTask == -1) {
    // in idle state so enter moving forward state
    MainTask = 50; stepperON(); PenMax(); ServoOFF();
    PhaseDirL = -1; PhaseDirR = -1; // set direction of stepper
    intervalJC = 40000;
    intervalJL = 40000; nextJLMicros = micros() + intervalJL;
    intervalJR = 40000; nextJRMicros = micros() + intervalJR;
    interval = 40000; nextMicros = micros() + interval;
  }
}

// ----------------------------------------------------------------------

void joyRight() {
  // perform joystick move right function  
  if (MainTask == 50) {
    // already in moving state, so check direction & vary speed
    if ((PhaseDirL == PhaseDirR) && (PhaseDirL != 0)) {
      // going in same direction to steer to the right by slowing it down
      intervalJR = (intervalJR * 9)/8;
      intervalJR = min(intervalJR, 40000);
      // increase speed of L-H wheel
      intervalJL = (intervalJL * 7)/8;
      intervalJL = max(intervalJL, 1000);
    }
    else {
      // already neutral turning so increase speed of both wheels
      PhaseDirL = 1; PhaseDirR = -1;
      intervalJL = (intervalJL * 7)/8;
      intervalJL = max(intervalJL, 1000);
      intervalJR = (intervalJR * 7)/8;
      intervalJR = max(intervalJR, 1000);
    }
  } else if (MainTask == -1) {
    // in idle state so enter rotating state
    MainTask = 50; stepperON(); PenMax(); ServoOFF();
    PhaseDirL = 1; PhaseDirR = -1; // set direction of stepper
    intervalJC = 0; // combined forward/reverse speed = 0
    intervalJL = 17500; nextJLMicros = micros() + intervalJL;
    intervalJR = 17500; nextJRMicros = micros() + intervalJR;
    interval = 20000; nextMicros = micros() + interval;
  } joyKey = 5; // set steering correction delay
}

// ----------------------------------------------------------------------

void joyStop() {
  // perform joystick stop function  
  if (MainTask == 50) {
    DrawEnd();
  }
}

// ----------------------------------------------------------------------

void PenDwn() {
  // lower the pen into the drawing position
  if (drawEn > 0) {
    if (servoEn < 1) {ServoON();}
    if (servoVal != liftDwn) {
      servoLift.writeMicroseconds(liftDwn);
      servoVal = liftDwn; delay(servoWait);
    }
  }
}

// ----------------------------------------------------------------------

void PenMax() {
  // raise the pen to the max height position
  if (servoEn < 1) {ServoON();}
  if (servoVal != liftMax) {
    servoLift.writeMicroseconds(liftMax);
    servoVal = liftMax; delay(servoWait);
  }
}

// ----------------------------------------------------------------------

void PenMode(int zMode) {
  // sets the current line drawing mode
  // 0 = do not draw
  // 1 = draw shapes - continuous line
  // 2 = draw shapes - dashed line
  penDrawMode = zMode;
}

// ----------------------------------------------------------------------

void PenUp() {
  // raise the pen off the paper
  if (servoEn < 1) {ServoON();}
  if (servoVal != liftUp) {
    servoLift.writeMicroseconds(liftUp);
    servoVal = liftUp;
    delay(servoWait);
  }
}

// ----------------------------------------------------------------------

void PolyLeft(float zNS,float zS1) {
  // draw a zNS sided polygon, turning left after side zS1
  int zN = zNS; zNS = 360.0 / zNS;
  for (int zL = 0; zL < zN; zL++) {
    DrawLine(zS1); TurnLeft(zNS,1);
  }
}

// ----------------------------------------------------------------------

void PolyRight(float zNS,float zS1) {
  // draw a zNS sided polygon, turning right after side zS1
  int zN = zNS; zNS = 360.0 / zNS;
  for (int zL = 0; zL < zN; zL++) {
    DrawLine(zS1); TurnRight(zNS,1);
  }
}

// ----------------------------------------------------------------------

void RectLeft(float zS1, float zS2) {
  // draw a rectangle turning to the left after side zS1
  DrawLine(zS1); TurnLeft(90,1);
  DrawLine(zS2); TurnLeft(90,1);
  DrawLine(zS1); TurnLeft(90,1);
  DrawLine(zS2);
}

// ----------------------------------------------------------------------

void RectRight(float zS1, float zS2) {
  // draw a rectangle turning to the right after side zS1
  DrawLine(zS1); TurnRight(90,1);
  DrawLine(zS2); TurnRight(90,1);
  DrawLine(zS1); TurnRight(90,1);
  DrawLine(zS2);
}

// ----------------------------------------------------------------------

void SetOrigin(float zX, float zY,float zAng) {
  // sets the origin for cartesian drawing functions
  // if zAng = 0.0 your heading points along the Y-axis
  // zAng has the range of -180.0 to 0.0 to +180.0
  CartX = zX; CartY = zY; CartAng = zAng; CartF = 1;
  CheckCartAng(); // ensure angle is within limits
  Serial.println("\nSetOrigin...");
  Serial.print("CartX = "); Serial.println(CartX);
  Serial.print("CartY = "); Serial.println(CartY);
  Serial.print("CAng = "); Serial.println(CartAng);
}

// ----------------------------------------------------------------------

void TurnLeft(float zDegrees, int zPen) {
  // neutral turn to the left by zDegrees
  zDegrees = zDegrees * (axleWidth / wheelDia) * 11.3777;
  anyL = int(zDegrees);
  if (zPen < 1) {PenUp();} else {PenDwn();}
  interval = intervalMax;
  Brake = 0; int zB = abs(anyL - intervalBTT); // set brake threshold
  if (anyL < 136) {zB = (anyL * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  for (int zL = 0; zL < anyL; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    decPhaseL(); decPhaseR();
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void TurnRight(float zDegrees, int zPen) {
  // neutral turn to the right by zDegrees
  zDegrees = zDegrees * (axleWidth / wheelDia) * 11.3777;
  anyL = int(zDegrees);
  if (zPen < 1) {PenUp();} else {PenDwn();}
  interval = intervalMax;
  Brake = 0; int zB = abs(anyL - intervalBTT); // set brake threshold
  if (anyL < 136) {zB = (anyL * intervalInc) / (intervalInc + intervalDec);}
  nextMicros = micros() + interval;
  for (int zL = 0; zL < anyL; zL++) {
    if (drawEn < 1) break;
    if (zL >= zB) Brake = 1; // apply the brake
    incPhaseR(); incPhaseL();
    SetPhaseLeft(); SetPhaseRight(); delayStep();
  }
}

// ----------------------------------------------------------------------

void TurnTo(float zAng) {
  // Turn at the current point to a new CartAng heading
  Serial.println("\nTurnTo...");
  float zTurn = CartAng - zAng; // determine heading difference
  if (zTurn > 0.0) {TurnLeft(zTurn, 0);}
  if (zTurn < 0.0) {TurnRight(zTurn, 0);}
  CartAng = zAng;
  Serial.print("zAng = "); Serial.println(zAng);
  Serial.print("CAng = "); Serial.println(CartAng);
}

// ----------------------------------------------------------------------





