  //##############################################################################
  //
  //  Cube Frame Task Functions
  //
  //##############################################################################


void doFrameTasks() {
  // called every 20ms (variable) to progress task actions
  // tasks functions can be delayed several cycles by setting FrameDel > 0
  // tasks can be blocked until buttons are released by WaitUp == true
  // task variables can be reset to their defaults by setting FrameTask < 0
  if (PAUSE) {return;}  // PAUSE flag set by the user, stops all animations
  if (FrameDel > 0) {FrameDel--; FrameBuild = false; return;} // set > 0 to reduce/delay task rate
  if (WaitUp) {if ((sw0State == LOW) || (sw1State == LOW) || (sw2State == LOW)) {return;} else {WaitUp = false;}}
  
  // if FrameTask is -ve we set variables to their default values before starting it
  if (FrameTask < 0) {setFrameTask(-FrameTask);}
  
  // FrameTasks with low numbers are LED pattern generators.
  // Whilst FrameBuild == true these tasks are called repeatedly. However, once a
  // task has done what it needs to do, it will set FrameBuild = false.
  // If you shorten or extend this list you must change the FrameMax definition.
  FrameMin = 1; FrameMax = 20;
  switch(FrameTask) {
    case  0: FrameBuild = false; break; // do nothing, default state
    case  1: Frame_Light(); break;      // white light
    case  2: Frame_Mood(); break;       // mood light
    case  3: Frame_Rain(); break;       // raining simulator
    case  4: Frame_Matrix(); break;     // Matrix simulator
    case  5: Frame_Worms(); break;      // rampent worms
    case  6: Frame_Burst(); break;      // rainbow burst
    case  7: Frame_SunSpots(); break;   // solar sun spots
    case  8: Frame_Spirals(); break;    // coloured spiral faces
    case  9: Frame_ASCII(); break;      // scrolling alphabet
    case 10: Frame_SpiralRoll(); break; // rotating spiral drum
    case 11: Frame_VuMeter(); break;    // scrolling VU sound meter
    case 12: Frame_VuSpin(); break;     // spinning VU speed animation
    case 13: Frame_VuFace(); break;     // sound animated face
    case 14: Frame_Discharge(); break;  // draw random connected lines, with discharge effect
    case 15: Frame_Level(); break;      // spirit level mode
    // case 16: Frame_Ethan(); break;      // Display messages to Ethan
    case 16: Frame_UTC(); break;        // Displays UTC promotional messages
    case 17: Frame_CoderDojo(); break;  // Displays CoderDojo promotional messages
    case 18: Frame_Lines(); break;      // Displays random lines on all surfaces
    case 19: Frame_Bars(); break;       // Displays random 2x2 solid bars on all surfaces
    case 20: Frame_Volcano(); break;    // Volcano simulation
    
    // FrameTasks with the following are special functions
    case 80: Frame_TEST(); break;       // only used in TEST mode when building cube
    case 91:
      Frame_CubeClr(); break; // clear cube faces then pause
    case 92:
      // NULL task used as a task end waiting for another event
      FrameBuild = false; break;
    case 93:
      // clear cube and display a Gfx character left justified
      Frame_GfxLft(); break;
    case 94:
      // clear cube and display a Gfx character right justified
      Frame_GfxRht(); break;
    case 95:
      // called after Frame_Text() to display the message
      Frame_TextDisplay(); break;
    case 96:
      // called to change from one task to another
      FastLED.clear(); FastLED.show(); setFrameTask(FrameNum); break;
    case 97:
      // called to terminate current task due to a switch operation
      FrameOFF(); break;
    case 98:
      // SW0 & SW1 & SW2 released before invoking MainNext
      if ((sw0State == HIGH) && (sw1State == HIGH) && (sw2State == HIGH)) {FrameTask = FrameNext;}
      break;
    case 99:
      // delay before invoking MainNext task, MainCnt = 50 for 1 second delay
      FrameCnt--; if (FrameCnt < 1) {FrameTask = FrameNext;}
      break;
    case 100:
      Frame_PwrOn(); break; // rising light bars on power on
    case 999:
      Frame_REBOOT(); break;  // red melt-down
    default:
      // error - task not defined so clear the pointer
      Serial.println("ERROR - Frame unknown!");
      setFrameTask(0); break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_ASCII() {
  // displays a range of ASII characters, and rotate them around the cube in
  // random directions and in random colours
  switch(LED_Task) {
    case 0:
      // initialise this task
      Serial.println("Frame_ASCII");
      Gfx_SetFace(0);   // set face data
      Mask = 0b00000111; // centre characters + clear pixels
      FramePeriod = 40; // slow frame rate down to 25 Hz
      // choose a random start point
      LED_Pnt = random8(4);
           if (LED_Pnt == 0) {LED_Pnt = '!';}  // start with this character
      else if (LED_Pnt == 1) {LED_Pnt = '0';}  // start with this character
      else if (LED_Pnt == 2) {LED_Pnt = 'A';}  // start with this character
      else if (LED_Pnt == 3) {LED_Pnt = 'a';}  // start with this character
      LED_Task++; break;
    case 1:
      // draw a character on Face 0 and copy to the others
      Gfx_ClrFace(0); Gfx_RndCol(Col11);
      Gfx_DrawChar(0,3,0,LED_Pnt,Mask);
      Gfx_CopyFace(0,1); Gfx_CopyFace(0,2); Gfx_CopyFace(0,3); Gfx_CopyFace(0,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt = 8; FrameDel = 5; LED_Rnd = random(9);
      LED_Task++; break;
    case 2:
      // move or dim the faces
      switch(LED_Rnd) {
        case 0: Gfx_RotRht(0); break;               // rotate the face right by 1 pixel
        case 1: Gfx_RotLft(0); break;               // rotate the face left by 1 pixel
        case 2: Gfx_RotDwn(0); break;               // rotate the face down by 1 pixel
        case 3: Gfx_RotUp(0); break;                // rotate the face up by 1 pixel
        case 4: Gfx_DimFace(0,1); break;            // dim face pixels by 1
        case 5: // rotate right down
          Gfx_ClrFace(0);
          switch(LED_Cnt) {
            case 8: Gfx_DrawChar(0,4,1,LED_Pnt,Mask); Gfx_DrawChar(0,-4,-7,LED_Pnt,Mask); break;
            case 7: Gfx_DrawChar(0,5,2,LED_Pnt,Mask); Gfx_DrawChar(0,-3,-6,LED_Pnt,Mask); break;
            case 6: Gfx_DrawChar(0,6,3,LED_Pnt,Mask); Gfx_DrawChar(0,-2,-5,LED_Pnt,Mask); break;
            case 5: Gfx_DrawChar(0,7,4,LED_Pnt,Mask); Gfx_DrawChar(0,-1,-4,LED_Pnt,Mask); break;
            case 4: Gfx_DrawChar(0,0,5,LED_Pnt,Mask); Gfx_DrawChar(0, 0,-3,LED_Pnt,Mask); break;
            case 3: Gfx_DrawChar(0,1,6,LED_Pnt,Mask); Gfx_DrawChar(0, 1,-2,LED_Pnt,Mask); break;
            case 2: Gfx_DrawChar(0,2,7,LED_Pnt,Mask); Gfx_DrawChar(0, 2,-1,LED_Pnt,Mask); break;
            case 1: Gfx_DrawChar(0,3,0,LED_Pnt,Mask); break;
          } break;
        case 6: // rotate right up
          Gfx_ClrFace(0);
          switch(LED_Cnt) {
            case 8: Gfx_DrawChar(0,4,-1,LED_Pnt,Mask); Gfx_DrawChar(0,-4,7,LED_Pnt,Mask); break;
            case 7: Gfx_DrawChar(0,5,-2,LED_Pnt,Mask); Gfx_DrawChar(0,-3,6,LED_Pnt,Mask); break;
            case 6: Gfx_DrawChar(0,6,-3,LED_Pnt,Mask); Gfx_DrawChar(0,-2,5,LED_Pnt,Mask); break;
            case 5: Gfx_DrawChar(0,7,-4,LED_Pnt,Mask); Gfx_DrawChar(0,-1,4,LED_Pnt,Mask); break;
            case 4: Gfx_DrawChar(0,0,-5,LED_Pnt,Mask); Gfx_DrawChar(0, 0,3,LED_Pnt,Mask); break;
            case 3: Gfx_DrawChar(0,1,-6,LED_Pnt,Mask); Gfx_DrawChar(0, 1,2,LED_Pnt,Mask); break;
            case 2: Gfx_DrawChar(0,2,-7,LED_Pnt,Mask); Gfx_DrawChar(0, 2,1,LED_Pnt,Mask); break;
            case 1: Gfx_DrawChar(0,3,0,LED_Pnt,Mask); break;
          } break;
        case 7: // rotate left & down by 1 pixel
          Gfx_ClrFace(0);
          switch(LED_Cnt) {
            case 8: Gfx_DrawChar(0, 2,1,LED_Pnt,Mask); Gfx_DrawChar(0,10,-7,LED_Pnt,Mask); break;
            case 7: Gfx_DrawChar(0, 1,2,LED_Pnt,Mask); Gfx_DrawChar(0, 9,-6,LED_Pnt,Mask); break;
            case 6: Gfx_DrawChar(0, 0,3,LED_Pnt,Mask); Gfx_DrawChar(0, 8,-5,LED_Pnt,Mask); break;
            case 5: Gfx_DrawChar(0,-1,4,LED_Pnt,Mask); Gfx_DrawChar(0, 7,-4,LED_Pnt,Mask); break;
            case 4: Gfx_DrawChar(0,-2,5,LED_Pnt,Mask); Gfx_DrawChar(0, 6,-3,LED_Pnt,Mask); break;
            case 3: Gfx_DrawChar(0,-3,6,LED_Pnt,Mask); Gfx_DrawChar(0, 5,-2,LED_Pnt,Mask); break;
            case 2: Gfx_DrawChar(0,-4,7,LED_Pnt,Mask); Gfx_DrawChar(0, 4,-1,LED_Pnt,Mask); break;
            case 1: Gfx_DrawChar(0,3,0,LED_Pnt,Mask); break;
          } break;
        case 8: // rotate left & up by 1 pixel
          Gfx_ClrFace(0);
          switch(LED_Cnt) {
            case 8: Gfx_DrawChar(0, 2,-1,LED_Pnt,Mask); Gfx_DrawChar(0,10,7,LED_Pnt,Mask); break;
            case 7: Gfx_DrawChar(0, 1,-2,LED_Pnt,Mask); Gfx_DrawChar(0, 9,6,LED_Pnt,Mask); break;
            case 6: Gfx_DrawChar(0, 0,-3,LED_Pnt,Mask); Gfx_DrawChar(0, 8,5,LED_Pnt,Mask); break;
            case 5: Gfx_DrawChar(0,-1,-4,LED_Pnt,Mask); Gfx_DrawChar(0, 7,4,LED_Pnt,Mask); break;
            case 4: Gfx_DrawChar(0,-2,-5,LED_Pnt,Mask); Gfx_DrawChar(0, 6,3,LED_Pnt,Mask); break;
            case 3: Gfx_DrawChar(0,-3,-6,LED_Pnt,Mask); Gfx_DrawChar(0, 5,2,LED_Pnt,Mask); break;
            case 2: Gfx_DrawChar(0,-4,-7,LED_Pnt,Mask); Gfx_DrawChar(0, 4,1,LED_Pnt,Mask); break;
            case 1: Gfx_DrawChar(0,3,0,LED_Pnt,Mask); break;
          } break;
      }
      Gfx_CopyFace(0,1); Gfx_CopyFace(0,2); Gfx_CopyFace(0,3); Gfx_CopyFace(0,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task++;}
      break;
    case 3:
      // choose next character or restart sequence
      LED_Pnt++; if (LED_Pnt > '~') {LED_Pnt = '!';}
      LED_Task = 1; FrameDel = 5; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Bars() {
  // Draws random bars on all 5 faces
  // Use the 'F***.' serial commands to tune this animation
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Lines");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      FrameBright = 8;        // brightness factor
      FrameCnt = 2;           // number of lines drawn
      FrameDim = 2;           // dim increment
      FramePeriod = 100;      // slow frame rate down
      LED_Task++; break;
    case 1:
      // Dim down all faces
      Gfx_Dim(0,NumLEDsM1,FrameDim);  // dim down all LEDs
      // Draw random lines on each face
      for (int16_t zF = 0;zF < 5; zF++) {
        Gfx_SetFace(zF);
        for (int16_t zC = 0;zC < FrameCnt;zC++) {
          Gfx_RndCol(FrameBright);
          Gfx_Bar(random8(7),random8(7),2,2);
        }
      }
      // Use the mic to set the frame period
      micVU = map(micPkTrk,micLvl,micMax,100,30);
      if (micVU < 30) {micVU = 30;} else if (micVU > 100) {micVU = 100;}
      FramePeriod = micVU;
      // Now display this
      LEDshow = true; FrameBuild = false;
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Burst() {
  // Rainbow burst, starting on a random face and spreading to the opposite face
  // the depth of the burst also varies randomly
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Burst");
      FramePeriod = 40; // slow frame rate down to 25 Hz
      LED_Task++; break;
    case 1:
      // chooose a face
      LED_Rnd = random8(0,6);               // set random face source
      LED_Col = random8(0,12);              // set random start colour
      FramePeriod = 40 - (4 * random8(6));  // set random speed
      LED_End = random8(11,23);             // set random length of pattern
      LED_Pnt = 0; LED_Cnt = 0;
      LED_Task++; break;
    case 2:
      // animate selected face
      switch (LED_Rnd) {
        case 0:
          // top face 0 source
          Gfx_GetFace(0); Gfx_PicXY(0,0);                           // get top edge colour
          Gfx_RotDwn(1); Gfx_GetFace(1); Gfx_Line(0,0,7,0);         // rotate down and draw new line
          Gfx_CopyFace(1,2); Gfx_CopyFace(1,3); Gfx_CopyFace(1,4);  // copy faces
          // animate top face
          Gfx_GetFace(0); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
          Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
          Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
          Gfx_SetCol(LED_Col); Gfx_Box(3,3,2,2);
          LED_Pnt++;
          if (LED_Pnt > LED_End) {LED_Col = 12; LED_Cnt++;}         // clear colour
          else {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}        // cycle colours
          if (LED_Cnt > 11) {LED_Task = 1;}
          break;
        case 1:
          // rear face 1 source
          Gfx_GetFace(1); Gfx_PicXY(0,0);                     // get rear edge colour
          Gfx_RotDwn(0); Gfx_GetFace(0); Gfx_Line(0,0,7,0);   // rotate down and draw new line
          Gfx_CopyFaceRR(0,2); Gfx_CopyFaceRL(0,4);           // copy rotated faces
          // animate rear face
          Gfx_GetFace(1); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
          Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
          Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
          Gfx_SetCol(LED_Col); Gfx_Box(3,3,2,2);
          // animate front face
          Gfx_GetFace(3); Gfx_PicXY(2,2); Gfx_Box(3,3,2,2);
          Gfx_PicXY(1,1); Gfx_Box(2,2,4,4);
          Gfx_PicXY(0,0); Gfx_Box(1,1,6,6);
          Gfx_GetFace(0); Gfx_PicXY(7,7);
          Gfx_GetFace(3); Gfx_Box(0,0,8,8);
          LED_Pnt++;
          if (LED_Pnt > LED_End) {LED_Col = 12; LED_Cnt++;}              // clear colour
          else {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}        // cycle colours
          if (LED_Cnt > 15) {LED_Task = 1;}
          break;
        case 2:
          // front face 3 source
          Gfx_GetFace(3); Gfx_PicXY(0,0);                     // get rear edge colour
          Gfx_RotUp(0); Gfx_GetFace(0); Gfx_Line(0,7,7,7);    // rotate up and draw new line
          Gfx_CopyFaceRR(0,2); Gfx_CopyFaceRL(0,4);           // copy rotated faces
          // animate rear face
          Gfx_GetFace(3); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
          Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
          Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
          Gfx_SetCol(LED_Col); Gfx_Box(3,3,2,2);
          // animate rear face
          Gfx_GetFace(1); Gfx_PicXY(2,2); Gfx_Box(3,3,2,2);
          Gfx_PicXY(1,1); Gfx_Box(2,2,4,4);
          Gfx_PicXY(0,0); Gfx_Box(1,1,6,6);
          Gfx_GetFace(0); Gfx_PicXY(0,0);
          Gfx_GetFace(1); Gfx_Box(0,0,8,8);
          LED_Pnt++;
          if (LED_Pnt > LED_End) {LED_Col = 12; LED_Cnt++;}              // clear colour
          else {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}        // cycle colours
          if (LED_Cnt > 15) {LED_Task = 1;}
          break;
        case 3:
          // left face 4 source
          Gfx_GetFace(4); Gfx_PicXY(0,0);                     // get top edge colour
          Gfx_RotRht(0); Gfx_GetFace(0); Gfx_Line(0,0,0,7);   // rotate right and draw new line
          Gfx_CopyFace(0,3); Gfx_CopyFaceFH(0,1);             // copy and flipped faces
          // animate left face
          Gfx_GetFace(4); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
          Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
          Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
          Gfx_SetCol(LED_Col); Gfx_Box(3,3,2,2);
          // animate right face
          Gfx_GetFace(2); Gfx_PicXY(2,2); Gfx_Box(3,3,2,2);
          Gfx_PicXY(1,1); Gfx_Box(2,2,4,4);
          Gfx_PicXY(0,0); Gfx_Box(1,1,6,6);
          Gfx_GetFace(0); Gfx_PicXY(7,0);
          Gfx_GetFace(2); Gfx_Box(0,0,8,8);
          LED_Pnt++;
          if (LED_Pnt > LED_End) {LED_Col = 12; LED_Cnt++;}              // clear colour
          else {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}        // cycle colours
          if (LED_Cnt > 15) {LED_Task = 1;}
          break;
        case 4:
          // right face 2 source
          Gfx_GetFace(2); Gfx_PicXY(0,0);                     // get top edge colour
          Gfx_RotLft(0); Gfx_GetFace(0); Gfx_Line(7,0,7,7);   // rotate leftt and draw new line
          Gfx_CopyFace(0,3); Gfx_CopyFaceFH(0,1);             // copy and flipped faces
          // animate right face
          Gfx_GetFace(2); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
          Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
          Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
          Gfx_SetCol(LED_Col); Gfx_Box(3,3,2,2);
          // animate left face
          Gfx_GetFace(4); Gfx_PicXY(2,2); Gfx_Box(3,3,2,2);
          Gfx_PicXY(1,1); Gfx_Box(2,2,4,4);
          Gfx_PicXY(0,0); Gfx_Box(1,1,6,6);
          Gfx_GetFace(0); Gfx_PicXY(0,0);
          Gfx_GetFace(4); Gfx_Box(0,0,8,8);
          LED_Pnt++;
          if (LED_Pnt > LED_End) {LED_Col = 12; LED_Cnt++;}              // clear colour
          else {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}        // cycle colours
          if (LED_Cnt > 15) {LED_Task = 1;}
          break;
        case 5:
          // bottom face source
          // animate top face
          Gfx_GetFace(0); Gfx_PicXY(2,2); Gfx_Box(3,3,2,2);
          Gfx_PicXY(1,1); Gfx_Box(2,2,4,4);
          Gfx_PicXY(0,0); Gfx_Box(1,1,6,6);
          Gfx_GetFace(1); Gfx_PicXY(0,0);
          Gfx_GetFace(0); Gfx_Box(0,0,8,8);
          // animate side faces
          Gfx_RotUp(1); Gfx_GetFace(1);
          Gfx_SetCol(LED_Col); Gfx_Line(0,7,7,7);                   // rotate up and draw new line
          Gfx_CopyFace(1,2); Gfx_CopyFace(1,3); Gfx_CopyFace(1,4);  // copy faces
          LED_Pnt++;
          if (LED_Pnt > LED_End) {LED_Col = 12; LED_Cnt++;}              // clear colour
          else {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}        // cycle colours
          if (LED_Cnt > 12) {LED_Task = 1;}
          break;
      }
      LEDshow = true; FrameBuild = false;
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_CoderDojo() {
  // Displays CoderDojo messages.
  // The number of message is only limited by the number of FrameSubTask, so it is
  // easy to change, extend or shorten the sequence.
  LED_Pnt = 0; LED_Task = 0; FrameNext = FrameTask;;  // defaults for each text message
  FrameTask = 95;   // set default task for carousel display
  switch(FrameSubTask) {
    case 0:
      // initialise
      Serial.println("Frame_CoderDojo");
      Gfx_SetFace(1);   // text is drawn to the rear face, then scrolled round to the front
      FramePeriod = 40; // slow frame rate down to 25 Hz
      Mask = 0b00000000; // left-justified
      // first message
      LED_Text = "CoderDojo Darlington"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 1:
      // message
      LED_Text = "Learn + Design + Code"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 2:
      // message
      LED_Text = "Scratch, Python, BBC micro:bit, Arduino, and more..."; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 3:
      // message
      LED_Text = "Countless career opportunities"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 4:
      // message
      LED_Text = "Code saves lives and helps us communicate"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 5:
      // message
      LED_Text = "Become a code today!"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    default:
      // end of messages, so repeat from the beginning
      FrameSubTask = 0; FrameTask = FrameNext; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_CubeClr() {
  // clear ALL cube faces, RGB LED 1 - 320, immediately
  // then wait for a button switch to be released before running another task
  Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
  LEDshow = true; FrameBuild = false;
  WaitUp = true; FrameTask = 92;
}

// ---------------------------------------------------------------------------------

void Frame_Discharge() {
  // Draw random connected lines with a top face static discharge effect
  // Points on the four vertical edges are created randomly, with connecting lines
  // drawn on the four side faces. A previous point will move to new point progressively,
  // before a new point being generated. When a random point hits a top corner a bright
  // line is drawn on the top face, giving a discharge effect.
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Discharge");
      FramePeriod = 40; // set initial frame rate
      FramePrdLL = 25;  // set FramePeriod Lower limit, fastest
      FramePrdUL = 40;  // set FramePeriod upper limit, slowest
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      for (int zP = 0; zP < ArrayMax; zP++) {LED_Array[zP] = 7;}  // clear array
      LED_Col = 3; LED_Min = 3; LED_Cnt = 5; 
      Gfx_SetCol(random8(12)); Gfx_MemCol();
      LED_Task++; break;
    case 1:
      Gfx_ClrFace(1); Gfx_ClrFace(2); Gfx_ClrFace(3); Gfx_ClrFace(4);
      // set new points or move them
      if (LED_Array[1] == LED_Array[11]) {LED_Array[1] = random8(LED_Min,8);}
      else {if (LED_Array[1] > LED_Array[11]) {LED_Array[11]++;} else {LED_Array[11]--;}}
      if (LED_Array[2] == LED_Array[12]) {LED_Array[2] = random8(LED_Min,8);}
      else {if (LED_Array[2] > LED_Array[12]) {LED_Array[12]++;} else {LED_Array[12]--;}}
      if (LED_Array[3] == LED_Array[13]) {LED_Array[3] = random8(LED_Min,8);}
      else {if (LED_Array[3] > LED_Array[13]) {LED_Array[13]++;} else {LED_Array[13]--;}}
      if (LED_Array[4] == LED_Array[14]) {LED_Array[4] = random8(LED_Min,8);}
      else {if (LED_Array[4] > LED_Array[14]) {LED_Array[14]++;} else {LED_Array[14]--;}}
      LED_Task++; break;
    case 2:
      // use a random colour
      LED_Col--; if (LED_Col == 0) {
        LED_Col = 3; Gfx_SetCol(random8(12)); Gfx_MemCol();
        LED_Cnt--; if (LED_Cnt == 0) {if (LED_Min > 0) {LED_Min--; LED_Cnt = 5;}}
      }
      // draw line between points on side faces
      if ((LED_Array[11] == 0) || (LED_Array[14] == 0)) {Gfx_MultCol(4);} else {Gfx_RecCol();}
      Gfx_GetFace(1); Gfx_Line(0,LED_Array[11],7,LED_Array[14]);
      if ((LED_Array[12] == 0) || (LED_Array[11] == 0)) {Gfx_MultCol(4);} else {Gfx_RecCol();}
      Gfx_GetFace(2); Gfx_Line(0,LED_Array[12],7,LED_Array[11]);
      if ((LED_Array[13] == 0) || (LED_Array[12] == 0)) {Gfx_MultCol(4);} else {Gfx_RecCol();}
      Gfx_GetFace(3); Gfx_Line(0,LED_Array[13],7,LED_Array[12]);
      if ((LED_Array[14] == 0) || (LED_Array[13] == 0)) {Gfx_MultCol(4);} else {Gfx_RecCol();}
      Gfx_GetFace(4); Gfx_Line(0,LED_Array[14],7,LED_Array[13]);
      // dim face 0
      Gfx_DimFace(0,2);
      // draw flash lines on face 0 as a bright colour
      Gfx_GetFace(0); Gfx_RecCol(); Gfx_MultCol(4);
      if (LED_Array[11] == 0) {Gfx_Line(4,3,7,0); Gfx_Box(3,3,2,2);}
      if (LED_Array[12] == 0) {Gfx_Line(4,4,7,7); Gfx_Box(3,3,2,2);}
      if (LED_Array[13] == 0) {Gfx_Line(3,4,0,7); Gfx_Box(3,3,2,2);}
      if (LED_Array[14] == 0) {Gfx_Line(3,3,0,0); Gfx_Box(3,3,2,2);}
      LEDshow = true; FrameBuild = false;
      // map sound to speed
      FramePeriod = map(micVol,micLvl,micMax,FramePrdUL,FramePrdLL);
      if (FramePeriod < FramePrdLL) {FramePeriod = FramePrdLL;}
      else if (FramePeriod > FramePrdUL) {FramePeriod = FramePrdUL;}
      LED_Task = 1; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Ethan() {
  // Displays Ethan's messages for my grandson.
  // The number of message is only limited by the number of FrameSubTask, so it is
  // easy to change, extend or shorten the sequence.
  LED_Pnt = 0; LED_Task = 0; FrameNext = FrameTask;;  // defaults for each text message
  FrameTask = 95;   // set default task for carousel display
  switch(FrameSubTask) {
    case 0:
      // initialise
      Gfx_SetFace(1);   // text is drawn to the rear face, then scrolled round to the front
      FramePeriod = 40; // slow frame rate down to 25 Hz
      Mask = 0b00000000; // left-justified
      // first message
      LED_Text = "Ethan's LED Cube"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 1:
      // message
      LED_Text = "I have 320 LEDs on 5 faces"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 2:
      // message
      LED_Text = "Each one can display 255 colours"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 3:
      // message
      LED_Text = "I respond to sounds and movements"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 4:
      // message
      LED_Text = "My brain is very fast"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 5:
      // message
      LED_Text = "I've just done 160 million things"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    default:
      // end of messages, so repeat from the beginning
      FrameSubTask = 0; FrameTask = FrameNext; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_GfxLft() {
  // In animation mode, clear cube and display a Gfx character left justified
  // In white light mode, display a white Gfx character left justified
  if (FrameNum == 1) {
    // in white lamp mode, display a '<' in a colour dependant on light level
    if (WhiteLight > 1) {
      // the colour of the '<' character is red
      LED_Red = Col11; LED_Grn = 0; LED_Blu = 0;
    } else {
      // the colour of the '<' character is yellow
      LED_Red = Col12; LED_Grn = Col12; LED_Blu = 0;}
  } else if (FrameTask > 2) {
    // not in white lamp mode
    Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
    // the colour of the '<' character is yellow
    LED_Red = Col12; LED_Grn = Col12; LED_Blu = 0;
  } 
  Gfx_DrawChar(0,0,0,Gfx_Char,0);
  LEDshow = true; FrameBuild = false;
}

// ---------------------------------------------------------------------------------

void Frame_GfxRht() {
  // In animation mode, clear cube and display a Gfx character right justified
  if (FrameNum == 1) {
    // in white lamp mode, display a '>' in a colour dependant on light level
    if (WhiteLight < WhiteMax) {
      // the colour of the '>' character is red
      LED_Red = Col11; LED_Grn = 0; LED_Blu = 0;
    } else {
      // the colour of the '>' character is yellow
      LED_Red = Col12; LED_Grn = Col12; LED_Blu = 0;}
  } else if (FrameTask > 2) {
    // not in white lamp mode
    Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
    // the colour of the '>' character is yellow
    LED_Red = Col12; LED_Grn = Col12; LED_Blu = 0;
  }
  Gfx_DrawChar(0,7,0,Gfx_Char,0b00000001);
  LEDshow = true; FrameBuild = false;
}

// ---------------------------------------------------------------------------------

void Frame_Level() {
  // Spirit level mode, in which a line is drawn around the cube to indicate 'pitch'
  // and a graphics symbol on the top face indicates 'yaw'
  // As the angle increase we also change the colours used.
  // Accelerometer offsets are captured on entering this mode, so it is relative
  // PitchGyr is +ve leaning towards buttons, face 3
  // PitchGyr is -ve leaning back from buttons, face 3
  // RollGyr is +ve leaning right, when viewed from face 3
  // RollGyr is -ve leaning leftt, when viewed from face 3
  // Ensure that the MPU was initialised, otherwise abort ad move onto the next task
  if (!MPU_Init) {setFrameTask(-(FrameTask + 1));}
  
  uint8_t zPYF; uint8_t zPYR; // default level values
  int zPG; int zRG;           // temp gyro values
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Level");
      AcXOff = -AcXAvg;  // use current average as offset data
      AcZOff = -AcZAvg;  // use current average as offset data
      FramePeriod = 30;
      LED_Task++; break;
    case 1:
      // Check to see if we are not moving, if not, then inject movement as a wave
      // so that we get some animation in Exec mode
      if (StaticCnt < 999) {zPG = abs(PitchGyrInt);}
      else {
        // Cube is stationary, so induce movement
        PitchGyrInt = random(-30,30); zPG = abs(PitchGyrInt);
        FrameDel = 5;
      }
      // determine lines to draw for pitch
           if (zPG > 31) {zPG-= 31; LED_Red = Col11; LED_Grn =     0; LED_Blu =     0;}
      else if (zPG > 23) {zPG-= 23; LED_Red = Col12; LED_Grn =     0; LED_Blu = Col12;}
      else if (zPG > 15) {zPG-= 15; LED_Red =     0; LED_Grn =     0; LED_Blu = Col11;}
      else if (zPG >  7) {zPG-=  7; LED_Red =     0; LED_Grn = Col12; LED_Blu = Col12;}
      else {LED_Red = 0; LED_Grn = Col11; LED_Blu = 0;}
      switch (zPG) {
        case 0: zPYF = 3; zPYR = 3; break;
        case 1: zPYF = 3; zPYR = 4; break;
        case 2: zPYF = 2; zPYR = 4; break;
        case 3: zPYF = 2; zPYR = 5; break;
        case 4: zPYF = 1; zPYR = 5; break;
        case 5: zPYF = 1; zPYR = 6; break;
        case 6: zPYF = 0; zPYR = 6; break;
        case 7: zPYF = 0; zPYR = 7; break;
        default: zPYF = 0; zPYR = 7; break;
      }
      if (PitchGyrInt < 0) {Any8 = zPYF; zPYF = zPYR; zPYR = Any8;} // swap for -ve values
      Gfx_Fill(0,NumLEDsM1,0,0,0);  // clear the cube
      Gfx_GetFace(2); Gfx_Line(0,zPYF,7,zPYR);
      Gfx_GetFace(4); Gfx_Line(0,zPYR,7,zPYF);
      Gfx_GetFace(3); Gfx_Line(0,zPYF,7,zPYF);
      Gfx_GetFace(1); Gfx_Line(0,zPYR,7,zPYR);
      LED_Task++; break;
    case 2:
      // determine lines to draw for Roll
      if (StaticCnt < 999) {zRG = abs(RollGyrInt);}
      else {
        // Cube is stationary, so induce movement
        RollGyrInt = random(-30,30); zRG = abs(RollGyrInt);
        FrameDel = 5;
      }
           if (zRG > 31) {zRG-= 31; LED_Red = Col11; LED_Grn =      0; LED_Blu =     0;}
      else if (zRG > 23) {zRG-= 23; LED_Red = Col12; LED_Grn =      0; LED_Blu = Col12;}
      else if (zRG > 15) {zRG-= 15; LED_Red =      0; LED_Grn =     0; LED_Blu = Col11;}
      else if (zRG >  7) {zRG-=  7; LED_Red =      0; LED_Grn = Col12; LED_Blu = Col12;}
      else {LED_Red = 0; LED_Grn = Col11; LED_Blu = 0;}
      if (zRG > 7) {zRG = 7;}
      loadVector(zRG);
      if (RollGyrInt < 0) {Gfx_FlipGCRL();}   // flip character for -ve values
      Gfx_DrawGfx(0,0,0);                     // put vector character on top face
      LEDshow = true; FrameBuild = false;
      LED_Task = 1; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Light() {
  // A white light on all 5 faces
  // The user can adjust the brightness with long presses on SW1 or SW2
  // The brightness will vary between 0 - LED_Max
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Light");
      Gfx_Fill(0,NumLEDsM1,WhiteLight,WhiteLight,WhiteLight);
      LEDshow = true; FrameBuild = false;
      FrameTask = 92; break; // do nothing more

      // the following code does not normally run
      // comment out the above line if you want to see how bright the cube can become
      LED_Pnt = LED_Max;
      LED_Task++; break;
    case 1:
      if (LED_Pnt < 255) {
        LED_Pnt++; Gfx_Fill(0,NumLEDsM1,LED_Pnt,LED_Pnt,LED_Pnt);
        LEDshow = true; FrameBuild = false;
        Serial.println("Level = " + String(LED_Pnt));
      } else {LED_Task++;}
      FrameDel = 5; break;
    case 2:
      if (LED_Pnt > 1) {
        LED_Pnt--; Gfx_Fill(0,NumLEDsM1,LED_Pnt,LED_Pnt,LED_Pnt);
        LEDshow = true; FrameBuild = false;
        Serial.println("Level = " + String(LED_Pnt));
      } else {LED_Task = 1;}
      FrameDel = 5; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Lines() {
  // Draws random lines on all 5 faces
  // Use the 'F***.' serial commands to tune this animation
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Lines");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      FrameBright = 8;        // brightness factor
      FrameCnt = 1;           // number of lines drawn
      FrameDim = 2;           // dim increment
      FramePeriod = 100;      // slow frame rate down
      LED_Task++; break;
    case 1:
      // Dim down all faces
      Gfx_Dim(0,NumLEDsM1,FrameDim);  // dim down all LEDs
      // Draw random lines on each face
      for (int16_t zF = 0;zF < 5; zF++) {
        Gfx_SetFace(zF);
        for (int16_t zC = 0;zC < FrameCnt;zC++) {
          Gfx_RndCol(FrameBright);
          Gfx_Line(random8(8),random8(8),random8(8),random8(8));
        }
      }
      // Now display this
      LEDshow = true; FrameBuild = false;
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Matrix() {
  // Green 'The Matrix' simulation.
  // the microphone audio is used to control the amount of top face activity.
  uint8_t zMin = Col34;
  uint8_t zX; uint8_t zY;
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Matrix");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      Gfx_SetFace(0); LED_Blu = 0; LED_Red = 0;
      FramePeriod = 100;      // slow frame rate down
      LED_Inc = Col14; LED_Task++; break;
    case 1:
      // dim down existing LEDs
      Gfx_Dim(GfxP0,GfxMax,LED_Inc);  // dim down face 0
      // get the number of random drops
      FrameRnd = random8(12,24);
      // apply audio to increase number of dots
      micVU = map(micPkTrk,micLvl,micMax,0,30);
      if (micVU >= 0) {FrameRnd+= micVU; LED_Inc = Col14;} else {micVU = 0; LED_Inc = Col12;}
      // Apply the dots
      for (int zP = 0;zP < FrameRnd; zP++) {
        // Set the green colour
        LED_Grn = random8(zMin,(Col11<<1));
        // position the dot
        zX = random8(8);  zY = random8(8); Gfx_SetXY(zX,zY);
      }
      LEDshow = true; FrameBuild = false;
      if (Col14 > 1) {LED_Task++;} 
      break;
    case 2:
      // dim down existing dots
      Gfx_Dim(GfxP0,GfxMax,LED_Inc);
      LEDshow = true; FrameBuild = false;
      if (Col14 == 1) {LED_Task = 1;} else {LED_Task++;}
      break;
    case 3:
      // dim down existing dots
      Gfx_Dim(GfxP0,GfxMax,LED_Inc);
      LEDshow = true; FrameBuild = false;
      LED_Task = 1; break;
  }
  if (LED_Task < 1) {return;}
  
  // scroll down the side faces and dim
  LED_Inc = random8(0,2);
  Gfx_MoveDwn(1); Gfx_CopyRowsRev(0,0,1,0); Gfx_DimFace(1,LED_Inc);
  Gfx_MoveDwn(2); Gfx_CopyColRowsRev(0,7,2,0); Gfx_DimFace(2,LED_Inc);
  Gfx_MoveDwn(3); Gfx_CopyRows(0,7,3,0); Gfx_DimFace(3,LED_Inc);
  Gfx_MoveDwn(4); Gfx_CopyColRows(0,0,4,0); Gfx_DimFace(4,LED_Inc);
  Gfx_SetFace(0); // reset the face pointer
}

// ---------------------------------------------------------------------------------

void Frame_Mood() {
  // A coloured 'mood' light on all 5 faces, default = cyan.
  // This function also handles mood colour change settings, initiated by the user
  // briefly place a finger over the Lux sensor
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Mood");
      Gfx_Fill(0,NumLEDsM1,MoodR,MoodG,MoodB);
      LEDshow = true; FrameBuild = false;
      Mask = 0b00000000; // left justify + don't clear pixels
      LED_Pnt = 0;  // sets the colour brightness of the top face character
      LED_Inc = 1;  // brightness adjustment factor
      LED_Task++; break;
    case 1:
      // do nothing unless luxMode adjustment is set
      if (luxMode > 0) {LED_Task++;}
      break;
    case 2:
      // respond to luxTask status, which defines the colour being adjusted
      if (luxMode == 0) {LED_Task = 9; break;}  // exiting this mode
      
      // set the X position of the top face character depending on its brightness
      LED_Num = LED_Max/4; LED_Num = ((LED_Num + LED_Pnt) * 3)/LED_Max;
      switch(luxTask) {
        case 0:
          // adjust red component
          Gfx_Fill(0,NumLEDsM1,MoodR,MoodG,MoodB);
          LED_Red = LED_Pnt; LED_Grn = 0; LED_Blu = 0;
          Gfx_DrawChar(0,LED_Num,0,'R',Mask); break;
        case 1:
          // adjust green component
          Gfx_Fill(0,NumLEDsM1,MoodR,MoodG,MoodB);
          LED_Red = 0; LED_Grn = LED_Pnt; LED_Blu = 0;
          Gfx_DrawChar(0,LED_Num,0,'G',Mask); break;
        case 2:
          // adjust blue component
          Gfx_Fill(0,NumLEDsM1,MoodR,MoodG,MoodB);
          LED_Red = 0; LED_Grn = 0; LED_Blu = LED_Pnt;
          Gfx_DrawChar(0,LED_Num,0,'B',Mask); break;
      } 
      LEDshow = true; FrameBuild = false;
      
      // change the brightness of the top face character
      LED_Pnt+= LED_Inc;
           if (LED_Pnt >= LED_Max) {LED_Inc = -1;}
      else if (LED_Pnt < 1) {LED_Inc = 1;}
      break;
    case 9:
      // user has dropped out of luxMode
      Gfx_FillFace(0,MoodR,MoodG,MoodB);
      LEDshow = true; FrameBuild = false;
      LED_Task = 1; break;
  }
}

// ---------------------------------------------------------------------------------

void FrameOFF() {
  // called to turn off the display and move to FrameTask 0
  Serial.println("FrameOFF");
  Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
  LEDshow = true; FrameBuild = false;
  setFrameTask(0);  // return to default sleep task
  // Serial.println("FrameOFF()");
}

// ---------------------------------------------------------------------------------

void Frame_PwrOn() {
  // called after POST if not in TEST mode
  // Generates two rising rainbow bursts in quick succession
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_PwrOn");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      LED_Pnt = 0;            // set the colour pointer
      LED_Cnt = 0;            // set exit counter
      LED_Task++; break;
    case 1:
      // draw boxes on the top face using top side colour
      Gfx_SetFace(0); Gfx_PicXY(2,2); Gfx_Box(3,3,2,2);
      Gfx_PicXY(1,1); Gfx_Box(2,2,4,4);
      Gfx_PicXY(0,0); Gfx_Box(1,1,6,6);
      Gfx_SetFace(1); Gfx_PicXY(0,0); Gfx_SetFace(0); Gfx_Box(0,0,8,8);
      // draw colour bands and roll them up
      Gfx_SetCol(LED_Pnt);  // set colours from a palette
      if (LED_Pnt < 12) {LED_Pnt++;}
      Gfx_RotUp(1);
      Gfx_SetFace(1); Gfx_Line(0,7,7,7);
      Gfx_CopyFace(1,2); Gfx_CopyFace(1,3); Gfx_CopyFace(1,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt++;
      if (LED_Cnt == 18) {LED_Pnt = 0;}     // do it again
      if (LED_Cnt >= 42) {setFrameTask(0);} // go to idle state
      FrameDel = 1; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Rain() {
  // Rain fall simulation, randomly puts blue pixels on teh top face.
  // Tthese slow disappear as the top face is being constantly dimmed down.
  // Any edge pixels are copied onto the side faces and scrolled down the side.
  uint8_t zMin = Col11/4;
  uint8_t zX; uint8_t zY;
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Rain");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      Gfx_SetFace(0); LED_Grn = 0; LED_Red = 0;
      FramePeriod = 40;   // slow frame rate down to 25 Hz
      LED_Inc = Col14; LED_Task++; break;
    case 1:
      Gfx_Dim(GfxP0,GfxMax,LED_Inc);  // dim down face 0
      FrameRnd = random8(5);
      micVU = map(micPkTrk,micLvl,micMax,0,30);
      if (micVU >= 0) {FrameRnd+= micVU; LED_Inc = Col14;} else {micVU = 0; LED_Inc = Col12;}
      if (FrameRnd > 0) {
        for (int zP = 0;zP < FrameRnd; zP++) {
          LED_Blu = random8(zMin,Col11<<1); zX = random8(8);  zY = random8(8);
          Gfx_SetXY(zX,zY);
          if (micVU > 2) {
            // apply a puddle
            LED_Blu = 1; Gfx_SetXY(zX+1,zY); Gfx_SetXY(zX-1,zY);
            Gfx_SetXY(zX,zY+1); Gfx_SetXY(zX,zY-1);
          }
        }
      }
      LEDshow = true; FrameBuild = false;
      LED_Task++; break;
    case 2:
      Gfx_Dim(GfxP0,GfxMax,LED_Inc);
      LEDshow = true; FrameBuild = false;
      LED_Task++; break;
    case 3:
      Gfx_Dim(GfxP0,GfxMax,LED_Inc);
      LEDshow = true; FrameBuild = false;
      LED_Task = 1; break;
  }
  if (LED_Task < 1) {return;}
  
  Gfx_MoveDwn(1); Gfx_CopyRowsRev(0,0,1,0);
  Gfx_MoveDwn(2); Gfx_CopyColRowsRev(0,7,2,0);
  Gfx_MoveDwn(3); Gfx_CopyRows(0,7,3,0);
  Gfx_MoveDwn(4); Gfx_CopyColRows(0,0,4,0);
}

// ---------------------------------------------------------------------------------

void Frame_REBOOT() {
  // Display the red REBOOT melt-down bright red burst
  // Serial.println("REBOOT");
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_REBOOT");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      Gfx_SetFace(0); LED_Blu = 0; LED_Grn = 0; LED_Pnt = 64;
      LED_Inc = 8; LED_Task++; break;
    case 1:
      Gfx_SetFace(0); Gfx_PicXY(0,0);
      Gfx_RotDwn(1); Gfx_SetFace(1); Gfx_Line(0,0,7,0);
      Gfx_CopyFace(1,2); Gfx_CopyFace(1,3); Gfx_CopyFace(1,4);
      Gfx_SetFace(0); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
      Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
      Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
      LED_Red = LED_Pnt; Gfx_Box(3,3,2,2);
      LEDshow = true; FrameBuild = false;
      if (LED_Inc > 1) {LED_Inc = (LED_Pnt / 6) + 1;}
      LED_Pnt-= LED_Inc; if (LED_Pnt == 0) {LED_Cnt = 15; LED_Task++;}
      break;
    case 2:
      Gfx_SetFace(0); Gfx_PicXY(0,0);
      Gfx_RotDwn(1); Gfx_SetFace(1); Gfx_Line(0,0,7,0);
      Gfx_CopyFace(1,2); Gfx_CopyFace(1,3); Gfx_CopyFace(1,4);
      Gfx_SetFace(0); Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
      Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
      Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
      LED_Red = LED_Pnt; Gfx_Box(3,3,2,2);
      LEDshow = true; FrameBuild = false;
     
      LED_Cnt--; if (LED_Cnt < 1) {resetFunc();}
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_SpiralRoll() {
  // Rotating spiral drum, generated from the centre of a face, turning in the
  // other direction on the opposite face. The duration of the spiral is random,
  // before starting a new spiral on a different face.
  switch(LED_Task) {
    case -1:
      // change task
      FramePeriod = 20 + (4 * random8(0,6));  // period 20 - 40
      LED_Last = LED_Task;
      while (LED_Last == LED_Task) {LED_Task = 1 + (2 * random8(3));} // task 1,3 or 5
      LED_Inc = random(2);  // set random rotation direction 0 or 1
      LED_Cnt = random8(29,128); LED_Col = 0; Gfx_SetCol(LED_Col);
      break;
    case 0:
      // initialise task
      Serial.println("Frame_SpiralRoll");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      LED_Task = -1; break;
    case 1:
      // face 0 clockwise out rainbow
      Gfx_SetFace(0);
      if (LED_Inc < 1) {Gfx_FaceSpiralACO(0); Gfx_SetXY(3,3);} else {Gfx_FaceSpiralCO(0); Gfx_SetXY(4,3);}
      for (int zR = 0; zR <= 7; zR++) {
        Gfx_CopyRowsRev(0,0,1,zR);    // copy down face 1
        Gfx_CopyColRowsRev(0,7,2,zR); // copy down face 2
        Gfx_CopyRows(0,7,3,zR);       // copy down face 3
        Gfx_CopyColRows(0,0,4,zR);    // copy down face 4
      }
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Cnt = 64; LED_Task++; LED_Col = 12;}
      break;
    case 2:
      // face 0 clockwise out off
      if (LED_Inc < 1) {Gfx_FaceSpiralACO(0); Gfx_SetXY(3,3);} else {Gfx_FaceSpiralCO(0); Gfx_SetXY(4,3);}
      for (int zR = 0; zR <= 7; zR++) {
        Gfx_CopyRowsRev(0,0,1,zR);    // copy down face 1
        Gfx_CopyColRowsRev(0,7,2,zR); // copy down face 2
        Gfx_CopyRows(0,7,3,zR);       // copy down face 3
        Gfx_CopyColRows(0,0,4,zR);    // copy down face 4
      }
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
    case 3:
      // face 1 out rainbow, face 3 rotates opposite
      if (LED_Inc < 1) {Gfx_FaceSpiralCO(1); Gfx_SetFace(1); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(3); Gfx_SetFace(3); Gfx_SetXY(3,3);}
      else             {Gfx_FaceSpiralCO(3); Gfx_SetFace(3); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(1); Gfx_SetFace(1); Gfx_SetXY(3,3);}
      for (int zR = 0; zR <= 7; zR++) {
        Gfx_CopyRowsRev(1,0,0,zR);  // copy down face 0
        Gfx_CopyCols(1,0,2,zR);     // copy left face 2
        Gfx_CopyCols(1,7,4,zR);     // copy right face 4
      }
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Cnt = 64; LED_Task++; LED_Col = 12;}
      break;
    case 4:
      // face 1 out off, face 3 rotates opposite
      if (LED_Inc < 1) {Gfx_FaceSpiralCO(1); Gfx_SetFace(1); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(3); Gfx_SetFace(3); Gfx_SetXY(3,3);}
      else             {Gfx_FaceSpiralCO(3); Gfx_SetFace(3); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(1); Gfx_SetFace(1); Gfx_SetXY(3,3);}
      for (int zR = 0; zR <= 7; zR++) {
        Gfx_CopyRowsRev(1,0,0,zR);  // copy down face 0
        Gfx_CopyCols(1,0,2,zR);     // copy left face 2
        Gfx_CopyCols(1,7,4,zR);     // copy right face 4
      }
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
    case 5:
      // face 2 out rainbow, face 4 rotates opposite
      if (LED_Inc < 1) {Gfx_FaceSpiralCO(2); Gfx_SetFace(2); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(4); Gfx_SetFace(4); Gfx_SetXY(3,3);}
      else             {Gfx_FaceSpiralCO(4); Gfx_SetFace(4); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(2); Gfx_SetFace(2); Gfx_SetXY(3,3);}
      for (int zR = 0; zR <= 7; zR++) {
        Gfx_CopyRowColsRev(2,0,0,zR); // copy left face 0
        Gfx_CopyCols(2,7,1,zR);       // copy right to face 1
        Gfx_CopyCols(2,0,3,zR);       // copy left to face 3
      }
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Cnt = 64; LED_Task++; LED_Col = 12;}
      break;
    case 6:
      // face 2 out off, face 4 rotates opposite
      if (LED_Inc < 1) {Gfx_FaceSpiralCO(2); Gfx_SetFace(2); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(4); Gfx_SetFace(4); Gfx_SetXY(3,3);}
      else             {Gfx_FaceSpiralCO(4); Gfx_SetFace(4); Gfx_SetXY(4,3); Gfx_FaceSpiralACO(2); Gfx_SetFace(2); Gfx_SetXY(3,3);}
      for (int zR = 0; zR <= 7; zR++) {
        Gfx_CopyRowColsRev(2,0,0,zR); // copy left face 0
        Gfx_CopyCols(2,7,1,zR);       // copy right to face 1
        Gfx_CopyCols(2,0,3,zR);       // copy left to face 3
      }
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
  }
  // change rainbow colour for all tasks
  if (LED_Col != 12) {LED_Col++; if (LED_Col > 11) {LED_Col = 0;}}
  Gfx_SetCol(LED_Col);
}

// ---------------------------------------------------------------------------------

void Frame_Spirals() {
  // Rainbow coloured spirals are drawn on the top face, then copied to the other
  // faces whilst being rotated in turn, making each face look different. After a
  // random period the face will change to the most recent random colour.
  switch(LED_Task) {
    case -1:
      // change task
      FramePeriod = 20 + (5 * random8(0,9));  // period 20 - 60
      LED_Task = 1 + (2 * random8(4));        // task 1 - 7
      LED_Rnd = random8(11,33);
      break;
    case 0:
      // initialise task
      Serial.println("Frame_Spirals");
      FramePeriod = 40;
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      LED_Rnd = random8(11,33);
      LED_Task++; break;
    case 1:
      // spiral clockwise in
      Gfx_FaceSpiralCI(0);
      Gfx_SetCol(LED_Col); Gfx_SetXY(7,0);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt++; if (LED_Cnt > 2) {
        LED_Cnt = 0;
        LED_Col++; if (LED_Col > 11) {LED_Col = 0;} // cycle the colour
        LED_Rnd--; if (LED_Rnd < 1) {LED_Task++; LED_Cnt = 64 + random8(13);}
      }
      break;
    case 2:
      // run colour out
      Gfx_FaceSpiralCI(0);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
    case 3:
      // spiral anti-clockwise in
      Gfx_FaceSpiralACI(0);
      Gfx_SetCol(LED_Col); Gfx_SetXY(0,0);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt++; if (LED_Cnt > 2) {
        LED_Cnt = 0;
        LED_Col++; if (LED_Col > 11) {LED_Col = 0;} // cycle the colour
        LED_Rnd--; if (LED_Rnd < 1) {LED_Task++; LED_Cnt = 64 + random8(13);}
      }
      break;
    case 4:
      // run colour out
      Gfx_FaceSpiralACI(0);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
    case 5:
      // spiral clockwise out
      Gfx_FaceSpiralCO(0);
      Gfx_SetCol(LED_Col); Gfx_SetXY(4,3);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt++; if (LED_Cnt > 2) {
        LED_Cnt = 0;
        LED_Col++; if (LED_Col > 11) {LED_Col = 0;} // cycle the colour
        LED_Rnd--; if (LED_Rnd < 1) {LED_Task++; LED_Cnt = 64 + random8(13);}
      }
      break;
    case 6:
      // run colour out
      Gfx_FaceSpiralCO(0);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
    case 7:
      // spiral anti-clockwise out
      Gfx_FaceSpiralACO(0);
      Gfx_SetCol(LED_Col); Gfx_SetXY(3,3);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt++; if (LED_Cnt > 2) {
        LED_Cnt = 0;
        LED_Col++; if (LED_Col > 11) {LED_Col = 0;} // cycle the colour
        LED_Rnd--; if (LED_Rnd < 1) {LED_Task++; LED_Cnt = 64 + random8(13);}
      }
      break;
    case 8:
      // run colour out
      Gfx_FaceSpiralACO(0);
      Gfx_CopyFaceRR(0,1); Gfx_CopyFaceRR(1,2); Gfx_CopyFaceRR(2,3); Gfx_CopyFaceRR(3,4);
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task = -1;}
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_SunSpots() {
  // Solar sun spots places white spots randomly on the cube faces, which are being
  // constantly rotated in a clockwise direction, whilst the background colour of
  // the cube achanges from yellow to red.
  uint8_t zWhite; // white intensity
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_SunSpots");
      FramePeriod = 40;
      LED_Inc = 1; LED_Num = 0;
      BAK_Red = Col14; BAK_Grn = Col14; BAK_Blu = 0;
      Gfx_Fill(0,NumLEDsM1,BAK_Red,BAK_Grn,BAK_Blu);
      LEDshow = true; FrameBuild = false;
      LED_Task++; break;
    case 1:
      // slowly change background colour
      LED_Num++; if (LED_Num > 10) {
        LED_Num = 0;
        if (LED_Inc > 0) {
          // make it more red
          if (BAK_Red < Col11) {BAK_Red++; if (BAK_Grn > 0) {BAK_Grn--;}}
          else {LED_Inc = -1;}
        } else {
          // make it more yellow
          if (BAK_Red > Col14) {BAK_Red--; if (BAK_Grn < Col14) {BAK_Grn++;}}
          else {LED_Inc = 1;}
        }
      }
      // dim cube towards background colour
      Gfx_DimTo(1,NumLEDsM1,1,BAK_Red,BAK_Grn,BAK_Blu);
      zWhite = random8(Col11,Col11<<1);
      LED_Red = zWhite; LED_Grn = zWhite; LED_Blu = zWhite;
      Gfx_SetFace(0); Gfx_SetXY(random8(8),random8(8));
      Gfx_SetFace(1); Gfx_SetXY(random8(8),random8(8));
      Gfx_SetFace(2); Gfx_SetXY(random8(8),random8(8));
      Gfx_SetFace(3); Gfx_SetXY(random8(8),random8(8));
      Gfx_SetFace(4); Gfx_SetXY(random8(8),random8(8));
      LED_Task++; break;
    case 2:
      // perform rotation
      LED_Cnt++;
      if (LED_Cnt > 2) {
        LED_Cnt = 0;
        // rotate side faces clockwise around face 0
        Gfx_RotLft(4); Gfx_CopyCols(3,0,4,7);
        Gfx_RotLft(3); Gfx_CopyCols(2,0,3,7);
        Gfx_RotLft(2); Gfx_CopyCols(1,0,2,7);
        Gfx_RotLft(1); Gfx_CopyCols(4,0,1,7);
        // rotate top face
        Gfx_FaceRotC(0,0); Gfx_FaceRotC(0,1);
        Gfx_FaceRotC(0,2); Gfx_FaceRotC(0,3);
      }
      LEDshow = true; FrameBuild = false;
      LED_Task = 1; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_TEST() {
  // Called only when initialised in TEST = true mode. This generates a walking
  // pixel from face 0 to face 4, in changing re, green and blue colours.
  switch(LED_Task) {
    case 0: LED[LED_Cnt].setRGB(0, 0, 0); LED[LED_Pnt].setRGB(15, 0, 0); break;
    case 1: LED[LED_Cnt].setRGB(0, 0, 0); LED[LED_Pnt].setRGB( 0,15, 0); break;
    case 2: LED[LED_Cnt].setRGB(0, 0, 0); LED[LED_Pnt].setRGB( 0, 0,15); break;
  }
  LED_Cnt = LED_Pnt; LED_Pnt++;
  if (LED_Pnt >= NumLEDs) {
    LED_Pnt = 0; LED_Task ++;
    if (LED_Task > 2) {LED_Task = 0;}
  } LEDshow = true; FrameBuild = false;
}

// ---------------------------------------------------------------------------------

void Frame_Text(String zT,int zN) {
  // Called to display text zT as a carousel, then either repeat or goto another
  // task. If flag zN > 0 then repeat the cycle using the same text zT.
  // the text is actually displayed with Frame_TextDisplay() once set up here
  Gfx_SetFace(1);   // text is drawn to the rear face, then scrolled round to the front
  FramePeriod = 40; // slow frame rate down to 25 Hz
  LED_Text = zT;    // load the text buffer with the desired message
  LED_Pnt = 0;      // point at the fiorst character
  LED_Num = LED_Text.length();  // store the length of the string
  Mask = 0b00000000; // left-justified
  LED_Task = 0;
  if (zN > 0) {FrameNext = FrameTask;} else {FrameNext = 0;}
  FrameTask = 95;   // off we go...
}

// ---------------------------------------------------------------------------------

void Frame_TextDisplay() {
  // Displays the contents of LED_Text as a scrolling carousel banner.
  // To use this function, first call Frame_Text(t$) with a text string.
  uint8_t zF; uint8_t zT;
  switch(LED_Task) {
    case 0:
      // load the character onto face 1
      Gfx_ClrFace(1);
      LED_Char = LED_Text.charAt(LED_Pnt);
      // set random colour or blank
      if (LED_Char == 32) {Gfx_SetCol(12);} else {Gfx_RndCol(Col11);}
      Gfx_DrawChar(1,0,0,LED_Char,Mask);
      LED_Cnt = GW +1;
      LED_Task++; break;
    case 1:
      // rotate the four faces, pulling the character off face 1
      Gfx_GetFace(0);
      // rotate the top face
      // row 1,1 - 5,1
      zT = GfxMax - 41;
      zF = zT + 8; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
      // column 1,6 - 1,1
      zF = zT - 1; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
      // row 6,6 - 1,6
      zF = zT - 8; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
      // column 6,1 - 6,6
      zF = zT + 1; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
      // replace the 1st pixel with the latest character colour
      Gfx_SetXY(6,1);
      
      // copy and rotate faces
      Gfx_RotLft(4); Gfx_CopyCols(3,0,4,7);
      Gfx_RotLft(3); Gfx_CopyCols(2,0,3,7);
      Gfx_RotLft(2); Gfx_CopyCols(1,0,2,7);
      Gfx_RotLft(1); Gfx_ClrCol(1,7);
      LEDshow = true; FrameBuild = false;
      LED_Cnt--; if (LED_Cnt < 1) {LED_Task++;}
      // test for Exec and hold it off whilst displaying text
      if (Exec > 0) {if (Exec <= 5) {Exec = 5;}}
      break;
    case 2:
      LED_Pnt++;
      if (LED_Pnt < LED_Num) {LED_Task = 0;}
      else if (LED_Pnt < (LED_Num + 24)) {
        // reached the end of the string, so just show blanks
        Gfx_ClrFace(1); LED_Task = 1;
        LED_Red = 0; LED_Grn = 0; LED_Blu = 0;
      } else {
        if ((Exec > 0) && (Exec <= 5)) {LED_Task = 3; Exec = 1;}
        else {FrameTask = FrameNext;}
      } break;
    case 3:
      // wait for Exec to terminate this task
      // this should occur within 100ms
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_UTC() {
  // Displays UTC promotional messages
  // The number of message is only limited by the number of FrameSubTask, so it is
  // easy to change, extend or shorten the sequence.
  LED_Pnt = 0; LED_Task = 0; FrameNext = FrameTask;;  // defaults for each text message
  FrameTask = 95;   // set default task for carousel display
  switch(FrameSubTask) {
    case 0:
      // initialise
      Gfx_SetFace(1);   // text is drawn to the rear face, then scrolled round to the front
      FramePeriod = 40; // slow frame rate down to 25 Hz
      Mask = 0b00000000; // left-justified
      // first message
      LED_Text = "UTC Robots Rock!"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 1:
      // message
      LED_Text = "Learn Design Make"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 2:
      // message
      LED_Text = "Model your future on the UTC"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    case 3:
      // message
      LED_Text = "STEM careers are rewarding"; LED_Num = LED_Text.length();
      FrameSubTask++; break;  // off we go...
    default:
      // end of messages, so repeat
      FrameSubTask = 0; FrameTask = FrameNext; break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Volcano() {
  // Simulates a volcano effect
  // Use the 'F***.' serial commands to tune this animation
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Volcano");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      LED_Grn = 0; LED_Blu = 0; // turn off ink colours
      FrameBright = 18;          // brightness factor
      FrameDim = 3;             // dim increment
      FramePeriod = 150;        // slow frame rate down
      FrameSw = 0;              // initialise the switch pointer
      LED_Task++; break;
    case 1:
      // Dim down all side faces
      Gfx_DimFaceTo(1,FrameDim,1,0,0);  // dim down face LEDs towards dim red
      Gfx_DimFaceTo(2,FrameDim,1,0,0);  // dim down face LEDs towards dim red
      Gfx_DimFaceTo(3,FrameDim,1,0,0);  // dim down face LEDs towards dim red
      Gfx_DimFaceTo(4,FrameDim,1,0,0);  // dim down face LEDs towards dim red
      // Move the side faces down
      for (int16_t zF = 1;zF < 5; zF++) {Gfx_MoveDwn(zF);}
      // Copy down top colour, to side face top lines
      Gfx_SetFace(0); Gfx_PicXY(0,0);
      Gfx_SetFace(1); Gfx_Line(0,0,7,0);
      Gfx_SetFace(2); Gfx_Line(0,0,7,0);
      Gfx_SetFace(3); Gfx_Line(0,0,7,0);
      Gfx_SetFace(4); Gfx_Line(0,0,7,0);
      // Move the top face centre out
      Gfx_SetFace(0);
      Gfx_PicXY(1,1); Gfx_Box(0,0,8,8);
      Gfx_PicXY(2,2); Gfx_Box(1,1,6,6);
      Gfx_PicXY(3,3); Gfx_Box(2,2,4,4);
      // Draw the centre on the top face
      switch(FrameSw) {
        case 0: // Set the peak value and ramp rate
          FramePk = 1 + random(FrameBright+1);
          LED_Red = FramePk;
          if (FramePk >= (FrameBright/2)) LED_Grn = map(FramePk,FrameBright/2,FrameBright,1,FrameBright);
          else {LED_Grn = 0;}
          Gfx_Box(3,3,2,2);
          break;
      }
      FrameSw++; if (FrameSw > 1) {FrameSw = 0;}
      // Now display this
      LEDshow = true; FrameBuild = false;
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_VuFace() {
  // Presents an animated face using Gfx characters, that responds to sound levels.
  if (Exec > 0) {Exec = 1; ExecTimer = millis() - 100; return;} // this task does not run under Exec
  
  switch(LED_Task) {
    case 0:
      Serial.println("Frame_VuFace");
      Gfx_SetFace(0); LED_Flag = true;
      FramePeriod = 40;   // slow frame rate down to 25 Hz
      LED_Task++; break;
    case 1:
      // Serial.println("0," + String(micPkTrk) + "," + String(micMax) + "," + String(micLvl));
      LED_Blu = 0; LED_Grn = Col14;
      micVU = map(micVol,micLvl,micMax,10,100);
      if (micVU < 10) {micVU = 0;}
           if (micVU >= 58) {LED_Grn = 0; LED_Red = Col11; loadGfx(8);}
      else if (micVU >= 54) {LED_Grn = 0; LED_Red = Col11; loadGfx(7);}
      else if (micVU >= 50) {LED_Red = Col11; loadGfx(6);}
      else if (micVU >= 46) {LED_Red = Col34; loadGfx(5);}
      else if (micVU >= 42) {LED_Red = Col34; loadGfx(4);}
      else if (micVU >= 38) {LED_Red = Col34; loadGfx(3);}
      else if (micVU >= 34) {LED_Red = Col12; loadGfx(2);}
      else if (micVU >= 30) {LED_Red = Col12; loadGfx(1);}
      else    {LED_Red = Col12; loadGfx(0);}
      Gfx_DrawGfx(0,0,0);
      // put in the eyes
      LED_Red = 0; LED_Grn = 0; LED_Blu = 6;
      if (micVU < 34) {
        if (random(50) == 0) {LED_Flag = !LED_Flag;}
        if (LED_Flag) {Gfx_SetXY(1,1); Gfx_SetXY(5,1);} else {Gfx_SetXY(2,1); Gfx_SetXY(6,1);}
      }
      else if (micVU < 38) {Gfx_SetXY(1,1); Gfx_SetXY(6,1);}
      
      // copy the faces
      Gfx_CopyFace(0,1); Gfx_CopyFace(0,2); Gfx_CopyFace(0,3); Gfx_CopyFace(0,4);
      LEDshow = true; FrameBuild = false;
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_VuMeter() {
  // Presents a scrolling sound level display, which starts on the left of face 2 and
  // scrolls round to face 1. On the top face we simulate a speaker cone, with box
  // patterns coming from the centre.
  switch(LED_Task) {
    case 0:
      Serial.println("Frame_VuMeter");
      LED_Blu = 0; LED_Grn = Col11; LED_Red = 0;
      FramePeriod = 40;   // slow frame rate down to 25 Hz
      LED_Flag = false; LED_Task++; break;
    case 1:
      // Serial.println("0," + String(micPkTrk) + "," + String(micMax) + "," + String(micLvl));
      Gfx_RotLft(2); Gfx_CopyCols(1,0,2,7); // rotate face 2 left by 1 pixel, then copy in column
      Gfx_RotLft(1); Gfx_CopyCols(4,0,1,7); // rotate face 1 left by 1 pixel, then copy in column
      Gfx_RotLft(4); Gfx_CopyCols(3,0,4,7); // rotate face 2 left by 1 pixel, then copy in column
      Gfx_SetFace(3);
      Gfx_RotLft(3);        // rotate face 3 left by 1 pixel
      Gfx_ClrBar(7,0,1,8);  // clear the right most column
      Gfx_VU(7);            // draw the VU bar
      
      Gfx_SetFace(0); Gfx_VUBox();
      LEDshow = true; FrameBuild = false;
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_VuSpin() {
  // Presents a rotating patern that spins faster with sound levels.
  switch(LED_Task) {
    case 0:
      Serial.println("Frame_Spin");
      Gfx_SetFace(0); LED_Blu = 0; LED_Grn = Col11; LED_Red = 0;
      LED_Pnt = 0; LED_Task++; break;
    case 1:
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      // set colours and speed
      if (micVol == 0) {micVU = 0; LED_Red = 0; LED_Grn = 0; LED_Blu = 2; FramePeriod = 60;}
      else {micVU = map(micVol,micLvl,micMax,10,50); LED_Blu = 0;} 
      if (micVU > 10) {LED_Red =     0; LED_Grn = Col11; FramePeriod = 39;}
      if (micVU > 20) {LED_Red = Col14; LED_Grn = Col34; FramePeriod = 36;}
      if (micVU > 30) {LED_Red = Col12; LED_Grn = Col12; FramePeriod = 33;}
      if (micVU > 40) {LED_Red = Col34; LED_Grn = Col12; FramePeriod = 30;}
      if (micVU > 50) {LED_Red = Col11; LED_Grn = Col14; FramePeriod = 27;}
      if (micVU > 60) {LED_Red = Col11 + Col14; LED_Grn = Col14; FramePeriod = 24;}
      if (micVU > 70) {LED_Red = Col11 + Col12; LED_Grn =     0; FramePeriod = 21;}
      if (micVU > 80) {LED_Red = Col11 + Col11; LED_Grn =     0; FramePeriod = 18;}
      Gfx_SetFace(0);
      // draw top face line
      switch(LED_Pnt) {
        case 0: Gfx_Line(3,3,0,0); Gfx_Line(4,3,7,0); Gfx_Line(4,4,7,7); Gfx_Line(3,4,0,7); break;
        case 1: Gfx_Line(3,3,1,0); Gfx_Line(4,3,7,1); Gfx_Line(4,4,6,7); Gfx_Line(3,4,0,6); break;
        case 2: Gfx_Line(3,3,2,0); Gfx_Line(4,3,7,2); Gfx_Line(4,4,5,7); Gfx_Line(3,4,0,5); break;
        case 3: Gfx_Line(3,3,3,0); Gfx_Line(4,3,7,3); Gfx_Line(4,4,4,7); Gfx_Line(3,4,0,4); break;
        case 4: Gfx_Line(4,3,4,0); Gfx_Line(4,4,7,4); Gfx_Line(3,4,3,7); Gfx_Line(3,3,0,3); break;
        case 5: Gfx_Line(4,3,5,0); Gfx_Line(4,4,7,5); Gfx_Line(3,4,2,7); Gfx_Line(3,3,0,2); break;
        case 6: Gfx_Line(4,3,6,0); Gfx_Line(4,4,7,6); Gfx_Line(3,4,1,7); Gfx_Line(3,3,0,1); break;
        case 7: Gfx_Line(4,3,7,0); Gfx_Line(4,4,7,7); Gfx_Line(3,4,0,7); Gfx_Line(3,3,0,0); break;
      }
      // draw vertical line on face 1
      LED_Cnt = 7 - LED_Pnt; Gfx_SetFace(1); Gfx_Line(LED_Cnt,0,LED_Cnt,7);
      // copy to all side faces
      Gfx_CopyFace(1,2); Gfx_CopyFace(1,3); Gfx_CopyFace(1,4);
      LEDshow = true; FrameBuild = false;
      LED_Pnt++; if (LED_Pnt > 7) {LED_Pnt = 0;}
      break;
  }
}

// ---------------------------------------------------------------------------------

void Frame_Worms() {
  // Rampent worms crawling randomly on the cube, for a random distance. At which
  // point dissappear into the face and re-appear somewhere else. If a worm runs into
  // another worm it will also disappear into the cube face.
  // Audio levels affects frame rate and worm speed.
  int16_t zP; uint8_t zX; uint8_t zY;
  switch(LED_Task) {
    case 0:
      // initialise task
      Serial.println("Frame_Worms");
      Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      for (zP = 0; zP < ArrayMax; zP++) {LED_Array[zP] = 0;}  // clear array
      Gfx_SetFace(0); LED_Grn = 0; LED_Red = 0;
      FramePeriod = 100;   // slow frame rate down to 10 Hz
      LED_Pnt = 0;  // worm pointer
      LED_Cnt = 5;  // moves count
      LED_Task++; break;
    case 1:
      // move a worm defined by LED_Pnt
      // array elements are define as:
      // [0]    - direction
      // [1]    - number of moves
      // [2-7]  - worm body points
      // move body components first
      zP = LED_Pnt + 6; // point at the tail
      for (int zBP = 4; zBP >= 0; zBP--) {
        LED_Array[zP + 1] = LED_Array[zP]; zP--;
      }
      // now move the head, or start it off again
      if (LED_Array[LED_Pnt + 2] == 0) {
        // the worms head has not yet been placed on the face, so place it
        GfxMax = 64 + (64 * random8(5));                  // random face
        zX = 1 + random8(6); zY = 1 + random8(6);         // set the X,Y location
        LED_Array[LED_Pnt + 2] = GfxMax - zX - (8 * zY);  // point at the LED
        // create the move direction
        // 1 = up
        // 2 = right
        // 3 = down
        // 4 = left
        LED_Array[LED_Pnt] = random8(1,5);
        // set the number of moves that this worm will take
        LED_Array[LED_Pnt + 1] = random8(5,11);
      } else {
        // the head has already been placed so move it or hide it
        if (LED_Array[LED_Pnt + 1] > 0) {
          LED_Array[LED_Pnt + 1]--;           // decrement the number of moves
          // check to see if move will take worm off the current face
          Gfx_SetFaceXY(LED_Array[LED_Pnt + 2]);
          LED_Flag = false; // flag is set true if we leave the current face
               if ((LED_Array[LED_Pnt] == 1) && (LED_Y == 0)) {LED_Flag = true;}
          else if ((LED_Array[LED_Pnt] == 2) && (LED_X == 7)) {LED_Flag = true;}
          else if ((LED_Array[LED_Pnt] == 3) && (LED_Y == 7)) {LED_Flag = true;}
          else if ((LED_Array[LED_Pnt] == 4) && (LED_X == 0)) {LED_Flag = true;}
          if (!LED_Flag) {
            // still on the same frame so simply move in the direction
            zP = 1; // default move left
                 if (LED_Array[LED_Pnt] == 1) {zP =  8;}  // move up
            else if (LED_Array[LED_Pnt] == 2) {zP = -1;}  // move right
            else if (LED_Array[LED_Pnt] == 3) {zP = -8;}  // move down
            LED_Array[LED_Pnt + 2] += zP;
            // check to ensure target LED[] is unoccupied
            zP = LED_Array[LED_Pnt + 2];
            if ((LED[zP].r == 0) && (LED[zP].g == 0) && (LED[zP].b == 0)) {LED_Array[LED_Pnt + 2] = zP;} else {LED_Array[LED_Pnt + 2] = 0;}
          } else {
            // moving onto a new face so adjust point and move direction
            Gfx_SetFaceXY(LED_Array[LED_Pnt + 2]);
            switch(LED_Face) {
              case 0: // Face 0
                     if (LED_Array[LED_Pnt] == 1) {LED_Array[LED_Pnt] = 3; zP = Gfx_GetPnt(1,7 - LED_X,    0);} // over top edge onto face 1
                else if (LED_Array[LED_Pnt] == 2) {LED_Array[LED_Pnt] = 3; zP = Gfx_GetPnt(2,7 - LED_Y,    0);} // over right edge onto face 2
                else if (LED_Array[LED_Pnt] == 3) {LED_Array[LED_Pnt] = 3; zP = Gfx_GetPnt(3,    LED_X,    0);} // over bottom edge onto face 3
                else if (LED_Array[LED_Pnt] == 4) {LED_Array[LED_Pnt] = 3; zP = Gfx_GetPnt(4,    LED_Y,    0);} // over left edge onto face 4
                break;
              case 1: // Face 1
                     if (LED_Array[LED_Pnt] == 1) {LED_Array[LED_Pnt] = 3; zP = Gfx_GetPnt(0,7 - LED_X,    0);} // over top edge onto face 0
                else if (LED_Array[LED_Pnt] == 2) {LED_Array[LED_Pnt] = 2; zP = Gfx_GetPnt(4,        0,LED_Y);} // over right edge onto face 4
                else if (LED_Array[LED_Pnt] == 3) {LED_Array[LED_Pnt] = 1; zP = Gfx_GetPnt(3,7 - LED_X,    7);} // over bottom edge onto face 3
                else if (LED_Array[LED_Pnt] == 4) {LED_Array[LED_Pnt] = 4; zP = Gfx_GetPnt(2,7 - LED_X,LED_Y);} // over left edge onto face 2
                break;
              case 2: // Face 2
                     if (LED_Array[LED_Pnt] == 1) {LED_Array[LED_Pnt] = 4; zP = Gfx_GetPnt(0,        7,7 - LED_X);} // over top edge onto face 0
                else if (LED_Array[LED_Pnt] == 2) {LED_Array[LED_Pnt] = 2; zP = Gfx_GetPnt(1,        0,    LED_Y);} // over right edge onto face 1
                else if (LED_Array[LED_Pnt] == 3) {LED_Array[LED_Pnt] = 1; zP = Gfx_GetPnt(4,7 - LED_X,        7);} // over bottom edge onto face 4
                else if (LED_Array[LED_Pnt] == 4) {LED_Array[LED_Pnt] = 4; zP = Gfx_GetPnt(3,        7,    LED_Y);} // over left edge onto face 3
                break;
              case 3: // Face 3
                     if (LED_Array[LED_Pnt] == 1) {LED_Array[LED_Pnt] = 1; zP = Gfx_GetPnt(0,    LED_X,    7);} // over top edge onto face 0
                else if (LED_Array[LED_Pnt] == 2) {LED_Array[LED_Pnt] = 2; zP = Gfx_GetPnt(2,        0,LED_Y);} // over right edge onto face 2
                else if (LED_Array[LED_Pnt] == 3) {LED_Array[LED_Pnt] = 1; zP = Gfx_GetPnt(1,7 - LED_X,    7);} // over bottom edge onto face 1
                else if (LED_Array[LED_Pnt] == 4) {LED_Array[LED_Pnt] = 4; zP = Gfx_GetPnt(4,        7,LED_Y);} // over left edge onto face 4
                break;
              case 4: // Face 4
                     if (LED_Array[LED_Pnt] == 1) {LED_Array[LED_Pnt] = 2; zP = Gfx_GetPnt(0,        0,LED_X);} // over top edge onto face 0
                else if (LED_Array[LED_Pnt] == 2) {LED_Array[LED_Pnt] = 2; zP = Gfx_GetPnt(3,        0,LED_Y);} // over right edge onto face 3
                else if (LED_Array[LED_Pnt] == 3) {LED_Array[LED_Pnt] = 1; zP = Gfx_GetPnt(2,7 - LED_X,    7);} // over bottom edge onto face 2
                else if (LED_Array[LED_Pnt] == 4) {LED_Array[LED_Pnt] = 4; zP = Gfx_GetPnt(1,        7,LED_Y);} // over left edge onto face 1
                break;
              default:  // this should not happen, it terminates the move
                zP = 0; break;
            } 
            // test to ensure that LED is off, unoccupied, otherwise terminate run
            if ((LED[zP].r == 0) && (LED[zP].g == 0) && (LED[zP].b == 0)) {LED_Array[LED_Pnt + 2] = zP;} else {LED_Array[LED_Pnt + 2] = 0;}
          }
        } else {LED_Array[LED_Pnt + 2] = 0;}  // terminate move
      }
      // draw a worm
      LED_Pnt+= 2;  // point at the first posiiton value
      if (LED_Array[LED_Pnt] > 0) {LED[LED_Array[LED_Pnt]].setRGB(Col11,Col14,    0);} // head
      LED_Pnt++;
      if (LED_Array[LED_Pnt] > 0) {LED[LED_Array[LED_Pnt]].setRGB(Col34,    0,    0);}
      LED_Pnt++;
      if (LED_Array[LED_Pnt] > 0) {LED[LED_Array[LED_Pnt]].setRGB(Col12,    0,    0);}
      LED_Pnt++;
      if (LED_Array[LED_Pnt] > 0) {LED[LED_Array[LED_Pnt]].setRGB(Col14,    0,    0);}
      LED_Pnt++;
      if (LED_Array[LED_Pnt] > 0) {LED[LED_Array[LED_Pnt]].setRGB(    0,    0,Col14);} // tail end
      LED_Pnt++;
      if (LED_Array[LED_Pnt] > 0) {LED[LED_Array[LED_Pnt]].setRGB( 0,0,0);} // blank
      // move onto the next worm
      // each worm occupies 7 elements of the array
      LED_Pnt++; if (LED_Pnt > 49) {LED_Task++;}
      break;
    case 2:
      // display them and reset pointers
      LEDshow = true; FrameBuild = false;
      // FramePeriod = map(micVol,micLvl,micMax,70,10);
      LED_Pnt = 0;
      LED_Task = 1; break;
  }
}

// ---------------------------------------------------------------------------------
