
/*
 This tab contains a set of functions used in creating effects.
 The co-ordinate X,Y system is 0,0 at top left corner, 7,7 at bottom right
 
 Gfx_Bar(X,Y,W,H)                 - draw a solid bar on the current screen at X,Y, width W, height H
 Gfx_Box(X,Y,W,H)                 - draw a box on the current screen at X,Y, width W, height H
 Gfx_Clear(S,F)                   - sets LEDs S to F to OFF
 Gfx_ClrBar(X,Y,W,H)              - clears an area of a face from X,Y width W, height H
 Gfx_ClrCol(F,C)                  - clears column C (0-7) on face F (0-4)
 Gfx_ClrFace(F)                   - sets LEDs on face F to OFF
 Gfx_ClrXY(X,Y)                   - clears a pixel on the current face to the current colour
 Gfx_CopyColRows(F0,C0,F1,R1)     - copy column C0 on face F0 to row R1 on face F1
 Gfx_CopyColRowsRev(F0,C0,F1,R1)  - copies column zC0 from face zF0 to row zR1 on face zF1 in reverse order
 Gfx_CopyCols(F0,C0,F1,C1)        - copy column C0 on face F0 to column C1 on face F1
 Gfx_CopyFace(F,T)                - copy face F to face T
 Gfx_CopyFaceFH(F,T)              - copy face F to face T, flipped horizontally
 Gfx_CopyFaceFV(F,T)              - copy face F to face T, flipped vertically
 Gfx_CopyFaceRR(F,T)              - copy face F to face T, rotate right
 Gfx_CopyFaceRL(F,T)              - copy face F to face T, rotate left
 Gfx_CopyRowCols(F0,R0,F1,C1)     - copy row R0 on face F0 to column C1 on face F1
 Gfx_CopyRowColsRev(F0,R0,F1,C1)  - copy row R0 on face F0 to column C1 on face F1 in reverse order
 Gfx_CopyRows(F0,R0,F1,R1)        - copy row R0 on face F0 to row R1 on face F1
 Gfx_CopyRowsRev(F0,R0,F1,R1)     - copy row R0 on face F0 to row R1 on face F1 in reverse order
 Gfx_CopyXY(FX,FY,TX,TY)          - copies a single pixel from FX,FY to TX,TY
 Gfx_Dim(S,F,D)                   - dim LEDs in the range S to F by factor D
 Gfx_DimFace(F,D)                 - dim LEDs on face F by factor D
 Gfx_DimFaceTo(F,D,R,G,B)         - dim LEDs on face F by factor D towards target R,G,B
 Gfx_DimTo(S,F,D,R,G,B)           - dim LEDs in the range S to F by factor D towards target R,G,B
 Gfx_DrawChar(F,X,Y,C)            - draws a single character C on face F, at X,Y
 Gfx_DrawGFX(F,X,Y)               - draws a loaded Gfx character on face F, at X,Y
 Gfx_FaceSpiralACI(F)             - rotates face F as an anti-clockwise inward spiral, seeding from X=0,Y=0
 Gfx_FaceSpiralACO(F)             - rotates face zF as an anti-clockwise outward spiral, seeding from X=3,Y=3
 Gfx_FaceSpiralCI(F)              - rotates face F as a clockwise inward spiral, seeding from X=7,Y=0
 Gfx_FaceSpiralCO(F)              - rotates face zF as a clockwise outward spiral, seeding from X=4,Y=3
 Gfx_FaceRotC(F,B)                - rotate band B on face F clockwise, B == 0 outer, B == 3 centre
 Gfx_Fill(S,F,R,G,B)              - sets LEDs S to F to R,G,B colour
 Gfx_FillFace(F,R,G,B)            - sets LEDs on face F to R,G,B colour
 Gfx_FlipGCRL()                   - flips all of the GC array bytes, bit by bit right to left
 Gfx_GetFace(F)                   - sets the pointers for face F but doesn't change current face value
 Gfx_GetFaceNum{P)                - returns the face number for array point P
 Gfx_GetGfxMax(F)                 - returns GfxMax for face F
 Gfx_InvCol(R,G,B)                - set LED colours to the inverse of R,G,B
 Gfx_Line(X0,Y0,X1,Y1)            - draw a line on the current face from X0,Y0 to X1,Y1
 Gfx_MemCol()                     - place the current RGB colours in a memory
 Gfx_MoveDwn(F)                   - moves the contents of face F down one row, leaving top row unaffected
 Gfx_MultCol(M)                   - multiply the ink colours by factor M
 Gfx_PicXY(X,Y)                   - gets a pixel colour from the current face at X,Y
 Gfx_RecCol()                     - recall the colours stored in a memory with Gfx_MemCol()
 Gfx_RndCol(M);                   - sets a random ink colour, max strength M
 Gfx_RotDwn(F)                    - rotate face F one pixel down
 Gfx_RotLft(F)                    - rotate face F one pixel to the left
 Gfx_RotRht(F)                    - rotate face F one pixel to the right
 Gfx_RotUp(F)                     - rotate face F one pixel Up
 Gfx_SetBakXY(X,Y)                - resets a pixel on the current face to the BAK colour
 Gfx_SetCol(C)                    - sets RGB colour variables from a palette based on C
 Gfx_SetFace(F)                   - sets the current face for drawing functions
 Gfx_SetFaceXY{P)                 - sets the face F and X,Y from point P
 Gfx_SetXY(X,Y)                   - sets a pixel on the current face to the current colour
 Gfx_VU(CX)                       - draw a vertical VU bar in column CX on the current face
 Gfx_VUBox()                      - draw a centre out VU box on the current face

*/


// ---------------------------------------------------------------------------------

void Gfx_Bar(int zX,int zY,int zW,int zH) {
  // Draw a solid bar on the current screen at zX,zY, width zW, height zH
  int zX1 = zX + zW - 1; int zY1 = zY + zH - 1;
  // Serial.println(String(zX) + "\t" + String(zY) + "\t" + String(zX1) + "\t" + String(zY1));
  for (int16_t zR = 0;zR < zH; zR++) {
    Gfx_Line(zX,zY,zX1,zY); zY++;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_Box(int zX,int zY,int zW,int zH) {
  // Draw a box on the current screen at zX,zY, width zW, height zH
  int zX1 = zX + zW - 1; int zY1 = zY + zH - 1;
  // Serial.println(String(zX) + "\t" + String(zY) + "\t" + String(zX1) + "\t" + String(zY1));
  Gfx_Line(zX,zY,zX1,zY);
  Gfx_Line(zX,zY,zX,zY1);
  Gfx_Line(zX1,zY,zX1,zY1);
  Gfx_Line(zX,zY1,zX1,zY1);
}

// ---------------------------------------------------------------------------------

void Gfx_Clear(int zS,int zF) {
  // Sets LEDs zS to zF to OFF
  for (int zP = zS;zP <= zF; zP++) {LED[zP].setRGB(0,0,0);}
}

// ---------------------------------------------------------------------------------

void Gfx_ClrBar(int zX,int zY,int zW,int zH) {
  // Clears a region of the current face from X,Y, width W, height H
  zW = zX + zW; zH = zY + zH;
  for (zX = zX;zX < zW;zX++) {
    for (zY = zY;zY < zH;zY++) {Gfx_ClrXY(zX,zY);}
  }
}

// ---------------------------------------------------------------------------------

void Gfx_ClrCol(uint8_t zF,int zC) {
  // Clears column C (0-7) on face F (0-4)
  // zC == 0 is left column, zC == 7 is right column
  zC = Gfx_GetGfxMax(zF) - (8 * zC);  // get the top most LED pointer
  for (int zP = 0; zP <= 7; zP++) {LED[zC].setRGB(0,0,0); zC--;}
}

// ---------------------------------------------------------------------------------

void Gfx_ClrFace(uint8_t zF) {
  // Sets all LEDs on face zF to OFF
  Gfx_GetFace(zF); Gfx_Clear(GfxP0,GfxMax);
}

// ---------------------------------------------------------------------------------

void Gfx_ClrXY(int zX, int zY) {
  // Clears a pixel on the current face
  // GFX co-ordinate system is 0,0 at top left corner
  // Check values are within expected limits
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  LED_PXY = GfxMax - zY - (8 * zX);
  LED[LED_PXY].setRGB(0,0,0);
}

// ---------------------------------------------------------------------------------

void Gfx_CopyColRows(uint8_t zF0,int zC0,uint8_t zF1,int zR1) {
  // Copies column zC0 from face zF0 to row zR1 on face zF1
  // zCn = 0 is left column, zCn = 7 is right column
  // zRn = 0 is top row, zRn = 7 is bottom row
  zC0 = Gfx_GetGfxMax(zF0) - (8 * zC0);
  zR1 = Gfx_GetGfxMax(zF1) - zR1;
  for (int zP = 0; zP <= 7; zP++) {LED[zR1 - (8 * zP)] = LED[zC0]; zC0--;}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyColRowsRev(uint8_t zF0,int zC0,uint8_t zF1,int zR1) {
  // Copies column zC0 from face zF0 to row zR1 on face zF1 in reverse order
  // zCn = 0 is left column, zCn = 7 is right column
  // zRn = 0 is top row, zRn = 7 is bottom row
  zC0 = Gfx_GetGfxMax(zF0) - (8 * zC0);
  zR1 = Gfx_GetGfxMax(zF1) - zR1 - 56;
  for (int zP = 0; zP <= 7; zP++) {LED[zR1 + (8 * zP)] = LED[zC0]; zC0--;}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyCols(uint8_t zF0,int zC0,uint8_t zF1,int zC1) {
  // Copy column C0 on face F0 to column C1 on face F1
  // zCn = 0 is left column, zCn = 7 is right column
  zC0 = Gfx_GetGfxMax(zF0) - (8 * zC0);
  zC1 = Gfx_GetGfxMax(zF1) - (8 * zC1);
  for (int zP = 0; zP <= 7; zP++) {LED[zC1] = LED[zC0]; zC0--; zC1--;}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyFace(uint8_t zF, uint8_t zT) {
  // Copy face F to face T
  int zFF = Gfx_GetGfxP0(zF); int zTT = Gfx_GetGfxP0(zT);
  for (int zP = 0;zP < 64; zP++) {LED[zTT] = LED[zFF]; zTT++; zFF++;}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyFaceFH(uint8_t zF,uint8_t zT) {
  // Copy face zF to face zT, flipped horizontally
  // We copy collumns of data
  int zSF = Gfx_GetGfxMax(zF);      // point at first column on face zF
  int zST = Gfx_GetGfxMax(zT) - 56; // point at last column on face zT
  for (int zX = 0;zX <= 7; zX++) {
    int zFF = zSF;                  // from column flipped start pointer
    int zTF = zST;                  // target column flipped pointer
    for (int zY = 0; zY <= 7; zY++) {
      LED[zTF] = LED[zFF]; zTF--; zFF--;
    }
    zSF-= 8;                        // set next start column
    zST+= 8;                        // set next target column
  }
}

// ---------------------------------------------------------------------------------

void Gfx_CopyFaceFV(uint8_t zF,uint8_t zT) {
  // Copy face zF to face zT, flipped vertically
  // We copy rows of data
  int zSF = Gfx_GetGfxMax(zF);      // point at top row 0,0 on face zF
  int zST = Gfx_GetGfxMax(zT) - 7;  // point at bottom row 0,7 on face zT
  for (int zY = 0;zY <= 7; zY++) {
    int zFF = zSF;                  // new row start pointer
    int zTF = zST;                  // target row flipped pointer
    for (int zX = 0; zX <= 7; zX++) {
      LED[zTF] = LED[zFF]; zTF-= 8; zFF+= 8;
    }
    zSF--;                        // set next from row
    zST++;                        // set next target row
  }
}

// ---------------------------------------------------------------------------------

void Gfx_CopyFaceRL(uint8_t zF, uint8_t zT) {
  // Copy face F to face T, rotated left
  // We copy columns L-R on zF, to rows B-T on zT
  int zSF = Gfx_GetGfxMax(zF);      // point st first column on face zF
  int zST = Gfx_GetGfxMax(zT) - 7;  // point at first row on face zT
  for (int zY = 0;zY <= 7; zY++) {
    int zFF = zSF;                  // new column X,0 start pointer
    int zTR = zST;                  // target rotated 0,Y row pointer
    for (int zX = 0; zX <= 7; zX++) {
      LED[zTR] = LED[zFF]; zTR-= 8; zFF--;
    }
    zSF-= 8;                        // set next from column
    zST++;                          // set next target row
  }
}

// ---------------------------------------------------------------------------------

void Gfx_CopyFaceRR(uint8_t zF, uint8_t zT) {
  // Copy face F to face T, rotated right
  // We copy columns L-R on zF, to rows T-B on zT
  int zSF = Gfx_GetGfxMax(zF);      // point st first column on face zF
  int zST = Gfx_GetGfxMax(zT) - 56; // point at first row on face zT
  for (int zY = 0;zY <= 7; zY++) {
    int zFF = zSF;                  // new column start pointer
    int zTR = zST;                  // target rotated row pointer
    for (int zX = 0; zX <= 7; zX++) {
      LED[zTR] = LED[zFF]; zTR+= 8; zFF--;
    }
    zSF-= 8;                        // set next from column
    zST--;                          // set next target row
  }
}

// ---------------------------------------------------------------------------------

void Gfx_CopyRowCols(uint8_t zF0,int zR0,uint8_t zF1,int zC1) {
  // Copy row zR0 on face zF0 to column zC1 on face zF1
  // zRn = 0 is top row, zRn = 7 is bottom row
  // zCn = 0 is left column, zCn = 7 is right column
  zR0 = Gfx_GetGfxMax(zF0) - zR0;
  zC1 = Gfx_GetGfxMax(zF1) - (8 * zC1);
  for (int zP = 0; zP <= 7; zP++) {LED[zC1] = LED[zR0]; zR0-= 8; zC1--;}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyRowColsRev(uint8_t zF0,int zR0,uint8_t zF1,int zC1) {
  // Copy row zR0 on face zF0 to column zC1 on face zF1, flipped, with Cn7 = Rn0, Cn6 = Rn1, etc
  // zRn = 0 is top row, zRn = 7 is bottom row
  // zCn = 0 is left column, zCn = 7 is right column
  zR0 = Gfx_GetGfxMax(zF0) - zR0;           // row left pixel
  zC1 = Gfx_GetGfxMax(zF1) - (8 * zC1) - 7; // column bottom pixel
  for (int zP = 0; zP <= 7; zP++) {LED[zC1] = LED[zR0]; zR0-= 8; zC1++;}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyRows(uint8_t zF0,int zR0,uint8_t zF1,int zR1) {
  // Copies a row zR0 from face zF0 to row zR1 on face zF1
  // zRn = 0 is top row, zRn = 7 is bottom row
  zR0 = Gfx_GetGfxMax(zF0) - zR0;   // row left pixel on face zF0
  zR1 = Gfx_GetGfxMax(zF1) - zR1;   // row left pixel on face zF1
  for (int zP = 0; zP <= 56; zP+= 8) {LED[zR1 - zP] = LED[zR0 - zP];}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyRowsRev(uint8_t zF0,int zR0,uint8_t zF1,int zR1) {
  // Copies row zR0 from face zF0, to row zR1 on face zF1, reversed
  // zRn = 0 is top row, zRn = 7 is bottom row
  zR0 = Gfx_GetGfxMax(zF0) - zR0;       // row left pixel on face zF0
  zR1 = Gfx_GetGfxMax(zF1) - zR1 - 56;  // row right pixel on face zF1
  for (int zP = 0; zP <= 56; zP+= 8) {LED[zR1 + zP] = LED[zR0 - zP];}
}

// ---------------------------------------------------------------------------------

void Gfx_CopyXY(int zFX,int zFY,int zTX,int zTY) {
  // Copies a single pixel from zFX,zFY to zTX,zTY on the current face
  if ((zFX < 0) || (zFX > 7) || (zFY < 0) || (zFY > 7)) {return;}
  if ((zTX < 0) || (zTX > 7) || (zTY < 0) || (zTY > 7)) {return;}
  
  zFX = GfxMax - zFY - (8 * zFX);     // put the source pixel number in zFX
  zTX = GfxMax - zTY - (8 * zTX);     // put the target pixel number in zTX
  LED[zTX] = LED[zFX];
}

// ---------------------------------------------------------------------------------

void Gfx_Dim(int zS,int zF,uint8_t zD) {
  // Reduces the brightness of each LED in the range zS to zF by zD
  for (int zP = zS;zP <= zF; zP++) {
    if (LED[zP].r > zD) {LED[zP].r -= zD;} else {LED[zP].r = 0;}
    if (LED[zP].g > zD) {LED[zP].g -= zD;} else {LED[zP].g = 0;}
    if (LED[zP].b > zD) {LED[zP].b -= zD;} else {LED[zP].b = 0;}
  }
}

// ---------------------------------------------------------------------------------

void Gfx_DimFace(int zF,uint8_t zD) {
  // Dim LEDs on face zF by factor zD
  Gfx_GetFace(zF); Gfx_Dim(GfxP0,GfxMax,zD);
}

// ---------------------------------------------------------------------------------

void Gfx_DimFaceTo(int zF,uint8_t zD,uint8_t zR,uint8_t zG,uint8_t zB) {
  // Dim LEDs on face zF by factor zDtowards target R,G,B
  Gfx_GetFace(zF); Gfx_DimTo(GfxP0,GfxMax,zD,zR,zG,zB);
}

// ---------------------------------------------------------------------------------

void Gfx_DimTo(int zS,int zF,uint8_t zD,uint8_t zR,uint8_t zG,uint8_t zB) {
  // Dim LEDs in the range S to F by factor D towards target R,G,B
  // As target can be brighter, hence dim up/down
  for (int zP = zS;zP <= zF; zP++) {
    if (LED[zP].r > zR) {
      if (zD < (LED[zP].r - zR)) {LED[zP].r -= zD;} else {LED[zP].r = zR;}
    } else if (LED[zP].r < zR) {
      if (zD < (zR - LED[zP].r)) {LED[zP].r += zD;} else {LED[zP].r = zR;}
    }
    if (LED[zP].g > zG) {
      if (zD < (LED[zP].g - zG)) {LED[zP].g -= zD;} else {LED[zP].g = zG;}
    } else if (LED[zP].g < zG) {
      if (zD < (zG - LED[zP].g)) {LED[zP].g += zD;} else {LED[zP].g = zG;}
    }
    if (LED[zP].r > zB) {
      if (zD < (LED[zP].b - zB)) {LED[zP].b -= zD;} else {LED[zP].b = zB;}
    } else if (LED[zP].b < zB) {
      if (zD < (zB - LED[zP].b)) {LED[zP].b += zD;} else {LED[zP].b = zB;}
    }
  }
}

// ---------------------------------------------------------------------------------

void Gfx_DrawChar(uint8_t zF,int zX,int zY,uint8_t zC,uint8_t zM) {
  // Loads and draws a single character C on face F, at X,Y
  // You would normally clear the face before doing this
  // zM is a modifying mask
  // XXXXXX00 - don't modify, left-justify from zX
  // XXXXXX11 - centre on zX
  // XXXXXX01 - right-justify from zX
  // XXXX00XX - don't reset non-set pixels
  // XXXX01XX - reset non-set pixels to OFF
  // XXXX10XX - set non-set pixels to background colour BAK_Red, etc
  // this code sets and resets pixels
  loadChar(zC);       // put character font into array
  Gfx_SetFace(zF);    // set face and limits
  if ((zM & 0b00000011) > 0) {
    // perform character justification, by changing zX
         if ((zM & 0b00000011) == 0b00000011) {zX = zX - (GW/2);}  // centre on zX
    else if ((zM & 0b00000001) == 0b00000001) {zX = zX - GW + 1;}  // right justify from zX
  }
  for (int zA = 0;zA <= 7;zA++) {
    // go through each row of the character array
    // GW is the width of the current character
    if ((zM & 0b00001100) == 0) {
      // set pixels, but not background, character is transparent
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);}}
    } else if ((zM & 0b00001100) == 0b00000100) {
      // set pixels and clear non-set pixels, removing background
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {Gfx_ClrXY(zX,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {Gfx_ClrXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {Gfx_ClrXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {Gfx_ClrXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {Gfx_ClrXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {Gfx_ClrXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {Gfx_ClrXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {Gfx_ClrXY(zX+7,zY);}}
    } else if ((zM & 0b00001100) == 0b00001000) {
      // set char pixels, and set non-pixels to background
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {Gfx_SetBakXY(zX,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {Gfx_SetBakXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {Gfx_SetBakXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {Gfx_SetBakXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {Gfx_SetBakXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {Gfx_SetBakXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {Gfx_SetBakXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {Gfx_SetBakXY(zX+7,zY);}}
    }
    zY++;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_DrawGfx(uint8_t zF,int zX,int zY) {
  // Draws a loaded Gfx character on face F, at X,Y
  // Note this sets and resets all pixels
  Gfx_GetFace(zF);    // get face limits
  for (int zA = 0;zA <= 7;zA++) {
    // go through each row of the character array
    if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {Gfx_ClrXY(zX,zY);}
    if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {Gfx_ClrXY(zX+1,zY);}
    if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {Gfx_ClrXY(zX+2,zY);}
    if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {Gfx_ClrXY(zX+3,zY);}
    if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {Gfx_ClrXY(zX+4,zY);}
    if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {Gfx_ClrXY(zX+5,zY);}
    if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {Gfx_ClrXY(zX+6,zY);}
    if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {Gfx_ClrXY(zX+7,zY);}
    zY++;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_FaceRotC(int16_t zF,int16_t zB) {
  // Rotate band B on face F clockwise, B == 0 outer, B == 3 centre
  Gfx_GetFace(zF);    // Get face limits
  switch(zB) {
    case 0: // Outer band 0,0 to 7,7
      LEDX[0] = LED[GfxMax];  // grab the LED at 0,0
      zF = GfxMax--; zB = GfxMax;
      for (int zA = 0;zA < 7;zA++) {LED[zB] = LED[zF]; zF--; zB--;}     // scroll up the LH column
      zF = zB - 8;
      for (int zA = 0;zA < 7;zA++) {LED[zB] = LED[zF]; zF-= 8; zB-= 8;} // scroll bottom row left
      zF = zB + 1;
      for (int zA = 0;zA < 7;zA++) {LED[zB] = LED[zF]; zF++; zB++;}     // scroll down RH column
      zF = zB + 8;
      for (int zA = 0;zA < 7;zA++) {LED[zB] = LED[zF]; zF+= 8; zB+= 8;} // scroll top row right
      zB = GfxMax - 8; LED[zB] = LEDX[0]; // replace LED 1,0
      break;
    case 1: // band 1,1
      zB = GfxMax - 9;
      LEDX[0] = LED[zB];  // grab the LED at 1,1
      zF = zB - 1;
      for (int zA = 0;zA < 5;zA++) {LED[zB] = LED[zF]; zF--; zB--;}     // scroll up the LH column
      zF = zB - 8;
      for (int zA = 0;zA < 5;zA++) {LED[zB] = LED[zF]; zF-= 8; zB-= 8;} // scroll bottom row left
      zF = zB + 1;
      for (int zA = 0;zA < 5;zA++) {LED[zB] = LED[zF]; zF++; zB++;}     // scroll down RH column
      zF = zB + 8;
      for (int zA = 0;zA < 5;zA++) {LED[zB] = LED[zF]; zF+= 8; zB+= 8;} // scroll top row right
      zB = GfxMax - 17; LED[zB] = LEDX[0]; // replace LED 2,1
      break;
    case 2: // band 2,2
      zB = GfxMax - 18;
      LEDX[0] = LED[zB];  // grab the LED at 2,2
      zF = zB - 1;
      for (int zA = 0;zA < 3;zA++) {LED[zB] = LED[zF]; zF--; zB--;}     // scroll up the LH column
      zF = zB - 8;
      for (int zA = 0;zA < 3;zA++) {LED[zB] = LED[zF]; zF-= 8; zB-= 8;} // scroll bottom row left
      zF = zB + 1;
      for (int zA = 0;zA < 3;zA++) {LED[zB] = LED[zF]; zF++; zB++;}     // scroll down RH column
      zF = zB + 8;
      for (int zA = 0;zA < 3;zA++) {LED[zB] = LED[zF]; zF+= 8; zB+= 8;} // scroll top row right
      zB = GfxMax - 26; LED[zB] = LEDX[0]; // replace LED 3,2
      break;
    case 3: // band 3,3
      zB = GfxMax - 27;
      LEDX[0] = LED[zB];      // grab the LED at 3,3
      LED[zB] = LED[zB - 1]; LED[zB - 1] = LED[zB - 9]; LED[zB - 9] = LED[zB - 8];
      LED[zB - 8] = LEDX[0];  // replace LED 4,3
      break;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_FaceSpiralACI(int16_t zF) {
  // Rotates face zF as an anti-clockwise inward spiral, seeding from X=0,Y=0
  // Call this function, then seed a colour at X=0,Y=0
  Gfx_GetFace(zF);    // get face limits
  // Start at the centre
  int16_t zT = GfxMax - 35; zF = zT - 1; LED[zT] = LED[zF]; // Col 4,3 - 4,4
  zT = zF; zF+= 8; LED[zT] = LED[zF];                       // Row 3,4 - 4,4
  // Col 3,2 - 3,4 Down
  zT = zF; zF++; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 3,2 - 5,2 Left
  zF = zT - 8; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 5,2 - 5,5 Up
  zF = zT - 1; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 2,5 - 5,5 Right
  zF = zT + 8; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 2,1 - 2,5 Down
  zF = zT + 1; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 2,1 - 6,1 Left
  zF = zT - 8; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 6,1 - 6,6 Up
  zF = zT - 1; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 1,6 - 6,6 Right
  zF = zT + 8; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 1,0 - 1,6 Down
  zF = zT + 1; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 1,0 - 7,0 Left
  zF = zT - 8; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 7,0 - 7,7 Up
  zF = zT - 1; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 0,7 - 7,7 Right
  zF = zT + 8; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 0,0 - 0,7 Down
  zF = zT + 1; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
}

// ---------------------------------------------------------------------------------

void Gfx_FaceSpiralACO(int16_t zF) {
  // Rotates face zF as an anti-clockwise outward spiral, seeding from X=3,Y=3
  // call this function, then seed a colour at X=3,Y=3
  Gfx_GetFace(zF);    // get face limits
  // Start at the top right corner
  int16_t zT = GfxMax - 56; zF = zT - 1;
  // Col 7,0 - 7,7 Up
  for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 0,7 - 7,7 Right
  zF = zT + 8; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 0,0 - 0,7 Down
  zF = zT + 1; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 0,0 - 6,0 Left
  zF = zT - 8; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 6,0 - 6,6 Up
  zF = zT - 1; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 1,6 - 6,6 Right
  zF = zT + 8; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 1,1 - 1,6 Down
  zF = zT + 1; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 1,1 - 5,1 Left
  zF = zT - 8; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 5,1 - 5,5 Up
  zF = zT - 1; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 2,5 - 5,5 Right
  zF = zT + 8; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 2,2 - 2,5 Down
  zF = zT + 1; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 2,2 - 4,2 Left
  zF = zT - 8; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 4,2 - 4,4 Up
  zF = zT - 1; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 3,4 - 4,4 Right
  zF = zT + 8; LED[zT] = LED[zF];
  // Col 3,3 - 3,4 Down
  zT = zF; zF++; LED[zT] = LED[zF];
}

// ---------------------------------------------------------------------------------

void Gfx_FaceSpiralCI(int16_t zF) {
  // Rotates face zF as a clockwise inward spiral, seeding from X=7,Y=0
  // Call this function, then seed a colour at X=7,Y=0
  Gfx_GetFace(zF);    // get face limits
  // Start at the centre
  int16_t zT = GfxMax - 27; zF = zT - 1; LED[zT] = LED[zF]; // Col 3,3 - 3,4 Up
  zT = zF; zF-= 8; LED[zT] = LED[zF];                       // Row 3,4 - 4,4 Left
  // Col 4,2 - 4,4 Down
  zT = zF; zF++; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 2,2 - 4,2 Right
  zF = zT + 8; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 2,2 - 2,5 Up
  zF = zT - 1; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 2,5 - 5,5 Left
  zF = zT - 8; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 5,1 - 5,5 Down
  zF = zT + 1; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 1,1 - 5,1 Right
  zF = zT + 8; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 1,1 - 1,6 Up
  zF = zT - 1; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 1,6 - 6,6 Left
  zF = zT - 8; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 6,0 - 6,6 Down
  zF = zT + 1; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 0,0 - 6,0 Right
  zF = zT + 8; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 0,7 - 0,0 Up
  zF = zT - 1; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 0,7 - 7,7 Left
  zF = zT - 8; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 7,0 - 7,7
  zF = zT + 1; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
}

// ---------------------------------------------------------------------------------

void Gfx_FaceSpiralCO(int16_t zF) {
  // Rotates face zF as a clockwise outward spiral, seeding from X=4,Y=3
  // Call this function, then seed a colour at X=4,Y=3
  Gfx_GetFace(zF);    // get face limits
  // Start at the top left corner
  int16_t zT = GfxMax; zF = zT - 1;
  // Col 0,7 - 0,0 Up
  for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 0,7 - 7,7 Left
  zF = zT - 8; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 7,0 - 7,7 Down
  zF = zT + 1; for (int zA = 0;zA < 7;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 1,0 - 7,0 Right
  zF = zT + 8; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 1,0 - 1,6 Up
  zF = zT - 1; for (int zA = 0;zA < 6;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 1,6 - 6,6 Left
  zF = zT - 8; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 6,1 - 6,6 Down
  zF = zT + 1; for (int zA = 0;zA < 5;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 2,1 - 6,1 Right
  zF = zT + 8; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 2,1 - 2,5 Up
  zF = zT - 1; for (int zA = 0;zA < 4;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 2,5 - 5,5 Left
  zF = zT - 8; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT-= 8; zF-= 8;}
  // Col 5,2 - 5,5 Down
  zF = zT + 1; for (int zA = 0;zA < 3;zA++) {LED[zT] = LED[zF]; zT++; zF++;}
  // Row 3,2 - 3,5 Right
  zF = zT + 8; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT+= 8; zF+= 8;}
  // Col 3,2 - 3,4 Up
  zF = zT - 1; for (int zA = 0;zA < 2;zA++) {LED[zT] = LED[zF]; zT--; zF--;}
  // Row 3,4 - 4,4 Left
  zF = zT - 8; LED[zT] = LED[zF];
  // Column 4,3 - 4,4
  zT = zF; zF++; LED[zT] = LED[zF];
}

// ---------------------------------------------------------------------------------

void Gfx_Fill(int zS,int zF,uint8_t zR,uint8_t zG,uint8_t zB) {
  // Sets LEDs S to F to R,G,B colour
  if ((zS < 0) || (zS > zF) || (zS > NumLEDsM1) || (zF > NumLEDsM1)) {return;}

  for (int zP = zS;zP <= zF; zP++) {LED[zP].setRGB(zR,zG,zB);}
}

// ---------------------------------------------------------------------------------

void Gfx_FillFace(int zF,uint8_t zR,uint8_t zG,uint8_t zB) {
  // Sets all LEDs on face zF to zR,zG,zB colours
  Gfx_GetFace(zF);
  for (int zP = GfxP0;zP <= GfxMax; zP++) {LED[zP].setRGB(zR,zG,zB);}
}

// ---------------------------------------------------------------------------------

void Gfx_FlipGCRL() {
  // Flips all of the GC array bytes, bit by bit right to left
  uint8_t zFB;  // temp flip byte
  for (int zA = 0;zA <= 7;zA++) {
    zFB = 0;
    if (GC[zA] & 0b10000000) {zFB+= 0b00000001;}
    if (GC[zA] & 0b01000000) {zFB+= 0b00000010;}
    if (GC[zA] & 0b00100000) {zFB+= 0b00000100;}
    if (GC[zA] & 0b00010000) {zFB+= 0b00001000;}
    if (GC[zA] & 0b00001000) {zFB+= 0b00010000;}
    if (GC[zA] & 0b00000100) {zFB+= 0b00100000;}
    if (GC[zA] & 0b00000010) {zFB+= 0b01000000;}
    if (GC[zA] & 0b00000001) {zFB+= 0b10000000;}
    GC[zA] = zFB;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_GetFace(uint8_t zF) {
  // Sets pointers for face zF, but doesn't change the face reference
  switch(zF) {
    case 0: GfxP0 =   0; GfxMax =  63; break;  // top face
    case 1: GfxP0 =  64; GfxMax = 127; break;  // rear face
    case 2: GfxP0 = 128; GfxMax = 191; break;  // right face
    case 3: GfxP0 = 192; GfxMax = 255; break;  // front face
    case 4: GfxP0 = 256; GfxMax = 319; break;  // left face
  }
}

// ---------------------------------------------------------------------------------

uint8_t Gfx_GetFaceNum(int16_t zP) {
  // Returns the face number for LED array point zP
       if (zP <  64) {return 0;}
  else if (zP < 128) {return 1;}
  else if (zP < 192) {return 2;}
  else if (zP < 256) {return 3;}
  return 4;
}

// ---------------------------------------------------------------------------------

int Gfx_GetGfxMax(uint8_t zF) {
  // Returns the array reference for top left corner of face zF 
  int zGfxMax = 63; // default face 0
  switch(zF) {
    case 0: zGfxMax =  63; break;  // top face
    case 1: zGfxMax = 127; break;  // rear face
    case 2: zGfxMax = 191; break;  // right face
    case 3: zGfxMax = 255; break;  // front face
    case 4: zGfxMax = 319; break;  // left face
  } return zGfxMax;
}

// ---------------------------------------------------------------------------------

int Gfx_GetGfxP0(uint8_t zF) {
  // Returns the array reference for bottom right corner of face zF 
  int zGfxP0 = 0; // default face 0
  switch(zF) {
    case 0: zGfxP0 =   0; break;  // top face
    case 1: zGfxP0 =  64; break;  // rear face
    case 2: zGfxP0 = 128; break;  // right face
    case 3: zGfxP0 = 192; break;  // front face
    case 4: zGfxP0 = 256; break;  // left face
  } return zGfxP0;
}

// ---------------------------------------------------------------------------------

int16_t Gfx_GetPnt(int16_t zF,int16_t zX,int16_t zY) {
  // Returns an array pointer to zX,zY on face zF
  // GFX assumes 0,0 is top left of face
  // zY increases top to bottom, but LED Y increases bottom to top +1
  // zX increases left to rignt, but LED X increases right to left +8
  zF = (zF * 64) + (7 - zY) + ((7 - zX) * 8);
  return zF;
}

// ---------------------------------------------------------------------------------

void Gfx_InvCol(uint8_t zR,uint8_t zG,uint8_t zB) {
  // set LED colours to the inverse of zR,zG,zB based on LED_Max
  LED_Red = LED_Max - zR; LED_Grn = LED_Max - zG; LED_Blu = LED_Max - zB;
}

// ---------------------------------------------------------------------------------

void Gfx_Line(int zX0,int zY0,int zX1,int zY1) {
  // Draw a line on the current face from zX0,zY0 to zX1,zY1
  int zX; int zY; int zR2;
  if ((abs(zX1 - zX0)== 0) && (abs(zY1 - zY0) == 0)) {
    // Single point
    if ((zX >= 0) && (zX <= 7) && (zY >= 0) && (zY <= 7)) {
      LED_PXY = GfxMax - zY - (8 * zX);
      LED[LED_PXY].setRGB(LED_Red,LED_Grn,LED_Blu);
    }
  } else if (abs(zX1 - zX0) > abs(zY1 - zY0)) {
    // Line is longer in X than Y
    // Ensure zX1 > zX0
    if (zX1 < zX0) {zX = zX1; zX1 = zX0; zX0 = zX; zY = zY1; zY1 = zY0; zY0 = zY;} // ensure zY1 is largest
    zR2 = (zX1 - zX0)/2; if (zY1 < zY0) {zR2 = -zR2;}
    for (zX = zX0;zX <= zX1;zX++) {
      if ((zX >= 0) && (zX <= 7)) {
        // only set point if within face area
        zY = zY0 + (((zY1 - zY0) * (zX - zX0)) + zR2)/(zX1 - zX0);
        if ((zY >= 0) && (zY <= 7)) {
          LED_PXY = GfxMax - zY - (8 * zX);
          LED[LED_PXY].setRGB(LED_Red,LED_Grn,LED_Blu);
        }
      }
    }
  } else {
    // line is longer in Y than X
    // ensure zY1 > zY0
    if (zY1 < zY0) {zY = zY1; zY1 = zY0; zY0 = zY; zX = zX1; zX1 = zX0; zX0 = zX;} // ensure zX1 is largest
    zR2 = (zY1 - zY0)/2; if (zX1 < zX0) {zR2 = -zR2;}
    for (zY = zY0;zY <= zY1;zY++) {
      if ((zY >= 0) && (zY <= 7)) {
        // only set point if within face area
        zX = zX0 + (((zX1 - zX0) * (zY - zY0))/(zY1 - zY0));
        if ((zX >= 0) && (zX <= 7)) {
          LED_PXY = GfxMax - zY - (8 * zX);
          LED[LED_PXY].setRGB(LED_Red,LED_Grn,LED_Blu);
        }
      }
    }
  }
}

// ---------------------------------------------------------------------------------

void Gfx_MemCol() {
  // place the current RGB colours in a memory
  Mem_Red = LED_Red; Mem_Grn = LED_Grn; Mem_Blu = LED_Blu;
}

// ---------------------------------------------------------------------------------

void Gfx_MoveDwn(uint8_t zF) {
  // Moves the entire face zF down one line, leaving the top line as it was
  int zGfxP0 = Gfx_GetGfxP0(zF);
  for (int zR = 0;zR < 7;zR++) {
    // Work along each column, right to left, moving pixels down
    for (int zC = 0; zC <= 56; zC+= 8) {
      LED[zGfxP0 + zC] = LED[zGfxP0 + zC + 1];
    }
    zGfxP0++;   // point at the next target row
  }
}

// ---------------------------------------------------------------------------------

void Gfx_MultCol(uint8_t zM) {
  // Multiply the ink colours by factor zM
  LED_Red*= zM; LED_Grn*= zM; LED_Blu*= zM;
}

// ---------------------------------------------------------------------------------

void Gfx_PicXY(int zX,int zY) {
  // Gets a pixel colour from the current face at X,Y
  // Co-ordinate system is 0,0 at top left corner
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  LED_PXY = GfxMax - zY - (8 * zX);
  LED_Red = LED[LED_PXY].r; LED_Grn = LED[LED_PXY].g; LED_Blu = LED[LED_PXY].b;
}

// ---------------------------------------------------------------------------------

void Gfx_RecCol() {
  // recall the colours stored in a memory
  LED_Red = Mem_Red; LED_Grn = Mem_Grn; LED_Blu = Mem_Blu;
}

// ---------------------------------------------------------------------------------

void Gfx_RndCol(int zM) {
  // sets a random ink colour, max strength M
  int z12 = zM>>1; int z14 = zM>>2; int z34 = zM - z14;
  Again:
  uint8_t zR = random8(12);
  if (zR == RndCol) {goto Again;}
  switch(zR) {
    case  0: LED_Red =  zM; LED_Grn =   0; LED_Blu =   0; break;
    case  1: LED_Red = z34; LED_Grn = z14; LED_Blu =   0; break;
    case  2: LED_Red = z12; LED_Grn = z12; LED_Blu =   0; break;
    case  3: LED_Red = z14; LED_Grn = z34; LED_Blu =   0; break;
    case  4: LED_Red =   0; LED_Grn =  zM; LED_Blu =   0; break;
    case  5: LED_Red =   0; LED_Grn = z34; LED_Blu = z14; break;
    case  6: LED_Red =   0; LED_Grn = z12; LED_Blu = z12; break;
    case  7: LED_Red =   0; LED_Grn = z14; LED_Blu = z34; break;
    case  8: LED_Red =   0; LED_Grn =   0; LED_Blu =  zM; break;
    case  9: LED_Red = z14; LED_Grn =   0; LED_Blu = z34; break;
    case 10: LED_Red = z12; LED_Grn =   0; LED_Blu = z12; break;
    case 11: LED_Red = z34; LED_Grn =   0; LED_Blu = z14; break;
  } RndCol = zR;
}

// ---------------------------------------------------------------------------------

void Gfx_RotDwn(int zF) {
  // Rotate face F one row down. Top row becomes bottom row.
  int zGfxMax = Gfx_GetGfxMax(zF);
  for (int zX = 0;zX <= 7; zX++) {
    int zT = zGfxMax - 7; zF = zT + 1;
    LEDX[0] = LED[zT];           // Remember column bottom pixel value
    // Now rotate the column down
    for (int zY = 0;zY < 7;zY++) {
      LED[zT] = LED[zF]; zT++; zF++;
    }
    LED[zT] = LEDX[0];          // Replace column top pixel value
    zGfxMax-= 8;                // Move the column pointer right
  }
}

// ---------------------------------------------------------------------------------

void Gfx_RotLft(int zF) {
  // Rotate face F one column to the left. Right column becomes left column.
  int zGfxP0 = Gfx_GetGfxP0(zF);
  for (int zY = 0;zY <= 7;zY++) {
    int zT = zGfxP0 + 56; zF = zT - 8;
    LEDX[0] = LED[zT];          // Remember row left pixel values
    // Now rotate the row right
    for (int zX = 0;zX < 7; zX++) {
      LED[zT] = LED[zF]; zT-= 8; zF-= 8;
    }
    LED[zT] = LEDX[0];          // Replace row right pixel value
    zGfxP0++;                   // Move the row pointer up
  }
}

// ---------------------------------------------------------------------------------

void Gfx_RotRht(int zF) {
  // Rotate face F one column to the right. Left column becomes right column.
  int zGfxP0 = Gfx_GetGfxP0(zF);
  for (int zY = 0;zY <= 7;zY++) {
    int zT = zGfxP0; zF = zT + 8;
    LEDX[0] = LED[zT];            // Remember row right pixel values
    // Now rotate the row right
    for (int zX = 0;zX < 7; zX++) {
      LED[zT] = LED[zF]; zT+= 8; zF+= 8;
    }
    LED[zT] = LEDX[0];            // Replace row left pixel value
    zGfxP0++;                     // Move the row pointer
  }
}

// ---------------------------------------------------------------------------------

void Gfx_RotUp(int zF) {
  // Rotate face F one row up. Bottom row becomes top row.
  int zGfxMax = Gfx_GetGfxMax(zF);
  for (int zX = 0;zX <= 7; zX++) {
    int zT = zGfxMax; zF = zT - 1;
    LEDX[0] = LED[zT];            // Remember column top pixel values
    // Now rotate the column up
    for (int zY = 0;zY < 7;zY++) {
      LED[zT] = LED[zF]; zT--; zF--;
    }
    LED[zT] = LEDX[0];            // Replace column bottom pixel value
    zGfxMax-= 8;                  // Move the column pointer right
  }
}

// ---------------------------------------------------------------------------------

void Gfx_SetBakXY(int zX,int zY) {
  // Resets a pixel on the current face to the BAK colour
  // Co-ordinate system is 0,0 at top left corner
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  LED_PXY = GfxMax - zY - (8 * zX);
  LED[LED_PXY].setRGB(BAK_Red,BAK_Grn,BAK_Blu);
}

// ---------------------------------------------------------------------------------

void Gfx_SetCol(uint8_t zC) {
  // Sets RGB colour variables from a palette based on zC
  switch(zC) {
    case  0: LED_Red = Col11; LED_Grn =     0; LED_Blu =     0; break; // red
    case  1: LED_Red = Col34; LED_Grn = Col14; LED_Blu =     0; break; // red/yellow
    case  2: LED_Red = Col12; LED_Grn = Col12; LED_Blu =     0; break; // yellow
    case  3: LED_Red = Col14; LED_Grn = Col34; LED_Blu =     0; break; // green/yellow
    case  4: LED_Red =     0; LED_Grn = Col11; LED_Blu =     0; break; // green
    case  5: LED_Red =     0; LED_Grn = Col34; LED_Blu = Col14; break; // green/magenta
    case  6: LED_Red =     0; LED_Grn = Col12; LED_Blu = Col12; break; // magenta
    case  7: LED_Red =     0; LED_Grn = Col14; LED_Blu = Col34; break; // blue/magenta
    case  8: LED_Red =     0; LED_Grn =     0; LED_Blu = Col11; break; // blue
    case  9: LED_Red = Col14; LED_Grn =     0; LED_Blu = Col34; break; // blue/purple
    case 10: LED_Red = Col12; LED_Grn =     0; LED_Blu = Col12; break; // purple
    case 11: LED_Red = Col34; LED_Grn =     0; LED_Blu = Col14; break; // red/purple
    case 12: LED_Red =     0; LED_Grn =     0; LED_Blu =     0; break; // OFF
  }
}

// ---------------------------------------------------------------------------------

void Gfx_SetFace(uint8_t zF) {
  // Sets the current face reference to zF and sets pointers
  LED_Face = zF;
  switch(LED_Face) {
    case 0: GfxP0 =   0; GfxMax =  63; break;  // top face
    case 1: GfxP0 =  64; GfxMax = 127; break;  // rear face
    case 2: GfxP0 = 128; GfxMax = 191; break;  // right face
    case 3: GfxP0 = 192; GfxMax = 255; break;  // front face
    case 4: GfxP0 = 256; GfxMax = 319; break;  // left face
  }
}

// ---------------------------------------------------------------------------------

void Gfx_SetFaceXY(int16_t zP) {
  // Sets the face F and X,Y from LED pointer zP, 0 - 319
  LED_Face = zP/64;                       // returns 0 - 4, for zP = 0 - 319
  zP = zP - (LED_Face * 64);              // gets the magnitude of the face vector, Y + (8 * X)
  LED_X = 7 - (zP/8);                     // gets X
  LED_Y = 7 - (zP - ((7 - LED_X) * 8));   // gets Y
}

// ---------------------------------------------------------------------------------

void Gfx_SetXY(int zX,int zY) {
  // Sets a pixel on the current face to the current colour
  // Co-ordinate system is 0,0 at top left corner
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  LED_PXY = GfxMax - zY - (8 * zX);
  LED[LED_PXY].setRGB(LED_Red,LED_Grn,LED_Blu);
}

// ---------------------------------------------------------------------------------

void Gfx_VU(uint8_t zC) {
  // Draw a vertical VU bar in column zC on the current face
  LED_Red = 0; LED_Grn = Col11; LED_Blu = 0;
  if (micVol == 0) {
    micVU = 0; micVuCnt++;
    if (micVuCnt > 4) {
      micVuCnt = 0; LED_Grn = 0; LED_Blu = 1; Gfx_SetXY(zC,7);
      LED_Grn = Col11; LED_Blu = 0;
    }
  } else {micVU = map(micPkTrk,micLvl,micMax,10,100);} 
  if (micVU > 10) {Gfx_SetXY(zC,7);}
  if (micVU > 20) {LED_Red = Col14; LED_Grn = Col11; Gfx_SetXY(zC,6);}
  if (micVU > 30) {LED_Red = Col14; LED_Grn = Col11; Gfx_SetXY(zC,5);}
  if (micVU > 40) {LED_Red = Col12; LED_Grn = Col34; Gfx_SetXY(zC,4);}
  if (micVU > 50) {LED_Red = Col12; LED_Grn = Col12; Gfx_SetXY(zC,3);}
  if (micVU > 60) {LED_Red = Col34; LED_Grn = Col14; Gfx_SetXY(zC,2);}
  if (micVU > 70) {LED_Red = Col34; LED_Grn = Col14; Gfx_SetXY(zC,1);}
  if (micVU > 80) {LED_Red = Col11; LED_Grn =     0; Gfx_SetXY(zC,0);}
}

// ---------------------------------------------------------------------------------

void Gfx_VUBox() {
  // draw a centre out VU box on the current face
  LED_Red = 0; LED_Grn = Col11; LED_Blu = 0;
  if (micVol == 0) {micVU = 0; micVuCnt++;} else {micVU = map(micPkTrk,micLvl,micMax,10,100);}
  Gfx_ClrFace(LED_Face);
  if ((micVU >= 10) && (micVU < 20)) {
    LED_Red =     0; LED_Grn = Col14; Gfx_Box(3,3,2,2);}
  if ((micVU >= 20) && (micVU < 30)) {
    LED_Red =     0; LED_Grn = Col14; Gfx_Box(3,3,2,2);
    LED_Red =     0; LED_Grn = Col12; Gfx_Box(2,2,4,4);}
  if ((micVU >= 30) && (micVU < 40)) {
    LED_Red =     0; LED_Grn = Col14; Gfx_Box(3,3,2,2);
    LED_Red =     0; LED_Grn = Col14; Gfx_Box(2,2,4,4);
    LED_Red =     0; LED_Grn = Col12; Gfx_Box(1,1,6,6);}
  if ((micVU >= 40) && (micVU < 50)) {
    LED_Red =     0; LED_Grn = Col14; Gfx_Box(3,3,2,2);
    LED_Red =     0; LED_Grn = Col12; Gfx_Box(2,2,4,4);
    LED_Red =     0; LED_Grn = Col34; Gfx_Box(1,1,6,6);
    LED_Red =     0; LED_Grn = Col11; Gfx_Box(0,0,8,8);}
  if ((micVU >= 50) && (micVU < 60)) {
    LED_Red =     0; LED_Grn = Col12; Gfx_Box(3,3,2,2);
    LED_Red =     0; LED_Grn = Col34; Gfx_Box(2,2,4,4);
    LED_Red =     0; LED_Grn = Col11; Gfx_Box(1,1,6,6);
    LED_Red = Col14; LED_Grn = Col34; Gfx_Box(0,0,8,8);}
  if ((micVU >= 60) && (micVU < 70)) {
    LED_Red =     0; LED_Grn = Col34; Gfx_Box(3,3,2,2);
    LED_Red =     0; LED_Grn = Col11; Gfx_Box(2,2,4,4);
    LED_Red = Col14; LED_Grn = Col34; Gfx_Box(1,1,6,6);
    LED_Red = Col12; LED_Grn = Col12; Gfx_Box(0,0,8,8);}
  if ((micVU >= 70) && (micVU < 80)) {
    LED_Red =     0; LED_Grn = Col11; Gfx_Box(3,3,2,2);
    LED_Red = Col14; LED_Grn = Col34; Gfx_Box(2,2,4,4);
    LED_Red = Col12; LED_Grn = Col12; Gfx_Box(1,1,6,6);
    LED_Red = Col34; LED_Grn = Col14; Gfx_Box(0,0,8,8);}
  if (micVU >= 80) {
    LED_Red = Col14; LED_Grn = Col34; Gfx_Box(3,3,2,2);
    LED_Red = Col12; LED_Grn = Col12; Gfx_Box(2,2,4,4);
    LED_Red = Col34; LED_Grn = Col14; Gfx_Box(1,1,6,6);
    LED_Red = Col11; LED_Grn =     0; Gfx_Box(0,0,8,8);}
}

// ---------------------------------------------------------------------------------
