
// --------------------------------------------------------------------------------

void AvgAccelAll() {
  // average the raw accelerometer values over a given number of values
  AcXSum += AcX; AcYSum += AcY; AcZSum += AcZ;
  if (AcAvgDiv < AcAvgDivMax) {
    // increase the scope of the averaging factor if necessary
    // note at 20ms sample, AvgDiv = 50 == 1 second rolling average
    AcAvgDiv++;
  } else {
    // once the accumulator has the correct number of samples roll out the average
    AcXSum -= AcXAvg; AcYSum -= AcYAvg; AcZSum -= AcZAvg;
  }
  AcXAvg = AcXSum / AcAvgDiv;
  AcYAvg = AcYSum / AcAvgDiv;
  AcZAvg = AcZSum / AcAvgDiv;

  // use the following print data to determine offset values, wiht offsets set = 0
  // then add them in the default values
  // Serial.println(String(AcXAvg) + "\t" + String(AcYAvg) + "\t" + String(AcZAvg));
}

// ---------------------------------------------------------------------------------

void AvgGyrAll() {
  // called as a 20ms subtask, keeps a rolling average of the gyroscopes
  GyXSum += GyX; GyYSum += GyY; GyZSum += GyZ;
  if (GyAvgDiv < GyAvgDivMax) {
    // increase the scope of the averaging factor if necessary
    // note at 20ms sample, AvgDiv = 50 == 1 second rolling average
    GyAvgDiv++;
  } else {
    // once the accumulator has the correct number of samples roll out the average
    GyXSum -= GyXAvg; GyYSum -= GyYAvg; GyZSum -= GyZAvg;
  }

  GyXAvg = GyXSum/GyAvgDiv;
  GyYAvg = GyYSum/GyAvgDiv;
  GyZAvg = GyZSum/GyAvgDiv;

  // initially the Gyro is not calibrated, so we run a timer before using its output
  // as an offset value
  if (GyCal > 0) {
    GyCal--; 
    if (GyCal < 1) {
      // calibration period complete so use offsets determined by averaging
      GyXOff = -GyXAvg;
      GyYOff = -GyYAvg;
      GyZOff = -GyZAvg;
      PitchGyr = PitchAcc;  // set pitch gyro using Acc angle
      RotGyr = 0.0;         // zero the rotation gyro
      YawGyr = YawAcc;      // set yaw gyro using Acc angle
    }
  }
  // Serial.println(String(GyXAvg) + "\t" + String(GyYAvg) + "\t" + String(GyZAvg));
}

// ---------------------------------------------------------------------------------

void AudioRead() {
  // Called from the main loop every 10ms to read the MAX4466 microphone input
  if (!micFast) {
  // t0 = micros();
    micRaw = analogRead(MicPin);  // t1-t0 = 67 - 96 us
  // t1 = micros() - t0; Serial.println(t1);
    if ((micRaw - micAvg) > 0) {micAvg+= micInc;} else {micAvg-= micInc;}
    micInc--;
    // Serial.println("0," + String(micAvg));
    if (micInc < 1) {micFast = true;}
    return; // no need to calculate micVol in this phase
  }
  
  // Now in fast sense phase, micPk is measured during main loop.
  // micPk is the peak value of micUni
  int16_t micTh = micLvl + (micLvl/4);  // set the 125% threshold
  if (micVol < micPk) {
    micVol = micPk; // set to peak detector
    // mic volume level is less than the peak of samples
    if (micVol < micTh) {
      micVol = 0; // squelsh
    }
  } else {
    if (micVol > 0) {
      micVol-= (1 + (micVol/16));           // decay
      if (micVol < micLvl) {micVol = 0;}    // squelsh
      micCnt = 0;                           // prevent level adjustment
    }
  }
  // adjust the level threshold
  if (micPk > micLvl) {
    if (micCnt < 0) {micCnt = 0;}
    micCnt++; if (micCnt > 1) {micCnt = 0; micLvl++;}}
  else if (micPk < micLvl) {
    if (micCnt > 0) {micCnt = 0;}
    micCnt--; if (micCnt < -5) {micCnt = 0; micLvl--;}}
  else {micCnt = 0;}
  // Serial.println("0," + String(micPk) + "," + String(micVol) + "," + String(micLvl) + "," + String(micMax));
  // Serial.println("0," + String(micPk) + "," + String(micLvl));

  // track max volume for LED VU scaling
  if (micVol > micMax) {micMax = micVol; micCnt = 10;}  // peak detector
  else {
    if (micMax > micMaxDef) {micMax--;} // reduce the VU peak over time
    if (micVol == 0) {
      // audio has been squelshed to return micMax to default after a delay
      if (micCnt > 0) {micCnt--;}
      else {micMax = micMaxDef;}
    }
  }
  micPkTrk = micPk; micPk = 0;  // reset main loop peak detector
  // Serial.println(micInc);
  micInc = 0; // reset sample counter
}

// ---------------------------------------------------------------------------------

void calcAngles() {
  // called from main loop every 20ms to calculate the yaw, pitch and turn angles
  // PitchAcc +ve tip forward, -ve tip backwards
  // RollAcc +ve tip right, -ve tip left
  // we want to reverse the sign of these values
  PitchAcc = 57.2958 * asin((float)AcX/8192.0);   // angle in radians
  RollAcc = 57.2958 * asin((float)AcZ/8192.0);    // angle in radians
  if (MPUCal) {PitchAcc = 0.0; RollAcc = 0.0;}    // in calibration mode

  // only do these calculations once the calibration process is completed
  if (GyCal > 0) {
    // during calibration gyros are locked to zero
    PitchGyr = 0.0; RollGyr = 0.0; RotGyr = 0.0;
  } else {
    // gyros are set at +/-250 deg/sec for 16-bit ADC = 32,767 FSD
    // to convert gyro value to deg/sec we divide by 131.068 (ie. 32,767/250)
    // we have recorded time period in GyTD in milliseconds, between readings
    // so angle is = GyAng * (GyTD/1000)/131.068
    // GyZ = Pitch, +ve tipping back, -ve tipping forward
    // GyY = Yaw, rotate round
    // GyX = Roll, +ve tipping left, -ve tipping right 
    PitchGyr += (float)GyZ * (float)GyTD / 131068.0;
    // use 5% accelerometer angle difference to correct any drift
    PitchGyr = PitchGyr + ((PitchAcc - PitchGyr) * 0.05);
    // limit the angle
    if (PitchGyr > 360.0) {PitchGyr -= 360.0;}
    else if (PitchGyr < -360.0) {PitchGyr += 360.0;}
    PitchGyrInt = int(PitchGyr);  // grab integer form for faster decision making
  
    RollGyr -= (float)GyX * (float)GyTD / 131068.0;
    // use 5% accelerometer angle difference to correct any drift
    RollGyr = RollGyr + ((RollAcc - RollGyr) * 0.05);
    // limit the angle
    if (RollGyr > 360.0) {RollGyr -= 360.0;}
    else if (RollGyr < -360.0) {RollGyr += 360.0;}
    RollGyrInt = int(RollGyr);  // grab integer form for faster decision making

    // Test for movement.
    // If the cube is stationary for more than 3 seconds, StaticCnt = 99
    // If it is moving then StaticCnt = 0
    int zPG = abs(PitchGyrInt);
    int zRG = abs(RollGyrInt);
    if ((abs(zPG - PitchGyrLast) < 2) && (abs(zRG - RollGyrLast) < 2)) {
      // Movement is less than 2 degrees, on either axis
      StaticCnt++;
      if (StaticCnt >= 150) {
        if (StaticCnt == 150) {Serial.println("Stationary");}
        StaticCnt = 999;
      }
    } else {
      if (StaticCnt == 999) {Serial.println("Moving");}
      StaticCnt = 0;
    }
    PitchGyrLast = zPG; // record latest abs values
    RollGyrLast  = zRG;
  
    // perform long average offset adjustment to improve accuracy
    if (GyCal == 0) {
      if (PitchGyr < PitchAcc) {
        GyZOffCnt--;
        if (GyZOffCnt < -100) {GyZOffCnt = 0; GyZOff--;}
      } else {
        GyZOffCnt++;
        if (GyZOffCnt > 100) {GyZOffCnt = 0; GyZOff++;}
      }
      if (RollGyr < RollAcc) {
        GyXOffCnt--;
        if (GyXOffCnt < -100) {GyXOffCnt = 0; GyXOff++;}
      } else {
        GyXOffCnt++;
        if (GyXOffCnt > 100) {GyXOffCnt = 0; GyXOff--;}
      }
    }
  
    RotDif = (float)GyY * (float)GyTD / 131068.0;
  //  Serial.println(String(fabs(RotDif)) + "\t" + String(RotCnt));
    RotGyr += RotDif;
    // if some rotation is detected then set a timer, if not then
    // ignore the drift by setting the angle to zero
    if (fabs(RotDif) > 0.01) {RotCnt = 150;} else if (RotCnt == 0) {RotGyr = 0.0;}
    // switch the angle to zero after 3 sec time-out
    if (RotCnt > 0) {
      // we are or have been rotating
      RotCnt--; if (RotCnt < 1) {RotGyr = 0.0;} // zero the rotation gyro
    } else {
      // we appear to have stopped or rotation is small
      // in this condition perform long averaging to improve accuracy
      if (GyY > 0) {
        GyYOffCnt++;
        if (GyYOffCnt > 100) {GyYOffCnt = 0; GyYOff--;}
      } else {
        GyYOffCnt--;
        if (GyYOffCnt < -100) {GyYOffCnt = 0; GyYOff++;}
      }
    }
    // limit the angle
    if (RotGyr > 360.0) {RotGyr -= 360.0;}
    else if (RotGyr < -360.0) {RotGyr += 360.0;}
    RotGyrInt = int(RotGyr);
  }

  // Serial.println(GyCal);

  if (MPUrep) {  
    Serial.print(PitchAcc); Serial.print("\t");
    Serial.print(RollAcc); Serial.print("\t");
    Serial.print(PitchGyr); Serial.print("\t");
    Serial.println(RollGyr);
  }
}

// ---------------------------------------------------------------------------------

void decodeKey(int zkeyVal) {
  // Decodes zkeyVal and excutes commands on '.'
  if (zkeyVal == 10) {return;}  // ignore LF
  if (zkeyVal == 13) {return;}  // ignore CR
  keyChar = char(zkeyVal);
  switch (keyChar) {
    case '!': REBOOT = true; setFrameTask(999); return; // REBOOT
    case '.': doCmd(); return;          // command terminator
    case '-': cmdSgn = -1; return;      // numeric sign
    case '*': cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1; return; // ESC abort command
    case '0': extendCmdVal(0); return;
    case '1': extendCmdVal(1); return;
    case '2': extendCmdVal(2); return;
    case '3': extendCmdVal(3); return;
    case '4': extendCmdVal(4); return;
    case '5': extendCmdVal(5); return;
    case '6': extendCmdVal(6); return;
    case '7': extendCmdVal(7); return;
    case '8': extendCmdVal(8); return;
    case '9': extendCmdVal(9); return;
  }
  if (cmdMode == ' ') {
    // test for new Command Mode char?
    // only accept certain characters as valid commands, or ignore them
    cmdMode = keyChar;
    switch (keyChar) {
      // check for valid mode and convert lower-case
      case 'f': cmdMode = 'F'; break;
      case 'm': cmdMode = 'M'; break;
      case 'r': cmdMode = 'R'; break;
      case 'p': cmdMode = 'P'; break;
      case 't': cmdMode = 'T'; break;
    } cmdType = ' '; cmdVal = 0;
  } else {
    // test for Command Type char?
    cmdType = keyChar;
    switch (keyChar) {
      // check for lower-case and convert to upper-case
      case 'b': cmdType = 'B'; break;
      case 'c': cmdType = 'C'; break;
      case 'd': cmdType = 'D'; break;
      case 'l': cmdType = 'L'; break;
      case 'o': cmdType = 'O'; break;
      case 'p': cmdType = 'P'; break;
      case 'r': cmdType = 'R'; break;
      case 'u': cmdType = 'U'; break;
    }
  }
}

// ---------------------------------------------------------------------------------

void doCmd() {
  // a '.' has been received on the serial port so execute a command, if valid
  // Commands:
  // !      - RESET
  // F.     - stops current FrameTask
  // F?.    - lists available FrameTasks
  // FBnn.  - sets brightness for current frame simulation
  // FCnn.  - set FrameCnt variable 0 - 1000
  // FDnn.  - sets FrameDim factor 1 - 64
  // Fnn.   - activates FrameTask nn
  // FLnn.  - set frame period lower limit
  // FPnn.  - set the FramePeriod directly as nn
  // FUnn.  - set frame period upper limit
  // M.     - toggles MPU on/off
  // MO.    - report MPU offsets
  // MRn.   - turns MPU reporting n=1 ON/n=0 OFF
  // P.     - Pause ON/OFF, allows you to change settings, then continue
  // R.     - stops serial port reporting
  // Rnn.   - sets serial port reporting mode
  // R?.    - lists available reports
  // TDnn.  - task delay, after frame show()
  cmdVal*= cmdSgn;  // correct cmdVal with the sign value
  
  switch (cmdMode) {
    case ' ': break;
    case 'F': // Run/stop Frame tasks
      switch (cmdType) {
        case '?': Report = -2; break;
        case ' ': setFrameTask(cmdVal); break;
        case 'B': // sets brightness for current frame simulation
          FrameBright = cmdVal;
          if (FrameBright < 1) {FrameBright = 1;} else if (FrameBright > 64) {FrameBright = 64;}
          break;
        case 'C': // set FrameCnt variable 0 - 1000
          FrameCnt = cmdVal;
          if (FrameCnt < 0) {FrameCnt = 0;} else if (FrameCnt > 1000) {FrameCnt = 1000;}
          break;
        case 'D': // sets FrameDim factor 1 - 64
          FrameDim = cmdVal;
          if (FrameDim < 1) {FrameDim = 1;} else if (FrameDim > 64) {FrameDim = 64;}
          break;
        case 'L': // set frame period lower limit
          FramePrdLL = cmdVal;
          if (FramePrdLL < 25) {FramePrdLL = 25;} else if (FramePrdLL > 1000) {FrameDim = 1000;}
          if (FramePrdLL > FramePrdUL) {FramePrdLL = FramePrdUL;}
          break;
        case 'P': // set the FramePeriod directly as nn
          FramePeriod = cmdVal;
          if (FramePeriod < 25) {FramePeriod = 25;} else if (FramePeriod > 1000) {FramePeriod = 1000;}
          break;
        case 'U': // set frame period upper limit
          FramePrdUL = cmdVal;
          if (FramePrdUL < 25) {FramePrdUL = 25;} else if (FramePrdUL > 1000) {FramePrdUL = 1000;}
          if (FramePrdUL < FramePrdLL) {FramePrdUL = FramePrdLL;}
          break;
      } break;
    case 'M': // toggles MPU on/off
      switch (cmdType) {
        case ' ': // M. toggles MPU on/off
          if (MPU_Init) {MPU_Init = false;} else {MPU_Init = true;}
          if (MPU_Init) {initialiseMPU(); cmdTxt = " MPU initialise";}  // restart MPU from scratch
          else {cmdTxt = " MPU is OFF";}
          break;
        case 'O': // report MPU offsets
          Serial.println("AcXOff: " + String(AcXOff) + "  \tAcYOff: " + String(AcYOff) + "  \tAcZOff: " + String(AcZOff));
          Serial.println("GyXOff: " + String(GyXOff) + "  \tGyYOff: " + String(GyYOff) + "  \tGyZOff: " + String(GyZOff));
          cmdTxt = " MPU offsets"; break;
        case 'R': // turns MPU reporting n=1 ON/n=0 OFF
          if (cmdVal == 0) {MPUrep = false;} else {MPUrep = true;}
          break;
      } break;
    case 'P': // Pause toggle ON/OFF
      if (cmdType == ' ') { cmdTxt = " PAUSE: ";if (PAUSE) {PAUSE = false; cmdTxt += "OFF";} else {PAUSE = true; cmdTxt += "ON";}}
      break;
    case 'R': // Set/reset report mode
      Report = cmdVal; ReportDel = 0;
      if (cmdType == '?') {Report = -1;}
      else if (Report > 0) {cmdTxt = " Report: " + String(Report);} else {cmdTxt = " Report: OFF";}
      break;
    case 'T': // Task functions
      switch (cmdType) {
        case 'D': TaskDel_ms = cmdVal; break;
      } break;
    default:
      Serial.println("Unknown cmd: " + cmdMode + cmdType + String(cmdVal));
      break;
  }
  // Report the values received
  Serial.println(String(cmdMode) + String(cmdType) + String(cmdVal) + "." + cmdTxt);
  // Now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1; cmdTxt = "";
}

// --------------------------------------------------------------------------------

void EEPROM_Read() {
  // reads the FrameNum number from EEPROM
  // this function is only called once after power-on or RESET
  // a key is used to determine whether the EEPROM holds a previously stored Task number
  // a flag is RESET set, which is cleared in loop() after 3 seconds
  if ((EEPROM.read(EeAdd_K0) == EEPROM_Key0) && (EEPROM.read(EeAdd_K1) == EEPROM_Key1)) {
    // a previous value has been stored, so read it
    Bright = EEPROM.read(EeAdd_Bright);
    FrameNum = EEPROM.read(EeAdd_Task);
    MoodB = EEPROM.read(EeAdd_MoodB);
    MoodG = EEPROM.read(EeAdd_MoodG);
    MoodR = EEPROM.read(EeAdd_MoodR);
    WhiteLight = EEPROM.read(EeAdd_Light);
    Serial.println("EEPROM checksum is good");
    // check for a double-RESET
    if (EEPROM.read(EeAdd_Reset) > 0) {
      // double press on RESET switch so revert to default values
      Bright = EEPROM_DefBright;
      FrameNum = EEPROM_DefTask;
      MoodB = EEPROM_DefMoodB;
      MoodG = EEPROM_DefMoodG;
      MoodR = EEPROM_DefMoodR;
      WhiteLight = EEPROM_DefLight;
      EEPROM_Store(); // save the default velues
      Serial.println("EEPROM_RESET defaults written");
    }
  } else {
    // the key does not match so assume no Task value has yet been stored
    // so we write the default value and the key
    Serial.println("EEPROM_Read Checksum failed");
    EEPROM.write(EeAdd_K0, EEPROM_Key0); EEPROM.write(EeAdd_K1, EEPROM_Key1); // store the key
    Bright = EEPROM_DefBright;
    FrameNum = EEPROM_DefTask;
    MoodB = EEPROM_DefMoodB;
    MoodG = EEPROM_DefMoodG;
    MoodR = EEPROM_DefMoodR;
    WhiteLight = EEPROM_DefLight;
    EEPROM_Store();                                 // store the default Task value
  }
  // set the RESET flag so that we can detect a double-RESET action
  EEPROM.write(EeAdd_Reset, 1);
  EEPROM.commit();
}

// --------------------------------------------------------------------------------

void EEPROM_Store() {
  // store the current values in EEPROM
  // to preserve memory retention, if values have not changed 'commit()' wont overwrite them
  EEPROM.write(EeAdd_Bright, Bright);
  EEPROM.write(EeAdd_Light, WhiteLight);
  EEPROM.write(EeAdd_MoodR, MoodR);
  EEPROM.write(EeAdd_MoodG, MoodG);
  EEPROM.write(EeAdd_MoodB, MoodB);
  EEPROM.write(EeAdd_Task, FrameNum);
  EEPROM.commit();
  Serial.println("EEPROM_Store");
}

// ---------------------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = abs(cmdVal * 10) + zVal;
}

// -------------------------------------------------------------------------------

void initialiseMPU() {
  // By default the MPU-6050 sleeps. So we have to wake it up.
  Wire.beginTransmission(MPU_addr); //Start communication with the address found during search.
  Wire.write(0x6B); //We want to write to the PWR_MGMT_1 register (6B hex)
  Wire.write(0x00); //Set the register bits as 00000000 to activate the gyro
  I2CError = Wire.endTransmission();
  delayMicroseconds(10);                // Allow MPU time to respond
  if (I2CError == 0) {
    //Set the full scale of the gyro to +/- 250 degrees per second
    Wire.beginTransmission(MPU_addr); //Start communication with the address found during search.
    Wire.write(0x1B); //We want to write to the GYRO_CONFIG register (1B hex)
    Wire.write(0x00); //Set the register bits as 00000000 (250dps full scale)
    Wire.endTransmission();
    delayMicroseconds(10);              // Allow MPU time to respond
    //Set the full scale of the accelerometer to +/- 4g.
    Wire.beginTransmission(MPU_addr); //Start communication with the address found during search.
    Wire.write(0x1C); //We want to write to the ACCEL_CONFIG register (1A hex)
    Wire.write(0x08); //Set the register bits as 00001000 (+/- 4g full scale range)
    Wire.endTransmission();
    delayMicroseconds(10);              // Allow MPU time to respond
    //Set some filtering to improve the raw data.
    Wire.beginTransmission(MPU_addr); //Start communication with the address found during search
    Wire.write(0x1A); //We want to write to the CONFIG register (1A hex)
    Wire.write(0x03); //Set the register bits as 00000011 (Set Digital Low Pass Filter to ~43Hz)
    Wire.endTransmission();
    delayMicroseconds(10);              // Allow MPU time to respond
    MPU_Init = true;
    Serial.println("MPU Initialised"); 
  } else {
    MPU_Init = false;
    Serial.println("MPU Initialisation Failed!"); 
    Serial.println("I2C Error = " + String(I2CError)); 
  }
}

// -------------------------------------------------------------------------------

void readAccelAll() {
  // Read the X,Y,Z-axis acceleration values from the MPU
  // in this robot:
  // AcX = Pitch, +ve tip forward, -ve tip backward
  // AcY = up/down, -ve upside-down, +ve right way up
  // AcZ = Roll, +ve tip right, -ve tip left
  // t0 = micros();  // t1-t0 average = 225µs, longer with I2C bus contention
  AcXL = AcX; AcYL = AcY; AcZL = AcZ; // track previous values
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3B);  // starting with register 0x3B (ACCEL_XOUT_H)
  Wire.endTransmission();
  delayMicroseconds(10);              // Allow MPU time to respond

  Wire.requestFrom(MPU_addr,6,true);  // request 1 register read
  AcX = Wire.read()<<8|Wire.read();   // 0x3B (ACCEL_XOUT_H) & 0x3C (ACCEL_XOUT_L)
  AcY = Wire.read()<<8|Wire.read();   // 0x3D (ACCEL_YOUT_H) & 0x3E (ACCEL_YOUT_L)
  AcZ = Wire.read()<<8|Wire.read();   // 0x3F (ACCEL_ZOUT_H) & 0x40 (ACCEL_ZOUT_L)
  
  // apply calibration offsets to these raw values
  AcX += AcXOff;  // note AcXOff should be zero initially in order to determin its value
  AcY += AcYOff;  // note AcYOff should be zero initially in order to determin its value
  AcZ += AcZOff;  // note AcZOff should be zero initially in order to determin its value

  // if 1st reading then we don't have a valid previous value, so use current
  if (AcAvgDiv < 1) {AcXL = AcX; AcYL = AcY; AcZL = AcZ;}

  // Adjust for Pitch abd Roll offset error if not in calibration mode
  if (!MPUCal) {
         if (AcX > 0) {AcX = map(AcX,0,8192 + AcXOff,0,8192);}
    else if (AcX < 0) {AcX = map(AcX,-8192 + AcXOff,0,-8192,0);}
         if (AcZ > 0) {AcZ = map(AcZ,0,8192 + AcZOff,0,8192);}
    else if (AcZ < 0) {AcZ = map(AcZ,-8192 + AcZOff,0,-8192,0);}
  }

  // limit values to avoid trig function problems
  if (AcX > 8192) {AcX = 8192;} else if (AcX < -8192) {AcX = -8192;}
  if (AcY > 8192) {AcY = 8192;} else if (AcY < -8192) {AcY = -8192;}
  if (AcZ > 8192) {AcZ = 8192;} else if (AcZ < -8192) {AcZ = -8192;}

  // average over two readings to reduce random noise
  AcX = (AcX + AcXL)/2; AcY = (AcY + AcYL)/2; AcZ = (AcZ + AcZL)/2;

  if (AcY >= 0) {UpRht = true;} else {UpRht = false;} // set the upright flag
  // t1 = micros();
  //   Serial.print(AcX); Serial.print("\t");
  //   Serial.print(AcY); Serial.print("\t");
  //   Serial.println(AcZ);
}

// --------------------------------------------------------------------------------

void readBattery() {
  // Called every 200ms
  // Read the voltage on the analog input and average it over 20 counts
  BatVol = analogRead(BatPin);  // take the reading
  
  BatSum = BatSum + BatVol - BatAvg;
  BatAvg = BatSum/20;     // ADC is averaged over 20 readings to remove noise
  // uncomment the following line if you want to see readings on serial monitor
  // Serial.print("BattAv = "); Serial.print(BattAv); Serial.print("\t"); Serial.print((5.0 * BattAv)/608.9); Serial.println("v");
}

// -------------------------------------------------------------------------------

void readGyrosAll() {
  // read the X,Y,Z-axis gyroscope values from the MPU
  // in this robot:
  // GyX = yaw
  // GyY = rotate round
  // GyZ = pitch
  // note gyro values are in degrees/sec so we use time between measurements to 
  // determine the average displaced angle
  // t0 = micros();  // t1-t0 average = 315µs, longer with I2C bus contention
  GyXL = GyX; GyYL = GyY; GyZL = GyZ; // track previous values
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x43);  // starting with register 0x43 (GYRO_XOUT_H)
  Wire.endTransmission();
  delayMicroseconds(10);              // Allow MPU time to respond

  Wire.requestFrom(MPU_addr,6,true);  // request 1 register read
  GyX = Wire.read()<<8|Wire.read();  // 0x43 (GYRO_XOUT_H) & 0x44 (GYRO_XOUT_L)
  GyY = Wire.read()<<8|Wire.read();  // 0x45 (GYRO_YOUT_H) & 0x46 (GYRO_YOUT_L)
  GyZ = Wire.read()<<8|Wire.read();  // 0x47 (GYRO_ZOUT_H) & 0x48 (GYRO_ZOUT_L)
  if (GyT > 0) {
    GyTD = millis() - GyT;  // get the differential time between readings
    GyT = GyTD + GyT;       // set the timer for the next reading
  } else {
    // 1st reading after reset
    GyT = millis();
    GyTD = 40; GyXL = GyX; GyYL = GyY; GyZL = GyZ;
  }

  // apply calibration offsets to these raw values
  GyX += GyXOff;  // note GyXOff should be zero initially in order to determin its value
  GyY += GyYOff;  // note GyYOff should be zero initially in order to determin its value
  GyZ += GyZOff;  // note GyZOff should be zero initially in order to determin its value

  // average over two readings to reduce random noise
  GyX = (GyX + GyXL)/2; GyY = (GyY + GyYL)/2; GyZ = (GyZ + GyZL)/2;
  // t1 = micros();
}

// --------------------------------------------------------------------------------

void readSerial() {
  // Reads characters from the serial port and responds to commands
  keyVal = Serial.read();
  if (keyVal != -1) {decodeKey(keyVal);}
}

// --------------------------------------------------------------------------------

void readSW0() {
  // Read the left button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task
  if (REBOOT) {return;}

  sw0State = digitalRead(sw0Pin); // record button state
  if (TEST) {return;}             // block switch decoding if in TEST mode
  if (SW0_Nop) {return;}          // block this function whilst == true
  
  bool sw0Clr = false;
  if (sw0State == LOW) {
    //##############################################################################
    //
    // SW0 button is pressed down
    //
    //##############################################################################
    if (sw0LastState == HIGH) {
      // SW0 has just been pressed down
      if (Exec != 0) {Exec = 0;}  // terminate Exec if running
      RGB0[0].setRGB(LEDSwMax, 0, 0); RGBshow = true;
      if (luxMode == 0) {
        if (FrameTask != 0) {FrameTask = 97; waitWhileDwn(); return;}  // turn lamp OFF
        else {setFrameTask(FrameNum);}   // turn lamp ON
      } else {
        if (luxMode == 2) {luxTask++; if (luxTask > 2) {luxTask = 0;}}
      }
      sw0DwnTime = 0; // restart button down time counter
      sw0Cnt++; // count on the falling edge
      sw0Timer = 0; // reset the timer
    } else {
      sw0DwnTime++; // track button pressed time
    }
    // if long press for >1.5 seconds then ... not used
    if (sw0DwnTime > 150) {}
  } else {
    //##############################################################################
    //
    // SW0 button is released
    //
    //##############################################################################
    if (sw0LastState == LOW) {
      // SW0 has just been released
      RGB0[0].setRGB(0, 0,RGB_Blu); RGBshow = true;
    }
    if (sw0Timer >= 100) {
      // button released for 1 sec so assume valid button count
      sw0Cnt = 0; sw0Timer = 0; sw0DwnTime = 0;
    }
  }
  if (ESC0 && !sw0State) {return;}
  sw0LastState = sw0State; // record current state
  ESC0 = false;
  if (sw0Cnt > 0) {sw0Timer++;}
}

// --------------------------------------------------------------------------------

void readSW1() {
  // read the centre button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task
  if (REBOOT) {return;}

  sw1State = digitalRead(sw1Pin); // record button state
  if (TEST) {return;}             // block switch decoding if in TEST mode
  if (SW1_Nop) {return;}          // block this function whilst == true
  
  if (sw1State == LOW) {
    //##############################################################################
    //
    // SW1 button is pressed down
    //
    //##############################################################################
    if (sw1LastState == HIGH) {
      // SW1 has just been pressed down
      if (Exec != 0) {Exec = 0;}  // terminate Exec if running
      RGB0[0].setRGB( 0,LEDSwMax, 0); RGBshow = true;
      if (FrameTask == 0) {yield();}
      else if (luxMode == 0) {Gfx_Char = 0; setFrameTask(93);}  // display left arrow <
      sw1DwnTime = 0; // restart button down time counter
      sw1Cnt++;       // count on the falling edge
      sw1Mode = 0;    // reset the mode state
      sw1Timer = 0;   // reset the timer
    } else {
      sw1DwnTime++;   // track button pressed time
      if (FrameNum == 1) {
        // we are in white light mode
        if ((sw1DwnTime >= 50) && (sw1Mode == 0)){
          // button held down for > 500ms, so adjust light down
          sw1Mode = 1; sw1DwnTime = 0; sw1Cnt = 0;
          if (WhiteLight > 1) {WhiteLight--; Frame_Light();}
        }
        if ((sw1DwnTime >= 10) && (sw1Mode == 1)){
          // button held down for > 100 ms, so adjust light down
          sw1DwnTime = 0;
          if (WhiteLight > 1) {WhiteLight--; Frame_Light();}
        }
      } else if ((FrameTask == 2) && (luxMode == 2)) {
        // we are in Mood lux mode
        sw1Cnt = 0;
        if (sw1DwnTime > 10) {
          switch(luxTask) {
            case 0: if (MoodR > 0) {MoodR--;} // adjust red component down
              break;
            case 1: if (MoodG > 0) {MoodG--;} // adjust green component down
              break;
            case 2: if (MoodB > 0) {MoodB--;} // adjust blue component down
              break;
          } sw1DwnTime = 0;
        }
      } else {
        // we are not in white light or mood mode
        if ((sw1DwnTime >= 100) && (sw1Mode == 0)){
          // button held down for > 1 sec
          sw1Mode = 1; Gfx_Char = 2; setFrameTask(93);  // display |<
        }
        if ((sw1DwnTime >= 200) && (sw1Mode == 1)){
          // button held down for > 2 sec
          sw1Mode = 2; Gfx_Char = 4; setFrameTask(93);  // display [.]
        }
      }
    }
    if (ESC1 && !sw1State) {return;}
    ESC1 = false;
  } else {
    //##############################################################################
    //
    // sw1State == HIGH as SW1 button is not pressed
    //
    //##############################################################################
    if (sw1LastState == LOW) {
      // SW1 has just been released
      RGB0[0].setRGB( 0, 0,RGB_Blu); RGBshow = true;
           if (FrameTask == 0) {yield();}
      else if (FrameNum == 1) {setFrameTask(1); EEPROM_Store(); EEPROM.commit();}  // set all faces
      else if (luxMode == 2) {EEPROM_Store(); EEPROM.commit();}  // store changes
      else if (FrameNum > 2) {setFrameTask(91);}  // clear cube faces
    }
    if (sw1Timer > 100) {
      // button released for 1 sec so assume valid button count
      // switch task if not in OFF state
      if ((sw1Cnt > 0) && (FrameTask != 0)) {
        if (sw1Mode == 0) {FrameNum-= sw1Cnt; if (FrameNum < FrameMin) {FrameNum = FrameMax;}}
        if (sw1Mode == 1) {FrameNum = FrameMin;}
        if (sw1Mode == 2) {Exec = 1;} // start the Exec random task mode
        EEPROM_Store(); EEPROM.commit(); // store new frame number task in EEPROM
        FrameTask = 96; // switch to the new frame task
      }
      sw1Cnt = 0; sw1Timer = 0; sw1DwnTime = 0; sw1Mode = 0;
   }
  }
  sw1LastState = sw1State; // record current state
  if (sw1Cnt > 0) {sw1Timer++;}
}

// --------------------------------------------------------------------------------

void readSW2() {
  // read the right button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task
  if (REBOOT) {return;}

  sw2State = digitalRead(sw2Pin); // record button state
  if (TEST) {return;}             // block switch decoding if in TEST mode
  if (SW2_Nop) {return;}          // block this function whilst == true
  
  if (sw2State == LOW) {
    //##############################################################################
    //
    // SW2 button is pressed down
    //
    //##############################################################################
    if (sw2LastState == HIGH) {
      // SW2 has just been pressed down
      if (Exec != 0) {Exec = 0;}  // terminate Exec if running
      RGB0[0].setRGB( 0, 0,LEDSwMax); RGBshow = true;
           if (FrameTask == 0) {yield();}
      else if (luxMode == 0) {Gfx_Char = 1; setFrameTask(94);}  // display right arrow
      sw2DwnTime = 0; // restart button down time counter
      sw2Cnt++;       // count on the falling edge
      sw2Mode = 0;    // reset the mode state
      sw2Timer = 0;   // reset the timer
    } else {
      sw2DwnTime++;   // track button pressed time
      if (FrameNum == 1) {
        // we are in white light mode
        if ((sw2DwnTime >= 50) && (sw2Mode == 0)){
          // button held down for > 500ms, so adjust light up
          sw2Mode = 1; sw2DwnTime = 0; sw2Cnt = 0;
          if (WhiteLight < WhiteMax) {WhiteLight++; Frame_Light();}
        }
        if ((sw2DwnTime >= 10) && (sw2Mode == 1)){
          // button held down for > 100 ms, so adjust light down
          sw2DwnTime = 0;
          if (WhiteLight < WhiteMax) {WhiteLight++; Frame_Light();}
        }
      } else if ((FrameTask == 2) && (luxMode == 2)) {
        // we are in Mood mode
        sw2Cnt = 0;
        if (sw2DwnTime > 10) {
          switch(luxTask) {
            case 0: if (MoodR < LED_Max) {MoodR++;} // adjust red component down
              break;
            case 1: if (MoodG < LED_Max) {MoodG++;} // adjust green component down
              break;
            case 2: if (MoodB < LED_Max) {MoodB++;} // adjust blue component down
              break;
          } sw2DwnTime = 0;
        }
      } else {
        if ((sw2DwnTime >= 100) && (sw2Mode == 0)){
          // button held down for > 1 sec
          sw2Mode = 1; Gfx_Char = 3; setFrameTask(94);  // display >|
        }
        if ((sw2DwnTime >= 200) && (sw2Mode == 1)){
          // button held down for > 2 sec
          sw2Mode = 2; Gfx_Char = 4; setFrameTask(93);  // display [.]
        }
      }
    }
    if (ESC2 && !sw1State) {return;}
    ESC2 = false;
  } else {
    //##############################################################################
    //
    // sw1State == HIGH as SW1 button is not pressed
    //
    //##############################################################################
    if (sw2LastState == LOW) {
      // SW2 has just been released
      RGB0[0].setRGB( 0, 0,RGB_Blu); RGBshow = true;
           if (FrameTask == 0) {yield();}
      else if (FrameNum == 1) {setFrameTask(1); EEPROM_Store(); EEPROM.commit();}  // set all faces
      else if (luxMode == 2) {EEPROM_Store(); EEPROM.commit();}  // store changes
      else if (FrameNum > 2) {setFrameTask(91);}  // clear cube faces
    }
    if (sw2Timer > 100) {
      // button released for 1 sec so assume valid button count
      // switch task if not in OFF state
      if ((sw2Cnt > 0) && (FrameTask != 0)) {
        if (sw2Mode == 0) {FrameNum+= sw2Cnt; if (FrameNum > FrameMax) {FrameNum = FrameMin;}}
        if (sw2Mode == 1) {FrameNum = FrameMax;}
        if (sw2Mode == 2) {Exec = 1;} // start the Exec random task mode
        EEPROM_Store(); EEPROM.commit(); // store new frame number task in EEPROM
        FrameTask = 96; // switch to the new frame task
      }
      sw2Cnt = 0; sw2Timer = 0; sw2DwnTime = 0; sw2Mode = 0;
   }
  }
  sw2LastState = sw2State; // record current state
  if (sw2Cnt > 0) {sw2Timer++;}
}

// ---------------------------------------------------------------------------------

void ReportNow() {
  // Called every 20ms to report values to the serial port
  if (ReportDel > 0) {ReportDel--; return;}

  switch (Report) {
    case -2: // List FrameTask options
      Serial.println("FrameTask options:");
      Serial.println(" 1 - Frame_Light()");
      Serial.println(" 2 - Frame_Mood()");
      Serial.println(" 3 - Frame_Rain()");
      Serial.println(" 4 - Frame_Matrix()");
      Serial.println(" 5 - Frame_Worms()");
      Serial.println(" 6 - Frame_Burst()");
      Serial.println(" 7 - Frame_SunSpots()");
      Serial.println(" 8 - Frame_Spirals()");
      Serial.println(" 9 - Frame_ASCII()");
      Serial.println("10 - Frame_SpiralRoll()");
      Serial.println("11 - Frame_VuMeter()");
      Serial.println("12 - Frame_VuSpin()");
      Serial.println("13 - Frame_VuFace()");
      Serial.println("14 - Frame_Discharge()");
      Serial.println("15 - Frame_Level()");
      Serial.println("16 - Frame_UTC()");
      Serial.println("17 - Frame_CoderDojo()");
      Serial.println(""); Report = 0; break;  // cancel th is option
    case -1:  // List report options
      Serial.println("Report options:");
      Serial.println(" 1 - AcX, AcY, AcZ");
      Serial.println(" 2 - GyX, GyY, GyZ");
      Serial.println(" 3 - micRaw");
      Serial.println(" 4 - lux");
      Serial.println(""); Report = 0; break;  // cancel th is option
    case  1:
      Serial.print("AcX:" + String(AcX) + "\t");
      Serial.print("AcY:" + String(AcY) + "\t");
      Serial.print("AcZ:" + String(AcZ) + "\n");
      ReportDel = 5; break; // repeat every 5 x 20 ms
    case  2:
      Serial.print("GyX:" + String(GyX) + "\t");
      Serial.print("GyY:" + String(GyY) + "\t");
      Serial.print("GyZ:" + String(GyZ) + "\n");
      ReportDel = 5; break; // repeat every 5 x 20 ms
    case  3:
      Serial.println("micRaw:" + String(micRaw));
      ReportDel = 0; break; // repeat every 5 x 20 ms
    case  4:
      Serial.println("lux:" + String(lux));
      ReportDel = 4; break; // repeat every 5 x 20 ms
  }
}

// ---------------------------------------------------------------------------------

void setFrameTask(int zMT) {
  // Change the frame task pointer to zMT.
  // Set default conditions for starting/ending all tasks.
  // Task initialisation may modify these values.
  AcXOff = 0;         // clear drift offset when not using MPU6050
  AcYOff = 0;         // clear drift offset when not using MPU6050
  AcZOff = 0;         // clear drift offset when not using MPU6050
  FrameBuild = true;  // if == true then run a frame building task
  FrameCnt = 0;       // Frame task general counter/timer
  FrameDel = 0;       // Frame task delay counter, if > 0 holds off tasks
  FrameNext = 0;      // next FrameTask to be performed after a delay
  FramePeriod = 30;   // frame time period in ms, default = 30
  FramePrdLL = 18;    // frame time period lower limit in ms, fastest
  FramePrdUL = 60;    // frame time period upper limit in ms, slowest
  FrameSubTask = 0;   // reset sub task pointer
  LED_Blink = true;   // flag toggled to create blinking effects
  LED_Cnt = 0;        // general counter used in LED tasks
  LED_Col = 0;        // general colour value used in LED tasks
  LED_Del = 0;        // LED Task delay, adds pauses into the processes
  LED_Inc = 1;        // general increment used in LED tasks
  LED_Num = 0;        // general number stored used in LED tasks
  LED_Pnt = 0;        // general LED pointer
  LED_SubTask = 0;    // LED sub-task pointer
  LED_Task = 0;       // LED task pointer
  luxMode = 0;        // lux mode trigered by lux sensor
  micVuCnt = 0;       // counter usied in VU level meter
  
  FrameTask = abs(zMT);
  Serial.println("setFrameTask(" + String(FrameTask) + ")");
}

// --------------------------------------------------------------------------------

void waitFrameEnd() {
  // called by either switch function when a button is pressed during a task
  // Serial.println("waitFrameEnd");
  setFrameTask(0); waitWhileDwn();
}

// --------------------------------------------------------------------------------

void waitWhileDwn() {
  // called by any switch function when a button is pressed during a task
  while ((digitalRead(sw0Pin) == LOW) || (digitalRead(sw1Pin) == LOW) || (digitalRead(sw2Pin) == LOW)) {
    // do nothing whilst any button is pressed
    delay(200);
    // test for long SW0 press == RESET
    if (digitalRead(sw0Pin) == LOW) {
      ResetCnt++;
      if (ResetCnt >= 15) {
        // button held down for >= 3 seconds so force a soft reset
        REBOOT = true; setFrameTask(999); return; //call REBOOT
      }
    } else {ResetCnt = 0;}  // clear the timer
  }
  // clear SW0 lamp if in FrameTask 0
  if (FrameTask == 0) {LED[0].setRGB(0, 0, 0); LEDshow = true;}
  RGB0[0].setRGB(0, 0,RGB_Blu); RGBshow = true;   // clear RG LED
  
  // reset switch variables
  sw0Cnt = 0; sw0Timer = 0; sw0LastState = HIGH; ESC0 = false; SW0_Nop = false;
  sw1Cnt = 0; sw1Timer = 0; sw1LastState = HIGH; ESC1 = false; SW1_Nop = false;
  sw2Cnt = 0; sw2Timer = 0; sw2LastState = HIGH; ESC2 = false; SW2_Nop = false;
  synchLoopTimers();
}

// ---------------------------------------------------------------------------------
