// ################################################################################
//
//  Cube 8x8x5 HD vR1.0
//
//  Released:  30/01/2025
//
//  Author: TechKnowTone
//
// #################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    microcontroller:  ESP32C3 Dev Module   v3.1.1

    also Tools/
      Use CDC On Boot "Enabled"
      Flash Mode "QIO"
      JTAG Adapter "Disabled"
      Upload Speed "115200"

    This code controls an 8x8x5 RGB LED light cube, which also contains sensors for
    sound, ambient light and motion. It is not using wireless features, and the tasks
    run on a single core.

    * tasks that construct LED patterns
    * driving NeoPixel LEDs using FastLED
    * reading of sound waves from Mic MAX4466 at 1 kHz
    * reading motions from MPU6050 3-axis Acc/Gyro
    * reading BH1750 lux sensor
    * reading button switches SW0 - SW2

    TEST = true initially for testing circuitry whilst building the cube. This ouputs
    a walking red LED light pattern and sends sensor data to the serial port. This
    confirm everything is connected and working before soldering the wire wraps.
    
    TEST = false is the normal mode of operation once the cube is built and tested.

    #############
    # IMPORTANT # There appears to be a problem with the FastLED library, and the RMT
    ############# device within the ESP32C3, that cause the code to fail when using the
    latest version of the library. If you experience problems, when compiling or running
    the code (observe Serial Monitor Boot statements), you should roll back your library
    components to match those shown below. These work, I can't guarantee later versions!
*/
// Declare Libraries
#include <Wire.h>             // I2C library                v3.1.1
#include <FastLED.h>          // Neopixel library           v3.9.4
#include <BH1750FVI.h>        // BH1750 light sensor        v1.1.1
#include <EEPROM.h>           // EEPROM flash library       v3.1.1

// Configuration
String Release = "Cube 8x8x5 HD vB.0";
String Date = "30/01/2025";
 
// I2C interface pins for ESP32C3
const int SCL_PIN =  9;
const int SDA_PIN =  8;

// Declare and initialise light sensor global variables
uint8_t ADDRESSPIN = 2;   // assigned but not used in this code
uint16_t lux;             // light level value returned from the BH1750 sensor
BH1750FVI::eDeviceAddress_t DEVICEADDRESS = BH1750FVI::k_DevAddress_L;
BH1750FVI::eDeviceMode_t DEVICEMODE = BH1750FVI::k_DevModeContHighRes;

// Create light sensor instance
BH1750FVI LightSensor(ADDRESSPIN, DEVICEADDRESS, DEVICEMODE);

// 3-axis MPU 6050 variables.
// Default values are assigned by calling setDefaults() in setup()
const int MPU_addr=0x68;      // I2C address of the MPU-6050
long AcAvgDiv;                // dividing factor used in averaging
long AcAvgDivMax;             // max dividing factor for averaging
int16_t AcX;                  // MPU raw X-acceleration value
long AcXAvg;                  // averaged AcX value to reduce noise
int16_t AcXL;                 // previous MPU X-acceleration value
int16_t AcXOff;               // offset to be added to raw X-acceleration value
long AcXSum;                  // averaging accumulator
int16_t AcY;                  // MPU raw Y-acceleration value
long AcYAvg;                  // averaged AcY value to reduce noise
float AcYF;                   // floating point Y acceleration value
int16_t AcYL;                 // previous MPU Y-acceleration value
int16_t AcYOff;               // offset to be added to raw Y-acceleration value
bool AcYsgn;                  // positive/negative sign flag
long AcYSum;                  // averaging accumulator
int16_t AcZ;                  // MPU raw Z-acceleration value
long AcZAvg;                  // averaged AcZ value to reduce noise
int16_t AcZL;                 // previous MPU Z-acceleration value
int16_t AcZOff;               // offset to be added to raw Z-acceleration value
long AcZSum;                  // averaging accumulator
byte GC[8];                   // buffer for Gfx charcaters
byte GH,GW;                   // Gfx character height and width
long GyAvgDiv;                // dividing factor used in averaging
int GyCal;                    // counter used to determine gyro calibration period
unsigned long GyT;            // time of gyro readings in milliseconds
unsigned long GyTD;           // time between successive gyro readings in milliseconds
int16_t GyX;                  // MPU raw X-gyroscope value
long GyXAvg;                  // averaged GyX value to reduce noise
int16_t GyXL;                 // previous MPU X-gyroscope value
int16_t GyXOff;               // offset to be added to raw X-gyroscope value
int16_t GyXOffCnt;            // offset counter for gyro centering
long GyXSum;                  // averaging accumulator
int16_t GyY;                  // MPU raw Y-gyroscope value
long GyYAvg;                  // averaged GyY value to reduce noise
int16_t GyYL;                 // previous MPU Y-gyroscope value
int16_t GyYOff;               // offset to be added to raw Y-gyroscope value
int16_t GyYOffCnt;            // offset counter for gyro centering
long GyYSum;                  // averaging accumulator
int16_t GyZ;                  // MPU raw Z-gyroscope value
long GyZAvg;                  // averaged GyZ value to reduce noise
int16_t GyZL;                 // previous MPU Z-gyroscope value
int16_t GyZOff;               // offset to be added to raw Z-gyroscope value
int16_t GyZOffCnt;            // offset counter for gyro centering
long GyZSum;                  // averaging accumulator
bool MPUCal;                  // == true during calibration
bool MPU_Init;                // == true if MPU read is enabled, default = true
bool MPUrep;                  // MPU report flag
float PitchAcc;               // pitch angle in degrees base on accelerometer
float PitchGyr;               // pitch angle in degrees base on gyro + acc
int PitchGyrInt;              // integer version of PitchGyr for quicker calcs
int16_t PitchGyrLast;         // previous PitchGyrInt value
float RollAcc;                // roll angle in degrees based on accelerometers
float RollGyr;                // roll angle in degrees based on gyro + acc
int RollGyrInt;               // integer version of RollGyr for quicker calcs
int16_t RollGyrLast;          // previous RollGyrInt value
bool UpRht;                   // set == true when the robot is the correct way up
float YawAcc;                 // roll angle in degrees based on accelerometers
float YawGyr;                 // roll angle in degrees based on gyro + acc
int YawGyrInt;                // integer version of RollGyr for quicker calcs

// EEPROM storage address map
#define EeAdd_K0 0            // EEPROM checksum key LSB
#define EeAdd_K1 1            // EEPROM checksum key MSB
#define EeAdd_Task 2          // latest selected frame task
#define EeAdd_Reset 3         // RESET to defaults flag
#define EeAdd_Light 4         // white light brightness level
#define EeAdd_MoodR 5         // mood light red component
#define EeAdd_MoodG 6         // mood light green component
#define EeAdd_MoodB 7         // mood light blue component
#define EeAdd_Bright 8        // FastLED brightness value


// EEPROM default values for 1st time run
#define EEPROM_DefBright 255  // default FastLED brightness level
#define EEPROM_DefLight 1     // default white light brightness level
#define EEPROM_DefMoodB 8     // default mood light blue component
#define EEPROM_DefMoodG 4     // default mood light green component
#define EEPROM_DefMoodR 0     // default mood light red component
#define EEPROM_DefTask 1      // default task number
#define EEPROM_Key0 0x55      // EEPROM checksum key LSB
#define EEPROM_Key1 0xAA      // EEPROM checksum key MSB
#define EEPROM_SIZE 9         // define the size of the reserved flash area, see map above

// Declare global general constants
#define ArrayMax 60           // LED_Array[] size for GFX functions
#define BatCal 377.1          // calibrated voltage multiplying factor, default 404.85 
#define BatCritical 2489      // critical battery threshold @6.6v, default = 2672, set == 0 to ignore
#define BatMax 3092           // A0 for fully charged battery voltage, == 8.2v
#define BatPin  1             // ADC battery pin
#define BatUSB 1858           // if <= then USB supply assumed
#define GyAvgDivMax 250       // max averaging divider, 250 == 10 second average
#define LED_Pin 7             // assign GPIO7 for 8x8x5 LEDs
#define LEDSwMax 32           // max LED brightness for switch operations
#define micMaxDef 200         // default micMax value
#define MicPin  0             // MAX4466 mic input pin
#define NumLEDs 320           // number of DotStar LEDs
#define NumLEDsM1 319         // number of DotStar LEDs minus 1
#define RGB_Pin 10            // GPIO10 is connected to the built-in RGB LED
#define sw0Pin  3             // left switch SW0 assigned to GPIO15
#define sw1Pin  4             // centre switch SW1 assigned to GPIO02
#define sw2Pin  5             // right switch SW2 assigned to GPIO00
#define TEST false            // normally false, set = true when building and testing the project
#define WhiteMax 64           // max value used in white light mode
 
// Create a FastLED instances and define the total number of LEDs in the cube
CRGB LED[NumLEDs];            // Cube matrix 8x8x5 strip
CRGB RGB0[1];                 // built-in LED
CRGB LEDX[1];                 // temp LED value, for GFX pixel manipulation

// define data structures
struct point_rgb {
  uint8_t r;
  uint8_t g;
  uint8_t b;
};
struct point_rgb point;

struct array_8 {
  uint8_t r[8];
  uint8_t g[8];
  uint8_t b[8];
};
struct array_8 LED_Block;

// Declare global variables which are initialised in setDefaults() immediately after RESET
int16_t Any;                  // any temp value
uint8_t Any8;                 // any temp byte value
long AnyLong;                 // any temp value
String Any$;                  // temporary string
uint8_t BAK_Blu;              // current background blue colour
uint8_t BAK_Grn;              // current background green colour
uint8_t BAK_Red;              // current background red colour
int16_t BatAvg;               // preload averaging function with high value
long BatSum;                  // preload averaging sum with 20x max value
int16_t BatVol;               // instantaneous battery voltage
uint8_t Bright;               // FastLED brightness
char cmdMode;                 // command mode
int16_t cmdSgn;               // cmdVal sign, 0 or -1
char cmdType;                 // command mode type
String cmdTxt;                // text sent to the serial port relating to cmdMode
int16_t cmdVal;               // value associated with a cmdType
uint8_t Col11;                // ambient max colour
uint8_t Col34;                // ambient 3/4 colour
uint8_t Col12;                // ambient 1/2 colour
uint8_t Col14;                // ambient 1/4 colour
bool Echo;                    // if true send print messages
String Error$ = "";           // any error message
bool ESC0;                    // SW0 ESC exit current Main Task flag
bool ESC1;                    // SW1 ESC exit current Main Task flag
bool ESC2;                    // SW2 ESC exit current Main Task flag
int16_t Exec;                 // timer/flag controls Exec overide mode
unsigned long ExecTimer;      // 100ms Exec timer
int16_t FrameBright;          // brightness factor used in tasks
bool FrameBuild;              // if == true then run a frame building task
int16_t FrameCnt;             // Main Task general counter/timer
int16_t FrameDel;             // Main task delay counter, if > 0 holds off tasks
int16_t FrameDim;             // Value used in frame dimming functions
int16_t FrameMax;             // max of FrameTask
int16_t FrameMin;             // min of FrameTask
int16_t FrameNext;            // next FrameTask to be performed after a delay
int16_t FrameNum;             // current FrameTask to be performed after a delay
unsigned long FramePeriod;    // frame time period in ms, default = 20
int16_t FramePk;              // general peak value used in frame tasks
unsigned long FramePrdLL;     // frame time period lower limit in ms
unsigned long FramePrdUL;     // frame time period upper limit in ms
int8_t FrameRnd;              // any random number 0 - 255
int16_t FrameSubTask;         // general subtask pointer
int16_t FrameSw;              // General frame switch() pointer
int16_t FrameTask;            // task pointer for frame construction
unsigned long FrameTimer;     // frame timer in ms
uint8_t Gfx_Char;             // pointer to a Gfx character
int16_t GfxMax;               // points to last element of current face, top left corner
int16_t GfxP0;                // points to first element of current face, bottom right corner
byte I2CError;                // used to detect I2C transmission errors
char keyChar;                 // any keyboard character
int16_t keyVal;               // any keyboard value
int16_t LED_Array[ArrayMax];  // array used in frame tasks
bool LED_Blink;               // flag toggled to create blinking effects
uint8_t LED_Blu;              // current blue colour
uint8_t LED_Char;             // general char value used in LED tasks
uint8_t LED_Col;              // general colour value used in LED tasks
int16_t LED_Cnt;              // general counter used in LED tasks
int16_t LED_Del;              // LED Task delay, adds pauses into the processes
uint8_t LED_End;              // general end counter used in LED tasks
int8_t LED_Face;              // cube face 0 - 4
bool LED_Flag;                // any true/false flag
uint8_t LED_Grn;              // current green colour
int16_t LED_Inc;              // general increment used in LED tasks
int16_t LED_Last;             // previous LED task pointer
uint8_t LED_Max;              // Max brightness level used by tasks
uint8_t LED_Min;              // general min number used by tasks
bool LED_Nop;                 // == true to block LED tasks, default = false
int16_t LED_Num;              // general number stored used in LED tasks
bool LED_ON;                  // built-in LED ON flag
int16_t LED_Pnt;              // general LED pointer
int16_t LED_PXY;              // LED array pointer
uint8_t LED_Red;              // current red colour
uint8_t LED_Rnd;              // 8-bit random number
bool LEDshow;                 // LED show flag for mouth LEDs
int16_t LED_SubTask;          // LED sub-task pointer
bool LED_SwDwn;               // == true if a button switch has been pressed
int16_t LED_Task;             // LED task pointer
String LED_Text;              // any string used in LED tasks
byte LED_Val;                 // values used in LED tasks
uint8_t LED_X;                // any X face co-ordinates
uint8_t LED_Y;                // any Y face co-ordinates
int16_t luxCnt;               // lux counter
int16_t luxMode;              // lux mode trigered by lux sensor
int16_t luxTask;              // lux task
bool luxState;                // true if lux >= 5
bool luxLastState;            // previous luxState
uint8_t Mask;                 // any 8-bit mask
uint8_t Mem_Blu;              // memory of current blue colour
uint8_t Mem_Grn;              // memory of current green colour
uint8_t Mem_Red;              // memory of current red colour
int16_t micAvg;               // average microphone value
int16_t micAvgCnt;            // counter for average microphone value
int16_t micCnt;               // delay counter usedin squelsh function
bool micFast;                 // == true to enable audio fast sampling
int16_t micInc;               // previous microphone value
int16_t micLvl;               // level tracker for squelch function
int16_t micMax;               // max micVol for scaling
int16_t micPk;                // peak measurement during 10ms cycle
int16_t micPkTrk;             // latest peak measurement during 10ms cycle
int16_t micRaw;               // raw value read from microphone input
int16_t micUni;               // unipolar microphone output
int16_t micVol;               // unipolar microphone volume
int16_t micVU;                // unipolar microphone VU level
int16_t micVuCnt;             // counter usied in VU level meter
uint8_t MoodB;                // mood light blue colour
uint8_t MoodG;                // mood light green colour
uint8_t MoodR;                // mood light red colour
unsigned long next10ms;       //  10ms loop timer
unsigned long next20ms;       //  20ms loop timer
unsigned long next200ms;      // 200ms loop timer
unsigned long nextSec;        // 1 second loop timer
bool Once;                    // flag used with RESET function, only true for a while
bool PAUSE;                   // flag set by user, if set stops animations being run, default = false
bool REBOOT;                  // set true when reboot is in progress to stop all tasks
bool RGBshow;                 // if == true then trigger a FastLED show for the RGB LED
int16_t Report;               // if > 0 value selects serial port reporting mode
int16_t ReportDel;            // if > 0 then skip report cycles
byte ResetCnt;                // timer used to invoke soft RESET
uint8_t RGB_Blu;              // built-in RGB LED blue OFF state, dims to off
uint8_t RndCol;               // previous random colour
int16_t RotCnt;               // auto zero delay counter for Z-axis
float RotDif;                 // rotation angle difference in degrees based on gyro + acc
float RotGyr;                 // rotation angle in degrees based on gyro + acc
int16_t RotGyrInt;            // rotation angle in degrees as an integer
int16_t StaticCnt;            // counter used in movement detection
int16_t sw0Cnt;               // button switch counter
int16_t sw0DwnTime;           // button pressed down time
bool sw0LastState;            // previous state of button switch, HIGH/LOW
bool SW0_Nop;                 // == true to block SW0 read tasks, default = false
bool sw0State;                // state of read button switch pin
int16_t sw0Timer;             // timer used to detemine button sequences
int16_t sw1Cnt;               // button switch counter
int16_t sw1DwnTime;           // button pressed down time
bool sw1LastState;            // previous state of button switch, HIGH/LOW
uint8_t sw1Mode;              // switch mode flag
bool SW1_Nop;                 // == true to block SW2 read tasks, default = false
bool sw1State;                // state of read button switch pin
int16_t sw1Timer;             // timer used to detemine button sequences
int16_t sw2Cnt;               // button switch counter
int16_t sw2DwnTime;           // button pressed down time
bool sw2LastState;            // previous state of button switch, HIGH/LOW
uint8_t sw2Mode;              // switch mode flag
bool SW2_Nop;                 // == true to block SW2 read tasks, default = false
bool sw2State;                // state of read button switch pin
int16_t sw2Timer;             // timer used to detemine button sequences
unsigned long t0,t1;          // millisecond code timers, report when t1 > t0
int16_t Task10ms;             // 10ms loop timer tasks
int16_t Task20ms;             // 20ms loop timer tasks
int16_t Task200ms;            // 200ms loop timer tasks
int16_t Task_Del;             // Delay allows FastLED RMT function
int16_t TaskDel_ms;           // Adjustable delay for debug, default = 4
uint32_t Task_ms;             // ms timer used with Task_Del
bool USBPwr;                  // true if POST detects USB power voltage
bool WaitUp;                  // true if task is to wait for button release
uint8_t WhiteLight;           // white light value

// ---------------------------------------------------------------------------------

void setup() {
  // This function runs immediately after RESET
  // Initialise I/O pins
  pinMode(BatPin,INPUT);        // ADC battery divider
  pinMode(MicPin,INPUT);        // ADC audio mic
  pinMode(sw0Pin,INPUT_PULLUP); // button switch sw0 Pin
  pinMode(sw1Pin,INPUT_PULLUP); // button switch sw1 Pin
  pinMode(sw2Pin,INPUT_PULLUP); // button switch sw2 Pin

  setDefaults();                // reset variables

  // Initialise the USB serial port.
  // On the ESP32C3 this is internal hardware, not a separate chip
  Serial.begin(115200);         // baud rate for serial monitor USB Rx/Tx comms
  delay(1000);                  // allow USB hardware to initialise
  Serial.println(Release);
  Serial.println(Date + "\n");

  // Initialise the FastLED components
  // Start with the 320 LEDs in the cube
  Serial.println("FastLEDs LED[]");
  FastLED.addLeds<WS2812B, LED_Pin, GRB>(LED, NumLEDs);
  FastLED.setBrightness(Bright);
  FastLED.clear(); FastLED.show();
  // Next, the single LED built-in to the ESP32C3 micro
  Serial.println("FastLEDs RGB0[]");
  FastLED.addLeds<WS2812B, RGB_Pin, RGB>(RGB0, 1);
  RGB0[0].setRGB(0,0,1); FastLED.show();     // dim blue at first

  // Wait for the RGB LEDs to be clocked out before accessing the I2C interface
  delay(20);  // It should only take 10ms but we can afford to be generous here

  // Initialise the I2C bus interface
  Serial.println("Wire.begin()...");
  Wire.begin(SDA_PIN, SCL_PIN); // initialise the I2C interface
  
  // Initialize EEPROM memory with predefined size
  Serial.println("EEPROM begin...");
  EEPROM.begin(EEPROM_SIZE);
    
  // Run the power-on self test (POST) code before running the main loop()
  Serial.println("RunPOST()...");
  RunPOST();

  // Set global loop timers for multi-tasking events
  Serial.println("synchLoopTimers()...");
  synchLoopTimers();
}

// ---------------------------------------------------------------------------------

  // the cube will react to a soft RESET event when SW0 is held down
  // this function effectively restarts the code as if from power-up
void(* resetFunc) (void) = 0; //declare reset function at address 0

// ---------------------------------------------------------------------------------


void loop() {
  // Main code which runs repeatedly as multi-tasking functions
  // Check serial port for user input every cycle
  if (Serial.available()) {readSerial();}   // check serial port for incoming data, not used in this project

  //###############################################################################
  //
  //  Frame tasks 20+ms (<=50Hz) variable
  //
  //###############################################################################
  // These tasks are run at a variable rate, so that LED effects can be speeded up
  // and slowed down in a smooth manner by changing the variable FramePeriod.
  // When the frame timer triggers it 'shows' the LEDs, if they are set to be shown
  // then it sets about building the next frame during the following period
  if (FrameBuild) {
    // Whilst FrameBuild == true we construct the new frame, then clear FrameBuild.
    // So complex frames can be built as a series of tasks progressively.
    doFrameTasks();
    // Has FrameTask cleared FrameBuild?
    if (!FrameBuild) {
      // Frame building has just completed.
    }
  } else {
    // FrameBuild is no longer active, so check for a show flag
    if (LEDshow || RGBshow) {
      // Update LEDs if LEDshow == true OR RGBshow = true;
      // This will also hold off loop_Tasks() for a Task_Del period
      FastLED.show();
      LEDshow = false; RGBshow= false;
      // Allow time for the LEDs to be updated, or output corruption will occur.
      // The FastLED output appears to compete with I2C timing, so hold off
      // reading I2C sernsors for a few milliseconds.
      Task_Del = TaskDel_ms; Task_ms = millis();
    }
  }
  if ((millis() - FrameTimer) >= FramePeriod) {
    // We have reached the end of the FrameTimer cycle period.
    FrameTimer = millis();  // reset the timer for the next period
    FrameBuild = true;      // re-initiate the next frame build process/sequence
  }

  //###############################################################################
  //
  //  Loop tasks
  //
  //###############################################################################
  // Loop tasks are held off immediately after a show() event for Task_Del milliseconds.
  // This is because the FastLED output interfers with the I2C interface
  if (Task_Del > 0) {
    // Decrement Task_Del every millisecond, until zero
    if (millis() != Task_ms) {Task_ms = millis(); Task_Del--;}
  } else {
    // If no delay, then perform loop_Tasks() on every loop() cycle
    loop_Tasks();
  }
    
  //###############################################################################
  //
  //  10 ms non-critical subtasks (100Hz)
  //
  //###############################################################################
  // These tasks are run in a consequitive ripple to reduce the overhead on more
  // critical code. They are performed in reverse case order and will all complete
  // within the 10ms period. None of these tasks use I2C, as it would interfers with
  // the FastLED output process
  if (Task10ms > 0) {
    switch(Task10ms) {
      case 1: AudioRead(); break;
      case 2: readSW2(); break;  // read right button switch SW2
      case 3: readSW1(); break;  // read centre button switch SW1
      case 4: readSW0(); break;  // read left button switch SW0
    }
    Task10ms--; // count down the sub-tasks from 4 - 1 as we loop
  } else if ((millis() - next10ms) >= 10) {
    // end of 10ms period, so set the timer for the next 10ms, and start tasks
    next10ms = millis();
    Task10ms = 4; // re-seed the task pointer
  }


  //###############################################################################
  //
  //  100ms Exec Timer
  //
  //###############################################################################
  // If the Exec mode is set, by a long press on SW1 or SW2, we will change the
  // display mode randomly after a defined period
  if ((millis() - ExecTimer) >= 100) {
    // we have reached the end of the 100ms period
    ExecTimer = millis(); // reset the timer for the next 100ms period
    // Test for Exec function, only when PAUSE is not active
    if (!PAUSE && (Exec > 0)) {
      // Exec mode automatically changes frame tasks randomly after a time delay
      // to exit Exec mode set Exec = 0, normally on a button press
      if (Exec > 1) {
        // the Exec counter timer is not decremented to zero, so we simply wait
        // for it to count down to 1, at a 100ms rate.
        Exec--;
      } else {
        // Exec == 1 so time to change to another task randomly
        // set Exec value to time the next change, timed in 100ms
        Exec = 100; // 10 seconds
        // some lower tasks (lamp and mood) and upper tasks are ignored
        // if they happen to be randomly selected they will re-invoke task change
        FrameNext = FrameTask;
        while (FrameNext == FrameTask) {FrameNext = random8(3,(FrameMax + 1));}
      //  Serial.println("Exec Task " + String(FrameNext));
        FrameTask = -FrameNext; // start a new frame task
        Gfx_Clear(0,NumLEDsM1); // clear all LEDs on cube faces
      }
    }
  }


  //###############################################################################
  //
  //  1 sec Tasks - TEST mode printing
  //
  //###############################################################################
  // This code is published with TEST == true, so that it generates a simple walking
  // LED pattern and reports sensor and switch data to the serial monitor. Use it to
  // check your wiring during the build process.
  // Once happy that everything is connected and working, set TEST = false to gain
  // the full functionality of the other code
  if ((millis() - nextSec) >= 1000) {
    // toggle built-in LEDS every second to show that the code is still working
    // this is only performed in TEST == true mode
    nextSec = millis();
    if (TEST) {
        LED_ON = !LED_ON;
        if (LED_ON) {
          digitalWrite(LED_BUILTIN,HIGH);
    //      Serial.println("LED ON");
        } else {
          digitalWrite(LED_BUILTIN,LOW);
    //      Serial.println("LED OFF");
        }
        // report sensor data every second in TEST == true mode
        Serial.println("");
        Serial.println("Report " + String(millis()/1000));
        Serial.println("BatVol= " + String(BatVol));
        Serial.println("Batt= " + String(float(BatAvg)/BatCal) + "v " + String(AnyLong) + "%");
        Serial.println("SW0= " + String(sw0State)+ "\tSW1= " + String(sw1State) + "\tSW2= " + String(sw2State));
        Serial.println("Lux= " + String(lux));
        Serial.println("micRaw= " + String(micRaw) + "\tmicUni= " + String(micUni) + "\tmicPk= " + String(micPk));
        Serial.println("AcX= " + String(AcXAvg) + "\tAcY= " + String(AcYAvg) + "\tAcZ= " + String(AcZAvg));
        Serial.println("GyX= " + String(GyXAvg) + "\tGyY= " + String(GyYAvg) + "\tGyZ= " + String(GyZAvg));
    } else {
      // if RGB_Blu > 0 then slowly dim the built-in RGB LED, until it is off
      if (RGB_Blu > 0) {RGB_Blu--; RGB0[0].setRGB( 0, 0,RGB_Blu); RGBshow = true;}
    }
  }

  //###############################################################################
  //
  //  Mic audio fast read task
  //
  //###############################################################################
  // Read mic input if in fast mode, every loop() cycle
  if (micFast) {
    micRaw = analogRead(MicPin);  // t1-t0 = 67 - 96 us
    if ((micRaw - micAvg) > 0) {micAvg+= micInc;} else {micAvg-= micInc;}
    micUni = abs(micRaw - micAvg);        // make volume unipolar
    if (micPk < micUni) {micPk = micUni;} // peak detector, reset by AudioRead
    micInc++; // count how many samples are being taken
  }
}

// ---------------------------------------------------------------------------------

void loop_Tasks() {
  // These functions are checked every loop() cycle, once Task_Del has expired.

  //###############################################################################
  //
  //  20 ms Tasks (50Hz)
  //
  //###############################################################################
  // These tasks are run in a consequitive ripple to reduce the overhead on more
  // critical code. They are performed in reverse case order and will all complete
  // within the 20ms period.
  if (Task20ms > 0) {
    switch(Task20ms) {
      case 1: if (Report) {ReportNow();} break; // report to serial port
      case 2: calcAngles(); break;              // calculate roll, pitch and yaw
      case 3: AvgGyrAll(); break;               // average sensor Gyro values
      case 4: AvgAccelAll(); break;             // average sensor Accel values
      case 5: readAccelAll(); break;            // read MPU6050 accelerometers
      case 6: readGyrosAll(); break;            // read MPU6050 gyroscopes as prioroty task
    }
    Task20ms--; // count down the sub-tasks from 6 - 1 as we loop
  } else if ((millis() - next20ms) >= 20) {
    // end of 20ms period, so set the timer for the next 20ms, and start tasks
    next20ms = millis();
    Task20ms = 6;   // re-seed the task pointer
    // If the MPU6050 fails to initialise we do not run those tasks
    if (!MPU_Init) {Task20ms = 1;}              // controls whether MPU is read or not
  }
  
  
  //###############################################################################
  //
  //  200 ms Tasks (5Hz)
  //
  //###############################################################################
  // Ambient light measurements are taken every 200ms as the BH1750 sensor needs
  // about 180ms in continuous mode. We use a 200ms timer to get that reading.
  // we can do other tasks that don't need to run quickly.
  if (Task200ms > 0) {
    switch(Task200ms) {
      case 1: readBattery(); break;// take a battery reading
    }
    Task200ms--;      // count down the sub-tasks from 5 - 1 as we loop
  } else if ((millis() - next200ms) >= 200) {
    // end of 200ms period, so set the timer for the next 200ms, and start tasks
    next200ms = millis();
    Task200ms = 1;    // re-seed the task pointer
    
    // over I2C it takes 383us to get a light reading
    // the light reading influences the brightness of the cube by setting default colours
    // it is also used as a mode setting switch when it is changed briefly, Hi-Lo-Hi
  //  t0 = micros();
    lux = LightSensor.GetLightIntensity(); // read the light sensor
  //  t0 = micros() - t0;
  //  Serial.println("Lux " + String(lux));
    // set colour levels base on ambient
    // on a bright day lux can exceed 800, so we will set 
    // >=600  - max brightness
    //   <50  - min brightness
          if (lux <=  50) {Col14 = 1;}
    else if (lux <= 100) {Col14 = 2;}
    else if (lux <= 150) {Col14 = 3;}
    else if (lux <= 250) {Col14 = 4;}
    else if (lux <= 450) {Col14 = 5;}
    else {Col14 = 6;}
    Col12 = Col14<<1; Col11 = Col14<<2; Col34 = Col11 - Col14;
    
    // Test for lux switching function, which is used to start a special mode in
    // which you can adjust the colour of the mood lighting.
    if (FrameTask > 0) {
      luxState = (lux >= 5);
      if (!luxState) {
        // the light sensor is below 5 lux
        if (luxLastState) {
          // the light sensor has just changed state to LOW
          luxCnt = 0; // reset the LOW counter
          Serial.println("Lux LOW " + String(lux));
        } luxCnt++;
      } else {
        // the light sensor is >= 5 lux
        if (!luxLastState) {
          // the light sensor has just changed state to HIGH
          Serial.println("Lux HIGH " + String(lux));
          if (luxCnt <= 5) {
            // short press detected, so actions are taken depending on FrameTask
            // if we are already in luxMode then simply cancel the mode
            if (luxMode > 0) {luxMode = 0;}           // cancelling luxMode
            else {luxMode = FrameTask; luxTask = 0;}  // set luxMode to current FrameTask
          } luxCnt = 0;
        }
      }
    } else {luxState = true;}
    luxLastState= luxState;

    // Test for long SW0 press == RESET
    if (digitalRead(sw0Pin) == LOW) {
      ResetCnt++;
      if (!REBOOT && (ResetCnt >= 15)) {
        // button held down for >= 3 seconds so force a soft reset
        REBOOT = true; setFrameTask(999); //call reset
      }
    } else {ResetCnt = 0;}  // clear the timer, when not pressed

    // Clear power-on reset flag 'Once' used as follows
    // if we want to reset the EEPROM memory back to default values we simply
    // need to press the micros RESET button twice in quick succession.
    if (Once) {
      // after 3 seconds we clear the RESET flag stored in EEPROM
      if (millis() >= 10000) {
        // reset the RESET once flag
        Once = false; EEPROM.write(EeAdd_Reset, 0); EEPROM.commit();
        Serial.println("RESET flag cleared");
      }
    }
  }
}

// ---------------------------------------------------------------------------------

void loopDelay(long zDelay) {
  // Called to run the main loop for a zDelay period in ms before returning.
  // Note the calling function must block being called from loop() to prevent stack
  // overflow conditions
  zDelay += millis(); // set the exit time
  while (millis() < zDelay) {
    loop();   // run tasks in the main loop
    yield();  // pass control to background functions
  }
}

// ---------------------------------------------------------------------------------

void RunPOST() {
  // Start-up initialisation processes, called from setup()
  if (TEST) {Serial.println("\n\nTEST mode ACTIVE\n");}

  // Take a quick battery reading
  readBattery();  // take a battery reading

  // Initialise the light sensor
  Serial.print("Light sensor... ");
  LightSensor.begin();
  Serial.println("initialised.");

  // Initialise the MPU6050 motion sensor
  Serial.println("Initialising MPU... ");
  initialiseMPU();
  
  if (TEST) {
    setFrameTask(-80);  // in TEST mode so force moving dot task
    Serial.println("Entering Report Mode");
  } else {
    EEPROM_Read();      // get default frame task from reset
    setFrameTask(100);  // set initial task to power-on burst
    Serial.println("EEPROM FrameNum = " + String(FrameNum));
  }
  Serial.println("\nPOST completed\n\n");
}

// ---------------------------------------------------------------------------------

void setDefaults() {
  // Load default values, which would normally be set at the start of your code.
  // This enables the variables to be reset at any time.
  AcAvgDiv = 0;           // dividing factor used in averaging
  AcAvgDivMax = 50;       // max dividing factor for averaging
  AcXAvg = 0;             // averaged AcX value to reduce noise
  AcXOff = 0;             // offset to be added to raw X-acceleration value
  AcXSum = 0;             // averaging accumulator
  AcYAvg = 0;             // averaged AcX value to reduce noise
  AcYOff = 0;             // offset to be added to raw Y-acceleration value
  AcYSum = 0;             // averaging accumulator
  AcZAvg = 0;             // averaged AcX value to reduce noise
  AcZOff = 0;             // offset to be added to raw Z-acceleration value
  AcZSum = 0;             // averaging accumulator
  BAK_Blu = 0;            // current background blue colour
  BAK_Grn = 0;            // current background green colour
  BAK_Red = 0;            // current background red colour
  BatAvg = 4095;          // preload averaging function with high value
  BatSum = 72320;         // preload averaging sum with 20x max value
  BatVol = 0;             // instantaneous battery voltage
  Bright = EEPROM_DefBright;  // FastLED brightness
  cmdMode = ' ';          // command mode
  cmdSgn = 1;             // cmdVal sign, 1 or -1
  cmdType = ' ';          // command mode type
  cmdTxt = "";            // text sent to the serial port relating to cmdMode
  cmdVal = 0;             // value associated with a cmdType
  Col11 = 4;              // ambient max colour
  Col34 = 3;              // ambient 3/4 colour
  Col12 = 2;              // ambient 1/2 colour
  Col14 = 1;              // ambient 1/4 colour
  Exec = 0;               // timer/flag controls Exec overide mode
  FrameBright = 16;       // brightness factor used in tasks
  FrameBuild = true;      // if == true then run a frame building task
  FrameCnt = 0;           // Main Task general counter/timer
  FrameDel = 0;           // Main task delay counter, if > 0 holds off tasks
  FrameMax = 17;          // max of FrameTask
  FrameMin = 1;           // min of FrameTask
  FrameNext = 0;          // next MainTask to be performed after a delay
  FrameNum = EEPROM_DefTask;  // default FrameTask
  FramePeriod = 30;       // frame time period in ms, defaulty = 20
  FramePrdLL = 18;        // frame time period lower limit in ms, fastest
  FramePrdUL = 60;        // frame time period upper limit in ms, slowest
  FrameTask = -1;         // task pointer for frame construction
  GfxP0 = 1;              // points to first element of current face, bottom right corner
  GyAvgDiv = 0;           // dividing factor used in averaging
  GyCal = 250;            // counter used to determine gyro calibration period
  GyX = 0;                // MPU raw X-gyroscope value
  GyXAvg = 0;             // averaged GyX value to reduce noise
  GyXOff = 0;             // offset to be added to raw X-gyroscope value
  GyXOffCnt = 0;          // offset counter for gyro centering
  GyXSum = 0;             // averaging accumulator
  GyY = 0;                // MPU raw Y-gyroscope value
  GyYAvg = 0;             // averaged GyY value to reduce noise
  GyYOff = 0;             // offset to be added to raw Y-gyroscope value
  GyYOffCnt = 0;          // offset counter for gyro centering
  GyYSum = 0;             // averaging accumulator
  GyZ = 0;                // MPU raw Z-gyroscope value
  GyZAvg = 0;             // averaged GyZ value to reduce noise
  GyZOff = 0;             // offset to be added to raw Z-gyroscope value
  GyZOffCnt = 0;          // offset counter for gyro centering
  GyZSum = 0;             // averaging accumulator
  I2CError = false;       // used to detect I2C transmission errors
  LED_Blink = false;      // flag toggled to create blinking effects
  LED_Blu = 0;            // current blue colour
  LED_Face = 0;           // cube face 0 - 4, default = 0, top face
  LED_Grn = 0;            // current green colour
  LED_Red = 0;            // current red colour
  LED_Cnt = 0;            // general counter used in LED tasks
  LED_Del = 0;            // LED Task delay, adds pauses into the processes
  LED_Inc = 1;            // general increment used in LED tasks
  LED_Max = 32;           // Max brightness level used by tasks
  LED_Nop = false;        // == true to block LED tasks, default = false
  LED_ON = false;         // built-in LED ON flag
  LEDshow = false;        // LED show flag for mouth LEDs
  LED_SubTask = 0;        // LED sub-task pointer
  LED_SwDwn = false;      // == true if a button switch has been pressed
  LED_Task = 0;           // LED task pointer
  LED_Val = 0;            // values used in LED tasks
  luxCnt = 0;             // lux counter
  luxMode = 0;            // lux mode trigered by lux sensor
  luxTask = 0;            // lux task
  luxState = true;        // true if lux >= 5
  luxLastState = true;    // previous luxState
  Mem_Blu = 0;            // memory of current blue colour
  Mem_Grn = 0;            // memory of current green colour
  Mem_Red = 0;            // memory of current red colour
  micAvg = 1650;          // average microphone value
  micAvgCnt = 0;          // counter for average microphone value
  micCnt = 0;             // delay counter usedin squelsh function
  micFast = false;        // == true to enable audio fast sampling
  micInc = 100;           // previous microphone value
  micLvl= 50;             // level tracker for squelch function
  micMax = 50;            // max micVol for scaling
  micPk = 0;              // peak measurement during 10ms cycle
  micVol = 0;             // unipolar microphone volume
  MoodB = EEPROM_DefMoodB;  // mood light blue colour
  MoodG = EEPROM_DefMoodG;  // mood light green colour
  MoodR = EEPROM_DefMoodR;  // mood light red colour
  MPUCal = false;         // == true during calibration
  MPU_Init = true;        // == true if MPU read is enabled, default = true
  MPUrep = false;         // MPU report flag
  Once = true;            // flag used with RESET function, only true for a while
  PAUSE = false;          // flag set by user, if set stops animations being run, default = false
  PitchAcc = 0.0;         // pitch angle in degrees base on accelerometer
  PitchGyr = 0.0;         // pitch angle in degrees base on gyro + acc
  PitchGyrLast = 0;       // previous PitchGyrInt value
  REBOOT = false;         // set true when reboot is in progress to stop all tasks
  Report = 0;             // if > 0 value selects serial port reporting mode
  ReportDel = 0;          // if > 0 then skip report cycles
  ResetCnt = 0;           // timer used to invoke soft RESET
  RGB_Blu = 5;            // built-in RGB LED blue OFF state, dims to off
  RGBshow = false;        // if == true then trigger a FastLED show for the RGB LED
  RndCol = 255;           // previous random colour
  RollGyrLast = 0;        // previous PitchGyrInt value
  StaticCnt = 0;          // counter used in movement detection
  sw0Cnt = 0;             // button switch counter
  sw0DwnTime = 0;         // button pressed down time
  sw0LastState = HIGH;    // previous state of button switch, HIGH/LOW
  SW0_Nop = false;        // == true to block SW0 read tasks, default = false
  sw0State = HIGH;        // state of read button switch pin
  sw0Timer = 0;           // timer used to detemine button sequences
  sw1Cnt = 0;             // button switch counter
  sw1LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw1Mode = 0;            // switch mode flag
  SW1_Nop = false;        // == true to block SW1 read tasks, default = false
  sw1State = HIGH;        // state of read button switch pin
  sw1Timer = 0;           // timer used to detemine button sequences
  sw2Cnt = 0;             // button switch counter
  sw2LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw2Mode = 0;            // switch mode flag
  SW2_Nop = false;        // == true to block SW2 read tasks, default = false
  sw2State = HIGH;        // state of read button switch pin
  sw2Timer = 0;           // timer used to detemine button sequences
  t0 = 0;                 // microsecond timer used for code speed checks
  t1 = 0;                 // microsecond timer used for code speed checks
  Task10ms = 0;           // 10ms loop timer tasks
  Task20ms = 0;           // 20ms loop timer tasks
  Task200ms = 0;          // 200ms loop timer tasks
  Task_Del = 0;           // Delay allows FastLED RMT function
  TaskDel_ms = 10;        // Adjustable delay for debug, default = 10
  USBPwr = false;         // true if POST detects USB power voltage
  WaitUp = false;         // true if task is to wait for button release
  WhiteLight = EEPROM_DefLight; // white light value
  YawAcc = 0.0;           // yaw angle in degrees based on accelerometers
  YawGyr = 0.0;           // yaw angle in degrees based on gyro + acc
}

// ---------------------------------------------------------------------------------

void synchLoopTimers() {
  // Reset main loop timers.
  // Phase offsets are introduced to prevent simultaneous task triggering.
  // The timers must be set to millis() or a value less than millis(), in order to
  // work correctly.
  while (millis() < 20) {yield();}

  ExecTimer = millis();       // 100ms Exec timer
  FrameTimer = millis() - 2;  // variable frame timer offset by 18 initially
  next10ms = millis() - 4;    // 10ms loop timer
  next20ms = next10ms - 2;    // 20ms loop timer offset by 4
  next200ms = next10ms - 4;   // 200ms loop timer offset by 6
  nextSec = millis() - 10;    // 1 sec loop timer offset by 8
}

// --------------------------------------------------------------------------------
