
// ---------------------------------------------------------------------------------

void getHeadAngle() {
  // converts the present head servo PWM to an angle
  HeadAngle = map(SPP[10],SLL[10],SUL[10],30,150);
}

// ---------------------------------------------------------------------------------

void getSpeedJXY(int zJXY) {
  // Joystick vector JoyV varies from 12 - 127 which is mapped into a speed profile
  // low to medium demands varies speed between SpdMin and SpdTop, where SpdTop = SpdMid
  // higher demands increase SpdTop up to SpdMax, extending the SpdTop speed range
  // reduced demand lowers SpdTop towards SpdMid
  // Gear changes from 1 - 5 to represent speed from SpdMin - SpdMax
  if (zJXY >= 120) {
    // high demand so increase speed progressively
     SpeedCnt++; if (SpeedCnt >= 3) {SpeedCnt = 0; SpdTop++;}
  } else if (zJXY <= 40) {
    // low demand so decrease speed progressively
     SpeedCnt++; if (SpeedCnt >= 3) {SpeedCnt = 0; SpdTop--;}
  } else {SpeedCnt = 0;}
  SpdTop = min(SpdTop,SpdMax); // limit maximum top speed
  SpdTop = max(SpdTop,SpdMid); // limit minimum top speed
  
  // limit rate of change of speed to avoid surges in demand
  SpeedTgt = map(zJXY,12,127,SpdMin,SpdTop);
  if (Speed < SpeedTgt) {Speed += 1 + ((SpeedTgt - Speed)/4);}
  else if (Speed > SpeedTgt) {Speed -= 1 + ((Speed - SpeedTgt)/4);}
}

// ---------------------------------------------------------------------------------

void getSteps() {
  // scans the target values for the largest difference and returns it in steps
  Steps = 4; int zD;
  for (int zS = 0; zS < 16; zS++) {
    zD = abs(STG[zS] - SPP[zS]); if (zD > Steps) {Steps = zD;} 
  }
//  PrintTx += "\nSteps = " + String(Steps) + "\n";
}

// ---------------------------------------------------------------------------------

void getServoAngle(int zCh) {
  // Converts the servo channel zCh PWM to an angle value in degrees
  int zA0 = 30; int zA1 = 150;
  switch(zCh) {
    case SL1: zA0 = 150; zA1 =  30; break;
    case SL2: zA0 = 150; zA1 =  30; break;
    case SL3: zA0 = 150; zA1 =  30; break;
    case SL4: zA0 =  70; zA1 = 110; break;
    case SR4: zA0 =  70; zA1 = 110; break;
  }

  // non-linearity of servo is corrected
  if (SPP[zCh] >= SRV[zCh]) {Angle = map(SPP[zCh],SRV[zCh],SUL[zCh],90,zA1);}
  else {Angle = map(SPP[zCh],SLL[zCh],SRV[zCh],zA0,90);}
  
//  Angle = map(SPP[zCh],SLL[zCh],SUL[zCh],zA0,zA1);
}

// ---------------------------------------------------------------------------------

void GoToRest() {
  // called at start when servo positions are unknown
  // immediately drives servos to default PWM rest positions
  // waits one second then removes pulse and OE
  for (int zS = 0; zS < 16; zS++) {
    I2Cpwm.setPWM(zS, 0, SRV[zS]);
  }
  digitalWrite(Pin_OE, LOW); // enable all PWM outputs
  AtRest = true; AutoOffTimer = 200; // set 2s timeout
}

// ---------------------------------------------------------------------------------

void Head_Engine() {
  // head movement uses a separate timer so that the speed of movement can be set
  // independantly  
  if (HeadTOF) {
    // timer overflow has occured so wait for micros() timer to overflow too
    if (micros() < lastHeadMicros) {HeadTOF = false;}  // back to normal, clear the flag
    else {return;}  // return to do other things in the mean time
  }
  headMicros = micros() + headInterval; // set next time point
  lastHeadMicros = micros(); if (headMicros < micros()) {HeadTOF = true;} // time point has over-flowed, so set overflow flag
  if (Pause) {return;}  // pause flag causes movement to be put on hold
  if (Head > 0) {
    if (servoOE) {SetServoOE(0);} // turn the servos ON
    if (HeadStep > 1) {
      SPP[SH0] = STG[SH0] - (((STG[SH0] - SSP[SH0])*HeadStep)/HeadSteps);
      I2Cpwm.setPWM(SH0, 0, SPP[SH0]); AutoOffTimer = 200;
      HeadStep--;
    } else if (HeadStep == 1) {
      SPP[SH0] = STG[SH0]; SSP[SH0] = STG[SH0];
      I2Cpwm.setPWM(SH0, 0, SPP[SH0]); AutoOffTimer = 200;
      HeadStep = 0;
    } else {
      Head = 0;
    }
  }
}

// ---------------------------------------------------------------------------------

void MemSet(int zR0,int zR1,int zR2,int zR3,int zR4,int zL0,int zL1,int zL2,int zL3,int zL4,int zSt) {
  // function is called to load a set of angle values into the move array
  // the value of MemRow must be set before calling this function, normally = 0
  // at the end MemRow will be incremented to point at the next row so subsequent
  // calls will load the memory sequentially
  int zP = MemRow * MemWidth; MemRow ++; MemMax = MemRow;
  if ((!Trans) || (zR0 < 0)) {
    // load in angles normally
    // this is ALWAYS true for command rows which must not be transposed
    Mem[zP] = zR0; zP++;
    Mem[zP] = zR1; zP++;
    Mem[zP] = zR2; zP++;
    Mem[zP] = zR3; zP++;
    Mem[zP] = zR4; zP++;
    
    Mem[zP] = zL0; zP++;
    Mem[zP] = zL1; zP++;
    Mem[zP] = zL2; zP++;
    Mem[zP] = zL3; zP++;
    Mem[zP] = zL4; zP++;
  } else {
    // transpose the loading of angles
    Mem[zP] = 180 - zL0; zP++;
    Mem[zP] = zL1; zP++;
    Mem[zP] = zL2; zP++;
    Mem[zP] = zL3; zP++;
    Mem[zP] = 180 - zL4; zP++;
    
    Mem[zP] = 180 - zR0; zP++;
    Mem[zP] = zR1; zP++;
    Mem[zP] = zR2; zP++;
    Mem[zP] = zR3; zP++;
    Mem[zP] = 180 - zR4; zP++;
  }
  Mem[zP] = zSt;
}

// ---------------------------------------------------------------------------------

void MoveBkd() {
  // called from move state machine
  if (MoveState != 5) {
    // just been asked to move backwards
    MoveState = 5; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    if (Gear <= 2) {
      MoveToSkiBkd();
    } else {
      MoveToWalkBkd();
    } MoveState = 5; LEDMthTask = 20;
  }
  if (JoyD == 5) {
    // demand is still there so keep moving at demanded speed
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
  } else {
    // -Y demand has dropped so return to ready state
    MoveState = -1;
  }
}

// ---------------------------------------------------------------------------------

void MoveFwd() {
  // called from move state machine
  if (MoveState != 1) {
    // just been asked to move forward
    MoveState = 1; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    if (Gear <= 2) {
      MoveToSkiFwd();
    } else {
      MoveToWalkFwd();
    } 
  }
  if (JoyD == 1) {
    // demand is still there so keep moving at demanded speed
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
  } else {
    // Y demand has dropped so return to ready state
    MoveState = -1;
  }
}

// ---------------------------------------------------------------------------------

void MoveFwdLft() {
  // move forward with a bias twist towards the left  
  if (MoveState != 8) {
    // just been asked to move forward towards the left
    MoveState = 8; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    if (Gear <= 2) {
      MoveToSkiFwdLft();
    } else {
      MoveToWalkFwdLft();
    } 
  }
  if (JoyD == 8) {
    // demand is still there so keep moving at demanded speed
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
  } else {
    // Y demand has dropped so return to ready state
    MoveState = -1;
  }
}

// ---------------------------------------------------------------------------------

void MoveFwdRht() {
  // move forward with a bias twist towards the right  
  if (MoveState != 2) {
    // just been asked to move forward towards the left
    MoveState = 2; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    if (Gear <= 2) {
      MoveToSkiFwdRht();
    } else {
      MoveToWalkFwdRht();
    }
  }
  if (JoyD == 2) {
    // demand is still there so keep moving at demanded speed
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
  } else {
    // Y demand has dropped so return to ready state
    MoveState = -1;
  }
}

// ---------------------------------------------------------------------------------

void MoveHeadTo(int zAngle, int zSpeed) {
  // move the head to specific angle  
    SetStartsToPPs();
    setHeadTgt(zAngle); HeadSteps = abs(STG[SH0] - SPP[SH0]); HeadStep = HeadSteps;
    setHeadSpeed(zSpeed); Head = 1;
}

// ---------------------------------------------------------------------------------

bool MoveMemTgtLoader(int zRow) {
  // load Mem[zRow+] angle target values until a row with all zeros is encountered
  // or zR >= MemMax, at which point we reset to zero
  // This function also detects the following embedded commands:
  //  0 - 'Blank' line, ignore
  //  1 - goto a specified row, branch instruction used to loop the move
  //  2 - set the speed to a new value
  //  3 - merged rows where step = 0, used for turning by JoyX
  Start:
  int zRowRef = zRow; // remember row value passed to this function
  // only report memory load if connected to the MoveTrainer
  if (LEDTaskPnt == 3) {PrintTx += "MemTgt " + String(zRow) + "\n";}
  bool zVal = false;  // flag used to determine if any targets are set
  WalkMem = zRow + 1; // set the pointer for next time
  zRow *= MemWidth;
  
  // look for command lines, wiht first value < 0
  if (Mem[zRow] < 0) {
    int zCmd = Mem[zRow + 1]; // get the command type
    // this is a command row, so decode it
    if (zCmd == 0) {
      // Blank line, so skip over it
      if (MTM_Step) {return zVal;}  // single stepping so don't go to next row
      if ((zRowRef + 1) >= MemMax) {return zVal;} // we are at the last row
      zRow = zRowRef + 1; goto Start;
    }
    if (zCmd == 1) {
      // this is a goto command, so load revised row pointer and loop back
      if (WalkMode < 1) {return zVal;}  // don't branch if WalkMode == 0 one shot mode
      zRow = Mem[zRow + 2]; moveCycleCnt++; goto Start;
    }
    if (zCmd == 2) {
      // this is a speed command, so load revised speed and loop back for next row
      Speed = Mem[zRow + 2]; // set revised speed
//      PrintTx += "Speed " + String(Speed) + "\n";
      if (MTM_Step) {return zVal;}  // single stepping so don't go to next row
      if ((zRowRef + 1) >= MemMax) {return zVal;} // we are at the last row
      zRow = zRowRef + 1; goto Start;
    }
  }
  
  WalkCnt += WalkInc; // adjust the walk movement counter
  
  // load angles into servo targets if > 0
  for (int zP = 0; zP < 10; zP++) {
    if (Mem[zRow + zP] > 0) {
      // servo angle is non-zero so use it
      int zA = Mem[zRow + zP];
      setServoTgt(zA,ChMap[zP]); zVal = true;
    }
  }

  // separate steps from rate profile
  // you must set the speed after loading a row as steps and profile can change
  zRow += 10; 
  LegSteps = Mem[zRow]; RateProf = 0;           // constant speed
//  PrintTx += "LegSteps " + String(LegSteps) + "\n";
  if (LegSteps > 600) {LegSteps -= 600; RateProf = 3;}  // acceleration then deceleration
  if (LegSteps > 400) {LegSteps -= 400; RateProf = 2;}  // deceleration
  if (LegSteps > 200) {LegSteps -= 200; RateProf = 1;}  // acceleration
  LegStep = LegSteps;
//  PrintTx += "RateProf " + String(RateProf) + "\n";
  
  if (Mem[zRow] > 0) {zVal = true;}  // steps set for this row so flag as ok
  return zVal;
}

// ---------------------------------------------------------------------------------

void MoveStepLeft() {
  // move by stepping to the left
  if (MoveState != 17) {
    // just been asked to step left
    MoveState = 17; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    MoveToStepLeft();
  }
  if (JoyD == 7) {
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
//    Serial.println(Steer);
  } else {
    // Y demand has dropped so return to ready state
    MoveState = -1;
  }
}

// ---------------------------------------------------------------------------------

void MoveStepRight() {
  // move by stepping to the right
  if (MoveState != 13) {
    // just been asked to move
    MoveState = 13; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    MoveToStepRight();
  }
  if (JoyD == 3) {
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
//    Serial.println(Steer);
  } else {
    // Y demand has dropped so return to ready state
    MoveState = -1;
  }
}

// ---------------------------------------------------------------------------------

void MoveTgtLoader(int zR0,int zR1,int zR2,int zR3,int zR4,int zL0,int zL1,int zL2,int zL3,int zL4) {
  // loads PWM target values into the target array based on servo angles
  // if the specified angle == 0 then no action is taken
  if (zR0 > 0) {setServoTgt(zR0,SR0);}
  if (zR1 > 0) {setServoTgt(zR1,SR1);}
  if (zR2 > 0) {setServoTgt(zR2,SR2);}
  if (zR3 > 0) {setServoTgt(zR3,SR3);}
  if (zR4 > 0) {setServoTgt(zR4,SR4);}
  if (zL0 > 0) {setServoTgt(zL0,SL0);}
  if (zL1 > 0) {setServoTgt(zL1,SL1);}
  if (zL2 > 0) {setServoTgt(zL2,SL2);}
  if (zL3 > 0) {setServoTgt(zL3,SL3);}
  if (zL4 > 0) {setServoTgt(zL4,SL4);}
  // we also set the start positions to save doing this elsewhere
  SetStartsToPPs();
}

// ---------------------------------------------------------------------------------

void MoveToAllAtRest() {
  // set up moves to retun all servos to rest position
  clearMem(); // prime memory loader

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,430);

  AtRest = true; HeadTask = 0; LEDTaskPnt = -1;
  Move_As_Memory(0,0,50); // row,mode,speed
//  Serial.println("MoveToRest");
}

// ---------------------------------------------------------------------------------

void MoveToBowStretch() {
  // initiate a single bow and stretch move
  clearMem(); // prime memory loader

  // this works fine for all feet type tested
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,420);
  MemSet( 90, 30,180,126, 90, 90, 30,180,126, 90,230);
  MemSet( 90, 30,180,126, 90, 90, 30,180,126, 90, 30);
  MemSet( 90, 30,180,150, 90, 90, 30,180,150, 90,415);
  MemSet( 90, 30,180,150, 90, 90, 30,180,150, 90, 30);
  MemSet( 90, 30,180,115, 90, 90, 30,180,115, 90,430);
  MemSet( 90, 30,180,115, 90, 90, 30,180,115, 90, 30);
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 30);
  MemSet( 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,430);
  MemSet( 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 30);
  MemSet( 90,137, 30,127, 90, 90,137, 30,127, 90,425);
  MemSet( 90,137, 30,127, 90, 90,137, 30,127, 90,430);
  MemSet( 90,137, 30, 79, 90, 90,140, 30, 79, 90,430);
  MemSet( 90,137, 30,115, 90, 90,142, 30,115, 90,420);
  MemSet( 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,230);
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,425);

  eyeR = 0; eyeG = 0; eyeB = 10; LEDTaskPnt = -2; LEDMthTask = 20;
  HeadTask = 0; // centralise head

  Move_As_Memory(0,0,40); // row,mode,speed
//  Serial.println("MoveToBowStretch");
}

// ---------------------------------------------------------------------------------

void MoveToFeetTwist(int zA, int zSpeed) {
  // move feet in oposite directions raising the droid
  // the angle zA is a relative figure centred on 90
  setServoTgt(90 - zA, SR0); setServoTgt(90 + zA, SL0);
  SetStartsToPPs();
  getSteps(); LegSteps = Steps; LegStep = Steps;
  setLegSpeed(zSpeed); Walk = 1;
}

// ---------------------------------------------------------------------------------

void MoveToReady(byte zM) {
  // bend both legs into the ready posture
  // zM = 0 only move legs
  // zM = 1 set head in random mode
  clearMem(); // prime memory loader

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,430);

  AtRest = false;
  if (zM > 0) {
    WiFiRestCnt = WiFiRestDef; HeadTask = 10;
    eyeB = 10; eyeG = 0; eyeR = 0; LEDTaskPnt = -2;
  }
  Move_As_Memory(0,0,50); // row,mode,speed
  AtRest = false; // clear droid at rest flag
  Trans = !Trans; // toggle the transpose flag so that movement starts differently each time
//  Serial.println("MoveToReady");
}

// ---------------------------------------------------------------------------------

void MoveToRndBodyPitch(int zSpeed) {
  // determine a random body pitch and set up movement  
    getServoAngle(SR3);
    if (Angle < 90) {
      Angle = 90 + (5 * random(11));  // set an angle between
    } else {
      Angle = 40 + (5 * random(11));  // set an angle between 
    }
    setServoTgt(Angle, SR3); setServoTgt(Angle, SL3);
    SetStartsToPPs();
    getSteps(); LegSteps = Steps; LegStep = Steps;
//    PrintTx += "LegStep = " + String(LegSteps) + "\n";
//    PrintTx += String(STG[SR3]) + "\t" + String(STG[SL3]) + "\n";
    setLegSpeed(zSpeed); Walk = 1;
}

// ---------------------------------------------------------------------------------

void MoveToRndHeadPos(int zSpeed) {
  // determine a random head position and set things up  
    getHeadAngle(); // get the current head angle
    if (HeadAngle < 90) {
      HeadAngle = 95 + (5 * random(12));  // set an angle between 95 to 150
    } else {
      HeadAngle = 85 - (5 * random(12));  // set an angle between 30 to 85
    }
    SetStartsToPPs();
    setHeadTgt(HeadAngle); HeadSteps = abs(STG[SH0] - SPP[SH0]); HeadStep = HeadSteps;
    setHeadSpeed(zSpeed); Head = 1;
}

// ---------------------------------------------------------------------------------

void MoveToSquat() {
  // move to squat position, normal when critical power level is reached
  clearMem(); // prime memory loader

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 37,180,139, 90, 90, 37,180,139, 90,430);


  setHeadTgt(90); HeadSteps = 30; HeadStep = Steps;
  setHeadSpeed(40); Head = 1;

  Move_As_Memory(0,0,50); // row,mode,speed
//  Serial.println("MoveToSquat");
}

// ---------------------------------------------------------------------------------

void MoveToSkiBkd() {
  // initiate a continuous skiing walking backward move
  clearMem(); // prime memory loader

  // smooth feet
//  MemSet( 99, 70,130,114, 99, 99, 70,130,114, 99,418);
//  MemSet(100, 70,130,114, 99,108, 72,130,114, 99,  9);
//  MemSet( 91, 86,121,121, 87,101, 72,130,113, 86, 16);
//  MemSet( 83,106,112,132, 75, 84, 70,130,117, 81,420);
//  MemSet( 83,103,112,129, 75, 81, 70,130,114, 81,  3);
//  MemSet( 72, 98,121,133, 75, 81, 70,130,115, 81, 11);
//  MemSet( 72, 70,130,114, 81, 81, 70,130,114, 81, 30);
//  MemSet( 79, 70,130,115, 94, 89, 88,121,123, 93,418);
//  MemSet( 96, 70,130,114, 99, 97,106,112,129,105, 18);
//  MemSet( 99, 70,130,114, 99, 97,106,112,129,105,  3);
//  MemSet( 99, 70,130,114, 99,108,100,121,132,105, 11);
//  MemSet( 99, 70,130,114, 99,108, 70,130,114, 99, 30);
//  MemSet( -1,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0);

  // felt feet
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,420);
  MemSet( 99, 70,130,116, 99,102, 70,130,116, 99,620);
  MemSet(100, 69,132,120, 99,117, 71,132,120, 99,  5);
  MemSet(100, 79,135,138, 99,117, 59,158,131,105,616);
  MemSet( 96, 79,135,138, 99,100, 53,158,131,105,  5);
  MemSet( 80, 79,135,132, 83, 84, 53,158,125, 93,630);
  MemSet( 63, 83,135,135, 83, 80, 53,158,128, 93,  5);
  MemSet( 63, 65,162,138, 79, 80, 62,155,133, 87,215);
  MemSet( 63, 54,158,128, 75, 80, 76,141,138, 81,415);
  MemSet( 78, 53,152,122, 75, 80, 76,141,135, 81,  5);
  MemSet(100, 53,158,125, 87,104, 87,131,130, 97,630);
  MemSet(100, 53,158,125, 87,117, 87,135,132, 97,  5);
  MemSet(100, 62,155,137, 93,117, 67,162,142,101,215);
  MemSet(100, 76,141,141, 99,117, 56,162,133,105,415);
  MemSet(100, 76,141,138, 99,102, 53,152,125,105,  5);
  MemSet( -1,  1,  5,  0,  0,  0,  0,  0,  0,  0,  0);



  SpdMin = 25; SpdMax = 45;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToSkiBkd");
}

// ---------------------------------------------------------------------------------

void MoveToSkiFwd() {
  // initiate a continuous skiing walking forward move
  clearMem(); // prime memory loader

  // smooth feet
//  MemSet( 99, 70,130,112, 99,108, 70,130,112, 99,418);
//  MemSet( 99, 70,130,112, 99,108,100,121,130,105,430);
//  MemSet( 99, 70,130,110, 99, 97,106,112,125,105, 11);
//  MemSet( 96, 70,130,110, 99, 97,106,112,125,105,403);
//  MemSet( 79, 70,130,111, 94, 89, 88,121,119, 93,418);
//  MemSet( 72, 70,130,112, 81, 81, 70,130,112, 81,418);
//  MemSet( 72,100,121,130, 75, 81, 70,130,112, 81,430);
//  MemSet( 83,106,112,125, 75, 81, 70,130,110, 81, 11);
//  MemSet( 83,106,112,125, 75, 84, 70,130,110, 81,403);
//  MemSet( 91, 88,121,119, 87,101, 70,130,111, 86,418);
//  MemSet( 99, 70,130,112, 99,108, 70,130,112, 99,418);
//  MemSet( 99, 70,130,112, 99,108,100,121,130,105,430);
//  MemSet( 99, 70,130,110, 99, 97,106,112,125,105, 11);
//  MemSet( -1,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0);

  // felt feet
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,430);
  MemSet( 99, 70,130,112, 99,112, 70,130,112, 99,422);
  MemSet( 99, 70,130,112, 99,112,100,121,130,105,430);
  MemSet( 99, 70,130,110, 99, 99,106,112,125,105,  5);
  MemSet( 81, 47,148,103, 74, 81, 64,142,117, 81,430);
  MemSet( 68, 70,130,112, 81, 81, 70,130,112, 81, 23);
  MemSet( 72,100,121,130, 75, 81, 70,130,112, 81,430);
  MemSet( 81,106,112,125, 75, 81, 70,130,110, 81,  5);
  MemSet( 99, 64,142,117, 99, 99, 47,148,103,106,430);
  MemSet( 99, 70,130,112, 99,112, 70,130,112, 99, 23);
  MemSet( 99, 70,130,112, 99,108,100,121,130,105,430);
  MemSet( 99, 70,130,110, 99, 99,106,112,125,105,  5);
  MemSet( -1,  1,  4,  0,  0,  0,  0,  0,  0,  0,  0);
  

  SpdMin = 25; SpdMax = 100;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToSkiFwd");
}

// ---------------------------------------------------------------------------------

void MoveToSkiFwdLft() {
  // initiate a continuous skiing walking forward towards the left
  clearMem(); // prime memory loader
  Trans = false;  // we don't transpose turning movements

  // works with all feet
  MemSet( 72, 70,130,112, 81, 81, 70,130,112, 81,418);
  MemSet( 72,100,121,130, 75, 81, 70,130,112, 81,430);
  MemSet( 83,106,112,127, 75, 81, 70,130,112, 81, 11);
  MemSet( 83,106,112,127, 75, 84, 70,130,112, 81,403);
  MemSet( 90, 65,136,112, 82, 91, 67,130,111, 88, 30);
  MemSet( 91, 65,141,116, 87,101, 62,140,116, 86,410);
  MemSet( 99, 70,130,112, 99,108, 70,130,112, 99,413);
  MemSet( 99, 70,130,112, 99,108, 82,133,124,105,412);
  MemSet( 99, 70,130,110, 99, 97, 90,126,115,105, 11);
  MemSet( 96, 70,130,110, 99, 97, 82,138,123,105,408);
  MemSet( 79, 70,130,111, 94, 89, 88,121,119, 93,417);
  MemSet( 72, 70,130,112, 81, 81, 70,130,112, 81,418);
  MemSet( 72,100,121,130, 75, 81, 70,130,112, 81,430);
  MemSet( 83,106,112,125, 75, 81, 70,130,110, 81, 11);
  MemSet( -1,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0);

  SpdMin = 25; SpdMax = 70;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToSkiFwdLft");
}

// ---------------------------------------------------------------------------------

void MoveToSkiFwdRht() {
  // initiate a continuous skiing walking forward towards the left
  clearMem(); // prime memory loader
  Trans = false;  // we don't transpose turning movements

  // works with all feet
  MemSet( 99, 70,130,112, 99,108, 70,130,112, 99,418);
  MemSet( 99, 70,130,112, 99,108,100,121,130,105,430);
  MemSet( 99, 70,130,112, 99, 97,106,112,127,105, 11);
  MemSet( 96, 70,130,112, 99, 97,106,112,127,105,403);
  MemSet( 89, 67,130,111, 92, 90, 65,136,112, 98, 30);
  MemSet( 79, 62,140,116, 94, 89, 65,141,116, 93,410);
  MemSet( 72, 70,130,112, 81, 81, 70,130,112, 81,413);
  MemSet( 72, 82,133,124, 75, 81, 70,130,112, 81,412);
  MemSet( 83, 90,126,115, 75, 81, 70,130,110, 81, 11);
  MemSet( 83, 82,138,123, 75, 84, 70,130,110, 81,408);
  MemSet( 91, 88,121,119, 87,101, 70,130,111, 86,417);
  MemSet( 99, 70,130,112, 99,108, 70,130,112, 99,418);
  MemSet( 99, 70,130,112, 99,108,100,121,130,105,430);
  MemSet( 99, 70,130,110, 99, 97,106,112,125,105, 11);
  MemSet( -1,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0);

  SpdMin = 25; SpdMax = 70;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToSkiFwdRht");
}

// ---------------------------------------------------------------------------------

void MoveToStepLeft() {
  // initiate a continuous stepping left walking move
  clearMem(); // prime memory loader
  Trans = false;  // we don't transpose turning movements

  // tested on felt coated feet
    //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 19);
  MemSet(103, 70,130,112,100,109, 70,134,112,102,419);
  MemSet(103, 70,130,112,100,109, 41,174,128,106,408);
  MemSet( 97, 70,130,112, 96,114, 96, 92, 93,130, 25);
  MemSet( 72, 70,130,112, 72, 95, 68,142,118, 92, 30);
  MemSet( 53, 70,130,112, 72, 95, 68,142,118, 92,  3);
  MemSet( 53, 70,130,112, 63, 83, 68,142,118, 83,  9);
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 10);
  MemSet( -1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0);


  SpdMin = 25; SpdMax = 45;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToStepLeft");
}

// ---------------------------------------------------------------------------------

void MoveToStepRight() {
  // initiate a continuous stepping right walking move
  clearMem(); // prime memory loader
  Trans = false;  // we don't transpose turning movements

  // tested on felt coated feet
    //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 19);
  MemSet( 71, 70,134,112, 78, 77, 70,130,112, 80,419);
  MemSet( 71, 41,174,128, 74, 77, 70,130,112, 80,408);
  MemSet( 66, 96, 92, 93, 50, 83, 70,130,112, 84, 25);
  MemSet( 85, 68,142,118, 88,108, 70,130,112,108, 30);
  MemSet( 85, 68,142,118, 88,127, 70,130,112,108,  3);
  MemSet( 97, 68,142,118, 97,127, 70,130,112,117,  9);
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 10);
  MemSet( -1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0);


  SpdMin = 25; SpdMax = 45;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToStepRight");
}

// ---------------------------------------------------------------------------------

void MoveToSwayAngle(int zA, int zSpeed) {
  // set up hips and feet to move to a sway angle
  // servo map: SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4 
  MoveTgtLoader( zA,  0,  0,  0, zA, zA,  0,  0,  0, zA);  // load angles into target arrays
  getSteps(); LegSteps = Steps; LegStep = Steps;
  setLegSpeed(zSpeed); Walk = 1;
}

// ---------------------------------------------------------------------------------

void MoveToSwayJig() {
  // initiate a single sway and jig move
  clearMem(); // prime memory loader
  Trans = !Trans; // toggle the transpose flag so that movement starts differently each time

  // tested on felt coated feet
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 15);
  MemSet(105, 70,130,110,105,105, 72,126,108,105,435);
  MemSet(105, 70,130,144,105,105, 78,120,144,105, 34);
  MemSet(105, 70,130, 83,105,105, 70,130, 77,105, 61);
  MemSet(105, 70,130,110,105,105, 70,130,110,105, 27);
  MemSet(107, 70,130,106,105,107, 29,180,117,112, 41);
  MemSet(110, 70,130,106,105,110, 29,180,117,130,418);
  MemSet(107, 70,130,106,105,107, 29,180,117,112,418);
  MemSet(105, 70,130,110,105,105, 70,130,110,105, 41);
  MemSet( 75, 70,130,110, 75, 75, 70,130,110, 75,460);
  MemSet( 75, 73,130,150, 75, 75, 68,132,144, 75, 40);
  MemSet( 75, 70,130, 78, 75, 75, 70,130, 83, 75, 72);
  MemSet( 75, 70,130,110, 75, 75, 70,130,110, 75, 32);
  MemSet( 73, 29,180,117, 68, 73, 70,130,106, 75, 41);
  MemSet( 70, 29,180,117, 50, 70, 70,130,106, 75,418);
  MemSet( 73, 29,180,117, 68, 73, 70,130,106, 75,418);
  MemSet( 75, 70,130,110, 75, 75, 70,130,110, 75, 41);
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,430);


  eyeR = 0; eyeG = 0; eyeB = 10; LEDTaskPnt = -2; LEDMthTask = 20;
  HeadTask = 10; // move head randomly

  Move_As_Memory(0,0,40); // row,mode,speed
//  Serial.println("MoveToSwayJig");
}

// ---------------------------------------------------------------------------------

void MoveToTurnLeft() {
  // initiate a continuous turning left walking move
  clearMem();     // prime memory loader
  Trans = false;  // we don't transpose turning movements

   // tested on felt coated feet
 //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 20);
  MemSet( 70, 61,150,124, 84, 80, 59,152,125, 91, 15);
  MemSet( 70, 98,120,133, 85, 80, 59,152,125, 91, 25);
  MemSet( 90,101,120,137, 84, 90, 64,142,124, 84,415);
  MemSet( 90, 70,130,114, 90, 90, 70,130,114, 90,420);
  MemSet( -1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0);


  SpdMin = 25; SpdMax = 70;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToTurnLeft");
}

// ---------------------------------------------------------------------------------

void MoveToTurnRight() {
  // initiate a continuous turning right walking move
  clearMem(); // prime memory loader
  Trans = false;  // we don't transpose turning movements

   // tested on felt coated feet
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 20);
  MemSet(100, 59,152,125, 89,110, 61,150,124, 96, 15);
  MemSet(100, 59,152,125, 89,110, 98,120,133, 95, 25);
  MemSet( 90, 64,142,124, 96, 90,101,120,137, 96,415);
  MemSet( 90, 70,130,114, 90, 90, 70,130,114, 90,420);
  MemSet( -1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0);


  SpdMin = 25; SpdMax = 70;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToTurnRight");
}

// ---------------------------------------------------------------------------------

void MoveToWalkBkd() {
  // initiate a continuous backward walking move
  clearMem(); // prime memory loader

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
//  MemSet(102, 58,132,100, 87, 98, 75,113, 90, 92,432);
//  MemSet(102, 60,136,110, 87, 98, 34,180,119, 92, 36);
//  MemSet(102, 60,156,132, 87,100, 56,144,105, 92, 36);
//  MemSet( 91, 66,144,126, 82, 87, 56,144,105, 87, 36);
//  MemSet( 82, 75,113,100, 88, 78, 64,132,110, 93,432);
//  MemSet( 82, 34,180,119, 88, 78, 60,136,110, 93, 36);
//  MemSet( 82, 56,144,110, 88, 78, 60,156,132, 93, 36);
//  MemSet( 93, 51,142,105, 93, 89, 66,144,126, 98, 36);
//  MemSet( -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0);  // loop to row 0

   // tested on felt coated feet
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90,420);
  MemSet(102, 58,132,102, 87,101, 71,113, 92, 92,430);
  MemSet(102, 60,136,110, 87, 98, 34,180,119, 92,220);
  MemSet(102, 60,156,132, 87,103, 56,144,105, 92,425);
  MemSet( 87, 66,144,126, 82, 87, 56,144,105, 87,230);
  MemSet( 82, 73,113,100, 88, 78, 60,140,114, 93,432);
  MemSet( 82, 34,180,119, 88, 78, 60,136,110, 93,620);
  MemSet( 79, 56,144,110, 88, 78, 60,156,132, 93,625);
  MemSet( 90, 51,142,105, 90, 90, 66,144,126, 90,230);
  MemSet( -1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0);


  SpdMin = 25; SpdMax = 65;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToWalkBkd");
}

// ---------------------------------------------------------------------------------

void MoveToWalkFwd() {
  // initiate a continuous forward walking move
  clearMem(); // prime memory loader

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
//  MemSet(102, 58,132,104, 87, 98, 75,113, 92, 92,420);
//  MemSet(102, 51,136, 97, 87, 98, 34,180,116, 92, 41);
//  MemSet(100, 45,140, 97, 85, 96, 70,134,118, 90,436);
//  MemSet( 90, 45,140, 97, 90, 90, 62,138,112, 90, 10);
//  MemSet( 82, 69,118, 92, 88, 78, 58,132,104, 93,430);
//  MemSet( 82, 34,180,121, 88, 78, 51,136, 99, 93, 41);
//  MemSet( 84, 70,134,118, 90, 80, 45,140, 97, 95,436);
//  MemSet( 90, 62,138,112, 90, 90, 45,140, 97, 90, 10);
//  MemSet(102, 58,132,104, 87, 98, 69,118, 92, 92,430);
//  MemSet(102, 51,136,101, 87, 98, 34,180,123, 92, 41);
//  MemSet(100, 45,140, 97, 85, 96, 70,134,118, 90,436);
//  MemSet( -1,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0);  // loop to row 3

   // tested on felt coated feet
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet( 90, 70,130,110, 90, 90, 70,130,110, 90, 20);
  MemSet(106, 55,144,111,103,108, 56,147,113,109,630);
  MemSet(106, 55,144,111,103,108, 56,147,113,109,  8);
  MemSet(106, 55,144,111,103,107, 34,181,128,112,615);
  MemSet(102, 47,150,105, 95,104, 78,142,130,102,430);
  MemSet(102, 47,150,105, 95,104, 78,142,130,102,  8);
  MemSet( 90, 49,146,107, 90, 90, 81,138,132, 90,220);
  MemSet( 73, 37,152,102, 70, 75, 55,156,129, 75,420);
  MemSet( 73, 37,152,102, 70, 75, 55,156,129, 75,  8);
  MemSet( 73, 32,181,129, 68, 74, 55,144,112, 77,220);
  MemSet( 76, 74,150,134, 78, 78, 47,150,105, 80,425);
  MemSet( 76, 74,150,134, 78, 78, 47,150,105, 80,  8);
  MemSet( 90, 81,138,132, 90, 90, 49,146,107, 90,220);
  MemSet(105, 55,156,129,105,107, 37,152,102,110,420);
  MemSet(105, 55,156,129,105,107, 37,152,102,110,  8);
  MemSet(106, 55,144,112,103,107, 32,181,129,110,220);
  MemSet(102, 47,150,105,100,104, 74,150,134,102,425);
  MemSet(102, 47,150,105,100,104, 74,150,134,102,  8);
  MemSet( -1,  1,  5,  0,  0,  0,  0,  0,  0,  0,  0);





  SpdMin = 25; SpdMax = 60;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToWalkFwd");
}

// ---------------------------------------------------------------------------------

void MoveToWalkFwdLft() {
  // initiate a continuous forward walking move biased towards the left
  clearMem(); // prime memory loader
  Trans = false;  // we can't transpose a turning movement

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet(102, 58,132,104, 87, 98, 75,113, 92, 92,420);
  MemSet(102, 51,136,101, 87, 98, 34,180,120, 92, 41);
  MemSet( 97, 49,140,103, 85, 96, 63,110, 96, 90,436);
  MemSet( 90, 52,140,106, 90, 90, 55,138,107, 90, 10);
  MemSet( 82, 69,118, 92, 88, 78, 58,132,104, 93,430);
  MemSet( 82, 34,180,123, 88, 78, 51,136,101, 93, 41);
  MemSet( 84, 70,134,118, 90, 80, 45,140, 97, 95,436);
  MemSet( 90, 54,138,106, 90, 90, 54,140,108, 90, 10);
  MemSet(102, 58,132,104, 87, 98, 69,118, 92, 92,430);
  MemSet(102, 51,136,101, 87, 98, 34,180,123, 92, 41);
  MemSet( 97, 49,140,103, 85, 96, 63,110, 96, 90,436);
  MemSet( -1,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0);

  SpdMin = 25; SpdMax = 60;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToWalkFwdLft");
}

// ---------------------------------------------------------------------------------

void MoveToWalkFwdRht() {
  // initiate a continuous forward walking move biased towards the right
  clearMem(); // prime memory loader
  Trans = false;  // we can't transpose a turning movement

  //     SR0,SR1,SR2,SR3,SR4,SL0,SL1,SL2,SL3,SL4,Stp 
  MemSet(102, 58,132,104, 87, 98, 75,113, 92, 92,420);
  MemSet(102, 51,136,101, 87, 98, 34,180,120, 92, 41);
  MemSet(100, 45,140, 97, 85, 96, 70,134,118, 90,436);
  MemSet( 90, 54,140,108, 90, 90, 54,138,106, 90, 10);
  MemSet( 82, 69,118, 92, 88, 78, 58,132,104, 93,430);
  MemSet( 82, 34,180,123, 88, 78, 51,136,101, 93, 41);
  MemSet( 84, 63,110, 96, 90, 83, 49,140,103, 95,436);
  MemSet( 90, 55,138,107, 90, 90, 52,140,106, 90, 10);
  MemSet(102, 58,132,104, 87, 98, 69,118, 92, 92,430);
  MemSet(102, 51,136,101, 87, 98, 34,180,123, 92, 41);
  MemSet(100, 45,140, 97, 85, 96, 70,134,118, 90,436);
  MemSet( -1,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0);

  SpdMin = 25; SpdMax = 60;
  Move_As_Memory(0,1,SpdMin); // row,mode,speed
//  Serial.println("MoveToWalkFwdRht");
}

// ---------------------------------------------------------------------------------

void MoveTurnLeft() {
  // called from move state machine
  if (MoveState != 7) {
    // just been asked to turn left
    MoveState = 7; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    MoveToTurnLeft();
  }
  if (JoyD == 7) {
    // demand is still there so keep moving at demanded speed
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
  } else {
    // -X demand has dropped so return to ready state
    MoveToReady(1); loopWhileWalk();
    MoveState = 0;
  }
}

// ---------------------------------------------------------------------------------

void MoveTurnRight() {
  // called from move state machine
  if (MoveState != 3) {
    // just been asked to turn right
    MoveState = 3; LEDMthTask = 20;
    if (AtRest) {
      // at reast so we need to move to ready position
      MoveToReady(1); loopWhileWalk();
    }
    MoveToTurnRight();
  }
  if (JoyD == 3) {
    // demand is still there so keep moving at demanded speed
    getSpeedJXY(JoyV);
    setLegSpeed(Speed);
  } else {
    // Y demand has dropped so return to ready state
    MoveToReady(1); loopWhileWalk();
    MoveState = 0;
  }
}

// ---------------------------------------------------------------------------------

void setHeadSpeed(int zS) {
  // sets the head timer to match a speed of zS steps per second
  // max speed is 250 per second (4ms steps)
  headInterval = 1000000/zS;
  if (headInterval < 4000) {headInterval = 4000;} // limit to 250 steps / sec
  headMicros = micros() + 1000; // set head loop timer to trigger after 1ms
}

// ---------------------------------------------------------------------------------

void setHeadTgt(int zHA) {
  // sets the servo target value to match an angle of zHA°
  // ensure that zHA is within the limits set for this servo
  if (zHA < 30) {zHA = 30;}
  else if (zHA > 150) {zHA = 150;}
  STG[10] = map(zHA,30,150,SLL[10],SUL[10]);
}

// ---------------------------------------------------------------------------------

void setLegSpeed(int zS) {
  // sets the leg timer to match a speed of zS steps (degrees) per second
  // max speed is 180 degrees per second (5.5ms steps) servos can't achieve this
  // RateProf sets the type of timer profile to use
  // a profile consists of 10 speeds: Prof1[] = {200,155,120, 93, 72, 56, 44, 34, 26, 20}
  // where 20 represents normal speed and 200 is 10 times slower
  if (zS < 10) {zS = 10;} // limit minimum speed setting to 10
  walkInterval = 1000000/zS;  // default walk interval
  
//  if (RateProf > 0) {walkInterval = 50000/zS;}  // will be multiplied by 20 - 200 from profile
  if (walkInterval < 4000) {walkInterval = 4000;} // limit to 250 steps / sec
  walkMicros = micros() + 1000; // set walk loop timer to trigger after 1ms
}

// ---------------------------------------------------------------------------------

void setServoAng(int zA, int zCh) {
  // sets the servo value to match an angle of zA°
  // ensure that angle zA° is within the limits set for this servo
  int zAL = 30; int zAU = 170; // angle limits
  int zSL = 30; int zSU = 150; // servo calibration limits
  byte zM = 0;
  switch(zCh) {
    case SH0:           zAU = 150; break;
    
    case SL0: zAL = 10;            break;
    case SL1: zAL = 10;            zM = 1; break;
    case SL2:           zAU = 180; zM = 1; break;
    case SL3:           zAU = 150; zM = 1; break;
    case SL4: zAL = 70; zAU = 130; zSL = 70; zSU = 110; break;
    
    case SR0: zAL = 10;            break;
    case SR1: zAL = 10;            break;
    case SR2:           zAU = 180; break;
    case SR3:           zAU = 150; break;
    case SR4: zAL = 50; zAU = 110; zSL = 70; zSU = 110; break;
  }
  if (servoOE) {SetServoOE(0);} // turn the servos ON
  if (zA < zAL) {zA = zAL;} else if (zA > zAU) {zA = zAU;}

  // non-linearity of servo is corrected
  if (zM < 1) {
    if (zA >= 90) {
      zA = map(zA,90,zSU,SRV[zCh],SUL[zCh]);
    } else {
      zA = map(zA,zSL,90,SLL[zCh],SRV[zCh]);
    }
  } else {
    if (zA >= 90) {
      zA = map(zA,90,zSU,SRV[zCh],SLL[zCh]);
    } else {
      zA = map(zA,zSL,90,SUL[zCh],SRV[zCh]);
    }
  }

  SPP[zCh] = zA; I2Cpwm.setPWM(zCh, 0, zA); AutoOffTimer = 200;
//  if (SI < 1) {PrintTx += "PWM=" + String(zA) + "\n";}
}

// ---------------------------------------------------------------------------------

void setServoTgt(int zA, int zCh) {
  // sets the servo target value to match an angle of zA°
  // ensure that angle zA° is within the limits set for this servo
  int zAL = 30; int zAU = 170; // angle limits
  int zSL = 30; int zSU = 150; // servo calibration limits
  byte zM = 0;
  switch(zCh) {
    case SH0:           zAU = 150; break;
    
    case SL0: zAL = 10;            break;
    case SL1: zAL = 10;            zM = 1; break;
    case SL2:           zAU = 180; zM = 1; break;
    case SL3:           zAU = 150; zM = 1; break;
    case SL4: zAL = 70; zAU = 130; zSL = 70; zSU = 110; break;
    
    case SR0: zAL = 10;            break;
    case SR1: zAL = 10;            break;
    case SR2:           zAU = 180; break;
    case SR3:           zAU = 150; break;
    case SR4: zAL = 50; zAU = 110; zSL = 70; zSU = 110; break;
  }
  if (zA < zAL) {zA = zAL;} else if (zA > zAU) {zA = zAU;}

  // non-linearity of servo is corrected
  if (zM < 1) {
    if (zA >= 90) {
      STG[zCh] = map(zA,90,zSU,SRV[zCh],SUL[zCh]);
    } else {
      STG[zCh] = map(zA,zSL,90,SLL[zCh],SRV[zCh]);
    }
  } else {
    if (zA >= 90) {
      STG[zCh] = map(zA,90,zSU,SRV[zCh],SLL[zCh]);
    } else {
      STG[zCh] = map(zA,zSL,90,SUL[zCh],SRV[zCh]);
    }
  }
}

// ---------------------------------------------------------------------------------

void SetStartsToPPs() {
  // sets all start pointers to present positions
  for (int zP = 0; zP < 16; zP++) {SSP[zP] = SPP[zP];}
}

// ---------------------------------------------------------------------------------

void SetTargetsToPPs() {
  // sets all target pointers to present positions, effectively stopping any movement
  for (int zP = 0; zP < 16; zP++) {STG[zP] = SPP[zP];}
}

// ---------------------------------------------------------------------------------

void SetTargetsToSRVs() {
  // sets all target pointers to default rest values
  for (int zP = 0; zP < 16; zP++) {STG[zP] = SRV[zP];}
}

// ---------------------------------------------------------------------------------

void WaitWhileMove() {
  // called to complete a single move like Rest or Ready
  // do not call for a move which repeats or it will never end
  while ((Head > 0) || (Walk > 0)) {
    // complete squat move
    if (micros() >= walkMicros) {Walk_Engine();}
    if (micros() >= headMicros) {Head_Engine();}
  } synchLoopTimers();
}

// ---------------------------------------------------------------------------------

void Walk_Engine() {
  // walking uses a separate timer so that the speed of walking can be controlled
  // before walking is enabled ensure that start conditions are set:
  // Walk > 0     - must be 1 or mote for the function to run
  // LegStep > 0  - a count down which determines the angle position
  // WalkMem > 0  - points to the next row of data in the memory incremented during load function
  // WalkMode > 1 - if 0 onloy read memory values once, if > 0 reload from start
  walkMicros = micros();  // grab the current timeer
  if (WalkTOF) {
    // timer overflow has occured so wait for micros() timer to overflow too
    if (micros() < lastWalkMicros) {WalkTOF = false;}  // back to normal, clear the flag
    else {return;}  // return to do other things in the mean time
  }
  if (Walk > 0) {
    // set the Walk time period based on the rate profile, fast is normal rate
    // 0 = default rate, constant
    // 1 = accelerating rate from slow to fast
    // 2 = decelerating rate from fast to slow
    // 3 = accelerating/decelerating slow to fast to slow
    if (RateProf < 1) {
      walkMicros += walkInterval;
    } else {
      ProfPnt = 9 - ((LegStep * 9)/LegSteps);
      if (ProfPnt > 9) {ProfPnt = 9;}
      if (ProfPnt < 0) {ProfPnt = 0;}
//      PrintTx += "ProfPnt " + String(ProfPnt) + "\n";
      switch(RateProf) {
        case 1: // decelerating profile
          walkMicros += (walkInterval * Prof1[ProfPnt])/20; break;
        case 2: // accelerating profile
          walkMicros += (walkInterval * Prof2[ProfPnt])/20; break;
        case 3: // accel/decelerating profile
          walkMicros += (walkInterval * Prof3[ProfPnt])/20; break;
      }
    }
  } else {
    // not walking so set timer to default 20ms interval
    walkMicros += walkIntDef;
  }

  lastWalkMicros = micros(); if (walkMicros < micros()) {WalkTOF = true;} // time point has over-flowed, so set overflow flag
//    LedTick = true; // perform LED sequences at same rate as walking
  if (Pause) {return;}  // pause flag causes movement to be put on hold
  if (Walk > 0) {
    switch (Walk) {
      // This Walk_Engine function moves servo values from their start positions to
      // their target positions, tracked by their present position as they are moved.
      case 1: // moving leg servos from start positions to targets 
//        PrintTx += String(LegStep) + "\n";
        if (servoOE) {SetServoOE(0);} // turn the servos ON
        if (LegStep > 1) {
          LegStep--;  // reduce the step value towards zero
          // only adjust values that have start points different from their targets
          for (ChMapPnt = 0; ChMapPnt < 10;ChMapPnt++) {
            SerPnt = ChMap[ChMapPnt];
            if (SSP[SerPnt] != STG[SerPnt]) {
              SPP[SerPnt] = STG[SerPnt] - (((STG[SerPnt] - SSP[SerPnt])*LegStep)/LegSteps);
              I2Cpwm.setPWM(SerPnt, 0, SPP[SerPnt]);
            }
          }
        } else if (LegStep == 1 ){
          // last step, so move all servos to target positions
          // set all start positions == target positions, for next move
          for (ChMapPnt = 0; ChMapPnt < 10;ChMapPnt++) {
            SerPnt = ChMap[ChMapPnt];
            if (SSP[SerPnt] != STG[SerPnt]) {SPP[SerPnt] = STG[SerPnt]; SSP[SerPnt] = STG[SerPnt]; I2Cpwm.setPWM(SerPnt, 0, SPP[SerPnt]);}
          }
          LegStep = 0; Walk = 0;
          // at end of leg move. Check to see if values are being loaded from Mem[]

          if (WalkMem > 0) {
            // values are being loaded from memory, so load the next row
            if (WalkMode < 1) {
              // single move sequence, only read data up to end of memory
              if (WalkMem < MemMax) {
                if (MoveMemTgtLoader(WalkMem)) {
                  // data loaded ok so carry on
                  setLegSpeed(Speed); // set walk speed
                  Walk = 1; // re-initiate walking with the new targets loaded
                }
              }
            } else {
              // continuous move sequence, only stop if an error occurs
              if (WalkMem >= MemMax) {WalkMem = 0;} // reset pointer if end of memory reached
              if (MoveMemTgtLoader(WalkMem)) {
                // data loaded ok so carry on
                setLegSpeed(Speed); // set walk speed
                Walk = 1; // re-initiate walking with the new targets loaded
              }
            }
          }
        }
//        PrintTx += String(SPP[SR0]) + "," + String(SPP[SL0]) + ",";
//        PrintTx += String(SPP[SR1]) + "," + String(SPP[SL1]) + ",";
//        PrintTx += String(SPP[SR2]) + "," + String(SPP[SL2]) + ",";
//        PrintTx += String(SPP[SR3]) + "," + String(SPP[SL3]) + ",";
//        PrintTx += String(SPP[SR4]) + "," + String[SL4]) + "\n";
        AutoOffTimer = 200; break;
    }
  } else {
    // not walking so reset the walk timer
    if (walkInterval != walkIntDef) {walkInterval = walkIntDef;}
  }
}

// ---------------------------------------------------------------------------------
