
// ---------------------------------------------------------------------------------

void AddToMem() {
  // a comma has been received on the serial port, so load cmdVal into memory if
  // a memory load command has initiated this sequence of events
  // the memory must be filled from the bottom up
  if (MemRow < 0) {cmdVal = 0; return;}
  if (MemPnt < MemWidth) {
    // only add values if the pointer is less than the memory row width, otherwise
    // the process wraps onto the next row
    Mem[MemRow + MemPnt] = cmdVal;
    PrintTx += String(cmdVal) + ",";
    MemPnt++;
    if (SI > 0) {
      // in SI mode apply angles directly to servos as recevied
      if (cmdVal < 0) {cmdVal = 0;} else if (cmdVal > 180) {cmdVal = 180;}
      setServoAng(cmdVal,ChMap[MemPnt - 1]);
      if (MemPnt == SI) {
        // data has filled row 0 servos so load them immediately
        PrintTx += "\n";      // add a line feed to print buffer
        SI = 0; MemPnt = 0;   // reset pointers
      }
    } else if (MemPnt == MemWidth) {
      // data has filled the row
      PrintTx += "\n";      // add a line feed to print buffer
      MemRow++; MemPnt = 0; // point at next row
      MemMax = (MemRow / MemWidth) + 1; // record the row filled as the max row number
//      PrintTx += "MemMax " + String(MemMax) + "\n";
    }
  }
  // zero cmdVal for receipt of next value
  cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 0;
}

// ---------------------------------------------------------------------------------

void AutoPwrOFF() {
  // turn OFF servos if no activity and AtRest == true
  if (AutoOffTimer > 0) {
    AutoOffTimer--;
//    PrintTx += String(AutoOffTimer) + "\n";
    if ((AutoOffTimer < 1) && AtRest) {
      SetServoOE(1);
      PrintTx += "Auto Power OFF\n";
    }
  }
}

// ---------------------------------------------------------------------------------

void clearMem() {
  // Clear the contents of the move memory Mem[]
  int zMax = MemMax * MemWidth;
  for (int zM = 0; zM < zMax; zM++) {Mem[zM] = 0;}
  MemRow = 0; MemMax = 0;
}

// ---------------------------------------------------------------------------------

void ClearPCAop() {
  // sets all 16 PWM channels to zero pulses, effectively OFF
  // even with OE enabled the servos will not receive any pulses
  for (int zC=0; zC < 16; zC++) {
    I2Cpwm.setPWM(zC, 0, 0);  // if OFFtime == ONtime the op never goes HIGH
  }
}

// ---------------------------------------------------------------------------------

void decodeKey(int zkeyVal) {
  // decodes zkeyVal and excutes commands on '.'
  // every time we receive a character set response to 10ms later, just in case
  // there is another packet behind it coming from the controller
  // when data is received set AutoOffTimer and eye LEDs to red
  next40ms = millis() + 10; AutoOffTimer = 200; LEDTaskPnt = 3;
  if (zkeyVal == 10) {return;}
  if (zkeyVal == 13) {return;}
  keyChar = char(zkeyVal);
  switch (keyChar) {
    case '.': doCmd(); return;          // command terminator
    case '+': cmdVal = -1; return;      // set flag for ML+. command
    case '-': cmdSgn = -1; return;      // numeric sign
    case ',': AddToMem(); return;       // assume cmdVal contains a memory value
    case '!': SetMainTask(98); return;  // return to 'Rest' position
    case '#': Ping++; AtRest = false; return; // 'ping' sent from application
    case '*': cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 0; return; // ESC abort command
    case '0': extendCmdVal(0); return;
    case '1': extendCmdVal(1); return;
    case '2': extendCmdVal(2); return;
    case '3': extendCmdVal(3); return;
    case '4': extendCmdVal(4); return;
    case '5': extendCmdVal(5); return;
    case '6': extendCmdVal(6); return;
    case '7': extendCmdVal(7); return;
    case '8': extendCmdVal(8); return;
    case '9': extendCmdVal(9); return;
  }
  if (cmdMode == ' ') {
    // test for new Command Mode char?
    switch (keyChar) {
      // check for lower-case and convert
      case 'm': cmdMode = 'M'; break;
      case 'M': cmdMode = 'M'; break;
      case 'r': cmdMode = 'R'; break;
      case 'R': cmdMode = 'R'; break;
      case 's': cmdMode = 'S'; break;
      case 'S': cmdMode = 'S'; break;
    } cmdType = ' '; cmdVal = 0;
  } else {
    // test for Command Type char?
    cmdType = keyChar;
    switch (keyChar) {
      // check for lower-case and convert to upper-case
      case 'a': cmdType = 'A'; break;
      case 'c': cmdType = 'C'; break;
      case 'e': cmdType = 'E'; break;
      case 'i': cmdType = 'I'; break;
      case 'l': cmdType = 'L'; break;
      case 'o': cmdType = 'O'; break;
      case 'p': cmdType = 'P'; break;
      case 'r': cmdType = 'R'; break;
      case 's': cmdType = 'S'; break;
      case 'x': cmdType = 'X'; break;
    }
  }
}

// ---------------------------------------------------------------------------------

void disableWiFi() {
  // when either a switch is pressed or WiFi disable is required
  WiFiEn = false; WiFiCntC = 0; WiFiRestCnt = 0; C_Dn = 8; Z_Dn = 8;
}

// ---------------------------------------------------------------------------------

void doCmd() {
  // a '.' has been received on the serial port so execute command if valid
  // Commands:
  // MC.        - clears memory contents
  // ME.        - export the contents of the memory
  // MLnn.n,n,. - start a memory load sequence, values separate by a comma, terminated by a comma
  // ML+.n,n,.  - continue memory load sequence
  // MO.        - run the contents of the memory once only.
  // MP.        - toggle walk pause if walking
  // MR.        - run the contents of the memory continuously.
  // MS.        - STOP a movement sequence
  // MXnn.      - run one row of the memory
  // RS.        - move to rest position.
  // SAnn.      - set target servo angle to nn immediately.
  // SEn.       - set the state of the echo flag, 0 = OFF, 1 = ON
  // SI.n,n,.   - set all ten angles, values separate by a comma, terminated by a comma
  // SPnn.      - set leg speed to nn
  // STnn.      - set the servo target channel for future angle data. Default = 0.

  switch (cmdMode) {
    case ' ': break;
    case 'M':
      switch (cmdType) {
        case 'C': 
          // clears memory contents and exports it
          PrintTx += "MC.\n";
          clearMem(); break;
        case 'E': 
          // export the contents of the memory
          PrintTx += "ME.\n";
          exportMem(); break;
        case 'L': 
          // start a memory load sequence, values separate by a comma, terminated by a period '.'
          if (cmdVal >= 0) {
            // initiate a memory load at a specific memory location
            MemRow = MemWidth * cmdVal;
            PrintTx += "ML" + String(cmdVal) + ".";
          } else {
            // initiate a memory load at the next row
            MemRow += MemWidth;
            PrintTx += "ML+.";
          } MemPnt = 0; Trans = false; break;
        case 'O':
          // run the contents of the memory once only
          PrintTx += "MO.\n";
          SetMainTask(96); break; // run sequence from Mem[] once
        case 'P':
          if (Pause) {Pause = false;} // toggle pause flag
          else {Pause = true;}
          PrintTx += "MP.\n";
           break;
        case 'R':
          // run the contents of the memory continuously
          PrintTx += "MR.\n";
          SetMainTask(97); break; // run sequence from Mem[] continuously
        case 'S':
          // stop movements by terminating tasks
          PrintTx += "STOP!!!\n";
          SetMainTask(0); break;
        case 'X':
          // move to the memory points Mem[MemRow]
          PrintTx += "MX" + String(cmdVal) + ".\n";
          MemRow = cmdVal; SetMainTask(95); break;
      } break;
    case 'R':
      switch (cmdType) {
        case 'S':
          // move to rest position
          PrintTx += "RS.\n";
          SetMainTask(98); break;
      } break;
    case 'S':
      switch (cmdType) {
        case 'A':
          // set target servo angle to nn immediately
          if (cmdVal < 0) {cmdVal = 0;} else if (cmdVal > 180) {cmdVal = 180;}
          PrintTx += "SA" + String(cmdVal) + ".\n";
          setServoAng(cmdVal,servoTgt); break;
        case 'E':
          // set the state of the Echo flag
          if (cmdVal > 0) {Echo = true; PrintTx += "Echo ON\n";} else {Echo = false;}
          break;
        case 'I':
          // set ten values immediately. Mem row 0 is used for this.
          PrintTx += "SI.";
          SI = MemWidth - 1; MemRow = 0; MemPnt = 0; break;
        case 'P':
          // set leg speed to cmdVal
          PrintTx += "SP" + String(cmdVal) + ".\n";
          Speed = cmdVal; setLegSpeed(cmdVal); break; // set a new leg speed
        case 'T':
          // set the servo target channel for future angle data. Default = 0
          PrintTx += "ST" + String(cmdVal) + ".\n";
          servoTgt = cmdVal; break;
      } break;
    default:
      PrintTx += "Unknown cmd: " + cmdMode + cmdType + String(cmdVal) + "\n";
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 0;
}

// ---------------------------------------------------------------------------------

void exportMem() {
  // sends the contents of the move memory Mem[] to the serial port
  int zMemSize = MemRows * MemWidth;
  for (int zRow = 0; zRow < zMemSize; zRow+=11) {
    PrintTx += "\nML" + String(zRow/10) + ".";
    for (int zPnt = 0; zPnt < 11; zPnt++) {
      if (Mem[zRow + zPnt] < 99) {
        PrintTx += " ";
      }
      if (Mem[zRow + zPnt] < 10) {
        PrintTx += " ";
      }
      PrintTx += String(Mem[zRow + zPnt]) + ",";
    } Serial.flush(); Serial1.flush();
  } PrintTx += "\n";
}

// ---------------------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = abs(cmdVal * 10) + zVal;
  if (cmdSgn < 0) {cmdVal = -cmdVal;}
}

// --------------------------------------------------------------------------------

void flashLEDs(int zCnt) {
  // flash the left and right LED swCnt times  
  for (int zP = 0; zP < zCnt; zP++) {
    NeoPixel_SetAllEyeLEDs(200, 0, 0); strip1.show(); delay(30);
    NeoPixel_SetAllEyeLEDs(0, 0, 0);strip1.show(); delay(270);
  }
}

// -------------------------------------------------------------------------------

String get3Digit(int zV) {
  // returns a 3 character string from 0 - 999
  String zS = String(zV);
  if (zV < 100) {zS = "0" + zS;}
  if (zV < 10) {zS = "0" + zS;}
  return zS;
}

// -------------------------------------------------------------------------------

void getLedCol(int zV) {
  // returns a colour for a specific zV
  switch (zV) {
    case 0: colR = 0; colG = 0; colB = 0; break;
    case 1: colR = 0; colG = 10; colB = 0; break;
    case 2: colR = 5; colG = 5; colB = 0; break;
    case 3: colR = 0; colG = 0; colB = 10; break;
    case 4: colR = 5; colG = 0; colB = 5; break;
    case 5: colR = 10; colG = 0; colB = 0; break;
  }
}

// -------------------------------------------------------------------------------

void getRangeCol(int zR) {
  // returns a colour for a range value
  // 0 = OFF
  // 1 = Green
  // 2 = Green/Yellow
  // 3 = Yellow
  // 4 = Orange
  // 5 = Red
  colR = 10; colG = 0; colB = 0;  // default is zR >= 5 Red
  switch (zR) {
    case 0: colR = 0; colG = 0; colB = 0; break;
    case 1: colR = 0; colG = 10; colB = 0; break;
    case 2: colR = 3; colG = 7; colB = 0; break;
    case 3: colR = 5; colG = 5; colB = 0; break;
    case 4: colR = 7; colG = 3; colB = 0; break;
  }
}

// -------------------------------------------------------------------------------

void Head_Tasks() {
  // called every 20ms to perform tasks based on a HeadTask pointer
//  Serial.println(HeadTask);
  switch(HeadTask) {
    case 0: //move head to centre forward and stop
      MoveHeadTo(90,100); HeadTask++; break;
    case 1: break;  // default NOP state

    case 10:  // move head to random positions
      MoveToRndHeadPos(50 + (6 * random(11)));  // random speed 50 to 110
      HeadTask++; break;
    case 11:  // wait for move to end
      if (Head == 0) {HeadTask++; HeadCnt = 20 + random(150);}
      break;
    case 12: // wait for random delay, then start the cycle over
      HeadCnt--; if (HeadCnt < 1) {HeadTask = 10;}
      break;
    case 13:  // stop the head from move at its current position
      Head = 0; break;

    case 20:  // scan head left/right and send data
      if (LTOF_On < 1) {VL53L0X_ON ();} // turn ON LTOF if it was OFF
      getHeadAngle(); LEDTaskPnt = 4;
      HdAngMin = 30; HdAngMax = 150; HeadInc = 4; Detect = false; DetectCnt = 0;
      HeadTask++; break;
    case 21:
      // report range and angle
      PrintTx += "RA" + get3Digit(RangeFtd) + get3Digit(HeadAngle) + "\n";
      if (RangeFtd <= 500) {
        DetectCnt = 3;
        if (!Detect) {
          // object just detected, so adjust limits
          if (HeadInc > 0) {HdAngMin = HeadAngle; HeadInc = 1;}
          else {HdAngMax = HeadAngle; HeadInc = -1;}
        } Detect = true;
      } else {Detect = false;}
      RangeNew = false; HeadTask++; break;
    case 22:
      // initiate a head move, and at the limits change direction
      HeadAngle += HeadInc;
      if (HeadAngle >= 150) {
        // reached upper limit so change direction
        HeadAngle = 150; DetectCnt = 0; HdAngMax = 150;
        HeadInc = -1; // change the sign of HeadInc
      } else if (HeadAngle <= 30) {
        // reached lower limit so change direction
        HeadAngle = 30; DetectCnt = 0; HdAngMin = 30;
        HeadInc = 1;  // change the sign of HeadInc
      } else {
        DetectCnt--;
        if (DetectCnt == 1) {
          // force a change in direction as we come out of the object detected zone
          if (HeadInc > 0) {HdAngMax = HeadAngle; HeadInc = -1;}  // we were increasing angle
          else {HdAngMin = HeadAngle; HeadInc = 1;} // we were decreasing angle
        } else if (DetectCnt == -12) {
          // an object has not been detected so revert back to normal speed after a delay
          HdAngMin = 30; HdAngMax = 150;
          if (HeadInc > 0) {HeadInc = 1;} else {HeadInc = -1;}
        }
      }
      // set the value of HeadInc based on its sign value
      if (abs(HeadInc) == 1) {HeadInc *= max((4 * (HdAngMax - HdAngMin))/120,1);}
      
      setServoAng(HeadAngle,SH0); // move the head immediately to the new angle
      HeadTask++; break;
    case 23:
      // wait for new range
      if (RangeNew) {HeadTask = 21;}  // new value received so report it
      break;
  }
}

// -------------------------------------------------------------------------------

void initialiseMPU() {
  // By default the MPU-6050 sleeps. So we have to wake it up.
  Wire.beginTransmission(MPU_addr); //Start communication with the address found during search.
  Wire.write(0x6B); //We want to write to the PWR_MGMT_1 register (6B hex)
  Wire.write(0x00); //Set the register bits as 00000000 to activate the gyro
  I2CError = Wire.endTransmission();
  if (I2CError == 0) {
    //Set the full scale of the gyro to +/- 250 degrees per second
    Wire.beginTransmission(MPU_addr); //Start communication with the address found during search.
    Wire.write(0x1B); //We want to write to the GYRO_CONFIG register (1B hex)
    Wire.write(0x00); //Set the register bits as 00000000 (250dps full scale)
    Wire.endTransmission();
    //Set the full scale of the accelerometer to +/- 4g.
    Wire.beginTransmission(MPU_addr); //Start communication with the address found during search.
    Wire.write(0x1C); //We want to write to the ACCEL_CONFIG register (1A hex)
    Wire.write(0x08); //Set the register bits as 00001000 (+/- 4g full scale range)
    Wire.endTransmission();
    //Set some filtering to improve the raw data.
    Wire.beginTransmission(MPU_addr); //Start communication with the address found during search
    Wire.write(0x1A); //We want to write to the CONFIG register (1A hex)
    Wire.write(0x03); //Set the register bits as 00000011 (Set Digital Low Pass Filter to ~43Hz)
    Wire.endTransmission();
    PrintTx = "MPU Initialised"; 
    if (SerialRx) {Serial1.println(PrintTx);} else {Serial.println(PrintTx);}
  } else {
    PrintTx = "MPU Initialisation Failed!"; 
    if (SerialRx) {Serial1.println(PrintTx);} else {Serial.println(PrintTx);}
    PrintTx = "I2C Error = " + String(I2CError); 
    if (SerialRx) {Serial1.println(PrintTx);} else {Serial.println(PrintTx);}
  }
  
}

// --------------------------------------------------------------------------------

void LED_Task() {
  // called from main loop every 10 ms to control LEDs
  if (!sw0State || !sw1State) {
    // if either switch pressed light eye LEDs
    NeoPixel_SetAllEyeLEDs(200, 0, 0);
    NeoPixel_SetAllMthLEDs(0,0,0);
    LEDMask = true;
  }else if (LEDMask) {
    // turns OFF eye LEDs for 100ms
    NeoPixel_SetAllEyeLEDs(0, 0, 0);
    if (LEDTaskPnt != 99) {LEDNextTask = LEDTaskPnt; LEDTaskPnt = 99;}
    LEDTaskWait = 100; LEDMask = false;
  } else {
    if (LEDTaskPnt < -1) {LEDTaskPnt = abs(LEDTaskPnt); LEDSubTask = 0;}
    switch(LEDTaskPnt) {
      case -1:
        // reset task which turns OFF the eye LEDs
        eyeB = 0; eyeG = 0; eyeR = 0;
        NeoPixel_SetAllEyeLEDs(eyeR,eyeG,eyeB);
        LEDTaskPnt++; LEDSubTask = 0; break;
      case 0: break;  // do nowt
      case 1: LED_Task1_DimUpDwn(); break;
      case 2: LED_Task2_BlinkWhenStop(); break;
      case 3: LED_Task3_RedIfPowerOn(); break;
      case 4: LED_Task4_Red(); break;
      case 5: LED_Task5_Gear(); break;
      case 99:
        // wait for a predefined number of 10ms intervals
        LEDTaskWait--; if (LEDTaskWait < 1) {LEDTaskPnt = LEDNextTask;} break;
    }
  }
}

// --------------------------------------------------------------------------------

void LED_Task1_DimUpDwn() {
  // dim eye and mouth white LEDs up and down
  LEDTaskWait = 50;         // set the task wait timer in 1ms steps
  LEDSubTask++;
  if (LEDSubTask > (10 + random(10))) {
    LEDSubTask = 0; NeoPixel_SetAllMthLEDs(0,0,0);
    LEDTaskWait = 50 + (100 * random(5));
  }
  NeoPixel_SetAllEyeLEDs(LEDSubTask, LEDSubTask, LEDSubTask);
  LEDNextTask = LEDTaskPnt; // return to this task
  LEDTaskPnt = 99;          // invoke the wait task
}

// --------------------------------------------------------------------------------

void LED_Task2_BlinkWhenStop() {
  // track head movement and blink when it stops
  // if stationary then also blink after a random delay
  switch (LEDSubTask) {
    case 0:
      LEDTaskWait = 200 + random(800);  // random blink delay 2 - 10s, used if stattionary
      if (!AtRest) {NeoPixel_SetAllEyeLEDs(eyeR,eyeG,eyeB);}
      LEDSubTask++; break;
    case 1: 
      // wait for moves to start or for end of a random delay
      LEDTaskWait--;
      if ((Head) || (Walk) || (LEDTaskWait < 1)) {LEDSubTask++;}
      break;
    case 2: if ((!Head) && (!Walk)) {LEDSubTask++;} break; // wait for moves to stop
    case 3:
      LEDTaskWait = 20 + random(20); LEDNextTask = LEDTaskPnt; LEDTaskPnt = 99;
      LEDSubTask++; break;  // restart the whole process
    case 4:
      // blink, then restart the whole process
      NeoPixel_SetAllEyeLEDs(0,0,0);
      LEDTaskWait = 10 + random(5); LEDNextTask = LEDTaskPnt; LEDTaskPnt = 99;
      LEDSubTask = 0; break;
  }
}

// --------------------------------------------------------------------------------

void LED_Task3_RedIfPowerOn() {
  // set the eyes to dim red when power is on
   eyeG = 0; eyeB = 0;
  if (AutoOffTimer > 0) {eyeR = 1;}
  else {eyeR = 0; LEDTaskPnt = 1;} // drop task when power goes off
  NeoPixel_SetAllEyeLEDs(eyeR,eyeG,eyeB);
}

// --------------------------------------------------------------------------------

void LED_Task4_Red() {
  // set the eyes to dim red, normally when LTOF is on
   eyeG = 0; eyeB = 0;
  if (LTOF_On > 0) {eyeR = 5;} else {eyeR = 0;}
  NeoPixel_SetAllEyeLEDs(eyeR,eyeG,eyeB);
}

// --------------------------------------------------------------------------------

void LED_Task5_Gear() {
  // temporarily display the gear colour in the eyes
  // you must save the current task pointer before calling this task
  switch (LEDSubTask) {
    case 0:
      getLedCol(Gear); NeoPixel_SetAllEyeLEDs(colR,colG,colB);
      LEDTaskWait = 20;
      if (LEDNextTask <= 0) {LEDNextTask = -1;} // LEDs were OFF so switch them OFF again
      LEDSubTask++; break;
    case 1:
      LEDTaskWait--;
      if (LEDTaskWait < 1) {
        LEDTaskPnt = LEDNextTask;
        NeoPixel_SetAllEyeLEDs(eyeR,eyeG,eyeB);
      } break;
  }
}

// --------------------------------------------------------------------------------

void LTOF_Read_Tasks() {
  // called from the main loop when enabled to read the laser range finder
  switch(LTOF_On) {
    // this is the VL53L0X_readRangeContinuousMillimeters() function broken down
    // into a state machine which take measurements when LTOF_On > 0
    // use the VL53L0X_Enable/Disable functions to switch the laser ON/OFF
    case 0: break;
    case 1: 
      // Returns a range reading in millimeters when continuous mode is active
      // (readRangeSingleMillimeters() also calls this function after starting a
      // single-shot range measurement)
      startTimeout(); LTOF_On++; break;
    case 2:
      // This is the main loop where we are waiting for the measurement to be completed
      // so we can branch to a multitasking loop from here whilst waiting
      // note that readReg takes about 500 us so not a fast while loop
      if ((VL53L0X_readReg(RESULT_INTERRUPT_STATUS) & 0x07) == 0) {
        // if no interrupt flag then check to see if a timeout has occured
        if (checkTimeoutExpired()) {
          // a timeout has occured
          did_timeout = true;
//          return 65535;
          Serial.println(F("LTOF timeout error!"));
        }
      } else {
        // an interrupt has occured suggesting a valid measurement
//        t1 = millis(); Serial.println(t1 - t0); t0 = t1; 
       LTOF_Wait = millis() + 32;  // default is +28
       LTOF_On++;
      } break;
    case 3:
      // assumptions: Linearity Corrective Gain is 1000 (default);
      // fractional ranging is not enabled
      Range = VL53L0X_readReg16Bit(RESULT_RANGE_STATUS + 10);
      Status = (VL53L0X_readReg(RESULT_RANGE_STATUS) & 0x78) >> 3;
      RangeNew = true;  // flag availabiulity of new range, this must be cleared by monitoring code
      VL53L0X_writeReg(SYSTEM_INTERRUPT_CLEAR, 0x01);
//      Serial.println(Range);
      LTOF_On++; break;
    case 4:
      // apply filtering to raw Range measurement
      RangeFtd = Range;
      if (Status != 11) {RangeFtd = 999;}
      if (RangeFtd >= 999) {
        RangeFtd = 999;
        if ((RangeFtd - (int)RangeLast) > 200) {
          Pulse++;
          if (Pulse < 3) {RangeFtd = RangeLast;}
        } else {Pulse = 0;}
      } else {
        if (((int)RangeLast - RangeFtd) > 200) {
          Pulse++;
          if (Pulse < 3) {RangeFtd = RangeLast;}
        } else {Pulse = 0;}
        if (RangeFtd < 50) {RangeFtd = 50;} // set minimum distance
      }
      RangeLast = RangeFtd; RangeFtd = min(RangeFtd,600);
//      Serial.print(min(Range,1000)); Serial.print(","); Serial.println(RangeFtd);
      LTOF_On++; break;
    case 5:
      // as the sensor only returns measurements every 30-32ms, there is no point in
      // spending time reading interrupt registers
      if (millis() >= LTOF_Wait) {LTOF_On = 1;}
      break;
  }
}

// --------------------------------------------------------------------------------

void MouthTask() {
  // called every 20ms from main loop to perform a range mouth LED of tasks
  switch(LEDMthTask) {
    case -1:
      // normal exit task, turns OFF LEDs and goes back to task 0
      NeoPixel_SetAllMthLEDs(0,0,0); LEDMthTask = 0; break;
    case 0:
      // default task simply displays battery voltage every 2 seconds
      // there are 5 LEDs in the mouth, 0 - 4, left to right
      mouthCnt--; getLedCol(Gear);
      if (mouthCnt == 10) {NeoPixel_SetMthLED(map(BattAv,BattMin,1023,0,5), colR, colG, colB);}
      else if (mouthCnt == 8) {NeoPixel_SetAllMthLEDs(0,0,0);}
      else if (mouthCnt < 1) {mouthCnt = 100;} // reset 2 second timer
      break;
    
    case 10: // bar graph task
      // LEDMthVal 0 - 5 determines which LEDs are lit
      getRangeCol(LEDMthVal);
      for (int zP = 0; zP < 5; zP++) {
        if (LEDMthVal > zP) {NeoPixel_SetMthLED(zP,colR,colG,colB);}
        else {NeoPixel_SetMthLED(zP, 0, 0, 0);}
      } break;


    case 20:
      // movement patterns. Colour determined by Speed value, green to red.
      NeoPixel_Clear_Array(); getRangeCol(map(Speed,SpdMin,SpdMax,1,5));
      colBlu[0] = colB; colGrn[0] = colG; colRed[0] = colR;
      LEDMthPnt = 0; LEDMthCnt = 1;
      LEDMthTask += 1;  // moving forward by default
      if (MoveState ==  3) {LEDMthTask = 23;} // turning right
      if (MoveState ==  7) {LEDMthTask = 22;} // turning left
      if (MoveState ==  5) {LEDMthTask = 24;} // moving backwards
      if (MoveState == 13) {LEDMthTask = 23;} // turning right
      if (MoveState == 17) {LEDMthTask = 22;} // turning left
      LEDMthNext = LEDMthTask;
      break;
    case 21:
      // rotate the LEDs for forward movement
      // we only need to specify the values that change
      switch(LEDMthPnt) {
        case 0: LP[0] = 1; LP[1] = 1; LP[2] = 1; LP[3] = 1; LP[4] = 1; break;
        case 1:                       LP[2] = 0;                       break;
        case 2:            LP[1] = 0;            LP[3] = 0;            break;
        case 3: LP[0] = 0; LP[1] = 1; LP[2] = 1; LP[3] = 1; LP[4] = 0; break;
      } LEDMthTask = 25; break;
    case 22:
      // rotate the LEDs for turning left movement
      switch(LEDMthPnt) {
        case 0: LP[0] = 0; LP[1] = 1; LP[2] = 1; LP[3] = 1; LP[4] = 0; break;
        case 1: LP[0] = 1;                       LP[3] = 0; LP[4] = 1; break;
        case 2:                       LP[2] = 0; LP[3] = 1;            break;
        case 3:            LP[1] = 0; LP[2] = 1;                       break;
      } LEDMthTask = 25; break;
    case 23:
      // rotate the LEDs for turning right movement
      switch(LEDMthPnt) {
        case 0: LP[0] = 0; LP[1] = 1; LP[2] = 1; LP[3] = 1; LP[4] = 0; break;
        case 1: LP[0] = 1; LP[1] = 0;                       LP[4] = 1; break;
        case 2:            LP[1] = 1; LP[2] = 0;                       break;
        case 3:                       LP[2] = 1; LP[3] = 0;            break;
      } LEDMthTask = 25; break;
    case 24:
      // rotate the LEDs for backward movement
      switch(LEDMthPnt) {
        case 0: LP[0] = 1; LP[1] = 1; LP[2] = 1; LP[3] = 1; LP[4] = 1; break;
        case 1: LP[0] = 0;                                  LP[4] = 0; break;
        case 2: LP[0] = 1; LP[1] = 0;            LP[3] = 0; LP[4] = 1; break;
        case 3:            LP[1] = 1; LP[2] = 0; LP[3] = 1;            break;
      } LEDMthTask = 25; break;
      break;
    case 25:
       LEDMthPnt++; if (LEDMthPnt > 3) {LEDMthPnt = 0;}
      for (int zP = 0; zP < 5; zP++) {
        strip0.setPixelColor(zP,strip0.Color(colRed[LP[zP]],colGrn[LP[zP]],colBlu[LP[zP]]));
      } LEDshow0 = true;
      LEDMthTask++; break;
    case 26:
      LEDMthCnt--; if (LEDMthCnt < 1) {LEDMthCnt = 6; LEDMthTask = LEDMthNext;}
      break;


      case 30:  // linked to the head scanning task display colour quadrants
        NeoPixel_SetAllMthLEDs(0,0,0);
        LEDMthTask++; break;
      case 31:
        LEDMthPnt = map(HeadAngle,30,150,4,0);
        getRangeCol(map(RangeFtd,0,600,5,0));
        strip0.setPixelColor(LEDMthPnt,strip0.Color(colR,colG,colB)); LEDshow0 = true;
        break;
  }

  if ((LEDMthTask >= 21) && (LEDMthTask <= 24)) {
    // when moving change the LEDs colour to match the Speed
    getRangeCol(map(Speed,SpdMin,SpdMax,1,6));
    colBlu[0] = colB; colGrn[0] = colG; colRed[0] = colR;
  }
}

// --------------------------------------------------------------------------------

void NeoPixel_Clear_Array() {
  // clears the contents of the colour array
  for (int zL = 0; zL < 5; zL++) {
    colBlu[zL] = 0; colGrn[zL] = 0; colRed[zL] = 0;
  }
}

// --------------------------------------------------------------------------------

void NeoPixel_Load_Array(byte zP) {
  // load the contents of the colour array into the mouth LEDs
  // as the colour array is twice the depth as the number of mouth LEDs you can
  // pre-load the array and then simply
  for (int zL = 0; zL < 5; zL++) {
    strip0.setPixelColor(zL,strip0.Color(colRed[zL + zP],colGrn[zL + zP],colBlu[zL + zP]));
  } LEDshow0 = true;
}

// --------------------------------------------------------------------------------

void NeoPixel_SetAllEyeLEDs(byte zR, byte zG, byte zB) {
  // set all eye LEDs to one colour
  strip1.setPixelColor(0,strip1.Color(zR,zG,zB));
  strip1.setPixelColor(1,strip1.Color(zR,zG,zB));
  LEDshow1 = true;
}

// --------------------------------------------------------------------------------

void NeoPixel_SetAllMthLEDs(byte zR, byte zG, byte zB) {
  // set all mouth LEDs to one colour
  for (int zP = 0; zP < ledNum0; zP++) {
    colBlu[zP] = zB; colGrn[zP] = zG; colRed[zP] = zR;
    strip0.setPixelColor(zP,strip0.Color(colRed[zP],colGrn[zP],colBlu[zP]));
  } LEDshow0 = true;
}

// --------------------------------------------------------------------------------

void NeoPixel_SetEyeLED(byte zPixNum, byte zR, byte zG, byte zB) {
  // set the colour of eye LED zPixNum.
  colBlu[zPixNum] = zB; colGrn[zPixNum] = zG; colRed[zPixNum] = zR;
  strip1.setPixelColor(zPixNum,strip1.Color(colRed[zPixNum],colGrn[zPixNum],colBlu[zPixNum]));
  LEDshow1 = true;
}

// --------------------------------------------------------------------------------

void NeoPixel_SetMthLED(byte zPixNum, byte zR, byte zG, byte zB) {
  // set the colour of mouth LED zPixNum.
  colBlu[zPixNum] = zB; colGrn[zPixNum] = zG; colRed[zPixNum] = zR;
  strip0.setPixelColor(zPixNum,strip0.Color(colRed[zPixNum],colGrn[zPixNum],colBlu[zPixNum]));
  LEDshow0 = true;
}

// -------------------------------------------------------------------------------

void readAccelX() {
  // read the X-axis acceleration value from the MPU
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3B);  // starting with register 0x3B (ACCEL_XOUT_H)
  Wire.endTransmission(false);
  Wire.requestFrom(MPU_addr,2,true);  // request 1 register read
  AcX = Wire.read()<<8|Wire.read();  // 0x3B (ACCEL_XOUT_H) & 0x3C (ACCEL_XOUT_L)
  PrintTx += "AcX Raw = " + String(AcX) + "\n";
}

// -------------------------------------------------------------------------------

void readAccelY() {
  // read the Y-axis acceleration value from the MPU
  // AcY = 8,192 at 1g. In linerar region divide by 138.5 to get degrees
  Wire.beginTransmission(MPU_addr);
  Wire.write(0x3D);  // starting with register 0x3D (ACCEL_YOUT_H)
  I2CError = Wire.endTransmission(false);
  if (I2CError == 0) {
    Wire.requestFrom(MPU_addr,2,true);  // request 1 register read
    if (Wire.available() > 1) {
      AcY = Wire.read()<<8|Wire.read();  // 0x3D (ACCEL_YOUT_H) & 0x3E (ACCEL_YOUT_L)
      PrintTx += "AcY Raw = " + String(AcY) + "\n";
    }
  }
  // we apply an offset if needed, depending on how it has been mounted
  // uncomment the next line to determine the raw value at rest, AcYrest
  // then set AcYOff = -AcYrest to zero out the offset
  return;
  AcYsum += (long)AcY; AcYsum -= AcYaV;   // update the rolling accumulator
  AcYaV = AcYsum/4;                       // calc the average value
//  if (Fn) {AcYaV = AcYsum/40;}            // greater average in special function mode
//  AcYaV = AcY;                          // don't calc the average value
    PrintTx += "AcY Av = " + String(AcYaV) + "\n";
//  AcYF = ((float)((AcYaV + AcYOff)/69))/2.0;         // convert to an approximate angle
//  AcYsgn = true; // set sign flag as positive
//  if (AcYF < 0.0){AcYF = -AcYF; AcYsgn = false;}; // set sign flag as negative
}

// --------------------------------------------------------------------------------

void readBattery(int zM) {
  // read the voltage on the analog input and average it over ten counts
  // zM = 0 - simple read + average
  // zM = 1 - calc float and percentage
  BattVol = analogRead(BattPin);
  BattAv = BattSum / 10; BattSum = BattSum + BattVol - BattAv;
  // uncomment the following line if you want to see readings on serial monitor
//  Serial.print(F("BattAv = ")); Serial.print(BattAv); Serial.print(F("\t")); Serial.print((5.0 * BattAv)/608.9); Serial.println(F("v"));
  
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  critical battery test
  //
  ////////////////////////////////////////////////////////////////////////////////
  // test for battery low condition, 804 == 6.6v
  if (BattAv < BattMin) {
    // battery has reached critical level so squat down and blink centre LED
    NeoPixel_SetAllEyeLEDs(0,0,0); strip1.show(); // eye LEDs OFF
    NeoPixel_SetAllMthLEDs(0,0,0); strip0.show(); // mouth LEDs OFF
    MoveToSquat(); WaitWhileMove();
    SetServoOE(1);  // turn servos OFF
    while (true) {
      // blink mouth centre LED red forever
      delay(20); BattCnt--;
      if (BattCnt < 1) {
        BattCnt = 100; NeoPixel_SetMthLED(2, 0, 0, 0); strip0.show();
      }
      if (BattCnt == 5) {NeoPixel_SetMthLED(2, 1, 0, 0); strip0.show();}
      // continue to take readings in case this is a deliberate power-down
      BattVol = analogRead(BattPin);
      BattAv = BattSum / 10; BattSum = BattSum + BattVol - BattAv;
      if ((digitalRead(sw0Pin) == LOW) && (BattAv > (BattMin + 10))) {break;}
    }
    // user has pressed button during power-down
    // battery looks ok so revive
    MoveToAllAtRest(); WaitWhileMove();
    setDefaults();
    synchLoopTimers();
  }
  
  if (Ping > 9) {Ping = 0; zM = 1;} // force a battery report every 10 pings
  if (zM < 1) {return;} // exit if not a report measurement
  
  // send battery voltage and % to serial port
//  BattV = 0.00821 * BattAv; BattP = (int)((100L * (BattAv - 804))/219);
//  PrintTx += "Batt " + String(BattAv) + "  " + BattV + "v  " + String(BattP) + "%\n";
}

// --------------------------------------------------------------------------------

void readSerial() {
  // reads characters from the two serial ports and responds to commands
  // Arduino NANO EVERY has two serial ports, Serial and Serial1
  //  usb Serial  port will connect to Windows app so handle as text
  // WiFi Serial1 port could be Wii Nunchuk so pass through stat machine first
  keyVal = Serial.read();
  if (keyVal != -1) {SerialRx = 0; decodeKey(keyVal); return;}
  
  RxVal = Serial1.read(); // this is the WiFi port
  if (RxVal != -1) {
    // receiver contains a data byte
    SerialRx = 1; // flag that data was received on the Serial1 WiFi port
    switch (RxState) {
      case 0: // check for first key value
        // normal text should not contain charcxater 0x0AA, so unless we are
        // connected to WiFi Nunchuk we should remain in RxState = 0
        if (RxVal == 0x0AA) {
          checksum = 0x0AA; RxState++;
//          Serial1.println("\n");
//          Serial1.println("K0");
        } else if (RxVal == 'Q') {RxState = 10;} // check for WiFi QM. request
        else {decodeKey(RxVal); return;} // text data from Move Controller
        break;
      case 1: // check for second key value
        if (RxVal == 0x055) {
          checksum = checksum ^ RxVal; RxState++;
//          Serial1.println("K1");
        } else {
          RxState = 0;
//          Serial1.println("Invalid K1");
        } break;
      case 2: // receive JoyX value, centre value = 127
        checksum = checksum ^ RxVal;
        RxJoyX = RxVal; RxState++;
        break;
      case 3: // receive JoyY value, centre value = 128
        checksum = checksum ^ RxVal;
        RxJoyY = RxVal; RxState++;
        break;
      case 4: // receive CZ value
        checksum = checksum ^ RxVal;
        RxCZ = RxVal; RxState++;
        break;
      case 5: // receive checksum value and confirm it is valid
        if (checksum == RxVal) {
          // frame checksum byte is confirmed as valid
          JoyX = RxJoyX; JoyY = RxJoyY; CZ = RxCZ;
          RxRec = true;
          Echo = false; // comment this line out if you want to return messages associated with Wii NUnchuk demands
          // Only use print statements at the end of the received frame
          // you can only send up to 30 characters
//            PrintBinary(JoyX);
//            PrintBinary(JoyY);
//            PrintBinary(CZ);
        } else {
          // frame checksum byte is invalid
//          Serial1.println("Invalid checksum");
//          PrintBinary(RxVal);
        }
        RxState = 0; // reset state machine
//        Serial1.println(".");
//        PrintBinary(checksum);
        break;
      case 10:
        // check for an 'M' after the 'Q' in the QM querie
        if (RxVal == 'M') {RxState++;} // check for 'M' in QM. request
        else {RxState = 0;}             // invalid character
        break;
      case 11:
        // on receipt of a QM. message the droid will respond with MQ2.
        // this advises the wireless ocntroller that it is a mode 2 device, which
        // should only be sent Wii data when it has changed. This frees up the 
        // comms channel for use with the Windows 'Move Trainer' app
        // check for '.' after 'QM'
        if (RxVal == '.') {PrintTx += "MQ2.";} // check for '.' in QM. request
        RxState = 0; break;
    }
  }
}

// --------------------------------------------------------------------------------

void readSW0() {
  // read the left button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task
  sw0State = digitalRead(sw0Pin);
  if ((sw0LastState == HIGH) && (sw0State == LOW)) {
    // SW0 has just been pressed down
    if (WiFiEn) {disableWiFi();}  // immediately stop WiFi enabled tasks
//    PrintTx += "SW0 Hi-LO\n";
    sw0Cnt++; // count on the falling edge
    sw0Timer = 0; // reset the timer
    if (MainTask) {sw0Cnt = 0; SetMainTask(98); ESC = true;}
  }
  sw0LastState = sw0State; // record current state
  if (ESC && !sw0State) {return;}
  ESC = false;
  if (sw0Cnt > 0) {sw0Timer++;}
  if ((sw0Timer >= 100) && (sw0State == HIGH)) {
    // button released for 1 sec so assume valid button count
//    Serial.print(F("swCnt = ")); Serial.println(swCnt);
    flashLEDs(sw0Cnt); // indicate count by flashing
    if (sw0Cnt == 1) {SetMainTask(1);}  // toggle LTOF
    if (sw0Cnt == 2) {SetMainTask(2);}; // backaway from target mode
    if (sw0Cnt == 3) {SetMainTask(3);}; // track target mode
    if (sw0Cnt == 4) {SetMainTask(4);}; // head scanning mode
    sw0Cnt = 0; sw0Timer = 0; synchLoopTimers();
  } else if((sw0Timer >= 200) && (sw0State == LOW)) {
    // button held down for >= 2 seconds
    // droid will enter battery LOW mode
    sw0Cnt = 0; sw0Timer = 0; // LedMode = 0;
    BattSum = 0;
  }
}

// --------------------------------------------------------------------------------

void readSW1() {
  // read the right button switch and respond accordingly, pressed == LOW
  // a button press will automatical drop the current task
  sw1State = digitalRead(sw1Pin);
  if ((sw1LastState == HIGH) && (sw1State == LOW)) {
    // SW1 has just been pressed down
    if (WiFiEn) {disableWiFi();}  // immediately stop WiFi enabled tasks
//    PrintTx = "SW1 Hi-LO\n";
    sw1Cnt++; // count on the falling edge
    sw1Timer = 0; // reset the timer
    if (MainTask) {sw1Cnt = 0; SetMainTask(98); ESC = true;}
  }
  sw1LastState = sw1State; // record current state
  if (ESC && !sw1State) {return;}
  ESC = false;
  if (sw1Cnt > 0) {sw1Timer++;}
  if ((sw1Timer >= 100) && (sw1State == HIGH)) {
    // button released for 1 sec so assume valid button count
//    Serial.print(F("swCnt = ")); Serial.println(swCnt);
    flashLEDs(sw1Cnt); // indicate count by flashing
    mainMode = sw1Cnt; SetMainTask(10); // do a movement task
    sw1Cnt = 0; sw1Timer = 0; synchLoopTimers();
  } else if((sw0Timer >= 200) && (sw0State == LOW)) {
    // button held down for >= 2 seconds
    sw1Cnt = 1; sw1Timer = 0; // LedMode = 0;
  }
}

// ---------------------------------------------------------------------------------

void readWii() {
  // called every 20ms, but wireless transceiver only transmits every 40ms
  // considers data received from the Wii Nunchuk controller over WiFi and 
  // manages all movements demanded by the controller
  readWiiCall = true; // prevent re-entry of this function
  
  if (RxRec) {
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  WiFi Nunchuk data received
  //
  ////////////////////////////////////////////////////////////////////////////////
    RxRec = false; overRun = 5;
//    PrintTx += "JX" + String(JoyX) + " JY" + String(JoyY) + " CZ" + String(CZ) + "\n";
//    Serial.print("CZ "); Serial.println(CZ);
    if (!WiFiEn) {
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  WiFi is currently disabled
  //
  ////////////////////////////////////////////////////////////////////////////////
      // we have not received enough 'C' buttons to make the system active
      if (CZ == 1) {
        // C button only is being pressed
        WiFiCntC += 4;
        if (WiFiCntC > 60) {
          // going active, user has held in 'C' button for sufficient time
          WiFiCntC = 64; WiFiEn = true; C_Dn = 8;
          if (MainTask != 0) {SetMainTask(0);}
          MoveToReady(1); LEDMthTask = -1; loopWhileWalk();
          // set defaults here
          Gear = 1;
        }
      }
    } else {
      
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  WiFi is enabled
  //
  ////////////////////////////////////////////////////////////////////////////////
      // WiFi enabled so check for power down
      if (CZ == 0) {
        // both C and Z buttons are being pressed so move towards powering down
        WiFiCntC -= 5;
        if (WiFiCntC <= 0) {
          // power-down condition satisfied, so go to rest
          disableWiFi();
          MoveToAllAtRest(); loopWhileWalk(); LEDMthTask = -1;
          // set defaults here
        }
      } else {
        // Check CZ buttons to see if a 'Gear' change is wanted
        if ((CZ & 2) == 0) {
          // C button is pressed, assume gear change
          C_Dn++; if (C_Dn == 2) {
            C_Dn = 8; if (Gear < 5) {Gear++;}
            if (LEDTaskPnt != 99) {LEDNextTask = LEDTaskPnt;}
            LEDTaskPnt = -5; // display 'Gear' colour in the eyes
          }
//          Serial.print("Gear "); Serial.println(Gear);
        } else {C_Dn = 0;} // C button released
        // Check Z button. It has two functions. Short press change gear, but if
        // held in it changes the mode of the joysticks
        if ((CZ & 1) == 0) {
          // Z button is pressed
          Z_Dn++; if (Z_Dn > 12) {Z_Dn = 50;} // Z_Dn capped at 50, held for >= 2 seconds
//          Serial.print("Z_Dn "); Serial.println(Z_Dn);
        } else {
          if ((Z_Dn > 0) && (Z_Dn <= 10)) {
            // held down for less than 400 ms @ 40ms Rx cycle
            if (Gear > 1) {Gear--;}  // short press so change gear if possible
            if (LEDTaskPnt != 99) {LEDNextTask = LEDTaskPnt;}
            LEDTaskPnt = -5; // display 'Gear' colour in the eyes
//            Serial.print("Gear "); Serial.println(Gear);
          } Z_Dn = 0; // Z button released
        }
//        Serial.print("Z_Dn "); Serial.println(Z_Dn);
      }

      
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  Wii joystick demands
  //
  ////////////////////////////////////////////////////////////////////////////////
      // now respond to joystick demands. We use a state machine to control how
      // the joystick demands are responded to. But first we resolve the JoyX and
      // JoyY values into a strength vector JoyV, and a direction JoyD
//      Serial.print("X"); Serial.print(JoyX); Serial.print("  Y"); Serial.println(JoyY);
      
      JoyXC = JoyX - 128; JoyYC = JoyY - 128;
      if ((abs(JoyXC) < 12) && (abs(JoyYC) < 12)) {
        // joystick is in centre deadband region
        JoyV = 0; JoyD = 0; // clear vector and direction values
      } else {
        // calculate vector outside of centre region
        JoyV = (int)sqrt((sq((float)JoyXC) + sq((float)JoyYC)));
        JoyV = min(JoyV,127); // cap max value, which can reach 181
        if (JoyXC == 0) {JoyXC = 1;}  // prevent divide by zero
        if (JoyYC == 0) {JoyYC = 1;}
        
        JoyD = 0; // reset direction pointer to default
        if (JoyYC >= 12) {
          // in upper hemisphere
          if (abs(JoyYC/JoyXC) >= 2) {JoyD = 1;}      // pointing N
          else if (abs(JoyXC/JoyYC) >= 2) {
            if (JoyXC > 0) {JoyD = 3;}                // pointing E
            else {JoyD = 7;}                          // pointing W
          } else {
            if (JoyXC > 0) {JoyD = 2;}                // pointing NE
            else {JoyD = 8;}                          // pointing NW
          }
        } else if (JoyYC > -12) {
          // in middle area
          if (JoyXC >= 12) {JoyD = 3;}                // pointing E
          else {JoyD = 7;}                            // Pointing W
        } else {
          // in lower hemisphere
          if (abs(JoyYC/JoyXC) >= 2) {JoyD = 5;}      // pointing S
          else if (abs(JoyXC/JoyYC) >= 2) {
            if (JoyXC > 0) {JoyD = 3;}                // pointing E
            else {JoyD = 7;}                          // pointing W
          } else {
            if (JoyXC > 0) {JoyD = 4;}                // pointing SE
            else {JoyD = 6;}                          // pointing SW
          }
        }
      }
//      Serial.print("JoyV "); Serial.print(JoyV);
//      Serial.print("   JoyD "); Serial.println(JoyD);
//      PrintTx += String(JoyD) + "\t" + String(JoyV) + "\n"; JoyD = 0; // print joystick vectors but don't move
//      if (MoveState > 0) {PrintTx += String(Speed) + "\n";}
      
      
      switch(MoveState) {
        case 0:
          // default state, droid will be in ready state or rest state
          LEDMthTask = -1;
          if (Z_Dn < 12) {
            // 'Z' button is not held down
            if (JoyD == 1) {MoveFwd();}
            else if (JoyD == 2) {MoveFwdRht();}
            else if (JoyD == 3) {MoveTurnRight();}
            else if (JoyD == 5) {MoveBkd();}
            else if (JoyD == 7) {MoveTurnLeft();}
            else if (JoyD == 8) {MoveFwdLft();}
          } else {
            // 'Z' button is held down
            if (JoyD == 1) {MoveState = 20; MoveToBowStretch();}
            else if (JoyD == 3) {MoveStepRight();}
            else if (JoyD == 5) {MoveState = 20; MoveToSwayJig();}
            else if (JoyD == 7) {MoveStepLeft();}
          } break;
        case 1: // moving forward
          MoveFwd(); break;
        case 2: // moving forward bearing to the right
          MoveFwdRht(); break;
        case 3: // turning right
          MoveTurnRight(); break;
        case 5: // moving backward
          MoveBkd(); break;
        case 7: // turning right
          MoveTurnLeft(); break;
        case 8: // moving forward bearing to the left
          MoveFwdLft(); break;
        case 13: // stepping right
          MoveStepRight(); break;
        case 17: // stepping left
          MoveStepLeft(); break;
        case 99:
          // wait for joystick to centralise
          if (JoyD == 0) {MoveState = 0; Trans = !Trans;} break;
      }
      if (MoveState > 0) {
        // responding to joystick so reset rest counter
        WiFiRestCnt = WiFiRestDef;
      }
    }
  } else {
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  No WiFi Nunchuk received
  //
  ////////////////////////////////////////////////////////////////////////////////
    if (!WiFiEn) {
      // droid is not enabled for Wii WiFi control
      if (WiFiCntC > 0) {
        // reduce the C-button counter, which has to be sustained to enable WiFi mode
        WiFiCntC--; LEDMthVal = 1 + (WiFiCntC/10);
        if (WiFiCntC == 0) {LEDMthTask = -1;} else {LEDMthTask = 10;}
      }
    } else {
      // droid is enabled for Wii WiFi control
      // maintain Wii enabled counter
      if (WiFiCntC < 64) {
        WiFiCntC++; LEDMthVal = WiFiCntC/10;
        if (WiFiCntC == 64) {LEDMthTask = -1;} else {LEDMthTask = 10;}
      }
//      // if moving check that mouth LEDs have been set correctly
//      if (MoveState > 0) {if (LEDMthTask < 20) {LEDMthTask = 20;}}
      
      // if in MoveState == 20 complete the move first, otherwise do these checks
      if (MoveState != 20) {
        // if moving ensure that demands are still being received, if not goto ready
        if (MoveState != 0) {
          overRun --;
          if (overRun < 1) {MoveState = 0; MoveToReady(1); loopWhileWalk(); LEDMthTask = -1;}
        }
        // perform auto-rest function if no demands
        if (WiFiRestCnt > 0) {WiFiRestCnt--;}
        if (WiFiRestCnt == 1) {
          // if not at rest go there
          if (!AtRest) {MoveToAllAtRest(); loopWhileWalk();}
        }
      }
//      Serial.println(WiFiRestCnt);
    }
  }

  if (WiFiEn) {
  ////////////////////////////////////////////////////////////////////////////////
  //
  //  WiFi background tasks when WiFi is enabled
  //
  ////////////////////////////////////////////////////////////////////////////////
    // only do these functions if WiFi is enabled
    if (MoveState == 20) {
      // long un-demanded moves like bowing are watched until they complete
      if (Walk < 1) {MoveState = 0; HeadTask = 10; LEDMthTask = -1;}
    }
    
    if (MoveState == -1) {
      // at the end of a joystick move we want to set things back to normal
      MoveToReady(1); loopWhileWalk(); LEDMthTask = -1; SetSpeedDefaults();
      MoveState = 0;
    }
  }
  
  readWiiCall = false; // re-allow calls to this function
//  Serial.print("CntZ "); Serial.println(WiFiCntZ);
//   Serial.println(MoveState);
}

// ---------------------------------------------------------------------------------

void SetMainTask(int zM) {
  // called to change current task and set initial conditions
  Detect = false;   // target detection flag; true detected
  DetectCnt = 0;    // target detection counter
  Head = 0;         // terminate any head movement
  HeadTask = 0;     // head task pointer
  LEDSubTask = 0;   // start subtasks from 0
  if (LTOF_On > 0) {VL53L0X_OFF();}
  mainEn = true;    // enable main tasks
  mainSubCnt = 0;   // zero the counter
  mainSubTask = 0;  // start subtasks from 0
  Pause = false;    // pause flag causes movement to be put on hold
  Walk = 0;         // terminate any leg movement
  WalkMem = 0;      // pointer to next move row in Mem[] to be read, 0 == do nothing
  WalkMode = 0;     // set the mode when moving using Mem[] targets, 0 == once, 1 = continuous
  
  LEDTaskPnt = 2;
  switch(zM) {
//    case 1: LEDTaskPnt = 2; break;
  }
  MainTask = zM;
  PrintTx += "\nMainTask = " + String(MainTask) + "\n";
  synchLoopTimers(); // ensure sync has not been lost
}

// ---------------------------------------------------------------------------------

void SetPCAop() {
  // sets all PCA channels to there default values
  for (int zC=0; zC < 16; zC++) {
    I2Cpwm.setPWM(zC, 0, SRV[zC]);  // if OFFtime == ONtime the op never goes HIGH
  }
}

// ---------------------------------------------------------------------------------

void SetServoOE(int zVal) {
  // manually set the servo board OE to HIGH or LOW
  // 0 - servos ON with outpus set
  // 1 - servoos OFF with all PWM set to zero
  if (zVal != 1) {
    // enabling PCA9685 outputs
    servoOE = 0; SetPCAop();
    digitalWrite(Pin_OE, LOW); // enable all PWM outputs
//    Serial.println("OE=LO [ON]");
  } else {
    // disabling PCA9685 outputs. As this is asynchronous there is a risk that this
    // action will shorten one or more PWM pulses, so we remove them first
    servoOE = 1; ClearPCAop();
    delay(10 + (1000/int(pwmFreq)));  // wait longer than one PWM cycle
    digitalWrite(Pin_OE, HIGH); // disable all PWM outputs
//    Serial.println("OE=HI [OFF]");
    synchLoopTimers();
  }
}

// ---------------------------------------------------------------------------------

void SetServoPW(int zCh, int zPWM) {
  // set the pulse width of a specified channel on the PCA9685 board
  // format: setPWM(channel,on,off)
  // setting 'on' = 0 means that the pulse 'off' can be 0 - 4095 (12-bits)
  I2Cpwm.setPWM(zCh, 0, zPWM);
  if (servoOE ==1) {SetServoOE(0);}  // if board is not enabled, enable it
}

// ---------------------------------------------------------------------------------

void SetSpeedDefaults() {
  // called to set default speed range values
  SpdMax = 100;            // maximum speed when moving
  SpdMid = 50;            // middle speed when moving
  SpdMin = 20;            // minimum speed when moving
  SpdTop = SpdMid;        // current top speed when moving
  Speed = SpdMin;         // default speed value
}

// ---------------------------------------------------------------------------------

void VL53L0X_OFF () {
  // disable the VL53L0X for reading ranges
  digitalWrite(XSHUT,LOW);  // VL53L0X reset is active LOW
  LTOF_On = 0;              // > 0 when laser ranging is enabled, default is 0 == OFF
}


// ---------------------------------------------------------------------------------

void VL53L0X_ON () {
  // enable the VL53L0X for reading ranges
  digitalWrite(XSHUT,HIGH); // enable the VL53L0X
  VL53L0X_init(true); delay(10);
  VL53L0X_setTimeout(500);

  // Start continuous back-to-back mode (take readings as
  // fast as possible).  To use continuous timed mode
  // instead, provide a desired inter-measurement period in
  // ms (e.g. sensor.startContinuous(100)).
  VL53L0X_startContinuous(); delay(10);
  LTOF_On = 1; // restart the measurement statemachine
}
