// ################################################################################
//
//  Biped Droid Release v0.00
//
//  Released:  20/06/2020
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    microcontroller:  NANO Every

    This code controls a Biped surveillance droid. It assumes that the servo calibration
    process has been completed successfully and that the limits are included here.

    This version includes:

    * Power-on stance, to limit random servo moves
    * reading LTOF range sensor
    * driving neopixel RGB LEDs on two buses
    * driving a PCA9685 16 x 12-bit PWM driver
    * move sequence memory, loaded from serial port
    * speed govenor for joystick demands
    * reading a MPU6050 gyro - TBD
*/
// Declare Libraries
#include <Wire.h>
#include <Adafruit_PWMServoDriver.h>
#include <Adafruit_NeoPixel.h>

// create I2C PWM driver instance for PCA9685 board
Adafruit_PWMServoDriver I2Cpwm = Adafruit_PWMServoDriver(0x40);

// define VL53LOX constants
#define ADDRESS_DEFAULT 0b0101001 // I2C address 0x29 HEX
#define calcMacroPeriod(vcsel_period_pclks) ((((uint32_t)2304 * (vcsel_period_pclks)* 1655) + 500) / 1000)
#define checkTimeoutExpired() (io_timeout > 0 && ((uint16_t)millis() - timeout_start_ms) > io_timeout)
#define decodeVcselPeriod(reg_val) (((reg_val) + 1) << 1)
#define encodeVcselPeriod(period_pclks) (((period_pclks) >> 1) - 1)
#define startTimeout() (timeout_start_ms = millis())
#define XSHUT 2  // D2 is connected to the XSHUT pin

// Declare register list from VL53L0X.h API, ordered as listed there
#define SYSRANGE_START 0x00
#define SYSTEM_THRESH_HIGH 0x0C
#define SYSTEM_THRESH_LOW 0x0E
#define SYSTEM_SEQUENCE_CONFIG 0x01
#define SYSTEM_RANGE_CONFIG 0x09
#define SYSTEM_INTERMEASUREMENT_PERIOD 0x04
#define SYSTEM_INTERRUPT_CONFIG_GPIO 0x0A
#define GPIO_HV_MUX_ACTIVE_HIGH 0x84
#define SYSTEM_INTERRUPT_CLEAR 0x0B
#define RESULT_INTERRUPT_STATUS 0x13
#define RESULT_RANGE_STATUS 0x14
#define RESULT_CORE_AMBIENT_WINDOW_EVENTS_RTN 0xBC
#define RESULT_CORE_RANGING_TOTAL_EVENTS_RTN 0xC0
#define RESULT_CORE_AMBIENT_WINDOW_EVENTS_REF 0xD0
#define RESULT_CORE_RANGING_TOTAL_EVENTS_REF 0xD4
#define RESULT_PEAK_SIGNAL_RATE_REF 0xB6
#define ALGO_PART_TO_PART_RANGE_OFFSET_MM 0x28
#define I2C_SLAVE_DEVICE_ADDRESS 0x8A
#define MSRC_CONFIG_CONTROL 0x60
#define PRE_RANGE_CONFIG_MIN_SNR 0x27
#define PRE_RANGE_CONFIG_VALID_PHASE_LOW 0x56
#define PRE_RANGE_CONFIG_VALID_PHASE_HIGH 0x57
#define PRE_RANGE_MIN_COUNT_RATE_RTN_LIMIT 0x64
#define FINAL_RANGE_CONFIG_MIN_SNR 0x67
#define FINAL_RANGE_CONFIG_VALID_PHASE_LOW 0x47
#define FINAL_RANGE_CONFIG_VALID_PHASE_HIGH 0x48
#define FINAL_RANGE_CONFIG_MIN_COUNT_RATE_RTN_LIMIT 0x44
#define PRE_RANGE_CONFIG_SIGMA_THRESH_HI 0x61
#define PRE_RANGE_CONFIG_SIGMA_THRESH_LO 0x62
#define PRE_RANGE_CONFIG_VCSEL_PERIOD 0x50
#define PRE_RANGE_CONFIG_TIMEOUT_MACROP_HI 0x51
#define PRE_RANGE_CONFIG_TIMEOUT_MACROP_LO 0x52
#define SYSTEM_HISTOGRAM_BIN 0x81
#define HISTOGRAM_CONFIG_INITIAL_PHASE_SELECT 0x33
#define HISTOGRAM_CONFIG_READOUT_CTRL 0x55
#define FINAL_RANGE_CONFIG_VCSEL_PERIOD 0x70
#define FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI 0x71
#define FINAL_RANGE_CONFIG_TIMEOUT_MACROP_LO 0x72
#define CROSSTALK_COMPENSATION_PEAK_RATE_MCPS 0x20
#define MSRC_CONFIG_TIMEOUT_MACROP 0x46
#define SOFT_RESET_GO2_SOFT_RESET_N 0xBF
#define IDENTIFICATION_MODEL_ID 0xC0
#define IDENTIFICATION_REVISION_ID 0xC2
#define OSC_CALIBRATE_VAL 0xF8
#define GLOBAL_CONFIG_VCSEL_WIDTH 0x32
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_0 0xB0
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_1 0xB1
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_2 0xB2
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_3 0xB3
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_4 0xB4
#define GLOBAL_CONFIG_SPAD_ENABLES_REF_5 0xB5
#define GLOBAL_CONFIG_REF_EN_START_SELECT 0xB6
#define DYNAMIC_SPAD_NUM_REQUESTED_REF_SPAD 0x4E
#define DYNAMIC_SPAD_REF_EN_START_OFFSET 0x4F
#define POWER_MANAGEMENT_GO1_POWER_FORCE 0x80
#define VHV_CONFIG_PAD_SCL_SDA__EXTSUP_HV 0x89
#define ALGO_PHASECAL_LIM 0x30
#define ALGO_PHASECAL_CONFIG_TIMEOUT 0x30

// define VL53L0C data structures
struct SequenceStepEnables {boolean tcc, msrc, dss, pre_range, final_range;};
struct SequenceStepTimeouts {
      uint16_t pre_range_vcsel_period_pclks, final_range_vcsel_period_pclks;
      uint16_t msrc_dss_tcc_mclks, pre_range_mclks, final_range_mclks;
      uint32_t msrc_dss_tcc_us, pre_range_us, final_range_us;};

// declare VL53L0X variables
uint8_t address = ADDRESS_DEFAULT;
bool did_timeout;
uint16_t io_timeout;
uint8_t last_status;      // status of last I2C transmission
uint32_t measurement_timing_budget_us;
byte Pulse = 0;           // unfiltered Range transient pulse counter
uint16_t Range;           // raw 16-bit range value
int RangeFtd;             // filtered range value used by general code
uint16_t RangeLast;       // 16-bit range value
bool RangeNew;            // set to true when a new Range value is determined
uint8_t Status;           // range status data, indicates quality of range value
uint8_t stop_variable;    // read by init and used when starting measurement; is StopVariable field of VL53L0X_DevData_t structure in API
uint16_t timeout_start_ms;
enum vcselPeriodType {VcselPeriodPreRange, VcselPeriodFinalRange};
void VL53L0X_startContinuous(uint32_t period_ms = 0);
inline void VL53L0X_setTimeout(uint16_t timeout) { io_timeout = timeout; }
bool VL53L0X_timeoutOccurred(void);
bool VL53L0X_getSpadInfo(uint8_t * count, bool * type_is_aperture);

// Define general constants
#define AT_Cmd 3          // pin assigned to Open Smart AT command signal
#define BattMin 804       // minimum A0 reading to trigger a battery LOW event
#define BattPin A0        // analogue pin used for battery monitoring
#define ColBlu 0x000088;  // pallet colour
#define ColCya 0x004444;  // pallet colour
#define ColGrn 0x008800;  // pallet colour
#define ColPur 0x440044;  // pallet colour
#define ColRed 0x880000;  // pallet colour
#define ColYel 0x444400;  // pallet colour
#define ledPin0 9         // RGB LED serial pin for body LEDs
#define ledPin1 10        // RGB LED serial pin for eye LEDs
#define ledNum0 5         // number of body RGB LEDs connected
#define ledNum1 2         // number of eye RGB LEDs connected
#define LtofDet 335       // max range for target detection
#define LtofMax 350       // out of range range value
#define MemRows 100       // no. rows in the move memory array
#define MemWidth 11       // no. of bytes in a memory row, servo angles + steps
#define Pin_OE 5          // define PCA9685 OE pin as D5
#define SteerMax 16       // steering gain, 16 == 50%, larger reduces gain
#define SteerRng 8        // range of JoyX steering +/-
#define sw0Pin 11         // left switch SW0 assigned to pin 11
#define sw1Pin 12         // right switch SW1 assigned to pin 12
#define walkIntDef 20000  // default walk interval 20ms (50Hz)
#define WiFiRestDef 500   // go to rest delay

// NeoPixel parameters
// Parameter 1 = number of pixels in strip
// Parameter 2 = pin number (most are valid)
// Parameter 3 = pixel type flags, add together as needed:
//   NEO_RGB     Pixels are wired for RGB bitstream
//   NEO_GRB     Pixels are wired for GRB bitstream
//   NEO_KHZ400  400 KHz bitstream (e.g. FLORA pixels)
//   NEO_KHZ800  800 KHz bitstream (e.g. High Density LED strip)
//Adafruit_NeoPixel strip = Adafruit_NeoPixel(ledNum, ledPin, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel strip0 = Adafruit_NeoPixel(ledNum0, ledPin0, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel strip1 = Adafruit_NeoPixel(ledNum1, ledPin1, NEO_GRB + NEO_KHZ800);

// Define servo mapping to PCA9685 channel numbers
// Not all channel numbers are included. Refer to calibration documentation.
// Channels 5, 6, 7, 8, and 9 are not connected
#define SH0 10  // head servo
#define SL0 0   // left foot
#define SL1 1   // left ankle
#define SL2 2   // left knee
#define SL3 3   // left hip 'swing'
#define SL4 4   // left hip 'sway'
#define SR0 15  // right foot
#define SR1 14  // right ankle
#define SR2 13  // right knee
#define SR3 12  // right hip 'swing'
#define SR4 11  // right hip 'sway'

// Define PWM freq and servo calibration arrays
#define Freq 50  // MG996R servos run at ~50 Hz updates
// Servo:   { 0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F }
int SLL[] = {164,180,221,162,258,163,163,163,163,163,189,237,189,118,177,176};  // servo lower limit values
int SRV[] = {295,310,367,298,295,306,306,306,306,306,315,271,326,248,310,309};  // reference values used to set a servo
int SUL[] = {435,442,491,435,331,449,449,449,449,449,448,304,456,388,451,442};  // servo upper limit values

// 3-axis MPU 6050 variables
const int MPU_addr=0x68;      // I2C address of the MPU-6050
int16_t AcX;                  // MPU raw X-acceleration value
int16_t AcXOff = 0;           // offset to be added to raw X-acceleration value
int16_t AcY;                  // MPU raw Y-acceleration value
long AcYaV;                   // averaged AcY value to reduce noise
float AcYF;                   // floating point Y acceleration value
int16_t AcYOff = 0;           // offset to be added to raw X-acceleration value
bool AcYsgn = false;          // positive/negative sign flag
long AcYsum;                  // averaging accumulator


// Declare and initialise global variables
int Angle;                    // temp angle variable used in angle calculations
int Any;                      // temp integer
int AnyAngle;                 // temp angle variable used in angle calculations
long AnyPWM;                  // temp PWM value used in movements
bool AtRest = false;          // = true when servos driven to rest position
int AutoOffTimer;             // timer used to auto-power off servos
int BattAv;                   // average battery voltage
int BattCnt;                  // battery failed 1s time-out counter
int BattP = 0;                // battery estimated percentage of capacity
int BattSum;                  // cumulative battery voltage
float BattV = 0.0;            // battery voltage
int BattVol;                  // instantaneous battery voltage
int C_Cnt;                    // counter used in 'C' button detection
byte C_Dn;                    // counter for 'C' button down period
byte C_Up;                    // counter for 'C' button up period
byte checksum;                // XOR Rx data to detect corruption at edge of range
int ChMap[] = {15,14,13,12,11,0,1,2,3,4}; // row point to servo channel map
int ChMapPnt = 0;             // variable used in array priming
char cmdMode = ' ';           // command mode
int cmdSgn = 0;               // cmdVal sign, 0 or -1
char cmdType = ' ';           // command mode type
int cmdVal = 0;               // value associated with a cmdType
uint8_t colBlu[] = {0,0,0,0,0,0,0,0,0,0}; // mouth colour array - blue
uint8_t colB = 255;           // mouth colour flag - blue
uint8_t colGrn[] = {0,0,0,0,0,0,0,0,0,0}; // mouth colour array - green
uint8_t colG = 255;           // mouth colour flag - green
uint8_t colRed[] = {0,0,0,0,0,0,0,0,0,0}; // mouth colour array - red
uint8_t colR = 255;           // mouth colour flag - red
int CZ = 0;                   // received button values
boolean CZ_Wait;              // flag used for button release wait function
bool Detect;                  // target detection flag; true detected
int DetectCnt;                // target detection counter
bool Echo;                    // if true send print messages
bool ESC = false;             // exit current Main Task flag
uint8_t eyeB = 10;            // eye colour flag - blue
uint8_t eyeG = 0;             // eye colour flag - green
uint8_t eyeR = 0;             // eye colour flag - red
byte Gear;                    // gear for speed range, default = 1
int Head;                     // if > 0 then head is moving
int HeadAngle;                // any angle assoicated with head movement
int HdAngMin,HdAngMax;        // lower and upper limits in head scanning
int HeadCnt;                  // counter used in head tasks
int HeadInc;                  // angle increment used in head movement
unsigned long headInterval;   // main loop head movement interval in microseconds
unsigned long headMicros;     // main loop head moving task interval timer in microseconds
long HeadStep;                // the number of steps left in a head move
long HeadSteps;               // the total number of steps in a head move
int HeadTask;                 // head task pointer
bool HeadTOF = false;         // head timer overflow flag
byte I2CError = true;         // used to detect I2C transmission errors
int JoyD;                     // direction value 0 - 8
int JoyV;                     // joystick vector  = SQRT(SQ(JX) + SQ(JY))
int JoyX;                     // received value
int JoyXC;                    // received value offset
int JoyY;                     // received value
int JoyYC;                    // received value offset
char keyChar;                 // any keyboard character
int keyVal;                   // any keyboard value
unsigned long lastHeadMicros; // timer record used in overflow
unsigned long lastWalkMicros; // timer record used in overflow
int LEDCnt;                   // timer counter used for RGB LEDs
bool LEDMask = false;         // flag used to inhibit LED tasks
int LEDMthCnt = 0;            // general counter for mouth LEDs
int LEDMthNext = 0;           // next task pointer for mouth LEDs
int LEDMthPnt = 0;            // general pointer for mouth LEDs
int LEDMthTask = 0;           // task pointer for mouth LEDs
byte LEDMthVal = 0;           // values used in LED tasks
int LEDNextTask;              // task to return to after a wait period
byte LEDPhase = 0;            // LED show phase
bool LEDshow0 = false;        // LED show flag for mouth LEDs
bool LEDshow1 = false;        // LED show flag for eye LEDs
byte LEDSubTask;              // LED subtask pointer, 0 = default
int LEDTaskPnt;               // LED task pointer, 0 = no task
int LEDTaskWait;              // task wait counter
long LegStep;                 // the number of steps left in a leg move
long LegSteps;                // the total number of steps in a leg move
byte LP[] = {0,0,0,0,0};      // general pointers for LED mapping
int LTOF_On;                  // > 0 when laser ranging is enabled
unsigned long LTOF_Wait;      // delay used to prevent wasted time in LTOF readings
bool mainEn = true;           // when == false, used as a main task blocking flag
int mainHedCnt;               // any MainTask head movement counter
byte mainMode;                // used to control specific tasks
int mainSubCnt;               // any MainTask subtask counter
int mainSubNext;              // the next subtask performed after a wait
int mainSubTask;              // any MainTask subtask pointer
int mainSubWait;              // any subtask wait value
int MainTask;                 // main task pointer
int MainTaskNext;             // next main task pointer
int MainTaskWait;             // main task wait timer in 10ms steps
int Mem[MemRows * MemWidth];  // memory array used to develop sequences, 10 bytes are used for each move position
byte MemMax;                  // number of rows loaded into memory
byte MemPnt;                  // a pointer used in loading the Mem[] array
int MemRow = -1;              // Mem[] row pointer used in loading the array
int mouthCnt = 0;             // counter used in mouth LED tasks
int moveCycleCnt;             // counter the number of repeat cycles in movements
bool MovePause = false;       // flag used to introduce pauses in moves
int MoveState;                // move state machine pointer
bool MTM_Step = false;        // set = true if single stepping moves
unsigned long next10ms;       // 10ms loop timer
unsigned long next20ms;       // 20ms loop timer
unsigned long next40ms;       // 40ms loop timer
int overRun;                  // counter used to prevent undemanded movement
bool Pause;                   // if moving pause if true, continue if false
int Ping = 0;                 // 'ping' counter
String PrintTx = "";          // printed strings
uint8_t Prof1[] = { 60, 45, 30, 20, 20, 20, 20, 20, 20, 20}; // accelerating period
uint8_t Prof2[] = { 20, 20, 20, 20, 20, 20, 20, 30, 45, 60}; // decelerating period
uint8_t Prof3[] = { 60, 45, 30, 20, 20, 20, 20, 30, 45, 60}; // acc/decelerating period
int ProfPnt;                  // pointer to the Profn[] arrays
float pwmFreq = Freq;         // MG996R servos run at ~50 Hz updates
byte PWM_Phase;               // PWM ramp phase 1 = up, 0 = down
byte RateProf = 0;            // speed rate profile, 0 = constant
bool readWiiCall;             // flag prevents call stack overflow
int RTxTimeout;               // counter used to auto-reset if WiFi fails
int RxCZ;                     // buffered received CZ value
int RxJoyX;                   // buffered received JoyX value
int RxJoyY;                   // buffered received JoyY value
bool RxRec;                   // set true when a valid frame of data is received
int RxState;                  // receiver state machine state
int RxVal = -1;               // value received from serial Rx
byte SerialRx = 0;            // serial receive flag; 0 = Serial, 1 = Serial1
byte servoOE;                 // servo board Output Enable; default = 1  OFF
byte servoTgt;                // serevo target channel for future angle data.
int SI;                       // flag used in blanket servo loading
int SL1Av;                    // average angle used in steering
int SL2Av;                    // average angle used in steering
int SL3Av;                    // average angle used in steering
int SerPnt = 0;               // any servo pointer
int SpdMax;                   // maximum speed when moving
int SpdMid;                   // middle speed when moving
int SpdMin;                   // minimum speed when moving
int SpdTop;                   // current top speed when moving
int Speed;                    // default speed value
int SpeedCnt = 0;             // used to progressively increase speed on full demand
int SpeedTgt;                 // target speed figure, prevents sudden changes
int SR1Av;                    // average angle used in steering
int SR2Av;                    // average angle used in steering
int SR3Av;                    // average angle used in steering
//int Steer;                    // steering angle applied when walking
int sw0Cnt;                   // button switch counter
boolean sw0LastState;         // previous state of button switch, HIGH/LOW
boolean sw0State;             // state of read button switch pin
int sw0Timer;                 // timer used to detemine button sequences
int sw1Cnt;                   // button switch counter
boolean sw1LastState;         // previous state of button switch, HIGH/LOW
boolean sw1State;             // state of read button switch pin
int sw1Timer;                 // timer used to detemine button sequences
long SPP[16];                 // array of servo movement present positions
long SSP[16];                 // array of servo movement start positions
long Steps;                   // max number of steps to perform in a move
long STG[16];                 // array of servo movement targets
unsigned long t0,t1;          // millisecond code timers
int Task10ms = 0;             // 10ms subtask timer
int Task20ms = 0;             // 20ms subtask timer
int tgtCnt;                   // counter used in automatic servo move to target actions
bool Track;                   // false = retreat, true = track mode
bool Trans;                   // if true transpose angles placed in memory
int Walk;                     // if > 0 then walk in one of four directions
long WalkCnt;                 // a counter which records the walking process
long WalkInc;                 // counter increment value
unsigned long walkInterval;   // main loop walking interval in microseconds
int WalkLast;                 // previousl Walk value if any
long WalkLftDrive;            // drive factor for left-hand side, 128 = max, 0 = min
byte WalkMax;                 // speed factor 4 - 12
int WalkMem;                  // pointer to next move row in Mem[] to be read
byte WalkMode;                // set the mode when moving using Mem[] targets
unsigned long walkMicros;     // main loop walking task interval timer in microseconds
int WalkNext;                 // used to store the next walking mode
long WalkRgtDrive;            // drive factor for right-hand side, 128 = max, 0 = min
int WalkSpeed;                // value used to determine WalkMax
int WalkSpeedTO;              // Time-Out counter for WalkSpeed auto-reduction, counts to zero
bool WalkTOF = false;         // walk timer overflow flag
int WalkWait;                 // delay counter used in LTOF mode
int WiFiCntC = 0;             // C button counter for WiFi enable
bool WiFiEn = false;          // true when WiFi is enabled with prolonged 'Z' button activity
int WiFiRestCnt;              // counter used to goi to rest if no WiFi demands
byte Z_Dn;                    // button pressed state; 0 == UP, >= 1 == Down


void setup() {
  // Initialise 16 x 12-bit PWM driver I2C interface immediately to limit any kick from
  // some servos going rogue at power-up
  pinMode(Pin_OE, OUTPUT);
  digitalWrite(Pin_OE, HIGH); // disable all PWM outputs to start with
  Wire.begin();
  I2Cpwm.begin();
  I2Cpwm.setPWMFreq(pwmFreq); // set PCA9685 PWM frequency
  GoToRest();                 // drive servos to default positions

  pinMode(XSHUT,OUTPUT);
  digitalWrite(XSHUT,LOW);   // VL53L0X reset is active LOW

  pinMode(BattPin,INPUT); BattVol = analogRead(BattPin);  // capture the battery voltage
  pinMode(sw0Pin,INPUT_PULLUP); // pullup resistor for switch SW0
  pinMode(sw1Pin,INPUT_PULLUP); // pullup resistor for switch SW0
  
  pinMode(AT_Cmd,OUTPUT);
  digitalWrite(AT_Cmd,HIGH);  //  take WiFi transceiver out of AT command mode
  
  setDefaults();
  
  Serial.begin(115200);   // baud rate for Rx/Tx comms
  Serial1.begin(115200);  // baud rate for WiFi Rx/Tx comms

  // Initialise neopixel RGB LEDs
  strip0.begin();
  strip0.fill(0,0,5);  // strip.fill(color, first, count);
  strip0.show();       // Initialize all pixels to 'off'
  strip1.begin();
  strip1.fill(0,0,2);  // strip.fill(color, first, count);
  strip1.show();       // Initialize all pixels to 'off'

  runPOST();
  synchLoopTimers();    // reset loop timers
}

// ---------------------------------------------------------------------------------


void loop() {
  // main code which runs repeatedly as multi-tasking functions
  // continuously read the laser range finder, when enabled
  if (LTOF_On > 0) LTOF_Read_Tasks();
  if (micros() >= walkMicros) {Walk_Engine();}  // repeated to reduce task timing jitter
  if (micros() >= headMicros) {Head_Engine();}
  readSerial();

  ////////////////////////////////////////////////////////////////////////////////
  //
  //  10 ms non-critical subtasks
  //
  ////////////////////////////////////////////////////////////////////////////////
  // These tasks are run in a consequtive ripple to reduce the overhead on more
  // critical code. They performed in reverse case order and will all complete
  // within the 10ms period
  if (Task10ms > 0) {
    switch(Task10ms) {
      case 1: if (LEDshow0) {strip0.show(); LEDshow0 = false;} break;
      case 2: if ((Head < 1) && (Walk < 1)) {AutoPwrOFF();} break;
      case 3: readSW1(); break;
      case 4: readSW0(); break;
      case 5: if (LEDshow1) {strip1.show(); LEDshow1 = false;} break;
      case 6: LED_Task(); break;  // perform head LED tasks
    } Task10ms--; // count down the sub-tasks from 6 - 0 as we loop
  }
  else if (millis() >= next10ms) {next10ms = millis() + 10; Task10ms = 6;}

  if (micros() >= walkMicros) {Walk_Engine();}  // repeated to reduce task timing jitter

  ////////////////////////////////////////////////////////////////////////////////
  //
  //  20 ms Main subtasks
  //
  ////////////////////////////////////////////////////////////////////////////////
  // Every 20ms we check for a main task and do that if one is invoked. Then we
  // perform a series of none critical in a consequitive sequence.
  if (Task20ms > 0) {
    switch(Task20ms) {
      case 1: readBattery(0); break;  // battery monitoring task
//      case 2: readAccelY(); break;
//      case 3: readAccelX(); break;
      case 4: Head_Tasks(); break;    // head movement tasks
      case 5: MouthTask(); break;     // mouth LED tasks
      case 6: if (!readWiiCall) {readWii();} break; // respond to Wii commands
    } Task20ms--; // count down the sub-tasks from 2 - 0 as we loop
  }
  else if (millis() >= next20ms) {
    next20ms = millis() + 20; // set next targer time
    if (mainEn) {Do_Main_Tasks();}  // do main task unless it is blocked
    Task20ms = 6; // set the subtask pointer to the highest task
  }

  if (micros() >= walkMicros) {Walk_Engine();}  // repeated to reduce task timing jitter

  ////////////////////////////////////////////////////////////////////////////////
  //
  //  40 ms Print subtask
  //
  ////////////////////////////////////////////////////////////////////////////////
  // Every 40ms we check to see if there is anything in the print buffer and if so
  // we send a portion (32 chars) of it.
  if (millis() >= next40ms) {
    next40ms = millis() + 40; // set next targer time
    if (!Echo) {PrintTx = "";}  // Echo is false so empty print buffer
    if (PrintTx.length() > 0) {
      // characters in string buffer so send some/all of them
      if (PrintTx.length() <= 32) {
        if (SerialRx) {Serial1.print(PrintTx);} else {Serial.print(PrintTx);}
        PrintTx = "";   // empty the buffer
      } else {
        if (SerialRx) {Serial1.print(PrintTx.substring(0,32));} else {Serial.print(PrintTx.substring(0,32));}
        PrintTx = PrintTx.substring(32);
      }
      // if sending data we shorten the inter-packet delay as receipt will hold off the app sending more
      next40ms = millis() + 10;
    }
  }
}

// ---------------------------------------------------------------------------------

void loopWhileWalk() {
  // allows functions to rum the main loop whilst being called
  while (Walk > 0) {loop();}  
}

// ---------------------------------------------------------------------------------

void runPOST() {
  // called during start-up
  PrintTx = "\n\nBiped Droid v0.14";
  Serial.println(PrintTx); Serial1.println(PrintTx); delay(40);
  PrintTx = "Starting POST...";
  Serial.println(PrintTx); Serial1.println(PrintTx); delay(40);
  for (int zR = 0; zR < 10; zR++) {
    readBattery(0);
  } PrintTx = ""; readBattery(1);
  Serial.print(PrintTx); Serial1.print(PrintTx); delay(40);
  PrintTx = "LED test";
  Serial.println(PrintTx); Serial1.println(PrintTx); delay(40);
  NeoPixel_SetAllEyeLEDs(10,10,10); // turn all eye LEDs dim white
  strip1.show(); delay(10);
  NeoPixel_SetAllMthLEDs(10,10,10); // turn all mouth LEDs dim white
  strip0.show();
  initialiseMPU();                                        // initialise the accelerometer
  PrintTx = "Accelerometer initialised.";
  Serial.println(PrintTx); Serial1.println(PrintTx); delay(40);
  delay(1000);
  NeoPixel_SetAllEyeLEDs(0,0,0); // turn all eye LEDs OFF
  NeoPixel_SetAllMthLEDs(0,0,0); // turn all mouth LEDs OFF
  strip0.show(); delay(10);
  AutoOffTimer = 100; // set 1s timeout
  PrintTx = "POST complete!";
  Serial.println(PrintTx); Serial1.println(PrintTx); delay(40);
  PrintTx = "";
}

// ---------------------------------------------------------------------------------

void setDefaults() {
  // load default values
  AutoOffTimer = 0;       // timer used to auto-power off servos
  BattAv = 915;           // average battery voltage
  BattCnt = 0;            // battery failed 1s time-out counter
  BattSum = BattMin * 15; //  pre-set cumulative battery voltage
  C_Cnt = 0;              // counter used in 'C' button detection
  C_Dn = 0;               // counter for 'C' button down period
  C_Up = 50;              // counter for 'C' button up period
  CZ_Wait = false;        // flag used for button release wait function
  Detect = false;         // target detection flag; true detected
  DetectCnt = 0;          // target detection counter
  Echo = true;            // if true send print messages
  Gear = 1;               // gear for speed range, default = 1
  Head = 0;               // if > 0 then head is moving
  headInterval = 20000;   // main loop head movement interval in microseconds
  HeadTask = 0;           // head task pointer
  JoyX= 0;                // received value
  JoyY = 0;               // received value
  LEDCnt = 0;             // timer counter used for RGB LEDs
  LEDSubTask = 0;         // LED subtask pointer, 0 = default 
  LEDTaskPnt = 2;         // LED task pointer, 0 = no task
  LTOF_On = 0;            // > 0 when laser ranging is enabled, default is 0 == OFF
  MainTask = 0;           // main task pointer
  MemMax = MemRows;       // number of rows loaded into memory
  MoveState = 0;          // move state machine pointer
  overRun = 0;            // counter used to prevent undemanded movement
  Pause = false;          // if moving pause if true, continue if false
  ProfPnt = 0;            // pointer to the Profn[] arrays
  RangeLast = 600;        // 16-bit range value
  readWiiCall = false;    // flag prevents call stack overflow
  RangeNew = false;       // set to true when a new Range value is determined
  RTxTimeout = 0;         // counter used to auto-reset if WiFi fails
  RxCZ = 0;               // buffered received CZ value
  RxJoyX = 0;             // buffered received JoyX value
  RxJoyY = 0;             // buffered received JoyY value
  RxRec = false ;         // set true when a valid frame of data is received
  RxState = 0;            // receiver state machine state
  RxVal = -1;             // value received from serial Rx
  servoOE = 1;            // servo board Output Enable; default = 1  OFF
  servoTgt = 0;           // serevo target channel for future angle data.
  SI = 0;                 // flag used in blanket servo loading
  SetSpeedDefaults();
  SpeedCnt = 0;           // used to progressively increase speed on full demand
  sw0Cnt = 0;             // button switch counter
  sw0LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw0State = HIGH;        // state of read button switch pin
  sw0Timer = 0;           // timer used to detemine button sequences
  sw1Cnt = 0;             // button switch counter
  sw1LastState = HIGH;    // previous state of button switch, HIGH/LOW
  sw1State = HIGH;        // state of read button switch pin
  sw1Timer = 0;           // timer used to detemine button sequences
  tgtCnt = 0;             // counter used in automatic servo move to target actions
  Trans = false;          // if true transpose angles placed in memory
  Walk = 0;               // if >0 then walk in one of four directions
  walkInterval = walkIntDef;   // main loop walking interval in microseconds
  WalkLast = 0;           // previousl Walk value if any
  WalkLftDrive = 128;     // drive factor for left-hand side, 128 = max, 0 = min
  WalkMax = 4;            // speed factor 4 - 12
  WalkRgtDrive = 128;     // drive factor for right-hand side, 128 = max, 0 = min
  WalkSpeed = 1;          // value used to determine WalkMax, 1 - 5
  WalkSpeedTO = 0;        // Time-Out counter for WalkSpeed auto-reduction, counts to zero
  Z_Dn = 0;               // button pressed state; 0 = UP, 1 = Down

  // fill arrays with default values
  for (int zP = 0; zP < 16; zP++) {SPP[zP] = SRV[zP];}
  SetTargetsToPPs();
  clearMem();
}

// ---------------------------------------------------------------------------------

void synchLoopTimers() {
  // reset main loop timers
  headMicros = micros() + headInterval; // set head loop timer
  walkMicros = micros() + walkInterval; // set walk loop timer
  next10ms = millis() + 10;             // reset 10ms loop timer
  next20ms = millis() + 13;             // reset 20ms loop timer with 3ms phase to 10ms timer
  next40ms = millis() + 16;             // reset 40ms loop timer with 6ms phase to 10ms timer
}

// --------------------------------------------------------------------------------
