// ################################################################################
//
//  16 Ch Controller v0.00
//
//  Released:  20/06/2020
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    microcontroller:  NANO Every

    This code works in conjunction with a 'Processing' application to enable the user
    to control up to 16 servo motors connected to a PCA9685 controller. It responds
    to simple commands received on its serial ports, either from usb or WiFi.

    The following commands are supported:
    !       - RESET and export default values to 16-Ch controller app
    SFnn.   - set PWM frequency, nominally 50 Hz
    SLnn.   - set 'target' lower limit to nn
    SOn.    - set servo board OE. 0 = LOW, 1 = HIGH
    STnn.   - set 'target' servo No. to nn
    SUnn.   - set 'target' upper limit to nn
    SVnn.   - set 'target' servo PWM value to nn
    QM.     - respond with MQ2. for WiFi mode 2
*/
// Declare Libraries
#include <Wire.h>
#include <Adafruit_PWMServoDriver.h>
#include <Adafruit_NeoPixel.h>  // only included to turn OFF random pixels on power-up

// create I2C PWM driver instance for PCA9685 board
Adafruit_PWMServoDriver I2Cpwm = Adafruit_PWMServoDriver(0x40);

// Define general constants
#define AT_Cmd 3          // pin assigned to Open Smart AT command signal
#define BattMin 804       // minimum A0 reading to trigger a battery LOW event
#define BattPin A0        // analogue pin used for battery monitoring
#define ledPin0 9         // RGB LED serial pin for body LEDs
#define ledPin1 10        // RGB LED serial pin for eye LEDs
#define ledNum0 5         // number of body RGB LEDs connected
#define ledNum1 2         // number of eye RGB LEDs connected
#define Pin_OE 5          // define PCA9685 OE pin as D5

// NeoPixels code only included to reset pixels on power up
//Adafruit_NeoPixel strip = Adafruit_NeoPixel(ledNum, ledPin, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel strip0 = Adafruit_NeoPixel(ledNum0, ledPin0, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel strip1 = Adafruit_NeoPixel(ledNum1, ledPin1, NEO_GRB + NEO_KHZ800);

// Define PWM freq and servo calibration arrays
#define Freq 50  // MG996R servos run at ~50 Hz updates
// Servo:   { 0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F }
int SLL[] = {164,180,221,162,258,163,163,163,163,163,189,237,189,118,177,176};  // servo lower limit values
int SRV[] = {295,310,367,298,295,306,306,306,306,306,315,271,326,248,310,309};  // reference values used to set a servo
int SUL[] = {435,442,491,435,331,449,449,449,449,449,448,304,456,388,451,442};  // servo upper limit values

// Declare and initialise global variables
int BattAv;               // average battery voltage
int BattSum;              // cumulative battery voltage
int BattVol;              // instantaneous battery voltage
bool Cal = false;         // if == true enables immediate rest, only set after calibration
char cmdMode = ' ';       // command mode
char cmdType = ' ';       // command mode type
int cmdVal = 0;           // value associated with a cmdType
char keyChar;             // any keyboard character
int keyVal;               // any keyboard value
int mouthCnt = 0;         // counter used in mouth LED tasks
unsigned long next20ms;   // 20ms loop timer
unsigned long next40ms;   // 40ms print loop timer
String PrintTx = "";      // printed strings
float pwmFreq = Freq;     // MG996R servos run at ~50 Hz updates
byte SerialRx = 0;        // serial receive flag; 0 = Serial, 1 = Serial1
byte servoOE = 1;         // servo board Output Enable; default = 1  OFF
int Servo_PWM;            // servo pulse width 800 - 2200
int servoTgt = 0;         // target servo for manual setting; default = 0

void setup() {
  // Setup code, runs once:
  // Initialise 16 x 12-bit PWM driver I2C interface immediately
  pinMode(Pin_OE, OUTPUT);
  digitalWrite(Pin_OE, HIGH); // disable all PWM outputs to start with

  pinMode(AT_Cmd,OUTPUT);
  digitalWrite(AT_Cmd,HIGH);  //  take WiFi transceiver out of AT command mode
  pinMode(BattPin,INPUT); BattVol = analogRead(BattPin);  // capture the battery voltage

  Serial.begin(115200);   // baud rate for Rx/Tx comms
  Serial1.begin(115200);  // baud rate for WiFi Rx/Tx comms

  // Initialise neopixel RGB LEDs
  strip0.begin();
  strip0.fill(0,0,5);  // strip.fill(color, first, count);
  strip0.show();       // Initialize all pixels to 'off'
  strip1.begin();
  strip1.fill(0,0,2);  // strip.fill(color, first, count);
  strip1.show();       // Initialize all pixels to 'off'

  Wire.begin();
  I2Cpwm.begin();
  I2Cpwm.setPWMFreq(pwmFreq);  // set PCA9685 PWM frequency
  if (Cal) {GoToRest();}      // go immediately to rest position
  ClearPCAop();

  PrintTx = "\n\n16-Ch Controller v0.00\n";
  SetDefaults();
  next20ms = millis() + 10; // reset 20ms loop timer
  next40ms = millis() + 20; // reset 40ms loop timer with 10ms phase to 20ms timer
  ExportArrays();           // send the contents of the arrays to the 16-Ch controller app
}

// ---------------------------------------------------------------------------------

void(* resetFunc) (void) = 0;//declare reset function at address 0

// ---------------------------------------------------------------------------------

void loop() {
  // main code which runs repeatedly as multi-tasking functions
  keyVal = Serial.read(); if (keyVal != -1) {SerialRx = 0; decodeKey(keyVal);}
  keyVal = Serial1.read(); if (keyVal != -1) {SerialRx = 1; decodeKey(keyVal);}

  ////////////////////////////////////////////////////////////////////////////////
  //
  //  20 ms tasks
  //
  ////////////////////////////////////////////////////////////////////////////////
  // Every 20ms we read the battery voltage and set a lamp periodically to indicate
  // is capacity
  if (millis() >= next20ms) {
    next20ms = millis() + 20; // set next target time
    readBattery();
  }

  ////////////////////////////////////////////////////////////////////////////////
  //
  //  40 ms Print subtask
  //
  ////////////////////////////////////////////////////////////////////////////////
  // Every 40ms we check to see if there is anything in the print buffer and if so
  // we send a portion (32 chars) of it.
  if (millis() >= next40ms) {
    next40ms = millis() + 40; // set next target time
    PrintTxPacket();
  }
}

// ---------------------------------------------------------------------------------

void SetDefaults() {
  // load a range of default values  
  BattAv = 915;           // average battery voltage
  BattSum = BattMin * 12; //  pre-set cumulative battery voltage
}
