// ################################################################################
//
//  Wii WiFi Auto v0.00
//
//  Released:  07/06/2019
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Elements of this software were taken from the public domain. It uses
    libraries which must be installed on your system.

    This code is for the Nunchuk wireless controller unit. It sends user demands
    over the Wi-Fi and gives indication of this on the LED. Data is sent for a
    joystick demand and for buttons C and Z. Depending on the mode of operation it
    will either send data formated for the Balance Bot or data formatted for the
    critter 'RC' robot. Data received from either robot is sent transparently to
    the serial monitor when reporint mode is disabled. The Z button status is read
    during POST to set the mode of operation, and therefore the type of data sent
    to the transceiver.

    This version also includes battery monitoring, which stops the code from running
    once a critical low voltage level is reached. As the centre points of Nunchuks
    can vary, the code sets deadband limts based on the initial readings. During POST
    it sends AT+ commands to the transciever such that it works in high speed
    115200 baud for data transfer and graphing.

*/
// Declare constants
#define AT_Cmd 10   // pin connected to transceiver CMD input
#define BatCritical 634 // critical battery voltage threshold
#define BatVolPin A0   // analog input pin connected to battery voltage divider
#define BatWarning 655 // warning battery voltage threshold
#define DbHalf 8 // value used as half of the joystick deadbands
#define interval 40 // main loop period is 40 ms (25Hz)
#define LEDPin 13   // LED pin
#define WiiErrRet 19 // error threshold used in I2C initialisation

// Declare libraries
#include <Wire.h>           // Include the Wire.h library for I2C comms
#include <SoftwareSerial.h> // Included serial for WiFi comms

// Create serial ports over which data will be sent to the transceiver 
SoftwareSerial SoftSerial(12, 11);  // Rx pin = 12, Tx pin = 11

// Declare and initialise global variables
int BatAvg = 1023;                  // preload averaging function with high value
bool BatPwr = false;                // true if battery supply detected
int BatQ;                           // a measure of battery quality 1 - 5
int BatSum = 10230;                 // preload averaging sum with high value
int BatVol = 0;                     // instantaneous battery voltage
byte checksum;                      // checksum byte for Critter RC data
char cmdMode = ' ';                 // command mode
char cmdType = ' ';                 // command type
byte DbLLX = 80;                    // BalanceBot joystick deadband lower limit
byte DbULX = 170;                   // BalanceBot joystick deadband upper limit
byte DbLLY = 80;                    // BalanceBot joystick deadband lower limit
byte DbULY = 170;                   // BalanceBot joystick deadband upper limit
int extraMillis = 0;                // time added if receiving data over WiFi
byte JoyXOff = 128;                 // value to correct stick centre X offset
byte JoyYOff = 128;                 // value to correct stick centre Y offset
char keyChar;                       // any keyboard character
int keyVal;                         // any keyboard value
int LedMax = 30;                    // LED slow flash timer
int LedON = 30;                     // LED flash counter
int Mode = 0;                       // data mode, 0 = Balance Bot, 1 = Critter
unsigned long nextMillis = 0;       // loop timer trigger
int nunchuk_address = 0x52;         // Nunchuk I2C address (0x52)
byte received_data[6];              // Array storing data received from Nunchuk
int reportEn = 0;                   // reports data for testing if >= 1. Set this to 0 for transparent telemetry mode
byte RxByte = 0;                    // byte received from WiFi serial link
byte send_byte;                     // bit pattern sent to BalanceBot over WiFi
byte TxByte = 0;                    // byte sent to WiFi serial link
int WiiErrCnt = 0;                  // number of attempts to initialse the Nunchuk
byte WiiError = 1;                  // used to detect transmission errors
bool XYOffAdj = true;               // if true adjust XY centre offsets

void setup(){
  // Start serial devices and connect to the Nunchuk over I2C
  pinMode(LEDPin, OUTPUT); digitalWrite(LEDPin, LOW);   // turn LED OFF initially
  pinMode(AT_Cmd, OUTPUT); digitalWrite(AT_Cmd, HIGH); // set transceiver to work in transparent mode
  Serial.begin(115200);                         // Start the USB serial port at 115200 bps
  SoftSerial.begin(115200);                     // Start the software serial port at 9600 bps
  SoftSerial.listen();                          // assign listening to this port
  runPOST();                                    // send power-on message
  nextMillis = millis() + interval;             // prime loop timer
}

// --------------------------------------------------------------------------------

void loop(){
  // read the Nunchuk every 40ms and send joystick data if detected
  // X/Y joystick readings are 0 - 255, centre 127/128 respectively.
  // We include a deadband of 80 - 170 for the BalanceBot.
  // The LED is ON when a joystick or button demand is being sent
  // Data is continuously sent for the Critter RC robot, including a checksum byte.
  ReadNunchukData(); // read the Nunchuk data registers
  // the default Mode = 0 is to send Balance Bot data, for which we construct a
  // single byte of data
  send_byte = B00000000;                        //Set the send_byte variable to 0
  if(received_data[0] < DbLLX) send_byte |= B00000001;   //If the variable received_data[0] is smaller then 80 set bit 0 of the send byte variable
  if(received_data[0] > DbULX) send_byte |= B00000010;  //If the variable received_data[0] is larger then 170 set bit 1 of the send byte variable
  if(received_data[1] < DbLLY) send_byte |= B00001000;   //If the variable received_data[1] is smaller then 80 set bit 3 of the send byte variable
  if(received_data[1] > DbULY) send_byte |= B00000100;  //If the variable received_data[1] is larger then 170 set bit 2 of the send byte variable
  if((received_data[5] & 1) == 0) send_byte |= B00010000;  //If Z button pressed set bit 4
  if((received_data[5] & 2) == 0) send_byte |= B00100000;  //If C button pressed set bit 5
  if(send_byte > 0){
      // valid data to be sent
    LedON = LedMax; // reset the LED blink timer
    digitalWrite(LEDPin, HIGH); // Set LED to come ON constantly
    if (Mode < 1) {
        // Mode == 0 hence send Balance Bot data byte
        SoftSerial.print("<");                // send data packet header
        SoftSerial.print((char)send_byte);    // send the send_byte variable if it's value is larger then 0
        SoftSerial.print(">");                // send data packet header
    } 
  }
  if (Mode > 0) {
    // Mode == 1 hence constantly send Critter RC data, irrespective of send_byte
    // baud rate = 9600, gives approx 960 characters per second
    // at 25Hz we can send blocks of 38 characters max, so 6 is no problem
    // we send a 6 byte frame = [ 2 byte key + 3 bytes data + 1 byte checksum ]
    checksum = 0xAA ^ 0x55 ^ received_data[0] ^ received_data[1] ^ received_data[5];
    SoftSerial.write(0x0AA);    // send 1st part of key 1010 1010 = AA Hex
    SoftSerial.write(0x055);    // send 2nd part of key 0101 0101 = 55 Hex
    SoftSerial.write(received_data[0]);    // send Nunchuk joystick-X
    SoftSerial.write(received_data[1]);    // send Nunchuk joystick-Y
    SoftSerial.write(received_data[5]);    // send Nunchuk buttons C and Z
    SoftSerial.write(checksum);    // send frame checksum byte
  }
  // see if Nunchuk data reporting is enabled?
  if (reportEn > 0) {
    while (SoftSerial.available() > 0) {
      RxByte = SoftSerial.read(); // discard any received data from BalanceBot
    } reportEn++;
    if (reportEn > 5) {
      // report every 200ms
      reportEn = 1; ReportNunchukData();
    }
  }

  // read battery voltage and take a rolling average over 10 readings
  if (BatPwr) {
    BatVol = analogRead(BatVolPin);
    BatAvg = BatSum / 10; BatSum = BatSum - BatAvg + BatVol;
  //  Serial.println(BatAvg);
    if (BatAvg <= BatWarning) {LedMax = 8;} // flash LED quickly as a warning
    if (BatAvg < BatCritical) {
      // the battery is too low for the system to function reliably so STOP
      BatteryLow();
    }
  }

  // now wait for loop timer, whilst checking transceiver serial port
  // data reception will in effect slow down the sending of Nunchuk data so
  // as to avoid collisions and adat corruption
  extraMillis = 4; // zero additional loop time until received data is detected
  while (nextMillis > millis()) {
    // wait for timer to catch up
    if (reportEn < 1) {
      // running in transparent mode, where all data received from USB port is
      // sent over the WiFi link, and all data received from the WiFi link is
      // sent to the USB serial port
      if (SoftSerial.available() > 0) {
        RxByte = SoftSerial.read(); // byte received from wireless transceiver
        Serial.write(RxByte);       // pass this onto the USB serial port
        extraMillis++;              // count in order to extend timer
      }
      if (Serial.available() > 0) {
        TxByte = Serial.read();     // byte received from USB port
        SoftSerial.write(TxByte);   // pass this onto the wireless transceiver
        extraMillis++;              // count in order to extend timer
      }
      if (extraMillis >= 5) {
        // for every 5 bytes received extend the time by 1 ms
        // at 115200 baud one character is 86 µs long at 10 bits
        extraMillis -= 5; nextMillis++;
      }
    }
  }
  // hardware timer has reached count, so restart the loop
  nextMillis = nextMillis + interval;     //Create a 40+ millisecond loop delay
  
  // flash LED if Nunchuk data is being sent
  // or flash once every second if in Balance Bot mode or twice if in Critter mode
  if (LedON == 4) {
    // turn ON LED if in critter mode, to make it flash twice per cycle
    if (Mode > 0) {digitalWrite(LEDPin, HIGH);}
  } else if (LedON <= 0) {
    // blink every second if no demand is being sent
    digitalWrite(LEDPin, HIGH); LedON = LedMax; // reset 1 sec timer
  } else {
    // otherwise turn the LED OFF at the start of every cycle
    digitalWrite(LEDPin, LOW);
   }
  LedON--; // constantly decrement the LED flash counter
}

// --------------------------------------------------------------------------------

void BatteryLow() {
  // code comes here if battery voltage drops below critical limit
  // it never exits this function
  Serial.println(F("Critical voltage shutdown"));
  digitalWrite(LEDPin, LOW); // turn the LED OFF
  while (true) {} // endless loop that never ends, doing nothing
}

// --------------------------------------------------------------------------------

void FlashLED(int zNum) {
  // briefly flash the LED zNum times
  for (int zi = 1;zi < zNum + 1; zi++) {
    digitalWrite(LEDPin, HIGH); delay(10);
    digitalWrite(LEDPin, LOW); delay(200);
  }
}

// --------------------------------------------------------------------------------

void FlushRxBuffer() {
  // Read and ignore all characters in the serial port buffer  
  while (Serial.available() > 0) {
    // read data to empty buffer
    TxByte = Serial.read();
  }
}

// --------------------------------------------------------------------------------

void ReadNunchukData() {
  // read a block of data from the Nunchuk controller
  Wire.beginTransmission(nunchuk_address); //Start communication with the Nunchuck.
  Wire.write(0x00);                        //We want to start reading at register (00 hex)
  WiiError = Wire.endTransmission();       //End the transmission and check ACK
  delay(1);
  //
  // check to ensure that the Nunchuk has responded
  if (WiiError == 0) {
    // no error reported so read the six Nunchuk registers
    Wire.requestFrom(nunchuk_address,6);      //Request 6 bytes from the Nunchuck
    for(byte i = 0; i < 6; i++) received_data[i] = Wire.read(); //Copy the bytes to the received_data array
    received_data[5] = received_data[5] & 0x03; // mask off accelerometer bits
    // check data validity in case Nunchuk has crashed. Under such cercumstances
    // all data registers will appear to hold a value of 255
    // we ignore the joystick registers as they may actually be at 255
    if ((received_data[2] == 255) && (received_data[3] == 255) && (received_data[4] == 255)) {
      WiiError = 1;
    } else {
      // Nunchuk appears to be working correctly, so we will remove any offset
      // found during POST, unless this feature was disabled with the 'C' button
      if (XYOffAdj) {
        // XY offset adjustment enabled, so re-map received values
        if (received_data[0] <= JoyXOff) {
          received_data[0] = map(received_data[0],0,JoyXOff,0,128);
        } else {received_data[0] = map(received_data[0],JoyXOff,255,128,255);}
        if (received_data[1] <= JoyYOff) {
          received_data[1] = map(received_data[1],0,JoyYOff,0,128);
        } else {received_data[1] = map(received_data[1],JoyYOff,255,128,255);}
      }
    }
  }
  if (WiiError != 0) {
    // we re-initialise the Nunchuk if a failure in the I2C comms occurs
    // this loop is sustained indefinately until a device is detected
    WiiInitialise();
    while (WiiErrCnt > WiiErrRet) {
      Serial.println(F("Nunchuk initialisation failed!"));
      WiiInitialise();
    } Serial.print(F("Nunchuk initialised in ")); Serial.println(WiiErrCnt);
    nextMillis = millis() + interval; // re-prime loop timer
  }
}

// --------------------------------------------------------------------------------

void ReportNunchukData() {
  // send nunchuk data to the serial port, with formatting
  Serial.println(F("\nNunchuk Data:")); // send title
  for(byte i = 0; i < 6; i++) {
    Serial.print(F("Bite ")); Serial.print(i); Serial.print(F(" = "));
    if (received_data[i] < 100) Serial.print(F(" "));
    if (received_data[i] < 10) Serial.print(F(" "));
    Serial.print(received_data[i]);       //output I2C data from nunchuk
    Serial.print(F("\t"));
    // this section packs out the binary field to a constant 8 characters
    if (received_data[i] < 128) Serial.print(F("0"));
    if (received_data[i] < 64) Serial.print(F("0"));
    if (received_data[i] < 32) Serial.print(F("0"));
    if (received_data[i] < 16) Serial.print(F("0"));
    if (received_data[i] < 8) Serial.print(F("0"));
    if (received_data[i] < 4) Serial.print(F("0"));
    if (received_data[i] < 2) Serial.print(F("0"));
    Serial.println(received_data[i],BIN);       //output I2C data from nunchuk
  }
}

// --------------------------------------------------------------------------------

void runPOST() {
  // send power-on messages  
  Serial.println(F("\n\n\n\nWii WiFi Auto v0.00\n"));
  //
  ///////////////////////////////////////////////////////////////////////////////
  // Check battery health
  ///////////////////////////////////////////////////////////////////////////////
  // first thing to do is to check the barrery voltage
  Serial.println(F("Checking battery voltage..."));
  BatVol = analogRead(BatVolPin); // ignore the first reading
  delay(10); BatVol = analogRead(BatVolPin); // take the reading
  Serial.print(F("ADC = "));
  Serial.print(BatVol); Serial.print(F(" == "));
  // test for USB powered board @5v +0.25/-0.6
  if ((BatVol >= 440) && (BatVol <= 537)) {
    Serial.print((float(BatVol) * 10.0)/1023.0); Serial.println(F(" volts"));
    Serial.println(F("USB port connection assumed")); BatQ = 5;
  } else {
    BatPwr = true;
    Serial.print(((float(BatVol) * 10.0)/1023.0) + 0.77); Serial.println(F(" volts"));
    if (BatVol < BatCritical) {
      // battery voltage is at or below the critical limit
      BatteryLow();
    } else {
      // rate the quality of the battery voltage 1 - 5
      // my fully charged battery was measured at 8.65v
      // so fully charged limit is: (((8.65 - 0.77)/2) * 1023/5) = 806
      BatQ = map(BatVol,BatCritical,806,1,5);
    }
  } 
  Serial.print(F("Battery quality = "));Serial.println(BatQ);
  
  FlashLED(BatQ);
  delay(1000);

  
  ///////////////////////////////////////////////////////////////////////////////
  // Flash Transceiver for 115200 baud
  ///////////////////////////////////////////////////////////////////////////////
  // If new the transceiver should be programmed from it's default settings
  // To do this uncomment the following lines of code and run once to flash
  // the new settings into the transceiver, then comment out again.
//  Serial.println(F("\nFlashing transciever settings..."));
//  digitalWrite(AT_Cmd, LOW);          // set transceiver to AT+ command mode
//  SoftSerial.begin(9600);             // reset the baud rate to run AT+ commands
//  delay(50);                         // allow time for mode change      
//  SoftSerial.print(F("AT+BAUD=7"));   // set baud rate to 115200
//  delay(50);
//  SoftSerial.print(F("AT+RETRY=05")); // set failed retries to 5
//  delay(50);
//  SoftSerial.print(F("AT+POWER=09")); // set transmit power to -6dBm
//  delay(50);
//  digitalWrite(AT_Cmd,HIGH);          // return transceiver to transparent mode
//  SoftSerial.begin(115200);
//  SoftSerial.listen();                // assign listening to this port
//  // report new settings
//  Serial.println(F("Baud rate = 115200"));
//  Serial.println(F("Retries = 5"));
//  Serial.println(F("Tx Power = -6dBm"));


  ///////////////////////////////////////////////////////////////////////////////
  // Initialise the Nunchuk controller
  ///////////////////////////////////////////////////////////////////////////////
  // iniaitalise the Nunchuk and confirm its presence
  Serial.println(F("\nInitialising Nunchuk over I2C..."));
  WiiInitialise();
  while (WiiErrCnt > WiiErrRet) {
    Serial.println(F("Nunchuk initialisation failed!"));
    WiiInitialise();
  } Serial.print(F("Nunchuk initialised in ")); Serial.println(WiiErrCnt);
  ReadNunchukData();



  ///////////////////////////////////////////////////////////////////////////////
  // Establish robot control mode
  ///////////////////////////////////////////////////////////////////////////////
  // A command is sent to the robot for it to return the mode of control it is
  // expecting to use. If not response is received during the time period then
  // Mode = 0 will be assumed for self-balancing robots.
  Serial.println(F("\nChecking robot 'Mode' response..."));
  // turn ON the LED briefly to indicate WiFi signalling is occuring
  digitalWrite(LEDPin, HIGH); //Briefly turn ON the LED
  // check the Z button 100 times over a 1 second period
  int zCnt = 0; // count of number of responses received
  for (int zI = 0; zI < 5; zI++) {
    // five requests will be sent to the robot to gain a response
    // these will be sent at 200ms periods
    SoftSerial.print(F("QM.")); // send the request command
    unsigned long zTimeout = millis() + 200;
    cmdMode = ' '; cmdType = ' ';
    while (millis() < zTimeout) {
      // in this 200ms period we look for the MQn response from the robot
      // not that this process naturally flushed the SoftSerial i/p buffer
      if (SoftSerial.available() > 0) {
        // serial data is available
        keyVal = SoftSerial.read(); keyChar = char(keyVal);
        // check for the MQn. sequence and determine mode 'n'
        if (cmdMode == ' ') {
          if (keyChar == 'M') {cmdMode = 'M';}
        } else if (cmdType == ' ') {
          if (keyChar == 'Q') {cmdType = 'Q';}
          // if 'Q' does not follow 'M' we reset the process
          else {cmdMode = ' '; cmdType = ' ';}
        } else {
          // 'MQ' received so the next value is assumed to be a number
          if (keyChar == '0') {Mode = 0;}
          if (keyChar == '1') {Mode = 1;}
          zCnt++; cmdMode = ' '; cmdType = ' ';
        }
      }
    }
  }
  Serial.print(F("Received ")); Serial.print(zCnt); Serial.println(F(" MQn. responses"));
  // report the Mode received, or Mode = 0 is assumed
  switch (Mode) {
    case 0: Serial.println(F("Mode = 0 for BalanceBot")); break;
    case 1: Serial.println(F("Mode = 1 for Critter 'RC'")); LedMax = 34; break;
  }
  

  ///////////////////////////////////////////////////////////////////////////////
  // Check for Offset adjustment
  ///////////////////////////////////////////////////////////////////////////////
  // if the 'C' button was pressed at the end of this period turn OFF X/Y offset
  // correction, which is ON by default.
  ReadNunchukData();
  Serial.print(F("\nXY Offset Adjust is "));
  if((received_data[5] & 2) == 0) {
    XYOffAdj = false;
    Serial.println(F("OFF"));
  } else {
    // XY offset adjustment is on by default, so read stick offsets
    JoyXOff = received_data[0]; JoyYOff = received_data[1];
    Serial.println(F("ON"));
    Serial.print(F("Joy-X Offset = "));Serial.println(JoyXOff);
    Serial.print(F("Joy-Y Offset = "));Serial.println(JoyYOff);
  }
  
  digitalWrite(LEDPin, LOW); // turn LED OFF
  delay(50); // this pause puts a short blip in the LED to indicate now waiting for Z button release
  // now wait for button(s) to be released, if either is pressed
  ReadNunchukData();
  while (received_data[5] != 3) {
    // stay here until both the 'C' & 'Z' buttons are released
    digitalWrite(LEDPin, HIGH); // turn ON the LED whilst waiting for button release
    delay(10); ReadNunchukData(); // read the Nunchuk every 10ms
  } digitalWrite(LEDPin, LOW); // turn LED OFF

  // narrow the joystick deadbands to get more range
  ReadNunchukData();
  DbLLX = received_data[0] - DbHalf; DbULX = received_data[0] + DbHalf;
  DbLLY = received_data[1] - DbHalf; DbULY = received_data[1] + DbHalf;
  
  Serial.println(F("\nPOST completed."));
  if (reportEn < 1) {
    // reportEn is set by you at the front of this code, default = 0
    // change this to 1 to receive regular reports of Nunchuk data on serial monitor
    Serial.println(F("\nEntering transparent mode...\n"));
  } else {
    // reporting mode is enabled, so regular reports of data will be sent
    Serial.println(F("\nReport mode enabled...\n"));
  }
}

// --------------------------------------------------------------------------------

void WiiInitialise() {
  // innitialise the Wii over the I2C bus
  WiiError = 1; WiiErrCnt = 0;
  while (WiiError != 0) {
    // wait in this loop initialising the Wii until it responds
    Wire.begin();                                 //Start the I2C as master
    TWBR = 12;                                    //Set the I2C clock speed to 400kHz
    Wire.begin();                                 //Start the I2C bus as master
    delay(10);                                    //Short delay
    Wire.beginTransmission(nunchuk_address);      //Start communication with the Nunchuck
    Wire.write((byte)0xF0);                       //We want to write to register (F0 hex)
    Wire.write((byte)0x55);                       //Set the register bits as 01010101
    Wire.endTransmission();                       //End the transmission
    delay(10);                                    //Short delay
    Wire.beginTransmission(nunchuk_address);      //Start communication with the Nunchuck
    Wire.write((byte)0xFB);                       //We want to write to register (FB hex)
    Wire.write((byte)0x00);                       //Set the register bits as 00000000
    WiiError = Wire.endTransmission();            //End the transmission
    delay(10);                                    //Short delay
    WiiErrCnt++;                                  //Track the number of attempts
    if (WiiErrCnt > WiiErrRet) {return;}          // no Nunchuk detected
  }
}

// --------------------------------------------------------------------------------
