// ################################################################################
//
//  Ball Balance Bot 3x3 vR1
//
//  Released:  10/04/2022
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    This code controls a self-balancing robot using two independant PID controllers
    that respond to angular errors, based on vertical X & Y channel setpoints.

    This self-balancing version includes:
    - A Main loop running at 4ms (250Hz)
    - A battery monitoring function to prevent over-discharge
    - LED sequences which are brightness controlled
    - Two PID controllers X & Y combined into one output and an angle vector
    - Speed limiting is applied to each error vector to prevent run-away
    - Auto wireless 2.4GHz link setup with modified Tx/Rx ID
    - Reporting of PID variables in response to a request.
    - Upside down motor drive demonstrations
    - PID P-gain only demonstration

    
*/
// Declare libraries
#include <Wire.h> //Include the Wire.h library so we can communicate with the gyro

#define Released "10/04/2022"

// define constants
#define acc_1g  8192                // accelerometer 1g value
#define acc_calibration_valueX -221 // Enter the accelerometer calibration value, default 1000
#define acc_calibration_valueY 345  // 257 Enter the accelerometer calibration value, default 1000
#define acc_ramp_rate 30            // rate limit on accelerometers to filter vibration at small angles
#define AccVectSafe 714.0           // off-balance safe zone >= 5 degrees
#define AccVectTrgt 72.0            // 0.5 degrees horizontal balance target @ 8,192 LSB/g
#define AT_Cmd 2                    // pin attached to the transceiver CMD pin
#define BattMin 540                 // minimum average A0 reading to trigger a battery LOW event
#define BattSw0 16                  // define SW0 LOW threshold
#define Calibrate false             // default = false; set to true to list sensor values
#define DelValRes 50                // LED delay loop count in 4ms steps
#define DIRA 10                     // DIR pin for motor A left
#define DIRB 4                      // DIR pin for motor B right
#define DIRC 6                      // DIR pin for motor C rear
#define gyro_address 0x68           // MPU-6050 I2C address (0x68 or 0x69)
#define LEDBckRed A2                // rear red LED pin
#define LEDBckGrn A1                // rear green LED pin
#define LEDLftRed 8                 // left red LED pin
#define LEDLftGrn 7                 // left green LED pin
#define LEDRhtRed 12                // right red LED pin
#define LEDRhtGrn A3                // right green LED pin

// these are the critical controller settings to achieve self-balance
#define pid_p_gain_ref 80.0         // 80.0 Gain setting for the P-controller (1.0 - 100.0)
#define pid_i_gain_ref 3.2          // 3.2 Gain setting for the I-controller (1.18 - 5.0)
#define pid_d_gain_ref 32.0         // 32.0 Gain setting for the D-controller (10 - 60)
#define pid_s_gain_ref 0.0          // 0.005 Gain setting for PID speed governor (0.05 - 0.001)

#define pid_s_thresh 0.0            // cut-in threshold for speed governor
#define pid_output_trend_limit 100  // self balancing trend limit. Max angle = pid_output_trend_limit * self_balance_pid_inc
#define PWMA 11                     // PWM pin for motor A left
#define PWMB 3                      // PWM pin for motor B right
#define PWMC 9                      // PWM pin for motor C rear
#define REPORT false                // set to true if you want to send serial data every main cycle
#define TEST false                  // default = false. True invokes accelerometer drift measurement
#define Topple 60.0                 // from vertical (0°) angle limit causing motor shut-down
#define X_Axis_En true              // if false, shuts off X-axis calcs and drive
#define Y_Axis_En true              // if false, shuts off Y-axis calcs and drive

// DEBUG variables, not used by the main program
char cmdMode = ' ';                 // command mode
int cmdRec;                         // > 0 if a '~' has been received
int cmdSgn = 1;                     // value polarity multiplier
int cmdSteer = 0;                   // values used in motor steering
char cmdType = ' ';                 // command type
int cmdVal = 0;                     // value associated with a cmdType
int cmdXDrive = 0;                  // values used in motor testing
int cmdYDrive = 0;                  // values used in motor testing
char keyChar;                       // any keyboard character
int keyVal;                         // any keyboard value
int PID_SPJXC = 0;                  // PID setpoint X jolt counter
int PID_SPJXI = 1;                  // PID setpoint X jolt counter increment
bool PID_SPJX = false;              // PID setpoint X jolt flag
int PID_SPJYC = 0;                  // PID setpoint Y jolt counter
int PID_SPJYI = 1;                  // PID setpoint Y jolt counter increment
bool PID_SPJY = false;              // PID setpoint Y jolt flag
int RPnn = 0;                       // reporting reference
bool Steer_En = false;              // steering enabled flag
bool X_DriveEn = false;             // motor X-drive enabled flag
bool Y_DriveEn = false;             // motor Y-drive enabled flag

// Declare PID controller coeficients
bool PID_d_En = true;               // PID d-gain enable true=ON, false=OFF
float pid_p_gain = 88.0;            // Gain setting for the P-controller (1.0 - 15.0)
bool PID_i_En = true;               // PID i-gain enable true=ON, false=OFF
float pid_i_gain = 4.00;            // Gain setting for the I-controller (1.18 - 1.5)
float pid_d_gain = 32.0;            // Gain setting for the D-controller (10 - 30)
float pid_out_max = 255.0;          // limit PID output to prevent PWM overload
float pid_sb_dbX = 0.0;             // Self-balancing dead band limit
float pid_sb_dbY = 0.0;             // Self-balancing dead band limit
float pid_s_gain = 0.005;           // Gain setting for PID speed governor (0.05 - 0.001)

// PID variables
int acc_cal_valueX = acc_calibration_valueX;   // Enter the accelerometer calibration value, default 1000
int acc_cal_valueY = acc_calibration_valueY;   // Enter the accelerometer calibration value, default 1000
int accel_delta = 0;                  // raw accelerometer instantanious increase
int accelerometer_data_X;             // accelerometer data for the forward X axis
int accelerometer_data_Y;             // accelerometer data for the sideways Y axis
int accelerometer_data_rawX_i;        // instantaneous record for averaging
int accelerometer_data_rawX_OA;       // instantaneous record offset adjusted
int accelerometer_data_rawX_OALast;   // previous instantaneous record offset adjusted
int accelerometer_data_rawY_i;        // instantaneous record for averaging
int accelerometer_data_rawY_OA;       // instantaneous record offset adjusted
int accelerometer_data_rawY_OALast;   // instantaneous record offset adjusted
int accelerometer_data_rawZ_i;        // instantaneous record for averaging
int acc_Z_Trck;                       // Z accelerator tracking filter value           
float angle_accX;                     // accelerometer angle average value
float angle_accY;                     // accelerometer angle average value
float angle_accX_i;                   // accelerometer angle instantaneous value
float angle_accY_i;                   // accelerometer angle instantaneous value
float angle_gyroFX;                   // rate limited gyro X angle
float angle_gyroFY;                   // rate limited gyro Y angle
float angle_gyroX;                    // compound gyro angle
float angle_gyroY;                    // compound gyro angle
float angle_offset;                   // small offset to correct gyro angle drift
long gyro_pitch_calibration_value;    // gyro drift compensation value
int gyro_pitch_data_raw;              // gyro value read from MPU6050
long gyro_yaw_calibration_value;      // gyro drift compensation value
int gyro_yaw_data_raw;                // gyro value read from MPU6050
int PID_En_Cnt = 0;                   // counter used to enter gain disable modes
float pid_error_angle;                // angle determined from PID X & Y outputs
float pid_error_lastX;                // used in PID derivative calcs
float pid_error_lastY;                // used in PID derivative calcs
float pid_error_newX;                 // error vector used in PID calcs
float pid_error_newY;                 // error vector used in PID calcs
float pid_error_speedX;               // angular speed of robot used in PID calculations
float pid_error_speedY;               // angular speed of robot used in PID calculations
float pid_error_tempX;                // current PID X gyro error
float pid_error_tempY;                // current PID Y gyro error
float pid_d_ValX = 0.0;               // PID d gain X element
float pid_d_ValY = 0.0;               // PID d gain Y element
float pid_i_memX = 0.0;               // used in PID X integral calcs
float pid_i_memY = 0.0;               // used in PID Y integral calcs
float pid_last_d_errorX;              // used in PID derivative calcs
float pid_last_d_errorY;              // used in PID derivative calcs
float pid_output;                     // combined PDI output
float PID_Output_Power;               // used in soft start
float pid_outputX;                    // X channel PDI output
float pid_outputY;                    // Y channel PDI output
int pid_output_trendX = 0;            // tracker used in self-balancing
int pid_output_trendY = 0;            // tracker used in self-balancing
float pid_p_ValX = 0.0;               // PID p gain X element
float pid_p_ValY = 0.0;               // PID p gain Y element
float pid_setpointX;                  // setpoint angle when moving
float pid_setpointY;                  // setpoint angle when moving
int PID_TuneX;                        // > 0 when in tuning mode
int PID_TuneY;                        // > 0 when in tuning mode
float self_balance_pid_incX;          // values used to inc/dec balance setpoint (0.0015)
float self_balance_pid_incY;          // values used to inc/dec balance setpoint (0.0015)
float self_balance_pid_setpointX;     // offset used to set natural balance point
float self_balance_pid_setpointY;     // offset used to set natural balance point

// Declare and initialise general global variables
float AccAbsAng;          // abs acceleration angle expressed as a ratio
float AccVect;            // acceleration vector
float AccVectTrk;         // acceleration vector tracking filter
int ACM = 0;              // set speed compensation to 50%, determined during calibration
int anyCnt;               // temporary counter
int AnyD;                 // temp variable
float anyFX;              // temporary floating point variable
float anyFY;              // temporary floating point variable
float anyFloat;           // temporary floating point variable
int BattAv;               // average battery voltage
int BattCnt;              // battery failed 1s time-out counter
bool BattPhase;           // alternating phase controls battery/switch reading
int BattSum;              // cumulative battery voltage
int BattVol;              // instantaneous battery voltage
int C_Button;             // button pressed state; 0 = UP, 1 = Down
int DelVal = DelValRes;   // LED step delay counter
bool DIR_A;               // value writtent to H-bridge pin
bool DIR_B;               // value writtent to H-bridge pin
bool DIR_C;               // value writtent to H-bridge pin
int DithCnt;              // counter used in dither application
int DithVal;              // peak dither value
bool DriveEn;             // if == true then output PWM to motors
int GoActive;             // counter used at initial balance point
bool GyroOnce;            // sets the initial angles for the gyros based on the accelerometers
unsigned long GyT;        // time of gyro readings in milliseconds
unsigned long GyTD;       // time between successive gyro readings in milliseconds
byte I2C_Error;           // flag returned from I2C comms
unsigned long interval = 4000; // main loop timer in microseconds
long Joy_Y;               // > 0 when joystick Y is pressed
int LEDA_Cnt;             // counter preset used in PWM LED flashing
int LEDA_CntLoad;         // counter preset used in PWM LED flashing
bool LEDBckRedOn;         // true if LED is ON, default false
bool LEDBckGrnOn;         // true if LED is ON, default false
int LEDB_Cnt;             // counter preset used in PWM LED flashing
int LEDB_CntLoad;         // counter preset used in PWM LED flashing
unsigned long LEDBrightness; // LED brightness factor 0 - 10
int LEDC_Cnt;             // counter preset used in PWM LED flashing
int LEDC_CntLoad;         // counter preset used in PWM LED flashing
int LEDCnt;               // counter used with LEDs
bool LEDLftRedOn;         // true if LED is ON, default false
bool LEDLftGrnOn;         // true if LED is ON, default false
int LEDMode;              // LED mode pointer
bool LEDs_ON;             // true if LEDs are ON, being timed, false if OFF
unsigned long LEDOnTime;  // tinmer used in brightness control
int LED_Override;         // if >0 counts down locking the current LED settings
bool LEDPhase;            // alternating LED phase cycle
bool LEDRhtGrnOn;         // true if LED is ON, default false
bool LEDRhtRedOn;         // true if LED is ON, default false
int LED_Task;             // LED task pointer
int MainMode;             // controls which tasks are performed, default = 0
bool MotorCal;            // controls motor PWM calibration, default = true
float ModeAngle;          // angle counter used in MainMode tasks
int ModeCnt;              // counter used in MainMode tasks
int ModeCntX;             // X counter used in MainMode tasks
int ModeCntY;             // Y counter used in MainMode tasks
int ModeCycle;            // cycle counter used in MainMode tasks
float ModePID;            // PID output demand used in MainMode tasks
float ModeSteer;          // steering demand used in MainMode tasks
int ModeTask;             // task pointer used in MainMode tasks
int ModeTaskNext;         // task pointer used in MainMode tasks
int ModeValX;             // value used in MainMode tasks
int ModeValY;             // value used in MainMode tasks
int MotorDriveA;          // motor drive value
int MotorDriveB;          // motor drive value
int MotorDriveC;          // motor drive value
int MotorMaxPWM;          // maximum PWM in either rotation, default = 255
int MotorMinPWMip;        // minimum PWM demand point, default = 10
int MotorMinPWMop;        // minimum PWM output point, default = 50
int MotorPWM50;           // remapped PWM = 50 in anti-clockwise rotation
int MotorPWM100;          // remapped PWM = 100 in anti-clockwise rotation
int MotorPWM150;          // remapped PWM = 150 in anti-clockwise rotation
int MotorPWM200;          // remapped PWM = 200 in anti-clockwise rotation
int Mot_Task;             // motor task pointer
unsigned long nextMicros; // main loop timer in microseconds
int OffBalance;           // flag capturing initial off-balance condition
int PWM_A;                // PWM value destined for H-bridge driver
int PWM_DA;               // PWM value written to H-bridge driver
int PWM_B;                // PWM value destined for H-bridge driver
int PWM_DB;               // PWM value written to H-bridge driver
int PWM_C;                // PWM value destined for H-bridge driver
int PWM_DC;               // PWM value written to H-bridge driver
int readTask;             // task pointer used in decoding received control data
int receive_counter;      // used to extend received data for 100 ms
byte received_byte;       // received byte from WiFi reciever
bool received_cmd;        // set true if valid control has ben recevied
long RPms = 0;            // time between receiving lasp R. request
bool RP_ON;               // if true then report values in DEBUG mode
int RPP;                  // reporting phase pointer
long RPt0 = millis();     // millis() tracker for R. requests
bool RP_Title;            // == true if title sent to the PID controller graph
int safeMode;             // start mode of safety state machine
int safeModeLast;         // previous safe mode
int safePntCnt;           // print timer counter
bool startEn;             // true to perform motor calcs and enable output
float Steer;              // steering demand, +ve = right, -ve = left, max 255
int swCnt;                // button switch counter
boolean swLastState;      // previous state of button switch, HIGH/LOW
boolean swState;          // state of read button switch pin
int swTimer;              // timer used to detemine button sequences
bool swWaitHi;            // set HIGH to wait for SW1 release
unsigned long TimeLeft;   // time remaining before nextMicros event in microseconds
bool WiFi_ON = false;     // WiFi/data has not yet been received
int Z_Button;             // button pressed state; 0 = UP, 1 = Down
int Z_orientation = 0;    // orientation flag based on z accelerometer, default 0 on its side

void setup() {
  // define pinmodes and initialise variables
  pinMode(AT_Cmd, OUTPUT); digitalWrite(AT_Cmd,HIGH); // default state for transceiver CMD pin
  pinMode(PWMA, OUTPUT); analogWrite(PWMA, 0); // initialise PWM on Motor A left
  pinMode(PWMB, OUTPUT); analogWrite(PWMB, 0); // initialise PWM on Motor B right
  pinMode(PWMC, OUTPUT); analogWrite(PWMC, 0); // initialise PWM on Motor C rear
  pinMode(DIRA, OUTPUT); digitalWrite(DIRA, LOW); // set direction for Motor A left
  pinMode(DIRB, OUTPUT); digitalWrite(DIRB, LOW); // set direction for Motor B right
  pinMode(DIRC, OUTPUT); digitalWrite(DIRC, LOW); // set direction for Motor A left
  pinMode(LEDBckRed, OUTPUT); digitalWrite(LEDBckRed, LOW);
  pinMode(LEDBckGrn, OUTPUT); digitalWrite(LEDBckGrn, LOW);
  pinMode(LEDLftRed, OUTPUT); digitalWrite(LEDLftRed, LOW);
  pinMode(LEDLftGrn, OUTPUT); digitalWrite(LEDLftGrn, LOW);
  pinMode(LEDRhtRed, OUTPUT); digitalWrite(LEDRhtRed, LOW);
  pinMode(LEDRhtGrn, OUTPUT); digitalWrite(LEDRhtGrn, LOW);
  setDefaults();
  Serial.begin(115200); // use this default baud rate for Nunchuk wireless control
  Wire.begin(); //Start the I2C bus as master
  POST(); // perform calibration sequence
  nextMicros = micros() + interval; // seed 250Hz loop timer
}

void loop() {
  //###############################################################################
  //
  // Main loop code
  //
  //###############################################################################
  if ((micros() - nextMicros) >= interval) {
    // this part runs at 4ms (250Hz)
    nextMicros = micros();

    // read the MPU 6050 registers every cycle
    // as gyro angle is calculated from gyro rate we must read them at preciose intervals
    readPitchYaw(); // read pitch and yaw gyros and get time delays for precision
    readAccelAll(); // read accelerometers as correction factors and topple


    //#############################################################################
    // check robot orientation using Z-axis accelerometer
    //#############################################################################
    // this data is rate limited to filter out noise
    // the trigger points are determined by 8192 * sin(angle°)
    acc_Z_Trck = acc_Z_Trck + ((accelerometer_data_rawZ_i - acc_Z_Trck)/16);
    if (acc_Z_Trck < -4096) {
      Z_orientation = 1; // right way up by > 30°
    } else if (acc_Z_Trck > 4096) {
      Z_orientation = -1; // upside down > 30°
    } else {Z_orientation = 0;} // on its side
    
    
    
    //#############################################################################
    // check wireless receiver buffer for control byte
    //#############################################################################
    // control bytes are only recognised during self-balancing mode, otherwise they
    // are cleared to zero
    if (!startEn) {
       received_byte = 0x00;                    // self-balancing is not yet enabled
       received_cmd = false;
    } else {  
      if(received_cmd){                         // serial data is available
          received_cmd = false;                 // reset the received flag
          receive_counter = 0;                  // Reset the receive_counter variable
          safePntCnt = 30;                      // block data reporting
      }
      if(receive_counter <= 25) {
        receive_counter ++;  // The received byte will be valid for 25 program loops (100 milliseconds)
      } else {received_byte = 0x00; PID_En_Cnt = 0;}  // After 100 milliseconds the received byte is deleted
    }


    //#############################################################################
    // branch to MainMode routines
    //#############################################################################
    switch(MainMode) {
      case 0: MainMode_0(); break; // default after reset, slow LED spin
      case 1: MainMode_1(); break; // user initiates self-balancing
      case 2: MainMode_2(); break; // phantom MainMode 1 with only PID P-gain
      case 3: MainMode_3(); break; // in active self-balancing mode
      case 4: MainMode_4(); break; // upside down drive demo
    }


    //#############################################################################
    // Angle calculations
    //#############################################################################
    // calculate angle for X axis using accelerometer value

    accel_delta = abs(accelerometer_data_rawX_OA - accelerometer_data_rawX_OALast);
    // test for rapid overshoot due to vibration around zero angle point
    if (accelerometer_data_rawX_OA < accelerometer_data_rawX_OALast) {
      if (accel_delta > acc_ramp_rate) {accelerometer_data_rawX_OA += acc_ramp_rate;}
      else {accelerometer_data_rawX_OA += accel_delta;}
    }
    else if (accelerometer_data_rawX_OA > accelerometer_data_rawX_OALast) {
      if (accel_delta > acc_ramp_rate) {accelerometer_data_rawX_OA -= acc_ramp_rate;}
      else {accelerometer_data_rawX_OA -= accel_delta;}
    } accelerometer_data_X = accelerometer_data_rawX_OA;
    
    if (accelerometer_data_X > 8192.0) {accelerometer_data_X = 8192.0;}
    else if (accelerometer_data_X < -8192.0) {accelerometer_data_X = -8192.0;}
    angle_accX = asin((float)accelerometer_data_X/8192.0)* 57.296;           //Calculate the current angle according to the accelerometer

    // repeat calculation for Y axis
    
    accel_delta = abs(accelerometer_data_rawY_OA - accelerometer_data_rawY_OALast);
    // test for rapid overshoot due to vibration around zero angle point
    if (accelerometer_data_rawY_OA < accelerometer_data_rawY_OALast) {
      if (accel_delta > acc_ramp_rate) {accelerometer_data_rawY_OA += acc_ramp_rate;}
      else {accelerometer_data_rawY_OA += accel_delta;}
    }
    else if (accelerometer_data_rawY_OA > accelerometer_data_rawY_OALast) {
      if (accel_delta > acc_ramp_rate) {accelerometer_data_rawY_OA -= acc_ramp_rate;}
      else {accelerometer_data_rawY_OA -= accel_delta;}
    } accelerometer_data_Y = accelerometer_data_rawY_OA;
    
    if (accelerometer_data_Y > 8192.0) {accelerometer_data_Y = 8192.0;}
    else if (accelerometer_data_Y < -8192.0) {accelerometer_data_Y = -8192.0;}
    angle_accY = -asin((float)accelerometer_data_Y/8192.0)* 57.296;           //Calculate the current angle according to the accelerometer

    // determine gyro angles based on gyro rate o/p and sampling time
    // gyros are set at +/-250 deg/sec for 16-bit ADC = 32,767 FSD
    // to convert gyro value to deg/sec we divide reading by 131.068 (ie. 32,767/250)
    // we have recorded time period in GyTD in microseconds, between readings
    // so angle is = GyAng * (GyTD/1000000)/131.068
    // in this case GyTD is nominally 4000 microseconds (ie. 4ms loop)
//    Serial.println(GyTD);
    float zDiv = (float)GyTD/131068000; // nominally 0.0000305 at 4ms
    gyro_pitch_data_raw -= gyro_pitch_calibration_value; // Add the gyro calibration value
    angle_gyroX += (float)gyro_pitch_data_raw * 0.0000305;      // Calculate the angle travelled during this 10ms loop and add this to the angle_gyro variable
    if (angle_gyroX > 90.0) {angle_gyroX = 90.0;} // limit gyro swing
    else if (angle_gyroX < -90.0) {angle_gyroX = -90.0;} // limit gyro swing
    gyro_yaw_data_raw -= gyro_yaw_calibration_value;     // Add the gyro calibration value
    angle_gyroY += (float)gyro_yaw_data_raw * 0.0000305;        // Calculate the angle travelled during this 10ms loop and add this to the angle_gyro variable
    if (angle_gyroY > 90.0) {angle_gyroY = 90.0;} // limit gyro swing
    else if (angle_gyroY < -90.0) {angle_gyroY = -90.0;} // limit gyro swing

    //#############################################################################
    // MPU-6050 gyro offset compensation
    //#############################################################################
    // As gyro angle is calculated from the rate/time an offset error can easily creep
    // in affecting the overall accuracy of the gyro. Particularly if the robot shakes
    // violently or falls over. So we use the noisier accelerometer angle to provide
    // a small ammount of offset adjustment as an integrator.
    if (GyroOnce) {
      // whilst balancing we use the accelerometer angles as small correction factors
      // this must be small, otherwise the angle will be too greatly affected by movement
      angle_offset = -(angle_gyroX - angle_accX)* 0.001; // was 0.02
      angle_gyroX += angle_offset; // Correct the drift of the gyro angle with the accelerometer angle
      angle_offset = -(angle_gyroY - angle_accY)* 0.001;
      angle_gyroY += angle_offset; // Correct the drift of the gyro angle with the accelerometer angle
    } else {
      // at the start of balance use the accelerometer values as the initial angle
      angle_gyroX = angle_accX; // Initialise the gyro angle using the accelerometer
      angle_gyroY = angle_accY; // Initialise the gyro angle using the accelerometer
      GyroOnce = true;
    }
 

    //#############################################################################
    // Gyro rate limit
    //#############################################################################
    // remove gyro noise spikes by applying a rate limiting filter
    angle_gyroFX = angle_gyroX;
    angle_gyroFY = angle_gyroY;


    //#############################################################################
    // PID controller calculations
    //#############################################################################
    // The balancing robot is angle driven. First the difference between the desired angel (setpoint) and actual angle (process value)
    // is calculated. The self_balance_pid_setpoint variable is automatically changed to make sure that the robot stays balanced all the time.
    // The (pid_setpoint - pid_output * 0.015) part functions as a speed limiting function.
    if (X_Axis_En) {
      pid_error_lastX = pid_error_newX;  //Store the true error for the next loop
      pid_error_newX = angle_gyroFX - self_balance_pid_setpointX - pid_setpointX;
      pid_error_tempX = pid_error_newX;
    } else {pid_error_newX = 0.0; pid_error_tempX = 0.0; angle_gyroFX = 0.0;} // error and gyro are set to zero if axis is off
    if (Y_Axis_En) {
      pid_error_lastY = pid_error_newY;  //Store the true error for the next loop
      pid_error_newY = angle_gyroFY - self_balance_pid_setpointY - pid_setpointY;
      pid_error_tempY = pid_error_newY;
    } else {pid_error_newY = 0.0; pid_error_tempY = 0.0; angle_gyroFY = 0.0;} // error and gyro are set to zero if axis is off


//    ///////////////////////////////////////////////////////////////////////////////////
//    // PID speed govenor
//    ///////////////////////////////////////////////////////////////////////////////////
//    // apply a speed governor to the X & Y error signals
//    // the rotational speeds are in degrees/sec
//    pid_error_speedX = abs(pid_error_tempX - pid_error_lastX)/0.004;
//    if (pid_error_speedX >= pid_s_thresh) {
//      if (pid_error_tempX >= 0.0) {pid_error_tempX -= pid_error_speedX * pid_s_gain;}
//      else {pid_error_tempX += pid_error_speedX * pid_s_gain;}
//    }
//    pid_error_speedY = abs(pid_error_tempY - pid_error_lastY)/0.004;
//    if (pid_error_speedY >= pid_s_thresh) {
//      if (pid_error_tempX >= 0.0) {pid_error_tempY -= pid_error_speedY * pid_s_gain;}
//      else {pid_error_tempY += pid_error_speedY * pid_s_gain;}
//    }
    

    //#############################################################################
    // PID calculations
    //#############################################################################
    //Calculate the PID output values; always include the P element
    pid_p_ValX = pid_p_gain * pid_error_tempX;    // p - proportional element
    pid_outputX = pid_p_ValX;                     // always include p element
    pid_p_ValY = pid_p_gain * pid_error_tempY;    // p - proportional element
    pid_outputY = pid_p_ValY;                     // always include p element

    if (PID_i_En && startEn) {
      // optional i - integrator element
      pid_i_memX += pid_i_gain * pid_error_tempX;     //Calculate the I-controller value and add it to the pid_i_mem variable
      if (pid_i_memX > 255.0) {pid_i_memX = 255.0;}   // limit integrator output
      else if (pid_i_memX < -255.0) {pid_i_memX = -255.0;} // limit integrator output
      pid_outputX += pid_i_memX;                      // add in PID i gain value
      pid_i_memY += pid_i_gain * pid_error_tempY;     //Calculate the I-controller value and add it to the pid_i_mem variable
      if (pid_i_memY > 255.0) {pid_i_memY = 255.0;}   // limit integrator output
      else if (pid_i_memY < -255.0) {pid_i_memY = -255.0;} // limit integrator output
      pid_outputY += pid_i_memY;                      // add in PID i gain value
    } else {pid_i_memX = 0.0; pid_i_memY = 0.0;}
    
    if (PID_d_En && startEn) {
      // optional d - deterministic element
      pid_d_ValX = pid_d_gain * (pid_error_tempX - pid_error_lastX);
      pid_outputX += pid_d_ValX;            // add in PID d gain value
      pid_d_ValY = pid_d_gain * (pid_error_tempY - pid_error_lastY);
      pid_outputY += pid_d_ValY;            // add in PID d gain value
    } else {pid_d_ValX = 0.0; pid_d_ValY = 0.0;} // clear value for graphing purposes

   
    //#############################################################################
    // Auto-setpoint adjustment
    //#############################################################################
    // The self balancing point is adjusted when there is no forward or backwards movement from the transmitter.
    // This way the robot will always find it's balancing point
    // The trend function prevents the setpoint from being wound up via pushing or slopes
    // To prevent constant swinging the increment is slowly reduced by 95% on each swing
    if((pid_setpointX == 0.0) && (PID_TuneX == 0)){
      // If the PID setpoint is zero degrees there is no external controller demand
      if(pid_outputX < -pid_sb_dbX){
        pid_output_trendX--;
        if (pid_output_trendX > 0) {
          // swing just changed direction
          pid_output_trendX = 0;
        }
        if (pid_output_trendX < -pid_output_trend_limit) {pid_output_trendX = -pid_output_trend_limit;}
        else {self_balance_pid_setpointX += self_balance_pid_incX;}  // Increase the self_balance_pid_setpoint if the robot is still moving forewards
      }
      else if(pid_outputX > pid_sb_dbX){
        pid_output_trendX++;
        if (pid_output_trendX < 0) {
          // swing just changed direction
          pid_output_trendX = 0;
        }
        if (pid_output_trendX > pid_output_trend_limit) {pid_output_trendX = pid_output_trend_limit;}
        else {self_balance_pid_setpointX -= self_balance_pid_incX;}   //Decrease the self_balance_pid_setpoint if the robot is still moving backwards
      } else {pid_output_trendX = 0;}
    }
    if((pid_setpointY == 0.0) && (PID_TuneY == 0)){
      // If the setpoint is zero degrees there is no external controller demand
      if(pid_outputY < -pid_sb_dbY){
        pid_output_trendY--;
        if (pid_output_trendY > 0) {
          // swing just changed direction
          pid_output_trendY = 0;
        }
        if (pid_output_trendY < -pid_output_trend_limit) {pid_output_trendY = -pid_output_trend_limit;}
        else {self_balance_pid_setpointY += self_balance_pid_incY;}  // Increase the self_balance_pid_setpoint if the robot is still moving forewards
      }
      else if(pid_outputY > pid_sb_dbY){
        pid_output_trendY++;
        if (pid_output_trendY < 0) {
          // swing just changed direction
          pid_output_trendY = 0;
        }
        if (pid_output_trendY > pid_output_trend_limit) {pid_output_trendY = pid_output_trend_limit;}
        else {self_balance_pid_setpointY -= self_balance_pid_incY;}   //Decrease the self_balance_pid_setpoint if the robot is still moving backwards
      } else {pid_output_trendY = 0;}
    }

    //#############################################################################
    // PID vector magnitude
    //#############################################################################
    // this is the vector resulting from the two PID outputs
    pid_output = sqrt(sq(pid_outputX) + sq(pid_outputY));


    //#############################################################################
    // PID vector angle
    //#############################################################################
    // calculate the drive direction angle to correct this error vector
    // adopting the convention of forward facing is 0°
    // Leaning forward so drive angle is 180° ie. backwards
    // Leaning right so drive angle is 0° to -180° ie. to the left
    // Leaning left so drive angle is 0° to +180° ie. to the right
    if (abs(pid_outputX) < 0.0001) {pid_outputX = 0.0001;} // avoid divide by zero
    if (pid_outputX >= 0.0) {
      // robot is leaning forward
      if (pid_outputY >= 0.0) {
        // robot is leaning forward left, angle is 0° to -90°
        pid_error_angle = 180.0 - (57.2958 * atan(pid_outputY/pid_outputX)); // angle in degrees
      } else {
        // robot is leaning forward right, angle is 0° to +90°
        pid_error_angle = -180.0 - (57.2958 * atan(pid_outputY/pid_outputX)); // angle in degrees
      }
    } else {
      // robot is leaning backward
      if (pid_outputY >= 0.0) {
        // robot is leaning backward left, angle is -90° to -180°
        pid_error_angle = -57.2958 * atan(pid_outputY/pid_outputX); // angle in degrees
      } else {
        // robot is leaning backward right, angle is +90° to +180°
        pid_error_angle = -57.2958 * atan(pid_outputY/pid_outputX); // angle in degrees
      }
    }


    //#############################################################################
    // limit pid_outputs to pid_out_max
    //#############################################################################
    if (pid_output > pid_out_max) {pid_output = pid_out_max;}     // Limit the PI-controller to the maximum controller output
    else if (pid_output < -pid_out_max) {pid_output = -pid_out_max;}

  
    //#############################################################################
    // limit output power during initial switch into self-balancing mode
    //#############################################################################
    if (PID_Output_Power < 100.0) {
      // PID_Output_Power can range from 0 - 100%
      pid_output = pid_output * PID_Output_Power/100.0;
      PID_Output_Power += 1.0;  // slowly increase power output
    }

    
    //#############################################################################
    // test for robot falling over or being over-tilted at any time  
    //#############################################################################
    if (acc_Z_Trck > -4096 || angle_gyroX > Topple || angle_gyroX < -Topple || angle_gyroY > Topple || angle_gyroY < -Topple) {
      // If the robot tips over or the start variable is zero
      // balance angle exceeded so drop out of any active mode
      if ((MainMode != 0) && (MainMode != 4)) {
        PWM_OFF(); setDefaults();
        setMainMode(0); // return to default mode
        Serial.println(F("Topple!"));
      }
    }

    //#############################################################################
    // Nunchuk Control calculations
    //#############################################################################
    // B00000001  X pressed to the left
    // B00000010  X pressed to the right
    // B00000100  Y pressed to the front
    // B00001000  Y pressed to the back
    // B00010000  'Z' button pressed
    // B00100000  'C' button pressed
    // we inc/dec steering and drive factors so as not to introduce sudden movements
    if(received_byte & B00000001){
      // Joystick pressed left -X
      if (Steer > -45.0) {Steer -= 0.5;}
    } else if (received_byte & B00000010){
      // Joystick pressed right +X
      if (Steer < 45.0) {Steer += 0.5;}
    } else {
      // return steering to zero
      if (Steer > 0.5) {Steer -= 1.0;}
      else if (Steer < -0.5) {Steer += 1.0;}
      else {Steer = 0.0;}
    }
    
    if(received_byte & B00000100){
      // Joystick pressed forward +Y
      if (pid_setpointX < 1.2) {pid_setpointX += 0.01;}
      else if (pid_setpointX < 2.0) {pid_setpointX += 0.001;}
    } else if (received_byte & B00001000){
      // Joystick pressed backward -Y
      if (pid_setpointX > -1.2) {pid_setpointX -= 0.01;}
      if (pid_setpointX > -2.0) {pid_setpointX -= 0.001;}
    } else {
      if (pid_setpointX > 0.01) {pid_setpointX -= 0.01;}
      else if (pid_setpointX < -0.01) {pid_setpointX += 0.01;}
      else {pid_setpointX = 0.0;} // ensure it returns to zero
    }
    

    //#############################################################################
    // Motor PWM calculations
    //#############################################################################
    if (startEn) {
      // motor power shaping is done in a later function
      // the pid_ouput is set -ve as motor connections have been reversed on this
      // robot
      driveMotors(pid_output,pid_error_angle,Steer);
    }
    

    //#############################################################################
    // Motor drive with dither calculations
    //#############################################################################
    // dither is applied at 30Hz to all motor drive signals when enabled
    // these tasks run a 4ms intervals and we apply dither over 8 cycles
    // giving a dither frequency of 31.25Hz (1000/32)
    if (startEn || DriveEn) {
      // update H-bridge drivers with PWM_Dn signals every cycle
      DithVal = 40;
//      Serial.println(JX);
      if (DithCnt < 4) {
        // switch to positive dither phase
        AnyD = DithVal; if (PWM_A < DithVal) {AnyD = PWM_A;}
        else {AnyD = DithVal - (PWM_A - DithVal); if (AnyD < 1) {AnyD = 0;}}
//        Serial.println(AnyD);
        PWM_DA = PWM_A + AnyD;
        AnyD = DithVal; if (PWM_B < DithVal) {AnyD = PWM_B;}
        else {AnyD = DithVal - (PWM_B - DithVal); if (AnyD < 1) {AnyD = 0;}}
        PWM_DB = PWM_B + AnyD;
        AnyD = DithVal; if (PWM_C < DithVal) {AnyD = PWM_C;}
        else {AnyD = DithVal - (PWM_C - DithVal); if (AnyD < 1) {AnyD = 0;}}
        PWM_DC = PWM_C + AnyD;
      } else {
        // switch to negative dither phase
        AnyD = DithVal; if (PWM_A < DithVal) {AnyD = PWM_A;}
        else {AnyD = DithVal - (PWM_A - DithVal); if (AnyD < 1) {AnyD = 0;}}
//        Serial.println(AnyD);
        PWM_DA = PWM_A - AnyD;
        AnyD = DithVal; if (PWM_B < DithVal) {AnyD = PWM_B;}
        else {AnyD = DithVal - (PWM_B - DithVal); if (AnyD < 1) {AnyD = 0;}}
        PWM_DB = PWM_B - AnyD;
        AnyD = DithVal; if (PWM_C < DithVal) {AnyD = PWM_C;}
        else {AnyD = DithVal - (PWM_C - DithVal); if (AnyD < 1) {AnyD = 0;}}
        PWM_DC = PWM_C - AnyD;
      }
      if (DIR_A) {analogWrite(PWMA,255-PWM_DA);
      } else {analogWrite(PWMA,PWM_DA);
      } digitalWrite(DIRA,DIR_A);
      if (DIR_B) {analogWrite(PWMB,255-PWM_DB);
      } else {analogWrite(PWMB,PWM_DB);
      } digitalWrite(DIRB,DIR_B);
      if (DIR_C) {analogWrite(PWMC,255-PWM_DC);
      } else {analogWrite(PWMC,PWM_DC);
      } digitalWrite(DIRC,DIR_C);
      DithCnt++; if (DithCnt > 7) {DithCnt = 0;}
    } else {DithCnt = 0;}
    
    
    //#############################################################################
    // read battery and button switch every 8ms
    //#############################################################################
    // switch SW1 is integrated with battery voltage reading
    if (BattPhase) {BattVol = analogRead(A0);}  // sample battery voltage
    else {
      readSwitch();             // read the state of the button switch
      if (swState == HIGH) {
        // only perform battery voltage check when switch is not pressed
        // check battery voltage averaged over 16 cycles
        BattAv = BattSum >> 4; BattSum = BattSum + BattVol - BattAv;
        if (BattAv <= BattMin) {
          // battery voltage has reached minimum level
          // a one second timeout counter avoids brownouts and glitches
          BattCnt++; if (BattCnt > 50) {BatteryFailed();}
        } else {BattCnt = 0;} // reset battery LOW timeout
      }
    } BattPhase = !BattPhase; // alternate phase


    ///////////////////////////////////////////////////////////////////////
    // drive LEDs depending on LEDMode
    ///////////////////////////////////////////////////////////////////////
    if (LED_Override > 0) {
      // LED override is set and timing out, LED state will not change
      LED_Override--;
    } else {
      switch (LEDMode) {
        case -1: break; // no LED task performed
        case 0: LEDs_OFF(); break; // LEDs are OFF
        case 1: // slow rotating sequence
          if (Z_orientation > 0) {
            // robot is upright so do this sequencing
            DelVal--;
            if (DelVal < 1) {
              DelVal = DelValRes;
              if (WiFi_ON) {
                LED_Task++; if (LED_Task > 5) {LED_Task = 0;}
              } else {
                LED_Task--; if (LED_Task < 0) {LED_Task = 5;}
              }
              Do_LED_Task(false);
            }
          } else {
            // robot is not upright so go red
            LEDLftRedOn = HIGH; LEDRhtRedOn = HIGH; LEDBckRedOn = HIGH;
            LEDLftGrnOn = LOW; LEDRhtGrnOn = LOW; LEDBckGrnOn = LOW;
            LED_Task = -2; // set task to turn OFF LEDs when returned to upright
          } break;
        case 2: // LEDs track motor PWM and DIR
          if (PWM_A != 0 ) {
            LEDA_CntLoad = 2000/PWM_A;
            if (LEDA_CntLoad > 50) {LEDA_CntLoad = 50;}
            if (PWM_A >= 250) {LEDA_Cnt = 3;} // LED will be permanently ON
            LEDA_Cnt--; // decrement LED flash counter
            if (LEDA_Cnt == 2) {
              if (DIR_A) {LEDLftGrnOn = HIGH;}
              else {LEDLftRedOn = HIGH;}
            }
          } else {LEDA_Cnt = 0; LEDA_CntLoad = 50;}
          if (LEDA_Cnt == 0) {
            LEDA_Cnt = LEDA_CntLoad; LEDLftGrnOn = LOW;  LEDLftRedOn = LOW;
          }
          if (PWM_B != 0 ) {
            LEDB_CntLoad = 2000/PWM_B;
            if (LEDB_CntLoad > 50) {LEDB_CntLoad = 50;}
            if (PWM_B >= 250) {LEDB_Cnt = 3;} // LED will be permanently ON
            LEDB_Cnt--; // decrement LED flash counter
            if (LEDB_Cnt == 2) {
              if (DIR_B) {LEDRhtGrnOn = HIGH;}
              else {LEDRhtRedOn = HIGH;}
            }
          } else {LEDB_Cnt = 0; LEDB_CntLoad = 50;}
          if (LEDB_Cnt == 0) {
            LEDB_Cnt = LEDB_CntLoad; LEDRhtGrnOn = LOW;  LEDRhtRedOn = LOW;
          }
          if (PWM_C != 0 ) {
            LEDC_CntLoad = 2000/PWM_C;
            if (LEDC_CntLoad > 50) {LEDC_CntLoad = 50;}
            if (PWM_C >= 250) {LEDC_Cnt = 3;} // LED will be permanently ON
            LEDC_Cnt--; // decrement LED flash counter
            if (LEDC_Cnt == 2) {
              if (DIR_C) {LEDBckGrnOn = HIGH;}
              else {LEDBckRedOn = HIGH;}
            }
          } else {LEDC_Cnt = 0; LEDC_CntLoad = 50;}
          if (LEDC_Cnt == 0) {
            LEDC_Cnt = LEDC_CntLoad; LEDBckGrnOn = LOW;  LEDBckRedOn = LOW;
          }
          break;
      }
    }

    

    // comment out the following line in the final version
    if (REPORT) {DebugReport(RPnn);} // if true run the debug report routine
  
    ///////////////////////////////////////////////////////////////////////
    // Set LED states and brightness timer
    ///////////////////////////////////////////////////////////////////////
    // multiplexed LEDs run as Red / Green phases and are turned ON by setting
    // flags in other parts of the code
    LEDPhase = !LEDPhase; // toggle phase on each cycle at (250Hz)
    if (LEDPhase) {
      // red phase
      digitalWrite(LEDLftRed,LEDLftRedOn); digitalWrite(LEDRhtRed,LEDRhtRedOn); digitalWrite(LEDBckRed,LEDBckRedOn);
      digitalWrite(LEDLftGrn,LOW); digitalWrite(LEDRhtGrn,LOW); digitalWrite(LEDBckGrn,LOW);
    } else {
      // green phase
      digitalWrite(LEDLftGrn,LEDLftGrnOn); digitalWrite(LEDRhtGrn,LEDRhtGrnOn); digitalWrite(LEDBckGrn,LEDBckGrnOn);
      digitalWrite(LEDLftRed,LOW); digitalWrite(LEDRhtRed,LOW); digitalWrite(LEDBckRed,LOW);
    }
    // we will assume that there is at least 5ms left in each cycle which can be
    // used for LED brightness control. LEDBrightness is between 0 - 10
    TimeLeft = nextMicros - micros();   // see what time is remaining
    LEDOnTime = micros() + ((LEDBrightness * (int)TimeLeft)/10); LEDs_ON = true;


    ///////////////////////////////////////////////////////////////////////
    // test for code over-run in each cycle
    ///////////////////////////////////////////////////////////////////////
    // code used during development.
    // uncomment the following lines if cycle over-run is suspected
//    if (micros() > nextMicros) {
//      // Oh dear, code over-run has occured!
//      Serial.println(F("!"));
//    } else {
//      // print what spare time there is before next 10ms cycle
//      Serial.println(nextMicros - micros());
//    }


    ///////////////////////////////////////////////////////////////////////
    // End of 4ms main loop
    ///////////////////////////////////////////////////////////////////////
  }

  
  ///////////////////////////////////////////////////////////////////////
  // the following code runs on every loop cycle, outside of the main tasks
  ///////////////////////////////////////////////////////////////////////
  readKey(); // respond to received serial data

  if (LEDs_ON) {
    // LED brightness timer-out is running
    if (micros() >= LEDOnTime) {
      LEDs_ON = false; LEDs_OFF();
    }
  }
}

// --------------------------------------------------------------------------------

void POST () {
  // run once at boot up
  Serial.println(F("\n\n\nBallBalanceBot vR1"));
  Serial.print(F("Released: ")); Serial.println(Released);
  Serial.println(F("Running POST..."));
  Serial.println(F("Initialising Gyros..."));
  Initialise_MPU_6050();
  
  if (!TEST) {
    // during POST determine sensor drift/offset values
    Calibrate_Gyros();
    
    // set initial accelerometer values prior to filtering
    readAccelAll();
  } else {
    // in TEST provide a read-out of the 3 accelerometer offsets
    Calibrate_Accelerometers();

    // then wait
    while (true) {}
  }
  
  // set initial accelerometer values prior to filtering
  readAccelAll();

  // switch OFF all LEDs
  ClrLEDFlags(); LEDs_OFF();

  
  ///////////////////////////////////////////////////////////////////////////////
  // Flash Transceiver for 115200 baud
  ///////////////////////////////////////////////////////////////////////////////
  // If new the transceiver should be programmed from it's default settings
  // To do this uncomment the following lines of code and run once to flash
  // the new settings into the transceiver, then comment out again.
//  WiFi_Reset();   // restore factory settings, if necessary
//  WiFi_Flash();   // flash the Open Smart transceiver. Only needed once.
//  WiFi_Report();  // report and confirm your settings

  Serial.flush(); // empty serial send buffer before switching baud rates
  Serial.begin(115200);
  Serial.println(F("Baud rate = 115200"));
  Serial.println(F("Retries = 5"));
  Serial.println(F("Power = -6dBm"));

  Serial.println(F("POST Complete!"));
  setMainMode(0); // start in waiting mode
  Serial.flush(); // empty serial send buffer before running main code
}

// --------------------------------------------------------------------------------

void setDefaults() {
  // load default values
  AccAbsAng = 0.0;                // acceleration vector
  accelerometer_data_rawX_OA = 0; // instantaneous record offset adjusted
  accelerometer_data_rawY_OA = 0; // instantaneous record offset adjusted
  ACM = 50;                       // set speed compensation to 50%, determined during calibration
  AccVect = 0.0;                  // acceleration vector
  AccVectTrk = 0.0;               // acceleration vector tracking filter
  acc_Z_Trck = -acc_1g;           // Z accelerator tracking filter value           
  BattAv = 915;                   // average battery voltage
  BattCnt = 0;                    // battery failed 1s time-out counter
  BattPhase = true;               // alternating phase controls battery/switch reading
  BattSum = BattMin << 4;         //  pre-set cumulative battery voltage
  C_Button = 0;                   // button pressed state; 0 = UP, 1 = Down
  DIR_A = false;                  // value writtent to H-bridge pin
  DIR_B = false;                  // value writtent to H-bridge pin
  DIR_C = false;                  // value writtent to H-bridge pin
  DriveEn = false;                // if == true then output PWM to motors
  GoActive = 0;                   // counter used at initial balance point
  GyroOnce = false;               // sets the initial angles for the gyros based on the accelerometers
  Joy_Y = 0;                      // > 0 when joystick Y is pressed
  LEDBckRedOn = false;            // true if LED is ON, default false
  LEDBckGrnOn = false;            // true if LED is ON, default false
  LEDCnt = 0;                     // counter used with LEDs
  LEDLftRedOn = false;            // true if LED is ON, default false
  LEDLftGrnOn = false;            // true if LED is ON, default false
  LEDMode = 1;                    // LED mode pointer, default = 1, slow rotation
  LED_Override = 0;               // if >0 counts down locking the current LED settings
  LEDPhase = false;               // alternating LED phase cycle
  LEDRhtGrnOn = false;            // true if LED is ON, default false
  LEDRhtRedOn = false;            // true if LED is ON, default false
  LED_Task = 0;                   // LED task pointer
  MainMode = 0;                   // controls which tasks are performed, default = 0
  ModeCnt = 0;                    // counter used in MainModes
  ModeTask = 0;                   // task poiinter used in MainModes
  MotorCal = true;                // controls motor PWM calibration, default = true
  MotorMaxPWM = 255;              // maximum PWM in either rotation, default = 255
  MotorMinPWMip = 10;             // minimum PWM demand point, default = 10
  MotorMinPWMop = 30;             // minimum PWM output point, default = 50
  MotorPWM50 = 47;                // remapped PWM = 50 in anti-clockwise rotation
  MotorPWM100 = 64;               // remapped PWM = 100 in anti-clockwise rotation
  MotorPWM150 = 98;               // remapped PWM = 150 in anti-clockwise rotation
  MotorPWM200 = 159;              // remapped PWM = 200 in anti-clockwise rotation
  Mot_Task = 0;                   // motor task pointer
  OffBalance = 0;                 // flag capturing initial off-balance condition

  pid_p_gain = pid_p_gain_ref;    // Gain setting for the P-controller
  pid_i_gain = pid_i_gain_ref;    // Gain setting for the I-controller
  pid_d_gain = pid_d_gain_ref;    // Gain setting for the D-controller
  pid_s_gain = pid_s_gain_ref;    // Gain setting for PID speed governor
  
  pid_output_trendX = 0;          // tracker used in self-balancing
  pid_output_trendY = 0;          // tracker used in self-balancing
  pid_output = 0.0;               // current PDI output
  pid_setpointX = 0.0;            // X setpoint angle when moving
  pid_setpointY = 0.0;            // Y setpoint angle when moving
  PID_TuneX = 0;                  // > 0 when in tuning mode
  PID_TuneY = 0;                  // > 0 when in tuning mode
  PWM_A = 0;                      // PWM value written to H-bridge driver
  PWM_B = 0;                      // PWM value written to H-bridge driver
  PWM_C = 0;                      // PWM value written to H-bridge driver
  readTask = 0;                   // task pointer used in decoding received control data
  received_cmd = false;           // set true if valid control has ben recevied
  RP_ON = false;                  // if true then report values in DEBUG mode
  safeMode = 0;                   // start mode of safety state machine
  safeModeLast = -1;              // previous safe mode
  safePntCnt = 0;                 // print timer counter
  self_balance_pid_incX = 0.001;  // values used to inc/dec balance setpoint (0.0015)
  self_balance_pid_incY = 0.001;  // values used to inc/dec balance setpoint (0.0015)
  self_balance_pid_setpointX = 0.0;  // offset used to set natural balance point
  self_balance_pid_setpointY = 0.0;  // offset used to set natural balance point
  startEn = false;                // true to perform motor calcs and enable output
  Steer = 0.0;                    // steering demand, +ve = right, -ve = left, max 255
  swCnt = 0;                      // button switch counter
  swLastState = HIGH;             // previous state of button switch, HIGH/LOW
  swState = HIGH;                 // stat of read button switch pin
  swTimer = 0;                    // timer used to detemine button sequences
  swWaitHi = false;               // set HIGH to wait for SW1 release
  Z_Button = 0;                   // button pressed state; 0 = UP, 1 = Down

  motorCal();                     // invoke motor PWM mapping as default
}

// --------------------------------------------------------------------------------
