/*
  Quick reference guide to Bodmers TFT_eSPI library functions.

  For more detailed information: https://doc-tft-espi.readthedocs.io/tft_espi/methods/drawbitmap/
  
  tft.init()                    - start the tft display
  tft.begin()                   - start the tft display, included for backwards compatibility only
  tft.setRotation(uint8_t m)    - set rotation, 0 = none, 1 = 90 CW, 2 = 180 CW, 3 = 270 CW

  tft.drawArc(int32_t x, int32_t y, int32_t r, int32_t ir, uint32_t startAngle, uint32_t endAngle, uint32_t fg_color, uint32_t bg_color, bool smoothArc = true)
  tft.drawSmoothArc(int32_t x, int32_t y, int32_t r, int32_t ir, uint32_t startAngle, uint32_t endAngle, uint32_t fg_color, uint32_t bg_color, bool roundEnds = false)
  tft.drawBitmap(int16_t x, int16_t y, const uint8_t *bitmap, int16_t w, int16_t h, uint16_t color)
  tft.drawBitmap(int16_t x, int16_t y, const uint8_t *bitmap, int16_t w, int16_t h, uint16_t fgcolor, uint16_t bgcolor)
  tft.drawXBitmap(int16_t x, int16_t y, const uint8_t *bitmap, int16_t w, int16_t h, uint16_t color)
  tft.drawXBitmap(int16_t x, int16_t y, const uint8_t *bitmap, int16_t w, int16_t h, uint16_t color, uint16_t bgcolor)
  tft.drawCentreString(const char *string, int32_t x, int32_t y, uint8_t font)
  tft.drawCentreString(const String& string, int32_t x, int32_t y, uint8_t font)
  tft.drawChar(int32_t x, int32_t y, uint16_t c, uint32_t color, uint32_t bg, uint8_t size)
  tft.drawChar(uint16_t uniCode, int32_t x, int32_t y)
  tft.drawChar(uint16_t uniCode, int32_t x, int32_t y, uint8_t font)
  tft.drawCircle(int32_t x, int32_t y, int32_t r, uint32_t color)
  tft.drawSmoothCircle(int32_t x, int32_t y, int32_t r, uint32_t fg_color, uint32_t bg_color)
  tft.drawCircleHelper( int32_t x, int32_t y, int32_t r, uint8_t cornername, uint32_t color)
  tft.drawEllipse(int16_t x0, int16_t y0, int32_t rx, int32_t ry, uint16_t color)
  tft.drawFastHLine(int32_t x,int32_t y,int32_t w,uint32_t color)
  tft.drawFastVLine(int32_t x,int32_t y,int32_t h,uint32_t color)
  tft.drawFloat(float floatNumber, uint8_t decimal, int32_t x, int32_t y, uint8_t font)
  tft.drawFloat(float floatNumber, uint8_t decimal, int32_t x, int32_t y)
  tft.drawLine(int32_t x0,int32_t y0,int32_t x1,int32_t y1,uint32_t color)
  tft.drawNumber(long intNumber, int32_t x, int32_t y, uint8_t font)
  tft.drawNumber(long intNumber, int32_t x, int32_t y)
  tft.drawPixel(int32_t x,int32_t y,uint32_t colour)
  tft.drawPixel(int32_t x,int32_t y,uint32_t colour,uint8_t alpha,uint32_t bgt_coolour)
  tft.drawRect(int32_t x,int32_t y,int32_t w,int32_t h,uint32_t color)
  tft.drawRightString(const String& string, int32_t dX, int32_t poY, uint8_t font)
  tft.drawRightString(const char *string, int32_t dX, int32_t poY, uint8_t font)
  tft.drawRoundRect(int32_t x, int32_t y,int32_t w,int32_t h,int32_t r,uint32_t color)
  tft.drawSmoothRoundRect(int32_t x,int32_t y,int32_t r,int32_t ir,int32_t w,int32_t h,uint32_t fg_color,uint32_t bg_color = 0x00FFFFFF,uint8_t quadrants = 0xF)
  tft.drawSpot(float ax, float ay, float r, uint32_t fg_color, uint32_t bg_color = 0x00FFFFFF)
  tft.drawString(const char *string, int32_t x, int32_t y, uint8_t font)
  tft.drawString(const char *string, int32_t x, int32_t y)
  tft.drawString(const String& string, int32_t x, int32_t y, uint8_t font)
  tft.drawString(const String& string, int32_t x, int32_t y)
  tft.drawTriangle(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t x2, int32_t y2, uint32_t color)
  tft.drawWedgeLine(float ax, float ay, float bx, float by, float ar, float br, uint32_t fg_color, uint32_t bg_color = 0x00FFFFFF)
  tft.drawWideLine(float ax, float ay, float bx, float by, float wd, uint32_t fg_color, uint32_t bg_color = 0x00FFFFFF)

  tft.fillCircle(int32_t x, int32_t y, int32_t r, uint32_t color)
  tft.fillCircle(int32_t x, int32_t y, int32_t r, uint32_t color, uint32_t bg_color = 0x00FFFFFF)
  tft.fillCircleHelper(int32_t x, int32_t y, int32_t r, uint8_t cornername, int32_t delta, uint32_t color)
  tft.fillEllipse(int16_t x0, int16_t y0, int32_t rx, int32_t ry, uint16_t color)
  tft.fillRect(int32_t x, int32_t y, int32_t w, int32_t h, uint32_t color)
  tft.fillRectHGradient(int16_t x, int16_t y, int16_t w, int16_t h, uint32_t color1, uint32_t color2)
  tft.fillRectVGradient(int16_t x, int16_t y, int16_t w, int16_t h, uint32_t color1, uint32_t color2)
  tft.fillRoundRect(int32_t x, int32_t y, int32_t w, int32_t h, int32_t radius, uint32_t color)
  tft.fillScreen(uint32_t color)
  tft.fillSmoothCircle(int32_t x, int32_t y, int32_t r, uint32_t color, uint32_t bg_color)
  tft.fillSmoothRoundRect(int32_t x, int32_t y, int32_t w, int32_t h, int32_t radius, uint32_t color, uint32_t bg_color = 0x00FFFFFF)
  img.fillSprite(TFT_TRANSPARENT);
  tft.fillTriangle(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t x2, int32_t y2, uint32_t color)

  tft.getCursorX(void)      - returns int16_t
  tft.getCursorY(void)      - returns int16_t
  tft.getTextDatum(void)    - returns uint8_t
  tft.getTextPadding(void)  - returns uint16_t

  tft.height(void)          - returns screen height as int16_t
  tft.width(void)           - returns screen width as int16_t

  tft.print(val)
  tft.print(val, format)
  tft.println(val)
  tft.println(val, format)

  img.createSprite(int16_t W, int16_t H)
  img.deleteSprite()
  img.fillSprite(uint16_t color)
  img.pushSprite(int16_t x, int16_t y)
  img.pushSprite(int16_t x, int16_t y,uint32_t colour transparent)

  tft.setCursor(int16_t x, int16_t y)
  tft.setCursor(int16_t x, int16_t y, uint8_t font)
  tft.setTextColor(uint16_t color)
  tft.setTextColor(uint16_t fgcolor, uint16_t bgcolor, bool bgfill = false)
  tft.setTextDatum(uint8_t d)
  tft.setTextFont(uint8_t f)
  tft.setTextPadding(uint16_t x_width)
  tft.setTextSize(uint8_t s)
  tft.setTextWrap(bool wrapX, bool wrapY)
  tft.setViewport(int32_t x, int32_t y, int32_t w, int32_t h, bool vpDatum)
  tft.setRotation(uint8_t m)    - set rotation, 0 = none, 1 = 90 CW, 2 = 180 CW, 3 = 270 CW

  tft.textWidth(const char *string, uint8_t font)   - returns pixel width of string in specified font
  tft.textWidth(const char *string)                 - returns pixel width of string in current font
  tft.textWidth(const String& string, uint8_t font) - as above for string types
  tft.textWidth(const String& string)               - 
  tft.fontHeight(uint8_t font)                      - returns pixel height of specified font
  tft.fontHeight(void)                              - returns pixel height of current font

  Basic colours already defined:
  TFT_BLACK       0x0000
  TFT_BLUE        0x001F
  TFT_CYAN        0x07FF
  TFT_DARKCYAN    0x03EF
  TFT_DARKGREEN   0x03E0
  TFT_DARKGREY    0x7BEF
  TFT_GREEN       0x07E0
  TFT_GREENYELLOW 0xB7E0
  TFT_LIGHTGREY   0xC618
  TFT_MAGENTA     0xF81F
  TFT_MAROON      0x7800
  TFT_NAVY        0x000F
  TFT_OLIVE       0x7BE0
  TFT_ORANGE      0xFDA0
  TFT_PINK        0xFC9F
  TFT_PURPLE      0x780F
  TFT_RED         0xF800
  TFT_TRANSPARENT         // used for text and sprites as background colour
  TFT_WHITE       0xFFFF
  TFT_YELLOW      0xFFE0

  Included fonts:
  GLCD    // Font 1. Original Adafruit 8 pixel font needs ~1820 bytes in FLASH
  FONT2   // Font 2. Small 16 pixel high font, needs ~3534 bytes in FLASH, 96 characters
  FONT4   // Font 4. Medium 26 pixel high font, needs ~5848 bytes in FLASH, 96 characters
  FONT6   // Font 6. Large 48 pixel font, needs ~2666 bytes in FLASH, only characters 1234567890:-.apm
  FONT7   // Font 7. 7 segment 48 pixel font, needs ~2438 bytes in FLASH, only characters 1234567890:-.
  FONT8   // Font 8. Large 75 pixel font needs ~3256 bytes in FLASH, only characters 1234567890:-.
  GFXFF   // FreeFonts. Include access to the 48 Adafruit_GFX free fonts FF1 to FF48 and custom fonts
*/

// --------------------------------------------------------------------------------

void GFX_ClrRun() {
  // Called at the end of an eye GFX_Run sequence to reset values for the next run
  GFX_Run = 0; GFX_Task = 0; 
}

// --------------------------------------------------------------------------------

void GFX_EyeTasks() {
  // GFX task sequences are run when GFX_Run > 0
  // Each sequence needs to have parameters preset before running it
  // Once completed GFX_Run is set to zero, so that another can be defined and called
  if (GFX_Del > 0) {GFX_Del--; return;}           // GFX_Run task delay in milliseconds

  // Choose a task to run based on GFX_Run
  switch (GFX_Run) {
    case 0: break;                                // default null, end of tasks returns here
    case GFX_EyeMove: GFX_Task_EyeMove(); break;  // draw an eye at a new location
    case GFX_EyeXY: GFX_Task_EyeXY(); break;      // blank display, then draw an eye at a preset location
    case GFX_Pupil: GFX_Task_Pupil(); break;      // draw an eye pupil
    default: // safety net
      GFX_ClrRun(); break;
  }
}

// --------------------------------------------------------------------------------

void GFX_Reset() {
  // Called during boot and whenever the screen mode changes or orientation changes
  // First set orientation
  switch (Upright) {
    case -1: // Upside down
      GFX_Rot = 0; break;
    case  0: // On its side
           if (Upside ==  1) {GFX_Rot = 3;}
      else if (Upside == -1) {GFX_Rot = 1;}
      break;
    case  1: // Upright
      GFX_Rot = 2; break;
  }
  tft.setRotation(GFX_Rot);   // set screen rotation
  // Now clear the screen
  if (GFX_Mode > 1) {// Black text display
    MPU_En = false; tft.fillScreen(TFT_BLACK); MPU_En = true;
    BakCol = TFT_BLACK; TxtCol = TFT_WHITE;
    Eye_En = false; Txt_En = true; GFX_Run = 0;
  } else {// white eye/text display
    // The background will be filled as part of the GFX_EyeXY() function
    BakCol = TFT_WHITE; TxtCol = TFT_BLACK;
    if (GFX_Mode == 0) {
      // This is a full eye
      GFX_R0 = 70; GFX_R1 = GFX_R0/2; // eye and cornea radii
      GFX_X = 120; GFX_Y = 120;}
    else if (GFX_Mode == 1) {
      // This is a partial eye
      GFX_R0 = 60; GFX_R1 = GFX_R0/2; // eye and cornea radii
      GFX_X = 120; GFX_Y =  80;}
    Eye_En = true; if (GFX_Mode == 1) {Txt_En = true;} else {Txt_En = false;}
    GFX_Run = GFX_EyeXY;
  }
  // Now set eye variables
  GFX_Task = 0; GFX_Eng = 0; GFX_Wait = 0; GFX_Del = 10;
  // Now set text variables
  GFX_Txt = 0; Txt_Del = 10; for (int16_t zP = 0; zP < TxtDepth; zP++) {TxtFP[zP] = 0;}
  // Clear the reset flag
  GFX_RST = false;
}

// --------------------------------------------------------------------------------

void GFX_Task_EyeXY() {
  // A GFX_Run task which draws an eye centred at GFX_X,GFX_Y
  // There insufficient RAM to support full screen 240x240 sprites,
  // so sprite mode is not used for this function.
  switch (GFX_Task) {
    case 0: // draw sclera background
      // If MPU comms has failed then this will be in red
      // t0 = micros();
      if (I2C_MPU) {BakCol = TFT_WHITE;} else {BakCol = TFT_RED;}
      TxtCol = TFT_BLACK;
      MPU_En = false; tft.fillScreen(BakCol); MPU_En = true;
      break;
    case 1: // draw blue or red iris
      MPU_En = false;
      // set colour                                            rrrrrggggggbbbbb
      if (JoyEye == 0) {tft.fillCircle(GFX_X, GFX_Y, GFX_R0, 0b0010010010011111);}  // blue
                  else {tft.fillCircle(GFX_X, GFX_Y, GFX_R0, 0b1111100001000010);}  // red
      MPU_En = true;
      break;
    case 2: // draw black pupil
      MPU_En = false; tft.fillCircle(GFX_X, GFX_Y, GFX_R1, TFT_BLACK);  MPU_En = true;
      GFX_XL = GFX_X; GFX_YL = GFX_Y; // record the new position
      GFX_ClrRun();
      // Serial.println(micros() - t0);
      return;           // terminate the run
    default: // end of drawing tasks, so reset system
      GFX_ClrRun();
      return;           // avoid incrementing the task
  } GFX_Task++;
}

// --------------------------------------------------------------------------------

void GFX_Task_EyeMove() {
  // A GFX_Run task which redraws an eye centred at GFX_X,GFX_Y
  // Previous eye was centred on GFX_XL,GFX_YL
  // Eye radius = GFX_R0
  // Pupil radiuous = GFX_R1
  // Called from Loop0 Core0 when GFX_Run > 0
  switch (GFX_Task) {
    case 0: // draw sclera background bar to obliterate previous iris
      // t0 = micros();
      if (GFX_Steps > 1) {
        // There are more than one movement steps to be completed
        GFX_X = GFX_XL + ((GFX_XT - GFX_XL)/GFX_Steps);
        GFX_Y = GFX_YL + ((GFX_YT - GFX_YL)/GFX_Steps);
      }
      GFX_Sprites = Sprites;
      if (!GFX_Sprites) {
        // normal, non-sprites drawing mode
        MPU_En = false;
        tft.fillRect(GFX_XL-GFX_R0, GFX_YL-GFX_R0, 1 + (2 * GFX_R0), 1 + (2 * GFX_R0), BakCol);
        MPU_En = true;
      } else {
        // Draw eye refresh using sprite mode (best)
        // Create a sprite canvas which covers old and new eye locations
        if (GFX_X < GFX_XL) {GFX_IX0 = GFX_X-GFX_R0; GFX_IX1 = GFX_XL+GFX_R0; GFX_IX = GFX_R0;}
        else {GFX_IX0 = GFX_XL-GFX_R0; GFX_IX1 = GFX_X+GFX_R0; GFX_IX = GFX_IX1-GFX_IX0-GFX_R0;}
        if (GFX_Y < GFX_YL) {GFX_IY0 = GFX_Y-GFX_R0; GFX_IY1 = GFX_YL+GFX_R0; GFX_IY = GFX_R0;}
        else {GFX_IY0 = GFX_YL-GFX_R0; GFX_IY1 = GFX_Y+GFX_R0;GFX_IY = GFX_IY1-GFX_IY0-GFX_R0;}
        GFX_IW = GFX_IX1 - GFX_IX0 + 1; GFX_IH = GFX_IY1 - GFX_IY0 + 1;
        img.createSprite(GFX_IW,GFX_IH); // width, height
        img.fillSprite(BakCol);                               // fill with background colour
      } break;
    case 1:
      // draw new iris
      if (!GFX_Sprites) {
        // normal, non-sprites drawing mode
        MPU_En = false;
        // set colour                                            rrrrrggggggbbbbb
        if (JoyEye == 0) {tft.fillCircle(GFX_X, GFX_Y, GFX_R0, 0b0010010010011111);}  // blue
                    else {tft.fillCircle(GFX_X, GFX_Y, GFX_R0, 0b1111100001000010);}  // red
        MPU_En = true;
      } else {
        // draw eye refresh using sprite mode (best)
        // set colour                                              rrrrrggggggbbbbb
        if (JoyEye == 0) {img.fillCircle(GFX_IX, GFX_IY, GFX_R0, 0b0010010010011111);}  // blue
                    else {img.fillCircle(GFX_IX, GFX_IY, GFX_R0, 0b1111100001000010);}  // red
      }
      // Serial.print(micros() - t0); Serial.print("\t");
      break;
    case 2: // draw black pupil
      if (!GFX_Sprites) {
        // normal, non-sprites drawing mode
        MPU_En = false;
        tft.fillCircle(GFX_X, GFX_Y, GFX_R1,TFT_BLACK);
      } else {
        // draw eye refresh using sprite mode (best)
        img.fillCircle(GFX_IX, GFX_IY, GFX_R1,TFT_BLACK);
        MPU_En = false;
        img.pushSprite(GFX_IX0, GFX_IY0);   // copy sprite to screen
        img.deleteSprite();                 // free up RAM
      }
      MPU_En = true;
      GFX_XL = GFX_X; GFX_YL = GFX_Y; // record the new eye position
      // If GFX_Steps > 1 then this process will repeat until all of the steps are completed
      if (GFX_Steps > 0) {GFX_Steps--; GFX_Task = 0;} else {GFX_ClrRun();}
      // Serial.println(micros() - t0);
      return;           // terminate the run
    default: // end of drawing tasks, so reset system
      GFX_ClrRun(); return;           // avoid incrementing the task
  } GFX_Task++;
}

// --------------------------------------------------------------------------------

void GFX_Task_Pupil() {
  // A GFX_Run task which draws an eye pupil centred at GFX_XL,GFX_YL
  // Called from Loop0 Core0 when GFX_Run > 0
  if (GFX_R1 < GFX_R1Tgt) {GFX_R1++;}
  else if (GFX_R1 > GFX_R1Tgt) {GFX_R1--;}
  MPU_En = false; tft.fillCircle(GFX_X, GFX_Y, GFX_R1, TFT_BLACK); MPU_En = true;
  GFX_ClrRun();
}

// --------------------------------------------------------------------------------

void GFX_TxtTasks() {
  // GFX text task sequences are run when GFX_Txt > 0
  // Each sequence needs to have parameters preset before running it
  // Once completed GFX_Txt is set to zero, so that another can be defined and called
  while (GFX_Txt > 0) {
    GFX_Txt--;
    if (TxtFP[GFX_Txt] > 0) {
      // Previous text had been drawn, so rewrite it in background colour
      tft.setTextColor(BakCol);
           if (TxtAP[GFX_Txt] == 0) {TFT_Text_Rht(TxtXP[GFX_Txt],TxtYP[GFX_Txt],TxtFP[GFX_Txt],TxtP$[GFX_Txt]);}
      else if (TxtAP[GFX_Txt] == 1) {TFT_Text_Ctr(TxtXP[GFX_Txt],TxtYP[GFX_Txt],TxtFP[GFX_Txt],TxtP$[GFX_Txt]);}
      else if (TxtAP[GFX_Txt] == 2) {TFT_Text_Lft(TxtXP[GFX_Txt],TxtYP[GFX_Txt],TxtFP[GFX_Txt],TxtP$[GFX_Txt]);}
      TxtFP[GFX_Txt] = 0;
    }
    // Now draw new text if font size > 0
    if (TxtFN[GFX_Txt] > 0) {
      // New text has been defined
      tft.setTextColor(TxtCol);
           if (TxtAN[GFX_Txt] == 0) {TFT_Text_Rht(TxtXN[GFX_Txt],TxtYN[GFX_Txt],TxtFN[GFX_Txt],TxtN$[GFX_Txt]);}
      else if (TxtAN[GFX_Txt] == 1) {TFT_Text_Ctr(TxtXN[GFX_Txt],TxtYN[GFX_Txt],TxtFN[GFX_Txt],TxtN$[GFX_Txt]);}
      else if (TxtAN[GFX_Txt] == 2) {TFT_Text_Lft(TxtXN[GFX_Txt],TxtYN[GFX_Txt],TxtFN[GFX_Txt],TxtN$[GFX_Txt]);}
      // Store references in previous for erasure process
      TxtAP[GFX_Txt] = TxtAN[GFX_Txt]; TxtFP[GFX_Txt] = TxtFN[GFX_Txt];
      TxtXP[GFX_Txt] = TxtXN[GFX_Txt]; TxtYP[GFX_Txt] = TxtYN[GFX_Txt];
      TxtP$[GFX_Txt] = TxtN$[GFX_Txt];
      break;  // exit after drawing new text string
    }
  }
}

// --------------------------------------------------------------------------------

void TFT_Text_Ctr(int16_t zX,int16_t zY,uint8_t zS,String zT$) {
  // Display text zT$ centred on zX,zY, in size zS
  // Standard font in GFX is 6 pixels wide, 8 pixels high, allowing 1 pixel border
  // Display co-ordinate system 0,0 is screen top left
  // Text co-ordinate system 0,0 is character top left
  tft.setTextSize(zS);                    // set the font size first
  zX = zX - ((zT$.length() * 6 * zS)/2);  // centre text based on half width
  tft.setCursor(zX, zY);
  MPU_En = false; tft.print(zT$); MPU_En = true;
}

// --------------------------------------------------------------------------------

void TFT_Text_Lft(int16_t zX,int16_t zY,uint8_t zS,String zT$) {
  // Display text zT$ left justified zX,zY, in size zS
  // Standard font in GFX is 6 pixels wide, 8 pixels high, allowing 1 pixel border
  // Display co-ordinate system 0,0 is screen top left
  // Text co-ordinate system 0,0 is character top left
  tft.setTextSize(zS);                    // set the font size first
  tft.setCursor(zX, zY);                  // place text at zX,zY top left
  MPU_En = false; tft.print(zT$); MPU_En = true;
}

// --------------------------------------------------------------------------------

void TFT_Text_Rht(int16_t zX,int16_t zY,uint8_t zS,String zT$) {
  // Display text zT$ right justified at zX,zY, in size zS
  // Standard font in GFX is 6 pixels wide, 8 pixels high, allowing 1 pixel border
  // Display co-ordinate system 0,0 is screen top left
  // Text co-ordinate system 0,0 is character top left
  tft.setTextSize(zS);                    // set the font size first
  zX = zX - (zT$.length() * 6 * zS);      // right justify text from zX
  tft.setCursor(zX, zY);
  MPU_En = false; tft.print(zT$); MPU_En = true;
}

// --------------------------------------------------------------------------------

