
// --------------------------------------------------------------------------------

int16_t abs16t(int16_t zV) {
  // returns the absolute value of zV
  if (zV >= 0) {return zV;}
  else {return -zV;}
}

// --------------------------------------------------------------------------------

void CalcLimits() {
  // Called to set soft limits, at start or during TEST mode changes
  // Battery multiplier used in PWM limiting
  BatMult = 255.0 * V_max;

  // Set PWM colour thresholds
  Mtr_10 = MtrMin + 10; Mtr_20 = MtrMin + 20; Mtr_30 = MtrMin + 30;
  Mtr_40 = MtrMin + 40; Mtr_50 = MtrMin + 50; Mtr_60 = MtrMin + 60;
  Mtr_70 = MtrMin + 70; Mtr_80 = MtrMin + 80;

  // Start limits based on PtchACSbSpSa and PtchBDSbSpSa
  PtchACSaN = PtchACSbSpSa - 0.5;
  PtchBDSaN = PtchBDSbSpSa - 0.5;
  PtchACSaP = PtchACSbSpSa + 0.5;
  PtchBDSaP = PtchBDSbSpSa + 0.5;
}

// ---------------------------------------------------------------------------------

void decodeKey(int zkeyVal) {
  // Decodes zkeyVal and excutes commands on '.'
  // Slow down the sending function when receiving
  // if ((millis() - print40ms) > 6) {print40ms -= 6;}

  if (zkeyVal == 10) {return;}
  if (zkeyVal == 13) {return;}
  keyChar = char(zkeyVal);
  switch (keyChar) {
    case '.': doCmd(); return;                        // command terminator
    case '!': PWM_OFF(); ESP.restart(); return;       // RESET and restart the code
    case '#': // if AppCnt > 0, conected to Windows app via USB or WiFi
      if (AppCnt == 0) {setAppMode();}
      AppCnt = 125; return;   // set 5 sec timeout                   
    case '~': // connected to OLED Mirror app via USB or WiFi
      if (!DispMon) {DispMon = true; Display_Intro();} // display mirrored message
      DispTx = 50; return;
    case '-': cmdSgn = -1; return;                    // numeric sign
    case '*': cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1; return; // ESC abort command
    case '0': extendCmdVal(0); return;
    case '1': extendCmdVal(1); return;
    case '2': extendCmdVal(2); return;
    case '3': extendCmdVal(3); return;
    case '4': extendCmdVal(4); return;
    case '5': extendCmdVal(5); return;
    case '6': extendCmdVal(6); return;
    case '7': extendCmdVal(7); return;
    case '8': extendCmdVal(8); return;
    case '9': extendCmdVal(9); return;
  }
  if (cmdMode == ' ') {
    // test for new Command Mode char?
    // only accept certain characters as valid commands, or ignore them
    cmdMode = keyChar;
    switch (keyChar) {
      // check for valid mode and convert upper-case
      case 'd': cmdMode = 'D'; break;
      case 'g': cmdMode = 'G'; break;
      case 'p': cmdMode = 'P'; break;
      case 's': cmdMode = 'S'; break;
    } cmdType = ' '; cmdVal = 0;
  } else {
    // test for Command Type char?
    cmdType = keyChar;
    switch (keyChar) {
      // check for lower-case and convert to upper-case
      case 'd': cmdType = 'D'; break;
      case 'e': cmdType = 'E'; break;
      case 'l': cmdType = 'L'; break;
      case 'm': cmdType = 'M'; break;
      case 'p': cmdType = 'P'; break;
      case 's': cmdType = 'S'; break;
      case 't': cmdType = 'T'; break;
      case 'v': cmdType = 'V'; break;
      case 'w': cmdType = 'W'; break;
    }
  }
}

// --------------------------------------------------------------------------------

void displayLft() {
  // change the DispMode pointer to the previous display
  if (DispLock) {return;} // ignore mouse click when display is locked
  if (DmDwn) {return;}    // Only respond to one mouse down event

  DispMode--; if (DispMode < DM_Min) {DispMode = DM_Max;}
  DispDel = 1; DispCnt = 0; DmDwn = true;
}

// --------------------------------------------------------------------------------

void displayRht() {
  // Change the DispMode pointer to the next display
  if (DispLock) {return;} // ignore mouse click when display is locked
  if (DmDwn) {return;}    // Only respond to one mouse down event

  DispMode++; if (DispMode > DM_Max) {DispMode = DM_Min;}
  DispDel = 1; DispCnt = 0; DmDwn = true;
}

// --------------------------------------------------------------------------------

void DM_BrkPntInc() {
  // Adjust BrkPnt value
  float zI = 0.1;
       if (DmX <= 34) {zI = 10.0;}
  else if (DmX <= 40) {zI =  1.0;}
  else if (DmX <= 45) {zI =  0.1;}
  if (DmB) {BrkPnt-= zI;} else {BrkPnt+= zI;}
  if (BrkPnt > 99.9) {BrkPnt = 99.9;} else if (BrkPnt < 0.0) {BrkPnt = 0.00;}
}

// --------------------------------------------------------------------------------

void DM_BrkValInc() {
  // Adjust BrkVal value
  float zI = 0.001;
       if (DmX <= 79) {zI =  1.0;}
  else if (DmX <= 84) {zI =  0.1;}
  else if (DmX <= 88) {zI =  0.01;}
  if (DmB) {BrkVal-= zI;} else {BrkVal+= zI;}
  if (BrkVal > 99.9) {BrkVal = 99.9;} else if (BrkVal < 0.0) {BrkVal = 0.00;}
}

// --------------------------------------------------------------------------------

void DM_Control_() {
  //##################################
  // Control display
  //##################################
  // user has clicked on a blue field in Monitor+ display
  if (DmX < 50) {displayLft();}
  else {
    // Has user clicked on a blue field?
         if ((DmX >= 50) && (DmX <= 64) && (DmY >= 25) && (DmY <= 36)) {ToggleTestMode(); DmDwn = true;}
    else if ((DmX >= 50) && (DmX <= 64) && (DmY >= 43) && (DmY <= 54)) {BatAvg = 0; BatSum = 0; DmDwn = true;}
    else if ((DmX >= 66) && (DmX <= 78) && (DmY >= 62) && (DmY <= 73)) {setMainMode(MainTest); DmDwn = true;}
    else if ((DmX >= 84) && (DmX <= 88) && (DmY >= 62) && (DmY <= 73)) {DM_IncMainTest(); DmDwn = true;}
    else if ((DmX >= 50) && (DmX <= 66) && (DmY >= 80) && (DmY <= 91)) {WiFiKill(); DmDwn = true;}
    else {displayRht();}
  }
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_DriveCtrl_() {
  //##################################
  // Drive Control display
  //##################################
  // user has clicked on a blue field in Monitor+ display
  if (DmX < 50) {displayLft();}
  else {
    // Has user clicked on a blue field?
         if ((DmX <= 64) && (DmY >= 62) && (DmY <= 73)) {DriveMinInc(); DmDwn = true;}
    else if ((DmX <= 64) && (DmY >= 80) && (DmY <= 91)) {DriveMaxInc(); DmDwn = true;}
    else {displayRht();}
  }
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_D_gainInc() {
  // Adjust pid_d_gain value
  float zI = 0.01;
       if (DmX <= 53) {zI = 10.0;}
  else if (DmX <= 59) {zI =  1.0;}
  else if (DmX <= 64) {zI =  0.1;}
  if (DmB) {pid_d_gain-= zI;} else {pid_d_gain+= zI;}
  if (pid_d_gain > 99.99) {pid_d_gain = 99.99;} else if (pid_d_gain < 0.00) {pid_d_gain = 0.00;}
}

// --------------------------------------------------------------------------------

void DM_EyeEng_() {
  //##################################
  // Eye Engine display
  //##################################
  // user has clicked on a blue field in Monitor+ display
  if (DmX < 50) {displayLft();}
  else {
    // Has user clicked on a blue field?
         if ((DmX >= 51) && (DmX <= 59) && (DmY >= 22) && (DmY <= 30)) {Eye_En = !Eye_En; DmDwn = true;}
    else if ((DmX >= 51) && (DmX <= 59) && (DmY >= 37) && (DmY <= 46)) {Sprites = !Sprites; DmDwn = true;}
    else {displayRht();}
  }
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_GyAcModInc() {
  // Adjust GyAcMod value
  float zI = 0.001;
       if (DmX <= 48) {zI = 1.000;}
  else if (DmX <= 54) {zI = 0.100;}
  else if (DmX <= 57) {zI = 0.010;}
  else if (DmX <= 62) {zI = 0.001;}
  if (DmB) {GyAcMod-= zI;} else {GyAcMod+= zI;}
  if (GyAcMod > 1.000) {GyAcMod = 1.000;} else if (GyAcMod < 0.000) {GyAcMod = 0.000;}
}

// ---------------------------------------------------------------------------------

void DM_I_gainInc() {
  // Adjust pid_i_gain value
  float zI = 0.01;
       if (DmX <= 53) {zI = 10.0;}
  else if (DmX <= 59) {zI =  1.0;}
  else if (DmX <= 64) {zI =  0.1;}
  if (DmB) {pid_i_gain-= zI;} else {pid_i_gain+= zI;}
  if (pid_i_gain > 99.99) {pid_i_gain = 99.99;} else if (pid_i_gain < 0.00) {pid_i_gain = 0.00;}
}

// --------------------------------------------------------------------------------

void DM_MPU_Acc_() {
  //##################################
  // MPU Acc display
  //##################################
  // user has clicked on a blue field in Monitor+ display
  if (DmX < 50) {displayLft();}
  else {
    // has user clicked on a blue field
    if ((DmX >= 67) && (DmX <= 72) && (DmY >= 9) && (DmY <= 19)) {AcOffEn = !AcOffEn; DmDwn = true;}
    else {displayRht();}
  }
  DispDel = 1;  // Update after 40ms
}

// --------------------------------------------------------------------------------

void DM_MPU_Gyr_() {
  //##################################
  // MPU Gyros display
  //##################################
  // user has clicked on a blue field in Monitor+ display
  if (DmX < 50) {displayLft();}
  else {
    // has user clicked on a blue field
    if ((DmX >= 70) && (DmX <= 74) && (DmY >= 9) && (DmY <= 19)) {GyOffEn = !GyOffEn; DmDwn = true;}
    else {displayRht();}
  }
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_IncMainTest() {
  // Increment/decrement MainTest value
  if (DmB == 0) {MainTest++;} else {MainTest--;}
  if (MainTest > 4) {MainTest = 0;}
  else if (MainTest < 0) {MainTest = 4;}
}

// ---------------------------------------------------------------------------------

void DM_Imax_gainInc() {
  // Adjust pid_i_max value
  float zI = 0.1;
       if (DmX <= 81) {zI = 100.0;}
  else if (DmX <= 85) {zI =  10.0;}
  else if (DmX <= 91) {zI =   1.0;}
  if (DmB) {pid_i_max-= zI;} else {pid_i_max+= zI;}
  if (pid_i_max > 255.00) {pid_i_max = 255.00;} else if (pid_i_max < 0.00) {pid_i_max = 0.00;}
}

// ---------------------------------------------------------------------------------

void DM_Limits_() {
  //##################################
  // Limits display
  //##################################
  // User has clicked on a blue field in Monitor+ display
  if (TEST) {
        if (DmY < 22) {if (DmX < 55) {displayLft();} else {displayRht();}}
    else if (DmY < 75) {
      if (DmX < 42) {displayLft();}
      else {
        // has user clicked on a blue field
            if ((DmX >= 43) && (DmX <= 57) && (DmY >= 22) && (DmY <= 32)) {DM_V_MaxInc();}
        else if ((DmX >= 43) && (DmX <= 55) && (DmY >= 40) && (DmY <= 51)) {DM_MtrMinInc();}
        else if ((DmX >= 85) && (DmX <= 96) && (DmY >= 40) && (DmY <= 51)) {DM_MtrSrtInc();}
        else if ((DmX >= 43) && (DmX <= 61) && (DmY >= 59) && (DmY <= 69)) {DM_GyAcModInc();}
        else {displayRht();}
      }
    }
    else if (DmX < 30) {displayLft();}
    else if (DmX <= 45) {DM_BrkPntInc();}
    else if (DmX <= 72) {displayRht();}
    else {DM_BrkValInc();}
  } else {if (DmX < 50) {displayLft();} else {displayRht();}}
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_MtrInc0(int16_t zP) {
  // Adjust PWM_?0 value
  int16_t zI = 1; if (DmSft) {zI = 10;}
  switch (zP) {
    case 0:
      if (DmB) {PWM_AC0-= zI;} else {PWM_AC0+= zI;}
      if (PWM_AC0 > 255) {PWM_AC0 = 255;} else if (PWM_AC0 < 0) {PWM_AC0 = 0;}
      analogWrite(Pin_AC0, PWM_AC0); PWM_AC = getPWM(PWM_AC0,PWM_AC1,PWM_AC_); break;
    case 1:
      if (DmB) {PWM_BD0-= zI;} else {PWM_BD0+= zI;}
      if (PWM_BD0 > 255) {PWM_BD0 = 255;} else if (PWM_BD0 < 0) {PWM_BD0 = 0;}
      analogWrite(Pin_BD0, PWM_BD0); PWM_BD = getPWM(PWM_BD0,PWM_BD1,PWM_BD_); break;
  }
}

// ---------------------------------------------------------------------------------

void DM_MtrInc1(int16_t zP) {
  // Adjust PWM_?1 value
  int16_t zI = 1; if (DmSft) {zI = 10;}
  switch (zP) {
    case 0:
      if (DmB) {PWM_AC1-= zI;} else {PWM_AC1+= zI;}
      if (PWM_AC1 > 255) {PWM_AC1 = 255;} else if (PWM_AC1 < 0) {PWM_AC1 = 0;}
      analogWrite(Pin_AC1, PWM_AC1); PWM_AC = getPWM(PWM_AC0,PWM_AC1,PWM_AC_); break;
    case 1:
      if (DmB) {PWM_BD1-= zI;} else {PWM_BD1+= zI;}
      if (PWM_BD1 > 255) {PWM_BD1 = 255;} else if (PWM_BD1 < 0) {PWM_BD1 = 0;}
      analogWrite(Pin_BD1, PWM_BD1); PWM_BD = getPWM(PWM_BD0,PWM_BD1,PWM_BD_); break;
  }
}

// ---------------------------------------------------------------------------------

void DM_MtrInc2(int16_t zP) {
  // Adjust PWM_? value
  int16_t zI = 1; if (DmSft) {zI = 10;}
  switch (zP) {
    case 0:
      if (DmB) {PWM_AC-= zI;} else {PWM_AC+= zI;}
      if (PWM_AC > 255) {PWM_AC = 255;} else if (PWM_AC < -255) {PWM_AC = -255;}
      if (PWM_AC_) {
             if (PWM_AC > 0) {PWM_AC0 = 255 - PWM_AC; PWM_AC1 = 255;}
        else if (PWM_AC < 0) {PWM_AC0 = 255; PWM_AC1 = 255 + PWM_AC;}
        else {PWM_AC0 = 255; PWM_AC1 = 255;}
      } else {
             if (PWM_AC > 0) {PWM_AC0 = 0; PWM_AC1 = PWM_AC;}
        else if (PWM_AC < 0) {PWM_AC0 = -PWM_AC; PWM_AC1 = 0;}
        else {PWM_AC0 = 0; PWM_AC1 = 0;}
      }
      analogWrite(Pin_AC0, PWM_AC0); analogWrite(Pin_AC1, PWM_AC1); break;
    case 1:
      if (DmB) {PWM_BD-= zI;} else {PWM_BD+= zI;}
      if (PWM_BD > 255) {PWM_BD = 255;} else if (PWM_BD < -255) {PWM_BD = -255;}
      if (PWM_BD_) {
             if (PWM_BD > 0) {PWM_BD0 = 255 - PWM_BD; PWM_BD1 = 255;}
        else if (PWM_BD < 0) {PWM_BD0 = 255; PWM_BD1 = 255 + PWM_BD;}
        else {PWM_BD0 = 255; PWM_BD1 = 255;}
      } else {
             if (PWM_BD > 0) {PWM_BD0 = 0; PWM_BD1 = PWM_BD;}
        else if (PWM_BD < 0) {PWM_BD0 = -PWM_BD; PWM_BD1 = 0;}
        else {PWM_BD0 = 0; PWM_BD1 = 0;}
      }
      analogWrite(Pin_BD0, PWM_BD0); analogWrite(Pin_BD1, PWM_BD1); break;
  }
}

// ---------------------------------------------------------------------------------

void DM_MtrMinInc() {
  // Adjust MtrMin value
  if (!ModPWM) {return;}

  int16_t zI = 1;
       if (DmX <= 47) {zI = 100;}
  else if (DmX <= 51) {zI =  10;}
  else if (DmX <= 55) {zI =   1;}
  if (DmB) {MtrMin-= zI;} else {MtrMin+= zI;}
  if (MtrMin > 150) {MtrMin = 150;} else if (MtrMin < 0) {MtrMin = 0;}
}

// ---------------------------------------------------------------------------------

void DM_MtrSrtInc() {
  // Adjust MtrSrt value
  if (!ModPWM) {return;}

  int16_t zI = 1;
       if (DmX <= 88) {zI = 100;}
  else if (DmX <= 93) {zI =  10;}
  else if (DmX <= 97) {zI =   1;}
  if (DmB) {MtrMinStrt-= zI;} else {MtrMinStrt+= zI;}
  if (MtrMinStrt > 150) {MtrMinStrt = 150;} else if (MtrMinStrt < 0) {MtrMinStrt = 0;}
}

// ---------------------------------------------------------------------------------

void DM_MtrOff(int16_t zP) {
  // Turn off PWM_?
  switch (zP) {
    case 0:
      PWM_AC = 0;
      if (DmB) {PWM_AC0 = 0; PWM_AC1 = 0; PWM_AC_ = 0;} else {PWM_AC0 = 255; PWM_AC1 = 255; PWM_AC_ = 1;}
      analogWrite(Pin_AC0, PWM_AC0); analogWrite(Pin_AC1, PWM_AC1); break;
    case 1:
      PWM_BD = 0;
      if (DmB) {PWM_BD0 = 0; PWM_BD1 = 0; PWM_BD_ = 0;} else {PWM_BD0 = 255; PWM_BD1 = 255; PWM_BD_ = 1;}
      analogWrite(Pin_BD0, PWM_BD0); analogWrite(Pin_BD1, PWM_BD1); break;
  }
}

// ---------------------------------------------------------------------------------

void DM_MtrTest_() {
  //##################################
  // Motor Test display
  //##################################
  // user has clicked on a blue field in Monitor+ display
  if (DmY < 27) {
    if (DmX < 50) {displayLft();} else {displayRht();}
  } else {
    if (DmX < 43) {
      // has user clicked on a blue field
      if ((DmX >= 14) && (DmX <= 27)) {
            if ((DmY >= 27) && (DmY <= 38)) {DM_MtrInc0(0);}
        else if ((DmY >= 45) && (DmY <= 55)) {DM_MtrInc0(1);}
        else if ((DmY >= 63) && (DmY <= 74)) {DM_MtrInc0(2);}
        else if ((DmY >= 82) && (DmY <= 92)) {DM_MtrInc0(3);}
      } else {displayLft();}
    } else {
      // has user clicked on a blue field
      if ((DmX >= 43) && (DmX <= 55)) {
            if ((DmY >= 27) && (DmY <= 38)) {DM_MtrInc1(0);}
        else if ((DmY >= 45) && (DmY <= 55)) {DM_MtrInc1(1);}
        else if ((DmY >= 63) && (DmY <= 74)) {DM_MtrInc1(2);}
        else if ((DmY >= 82) && (DmY <= 92)) {DM_MtrInc1(3);}
      }
      else if ((DmX >= 64) && (DmX <= 79)) {
            if ((DmY >= 27) && (DmY <= 38)) {DM_MtrInc2(0);}
        else if ((DmY >= 45) && (DmY <= 55)) {DM_MtrInc2(1);}
        else if ((DmY >= 63) && (DmY <= 74)) {DM_MtrInc2(2);}
        else if ((DmY >= 82) && (DmY <= 92)) {DM_MtrInc2(3);}
      }
      else if ((DmX >= 82) && (DmX <= 96)) {
            if ((DmY >= 27) && (DmY <= 38)) {DM_MtrOff(0);}
        else if ((DmY >= 45) && (DmY <= 55)) {DM_MtrOff(1);}
        else if ((DmY >= 63) && (DmY <= 74)) {DM_MtrOff(2);}
        else if ((DmY >= 82) && (DmY <= 92)) {DM_MtrOff(3);}
      }
      else {displayRht();}
    }
  }
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_P_gainInc() {
  // Adjust pid_p_gain value
  float zI = 0.1;
       if (DmX <= 53) {zI = 100.0;}
  else if (DmX <= 58) {zI =  10.0;}
  else if (DmX <= 63) {zI =   1.0;}
  if (DmB) {pid_p_gain-= zI;} else {pid_p_gain+= zI;}
  if (pid_p_gain > 999.9) {pid_p_gain = 999.9;} else if (pid_p_gain < 0.00) {pid_p_gain = 0.00;}
}

// ---------------------------------------------------------------------------------

void DM_PID_() {
  //##################################
  // PID Controller display
  //##################################
  // User has clicked on a red or blue field in Monitor+ display
  if (TEST) {
          if (DmY < 26) {
              if (DmX < 15) {setDefPID();}
              else if (DmX < 50) {displayLft();} else {displayRht();}}
      else if ((DmY >= 26) && (DmY < 74)) {
          if (DmX < 50) {displayLft();}
      else if ((DmX >= 50) && (DmX <= 68) && (DmY <= 36)) {DM_P_gainInc();}
      else if ((DmX >= 86) && (DmX <= 96) && (DmY <= 36)) {DM_PID_dbInc();}
      else if ((DmX >= 50) && (DmX <= 68) && (DmY >= 44) && (DmY <= 54)) {DM_I_gainInc();}
      else if ((DmX >= 74) && (DmX <= 96) && (DmY >= 44) && (DmY <= 54)) {DM_Imax_gainInc();}
      else if ((DmX >= 50) && (DmX <= 68) && (DmY >= 62) && (DmY <= 73)) {DM_D_gainInc();}
      else {displayRht();}
    }
    else if ((DmY >= 81) && (DmY <= 91)) {
          if ((DmX >= 29) && (DmX <= 47)) {DM_PtchACSaInc();}
      else if ((DmX >= 77) && (DmX <= 97)) {DM_PtchBDSaInc();}
      else if (DmX < 50) {displayLft();} else {displayRht();}
    }
    else if (DmX < 50) {displayLft();} else {displayRht();}
  } else if (DmX < 50) {displayLft();} else {displayRht();}
  DispDel = 1;  // Update after 40ms
}

// ---------------------------------------------------------------------------------

void DM_PID_dbInc() {
  // Adjust PID_db value
  float zI = 0.1;
  if (DmX <= 91) {zI = 1.0;}
  if (DmB) {PID_db-= zI;} else {PID_db+= zI;}
  if (PID_db > 4.9) {PID_db = 4.9;} else if (PID_db < 0.0) {PID_db = 0.0;}
}

// ---------------------------------------------------------------------------------

void DM_PtchACSaInc() {
  // Adjust PtchACSbSpSa value
  float zI = 0.01;
       if (DmX <= 37) {zI = 1.0;}
  else if (DmX <= 42) {zI = 0.1;}
  if (PtchACSbSpSa < 0.0) {zI = -zI;} // reverse adjustment for negative values
  if (DmB) {PtchACSbSpSa-= zI;} else {PtchACSbSpSa+= zI;}
  if (PtchACSbSpSa > 9.99) {PtchACSbSpSa = 9.99;} else if (PtchACSbSpSa < -9.99) {PtchACSbSpSa = -9.99;}
}

// ---------------------------------------------------------------------------------

void DM_PtchBDSaInc() {
  // Adjust PtchBDSbSpSa value
  float zI = 0.01;
       if (DmX <= 84) {zI = 1.0;}
  else if (DmX <= 89) {zI = 0.1;}
  if (PtchBDSbSpSa < 0.0) {zI = -zI;} // reverse adjustment for negative values
  if (DmB) {PtchBDSbSpSa-= zI;} else {PtchBDSbSpSa+= zI;}
  if (PtchBDSbSpSa > 9.99) {PtchBDSbSpSa = 9.99;} else if (PtchBDSbSpSa < -9.99) {PtchBDSbSpSa = -9.99;}
}

// ---------------------------------------------------------------------------------

void DM_V_MaxInc() {
  // Adjust V_max value
  float zI = 0.1;
       if (DmX <= 48) {zI = 1.00;}
  else if (DmX <= 53) {zI = 0.10;}
  else if (DmX <= 57) {zI = 0.01;}
  if (DmB) {V_max-= zI;} else {V_max+= zI;}
  if (V_max > 8.20) {V_max = 8.20;} else if (V_max < 6.60) {V_max = 6.60;}
  CalcLimits();   // update soft limits
}

// ---------------------------------------------------------------------------------

void doCmd() {
  // a '.' has been received on the serial port so execute command if valid
  // Commands:
  // DDn.     - increase (1) or decrease (-1) DispDel global delay offset DispDelGbl
  // DLn.     - toggle display lock, n=0 OFF, n=1 ON
  // DMxxyy.  - display monitor left mouse click, xx = 0-99, yy = 0-99
  // DM-.     - display monitor mouse up event
  // DS.      - SHIFT-key added to DMxxyy. message
  // G.       - report Scope data for graphing app
  // GE.      - end of Scope graphing mode
  // GPnn.    - nn points to data to be sent
  // PMnn.    - sets ModPWM flag, 0 = false, 1 = true
  // PTnn.    - set target wheel PWM pointer
  // PVnn.    - set PWM value
  // PW.      - list modified PWM table to the serial port
  // SLnn.    - set an LED mode directly.

  cmdVal*= cmdSgn;  // correct cmdVal with the sign value
  
  switch (cmdMode) {
    case ' ':
      PrintTx = "Hello\n"; break;
    case 'D':
      switch (cmdType) {
        case 'D': // DDn. command, adjust global delay offset
          if (cmdVal > 0) {DispDelGbl ++;} else {DispDelGbl --;}
          if (DispDelGbl < 0) {DispDelGbl = 0;}
          // display this in the Rx field of Monitor+
          PrintTx+= "$DispDelGbl: " + String(DispDelGbl) +"\n";
          break;
        case 'L':
          if (cmdVal != 0) {DispLock = true; DispOveride = DispMode;}
          else {DispLock = false;}
          break;
        case 'M':
          if (cmdSgn < 0) {  // mouse click released
            DmDwn = false;
            // Serial.println("DwDwn=false");
          } else {
            if (cmdVal >= 10000) {
              // this is a Display Monitor mouse right-click
              // determine whether it is a left or right button click
              cmdVal /= 10000; cmdVal --; DmB = 1;
              DmY = cmdVal % 100; DmX = cmdVal/100;
            } else {
              // this is a Display Monitor mouse left-click
              DmY = cmdVal % 100; DmX = cmdVal/100; DmB = 0;
            }
            if (!DmDwn) {
                  if (DispMode == DM_Control) {DM_Control_();}
              else if (DispMode == DM_DriveCtrl) {DM_DriveCtrl_();}
              else if (DispMode == DM_EyeEng) {DM_EyeEng_();}
              else if (DispMode == DM_MPU_Acc) {DM_MPU_Acc_();}
              else if (DispMode == DM_MPU_Gyros) {DM_MPU_Gyr_();}
              else if (DispMode == DM_Limits) {DM_Limits_();}
              else if (DispMode == DM_MtrTest) {DM_MtrTest_();}
              else if (DispMode == DM_PID) {DM_PID_();}
              else {
                // this is not a special 'clickable' display, so simply change it
                if (DmX < 50) {displayLft();  // left side click
                } else {displayRht();}       // right side click
              }
            }
            // Serial.println("DwDwn=true");
          } DmSft = false; break;
        case 'S': // embedded SHIFT key
          DmSft = true; break;
        break;
      } break;
    case 'G':
      // Scope graphing app report commands
      switch (cmdType) {
        case ' ': 
          // Report the same set of values twice as WiFi data can easily be
          // corrupted. Therefore two matching consequitives messages must be
          // received for the graphing function to work
          // if in Display Monitor mode then switch that off
          if (DispMon) {DispMon = false; PrintTx = ""; DispTx = -1;}
          // RPms = millis() - RPt0; RPt0 = millis();
          ScopeReport(GPnn); ScopeReport(GPnn);
          break;
        case 'E': // GE. enter monitor mode when dropping out of graphing mode
          // return to Display Monitor mode if it was previously active
          if (DispTx < 0) {DispMon = true; PrintTx = ""; DispTx = 50; DispDel = 0; DispCnt = 0;}
          break;
        case 'P': // GPnn. sets the reporting mode and triggers title reporting
          if (DispMon) {DispMon = false; PrintTx = ""; DispTx = -1;}
          GPnn = cmdVal; 
          if (G_once) {G_once = false; GPnn = 0;}
          GP_Title = true;
          PrintTx += "GP" + String(GPnn) + "\n";
          break;
      } break;
    case 'P':
      // keyboard/serial commands used to set target wheel PWM pointer
      switch (cmdType) {
        case 'M': //sets ModPWM flag, 0 = false, 1 = true
          if (cmdVal == 1) {ModPWM = true; Serial.println("ModPWM enabled");}
          else {ModPWM = false; Serial.println("ModPWM disabled");}
          break;
        case 'T': // set target servo pointer in range 0 - 5
          if (cmdVal < 0) {cmdVal = MotorPnt;}
          if (cmdVal > 3) {cmdVal = MotorPnt;}
          MotorPnt = cmdVal; break;
        case 'V': // set wheel PWM values 0 - 255
          cmdVal = constrain(cmdVal,0,255);
          // set the PWM value, and report the action
               if (MotorPnt ==  0) {analogWrite(Pin_AC0, cmdVal); PrintTx += "analogWrite(Pin_AC0, " + String(cmdVal) + ")";}
          else if (MotorPnt ==  1) {analogWrite(Pin_AC1, cmdVal); PrintTx += "analogWrite(Pin_AC1, " + String(cmdVal) + ")";}
          else if (MotorPnt ==  2) {analogWrite(Pin_BD0, cmdVal); PrintTx += "analogWrite(Pin_BD0, " + String(cmdVal) + ")";}
          else if (MotorPnt ==  3) {analogWrite(Pin_BD1, cmdVal); PrintTx += "analogWrite(Pin_BD1, " + String(cmdVal) + ")";}
          break;
        case 'W': // list modified PWM table to the serial port
          getModPWMPrint();
          break;
      } break;
    case 'S':
      switch (cmdType) {
        case 'L': // set LED mode directly
          SetLEDmode(cmdVal);
          Serial.println("SL" + String(cmdVal) + ".");
          break;
      }
      break;
   default:
      PrintTx += "Unknown cmd: " + cmdMode + cmdType + String(cmdVal) + "\n";
      break;
  }
  // now reset the variables
  cmdMode = ' '; cmdType = ' '; cmdVal = 0; cmdSgn = 1; //cmdBlock = false;
}

// --------------------------------------------------------------------------------

void Do_LED_Task(bool zM) {
  // set LEDs or flags according to LED_Task value
  // zM = true turn LEDs ON/OFF
  // zM = false set LED flags
  // if (zM) {
  //   // it modifies LED outputs based on LED_Task value
  //   // the LOW state is used as the LED OFF state
  //   switch (LED_Task) {
  //     case -1: LEDs_OFF(); break; // ensure all are in the OFF state (LOW)
  //     case 0: // turns ON front blue LED_AB and LED_HL
  //       pinMode(LedCol2,INPUT_PULLUP); digitalWrite(LedRow1,LOW);
  //       digitalWrite(LedRow0, HIGH); digitalWrite(LedRow2, HIGH);
  //       pinMode(LedCol2,OUTPUT); digitalWrite(LedCol2, LOW); break;
  //     case 1: // turns ON right yellow LED_BY
  //       pinMode(LedCol2,INPUT_PULLUP); digitalWrite(LedRow0,LOW); digitalWrite(LedRow2, LOW);
  //       digitalWrite(LedRow1,HIGH); 
  //       pinMode(LedCol1,OUTPUT); digitalWrite(LedCol1,LOW); break;
  //     case 2: // turns ON right blue LED_BB
  //       pinMode(LedCol1,INPUT_PULLUP);
  //       digitalWrite(LedRow1,HIGH);
  //       pinMode(LedCol0,OUTPUT); digitalWrite(LedCol0,LOW); break;
  //     case 3: // turns ON rear yellow LED_CY
  //       pinMode(LedCol0,INPUT_PULLUP); digitalWrite(LedRow1,LOW);
  //       digitalWrite(LedRow2,HIGH);
  //       pinMode(LedCol1,OUTPUT); digitalWrite(LedCol1,LOW); break;
  //     case 4: // turns ON rear blue LED_CB
  //       pinMode(LedCol1,INPUT_PULLUP);
  //       digitalWrite(LedRow2,HIGH);
  //       pinMode(LedCol0,OUTPUT); digitalWrite(LedCol0,LOW); break;
  //     case 5: // turns ON left yellow LED_DY
  //       pinMode(LedCol0,INPUT_PULLUP); digitalWrite(LedRow2,LOW);
  //       digitalWrite(LedRow0,HIGH);
  //       pinMode(LedCol1,OUTPUT); digitalWrite(LedCol1,LOW); break;
  //     case 6: // turns ON left blue LED_DB
  //       pinMode(LedCol1,INPUT_PULLUP);
  //       digitalWrite(LedRow0,HIGH);
  //       pinMode(LedCol0,OUTPUT); digitalWrite(LedCol0,LOW); break;
  //     case 7: // turns ON front yellow LED_AY and LED_HL
  //       pinMode(LedCol0,INPUT_PULLUP); digitalWrite(LedRow0,LOW);
  //       digitalWrite(LedRow1,HIGH);  digitalWrite(LedRow2, HIGH);
  //       pinMode(LedCol2,OUTPUT); digitalWrite(LedCol2,LOW); break;
  //   }
  // } else {
  //   // this code is used in LEDMode = 2
  //   // it modifies flags based on LED_Task value
  //   ClrLEDFlags(); // set all flags to the OFF state (false)
  //   switch (LED_Task) {
  //     case 0: LED_AB = HIGH; break;
  //     case 1: LED_BY = HIGH; break;
  //     case 2: LED_BB = HIGH; break;
  //     case 3: LED_CY = HIGH; break;
  //     case 4: LED_CB = HIGH; break;
  //     case 5: LED_DY = HIGH; break;
  //     case 6: LED_DB = HIGH; break;
  //     case 7: LED_AY = HIGH; break;
  //   }
  // }
}

// ---------------------------------------------------------------------------------

void DriveMaxInc() {
  // Adjust DrvSpMax value
  float zI = 0.01;
       if (DmX <= 55) {zI = 1.0;}
  else if (DmX <= 59) {zI = 0.1;}
  if (DmB) {DrvSpMax-= zI;} else {DrvSpMax+= zI;}
  if (DrvSpMax > 9.99) {DrvSpMax = 9.99;} else if (DrvSpMax < 0.00) {DrvSpMax = 0.00;}
}

// ---------------------------------------------------------------------------------

void DriveMinInc() {
  // Adjust DrvSpMin value
  float zI = 0.01;
       if (DmX <= 55) {zI = 1.0;}
  else if (DmX <= 59) {zI = 0.1;}
  if (DmB) {DrvSpMin-= zI;} else {DrvSpMin+= zI;}
  if (DrvSpMin > 9.99) {DrvSpMin = 9.99;} else if (DrvSpMin < 0.00) {DrvSpMin = 0.00;}
}

// --------------------------------------------------------------------------------

void driveMotors(float zPID_X,float zPID_Y) {
  //#################################################################################
  // Motor PWM calculations
  //#################################################################################
  // Uses pid_outputs to develop a motor demand for all 4 motors,
  MotorDriveAC = (int)zPID_Y;
  MotorDriveBD = (int)zPID_X;

  // as adding in steering can exceed the max PWM value we need to apply limits
  if (MotorDriveAC > 255) {MotorDriveAC = 255;}
  else if (MotorDriveAC < -255) {MotorDriveAC = -255;}
  if (MotorDriveBD > 255) {MotorDriveBD = 255;}
  else if (MotorDriveBD < -255) {MotorDriveBD = -255;}
  
  // Motor drive wires are connected in reverse so we need to reverse the polarity
  // of the drive signals at this point. This depends on how you have connected
  // the DC motors and may not be necessary.
  setMotorAC_PWM(MotorDriveAC); setMotorBD_PWM(MotorDriveBD);
}

// ----------------------------------------------------------------------

void exitAppMode() {
  // Called when serial data from an external app is no longer deteceted
  GFX_Mode = 1; GFX_RST = true; Txt_Mode = 0;
}

// ---------------------------------------------------------------------------------

void extendCmdVal(int zVal) {
  // adds a new digit to the right-hand end of cmdVal
  cmdVal = abs(cmdVal * 10) + zVal;
}

// --------------------------------------------------------------------------------

void FlushRxBuffer() {
  // Read and ignore all characters in the serial port buffer  
  // while (Serial.available() > 0) {
  //     // read data to empty buffer
  //   keyVal = Serial.read();
  // }
}

// --------------------------------------------------------------------------------

String get3Digits(int16_t zV) {
  // Returns a 3-digit numbe from 000 to 999
  String zT$ = String(zV);
       if (zT$.length() < 2) {zT$ = "00" + zT$;}
  else if (zT$.length() < 3) {zT$ = "0" + zT$;}
  return zT$;
}

// --------------------------------------------------------------------------------

String Get4BIN(byte zVal) {
  // returns a 4 character binary representation of zVal
  String zBIN$ = "";
  // convert zVal into binary
  if (zVal & 0b1000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0b0100) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0b0010) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0b0001) {zBIN$ += "1";} else {zBIN$ += "0";}
  return zBIN$;
}

// --------------------------------------------------------------------------------

String getAtState() {
  // returns the AtState value as a string
  String zT$ = "";
  GAScnt++; if (GAScnt > 2) {GAScnt = 0;}
  switch (GAScnt) {
    case 0: zT$ = "..z  "; break;
    case 1: zT$ = ".zz "; break;
    case 2: zT$ = "zzZ"; break;
  }
  Any$ = "???";
  switch (AtState) {
    case CAL:       Any$ = "READY!"; break;
    case At_DRIVE:  Any$ = "Drive Mode"; break;
    case At_OMNI:   Any$ = "Omni Mode"; break;
    case At_READY:  Any$ = "READY!"; break;
    case At_RESET:  Any$ = "RESET..."; break;
    case At_REST:   Any$ = "Resting " + zT$; break;
  }
  if (Upright  <  0) {
  switch (GAScnt) {
    case 0: Any$ = "Upside"; break;
    case 1: Any$ = "Down"; break;
    case 2: Any$ = "???"; break;
  }
  }
  return Any$;
}

// --------------------------------------------------------------------------------

String getBAhex() {
  // Rerturns the current Broardcast Address MAC, ie: 50:02:91:68:F7:3F
  String zRet$ = ""; uint8_t zV;
  for (int zP = 0;zP < 6;zP++) {
    zV = broadcastAddress[zP]/16;
    if (zV < 10) {zRet$ += char(zV + 48);} else {zRet$ += char(zV + 55);}
    zV = broadcastAddress[zP] - (zV *16);
    if (zV < 10) {zRet$ += char(zV + 48);} else {zRet$ += char(zV + 55);}
    if (zP < 5) {zRet$ += ":";}
  }
  return zRet$;
}

// --------------------------------------------------------------------------------

int16_t getBatV(int32_t zAvg) {
  // Determine the battery voltage, and percentage, by mapping the ADC readings onto threshold Voltages
  // These threshold voltages were determined using a multimeter, to calibrate the ESP32's ADC
  // if the voltage exceeds BatMax, the estimated voltage is displayed, but battery capacity will be limited to 100%
       if (zAvg > Bat8v2)  {BatV = map(zAvg,Bat7v8,Bat8v4,780,840); BatPc = 100;} // >= 100%
  else if (zAvg > Bat7v8) {BatV = map(zAvg,Bat7v8,Bat8v2,780,820); BatPc = map(zAvg,Bat7v8,Bat8v2,80,100);}  // >= 80%
  else if (zAvg > Bat7v2) {BatV = map(zAvg,Bat7v2,Bat7v8,720,780); BatPc = map(zAvg,Bat7v2,Bat7v8,20, 80);}  // >= 20%
  else {BatV = map(zAvg,Bat6v6,Bat7v2,660,720); BatPc = map(zAvg,Bat6v6,Bat7v2,0,20);}  // < 20%
  if (BatPc < 0) {BatPc = 0;}   // Ignore < 0 percentages
  return BatV;
}

// --------------------------------------------------------------------------------

String GetBIN(byte zVal) {
  // returns an 8 character binary representation of zVal
  String zBIN$ = "";
  // convert zVal into binary
  if (zVal & 0B10000000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B01000000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00100000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00010000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00001000) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000100) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000010) {zBIN$ += "1";} else {zBIN$ += "0";}
  if (zVal & 0B00000001) {zBIN$ += "1";} else {zBIN$ += "0";}
  return zBIN$;
}

// --------------------------------------------------------------------------------

uint32_t getCol32(uint32_t zR,uint32_t zG,uint32_t zB) {
  // Return a 32-bit unsigned colour based on zR,zG,zB
  return ((zR<<16) + (zG<<8) + zB);
}

// --------------------------------------------------------------------------------

String getHmsTime(unsigned long zMillis) {
  // returns a 'time' string in Hrs:Min:sec base on zMillis
  String zTime$ = "";
  long zU = zMillis/3600000; zMillis = zMillis%3600000;
  if (zU > 0) {zTime$ = String(zU) + ":";}
  zU = zMillis/60000; zMillis = zMillis%60000;
  if (zU < 10) {zTime$ += "0";}
  zTime$ += String(zU);
  zU = zMillis/1000;
  if (zU < 10) {zTime$ += ":0";} else {zTime$ += ":";}
  zTime$ += String(zU);
  return zTime$;
}

// --------------------------------------------------------------------------------

String getMinSec(int32_t zTick) {
  // Used by the RECall function, to convert RECtime[RECpnt] into min:sec
  // zTick are in 40ms units
  String zT = "";
  int16_t zMin = zTick/1500;
  if (zMin > 0) {zTick -= (zMin * 1500);}
  int16_t zSec = zTick/25;
  if (zMin < 10) {zT = "0" + String(zMin);} else {zT = String(zMin);}
  zT+= ":";
  if (zSec < 10) {zT+= "0" + String(zSec);} else {zT+= String(zSec);}
  return zT;
}

// --------------------------------------------------------------------------------

int16_t getModPWM(int16_t zPWM) {
  // Returns a modified PWM value, which is offset by MtrMin and scaled by PWM_Max from V_max
  // Note that it is assumed zPWM is in the range of -255 to 0 to +255
  // If not enabled, return without modification
  if (!ModPWM) {return zPWM;}

  // If zPWM is 0 then return 0
  if (zPWM == 0) {return 0;}
  if (zPWM > 0) {
    if (zPWM >= MtrMinStrt) {zPWM = map(zPWM,MtrMinStrt,255,MtrMin,PWM_Max);}
    else {zPWM = map(zPWM,0,MtrMinStrt,0,MtrMin);}
  } else {
    if (zPWM <= -MtrMinStrt) {zPWM = map(zPWM,-MtrMinStrt,-255,-MtrMin,-PWM_Max);}
    else {zPWM = map(zPWM,0,-MtrMinStrt,0,-MtrMin);}
  }
  if (zPWM >  PWM_Max) {zPWM =  PWM_Max;}
  else if (zPWM < -PWM_Max) {zPWM = -PWM_Max;}
  return zPWM;
}

// --------------------------------------------------------------------------------

void getModPWMPrint() {
  // Calls the getModPWM() function for the full range of PWM values -255,0,255
  // and send them to the serial port. This function tasks time!
  // We are sending up to 7 bytes per line, @115200 baud == 0.6ms
  Serial.println("\n\ngetModPWM() table");
  Serial.println("PWM\tModPWM");
  for (int16_t zPWM = -255;zPWM < 256;zPWM++) {
    Serial.println(String(zPWM) + "\t" + String(getModPWM(zPWM)));
    delay(1);         // prevents buffer ov erload
  }
  Serial.println("");
  synchLoopTimers();  // keep timers in synch
}

// --------------------------------------------------------------------------------

int16_t getPWM(int16_t z0,int16_t z1,int16_t zF) {
  // Returns the combined PWM value, as in PWM_A, for PWM_A0, PWM_A1 and PWM_A_
  // Note this value is +/-
  int16_t zV = 0;
  if (zF) {
         if (z0 == 255) {zV = z1 - 255;}
    else if (z1 == 255) {zV = 255 - z0;}
    else if (z0  >  z1) {zV = z1 - z0;}
    else if (z0  <  z1) {zV = z1 - z0;}
  } else {
         if (z0 ==   0) {zV =  z1;}
    else if (z1 ==   0) {zV = -z0;}
    else if (z0  >  z1) {zV = z1 - z0;}
    else if (z0  <  z1) {zV = z1 - z0;}
  }
  return zV;
}

// --------------------------------------------------------------------------------

uint32_t getPwmColr(int16_t zPWM) {
  // This function returns an RGB colour value, based on the absolute zPWM value
  zPWM = abs(zPWM);
  if (zPWM > Mtr_80) {return 0x080000;}
  if (zPWM > Mtr_70) {return 0x060002;}
  if (zPWM > Mtr_60) {return 0x040004;}
  if (zPWM > Mtr_50) {return 0x020006;}
  if (zPWM > Mtr_40) {return 0x000008;}
  if (zPWM > Mtr_30) {return 0x000206;}
  if (zPWM > Mtr_20) {return 0x000404;}
  if (zPWM > Mtr_10) {return 0x000602;}
  return 0x000800;    // defualt
}

// --------------------------------------------------------------------------------

void GetRndCol(byte zB) {
  // return a random colour of strength zB
  if (zB < 2) {zB = 2;}
  switch(random(6)) {
    case 0: ColR =   zB; ColG =    0; ColB =    0; break;  // red
    case 1: ColR = zB/2; ColG = zB/2; ColB =    0; break;  // yellow
    case 2: ColR =    0; ColG =   zB; ColB =    0; break;  // green
    case 3: ColR =    0; ColG = zB/2; ColB = zB/2; break;  // cyan
    case 4: ColR =    0; ColG =    0; ColB =   zB; break;  // blue
    case 5: ColR = zB/2; ColG =    0; ColB = zB/2; break;  // magenta
  }
}

// --------------------------------------------------------------------------------

int16_t getRowAng(float zAng) {
  // Returns a row number, based on the angle.
  // DotStar panel rows are nunmbered 0 - top, 7 - bottom
  int16_t zRow = 7; // default bottom row
       if (zAng >=  10.0) {zRow = 0;}
  else if (zAng >=   6.0) {zRow = 1;}
  else if (zAng >=   3.0) {zRow = 2;}
  else if (zAng >=   1.0) {zRow = 3;}
  else if (zAng >=  -1.0) {zRow = 4;}
  else if (zAng >=  -3.0) {zRow = 5;}
  else if (zAng >=  -6.0) {zRow = 6;}
  else if (zAng >= -10.0) {zRow = 7;}
  return zRow;
}

// --------------------------------------------------------------------------------

void initBatteryRead() {
  // called during runPOST() or on a soft reset
  // take 100 battery readings over a 100ms period to get an average
  BatVol = 0;
  for (int zI = 0;zI < 100;zI++) {BatVol+= analogRead(BatPin); delay(1);}
  BatVol/= 100;

  BatAvg = BatVol; BatSum = BatVol * 50;  // preload the battery sum for averaging
  BatV = getBatV(BatVol); BatVfp = (float)BatV/100.0;
  Serial.println("Battery " + String(BatVol) + "\t" + String(BatVfp) + "v\t" + String(BatPc) + "%");
  randomSeed(BatVol); // use the battery voltage as the random seed factor
  // if battery is less than 5.5v then USB mode is assumed, which makes servos inoperable
  if (BatAvg <= BatUSB) {USB = true; Serial.println(F("USB power assumed."));}
  // Convert BatAvg to a mapped voltage and floating point equivalent
  BatV = getBatV(BatAvg); BatVfp = (float)BatV/10.0;
}

// --------------------------------------------------------------------------------

void Init_ESP_NOW() { 
  // called to initiate an ESEP-NOW link or connect to another trannsceiver
  // each time this function is called it will attempt to connect to a different
  // boradcast address, cycling through them, over and over
  //  Serial.println("Init_ESP_NOW()");

  // if we have previously tried a connection then we need to de-initialise it
  if (ESP_NOW_Init) {
    // already attempted to initialise ESP-NOW so strip everything back
    Serial.println("Closing ESP-NOW");
    esp_now_deinit();
  }
  
  // Init ESP-NOW and returns its status
  if (esp_now_init() != ESP_OK) {
    Serial.println("Error initializing ESP-NOW");
    return;
  }
  ESP_NOW_Init = true;
  
  // change the broadcast address pointer each time
  ESP_NOW_BA++;
  if (ESP_NOW_BA > 3) {
    // Reached the end of the broadcast address list.
    // So extend pause time and reset the cycle.
    ESP_NOW_BA = -1;      // 1st case will be 0
    WiFiConCnt = 75;      // 3 sec delay between trying cycles
    // Turn off serial port reporting after first pass, unless connection fails
    WiFiTryOnce = false;  // don't report trying after one cycle
    ESP_NOW_Init = false;
    return;
  }
  // Serial.println("ESP_NOW_BA = " + String(ESP_NOW_BA));
  
  switch (ESP_NOW_BA) {
    case 0: // Wii Transciever unique MAC: 50:02:91:68:F7:3F
      broadcastAddress[0] = 0x50;
      broadcastAddress[1] = 0x02;
      broadcastAddress[2] = 0x91;
      broadcastAddress[3] = 0x68;
      broadcastAddress[4] = 0xF7;
      broadcastAddress[5] = 0x3F;
      break;
    case 1: // Wii Transciever unique MAC: 50:02:91:68:F0:D2
      broadcastAddress[0] = 0x50;
      broadcastAddress[1] = 0x02;
      broadcastAddress[2] = 0x91;
      broadcastAddress[3] = 0x68;
      broadcastAddress[4] = 0xF0;
      broadcastAddress[5] = 0xD2;
      break;
    case 2: // Wii D1 Mk1 Transciever unique MAC: 58:BF:25:DB:27:A2
      broadcastAddress[0] = 0x58;
      broadcastAddress[1] = 0xBF;
      broadcastAddress[2] = 0x25;
      broadcastAddress[3] = 0xDB;
      broadcastAddress[4] = 0x27;
      broadcastAddress[5] = 0xA2;
      break;
    case 3: // Wii ESP32-C3 Zero Transciever unique MAC: EC:DA:3B:BD:4B:AC
      broadcastAddress[0] = 0xEC;
      broadcastAddress[1] = 0xDA;
      broadcastAddress[2] = 0x3B;
      broadcastAddress[3] = 0xBD;
      broadcastAddress[4] = 0x4B;
      broadcastAddress[5] = 0xAC;
      break;
  }

  // Once ESPNow is successfully Init, we will register for Send CB to
  // get the status of Trasnmitted packet
  esp_now_register_send_cb((esp_now_send_cb_t)OnDataSent);

  // Register peer
  esp_now_peer_info_t peerInfo; //initialize and assign the peer information as a pointer to an addres
  memset(&peerInfo, 0, sizeof(peerInfo));
  memcpy(peerInfo.peer_addr, broadcastAddress, 6); //copy the value of  broadcastAddress with 6 bytes to peerInfo.peer_addr
  peerInfo.channel = 0;     //channel at which the esp talk. 0 means undefined and data will be sent on the current channel. 1-14 are valid channels which is the same with the local device 
  peerInfo.encrypt = false; //not encrypted

  //Add the device to the paired device list 
  if (esp_now_add_peer(&peerInfo) != ESP_OK){ 
    Serial.println("Failed to add peer");
    return;
  }

  // Register for a callback function that will be called when data is received
  esp_now_register_recv_cb(OnDataRecv);
}

// --------------------------------------------------------------------------------

void LED_Balancing() {
  // Live balancing mode
  // Top plate shows balance ring.
  // Wheel LEDs represent motor drive values, with colour and brightness
  if (LEDonce) {
    GFX_AY = 3; StripA.clear(); StripC.clear();
    GFX_BY = 3; StripB.clear(); StripD.clear();
    LEDonce = false; LedA = -2; LedB = -2;
  }
  switch(LED_Task) {
    case 0: // top plate spirit level
      LEDshowSkip = true; LED_SpiritLevel();
      LED_Task++; break;
    case 1: // PitchAC drive 
      if (abs(PWM_AC) < 5) { // no PWM drive
        // draw a horizontal line across the panel
        if (LedA != 0) {
          StripA.clear(); GFX_AY = 3;
          GFX_FillRow(1,GFX_AY,0x000800);
          GFX_CopyPanel(1,3);
          DotColr = 0x000008; LedA = 0;
        }
      } else if (PWM_AC > 0) {
        if (LedA != 1) {
          StripA.clear();LedA = 1;
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(1,0,0,5,0);
          GFX_CopyPanelFlipV(1,3);
        } else {
          GFX_NewCol(1,getPwmColr(PWM_AC));
          GFX_RollDwn(1);
          GFX_CopyPanelFlipV(1,3);
        }
      } else {
        if (LedA != -1) {
          StripA.clear();LedA = -1;
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(1,0,0,4,0);
          GFX_CopyPanelFlipV(1,3);
        } else {
          GFX_NewCol(1,getPwmColr(PWM_AC));
          GFX_RollUp(1);
          GFX_CopyPanelFlipV(1,3);
        }
      }
      LED_Task++; break;
    case 2: // PitchBD drive
      if (abs(PWM_BD) < 5) { // no PWM drive
        // draw a horizontal line across the panel
        if (LedB != 0) {
          StripB.clear();
          GFX_FillRow(2,GFX_AY,0x000800);
          GFX_CopyPanel(2,4);
          DotColr = 0x000008; LedB = 0;
        }
      } else if (PWM_BD > 0) {
        if (LedB != 1) {
          StripB.clear();LedB = 1;
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(2,0,0,4,0);
          GFX_CopyPanelFlipV(2,4);
        } else {
          GFX_NewCol(2,getPwmColr(PWM_BD));
          GFX_RollUp(2);
          GFX_CopyPanelFlipV(2,4);
        }
      } else {
        if (LedB != -1) {
          StripB.clear();LedB = -1;
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(2,0,0,5,0);
          GFX_CopyPanelFlipV(2,4);
        } else {
          GFX_NewCol(2,getPwmColr(PWM_BD));
          GFX_RollDwn(2);
          GFX_CopyPanelFlipV(2,4);
        }
      }
      LED_Del = 5; LEDshow = true;
      LED_Task = 0; break;
  }
}

// --------------------------------------------------------------------------------

void LED_BalDrv(float zOp,int16_t zL) {
  // Determine the colour and brightness of LED zL for output zOp
  // Colour will go from green to red, brightness limited to 64
  // ColR = 0; ColG = 0; ColB = 0;
  // if (zOp > 64) {zOp = 64;}                       // limit zOp
  //      if (zOp <=  8.0) {ColR =  2; ColG = 2;}    // yellow
  // else if (zOp <= 16.0) {ColG =  6;}              // green
  // else if (zOp <= 24.0) {ColG =  4; ColB = 4;}    // turquise
  // else if (zOp <= 32.0) {ColB = 10;}              // blue
  // else if (zOp <= 40.0) {ColR =  6; ColB = 6;}    // purple
  // else if (zOp <= 48.0) {ColR = 16;}              // red
  //    else {ColR = 32;}                            // bright red    
  // LED[zL].setRGB(ColR,ColG,ColB);
}

// --------------------------------------------------------------------------------

void LED_Blue_Rot() {
  // Fill and rotate a Blue band, when WiFi connection is present
  if (!WiFiConnected) {SetLEDmode(1); return;}  // Switch to blue when WiFi is connected

  switch (LED_Task) {
    case 0: // set initial state
      Strip0.clear();
      Strip0.setPixelColor( 0,0,0,1); Strip0.setPixelColor( 1,0,0,2); Strip0.setPixelColor( 2,0,0,4);
      Strip0.setPixelColor( 4,0,0,1); Strip0.setPixelColor( 5,0,0,2); Strip0.setPixelColor( 6,0,0,4);
      Strip0.setPixelColor( 8,0,0,1); Strip0.setPixelColor( 9,0,0,2); Strip0.setPixelColor(10,0,0,4);
      Strip0.setPixelColor(12,0,0,1); Strip0.setPixelColor(13,0,0,2); Strip0.setPixelColor(14,0,0,4);
      LED_Task++; break;
    case 1: // rotate LEDs
      LED_Rot_Lft();                  // rotate the LEDs on the micro plate
      StripA.clear();                 // clear panel 'A' 
      GFX_FillRow(1,LedCnt,0x000002); // draw a horizontal blue line based on LedCnt
      GFX_CopyPanel(1,2); GFX_CopyPanel(1,3); GFX_CopyPanel(1,4);
      if (LedCnt == 7) {LedInc = -1;}
      else if (LedCnt == 0) {LedInc = 1;}
      LedCnt += LedInc;
      break;
  }
  LED_Del = 5; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Drive_Bwd() {
  // Drive backwards sequence.
  // Note L0 is on the left-hand side, L4 is at the front, L8 is on the right, and L12 is at the rear
  switch (LED_Task) {
    case 0: // set initial state
      Strip0.clear();
      Strip0.setPixelColor(12,0,0,1); Strip0.setPixelColor(11,0,0,2); Strip0.setPixelColor(10,2,0,4); Strip0.setPixelColor( 9,4,0,4);
      Strip0.setPixelColor( 8,8,0,2); Strip0.setPixelColor( 7,8,2,0); Strip0.setPixelColor( 6,8,8,0); Strip0.setPixelColor( 5,0,0,0);

      Strip0.setPixelColor(13,0,0,2); Strip0.setPixelColor(14,0,0,2); Strip0.setPixelColor(15,4,0,4); Strip0.setPixelColor( 0,8,0,2);
      Strip0.setPixelColor( 1,8,2,0); Strip0.setPixelColor( 2,8,8,0); Strip0.setPixelColor( 3,0,0,0); Strip0.setPixelColor( 4,0,0,0);
      LED_Task++; break;
    case 1: // Rotate the colours towards the rear
      DotColr = Strip0.getPixelColor(4);
      Strip0.setPixelColor( 4,Strip0.getPixelColor( 5));
      Strip0.setPixelColor( 5,Strip0.getPixelColor( 6));
      Strip0.setPixelColor( 6,Strip0.getPixelColor( 7));
      Strip0.setPixelColor( 7,Strip0.getPixelColor( 8));
      Strip0.setPixelColor( 8,Strip0.getPixelColor( 9));
      Strip0.setPixelColor( 9,Strip0.getPixelColor(10));
      Strip0.setPixelColor(10,Strip0.getPixelColor(11));
      Strip0.setPixelColor(11,Strip0.getPixelColor(12));
      Strip0.setPixelColor( 3,Strip0.getPixelColor( 5));
      Strip0.setPixelColor( 2,Strip0.getPixelColor( 6));
      Strip0.setPixelColor( 1,Strip0.getPixelColor( 7));
      Strip0.setPixelColor( 0,Strip0.getPixelColor( 8));
      Strip0.setPixelColor(15,Strip0.getPixelColor( 9));
      Strip0.setPixelColor(14,Strip0.getPixelColor(10));
      Strip0.setPixelColor(13,Strip0.getPixelColor(11));
      Strip0.setPixelColor(12,DotColr);
  }

  // Clear all panels at start
  if (LEDonce) {LEDonce = false; StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();}

  // Fill rows based on top plate LED pattern
  if (PWM_AC != 0) {
    // PWM applied so set all rows
    GFX_FillRow(1,7,Strip0.getPixelColor(11));
    GFX_FillRow(1,6,Strip0.getPixelColor(10));
    GFX_FillRow(1,5,Strip0.getPixelColor( 9));
    GFX_FillRow(1,4,Strip0.getPixelColor( 8));
    GFX_FillRow(1,3,Strip0.getPixelColor( 7));
    GFX_FillRow(1,2,Strip0.getPixelColor( 6));
    GFX_FillRow(1,1,Strip0.getPixelColor( 5));
    GFX_FillRow(1,0,Strip0.getPixelColor( 4));
  } else {StripA.clear(); GFX_FillRow(1,4,0x000400);}

  // Reverse the order for the rear panel
  if (PWM_BD != 0) {
    // PWM applied so set all rows
    GFX_FillRow(2,0,Strip0.getPixelColor(11));
    GFX_FillRow(2,1,Strip0.getPixelColor(10));
    GFX_FillRow(2,2,Strip0.getPixelColor( 9));
    GFX_FillRow(2,3,Strip0.getPixelColor( 8));
    GFX_FillRow(2,4,Strip0.getPixelColor( 7));
    GFX_FillRow(2,5,Strip0.getPixelColor( 6));
    GFX_FillRow(2,6,Strip0.getPixelColor( 5));
    GFX_FillRow(2,7,Strip0.getPixelColor( 4));
  } else {StripB.clear(); GFX_FillRow(2,4,0x000400);}

  // Copy pattern onto all 3 panels
  GFX_CopyPanelFlipV(1,3); GFX_CopyPanelFlipV(2,4);

  LED_Del = map(LedSpd,PWM_StartMin,255,10,1); 
  LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Drive_Fwd() {
  // Drive forward sequence.
  // Note L0 is on the left-hand side, L4 is at the front, L8 is on the right, and L12 is at the rear
  switch (LED_Task) {
    case 0: // set initial state, LED 4 is the front LED, and LED 12 is the rear LED.
      Strip0.clear();
      Strip0.setPixelColor( 4,0,0,1); Strip0.setPixelColor( 5,0,0,2); Strip0.setPixelColor( 6,2,0,4); Strip0.setPixelColor( 7,4,0,4);
      Strip0.setPixelColor( 8,8,0,2); Strip0.setPixelColor( 9,8,2,0); Strip0.setPixelColor(10,8,8,0); Strip0.setPixelColor(11,0,0,0);

      Strip0.setPixelColor(12,0,0,0); Strip0.setPixelColor(13,0,0,0); Strip0.setPixelColor(14,8,8,0); Strip0.setPixelColor(15,8,2,0);
      Strip0.setPixelColor( 0,8,0,2); Strip0.setPixelColor( 1,4,0,4); Strip0.setPixelColor( 2,2,0,4); Strip0.setPixelColor( 3,0,0,2);
      LED_Task++; break;
    case 1: // Rotate the colours towards the rear
      DotColr = Strip0.getPixelColor(12);
      Strip0.setPixelColor(12,Strip0.getPixelColor(11));
      Strip0.setPixelColor(11,Strip0.getPixelColor(10));
      Strip0.setPixelColor(10,Strip0.getPixelColor( 9));
      Strip0.setPixelColor( 9,Strip0.getPixelColor( 8));
      Strip0.setPixelColor( 8,Strip0.getPixelColor( 7));
      Strip0.setPixelColor( 7,Strip0.getPixelColor( 6));
      Strip0.setPixelColor( 6,Strip0.getPixelColor( 5));
      Strip0.setPixelColor( 5,Strip0.getPixelColor( 4));
      Strip0.setPixelColor( 3,Strip0.getPixelColor( 5));
      Strip0.setPixelColor( 2,Strip0.getPixelColor( 6));
      Strip0.setPixelColor( 1,Strip0.getPixelColor( 7));
      Strip0.setPixelColor( 0,Strip0.getPixelColor( 8));
      Strip0.setPixelColor(15,Strip0.getPixelColor( 9));
      Strip0.setPixelColor(14,Strip0.getPixelColor(10));
      Strip0.setPixelColor(13,Strip0.getPixelColor(11));
      Strip0.setPixelColor( 4,DotColr);
  }

  // Clear all panels at start
  if (LEDonce) {LEDonce = false; StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();}

  // Fill rows based on top plate LED pattern
  if (PWM_AC != 0) {
    // PWM applied so set all rows
    GFX_FillRow(1,7,Strip0.getPixelColor(12));
    GFX_FillRow(1,6,Strip0.getPixelColor(11));
    GFX_FillRow(1,5,Strip0.getPixelColor(10));
    GFX_FillRow(1,4,Strip0.getPixelColor( 9));
    GFX_FillRow(1,3,Strip0.getPixelColor( 8));
    GFX_FillRow(1,2,Strip0.getPixelColor( 7));
    GFX_FillRow(1,1,Strip0.getPixelColor( 6));
    GFX_FillRow(1,0,Strip0.getPixelColor( 5));
  } else {StripA.clear(); GFX_FillRow(1,4,0x000400);}

  // Fill rows based on top plate LED pattern
  if (PWM_BD != 0) {
    // PWM applied so set all rows
    GFX_FillRow(4,7,Strip0.getPixelColor(12));
    GFX_FillRow(4,6,Strip0.getPixelColor(11));
    GFX_FillRow(4,5,Strip0.getPixelColor(10));
    GFX_FillRow(4,4,Strip0.getPixelColor( 9));
    GFX_FillRow(4,3,Strip0.getPixelColor( 8));
    GFX_FillRow(4,2,Strip0.getPixelColor( 7));
    GFX_FillRow(4,1,Strip0.getPixelColor( 6));
    GFX_FillRow(4,0,Strip0.getPixelColor( 5));
  } else {StripD.clear();GFX_FillRow(4,4,0x000400);}

  // Copy pattern onto all 4 panels
  GFX_CopyPanelFlipV(1,3); GFX_CopyPanelFlipV(4,2);

  LED_Del = map(LedSpd,PWM_StartMin,255,10,1); 
  LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Drive_Lft() {
  // Drive left sequence.
  // Note L0 is on the left-hand side, L0 is at the front, L4 is on the right, and L8 is at the rear
  switch (LED_Task) {
    case 0: // set initial state, LED 0 is the front LED, and LED 8 is the rear LED.
      Strip0.clear();
      Strip0.setPixelColor( 0,0,0,1); Strip0.setPixelColor( 1,0,0,2); Strip0.setPixelColor( 2,2,0,4); Strip0.setPixelColor( 3,4,0,4);
      Strip0.setPixelColor( 4,8,0,2); Strip0.setPixelColor( 5,8,2,0); Strip0.setPixelColor( 6,8,8,0); Strip0.setPixelColor( 7,0,0,0);

      Strip0.setPixelColor( 8,0,0,0); Strip0.setPixelColor( 9,0,0,0); Strip0.setPixelColor(10,8,8,0); Strip0.setPixelColor(11,8,2,0);
      Strip0.setPixelColor(12,8,0,2); Strip0.setPixelColor(13,4,0,4); Strip0.setPixelColor(14,2,0,4); Strip0.setPixelColor(15,0,0,2);
      LED_Task++; break;
    case 1: // Rotate the colours towards the right
      DotColr = Strip0.getPixelColor(8);
      Strip0.setPixelColor( 8,Strip0.getPixelColor( 7));
      Strip0.setPixelColor( 7,Strip0.getPixelColor( 6));
      Strip0.setPixelColor( 6,Strip0.getPixelColor( 5));
      Strip0.setPixelColor( 5,Strip0.getPixelColor( 4));
      Strip0.setPixelColor( 4,Strip0.getPixelColor( 3));
      Strip0.setPixelColor( 3,Strip0.getPixelColor( 2));
      Strip0.setPixelColor( 2,Strip0.getPixelColor( 1));
      Strip0.setPixelColor( 1,Strip0.getPixelColor( 0));
      Strip0.setPixelColor(15,Strip0.getPixelColor( 1));
      Strip0.setPixelColor(14,Strip0.getPixelColor( 2));
      Strip0.setPixelColor(13,Strip0.getPixelColor( 3));
      Strip0.setPixelColor(12,Strip0.getPixelColor( 4));
      Strip0.setPixelColor(11,Strip0.getPixelColor( 5));
      Strip0.setPixelColor(10,Strip0.getPixelColor( 6));
      Strip0.setPixelColor( 9,Strip0.getPixelColor( 7));
      Strip0.setPixelColor( 0,DotColr);
  }

  // Clear all panels at start
  if (LEDonce) {LEDonce = false; StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();}

  // Fill rows based on top plate LED pattern
  if (PWM_BD != 0) {
    // PWM applied so set all rows
    GFX_FillRow(4,7,Strip0.getPixelColor( 7));
    GFX_FillRow(4,6,Strip0.getPixelColor( 6));
    GFX_FillRow(4,5,Strip0.getPixelColor( 5));
    GFX_FillRow(4,4,Strip0.getPixelColor( 4));
    GFX_FillRow(4,3,Strip0.getPixelColor( 3));
    GFX_FillRow(4,2,Strip0.getPixelColor( 2));
    GFX_FillRow(4,1,Strip0.getPixelColor( 1));
    GFX_FillRow(4,0,Strip0.getPixelColor( 0));
  } else {StripD.clear(); GFX_FillRow(4,4,0x000400);}

  // Reverse the order for the rear panel
  if (PWM_AC != 0) {
    // PWM applied so set all rows
    GFX_FillRow(1,7,Strip0.getPixelColor( 0));
    GFX_FillRow(1,6,Strip0.getPixelColor( 1));
    GFX_FillRow(1,5,Strip0.getPixelColor( 2));
    GFX_FillRow(1,4,Strip0.getPixelColor( 3));
    GFX_FillRow(1,3,Strip0.getPixelColor( 4));
    GFX_FillRow(1,2,Strip0.getPixelColor( 5));
    GFX_FillRow(1,1,Strip0.getPixelColor( 6));
    GFX_FillRow(1,0,Strip0.getPixelColor( 7));
  } else {StripA.clear(); GFX_FillRow(1,4,0x000400);}

  // Copy pattern onto all 4 panels
  GFX_CopyPanelFlipV(1,3); GFX_CopyPanelFlipV(4,2);

  LED_Del = map(LedSpd,PWM_StartMin,255,10,1); 
  LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Drive_Rht() {
  // Drive left sequence.
  // Note L0 is on the left-hand side, L4 is at the front, L8 is on the right, and L12 is at the rear
  switch (LED_Task) {
    case 0: // set initial state, LED 8 is the front LED, and LED 0 is the rear LED.
      Strip0.clear();
      Strip0.setPixelColor( 0,0,0,1); Strip0.setPixelColor( 1,0,0,2); Strip0.setPixelColor( 2,2,0,4); Strip0.setPixelColor( 3,4,0,4);
      Strip0.setPixelColor( 4,8,0,2); Strip0.setPixelColor( 5,8,2,0); Strip0.setPixelColor( 6,8,8,0); Strip0.setPixelColor( 7,0,0,0);

      Strip0.setPixelColor( 8,0,0,0); Strip0.setPixelColor( 9,0,0,0); Strip0.setPixelColor(10,8,8,0); Strip0.setPixelColor(11,8,2,0);
      Strip0.setPixelColor(12,8,0,2); Strip0.setPixelColor(13,4,0,4); Strip0.setPixelColor(14,2,0,4); Strip0.setPixelColor(15,0,0,2);
      LED_Task++; break;
    case 1: // Rotate the colours towards the right
      DotColr = Strip0.getPixelColor(0);
      Strip0.setPixelColor( 0,Strip0.getPixelColor(15));
      Strip0.setPixelColor(15,Strip0.getPixelColor(14));
      Strip0.setPixelColor(14,Strip0.getPixelColor(13));
      Strip0.setPixelColor(13,Strip0.getPixelColor(12));
      Strip0.setPixelColor(12,Strip0.getPixelColor(11));
      Strip0.setPixelColor(11,Strip0.getPixelColor(10));
      Strip0.setPixelColor(10,Strip0.getPixelColor( 9));
      Strip0.setPixelColor( 9,Strip0.getPixelColor( 8));
      Strip0.setPixelColor( 7,Strip0.getPixelColor( 9));
      Strip0.setPixelColor( 6,Strip0.getPixelColor(10));
      Strip0.setPixelColor( 5,Strip0.getPixelColor(11));
      Strip0.setPixelColor( 4,Strip0.getPixelColor(12));
      Strip0.setPixelColor( 3,Strip0.getPixelColor(13));
      Strip0.setPixelColor( 2,Strip0.getPixelColor(14));
      Strip0.setPixelColor( 1,Strip0.getPixelColor(15));
      Strip0.setPixelColor( 8,DotColr);
  }

  // Clear all panels at start
  if (LEDonce) {LEDonce = false; StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();}

  // Fill rows based on top plate LED pattern
  if (PWM_BD != 0) {
    // PWM applied so set all rows
    GFX_FillRow(2,0,Strip0.getPixelColor( 8));
    GFX_FillRow(2,1,Strip0.getPixelColor( 7));
    GFX_FillRow(2,2,Strip0.getPixelColor( 6));
    GFX_FillRow(2,3,Strip0.getPixelColor( 5));
    GFX_FillRow(2,4,Strip0.getPixelColor( 4));
    GFX_FillRow(2,5,Strip0.getPixelColor( 3));
    GFX_FillRow(2,6,Strip0.getPixelColor( 2));
    GFX_FillRow(2,7,Strip0.getPixelColor( 1));
  } else {StripB.clear(); GFX_FillRow(2,4,0x000400);}

  // Reverse the order for the rear panel
  if (PWM_AC != 0) {
    // PWM applied so set all rows
    GFX_FillRow(3,0,Strip0.getPixelColor( 0));
    GFX_FillRow(3,1,Strip0.getPixelColor( 1));
    GFX_FillRow(3,2,Strip0.getPixelColor( 2));
    GFX_FillRow(3,3,Strip0.getPixelColor( 3));
    GFX_FillRow(3,4,Strip0.getPixelColor( 4));
    GFX_FillRow(3,5,Strip0.getPixelColor( 5));
    GFX_FillRow(3,6,Strip0.getPixelColor( 6));
    GFX_FillRow(3,7,Strip0.getPixelColor( 7));
  } else {StripC.clear(); GFX_FillRow(3,4,0x000400);}

  // Copy pattern onto all 3 panels
  GFX_CopyPanelFlipV(2,4); GFX_CopyPanelFlipV(3,1);

  LED_Del = map(LedSpd,PWM_StartMin,255,10,1); 
  LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Fill(uint8_t zR,uint8_t zG,uint8_t zB) {
  // Fill the LED array with a specific colour
  for(int i = 0; i < 16; i++) {Strip0.setPixelColor(i, Strip0.Color(zR, zG, zB));}
}

// --------------------------------------------------------------------------------

void LED_Green_Rot() {
  // Fill and rotate a Green band, when no WiFi connection is present
  if (WiFiConnected) {SetLEDmode(2); return;}  // Switch to blue when WiFi is connected

  switch (LED_Task) {
    case 0: // set initial state
      Strip0.clear();
      Strip0.setPixelColor( 0,0,4,0); Strip0.setPixelColor( 1,0, 2,0); Strip0.setPixelColor( 2,0, 1,0);
      Strip0.setPixelColor( 4,0,4,0); Strip0.setPixelColor( 5,0, 2,0); Strip0.setPixelColor( 6,0, 1,0);
      Strip0.setPixelColor( 8,0,4,0); Strip0.setPixelColor( 9,0, 2,0); Strip0.setPixelColor(10,0, 1,0);
      Strip0.setPixelColor(12,0,4,0); Strip0.setPixelColor(13,0, 2,0); Strip0.setPixelColor(14,0, 1,0);
      LED_Task++; break;
    case 1: // rotate LEDs
      LED_Rot_Rht();                  // rotate the LEDs on the micro plate
      StripA.clear();                 // clear panel 'A' 
      GFX_FillRow(1,LedCnt,0x000200); // draw a horizontal green line based on LedCnt
      GFX_CopyPanel(1,2); GFX_CopyPanel(1,3); GFX_CopyPanel(1,4);
      if (LedCnt == 7) {LedInc = -1;}
      else if (LedCnt == 0) {LedInc = 1;}
      LedCnt += LedInc;
      break;
  }
  LED_Del = 8; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Main() {
  // Tasks for manipulating RGB LED patterns
  // Normally called at 20ms intervals, but variable if moving
  // Serial.print(String(LedMode) + " ");
  if (LedNop) {return;}
  
  // Blink background clock
  Blink = false; BlinkCnt++; if (BlinkCnt > 30) {BlinkCnt = 0; Blink = true;}
  // Check button switch over-rides
  if (sw0State == LOW) {
    // SW0 is LOW
    if (!LED_SW) {
      Strip0.clear();  // turn OFF all LEDs
      Strip0.setPixelColor(0,0,0,24); Strip0.setPixelColor(1,0,0,24); Strip0.setPixelColor(15,0,0,24); LEDshow = true;
    } LED_SW = true; return;
  }
  else if (sw1State == LOW) {
    // SW1 is LOW
    if (!LED_SW) {
      Strip0.clear();  // turn OFF all LEDs
      Strip0.setPixelColor(7,0,0,24); Strip0.setPixelColor(8,0,0,24); Strip0.setPixelColor(9,0,0,24); LEDshow = true;
    } LED_SW = true; return;
  }
  if (LED_Del > 0) {LED_Del--; return;}                 // delay by skipping tasks
  if (LED_SW) {LED_SW = false; SetLEDmode(-LedMode);}   // return to previous LED mode

  // Serial.println(LedMode);
  switch(LedMode) {
    case  0: break;                       // do nothing
    case  1: LED_Green_Rot(); break;      // rotating green band
    case  2: LED_Blue_Rot(); break;       // rotating blue band
    case  3: LED_Purple_Rot(); break;     // rotating purple band
    case  5: LED_RandomColours(); break;  // random colours for WiFi connect/disconnect
    case 10: LED_Yellow_Ring(); break;    // pulsating yellow ring
    case 12: LED_SpiritLevel(); break;    // top plate spirit level, to initiate balancing
    case 13: LED_SideLevels(); break;     // side plate spirit levels, triggered from LedMode == 12
    case 20: LED_Red_Pulse(); break;      // drive mode stationary
    case 21: LED_Drive_Fwd(); break;      // drive forwards
    case 22: LED_Drive_Bwd(); break;      // drive backwards
    case 23: LED_Drive_Rht(); break;      // drive right
    case 24: LED_Drive_Lft(); break;      // drive left
    case 30: LED_Balancing(); break;      // LIVE balancing
    case 40: LED_WheelTest(); break;      // wheel friction testing
  }
}

// --------------------------------------------------------------------------------

void LED_Purple_Rot() {
  // Rotating Purple band
  switch (LED_Task) {
    case 0: // set initial state
      Strip0.clear();
      Strip0.setPixelColor( 0,4,0,4); Strip0.setPixelColor( 1,2,0,2); Strip0.setPixelColor( 2,1,0,1);
      Strip0.setPixelColor( 4,4,0,4); Strip0.setPixelColor( 5,2,0,2); Strip0.setPixelColor( 6,1,0,1);
      Strip0.setPixelColor( 8,4,0,4); Strip0.setPixelColor( 9,2,0,2); Strip0.setPixelColor(10,1,0,1);
      Strip0.setPixelColor(12,0,0,4); Strip0.setPixelColor(13,2,0,2); Strip0.setPixelColor(14,1,0,1);
      LED_Task++; break;
    case 1: // rotate LEDs
      LED_Rot_Rht();                    // rotate the LEDs on the micro plate
      StripA.clear();                   // clear panel 'A' 
      GFX_FillColm(1,LedCnt,0x010001);  // draw a vertical Purple line based on LedCnt
      GFX_CopyPanel(1,2); GFX_CopyPanel(1,3); GFX_CopyPanel(1,4);
      if (LedCnt == 7) {LedInc = -1;}
      else if (LedCnt == 0) {LedInc = 1;}
      LedCnt += LedInc;
      break;
  }
  LED_Del = 5; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_RandomColours() {
  // Random colours for WiFi connect/disconnect
  GetRndCol(16); Strip0.setPixelColor(random(15),ColR,ColG,ColB);
  GetRndCol(32); Strip0.setPixelColor(random(15),ColR,ColG,ColB);
  GetRndCol(64); Strip0.setPixelColor(random(15),ColR,ColG,ColB);
  GetRndCol(16); Strip0.setPixelColor(random(15),ColR,ColG,ColB);
  GetRndCol(32); Strip0.setPixelColor(random(15),ColR,ColG,ColB);
  GetRndCol(64); Strip0.setPixelColor(random(15),ColR,ColG,ColB);
  // First time round clear the DotStar panels
  if (LEDonce) {
    // Clear the DotStar panels
    StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();
    LEDonce = false;
  }

  // Draw a line depending on WiFiCntC
  if (WiFiCntC >   0) {GFX_FillRow(1,7,0x000001);} else {GFX_FillRow(1,7,0x000000);}
  if (WiFiCntC >=  8) {GFX_FillRow(1,6,0x000002);} else {GFX_FillRow(1,6,0x000000);}
  if (WiFiCntC >= 16) {GFX_FillRow(1,5,0x000003);} else {GFX_FillRow(1,5,0x000000);}
  if (WiFiCntC >= 24) {GFX_FillRow(1,4,0x000004);} else {GFX_FillRow(1,4,0x000000);}
  if (WiFiCntC >= 32) {GFX_FillRow(1,3,0x000005);} else {GFX_FillRow(1,3,0x000000);}
  if (WiFiCntC >= 40) {GFX_FillRow(1,2,0x000006);} else {GFX_FillRow(1,2,0x000000);}
  if (WiFiCntC >= 48) {GFX_FillRow(1,1,0x000007);} else {GFX_FillRow(1,1,0x000000);}
  if (WiFiCntC >= 56) {GFX_FillRow(1,0,0x000008);} else {GFX_FillRow(1,0,0x000000);}

  // Copy pattern onto all 3 panels
  GFX_CopyPanel(1,2); GFX_CopyPanel(1,3); GFX_CopyPanel(1,4);
  LED_Del = 0; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Red_Pulse() {
  // Pulsing red ring and panels
  switch (LedCnt) {
    case 0: Strip0.fill(0x000000); break;
    case 1: Strip0.fill(0x010000); break;
    case 2: Strip0.fill(0x020000); break;
    case 3: Strip0.fill(0x040000); break;
    case 4: Strip0.fill(0x080000); break;
  } LedCnt++; if (LedCnt > 4) {LedCnt = 0;}
  // This first part can be called as part of another function, using the skip technique
  if (LEDshowSkip) {LEDshowSkip = false; return;}

  // Clear all panels at start
  if (LEDonce) {LEDonce = false; StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();}

  // Now draw box on panel 'A'
  GFX_GetColor(1,1,1); GFX_Box(1,0,0,8,8);
  GFX_GetColor(1,2,2); GFX_Box(1,1,1,6,6);
  GFX_GetColor(1,3,3); GFX_Box(1,2,2,4,4);
  GFX_GetColor(0,0,0); GFX_Box(1,3,3,2,2);  // draw centre box, colour from top ring

  // Copy pattern onto all 3 panels
  GFX_CopyPanel(1,2); GFX_CopyPanel(1,3); GFX_CopyPanel(1,4);

  LED_Del = 5; LEDshow = true;
}

// --------------------------------------------------------------------------------

void LED_Rot_Lft() {
  // Rotate all of the top LEDs by one to the left, anti-clockwise when viewed from above
  Strip0.setPixelColor(16,Strip0.getPixelColor(15));
  for (int zL = 15; zL > 0;zL--) {Strip0.setPixelColor(zL,Strip0.getPixelColor(zL-1));}
  Strip0.setPixelColor(0,Strip0.getPixelColor(16));
}

// --------------------------------------------------------------------------------

void LED_Rot_Rht() {
  // Rotate all of the top LEDs by one to the right, clockwise when viewed from above
  Strip0.setPixelColor(16,Strip0.getPixelColor(0));
  for (int zL = 0; zL < 15;zL++) {Strip0.setPixelColor(zL,Strip0.getPixelColor(zL+1));}
  Strip0.setPixelColor(15,Strip0.getPixelColor(16));
}

// --------------------------------------------------------------------------------

void LED_SetNum(int16_t zP) {
  // Called to set a specific top LED to preset colours
  // If the pointer zP exceeds array limits it is corrected
       if (zP <  0) {zP += 16;}
  else if (zP > 15) {zP -= 16;}
  Strip0.setPixelColor(zP,ColR,ColG,ColB);
}

// --------------------------------------------------------------------------------

void LED_SetNumRGB(int16_t zP,uint8_t zR,uint8_t zG,uint8_t zB) {
  // Called to set a specific top LED to preset colours
  // If the pointer zP exceeds array limits it is corrected
       if (zP <  0) {zP += 16;}
  else if (zP > 15) {zP -= 16;}
  Strip0.setPixelColor(zP,zR,zG,zB);
}

// --------------------------------------------------------------------------------

void LED_SideLevels() {
  // Draws lines on side plates to mimic spirit level angles
  // Start by clearling panels
  StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();
  DotColr = getCol32(8,8,0);
  // Look at tilt angles
  if (ACErrAbs > BDErrAbs) {
    // AC axis has the largest angle
    int16_t zRowA = getRowAng( ACErr); GFX_FillRow(1,zRowA,DotColr);
    int16_t zRowC = getRowAng(-ACErr); GFX_FillRow(3,zRowC,DotColr);
    GFX_Line(2, 0,zRowC, 7,zRowA);
    GFX_Line(4, 0,zRowA, 7,zRowC);
  } else {
    // BD axis has the largest angle
    int16_t zRowB = getRowAng(-BDErr); GFX_FillRow(2,zRowB,DotColr);
    int16_t zRowD = getRowAng( BDErr); GFX_FillRow(4,zRowD,DotColr);
    GFX_Line(1, 0,zRowB, 7,zRowD);
    GFX_Line(3, 0,zRowD, 7,zRowB);
  }
  // Go back to top ring swpirit level mode
  if (SafeMode < 4) {
    LedMode = 12;
    // if the LEDshowSkip flag is true we don't trigger a show event
    if (!LEDshowSkip) {LED_Del = 3; LEDshow = true;} else {LEDshowSkip = false;}
  } else {
    // if the LEDshowSkip flag is true we don't trigger a show event
    SetLEDmode(30);
  }
}

// --------------------------------------------------------------------------------

void LED_SpiritLevel() {
  // Spirit level ring, top plate
  // This runs first, before side plate task
  LedAng = 3.0;     // Pitch max angle error
  LedPnt = 0;       // target LED, 0 - 15
  LedNum = 0;       // pointer to the number of LEDs to light, 0 - 8
  // Find the LED which is the highest
  if (AcY_OA == 0) {AcY_OA = 1;}  // avoid divide by zero
  ACErr = PitchAC - PtchACSbSpSa; ACErrAbs = fabs(ACErr);
  BDErr = PitchBD - PtchBDSbSpSa; BDErrAbs = fabs(BDErr);
  // determine which is the highest point/lartgest angle
  if (ACErrAbs > BDErrAbs) {
    // The AC pitch axis it tilting at a larger angle
    // Use AC as the lead angle
    AccAbsAng = ACErrAbs/BDErrAbs;
    LedAng = ACErrAbs;
    // determine which side of the ring is the highest
    if (PitchAC > PtchACSbSpSa) {
      // the 'A' motor is highest
      LedPnt = 6;
      // look for BD axis being level
      if (PitchBD > PtchBDSbSpSa) {
        // motor 'B' is higher than motor 'D'
             if (AccAbsAng > 5.0273) {LedPnt = 6;}
        else if (AccAbsAng > 1.4966) {LedPnt = 5;}
        else if (AccAbsAng > 0.6682) {LedPnt = 4;}
      } else {
        // motor 'D' is higher than motor 'B'
             if (AccAbsAng > 5.0273) {LedPnt = 6;}
        else if (AccAbsAng > 1.4966) {LedPnt = 7;}
        else if (AccAbsAng > 0.6682) {LedPnt = 8;}
      }
    } else {
      // the 'C' motor is highest
      LedPnt = 14;
      // look for BD axis being level
      if (PitchBD > PtchBDSbSpSa) {
        // motor 'B' is higher than motor 'D'
             if (AccAbsAng > 5.0273) {LedPnt = 14;}
        else if (AccAbsAng > 1.4966) {LedPnt = 15;}
        else if (AccAbsAng > 0.6682) {LedPnt = 0;}
      } else {
        // motor 'D' is higher than motor 'B'
             if (AccAbsAng > 5.0273) {LedPnt = 14;}
        else if (AccAbsAng > 1.4966) {LedPnt = 13;}
        else if (AccAbsAng > 0.6682) {LedPnt = 12;}
      }
    }
  } else {
    // The BD pitch axis it tilting at a larger angle
    // Use BD as the lead angle
    AccAbsAng = BDErrAbs/ACErrAbs;
    LedAng = BDErrAbs;
    // determine which side of the ring is the highest
    if (PitchBD > PtchBDSbSpSa) {
      // the 'B' motor is highest
      LedPnt = 2;
      // look for BD axis being level
      if (PitchAC > PtchACSbSpSa) {
        // motor 'B' is higher than motor 'D'
             if (AccAbsAng > 5.0273) {LedPnt = 2;}
        else if (AccAbsAng > 1.4966) {LedPnt = 3;}
        else if (AccAbsAng > 0.6682) {LedPnt = 4;}
      } else {
        // motor 'D' is higher than motor 'B'
             if (AccAbsAng > 5.0273) {LedPnt = 2;}
        else if (AccAbsAng > 1.4966) {LedPnt = 1;}
        else if (AccAbsAng > 0.6682) {LedPnt = 0;}
      }
    } else {
      // the 'D' motor is highest
      LedPnt = 10;
      // look for BD axis being level
      if (PitchAC > PtchACSbSpSa) {
        // motor 'B' is higher than motor 'D'
             if (AccAbsAng > 5.0273) {LedPnt = 10;}
        else if (AccAbsAng > 1.4966) {LedPnt = 9;}
        else if (AccAbsAng > 0.6682) {LedPnt = 8;}
      } else {
        // motor 'D' is higher than motor 'B'
             if (AccAbsAng > 5.0273) {LedPnt = 10;}
        else if (AccAbsAng > 1.4966) {LedPnt = 11;}
        else if (AccAbsAng > 0.6682) {LedPnt = 12;}
      }
    }
  }

  // Use angle zV and LED pointer LedPnt to determine number of LEDs to light
  // The larger the angle the fewer the LEDs, and colour turns from green towards red
  float zLedAng = LedAng;
       if (zLedAng <= 0.6) {LedNum = 8; ColR =  0; ColG =  8; ColB =  0;}   // 16 green LEDs
  else if (zLedAng <= 0.9) {LedNum = 7; ColR =  0; ColG =  6; ColB =  2;}   // 15 green LEDs
  else if (zLedAng <= 1.2) {LedNum = 6; ColR =  0; ColG =  6; ColB =  6;}   // 13 tourquis LEDs
  else if (zLedAng <= 1.5) {LedNum = 5; ColR =  0; ColG =  3; ColB =  9;}   // 11 blue LEDs
  else if (zLedAng <= 1.8) {LedNum = 4; ColR =  0; ColG =  0; ColB = 12;}   //  9 blue LEDs
  else if (zLedAng <= 2.1) {LedNum = 3; ColR =  4; ColG =  0; ColB =  8;}   //  7 purple LEDs
  else if (zLedAng <= 2.4) {LedNum = 2; ColR =  8; ColG =  0; ColB =  8;}   //  5 purple LEDs
  else if (zLedAng <= 2.7) {LedNum = 1; ColR = 10; ColG =  0; ColB =  6;}   //  3 purple LEDs
                 else {LedNum = 0; ColR = 16; ColG =  0; ColB =  0;}   //  1 red LED
  // Now set the LEDs
  Strip0.clear();
  switch (LedNum) {
    case 0: LED_SetNum(LedPnt); break;
    case 1: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1); break;
    case 2: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); break;
    case 3: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); LED_SetNum(LedPnt+3); LED_SetNum(LedPnt-3); break;
    case 4: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); LED_SetNum(LedPnt+3); LED_SetNum(LedPnt-3);
     LED_SetNum(LedPnt+4); LED_SetNum(LedPnt-4); break;
    case 5: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); LED_SetNum(LedPnt+3); LED_SetNum(LedPnt-3);
     LED_SetNum(LedPnt+4); LED_SetNum(LedPnt-4); LED_SetNum(LedPnt+5); LED_SetNum(LedPnt-5); break;
    case 6: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); LED_SetNum(LedPnt+3); LED_SetNum(LedPnt-3);
     LED_SetNum(LedPnt+4); LED_SetNum(LedPnt-4); LED_SetNum(LedPnt+5); LED_SetNum(LedPnt-5);
     LED_SetNum(LedPnt+6); LED_SetNum(LedPnt-6); break;
    case 7: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); LED_SetNum(LedPnt+3); LED_SetNum(LedPnt-3);
     LED_SetNum(LedPnt+4); LED_SetNum(LedPnt-4); LED_SetNum(LedPnt+5); LED_SetNum(LedPnt-5);
     LED_SetNum(LedPnt+6); LED_SetNum(LedPnt-6); LED_SetNum(LedPnt+7); LED_SetNum(LedPnt-7); break;
    case 8: LED_SetNumRGB(LedPnt,16,0,0); LED_SetNum(LedPnt+1); LED_SetNum(LedPnt-1);
     LED_SetNum(LedPnt+2); LED_SetNum(LedPnt-2); LED_SetNum(LedPnt+3); LED_SetNum(LedPnt-3);
     LED_SetNum(LedPnt+4); LED_SetNum(LedPnt-4); LED_SetNum(LedPnt+5); LED_SetNum(LedPnt-5);
     LED_SetNum(LedPnt+6); LED_SetNum(LedPnt-6); LED_SetNum(LedPnt+7); LED_SetNum(LedPnt-7); break;
    default: Strip0.fill(0x110000); break;
  }
  // Go on to side plate swpirit level mode, if not switched to self balancing
  if (SafeMode < 4) {LedMode = 13;}
  else {
    // if the LEDshowSkip flag is true we don't trigger a show event
    SetLEDmode(30);
  }
}

// --------------------------------------------------------------------------------

void LED_WheelTest() {
  // Wheel friction drive test mode.
  // Top plate shows red drive ring.
  // Wheel LEDs represent motor drive values, with colour and brightness
  if (LEDonce) {
    GFX_AY = 3; StripA.clear(); StripC.clear();
    GFX_BY = 3; StripB.clear(); StripD.clear();
    LEDonce = false; LedA = -2; LedB = -2;
  }
  switch(LED_Task) {
    case 0: // top plate spirit level
      LEDshowSkip = true; LED_Red_Pulse();
      BakColr = 0x000000;
      LED_Task++; break;
    case 1: // PitchAC drive 
      if (PWM_AC == 0) { // no PWM drive
        // draw a horizontal line across the panel
        if (LedA != 0) {
          StripA.clear(); GFX_AY = 3; LedA = 0;
          DotColr = 0x080000; GFX_FillRow(1,GFX_AY,0x080000);
          GFX_CopyPanel(1,3);
        }
      } else if (PWM_AC > 0) { // motor drive is +ve
        if (LedA != 1) {
          LedA = 1; StripA.clear();
          DotColr = 0x080000;       // red
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(1,0,0,0,0b00000000);   // load and draw GFC char 0
          DotColr = 0x020202;       // white
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(1,0,0,2,0b00000000);   // load and draw GFC char 2
        } else {GFX_RollUp(1);}
        GFX_CopyPanelFlipV(1,3);
      } else { // motor drive is -ve
        if (LedA != -1) {
          LedA = -1; StripA.clear();
          DotColr = 0x080000;       // red
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(1,0,0,1,0b00000000);   // load and draw GFC char 1
          DotColr = 0x020202;       // white
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(1,0,0,3,0b00000000);   // load and draw GFC char 3
        } else {GFX_RollDwn(1);}
        GFX_CopyPanelFlipV(1,3);
      }
      LED_Task++; break;
    case 2: // PitchBD drive
      if (PWM_BD == 0) { // no PWM drive
        // draw a horizontal line across the panel
        if (LedB != 0) {
          StripB.clear(); GFX_AY = 3; LedB = 0;
          GFX_FillRow(2,GFX_AY,0x080000);
          GFX_CopyPanel(2,4);
        }
      } else if (PWM_BD > 0) { // motor drive is +ve
        if (LedB != 1) {
          LedB = 1; StripB.clear();
          DotColr = 0x080000;       // red
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(2,0,0,1,0b00000000);   // load and draw GFC char 1
          DotColr = 0x020202;       // white
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(2,0,0,3,0b00000000);   // load and draw GFC char 3
        } else {GFX_RollDwn(2);}
        GFX_CopyPanelFlipV(2,4);
      } else { // motor drive is -ve
        if (LedB != -1) {
          LedB = -1; StripB.clear();
          DotColr = 0x080000;       // red
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(2,0,0,0,0b00000000);   // load and draw GFC char 0
          DotColr = 0x020202;       // white
          //         (F,X,Y,C,M)
          Gfx_DrawGFX(2,0,0,2,0b00000000);   // load and draw GFC char 2
        } else {GFX_RollUp(2);}
        GFX_CopyPanelFlipV(2,4);
      }
      LED_Del = 3; LEDshow = true;
      LED_Task = 0; break;
  }
}

// --------------------------------------------------------------------------------

void LED_Yellow_Ring() {
  // Pulsing yellow ring, used in self-balance sequence, 1st stage
  // LedCnt += LedInc;
  GFX_FillRGB(0,LedCnt,LedCnt,0);
  StripA.clear();                 // clear panel 'A'
  DotColr = LedCnt/2;
  DotColr = (DotColr<<16) + (DotColr<<8);
  if (LedCnt >=  1) {GFX_FillRow(1,7,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >=  3) {GFX_FillRow(1,6,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >=  5) {GFX_FillRow(1,5,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >=  7) {GFX_FillRow(1,4,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >=  9) {GFX_FillRow(1,3,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >= 11) {GFX_FillRow(1,2,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >= 13) {GFX_FillRow(1,1,DotColr);} // draw a horizontal yellow line based on LedCnt
  if (LedCnt >= 15) {GFX_FillRow(1,0,DotColr);} // draw a horizontal yellow line based on LedCnt
  GFX_CopyPanel(1,2); GFX_CopyPanel(1,3); GFX_CopyPanel(1,4);
  LedCnt++; if (LedCnt > 16) {LedCnt = 0;}
  LED_Del = 3; LEDshow = true;
}

// --------------------------------------------------------------------------------

int Limit255(int zI) {
  // return an integer value between 0 and 255
  if (zI > 255) {zI = 255;}
  if (zI < 0) {zI = 0;}
  return zI;
}

// --------------------------------------------------------------------------------

int16_t map16(int16_t x,int16_t in_min,int16_t in_max,int16_t out_min,int16_t out_max) {
  // standard map function, but for int16_t variables
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// --------------------------------------------------------------------------------

float mapFp(float x,float in_min,float in_max,float out_min,float out_max) {
  // floating point version of the map() function
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// --------------------------------------------------------------------------------

void MPU_6050_Initialise() {
  // set up MPU6050 3-axis motion seonsor hardware
  // By default the MPU-6050 sleeps. So we have to wake it up.
  Wire.beginTransmission(MPU_address);  // Start communication with the address found during search.
  Wire.write(0x6B);                     // We want to write to the PWR_MGMT_1 register (6B hex)
  Wire.write(0x00);                     // Set the register bits as 00000000 to activate the gyro
  I2C_Err = Wire.endTransmission();      // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond

  //Set the full scale of the gyro to +/- 250 degrees per second
  Wire.beginTransmission(MPU_address);  // Start communication with the address found during search.
  Wire.write(0x1B);                     // We want to write to the GYRO_CONFIG register (1B hex)
  Wire.write(0x00);                     // Set the register bits as 00000000 (250dps full scale)
  I2C_Err = Wire.endTransmission();      // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond

  //Set the full scale of the accelerometer to +/- 4g.
  Wire.beginTransmission(MPU_address);  // Start communication with the address found during search.
  Wire.write(0x1C);                     // We want to write to the ACCEL_CONFIG register (1A hex)
  //  Wire.write(0x00);                 // Set the register bits as 00000000 (+/- 2g full scale range)
  Wire.write(0x08);                     // Set the register bits as 00001000 (+/- 4g full scale range)
  I2C_Err = Wire.endTransmission();      // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond

  //Set some filtering to improve the raw data.
  Wire.beginTransmission(MPU_address);  // Start communication with the address found during search
  Wire.write(0x1A);                     // We want to write to the CONFIG register (1A hex)
  Wire.write(0x03);                     // Set the register bits as 00000011 (Set Digital Low Pass Filter to ~43Hz)
  I2C_Err = Wire.endTransmission();      // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond
}

// ---------------------------------------------------------------------------------

void PwmAttach(uint8_t zCh) {
  // Attached the micros internal PWM counter to an output pin
  bool zRet = false; String zT$ = "";
  switch (zCh) {
    // case 0: analogWrite(Pin_AC0, 0); break;
    // case 1: analogWrite(Pin_AC1, 0); break;
    // case 2: analogWrite(Pin_BD0, 0); break;
    // case 3: analogWrite(Pin_BD1, 0); break;
    case 0: zRet = ledcAttachChannel(Pin_AC0, PwmFreq, PwmBits,12); zT$ = "AC0"; break;
    case 1: zRet = ledcAttachChannel(Pin_AC1, PwmFreq, PwmBits,13); zT$ = "AC1"; break;
    case 2: zRet = ledcAttachChannel(Pin_BD0, PwmFreq, PwmBits,14); zT$ = "BD0"; break;
    case 3: zRet = ledcAttachChannel(Pin_BD1, PwmFreq, PwmBits,15); zT$ = "BD1"; break;
  }
  Serial.print("PwmAttach(" + zT$ + ")");
  if (zRet) {Serial.println(" - success!");} else {Serial.println(" - failed!");}
}

// ---------------------------------------------------------------------------------

void PwmSetup(uint8_t zCh) {
  // Set the PWM counter to an output pin
  switch (zCh) {
    case 0: analogWriteFrequency(Pin_AC0, PwmFreq); analogWriteResolution(Pin_AC0, PwmBits); break;
    case 1: analogWriteFrequency(Pin_AC1, PwmFreq); analogWriteResolution(Pin_AC1, PwmBits); break;
    case 2: analogWriteFrequency(Pin_BD0, PwmFreq); analogWriteResolution(Pin_BD0, PwmBits); break;
    case 3: analogWriteFrequency(Pin_BD1, PwmFreq); analogWriteResolution(Pin_BD1, PwmBits); break;
  }
}

// --------------------------------------------------------------------------------

void PWM_OFF() {
  // turn OFF motor drives
  StartEn = false; DriveEn = false;       // disable drive task
  PWM_AC0 = 0; PWM_AC1 = 0; PWM_AC = 0;   // stop Motor A & C
  PWM_BD0 = 0; PWM_BD1 = 0; PWM_BD = 0;   // stop Motor B & D
  analogWrite(Pin_AC0, PWM_AC0); analogWrite(Pin_AC1, PWM_AC1);
  analogWrite(Pin_BD0, PWM_BD0); analogWrite(Pin_BD1, PWM_BD1);
}

// --------------------------------------------------------------------------------

void read_Battery() {
  // Called to read the battery voltage every 8ms
  // In TEST mode we don't apply critical checks, so that we can adjust voltage
  // feeds freely for test purposes.
  BatVol = analogRead(BatPin);
  BatSum = BatSum + BatVol - BatAvg;
  BatAvg = BatSum/50;     // ADC is rolling averaged over 50 readings to remove noise
  // Serial.println(BatAvg);

  if (USB && (BatAvg > Bat6v6)) {
    // power has returned from USB power
    USB = false;
    synchLoopTimers();
  }
  if (BatLow && (BatAvg > Bat6v6)) {
    // power has returned from low power state
    BatLow = false;
    if (!TEST) {ESP.restart();}
  }
  if (BatAvg < BatUSB) {USB = true;}
  if (USB && !TEST) {return;}

  BatV = getBatV(BatAvg); BatVfp = (float)BatV/100.0;

  // Determine the max PWM value for this battery voltage
       if (BatVfp <= V_max) {PWM_Max = 255;}
  else if (BatVfp <= 8.20) {PWM_Max = (BatMult/BatVfp);}  // BatMult = 255.0 * V_max
  else {PWM_Max = 236;}

  // report WARNING state
  if ((BatPc < 20) && !TEST) {Display_BatWarning();}

  // If battery voltage falls below critical threshold, trigger a shutdown
  if (BatAvg < Bat6v6) {
    // Below critical threshold
    if (!BatLow) {
      // Respond to low power state transition
      BatLow = true;  // prevent this from repeating
      // Ensure motors are OFF
      PWM_OFF();
      // Clear eye updates
      GFX_Txt = 0; GFX_Run = 0; LEDshow= false;

      // Turn OFF LEDs
      Strip0.clear(); StripA.clear(); StripB.clear(); StripC.clear(); StripD.clear();
      Strip0.show(); StripA.show(); StripB.show(); StripC.show(); StripD.show();

      if (!TEST) {
        // If Monitor+ is running, send warning, then terminate
        // We don't do this in TEST mode
        PrintTx = ""; Display_Batt_Low();
        while (PrintTx.length() > 0) {PrintTxHandler(); delay(40);}
        DispMon = false;
      }
      // Put a message on the eye display
      tft.fillScreen(TFT_BLACK);
      tft.setTextColor(TFT_RED);
      TFT_Text_Ctr(120,80,5,"Battery");
      TFT_Text_Ctr(120,128,5,"LOW");
      delay(1000);  // allow time for messages
      if (!TEST) {
        // Enter low power state
        // We don't do this in TEST mode
        esp_deep_sleep_start();
        // Endless loop...
        while (true) {yield();}
      }
    }
  }
}

// --------------------------------------------------------------------------------

void readSerial() {
  // Reads characters from the serial port and responds to commands
  // set PrintTgt to indicate source as serial port for responses
  // This function empties the serial buffer, to prevent overload
  while (Serial.available() > 0) {
    keyVal = Serial.read();
    SerialRx = true; PrintTgt = 0;
    decodeKey(keyVal);
  }
}

// --------------------------------------------------------------------------------

void read_SW0() {
  // Read the left SW0 button switch and respond accordingly, pressed == LOW
  // A button press will automatical drop the current task
  // this function is called every 8ms which is the swTimer increment period
  sw0State = digitalRead(sw0Pin);
  if ((sw0WaitHi) && (sw0State == LOW)) {return;} // waiting for HIGH release
  if (sw0WaitHi) {sw0WaitHi = false; sw0LastState = HIGH; return;}  // clear hold flag
  
  if (!sw0State) {
    // Test for Hi-Lo transition
    if (sw0LastState != sw0State) {
      // SW0 switch has just gone LOW, this is the falling edge
      Serial.println("SW0 Lo");
      if (MainMode != 0) {setMainMode(0); sw0WaitHi = true; return;}
      sw0Cnt++;      // count on the falling edge
      sw0Timer = 0;  // reset the timer
    }
    if (sw0Timer >= 125) {
      // SW0 held down for 1 second so change eye display mode
      GFX_Mode++; if (GFX_Mode > 2) {GFX_Mode = 0;}
      GFX_RST = true; sw0WaitHi = true; sw0Cnt = 0; sw0Timer = 0; return;
    }
    sw0Timer++;

  } else {
    // Test for Lo-Hi transition
    if (sw0LastState != sw0State) {
      // rising edge of SW0
      Serial.println("SW0 Hi");
           if (MainMode != 0) {setMainMode(0);}  // active so go back to waiting ManMode 0
      else if (Upright < 0) {setMainMode(4);}    // motor drive demonstration
    }
    if (sw0Cnt > 0) {sw0Timer++;} // button was pressed so run timer
    if (sw0Timer >= 125) {
      // Button released for 1 sec so assume valid button count
    } else if(sw0Timer >= 250) {
      sw0Cnt = 0; sw0Timer = 0;
    }
  }
  sw0LastState = sw0State;      // record current state
}

// --------------------------------------------------------------------------------

void read_SW1() {
  // Read the right SW1 button switch and respond accordingly, pressed == LOW
  // A button press will automatical drop the current task
  // this function is called every 8ms which is the swTimer increment period
  sw1State = digitalRead(sw1Pin);
  if ((sw1WaitHi) && (sw1State == LOW)) {return;} // waiting for HIGH release
  if (sw1WaitHi) {sw1WaitHi = false; sw1LastState = HIGH; return;}  // clear hold flag
  
  if (!sw1State) {
    // Test for Hi-Lo press transition
    if (sw1LastState != sw1State) {
      // SW0 switch has just gone LOW, this is the falling edge
      Serial.println("SW1 Lo");
      if (MainMode != 0) {setMainMode(0); sw1WaitHi = true; return;}
      sw1Cnt++;      // count on the falling edge
      sw1Timer = 0;  // reset the timer
    }
  } else {
    // Test for Lo-Hi release transition
    if (sw1LastState != sw1State) {
      // rising edge of SW1
      Serial.println("SW1 Hi");
    }
    if (sw1Timer >= 125) { // 1 sec timeout on release
      if (Upright  >  0) { // upright, so use count to determine action
             if (sw1Cnt == 1) {setMainMode(1);}   // try to enter self balance sequence, if vertical is within limits
        else if (sw1Cnt == 2) {setMainMode(5);}
      } else if (Upright  <  0) { // upside down, so ignore the count
        setMainMode(4);           // motor drive demonstration
      }
      sw1Cnt = 0; sw1Timer = 0;   // clear the count and timer
    }
  }
  if (sw1Cnt > 0) {sw1Timer++;}   // increment timer, only if switch has been pressed
  sw1LastState = sw1State;        // record current state
}

// ---------------------------------------------------------------------------------

void readWii() {
  // Called every 20ms, but wireless transceiver only transmits every 40ms
  // Considers data received from the Wii Nunchuk controller over WiFi and 
  // manages all movements demanded by the controller
  if (readWiiCall) {return;}  // avoid stack overflow

  readWiiCall = true;         // prevent re-entry of this function
  
  if (RxRec) {
  //##############################################################################
  //
  //  WiFi Nunchuk data received
  //
  //##############################################################################
    // Wii data received but we may not have enabled WiFi mode yet
    RxRec = false; RTxTimeout = 50; // set timeout to approx 1 sec
  //    PrintTx += "JX" + String(JoyX) + " JY" + String(JoyY) + " CZ" + String(CZ) + "\n";
  //    Serial.print("CZ "); Serial.println(CZ);
    if (!WiFiEn) {
  //##############################################################################
  //
  //  WiFi is currently disabled, listening
  //
  //##############################################################################
      // We have not received enough 'C' buttons to make the system active
      // the 'C' button alone must be held in to activate the Wii WiFi link
      if (CZ == 1) {
        // C button only is being pressed
        if (WiFiCntC < 1) {Display_Text2("Waking","Up!");}
        WiFiCntC += 2;
        //  Serial.println("WiFiCnt = " + String(WiFiCntC));
        if (WiFiCntC >= 64) {
          // going active, user has held in 'C' button for sufficient time
          //  Serial.println("WiFiEn = true");
          WiFiCntC = 64;  // fix count at max.
          setMainMode(3); // go to Drive mode
          C_Dn = 8;
        } else {
          // still counting C buttons
          LEDVal = WiFiCntC/10; SetLEDmode(5);  // random colours
        }
      } else {
        // C button not pressed or released early so check for a inadequate count
        if (WiFiCntC > 0) {
          // reduce the C-button counter, which has to be sustained to enable WiFi mode
          WiFiCntC--; LEDVal = 1 + (WiFiCntC/10);
          if (WiFiCntC < 1) {
            Display_Text2("Wakeup","Cancelled");
            setMainMode(0);
          } else {SetLEDmode(0);}
        }
      }
    } else {
      
  //##############################################################################
  //
  //  WiFi is enabled
  //
  //##############################################################################
      // WiFi enabled so check for power down
      // CZ button stats are active LOW
      if (CZ == 0) {
        // both C and Z buttons are being pressed so move towards powering down
        if (WiFiCntC > 0) {
          if (WiFiCntC == 64) {Display_Text2("Going To","Sleep..."); DispMode = 0;}
          WiFiCntC -= 2;
        //  Serial.println("WiFiCnt = " + String(WiFiCntC));
          if (WiFiCntC <= 0) {
            // Power-down condition satisfied, so go to rest
            //  Serial.println("WiFi mode disabled!");
            setMainMode(0);
            if (JoyEye > 0) {
              // The eye is in user cyclops mode, so drop out
              JoyEye = 0; GFX_Mode = GFX_ModeLast; GFX_RST = true;
            }
          } else {
            // still counting down CZ buttons
            SetLEDmode(-5);
          }
        }
      } else {
        // Check CZ buttons to see if a speed change is wanted
        if ((CZ & 2) == 0) {
          // C button is pressed, assume a speed change is needed
          C_Dn++;
          if (C_Dn == 2) {
            C_Dn = 8;
            if (!BSe) {
              // Normal Gear select
              if (Gear < GearMax) {Gear++;}
              if (DispMon) {Display_Text2S24(String(Gear),"Gear");}
            } else {
              // SELECT + C so change to GearMax
              GearMax = 5;
              if (DispMon) {Display_Text2S24("Gear","Max");}
            }
          }
          //  Serial.println("Gear =" + String(Gear));
        } else {C_Dn = 0;} // C button released
        // Check Z button. It has two functions. Short press change speed, but if
        // held in it changes the mode of the joysticks
        if ((CZ & 1) == 0) {
          // Z button is pressed
          Z_Dn++; if (Z_Dn >= 12) {
            // Switching between slide mode and normal only works for Nunchuk controllers
            if ((Z_Dn == 12) && (WiiType == 'N')) {
              Z_Mode = !Z_Mode; // toggle joystick motion mode flag
              Border = true;    // turn on boarder
              if (Z_Mode) {Display_Text2("Slide","Drive");}
              else {Display_Text2("Normal","Drive");} DispDel = 12;        // short display 480ms
              Border = false;   // thick boarder
            } Z_Dn = 50;        // Z_Dn capped at 50, held for >= 2 seconds
          }
          //  Serial.print("Z_Dn "); Serial.println(Z_Dn);
        } else {
          // Z button is not pressed
          if ((Z_Dn > 1) && (Z_Dn <= 10)) {
            // held down for less than 400 ms @ 40ms Rx cycle
            if (!BSe) {
              // Normal Gear down select
              if (Gear > 1) {Gear--;}
              if (DispMon) {Display_Text2S24(String(Gear),"Gear");}
            } else {
              // SE£LECT button held down, so reduce GearMax
              GearMax = 2;
              if (Gear > GearMax) {Gear = GearMax;}
              if (DispMon) {Display_Text2S24("Gear","Min");}
            }
          } Z_Dn = 0; // Z button released so clear down counter
        }
        //  Serial.print("Z_Dn "); Serial.println(Z_Dn);

        if (WiFiCntC < 60) {
          // C-button counter is not at full strength, assume WiFi off released prematurely
          WiFiCntC++; LEDVal = 1 + (WiFiCntC/10);
          if (WiFiCntC == 60) {SetLEDmode(-1);} else {SetLEDmode(-5);}
        }
      }
    }
      
  //##############################################################################
  //
  //  Wii joystick demands
  //
  //##############################################################################
    // joystick movements are ignored if:
    // BallBot is not active
    // Wii Joystick has not sent new data
    // a MainTask is running
    if (BotActive && WiFiEn && JoyActive && (MainMode == 3)) {
      // read the X,Y joysticks and drive if demanded
      JoyActive = false; // prevent stack overflow during moves that call loop()
      // respond to joystick demands every 40ms (25Hz)
      // note direction of travel will only change once stick has been centred
      if ((JoyLX != 32) || (JoyLY != 32)) {
        // The presence of a left Classic joystick is detected, so place eye in
        // joystick control mode
        int16_t zJLX, zJLY;
             if (JoyLX > 32) {zJLX = map(JoyLX,33,63,129,255);}
        else if (JoyLX < 32) {zJLX = map(JoyLX,0,31,0,127);}
           else {zJLX = 128;}
             if (JoyLY > 32) {zJLY = map(JoyLY,33,63,129,255);}
        else if (JoyLY < 32) {zJLY = map(JoyLY,0,31,0,127);}
           else {zJLY = 128;}
        JoyEyeMove(zJLX,zJLY);
      } else {
        // Left joystick demand not detected, so timeout eye demand mode if it
        // had been turned on. Also centre the eye.
        if (JoyEye > 0) {
          // Centre eye if not centred
          if ((EyeJx != 128) || (EyeJy != 128)) {GFX_Wait = 0;}
          EyeJx = 128; EyeJy = 128;
          // Reduce time-out
          JoyEye--;
          if (JoyEye == 0) {GFX_Mode = GFX_ModeLast; GFX_RST = true;}
        }
      }
      // Check for right joystick centre condition
      if ((JoyX >= DbLLX) && (JoyX <= DbULX) && (JoyY >= DbLLY) && (JoyY <= DbULY)) {
        // Joystick is in centre deadband position
        JoyMode = 0;  // reset the direction of travel mode
        JoyCnt = 0;   // reset the controller glitch filter
        AtState = At_DRIVE; DriveTxt = "Drive Mode";
          //  Serial.print(F("#"));
        if (Drive != 0) {
          // If we were moving, then stop the BallBot
          //  Serial.println(F("Stopping"));
          SetLEDmode(20);
          // stop driving if active, but remember direction
          DriveLast = Drive; Drive = 0;
          //  Serial.println("MotorPWM (0,0,0,0)");
          MotorPWM (0,0); Steer = 0;
          DriveTxt = "-";
        }
      } else {
        // Joystick not in centre deadband position. So set movement flag, and
        // respond to joystick demands
        if (JoyMode == 0) {
          // in normal mode a joystick selection locks us into that mode, until released
          JoyCnt++;
          if (JoyCnt == 3) {
            // We have ignored the first 2 values
            // we apply a deadband to the joystick before triggering movement
            // there are potentially 8 directions of travel
            if (!Z_Mode) {
              // Normal drive mode
                    if (JoyY > DbULY) {JoyMode = 1;} // drive forward
              else if (JoyY < DbLLY) {JoyMode = 5;} // drive backwards
              else if (JoyX > DbULX) {JoyMode = 3;} // drive right
              else if (JoyX < DbLLX) {JoyMode = 7;} // drive left
            }
          }
        }
      }
      switch (JoyMode) {
        case 1: JoyMoveFwd(); break;
        case 3: JoyMoveRht(); break;
        case 5: JoyMoveBwd(); break;
        case 7: JoyMoveLft(); break;
      }
    } JoyActive = true; // re-enable this code block
  } else {
  //##############################################################################
  //
  //  No WiFi Nunchuk received recently
  //
  //##############################################################################
    if (WiFiEn) {
      // Robot is enabled for Wii WiFi control
      // maintain Wii enabled counter
      if ((WiFiCntC < 64) && (CZ != 0)) {
        WiFiCntC++; LEDVal = WiFiCntC/10;
        if (WiFiCntC == 64) {SetLEDmode(-1);} else {SetLEDmode(-10);}
      }
    }
  }
  
  readWiiCall = false; // re-allow calls to this function
  //  Serial.print("CntZ "); Serial.println(WiFiCntZ);
  //  Serial.println(MoveState);
}

// --------------------------------------------------------------------------------

void ReadWiiClassic() {
  // Called when the Wii Classic controller is connected over Wi-Fi
  // The 'BLT', 'BRT' and JRX/JRY functions have already been handled
  // Rx bits are active LOW, but these button flags are active HIGH
  //##########################################################
  // front Z buttons
  //##########################################################
  BLT = !(RxWiFi[4]>>5 & 1);  // read L button
  BRT = !(RxWiFi[4]>>1 & 1);  // read R button
  //##########################################################
  // front Z buttons
  //##########################################################
  BZL = !(RxWiFi[5]>>7 & 1);  // read ZL button
  BZR = !(RxWiFi[5]>>2 & 1);  // read ZR button
  //##########################################################
  // Digital pad
  //##########################################################
  BDD = !(RxWiFi[4]>>6 & 1); // read BDD button
  BDL = !(RxWiFi[5]>>1 & 1); // read BDL button
  BDR = !(RxWiFi[4]>>7 & 1); // read BDR button
  BDU = !(RxWiFi[5]    & 1); // read BDU button

  // PLAY#### related code.
  if (WiFiEn) {
    //##########################################################
    // PLAYback functions
    //##########################################################
    // When in WiFi drive mode.
    // Handle digital pad special movements.
    // if (BDD) {Move_BDD();} else {BDD_ = false;}
    // if (BDL) {Move_BDL();} else {BDL_ = false;}
    // if (BDR) {Move_BDR();} else {BDR_ = false;}
    // if (BDU) {Move_BDU();} else {BDU_ = false;}
  }

  //##########################################################
  // Left joystick
  //##########################################################
  JoyLX = RxWiFi[0]; JoyLX &= 0x3f; // extract left joystick value
  JoyLY = RxWiFi[1]; JoyLY &= 0x3f; // extract right joystick value
  //##########################################################
  // Centre Buttons, Select,Home,Start
  //##########################################################
  BSe = !(RxWiFi[4]>>4 & 1);  // read select button
  BHm = !(RxWiFi[4]>>3 & 1);  // read home button
  BSt = !(RxWiFi[4]>>2 & 1);  // read start button

  // PLAY#### related code.
  if (WiFiEn) {
    //##########################################################
    // RECord & PLAYback functions
    //##########################################################
    // When in WiFi drive mode.
    // Handle RECording and PLAYback for Classic controllers
    if (WiiType == 'C') {    // Wii Classic
      // Pressing BZL + Start will start a RECording process.
      // Pressing BZL on its own will stop a RECording.
      // Pressing Start on its own will PLAY a recording.
      if (!BZL && BSt) {if (!REC && !PLAY && (RECtotal > 0)) {PLAY_CLR(); PLAY = true; Gear = 1;}}
      if ( BZL && BSt) {if (!REC && !RECstart) {REC_CLR(); REC = true; RECstart = true; Gear = 1;}}
      if ( BZL && REC && !RECstart) {REC = false;}
    }
    if (WiiType == 'P') {    // Wii Classic Pro
      // Pressing L + Start will start a RECording process.
      // Pressing L on its own will stop a RECording.
      // Pressing Start on its own will PLAY a recording.
      if (!BLT && BSt) {if (!REC && !PLAY && (RECtotal > 0)) {PLAY_CLR(); PLAY = true; Gear = 1;}}
      if ( BLT && BSt) {if (!REC && !RECstart) {REC_CLR(); REC = true; RECstart = true; Gear = 1;}}
      if ( BLT && REC && !RECstart) {REC = false;}
    }
  }

  //##########################################################
  // Right-hand Buttons, Y,X,b,a
  //##########################################################
  Ba = !(RxWiFi[5]>>4 & 1);  // read Ba button
  Bb = !(RxWiFi[5]>>6 & 1);  // read Bb button
  BX = !(RxWiFi[5]>>3 & 1);  // read BX button
  BY = !(RxWiFi[5]>>5 & 1);  // read BY button
}

// ----------------------------------------------------------------------

void setAppMode() {
  // Called when serial data from an external app is deteceted
  GFX_Mode = 2; GFX_RST = true; Txt_Mode = 4;
}

// ----------------------------------------------------------------------

void SetLEDmode(int16_t zM) {
  // Starts a new LED sequence on Core0
  // Set zM to -ve to restart that mode
  if (LedMode == zM) {return;}      // already in this mode, so exit

  LedModeNew = zM; SetLEDs = true;  // trigger Core0 to set this mode
}

// ----------------------------------------------------------------------

void SetLEDmodeNow(int16_t zM) {
  // Called from Core0 task, starts a new LED sequence
  if ((zM < 0) || (LedMode != zM)) {
    AccAbsAng = 0.0;        // absolute accelerometer angle ratio, abs(AcX_OA/AcY_OA)
    ACErr = 0.0;            // PitchAC angle AC error
    ACErrAbs = 0.0;         // abs(ACerr)
    BDErr = 0.0;            // PitchBD angle BD error
    BDErrAbs = 0.0;         // abs(BDErr)
    LedAng = 0.0;           // tilt angle in spirit level mode
    LedCnt = 0;             // counter used in LED sequencing
    LED_Del = 0;            // delay timer used in LED functions
    LedInc = 1;             // increment used in LED sequencing
    LedLast = 0;            // previous task when change set
    LedMem = -1;            // previous recorded LED variable, default -1
    LedNop = false;         // if == true then skip LED modes
    LedNum = 0;             // pointer to the number of LEDs to light in spirit level mode
    LEDonce = true;         // use to display things once
    LedPnt = 0;             // reset Led pointer for spirit level
    LEDshow = false;        // LED show flag for mouth LEDs
    LEDshowSkip = false;    // if == true an LED function can be called without triggering a show event
    LED_Task = 0;           // reset LED task pointer
    LEDVal = 0;             // values used in LED tasks
  }
  LedMode = abs(zM);        // set mode to zM
  // Serial.println("\nSetLEDmode: " + String(zM) + "\t" + String(LedMode));
}

// --------------------------------------------------------------------------------

void setMainMode(int16_t zMM) {
  // set the appropriate conditions for a given mode
  PWM_OFF();                // stop motors when changing modes
  BotActive = false;        // Disable BotActive when changing modes
  Gear = 1;                 // value used to determine PWM Max values
  GearMax = 2;              // restricts Gear selection, default = 2, max = 5
  LED_Del = 0;              // reset delay timer used in LED functions
  LedNop = false;           // remove update block
  MainDel = 0;                    // delay counter used with some Main Mode tasks
  MainSkip = 0;             // counter used to skip MainMOde tasks
  MainSkipCnt = 0;          // value used to pre-load MainSkip
  ModeCnt = 0;              // counter used in MainMode tasks
  ModeCycle = 0;            // cycle counter used in MainMode tasks
  ModeTask = 0;             // task poiinter used in MainMode tasks
  PitchAC_En = false;       // disable PID PitchAC error calcs
  PtchAC_pid_output = 0.0;  // Pitch PID p-gain output
  PitchBD_En = false;       // disable PID PitchBD error calcs
  PtchBD_pid_output = 0.0;  // Pitch PID p-gain output
  PwmEn = false;            // disable motor PWMs
  SafeMode = 0;             // default condition is safe
  SetMainSkip(0);           // reset Main Mode skip timer
  StartEn = false;          // true to perform motor calcs and enable output
  Steer = 0.0;              // steering demand, +ve = right, -ve = left, max 255
  WiFiEn = false;           // Disable WiFi enabled flag
  switch(zMM) {
    case 0: // default MainMode
      break;
    case 1: // user looking for balance start, with full PID settings
      // reset_PID();
      DispMode = DM_SaPtchPtch; DispDel = 1;
      break;
    case 2: // LIVE balancing, Safe Mode 4
      // This mode is never set via this function, to avoid reseting flags
      break;
    case 3: // Drive mode, over WiFi via Wii controller
      BotActive = true;   // Set bot to active
      PwmEn = true;       // Enable motor PWMs
      WiFiEn = true;      // Set WiFi is enabled flag
      break;
    case 4: // Motor drive demo, upside down wheel demo
      // LEDMode = 2; LEDA_Cnt = 1; LEDB_Cnt = 1; LEDC_Cnt = 1; LEDD_Cnt = 1;  // set LED mode
      break;
    case 5: // Wheel drive mode, checks wheel grip in all directions
      break;
  } MainMode = zMM; MainTest = zMM;
  // Serial.flush(); Serial.print(F("MM = ")); Serial.println(MainMode);
}

// --------------------------------------------------------------------------------

void SetMainSkip(int zCnt) {
  // Sets a delay counter for controlling the speed of some Mode functions
  MainSkipCnt = zCnt; MainSkip = MainSkipCnt;
}

// --------------------------------------------------------------------------------

void setMotorAC_PWM(int zPWM) {
  // Sets the motor drive based on the PWM value and sign
  // +ve = clockwise, -ve = anti-clockwise
  // clockwise  - set AC0 = 255 - zPWM,  AC1 = HIGH (255)
  // anti-clkw  - set AC0 = HIGH (255),  AC1 = 255 - zPWM
  // Serial.println(); Serial.print(zPWM);
  PWM_AC = getModPWM(zPWM);
  if (zPWM < 0) {
    // negative drive, anti-clockwise
    PWM_AC0 = 255; PWM_AC1 = 255 + PWM_AC;  // 
  } else if (zPWM == 0) {
    PWM_AC0 = 0; PWM_AC1 = 0;               // H-bridge brake mode
  } else {
    // positive drive, clockwise
    PWM_AC0 = 255 - PWM_AC; PWM_AC1 = 255;  // 
  }
  analogWrite(Pin_AC0, PWM_AC0); analogWrite(Pin_AC1, PWM_AC1);
}

// --------------------------------------------------------------------------------

void setMotorBD_PWM(int zPWM) {
  // sets the motor drive based on the PWM value and sign
  // +ve = clockwise, -ve = anti-clockwise
  // clockwise  - set BD0 = 255 - zPWM,  BD1 = HIGH (255)
  // anti-clkw  - set BD0 = HIGH (255),  BD1 = 255 - zPWM
  // Serial.print(","); Serial.print(zPWM);
  PWM_BD = getModPWM(zPWM);
  if (zPWM < 0) {
    // negative drive, anti-clockwise
    PWM_BD0 = 255; PWM_BD1 = 255 + PWM_BD;  // 
  } else if (zPWM == 0) {
    PWM_BD0 = 0; PWM_BD1 = 0;               // H-bridge brake mode
  } else {
    // positive drive, clockwise
    PWM_BD0 = 255 - PWM_BD; PWM_BD1 = 255;  // 
  }
  analogWrite(Pin_BD0, PWM_BD0); analogWrite(Pin_BD1, PWM_BD1);
}

// --------------------------------------------------------------------------------

void ToggleTestMode() {
  // Toggle the TEST value and settings
  TEST = !TEST;
  getDispRef();             // read references, without displaying screens
}

// ---------------------------------------------------------------------------------



  //###############################################################################
  //
  //  WiFi Code
  //
  //###############################################################################


// --------------------------------------------------------------------------------

void WiFiCheckConnected() {
  // If connected to WiFi do the following data reads
  if (WiFiConnected) {
    readWii(); // if connected to WiFi read Wii data and respond
    if ((WiiType == 'C') || (WiiType == 'P')) {ReadWiiClassic();}
  }
}

// --------------------------------------------------------------------------------

void WiFiClearRxBuff(int zNum) {
  // clears the contents of the receive buffer
  for (int zP = 0; zP < zNum;zP++) {Rx_Buff.ESPdata[zP] = 0;}
}

// --------------------------------------------------------------------------------

void WiFiClearTxBuff(int zNum) {
  // clears the contents of the transmit buffer
  for (int zP = 0; zP < zNum;zP++) {Tx_Buff.ESPdata[zP] = 0;}
}

// ---------------------------------------------------------------------------------

void WiFiDisable() {
  // when either a switch is pressed or WiFi disable is required
  WiFiEn = false; WiFiCntC = 0; WiFiRestCnt = 0; C_Dn = 8; Z_Dn = 8;
}

// --------------------------------------------------------------------------------

void WiFiDisconnect() {
  // enter the WiFi disconnected state
  WiFiConnected = false;
  WiFiClearRxBuff(WiFiTx_len);
  WiFiClearTxBuff(WiFiTx_len);
  WiFiTryCnt = 0;     // reset the connection retry counter
  WiFiTx_CB = 1;      // set ready to send flag
  WiFiRxRec = false;  // clear the data received flag
  Serial.println("WiFiDisconnected!");
}

// --------------------------------------------------------------------------------

void WiFiKill() {
  // Called to permanently disconnect WiFi
  // You need to press RESET to recover from this
  WiFi.disconnect();  // Disconnect existing link
  WiFiDisconnect();   // reset variables
  WiFiOff = true;     // Flag prevents reconnection attempts
}

// --------------------------------------------------------------------------------

void WiFiNotConnected() {
  // If not connected to WiFi then try to connect every second
  // only try to connect WiFi if no 'Pings' are being received from an application
  if (WiFiOff) {return;}  // WiFi has been turned off

  if (Ping == 0) {
    WiFiConCnt--; if (WiFiConCnt < 1) {
      WiFiConCnt = 10; WiFiTryToConnect();  // 400 ms retry time-out
      if (WiFiTryOnce) {Serial.println("Trying to connect: " + getBAhex());}
    }
  }
}

// --------------------------------------------------------------------------------

void WiFiPingHandler() {
  // WiFiPing is set to a value whenever data is received or sent over WiFi
  // For WiFi to remain connected, this must be continuous
  WiFiPing--;
  if (WiFiPing < 1) {
      // We have stopped receiving so disconnect WiFi
    WiFiDisconnect();
    ESP_NOW_BA = -1;
    WiFiTryOnce = true;
    Serial.println("WiFiPing == 0");
    // set defaults here
    WiFiEn = false; SetLEDmode(1);
    // Reset try to connect feature
    WiFiTryCnt = 0;     // WiFi try to connect count
    WiFiTryNum = 0;     // WiFi try to connect total count
  }
}

// --------------------------------------------------------------------------------

void WiFiReadRx() {
  // called from the main loop every cycle to check for ESP-NOW data  
  // a state machine is used to separate the data blocks by header type
  // DispDel = 500;  // hold off display updates for 10 sec
  switch(Rx_Task) {
    case 0: // determine the type of data to handle
           if (Rx_Buff.ESPdata[Rx_Pnt] == 'B') {Rx_Task = 3; break;}
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'C') {WiiType = 'C'; Rx_Task = 1; break;} // Classic
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'P') {WiiType = 'P'; Rx_Task = 1; break;} // Classic Pro
      else if (Rx_Buff.ESPdata[Rx_Pnt] == 'N') {WiiType = 'N'; Rx_Task = 1; break;} // Nunchuk
      else if (Rx_Buff.ESPdata[Rx_Pnt] == '-') {WiiType = '-'; Rx_Task = 1; break;}
      else if (Rx_Buff.ESPdata[Rx_Pnt] == '$') {Rx_Task = 2; break;}
      // oh dear! bloick header not recognised so discard the block and flag error
      else {WiFiRxErr++; WiFiRx = false;}
      break;
      
    case 1: // load Wii I2C data
      // after the header byte there are 6 data bytes + checksum
      // perform XOR on 6 bytes to check for errors
      if (Rx_Buff.ESPdata[7] == (Rx_Buff.ESPdata[1] ^ Rx_Buff.ESPdata[2] ^ Rx_Buff.ESPdata[3] ^ Rx_Buff.ESPdata[4] ^ Rx_Buff.ESPdata[5] ^ Rx_Buff.ESPdata[6])) {
        // data has passed the checksum test
        RxWiFi[0] = Rx_Buff.ESPdata[1];
        RxWiFi[1] = Rx_Buff.ESPdata[2];
        RxWiFi[2] = Rx_Buff.ESPdata[3];
        RxWiFi[3] = Rx_Buff.ESPdata[4];
        RxWiFi[4] = Rx_Buff.ESPdata[5];
        RxWiFi[5] = Rx_Buff.ESPdata[6];


        // This is PLAY#### related code.
        //#######################################################################
        // Adjust period timer, to display Rx rate in Hz
        RxPeriod = millis() - RxMs; RxMs += RxPeriod;
        if (WiFiEn) {
          // WiFi mode must be active for these functions to run.
          // If in RECording mode, compare this with the previous values.
          // If different, then store the change.
               if (REC) {REC_Chng_Check();}
          else if (PLAY) {PLAY_now();}
        } else if (PLAY) {PLAY_STOP();}
        //#######################################################################

        // if Nunchuk data strip off the JX,JY and CZ values
        RxRec = true; // indicate a valid frame
        if (WiiType == 'N') {           // Wii Nunchuk
          JoyX = RxWiFi[0];             // joystick X 0-255
          JoyY = RxWiFi[1];             // joystick Y 0-255
          CZ = RxWiFi[5] & 3;           // CZ buttons, C = 2/0, Z = 1/0, active low
        } else if (WiiType == 'C') {    // Wii Classic
          JoyX = WiiRightStickX();      // joystick X 0-248
               if (JoyX > 16) {JoyX = map(JoyX,17,31,129,255);}
          else if (JoyX < 16) {JoyX = map(JoyX, 0,15,  0,127);}
             else {JoyX = 128;}
          JoyY = WiiRightStickY();      // joystick Y 0-248
               if (JoyY > 16) {JoyY = map(JoyY,17,31,129,255);}
          else if (JoyY < 16) {JoyY = map(JoyY, 0,15,  0,127);}
             else {JoyY = 128;}
          CZ = ((RxWiFi[4] & 0b100000)>>5) + (RxWiFi[4] & 0b10); // read LT and RT buttons as C,Z
        } else if (WiiType == 'P') {    // Wii Classic Pro
          JoyX = WiiRightStickX();      // joystick X 0-248
               if (JoyX > 16) {JoyX = map(JoyX,17,31,129,255);}
          else if (JoyX < 16) {JoyX = map(JoyX, 0,15,  0,127);}
             else {JoyX = 128;}
          JoyY = WiiRightStickY();      // joystick Y 0-248
               if (JoyY > 16) {JoyY = map(JoyY,17,31,129,255);}
          else if (JoyY < 16) {JoyY = map(JoyY, 0,15,  0,127);}
             else {JoyY = 128;}
          CZ = (RxWiFi[4] & 0b10) + ((RxWiFi[5] & 0b100)>2); // read RT and ZR buttons as C,Z
        } else {
          // not Nunchuk so set defaults
          JoyX = 128; JoyY = 128; CZ = 3;
        }
      } else {
        WiFiRxErr++;
      }
      Rx_Task = 0;  // reset the task pointer to lok at another block if needed
      Rx_Pnt += 8;  // increase the data pointer beyond this block
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; Rx_len = 0;} // reached end of buffer
      break;

    case 2: // load text data as if from a serial port
      // read characters up to the end of the buffer
      Rx_Pnt++;
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; Rx_len = 0; break;} // reached end of buffer
      keyVal = Rx_Buff.ESPdata[Rx_Pnt]; PrintTgt = 1; decodeKey(keyVal);
      // Serial.write(keyVal);
      break;

    case 3: // load binary data block
      // data length is specified in second byte
      Rx_Pnt++;
      // at the moment just jump over it, binary transfer is not needed
      Rx_Pnt += Rx_Buff.ESPdata[Rx_Pnt] + 1;
      if (Rx_Pnt >= Rx_len) {WiFiRx = false; break;} // reached end of buffer
      break;
  }
}

// ---------------------------------------------------------------------------------

void WiFiRxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Rx buffer of length zlen
  Rx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    Rx_Chk = Rx_Chk ^ Rx_Buff.ESPdata[zP];
  }
}

// --------------------------------------------------------------------------------

void WiFiTryToConnect() {
  // Try to connect with the Wii Transceiver by sending devices name
  // Constantly trying to connect can impact on robot performance, so we give up
  // after a number of tries
  if (WiFiTryNum > 20) {return;}
  else if (WiFiTryNum == 20) {
    Serial.println("Connection attempts exhausted.");
    WiFiTryNum++; return; // maximum number of tries
  }
  // Initialise ESP-NOW link first
  Init_ESP_NOW();

  Tx_Buff.ESPdata[ 0] = 'B';
  Tx_Buff.ESPdata[ 1] = 'a';
  Tx_Buff.ESPdata[ 2] = 'l';
  Tx_Buff.ESPdata[ 3] = 'l';
  Tx_Buff.ESPdata[ 4] = 'B';
  Tx_Buff.ESPdata[ 5] = 'o';
  Tx_Buff.ESPdata[ 6] = 't';
  Tx_Buff.ESPdata[ 7] = '4';
  Tx_Buff.ESPdata[ 8] = 'x';
  Tx_Buff.ESPdata[ 9] = '4';
  Tx_Buff.ESPdata[10] = '(';
  Tx_Buff.ESPdata[11] = 'S';
  Tx_Buff.ESPdata[12] = ')';
  WiFiTx_len = 13;
  // t0 = micros();
  esp_err_t result = esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
  if (result == ESP_OK) {
    // Serial.println("Sent with success");
  } else {
    // Serial.println("Error sending the data");
  } 

  WiFiTryNum++; // count the total number of attempts
  // Serial.println("WiFiTryToConnect...");
}

// --------------------------------------------------------------------------------

void WiFiTxGetChecksum(int zlen) {
  // performs an XOR checksum calc on the Tx buffer of length zlen
  WiFiTx_Chk = 0x55;  // define checksum seed as 01010101
  for (int zP = 0;zP < zlen;zP++) {
    WiFiTx_Chk = WiFiTx_Chk ^ Tx_Buff.ESPdata[zP];
  }
}

// ----------------------------------------------------------------------

int WiiLeftStickX() {
  // returns a Wii Classic LX value 0/32/63
  return  ((RxWiFi[0] & (byte)0x3f));
}

// ----------------------------------------------------------------------

int WiiLeftStickY() {
  // returns a Wii Classic LY value 0/32/63
  return  ((RxWiFi[1] & (byte)0x3f));       
}

// ----------------------------------------------------------------------

boolean WiiPressedRowBit(byte row, byte bit) {
  // Wii Classic, read a bit to test for button presses.
  // Call directly using the following:
  //  values  (row,bit):
  //  RT      (4,1)
  //  Start   (4,2)
  //  Home    (4,3)
  //  Select  (4,4)
  //  LT      (4,5)
  //  D-Down  (4,6)
  //  D-Right (4,7)
  //  D-Up    (5,0)
  //  D-Left  (5,1)
  //  RZ      (5,2)
  //  X       (5,3)
  //  A       (5,4)
  //  Y       (5,5)
  //  B       (5,6)
  //  LZ      (5,7)

  byte mask = (1 << bit);           // gives 00001000 mask for (X,3) 
  return (!(RxWiFi[row] & mask ));  // gives 11110111 or 11111111
}

// ----------------------------------------------------------------------

int WiiRightStickX() {
  // returns a Wii Classic RX value 0/16/31
  return ((RxWiFi[0] & (byte)0xc0) >> 3) + ((RxWiFi[1] & (byte)0xc0) >> 5) +  ((RxWiFi[2] & (byte)0x80) >> 7);
}

// ----------------------------------------------------------------------

int WiiRightStickY() {
  // returns a Wii Classic RY value 0/16/31
  return RxWiFi[2] & (byte)0x1f;    
}

// --------------------------------------------------------------------------------

