
/* SSD1306Wire.h functions, see https://github.com/ThingPulse/esp8266-oled-ssd1306

  disply.clear()        - clear the local pixel buffer
  disply.display()      - write the buffer to the display
  disply.displayOff()   - turn the display OFF
  disply.displayOn()    - turn the display ON
  display.drawCircle(int16_t x, int16_t y, int16_t radius)
  display.drawFastImage(int16_t x, int16_t y, int16_t width, int16_t height, const uint8_t *image)
  display.drawHorizontalLine(int16_t x, int16_t y, int16_t length)
  display.drawLine(int16_t x0, int16_t y0, int16_t x1, int16_t y1)  - draw from x0k,y0 to x1,y1
  display.drawProgressBar(uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint8_t progress)
  display.drawRect(int16_t x, int16_t y, int16_t width, int16_t height)
  display.drawString(int16_t x, int16_t y, String text)
  display.drawStringMaxWidth(int16_t x, int16_t y, int16_t maxLineWidth, String text)
  display.drawVerticalLine(int16_t x, int16_t y, int16_t length)
  display.drawXbm(int16_t x, int16_t y, int16_t width, int16_t height, const char* xbm)
  display.fillCircle(int16_t x, int16_t y, int16_t radius)
  display.fillRect(int16_t x, int16_t y, int16_t width, int16_t height)
  display.flipScreenVertically()   - flip the current screen upside down
  disply.init()         - initialise the display
  disply.invertDisplay()- inverted display mode
  disply.end()          - free memory used by the display
  disply.displayOn()    - turn the display ON
  display.mirrorScreen() - draw the screen mirrored
  disply.normalDisplay()- normal display mode
  disply.reconnect()    - connect again through I2C
  display.setBrightness(uint8_t)
  display.setColor(OLEDDISPLAY_COLOR color) - BLACK, WHITE, INVERSE
  display.setContrast(uint8_t contrast, uint8_t precharge = 241, uint8_t comdetect = 64)
  display.setFont(const uint8_t* fontData)
  display.setPixel(int16_t x, int16_t y)  - set a pixel at x,y
  display.setTextAlignment(OLEDDISPLAY_TEXT_ALIGNMENT textAlignment)  - TEXT_ALIGN_LEFT, TEXT_ALIGN_CENTER, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER_BOTH

  W = display.getStringWidth(const char* text, uint16_t length)
  W = display.getStringWidth(String text)
  
  Display size is 128 x 64 (w x h) pixels
  Default fonts are ArialMT at 10, 16 & 24 pixels high
  Text options include:
    2 x 24, + 1 x 10 pnt lines
    1 x 24, + 2 x 16 pnt lines
    1 x 24, + 3 x 10 pnt lines
    4 x 16 pnt lines, no line spaces, decending characters will merge
    3 x 16, + 1 x 10 pnt lines
    6 x 10 pnt lines, no line spaces, decending characters will merge
*/

// --------------------------------------------------------------------------------

void Display_40ms() {
  // Called every 40ms from the main loop if DispMon == true
  // if DispDel > 0 then we wait for DispDel cycles to complete before update
  // if (DispNOP) {return;}  // this function is aborted, no I2C activity
  if (WinApp == 16) {return;}           // 16-channel controller is connected
  if (DispDel > 0) {DispDel--; return;} // don't update display if DispDel > 0
  if (DispClr) {Display_clear(); DispClr = false; return;}
  if (DispDisp) {Display_display(); DispDisp = false; DispClr = true; return;}

  // We refresh individual displays based on a counter, DispCnt
  DispCnt--;
  if (DispCnt < 1) {
    // DispCnt is reset for display updates, so can be different for each
    DispCnt = 0;  // prevent negative counting
    if (DispLock) {DispMode = DispOveride;} // force locked mode
    if (!TEST) {
      // Normal menu screens, display depends on menu modes
      switch(DispMode) {
        case -6: Display_MotorDemo(); break;
        case -5: Display_EyeEngine(); break;
        case -4: Display_Status_A(); break;
        case -3: Display_Status_B(); break;
        case -2: Display_WheelTest(); break;
        case -1: Display_Control(); break;
        case  0: Display_Battery(); break;
        case  1: Display_PitchPitchYaw(); break;
        case  2: Display_SaPtchPtch(); break;
        case  3: Display_SafeMode(); break;
        case  4: Display_PID(); break;
        case  5: Display_DriveCtrl(); break;
        case  6: Display_Limits(); break;
        case  7: Display_SpiritLevel(); break;
        case  8: Display_MPU_Acc(); break;
        case  9: Display_MPU_Gyro(); break;
        case 10: Display_MPU_AccAng(); break;
        case 11: Display_MPU_GyrAng(); break;
        case 12: Display_MPU_Levels(); break;
        case 13: Display_MPU_Rot(); break;
        case 14: Display_Joystick(); break;
        case 15: Display_Joystick_Norm(); break;
      }
      DM_Min = -6; DM_Max = 15;
    } else {
      // TEST menu screens, display depends on menu modes
      switch(DispMode) {
        case -6: Display_MotorDemo(); break;
        case -5: Display_MotorTest(); break;
        case -4: Display_Status_A(); break;
        case -3: Display_Status_B(); break;
        case -2: Display_WheelTest(); break;
        case -1: Display_Control(); break;
        case  0: Display_Battery_Plus(); break;
        case  1: Display_SafeMode(); break;
        case  2: Display_PID(); break;
        case  3: Display_DriveCtrl(); break;
        case  4: Display_Limits(); break;
        case  5: Display_SpiritLevel(); break;
        case  6: Display_MPU_Levels(); break;
        case  7: Display_MPU_Acc(); break;
        case  8: Display_MPU_Gyro(); break;
        case  9: Display_MPU_AccAng(); break;
        case 10: Display_MPU_GyrAng(); break;
        case 11: Display_WiFi(); break;
        case 12: Display_I2C(); break;
      }
      DM_Min = -6; DM_Max = 12;
    }
    // the following line puts a rectangle around every display, showing the boundry region
    // Border = true;
    // if ((Border) || TEST) {Display_drawRect(0,0,128,64);}

    // Add a global frame delay offset, if set
    DispDel += DispDelGbl;
  }
}

// --------------------------------------------------------------------------------

void Display_Battery() {
  // normal screen to be displayed
  DM_Batt = DispMode;     // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111     2222222222  3333333333     4444444444
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "BallBot 4x4 (S)");
  Display_setFont(10);
  Display_drawString(64,23, "Battery");
  if (!USB) {
    if (BatVfp > 9.99) {Any$= String(BatVfp,1) + "v  ";}
    else {Any$ = String(BatVfp,2) + "v  ";}
    Display_drawString(64,35, Any$ + String(BatPc) + "%");
  } else {
    Display_drawString(64,35, "USB 5v");
  }
  if (MainMode != 3) {Display_drawString(64,50, getAtState());}
  else {Display_drawString(64,50, DriveTxt);}

  Display_display();
  DispDel = 25; // update every 1s
}

// --------------------------------------------------------------------------------

void Display_Battery_Plus() {
  // DEtailed battery screen to be displayed
  DM_BatPlus = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //  1111111111111111    2222222222222222        3333333333333333
  //                           4444444444444444
  Display_setFont(16); Display_setColor(WHITE);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 1, "Battery+");
  if ((BatAvg > Bat6v6) || TEST) {
    // battery powered mode assumed
    if (BatVfp > 9.99) {Display_drawString(64, 21, String(BatVfp,1) + "v");}
    else {Display_drawString(64,21, String(BatVfp,2) + "v");}

    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_setFont(10);
    Display_drawString( 2,26, String(BatAvg));
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (!USB) {Display_drawString(125,26, String(BatPc) + "%");}
         else {Display_drawString(125,26, "USB 5v");}
    // display the time
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    Display_drawString( 2,45, getHmsTime(millis()));
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    Display_drawString(125,45, BatTime$);
  } else {
    // USB powered mode
    Display_setTextAlignment(TEXT_ALIGN_CENTER);
    Display_drawString(64,24, "USB Pwr");
    // display the time
    Display_drawString(64,45, getHmsTime(millis()));
  }
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 10; // update every 400ms
}

// --------------------------------------------------------------------------------

void Display_Batt_Low() {
  // Battery = 0%
  // This is the last message sent over the comms link
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                1111111111111111    2222222222222222
  Display_setFont(16); Display_setColor(RED);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,15, "Battery Low");
  Display_drawString(64,35, "0%");
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 1000; // hold off display update
}

// --------------------------------------------------------------------------------

void Display_BatWarning() {
  // Battery <= 20% warning
  // we need a timer here to prevent this being called too often
  DM_BatWarn = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  if (!DispMon) {return;}
  if ((millis() - DispBatWarn) < 30000)  {return;} // only action after 30 sec

  DispBatWarn = millis();  // reset the 30 sec timer
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                1111111111111111    2222222222222222
  Display_setFont(16); Display_setColor(RED);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,15, "Battery Low");
  Display_drawString(64,35, "< 20%");
  Display_drawRect(0,0,128,64);   // draw box outline

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 50; // hold off display update for 2sec
}

// --------------------------------------------------------------------------------

void Display_Control() {
  // Display control screen
  DM_Control = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  if (!DispMon) {return;}

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Control");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(63,14, "TEST:");
  Display_drawString(63,26, "Bat.Low:");
  Display_drawString(63,38, "MainMode:");
  Display_drawString(63,50, "WiFi:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(65,38, String(MainMode));
  Display_setColor(BLUE);
  if (TEST) {Display_drawString(65,14, "ON");} else {Display_drawString(65,14, "OFF");}
  Display_drawString(108,38, String(MainTest));

  Display_setColor(RED);
  Display_drawString(65,26, "Test");
  Display_drawString(85,38, "Run");
  Display_drawString(65,50, "Stop");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 10; // hold off display update for 2sec
}

// --------------------------------------------------------------------------------

void Display_DriveCtrl() {
  // Display Drive control screen
  DM_DriveCtrl = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()
  if (!DispMon) {return;}

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111   2222222222  3333333333   4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Drive Ctrl");
  Display_drawString(64,40, DriveTxt);

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 63,15, "Drive min:");
  Display_drawString( 63,27, "Drive max:");
  Display_drawString( 32,52, "ACsp:");
  Display_drawString( 95,52, "BDsp:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(33,52, String(PtchACSp,2));
  Display_drawString(96,52, String(PtchBDSp,2));
  if (TEST) {Display_setColor(BLUE);}
  Display_drawString(64,15, String(DrvSpMin,2));
  Display_drawString(64,27, String(DrvSpMax,2));

  Display_setColor(WHITE);
  Display_setFont(5);
  Display_drawString(2,2, Get4BIN(Drive_byte));
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // hold off display update for 2sec
}

// --------------------------------------------------------------------------------

void Display_EyeEngine() {
  // Display Eye engine screen
  DM_EyeEng = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  if (!DispMon) {return;}

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  22222222  33333333  44444444  55555555  66666666
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Eye Engine");

  Display_setFont(8);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(63,14, "Enabled:");
  Display_drawString(63,24, "Sprites:");
  Display_drawString(35,34, "JoyEye:");
  Display_drawString(99,34, "GFXMode:");
  Display_drawString(35,44, "EyeJx:");
  Display_drawString(99,44, "EyeJy:");
  Display_drawString(35,54, "GFX_X:");
  Display_drawString(99,54, "GFX_Y:");
  

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 36,34, String(JoyEye));
  Display_drawString(100,34, String(GFX_Mode));
  Display_drawString( 36,44, String(EyeJx));
  Display_drawString(100,44, String(EyeJy));
  Display_drawString( 36,54, String(GFX_X));
  Display_drawString(100,54, String(GFX_Y));

  Display_setColor(BLUE);
  if (Eye_En) {Display_drawString(65,14, "ON");} else {Display_drawString(65,14, "OFF");}
  if (Sprites) {Display_drawString( 65,24, "ON");} else {Display_drawString( 65,24, "OFF");}

  Display_setColor(GREEN);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(127,34, String(GFX_Steps));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // hold off display update for 2sec
}

// --------------------------------------------------------------------------------

void Display_I2C() {
  // display 6 bytes received from Wii I2C
  // display I2C values as decimal 0 - 255
  DM_I2C = DispMode;    // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   111111111122222222223333333333444444444455555555556666666666
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_setFont(10); DispCX = 76;
  Display_drawString(DispCX,  2, String(RxWiFi[0]));
  Display_drawString(DispCX, 12, String(RxWiFi[1]));
  Display_drawString(DispCX, 22, String(RxWiFi[2]));
  Display_drawString(DispCX, 32, String(RxWiFi[3]));
  Display_drawString(DispCX, 42, String(RxWiFi[4]));
  Display_drawString(DispCX, 52, String(RxWiFi[5]));
  // display I2C values as binary 00000000 - 11111111
  Display_setTextAlignment(TEXT_ALIGN_LEFT); DispCX = 82;
  Display_drawString(DispCX,  2, GetBIN(RxWiFi[0]));
  Display_drawString(DispCX, 12, GetBIN(RxWiFi[1]));
  Display_drawString(DispCX, 22, GetBIN(RxWiFi[2]));
  Display_drawString(DispCX, 32, GetBIN(RxWiFi[3]));
  Display_drawString(DispCX, 42, GetBIN(RxWiFi[4]));
  Display_drawString(DispCX, 52, GetBIN(RxWiFi[5]));
  // display 'I2C' in top left corner
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 11111111112222222222
  Display_drawString(2,  2, "Wii");
       if (WiiType == 'N') {Display_drawString(2, 12, "Nunchuk");}
  else if (WiiType == 'C') {Display_drawString(2, 12, "Classic");}
  else if (WiiType == 'P') {Display_drawString(2, 12, "Classic-Pro");}
  else {Display_drawString(2, 20, "???");}
  if (TEST) {Display_setColor(WHITE); Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 200ms
}

// --------------------------------------------------------------------------------

void Display_Intro() {
  // display robot type when DispMon is first connected
  // we clear current flags so that this is displayed immediately
  DM_Intro = DispMode;    // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  if (!DispMon) {return;}

  DispClr = false; DispDisp = false; DispCnt = 1;
  
  Display_clear();    // remove current display content
  Display_display();  // clears mirror display immediately
  if (Brightness != 255) {Display_setBrightness(255);}  // display may be in sleep mode
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111111111111111111 222222222222222222222222   3333333333
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "BallBot");
  Display_setFont(10);
  Display_drawString(64, 25, "4x4  Strider");
  Display_drawString(64,52, "Batt: " + String(BatPc) + "%");
  Display_display();  // display immediately

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  DispMode = DM_Batt; // set the default screen
  DispDel = 50;       // wait 2s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Joystick() {
  // display joystick valus and offsets
  DM_JoyOff = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111     2222222222222222     3333333333333333
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Joysticks");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_setFont(10);
  if (WiiType == 'N') {
    // Wii Nunchuk joystick data
    Display_drawString(16,25, "X = ");
    Display_drawString(44,25, String(RxWiFi[0]));
    Display_drawString(16,46, "Y = ");
    Display_drawString(44,46, String(RxWiFi[1]));
    } else if (WiiType == 'C') {
    // Wii Classic joystick data
    Display_drawString( 8,25, "XL= ");
    Display_drawString(40,25, String(RxWiFi[0] & 63));
    Display_drawString(72,25, "XR= ");
    AnyByte = ((RxWiFi[0] & 0b11000000)>>3) + ((RxWiFi[1] & 0b11000000)>>5) + ((RxWiFi[2] & 0b10000000)>>7);
    Display_drawString(104,25, String(AnyByte));
    Display_drawString( 8,46, "YL= ");
    Display_drawString(40,46, String(RxWiFi[1] & 63));
    Display_drawString(72,46, "YR= ");
    Display_drawString(104,46, String(RxWiFi[2] & 31));
    } else if (WiiType == 'P') {
    // Wii Classic Pro joystick data
    Display_drawString( 8,25, "XL= ");
    Display_drawString(40,25, String(RxWiFi[0] & 63));
    Display_drawString(72,25, "XR= ");
    AnyByte = ((RxWiFi[0] & 0b11000000)>>3) + ((RxWiFi[1] & 0b11000000)>>5) + ((RxWiFi[2] & 0b10000000)>>7);
    Display_drawString(104,25, String(AnyByte));
    Display_drawString( 8,46, "YL= ");
    Display_drawString(40,46, String(RxWiFi[1] & 63));
    Display_drawString(72,46, "YR= ");
    Display_drawString(104,46, String(RxWiFi[2] & 31));
  } else {
    // Wii type undefined
    Display_drawString(44,25, "-?-");
    Display_drawString(44, 46, "-?-");
    Display_drawString(86,25, "-?-");
    Display_drawString(86, 46, "-?-");
  }
  Display_display();
  DispDel = 3;
}

// --------------------------------------------------------------------------------

void Display_Joystick_Norm() {
  // display normalised joystick valus and offsets
  DM_JoyNorm = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //     1111111111111111     2222222222222222     3333333333333333
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 4, "Joysticks");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_setFont(10);
  if (WiiType == 'N') {
    // Wii Nunchuk joystick data
    Display_drawString(16,25, "X = ");
    Display_drawString(44,25, String(RxWiFi[0]));
    Display_drawString(16,46, "Y = ");
    Display_drawString(44,46, String(RxWiFi[1]));
    } else if (WiiType == 'C') {
    // Wii Classic joystick data
    Display_drawString( 8,25, "LX= ");
    Display_drawString(40,25, String(JoyLX));
    Display_drawString(72,25, "JX= ");
    Display_drawString(104,25, String(JoyX));
    Display_drawString( 8,46, "LY= ");
    Display_drawString(40,46, String(JoyLY));
    Display_drawString(72,46, "JY= ");
    Display_drawString(104,46, String(JoyY));
    } else if (WiiType == 'P') {
    // Wii Classic Pro joystick data
    Display_drawString( 8,25, "LX= ");
    Display_drawString(40,25, String(JoyLX));
    Display_drawString(72,25, "JX= ");
    Display_drawString(104,25, String(JoyX));
    Display_drawString( 8,46, "LY= ");
    Display_drawString(40,46, String(JoyLY));
    Display_drawString(72,46, "JY= ");
    Display_drawString(104,46, String(JoyY));
  } else {
    // Wii type undefined
    Display_drawString(44,25, "-?-");
    Display_drawString(44, 46, "-?-");
    Display_drawString(86,25, "-?-");
    Display_drawString(86, 46, "-?-");
  }
  Display_display();
  DispDel = 3;
}

// --------------------------------------------------------------------------------

void Display_Limits() {
  // Display 'Limits'' variables
  DM_Limits = DispMode; // note Display Mode assignment
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(63,2, "Limits");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 54,14, "V-max:"); Display_drawString(108,14, "PWM:");
  Display_drawString( 54,26, "MtrMin:");
  Display_drawString(108,26, "MtrSrt:");
  Display_drawString( 54,38, "GyAcMod:");
  Display_drawString( 38,50, "BrkPnt:");
  Display_drawString( 93,50, "BrkVal:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(109,14, String(PWM_Max));
  Display_drawString( 90,38, String(GyAcMod * 100.0,1) + "%");

  if (USB) {Display_setColor(RED);}
  else if (TEST) {Display_setColor(BLUE);} else {Display_setColor(WHITE);}
  Display_drawString( 55,14, String(V_max,2));
  if (ModPWM) {Display_drawString( 55,26, get3Digits(MtrMin));} else {Display_drawString( 55,26, "---");}
  if (ModPWM) {Display_drawString(109,26, get3Digits(MtrMinStrt));} else {Display_drawString(109,26, "---");}
  Display_drawString( 55,38, String(GyAcMod,3));
  Any$ = ""; if (BrkPnt < 10.00) {Any$ = "0";}
  Display_drawString( 39,50, Any$ + String(BrkPnt,1));
  Display_drawString( 94,50, Any$ + String(BrkVal,3));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 160ms
}

// --------------------------------------------------------------------------------

void Display_MPU_Acc() {
  // display MPU6050 accelerometer information
  DM_MPU_Acc = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //    1111111111      2222222222      3333333333      4444444444
  Display_setFont(10); 
  if (!I2C_MPU) {Display_setColor(RED);} else {Display_setColor(WHITE);}
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 3, "MPU Acc.");

  Display_setTextAlignment(TEXT_ALIGN_LEFT); Display_setColor(BLUE);
  if (AcOffEn) {Display_drawString(88, 3, "-");}      // offsets are being applied
           else {Display_drawString(86, 3, "R");}     // raw values (yellow)

  if (I2C_MPU) {if (AcOffEn) {Display_setColor(WHITE);} else {Display_setColor(YELLOW);}}
  else {Display_setColor(RED);}
  if (!AcOffEn) {Display_drawString( 2, 3, String(AcAvgDiv));}
  Display_drawString( 2,19, "AcX");
  if (AcOffEn) {Display_drawString(26,19, String(AcX - AcOffX));} else {Display_drawString(26,19, String(AcX));}
  // if (MainMode == 1) {
    if (AcOffEn) {Display_drawString(70,19, "AoX");} else {Display_drawString(70,19, "AvX");}
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (AcOffEn) {Display_drawString(125,19, String(AcOffX));} else {Display_drawString(125,19, String(AcXAvgL));}
  // }

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 2,35, "AcY");
  if (AcOffEn) {Display_drawString(26,35, String(AcY - AcOffY));} else {Display_drawString(26,35, String(AcY));}
  // if (MainMode == 1) {
    if (AcOffEn) {Display_drawString(70,35, "AoY");} else {Display_drawString(70,35, "AvY");}
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (AcOffEn) {Display_drawString(125, 35, String(AcOffY));} else {Display_drawString(125, 35, String(AcYAvgL));}
  // }

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 2,51, "AcZ");
  if (AcOffEn) {Display_drawString(26,51, String(AcZ - AcOffZ));} else {Display_drawString(26,51, String(AcZ));}
  // if (MainMode == 1) {
    if (AcOffEn) {Display_drawString(70,51, "AoZ");} else {Display_drawString(70,51, "AvZ");}
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (AcOffEn) {Display_drawString(125,51, "---");} else {Display_drawString(125, 51, String(AcZAvgL));}
  // }

  if (TEST) {Display_setColor(WHITE); Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 200ms
}

// --------------------------------------------------------------------------------

void Display_MPU_AccAng() {
  // displays the MPU Acc angles
  DM_MpuAccAng = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //    1111111111      2222222222      3333333333      4444444444
  Display_setFont(10); Display_setColor(WHITE);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 3, "MPU Acc Ang.");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62,19, "PitchAC");
  Display_drawString(62,35, "PitchBD");
  Display_drawString(62,51, "Yaw");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(66,19, String(AngAcY) + Deg);
  Display_drawString(66,35, String(AngAcX) + Deg);
  Display_drawString(66,51, "---");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 160ms
}

// --------------------------------------------------------------------------------

void Display_MPU_GyrAng() {
  // displays the MPU angles
  DM_MpuGyrAng = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //    1111111111      2222222222      3333333333      4444444444
  Display_setFont(10); Display_setColor(WHITE);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 3, "MPU Gyr Ang.");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62,19, "PitchAC");
  Display_drawString(62,35, "PitchBD");
  Display_drawString(62,51, "Yaw");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(66,19, String(AngGyX) + Deg);      // Pitch AC
  Display_drawString(66,35, String(AngGyY) + Deg);      // Pitch BD
  Display_drawString(66,51, String(AngGyZ) + Deg);      // Yaw

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 160ms
}

// --------------------------------------------------------------------------------

void Display_MPU_Gyro() {
  // display MPU-6050 gyro information
  DM_MPU_Gyros = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //    1111111111      2222222222      3333333333      4444444444
  Display_setFont(10);
  if (!I2C_MPU) {Display_setColor(RED);} else {Display_setColor(WHITE);}
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 3, "MPU Gyros");

  Display_setTextAlignment(TEXT_ALIGN_LEFT); Display_setColor(BLUE);
  if (GyOffEn) {Display_drawString(92, 3, "-");}
           else {Display_drawString(90, 3, "R");}

  if (I2C_MPU) {if (GyOffEn) {Display_setColor(WHITE);} else {Display_setColor(YELLOW);}}
  else {Display_setColor(RED);}
  if (!GyOffEn) {Display_drawString( 2, 3, String(GyAvgDiv));}
  Display_drawString( 2,19, "GyX");
  if (GyOffEn) {Display_drawString(26,19, String(GyX - GyOffX));} else {Display_drawString(26,19, String(GyX));}
  // if (MainMode == 1) {
    if (GyOffEn) {Display_drawString(70,19, "GoX");} else {Display_drawString(70,19, "GvX");}
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (GyOffEn) {Display_drawString(125,19, String(GyOffX));} else {Display_drawString(125,19, String(GyXAvgL));}
  // }

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 2,35, "GyY");
  if (GyOffEn) {Display_drawString(26,35, String(GyY - GyOffY));} else {Display_drawString(26,35, String(GyY));}
  // if (MainMode == 1) {
    if (GyOffEn) {Display_drawString(70,35, "GoY");} else {Display_drawString(70,35, "GvY");}
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (GyOffEn) {Display_drawString(125,35, String(GyOffY));} else {Display_drawString(125,35, String(GyYAvgL));}
  // }

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 2,51, "GyZ");
  if (GyOffEn) {Display_drawString(26,51, String(GyZ - GyOffZ));} else {Display_drawString(26,51, String(GyZ));}
  // if (MainMode == 1) {
    if (GyOffEn) {Display_drawString(70,51, "GoZ");} else {Display_drawString(70,51, "GvZ");}
    Display_setTextAlignment(TEXT_ALIGN_RIGHT);
    if (GyOffEn) {Display_drawString(125,51, String(GyOffZ));} else {Display_drawString(125,51, String(GyZAvgL));}
  // }

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 200ms
}

// --------------------------------------------------------------------------------

void Display_MPU_Levels() {
  // display MPU-6050 Pitch and Roll angles like spirit level
  DM_MpuLvl = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  Display_setFont(10);

  if ((fabs(PitchAC) <= 2.00) && (fabs(PitchBD) <= 2.00)) {
    // angles are <= 2.0°
    // Y64 vertical text spacing, from Y=0 to Y=63
    // 0123456789012345678901234567890123456789012345678901234567890123
    // 1111111111                 2222222222
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    if (PitchAC > 0.00) {
      Display_drawString(68, 0, String(PitchAC,1) + Deg);
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(60, 0, "F");
    } else {
      Display_drawString(68,53, String(PitchAC,1) + Deg);
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(60,53, "B");
    }
    Display_setTextAlignment(TEXT_ALIGN_LEFT);
    if (PitchBD > 0.00) {
      Display_drawString(100,26, String(PitchBD,1) + Deg);
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(28,26, "L");
    } else {
      Display_drawString(100,26, "R");
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(28,26, String(PitchBD,1) + Deg);
    }
    Display_drawLine(32,32,96,32);    // horizontal centre line
    DispCX = 62 + (16 * PitchBD);  // Roll bubble
    Display_drawRect(DispCX,30,4,4);  // horizontal bubble
    Display_drawLine(64,0,64,63);     // vertical centre line
    DispCY = 30 - (16 * PitchAC);  // Pitch bubble
    Display_drawRect(62,DispCY,4,4);  // vertical bubble
  } else {
    // angles are >= 2.0°
    if (PitchBD >= 0.00) {
      // rolling left
      DispCX = 96; DispCY = 2 * fabs(PitchBD);
      if (DispCY > 40.0) {DispCY = 40.0;}
      Display_drawLine(DispCX,32 - DispCY,DispCX,32 + DispCY);  // horizontal wedge right
      Display_drawLine(32,32,DispCX,32 - DispCY);
      Display_drawLine(32,32,DispCX,32 + DispCY);
      Display_setTextAlignment(TEXT_ALIGN_LEFT);
      Display_drawString(DispCX + 4,26, String(PitchBD,1) + Deg);
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(28,26, "L");
    } else {
      // rolling right
      DispCX = 32; DispCY = 2 * fabs(PitchBD);
      if (DispCY > 40.0) {DispCY = 40.0;}
      Display_drawLine(DispCX,32 - DispCY,DispCX,32 + DispCY);  // horizontal wedge left
      Display_drawLine(96,32,DispCX,32 - DispCY);
      Display_drawLine(96,32,DispCX,32 + DispCY);
      Display_setTextAlignment(TEXT_ALIGN_RIGHT);
      Display_drawString(DispCX - 4,26, String(PitchBD,1) + Deg);
      Display_setTextAlignment(TEXT_ALIGN_LEFT);
      Display_drawString(100,26, "R");
    }
    if (PitchAC >= 0.00) {
      // pitching forwards
      DispCX = 2 * fabs(PitchAC); DispCY = 1; 
      if (DispCX > 40.0) {DispCX = 40.0;}
      Display_drawLine(64 - DispCX,DispCY,64 + DispCX,DispCY);  // vertical wedge up
      Display_drawLine(64,64,64 - DispCX,DispCY);
      Display_drawLine(64,64,64 + DispCX,DispCY);
      if (PitchBD < 0.00) {
        // rolling right
        Display_setTextAlignment(TEXT_ALIGN_LEFT);
        Display_drawString(70,53, String(PitchAC,1) + Deg);
        Display_setTextAlignment(TEXT_ALIGN_RIGHT);
        Display_drawString(58,53, "F");
      } else {
        // rolling left
        Display_setTextAlignment(TEXT_ALIGN_RIGHT);
        Display_drawString(58,53, String(PitchAC,1) + Deg);
        Display_setTextAlignment(TEXT_ALIGN_LEFT);
        Display_drawString(70,53, "F");
      }
    } else {
      // pitching backwards
      DispCX = 2 * fabs(PitchAC); DispCY = 63; 
      if (DispCX > 40.0) {DispCX = 40.0;}
      Display_drawLine(64 - DispCX,DispCY,64 + DispCX,DispCY);  // vertical wedge down
      Display_drawLine(64,0,64 - DispCX,DispCY);
      Display_drawLine(64,0,64 + DispCX,DispCY);
      if (PitchBD < 0.00) {
        // rolling right
        Display_setTextAlignment(TEXT_ALIGN_LEFT);
        Display_drawString(70, 0, String(PitchAC,1) + Deg);
        Display_setTextAlignment(TEXT_ALIGN_RIGHT);
        Display_drawString(58, 0, "B");
      } else {
        // rolling left
        Display_setTextAlignment(TEXT_ALIGN_RIGHT);
        Display_drawString(58, 0, String(PitchAC,1) + Deg);
        Display_setTextAlignment(TEXT_ALIGN_LEFT);
        Display_drawString(70, 0, "B");
      }
    }
  }
  if (MainMode) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 160ms
}

// --------------------------------------------------------------------------------

void Display_MPU_Rot() {
  // display MPU-6050 rotation angle
  DM_MpuRot = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                         1111111111111111              2222222222
  Display_setFont(16);  // fonts are 10,16 or 24
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 24, String(Yaw,1) + Deg);
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(0,  0, "Yaw");
  Display_drawString(0, 54, String(YawDif,3) + Deg);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(127, 54, String(GyOffZ));
  
  Display_drawCircle(64,32,31);
  Display_drawLine(21,32,30,32);
  Display_drawLine(97,32,106,32);
  if (YawDif > 0.01) {
    Display_drawLine(21,32,26,21); Display_drawLine(26,21,30,32);
    Display_drawLine(97,32,102,41); Display_drawLine(102,41,106,32);
  } else if (YawDif < -0.01) {
    Display_drawLine(21,32,26,41); Display_drawLine(26,41,30,32);
    Display_drawLine(97,32,102,21); Display_drawLine(102,21,106,32);
  }
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 5;
}

// --------------------------------------------------------------------------------

void Display_MPU_Warning() {
  // Display warning when MPU fails
  DM_MpuWarn = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111111111111111111   2222222222222222   3333333333333333
  Display_setFont(24);
  Display_setColor(RED);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "WARNING");
  Display_setFont(16);
  Display_drawString(64,27, "MPU-6050");
  Display_drawString(64,46, "Has Failed!");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 50;       // wait 2s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_MotorDemo() {
  // Display motor demo stats
  DM_MtrDemo = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111    2222222222              4444444444
  //                             3333333333              5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Motor Demo");
  Display_drawString(64,28, ModeStr$);

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(32,16, "D:");
  Display_drawString(94,16, "A:");
  Display_drawString(32,40, "C:");
  Display_drawString(94,40, "B:");
  Display_drawString(126,52, String(ModeTask));

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(34,16, String(PWM_BD));
  Display_drawString(96,16, String(PWM_AC));
  Display_drawString(34,40, String(PWM_AC));
  Display_drawString(96,40, String(PWM_BD));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 3;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_MotorTest() {
  // Display motor test control display
  DM_MtrTest = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111   2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Motor Test");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(17,15, "A0:"); Display_drawString(53,15, "A1:"); Display_drawString(82,15, "A:");
  Display_drawString(17,27, "B0:"); Display_drawString(53,27, "B1:"); Display_drawString(82,27, "B:");
  Display_drawString(17,39, "C0:"); Display_drawString(53,39, "C1:"); Display_drawString(82,39, "C:");
  Display_drawString(17,51, "D0:"); Display_drawString(53,51, "D1:"); Display_drawString(82,51, "D:");

  Display_setColor(RED);
  if (!PWM_AC_) {Display_drawString(125,15, "off ");} else {Display_drawString(125,15, "OFF");}
  if (!PWM_BD_) {Display_drawString(125,27, "off ");} else {Display_drawString(125,27, "OFF");}

  Display_setTextAlignment(TEXT_ALIGN_LEFT); Display_setColor(BLUE);
  Display_drawString(18,15,String(PWM_AC0)); Display_drawString(54,15,String(PWM_AC1)); Display_drawString(83,15, String(PWM_AC));
  Display_drawString(18,27,String(PWM_BD0)); Display_drawString(54,27,String(PWM_BD1)); Display_drawString(83,27, String(PWM_BD));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 5;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_PID() {
  // display Pitch PID variables
  DM_PID = DispMode; // note Display Mode assignment
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(63,2, "PID Controller");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(63,14, "P-gain:"); Display_drawString(110,14, "db:");
  Display_drawString(63,26, "I-gain:");
  Display_drawString(63,38, "D-gain:");
  Display_drawString(36,50, "ACSa:"); Display_drawString(98,50, "BDSa:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  if (TEST) {Display_setColor(RED); Display_drawString(2,2, "RST");}
  if (USB) {Display_setColor(RED);}
  else if (TEST) {Display_setColor(BLUE);} else {Display_setColor(WHITE);}
  Any$ = ""; if (pid_p_gain < 100.00) {Any$ = "0";}
  if (pid_p_gain < 10.00) {Any$ += "0";}
  Display_drawString( 64,14, Any$ + String(pid_p_gain,1));
  Display_drawString(111,14, String(PID_db,1));
  if (PID_i_En) {
    Any$ = ""; if (pid_i_gain < 10.00) {Any$ = "0";}
    Display_drawString(64,26, Any$ + String(pid_i_gain,2));
    Any$ = ""; if (pid_i_max < 100.0) {Any$ = "0";}
    if (pid_i_max < 10.0) {Any$ += "0";}
    Display_drawString(100,26, Any$ + String(pid_i_max,1));
  } else {Display_drawString(64,26, "---.---");}
  if (PID_d_En) {
    Any$ = ""; if (pid_d_gain < 10.00) {Any$ = "0";}
    Display_drawString(64,38, Any$ + String(pid_d_gain,2));}
  else {Display_drawString(64,38, "---.---");}
  if (PtchACSbSpSa < 0.00) {Display_drawString( 37,50, "-");}
  Display_drawString( 41,50, String(fabs(PtchACSbSpSa),2) + Deg);
  if (PtchBDSbSpSa < 0.00) {Display_drawString( 98,50, "-");}
  Display_drawString(102,50, String(fabs(PtchBDSbSpSa),2) + Deg);

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 6; // update every 160ms
}

// --------------------------------------------------------------------------------

void Display_PitchPitchYaw() {
  // Display angles
  DM_PtchYaw = DispMode;  // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111        2222222222  3333333333  4444444444
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 2,16,6,16,6,16,2
  Display_drawString(64, 2, "Angles");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62,20, "PitchAC:");
  Display_drawString(62,32, "PitchBD:");
  Display_drawString(62,44, "Yaw:");
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(66,20, String(PitchAC) + Deg);
  Display_drawString(66,32, String(PitchBD) + Deg);
  Display_drawString(66,44, String(Yaw) + Deg);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 3;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Recall() {
  // Display 6 bytes received from Wii over ESP-NOW.
  // Display I2C values as decimal 0 - 255.
  DM_Recall = DispMode;     // note Display Mode assignment
  if (!DispEn) {return;}    // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 1111111111 222222223333333344444444555555556666666677777777
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  if (!PLAY) {Display_drawString(63,  0, "Record");}
  else {Display_drawString(63,  0, "Recall");}

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_setFont(8); DispCX = 89;
  Display_drawString(DispCX, 11, String(RxWiFi[0]));
  Display_drawString(DispCX, 19, String(RxWiFi[1]));
  Display_drawString(DispCX, 27, String(RxWiFi[2]));
  Display_drawString(DispCX, 35, String(RxWiFi[3]));
  Display_drawString(DispCX, 43, String(RxWiFi[4]));
  Display_drawString(DispCX, 51, String(RxWiFi[5]));
  // display I2C values as binary 00000000 - 11111111
  Display_setTextAlignment(TEXT_ALIGN_LEFT); DispCX = 93;
  Display_drawString(DispCX, 11, GetBIN(RxWiFi[0]));
  Display_drawString(DispCX, 19, GetBIN(RxWiFi[1]));
  Display_drawString(DispCX, 27, GetBIN(RxWiFi[2]));
  Display_drawString(DispCX, 35, GetBIN(RxWiFi[3]));
  Display_drawString(DispCX, 43, GetBIN(RxWiFi[4]));
  Display_drawString(DispCX, 51, GetBIN(RxWiFi[5]));
  // display Wii type in top left corner
  if (!WiFiEn) {
    Display_setColor(RED);
    Display_drawString(0, 11, "WiFi Disabled");
    Display_setColor(WHITE);
  } else {
         if (WiiType == 'C') {Display_drawString(0, 11, "Classic");}
    else if (WiiType == 'P') {Display_drawString(0, 11, "Classic Pro");}
    else if (WiiType == 'N') {Display_drawString(0, 11, "Nunchuk");}
  }

  Display_drawString(2, 27, "Size:" + String(RECpnt) + "/" + String(RECnum) + "  " + String((100.0 * RECpnt)/RECnum,0) + "%");
  Display_drawString(1, 35, "REC:");
  Display_drawString(34, 35, "PLAY:");
  if (!PLAY) {
    Display_drawString( 0, 43, "Time:" + String(RECtime[RECpnt]));
    Display_drawString(49, 43, getMinSec(RECtime[RECpnt]));
    Display_drawString(10, 51, "At:" + String(RECtotal));
    Display_drawString(49, 51, getMinSec(RECtotal));
  } else {
    Display_drawString( 0, 43, "Time:" + String(RECtime[PLAYpnt]));
    Display_drawString(49, 43, getMinSec(RECtime[PLAYpnt]));
    Display_drawString(10, 51, "At:" + String(PLAYtotal));
    Display_drawString(49, 51, getMinSec(PLAYtotal));
  }

  if (WiFiEn) {Display_setColor(BLUE);} else {Display_setColor(GREY);}
  Display_drawString(  0, 0, "COPY");
  Display_drawString(113, 0, "CLR");
  if (REC) {Display_drawString(19, 35, "Y");} else {Display_drawString(19, 35, "N");}
  if (PLAY) {Display_drawString(54, 35, "Y");} else {Display_drawString(54, 35, "N");}

  Display_setColor(GREEN);
  Display_drawString( 0,19, String(1000/RxPeriod) + "Hz");

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 8;        // wait before displaying next count
}

// --------------------------------------------------------------------------------

void Display_SafeMode() {
  // Display SafeMode + data
  DM_SafeMode = DispMode; // note Display Mode assignment
  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(63, 2, "SafeMode " + String(SafeMode));

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(35,14, "PchAC:");    Display_drawString(97,14, "PchBD:");
  Display_drawString(35,26, "SbSp:");    Display_drawString(97,26, "SbSp:");
  Display_drawString(35,38, "Err:");     Display_drawString(97,38, "Err:");
  Display_drawString(35,50, "AC-Op:");  Display_drawString(97,50, "BD-Op:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(36,14, String(PitchAC,1) + Deg);      Display_drawString(98,14, String(PitchBD,1) + Deg);
  Display_drawString(36,26, String(PtchACSbSp) + Deg);   Display_drawString(98,26, String(PtchBDSbSp) + Deg);
  Display_drawString(36,38, String(PtchACPidErr) + Deg); Display_drawString(98,38, String(PtchBDPidErr) + Deg);
  Display_drawString(36,50, String((int)PtchAC_pid_output));  Display_drawString(98,50, String((int)PtchBD_pid_output));

  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();
  DispDel = 5; // update every 160ms
}

// --------------------------------------------------------------------------------

void Display_SaPtchPtch() {
  // Display PitchAC & PicthBD relative to start angles
  DM_SaPtchPtch = DispMode;   // note Display Mode assignment
  if (!DispEn) {return;}      // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111        2222222222   3333333333
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  // Y64 = 2,16,6,16,6,16,2
  Display_drawString(64, 2, "Start Angles");
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(62,20, "PitchAC:");
  Display_drawString(62,33, "PitchBD:");
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(66,20, String((PitchAC - PtchACSbSpSa),1) + Deg);
  Display_drawString(66,33, String((PitchBD - PtchBDSbSpSa),1) + Deg);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();                        // display this immediately
  DispDel = 3;                              // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_SpiritLevel() {
  // Display spirit level data
  DM_SpiritLvl = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111  22222222  33333333  44444444  55555555  66666666
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(63, 2, "Spirit Level");

  Display_setFont(8);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 40,14, "ACErr:");
  Display_drawString(102,14, "ACErrAbs:");
  Display_drawString( 40,24, "BDErr:");
  Display_drawString(102,24, "BDErrAbs:");
  Display_drawString( 63,34, "AccAbsAng:");
  Display_drawString( 63,44, "LedAng:");
  Display_drawString( 40,54, "LedPnt:");
  Display_drawString(102,54, "LedNum:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 41,14, String(ACErr));
  Display_drawString(103,14, String(ACErrAbs));
  Display_drawString( 41,24, String(BDErr));
  Display_drawString(103,24, String(BDErrAbs));
  Display_drawString( 64,34, String(AccAbsAng));
  Display_drawString( 64,44, String(LedAng));
  Display_drawString( 41,54, String(LedPnt));
  Display_drawString(103,54, String(LedNum));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 5;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Status_A() {
  // Display status A
  DM_Status_A = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111   2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Status A");

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 63,15, "I2C Err:");
  Display_drawString(110,15, "Cnt:");
  Display_drawString( 63,27, "MPU Nop:");
  Display_drawString( 63,39, "WiFi:");
  Display_drawString( 63,51, "freeTime:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  if (!I2C_Err) {Display_drawString(65,15, "OK");} else {Display_drawString(65,15, "Yes");}
  Display_drawString(112,15, String(I2C_ErrCnt));
  Display_drawString( 65,27, String(MPU_Crash));
  if (WiFiConnected) {Display_drawString(65,39, "Connected");} else {Display_drawString(65,39, "-----");}
  Display_drawString( 65,51, String(freeTime));
  Display_drawString( 95,51, String(freeTime/40) + "%");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 3;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Status_B() {
  // Display status B
  DM_Status_B = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111   2222222222  3333333333  4444444444  5555555555
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Status B");

  Display_setFont(8);
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 63,15, "Main Mode:");
  Display_drawString(115,15, "Task:");
  Display_drawString( 63,27, "LedMode:");
  Display_drawString(115,27, "Task:");
  Display_drawString( 63,39, "Upright:");
  Display_drawString(115,39, "Upside:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 65,15, String(MainMode));
  Display_drawString(117,15, String(ModeTask));
  Display_drawString( 65,27, String(LedMode));
  Display_drawString(117,27, String(LED_Task));
  Display_drawString( 65,39, String(Upright));
  Display_drawString(117,39, String(Upside));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 4;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Text1x16(String zLine0,int16_t zDel) {
  // display one line of text in 16 font
  // 0123456789012345678901234567890123456789012345678901234567890123
  //                        1111111111111111
  Display_clear();    // remove current display content
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,23, zLine0);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = zDel;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Text2(String zT0,String zT1) {
  // display two lines of text of equal height
  if (DispLock) {return;} // prompts are disabled in DispLock mode

  Display_clear();
  Display_setBrightness(255);   // ensure display is bright
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //               1111111111111111    2222222222222222
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 14, zT0);
  Display_drawString(64, 34, zT1);
  if (Border) {Display_drawRect(0,0,128,64); Display_drawRect(1,1,126,62);} // thick boarder

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 50;       // wait 2 sec before display next screen
}

// --------------------------------------------------------------------------------

void Display_Text2x16(String zLine0,String zLine1,int16_t zDel) {
  // display two lines of text in 16 font
  // 0123456789012345678901234567890123456789012345678901234567890123
  //            1111111111111111    2222222222222222
  Display_clear();    // remove current display content
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,11, zLine0);
  Display_drawString(64,31, zLine1);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = zDel;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_Text2S16(String zT0,String zT1) {
  // display two lines of text, in 16 font
  if (DispLock) {return;} // prompts are disabled in DispLock mode

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //            1111111111111111    2222222222222222
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 11, zT0);
  Display_drawString(64, 31, zT1);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 25;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void Display_Text2S24(String zT0,String zT1) {
  // display two lines of text, 1st line is large
  if (DispLock) {return;} // prompts are disabled in DispLock mode

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //         111111111111111111111111        2222222222222222
  Display_setFont(24);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64,  8, zT0);
  Display_setFont(16);
  Display_drawString(64, 40, zT1);

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 25;       // wait 1 sec before display next screen
}

// --------------------------------------------------------------------------------

void Display_Warning() {
  // Display warning prior to servo movement
  if (!DispMon) {return;}
  
  Display_clear();    // remove current display content
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  // 111111111111111111111111   2222222222222222   3333333333333333
  Display_setFont(24);
  Display_setColor(RED);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 0, "WARNING");
  Display_setFont(16);
  Display_drawString(64,27, "Movement");
  Display_drawString(64,46, "Imminent!");
  Display_setColor(WHITE);
  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display immediately
  DispDel = 50;       // wait 2s before displaying next count
}

// --------------------------------------------------------------------------------

void Display_WheelTest() {
  // Display wheel test actions
  DM_WheelTest = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  Display_clear();
  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111   2222222222  3333333333  4444444444
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Wheel Test");
  Display_drawString(64,39, ModeStr$);

  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString( 41,15, "PwmAC:");
  Display_drawString(105,15, "PwmBD:");
  Display_drawString( 63,27, "Task:");

  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString( 42,15, String(PWM_AC));
  Display_drawString(106,15, String(PWM_BD));
  Display_drawString( 64,27, String(ModeTask));

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 3;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void Display_WiFi() {
  // display WiFi control
  DM_WiFi = DispMode; // note Display Mode assignment
  if (!DispEn) {return;}  // reading assignments in getDispRef()

  // Y64 vertical text spacing, from Y=0 to Y=63
  // 0123456789012345678901234567890123456789012345678901234567890123
  //   1111111111111111    2222222222     3333333333    4444444444
  Display_clear();    // remove current display content
  String zC = "1"; String zZ = "1";
  Display_setFont(16);
  Display_setTextAlignment(TEXT_ALIGN_CENTER);
  Display_drawString(64, 2, "Wi-Fi");
  Display_setFont(10);
  Display_setTextAlignment(TEXT_ALIGN_LEFT);
  Display_drawString(10,22, "JX:" + String(JoyX));
  Display_drawString(50,22, "JY:" + String(JoyY));
  if ((CZ & 2) == 0) {zC = "0";}
  if ((CZ & 1) == 0) {zZ = "0";}
  Display_drawString(90,22, "CZ: " + zC + zZ);
  if (WiiType == 'C') {
    Display_drawString(10,37, "LX:" + String(JoyLX));
    Display_drawString(50,37, "LY:" + String(JoyLY));
  }
  Display_setTextAlignment(TEXT_ALIGN_RIGHT);
  Display_drawString(126, 51, "Bat: " + String(BatVfp,1) + "v  " + String(BatPc) + "%");

  if (TEST) {Display_drawRect(0,0,128,64);} // draw 'Test' box outline
  Display_display();  // display this immediately
  DispDel = 3;        // wait 200ms before displaying next count
}

// --------------------------------------------------------------------------------

void getDispRef() {
  // Called during setup to extract all of the display screen references
  // This calls Disp_40ms multiple times, with DispEnd = false
  // It is called when entering and leaving TEST mode as it has differnt display options
  int16_t zDM = DispMode;         // preserve DispMode
  DispCnt = 0; DispDel = 0; DispClr = false; DispDisp = false; // clear flags
  DispEn = false;                 // disable the generation of display data
  setDMs();                       // set all DM references to their default value -99
  DispMode = 0; Display_40ms();   // read the default screen and get DM_Min and DM_Max
  for (DispMode = DM_Min;DispMode <= DM_Max; DispMode++) {
    if (DispMode != 0) {Display_40ms();} // extract all of the other references
  }
  DispMode = zDM;                 // restore DispMode
  DispEn = true;                  // enable the generation of display data
  DispDel = 1;
}

// --------------------------------------------------------------------------------

