/*
  DotStar GFX functions:

  GFX_Bar(P,X,Y,W,H)            - draw a solid bar on panel P at X,Y, width W, height H in 32-bit DotColr
  GFX_Box(P,X,Y,W,H)            - draw a box on panel P at X,Y, width W, height H in 32-bit DotColr
  GFX_ClrXY(X,Y)                - clears a pixel on LedFrame to the current background colour BakColr
  GFX_ClrXYP(F,X,Y)             - clears a pixel on frame F to the current background colour BakColr
  GFX_ClrXY(P,X,Y)              - clears a pixel on panel P to the 32-bit DotColr
  GFX_CopyPanel(PF,PT)          - copy the contents of panel PF to panel PT
  GFX_CopyPanelFlipV(PF,PT)     - copy the contents of panel PF to panel PT, flipped vertically
  Gfx_DrawChar(F,X,Y,C,M)       - draws a single character C on frame F, at X,Y using mode M
  Gfx_DrawGFX(F,X,Y,C,M)        - draws a single GFX character C on frame F, at X,Y using mode M
  GFX_GetXY(X,Y)                - returns the pixel pointer for X,Y in range 0 - 63
  GFX_GetColor(P,X,Y)           - puts the colour of pixel at X,Y on panel P in 32-bit DotColr
  GFX_FillColm(P,C,Col32)       - fills column C on panel P with 32-bit colour Col32
  GFX_FillColor(P,C)            - fills panel P with 32-bit colour C
  GFX_FillRGB(P,R,G,B)          - fills panel P with RGB colours
  GFX_FillRow(P,R,Col32)        - fills row R on panel P with 32-bit colour Col32
  GFX_Line(P,X0,Y0,X1,Y1)       - draw a line on panel P from X0,Y0 to X1,Y1 in 32-bit DotColr
  GFX_NewCol(P,C)               - change all pixels in panel P to new colour C
  GFX_RollDwn(P)                - roll the contents of panel P down vertically
  GFX_RollUp(P)                 - roll the contents of panel P up vertically
  Gfx_SetBakXY(X,Y)             - sets a pixel on the current frame to the BakColr colours
  Gfx_SetXY(X,Y)                - sets a pixel on the current frame to the DotColr colours
  Gfx_SetXYRGB(X,Y)             - sets a pixel on the current frame to the current RGB colours, ColR, ColG, ColB
*/

// ---------------------------------------------------------------------------------

void GFX_Bar(uint8_t zP,int zX,int zY,int zW,int zH) {
  // Draw a solid bar on panel zP at zX,zY, width zW, height zH in 32-bit DotColr
  int zX1 = zX + zW - 1; int zY1 = zY + zH - 1;
  // Serial.println(String(zX) + "\t" + String(zY) + "\t" + String(zX1) + "\t" + String(zY1));
  for (int16_t zR = 0;zR < zH; zR++) {
    GFX_Line(zP,zX,zY,zX1,zY1); zY++;
  }
}

// ---------------------------------------------------------------------------------

void GFX_Box(uint8_t zP,int zX,int zY,int zW,int zH) {
  // Draw a box on the current screen at zX,zY, width zW, height zH
  int zX1 = zX + zW - 1; int zY1 = zY + zH - 1;
  // Serial.println(String(zX) + "\t" + String(zY) + "\t" + String(zX1) + "\t" + String(zY1));
  GFX_Line(zP,zX,zY,zX1,zY);
  GFX_Line(zP,zX,zY,zX,zY1);
  GFX_Line(zP,zX1,zY,zX1,zY1);
  GFX_Line(zP,zX,zY1,zX1,zY1);
}

// ---------------------------------------------------------------------------------

void GFX_ClrXY(int zX, int zY) {
  // Clears a pixel on LedFrame to the current background colour BakColr
  // GFX co-ordinate system is 0,0 at top left corner
  // Check values are within expected limits
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  DotXYP = (8 * zX) + zY;
  switch (LedFrame) {
    case 0: Strip0.setPixelColor(DotXYP,BakColr); break;
    case 1: StripA.setPixelColor(DotXYP,BakColr); break;
    case 2: StripB.setPixelColor(DotXYP,BakColr); break;
    case 3: StripC.setPixelColor(DotXYP,BakColr); break;
    case 4: StripD.setPixelColor(DotXYP,BakColr); break;
  }
}

// ---------------------------------------------------------------------------------

void GFX_ClrXYP(uint8_t zP,int zX, int zY) {
  // Clears a pixel on frame zP to the current background colour BakColr
  // GFX co-ordinate system is 0,0 at top left corner
  // Check values are within expected limits
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  DotXYP = (8 * zX) + zY;
  switch (zP) {
    case 0: Strip0.setPixelColor(DotXYP,BakColr); break;
    case 1: StripA.setPixelColor(DotXYP,BakColr); break;
    case 2: StripB.setPixelColor(DotXYP,BakColr); break;
    case 3: StripC.setPixelColor(DotXYP,BakColr); break;
    case 4: StripD.setPixelColor(DotXYP,BakColr); break;
  }
}

// --------------------------------------------------------------------------------

void GFX_CopyPanel(uint8_t zPF,uint8_t zPT) {
  // Copy panel zPF to panel zPT
  if (zPF == zPT) {return;} // pointless
  switch (zPT) {
    case 1: for (int zI = 0;zI < 64; zI++) {
      // Copying to panel 'A'
      switch (zPF) {
        case 2: StripA.setPixelColor(zI,StripB.getPixelColor(zI)); break;
        case 3: StripA.setPixelColor(zI,StripC.getPixelColor(zI)); break;
        case 4: StripA.setPixelColor(zI,StripD.getPixelColor(zI)); break;
        default: break;
      }
    } break;
    case 2: for (int zI = 0;zI < 64; zI++) {
      // Copying to panel 'B'
      switch (zPF) {
        case 1: StripB.setPixelColor(zI,StripA.getPixelColor(zI)); break;
        case 3: StripB.setPixelColor(zI,StripC.getPixelColor(zI)); break;
        case 4: StripB.setPixelColor(zI,StripD.getPixelColor(zI)); break;
        default: break;
      }
    } break;
    case 3: for (int zI = 0;zI < 64; zI++) {
      // Copying to panel 'C'
      switch (zPF) {
        case 1: StripC.setPixelColor(zI,StripA.getPixelColor(zI)); break;
        case 2: StripC.setPixelColor(zI,StripB.getPixelColor(zI)); break;
        case 4: StripC.setPixelColor(zI,StripD.getPixelColor(zI)); break;
        default: break;
      }
    } break;
    case 4: for (int zI = 0;zI < 64; zI++) {
      // Copying to panel 'D'
      switch (zPF) {
        case 1: StripD.setPixelColor(zI,StripA.getPixelColor(zI)); break;
        case 2: StripD.setPixelColor(zI,StripB.getPixelColor(zI)); break;
        case 3: StripD.setPixelColor(zI,StripC.getPixelColor(zI)); break;
        default: break;
      }
    } break;
  }
}

// --------------------------------------------------------------------------------

void GFX_CopyPanelFlipH(uint8_t zPF,uint8_t zPT) {
  // Copy panel zPF to panel zPT flipped horizontally
  if (zPF == zPT) {return;} // pointless
  int zR = 0; int zC = 56;
  switch (zPT) {
    case 1: 
      for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'A'
        switch (zPF) {
          case 2: StripA.setPixelColor(zI,StripB.getPixelColor(zR+zC)); break;
          case 3: StripA.setPixelColor(zI,StripC.getPixelColor(zR+zC)); break;
          case 4: StripA.setPixelColor(zI,StripD.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR++; if (zR > 7) {zR = 0; zC-= 8;}
      } break;
    case 2:
       for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'B'
        switch (zPF) {
          case 1: StripB.setPixelColor(zI,StripA.getPixelColor(zR+zC)); break;
          case 3: StripB.setPixelColor(zI,StripC.getPixelColor(zR+zC)); break;
          case 4: StripB.setPixelColor(zI,StripD.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR++; if (zR > 7) {zR = 0; zC-= 8;}
      } break;
    case 3:
       for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'C'
        switch (zPF) {
          case 1: StripC.setPixelColor(zI,StripA.getPixelColor(zR+zC)); break;
          case 2: StripC.setPixelColor(zI,StripB.getPixelColor(zR+zC)); break;
          case 4: StripC.setPixelColor(zI,StripD.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR++; if (zR > 7) {zR = 0; zC-= 8;}
      } break;
    case 4:
      for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'D'
        switch (zPF) {
          case 1: StripD.setPixelColor(zI,StripA.getPixelColor(zR+zC)); break;
          case 2: StripD.setPixelColor(zI,StripB.getPixelColor(zR+zC)); break;
          case 3: StripD.setPixelColor(zI,StripC.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR++; if (zR > 7) {zR = 0; zC-= 8;}
      } break;
  }
}

// --------------------------------------------------------------------------------

void GFX_CopyPanelFlipV(uint8_t zPF,uint8_t zPT) {
  // Copy panel zPF to panel zPT flipped vertically
  if (zPF == zPT) {return;} // pointless
  int zR = 7; int zC = 0;
  switch (zPT) {
    case 1: 
      for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'A'
        switch (zPF) {
          case 2: StripA.setPixelColor(zI,StripB.getPixelColor(zR+zC)); break;
          case 3: StripA.setPixelColor(zI,StripC.getPixelColor(zR+zC)); break;
          case 4: StripA.setPixelColor(zI,StripD.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR--; if (zR < 0) {zR = 7; zC+= 8;}
      } break;
    case 2:
       for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'B'
        switch (zPF) {
          case 1: StripB.setPixelColor(zI,StripA.getPixelColor(zR+zC)); break;
          case 3: StripB.setPixelColor(zI,StripC.getPixelColor(zR+zC)); break;
          case 4: StripB.setPixelColor(zI,StripD.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR--; if (zR < 0) {zR = 7; zC+= 8;}
      } break;
    case 3:
       for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'C'
        switch (zPF) {
          case 1: StripC.setPixelColor(zI,StripA.getPixelColor(zR+zC)); break;
          case 2: StripC.setPixelColor(zI,StripB.getPixelColor(zR+zC)); break;
          case 4: StripC.setPixelColor(zI,StripD.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR--; if (zR < 0) {zR = 7; zC+= 8;}
      } break;
    case 4:
      for (int zI = 0;zI < 64; zI++) {
        // Copying to panel 'D'
        switch (zPF) {
          case 1: StripD.setPixelColor(zI,StripA.getPixelColor(zR+zC)); break;
          case 2: StripD.setPixelColor(zI,StripB.getPixelColor(zR+zC)); break;
          case 3: StripD.setPixelColor(zI,StripC.getPixelColor(zR+zC)); break;
          default: break;
        }
        zR--; if (zR < 0) {zR = 7; zC+= 8;}
      } break;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_DrawChar(uint8_t zF,int zX,int zY,uint8_t zC,uint8_t zM) {
  // Loads and draws a single character zC on frame zF, at zX,zY using mask zM.
  // You would normally clear the face before doing this, and set LedFrame.
  // zM is a modifying mask
  // XXXXXX00 - don't modify, left-justify from zX
  // XXXXXX11 - centre on zX
  // XXXXXX01 - right-justify from zX
  // XXXX00XX - don't reset non-set pixels
  // XXXX01XX - reset non-set pixels to OFF
  // XXXX10XX - set non-set pixels to background colour BAK_Red, etc
  // this code sets and resets pixels
  LedFrame = zF;
  loadChar(zC);       // put character font into array
  if ((zM & 0b00000011) > 0) {
    // perform character justification, by changing zX
         if ((zM & 0b00000011) == 0b00000011) {zX = zX - (GW/2);}  // centre on zX
    else if ((zM & 0b00000001) == 0b00000001) {zX = zX - GW + 1;}  // right justify from zX
  }
  for (int zA = 0;zA <= 7;zA++) {
    // go through each row of the character array
    // GW is the width of the current character
    if ((zM & 0b00001100) == 0) {
      // set pixels, but not background, character is transparent
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);}}
    } else if ((zM & 0b00001100) == 0b00000100) {
      // set pixels and clear non-set pixels, removing background
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {GFX_ClrXY(zX,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {GFX_ClrXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {GFX_ClrXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {GFX_ClrXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {GFX_ClrXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {GFX_ClrXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {GFX_ClrXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {GFX_ClrXY(zX+7,zY);}}
    } else if ((zM & 0b00001100) == 0b00001000) {
      // set char pixels, and set non-pixels to background
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {Gfx_SetBakXY(zX,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {Gfx_SetBakXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {Gfx_SetBakXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {Gfx_SetBakXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {Gfx_SetBakXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {Gfx_SetBakXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {Gfx_SetBakXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {Gfx_SetBakXY(zX+7,zY);}}
    }
    zY++;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_DrawGFX(uint8_t zF,int zX,int zY,uint8_t zC,uint8_t zM) {
  // Loads and draws a single GFX character zC on frame zF, at zX,zY using mask zM.
  // You would normally clear the face before doing this, and set LedFrame.
  // zM is a modifying mask
  // XXXXXX00 - don't modify, left-justify from zX
  // XXXXXX11 - centre on zX
  // XXXXXX01 - right-justify from zX
  // XXXX00XX - don't reset non-set pixels
  // XXXX01XX - reset non-set pixels to OFF
  // XXXX10XX - set non-set pixels to background colour ColR, etc
  // this code sets and resets pixels
  LedFrame = zF;
  loadGfx(zC);       // put character font into array
  if ((zM & 0b00000011) > 0) {
    // perform character justification, by changing zX
         if ((zM & 0b00000011) == 0b00000011) {zX = zX - (GW/2);}  // centre on zX
    else if ((zM & 0b00000001) == 0b00000001) {zX = zX - GW + 1;}  // right justify from zX
  }
  for (int zA = 0;zA <= 7;zA++) {
    // go through each row of the character array
    // GW is the width of the current character
    if ((zM & 0b00001100) == 0) {
      // set pixels, but not background, character is transparent
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);}}
    } else if ((zM & 0b00001100) == 0b00000100) {
      // set pixels and clear non-set pixels, removing background
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {GFX_ClrXY(zX,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {GFX_ClrXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {GFX_ClrXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {GFX_ClrXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {GFX_ClrXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {GFX_ClrXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {GFX_ClrXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {GFX_ClrXY(zX+7,zY);}}
    } else if ((zM & 0b00001100) == 0b00001000) {
      // set char pixels, and set non-pixels to background
      if (GW > 0) {if (GC[zA] & 0b10000000) {Gfx_SetXY(zX  ,zY);} else {Gfx_SetBakXY(zX,zY);}}
      if (GW > 1) {if (GC[zA] & 0b01000000) {Gfx_SetXY(zX+1,zY);} else {Gfx_SetBakXY(zX+1,zY);}}
      if (GW > 2) {if (GC[zA] & 0b00100000) {Gfx_SetXY(zX+2,zY);} else {Gfx_SetBakXY(zX+2,zY);}}
      if (GW > 3) {if (GC[zA] & 0b00010000) {Gfx_SetXY(zX+3,zY);} else {Gfx_SetBakXY(zX+3,zY);}}
      if (GW > 4) {if (GC[zA] & 0b00001000) {Gfx_SetXY(zX+4,zY);} else {Gfx_SetBakXY(zX+4,zY);}}
      if (GW > 5) {if (GC[zA] & 0b00000100) {Gfx_SetXY(zX+5,zY);} else {Gfx_SetBakXY(zX+5,zY);}}
      if (GW > 6) {if (GC[zA] & 0b00000010) {Gfx_SetXY(zX+6,zY);} else {Gfx_SetBakXY(zX+6,zY);}}
      if (GW > 7) {if (GC[zA] & 0b00000001) {Gfx_SetXY(zX+7,zY);} else {Gfx_SetBakXY(zX+7,zY);}}
    }
    zY++;
  }
}

// --------------------------------------------------------------------------------

void GFX_FillColm(uint8_t zP,uint8_t zX,uint32_t zC) {
  // fills column zX on panel zP with 32-bit colour zC
  zX = 8 * zX;
  switch (zP) {
    case 1: for (int zY = 0; zY < 8; zY++) {DotXYP = zX + zY; StripA.setPixelColor(DotXYP,zC);} break;
    case 2: for (int zY = 0; zY < 8; zY++) {DotXYP = zX + zY; StripB.setPixelColor(DotXYP,zC);} break;
    case 3: for (int zY = 0; zY < 8; zY++) {DotXYP = zX + zY; StripC.setPixelColor(DotXYP,zC);} break;
    case 4: for (int zY = 0; zY < 8; zY++) {DotXYP = zX + zY; StripD.setPixelColor(DotXYP,zC);} break;
  }
}

// --------------------------------------------------------------------------------

void GFX_FillColor(int8_t zP,uint32_t zC) {
  // Fils a DotStar panel zP with 32-bit colour zC
  switch (zP) {
    case 1: StripA.fill(zC); break;
    case 2: StripB.fill(zC); break;
    case 3: StripC.fill(zC); break;
    case 4: StripD.fill(zC); break;
  }
}

// --------------------------------------------------------------------------------

void GFX_FillRow(uint8_t zP,uint8_t zR,uint32_t zC) {
  // fills row R on panel P with 32-bit colour Col32
  switch (zP) {
    case 1: for (int zX = 0; zX < 8; zX++) {DotXYP = (8 * zX) + zR; StripA.setPixelColor(DotXYP,zC);} break;
    case 2: for (int zX = 0; zX < 8; zX++) {DotXYP = (8 * zX) + zR; StripB.setPixelColor(DotXYP,zC);} break;
    case 3: for (int zX = 0; zX < 8; zX++) {DotXYP = (8 * zX) + zR; StripC.setPixelColor(DotXYP,zC);} break;
    case 4: for (int zX = 0; zX < 8; zX++) {DotXYP = (8 * zX) + zR; StripD.setPixelColor(DotXYP,zC);} break;
  }
}

// --------------------------------------------------------------------------------

void GFX_FillRGB(int8_t zP,uint32_t zR,uint32_t zG,uint32_t zB) {
  // Fills LED ring or DotStar panel zP with RGB colours
  DotColr = (zR<<16) + (zG<<8) + zB;
  switch (zP) {
    case 0: Strip0.fill(DotColr); break;   // LED ring on micro plate
    case 1: StripA.fill(DotColr); break;
    case 2: StripB.fill(DotColr); break;
    case 3: StripC.fill(DotColr); break;
    case 4: StripD.fill(DotColr); break;
  }
}

// --------------------------------------------------------------------------------

uint8_t GFX_getXY(uint8_t zX,uint8_t zY) {
  // returns the pixel pointer for zX,zY in range 0 - 63
  return (zX * 8) + zY;
}

// --------------------------------------------------------------------------------

void GFX_GetColor(int8_t zP,uint8_t zX,uint8_t zY) {
  // Returns the colour of pixel zX,zY on panel zP as 32-bit DotColr
  // If panel 0 is specified then zX is used as the pixel ponter
  uint8_t zI = GFX_getXY(zX,zY);  // get the pixel pointer
  switch (zP) {
    case 0: DotColr = Strip0.getPixelColor(zX); break;  // ring on micro plate
    case 1: DotColr = StripA.getPixelColor(zI); break;  // panel 'A'
    case 2: DotColr = StripB.getPixelColor(zI); break;  // panel 'B'
    case 3: DotColr = StripC.getPixelColor(zI); break;  // panel 'C'
    case 4: DotColr = StripD.getPixelColor(zI); break;  // panel 'D'
  }
}

// ---------------------------------------------------------------------------------

void GFX_Line(uint8_t zP,int zX0,int zY0,int zX1,int zY1) {
  // Draw a line on panel zP, from zX0,zY0 to zX1,zY1 in 32-bit DotColr
  int zX; int zY; int zR2;
  if ((abs(zX1 - zX0)== 0) && (abs(zY1 - zY0) == 0)) {
    // Single pixel
    if ((zX0 >= 0) && (zX0 <= 7) && (zY0 >= 0) && (zY0 <= 7)) {
      DotXYP = (8 * zX0) + zY0;
      switch (zP) {
        case 1: StripA.setPixelColor(DotXYP,DotColr); break;
        case 2: StripB.setPixelColor(DotXYP,DotColr); break;
        case 3: StripC.setPixelColor(DotXYP,DotColr); break;
        case 4: StripD.setPixelColor(DotXYP,DotColr); break;
      }
    }
  } else if (abs(zX1 - zX0) > abs(zY1 - zY0)) {
    // Line is longer in X than Y
    // Ensure zX1 > zX0
    if (zX1 < zX0) {zX = zX1; zX1 = zX0; zX0 = zX; zY = zY1; zY1 = zY0; zY0 = zY;} // ensure zY1 is largest
    zR2 = (zX1 - zX0)/2; if (zY1 < zY0) {zR2 = -zR2;}
    for (zX = zX0;zX <= zX1;zX++) {
      if ((zX >= 0) && (zX <= 7)) {
        // only set point if within face area
        zY = zY0 + (((zY1 - zY0) * (zX - zX0)) + zR2)/(zX1 - zX0);
        if ((zY >= 0) && (zY <= 7)) {
          DotXYP = (8 * zX) + zY;
          switch (zP) {
            case 1: StripA.setPixelColor(DotXYP,DotColr); break;
            case 2: StripB.setPixelColor(DotXYP,DotColr); break;
            case 3: StripC.setPixelColor(DotXYP,DotColr); break;
            case 4: StripD.setPixelColor(DotXYP,DotColr); break;
          }
        }
      }
    }
  } else {
    // line is longer in Y than X
    // ensure zY1 > zY0
    if (zY1 < zY0) {zY = zY1; zY1 = zY0; zY0 = zY; zX = zX1; zX1 = zX0; zX0 = zX;} // ensure zX1 is largest
    zR2 = (zY1 - zY0)/2; if (zX1 < zX0) {zR2 = -zR2;}
    for (zY = zY0;zY <= zY1;zY++) {
      if ((zY >= 0) && (zY <= 7)) {
        // only set point if within face area
        zX = zX0 + (((zX1 - zX0) * (zY - zY0))/(zY1 - zY0));
        if ((zX >= 0) && (zX <= 7)) {
          DotXYP = (8 * zX) + zY;
          switch (zP) {
            case 1: StripA.setPixelColor(DotXYP,DotColr); break;
            case 2: StripB.setPixelColor(DotXYP,DotColr); break;
            case 3: StripC.setPixelColor(DotXYP,DotColr); break;
            case 4: StripD.setPixelColor(DotXYP,DotColr); break;
          }
        }
      }
    }
  }
}

// --------------------------------------------------------------------------------

void GFX_NewCol(uint8_t zP,uint32_t zC) {
  // Converts lit pixels to colour zC
  int zI;
  switch (zP) {
    case 0: // micro plate ring, only 16 LEDs
      for (zI = 0;zI < 16;zI+= 1) {if (Strip0.getPixelColor(zI) > 0) {Strip0.setPixelColor(zI,zC);}} break;
    case 1: // panel 'A'
      for (zI = 0;zI < 64;zI+= 1) {if (StripA.getPixelColor(zI) > 0) {StripA.setPixelColor(zI,zC);}} break;
    case 2: // panel 'B'
      for (zI = 0;zI < 64;zI+= 1) {if (StripB.getPixelColor(zI) > 0) {StripB.setPixelColor(zI,zC);}} break;
    case 3: // panel 'C'
      for (zI = 0;zI < 64;zI+= 1) {if (StripC.getPixelColor(zI) > 0) {StripC.setPixelColor(zI,zC);}} break;
    case 4: // panel 'D'
      for (zI = 0;zI < 64;zI+= 1) {if (StripD.getPixelColor(zI) > 0) {StripD.setPixelColor(zI,zC);}} break;
  }
}

// --------------------------------------------------------------------------------

void GFX_RollDwn(uint8_t zP) {
  // Roll the contents of panel zP down vertically
  uint32_t zPix; int zC; int zR;
  switch (zP) {
    case 1: // panel 'A'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripA.getPixelColor(zC+7);
        for (zR = 7;zR > 0;zR--) {StripA.setPixelColor(zC+zR,StripA.getPixelColor(zC+zR-1));}
        StripA.setPixelColor(zC+zR,zPix);
      } break;
    case 2: // panel 'B'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripB.getPixelColor(zC+7);
        for (zR = 7;zR > 0;zR--) {StripB.setPixelColor(zC+zR,StripB.getPixelColor(zC+zR-1));}
        StripB.setPixelColor(zC+zR,zPix);
      } break;
    case 3: // panel 'C'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripC.getPixelColor(zC+7);
        for (zR = 7;zR > 0;zR--) {StripC.setPixelColor(zC+zR,StripC.getPixelColor(zC+zR-1));}
        StripC.setPixelColor(zC+zR,zPix);
      } break;
    case 4: // panel 'D'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripD.getPixelColor(zC+7);
        for (zR = 7;zR > 0;zR--) {StripD.setPixelColor(zC+zR,StripD.getPixelColor(zC+zR-1));}
        StripD.setPixelColor(zC+zR,zPix);
      } break;
  }
}

// --------------------------------------------------------------------------------

void GFX_RollUp(uint8_t zP) {
  // Roll the contents of panel zP up vertically
  uint32_t zPix; int zC; int zR;
  switch (zP) {
    case 1: // panel 'A'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripA.getPixelColor(zC);
        for (zR = 0;zR < 7;zR++) {StripA.setPixelColor(zC+zR,StripA.getPixelColor(zC+zR+1));}
        StripA.setPixelColor(zC+zR,zPix);
      } break;
    case 2: // panel 'B'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripB.getPixelColor(zC);
        for (zR = 0;zR < 7;zR++) {StripB.setPixelColor(zC+zR,StripB.getPixelColor(zC+zR+1));}
        StripB.setPixelColor(zC+zR,zPix);
      } break;
    case 3: // panel 'C'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripC.getPixelColor(zC);
        for (zR = 0;zR < 7;zR++) {StripC.setPixelColor(zC+zR,StripC.getPixelColor(zC+zR+1));}
        StripC.setPixelColor(zC+zR,zPix);
      } break;
    case 4: // panel 'D'
      // Rool each column
      for (zC = 0;zC < 57;zC+= 8) {
        // Roll each column
        zPix = StripD.getPixelColor(zC);
        for (zR = 0;zR < 7;zR++) {StripD.setPixelColor(zC+zR,StripD.getPixelColor(zC+zR+1));}
        StripD.setPixelColor(zC+zR,zPix);
      } break;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_SetBakXY(int zX,int zY) {
  // Sets a pixel on the current frame to the BakColr colours
  // Co-ordinate system is 0,0 at top left corner
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  DotXYP = zY + (8 * zX);
  switch (LedFrame) {
    case 0: Strip0.setPixelColor(DotXYP,BakColr); break;
    case 1: StripA.setPixelColor(DotXYP,BakColr); break;
    case 2: StripB.setPixelColor(DotXYP,BakColr); break;
    case 3: StripC.setPixelColor(DotXYP,BakColr); break;
    case 4: StripD.setPixelColor(DotXYP,BakColr); break;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_SetXY(int zX,int zY) {
  // Sets a pixel on the current frame to the DotColr colours
  // Co-ordinate system is 0,0 at top left corner
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  DotXYP = zY + (8 * zX);
  switch (LedFrame) {
    case 0: Strip0.setPixelColor(DotXYP,DotColr); break;
    case 1: StripA.setPixelColor(DotXYP,DotColr); break;
    case 2: StripB.setPixelColor(DotXYP,DotColr); break;
    case 3: StripC.setPixelColor(DotXYP,DotColr); break;
    case 4: StripD.setPixelColor(DotXYP,DotColr); break;
  }
}

// ---------------------------------------------------------------------------------

void Gfx_SetXYRGB(int zX,int zY) {
  // Sets a pixel on the current frame to the current RGB colours
  // Co-ordinate system is 0,0 at top left corner
  if ((zX < 0) || (zX > 7) || (zY < 0) || (zY > 7)) {return;}
  
  DotXYP = zY + (8 * zX);
  switch (LedFrame) {
    case 0: Strip0.setPixelColor(DotXYP,ColR,ColG,ColB); break;
    case 1: StripA.setPixelColor(DotXYP,ColR,ColG,ColB); break;
    case 2: StripB.setPixelColor(DotXYP,ColR,ColG,ColB); break;
    case 3: StripC.setPixelColor(DotXYP,ColR,ColG,ColB); break;
    case 4: StripD.setPixelColor(DotXYP,ColR,ColG,ColB); break;
  }
}

// ---------------------------------------------------------------------------------

/*
  This section contains definitions of all of the characters used in the DotStars.
  Characters are defined by an 8-byte array.
  The array reference sets the vertical reference. ie. 0 = top row, Y = 0
  Bits in the byte define whether pixels are 1 = ON, 0 = OFF, X0 = MSB, X7 = LSB
  For each character the width GW and height GH are also specified 
*/

void loadChar(uint8_t zC) {
  // loads 8 bytes into the character array for use in graphics
  switch(zC) {
    case  0:  // < - task selection
      GC[0] = 0b00000000; GC[1] = 0b00010000; GC[2] = 0b00100000; GC[3] = 0b01000000;
      GC[4] = 0b01000000; GC[5] = 0b00100000; GC[6] = 0b00010000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case  1:  // > - task selection
      GC[0] = 0b00000000; GC[1] = 0b10000000; GC[2] = 0b01000000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b01000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case  2:  // |< - goto first task selection
      GC[0] = 0b00000000; GC[1] = 0b10010000; GC[2] = 0b10100000; GC[3] = 0b11000000;
      GC[4] = 0b11000000; GC[5] = 0b10100000; GC[6] = 0b10010000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case  3:  // >| - goto last task selection
      GC[0] = 0b00000000; GC[1] = 0b10010000; GC[2] = 0b01010000; GC[3] = 0b00110000;
      GC[4] = 0b00110000; GC[5] = 0b01010000; GC[6] = 0b10010000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case  4:  // [.]
      GC[0] = 0b00100100;
      GC[1] = 0b01100110;
      GC[2] = 0b11100111;
      GC[3] = 0b00011000;
      GC[4] = 0b00011000;
      GC[5] = 0b11100111;
      GC[6] = 0b01100110;
      GC[7] = 0b00100100;
      GW = 8; GH = 7; break;
    case  5:  // [:]
      GC[0] = 0b11100111;
      GC[1] = 0b11000011;
      GC[2] = 0b10000001;
      GC[3] = 0b00011000;
      GC[4] = 0b00011000;
      GC[5] = 0b10000001;
      GC[6] = 0b11000011;
      GC[7] = 0b11100111;
      GW = 8; GH = 7; break;

    case 32:  // SPACE
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00000000;
      GC[3] = 0b00000000;
      GC[4] = 0b00000000;
      GC[5] = 0b00000000;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 33:  // !
      GC[0] = 0b10000000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b10000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 1; GH = 7; break;
    case 34:  // "
      GC[0] = 0b10100000; GC[1] = 0b10100000; GC[2] = 0b10100000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 35:  // #
      GC[0] = 0b01010000; GC[1] = 0b01010000; GC[2] = 0b11111000; GC[3] = 0b01010000;
      GC[4] = 0b11111000; GC[5] = 0b01010000; GC[6] = 0b01010000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 36:  // $
      GC[0] = 0b00100000; GC[1] = 0b01111000; GC[2] = 0b10100000; GC[3] = 0b01110000;
      GC[4] = 0b00101000; GC[5] = 0b11110000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 37:  // %
      GC[0] = 0b11000000; GC[1] = 0b11000100; GC[2] = 0b00001000; GC[3] = 0b00010000;
      GC[4] = 0b00100000; GC[5] = 0b01000000; GC[6] = 0b10011000; GC[7] = 0b00011000;
      GW = 5; GH = 7; break;
    case 38:  // &
      GC[0] = 0b01100000; GC[1] = 0b10010000; GC[2] = 0b10100000; GC[3] = 0b01000000;
      GC[4] = 0b10101000; GC[5] = 0b10010000; GC[6] = 0b01101000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 39:  // '
      GC[0] = 0b11000000; GC[1] = 0b01000000; GC[2] = 0b10000000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 2; GH = 7; break;
    case 40:  // (
      GC[0] = 0b00100000; GC[1] = 0b01000000; GC[2] = 0b10000000; GC[3] = 0b10000000;
      GC[4] = 0b10000000; GC[5] = 0b01000000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 41:  // )
      GC[0] = 0b10000000; GC[1] = 0b01000000; GC[2] = 0b00100000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b01000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 42:  // *
      GC[0] = 0b00000000; GC[1] = 0b00100000; GC[2] = 0b10101000; GC[3] = 0b01110000;
      GC[4] = 0b10101000; GC[5] = 0b00100000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 43:  // +
      GC[0] = 0b00000000; GC[1] = 0b00100000; GC[2] = 0b00100000; GC[3] = 0b11111000;
      GC[4] = 0b00100000; GC[5] = 0b00100000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 44:  // ,
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b00000000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b11000000; GC[6] = 0b01000000; GC[7] = 0b10000000;
      GW = 2; GH = 7; break;
    case 45:  // -
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b00000000; GC[3] = 0b11111000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 46:  // .
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b00000000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b11000000; GC[7] = 0b11000000;
      GW = 2; GH = 7; break;
    case 47:  // /
      GC[0] = 0b00000000; GC[1] = 0b00001000; GC[2] = 0b00010000; GC[3] = 0b00100000;
      GC[4] = 0b01000000; GC[5] = 0b10000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;

    case 48:  // 0
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10011000; GC[3] = 0b10101000;
      GC[4] = 0b11001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 49:  // 1
      GC[0] = 0b00100000; GC[1] = 0b01100000; GC[2] = 0b10100000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b00100000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 50:  // 2
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b00001000; GC[3] = 0b00110000;
      GC[4] = 0b01000000; GC[5] = 0b10000000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 51:  // 3
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b00001000; GC[3] = 0b00110000;
      GC[4] = 0b00001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 52:  // 4
      GC[0] = 0b00010000; GC[1] = 0b00110000; GC[2] = 0b01010000; GC[3] = 0b10010000;
      GC[4] = 0b11111000; GC[5] = 0b00010000; GC[6] = 0b00010000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 53:  // 5
      GC[0] = 0b11111000; GC[1] = 0b10000000; GC[2] = 0b11110000; GC[3] = 0b00001000;
      GC[4] = 0b00001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 54:  // 6
      GC[0] = 0b00110000; GC[1] = 0b01000000; GC[2] = 0b10000000; GC[3] = 0b11110000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 55:  // 7
      GC[0] = 0b11111000; GC[1] = 0b00001000; GC[2] = 0b00010000; GC[3] = 0b00100000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b01000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 56:  // 8
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b01110000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 57:  // 9
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b01111000;
      GC[4] = 0b00001000; GC[5] = 0b00010000; GC[6] = 0b01100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
      
    case 58:  // :
      GC[0] = 0b00000000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b00000000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 1; GH = 7; break;
    case 59:  // ;
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b00000000; GC[3] = 0b01000000;
      GC[4] = 0b00000000; GC[5] = 0b01000000; GC[6] = 0b01000000; GC[7] = 0b10000000;
      GW = 2; GH = 7; break;
    case 60:  // <
      GC[0] = 0b00100000; GC[1] = 0b01000000; GC[2] = 0b10000000; GC[3] = 0b10000000;
      GC[4] = 0b10000000; GC[5] = 0b01000000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 61:  // =
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b11100000; GC[3] = 0b00000000;
      GC[4] = 0b11100000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 62:  // >
      GC[0] = 0b10000000; GC[1] = 0b01000000; GC[2] = 0b00100000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b01000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 63:  // ?
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b00001000; GC[3] = 0b00010000;
      GC[4] = 0b00100000; GC[5] = 0b00000000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 64:  // @
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10111000; GC[3] = 0b10101000;
      GC[4] = 0b10111000; GC[5] = 0b10000000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;

    case 65:  // A
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b11111000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 66:  // B
      GC[0] = 0b11110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b11110000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b11110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 67:  // C
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10000000; GC[3] = 0b10000000;
      GC[4] = 0b10000000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 68:  // D
      GC[0] = 0b11100000; GC[1] = 0b10010000;
      GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10010000;
      GC[6] = 0b11100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 69:  // E
      GC[0] = 0b11111000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b11110000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 70:  // F
      GC[0] = 0b11111000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b11110000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 71:  // G
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10000000; GC[3] = 0b10111000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 72:  // H
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b11111000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 73:  // I
      GC[0] = 0b11100000; GC[1] = 0b01000000; GC[2] = 0b01000000; GC[3] = 0b01000000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b11100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 74:  // J
      GC[0] = 0b00111000; GC[1] = 0b00010000; GC[2] = 0b00010000; GC[3] = 0b00010000;
      GC[4] = 0b10010000; GC[5] = 0b10010000; GC[6] = 0b01100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 75:  // K
      GC[0] = 0b10001000; GC[1] = 0b10010000; GC[2] = 0b10100000; GC[3] = 0b11000000;
      GC[4] = 0b10100000; GC[5] = 0b10010000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 76:  // L
      GC[0] = 0b10000000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b10000000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 77:  // M
      GC[0] = 0b10001000; GC[1] = 0b11011000; GC[2] = 0b10101000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 78:  // N
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b11001000; GC[3] = 0b10101000;
      GC[4] = 0b10011000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 79:  // O
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 80:  // P
      GC[0] = 0b11110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b11110000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 81:  // Q
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10101000; GC[6] = 0b01110000; GC[7] = 0b00001000;
      GW = 5; GH = 7; break;
    case 82:  // R
      GC[0] = 0b11110000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b11110000;
      GC[4] = 0b10100000; GC[5] = 0b10010000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 83:  // S
      GC[0] = 0b01110000; GC[1] = 0b10001000; GC[2] = 0b10000000; GC[3] = 0b01110000;
      GC[4] = 0b00001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 84:  // T
      GC[0] = 0b11111000; GC[1] = 0b00100000; GC[2] = 0b00100000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b00100000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 85:  // U
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 86:  // V
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b01010000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 87:  // W
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10101000;
      GC[4] = 0b10101000; GC[5] = 0b10101000; GC[6] = 0b01010000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 88:  // X
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b01010000; GC[3] = 0b00100000;
      GC[4] = 0b01010000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 89:  // Y
      GC[0] = 0b10001000; GC[1] = 0b10001000; GC[2] = 0b01010000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b00100000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 90:  // Z
      GC[0] = 0b11111000; GC[1] = 0b00001000; GC[2] = 0b00010000; GC[3] = 0b00100000;
      GC[4] = 0b01000000; GC[5] = 0b10000000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;

    case 91:  // [
      GC[0] = 0b11100000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b10000000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b11100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 92:  // forward slash
      GC[0] = 0b00000000; GC[1] = 0b10000000; GC[2] = 0b01000000; GC[3] = 0b00100000;
      GC[4] = 0b00010000; GC[5] = 0b00001000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 93:  // ]
      GC[0] = 0b11100000; GC[1] = 0b00100000; GC[2] = 0b00100000; GC[3] = 0b00100000;
      GC[4] = 0b00100000; GC[5] = 0b00100000; GC[6] = 0b11100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 94:  // ^
      GC[0] = 0b00100000; GC[1] = 0b01010000; GC[2] = 0b10001000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 95:  // _
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b00000000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 96:  // '
      GC[0] = 0b10000000; GC[1] = 0b01000000; GC[2] = 0b00000000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 2; GH = 7; break;

    case 97:  // a
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b01110000; GC[3] = 0b00001000;
      GC[4] = 0b01111000; GC[5] = 0b10001000; GC[6] = 0b01111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 98:  // b
      GC[0] = 0b10000000; GC[1] = 0b10000000; GC[2] = 0b11110000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b11110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 99:  // c
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b01110000; GC[3] = 0b10001000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b01111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 100:  // d
      GC[0] = 0b00001000; GC[1] = 0b00001000; GC[2] = 0b01111000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 101:  // e
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b01110000; GC[3] = 0b10001000;
      GC[4] = 0b11111000; GC[5] = 0b10000000; GC[6] = 0b01111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 102:  // f
      GC[0] = 0b00110000; GC[1] = 0b01000000; GC[2] = 0b11100000; GC[3] = 0b01000000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b01000000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case 103:  // g
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b01110000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b01111000; GC[6] = 0b00001000; GC[7] = 0b11110000;
      GW = 5; GH = 7; break;
    case 104:  // h
      GC[0] = 0b10000000; GC[1] = 0b10000000; GC[2] = 0b11110000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 105:  // i
      GC[0] = 0b01000000; GC[1] = 0b00000000; GC[2] = 0b11000000; GC[3] = 0b01000000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b11100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 106:  // j
      GC[0] = 0b00010000; GC[1] = 0b00000000; GC[2] = 0b00110000; GC[3] = 0b00010000;
      GC[4] = 0b00010000; GC[5] = 0b00010000; GC[6] = 0b10010000; GC[7] = 0b01100000;
      GW = 4; GH = 7; break;
    case 107:  // k
      GC[0] = 0b10000000; GC[1] = 0b10000000; GC[2] = 0b10010000; GC[3] = 0b10100000;
      GC[4] = 0b11000000; GC[5] = 0b10100000; GC[6] = 0b10010000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case 108:  // l
      GC[0] = 0b11000000; GC[1] = 0b01000000; GC[2] = 0b01000000; GC[3] = 0b01000000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b11100000; GC[7] = 0b00000000;
      GW = 3; GH = 7; break;
    case 109:  // m
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b11110000; GC[3] = 0b10101000;
      GC[4] = 0b10101000; GC[5] = 0b10101000; GC[6] = 0b10101000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 110:  // n
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b11110000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 111:  // o
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b01110000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 112:  // p
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b11110000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b11110000; GC[6] = 0b10000000; GC[7] = 0b10000000;
      GW = 5; GH = 7; break;
    case 113:  // q
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b00000000; GC[3] = 0b01111000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01111000; GC[7] = 0b00001000;
      GW = 5; GH = 7; break;
    case 114:  // r
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b10110000; GC[3] = 0b01001000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b01000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 115:  // s
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b01111000; GC[3] = 0b10000000;
      GC[4] = 0b01110000; GC[5] = 0b00001000; GC[6] = 0b11110000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 116:  // t
      GC[0] = 0b01000000; GC[1] = 0b01000000; GC[2] = 0b11100000; GC[3] = 0b01000000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b00110000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case 117:  // u
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b01111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 118:  // v
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b01010000; GC[5] = 0b01010000; GC[6] = 0b00100000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 119:  // w
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b10101000; GC[3] = 0b10101000;
      GC[4] = 0b10101000; GC[5] = 0b10101000; GC[6] = 0b01010000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 120:  // x
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b10001000; GC[3] = 0b01010000;
      GC[4] = 0b00100000; GC[5] = 0b01010000; GC[6] = 0b10001000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 121:  // y
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b01111000; GC[6] = 0b00001000; GC[7] = 0b01110000;
      GW = 5; GH = 7; break;
    case 122:  // z
      GC[0] = 0b00000000; GC[1] = 0b00000000; GC[2] = 0b11111000; GC[3] = 0b00010000;
      GC[4] = 0b00100000; GC[5] = 0b01000000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
      
    case 123:  // {
      GC[0] = 0b00110000; GC[1] = 0b01000000; GC[2] = 0b01000000; GC[3] = 0b10000000;
      GC[4] = 0b01000000; GC[5] = 0b01000000; GC[6] = 0b00110000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case 124:  // |
      GC[0] = 0b10000000; GC[1] = 0b10000000; GC[2] = 0b10000000; GC[3] = 0b00000000;
      GC[4] = 0b10000000; GC[5] = 0b10000000; GC[6] = 0b10000000; GC[7] = 0b00000000;
      GW = 1; GH = 7; break;
    case 125:  // }
      GC[0] = 0b11000000; GC[1] = 0b00100000; GC[2] = 0b00100000; GC[3] = 0b00010000;
      GC[4] = 0b00100000; GC[5] = 0b00100000; GC[6] = 0b11000000; GC[7] = 0b00000000;
      GW = 4; GH = 7; break;
    case 126:  // ~
      GC[0] = 0b00001000; GC[1] = 0b01110000; GC[2] = 0b10000000; GC[3] = 0b00000000;
      GC[4] = 0b00000000; GC[5] = 0b00000000; GC[6] = 0b00000000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;
    case 127:  // []
      GC[0] = 0b11111000; GC[1] = 0b10001000; GC[2] = 0b10001000; GC[3] = 0b10001000;
      GC[4] = 0b10001000; GC[5] = 0b10001000; GC[6] = 0b11111000; GC[7] = 0b00000000;
      GW = 5; GH = 7; break;

    default:  // Filled 8x8
      GC[0] = 0b11111111;
      GC[1] = 0b11111111;
      GC[2] = 0b11111111;
      GC[3] = 0b11111111;
      GC[4] = 0b11111111;
      GC[5] = 0b11111111;
      GC[6] = 0b11111111;
      GC[7] = 0b11111111;
      GW = 8; GH = 8; break;
  }
}

// ---------------------------------------------------------------------------------

void loadGfx(uint8_t zC) {
  // loads 8 bytes into the character array for use in graphics
  switch(zC) {
    case  0:  // Upwards pointing arrow
      GC[0] = 0b00000000;
      GC[1] = 0b00011000;
      GC[2] = 0b00100100;
      GC[3] = 0b01000010;
      GC[4] = 0b10000001;
      GC[5] = 0b11100111;
      GC[6] = 0b00111100;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  1:  // Downwards pointing arrow
      GC[0] = 0b00000000;
      GC[1] = 0b00111100;
      GC[2] = 0b11100111;
      GC[3] = 0b10000001;
      GC[4] = 0b01000010;
      GC[5] = 0b00100100;
      GC[6] = 0b00011000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  2:  // Upwards pointing arrow inside
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00011000;
      GC[3] = 0b00111100;
      GC[4] = 0b01111110;
      GC[5] = 0b00011000;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  3:  // Downwards pointing arrow inside
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00011000;
      GC[3] = 0b01111110;
      GC[4] = 0b00111100;
      GC[5] = 0b00011000;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;

    case  4:  // Upwards 2 pointing arrow
      GC[0] = 0b00111100;
      GC[1] = 0b01000010;
      GC[2] = 0b11111111;
      GC[3] = 0b00000000;
      GC[4] = 0b00111100;
      GC[5] = 0b01000010;
      GC[6] = 0b11111111;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;

    case  5:  // Downwards 2 pointing arrow
      GC[0] = 0b00000000;
      GC[1] = 0b11111111;
      GC[2] = 0b01000010;
      GC[3] = 0b00111100;
      GC[4] = 0b00000000;
      GC[5] = 0b11111111;
      GC[6] = 0b01000010;
      GC[7] = 0b00111100;
      GW = 8; GH = 8; break;

    default:  // Filled 8x8
      GC[0] = 0b11111111;
      GC[1] = 0b11111111;
      GC[2] = 0b11111111;
      GC[3] = 0b11111111;
      GC[4] = 0b11111111;
      GC[5] = 0b11111111;
      GC[6] = 0b11111111;
      GC[7] = 0b11111111;
      GW = 8; GH = 8; break;
  }
}

// ---------------------------------------------------------------------------------

void loadVector(uint8_t zC) {
  // loads 8 bytes into the character array for use in graphics
  switch(zC) {
    case  0:  // Level 0
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00000000;
      GC[3] = 0b00011000;
      GC[4] = 0b00011000;
      GC[5] = 0b00000000;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  1:  // Level 1
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00001100;
      GC[3] = 0b00010100;
      GC[4] = 0b00010100;
      GC[5] = 0b00001100;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  2:  // Level 2
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00001100;
      GC[3] = 0b00110100;
      GC[4] = 0b00110100;
      GC[5] = 0b00001100;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  3:  // Level 3
      GC[0] = 0b00000000;
      GC[1] = 0b00000110;
      GC[2] = 0b00001010;
      GC[3] = 0b00110010;
      GC[4] = 0b00110010;
      GC[5] = 0b00001010;
      GC[6] = 0b00000110;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  4:  // Level 4
      GC[0] = 0b00000000;
      GC[1] = 0b00000110;
      GC[2] = 0b00011010;
      GC[3] = 0b01100010;
      GC[4] = 0b01100010;
      GC[5] = 0b00011010;
      GC[6] = 0b00000110;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
    case  5:  // Level 5
      GC[0] = 0b00000011;
      GC[1] = 0b00000101;
      GC[2] = 0b00011001;
      GC[3] = 0b01100001;
      GC[4] = 0b01100001;
      GC[5] = 0b00011001;
      GC[6] = 0b00000101;
      GC[7] = 0b00000011;
      GW = 8; GH = 8; break;
    case  6:  // Level 6
      GC[0] = 0b00000011;
      GC[1] = 0b00001101;
      GC[2] = 0b00110001;
      GC[3] = 0b11000001;
      GC[4] = 0b11000001;
      GC[5] = 0b00110001;
      GC[6] = 0b00001101;
      GC[7] = 0b00000011;
      GW = 8; GH = 8; break;
    case  7:  // Level 7
      GC[0] = 0b00000111;
      GC[1] = 0b00011001;
      GC[2] = 0b01100001;
      GC[3] = 0b10000001;
      GC[4] = 0b10000001;
      GC[5] = 0b01100001;
      GC[6] = 0b00011001;
      GC[7] = 0b00000111;
      GW = 8; GH = 8; break;
    default:  // empty 8x8
      GC[0] = 0b00000000;
      GC[1] = 0b00000000;
      GC[2] = 0b00000000;
      GC[3] = 0b00000000;
      GC[4] = 0b00000000;
      GC[5] = 0b00000000;
      GC[6] = 0b00000000;
      GC[7] = 0b00000000;
      GW = 8; GH = 8; break;
  }
}

// --------------------------------------------------------------------------------

