/*
  OLED Display Mirroring functions
  Rel: 13/03/2022

  This version is for robots without a display that want to send data to the
  display mirroring app.
  
  Note the names of these functions start with a capital 'D' and use an underscore
  '_' instead of a period '.' so as not to be confused with the functions of the
  SSD1306Wire.h library. These functions only work when connected to a PC running
  the OLED Mirror app. I have only included the functions needed by this project
  but more could easily be added, using the same principles. Not all features can
  be included due to the limitations of Processing, like inverse ink colour.

  Display_clear()                 - clear the mirror pixel buffers
  Display_display()               - write the mirrored buffer to the display
  Display_drawCircle(X0,Y0,R0)    - draw a circle radius R0 at X0,Y0
  Display_drawLine(X0,Y0,X1,Y1)   - draw a line from X0,Y0 to X1,Y1
  Display_drawRect(x,y,w,h)       - draw a rectangle in the OLED Mirror app
  Display_drawString(x,y,S$)      - draw text string on teh mirrored display
  Display_fillRect(X,Y,W,H)       - draw a filled rectangle in the OLED Mirror app
  Display_setBrightness(Bright)   - sets teh brightness of the display 0 - 255
  Display_setColor(Ink)           - set the colour of the ink in OLED Mirror app, BLACK, WHITE
  Display_setFont(FontSize)       - set the point size of the mirrored font
  display.setTextAlignment(Align) - TEXT_ALIGN_LEFT, TEXT_ALIGN_CENTER, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER_BOTH

*/

// --------------------------------------------------------------------------------

void Display_clear() {
  // clear the OLED display memory
  // if DispMon then clear the mirror pixel buffers
  // display.clear();
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "\n~Clr\n"; // remove any broken junk from serial link and force a display clear
}

// --------------------------------------------------------------------------------

void Display_display() {
  // display update over I2C
  // if DisMon then write the mirrored commands to the display in OLED Mirror app
  // display.display();
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~Disp\n";
}

// --------------------------------------------------------------------------------

void Display_drawCircle(int16_t zX0,int16_t zY0,int16_t zR0) {
  // draw a circle radius zR0 at zX0,zY0
  // display.drawCircle(zX0,zY0,zR0);
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwCle(" + String(zX0) + "," + String(zY0) + "," + String(zR0) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_drawLine(int16_t zX0,int16_t zY0,int16_t zX1,int16_t zY1) {
  // draw a line from X0,Y0 to X1,Y1
  // display.drawLine(zX0,zY0,zX1,zY1);
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwLin(" + String(zX0) + "," + String(zY0) + "," + String(zX1) + "," + String(zY1) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_drawRect(int16_t zX,int16_t zY,int16_t zW,int16_t zH) {
  // draw a rectangle in OLED memory
  // if DispMOn then draw a rectangle in the OLED Mirror app
  // display.drawRect(zX,zY,zW,zH);
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwRec(" + String(zX) + "," + String(zY) + "," + String(zW) + "," + String(zH) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_drawString(int16_t zX,int16_t zY,String zS$) {
  // draw text at zX,zY in OLED memory
  // if DispMOn then draw text string on the mirrored display
  // display.drawString(zX,zY,zS$);
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwStr(" + String(zX) + "," + String(zY) + "," + zS$ + ")\n";
}

// --------------------------------------------------------------------------------

void Display_fillRect(int16_t zX,int16_t zY,int16_t zW,int16_t zH) {
  // draw a filled rectangle in OLED memory
  // if DispMon then draw a filled rectangle in the OLED Mirror app
  // display.fillRect(zX,zY,zW,zH);
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~DrwBar(" + String(zX) + "," + String(zY) + "," + String(zW) + "," + String(zH) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setBrightness(uint8_t zB) {
  // set the brightness level of the OLED display
  // Brightness = zB; display.setBrightness(zB);
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetLux(" + String(zB) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setColor(uint8_t zInk) {
  // set the colour of the ink used in OLED display graphics
  // if DispMon then set the colour of the ink in OLED Mirror app
  // the OLED MIrror app only supports two colours
  // if (zInk == BLACK) {display.setColor(BLACK);}
  // if (zInk == WHITE) {display.setColor(WHITE);}
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetCol(" + String(zInk) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setFont(uint8_t zFp) {
  // set the font size of the OLED text
  // if DispMon then set the point size of the mirrored font
  //       if (zFp == 10) {display.setFont(ArialMT_Plain_10);}
  // else if (zFp == 16) {display.setFont(ArialMT_Plain_16);}
  // else {display.setFont(ArialMT_Plain_24);}
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetFnt(" + String(zFp) + ")\n";
}

// --------------------------------------------------------------------------------

void Display_setTextAlignment(uint8_t zFa) {
  // set the font alignment
  // if DispMOn then set text alignment for OLED Mirror app 
  // zFa = 0  TEXT_ALIGN_LEFT
  // zFa = 1  TEXT_ALIGN_RIGHT
  // zFa = 2  TEXT_ALIGN_CENTER
  // zFa = 3  TEXT_ALIGN_CENTER_BOTH
  // if (zFa == 0) {display.setTextAlignment(TEXT_ALIGN_LEFT); }
  // if (zFa == 1) {display.setTextAlignment(TEXT_ALIGN_RIGHT); }
  // if (zFa == 2) {display.setTextAlignment(TEXT_ALIGN_CENTER); }
  // if (zFa == 3) {display.setTextAlignment(TEXT_ALIGN_CENTER_BOTH); }
  if (!DispMon) {return;} // do nothing if not connected to PC app

  PrintTx+= "~SetAlg(" + String(zFa) + ")\n";
}

// --------------------------------------------------------------------------------
