// ################################################################################
//
//  BallBot 4x4 ESP32 vR1
//
//  Released:  05/01/2025
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    microcontroller: ESP32 Dev Module   v3.0.5

    This code controls a self-balancing robot using two independant PID controllers
    that respond to angular errors, based on vertical X & Y channel setpoints. This
    robot has four wheels, two for each independant axis.

    This self-balancing version includes:
    - A Main loop running at 4ms (250Hz)
    - A battery monitoring function to prevent over-discharge
    - Eye presented on 1.28" circular TFT panel using SPI
    - RGB LED sequences which are brightness controlled
    - Two PID controllers X & Y, one for each axis
    - Speed limiting is applied to each error vector to prevent run-away
    - Auto wireless ESP-NOW 2.4GHz link setup
    - Control and reporting of PID variables via Monitor+.
    - Upside down motor drive demonstrations
    - PID P-gain only demonstration, to show i-gain benefit
    
    IMPORTANT - Espressif changed the way ESP-NOW works from v3.0, which broke the original
    code and caused compiler errors. This version has been modified to work with v3.0+.
    Enjoy!
*/
// Declare libraries
// #include <Wire.h> //Include the Wire.h library so we can communicate with the gyro
#include <Adafruit_GFX.h>       // Adafruit graphics lib      v1.11.9
#include <Adafruit_GC9A01A.h>   // Adafruit GC9A01A chip lib  v1.1.1
#include <FastLED.h>            // Neopixel library           v3.7.8

#include <esp_now.h>            // ESP-NOW WiFi link library  - defined by board version
#include <WiFi.h>               // ESP WiFi library           - defined by board version

String Release = "BallBot ESP32 4x4 (R1)";
String Date = "05/01/2025";

// Define a task for core 0
TaskHandle_t Core0;
 
// Initialize GC9A01A TFT library with hardware SPI module
#define TFT_CS    17      // TFT CS  pin is connected to ESP32 GPIO17
#define TFT_RST   16      // TFT RST pin is connected to ESP32 GPIO16
#define TFT_DC     4      // TFT DC  pin is connected to ESP32 GPIO4
#define TFT_CLK   18      // TFT CLK pin is connected to ESP32 GPIO18
#define TFT_DIN   23      // TFT DIN pin is connected to ESP32 GPIO23
Adafruit_GC9A01A tft(TFT_CS, TFT_DC);

// define constants for Display Monitor app
#define BLACK 0
#define BLUE 5
#define GREEN 4
#define INVERSE 2
#define MAGENTA 8
#define RED 3
#define TEXT_ALIGN_LEFT 0
#define TEXT_ALIGN_RIGHT 1
#define TEXT_ALIGN_CENTER 2
#define TEXT_ALIGN_CENTER_BOTH 3
#define TURQUOISE 7
#define WHITE 1
#define YELLOW 6

// define tasks for GFX_Run functions
#define GFX_EyeMove 1               // redraw an eye at GFX_X,GFX_Y
#define GFX_EyeXY 2                 // draw an eye at GFX_X,GFX_Y
#define GFX_Pupil 3                 // redraw an eye pupil

// define general constants
// Note that the accelerometer cal offsets are negative of what is measured in TEST mode
#define acc_1g  8192                // accelerometer 1g value
#define AcOffX 42                   // calibration offset applied to X-axis accelerometer (0)
#define AcOffY -460                 // calibration offset applied to Y-axis accelerometer (0)
#define AcOffZ 0                    // calibration offset applied to Z-axis accelerometer (0)
#define AcRampRate 30               // rate limit on accelerometers to filter vibration at small angles
// #define AccVectSafe 714.0           // off-balance safe zone >= 5 degrees
// #define AccVectTrgt 72.0            // 0.5 degrees horizontal balance target @ 8,192 LSB/g
#define Bat6v6 2482                 // 6.6v critical battery threshold, shutdown (single cell 3.3v)
#define Bat7v2 2734                 // 7.2v battery threshold 20% warning (single cell 3.6v)
#define Bat7v6 2906                 // 7.6v battery threshold 80% (single cell 3.8v)
#define Bat8v2 3190                 // 8.2v fully charged battery voltage, light load
#define Bat8v4 3305                 // 8.4v fully charged battery voltage, O/C
#define BatUSB 1500                 // minimum triggers US supply assumption
#define BatPin 36                   // analogue input pin ADC15 on GPIO36
#define CAL -1                      // in CALibration state
#define Calibrate false             // default = false; set to true to list sensor values
#define ctr_X 160                   // eye display centre X position
#define ctr_Y 120                   // eye display centre Y position
#define DbLLX 108                   // BallBot joystick deadband lower limit
#define DbULX 148                   // BallBot joystick deadband upper limit
#define DbLLY 108                   // BallBot joystick deadband lower limit
#define DbULY 148                   // BallBot joystick deadband upper limit
#define GyOffX -38                  // calibration offset taken from X-axis gyroscope (0)
#define GyOffY 44                   // calibration offset taken from Y-axis gyroscope (0)
#define GyOffZ -180                 // calibration offset taken from Z-axis gyroscope (0)
#define I2Cdel 10                   // I2C bus timing delay for ICM-42607-P
#define ICM_address 0x68            // ICM-42607-P I2C address (0x68 or 0x69)
#define LED_Pin  5                  // GPIO for RGB LEDs
#define NumLEDs 12                  // number of neopixel LEDs
#define Pin_A0 26                   // PWM pin for motor A front
#define Pin_A1 25                   // PWM pin for motor A front
#define Pin_B0 33                   // PWM pin for motor B right
#define Pin_B1 32                   // PWM pin for motor B right
#define Pin_C0  2                   // PWM pin for motor C rear
#define Pin_C1 19                   // PWM pin for motor C rear
#define Pin_D0 13                   // PWM pin for motor D left
#define Pin_D1 27                   // PWM pin for motor D left
#define PwmBits 8                   // PWM counter width 8 bits, 0 - 255
#define PwmFreq 30000               // PWM frequency is 30 kHz
#define READY 1                     // in READY state
#define RESET -2                    // in RESETing state
#define REST 0                      // in REST state
#define sw0Pin 15                   // GPIO15 assigned to SW0 switch input
#define sw1Pin 14                   // GPIO27 assigned to SW1 switch input
#define Topple 40.0                 // from vertical (0°) angle limit causing motor shut-down

// these are the critical controller settings to achieve self-balance
#define PtchPidOpTrendLim 200       // self balancing trend limit. Max angle = pid_output_trend_limit * self_balance_pid_inc
#define RollPidOpTrendLim 200       // self balancing trend limit. Max angle = pid_output_trend_limit * self_balance_pid_inc
 
// create a FastLED instance and define the total number of LEDs on the robot
CRGB LED[NumLEDs];        // LEDs connected to micro for o/p
CRGB LEDX[1];             // temp LED value

// my Wii Transciever unique MAC: 50:02:91:68:F7:3F
// you need to build a Wii Transceiver and get its MAC address
uint8_t broadcastAddress[] = {0x50, 0x02, 0x91, 0x68, 0xF7, 0x3F};

// 250 byte max for Tx/Rx data
// It must match the receivers data structure
typedef struct ESPNOW_Buff250 {
    char ESPdata[250];
} ESPNOW_Buff250;

// Create an 8 byte message called NunchukDataTx, to hold I2C data messages
ESPNOW_Buff250 Tx_Buff;

// Create an 8 byte message to hold incoming sensor readings
ESPNOW_Buff250 Rx_Buff;

// Declare and initialise global variables
long AcAvgDiv;                // accelerometer long averaging factor
int16_t AcDelta;              // difference in consequitive accelerometer readings
bool AcOffEn;                 // if == false then display RAW Acc values & run averaging
int16_t AcX;                  // instantaneous sensor read for averaging
long AcXAvg;                  // averaged AcX value to reduce noise
long AcXAvgL;                 // long averaged AcX value to reduce noise
int16_t AcXL;                 // previous ICM X-acceleration value
int16_t AcX_OA;               // offset adjusted AcX
int16_t AcX_OAL;              // previous offset adjusted AcX_OA
long AcXSum;                  // averaging accumulator
long AcXSumL;                 // long averaging accumulator
int16_t AcY;                  // instantaneous sensor read for averaging
long AcYAvg;                  // averaged AcY value to reduce noise
long AcYAvgL;                 // long averaged AcY value to reduce noise
int16_t AcYL;                 // previous ICM Y-acceleration value
int16_t AcY_OA;               // offset adjusted AcY
int16_t AcY_OAL;              // previous offset adjusted AcY_OA
long AcYSum;                  // averaging accumulator
long AcYSumL;                 // long averaging accumulator
long AcY_Trck;                // Y-axis tracking filter for orientation
int16_t AcZ;                  // instantaneous sensor read for averaging
long AcZAvg;                  // averaged AcZ value to reduce noise
long AcZAvgL;                 // longaveraged AcZ value to reduce noise
int16_t AcZL;                 // previous ICM Z-acceleration value
long AcZSum;                  // averaging accumulator
long AcZSumL;                 // long averaging accumulator
long AcZ_Trck;                // Z-axis tracking filter for orientation
float AngAcX;                 // X-axis accelerometer angle
float AngAcY;                 // Y-axis accelerometer angle
float AngAcZ;                 // Z-axis accelerometer angle
float AngGyX;                 // X gyro angle - Roll
float AngGyY;                 // Y gyro angle - Pitch
float AngGyZ;                 // Z gyro angle - Yaw
float AngOffset;              // small offset to correct gyro angle drift
int Any;                      // any temporary integer value
String Any$;                  // temporary string
uint8_t AnyByte;              // any byte value
int anyCnt;                   // temporary counter
int AnyD;                     // temp variable
int anyFor;                   // temporary for loop counter
float anyFX;                  // temporary floating point variable
float anyFY;                  // temporary floating point variable
float anyFloat;               // temporary floating point variable
long AnyLong;                 // any temp value
unsigned long anyMilli;       // any millisecond timer
int16_t anyVal;               // any temp variable
int16_t AtState;              // current at state, ie. REST, READY,STAND, etc...
int16_t BakCol;               // background colour
int32_t BatAvg;               // preload averaging function with high value
float BatMult;                // battery multiplier used in PWM limiting
int32_t BatPc;                // battery percentage of charge, as integer
int32_t BatSum;               // cumulative battery voltage
String BatTime$;              // time left displayed
int32_t BatV;                 // battery voltage * 10, as in 10.0v == 100
int32_t BatVol;               // instantaneous battery voltage
float BatVfp;                 // battery voltage as a decimal number. ie. 11.2v
bool Blink;                   // LED blink clock
int16_t BlinkCnt;             // counter for LED Blink clock
uint8_t Bright;               // FastLED brightness
uint8_t Brightness;           // Display brightness
float BrkPnt;                 // Threshold at which PID brake is applied
float BrkVal;                 // PID brake multiplier value
int C_Button;                 // button pressed state; 0 = UP, 1 = Down
int16_t C_Cnt;                // counter used in 'C' button detection
byte C_Dn;                    // counter for 'C' button down period
char cmdMode = ' ';           // command mode
int cmdRec;                   // > 0 if a '~' has been received
int cmdSgn = 1;               // value polarity multiplier
char cmdType = ' ';           // command type
int cmdVal = 0;               // value associated with a cmdType
uint8_t ColB;                 // RGB blue colour
uint8_t ColG;                 // RGB green colour
uint8_t ColR;                 // RGB red colour
byte C_Up;                    // counter for 'C' button up period
int16_t CZ;                   // received button values
char Deg = 176;               // ASCII for "°"
unsigned long DispBatWarn;    // timer usesd to prevent battery warning repeating too often
bool DispClr;                 // set == true to clear the memory contents
int16_t DispCnt;              // counter used in display updating
int8_t DispCX,DispCY;         // display cursor coordinates
int16_t DispDel;              // >0 display delay counter, stops updates over I2C
int16_t DispDelGbl;           // global DispDel offset, used to adjust display frame rates
bool DispDisp;                // set = true to immediately display memory contents
bool DispEn;                  // if == true then generate display messages, otherwise inhibit
int16_t DispLast;             // previous DispMode value
bool DispLock;                // if == true then don't allow display mode changes
int16_t DispMode;             // the type of display for a given mode
bool DispMon;                 // == true if connected to OLED Mirror app
int16_t DispNext;             // Used with temporary displays to direct follow-up screen
int16_t DispOveride;          // if DispLock == true then use this overide mode
int16_t DispTx;               // counter used with display mirror app
int16_t DmB,DmX,DmY;          // button and X,Y values associated with a display monitor mouse click
int16_t DM_Batt;              // Battery display assigned Display Mode
int16_t DM_BatPlus;           // Battery plus display assigned Display Mode
int16_t DM_BatWarn;           // Battery warning <20% display assigned Display Mode
int16_t DM_Control;           // Control display assigned Display Mode
int16_t DM_DriveCtrl;         // Drive Ctrl display assigned Display Mode
bool DmDwn;                   // == true whilst mouse is clicked in Monitor+ mode
int16_t DM_EyeEng;            // Eye Engine display assigned Display Manager
int16_t DM_I2C;               // I2C display assigned Display Mode
int16_t DM_Intro;             // Intro display assigned Display Mode
int16_t DM_JoyNorm;           // Joystick normal values display assigned Display Mode
int16_t DM_JoyOff;            // Joystick values and offsets display assigned Display Mode
int16_t DM_IcmLvl;            // 'ICM Levels' display assigned Display Mode
int16_t DM_Limits;            // Limits display assigned Display Mode
int16_t DM_Min,DM_Max;        // min/max Display Mode pointer values
int16_t DM_ICM_Acc;           // 'ICM Acc.' display assigned Display Mode
int16_t DM_IcmAccAng;         // 'ICM Acc Angles' display assigned Display Mode
int16_t DM_IcmGyrAng;         // 'ICM Gyro Angles' display assigned Display Mode
int16_t DM_ICM_Gyros;         // 'ICM Gyros' display assigned Display Mode
int16_t DM_IcmRot;            // 'ICM Rotation' display assigned Display Mode
int16_t DM_IcmWarn;           // 'ICM Warning' display assigned Display Mode
int16_t DM_MtrDemo;           // Motor demo assigned Display Mode
int16_t DM_MtrTest;           // Motor test assigned Display Mode
int16_t DM_PID;               // PID display assigned to Display Mode
int16_t DM_PtchYaw;           // Pitch n Yaw display assigned Display Mode
bool DmSft;                   // == true if SHIFT key held down when mouse clicked in Monitor+
int16_t DM_SafeMode;          // SafeMOde display assigned Display Mode
int16_t DM_SaPtchRoll;        // Start angle Pitch & Roll Display Mode
int16_t DM_Status_A;          // Status A display assigned Display Mode
int16_t DM_Status_B;          // Status B display assigned Display Mode
int16_t DM_WiFi;              // WiFi display assigned Display Mode
bool DriveEn;                 // if == true then output PWM to motors
float DrvSpMax;               // Max drive setpoint
float DrvSpMin;               // Min drive setpoint
bool Echo;                    // if true send print messages
int16_t end4ms;               // flag indicating end of timer run
int16_t ESP_NOW_BA;           // broadcast address pointer, -1 to start with
bool ESP_NOW_Init;            // == true once ESP-NOW has been initiaiised
bool Eye_En;                  // == true if eye engine is enabled, default = true
unsigned long freeTime;       // time spare in 4ms cycle
int16_t GAScnt;               // counter used in getAtState messages
bool G_once;                  // set == true to force initial graphing mode, default = false
int16_t GPmax,GPmin;          // start and end of graphing list, for Scope mode
int16_t GPnn;                 // pointer to current graphing items
bool GP_Title;                // if == true then report a title in Scope graphing mode
int16_t GFX_Cnt;              // eye GFX move increment counter
int16_t GFX_Del;              // millisecond delay timer, holds off GFX_Run tasks
int16_t GFX_Eng;              // eye GFX move engine task pointer
int16_t GFX_Last;             // previous GFX_Mode value
int16_t GFX_Mode;             // eye drawing mode, 0 - 2
int16_t GFX_R0, GFX_R1;       // eye and cornea radii
int16_t GFX_R1Tgt;            // eye and cornea radii
int16_t GFX_Rot;              // Eye display rotation value
bool GFX_RST;                 // if == true then reset GFX modes for eye display
int16_t GFX_Run;              // set > 0 to initiate Core0, eye drawing tasks
int16_t GFX_Task;             // task pointer for TFT drawing sequence, default = 0, reset
int16_t GFX_Txt;              // set > 0 to initiate Core0, text drawing tasks
int16_t GFX_Wait;             // eye GFX move engine pause/wait counter
int16_t GFX_X,GFX_Y;          // GFX drawing cursor co-ordinates
int16_t GFX_XL,GFX_YL;        // record of GFX drawing cursor co-ordinates
int16_t GFX_XTgt,GFX_YTgt;    // GFX target values
float GyAcMod;                // Gyro angle modified by Acc angle
long GyAvgDiv;                // accelerometer long averaging factor
bool GyOffEn;                 // if == false then display RAW Gyro values
bool GyrOnce;                 // sets the initial angles for the gyros based on the accelerometers
unsigned long GyT;            // time of gyro readings in milliseconds
unsigned long GyTD;           // time between successive gyro readings in milliseconds
int16_t GyX;                  // gyro value read from ICM-42607-P sensor
long GyXAvg;                  // averaged GyX value to reduce noise
long GyXAvgL;                 // long averaged GyX value to reduce noise
int16_t GyXL;                 // previous MPU X-gyroscope value
int16_t GyX_OA;               // GyX with calibration drift offset added
int16_t GyXOff;               // offset to be added to raw X-gyroscope value
int16_t GyXOffCnt;            // offset counter for gyro centering
long GyXSum;                  // averaging accumulator
long GyXSumL;                 // long averaging accumulator
int16_t GyY;                  // gyro value read from ICM-42607-P sensor
long GyYAvg;                  // averaged GyY value to reduce noise
long GyYAvgL;                 // long averaged GyY value to reduce noise
int16_t GyYL;                 // previous MPU Y-gyroscope value
int16_t GyY_OA;               // GyY with calibration drift offset added
int16_t GyYOff;               // offset to be added to raw Y-gyroscope value
int16_t GyYOffCnt;            // offset counter for gyro centering
long GyYSum;                  // averaging accumulator
long GyYSumL;                 // long averaging accumulator
int16_t GyZ;                  // gyro value read from ICM-42607-P sensor
long GyZAvg;                  // averaged GyZ value to reduce noise
long GyZAvgL;                 // long averaged GyZ value to reduce noise
int16_t GyZL;                 // previous MPU Z-gyroscope value
int16_t GyZ_OA;               // GyY with calibration drift offset added
int16_t GyZOff;               // offset to be added to raw Z-gyroscope value
int16_t GyZOffCnt;            // offset counter for gyro centering
long GyZSum;                  // averaging accumulator
long GyZSumL;                 // long averaging accumulator
byte I2C_Err;                 // flag returned from I2C comms
int16_t I2C_ErrCnt;           // counter that tracks I2C_Err events
bool I2C_ICM;                 // true if ICM-42607-P is detected during POST
int16_t ICM_Crash;            // ICM crash counter, default = 0
bool ICM_En;                  // if == true then read ICM over I2C
bool ICMread;                 // == true whilst reading ICM
int16_t JoyLX,JoyLY;          // Wii Classic left joystick values
int16_t JoyX, JoyY;           // received value
char keyChar;                 // any keyboard character
uint8_t keyVal;               // any keyboard value
int16_t LedCnt;               // counter used in LED sequencing
int16_t LedInc;               // increment used in LED sequencing
int16_t LED_Del;              // delay timer used in LED functions
int16_t LedLast;              // previous task when change set
int16_t LedMem;               // previous recorded LED variable, default -1
int16_t LedMode;              // LED mode pointer
int16_t LedModeNew;           // new mode implemented by Core0
int16_t LedNext;              // next LedMode to be used after a priotity task
bool LedNop;                  // if true then don't update LEDs
bool LEDshow;                 // LED show flag
bool LEDskip;                 // if == true then skip a Loop0 1ms cycle
bool LED_SW;                  // if == true then button switch is down
int16_t LED_Task;             // LED task pointer 
byte LEDVal;                  // values used in LED tasks
int16_t MainMode;             // determined after RESET, based on orientation
bool MainRun;                 // flag runs MainMode tasks every 8ms
int16_t MainSkip;             // counter used to skip MainMode tasks
int16_t MainSkipCnt;          // value used to pre-load MainSkip
int16_t MainTest;             // used to test specific MainModes
unsigned long mic4ms;         // 4ms (250Hz) task timer in microseconds
float ModeAngle;              // angle counter used in MainMode tasks
int16_t ModeCnt;              // counter used in MainModes
int16_t ModeCntX;             // X counter used in MainMode tasks
int16_t ModeCntY;             // Y counter used in MainMode tasks
int16_t ModeCycle;            // cycle counter used in MainMode tasks
int16_t ModeMem;              // recorded value used in MainMOdes
float ModePID;                // PID output demand used in MainMode tasks
int16_t ModeSteer;            // steering demand used in MainMode tasks
String ModeStr$;              // mode string displayed in displays
int16_t ModeTask;             // task poiinter used in MainModes
int16_t ModeTaskNext;         // task pointer used in MainMode tasks
int16_t ModeValX;             // value used in MainMode tasks
int16_t ModeValY;             // value used in MainMode tasks
float MotorDriveA;            // motor drive value
float MotorDriveB;            // motor drive value
float MotorDriveC;            // motor drive value
float MotorDriveD;            // motor drive value
int16_t MtrMin;               // minimum motor PWM, to overcome gearbox stiction
String myMAC;                 // MAC address if this device
unsigned long next_4ms;       // 4ms (250Hz) loop timer
float PID_db;                 // PID controller deadband angle
bool PID_d_En;                // PID d-gain enable true=ON, false=OFF
float pid_p_gain;             // Gain setting for the P-controller (0.0 - 100.0)
bool PID_i_En;                // PID i-gain enable true=ON, false=OFF
float pid_i_gain;             // Gain setting for the I-controller (0.0 - 10.0)
float pid_i_max;              // i-gain limit, o - 255
float pid_d_gain;             // Gain setting for the D-controller (10 - 40)
float pid_out_max;            // limit PID output to prevent PWM overload
float PID_Output_Power;       // used in soft start
int16_t Ping;                 // Rx 'ping' counter
float Pitch;                  // forward/backward leaning angle
bool Pitch_En;                // Pitch PID enable, default = false
float Ptch_d_Val;             // Pitch d-gain value
float Ptch_i_mem;             // Pitch PID i-gain memory
float PtchPidErr;             // Pitch angle error, relative to setpoints
float PtchPidErrLast;         // Previous Pitch angle error, relative to setpoints
float Ptch_pid_output;        // Pitch PID p-gain output
float PtchPidSb_db;           // Pitch self-balance deadband
int16_t PtchPidOpTrend;       // Pitch output trend
int16_t PtchPidTune;          // > 0 when in tuning mode
float Ptch_P_Val;             // Pitch p-gain value
float PtchSaN;                // Negative Pitch start angle trigger point
float PtchSaP;                // Positive Pitch start angle trigger point
float PtchSbInc;              // Pitch self-balance increment
float PtchSbSp;               // Pitch self-balance setpoint
float PtchSbSpSa;             // Pitch self-balance setpoint start angle
float PtchSp;                 // nominal Pitch setpoint, start angle
uint32_t print40ms;           // 40ms variable print timer
int16_t PrintTgt;             // 0 == serial port, 1 == WiFi ESP-NOW
String PrintTx = "";          // printed strings
int16_t PWM_A;                // PWM drive for motor A
int16_t PWM_A0;               // PWM value written to H-bridge driver
int16_t PWM_A1;               // PWM value written to H-bridge driver
int16_t PWM_A_;               // PWM off state value, 0 = 00 or 1 = FF
int16_t PWM_B;                // PWM drive for motor B
int16_t PWM_B0;               // PWM value written to H-bridge driver
int16_t PWM_B1;               // PWM value written to H-bridge driver
int16_t PWM_B_;               // PWM off state value, 0 = 00 or 1 = FF
int16_t PWM_C;                // PWM drive for motor C
int16_t PWM_C0;               // PWM value written to H-bridge driver
int16_t PWM_C1;               // PWM value written to H-bridge driver
int16_t PWM_C_;               // PWM off state value, 0 = 00 or 1 = FF
int16_t PWM_D;                // PWM drive for motor D
int16_t PWM_D0;               // PWM value written to H-bridge driver
int16_t PWM_D1;               // PWM value written to H-bridge driver
int16_t PWM_D_;               // PWM off state value, 0 = 00 or 1 = FF
int16_t PWM_Max;              // max PWM value, constrained by V_max
int readTask;                 // task pointer used in decoding received control data
int receive_counter;          // used to extend received data for 100 ms
byte received_byte;           // received byte from WiFi reciever
bool received_cmd;            // set true if valid control has ben recevied
byte received_mem;            // copy of received_byte
float Roll;                   // side to side rolling angle
float Roll_d_Val;             // Roll PID d-gain value
bool Roll_En;                 // Roll PID ewnable, default = false;
float Roll_i_mem;             // Roll PID i-gain memory
float RollPidErr;             // Roll angle error, relative to setpoints
float RollPidErrLast;         // Previous Roll angle error, relative to setpoints
float Roll_pid_output;        // Roll PID output
int16_t RollPidOpTrend;       // Roll output trend
float RollPidSb_db;           // Roll self-balance deadband
int16_t RollPidTune;          // > 0 when in tuning mode
float Roll_p_Val;             // Roll p-gain value
float RollSaN;                // Negative Roll start angle trigger point
float RollSaP;                // Positive Roll start angle trigger point
float RollSbInc;              // Roll self-balance increment
float RollSbSp;               // Roll self-balance setpoint
float RollSbSpSa;             // Roll self-balance setpoint start angle
float RollSp;                 // nominal Roll setpoint, start angle
float run4ms;                 // set == true to run 4ms tasks
int16_t Rx_Chk;               // Rx checksum
int16_t RxCZ;                 // buffered received CZ value
int16_t RxJoyX;               // buffered received JoyX value
int16_t RxJoyY;               // buffered received JoyY value
int16_t Rx_len;               // length of WiFi received data block
int16_t Rx_Pnt;               // data pointer user by Rx state machine
bool RxRec;                   // set true when a valid frame of data is received
bool RxRecvEvent;             // set == true when a OnDataRecv() call back event has occured
int16_t RxState;              // receiver state machine state
int16_t Rx_Task;              // task pointer user by Rx state machine
int16_t RTxTimeout;           // counter used to auto-reset if WiFi fails
int16_t RxVal;                // value received from serial Rx
byte RxWiFi[6];               // array holding 6 bytes of Wii received data
int16_t SafeMode;             // Safety mode state pointer
bool SerialRx;                // == true if any character has been received
bool SetLEDs;                 // if == true then Core0 sets a new LED mode
bool StartEn;                 // true to perform motor calcs and enable output
float Steer;                  // steering demand, +ve = right, -ve = left, max 255
int sw0Cnt;                   // button switch counter
bool sw0LastState;            // previous state of button switch, HIGH/LOW
bool sw0State;                // state of read button switch pin SW0
int sw0Timer;                 // timer used to detemine button sequences
bool sw0WaitHi;               // set HIGH to wait for SW0 release
int sw1Cnt;                   // button switch counter
bool sw1LastState;            // previous state of button switch, HIGH/LOW
bool sw1State;                // state of read button switch pin SW1
int sw1Timer;                 // timer used to detemine button sequences
bool sw1WaitHi;               // set HIGH to wait for SW1 release
int t0,t1;                    // general temp timers
int16_t Task8ms;              // task pointer for 8ms tasks
int16_t Task20ms;             // task pointer for 20ms tasks
int16_t Task40ms;             // task pointer for 40ms tasks
int16_t Tx_PingCnt;           // 1 second Tx ping timer
bool TEST = false;            // default = false. True invokes accelerometer drift measurement
float TurnMax;                // Max turn speed
float TurnMin;                // Min turn speed
int16_t TxtCol;               // Text colour
int16_t Txt_Del;              // text delay timer counter
bool Txt_En;                  // if == true then enable drawing of text in eye display
int16_t Txt_Mode;             // Text display function pointer
int16_t Upright;              // either -1,0,1 depending on orientation
int16_t Upside;               // either -1,0,1 depending on orientation
bool USB;                     // == true if initial battery is below 5.5v
float V_max;                  // voltage limit for motor PWM
int16_t WiFiCntC;             // C button counter for WiFi enable
int16_t WiFiConCnt;           // counter used in connection retry times
bool WiFiConnected;           // == true if a WiFi connection has been established
bool WiFiEn;                  // true when WiFi is enabled with prolonged 'C' button activity
bool WiFiOff;                 // if == true then kill WiFi, default = false
int16_t WiFiPing;             // Rx counter used to maintain connected state
int16_t WiFiRestCnt;          // counter used to go to rest if no WiFi demands
bool WiFiRx;                  // true when a block of data has been received
long WiFiRxBytes;             // number of bytes received over the WiFi link
int16_t WiFiRxErr;            // number of bytes received over the WiFi link
byte WiFiRxMacAddr[6];        // senders MAC address, which could be broadcast FF:FF:FF:FF:FF:FF
bool WiFiRxRec;               // true when a block of data has been successfully received
int16_t WiFiTryCnt;           // WiFi try to connect count on error count
int16_t WiFiTryNum;           // WiFi try to connect total count
bool WiFiTryOnce;             // == true if first pass of trying to connect
bool WiFiTx;                  // true when a block of data has been sent successfully
int16_t WiFiTx_CB;            // == 1 call back is clear to send, == -1 resend
int16_t WiFiTx_Chk;           // Tx checksum
int16_t WiFiTx_len;           // >0 when WiFi Tx buffer contains data to be sent
long WiFiTxBytes;             // number of bytes received over the WiFi link
int16_t WiFiTxErr;            // WiFi data error
int16_t WiFiTxErrCnt;         // WiFi Tx error count, leads to disconnection
byte WiFiTxMacAddr[6];        // transmit MAC address, which should match myMAC address
int16_t WiiType;              // device type, 0 == Nunchuk, 1 == Classic
int WinApp;                   // defines the Windows app connection, default = 0
float Yaw;                    // body swings left/right
int16_t YawCnt;               // 3 sec counter used to zero Yaw angle when turning has stopped
float YawDif;                 // Yaw gyro differential angle, between successive readings
float YawLast;                // previous Yaw angle
int16_t YawTimeout;           // timeout used in Yaw angle display
int Z_Button;                 // button pressed state; 0 = UP, 1 = Down
byte Z_Dn;                    // button pressed state; 0 == UP, >= 1 == Down
bool Z_Mode;                  // joystick motion flag, normally false, if == true then slide mode
int Z_orientation = 0;        // orientation flag based on z accelerometer, default 0 on its side

// Define arrays for eye text messages
const int16_t TxtDepth = 5;
String TxtN$[TxtDepth];       // new text strings
String TxtP$[TxtDepth];       // new text strings
int16_t TxtAN[TxtDepth];      // new text alignment, 0 = right, 1 = centre, 2 = left
int16_t TxtAP[TxtDepth];      // previous text alignment, 0 = right, 1 = centre, 2 = left
int16_t TxtFN[TxtDepth];      // new text font size, if > 0 text was drawn
int16_t TxtFP[TxtDepth];      // previous text font size, if > 0 text was drawn
int16_t TxtXN[TxtDepth];      // new text CX co-ordinate
int16_t TxtXP[TxtDepth];      // previous text CX co-ordinate
int16_t TxtYN[TxtDepth];      // new text CY co-ordinate
int16_t TxtYP[TxtDepth];      // previous text CY co-ordinate

// --------------------------------------------------------------------------------

void setup() {
  // Setup basic functions
  // Initialise PWM pins and output states
  pinMode(Pin_A0, OUTPUT); digitalWrite(Pin_A0,LOW);
  pinMode(Pin_A1, OUTPUT); digitalWrite(Pin_A1,LOW);
  pinMode(Pin_B0, OUTPUT); digitalWrite(Pin_B0,LOW);
  pinMode(Pin_B1, OUTPUT); digitalWrite(Pin_B1,LOW);
  pinMode(Pin_C0, OUTPUT); digitalWrite(Pin_C0,LOW);
  pinMode(Pin_C1, OUTPUT); digitalWrite(Pin_C1,LOW);
  pinMode(Pin_D0, OUTPUT); digitalWrite(Pin_D0,LOW);
  pinMode(Pin_D1, OUTPUT); digitalWrite(Pin_D1,LOW);

  // Initialise other pins
  pinMode(BatPin,INPUT);          // battery ADC
  pinMode(sw0Pin,INPUT_PULLUP);   // button switch SW0 input, with pullup resistor
  pinMode(sw1Pin,INPUT_PULLUP);   // button switch SW1 input, with pullup resistor

  setDefaults();
  getDispRef();             // read references, without displaying screens
  
  Serial.begin(115200);     // use this default baud rate for Nunchuk wireless control
  delay(100);
  I2C_Err = Wire.begin();   // Start the I2C bus as master
  if (!I2C_Err) {Serial.println("I2C Wire.begin() Error!");} else {Serial.println("I2C started");}
  I2C_Err = Wire.setClock(400000);    // set I2C 400 kHz Fast-mode, default = 100 kHz
  if (!I2C_Err) {Serial.println("I2C Setclock() Error!");} else {Serial.println("I2C clock freq set");}

  // initialise the FastLED component
  // a chain of 12 LEDs = 24 x 12 = 288 bits, taking 360us to show at 800kbps
  FastLED.addLeds<WS2812B, LED_Pin, GRB>(LED, NumLEDs);
  FastLED.setBrightness(Bright);
  FastLED.clear();
  FastLED.show();

  // Initialise the eight PWM channels, setting each to OFF
  for (int zP = 0;zP < 8;zP++) {
    PwmSetup(zP);
    PwmAttach(zP);
  }

  // Initialise the round TFT eye display
  tft.begin();
  tft.setRotation(2);
  tft.fillScreen(GC9A01A_BLACK);
    
  // initialise WiFi link
  WiFi.mode(WIFI_STA);
  // new code for ESP32 lib v3.0+
  while (!WiFi.STA.started()) {delay(10);}
  // Once started we can get the micros MAC address
  myMAC = WiFi.macAddress();
  Serial.println("my MAC= " + String(myMAC));

  // initialise the ESP-NOW link for the first time
  Init_ESP_NOW();
  
  run_POST();               // perform calibration sequence
  
  // define the Core 0 task parameters
  // Core 0 is normally used for WiFi, but here we use it to read sensors
  // leaving Core 1 to create the LED patterns and clock them out
  // Note, this code starts and runs immediately, before loop() does
  xTaskCreatePinnedToCore(
    loop0,       // Task function.
    "Core0",     // name of task.
    10000,       // Stack size of task
    NULL,        // parameter of the task
    1,           // priority of the task
    &Core0,      // Task handle to keep track of created task
    0);          // pin task to core 0

  // so off we go!...
  synchLoopTimers();
}

// --------------------------------------------------------------------------------

void loop() {
  // Main loop code, runs repeatedly.
  //###############################################################################
  //
  //  WiFi comms check & respond
  //
  //###############################################################################
  // if WiFi is conencted then check Rx buffer on every cycle, so that we respond to
  // received packets as soon as possible
  if (RxRecvEvent) {OnDataRecvHandler();}   // respond to ESP-NOW data received events

  if (WiFiConnected) {
    if (WiFiRx) {WiFiReadRx();} // WiFi Rx buffer contains unread data so deal with it before adding new
  }

  //###############################################################################
  //
  //  Read serial comms
  //
  //###############################################################################
  readSerial();                 // check serial port on every loop cycle

  //#############################################################################
  // Run MainMode tasks
  //#############################################################################
  // These tasks run in the alternate phase of the 8ms cascaded tasks
  if (MainRun) {
    MainRun = false;
    switch (MainMode) {
      case 0: MainMode_0(); break;      // Waiting for user input
      case 1: MainMode_1(); break;      // Self-balance start sequence
      case 4: MainMode_4(); break;      // Motor upside down demo
    }
  }

  //#############################################################################
  // Run 8ms tasks
  //#############################################################################
  // This sequence of tasks run in reverse order once Task8ms > 0
  if (Task8ms > 0) {
    switch (Task8ms) {
      case 1: read_SW1(); break;            // read the state of the button switch SW1
      case 2: read_SW0(); break;            // read the state of the button switch SW0
    }
    Task8ms--;  // point at the next task or stop
  }
 
  //#############################################################################
  // Run 20ms tasks
  //#############################################################################
  // This sequence of tasks run in reverse order once Task20ms > 0
  else if (Task20ms > 0) {
    switch (Task20ms) {
      case 1: read_Battery(); break;        // read the battery voltage
      case 2: LED_Main(); break;            // Update LED task
    }
    Task20ms--;  // point at the next task or stop
  }

  //#############################################################################
  // Run 40ms tasks
  //#############################################################################
  // This sequence of tasks run in reverse order once Task40ms > 0
  else if (Task40ms > 0) {
    switch (Task40ms) {
      case  1: if (DispMon) {Display_40ms();} break;
      case  2: if (Ping > 0) {Ping--;} break;  // time-out for app data received flag
      case  3: if (!WiFiConnected) {WiFiNotConnected();} break;
      case  4: if (WiFiPing > 0) {WiFiPingHandler();} break;
      case  5: Eye_Engine(); break;
      case  6: Txt_Engine(); break;
    }
    Task40ms--;  // point at the next task or stop
  }
  // We have got to the end of all of the cascaded tasks
  else if (end4ms == 0) {end4ms = 1;}   // set the flag for the end timer

  //###############################################################################
  //
  //  Main tasks and timers
  //
  //###############################################################################
  // This main loop runs on a 4ms timer (250Hz) to trigger the balancing calculations.
  // Once they are complete it then triggers a series of tasks to perform other
  // functions, at sub-multiples of 4ms. For example, an 8ms task is used to read
  // the button switches. All task must be completed within the 4ms period.

  if ((millis() - next_4ms) >= 4) {
    // The 4ms hardware timer has triggered
    // We don't re-trigger unless we have finished all of the tasks
    if (end4ms == 2) {
      next_4ms = millis();    // reset the 4ms loop timer
      run4ms = true;          // set the 4ms run flag
      end4ms = 0;             // end of tasks time flag
      mic4ms = micros();      // task timer for free space calcs
    }
  }
  if (run4ms) {
    //#############################################################################
    // Run 4ms tasks
    //#############################################################################
    // These tasks are triggered by the 4ms timer (250Hz)
    // Erasing the round display can crash the I2C bus, so we avoid using it at the same time
    // GFX display writes will set ICM_En = false to hold off the reading process
    if (ICM_En) {
      ICMread = true;         // Block Core0 tasks whilst reading
      loop_Read_ICM();        // read ICM-42607-P sensor registers
      loop_PID();             // perform PID calculations

      // Now adjust additional slower timers to trigger tasks as necessary
      // Only do this when tasks are complete, or you would miss tasks out
      if (Task8ms  <= 0) {Task8ms--;  if (Task8ms  < -1) {Task8ms  = 2;} else {MainRun = true;}}   // either re-trigger 8ms task sequence or run MainMode tasks
      if (Task20ms <= 0) {Task20ms--; if (Task20ms < -4) {Task20ms = 2;}}   // re-trigger 20ms task sequence
      if (Task40ms <= 0) {Task40ms--; if (Task40ms < -9) {Task40ms = 6;}}   // re-trigger 40ms task sequence
      run4ms = false;
    }
  }

  //#############################################################################
  // Run 40ms (variable) PrintTx Task
  //#############################################################################
  // This PrintTx task runs at a variable rate
  if ((millis() - print40ms) >= 40) {
    print40ms = millis();
    PrintTxHandler();
  }

  //#############################################################################
  // Test for slack time and code over-run in each cycle
  //#############################################################################
  // Code used during development.
  // The end4ms flag is set once all of the 4ms tasks have been completed.
  if (end4ms == 1) {
    // freeTime should always be positive. If not over-run has occured
    freeTime = 4000 - (int)(micros() - mic4ms);
    end4ms = 2;
  }


  //#############################################################################
  // End of 4ms main loop
  //#############################################################################
}

// ---------------------------------------------------------------------------------

void loop0(void * pvParameters) {
  // this function runs on Core 0, also used for ESP32 WiFi NOW
  // *** WARNING *** do not use for time consuming code like display, I2C, FastLED, etc
  // the following message, sent once at boot, confirms core 0 assignment
  Serial.print("Core0 task running on core "); Serial.println(xPortGetCoreID());

  for(;;){
    // infinite loop, just like loop() but running in Core 0
    // you must never use 'break' or 'return' statement here or it will crash
    // we tell the free RTOS system to call this function every 1 ms

    //###############################################################################
    //
    //  Control GFX_Run tasks and RGB LEDs
    //
    //###############################################################################
    // GFX_Txt tasks take priority over GFX_Run tasks
    // GFX_Run tasks take priority over LEDshow triggers
    if (!ICMread) {
           if (GFX_RST)     {GFX_Reset();}        // resets eye display, depening on GFX_Mode and orientation
      else if (GFX_Txt > 0) {GFX_TxtTasks();}     // perform GFX text drawing tasks
      else if (GFX_Run > 0) {GFX_EyeTasks();}     // perform GFX eye drawing tasks
      else if (LEDshow) {
        FastLED.show();
        LEDshow = false; LEDskip = true;
      }
      else if (SetLEDs) {SetLEDs = false; SetLEDmodeNow(LedModeNew);} // change LEDmode
      else if (LEDskip) {LEDskip = false;}        // miss one cycle after LEDshow
    }

    // return to RTOS system with a 1ms call back, allows time for Wi-Fi code
    vTaskDelay(1/portTICK_PERIOD_MS);
  }
}

// --------------------------------------------------------------------------------

void loop_Read_ICM() {
  // Read the ICM-42607-P registers every cycle
  // As gyro angle is calculated from gyro rate we must read them at precise intervals
  Wire.beginTransmission(ICM_address);              // Start communication with the MPU
  Wire.write(0x0B);                                 // Start reading ACCEL_XOUT_H at register OB hex
  I2C_Err = Wire.endTransmission(false);            // End the transmission
  delayMicroseconds(I2Cdel);                        // Allow ICM time to respond

  // As the registers are consequitive we read them all as a block
  if (!I2C_Err) {
    // No I2C bus error so preceed with the read, otherwise values do not change
    Wire.requestFrom(ICM_address, 12);              // Request 12 bytes from the ICM-42607-P sensor
    delayMicroseconds(I2Cdel);                      // Allow ICM time to respond
    AcX = Wire.read()<<8|Wire.read();               // read instantaneous Pitch acceleration, on X accelerometer
    AcY = Wire.read()<<8|Wire.read();               // read instantaneous Roll acceleration, on Y accelerometer
    AcZ = Wire.read()<<8|Wire.read();               // read instantaneous Vertical acceleration, on Z accelerometer
    GyX = Wire.read()<<8|Wire.read();               // read instantaneous Roll value, on X gyro
    GyY = Wire.read()<<8|Wire.read();               // read instantaneous Pitch value, on Y gyro
    GyZ = Wire.read()<<8|Wire.read();               // read instantaneous Yaw value, on Z gyro
  } else {I2C_ErrCnt++; return;}                    // record error event and exit
  // determine intermeasurement time for increased gyro angle precision
  if (GyT > 0) {
    GyTD = micros() - GyT;  // get the differential time between readings
    GyT = GyTD + GyT;       // set the timer for the next reading
  } else {
    // 1st reading after reset
    GyT = micros();
    GyTD = 4000;
  }
  ICMread = false;

  // Remember previous offset adjusted acceleromter values, but ignore AcZ
  // These are used in rate limiting calculations
  AcX_OAL = AcX_OA;
  AcY_OAL = AcY_OA;

  // Determine new offset adjusted values, by adding Cal. offsets to raw values
  // Z-axis is not included
  AcX_OA = AcX - AcOffX;
  AcY_OA = AcY - AcOffY;

  // Serial.print(String(AcX) + "," + String(AcX_OA) + "," + String(AcY) + "," + String(AcY_OA) + "\n");

  // ############################################################################
  // ICM crash detection
  // ############################################################################
  // If the ICM returns the same values on four consequitive readings
  if ((AcX_OA == AcX_OAL) && (AcY_OA == AcY_OAL)) {
    ICM_Crash++;
    if (ICM_Crash >= 4) {
      I2C_ICM = false; ICM_42607_Initialise();
      if (I2C_Err == 0) {I2C_ICM = true;}
    }
  } else {ICM_Crash = 0;}

  // ############################################################################
  // Gather long averages for accelerometer offsets
  // ############################################################################
  // These areaccessed via Monitor+ app, when AcOffEn can be set to zero
  if (!AcOffEn) {
    // Offset enable is switched off in Monitor+ app so determine long averages
    if (AcAvgDiv < 2048) {AcAvgDiv++;}   // set averaging count
    AcXSumL = AcXSumL + (long)AcX - AcXAvgL; AcXAvgL = AcXSumL/AcAvgDiv;
    AcYSumL = AcYSumL + (long)AcY - AcYAvgL; AcYAvgL = AcYSumL/AcAvgDiv;
    AcZSumL = AcZSumL + (long)AcZ - AcZAvgL; AcZAvgL = AcZSumL/AcAvgDiv;
  }

  // ############################################################################
  // Gather long averages for gyro offsets
  // ############################################################################
  if (!GyOffEn) {
    // Offset enable is switched off in Monitor+ app so determine long averages
    if (GyAvgDiv < 2048) {GyAvgDiv++;}   // set averaging count
    GyXSumL = GyXSumL + (long)GyX - GyXAvgL; GyXAvgL = GyXSumL/GyAvgDiv;
    GyYSumL = GyYSumL + (long)GyY - GyYAvgL; GyYAvgL = GyYSumL/GyAvgDiv;
    GyZSumL = GyZSumL + (long)GyZ - GyZAvgL; GyZAvgL = GyZSumL/GyAvgDiv;
  }

  // ############################################################################
  // Rate limit X & Y pitch and roll accelerometers
  // ############################################################################
  // Do pitch X-axis first
  AcDelta = AcX_OA - AcX_OAL; if (AcDelta < 0) {AcDelta = -AcDelta;}
  if (AcX_OAL < AcX_OA) { // acceleration is increasing
    if (AcDelta > AcRampRate) {AcX_OA = AcX_OAL + AcRampRate;}
  } else if (AcX_OAL > AcX_OA) { // acceleration is decreasing
    if (AcDelta > AcRampRate) {AcX_OA = AcX_OAL - AcRampRate;}
  }
  // Now do roll Y-axis
  AcDelta = AcY_OA - AcY_OAL; if (AcDelta < 0) {AcDelta = -AcDelta;}
  if (AcY_OAL < AcY_OA) { // acceleration is increasing
    if (AcDelta > AcRampRate) {AcY_OA = AcY_OAL + AcRampRate;}
  } else if (AcY_OAL > AcY_OA) { // acceleration is decreasing
    if (AcDelta > AcRampRate) {AcY_OA = AcY_OAL -AcRampRate;}
  }

  // ############################################################################
  // Limit X & Y accelerometer values, to prevent divide by zero errors
  // ############################################################################
       if (AcX_OA >  8192) {AcX_OA =  8192;}
  else if (AcX_OA < -8192) {AcX_OA = -8192;}
       if (AcY_OA >  8192) {AcY_OA =  8192;}
  else if (AcY_OA < -8192) {AcY_OA = -8192;}
       if (AcZ    >  8192) {AcZ    =  8192;}
  else if (AcZ    < -8192) {AcZ    = -8192;}
    
  // ############################################################################
  // Calculate accelerometer X & Y angles
  // ############################################################################
  AngAcX =  asin((float)AcX_OA/8192.0)* 57.296;  // Pitch accelerometer
  AngAcY = -asin((float)AcY_OA/8192.0)* 57.296;  // Roll accelerometer
  AngAcZ =  asin((float)AcZ   /8192.0)* 57.296;  // Z accelerometer, no offset applied
    
  // ############################################################################
  // Calculate gyro X & Y angles
  // ############################################################################
  // Determine gyro angles based on gyro rate o/p and sampling time.
  // Gyros are set at +/-250 deg/sec for 16-bit ADC = 32,767 FSD
  // to convert gyro value to deg/sec we divide reading by 131.068 (ie. 32,767/250)
  // we have recorded time period in GyTD in microseconds, between readings
  // so angle is = GyAng * (GyTD/1000000)/131.068
  // in this case GyTD is nominally 4000 microseconds (ie. 4ms loop)
  float zDiv = (float)GyTD/131068000;               // nominally 0.000305 at 4ms
  // Get gyro X angle - Roll
  GyX_OA = GyX - GyOffX;                            // Add the gyro calibration value
  AngGyX += (float)GyX_OA * zDiv;                   // Calculate the angle travelled during this 10ms loop and add this to the angle_gyro variable
       if (AngGyX >  180.0) {AngGyX =  180.0;}      // limit X gyro swing
  else if (AngGyX < -180.0) {AngGyX = -180.0;}      // limit X gyro swing
  // Get gyro Y angle - Pitch
  GyY_OA = GyY - GyOffY;                            // Add the gyro calibration value
  AngGyY += (float)GyY_OA * zDiv;                   // Calculate the angle travelled during this 10ms loop and add this to the angle_gyro variable
       if (AngGyY >  180.0) {AngGyY =  180.0;}      // limit X gyro swing
  else if (AngGyY < -180.0) {AngGyY = -180.0;}      // limit X gyro swing
  // Get gyro Z angle - Yaw
  GyZ_OA = GyZ - GyOffZ;                            // Add the gyro calibration value
  AngGyZ += (float)GyZ_OA * zDiv;                   // Calculate the angle travelled during this 10ms loop and add this to the angle_gyro variable
       if (AngGyZ >  180.0) {AngGyZ =  180.0;}      // limit X gyro swing
  else if (AngGyZ < -180.0) {AngGyZ = -180.0;}      // limit X gyro swing
  YawLast = Yaw;                                    // remember previous Yaw angle
  Yaw = AngGyZ;                                     // Get Yaw angle
  YawDif = Yaw - YawLast;                           // Change in Yaw angle
  // if some rotation is detected then set a timer, if not then
  // ignore the small drift by setting the angle to zero
  if (fabs(YawDif) >= 0.01) {YawCnt = 750;}
  // switch the angle to zero after 3 sec time-out
  if (YawCnt > 0) {
    // we are or have been rotating
    YawCnt--;
  } else {Yaw = 0.0; AngGyZ= 0.0;} // zero the rotation gyro

  //#############################################################################
  // ICM-42607-P gyro offset compensation
  //#############################################################################
  // As gyro angle is calculated from the rate/time an offset error can easily creep
  // in affecting the overall accuracy of the gyro. Particularly if the robot shakes
  // violently or falls over. So we use the noisier accelerometer angle to provide
  // a small ammount of offset adjustment as an integrator.
  if (GyrOnce) {
    // whilst balancing we use the accelerometer angles as small correction factors
    // this must be small, otherwise the angle will be too greatly affected by movement
    AngOffset = -(AngGyX - AngAcY)* GyAcMod;    // nominally 0.010
    AngGyX += AngOffset;                        // Correct the drift of the gyro angle with the accelerometer angle
    Roll = AngGyX;
    AngOffset = -(AngGyY - AngAcX)* GyAcMod;    // nominally 0.010
    AngGyY += AngOffset;                        // Correct the drift of the gyro angle with the accelerometer angle
    Pitch = AngGyY;
  } else {
    // at the start of balance use the accelerometer values as the initial angle
    AngGyX = AngAcY;                        // Initialise the gyro angle using the accelerometer
    AngGyY = AngAcX;                        // Initialise the gyro angle using the accelerometer
    Roll = AngGyX; Pitch = AngGyY;
    GyrOnce = true;
  }
}

// --------------------------------------------------------------------------------

void loop_PID() {
  // ############################################################################
  // ############################################################################
  //
  // PID controller code
  //
  // ############################################################################
  // ############################################################################
  // This is the main task which performs balance calculations every 4ms (250Hz)
  //#############################################################################
  // Check robot orientation using Z-axis accelerometer
  //#############################################################################
  // This data is averaged over 16 samples to filter out noise
  AcZ_Trck = AcZ_Trck + ((AcZ - AcZ_Trck)/16);
  if (AcZ_Trck < -5000) {
    Upright = 1;   // right way up by > 30°
  } else if (AcZ_Trck > 5000) {
    Upright = -1; // upside down > 30°
  } else if ((AcZ_Trck > -3000) && (AcZ_Trck < 3000)) {Upright = 0;} // on its side
  AcY_Trck = AcY_Trck + ((AcY_OA - AcY_Trck)/16);
  if (AcY_Trck < -5000) {
    Upside = 1;   // right-hand down by > 30°
  } else if (AcY_Trck > 5000) {
    Upside = -1; // left-hand down by > 30°
  } else if ((AcY_Trck > -3000) && (AcY_Trck < 3000)) {Upside = 0;} // on its backfront

  //#############################################################################
  // Pitch & Roll PID Controller Angle Error Calculations
  //#############################################################################
  // The balancing robot is angle driven. First the difference between the desired angel (setpoint) and actual angle (process value)
  // is calculated. The self_balance_pid_setpoint variable is automatically changed to make sure that the robot stays balanced all the time.
  // The (pid_setpoint - pid_output * 0.015) part functions as a speed limiting function.
  if (Pitch_En) {
    PtchPidErrLast = PtchPidErr;              // Store previous pitch PID error
    PtchPidErr = Pitch - PtchSbSp - PtchSp;   // Calculate new pitch error
    if ((PtchPidErr > -PID_db) && (PtchPidErr < PID_db)) {
      // Pitch deadband region so zero gains
      PtchPidErr = 0.0; PtchPidErrLast = 0.0; Ptch_i_mem = 0.0;
    }
    else if (Ptch_pid_output > BrkPnt || Ptch_pid_output < -BrkPnt) PtchPidErr += Ptch_pid_output * BrkVal ; // brake (0.015)
  } else {PtchPidErr = 0.0;}                  // Pitch error and angle are set to zero if axis is off
  if (Roll_En) {
    RollPidErrLast = RollPidErr;              // Store previous roll PID error
    RollPidErr = Roll - RollSbSp - RollSp;    // Calculate new pitch error
    if ((RollPidErr > -PID_db) && (RollPidErr < PID_db)) {
      // Roll deadband region so zero gains
      RollPidErr = 0.0; RollPidErrLast = 0.0; Roll_i_mem = 0.0;
    }
    else if (Roll_pid_output > BrkPnt || Roll_pid_output < -BrkPnt) RollPidErr += Roll_pid_output * BrkVal ; // brake (0.015)
  } else {RollPidErr = 0.0;}                  // Roll error and angle are set to zero if axis is off
  

  //#############################################################################
  // PID Gain Calculations
  //#############################################################################
  // Calculate the PID output values; always include the P element
  Ptch_P_Val = pid_p_gain * PtchPidErr;         // p - proportional element
  Ptch_pid_output = Ptch_P_Val;                 // output always include p element

  Roll_p_Val = pid_p_gain * RollPidErr;         // p - proportional element
  Roll_pid_output = Roll_p_Val;                 // output always include p element

  if (PID_i_En && StartEn) {
    // optional i - integrator element
    Ptch_i_mem += pid_i_gain * PtchPidErr;                // Calculate the I-controller value and add it to the pid_i_mem variable
         if (Ptch_i_mem >  pid_i_max) {Ptch_i_mem =  pid_i_max;}  // limit integrator output
    else if (Ptch_i_mem < -pid_i_max) {Ptch_i_mem = -pid_i_max;}  // limit integrator output
    Ptch_pid_output += Ptch_i_mem;                        // add in PID i gain value

    Roll_i_mem += pid_i_gain * RollPidErr;                // Calculate the I-controller value and add it to the pid_i_mem variable
         if (Roll_i_mem >  pid_i_max) {Roll_i_mem =  pid_i_max;}  // limit integrator output
    else if (Roll_i_mem < -pid_i_max) {Roll_i_mem = -pid_i_max;}  // limit integrator output
    Roll_pid_output += Roll_i_mem;                        // add in PID i gain value
  } else {Ptch_i_mem = 0.0; Roll_i_mem = 0.0;}            // i-gain not enabled
  
  if (PID_d_En && StartEn) {
    // optional d - deterministic element
    Ptch_d_Val = pid_d_gain * (PtchPidErr - PtchPidErrLast);
    Ptch_pid_output += Ptch_d_Val;                            // add in PID d gain value

    Roll_d_Val = pid_d_gain * (RollPidErr - RollPidErrLast);
    Roll_pid_output += Roll_d_Val;                            // add in PID d gain value
  } else {Ptch_d_Val = 0.0; Roll_d_Val = 0.0;}                // d-gain not enabled

  
  // #############################################################################
  // Auto-setpoint adjustment
  // #############################################################################
  // The self balancing point is adjusted when there is no forward or backwards demand.
  // This way the robot will always find it's balancing point
  // The trend function prevents the setpoint from being wound up via pushing or slopes
  if ((PtchSp == 0.00) && (PtchPidTune == 0)){
    // If the PID setpoint is zero degrees there is no external controller demand
    if (Ptch_pid_output < -PtchPidSb_db){
      PtchPidOpTrend--;
      if (PtchPidOpTrend > 0) {
        // swing just changed direction
        PtchPidOpTrend = 0;
      }
      if (PtchPidOpTrend < -PtchPidOpTrendLim) {PtchPidOpTrend = -PtchPidOpTrendLim;}
      else {PtchSbSp += PtchSbInc;}  // Increase the PtchSbSp if the robot is still moving forewards
    }
    else if(Ptch_pid_output > PtchPidSb_db){
      PtchPidOpTrend++;
      if (PtchPidOpTrend < 0) {
        // swing just changed direction
        PtchPidOpTrend = 0;
      }
      if (PtchPidOpTrend > PtchPidOpTrendLim) {PtchPidOpTrend = PtchPidOpTrendLim;}
      else {PtchSbSp -= PtchSbInc;}   // Decrease the self_balance_pid_setpoint if the robot is still moving backwards
    } else {PtchPidOpTrend = 0;}
  }
  
  if ((RollSp == 0.00) && (RollPidTune == 0)){
    // If the setpoint is zero degrees there is no external controller demand
    if (Roll_pid_output < -RollPidSb_db){
      RollPidOpTrend--;
      if (RollPidOpTrend > 0) {
        // swing just changed direction
        RollPidOpTrend = 0;
      }
      if (RollPidOpTrend < -RollPidOpTrendLim) {RollPidOpTrend = -RollPidOpTrendLim;}
      else {RollSbSp += RollSbInc;}  // Increase the self_balance_pid_setpoint if the robot is still moving forewards
    }
    else if (Roll_pid_output > RollPidSb_db){
      RollPidOpTrend++;
      if (RollPidOpTrend < 0) {
        // swing just changed direction
        RollPidOpTrend = 0;
      }
      if (RollPidOpTrend > RollPidOpTrendLim) {RollPidOpTrend = RollPidOpTrendLim;}
      else {RollSbSp -= RollSbInc;}   //Decrease the self_balance_pid_setpoint if the robot is still moving backwards
    } else {RollPidOpTrend = 0;}
  }


  //#############################################################################
  // Limit pid_outputs to pid_out_max
  //#############################################################################
  // Limit the PID-controllers to the maximum controller output
       if (Ptch_pid_output >  pid_out_max) {Ptch_pid_output =  pid_out_max;}
  else if (Ptch_pid_output < -pid_out_max) {Ptch_pid_output = -pid_out_max;}
       if (Roll_pid_output >  pid_out_max) {Roll_pid_output =  pid_out_max;}     // Limit the PI-controller to the maximum controller output
  else if (Roll_pid_output < -pid_out_max) {Roll_pid_output = -pid_out_max;}


  //#############################################################################
  // Limit output power during initial switch into self-balancing mode
  //#############################################################################
  if (PID_Output_Power < 100.0) {
    // PID_Output_Power can range from 0 - 100%
    Ptch_pid_output = Ptch_pid_output * PID_Output_Power/100.0;
    Roll_pid_output = Roll_pid_output * PID_Output_Power/100.0;
    PID_Output_Power += 1.0;  // slowly increase power output
  }
  
  if (SafeMode == 4) {
    // This code is only used during self-balancing mode
    //#############################################################################
    // Test for robot falling over or being over-tilted at any time  
    //#############################################################################
    if ((AcZ_Trck > -4096) || (fabs(Pitch) > Topple) || (fabs(Roll) > Topple)) {
      // If the robot tips over or the start variable is zero
      // balance angle exceeded so drop out of any active mode
      PWM_OFF();
      setMainMode(0); // return to default mode
      Display_Text2x16("Topple","Event",20);
      DispMode = 0;
      // Serial.println(F("Topple!"));
    }
    //#############################################################################
    // Test for Wii controller data
    //#############################################################################
    if (RxRec) {
      // convert Wii data into single byte binary format for use later
      // the balancebot does not use proportional joystick control
      received_byte = 0x00;                 // start with no demand
      if(JoyX < DbLLX)  received_byte |= 0b00000001;  // If the variable JoyX is smaller then 108 set bit 0 of the send byte variable
      if(JoyX > DbULX)  received_byte |= 0b00000010;  // If the variable JoyX is larger then 148 set bit 1 of the send byte variable
      if(JoyY < DbLLY)  received_byte |= 0b00001000;  // If the variable JoyY is smaller then 108 set bit 3 of the send byte variable
      if(JoyY > DbULY)  received_byte |= 0b00000100;  // If the variable JoyY is larger then 148 set bit 2 of the send byte variable
      if((CZ & 1) == 0) received_byte |= 0b00010000;  // If Z button pressed set bit 4
      if((CZ & 2) == 0) received_byte |= 0b00100000;  // If C button pressed set bit 5

      receive_counter = 0;                            // Reset the receive_counter variable
    } RxRec = false;                                  // clear the data received flag
      
    // for modes, like PILOT, we can overide the Wii demands to take control
    received_mem = received_byte;                     // remember Wii Rx demand
    
    if (receive_counter <= 25) {receive_counter ++;}  // The received byte will be valid for 25 program loops (100 milliseconds)
    else {
      // After 100 milliseconds the received byte is deleted
      received_byte = 0x00;
    }

    //#############################################################################
    // Wii Controller calculations
    //#############################################################################
    // 0b00000001  X pressed to the left
    // 0b00000010  X pressed to the right
    // 0b00000100  Y pressed to the front
    // 0b00001000  Y pressed to the back
    // 0b00010000  'Z' button pressed
    // 0b00100000  'C' button pressed
    // we inc/dec steering and drive factors so as not to introduce sudden movements
    //##############
    // Steering
    //##############
    if(received_byte & 0b00000001){
      // Joystick pressed left -X, so steer left
      if (Steer == 0.00) {GFX_Wait = 0;}
      if (received_byte & 0b00010000) {if (Steer < TurnMax) {Steer += 0.5;}} // 'Z' button reduces steering speed
      else {if (Steer < TurnMin) {Steer += 0.5;}}
    } else if (received_byte & 0b00000010){
      // Joystick pressed right +X, so steer right
      // drive motors appear weaker in this direction, so use a larger factor
      if (Steer == 0.00) {GFX_Wait = 0;}
      if(received_byte & 0b00010000){if (Steer > -TurnMax) {Steer -= 0.5;}} // 'Z' button reduces steering speed
      else {if (Steer > -TurnMin) {Steer -= 0.5;}}
    } else {
      // return steering to zero more rapidly
      if (Steer > 0.5) {Steer -= 1.0;}
      else if (Steer < -0.5) {Steer += 1.0;}
      else {Steer = 0.00;}
    }
  
    //##############
    // Drive forwards/backwards
    //##############
    // A Y-axis Joystick demand moves the Pitch setpoint, which is normally 0.0
    if(received_byte & 0b00000100){
      // Joystick pressed forward +Y
      // Limit power in forward drive
           if (PtchSp < DrvSpMin) {PtchSp += 0.01;}
      else if (PtchSp < DrvSpMax) {PtchSp += 0.001;}
    } else if (received_byte & 0b00001000){
      // Joystick pressed backward -Y
           if (PtchSp > -DrvSpMin) {PtchSp -= 0.01;}
      else if (PtchSp > -DrvSpMax) {PtchSp -= 0.001;}
    } else {
      // Joystick released, so slow down
      if (PtchSp >= 0.02) {PtchSp -= 0.01;}
      else if (PtchSp <= -0.02) {PtchSp += 0.01;}
      else {PtchSp = 0.00;} // ensure it returns to zero, to enable auto self-balance function
    }
  } else {received_byte = 0x00;}
  

  //#############################################################################
  // Motor PWM demands
  //#############################################################################
  if (StartEn) {
    // PWM offset and scaling is performed within driveMotors() functions
    // The polarity of PID outputs is reversed to drive motors correctly
    driveMotors(-Ptch_pid_output,-Roll_pid_output,Steer);
  }
}

// --------------------------------------------------------------------------------

void OnDataRecv(const esp_now_recv_info_t *info, const uint8_t *incomingData, int len) {
  // Call-back function oocurs when data has been received
  // This needs to be treated like an interrupt, so speed is the essence here
  // memcpy moves data in a block, instead of individual bytes if using for() loop
  memcpy(WiFiRxMacAddr, info->src_addr, 6);   // get the senders MAC/broadcast address
  memcpy(WiFiTxMacAddr, info->des_addr, 6);   // get destination MAC address, which should be assigned peer
  memcpy(&Rx_Buff, incomingData, len);        // copy the received data into Rx_Buff array
  Rx_len = len - 1; Rx_Pnt = 0;               // len contain the length of data
  RxRecvEvent = true;                         // set the flag to say data received
}

// --------------------------------------------------------------------------------

void OnDataRecvHandler() {
  // Called from the main loop() when has been set, to handle the received data
  RxRecvEvent = false;
  // incoming data will always be two or more characters, ie. text == $+++
  if (Rx_len < 2) {Serial.println("Rx Error - no Rx data"); return;}

  WiFiRxGetChecksum(Rx_len);  // generate a checksum from Rx data
  if (Rx_Chk != Rx_Buff.ESPdata[Rx_len]) {
    Serial.println("Rx Error - checksum");
  }
  // set flags before exiting
  WiFiRxBytes += Rx_len; Rx_Task = 0; WiFiRx = true;
  WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
  WiFiRxRec = true; // set flag indicating data has been recieved from Wii Transceiver
  //  Serial.print("Bytes received: "); Serial.println(Rx_len);
}

// --------------------------------------------------------------------------------

void OnDataSent(const uint8_t *mac_addr, esp_now_send_status_t status) {
  // Callback when data is sent over Wi-Fi, determines success or failure
  // t1 = micros();
  // Serial.print("Send Status: ");
  if (WiFiConnected) {
    // the link is already active so handle normal success/failure scenarios
    if (status == 0){
      // Serial.println("Success " + String(t1-t0));
      WiFiTx = true;    // flag WiFi Tx success
      WiFiTx_CB = 1;    // set flag to indicate that a call back state has occured
      WiFiTxErrCnt = 0; // clear the error flag in case of a glitch
      WiFiPing = 50;    // set link time-out to 2 seconds
    } else {
      // Serial.println("Fail " + String(t1 - t0));
      WiFiTx = false;   // flag WiFi Tx failure
      WiFiTx_CB = -1;   // set flag to indicate that a resend is required
      WiFiTxErrCnt++;
    }
  } else {
    // not connected so we must have tried to connect and may get an error
    if (status == 0){
      // successful so enter the connected state
      Serial.println("Connected!");
      WiFiConnected = true;
      WiFiPing = 50;    // set link time-out to 2 seconds, Wii Transceiver will keep setting this with Wii data
      WiFiTxErrCnt = 0; // reset error count for this connection
    } else {
      // Serial.println("Status = " + String(status));
      WiFiTxErrCnt++;
    }
    WiFiTx_CB = 1;
  }
}

// --------------------------------------------------------------------------------

void PrintTxHandler() {
  // This function handles the contents of the PrintTx string, which is used
  // to send data to either the serial port or WiFi system.
  // It is called at a 40ms rate, but speeds up when more data is to be sent.
  if (!Echo) {PrintTx = "";}   // Echo is false so empty print buffer
  if (PrintTx.length() > 0) {
    // characters in string buffer so send some/all of them
    print40ms = millis() - 34; // shorten the delay to 6ms for more data to send
    Tx_PingCnt = 0;            // reset the WiFi ping counter
    // PrintTgt = 0; // force printing to serial port
    if (PrintTgt == 0) {
      // default uses the serial port
      if (Serial.availableForWrite() >= 64) {
        if (PrintTx.length() <= 64) {
          Serial.print(PrintTx);
          // PrintTx = "";   // empty the buffer
          PrintTx.clear();   // empty the buffer
        } else {
          Serial.print(PrintTx.substring(0,64));
          PrintTx = PrintTx.substring(64);
        }
      }
    } else {
      // text data has been received previously from WiFi, so respond using ESP_NOW
      // ensure that we have had a call back from the previous frame
      if (WiFiConnected) {
        // only trannsmit PrintTx data if connected
        if (WiFiTx_CB == 1) {
          WiFiTx_CB = 0;  // clear the call back flag when sending
          if (PrintTx.length() <= 32) {
            WiFiTx_len = PrintTx.length();
            PrintTx.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
            WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
            WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
          //  t0 = micros();
            esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
            // PrintTx = "";   // empty the buffer
            PrintTx.clear();   // empty the buffer
          } else {
            WiFiTx_len = 32;
            Any$ = PrintTx.substring(0,WiFiTx_len);
            Any$.toCharArray(Tx_Buff.ESPdata,WiFiTx_len + 1);
            WiFiTxBytes += WiFiTx_len;  // track number of bytes sent
            WiFiTxGetChecksum(WiFiTx_len); Tx_Buff.ESPdata[WiFiTx_len] = WiFiTx_Chk; WiFiTx_len++;
          //  t0 = micros();
            esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
            PrintTx = PrintTx.substring(32);
          }
        } else if (WiFiTx_CB == -1) {
          // an error has occured so resend the data
          if (WiFiTryCnt < 5) {
            WiFiTryCnt++;   // track the number of retries
            WiFiTx_CB = 0;  // clear the call back flag when sending
          //  t0 = micros();
            esp_now_send(broadcastAddress, (uint8_t *) &Tx_Buff, WiFiTx_len);
          } else {
            // too many resends so disconnect WiFi link
            WiFiDisconnect();
            Serial.println("Failed to Tx!");
          }
        }
      } else {
        // not connected so clear PrintTxdata
        PrintTx = "";
      }
    }
  }
}

// --------------------------------------------------------------------------------

void run_POST() {
  // run once at boot up
  Serial.println("\n\n\n" + Release + "\n");
  Serial.print(F("Released: ")); Serial.println(Date);
  Serial.println(F("Running POST..."));
  
  // Print TEST mode?... holding down SW1 during boot forces this condition
  if (!TEST) {if (!digitalRead(sw0Pin) || !digitalRead(sw1Pin)) {TEST = true;}}
  if (TEST) {Serial.println(F("TEST mode active"));}

  // Test for ICM-42607-P 3-axis motion sensor and initialise
  Wire.beginTransmission(ICM_address);
  I2C_Err = Wire.endTransmission();
  if (I2C_Err == 0) {
    Serial.println(F("Initialising ICM..."));
    I2C_ICM = true; ICM_42607_Initialise();
  } else {I2C_ICM = false; Serial.println(F("ICM-42607-P Not Found!"));}
  
  // Set up battery reading
  initBatteryRead();

  // Set the LED ring low green
  LED_Fill(0,4,0);
  FastLED.show();

  // TFT display "Hello", centre display
  tft.setTextColor(GC9A01A_WHITE);
  TFT_Text_Ctr(120,100,5,"Hello");
  delay(2000);


  if (!TEST) {
    // Not in TEST mode.
  } else {
  }

  Serial.println(F("POST completed"));
}

// --------------------------------------------------------------------------------

void setDefaults() {
  // Load default values
  setDefPID();                    // set all PID default values

  // Declare Global variable initial values
  AcAvgDiv = 0;                   // accelerometer long averaging factor
  AcOffEn = true;                 // if == false then display RAW Acc values & run averaging
  AcZ_Trck = -8192;               // Z-axis tracking filter for orientation   
  AcY_Trck = 0;                   // Y-axis tracking filter for orientation   
  BatSum = Bat8v2 * 20;           // preload averaging sum with 20x max value
  Blink = false;                  // LED blink clock
  BlinkCnt = 0;                   // counter for LED Blink clock
  Bright = 255;                   // FastLED brightness
  C_Button = 0;                   // button pressed state; 0 = UP, 1 = Down
  DispClr = false;                // set == true to clear the memory contents
  DispCnt = 0;                    // counter used in display updating
  DispDel = 0;                    // >0 display delay counter, stops updates over I2C
  DispDelGbl = 0;                 // global DispDel offset, used to adjust display frame rates
  DispDisp = false;               // set = true to immediately display memory contents
  DispEn = true;                  // if == true then generate display messages, otherwise inhibit
  DispLock = false;               // if == true then don't allow display mode changes
  DispMode = 0;                   // the type of display for a given mode
  DispMon = false;                // == true if connected to OLED Mirror app
  DispOveride = 0;                // if DispLock == true then use this overide mode
  DispTx = 0;                     // counter used with display mirror app
  DmDwn = false;                  // == true whilst mouse is clicked in MOnitor+ mode
  DmSft = false;                  // == true if SHIFT key held down when mouse clicked in Monitor+
  DriveEn = false;                // if == true then output PWM to motors
  Echo = true;                    // if true send PrintTx print messages
  end4ms = 2;                     // flag indicating end of timer run
  ESP_NOW_BA = -1;                // broadcast address pointer, -1 to start with
  ESP_NOW_Init = false;           // == true once ESP-NOW has been initiaiised
  Eye_En = true;                  // == true if eye engine is enabled, default = true
  GFX_Cnt = 0;                    // eye GFX move increment counter
  GFX_Del = 0;                    // millisecond delay timer, holds off GFX_Run tasks
  GFX_Eng = 0;                    // eye GFX move engine task pointer
  GFX_Last = 1;                   // previous GFX_Mode value, default = 1
  GFX_Mode = 1;                   // eye drawing mode, 0 - 2, default = 1
  GFX_R0 = 70;                    // eye ball radius, full eye
  GFX_R1 = GFX_R0/2;              // eye cornea radius
  GFX_Rot = 2;                    // Eye display rotation value, default = 2 for upright
  GFX_RST = true;                 // if == true then reset GFX modes for eye display, default = true on boot
  GFX_Run = GFX_EyeXY;            // set > 0 to initiate a TFT draw task in Core0, default = 0, off
  GFX_Task = 0;                   // task pointer for TFT drawing sequence, default = 0, reset
  GFX_Txt = 0;                    // set > 0 to initiate Core0, text drawing tasks
  GFX_Wait = 0;                   // eye GFX move engine pause/wait counter
  GFX_X = 120;                    // GFX drawing cursor X co-ordinate
  GFX_Y = 120;                    // GFX drawing cursor Y co-ordinate
  GFX_XTgt = 120;                 // GFX X target values
  GFX_YTgt = 120;                 // GFX Y target values
  G_once = true;                  // set == true to force initial graphing mode, default = false
  GPmax = 0;                      // end of graphing list, for Scope mode
  GPmin = 0;                      // start of graphing list, for Scope mode
  GPnn = 0;                       // pointer to current graphing items, default = 0
  GP_Title = true;                // if == true then report a title in Scope graphing mode
  GyAvgDiv = 0;                   // accelerometer long averaging factor
  GyOffEn = true;                 // if == false then display RAW Gyro values
  GyrOnce = false;                // sets the initial angles for the gyros based on the accelerometers
  GyT = 0;                        // time of gyro readings in milliseconds
  I2C_ErrCnt = 0;                 // counter that tracks I2C_Err events
  ICM_Crash = 0;                  // ICM crash counter, default = 0
  ICM_En = true;                  // if == true then read ICM over I2C
  ICMread = false;                // == true whilst reading ICM
  LedCnt = 0;                     // counter used in LED sequencing
  LED_Del = 0;                    // delay timer used in LED functions
  LedInc = 1;                     // increment used in LED sequencing
  LedLast = 0;                    // previous task when change set
  LedMode = 1;                    // LED mode pointer, default = 1, slow rotation
  LedNop = false;                 // if true then don't update LEDs
  LEDshow = false;                // LED show flag for mouth LEDs
  LEDskip = false;                // if == true then skip a Loop0 1ms cycle
  LED_Task = 0;                   // LED task pointer
  LEDVal = 0;                     // values used in LED tasks
  MainMode = 0;                   // controls which tasks are performed, default = 0
  MainRun = false;                // flag runs MainMode tasks every 8ms
  MainSkip = 0;                   // counter used to skip MainMOde tasks
  MainTest = 0;                   // used to test specific MainModes
  ModeCnt = 0;                    // counter used in MainModes
  ModeStr$ = "";                  // mode string displayed in displays
  ModeTask = 0;                   // task poiinter used in MainModes
  Pitch_En = false;               // Pitch PID enable, default = false
  Ptch_d_Val = 0.0;               // Pitch d-gain value
  Ptch_i_mem = 0.0  ;             // Pitch PID i-gain memory
  PtchPidErrLast = 0.0;           // Previous Pitch angle error, relative to setpoints
  Ptch_pid_output = 0.0;          // Pitch PID p-gain output
  PtchPidOpTrend = 0;             // Pitch output trend
  PtchPidTune = 0;                // > 0 when in tuning mode
  Ptch_P_Val = 0.0;               // Pitch p-gain value
  PtchPidOpTrend = 0;             // tracker used in self-balancing
  Ping = 0;                       // 'ping' counter
  PrintTgt = 0;                   // 0 == serial port, 1 == WiFi ESP-NOW
  PWM_A = 0;                      // PWM drive for motor A
  PWM_A0 = 0;                     // PWM value written to H-bridge driver
  PWM_A1 = 0;                     // PWM value written to H-bridge driver
  PWM_A_ = 0;                     // PWM off state value, 0 = 00 or 1 = FF
  PWM_B = 0;                      // PWM drive for motor B
  PWM_B0 = 0;                     // PWM value written to H-bridge driver
  PWM_B1 = 0;                     // PWM value written to H-bridge driver
  PWM_B_ = 0;                     // PWM off state value, 0 = 00 or 1 = FF
  PWM_C = 0;                      // PWM drive for motor C
  PWM_C0 = 0;                     // PWM value written to H-bridge driver
  PWM_C1 = 0;                     // PWM value written to H-bridge driver
  PWM_C_ = 0;                     // PWM off state value, 0 = 00 or 1 = FF
  PWM_D = 0;                      // PWM drive for motor D
  PWM_D0 = 0;                     // PWM value written to H-bridge driver
  PWM_D1 = 0;                     // PWM value written to H-bridge driver
  PWM_D_ = 0;                     // PWM off state value, 0 = 00 or 1 = FF
  PWM_Max = 255;                  // max PWM value, constrained by V_max
  readTask = 0;                   // task pointer used in decoding received control data
  received_cmd = false;           // set true if valid control has ben recevied
  Roll = 0.0;                     // side to side rolling angle
  Roll_d_Val = 0;                 // Roll PID d-gain value
  Roll_En = false;                // Roll PID ewnable, default = false;
  Roll_i_mem = 0.0;               // Roll PID i-gain memory
  RollPidErr = 0.0;               // Roll angle error, relative to setpoints
  RollPidErrLast = 0.0;           // Previous Roll angle error, relative to setpoints
  Roll_pid_output = 0.0;          // Roll PID output
  RollPidOpTrend = 0;             // Roll output trend
  RollPidTune = 0;                // > 0 when in tuning mode
  Roll_p_Val = 0.0;               // Roll p-gain value
  RollPidOpTrend = 0;             // tracker used in self-balancing
  run4ms = false;                 // set == true to run 4ms tasks
  RTxTimeout = 0;                 // counter used to auto-reset if WiFi fails
  Rx_Chk = 0;                     // Rx checksum
  RxCZ = 0;                       // buffered received CZ value
  RxJoyX = 0;                     // buffered received JoyX value
  RxJoyY = 0;                     // buffered received JoyY value
  Rx_len = 0;                     // length of WiFi received data block
  RxRec = false ;                 // set true when a valid frame of data is received
  RxRecvEvent = false;            // set == true when a OnDataRecv() call back event has occured
  RxState = 0;                    // receiver state machine state
  Rx_Task = 0;                    // pointer user by Rx state machine
  RxVal = -1;                     // value received from serial Rx
  SafeMode = 0;                   // Safety mode state pointer
  SetLEDs = false;                // if == true then Core0 sets a new LED mode
  StartEn = false;                // true to perform motor calcs and enable output
  Steer = 0.0;                    // steering demand, +ve = right, -ve = left, max 255
  sw0Cnt = 0;                     // button switch counter
  sw0LastState = HIGH;            // previous state of button switch, HIGH/LOW
  sw0State = HIGH;                // stat of read button switch pin
  sw0Timer = 0;                   // timer used to detemine button sequences
  sw0WaitHi = false;              // set HIGH to wait for SW1 release
  sw1Cnt = 0;                     // button switch counter
  sw1LastState = HIGH;            // previous state of button switch, HIGH/LOW
  sw1State = HIGH;                // stat of read button switch pin
  sw1Timer = 0;                   // timer used to detemine button sequences
  sw1WaitHi = false;              // set HIGH to wait for SW1 release
  Task8ms = 0;                    // task pointer for 8ms tasks
  Task20ms = 0;                   // task pointer for 20ms tasks
  Tx_PingCnt = 0;                 // 1 second Tx ping timer
  Txt_Del = 0;                    // text delay timer counter
  Txt_En = true;                  // if == true then enable drawing of text in eye display
  Txt_Mode = 0;                   // Text display function pointer
  Upright = 1;                    // either -1,0,1 depending on orientation
  Upside = 0;                     // either -1,0,1 depending on orientation
  USB = false;                    // == true if initial battery is below 5.5v
  WiFiCntC = 0;                   // C button counter for WiFi enable
  WiFiConCnt = 0;                 // counter used in connection retry times
  WiFiConnected = false;          // == true if a WiFi connection has been established
  WiFiEn = false;                 // true when WiFi is enabled with prolonged 'Z' button activity
  WiFiOff = false;                // if == true then kill WiFi, default = false
  WiFiPing = 0;                   // Rx counter used to maintain connected state
  WiFiRx = false;                 // true when a block of data has been received
  WiFiRxBytes = 0;                // number of bytes receiv ed over the WiFi link
  WiFiRxErr = 0;                  // number of receive errors in the WiFi link
  WiFiRxRec = false;              // true when a block of data has been successfully received
  WiFiTryCnt = 0;                 // WiFi try to connect count
  WiFiTryNum = 0;                 // WiFi try to connect total count
  WiFiTryOnce = true;             // == true if first pass of trying to connect
  WiFiTx = false;                 // true when a block of data has been sent successfully
  WiFiTxBytes = 0;                // number of bytes receiv ed over the WiFi link
  WiFiTx_CB = 1;                  // == 1 call back is clear to send, == -1 resend
  WiFiTx_Chk = 0;                 // Tx checksum
  WiFiTx_len = 0;                 // >0 when WiFi Tx buffer contains data to be sent
  WiFiTxErr = 0;                  // WiFi data error
  WiFiTxErrCnt = 0;               // WiFi Tx error count, leads to disconnection
  WiiType = 0;                    // device type, 0 == Nunchuk, 1 == Classic
  Z_Button = 0;                   // button pressed state; 0 = UP, 1 = Down
  Z_Dn = 0;                       // button pressed state; 0 = UP, 1 = Down
  Z_Mode = false;                 // joystick motion flag, normally false, if == true then slide mode
}

// --------------------------------------------------------------------------------

void setDefPID() {
  // Declare PID controller coeficients
  //###################################################
  // When starting from scratch, use the default values
  //###################################################
  // These values are what I used in my project, but they are not guaranteed to
  // work for you. You should expect to have to tune your PID controllers.
  // It is recommended that you start with the following:
  // PID_db = 0.0;                   // PID controller deadband angle
  // pid_p_gain = 10.0;              // Gain setting for the P-controller (0.0 - 100.0), default = 10.0
  // pid_i_gain = 0.00;              // Gain setting for the I-controller (0.0 - 10.0), default = 0.0
  // pid_i_max =  50.0;              // i-gain limit, 0.0 - 255.0
  // pid_d_gain =  0.00;             // Gain setting for the D-controller, default = 0.0
  // 
  BrkPnt = 10.0;                  // Threshold at which PID brake is applied, default = 10.0
  BrkVal = 0.015;                 // PID brake multiplier value, default = 0.015
  DrvSpMax = 5.0;                 // Max drive setpoint
  DrvSpMin = 2.0;                 // Min drive setpoint
  GyAcMod = 0.005;                // Gyro angle modified by Acc angle
  MtrMin = 20;                    // minimum motor PWM, to overcome gearbox stiction, default = 20
  PID_db = 0.0;                   // PID controller deadband angle
  PID_d_En = true;                // PID d-gain enable true=ON, false=OFF, default = true
  pid_p_gain = 24.0;              // 88.0 Gain setting for the P-controller (0.0 - 100.0), default = 5.0
  PID_i_En = true;                // PID i-gain enable true=ON, false=OFF, default = false
  pid_i_gain = 0.60;              // 4.0 Gain setting for the I-controller (0.0 - 10.0), default = 0.0
  pid_i_max = 255.0;              // i-gain limit, 0.0 - 255.0
  pid_d_gain = 60.00;             // 32.0 Gain setting for the D-controller, default = 0.0
  pid_out_max = 255.0;            // limit PID output to prevent PWM overload
  PtchPidSb_db = 0.0;             // Self-balancing dead band limit, default = 0.0
  PtchSbInc = 0.0001;             // Pitch self-balance increment, default = 0.001
  PtchSbSpSa = -0.85;             // Pitch self-balance setpoint start angle, default = 0.0
  PtchSp = 0.00;                  // nominal Pitch setpoint, start angle
  RollPidSb_db = 0.0;             // Self-balancing dead band limit
  RollSbInc = 0.0001;             // Roll self-balance increment, default = 0.001
  RollSbSpSa = -0.1;              // Roll self-balance setpoint start angle, default = 0.0
  RollSp = 0.00;                  // nominal Roll setpoint, start angle
  TurnMax = 32.0;                 // Max turn speed
  TurnMin = 16.0;                 // Min turn speed
  V_max = 7.60;                   // voltage limit for motor PWM, 6.6 - 8.2v, default = 7.6
  CalcLimits();                   // determine and set soft limits
}

// --------------------------------------------------------------------------------

void setDMs() {
  // Sets the DM references to their default values of -99
  DM_Batt = -99;              // Battery display assigned Display Mode
  DM_BatPlus = -99;           // Battery plus display assigned Display Mode
  DM_BatWarn = -99;           // Battery warning <20% display assigned Display Mode
  DM_Control = -99;           // Control display assigned Display Mode
  DM_DriveCtrl = -99;         // Drive Ctrl display assigned Display Mode
  DM_EyeEng = -99;            // Eye Engine display assigned Display Manager
  DM_I2C = -99;               // I2C display assigned Display Mode
  DM_Intro = -99;             // Intro display assigned Display Mode
  DM_IcmLvl = -99;            // 'ICM Levels' display assigned Display Mode
  DM_ICM_Acc = -99;           // 'ICM Acc.' display assigned Display Mode
  DM_IcmAccAng = -99;         // 'ICM Acc Angles' display assigned Display Mode
  DM_IcmGyrAng = -99;         // 'ICM Gyro Angles' display assigned Display Mode
  DM_ICM_Gyros = -99;         // 'ICM Gyros' display assigned Display Mode
  DM_IcmRot = -99;            // 'ICM Rotation' display assigned Display Mode
  DM_IcmWarn = -99;           // 'ICM Warning' display assigned Display Mode
  DM_JoyNorm = -99;           // Joystick normal values display assigned Display Mode
  DM_JoyOff = -99;            // Joystick values and offsets display assigned Display Mode
  DM_Limits = -99;            // Limits display assigned Display Mode
  DM_MtrDemo = -99;           // Motor demo assigned Display Mode
  DM_MtrTest = -99;           // Motor test assigned Display Mode
  DM_PID = -99;               // PID display assigned to Display Mode
  DM_PtchYaw = -99;           // Pitch n Yaw display assigned Display Mode
  DM_SafeMode = -99;          // SafeMOde display assigned Display Mode
  DM_SaPtchRoll = -99;        // Start angle Pitch & Roll Display Mode
  DM_Status_A = -99;          // Status A display assigned Display Mode
  DM_Status_B = -99;          // Status B display assigned Display Mode
  DM_WiFi = -99;              // WiFi display assigned Display Mode
}

// --------------------------------------------------------------------------------

void synchLoopTimers() {
  // reset main loop timers
  while (millis() < 40) {}    // we must wait until millis() is at least 40ms from reset

  next_4ms = millis();        // set loop timer
  print40ms = millis() - 1;   // 40ms variable print timer
  // Serial.print("synchLoopTimers()");
}

// --------------------------------------------------------------------------------
