// ################################################################################
//
//  Cal_Offsets v0.0
//
//  Released:  28/01/2023
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    micro: NANO

    Elements of this software were taken from the public domain. It uses
    libraries which must be installed on your system.

    This code is provided as a means of gathering the accelerometer and gyroscope
    offsets present in your MPU6050. These devices are good, but not perfect, so
    use this code to eliminate their undesireable offsets.

    Note that not all MPU6050 devices are happy to work at 5v over the I2C bus, and
    much prefer 3v3 operation. So again this code will help you identify devices
    that will work well for you.
*/
// Declare libraries
#include <Wire.h>           // include the Wire.h library for I2C data transfer

// Define constants
#define Acc_X_address 0x3B  // MPU ACCEL_XOUT_H register address
#define Acc_Y_address 0x3D  // MPU ACCEL_YOUT_H register address
#define Acc_Z_address 0x3F  // MPU ACCEL_ZOUT_H register address
#define Gyr_X_address 0x43  // MPU GYRO_XOUT_H register address
#define Gyr_Y_address 0x45  // MPU GYRO_YOUT_H register address
#define Gyr_Z_address 0x47  // MPU GYRO_ZOUT_H register address
#define MPU_address 0x68    // MPU-6050 I2C bus address (0x68 or 0x69)

// configuration
String Release = "\n\n\nCal_Offsets v0.\n";

// Declare and initialise global variables
int Acc_X;                  // raw 16-bit X axis accelerometer value read from the MPU
long Acc_X_Sum;             // cumulative Acc_X for averaging
int Acc_Y;                  // raw 16-bit Y axis accelerometer value read from the MPU
long Acc_Y_Sum;             // cumulative Acc_Y for averaging
int Acc_Z;                  // raw 16-bit Z axis accelerometer value read from the MPU
long Acc_Z_Sum;             // cumulative Acc_Z for averaging
int Gyro_X;                 // raw 16-bit X axis gyro value read from the MPU
long Gyr_X_Sum;             // cumulative Gyro_X for averaging
int Gyro_Y;                 // raw 16-bit Y axis gyro value read from the MPU
long Gyr_Y_Sum;             // cumulative Gyro_Y for averaging
int Gyro_Z;                 // raw 16-bit Z axis gyro value read from the MPU
long Gyr_Z_Sum;             // cumulative Gyro_Z for averaging

// --------------------------------------------------------------------------------

void setup() {
  // Start the serial port and initialise the MPU6050
  Serial.begin(115200);      // Open serial port at 115200 bps
  Serial.println(Release);

  Wire.begin();               //Start the I2C bus as master
  
  //By default the MPU-6050 sleeps. So we have to wake it up.
  // note that we set scaling factors in the MPU to match those used in the robot
  Wire.beginTransmission(MPU_address);   // Set the I2C device address for MPU
  Wire.write(0x6B);           // Write 6B hex to the PWR_MGMT_1 register
  Wire.write(0x00);           // Set the register bits as 00000000 to activate the MPU
  Wire.endTransmission();     // End the transmission
  
  //Set the full scale of the gyro to +/- 250 degrees per second
  Wire.beginTransmission(MPU_address);   // Set the I2C device address for MPU
  Wire.write(0x1B);           // Write to the GYRO_CONFIG register (1B hex)
  Wire.write(0x00);           // Set the register bits as 00000000 (250dps full scale)
  Wire.endTransmission();     // End the transmission
  
  //Set the full scale of the accelerometer to +/- 4g.
  Wire.beginTransmission(MPU_address);   // Set the I2C device address for MPU
  Wire.write(0x1C);           // Write to the ACCEL_CONFIG register (1A hex)
  Wire.write(0x08);           // Set the register bits as 00001000 (+/- 4g full scale range)
  Wire.endTransmission();     //End the transmission with the gyro
  
  //Set some filtering to improve the raw data.
  Wire.beginTransmission(MPU_address);   // Set the I2C device address for MPU
  Wire.write(0x1A);           // Write to the CONFIG register (1A hex)
  Wire.write(0x03);           // Set the register bits as 00000011 (Set Digital Low Pass Filter to ~43Hz)
  Wire.endTransmission();     //End the transmission with the gyro

  delay(1000);                // allow the MPU 1 second to settle down
}

// --------------------------------------------------------------------------------

void loop() {
  // This main loop repeats every 5 seconds
  // in order for the serial port to be maganed in the background we use the yield()
  // function, otherwise the code would crash.

  Serial.println("Taking 1000 readings...\n");
  // clear variables for repeat run
  Acc_X_Sum = 0; Acc_Y_Sum = 0; Acc_Z_Sum = 0;
  Gyr_X_Sum = 0; Gyr_Y_Sum = 0; Gyr_Z_Sum = 0;

  Serial.println("AccX\tAccY\tAccZ\tGyrX\tGyrY\tGyrZ");
  for(int zTimes = 0; zTimes < 1000; zTimes++) { // perform 1000 loops
    Wire.beginTransmission(MPU_address);   // Set the I2C device address for MPU
    Wire.write(Acc_X_address);             // Set MPU register address pointer to 0x3B hex
    Wire.endTransmission();                // End the transmission

    Wire.requestFrom(MPU_address, 6);      // Request 6 bytes from the MPU
    Acc_X = Wire.read()<<8|Wire.read();   // get Hi + Lo bytes and combine into one 16-bit integer
    Acc_Y = Wire.read()<<8|Wire.read();   // get Hi + Lo bytes and combine into one 16-bit integer
    Acc_Z = Wire.read()<<8|Wire.read();   // get Hi + Lo bytes and combine into one 16-bit integer
    Wire.endTransmission();   //End the transmission with the gyro
    Acc_X_Sum+= Acc_X; Acc_Y_Sum+= Acc_Y; Acc_Z_Sum+= Acc_Z;
    yield();

    Wire.beginTransmission(MPU_address);   // Set the I2C device address for MPU
    Wire.write(Gyr_X_address);             // Set MPU register address pointer to 0x3B hex
    Wire.endTransmission();                // End the transmission

    Wire.requestFrom(MPU_address, 6);      // Request 6 bytes from the MPU
    Gyro_X = Wire.read()<<8|Wire.read();   // get Hi + Lo bytes and combine into one 16-bit integer
    Gyro_Y = Wire.read()<<8|Wire.read();   // get Hi + Lo bytes and combine into one 16-bit integer
    Gyro_Z = Wire.read()<<8|Wire.read();   // get Hi + Lo bytes and combine into one 16-bit integer
    Wire.endTransmission();   //End the transmission with the gyro
    Gyr_X_Sum+= Gyro_X; Gyr_Y_Sum+= Gyro_Y; Gyr_Z_Sum+= Gyro_Z;
    yield();

    // wait for previous serial data to be sent
    Serial.flush();
    // print out the results on the serial monitor as comma separated variables
    Serial.print(Acc_X); Serial.print("\t");
    Serial.print(Acc_Y); Serial.print("\t");
    Serial.print(Acc_Z); Serial.print("\t");
    Serial.print(Gyro_X); Serial.print("\t");
    Serial.print(Gyro_Y); Serial.print("\t");
    Serial.println(Gyro_Z);
  }
  // print average values
  Serial.println("\nAccX\tAccY\tAccZ\tGyrX\tGyrY\tGyrZ");
  Serial.print(Acc_X_Sum/1000); Serial.print("\t");
  Serial.print(Acc_Y_Sum/1000); Serial.print("\t");
  Serial.print(Acc_Z_Sum/1000); Serial.print("\t");
  Serial.print(Gyr_X_Sum/1000); Serial.print("\t");
  Serial.print(Gyr_Y_Sum/1000); Serial.print("\t");
  Serial.println(Gyr_Z_Sum/1000);

  // at the end of the run we can either stop indefinately in the while() loop
  // or if you comment out this line, repeat after a delay
  while (true) {yield();}
  Serial.println("Pausing before repeat.\n");
  delay(5000);
}

// --------------------------------------------------------------------------------


