// ################################################################################
//
//  Balance Bot Remote v0.03 Beta
//
//  Released:  01/09/2017
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Elements of this software were taken from the public domain. It uses
    libraries which must be installed on your system.

    This code is for the Nunchuk wireless controller unit. It sends user demands
    over the Wi-Fi and gives indication of this on the LED. Data is sent for a
    joystick demand and for buttons C and Z. Data received from the BalanceBot is
    sent transparently to the serial monitor. This version includes the sending of
    step responses from the keyboard to assess control loop stability:

    B - step backward (joystick -Y)
    F - step forward (joystick +Y)
    L - turn left (joystick -X)
    R - turn right (joystick +X)
    
*/
// Declare constants
#define interval 40 // main loop period is 40 ms
#define LEDPin 13   // LED pin

// Declare libraries
#include <Wire.h>           // Include the Wire.h library for I2C comms
#include <SoftwareSerial.h> // Included serial for WiFi comms

SoftwareSerial SoftSerial(12, 11);  // Rx pin = 12, Tx pin = 11

// Declare and initialise global variables
char cmdMode;                       // keyboard command
byte keyBit[32];                    // keyboard bit commands
char keyChar;                       // any keyboard character
int keyCnt = 0;                     // current number of bytes in keyBit[] array
int keyVal;                         // any keyboard value
int LedON = 0;                      // LED flash counter
unsigned long nextMillis = 0;       // loop timer trigger
int nunchuk_address = 0x52;         //Nunchuk I2C address (0x52)
byte received_data[6];              //Declare some global byte variables
int reportEn = 0;                   //reports data for testing if > 0
byte RxByte = 0;
byte send_byte;

void setup(){
  // Start serial devices and connect to the Nunchuk over I2C
  pinMode(LEDPin, OUTPUT); digitalWrite(LEDPin, HIGH); // turn LED ON initially
  Serial.begin(9600);                           //Start the serial port at 9600 kbps
  SoftSerial.begin(9600);                       //Start the software serial port at 9600 kbps
  Wire.begin();                                 //Start the I2C as master
  TWBR = 12;                                    //Set the I2C clock speed to 400kHz
  Wire.begin();                                 //Start the I2C bus as master
  delay(10);                                    //Short delay
  Wire.beginTransmission(nunchuk_address);      //Start communication with the Nunchuck
  Wire.write(0xF0);                             //We want to write to register (F0 hex)
  Wire.write(0x55);                             //Set the register bits as 01010101
  Wire.endTransmission();                       //End the transmission
  delay(10);                                    //Short delay
  Wire.beginTransmission(nunchuk_address);      //Start communication with the Nunchuck
  Wire.write(0xFB);                             //We want to write to register (FB hex)
  Wire.write(0x00);                             //Set the register bits as 00000000
  Wire.endTransmission();                       //End the transmission
  delay(10);                                    //Short delay
  runPOST();                                    // send power-on message
  nextMillis = millis() + interval;             // prime loop timer
}

// --------------------------------------------------------------------------------

void loop(){
  // read the Nunchuk every 40 ms and send joystick data if detected
  // X/Y joystick readings are 0 - 255, centre 125. We include a deadband of 80 - 170
  // the LED flashes when a demand is being sent
  Wire.beginTransmission(nunchuk_address);      //Start communication with the Nunchuck.
  Wire.write(0x00);                             //We want to start reading at register (00 hex)
  Wire.endTransmission();                       //End the transmission
  delay(1);
  Wire.requestFrom(nunchuk_address,6);          //Request 6 bytes from the Nunchuck
  for(byte i = 0; i < 6; i++) received_data[i] = Wire.read();       //Copy the bytes to the received_data array
  send_byte = B00000000;                        //Set the send_byte variable to 0
  if(received_data[0] < 80)send_byte |= B00000001;   //If the variable received_data[0] is smaller then 80 set bit 0 of the send byte variable
  if(received_data[0] > 170)send_byte |= B00000010;  //If the variable received_data[0] is larger then 170 set bit 1 of the send byte variable
  if(received_data[1] < 80)send_byte |= B00001000;   //If the variable received_data[1] is smaller then 80 set bit 3 of the send byte variable
  if(received_data[1] > 170)send_byte |= B00000100;  //If the variable received_data[1] is larger then 170 set bit 2 of the send byte variable
  if((received_data[5] & 1) == 0)send_byte |= B00010000;  //If Z button pressed set bit 4
  if((received_data[5] & 2) == 0)send_byte |= B00100000;  //If C button pressed set bit 5
  if(send_byte > 0){
    SoftSerial.print((char)send_byte);    //Send the send_byte variable if it's value is larger then 0
    if (LedON < 1) {LedON = 4;}           // Set LED to flash
  } else if (keyCnt > 0) {
    // if bytes in FIFO then send them
    send_byte = keyBit[0]; SoftSerial.print((char)send_byte);
    keyFIFO_shift();
  }

  // see if Nunchuk data reporting is enabled?
  if (reportEn > 0) {
    while (SoftSerial.available()) {
      RxByte = SoftSerial.read(); // discard any received data from BalanceBot
    } reportEn++;
    if (reportEn > 25) {
      // report every second
      reportEn = 1; ReportNunchukData();
    }
  }

  // now wait for loop timer
  while (nextMillis > millis()) {
    // wait for timer to catch up
    if (reportEn < 1) {
      // Rx transparent mode
      if (SoftSerial.available()) {RxByte = SoftSerial.read(); Serial.write(RxByte);}
    } readKey();
  }
  nextMillis = nextMillis + interval;     //Create a 40 millisecond loop delay
  
  // flash LED if joystick data is being sent
  if (LedON > 0) {
    if (LedON == 4) {digitalWrite(LEDPin, HIGH);}
    else {digitalWrite(LEDPin, LOW);}
  }
  else if (LedON < -25) {
    // blink every second if no demand is being sent
    digitalWrite(LEDPin, HIGH); LedON = 2;
  } LedON--;
}

// ----------------------------------------------------------------------

void doCmd() {
  // perform a keyboard command
    switch (cmdMode) {
      case ' ': break;
      case 'B':
        // store a 'backward' step to be sent
        keyBit[keyCnt] = B00001000; keyCnt++; break;
      case 'F':
        // store a 'forward' step to be sent
        keyBit[keyCnt] = B00000100; keyCnt++; break;
      case 'L':
        // store a 'left' step to be sent
        keyBit[keyCnt] = B00000001; keyCnt++; break;
      case 'R':
        // store a 'right' step to be sent
        keyBit[keyCnt] = B00000010; keyCnt++; break;
    }
    // now reset the variables
    cmdMode = ' ';
}

// ----------------------------------------------------------------------

void keyFIFO_shift () {
  // if bytes are in keyBit[] array then shift them down
  if (keyCnt == 1) {
    // clear last byte in the keyBit[] array
    keyBit[0] = 0; keyCnt = 0;
  } else if (keyCnt > 1) {
    // shift the bytes in the keyBit[] array down one and decrement counter
    for (int zI = 0; zI < keyCnt; zI++) {
      keyBit[zI] = keyBit[zI+1];
    } keyBit[keyCnt] = 0; keyCnt--;
  }
}

// ----------------------------------------------------------------------

void readKey() {
  // reads a key from the keyboard annd reacts accordingly
  keyVal = Serial.read();
  if (keyVal != -1) {
    keyChar = char(keyVal);
    switch (keyChar) {
      case 'b': cmdMode = 'B'; break;
      case 'B': cmdMode = 'B'; break;
      case 'f': cmdMode = 'F'; break;
      case 'F': cmdMode = 'F'; break;
      case 'l': cmdMode = 'L'; break;
      case 'L': cmdMode = 'L'; break;
      case 'r': cmdMode = 'R'; break;
      case 'R': cmdMode = 'R'; break;
    }
  } doCmd();
}

// --------------------------------------------------------------------------------

void ReportNunchukData() {
  // send nunchuk data to the serial port, with formatting
  Serial.println("Nunchuk Data:"); // send title
  for(byte i = 0; i < 6; i++) {
    Serial.print("Bite "); Serial.print(i); Serial.print(" = ");
    if (received_data[i] < 100) Serial.print(" ");
    if (received_data[i] < 10) Serial.print(" ");
    Serial.print(received_data[i]);       //output I2C data from nunchuk
    Serial.print("\t");
    if (received_data[i] < 128) Serial.print("0");
    if (received_data[i] < 64) Serial.print("0");
    if (received_data[i] < 32) Serial.print("0");
    if (received_data[i] < 16) Serial.print("0");
    if (received_data[i] < 8) Serial.print("0");
    if (received_data[i] < 4) Serial.print("0");
    if (received_data[i] < 2) Serial.print("0");
    Serial.println(received_data[i],BIN);       //output I2C data from nunchuk
  }
  Serial.print("\n\n\n\n\n"); // send 5 blank lines
}

// --------------------------------------------------------------------------------

void runPOST() {
  // send power-on messages  
  Serial.println("\n\n\n\nNunchuk Remote v0.03\n");
  if (reportEn < 1) {
    Serial.println("Entering transparent mode...\n");
  }
}

// --------------------------------------------------------------------------------


